import se.datadosen.jalbum.*;

import java.awt.*;
import java.awt.image.BufferedImage;

import java.util.*;

/**
 * JAlbum filter that makes sure that all images/thumbnails get the same shape
 * as specified by image size or thumbnail size. The effect is accomplished by cropping.
 * Always use with "prescale" attribute
 * @author David Ekholm, JAlbum 2014
 * @version 1.2
 */
public class FixedShapeFilter implements JAFilter, ModifiesSize {
    
    private float xWeight = 0.5f;
    private float yWeight = 0.2f;   

    public float getXWeight() {
        return xWeight;
    }

    /**
     * Control what area of the image to use when cropping occurs
     * Defaults to 50%, i.e focus on the middle part of the source image
     * Can be overridden by the "xWeight" user variable
     * @since 1.1
     */
    public void setXWeight(float xWeight) {
        this.xWeight = xWeight;
    }

    public float getYWeight() {
        return yWeight;
    }

    /**
     * Control what area of the image to use when cropping occurs
     * Defaults to 20%, i.e focus on the topmost part of the source image
     * Can be overridden by the "yWeight" user variable
     * @since 1.1
     */
    public void setYWeight(float yWeight) {
        this.yWeight = yWeight;
    }    
    
    @Override
    public String getName() {
        return "Fixed shape filter";
    }

    // Implements JAFilter
    @Override
    public String getDescription() {
        return "Makes all images the same fixed shape set by the user interface by cropping them if needed";
    }

    // Implements JAFilter
    @Override
    public BufferedImage filter(BufferedImage bi, java.util.Map vars) {
    	bi = doFilter(bi, vars);
    	return bi;
    }

    // Implements JAFilter
    private BufferedImage doFilter(BufferedImage bi, java.util.Map vars) {
        Dimension sDim = new Dimension(bi.getWidth(), bi.getHeight());
        Rectangle bounds = getBounds(sDim, vars);

        if (bounds.x != 0 || bounds.y != 0 || bounds.width != bi.getWidth() || bounds.height != bi.getHeight()) {
            return bi.getSubimage(bounds.x, bounds.y, bounds.width, bounds.height);
        } else {
            return bi; // Already right shape
        }
    }

    private Rectangle getBounds(Dimension sDim, Map vars) {
        AlbumBean engine = (AlbumBean)vars.get("engine");
        int stage = (vars.get("stage") != null) ? ((Integer)vars.get("stage")) : 4;
        Dimension dDim = engine.getSizeForStage(stage);

        float hw = xWeight;
        if (vars.containsKey("xWeight")) {
            Object xWeight = vars.get("xWeight");
            hw = (xWeight instanceof Double) ? ((Double)xWeight).floatValue() : Float.parseFloat((String)xWeight);
        }
        float vw = yWeight;
        if (vars.containsKey("yWeight")) {
            Object yWeight = vars.get("yWeight");
            vw = (yWeight instanceof Double) ? ((Double)yWeight).floatValue() : Float.parseFloat((String)yWeight);
        }
        
        double sAspekt = ((double)sDim.width) / sDim.height;
        double dAspekt = ((double)dDim.width) / dDim.height;

        if (sAspekt > dAspekt) { // Crop image to the left and right to fit

            int newWidth = (int)(dAspekt * (sDim.height+0.5));

            return new Rectangle((int)((sDim.width - newWidth) * hw), 0, newWidth, sDim.height);
        } else if (sAspekt < dAspekt) { // Crop image to the top and bottom to fit

            int newHeight = (int)((sDim.width+0.5) / dAspekt);

            return new Rectangle(0, (int)((sDim.height - newHeight) * vw), sDim.width, newHeight);
        } else {
            return new Rectangle(0,0, sDim.width, sDim.height); // Already right shape
        }        
    }
    
    private Dimension parseSize(String sizeString) {
        String size = sizeString.toLowerCase();
        StringTokenizer tokens = new StringTokenizer(size, "x ");
        int w;
        int h;
        w = Integer.parseInt(tokens.nextToken());
        h = Integer.parseInt(tokens.nextToken());

        return new Dimension(w, h);
    }

    // Implements ModifiesSize
    @Override
    public Dimension getModifiedSize(Dimension originalSize, Map vars) {
        Dimension dim = getBounds(originalSize, vars).getSize();
        return dim;
    }
}
