package tiger;

/*
	Gui.java - User Interface
*/

import java.io.*;
import java.util.*;
import java.awt.Color;
import java.awt.event.*;
import java.awt.BorderLayout;
import java.awt.Component;
import java.awt.Dimension;
import java.awt.Font;
import java.beans.PropertyChangeListener;
import java.net.MalformedURLException;
import java.util.logging.Level;
import java.util.logging.Logger;
import javax.swing.*;
import javax.swing.BorderFactory;
import javax.swing.JButton;
import javax.swing.border.Border;
import javax.swing.filechooser.FileNameExtensionFilter;
import edu.stanford.ejalbert.BrowserLauncher;
import com.kitfox.svg.app.beans.SVGIcon;

import icons.Lazaicon;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import net.jalbum.component.JDraggableList;

import se.datadosen.util.Item;
import se.datadosen.util.IO;
import se.datadosen.component.*;
import se.datadosen.component.DeferredSVGIcon;
import se.datadosen.jalbum.AlbumObject;
import se.datadosen.jalbum.JAlbumContext;
import se.datadosen.jalbum.PluginContext;
import se.datadosen.jalbum.EditPanel;
import se.datadosen.jalbum.SkinProperties;
import se.datadosen.jalbum.AccountManager;
import se.datadosen.jalbum.SignInManager;
import se.datadosen.jalbum.JAlbumSite;
import se.datadosen.jalbum.AlbumBean;
import se.datadosen.jalbum.Icons;
import se.datadosen.jalbum.JAlbum;
import se.datadosen.jalbum.JAlbumColor;
import se.datadosen.jalbum.JAlbumFrame;
import se.datadosen.jalbum.ParameterException;
import se.datadosen.jalbum.event.JAlbumAdapter;
import se.datadosen.jalbum.event.JAlbumEvent;
import se.datadosen.util.Languages;
import se.datadosen.util.VersionNumber;

import net.jalbum.browser.WebViewBrowser;

/*************************************************
 * 
 *					Tiger GUI
 * 
 */

public class Gui extends GuiBase {
		
	private long skinReadyAt = Long.MAX_VALUE;
	private final VersionNumber jalbumVersion = new VersionNumber(AlbumBean.getInternalVersion());
	private final String skinVer = new SkinProperties(skinDirectory).getProperty(SkinProperties.VERSION);
	private final String supportForum = new SkinProperties(skinDirectory).getProperty(SkinProperties.SUPPORT_FORUM);
	private final String helpRoot = "http://jalbum.net/help/en/Skin/Tiger3";
	
	private boolean isSkinReady() {
		// Within 1s of launch
		//System.out.println("skinReady()? " + (System.currentTimeMillis() - skinReadyAt));
		return (System.currentTimeMillis() - skinReadyAt) > 1000L;
	}
	
	private String getStyleName() {
		String style = engine.getStyle();
		return style.substring(0, style.indexOf("."));
	}
	
	private boolean isLightStyle() {
		String s = getStyleName();
		return "Cardboard,Creme,Exhibition,Hemp,Leather Light,Retro,Ricepaper,Shine,Stonewall,Wallpaper,White,Wood light,Yellow".contains(s);
	}
	
	private String getSkinName() {
		return engine.getSkin();
	}
	
	private String getLastSkinName() {
		String ls = null;
		
		try {
			ls = engine.getLastSkin();
		} catch(Throwable t) {
			JAlbum.logger.log(Level.FINER, "Last skin is unknown.");
		}
		
		return ls;
	}	
	
	private String unCamelCase(String s) {
		
		if (s.length() > 0) {
			s = s.replaceAll("([a-z])([A-Z]+)", "$1 $2");
			return Character.toUpperCase(s.charAt(0)) + s.substring(1);
		}
		
		return "";
	}
	
	private String getText(String t) {
		String s;
		
		try {
			s = texts.getString(t);
		} catch (java.util.MissingResourceException e) {
			JAlbum.logger.log(Level.FINE, "Missing text: \"{0}\"", t);
			if (t.startsWith("ui.")) {
				t = t.substring(3);
			}
			s = unCamelCase(t);
		}
		
		return s;
	}
	
	private final int uiHeight = 600;
	private final int uiWidth = 640;
	private final int previewWidth = 560;
	private final String tabStyle = "style='padding:3px 4px;margin:3px 4px;'";
	private final String CUSTOM = "<!--custom-->";
	
	private String getFonts(String html, String ff, String hf) {
		String	gf = "",
				fw = "",
				bw = "bold",
				hw = "";
		String[] w;
		
		if (ff.contains(":")) {
			// Google font
			gf = ff;
			fw = ff.split(":")[1];
			if (fw.length() > 0) {
				w = fw.split(",");
				bw = (w.length > 2)? w[2] : w[1];
				fw = w[0];
			}
			
			ff = ff.split(":")[0];
			if (ff.contains(" ")) {
				ff = "\"" + ff + "\"";
			}
			ff += ",sans-serif";
		}

		if (!hf.equals("")) {
			if (!hf.equals(gf)) {
				gf = (gf.length() > 0)? (gf + "|" + hf) : hf;
			}
			if (hf.contains(":")) {
				// Has weight info
				hw = hf.split(":")[1];
				if (hw.length() > 0) {
					// Get base weight
					hw = hw.split(",")[0];
				}
				hf = hf.split(":")[0];
			}
			if (hf.contains(" ")) {
				hf = "\"" + hf + "\"";
			}
			hf += ",sans-serif";
		}
		
		//System.out.println("Google font: " + gf);
		return html.replace("{googleFontsLink}", (gf.length() == 0)? "" : ("\t<link rel=\"stylesheet\" href=\"http://fonts.googleapis.com/css?family=" + gf.replaceAll(" ", "+") + "&display=swap\">\n"))
			.replace("{fontFamily}", ff)
			.replace("{fontWeight}", (fw.equals("")? "300" : fw))
			.replace("{boldWeight}", bw)
			.replace("{headlineFont}", (hf.equals(""))? "inherit" : hf)
			.replace("{headlineWeight}", (hw.equals("")? "400" : hw));
		
	}
	
	private String attemptSignIn() {
		SignInManager mgr = SignInManager.getInstance();
		if ( mgr != null && mgr.isSignedIn() ) {
			return "&cid=" + AccountManager.getCid(mgr.getUserName(), mgr.getPassword());
		}
		return "";
	}
	
	//private String licensee = licenseManager.isLicenseValid()? licenseManager.getUserName() : "";
	
	private String license = licenseManager.isLicenseValid()? licenseManager.getLicenseType() : "";
	
	private boolean commercialUseAllowed = license != null && license.length() > 0 && "pro".equals(license);
	
	private Font mono = new Font("monospaced", Font.PLAIN, 12);
	
	private Border emptyBorder = BorderFactory.createEmptyBorder();
	
	private String getFileContents(String name) {
		StringBuilder cont = new StringBuilder();
		String line;
		String nl = System.getProperty("line.separator");
		File f = new File(skinDirectory, name);
		
		if (f.exists()) {
			try {
				try (java.io.BufferedReader in = new java.io.BufferedReader(new java.io.FileReader(f))) {
					while ((line = in.readLine()) != null) {
						cont.append(line);
						cont.append(nl);
					}
				}
			} catch (FileNotFoundException e) {
				return null;
			} catch (IOException e) {
				return null;
			}
		}
		return cont.toString();
	}
	
	private void writeFile(String fileName, String cont) {
		File out = new File(fileName);
		
		try {
			FileWriter writer = new FileWriter(out, false);
			writer.write(cont);
			writer.close();
		} catch (IOException e) {
			JAlbum.logger.log(Level.FINE, "Error writing \"{0}\".", fileName);
		}
	}
	
	private final Double FULLY_TRANSP = 0.0001D;
	private final Double FULLY_OPAQUE = 0.9999D;
	
	private Color toColor(JColorSelector cs) {
		return toColor(cs.toString());
	}
	
	private static final Pattern RGBA_PATTERN = Pattern.compile("rgba?\\((\\d+),(\\d+),(\\d+)(,([\\d\\.]+))?\\)");
	
	private Color toColor(String c) {
		int a = 255;
		Color clr = Color.GRAY;
		
		//System.out.print("toColor(" + c + ") -> ");
		if (c.startsWith("rgb")) {
			Matcher matcher = RGBA_PATTERN.matcher(c);
			if (matcher.matches()) {
				// Get the color components from the matched groups
				if (matcher.groupCount() > 5) {
					a = Math.min(255, (int)Math.round(Double.parseDouble(matcher.group(5)) * 255.0));
				}
				
				if (a == 255) {
					clr = new Color(
						Math.min(255, Integer.parseInt(matcher.group(1))),
						Math.min(255, Integer.parseInt(matcher.group(2))),
						Math.min(255, Integer.parseInt(matcher.group(3)))
					);
				} else {
					clr = new Color(
						Math.min(255, Integer.parseInt(matcher.group(1))),
						Math.min(255, Integer.parseInt(matcher.group(2))),
						Math.min(255, Integer.parseInt(matcher.group(3))),
						a
					);
				}
			}
			
			return clr;
		}
		
		if (c.charAt(0) == '#') {
			c = c.substring(1);
		}
		
		if (c.length() > 7) {
			a = Integer.parseInt(c.substring(0, 2), 16);
			c = c.substring(2);
		}
		
		if (c.length() == 3) {
			clr = new Color(
					Math.min(255, Integer.parseInt(c.substring(0,1), 16) * 17),
					Math.min(255, Integer.parseInt(c.substring(1,2), 16) * 17),
					Math.min(255, Integer.parseInt(c.substring(2), 16) * 17)
				);
		}
		
		if (c.length() == 6) {
			clr = new Color(
					Math.min(255, Integer.parseInt(c.substring(0,2), 16)),
					Math.min(255, Integer.parseInt(c.substring(2,4), 16)),
					Math.min(255, Integer.parseInt(c.substring(4), 16)),
					a
				);
			//System.out.println(clr.getAlpha() + " alpha=" + a);
		}
		
		return clr;
	}
	
	private String getCssColor(JColorSelector cs) {
		
		return getCssColor(toColor(cs));
	}
	
	private String getCssColor(String c) {
		
		return getCssColor(toColor(c));
	}
	
	private String getCssColor(Color c) {
		
		if (c == null) {
			return "transparent";
		}
		
		Double a = c.getAlpha() / 255.0;
		
		if (a < FULLY_TRANSP) {
			// Transparent
			return "transparent";
		} else if (a < FULLY_OPAQUE) {
			// Semi transparent
			return "rgba(" + c.getRed() + "," + c.getGreen() + "," + c.getBlue() + "," + (Math.round(a * 1000) / 1000.0) + ")";
		}
		
		// Opaque
		return "rgba(" + c.getRed() + "," + c.getGreen() + "," + c.getBlue() + ",1)";
	}
	
	private Double getAlpha(JColorSelector cs) {
		Color c = toColor(cs);
		
		return c.getAlpha() / 255.0;
	}
	
	private Double getAlpha(Color c) {
		return c.getAlpha() / 255.0;
	}
	
	private Double getAlpha(String clr) {
		Color c = toColor(clr);
		
		return c.getAlpha() / 255.0;
	}
	
	private String setAlpha(String clr, Double a) {
		Color c = toColor(clr);
		
		return "rgba(" + c.getRed() + "," + c.getGreen() + "," + c.getBlue() + "," + a + ")";
	}
	
	private Color flatColor(JColorSelector bc, JColorSelector fc) {
		
		return flatColor(toColor(bc), toColor(fc));
	}
	
	private Color flatColor(Color bc, Color fc) {
		Double a = fc.getAlpha() / 255.0;
			
		if (a < FULLY_TRANSP) {
			return bc;
		} else if (a > FULLY_OPAQUE) {
			return fc;
		}
		
		return new Color(
			(int)Math.max(0, Math.min(Math.round(a * fc.getRed() + (1 - a) * bc.getRed()), 255)),
			(int)Math.max(0, Math.min(Math.round(a * fc.getGreen() + (1 - a) * bc.getGreen()), 255)),
			(int)Math.max(0, Math.min(Math.round(a * fc.getBlue() + (1 - a) * bc.getBlue()), 255))
		);
	}
	
	private String getMiddleColor(String c1, String c2) {
		Color c = getMiddleColor(toColor(c1), toColor(c2));
		
		return "rgba(" + c.getRed() + "," + c.getGreen() + "," + c.getBlue() + "," + (Double.valueOf(c.getAlpha()) / 255.0) + ")";
	}
	
	private Color getMiddleColor(Color c1, Color c2) {
					
		return new Color(
				(int)Math.round((c1.getRed() + c2.getRed()) / 2),
				(int)Math.round((c1.getGreen() + c2.getGreen()) / 2),
				(int)Math.round((c1.getBlue() + c2.getBlue()) / 2),
				(int)Math.round((c1.getAlpha() + c2.getAlpha()) / 2)
			);
	}

	
	private String getFlatColor(JColorSelector bc, JColorSelector fc) {
		
		return getCssColor(flatColor(toColor(bc), toColor(fc)));
	}
	
	private String getFlatColor(String bc, String fc) {
		
		return getCssColor(flatColor(toColor(bc), toColor(fc)));
	}
	
	private String getFlatColor(Color bc, Color fc) {
		
		return getCssColor(flatColor(bc, fc));
	}
	
	private Double getLuminosity(JColorSelector bg, JColorSelector fg) {
		
		return getLuminosity(flatColor(bg, fg));
	}
			
	private Double getLuminosity(JColorSelector cs) {
		
		return getLuminosity(toColor(cs));
	}
	
	private Double getLuminosity(String c) {
		
		return getLuminosity(toColor(c));
	}
	
	private Double getLuminosity(Color c) {
		
		return	0.0011725490196078 * c.getRed() + 
				0.0023019607843137 * c.getGreen() +
				0.0004470588235294118 * c.getBlue();
	}
	
	private boolean isLightColor(JColorSelector cs) {
		
		return getLuminosity(cs) > 0.5;
	}
	
	private boolean isLightColor(String fc) {
		
		return getLuminosity(fc) > 0.5;
	}
	
	private boolean isLightColor(JColorSelector fc, double threshold) {
		
		return getLuminosity(fc) > threshold;
	}
	
	private boolean isLightColor(String fc, double threshold) {
		
		return getLuminosity(fc) > threshold;
	}
	
	private boolean isLightColor(JColorSelector bc, JColorSelector fc) {
		
		return isLightColor(toColor(bc), toColor(fc), 0.5);
	}
	
	private boolean isLightColor(String bc, String fc) {
		
		return isLightColor(toColor(bc), toColor(fc), 0.5);
	}
	
	private boolean isLightColor(JColorSelector bc, JColorSelector fc, Double threshold) {
		
		return isLightColor(toColor(bc), toColor(fc), threshold);
	}
	
	private boolean isLightColor(String bc, String fc, Double threshold) {
		
		return isLightColor(toColor(bc), toColor(fc), threshold);
	}
	
	private boolean isLightColor(Color bc, Color fc, Double threshold) {
		
		Double a = fc.getAlpha() / 255.0;
		
		if (a > FULLY_OPAQUE) {
			return getLuminosity(fc) >= threshold;
		}
		
		return getLuminosity(flatColor(bc, fc)) >= threshold;
	}
	
	private String getLegibleColor(Color bc1, Color bc2, Color fc, Double f) {
		Color bc = flatColor(bc1, bc2);
		
		return getLegibleColor(bc, fc, f);
	}
		
	private String getLegibleColor(JColorSelector bc, JColorSelector fc) {
		return getLegibleColor(toColor(bc), toColor(fc), 0.6);
	}
	
	private String getLegibleColor(JColorSelector bc, JColorSelector fc, Double f) {
		return getLegibleColor(toColor(bc), toColor(fc), f);
	}
	
	private String getLegibleColor(JColorSelector bc1, JColorSelector bc2, JColorSelector fc) {
		return getLegibleColor(flatColor(bc1, bc2), toColor(fc), 0.6);
	}
	
	private String getLegibleColor(JColorSelector bc1, JColorSelector bc2, JColorSelector fc, Double f) {
		return getLegibleColor(flatColor(bc1, bc2), toColor(fc), f);
	}
	
	private String getLegibleColor(Color bc, Color fc, Double f) {
		int r, g, b;
		
		if (Math.abs(getLuminosity(bc) - getLuminosity(fc)) >= f) {
			return getCssColor(fc);
		}
		
		f *= 255.0;
		
		if (getLuminosity(bc) > 0.5) {
			// Darken
			r = (int)Math.round(Math.max(fc.getRed() - f, 0));
			g = (int)Math.round(Math.max(fc.getGreen() - f, 0));
			b = (int)Math.round(Math.max(fc.getBlue() - f, 0));
		} else {
			// Lighten
			r = (int)Math.round(Math.min(fc.getRed() + f, 255));
			g = (int)Math.round(Math.min(fc.getGreen() + f, 255));
			b = (int)Math.round(Math.min(fc.getBlue() + f, 255));
		}
		
		return getCssColor(new Color(r, g, b));
	}
	
	private final String[] imageFiles = new String[] { "jpg", "jpeg", "png", "gif", "svg" };
	private final FileChooser fc = ChooserFactory.createFileChooser(window);

	private void getFileToRes(String[] extensions, JTextField name, Component c) {
		fc.setFileFilter(new FileNameExtensionFilter(getText("ui.select"), extensions));
		int returnVal = fc.showOpenDialog(c);
		
		if (returnVal == JFileChooser.APPROVE_OPTION) {
			String fn = fc.getSelectedFile().toString();
			
			if (!fn.trim().equals("")) {
				File src = new File(fn);
				File dst = new File(context.getPluginContext().getRootFolder().getFile(), "res");
				if (!dst.exists()) {
					dst.mkdir();
				}
				if (src.exists() &&  dst.exists()) {
					try {
						IO.copyFile(src, dst);
						name.setText(src.getName());
					} catch (IOException e) {
						System.out.println(e);
					}
				}
			}
		}
	}
	
	private void backupProjectFile(String ver) {
		
		try {
			File projectFile = window.projectChooser.getSelectedFile();
			IO.copyFile(projectFile, new File(projectFile.getParentFile(), IO.baseName(projectFile) + "-" + ((ver == null)? "old" : ver) + ".jap"));
		} catch (IOException ex) {
			JAlbum.logger.log(Level.FINE, "Error backing up project file: {0}", window.projectChooser.getSelectedFile());
		}
	}
	
	private StateMonitor commercialMonitor = new StateMonitor() {
		@Override
		public void onChange() {
			
			if (((JCheckBox)source).isSelected() && !commercialUseAllowed) {
				if (isSkinReady()) {
					Object[] options = { 
						getText("ui.signUp"),
						getText("ui.noThanks")
					};
					int n = JOptionPane.showOptionDialog(window, 
						getText("ui.licenseWarningText"), 
						getText("ui.licenseWarningTitle"), 
						JOptionPane.YES_NO_OPTION, 
						JOptionPane.INFORMATION_MESSAGE,
						null,
						options,
						options[1]
					);
					if (n == 0) {
						try {
							BrowserLauncher.openURL(JAlbumSite.getTrueInstance().getMyJAlbumUpgradeUrl() + "/?referrer=" + skin + attemptSignIn());
						} catch (se.datadosen.tags.ElementException | IOException x) {
						}
					}
				}
				((JCheckBox)source).setSelected(false);
			}
		}
	};

	private static Icon icon(String basename) {
		return Icons.get(Lazaicon.class, "svg/" + basename + ".svg", 18);
	}
		
	private static Icon icon(String basename, int size) {
		return Icons.get(Lazaicon.class, "svg/" + basename + ".svg", size);
	}
	
	private static Icon svgIcon(String basename, Dimension dim) {
		return svgIcon(basename, dim, false);
	}
	
	private static Icon svgIcon(String basename, Dimension dim, boolean adapt) {
		DeferredSVGIcon icon = new DeferredSVGIcon(Gui.class, "graphics/" + basename + ".svg");
		
		icon.setAntiAlias(true);
		icon.setAdaptColors(adapt);
		icon.setPreferredSize(dim);
		icon.setAutosize(SVGIcon.AUTOSIZE_STRETCH);
		
		return icon;
	}
	
	private Object[] getPosition() {
		return new Object[] {
			new Item("left top", getText("ui.left_top")), 
			new Item("center top", getText("ui.center_top")), 
			new Item("right top", getText("ui.right_top")), 
			new Item("left center", getText("ui.left_middle")), 
			new Item("center center", getText("ui.center_middle")), 
			new Item("right center", getText("ui.right_middle")), 
			new Item("left bottom", getText("ui.left_bottom")), 
			new Item("center bottom", getText("ui.center_bottom")), 
			new Item("right bottom", getText("ui.right_bottom"))
		};
	}
	
	private int setSelectedValue(JComboBox<Item> box, Object val) {
		int		size = box.getItemCount();
		Item	it;
		
		for (int i = 0; i < size; i++) {
			it = box.getItemAt(i);
			if (it.item.equals(val)) {
				box.setSelectedIndex(i);
				return i;
			}
		}
		
		return -1;
	}
	
	private String getSelectedValue(Object o) {
		if (o instanceof JComboBox) {
			Object so = ((JComboBox)o).getSelectedItem();
			if (so instanceof Item) {
				return ((Item)so).value.toString();
			}
			return (so == null)? null : so.toString();
		}
		return (o == null)? null : o.toString();
	}
	
	private int getSpinnerValueAsInt(JSpinner o) {
		return (Integer)o.getValue();
	}
	
	private Double getSpinnerValueAsDouble(JSpinner o) {
		return (Double)o.getValue();
	}
	
	private Double getSelectedValueAsDouble(Object o) {
		if (o instanceof JComboBox) {
			Object so = ((JComboBox)o).getSelectedItem();
			if (so instanceof Item) {
				String s = ((Item)so).value.toString();
				return ((s != null) && (s.length() > 0))? Double.parseDouble(s) : null;
			}
			return null;
		}
		return (o == null)? null : Double.parseDouble(o.toString());
	}
	
	private Integer getSelectedValueAsInt(JComboBox o) {
		Object so = ((JComboBox)o).getSelectedItem();
		if (so instanceof Item) {
			return Integer.parseInt(((Item)so).value.toString());
		}
		return null;
	}
	
	private Double getThumbAspectRatio() {
		String[] cdim = engine.getThumbSize().split("x");
		Double	w = Double.parseDouble(cdim[0]),
				h = Double.parseDouble(cdim[1]);
		
		if (w != null && h != null) {
			return w / h;
		}
		
		JAlbum.logger.log(Level.FINE, "Invalid thumbnail dimensions: {0} x {1}", new Object[]{ w.toString(), h.toString() });
		return null;
	}
		
	private Date skinLoadedOn = new Date();
		
	private void allowHTMLEditing(JSmartTextArea ta) {
		try {
			ta.setAllowHTMLEditing(true).setFullHTMLEditing(true);
		} catch (Throwable e) {
		}
	}
	
	private void allowSpelling(JSmartTextArea ta) {
		try {
			ta.spelling();
		} catch (Throwable e) {
		}
	}
	
	private String majorVersion(String s) {
		int i = s.indexOf(".");
		if (i > 0) {
			return s.substring(0, i);
		}
		return s;
	}
	
	private int getMajorVersion(Object v) {
		
		if (v == null) {
			return -1;
		} else if (v instanceof Integer) {
			return (int)v;
		} else if (v instanceof String) {
			try {
				return Integer.valueOf(majorVersion((String)v));
			} catch(NumberFormatException ex) {
				return -1;
			}
		} else if (v instanceof Double) {
			return (int)Math.floor((Double)v);
		}
		return -1;
	}
	
	Boolean checkBooleanProperty(AlbumObject[] sel, String pn) {
		
		if (sel.length > 1) {
			boolean v = sel[0].getProperties().get(pn, false);
			
			for (AlbumObject ao : sel) {
				if (ao.getProperties().get(pn, false) != v) {
					return null;
				}
			}
			return v;
		} else if (sel.length == 1) {
			return sel[0].getProperties().get(pn, false);
		} else {
			return null;
		}
	}
	
	void createMenuItemForBooleanProperty(JMenu menu, String pn, String title) {
		
		createMenuItemForBooleanProperty(menu, window.albumExplorer.explorer.getSelectedAlbumObjects(), pn, title);
	}
	
	void createMenuItemForBooleanProperty(JMenu menu, AlbumObject[] sel, String pn, String title) {		
		Boolean cv = checkBooleanProperty(sel, pn);

		try {
			if (cv == null) {
				// Indefinite value -> Add submenu
				JMenu subMenu = new JMenu(title);
				JMenuItem all = new JMenuItem(getText("ui.all"));
				all.addActionListener(ae -> {
					for (AlbumObject ao : sel) {
						ao.getProperties().put(pn, true);
						ao.getProperties().save();
					}				
				});
				JMenuItem none = new JMenuItem(getText("ui.none"));
				none.addActionListener(ae -> {
					for (AlbumObject ao : sel) {
						ao.getProperties().put(pn, false);
						ao.getProperties().save();
					}				
				});
				subMenu.add(all);
				subMenu.add(none);
				menu.add(subMenu);
	
			} else {
				// All items has the same value -> invert them when clicked
				JCheckBoxMenuItem menuItem = new JCheckBoxMenuItem(title, null, cv);
	
				menuItem.addActionListener(ae -> {
					for (AlbumObject ao : sel) {
						ao.getProperties().put(pn, !cv);
						ao.getProperties().save();
					}
				});
	
				menu.add(menuItem);
			}
			
		} catch (Throwable ex) {
			JAlbum.logger.log(Level.INFO, "Right-click menu is available only in jAlbum v32 and newer!");
		}
	}
	
	JLinkLabel safeLinkLabel(javax.swing.JComponent comp, String url, String txt) {
		
		if (jalbumVersion.compareTo(new VersionNumber("32.1.4")) >= 0) {
			try {
				return new JLinkLabel(comp, txt);
			} catch (Throwable ex) {
				
			}
		}
		
		return new JLinkLabel(url, txt);
	}
	
	/*****************************************
	 *
	 *			Skin UI starts here
	 *
	 *****************************************/

	ControlPanel skinUi = new ControlPanel() {
		
		// UI-wide variables
		
		// Icons used multple places
		private Icon infoIcon = icon("info");
		private Icon mandatory = svgIcon("asterisk", new Dimension(16, 16));
		
		// JLinkLabels need to be difned globally 
		private JLabel siteDesignLink;
		private JLabel typographyLink;
		private JLabel lightboxDesignLink;
		private JLabel titleCaptionLink;
		private JLabel folderCaptionLink;
		private JLabel thumbCaptionLink;
		private JLabel imageCaptionLink;
		private JLabel markNewImagesLink;
		private JLabel newImagesSectionLink;
		private JLabel pagesLink;
		private JLabel topNavPagesLink;
		private JLabel topBarCCLink;
		private JLabel footerCCLink;
		private JLabel albumInfoLink;
		private JLabel customSectionLink;
		private JLabel customCodeLink;
		private JLabel customKeysLink;
		
		// silence further notifications
		private boolean skinChangeReported = false;
		
		private void selectComboBoxItem(JComboBox<Item> cb, JTextArea ta) {
			selectComboBoxItem(cb, ta.getText());
		}
		
		private void selectComboBoxItem(JComboBox<Item> cb, String val) {
			int size = cb.getModel().getSize();
			
			if (size == 0) {
				return;
			}
			
			ComboBoxModel<Item> model = cb.getModel();
			
			if (val != null) {
				for (int i = 0; i < size; i++) {
					if (model.getElementAt(i).value.equals(val)) {
						//System.out.println("Found preset " + i + " = \"" + val + "\"");
						cb.setSelectedIndex(i);
						return;
					}
				}
			}
			
			//System.out.println("Not found preset \"" + val + "\"");
			cb.setSelectedIndex(model.getSize() - 1);
		}
		/*
		public boolean compareObjects(Object obj1, Object obj2) {
			if (obj1 == obj2) {
				return true;
			}
			if (obj1 == null || obj2 == null) {
				return false;
			}

			if (obj1 instanceof String && obj2 instanceof String) {
				return ((String) obj1).equals((String) obj2);
			} else if (obj1 instanceof Integer && obj2 instanceof Integer) {
				return ((Integer) obj1).equals((Integer) obj2);
			} else if (obj1 instanceof Boolean && obj2 instanceof Boolean) {
				return ((Boolean) obj1).booleanValue() == ((Boolean) obj2).booleanValue();
			} else {
				// Handle other types or return false for incompatible types
				return false;
			}
		}
		*/
		@Override
		public void importVariables(Map<String, Object> vars) {
			
			//JAlbum.logger.log(Level.FINE, "Project file loaded: {0}", window.projectChooser.getSelectedFile());
			//JAlbum.logger.log(Level.FINE, "{0}", vars.keySet());
			
			if (skinChangeReported || window.projectChooser.getSelectedFile() == null) {
				return;
			}
			
			Map<String, Object> skinVars = engine.getSkinVariables();
			int	omv = getMajorVersion(skinVars.get("majorSkinVersion")),
				cmv = getMajorVersion(skinVer);
			/*
			for (Map.Entry<String, Object> entry : skinVars.entrySet()) {
				String key = entry.getKey();
				Object value = entry.getValue();
				Object val = vars.get(key);
				if (!compareObjects(value, val)) {
					System.out.println("\"" + key + "\" diff: \"" + value + "\" <> \"" + val + "\"");
				}
			}
			*/
			if (omv == -1) {
				// Fallback to Tiger 3.0.x variable
				omv = getMajorVersion(skinVars.get("skinVersion"));
				if (omv == -1) {
					// Guess if v3+
					if (skinVars.get("zoomSlider") != null) {
						omv = 4;
					}
				}
			} else {
				String ht = (String)vars.get("heroType");
				//String hit = (String)vars.get("heroImageType");
				if (ht != null) {
					// Version 3
					// Import hero bg image settings and color
					//JAlbum.logger.log(Level.FINE, "heroType=\"{0}\", heroImageType=\"{1}\"", new Object[] {(ht==null)? "null":ht, (hit==null)? "null":hit});

					vars.put("heroImageType", ht.equals("flatColor")? "" : ht);
					
					String hc = (String)vars.get("heroColor");
					if (hc != null) {
						vars.put("heroOverlayColor", hc);
						// Allow only if no background image was used
						vars.put("heroOverlayColorTransparent", !ht.equals("flatColor"));
					}

					// Old hoverEffect conversion
					Boolean he = (Boolean)vars.get("hoverEffectThumbs");
					if (he != null) {
						vars.put("hoverEffect", he? "hover-zoom" : "");
					}
				}
			}
			
			Boolean dsb = (Boolean)vars.get("dontStretchBehind");
			if (dsb != null) {
				vars.put("fitImagesBetween", (dsb == true)? "panels" : "screen");
			}
			
			Boolean emw = (Boolean)vars.get("enableMouseWheel");
			if (emw != null) {
				vars.put("mouseWheelAction", emw? "navigation" : "");
			}
			
			if (vars.get("topNavigationHideLong") == null) {
				// Version < 6.6
				// Variable omv is unreliable after skinModel got infected with the wrong value of "4"
				String gak = (String)vars.get("googleApiKey");
				if (gak != null && gak.trim().length() == 0) {
					Boolean sms = (Boolean)vars.get("showMapSection");
					if (sms) {
						vars.put("showMapSection", false);
					}
					sms = (Boolean)vars.get("showMap");
					if (sms) {
						vars.put("showMap", false);
					}
				}
			}
			
			String tl = (String)vars.get("thumbLayout");
			if (tl != null && tl.equals("horizontal")) {
				vars.put("thumbLayout", "justified");
			}
			
			//JAlbum.logger.log(Level.FINE, "skinVars = {" + skinVars.toString() + "}");
			// Old and current major versions
			// Old skin name
			String	lastSkin = getLastSkinName();
			boolean skinChanged = lastSkin != null && !getSkinName().equals(lastSkin);
			
			// Attempt writing out the current version
			if (cmv != -1) {
				skinVars.put("majorSkinVersion", cmv);
			}
			
			if (skinChanged || cmv > omv) {
				JNotification jn;
				
				if (skinChanged) {
					// Other skin
					JAlbum.logger.log(Level.FINE, "Skin changed: {0} -> {1} v{2}", new Object[]{lastSkin, skin, cmv});
					jn = new JNotification(getText("ui.madeWithAnotherSkin") + "\n" + 
							getText("ui.backupSaved"), JNotification.Type.SKIN);
					backupProjectFile((lastSkin != null)? lastSkin : "old");					
					jn.setExpiration(3);
					jn.setRememberDismissed(true);
					window.showNotification(jn);
				} else {
					// Major version change
					JAlbum.logger.log(Level.FINE, "Major version changed: {0} v{1} -> v{2}", new Object[]{skin, omv, cmv});
					/*
					jn = new JNotification(getText("ui.madeWithOldVersion").replaceAll("\\{0\\}", skin) + 
							((omv == -1)? "" : ("(" + omv + ")"))+ "\n" +
							"\n" + getText("ui.backupSaved"), JNotification.Type.SKIN);
					backupProjectFile((omv >= 0)? String.valueOf(omv) : "old");
					*/
				}
				
				skinChangeReported = true;
				// Have to process all subdirectories after a skin or major version change
				engine.setUpdatedDirsOnly(false);
			}	
		};
		/*
		private String typeOf(Object o) {
			Class c = o.getClass();
			return c.toString().replaceAll("class java\\.lang\\.", "").replaceAll("Boolean", "boolean").replaceAll("Integer", "int").replaceAll("Double", "double");
		}
		private String add_tab(String s, int tl) {
			return s + "\t\t\t\t\t\t\t\t\t\t\t\t\t\t\t\t\t\t\t\t\t\t\t".substring(0, 1 + Math.max(0, tl - s.length() - 1) / 4);
		}
		
		@Override
		public void importVariables(Map<String, Object> vars) {
			
			JAlbum.logger.log(Level.FINE, "Mapping skin variables");

			Map<String, Object> skinVars = engine.getSkinVariables();
			Map<String, Object> skinVarsSorted = new TreeMap<>(skinVars);
			String	key;
			String	ic,
					dc,
					fv,
					comment;
			Object	iv,
					dv;

			System.out.println("public class SkinModel\n{");
			for (Map.Entry<String, Object> entry : skinVarsSorted.entrySet()) {
				key = entry.getKey();
				dv = entry.getValue();
				dc = typeOf(dv);
				iv = vars.get(key);
				ic = (iv == null)? "" : typeOf(iv);
				
				if (iv == null || !dc.equals(ic)) {
					// No imported value or of different type
					fv = (dv == null)? "null" : (dc.equals("String")? ("\"" + dv.toString().replaceAll("\n", "\\n") + "\"") : dv.toString());
					comment = (iv == null)? "Missing key" : ("Incompatible types: " + dc + " =/= " + ic);
				} else {
					// Importing
					fv = ic.equals("String")? ("\"" + iv.toString().replaceAll("\"", "\\\\\"").replaceAll("\n", "\\\\n").replaceAll("\t", "\\\\t") + "\"") : iv.toString();
					comment = "";
				}
				System.out.println("\tpublic " + add_tab(dc, 12) + add_tab(key, 40) + "= " + ((comment.length() > 0)? (add_tab(fv + ";", 32) + "// " + comment) : (fv + ";")));
			}
			System.out.println("}");		
		}
		*/
		
		// styleName is stored as skin variable in order to check style change later
		private JTextField styleName = new JTextField(getStyleName());
		
		// skinVersion is stored as skin variable in order to check major skin version change later
		JTextField majorSkinVersion = new JTextField(majorVersion(skinVer));
		
		// Tracking extra size changes in order to be able to remove the stale files
		JTextField extraSizes = new JTextField(24);
		String lastExtraSizes = (String)engine.getSkinVariables().get("extraSizes");
		JCheckBox extraSizesChanged = new JCheckBox("", false);
		
		JTextField folderThumbSize = new JTextField("320x240");
		JTextField folderImageSize = new JTextField("1000x240");
		
		JComboBox maxPageWidth = new JComboBox(new Object[] {
			new Item("52.5rem", "840"),
			new Item("57.5rem", "920"),
			new Item("62.5rem", "1000"),
			new Item("67.5rem", "1080"),
			new Item("72.5rem", "1160"),
			new Item("77.5rem", "1240"),
			new Item("82.5rem", "1320"),
			new Item("87.5rem", "1400"),
			new Item("92.5rem", "1480"),
			new Item("97.5rem", "1560"),
			new Item("none", getText("ui.noLimit"))
		});
		
		// Background
		JTextField backgroundImageName = new JSmartTextField(10);
		JComboBox backgroundPos = new JComboBox(getPosition());
		JComboBox backgroundRepeat = new JComboBox(new Object[] {
			new Item("no-repeat", getText("ui.no_repeat")),
			new Item("repeat-x", getText("ui.repeat_x")),
			new Item("repeat-y", getText("ui.repeat_y")),
			new Item("repeat", getText("ui.repeat_both")),
			new Item("stretch", getText("ui.stretch")),
		});
		
		// Text, button
		JColorSelector textColor = new JColorSelector(getText("ui.textColor"), new JSmartTextField(10));
		JColorSelector linkColor = new JColorSelector(getText("ui.linkColor"), new JSmartTextField(10));
		JColorSelector hoverColor = new JColorSelector(getText("ui.hoverColor"), new JSmartTextField(10));
		JColorSelector buttonColor = new JColorSelector(getText("ui.button"), new JSmartTextField(10));
		JColorSelector backgroundColor = new JAlphaColorSelector(getText("ui.backgroundColor"), new JSmartTextField(10));
		JColorSelector topmenuColor = new JAlphaColorSelector(getText("ui.topBar"), new JSmartTextField(10));
		JColorSelector footerColor = new JAlphaColorSelector(getText("ui.footer"), new JSmartTextField(10));
		
		// Boxes
		JTextField boxBackgroundImageName = new JSmartTextField(10);
		JColorSelector boxBackgroundColor = new JAlphaColorSelector(getText("ui.backgroundColor"), new JSmartTextField(10));
		JCheckBox boxDropShadow = new JCheckBox(getText("ui.dropShadow"));
		JSpinner cornerRadius = new JSpinner(new SpinnerNumberModel(6, 0, 100, 1));
		JComboBox boxPadding = new JComboBox(new Object[] {
			new Item("none", getText("ui.none")),
			new Item("small", getText("ui.small")),
			new Item("medium", getText("ui.medium")),
			new Item("large", getText("ui.large")),
			new Item("x-large", getText("ui.xlarge"))
		});
		
		// Top bar
		JCheckBox topbarSticky = new JCheckBox(getText("ui.stickToTop"));
		JComboBox topnavigationAlignment = new JComboBox(new Object[] {
			new Item("left", getText("ui.left")),
			new Item("center", getText("ui.center")),
			new Item("right", getText("ui.right"))
		});
		JCheckBox useSearch = new JCheckBox(getText("ui.useSearch"));
		JCheckBox searchVisibleByDefault = new JCheckBox(getText("ui.searchVisibleByDefault"));
		
		// Hero
		JCheckBox heroFullWidth = new JCheckBox(getText("ui.fullWidth"));
		JSpinner folderImageHeight = new JSpinner(new SpinnerNumberModel(240, 110, 640, 10));
		JCheckBox heroImageDesaturate = new JCheckBox(getText("ui.blackAndWhite"), false);
		JCheckBox darkenTitleBackground = new JCheckBox(getText("ui.darkenTitleBackground"), false);
		JCheckBox darkenButtonBackground = new JCheckBox(getText("ui.darkenButtonBackground"), false);
		JCheckBox moveSearchToHero = new JCheckBox(getText("ui.moveSearchToHero"));
		JCheckBox showStartSlideshow = new JCheckBox(getText("ui.showStartSlideshow"), true);
		JCheckBox showHelp = new JCheckBox(getText("ui.showHelpButton"));
		JCheckBox showFullscreenButton = new JCheckBox(getText("ui.showFullscreenButton"));

		JComboBox folderTitlePosition = new JComboBox(new Object[] {
			new Item("top", getText("ui.left_top")),
			new Item("middle", getText("ui.left_middle")), 
			new Item("center", getText("ui.center_middle")), 
			new Item("bottom", getText("ui.left_bottom"))
		});
		
		JComboBox<Item> titleCaptionPresets = new JComboBox(new Object[] {
			new Item("", getText("ui.empty")),
			new Item("<h1>${title}</h1>", getText("ui.title")),
			new Item("<h1>${title}</h1>\n<div class=\"description\">${description}</div>", getText("ui.title") + " + " + getText("ui.description")),
			new Item("<h1>${title} <span class=\"date\">${folderModDate}</span></h1>\n<div class=\"description\">${description}</div>", getText("ui.title") + " + " + getText("ui.date") + " + " + getText("ui.description") + " (" + getText("ui.default") + ")"),
			new Item(CUSTOM, "[ " + getText("ui.custom") + " ]")
		});
		JTextArea titleCaptionTemplate = new JSmartTextArea(4, 28);

		JComboBox heroImageType = new JComboBox(new Object[] {
			new Item("", getText("ui.none")),
			new Item("folderImage", getText("ui.folderThumbnail")),
			new Item("albumImage", getText("ui.albumThumbnail")),
			new Item("randomImage", getText("ui.randomImage")),
			new Item("selected", getText("ui.selectedImages"))
		});

		JColorSelector heroOverlayColor = new JAlphaColorSelector(getText("ui.color"), new JSmartTextField("#222222", 10));
		JCheckBox heroOverlayColorTransparent = new JCheckBox(getText("ui.dontUse"), true);
		
		JComboBox heroPattern = new JComboBox(new Object[] {
			new Item("", getText("ui.none")),
			new Item("cardboard.png", "Cardboard"),
			new Item("circles.png", "Circles"),
			new Item("cross.png", "Cross"),
			new Item("dots.png", "Dots"),
			new Item("gplay.png", "GPlay"),
			new Item("hexa.png", "Hexagons"),
			new Item("linen.png", "Linen"),
			new Item("paper.png", "Paper"),
			new Item("pinstripe.png", "Pinstripe"),
			new Item("tiles.png", "Tiles"),
			new Item("wood.png", "Wood"),
			new Item("zebra.png", "Zebra")
		});
		
		// Typography
		private final JComboBox fontSuggestions = new JComboBox(new Object[] {
			"[" + getText("ui.suggestedFonts") +  "]",
			"Barlow Semi Condensed Light / Barlow",
			"Cinzel Decorative / Raleway",
			"Dosis Bold / Montserrat",
			"Exo 2 / Exo 2",
			"Fjalla One / Roboto",
			"IBM Plex Serif / IBM Plex Sans",
			"Jura / Roboto Condensed",
			"Martel / Roboto",
			"Merriweather / Merriweather Sans",
			"Oswald / Open Sans",
			"Raleway / Open Sans",
			"Roboto Condensed / Roboto",
			"Roboto Slab / Montserrat",
			"Special Elite / Yanone Kaffeesatz",
			"Unica One / Lato",
			"Yanone Kaffeesatz Bold / Muli"
		});

		private void setFontBoxes() {
			
			if (fontSuggestions.getSelectedIndex() == 0) {
				return;
			}
			String	s = fontSuggestions.getSelectedItem().toString(),
					hf = s.split("/")[0].trim(),
					ff = s.split("/")[1].trim();

			setSelectedValue(fontFamily, ff);
			setSelectedValue(headlineFont, hf);
			fontSuggestions.setSelectedIndex(0);
		};
		
		JComboBox fontFamily = new JComboBox(new Object[] {
			new Item("Arial, \"Helvetica Neue\", Helvetica, sans-serif", "Arial"),
			new Item("Baskerville, \"Baskerville Old Face\", \"Hoefler Text\", Garamond, \"Times New Roman\", serif", "Baskerville"),
			new Item("Calibri, Candara, Segoe, \"Segoe UI\", Optima, Arial, sans-serif", "Calibri"),
			new Item("Cambria, Georgia, Times, \"Times New Roman\", serif", "Cambria"),
			new Item("\"Century Gothic\", \"Apple Gothic\", \"Goudy Old Style\", sans-serif", "Century Gothic"),
			new Item("\"Comic Sans MS\", cursive", "Comic Sans"),
			new Item("Consolas, \"Lucida Console\", Monaco, monospace", "Consolas"),
			new Item("Constantia, Palatino, \"Palatino Linotype\", \"Palatino LT STD\", Georgia, serif", "Constantia"),
			new Item("\"Copperplate Light\", \"Copperplate Gothic Light\", serif", "Copperplate Light"),
			new Item("\"Courier New\", Courier, monospace", "Courier New"),
			new Item("\"Franklin Gothic Medium\", \"Arial Narrow Bold\", Arial, sans-serif", "Franklin Gothic"),
			new Item("Futura, \"Century Gothic\", AppleGothic, sans-serif", "Futura"),
			new Item("Garamond, \"Hoefler Text\", Times New Roman, Times, serif", "Garamond"),
			new Item("Geneva, \"Lucida Sans\", \"Lucida Grande\", \"Lucida Sans Unicode\", Verdana, sans-serif", "Geneva"),
			new Item("Georgia, Palatino, \"Palatino Linotype\", Times, \"Times New Roman\", serif", "Georgia"),
			new Item("\"Gill Sans\", \"Gill Sans MT\", Calibri, \"Trebuchet MS\", sans-serif", "Gill Sans"),
			new Item("\"Goudy Old Style\", Garamond, \"Big Caslon\", \"Times New Roman\", serif", "Goudy Old Style"),
			new Item("\"Helvetica Neue\", Helvetica, Arial, sans-serif", "Helvetica Neue"),
			new Item("\"Hoefler Text\", Constantia, Palatino, \"Palatino Linotype\", \"Book Antiqua\", Georgia, serif", "Hoefler Text"),
			new Item("Impact, Haettenschweiler, \"Arial Narrow Bold\", sans-serif", "Impact"),
			new Item("\"Lucida Sans\", \"Lucida Grande\", \"Lucida Sans Unicode\", Verdana, sans-serif", "Lucida Sans"),
			new Item("\"Lucida Bright\", Georgia, serif", "Lucida Bright"),
			new Item("Palatino, \"Palatino Linotype\", \"Book Antiqua\", Georgia, serif", "Palatino"),
			new Item("Segoe, \"Segoe UI\", Tahoma, Geneva, \"Nimbus Sans L\", sans-serif", "Segoe"),
			new Item("Tahoma, Geneva, Verdana, sans-serif", "Tahoma"),
			new Item("Times, \"Times New Roman\", Georgia, serif", "Times"),
			new Item("\"Trebuchet MS\", \"Lucida Sans Unicode\", \"Lucida Grande\", \"Lucida Sans\", Tahoma, sans-serif", "Trebuchet MS"),
			new Item("Verdana, Geneva, Tahoma, sans-serif", "Verdana"),
			new Item("Alegreya Sans:300,300i,500,500i", "Alegreya Sans Light"),
			new Item("Barlow:300,300i,600,600i", "Barlow Light"),
			new Item("Barlow:400,400i,700,700i", "Barlow"),
			new Item("Barlow Semi Condensed:300,300i,500,500i", "Barlow Semi Condensed"),
			new Item("Exo 2:300,300i,600,600i", "Exo 2"),
			new Item("Fira Sans:300,300i,600,600i", "Fira Sans"),
			new Item("IBM Plex Sans:300,300i,600,600i", "IBM Plex Sans"),
			new Item("Josefin Sans:300,300i,600,600i", "Josefin Sans"),
			new Item("Josefin Slab:300,300i,600,600i", "Josefin Slab"),
			new Item("Lato:300,300i,700,700i", "Lato"),
			new Item("Merriweather Sans:300,300i,700,700i", "Merriweather Sans"),
			new Item("Merriweather:300,300i,700,700i", "Merriweather Serif"),
			new Item("Montserrat:300,300i,600,600i", "Montserrat"),
			new Item("Muli:300,300i,700,700i", "Muli"),
			new Item("Open Sans:300,300i,600,600i", "Open Sans"),
			new Item("Raleway:300,300i,600,600i", "Raleway"),
			new Item("Roboto:300,300i,500,500i", "Roboto"),
			new Item("Roboto Condensed:300,300i,700,700i", "Roboto Condensed"),
			new Item("Saira:300,300i,600,600i", "Saira"),
			new Item("Source Sans Pro:300,300i,600,600i", "Source Sans Pro"),
			new Item("Work Sans:300,300i,600,600i", "Work Sans Light"),
			new Item("Yanone Kaffeesatz:300,500", "Yanone Kaffeesatz")
		});
		
		JComboBox headlineFont = new JComboBox(new Object[] {
			new Item("", "[ " + getText("ui.sameAsBaseFont") + " ]"),
			new Item("Abril Fatface", "Abril Fatface"),
			new Item("Alex Brush", "Alex Brush"),
			new Item("Amaranth", "Amaranth"),
			new Item("Amatic SC:700", "Amatic SC Bold"),
			new Item("Anton", "Anton"),
			new Item("Arapey", "Arapey"),
			new Item("Barlow:300", "Barlow Light"),
			new Item("Barlow:500", "Barlow Medium"),
			new Item("Barlow Semi Condensed:300", "Barlow Semi Condensed Light"),
			new Item("Barlow Condensed", "Barlow Condensed"),
			new Item("Cantata One", "Cantata One"),
			new Item("Cinzel Decorative", "Cinzel Decorative"),
			new Item("Cookie", "Cookie"),
			new Item("Dancing Script", "Dancing Script"),
			new Item("Dynalight", "Dynalight"),
			new Item("Dosis:600", "Dosis Bold"),
			new Item("Economica", "Economica"),
			new Item("Emilys Candy", "Emilys Candy"),
			new Item("Euphoria Script", "Euphoria Script"),
			new Item("Exo 2:300", "Exo 2"),
			new Item("Fauna One", "Fauna One"),
			new Item("Fjalla One", "Fjalla One"),
			new Item("Fredericka the Great", "Fredericka the Great"),
			new Item("Geo", "Geo"),
			new Item("Gilda Display", "Gilda Display"),
			new Item("Grand Hotel", "Grand Hotel"),
			new Item("Great Vibes", "Great Vibes"),
			new Item("Gruppo", "Gruppo"),
			new Item("Handlee", "Handlee"),
			new Item("IBM Plex Serif:300", "IBM Plex Serif"),
			new Item("IM Fell English", "IM Fell English"),
			new Item("Italiana", "Italiana"),
			new Item("Josefin Slab", "Josefin Slab"),
			new Item("Julius Sans One", "Julius Sans One"),
			new Item("Jura", "Jura"),
			new Item("La Belle Aurore", "La Belle Aurore"),
			new Item("Libre Baskerville", "Libre Baskerville"),
			new Item("Lobster", "Lobster"),
			new Item("Lobster Two", "Lobster Two"),
			new Item("Lora", "Lora"),
			new Item("Maiden Orange", "Maiden Orange"),
			new Item("Martel:300", "Martel"),
			new Item("Marvel:700", "Marvel Bold"),
			new Item("Medula One", "Medula One"),
			new Item("Merriweather:300", "Merriweather"),
			new Item("Mountains of Christmas", "Mountains of Christmas"),
			new Item("Noticia Text", "Noticia Text"),
			new Item("Noto Serif Display:300", "Noto Serif Light"),
			new Item("Old Standard TT", "Old Standard TT"),
			new Item("Oranienbaum", "Oranienbaum"),
			new Item("Oswald", "Oswald"),
			new Item("Philosopher", "Philosopher"),
			new Item("Poiret One", "Poiret One"),
			new Item("Prata", "Prata"),
			new Item("Princess Sofia", "Princess Sofia"),
			new Item("PT Mono", "PT Mono"),
			new Item("PT Sans Narrow", "PT Sans Narrow"),
			new Item("Raleway:300", "Raleway"),
			new Item("Raleway:600", "Raleway Bold"),
			new Item("Roboto Condensed", "Roboto Condensed"),
			new Item("Roboto Slab:300", "Roboto Slab"),
			new Item("Roboto Slab:600", "Roboto Slab Bold"),
			new Item("Rochester", "Rochester"),
			new Item("Shadows Into Light Two", "Shadows Into Light Two"),
			new Item("Scope One", "Scope One"),
			new Item("Six Caps", "Six Caps"),
			new Item("Sofia", "Sofia"),
			new Item("Sorts Mill Goudy", "Sorts Mill Goudy"),
			new Item("Special Elite", "Special Elite"),
			new Item("Squada One", "Squada One"),
			new Item("Strait", "Strait"),
			new Item("Unica One", "Unica One"),
			new Item("Vidaloka", "Vidaloka"),
			new Item("Work Sans", "Work Sans"),
			new Item("Yanone Kaffeesatz:300", "Yanone Kaffeesatz"),
			new Item("Yanone Kaffeesatz:500", "Yanone Kaffeesatz Bold")
		});
		
		// Folders
		JComboBox folderCols = new JComboBox(new Object[]{
			1,
			2,
			3,
			4,
			5,
			6
		});
		
		JCheckBox fixedShapeFolderThumbs = new JCheckBox(getText("ui.fixedShapeThumbs"), false);
		
		JComboBox folderThumbAspectRatio = new JComboBox(new Object[] {
			new Item("", "[ " + getText("ui.sameAsOnThumbnails") + " ]"),
			new Item("0.5", "1 : 2"), 
			new Item("0.6667", "2 : 3"), 
			new Item("0.75", "3 : 4"), 
			new Item("0.8", "4 : 5"), 
			new Item("0.8667", "5 : 6"), 
			new Item("1.0", "1 : 1"), 
			new Item("1.2", "6 : 5"), 
			new Item("1.25", "5 : 4"), 
			new Item("1.3333", "4 : 3"), 
			new Item("1.5", "3 : 2"),
			new Item("1.7778", "16 : 9"),
			new Item("2.0", "2 : 1"),
			new Item("3.0", "3 : 1")
		});
		
		JComboBox folderCaptionPlacement = new JComboBox(new Object[]{
			new Item("over", getText("ui.over")),
			new Item("below", getText("ui.below")),
			new Item("beside", getText("ui.beside"))
		});
		
		JComboBox<Item> folderCaptionPresets = new JComboBox(new Object[] {
			new Item("", getText("ui.empty")),
			new Item("<h3>${title}</h3>", getText("ui.title")),
			new Item("<h3>${title}</h3>\n<div class=\"comment\">${comment}</div>", getText("ui.title") + " + " + getText("ui.comment")),
			new Item("<h3>${title}</h3>\n<div class=\"comment\">${commentShort}</div>", getText("ui.title") + " + " + getText("ui.commentShort") + " (" + getText("ui.default") + ")"),
			new Item("<h3>${title} <span class=\"date\">${folderModDate}</span></h3>\n<div class=\"comment\">${comment}</div>", getText("ui.title") + " + " + getText("ui.date") + " + " + getText("ui.comment")),
			new Item("<h3>${title} <span class=\"date\">${folderModDate}</span></h3>\n<div class=\"comment\">${commentShort}</div>", getText("ui.title") + " + " + getText("ui.date") + " + " + getText("ui.commentShort")),
			new Item("<h3>${fileLabel}</h3>", getText("ui.fileLabel")),
			new Item("<h3>${fileLabel}</h3>\n<div class=\"comment\">${comment}</div>", getText("ui.fileLabel") + " + " + getText("ui.comment")),
			new Item(CUSTOM, "[ " + getText("ui.custom") + " ]")
		});
		JTextArea folderCaptionTemplate = new JSmartTextArea(4, 30);
		
		// Pages
		JComboBox showPages = new JComboBox(new Object[]{
			new Item("none", getText("ui.none")), 
			new Item("link", getText("ui.linkOnly")), 
			new Item("excerpt", getText("ui.withExcerpt")), 
			new Item("embedCustom", getText("ui.embedCustom")),
			new Item("embed", getText("ui.embedAll"))
		});

		// Weblocations
		JComboBox showWeblocations = new JComboBox(new Object[]{
			new Item("none", getText("ui.none")), 
			new Item("link", getText("ui.linkOnly")),
			new Item("smallThumb", getText("ui.smallThumbnail")),
			new Item("thumb", getText("ui.thumbnail")),
			new Item("asFolder", getText("ui.asFolder")),
			new Item("embed", getText("ui.embedFull"))
		});

		// Thumbnails
		JComboBox thumbLayout = new JComboBox(new Object[]{
			new Item("fixgrid", getText("ui.fixedShapeGrid")),
			new Item("grid", getText("ui.grid")),
			new Item("flexgrid", getText("ui.flexibleGrid")),
			//new Item("horizontal", getText("ui.masonry") + " (" + getText("ui.horizontal") + ")"),
			new Item("justified", getText("ui.justified"))
		});

		JComboBox thumbCols = new JComboBox(new Object[]{
			2,
			3,
			4,
			5,
			6,
			7,
			8
		});
		
		JComboBox thumbAspectRatio = new JComboBox(new Object[] {
			new Item("0.5", "1 : 2"), 
			new Item("0.6667", "2 : 3"), 
			new Item("0.75", "3 : 4"), 
			new Item("0.8", "4 : 5"), 
			new Item("0.8667", "5 : 6"), 
			new Item("1.0", "1 : 1"), 
			new Item("1.2", "6 : 5"), 
			new Item("1.25", "5 : 4"), 
			new Item("1.3333", "4 : 3"), 
			new Item("1.5", "3 : 2"),
			new Item("1.7778", "16 : 9"),
			new Item("2.0", "2 : 1"),
			new Item("3.0", "3 : 1")
		});

		JComboBox thumbGap = new JComboBox(new Object[] {
			new Item("none", getText("ui.none")),
			new Item("tiny", getText("ui.thin")),
			new Item("small", getText("ui.small")),
			new Item("medium", getText("ui.medium")),
			new Item("large", getText("ui.large"))
		});

		JComboBox captionPlacement = new JComboBox(new Object[] {
			new Item("over", getText("ui.over")),
			new Item("below", getText("ui.below"))
		});
		
		JComboBox thumbCaptionPresets = new JComboBox(new Object[] {
			new Item("", getText("ui.empty")),
			new Item("<div class=\"title\">${fileTitle}</div>", getText("ui.title")),
			new Item("<div class=\"title\">${fileTitle}</div>\n<div class=\"comment\">${comment}</div>", getText("ui.title") + " + " + getText("ui.comment")),
			new Item("<div class=\"title\">${fileTitle}</div>\n<div class=\"comment\">${commentShort}</div>", getText("ui.title") + " + " + getText("ui.commentShort") + " (" + getText("ui.default") + ")"),
			new Item("<div class=\"date\">${originalDate}</div>\n<div class=\"title\">${fileTitle}</div>\n<div class=\"comment\">${commentShort}</div>", getText("ui.date") + " + " + getText("ui.title") + " + " + getText("ui.commentShort")),
			new Item("<div class=\"date\">${originalTime}</div>\n<div class=\"title\">${fileTitle}</div>\n<div class=\"comment\">${commentShort}</div>", getText("ui.time") + " + " + getText("ui.title") + " + " + getText("ui.commentShort")),
			new Item("<span class=\"nr\">${imageNum}</span>\n<div class=\"title\">${fileTitle}</div>\n<div class=\"comment\">${commentShort}</div>", getText("ui.number") + " + " + getText("ui.title") + " + " + getText("ui.commentShort")),
			new Item("<div class=\"title\">${fileLabel}</div>", getText("ui.fileLabel")),
			new Item("<div class=\"title\">${fileLabel}</div>\n<div class=\"comment\">${comment}</div>", getText("ui.fileLabel") + " + " + getText("ui.comment")),
			new Item("<div class=\"title\">${fileLabel}</div>\n<div class=\"comment\">${commentShort}</div>", getText("ui.fileLabel") + " + " + getText("ui.commentShort")),
			new Item("<span class=\"nr\">${imageNum}</span>\n<div class=\"title\">${fileLabel}</div>\n<div class=\"comment\">${commentShort}</div>", getText("ui.number") + " + " + getText("ui.fileLabel") + " + " + getText("ui.commentShort")),
			new Item(CUSTOM, "[ " + getText("ui.custom") + " ]")
		});
		JTextArea thumbCaptionTemplate = new JSmartTextArea(4, 30);
		
		JComboBox hoverEffect = new JComboBox(new Object[] {
			new Item("", getText("ui.none")),
			new Item("hover-zoom", getText("ui.zoom")),
			new Item("hover-saturate", getText("ui.blackAndWhiteToColor"))
		});
		
		private void setFolderImageSize() {
			
			if (heroFullWidth.isSelected()) {
				folderImageSize.setText("1880x" + (int)Math.round(getSpinnerValueAsInt(folderImageHeight) * 1.5));
			} else {
				folderImageSize.setText(getPageWidth() + "x" + getSpinnerValueAsInt(folderImageHeight));
			}
		};
																		
		private void setFolderThumbSize() {
			String	ftar = (((Item)folderThumbAspectRatio.getSelectedItem()).value).toString(),
					tar = (((Item)thumbAspectRatio.getSelectedItem()).value).toString();
			int		pw = getPageWidth(),
					cols = Integer.parseInt((String)folderCols.getSelectedItem().toString());
			double	w = Math.max((cols > 3)? 300 : 620, (pw - (cols - 1) * 10)  / cols),
					ar = fixedShapeFolderThumbs.isSelected()? Float.parseFloat((ftar.length() > 0)? ftar : tar) : 1.333333;
			
			//System.out.println("setFolderThumbSize() -> " + (int)Math.round(w) + "x" + (int)Math.round(w / ar));
			folderThumbSize.setText((int)Math.round(w) + "x" + (int)Math.round(w / ar));
		};
																		
		private void setMakeSlides() {
			boolean ov = engine.isSlides(),
					nv = (showShare.isSelected() &&				// Share individual slides?
							(shareFacebook.isSelected() || shareThreads.isSelected() || shareBluesky.isSelected())) 
						|| 
						(writeSitemapXml.isSelected() && sitemapIncludeSlides.isSelected());
			
			if (ov != nv) {
				try {
					window.ui2Engine();
					engine.setSlides(nv);
					window.engine2UI();
				} catch (Exception ex) {
					throw new RuntimeException(ex);
				}
			}
		};
		
		private String lastThumbDimSuggestion = "";
		
		private int getPageWidth() {
			String	pw = ((Item)maxPageWidth.getSelectedItem()).value.toString();
			return pw.equals("none")? 1900 : Math.round(Float.parseFloat(pw.replace("rem","")) * 16);
		}
		
		private int getThumbGap() {
			switch (((Item)thumbGap.getSelectedItem()).value.toString()) {
				case "tiny":
					return 2;
				case "small":
					return 6;
				case "medium":
					return 12;
				case "large":
					return 20;
			}
			return 0;
		}
		
		private int getThumbPad() {
			switch (((Item)boxPadding.getSelectedItem()).value.toString()) {	
				case "small":
					return 4;
				case "medium":
					return 8;
				case "large":
					return 12;
				case "x-large":
					return 20;
			}
			return 0;
		}
		
		// Returns the thumb width
		private int getMaxThumbWidth() {
			
			return Integer.parseInt(engine.getThumbSize().split("x")[0]);
		}
		
		// Returns the thumb height
		private int getMaxThumbHeight() {
			
			return Integer.parseInt(engine.getThumbSize().split("x")[1]);
		}
		
		// Checks if number is within range of ref
		private boolean inRange(double n, double ref, double range) {
			
			return (n >= ref * (1.0 - range)) && (n < ref * (1.0 + range));
		}
		
		// Calculating proper thumb dimensions for the different thumb layouts
		private void setThumbDimensions() {
			
			if (!isSkinReady()) {
				return;
			}
			
			String	layout = ((Item)thumbLayout.getSelectedItem()).value.toString(),
					cdim = engine.getThumbSize(),
					ndim;

			int		cw = getPageWidth(),
					cols = Integer.parseInt(thumbCols.getSelectedItem().toString()),
					tw,
					w,
					h,
					mtw = getMaxThumbWidth(),
					mth = getMaxThumbHeight(),
					tg = getThumbGap(),
					tp = getThumbPad();
			
			//cw = (int) Math.ceil((pw - 30) / (layout.equals("horizontal")? Math.max(cols - 1, 1.5) : cols));
			// Thumb width
			tw = (int)Math.round(((cw - tg * 2 * (cols - 1)) / cols) - 2 * tp);
			//System.out.println("thumbWidth = (" + cw + " - " + tg + " * (" + cols + " - 1)) / " + cols + ") - 2 * " + tp + " => " + tw);
			
			switch (layout) {
				case "horizontal":
					w = (int)Math.round(tw * 1.66667);
					h = (int)Math.round(tw * 0.75);
					break;
					
				case "justified":
					w = (int)Math.round(tw * 1.6);
					h = (int)Math.round(tw * 0.8);
					/*
					w = (int)Math.round(tw * 1.4142);
					h = (int)Math.round(tw * 0.7071);
					*/
					break;
					
				default:
					w = tw;
					h = (int)Math.round(tw / Float.parseFloat((((Item)thumbAspectRatio.getSelectedItem()).value).toString()));
			}
			
			ndim = w + "x" + h;
			
			if (!lastThumbDimSuggestion.equals(ndim) &&					// Already set
				!(inRange(w, mtw, 0.05) && inRange(h, mth, 0.05) &&		// Width and Height is out of 5% range
				inRange(w / h, mtw / mth, 0.01))) {						// Aspect ratio is out of 1% range
				
				try {
					window.ui2Engine();
					engine.setThumbSize(ndim);
					JAlbum.logger.log(Level.FINE, "Thumb size has changed from {0}px to {1}px.", new Object[]{cdim, ndim});
					window.engine2UI();
				} catch (ParameterException ex) {
					throw new RuntimeException(ex);
				}
				
				lastThumbDimSuggestion = ndim;
			} else {
				JAlbum.logger.log(Level.FINE, "Thumb size change from {0}px to {1}px is within range.", new Object[]{cdim, ndim});
			}
		};
		
		// Lightbox global values
		
		JCheckBox lightboxUseMainBg = new JCheckBox(getText("ui.lightboxUseMainBackground"));
		JColorSelector lightboxBackgroundColor = new JAlphaColorSelector(getText("ui.backgroundColor"), new JSmartTextField("#e5111111", 8));
		JColorSelector lightboxBorderColor = new JAlphaColorSelector(getText("ui.borderColor"), new JSmartTextField("#eeeeee", 8));
		JSpinner lightboxBorderWidth = new JSpinner(new SpinnerNumberModel(10, 0, 100, 1));
		JSpinner imgCornerRadius = new JSpinner(new SpinnerNumberModel(0, 0, 200, 1));
		JSpinner fitPadding = new JSpinner(new SpinnerNumberModel(15, 0, 100, 1));
		JComboBox fitImagesBetween = new JComboBox(new Object[] {
			new Item("screen", getText("ui.wholeScreen")), 
			new Item("panels", getText("ui.topAndBottomPanels")),
			new Item("all", getText("ui.panelsAndControls"))
		});
		JCheckBox useThumbnailStrip = new JCheckBox(getText("ui.useThumbnailStrip"));
		JCheckBox thumbnailsVisible = new JCheckBox(getText("ui.thumbnailsVisible"));
		JComboBox thumbstripHeight = new JComboBox(new Object[] {
			new Item("60", getText("ui.tiny")), 
			new Item("80", getText("ui.small")), 
			new Item("110", getText("ui.medium")), 
			new Item("140", getText("ui.large"))
		});
		JCheckBox controlsUseText = new JCheckBox(getText("ui.showTextOnButtons"));
		JCheckBox autohideControls = new JCheckBox(getText("ui.autohideControls"));
		JCheckBox showFitToggle = new JCheckBox(getText("ui.showFitToggle"));
		JCheckBox zoomSlider = new JCheckBox(getText("ui.useZoomSlider"));
		JCheckBox showStartStop = new JCheckBox(getText("ui.showStartStop"));
		JCheckBox showFullscreen = new JCheckBox(getText("ui.showFullScreenButton"));
		
		JCheckBox fitImage = new JCheckBox(getText("ui.fitImagesByDefault"));
		JSpinner maxZoom = new JSpinner(new SpinnerNumberModel(1.4, 1.0, 3.0, 0.1));
		JCheckBox displayOriginals = new JCheckBox(getText("ui.displayOriginals"));
		
		JComboBox imageCaptionPresets = new JComboBox(new Object[] {
			new Item("", getText("ui.empty")),
			new Item("${originalTime}", getText("ui.time")),
			new Item("<div class=\"title\">${fileTitle}</div>", getText("ui.title")),
			new Item("<div class=\"title\">${fileTitle}</div>\n<div class=\"comment\">${comment}</div>", getText("ui.title") + " + " + getText("ui.comment")),
			new Item("<div class=\"title\">${fileTitle}</div>\n<div class=\"comment\">${commentShort}</div>", getText("ui.title") + " + " + getText("ui.commentShort")),
			new Item("<span>${originalTime}</span>\n<div class=\"title\">${fileTitle}</div>\n<div class=\"comment\">${comment}</div>", getText("ui.time") + " + " + getText("ui.title") + " + " + getText("ui.comment")),
			new Item("<div class=\"title\">${fileLabel}</div>", getText("ui.fileLabel")),
			new Item("<div class=\"title\">${fileLabel}</div>\n<div class=\"comment\">${comment}</div>", getText("ui.fileLabel") + " + " + getText("ui.comment") + " (" + getText("ui.default") + ")"),
			new Item("<div class=\"title\">${fileLabel}</div>\n<div class=\"comment\">${commentShort}</div>", getText("ui.fileLabel") + " + " + getText("ui.commentShort")),
			new Item(CUSTOM, "[ " + getText("ui.custom") + " ]")			
		});
		JTextArea imageCaptionTemplate = new JSmartTextArea(6, 30);

		JCheckBox showImageNumbers = new JCheckBox(getText("ui.showImageNumbers"));
		JCheckBox infoPanelVisible = new JCheckBox(getText("ui.infoPanelVisible"));
		JCheckBox infoPanelAdapt = new JCheckBox(getText("ui.adaptToContent"));

		JCheckBox buttonLabelsVisible = new JCheckBox(getText("ui.buttonLabelsVisible"));
		JCheckBox showMap = new JCheckBox(getText("ui.map"));
		JCheckBox showShopBtn = new JCheckBox(getText("ui.shoppingCart"));
		JCheckBox showFeedbackBtn = new JCheckBox(getText("ui.feedback"));
		JCheckBox showRegions = new JCheckBox(getText("ui.regions"));
		JCheckBox showShare = new JCheckBox(getText("ui.share"));
		JCheckBox printImageButton = new JCheckBox(getText("ui.print"));
		JCheckBox downloadBtn = new JCheckBox(getText("ui.download"));
		JCheckBox showPhotoData = new JCheckBox(getText("ui.photoData"));
		
		// These need to be global to watch independent slide pages requirement
		
		JCheckBox shareFacebook = new JCheckBox("Facebook");
		JCheckBox shareThreads = new JCheckBox("Threads");
		JCheckBox shareBluesky = new JCheckBox("Bluesky");
		JCheckBox writeSitemapXml = new JCheckBox(getText("ui.createSitemapXml"));
		JCheckBox sitemapIncludeSlides = new JCheckBox(getText("ui.includeSlides"), true);
		
		// Refreshing index page preview
		
		private final WebViewBrowser sitePreview = new WebViewBrowser();
		
		private void refreshIndexPreview() {
			Double		tar = getSelectedValueAsDouble(thumbAspectRatio),
						ftar = getSelectedValueAsDouble(folderThumbAspectRatio);
			double		backgroundLuminosity = getLuminosity(backgroundColor),
						topmenuAlpha = getAlpha(topmenuColor),
						ftheight,
						twidth,
						theight,
						thumbPad = getThumbPad(),
						cardGap = getThumbGap();

			int			brl = getSpinnerValueAsInt(cornerRadius),
						fcols = (int)folderCols.getSelectedItem(),
						cols = (int)thumbCols.getSelectedItem();

			String		html,
						sName = styleName.getText(),
						stylePath,
						patternDir,
						backgroundImage = backgroundImageName.getText(),
						bgRepeat = getSelectedValue(backgroundRepeat),
						boxBackgroundImage = boxBackgroundImageName.getText(),
						topmenuBgColor = getCssColor(topmenuColor),
						topmenuTextColor = getLegibleColor(backgroundColor, topmenuColor, textColor),
						footerBgColor = getCssColor(footerColor),
						footerTextColor = getLegibleColor(backgroundColor, footerColor, textColor),
						buttonTextColor = getLegibleColor(buttonColor, textColor, 0.7),
						boxShadow = boxDropShadow.isSelected()? ("1px 2px 6px rgba(0,0,0," + (0.55 - backgroundLuminosity * 0.4) + ")") : "none",
						fcplacement = getSelectedValue(folderCaptionPlacement),
						tlayout = getSelectedValue(thumbLayout);

			
			if (tar == null) {
				tar = 1.0;
			}
			if (ftar == null || ftar.equals("")) {
				ftar = tar;
			}
			
			ftheight = (520.0 - (fcols - 1) * cardGap) / fcols / ftar - 2 * thumbPad;

			if (fcplacement.equals("beside")) {
				ftheight *= 0.6;
			}

			twidth = (520.0 - (cols - 1) * cardGap) / cols;

			if (tlayout.equals("horizontal") || tlayout.equals("justified")) {
				twidth = Math.round(twidth * 1.2);
				theight = Math.round(twidth * 0.8);
			} else {
				theight = Math.round((twidth - 2 * thumbPad) / tar);
			}

			try {
				stylePath = (new File(skinDirectory, "styles/" + sName).toURI().toURL()).toString();
			} catch (MalformedURLException ex) {
				Logger.getLogger(Gui.class.getName()).log(Level.SEVERE, null, ex);
				stylePath = "";
			}
			
			try {
				patternDir = (new File(skinDirectory, "patterns").toURI().toURL()).toString();
			} catch (MalformedURLException ex) {
				Logger.getLogger(Gui.class.getName()).log(Level.SEVERE, null, ex);
				patternDir = "";
			}
			
			patternDir += (!heroImageType.equals("") && isLightStyle() || heroImageType.equals("") && isLightColor(backgroundColor, heroOverlayColor))? "light" : "dark";
			
			html = indexTemplate
				.replace("{fullWidth}", heroFullWidth.isSelected()? "full-width" : "page-wide")
				.replace("{htmlColor}", (backgroundImage.length() > 0)? 
						"transparent" 
						: 
						getCssColor(heroFullWidth.isSelected()? 
							flatColor(backgroundColor, topmenuColor) : toColor(backgroundColor)
						)
					)
				.replace("{mainBgColor}", getCssColor(backgroundColor))
				.replace("{backgroundImage}", (backgroundImage.length() > 0)? ("url(" + stylePath + "/" + backgroundImage + ")") : "none")
				.replace("{backgroundPosition}", getSelectedValue(backgroundPos))
				.replace("{backgroundRepeat}",  bgRepeat.equals("stretch")? "no-repeat" : bgRepeat)
				.replace("{backgroundSize}", bgRepeat.equals("stretch")? "cover" : "auto")
				.replace("{topbarSticky}", topbarSticky.isSelected()? "sticky" : "")
				.replace("{topmenuOpaque}", (topmenuAlpha < 0.05)? "transparent" : "opaque")
				.replace("{topmenuTextColor}", topmenuTextColor)
				.replace("{topmenuBgColor}", topmenuBgColor)
				.replace("{topnavAlign}", getSelectedValue(topnavigationAlignment))
				.replace("{footerTextColor}", footerTextColor)
				.replace("{footerBgColor}", footerBgColor)
				.replace("{searchVisible}", searchVisibleByDefault.isSelected()? "visible":"hidden")
				.replace("{searchPosition}", useSearch.isSelected()? (moveSearchToHero.isSelected()? "search-hero":"search-top") : "")
				.replace("{folderImageHeight}", (int)Math.round(getSpinnerValueAsInt(folderImageHeight) * 0.5) + "")
				.replace("{titlePosition}", getSelectedValue(folderTitlePosition))
				.replace("{heroType}", getSelectedValue(heroImageType).equals("")? "flat" : ("bg-image" + (heroImageDesaturate.isSelected()? " desaturate" : "")))
				.replace("{heroPattern}", patternDir + "/" + getSelectedValue(heroPattern))
				.replace("{heroOlyColor}", heroOverlayColorTransparent.isSelected()? "transparent" : getCssColor(heroOverlayColor))
				.replace("{heroTitleDarken}", darkenTitleBackground.isSelected()? "darken-bg":"")
				.replace("{heroButtonDarken}", darkenButtonBackground.isSelected()? "darken-btns":"")
				.replace("{headerTextColor}", "#fff")
				.replace("{textColor}", textColor.toString())
				.replace("{linkColor}", linkColor.toString())
				.replace("{hoverColor}", hoverColor.toString())
				.replace("{buttonTextColor}", buttonTextColor)
				.replace("{buttonBgColor}", getCssColor(buttonColor))
				.replace("{buttonHoverTextColor}", buttonTextColor)
				.replace("{buttonHoverBgColor}", getCssColor(buttonColor))
				.replace("{cardGap}", Double.toString(cardGap / 2))
				.replace("{folderWidth}", Double.toString(100.0 / fcols))
				.replace("{folderCaptionPlacement}", fcplacement)
				.replace("{folderThumbHeight}", Double.toString(Math.round(ftheight)))
				.replace("{folderThumbFit}", fixedShapeFolderThumbs.isSelected()? "fill" : "fit")
				.replace("{thumbLayout}", tlayout)
				.replace("{thumbPad}", Double.toString(thumbPad))
				.replace("{thumbWidth}", Double.toString(twidth))
				.replace("{thumbHeight}", Double.toString(theight))
				.replace("{captionPlacement}", getSelectedValue(captionPlacement))
				.replace("{hoverEffect}", getSelectedValue(hoverEffect))
				.replace("{boxTextColor}", getLegibleColor(backgroundColor, boxBackgroundColor, textColor))
				.replace("{boxBackgroundImage}", (boxBackgroundImage.length() > 0)? ("url(" + stylePath + "/" + boxBackgroundImage + ")") : "none")
				.replace("{boxBgColor}", getCssColor(boxBackgroundColor))
				.replace("{boxShadow}", boxShadow)
				.replace("{brs}", Math.round(brl * .333) + "")
				.replace("{brm}", Math.round(brl * .667) + "")
				.replace("{brl}", brl + "")
				.replace("{brlth}", (brl > 2? Math.round(Math.max(brl - thumbPad * 16, 2)) : 0) + "")
				.replace("{brth}", (brl > 3? Math.round(Math.max(brl * .667 - thumbPad * 16, 2)) : 0) + "");

			html = getFonts(html, getSelectedValue(fontFamily), (headlineFont.getSelectedIndex() == 0)? "" : getSelectedValue(headlineFont));

			final String html1 = html;
			
			SwingUtilities.invokeLater(() -> {
				sitePreview.loadContent(html1);
			});

			//writeFile("D:/Temp/index-preview.html", html);
		}
		
		// Refreshing lightbox page preview
		
		private final WebViewBrowser lightboxPreview = new WebViewBrowser();
		
		private void refreshLightboxPreview() {
			double		lightboxLuminosity = getLuminosity(backgroundColor, lightboxBackgroundColor);

			String		html,
						sName = styleName.getText(),
						stylePath,
						backgroundImage = backgroundImageName.getText(),
						bgRepeat = getSelectedValue(backgroundRepeat),
						boxShadow = boxDropShadow.isSelected()? ("1px 2px 6px rgba(0,0,0," + (0.55 - lightboxLuminosity * 0.4) + ")") : "none",
						boxBg,
						lightboxTextColor,
						lightboxButtonHover,
						lightboxBoxBg,
						lightboxBoxHoverBg;

			int			brl = getSpinnerValueAsInt(cornerRadius);

			try {
				stylePath = (new File(skinDirectory, "styles/" + sName).toURI().toURL()).toString();
			} catch (MalformedURLException ex) {
				Logger.getLogger(Gui.class.getName()).log(Level.SEVERE, null, ex);
				stylePath = "";
			}

			if (lightboxLuminosity > 0.6) {
				// Light
				lightboxTextColor = "#333333";
				lightboxButtonHover = "#000000";
				boxBg = (lightboxLuminosity > 0.92)? "239,239,239" : "255,255,255";
				lightboxBoxBg = "rgba(" + boxBg + "," + Double.toString(0.25 + lightboxLuminosity / 2) + ")";
				lightboxBoxHoverBg = "rgba(" + boxBg + "," + Double.toString(0.5 + lightboxLuminosity / 2) + ")";
			} else {
				// Dark
				lightboxTextColor = "#dddddd";
				lightboxButtonHover = "#ffffff";
				boxBg = (lightboxLuminosity < 0.1)? "21,21,21" : "0,0,0";
				lightboxBoxBg = "rgba(" + boxBg + "," + Double.toString(0.75 - lightboxLuminosity / 2) + ")";
				lightboxBoxHoverBg = "rgba(" + boxBg + "," + Double.toString(1 - lightboxLuminosity / 2) + ")";
			}

			html = lightboxTemplate
				.replace("{mainBgColor}", getFlatColor(backgroundColor, lightboxBackgroundColor))
				.replace("{backgroundImage}", (backgroundImage.length() > 0 && lightboxUseMainBg.isSelected())? ("url(" + stylePath + "/" + backgroundImage + ")") : "none")
				.replace("{backgroundPosition}", getSelectedValue(backgroundPos))
				.replace("{backgroundRepeat}",  bgRepeat.equals("stretch")? "no-repeat" : bgRepeat)
				.replace("{backgroundSize}", bgRepeat.equals("stretch")? "cover" : "auto")
				.replace("{textColor}", lightboxTextColor)
				.replace("{linkColor}", linkColor.toString())
				.replace("{hoverColor}", hoverColor.toString())
				.replace("{buttonTextColor}", lightboxTextColor)
				.replace("{buttonBgColor}", getCssColor(buttonColor))
				.replace("{buttonHoverTextColor}", lightboxButtonHover)
				.replace("{buttonHoverBgColor}", getCssColor(buttonColor))
				.replace("{boxShadow}", boxShadow)
				.replace("{brs}", Math.round(brl * .333) + "")
				.replace("{brm}", Math.round(brl * .667) + "")
				.replace("{brl}", brl + "")
				.replace("{lightboxBgColor}", getCssColor(lightboxBackgroundColor))
				.replace("{lightboxBoxBg}", lightboxBoxBg)
				.replace("{lightboxBoxHoverBg}", lightboxBoxHoverBg)
				.replace("{lightboxBorderColor}", getCssColor(lightboxBorderColor))
				.replace("{lightboxBorderWidth}", getSpinnerValueAsInt(lightboxBorderWidth) + "")
				.replace("{fitImage}", fitImage.isSelected()? "fit" : "")
				.replace("{fitPadding}", getSpinnerValueAsInt(fitPadding) + "")
				.replace("{maxImageWidth}", getSpinnerValueAsDouble(maxZoom) * 448 + "")
				.replace("{imgCornerRadius}", getSpinnerValueAsInt(imgCornerRadius) + "")
				.replace("{thumbstripHeight}", 40 + thumbstripHeight.getSelectedIndex() * 20 + "")
				.replace("{thumbsVisible}", "thumbs-" + ((useThumbnailStrip.isSelected() && thumbnailsVisible.isSelected())? "visible" : "hidden"))
				.replace("{useThumbstrip}", useThumbnailStrip.isSelected()? "use-thumbstrip" : "")
				.replace("{controlsText}", controlsUseText.isSelected()? "show-label" : "")
				.replace("{showFitToggle}", showFitToggle.isSelected()? "show-fit-toggle" : "")
				.replace("{useSlider}", zoomSlider.isSelected()? "use-slider" : "")
				.replace("{showStartStop}", showStartStop.isSelected()? "show-start-stop" : "")
				.replace("{showFullscreen}", showFullscreen.isSelected()? "show-fullscreen" : "")
				.replace("{showLabels}", buttonLabelsVisible.isSelected()? "labels" : "")
				.replace("{showNumbers}", showImageNumbers.isSelected()? "show" : "")
				.replace("{adaptInfoPanel}", infoPanelAdapt.isSelected()? "adapt" : "fixed")
				.replace("{showDownload}", downloadBtn.isSelected()? "show" : "")
				.replace("{showCart}", showShopBtn.isSelected()? "show" : "")
				.replace("{showFeedback}", showFeedbackBtn.isSelected()? "show" : "")
				.replace("{showFacetag}", showRegions.isSelected()? "show" : "")
				.replace("{showPrint}", printImageButton.isSelected()? "show" : "")
				.replace("{showMap}", showMap.isSelected()? "show" : "")
				.replace("{showPhotodata}", showPhotoData.isSelected()? "show" : "")
				.replace("{showShare}", showShare.isSelected()? "show" : "")
				.replace("{captionVisible}", "caption-" + (infoPanelVisible.isSelected()? "visible" : "hidden"));

			html = getFonts(html, getSelectedValue(fontFamily), (headlineFont.getSelectedIndex() == 0)? "" : getSelectedValue(headlineFont));

			final String html2 = html;
			
			SwingUtilities.invokeLater(() -> {
				lightboxPreview.loadContent(html2);
			});

			//writeFile("D:/Temp/lightbox-preview.html", html);
		};
		
		private PropertyChangeListener setupMonitors = pce -> {
			
			StateMonitor.monitoring(fontSuggestions).onChange(src -> {
				if (isSkinReady() && src != null) {
					setFontBoxes();
				}
			});

			StateMonitor.monitoring(extraSizes).onChange(src -> {
				extraSizesChanged.setSelected(lastExtraSizes == null || !lastExtraSizes.equals(extraSizes.getText()));
			});
			
			// Thumb size change
			StateMonitor.monitoring(
					maxPageWidth,
					thumbCols,
					thumbGap,
					thumbLayout,
					thumbAspectRatio).onChange(src -> {

				if (isSkinReady() && src != null) {
					//JAlbum.logger.log(Level.FINE, "stateMonitor: \"{0}\" changed", getSelectedValue(src));
					setThumbDimensions();
				}

			});

			// Folder image size change
			StateMonitor.monitoring(
					maxPageWidth,
					heroFullWidth,
					folderImageHeight).onChange(src -> {
				setFolderImageSize();
			});

			// Folder thumb size change
			StateMonitor.monitoring(
					maxPageWidth,
					folderCols,
					folderThumbAspectRatio,
					fixedShapeFolderThumbs).onChange(src -> {
				setFolderThumbSize();
			});
			
			// Separate slide page requirement
			StateMonitor.monitoring(
					shareFacebook,
					shareThreads,
					shareBluesky,
					showShare,
					writeSitemapXml,
					sitemapIncludeSlides).onChange(src -> {
				setMakeSlides();
			});

			// Index preview update
			StateMonitor.monitoring(styleName,
					heroFullWidth,
					heroPattern,
					heroOverlayColor,
					heroOverlayColorTransparent,
					heroImageType,
					heroImageDesaturate,
					darkenTitleBackground,
					darkenButtonBackground,
					topbarSticky,
					topmenuColor,
					topnavigationAlignment,
					footerColor,
					backgroundColor,
					backgroundPos,
					backgroundRepeat,
					backgroundImageName,
					textColor,
					linkColor,
					hoverColor,
					buttonColor,
					boxBackgroundColor,
					boxBackgroundImageName,
					boxDropShadow,
					boxPadding,
					cornerRadius,
					fontFamily,
					useSearch,
					searchVisibleByDefault,
					moveSearchToHero,
					headlineFont,
					folderImageHeight,
					folderTitlePosition,
					folderCols,
					folderThumbAspectRatio,
					folderCaptionPlacement,
					fixedShapeFolderThumbs,
					thumbCols,
					thumbLayout,
					thumbAspectRatio,
					thumbGap,
					hoverEffect,
					captionPlacement
				).onChange((Object src) -> {

				if (isSkinReady() && src != null) {
					refreshIndexPreview();
				}
			});

			// Lightbox preview update
			StateMonitor.monitoring(
					styleName,
					backgroundColor,
					backgroundPos,
					backgroundRepeat,
					backgroundImageName,
					lightboxUseMainBg,
					lightboxBackgroundColor,
					textColor,
					linkColor,
					hoverColor,
					buttonColor,
					boxDropShadow,
					cornerRadius,
					fontFamily, 
					headlineFont,
					useThumbnailStrip,
					thumbstripHeight,
					thumbnailsVisible,
					controlsUseText,
					lightboxBorderColor,
					lightboxBorderWidth,
					imgCornerRadius,
					fitImage,
					maxZoom,
					fitPadding,
					showFitToggle,
					zoomSlider,
					showStartStop,
					showFullscreen,
					showImageNumbers,
					infoPanelAdapt,
					buttonLabelsVisible,
					showMap,
					showShopBtn,
					showFeedbackBtn,
					showRegions,
					showShare,
					printImageButton,
					downloadBtn,
					showPhotoData,
					infoPanelVisible
				).onChange((Object src) -> {

				if (isSkinReady() && src != null) {
					refreshLightboxPreview();
				}
			});

			// Initial setup
			setThumbDimensions();
			setFolderImageSize();
			setFolderThumbSize();

			// Initial preview rendering
			
			refreshIndexPreview();
			refreshLightboxPreview();
		};
				
		
		/*	---------------------------------------------------------------
								Design or Site tab
			--------------------------------------------------------------- */
		
		ControlPanel design = new ControlPanel() {

			private JButton selectImage = new JButton();
			private JButton selectBoxImage = new JButton();
			
			JComboBox modalWindowsTheme = new JComboBox(new Object[] {
				new Item("auto", getText("ui.auto")),
				new Item("light", getText("ui.light")),
				new Item("dark", getText("ui.dark"))
			});
			
			JComboBox iconStyle = new JComboBox(new Object[] {
				new Item("thin", getText("ui.thin")),
				new Item("fat", getText("ui.fat"))
			});
			
			//	Colors

			ControlPanel siteDesign = new ControlPanel() {

				JComboBox<Item> language = new JComboBox<Item>() {
					{
						setModel(Languages.modelFrom(new File(skinDirectory, "texts")));
						insertItemAt(new Item("jalbum", "[ " + getText("ui.jAlbumPreference") + " ]"), 0);
						setSelectedIndex(0);
					}
				};
				
				{
					maxPageWidth.setToolTipText(getText("ui.maxPageWidthInfo"));
					language.setToolTipText(getText("ui.languageInfo"));
					selectImage.setText(getText("ui.select"));
					selectImage.addActionListener((ActionEvent e) -> {
						getFileToRes(imageFiles, backgroundImageName, skinUi);
					});
					selectBoxImage.setText(getText("ui.select"));
					selectBoxImage.addActionListener((ActionEvent e) -> {
						getFileToRes(imageFiles, boxBackgroundImageName, skinUi);
					});
					backgroundColor.getTextComponent().setFont(mono);
					backgroundColor.setToolTipText(getText("ui.backgroundColorInfo"));
					topmenuColor.getTextComponent().setFont(mono);
					footerColor.getTextComponent().setFont(mono);
					boxBackgroundColor.getTextComponent().setFont(mono);
					textColor.getTextComponent().setFont(mono);
					linkColor.getTextComponent().setFont(mono);
					hoverColor.getTextComponent().setFont(mono);
					buttonColor.getTextComponent().setFont(mono);
			
					add("", new JLabelFor(getText("ui.maxPageWidth"), maxPageWidth));
					add("tab", maxPageWidth);
					add(new JLabel("px"));
					add("br", new JLabelFor(getText("ui.language"), language));
					add("tab", language);
					add("br", new JLabelFor(getText("ui.backgroundImage"), backgroundImageName));
					add("tab", backgroundImageName);
					add(" ", selectImage);
					add("br", new JLabelFor(getText("ui.position"), backgroundPos));
					add("tab", backgroundPos);
					add("br tab", backgroundRepeat);
					add("br", new JLabelFor(getText("ui.backgroundColor"), backgroundColor));
					add("tab", backgroundColor);
					add("tab", backgroundColor.getTextComponent());
					add("br", new JLabelFor(getText("ui.textColor"), textColor));
					add("tab", textColor);
					add("tab", textColor.getTextComponent());
					add("br", new JLabelFor(getText("ui.linkColor"), linkColor));
					add("tab", linkColor);
					add("tab", linkColor.getTextComponent());
					add("br", new JLabelFor(getText("ui.hoverColor"), hoverColor));
					add("tab", hoverColor);
					add("tab", hoverColor.getTextComponent());
					add("br", new JLabelFor(getText("ui.buttonColor"), buttonColor));
					add("tab", buttonColor);
					add("tab", buttonColor.getTextComponent());
					add("br", new JLabelFor(getText("ui.topBar"), topmenuColor));
					add("tab", topmenuColor);
					add("tab", topmenuColor.getTextComponent());
					add("br", new JLabelFor(getText("ui.footer"), footerColor));
					add("tab", footerColor);
					add("tab", footerColor.getTextComponent());
					add("br", new JLabel(texts.getString("ui.mouseOverEffect")));
					add("tab", hoverEffect);
					add("br", new JLabelFor(getText("ui.modalWindowsTheme"), modalWindowsTheme));
					add("tab", modalWindowsTheme);
					add("br", new JLabel(getText("ui.iconStyle")));
					add("tab", iconStyle);
				}
			};
						
			// Boxes

			ControlPanel boxes = new ControlPanel() {

				{
					cornerRadius.setToolTipText(getText("ui.cornerRadiusInfo"));
					
					add("", new JLabelFor(getText("ui.backgroundColor"), boxBackgroundColor));
					add("tab", boxBackgroundColor);
					add("tab", boxBackgroundColor.getTextComponent());
					add("br", new JLabelFor(getText("ui.boxBackground"), boxBackgroundImageName));
					add("tab", boxBackgroundImageName);
					add(" ", selectBoxImage);
					add("br", new JLabel(getText("ui.cornerRadius")));
					add("tab", cornerRadius);
					add("", new JLabel("px"));
					add("br", new JLabelFor(getText("ui.padding"), boxPadding));
					add("tab", boxPadding);
					add("br", new JLabelFor(getText("ui.gap"), thumbGap));
					add("tab", thumbGap);
					add("br tab", boxDropShadow);
				}
			};
			
			JTabbedPane siteDesignTabs = new JTabbedPane() {
				
				{
					siteDesignLink = safeLinkLabel(siteDesign, helpRoot + "/Site/Design", getText("ui.genericDesign"));
					
					addTab(getText("ui.site"), siteDesign);
					addTab(getText("ui.boxes"), boxes);
				}
				
			};
			
			{
				((RiverLayout)(getLayout())).setVgap(0);
				((RiverLayout)(getLayout())).setHgap(0);
				add("hfill", siteDesignTabs);
				
				putClientProperty("helpPage", helpRoot + "/Site/Design");
			}
		};
		
		// Typography
		
		ControlPanel typography = new ControlPanel() {

			JComboBox baseFontSize = new JComboBox(new Object[] {
				new Item("11", getText("ui.smallest") + " (11px)"),
				new Item("12", getText("ui.tiny") + " (12px)"),
				new Item("13", getText("ui.small") + " (13px)"),
				new Item("14", getText("ui.smaller") + " (14px)"),
				new Item("15", getText("ui.normal") + " (15px)"),
				new Item("16", getText("ui.normal") + " (16px)"),
				new Item("17", getText("ui.larger") + " (17px)"),
				new Item("18", getText("ui.large") + " (18px)"),
				new Item("20", getText("ui.largest") + " (20px)"),
			});
								
			{
				typographyLink = safeLinkLabel(fontFamily, helpRoot + "/Site/Typography", getText("ui.fonts"));
				
				headlineFont.setEditable(true);

				add(new JLabelFor(getText("ui.pairingSuggestions"), fontSuggestions));
				add("tab", fontSuggestions);
				add("br", new JLabelFor(getText("ui.fontFamily"), fontFamily));
				add("tab", fontFamily);
				add("br", new JLabelFor(getText("ui.baseFontSize"), baseFontSize));
				add("tab", baseFontSize);
				add("br", new JLabelFor(getText("ui.headlineFont"), headlineFont));
				add("tab", headlineFont);
				
				putClientProperty("helpPage", helpRoot + "/Site/Typography");
			}
		};

		// Top bar
		
		ControlPanel topbar = new ControlPanel() {
			
			//JCheckBox topbarSticky = new JCheckBox(getText("ui.stickToTop"));
			
			ControlPanel logo = new ControlPanel(getText("ui.logo")) {

				JTextField logoName = new JSmartTextField(18);
				JButton selectLogo = new JButton(getText("ui.select"));
				JComboBox logoLinking = new JComboBox(new Object[] {
					new Item("toplevel", getText("ui.topLevelPage")),
					new Item("external", getText("ui.externalHomePage")),
					new Item("none", getText("ui.none")),
				});
				
				{
					selectLogo.addActionListener(new ActionListener() { 
						@Override
						public void actionPerformed(ActionEvent e) {
							getFileToRes(imageFiles, logoName, skinUi);
					}});

					add("hfill", logoName);
					add("", selectLogo);
					add("br", new JLabel(getText("ui.linkTo")));
					add(" ", logoLinking);
				}
			};

			ControlPanel topNavigation = new ControlPanel(getText("ui.topNavigation")) {
				JCheckBox topNavigationIncludeFolders = new JCheckBox(getText("ui.includeFolders"));
				JCheckBox topNavigationIncludePages = new JCheckBox(getText("ui.includePages"));
				JCheckBox topNavigationIncludeWebLocations = new JCheckBox(getText("ui.includeWebLocations"));
				JSpinner topNavigationDepth = new JSpinner(new SpinnerNumberModel(4, 1, 6, 1));
				JSpinner topNavigationHideLong = new JSpinner(new SpinnerNumberModel(16, 1, 200, 1));
				JSpinner maxTitleLength = new JSpinner(new SpinnerNumberModel(48, 24, 160, 8));
				{
					topNavPagesLink = safeLinkLabel(topNavigationIncludePages, "/Site/Top_bar", getText("ui.topNavigation"));
					topNavigationHideLong.setToolTipText(getText("ui.useButtonAboveInfo"));
					
					add("", topNavigationIncludeFolders);
					add("br", topNavigationIncludePages);
					add("br", topNavigationIncludeWebLocations);
					add("br", new JLabel(getText("ui.depth")));
					add("tab", topNavigationDepth);
					add("", new JLabel(getText("ui.levels")));
					add("br", new JLabel(getText("ui.useButtonAbove")));
					add("tab", topNavigationHideLong);
					add("", new JLabel(getText("ui.elements")));
					add("br", new JLabel(getText("ui.maxTitleLength")));
					add("tab", maxTitleLength);
					add("", new JLabel(getText("ui.characters")));
					add("br", new JLabel(getText("ui.align")));
					add("tab", topnavigationAlignment);
				}
			};

			ControlPanel search = new ControlPanel(getText("ui.search")) {

				JTextField searchFields = new JSmartTextField(18);
				//WrappableJLabel fieldsInfo = new WrappableJLabel("<html><i>" + getText("ui.searchFieldsInfo") + "</i></html>");

				{
					searchVisibleByDefault.setToolTipText(getText("ui.searchVisibleByDefaultInfo"));
					searchFields.setToolTipText("<html>" + getText("ui.searchFieldsInfo") + "</html>");
					//fieldsInfo.setPreferredWidth(uiWidth - 340);
					ComponentUtilities.whenSelectedEnable(useSearch, new JComponent[]{moveSearchToHero, searchVisibleByDefault, searchFields});
					ComponentUtilities.whenSelectedDisable(moveSearchToHero, searchVisibleByDefault);
					
					add("", useSearch);
					add("br", moveSearchToHero);
					add("br", searchVisibleByDefault);
					add("br", new JLabelFor(getText("ui.fields"), searchFields));
					add("tab hfill", searchFields);
					add("br", new JLabel(""));
					//add("tab", fieldsInfo);
				}
			};

			ControlPanel topbarDesign = new ControlPanel() {
				
				{
					add(topbarSticky);
					add("br hfill", logo);
					add("br hfill", topNavigation);
					add("br hfill", search);
				}
			};
			
			ControlPanel aboveTopbar = new ControlPanel() {
				
				JSmartTextArea bodyTopHook = new JSmartTextArea(5, 24);
				JScrollPane bodyTopHookPane = new JScrollPane(bodyTopHook);
				JCheckBox bodyTopFitWidth = new JCheckBox(getText("ui.constrainPageWidth"), true);
				JCheckBox bodyTopTopLevelOnly = new JCheckBox(getText("ui.topLevelOnly"), false);
				//private WrappableJLabel instr = new WrappableJLabel("<html><i>" + getText("ui.bodyTopText") + "</i></html>");

				{
					topBarCCLink = safeLinkLabel(bodyTopHookPane, helpRoot + "/Site/Top_bar", getText("ui.topBar"));
					bodyTopHook.setEditable(true);
					bodyTopHook.setLineWrap(false);
					bodyTopHook.setWrapStyleWord(true);
					bodyTopHook.setFont(mono);
					bodyTopHook.setTabSize(2);
					bodyTopHook.setBorder(emptyBorder);
					bodyTopHookPane.setBorder(BorderFactory.createTitledBorder(getText("ui.bodyTopText")));
					allowHTMLEditing(bodyTopHook);
					//instr.setPreferredWidth(uiWidth - 300);

					add("hfill vfill", bodyTopHookPane);
					add("br", bodyTopFitWidth);
					add("br", bodyTopTopLevelOnly);
					//add("br", instr);
				}
				
			};
			
			JTabbedPane topbarTabs = new JTabbedPane() {
				
				{
					addTab(getText("ui.functions"), topbarDesign);
					addTab(getText("ui.aboveTopBar"), aboveTopbar);
				}
			};
			
			{
				((RiverLayout)(getLayout())).setVgap(0);
				((RiverLayout)(getLayout())).setHgap(0);
				add("hfill", topbarTabs);
				
				putClientProperty("helpPage", helpRoot + "/Site/Top_bar");
			}
		};

		// Hero design
		
		ControlPanel hero = new ControlPanel() {

			JCheckBox alwaysRestartSlideshow = new JCheckBox(getText("ui.alwaysRestart"), false);
			JSpinner folderImageHeightPages = new JSpinner(new SpinnerNumberModel(80, 80, 640, 10));
			JSpinner folderImageHeightSubAlbums = new JSpinner(new SpinnerNumberModel(240, 110, 640, 10));
			JCheckBox preferThemeImage = new JCheckBox(getText("ui.preferThemeImage"), false);
			JSpinner heroSlideshowDelay = new JSpinner(new SpinnerNumberModel(4000, 100, 10000, 100));
			JLabel heroSlideshowDelayLabel = new JLabelFor(getText("ui.timing"), heroSlideshowDelay);
			JScrollPane titleCaptionPane = new JScrollPane(titleCaptionTemplate);
			JComboBox folderDateSource = new JComboBox(new Object[]{
				new Item("none", getText("ui.none")),
				new Item("fileDate", getText("ui.fileDate")),
				new Item("folderModDate", getText("ui.folderModDate")),
				new Item("lastCameraDate", getText("ui.lastCameraDate")),
				new Item("cameraDateRange", getText("ui.cameraDateRange"))
			});
			JLabel folderDateSourceLabel = new JLabelFor(getText("ui.dateSource"), folderDateSource);
			JComboBox showBreadcrumbPath = new JComboBox(new Object[]{
				new Item("none", getText("ui.none")),
				new Item("above", getText("ui.aboveTitle")),
				new Item("tooltip", getText("ui.asTooltip"))
			});
			
			ControlPanel heroDesign = new ControlPanel() {
				
				WrappableJLabel heroDesc = new WrappableJLabel("<html><i>" + getText("ui.heroDescription") + "</i></html>");
					
				{
					heroFullWidth.setToolTipText(getText("ui.fullWidthInfo"));
					showStartSlideshow.setToolTipText(getText("ui.showStartSlideshowInfo"));
					folderImageHeight.setToolTipText(getText("ui.folderImageHeightInfo"));
					folderImageHeightPages.setToolTipText(getText("ui.heightOnPagesInfo"));
					folderImageHeightSubAlbums.setToolTipText(getText("ui.heightOnSubFoldersInfo"));
					ComponentUtilities.whenSelectedEnable(showStartSlideshow, new JComponent[] { alwaysRestartSlideshow });
					ComponentUtilities.whenSelectedDisable(heroOverlayColorTransparent, new JComponent[] { heroOverlayColor });
					preferThemeImage.setToolTipText(getText("ui.preferThemeImageInfo"));
					heroOverlayColor.getTextComponent().setFont(mono);
					//heroParallax.setToolTipText(getText("ui.parallaxEffectInfo"));
					heroDesc.setPreferredWidth(uiWidth - 260);
					new StateMonitor() {
						@Override
						public void onChange() {
							boolean imageBg = heroImageType.getSelectedIndex() > 0,
								ssBg = heroImageType.getSelectedIndex() > 2;
							darkenTitleBackground.setSelected(imageBg);
							heroImageDesaturate.setVisible(imageBg);
							preferThemeImage.setVisible(imageBg && !ssBg);
							heroSlideshowDelayLabel.setVisible(ssBg);
							heroSlideshowDelay.setVisible(ssBg);
						}
					}.add(heroImageType).done();
					darkenButtonBackground.setToolTipText(getText("ui.darkenButtonBackgroundInfo"));
									
					add("", heroFullWidth);
					add("br", new JLabel(getText("ui.height")));
					add("tab", folderImageHeight);
					add("br", new JLabel(getText("ui.heightOnSubFolders")));
					add("tab", folderImageHeightSubAlbums);
					add("br", new JLabel(getText("ui.heightOnPages")));
					add("tab", folderImageHeightPages);
					add("br", new JLabelFor(getText("ui.backgroundImage"), heroImageType));
					add("tab", heroImageType);
					add("", heroImageDesaturate);
					add("br", heroSlideshowDelayLabel);
					add("tab", heroSlideshowDelay);
					add("br tab", preferThemeImage);
					add("br", new JLabelFor(getText("ui.pattern"), heroPattern));
					add("tab", heroPattern);
					add("br", new JLabelFor(getText("ui.overlayColor"), heroOverlayColor));
					add("tab", heroOverlayColor);
					add("", heroOverlayColor.getTextComponent());
					add("", heroOverlayColorTransparent);
					add("br", darkenTitleBackground);
					add("br", darkenButtonBackground);
					add("br", showHelp);
					add("br", showStartSlideshow);
					add("br", alwaysRestartSlideshow);
					add("br", showFullscreenButton);
					add("br", new JLabel(infoIcon));
					add("", heroDesc);
				}
			};

			//JCheckBox heroParallax = new JCheckBox(getText("ui.parallaxEffect"));
			
			ControlPanel heroTitle = new ControlPanel() {
				
				{
					titleCaptionLink = safeLinkLabel(titleCaptionTemplate, helpRoot + "/Site/Hero", getText("ui.titleCaption"));
					titleCaptionPresets.addItemListener(listener -> {
						if (isSkinReady()) {
							String s = getSelectedValue(titleCaptionPresets);
							if (!s.equals(CUSTOM)) {
								titleCaptionTemplate.setText(s);
							}
						}
					});	
					new StateMonitor() {
						public void onChange() {
							String s = getSelectedValue(titleCaptionPresets);
							boolean custom = s.equals(CUSTOM),
								hasDate = custom || s.contains("${folderModDate}");
							titleCaptionPane.setVisible(custom);
							folderDateSourceLabel.setVisible(hasDate);
							folderDateSource.setVisible(hasDate);
						}
					}.add(titleCaptionPresets).done();						
					titleCaptionTemplate.setEditable(true);
					titleCaptionTemplate.setLineWrap(true);
					titleCaptionTemplate.setWrapStyleWord(true);
					titleCaptionTemplate.setFont(mono);
					titleCaptionTemplate.setTabSize(2);
					titleCaptionPane.setBorder(BorderFactory.createTitledBorder(getText("ui.customCode")));
					showBreadcrumbPath.setToolTipText(getText("ui.showBreadcrumbPathInfo"));
					
					add(new JLabelFor(getText("ui.position"), folderTitlePosition));
					add("tab", folderTitlePosition);
					add("br", new JLabelFor(getText("ui.caption"), titleCaptionPresets));
					add("tab", titleCaptionPresets);
					add("br hfill", titleCaptionPane);
					add("br", folderDateSourceLabel);
					add("", folderDateSource);
					add("br", new JLabelFor(getText("ui.showBreadcrumbPath"), showBreadcrumbPath));
					add("", showBreadcrumbPath);
					
				}
			};
			
			JTabbedPane heroDesignTabs = new JTabbedPane() {
				
				{
					addTab(getText("ui.design"), heroDesign);
					addTab(getText("ui.title"), heroTitle);
				}
			};
			
			{
				((RiverLayout)(getLayout())).setVgap(0);
				((RiverLayout)(getLayout())).setHgap(0);
				add("hfill", heroDesignTabs);
				
				putClientProperty("helpPage", helpRoot + "/Site/Hero");
			}
		};
		
		// Footer
		
		ControlPanel footerSettings = new ControlPanel() {

			ControlPanel bottomNavigationPanel = new ControlPanel(getText("ui.bottomNavigation")) {
				//JCheckBox showBottomNavigation = new JCheckBox(getText("ui.showBottomNavigation"), false);	
				JCheckBox bottomNavigationIncludeFolders = new JCheckBox(getText("ui.includeFolders"), true);
				JCheckBox bottomNavigationIncludePages = new JCheckBox(getText("ui.includePages"), true);
				JCheckBox bottomNavigationIncludeWebLocations = new JCheckBox(getText("ui.includeWebLocations"), true);
				
				@Override
				public void importVariables(Map<String, Object> vars) {
					Object sbn = vars.get("showBottomNavigation");
					if (sbn != null && vars.get("bottomNavigationIncludeFolders") == null) {
						vars.put("bottomNavigationIncludeFolders", sbn);
						vars.put("bottomNavigationIncludePages", sbn);
						vars.put("bottomNavigationIncludeWebLocations", sbn);
					}
				};
				
				{
					//showBottomNavigation.setToolTipText(getText("ui.showBottomNavigationInfo"));
					//add("", showBottomNavigation);
					add("", bottomNavigationIncludeFolders);
					add("br", bottomNavigationIncludePages);
					add("br", bottomNavigationIncludeWebLocations);
				}
			};

			ControlPanel customLinkPanel = new ControlPanel(getText("ui.customLink")) {

				JTextField customLink = new JSmartTextField();
				JTextField customLinkText = new JSmartTextField();

				{
					customLink.setToolTipText(getText("ui.customLinkInfo"));
					customLinkText.setToolTipText(getText("ui.customLinkTextInfo"));

					add(new JLabel("URL"));
					add("tab hfill", customLink);
					add("br", new JLabel(getText("ui.customLinkText")));
					add("tab hfill", customLinkText);
				}
			};

			ControlPanel customFooterContent = new ControlPanel(getText("ui.customContent")) {

				JSmartTextArea footer = new JSmartTextArea(6, 20);
				JScrollPane footerPane = new JScrollPane(footer, JScrollPane.VERTICAL_SCROLLBAR_AS_NEEDED, JScrollPane.HORIZONTAL_SCROLLBAR_AS_NEEDED);
				JCheckBox footerFitWidth = new JCheckBox(getText("ui.constrainPageWidth"), true);
				JCheckBox footerTopLevelOnly = new JCheckBox(getText("ui.topLevelOnly"), false);
				private WrappableJLabel instr = new WrappableJLabel("<html><i>" + getText("ui.pageBottomText") + "</i></html>");
				
				{
					footer.setEditable(true);
					footer.setLineWrap(true);
					footer.setWrapStyleWord(true);
					footer.setFont(mono);
					footer.setTabSize(2);
					footer.setToolTipText(getText("ui.customContentInfo"));
					footerPane.setPreferredSize(new Dimension(uiWidth - 260, 64));
					allowHTMLEditing(footer);
					instr.setPreferredWidth(uiWidth - 260);

					add("hfill vfill", footerPane);
					add("br", footerFitWidth);
					add("br", footerTopLevelOnly);
					add("br", instr);

				}
			};
			
			JCheckBox showModifiedDate = new JCheckBox(getText("ui.showModifiedDate"));	
			JCheckBox showImageCount = new JCheckBox(getText("ui.showFolderImageCount"));

			{
				footerCCLink = safeLinkLabel(customFooterContent, helpRoot + "/Site/Footer", getText("ui.footer"));
				
				showImageCount.setToolTipText(getText("ui.showFolderImageCountInfo"));

				add("hfill", bottomNavigationPanel);
				add("br hfill", customLinkPanel);				
				add("br", showModifiedDate);
				add("br", showImageCount);
				add("br hfill vfill", customFooterContent);
				
				putClientProperty("helpPage", helpRoot + "/Site/Footer");
			}

		};
			
		// Rating
		
		ControlPanel rating = new ControlPanel() {

			JCheckBox useRating = new JCheckBox(getText("ui.displayRating"));
			JCheckBox visitorRating = new JCheckBox(getText("ui.allowVisitorRatings"));
			JCheckBox useJalbumRating = new JCheckBox(getText("ui.useJalbumRatings"));
			
			{
				useRating.setToolTipText(getText("ui.displayRatingInfo"));
				visitorRating.setToolTipText(getText("ui.allowVisitorRatingsInfo"));
				useJalbumRating.setToolTipText(getText("ui.useJalbumRatingsInfo"));
				ComponentUtilities.whenSelectedEnable(useRating, new JComponent[]{ visitorRating, useJalbumRating});
				
				add("br", useRating);
				add("br", useJalbumRating);
				add("br", visitorRating);
				
				putClientProperty("helpPage", helpRoot + "/Site/Rating");
			}
		};
				
		// Mark new files
		
		ControlPanel markNew = new ControlPanel() {

			JCheckBox markFilesNew = new JCheckBox(getText("ui.markFilesNew"));	
			JTextField newDaysCount = new JSmartTextField("60", 3);
			JComboBox newDaysRef = new JComboBox(new Object[] {
				new Item("dateTaken", getText("ui.dateTaken")),
				new Item("fileModified", getText("ui.fileModified")),
				new Item("added", getText("ui.addedToAlbum"))
			});
			JComboBox newDaysMark = new JComboBox(new Object[] {
				new Item("icon", getText("ui.icon")),
				new Item("text", getText("ui.text"))
			});
			JTextField newDaysText = new JSmartTextField(getText("new"), 6);
			JLabel newIcon = new JLabel(icon("new-fill"));
			
			{
				newDaysCount.setToolTipText(getText("ui.newDaysCountInfo"));
				newDaysMark.addItemListener(listener -> {
					int i = newDaysMark.getSelectedIndex();
					newDaysText.setVisible(i == 1);
					newIcon.setVisible(i == 0);
				});
				newDaysText.setVisible(false);
				ComponentUtilities.whenSelectedEnable(markFilesNew, new JComponent[]{ newDaysCount, newDaysRef, newDaysMark, newDaysText, newIcon });
				
				add("", markFilesNew);
				add("", newDaysCount);
				add(new JLabel(getText("ui.daysOld")));
				add("br", new JLabel(getText("ui.reference")));
				add("tab", newDaysRef);
				add("br", new JLabel(getText("ui.marker")));
				add("tab", newDaysMark);
				add(newIcon);
				add(newDaysText);
				
				putClientProperty("helpPage", helpRoot + "/Site/Mark_new_files");
			}
		};
		
		//	Extra
		
		ControlPanel download = new ControlPanel() {

			private WrappableJLabel instr = new WrappableJLabel("<html><i>" + getText("ui.extraSizesInfo") + "</i></html>");
			JTextField downloadLabel = new JSmartTextField(16);
			JComboBox zipImages = new JComboBox(new Object[] {
				new Item("none", getText("ui.nothing")),
				new Item("slides", getText("ui.scaledDown")),
				new Item("originals", getText("ui.originals")),
				new Item("included", getText("ui.includedOriginals"))
			}); 
			JLabel zipInfo = new JLabel(infoIcon);
			
			{
				instr.setPreferredWidth(uiWidth - 300);
				downloadLabel.setToolTipText(getText("ui.default") + ": " + getText("ui.downloadImages"));
				zipInfo.addMouseListener(new MouseAdapter() {  
					@Override
					public void mouseReleased(MouseEvent e) {
						JOptionPane.showMessageDialog(window, getText("ui.nonAsciiWarning"), getText("ui.warning"), JOptionPane.WARNING_MESSAGE);
				}});
				
				add("br", new JLabel(getText("ui.extraSizes")));
				add("tab", extraSizes);
				add("br tab", instr);
				add("br", new JLabelFor(getText("ui.offerDownload"), zipImages));
				add("", zipImages);
				add("", zipInfo);
				
				putClientProperty("helpPage", helpRoot + "/Site/Download");
			}
		};

		// Social settings
		
		ControlPanel social = new ControlPanel() {

			ControlPanel buttons = new ControlPanel(getText("ui.buttons")) {
				JCheckBox facebookLike = new JCheckBox(getText("ui.facebookLikeButton"));
				JCheckBox twitterTweet = new JCheckBox(getText("ui.tweetButton"));
				JCheckBox pinItButton = new JCheckBox(getText("ui.pinItButton"));

				{
					facebookLike.setToolTipText(getText("ui.facebookLikeButtonInfo"));
					twitterTweet.setToolTipText(getText("ui.tweetButtonInfo"));
					pinItButton.setToolTipText(getText("ui.pinItButtonInfo"));

					add(facebookLike);
					add(" ", twitterTweet);
					add(" ", pinItButton);
				}
			};

			ControlPanel shares = new ControlPanel(getText("shareOn")) {

				JCheckBox shareTwitter = new JCheckBox("X");
				JCheckBox sharePinterest = new JCheckBox("Pinterest");
				JCheckBox shareLinkedin = new JCheckBox("LinkedIn");
				JCheckBox shareDigg = new JCheckBox("Digg");
				JCheckBox shareReddit = new JCheckBox("Reddit");
				JCheckBox shareTumblr = new JCheckBox("Tumblr");

				{
					setLayout(new RiverLayout(4, 5));
					add(new JLabel(icon("facebook")));
					add(shareFacebook);
					add("br", new JLabel(icon("threads")));
					add(shareThreads);
					add("br", new JLabel(icon("bluesky")));
					add(shareBluesky);
					add("br", new JLabel(icon("x")));
					add(shareTwitter);
					add("br", new JLabel(icon("reddit")));
					add(shareReddit);
					add("br", new JLabel(icon("pinterest")));
					add(sharePinterest);
					add("br", new JLabel(icon("tumblr")));
					add(shareTumblr);
					add("br", new JLabel(icon("linkedin")));
					add(shareLinkedin);
					add("br", new JLabel(icon("digg")));
					add(shareDigg);
				}
			};

			JTextField facebookAppId = new JSmartTextField("");
			JLabel mandatoryInfo = new JLabel(mandatory);
			
			ControlPanel links = new ControlPanel() {
				
				JCheckBox shareEmail = new JCheckBox(getText("ui.email"));
				JLabel emailSubjectLabel = new JLabel(getText("ui.subject"));
				JTextField emailSubject = new JSmartTextField(18);
				JTextArea emailBody = new JSmartTextArea(4,22);
				JScrollPane emailBodyPane = new JScrollPane(emailBody);
				JCheckBox shareLink = new JCheckBox(getText("ui.link"));
				
				{
					emailBody.setEditable(true);
					emailBody.setLineWrap(true);
					emailBody.setWrapStyleWord(true);
					emailBody.setToolTipText(getText("ui.emailBodyInfo"));
					emailBodyPane.setBorder(BorderFactory.createTitledBorder(getText("ui.body")));
					ComponentUtilities.whenSelectedEnable(shareEmail, new JComponent[]{emailSubjectLabel, emailSubject, emailBodyPane});
					
					add("", new JLabel(icon("email")));
					add("tab", shareEmail);
					add("br tab", emailSubjectLabel);
					add("", emailSubject);
					add("br tab", emailBodyPane);
					add("br", new JLabel(icon("link")));
					add("tab", shareLink);
					add("br", new JLabelFor(getText("ui.facebookAppId"), facebookAppId));
					add(mandatoryInfo);
					add("hfill", facebookAppId);
					add("", new JLinkLabel("https://developers.facebook.com/apps", getText("ui.signUp")));
				}
			};
						
			ControlPanel fbBox = new ControlPanel(getText("ui.facebookBox")) {

				JCheckBox useFacebookBox = new JCheckBox(getText("ui.useFacebookBox"));
				JTextField facebookPageId = new JSmartTextField(16);
				JLabel mandatoryInfo = new JLabel(mandatory);
				JCheckBox facebookFaces = new JCheckBox(getText("ui.showFaces"));
				JCheckBox facebookHeader = new JCheckBox(getText("ui.showHeader"));
				WrappableJLabel fbBoxInfo = new WrappableJLabel("<html><i>" + getText("ui.facebookBoxInfo") + " </i></html>");

				{
					mandatoryInfo.setToolTipText(getText("ui.mandatory"));
					fbBoxInfo.setPreferredWidth(uiWidth - 260);
					ComponentUtilities.whenSelectedEnable(useFacebookBox, new JComponent[] {facebookPageId, facebookFaces, facebookHeader});
					
					add(useFacebookBox);
					add("br", new JLabelFor(getText("ui.facebookPageId"), facebookPageId));
					add(mandatoryInfo);
					add("tab hfill", facebookPageId);
					add("br", facebookHeader);
					add("", facebookFaces);
					add("br", fbBoxInfo);
				}

			};
			
			{
				mandatoryInfo.setToolTipText(getText("ui.mandatory"));
				links.setBorder(emptyBorder);
				
				add(shares);
				add("hfill", links);
				add("br", buttons);
				add("br hfill", fbBox);

				putClientProperty("helpPage", helpRoot + "/Site/Social");
			}
		};
		
		//	Map settings
		
		ControlPanel mapSettings = new ControlPanel() {

			JComboBox mapType = new JComboBox(new Object[]{
				new Item("roadmap", getText("ui.roadmap")),
				new Item("satellite", getText("ui.satellite")),
				new Item("hybrid", getText("ui.hybrid")),
				new Item("terrain", getText("ui.terrain"))
			});
			JSlider mapZoom = new JSlider(JSlider.HORIZONTAL, 1, 20, 18);
			JTextField googleApiKey = new JSmartTextField(24);
			JLabel googleApiInfo = new JLabel(infoIcon);
			WrappableJLabel note = new WrappableJLabel("<html><i>" + getText("ui.mapApiKeyInfo") + "</i></html>");
			WrappableJLabel instr = new WrappableJLabel("<html><i>" + getText("ui.selectIndexOrLightbox") + "</i></html>");
			
			{
				mapZoom.setOrientation(JSlider.HORIZONTAL);
				mapZoom.setMinimum(0);
				mapZoom.setMaximum(20);
				mapZoom.setValue(18);
				mapZoom.setMajorTickSpacing(10);
				mapZoom.setMinorTickSpacing(1);
				mapZoom.setPaintTicks(true);
				mapZoom.setPaintLabels(true);
				mapZoom.setSnapToTicks(true);
				instr.setPreferredWidth(uiWidth - 240);
				note.setPreferredWidth(uiWidth - 340);
				googleApiInfo.addMouseListener(new MouseAdapter() {  
					@Override
					public void mouseReleased(MouseEvent e) {
						JOptionPane.showMessageDialog(window, "<html><i>" + getText("ui.gooleMapsApiNote") + "</i></html>", getText("ui.format"), JOptionPane.INFORMATION_MESSAGE);
				}});
				
				add(new JLabelFor(getText("ui.initialView"), mapType));
				add("tab", mapType);
				add("br", new JLabelFor(getText("ui.initialZoom"), mapZoom));
				add("tab", mapZoom);
				add("br", new JLabelFor(getText("ui.googleApiKey"), googleApiKey));
				add("tab", googleApiKey);
				add("br tab", new JLinkLabel("https://console.developers.google.com/apis/credentials", getText("ui.createNew")));
				add("", googleApiInfo);
				add("br tab", note);
				add("br tab", instr);

				putClientProperty("helpPage", helpRoot + "/Site/Map");
			}

		};			
				
		// Audio clips

		ControlPanel audioClips = new ControlPanel() {

			JCheckBox useAudioClipButton = new JCheckBox(getText("ui.playAudioClips"), false);
			JSpinner audioClipVolume = new JSpinner(new SpinnerNumberModel(75, 1, 100, 1));
			JCheckBox copyAudioClips = new JCheckBox(getText("ui.copyAudioClips"));

			{
				useAudioClipButton.setToolTipText(getText("ui.playAudioClipsInfo"));
				audioClipVolume.setToolTipText(getText("ui.audioClipVolumeInfo"));
				copyAudioClips.setToolTipText(getText("ui.copyAudioClipsInfo") + " (" + getText("ui.oldMethod") + ")");
				
				add("", useAudioClipButton);
				add("br", new JLabel(getText("ui.initialVolume")));
				add("", audioClipVolume);
				add("", new JLabel("%"));
				add("br", copyAudioClips);
				
				putClientProperty("helpPage", helpRoot + "/Site/Audio_clips");
			}
		};
		
		// Background music

		ControlPanel backgroundMusic = new ControlPanel() {

			JPlaylist backgroundAudio = new JPlaylist();
			JCheckBox backgroundAudioAutoPlay = new JCheckBox(getText("ui.autoStart"));
			JSpinner backgroundAudioVolume = new JSpinner(new SpinnerNumberModel(25, 1, 100, 1));
			JCheckBox backgroundAudioSlideshowControl = new JCheckBox(getText("ui.slideshowControl"));
			JCheckBox muteBackgroundAudio = new JCheckBox(getText("ui.muteBackgroundAudio"), true);
			JCheckBox backgroundAudioLoop = new JCheckBox(getText("ui.loop"));
			JCheckBox backgroundAudioRetainPosition = new JCheckBox(getText("ui.retainPosition"), true);
			WrappableJLabel autostartNotice = new WrappableJLabel("<html><i>" + getText("ui.autostartNotice") + "</i></html>");

			{
				backgroundAudioSlideshowControl.setToolTipText(getText("ui.slideshowControlInfo"));
				muteBackgroundAudio.setToolTipText(getText("ui.muteBackgroundAudioInfo"));
				backgroundAudioRetainPosition.setToolTipText(getText("ui.retainPositionInfo"));
				autostartNotice.setPreferredWidth(uiWidth - 260);
				
				add("br hfill", backgroundAudio);
				add("br", new JLabel(getText("ui.initialVolume")));
				add("", backgroundAudioVolume);
				add("", new JLabel("%"));
				add("br", backgroundAudioAutoPlay);
				add("tab", backgroundAudioLoop);
				add("br", backgroundAudioRetainPosition);
				add("br", backgroundAudioSlideshowControl);
				add("br", muteBackgroundAudio);
				add("br", new JLabel(infoIcon));
				add("", autostartNotice);
				
				putClientProperty("helpPage", helpRoot + "/Site/Background_music");
			}
		};
		
		// Album info popup
		
		ControlPanel albumInfo = new ControlPanel() {

			JCheckBox headerTopLevelOnly = new JCheckBox(getText("ui.topLevelOnly"), true);
			JSmartTextArea header = new JSmartTextArea(6, 30);
			JScrollPane headerPane = new JScrollPane(header);

			{
				header.setEditable(true);
				header.setLineWrap(false);
				header.setFont(mono);
				header.setTabSize(2);
				header.setToolTipText(getText("ui.customContentInfo"));
				allowHTMLEditing(header);
				headerPane.setBorder(BorderFactory.createTitledBorder(getText("ui.albumInfoText")));
				
				add("hfill vfill", headerPane);
				add("br", headerTopLevelOnly);
				
				putClientProperty("helpPage", helpRoot + "/Site/Album_info_window");
			}
		};
		

		/*	---------------------------------------------------------------
								Sections / Main content
			--------------------------------------------------------------- */
		
		// Folders section
		
		ControlPanel folders = new ControlPanel() {

			ButtonGroup defaultFolderIcon = new ButtonGroup();
			private JRadioButton defaultFolderIconBook = new JRadioButton("");
			private JRadioButton defaultFolderIconYellow = new JRadioButton("");
			private JRadioButton defaultFolderIconBlue = new JRadioButton("");
			private JRadioButton defaultFolderIconHollow = new JRadioButton("");
			
			JCheckBox folderMosaic = new JCheckBox(getText("ui.addMorePictures"), false);
			JSpinner folderMosaicCount = new JSpinner(new SpinnerNumberModel(6, 3, 12, 1));
			
			JScrollPane folderCaptionPane = new JScrollPane(folderCaptionTemplate);
			JCheckBox linkFolderTitle = new JCheckBox(getText("ui.linkFolderTitle"), true);
			JCheckBox showFolderImageCount = new JCheckBox(getText("ui.showFolderImageCount"), true);
			JCheckBox gatherWeblocationInfo = new JCheckBox(getText("ui.gatherWeblocationInfo"), true);
			JCheckBox webLocationOpenNew = new JCheckBox(getText("ui.webLocationsOpenInNewWindow"));
			JCheckBox hideWeblocations = new JCheckBox(getText("ui.hideWeblocations"));
			
			@Override
			public void importVariables(Map<String, Object> vars) {
				Object ftar = vars.get("folderThumbAspectRatio");
				
				if (ftar instanceof Double) {
					vars.put("folderThumbAspectRatio", String.format("%.4f", ftar).replace(".0000", ""));
				}
			};
			
			{	
				ComponentUtilities.whenSelectedEnable(folderMosaic, folderMosaicCount);
				folderCaptionLink = safeLinkLabel(folderCaptionTemplate, helpRoot + "/Sections/Folders", getText("ui.folderThumbCaption"));
				folderCaptionPresets.addItemListener(listener -> {
					if (isSkinReady()) {
						String s = getSelectedValue(folderCaptionPresets);
						if (!s.equals(CUSTOM)) {
							folderCaptionTemplate.setText(s);
						}
					}
				});	
				new StateMonitor() {
					public void onChange() {
						String s = getSelectedValue(folderCaptionPresets);
						folderCaptionPane.setVisible(s.equals(CUSTOM));
					}
				}.add(folderCaptionPresets).done();						
				
				folderCols.setToolTipText(getText("ui.folderColsInfo"));
				fixedShapeFolderThumbs.setToolTipText(getText("ui.fixedShapeThumbsInfo"));
				folderThumbAspectRatio.setToolTipText(getText("ui.aspectRatioInfo"));
				folderMosaic.setToolTipText(getText("ui.folderMosaicInfo"));
				defaultFolderIcon.add(defaultFolderIconBook);
				defaultFolderIcon.add(defaultFolderIconYellow);
				defaultFolderIcon.add(defaultFolderIconBlue);
				defaultFolderIcon.add(defaultFolderIconHollow);
				defaultFolderIconBook.setSelected(true);
				defaultFolderIconBook.setActionCommand("book");
				defaultFolderIconYellow.setActionCommand("yellow");
				defaultFolderIconBlue.setActionCommand("blue");
				defaultFolderIconHollow.setActionCommand("hollow");
				
				folderCaptionTemplate.setEditable(true);
				folderCaptionTemplate.setLineWrap(true);
				folderCaptionTemplate.setWrapStyleWord(true);
				folderCaptionTemplate.setFont(mono);
				folderCaptionTemplate.setTabSize(2);
				folderCaptionPane.setBorder(BorderFactory.createTitledBorder(getText("ui.customCode")));
				ComponentUtilities.whenSelectedEnable(fixedShapeFolderThumbs, new JComponent[]{folderThumbAspectRatio});
				showFolderImageCount.setToolTipText(getText("ui.showFolderImageCountInfo"));
				gatherWeblocationInfo.setToolTipText(getText("ui.gatherWeblocationInfoInfo"));
				
				add(new JLabelFor(getText("ui.folderCols"), folderCols));
				add("tab", folderCols);
				add("tab", fixedShapeFolderThumbs);
				add("br", new JLabelFor(getText("ui.aspectRatio"), folderThumbAspectRatio));
				add("tab", folderThumbAspectRatio);
				add("br", new JLabelFor(getText("ui.placeThumbCaptions"), folderCaptionPlacement));
				add("tab", folderCaptionPlacement);
				add("br", folderMosaic);
				add("tab", folderMosaicCount);
				add("br", new JLabel(getText("ui.defaultFolderIcon")));
				add("tab", defaultFolderIconBook);
				add(new JLabel(svgIcon("folder-s-book", new Dimension(32, 32))));
				add("", defaultFolderIconYellow);
				add(new JLabel(svgIcon("folder-s-yellow", new Dimension(32, 32))));
				add("", defaultFolderIconBlue);
				add(new JLabel(svgIcon("folder-s-blue", new Dimension(32, 32))));
				add("", defaultFolderIconHollow);
				add(new JLabel(svgIcon("folder-s-hollow", new Dimension(32, 32))));
				add("br", new JLabelFor(getText("ui.caption"), folderCaptionPresets));
				add("tab", folderCaptionPresets);
				add("br hfill", folderCaptionPane);
				add("br", linkFolderTitle);
				add("br", showFolderImageCount);
				add("br", gatherWeblocationInfo);
				add("br", hideWeblocations);
				add("br", webLocationOpenNew);
				
				putClientProperty("helpPage", helpRoot + "/Sections/Folders");
			}
		};

		// Pages section
		
		ControlPanel pages = new ControlPanel() {

			JCheckBox useBoxForPages = new JCheckBox(getText("ui.useBoxForPages"), false);
			
			{
				useBoxForPages.setToolTipText(getText("ui.useBoxForPagesInfo"));
				
				add(new JLabel(getText("ui.showPages")));
				add("tab", showPages);
				add("br", useBoxForPages);
				
				putClientProperty("helpPage", helpRoot + "/Sections/Pages");
			}
		};
		
		// Thumbnails (images) section

		ControlPanel images = new ControlPanel() {
			
			JLabel layoutInfo = new JLabel(infoIcon);
			JLabel arLabel = new JLabel(getText("ui.aspectRatio"));
			JLabel columnsInfo = new JLabel(infoIcon);
			JScrollPane thumbCaptionPane = new JScrollPane(thumbCaptionTemplate);
			JCheckBox focusCurrentThumb = new JCheckBox(getText("ui.focusCurrentThumb"), true);
			JCheckBox captionShowOnHover = new JCheckBox(getText("ui.showOnlyOnMouseOver"), true);
			JTextField maxHorizontalAR = new JTextField(8);
			JTextField minVerticalAR = new JTextField(8);

			{
				thumbCaptionLink = safeLinkLabel(thumbCaptionTemplate, helpRoot + "/Sections/Images", getText("ui.thumbnailCaption"));
				
				layoutInfo.addMouseListener(new MouseAdapter() {  
					@Override
					public void mouseReleased(MouseEvent e) {
						WrappableJLabel lbl = new WrappableJLabel("<html>" +
								"<br><p><b>" + getText("ui.fixedShapeGrid") + "</b><br>" + getText("ui.fixedShapeGridInfo") + "</p>" + 
								"<br><p><b>" + getText("ui.grid") + "</b><br>" + getText("ui.gridInfo") + "</p>" + 
								"<br><p><b>" + getText("ui.flexibleGrid") + "</b><br>" + getText("ui.flexibleGridInfo") + "</p>" + 
								"<br><p><b>" + getText("ui.justified") + "</b><br>" + getText("ui.justifiedInfo") + "</p>" +
							"</html>");
						
						lbl.setPreferredWidth(360);
						JOptionPane.showMessageDialog(window, lbl, getText("ui.thumbnailLayout"), JOptionPane.INFORMATION_MESSAGE);
				}});
				thumbCols.setToolTipText(getText("ui.thumbColsInfo"));
				columnsInfo.addMouseListener(new MouseAdapter() {  
					@Override
					public void mouseReleased(MouseEvent e) {
						WrappableJLabel lbl = new WrappableJLabel("<html><p>" + getText("ui.columnsInfo") + "</p><br><p>" + getText("ui.masonryColumnsInfo") + "</p></html>");
						lbl.setPreferredWidth(360);
						JOptionPane.showMessageDialog(window, lbl, getText("ui.thumbCols"), JOptionPane.INFORMATION_MESSAGE);
				}});
				thumbCaptionPresets.addItemListener(listener -> {
					if (isSkinReady()) {
						String s = getSelectedValue(thumbCaptionPresets);
						if (!s.equals(CUSTOM)) {
							thumbCaptionTemplate.setText(s);
						}
					}
				});	
				new StateMonitor() {
					public void onChange() {
						String s = getSelectedValue(thumbCaptionPresets);
						thumbCaptionPane.setVisible(s.equals(CUSTOM));
					}
				}.add(thumbCaptionPresets).done();						

				captionPlacement.addActionListener((ActionEvent e) -> {
					captionShowOnHover.setEnabled(captionPlacement.getSelectedIndex() == 0);
				});
				thumbLayout.addActionListener((ActionEvent e) -> {
					boolean f = thumbLayout.getSelectedIndex() == 0;
					thumbAspectRatio.setEnabled(f);
					arLabel.setEnabled(f);
				});
				thumbCaptionPane.setBorder(BorderFactory.createTitledBorder(getText("ui.customCode")));
				thumbCaptionTemplate.setEditable(true);
				thumbCaptionTemplate.setLineWrap(true);
				thumbCaptionTemplate.setWrapStyleWord(true);
				thumbCaptionTemplate.setFont(mono);
				thumbCaptionTemplate.setTabSize(2);
				focusCurrentThumb.setToolTipText(getText("ui.focusCurrentThumbInfo"));
				maxHorizontalAR.setToolTipText(getText("ui.aspectRatioInfo") + " (> 1)");
				minVerticalAR.setToolTipText(getText("ui.aspectRatioInfo") + " (< 1)");
				//ComponentUtilities.whenSelectedDisable(fixedShapeThumbs, new JComponent[]{maxHorizontalAR, minVerticalAR});

				add(new JLabelFor(getText("ui.thumbnailLayout"), thumbLayout));
				add("tab", thumbLayout);
				add("", layoutInfo);
				add("br", arLabel);
				add("tab", thumbAspectRatio);
				add("br", new JLabelFor(getText("ui.thumbCols"), thumbCols));
				add("tab", thumbCols);
				add("", columnsInfo);
				add("br", new JLabelFor(getText("ui.placeThumbCaptions"), captionPlacement));
				add("tab", captionPlacement);
				add("br tab", captionShowOnHover);
				add("br", new JLabelFor(getText("ui.caption"), thumbCaptionPresets));
				add("tab", thumbCaptionPresets);
				add("br hfill", thumbCaptionPane);
				add("br", focusCurrentThumb);
				add("br", new JLabelFor(getText("ui.maxHorizontalAspectRatio"), maxHorizontalAR));
				add("tab", maxHorizontalAR);
				add("br", new JLabelFor(getText("ui.minVerticalAspectRatio"), minVerticalAR));
				add("tab", minVerticalAR);
				
				putClientProperty("helpPage", helpRoot + "/Sections/Images");
			}
		};
		
		// Map section

		ControlPanel map = new ControlPanel() {

			JCheckBox showMapSection = new JCheckBox(getText("ui.showMapSection"));
			
			{
				showMapSection.setToolTipText(getText("ui.showMapSectionInfo"));
				//ComponentUtilities.whenSelectedEnable(showMapSection, mapSettings);
				
				add(showMapSection);
				add("br", new JLabel(infoIcon));
				add(new JLabel("<html><i>" + getText("ui.findCorrespondingSettings") + " " + getText("ui.site") + " / " + getText("ui.map") + "</i></html>"));

				//add("br", mapSettings);
				putClientProperty("helpPage", helpRoot + "/Sections/Map");
			}
		};

		// Filtering section
		
		ControlPanel filteringAndSort = new ControlPanel() {
						
			ControlPanel filtering = new ControlPanel(getText("ui.filtering")) {
				
				JCheckBox useFilters = new JCheckBox(getText("ui.useFilters"));
				JTextField filterLabel = new JSmartTextField(getText("ui.filter"), 16);
				JCheckBox showEmptyFilters = new JCheckBox(getText("ui.stayVisibleIfNothingToFilter"));
				JVariablesPanel filtersPanel = new JVariablesPanel(texts);
				JTable filterData = filtersPanel.getTable();
				
				{
					showEmptyFilters.setToolTipText(getText("ui.stayVisibleIfNothingToFilterInfo"));
					filtersPanel.setDimensions(new Dimension(320, 160));
					ComponentUtilities.whenSelectedEnable(useFilters, new JComponent[]{ filterLabel, showEmptyFilters, filtersPanel });
					
					add("", useFilters);
					add("tab", new JLabelFor(getText("ui.boxTitle"), filterLabel));
					add("tab", filterLabel);
					add("tab", showEmptyFilters);
					add("br hfill", filtersPanel);
				}
			};
			
			ControlPanel sorting = new ControlPanel(getText("ui.sorting")) {
				
				JCheckBox useSort = new JCheckBox(getText("ui.useSort"));
				JTextField sortLabel = new JSmartTextField(getText("ui.sortBy"), 16);
				JVariablesPanel sortPanel = new JVariablesPanel(texts);
				JTable sortData = sortPanel.getTable();
				
				{
					sortPanel.setDimensions(new Dimension(320, 80));
					ComponentUtilities.whenSelectedEnable(useSort, new JComponent[]{ sortLabel, sortPanel });
					
					add("", useSort);
					add("tab", new JLabelFor(getText("ui.boxTitle"), sortLabel));
					add("tab", sortLabel);
					add("br hfill", sortPanel);
				}
			};
			
			JCheckBox filteringBoxSticky = new JCheckBox(getText("ui.alwaysVisible"));
			
			{
				add("", filteringBoxSticky);
				add("br hfill", filtering);
				add("br hfill", sorting);
				if (jalbumVersion.compareTo(new VersionNumber("18.3b4")) < 0) {
					add("br", new JLabel(getText("ui.cantSaveThisWithVersion").replace("{0}", "18.3")));
				}
				
				putClientProperty("helpPage", helpRoot + "/Sections/Filtering");
			}
		};
		
		//	Shopping cart section
		
		ControlPanel shoppingCart = new ControlPanel() {

			JCheckBox showShop = new JCheckBox(getText("ui.usePaypal"));

			JCheckBox showShopOnImagePagesOnly = new JCheckBox(getText("ui.showShopOnImagePagesOnly"), true);
			JCheckBox shopBoxSticky = new JCheckBox(getText("ui.alwaysVisible"));
			//JCheckBox shopFloatButton = new JCheckBox(getText("ui.useFloatingButton"));
			JTextField shopLabel = new JTextField(24);
			JTextField shopId = new JTextField(24);
			JLabel mandatoryInfo = new JLabel(mandatory);
			/*JComboBox shopPlacement = new JComboBox(new Object[]{ 
				new Item("aboveThumbs", getText("ui.aboveThumbs")),
				new Item("sidebar", getText("ui.inTheSidebar"))
			});*/
			JComboBox shopCurrency = new JComboBox(new Object[]{ 
				new Item("USD", "United States Dollar"),
				new Item("EUR", "Euro"),
				new Item("GBP", "British Pound"),
				new Item("CAD", "Canadian Dollar"),
				new Item("AUD", "Australian Dollar"),
				//new Item("RUB", "Russia Rubles"),
				new Item("JPY", "Japanese Yen"),
				//new Item("INR", "India Rupees"),
				new Item("NZD", "New Zealand Dollar"),
				new Item("CHF", "Swiss Franc"),
				//new Item("ARS", "Argentina Pesos"),
				//new Item("BHD", "Bahrain Dinars"),
				//new Item("BYR", "Belarus Rubles"),
				//new Item("BAM", "Bosnia & Herzegovina C.Marka"),
				new Item("BRL", "Brazilian Real"),
				//new Item("BGN", "Bulgaria Leva"),
				//new Item("CLP", "Chile Pesos"),
				//new Item("CNY", "China Yuan Renminbi"),
				//new Item("COP", "Colombia Pesos"),
				//new Item("CRC", "Costa Rica Colones"),
				//new Item("HRK", "Croatia Kuna"),
				new Item("CZK", "Czech Koruna"),
				new Item("DKK", "Danish Krone"),
				//new Item("EGP", "Egypt Pounds"),
				//new Item("EEK", "Estonia Krooni"),
				//new Item("GTQ", "Guatemala Quetzales"),
				new Item("HKD", "Hong Kong Dollar"),
				new Item("HUF", "Hungary Forint"),
				//new Item("ISK", "Iceland Kronur"),
				//new Item("IDR", "Indonesia Rupiahs"),
				//new Item("IQD", "Iraq Dinars"),
				new Item("ILS", "Israel New Shekel"),
				//new Item("JMD", "Jamaica Dollars"),
				//new Item("JOD", "Jordan Dinars"),
				//new Item("KWD", "Kuwait Dinars"),
				//new Item("LVL", "Latvia Lati"),
				//new Item("LBP", "Lebanon Pounds"),
				//new Item("LTL", "Lithuania Litai"),
				//new Item("MKD", "Macedonia Denars"),
				new Item("MYR", "Malaysian Ringgit"),
				new Item("MXN", "Mexican Peso"),
				//new Item("MDL", "Moldova Lei"),
				//new Item("MAD", "Morocco Dirhams"),
				new Item("NOK", "Norway Krone"),
				//new Item("PEN", "Peru Nuevos Soles"),
				new Item("PHP", "Philippines Peso"),
				new Item("PLN", "Poland Zlotych"),
				//new Item("RON", "Romania New Lei"),
				new Item("RUB", "Russian Ruble"),
				//new Item("SAR", "Saudi Arabia Riyals"),
				//new Item("RSD", "Serbia Dinars"),
				new Item("SGD", "Singapore Dollar"),
				//new Item("ZAR", "South Africa Rand"),
				//new Item("KRW", "South Korea Won"),
				new Item("SEK", "Sweden Krona"),
				new Item("TWD", "New Taiwan Dollar"),
				new Item("THB", "Thailand Baht"),
				new Item("TRY", "Turkish Lira"),
				//new Item("UAH", "Ukraine Hryvnia"),
				//new Item("AED", "United Arab Emirates Dirhams"),
				//new Item("UYU", "Uruguay Pesos"),
				//new Item("VND", "Vietnam Dong")
			});
			JTextField shopSuccessUrl = new JSmartTextField(24);
			JCheckBox shopSameWindowCheckout = new JCheckBox(getText("ui.sameWindowCheckout"));
			JCheckBox shopSendAlbumName = new JCheckBox(getText("ui.sendAlbumName"));
			JCheckBox shopSendFolderPath = new JCheckBox(getText("ui.sendFolderPath"));
			JComboBox shopSendItemName = new JComboBox(new Object[]{ 
				new Item("name", getText("ui.filename")),
				new Item("title", getText("ui.titleOrFilename")),
				new Item("comment", getText("ui.commentOrFilename"))
			});
			JCheckBox shopAskPermissionToEmpty = new JCheckBox(getText("ui.askPermisssionToEmptyCart"));
			
			JComboBox showPriceRange = new JComboBox(new Object[]{ 
				new Item("", getText("ui.none")),
				new Item("lowest", getText("ui.lowestPriceOnly")),
				new Item("minmax", getText("ui.minMax"))
			});
			JCheckBox usePriceAsSingleOption = new JCheckBox(getText("ui.usePriceAsSingleOption"));
			JComboBox shopOptionSelectMethod = new JComboBox(new Object[]{ 
				new Item("combo", getText("ui.dropdownBox")),
				new Item("radio", getText("ui.radioButtons"))
			});
			JTextArea shopOptions = new JSmartTextArea(8, 28);
			JScrollPane shopOptionsPane = new JScrollPane(shopOptions);
			JLabel shopOptionsInfo = new JLabel(infoIcon);

			JTextField shopMinimalAmount = new JSmartTextField(8);
			JTextField shopHandling = new JSmartTextField(8);
			JLabel shopHandlingInfo = new JLabel(infoIcon);
			JTextField shopTax = new JSmartTextField(8);
			JTextField shopQuantityCap = new JSmartTextField(8);

			JTextArea shopCoupons = new JSmartTextArea(4, 16);
			JScrollPane shopCouponsPane = new JScrollPane(shopCoupons);
			JLabel shopCouponsInfo = new JLabel(infoIcon);

			JTextField shopDiscountRate = new JSmartTextField(8);
			JTextField shopDiscountMinQuantity = new JSmartTextField(8);
			JTextField shopDiscountMinAmount = new JSmartTextField(8);

			JSmartTextArea shopInstructions = new JSmartTextArea(6, 24);	
			JScrollPane shopInstructionsPane = new JScrollPane(shopInstructions);
			JSmartTextArea shopInstructionsBox = new JSmartTextArea(6, 24);	
			JScrollPane shopInstructionsBoxPane = new JScrollPane(shopInstructionsBox);
				
			ControlPanel shoppingCartOptions = new ControlPanel() {

				{
					shopId.setToolTipText(getText("ui.sellerIdInfo"));
					mandatoryInfo.setToolTipText(getText("ui.mandatory"));
					shopLabel.setToolTipText(getText("ui.default") + ": " + getText("ui.shoppingCart"));
					shopCurrency.setEditable(false);
					shopCurrency.setToolTipText(getText("ui.currencyInfo"));
					shopSuccessUrl.setToolTipText(getText("ui.shopSuccessUrlInfo"));
					usePriceAsSingleOption.setToolTipText(getText("ui.usePriceAsSingleOptionInfo"));
					shopSameWindowCheckout.setToolTipText(getText("ui.sameWindowCheckoutInfo"));
					shopSendAlbumName.setToolTipText(getText("ui.sendAlbumNameInfo"));

					add(showShopOnImagePagesOnly);
					add("br", shopBoxSticky);
					add("br", new JLabelFor(getText("ui.boxTitle"), shopLabel));
					add("tab", shopLabel);
					add("br", new JLabelFor(getText("ui.sellerId"), shopId));
					add(mandatoryInfo);
					add("tab", shopId);
					add("br tab", new JLinkLabel("https://www.paypal.com/signup/account", getText("ui.signUp")));
					add("br", new JLabelFor(getText("ui.currency"), shopCurrency));
					add("tab", shopCurrency);
					add("br", new JLabelFor(getText("ui.shopSuccessUrl"), shopSuccessUrl));
					add("tab", shopSuccessUrl);
					add("br", new JLabel(getText("ui.showPriceRange")));
					add("tab", showPriceRange);
					add("br", usePriceAsSingleOption);
					add("br", shopSendAlbumName);
					add("tab", shopSendFolderPath);
					add("tab", shopSendItemName);
					add("", new JLabel(getText("ui.max128Chars")));
					add("br", shopSameWindowCheckout);
					add("br", shopAskPermissionToEmpty);
				}
			};

			ControlPanel shoppingCartPricing = new ControlPanel() {

				{
					shopOptionsInfo.addMouseListener(new MouseAdapter() {  
						@Override
						public void mouseReleased(MouseEvent e) {
							String s = "<html><b><kbd>" + getText("ui.shopOptionsFormat") + "</kbd></b><br>" +
									"<br><b>" + getText("ui.examples") + ":</b><br>" +
									"<br><kbd>" + "Print A4=2.0" + "</kbd><br>" +
									"<i>" + getText("ui.optionExample1Hint") + "</i><br>" +
									"<br><kbd>" + "Print A2=10.0+0.5" + "</kbd><br>" +
									"<i>" + getText("ui.optionExample2Hint") + "</i><br>" +
									"<br><kbd>" + "Print A2=10.0+0.5+0" + "</kbd><br>" +
									"<i>" + getText("ui.optionExample3Hint") + "</i><br>" +
									"<br><i>" + getText("ui.noCurrencySign") + "</i></html>";
							JOptionPane.showMessageDialog(window, s, getText("ui.format"), JOptionPane.INFORMATION_MESSAGE);
					}});
					shopOptions.setToolTipText(getText("ui.shopOptionsInfo"));
					shopOptions.setEditable(true);
					shopOptions.setLineWrap(false);
					shopOptions.setFont(mono);
					shopOptionsPane.setBorder(BorderFactory.createTitledBorder(getText("ui.shopOptions")));
					shopHandlingInfo.addMouseListener(new MouseAdapter() {  
						@Override
						public void mouseReleased(MouseEvent e) {
							String s = "<html><b><kbd>" + getText("ui.handlingFormat") + "</kbd></b><br>" +
									"<br><b>" + getText("ui.examples") + ":</b><br>" +
									"<br><kbd>" + "2.0" + "</kbd><br>" +
									"<i>" + getText("ui.handlingExample1Hint") + "</i><br>" +
									"<br><kbd>" + "0.0+0.1" + "</kbd><br>" +
									"<i>" + getText("ui.handlingExample2Hint") + "</i><br>" +
									"<br><kbd>" + "1.0+0+0.1" + "</kbd><br>" +
									"<i>" + getText("ui.handlingExample3Hint") + "</i><br>" +
									"<br><i>" + getText("ui.noCurrencySign") + "</i></html>";
							JOptionPane.showMessageDialog(window, s, getText("ui.format"), JOptionPane.INFORMATION_MESSAGE);
					}});
					shopHandling.setToolTipText(getText("ui.handlingInfo") + " " + getText("ui.leaveEmpty"));
					shopTax.setToolTipText(getText("ui.taxInfo") + " " + getText("ui.leaveEmpty"));
					shopQuantityCap.setToolTipText(getText("ui.shopQuantityCapInfo") + " " + getText("ui.leaveEmpty"));
					shopMinimalAmount.setToolTipText(getText("ui.shopMinimalCartValueInfo"));
					shopDiscountRate.setToolTipText(getText("ui.shopDiscountRateInfo") + " " + getText("ui.leaveEmpty"));
					shopDiscountMinQuantity.setToolTipText(getText("ui.shopDiscountMinQuantityInfo"));
					shopDiscountMinAmount.setToolTipText(getText("ui.shopDiscountMinAmountInfo"));
					shopCoupons.setToolTipText(getText("ui.shopCouponsInfo"));
					shopCoupons.setEditable(true);
					shopCoupons.setLineWrap(false);
					shopCoupons.setFont(mono);
					shopCouponsInfo.addMouseListener(new MouseAdapter() {  
						@Override
						public void mouseReleased(MouseEvent e) {
							String s = "<html><b><kbd>" + getText("ui.shopCouponsFormat") + "</kbd></b><br>" +
									"<br><b>" + getText("ui.examples") + ":</b><br>" +
									"<br><kbd>" + "5OFF=5.0" + "</kbd><br>" +
									"<i>" + getText("ui.couponExample1Hint") + "</i><br>" +
									"<br><kbd>" + "XMAS=20% &lt;2016-12-24" + "</kbd><br>" +
									"<i>" + getText("ui.couponExample2Hint") + "</i><br>" +
									"<br><kbd>" + "MIN10=30% 10+" + "</kbd><br>" +
									"<i>" + getText("ui.couponExample3Hint") + "</i><br>" +
									"<br><i>" + getText("ui.noCurrencySign") + "</i></html>";
							JOptionPane.showMessageDialog(window, s, getText("ui.format"), JOptionPane.INFORMATION_MESSAGE);
					}});
					shopCouponsPane.setBorder(BorderFactory.createTitledBorder(getText("ui.shopCoupons")));

					add("", new JLabelFor(getText("ui.handling"), shopHandling));
					add(shopHandlingInfo);
					add("tab", shopHandling);
					add("tab", new JLabelFor(getText("ui.tax") + " (%)", shopTax));
					add("tab", shopTax);
					add("br", new JLabelFor(getText("ui.shopQuantityCap"), shopQuantityCap));
					add("tab", shopQuantityCap);
					add("tab", new JLabelFor(getText("ui.shopMinimalCartValue"), shopMinimalAmount));
					add("tab", shopMinimalAmount);
					add("br", new JLabelFor(getText("ui.shopDiscountRate") + " (%)", shopDiscountRate));
					add("tab", shopDiscountRate);
					add("tab", new JLabelFor(getText("ui.minQuantity"), shopDiscountMinQuantity));
					add("tab", shopDiscountMinQuantity);
					add("tab", new JLabelFor(getText("ui.minAmount"), shopDiscountMinAmount));
					add("tab", shopDiscountMinAmount);
					add("br hfill", shopOptionsPane);
					add(shopOptionsInfo);
					add("br hfill", shopCouponsPane);
					add(shopCouponsInfo);
					add("br", new JLabelFor(getText("ui.selectMethod"), shopOptionSelectMethod));
					add("tab", shopOptionSelectMethod);

				}
			};

			ControlPanel shoppingCartInstructions = new ControlPanel() {

				{
					shopInstructions.setToolTipText(getText("ui.shopInstructionsInfo"));
					shopInstructions.setEditable(true);
					shopInstructions.setLineWrap(true);
					shopInstructions.setWrapStyleWord(true);
					shopInstructions.setFont(mono);
					allowHTMLEditing(shopInstructions);
					shopInstructionsPane.setBorder(BorderFactory.createTitledBorder(getText("ui.shopInstructions")));
					shopInstructionsBox.setToolTipText(getText("ui.shopInstructionsBoxInfo"));
					shopInstructionsBox.setEditable(true);
					shopInstructionsBox.setLineWrap(true);
					shopInstructionsBox.setWrapStyleWord(true);
					shopInstructionsBox.setFont(mono);
					shopInstructionsBox.setTabSize(2);
					allowHTMLEditing(shopInstructionsBox);
					shopInstructionsBoxPane.setBorder(BorderFactory.createTitledBorder(getText("ui.shopInstructionsBox")));
					
					add("hfill", shopInstructionsPane);
					add("br hfill", shopInstructionsBoxPane);
				}
			};

			//WrappableJLabel instr = new WrappableJLabel("<html><i>" + getText("ui.selectIndexOrLightbox") + "</i></html>");
			
			JTabbedPane shopTabs = new JTabbedPane() {
								
				{
					shoppingCartOptions.setBorder(emptyBorder);
					shoppingCartPricing.setBorder(emptyBorder);
					shoppingCartInstructions.setBorder(emptyBorder);
					
					addTab(getText("ui.settings"), shoppingCartOptions);
					addTab(getText("ui.pricing"), shoppingCartPricing);
					addTab(getText("ui.instructions"), shoppingCartInstructions);
				}
			};

			{
				if (!commercialUseAllowed) {
					showShop.setSelected(false);
					commercialMonitor.add(showShop);
				}
				//instr.setPreferredSize(new Dimension(uiWidth - 210, 32));
				showShop.setToolTipText(getText("ui.sellPhotosInfo"));
				ComponentUtilities.whenSelectedEnable(showShop, new JComponent[]{ shoppingCartOptions, shoppingCartPricing, shoppingCartInstructions } );
				
				add("", showShop);
				add("br hfill", shopTabs);
				//add("br", new JLabel(infoIcon));
				//add(instr);

				putClientProperty("helpPage", helpRoot + "/Sections/Shopping_cart");
			}
		};
		
		// Feedback section
		
		ControlPanel feedback = new ControlPanel() {

			JCheckBox showFeedback = new JCheckBox(getText("ui.useFeedback"));

			ControlPanel feedbackOptions = new ControlPanel() {

				JCheckBox showFeedbackOnImagePagesOnly = new JCheckBox(getText("ui.showOnImagePagesOnly"), true);
				JCheckBox feedbackBoxSticky = new JCheckBox(getText("ui.alwaysVisible"));
				JCheckBox feedbackAddMultiple = new JCheckBox(getText("ui.canAddMultipleTimes"));
				JTextField feedbackLabel = new JTextField(24);
				JTextField feedbackEmail = new JTextField(24);
				JLabel mandatoryInfo = new JLabel(mandatory);
				JComboBox feedbackFormatting = new JComboBox(new Object[] {
					new Item("human", getText("ui.humanReadable")),			
					new Item("serialized", getText("ui.serialized"))
				});
				JTextField feedbackViewButtonLabel = new JTextField(24);
				JTextField feedbackAddButtonLabel = new JTextField(24);
				JTextField feedbackAddButtonTooltip = new JTextField(24);
				JTextField feedbackModalTitle = new JTextField(24);
				JTextField feedbackCopyButtonLabel = new JTextField(24);
				JCheckBox useFeedbackSendButton = new JCheckBox(getText("ui.useSendButton"), true);
				JTextField feedbackSendButtonLabel = new JTextField(24);
				
				{
					feedbackLabel.setToolTipText(getText("ui.feedbackLabelInfo"));
					//ComponentUtilities.whenSelectedDisable(feedbackBoxSticky, feedbackFloatButton);
					mandatoryInfo.setToolTipText(getText("ui.mandatory"));
					feedbackViewButtonLabel.setToolTipText(getText("ui.feedbackViewButtonLabelInfo"));
					feedbackAddButtonLabel.setToolTipText(getText("ui.feedbackAddButtonLabelInfo"));
					feedbackAddButtonTooltip.setToolTipText(getText("ui.feedbackActionButtonTooltipInfo"));
					feedbackModalTitle.setToolTipText(getText("ui.feedbackModalTitleInfo"));
					feedbackCopyButtonLabel.setToolTipText(getText("ui.feedbackCopyButtonLabelInfo"));
					feedbackSendButtonLabel.setToolTipText(getText("ui.feedbackSendButtonLabelInfo"));
					feedbackEmail.setToolTipText(getText("ui.feedbackEmailInfo"));
					
					ComponentUtilities.whenSelectedEnable(useFeedbackSendButton, feedbackSendButtonLabel);
					
					add(showFeedbackOnImagePagesOnly);
					add("br", feedbackBoxSticky);
					add("br", feedbackAddMultiple);
					add("br", new JLabelFor(getText("ui.boxTitle"), feedbackLabel));
					add("tab", feedbackLabel);
					add("tab", new JLabel("<html><i>" + getText("ui.default") + ": \"" + getText("feedback") + "\"</i></html>"));
					add("br", new JLabelFor(getText("ui.feedbackEmail"), feedbackEmail));
					add(mandatoryInfo);
					add("tab", feedbackEmail);
					add("br", new JLabelFor(getText("ui.formatting"), feedbackFormatting));
					add("tab", feedbackFormatting);
					add("br", new JLabelFor(getText("ui.viewButtonLabel"), feedbackViewButtonLabel));
					add("tab", feedbackViewButtonLabel);
					add("tab", new JLabel("<html><i>" + getText("ui.default") + ": \"" + getText("editFeedback") + "\"</i></html>"));
					add("br", new JLabelFor(getText("ui.addButtonLabel"), feedbackAddButtonLabel));
					add("tab", feedbackAddButtonLabel);
					add("tab", new JLabel("<html><i>" + getText("ui.default") + ": \"" + getText("writeFeedback") + "\"</i></html>"));
					add("br", new JLabelFor(getText("ui.addButtonTooltip"), feedbackAddButtonTooltip));
					add("tab", feedbackAddButtonTooltip);
					add("tab", new JLabel("<html><i>" + getText("ui.default") + ": \"" + getText("provideFeedbackOnSelectedItems") + "\"</i></html>"));
					add("br", new JLabelFor(getText("ui.modalTitle"), feedbackModalTitle));
					add("tab", feedbackModalTitle);
					add("tab", new JLabel("<html><i>" + getText("ui.default") + ": \"" + getText("feedbackOnAlbum") + "\"</i></html>"));
					add("br", new JLabelFor(getText("ui.copyButtonLabel"), feedbackCopyButtonLabel));
					add("tab", feedbackCopyButtonLabel);
					add("tab", new JLabel("<html><i>" + getText("ui.default") + ": \"" + getText("copyFeedback") + "\"</i></html>"));
					add("br", useFeedbackSendButton);
					add("br", new JLabelFor(getText("ui.sendButtonLabel"), feedbackSendButtonLabel));
					add("tab", feedbackSendButtonLabel);
					add("tab", new JLabel("<html><i>" + getText("ui.default") + ": \"" + getText("sendFeedback") + "\"</i></html>"));
				}
			};
			
			ControlPanel feedbackForm = new ControlPanel() {
				
				JTextArea feedbackTemplate = new JSmartTextArea(5, 24);
				JScrollPane feedbackTemplatePane = new JScrollPane(feedbackTemplate);
				//JLabel feedbackTemplateInfo = new JLabel(infoIcon);
				
				{
					feedbackTemplate.setToolTipText(getText("ui.feedbackTemplateInfo"));
					feedbackTemplate.setEditable(true);
					feedbackTemplate.setLineWrap(true);
					feedbackTemplate.setWrapStyleWord(true);
					feedbackTemplate.setFont(mono);
					feedbackTemplate.setTabSize(2);
					feedbackTemplatePane.setBorder(BorderFactory.createTitledBorder(getText("ui.feedbackTemplate")));
					
					add("hfill vfill", feedbackTemplatePane);
					
				}
			};
			
			ControlPanel feedbackInstruction = new ControlPanel() {
				
				JSmartTextArea feedbackInstructions = new JSmartTextArea(6, 24);	
				JScrollPane feedbackInstructionsPane = new JScrollPane(feedbackInstructions);
				JSmartTextArea feedbackInstructionsBox = new JSmartTextArea(6, 24);	
				JScrollPane feedbackInstructionsBoxPane = new JScrollPane(feedbackInstructionsBox);
				
				{
					
					feedbackInstructions.setToolTipText(getText("ui.feedbackInstructionsInfo"));
					feedbackInstructions.setEditable(true);
					feedbackInstructions.setLineWrap(true);
					feedbackInstructions.setWrapStyleWord(true);
					feedbackInstructions.setFont(mono);
					feedbackInstructions.setTabSize(2);
					allowHTMLEditing(feedbackInstructions);
					feedbackInstructionsPane.setBorder(BorderFactory.createTitledBorder(getText("ui.feedbackInstructions")));
					
					feedbackInstructionsBox.setToolTipText(getText("ui.feedbackInstructionsBoxInfo"));
					feedbackInstructionsBox.setEditable(true);
					feedbackInstructionsBox.setLineWrap(true);
					feedbackInstructionsBox.setWrapStyleWord(true);
					feedbackInstructionsBox.setFont(mono);
					feedbackInstructionsBox.setTabSize(2);
					allowHTMLEditing(feedbackInstructionsBox);
					feedbackInstructionsBoxPane.setBorder(BorderFactory.createTitledBorder(getText("ui.feedbackInstructionsBox")));

					add("hfill", feedbackInstructionsPane);
					add("br hfill", feedbackInstructionsBoxPane);
				}
			};
			
			JTabbedPane feedbackTabs = new JTabbedPane() {
								
				{
					feedbackOptions.setBorder(emptyBorder);
					feedbackForm.setBorder(emptyBorder);
					feedbackInstruction.setBorder(emptyBorder);
					
					addTab(getText("ui.settings"), feedbackOptions);
					addTab(getText("ui.feedbackForm"), feedbackForm);
					addTab(getText("ui.instructions"), feedbackInstruction);
				}
			};
			
			{
				if (!commercialUseAllowed) {
					showFeedback.setSelected(false);
					commercialMonitor.add(showFeedback);
				}
				showFeedback.setToolTipText(getText("ui.useFeedbackInfo"));
				ComponentUtilities.whenSelectedEnable(showFeedback, new JComponent[] {feedbackOptions,feedbackForm,feedbackInstruction});
				feedbackOptions.setBorder(emptyBorder);

				add(showFeedback);
				add("br vfill hfill", feedbackTabs);

				putClientProperty("helpPage", helpRoot + "/Sections/Feedback");
			}

		};
		
		// Tag cloud section

		ControlPanel tagCloud = new ControlPanel() {

			JComboBox tagCloudSource = new JComboBox(new Object[] {
				new Item("none", getText("ui.noTagCloud")),			
				new Item("current", getText("ui.currentFolder")),			
				new Item("subfolders", getText("ui.subfolders")),
				new Item("tree", getText("ui.wholeAlbum"))
			});
			JCheckBox tagCloudUseFolders = new JCheckBox(getText("ui.folders"), true);
			JCheckBox tagCloudUsePages = new JCheckBox(getText("ui.pages"));
			JCheckBox tagCloudUseWebLocations = new JCheckBox(getText("ui.webLocations"));
			JLabel tagCloudSkipLevelsLabel = new JLabel(getText("ui.skipLevels"));
			JSpinner tagCloudSkipLevels = new JSpinner(new SpinnerNumberModel(1, 0, 5, 1));
			JTextField tagCloudLabel = new JSmartTextField(12);
			JTextField tagCloudFields = new JSmartTextField();
			JComboBox tagCloudSort = new JComboBox(new Object[] {
				new Item("none", getText("ui.unsorted")),			
				new Item("name", getText("ui.name")),			
				new Item("frequency", getText("ui.frequency"))
			});
			JCheckBox tagCloudFontVaries = new JCheckBox(getText("ui.tagCloudFontVaries"));
			JCheckBox tagCloudSearch = new JCheckBox(getText("ui.addSearchBox"));

			{
				tagCloudLabel.setText(getText("ui.labels"));
				tagCloudSource.addActionListener(new ActionListener() {
					@Override
					public void actionPerformed(ActionEvent e) {
						int tsi = tagCloudSource.getSelectedIndex();
						tagCloudUseFolders.setEnabled(tsi > 0);
						tagCloudUsePages.setEnabled(tsi > 0);
						tagCloudUseWebLocations.setEnabled(tsi > 0);
						tagCloudLabel.setEnabled(tsi > 0);
						tagCloudFields.setEnabled(tsi > 0);
						tagCloudSort.setEnabled(tsi > 0);
						tagCloudFontVaries.setEnabled(tsi > 0);
						tagCloudSearch.setEnabled(tsi > 0);
						tagCloudSkipLevels.setVisible(tsi == 2);
						tagCloudSkipLevelsLabel.setVisible(tsi == 2);
				}});
				tagCloudSkipLevels.setToolTipText(getText("ui.skipLevelsInfo"));
				tagCloudFields.setToolTipText("<html>" + getText("ui.searchFieldsInfo") + "</html>");
				tagCloudFontVaries.setToolTipText(getText("ui.tagCloudFontVariesInfo"));

				add(new JLabelFor(getText("ui.collectFrom"), tagCloudSource));
				add("tab", tagCloudSource);
				add("tab", tagCloudSkipLevelsLabel);
				add("", tagCloudSkipLevels);
				add("", tagCloudUseFolders);
				add("", tagCloudUsePages);
				add("", tagCloudUseWebLocations);
				add("br", new JLabelFor(getText("ui.boxTitle"), tagCloudLabel));
				add("tab hfill", tagCloudLabel);
				add("br", new JLabelFor(getText("ui.fields"), tagCloudFields));
				add("tab hfill", tagCloudFields);
				add("br", new JLabelFor(getText("ui.sortBy"), tagCloudFields));
				add("tab", tagCloudSort);
				add("tab", tagCloudFontVaries);
				add("br tab", tagCloudSearch);
				
				putClientProperty("helpPage", helpRoot + "/Sections/Tag_cloud_box");
			}
		};

		// Search new section
		
		ControlPanel searchNew = new ControlPanel() {

			JComboBox searchNewSource = new JComboBox(new Object[] {
				new Item("none", getText("ui.noSearchNew")),			
				new Item("current", getText("ui.currentFolder")),			
				new Item("subfolders", getText("ui.subfolders")),
				new Item("tree", getText("ui.wholeAlbum"))
			});
			JTextField searchNewLabel = new JSmartTextField(12);
			JTextField searchNewDays = new JSmartTextField(12);
			JComboBox searchNewReference = new JComboBox(new Object[] {
				new Item("dateTaken", getText("ui.dateTaken")),
				new Item("fileModified", getText("ui.fileModified")),
				new Item("added", getText("ui.addedToAlbum"))
			});
			JCheckBox searchNewSinceLastVisit = new JCheckBox(getText("ui.searchNewSinceLastVisit"));

			{
				searchNewSource.addActionListener(new ActionListener() {
					@Override
					public void actionPerformed(ActionEvent e) {
						boolean on = searchNewSource.getSelectedIndex() > 0;
						searchNewLabel.setEnabled(on);
						searchNewDays.setEnabled(on);
						searchNewReference.setEnabled(on);
						searchNewSinceLastVisit.setEnabled(on);
				}});
				searchNewLabel.setText(getText("ui.newImages"));
				searchNewDays.setToolTipText(getText("ui.searchNewDaysInfo"));
				searchNewSinceLastVisit.setToolTipText(getText("ui.searchNewSinceLastVisitInfo"));

				add(new JLabelFor(getText("ui.collectFrom"), searchNewSource));
				add("tab", searchNewSource);
				add("br", new JLabelFor(getText("ui.boxTitle"), searchNewLabel));
				add("tab hfill", searchNewLabel);
				add("br", new JLabelFor(getText("ui.days"), searchNewDays));
				add("tab hfill", searchNewDays);
				add("br", new JLabelFor(getText("ui.reference"), searchNewReference));
				add("tab", searchNewReference);
				add("br tab", searchNewSinceLastVisit);
				
				putClientProperty("helpPage", helpRoot + "/Sections/Search_new_images");
			}
		};

		// Neighboring folders section

		ControlPanel neighboringFolders = new ControlPanel() {
			
			JCheckBox linkNeighboringFolders = new JCheckBox(getText("ui.linkNeighboringFolders"), true);
			JCheckBox neighboringFolderBgImage = new JCheckBox(getText("ui.neighboringFolderBgImage"), true);
			JCheckBox neighboringFolderSkipLevels = new JCheckBox(getText("ui.skipFolderLevels"), false);
			JCheckBox neighboringFolderLoop = new JCheckBox(getText("ui.neighboringFolderLoop"), false);

			{
				linkNeighboringFolders.setToolTipText(getText("ui.linkNeighboringFoldersInfo"));
				neighboringFolderBgImage.setToolTipText(getText("ui.neighboringFolderBgImageInfo"));
				neighboringFolderSkipLevels.setToolTipText(getText("ui.skipFolderLevelsInfo"));
				neighboringFolderLoop.setToolTipText(getText("ui.neighboringFolderLoopInfo"));
				ComponentUtilities.whenSelectedEnable(linkNeighboringFolders, new JComponent[]{ neighboringFolderBgImage, neighboringFolderSkipLevels, neighboringFolderLoop });

				add(linkNeighboringFolders);
				add("br", neighboringFolderSkipLevels);
				add("br", neighboringFolderBgImage);
				add("br", neighboringFolderLoop);
				
				putClientProperty("helpPage", helpRoot + "/Sections/Neighboring_folders");
			}
		};
				
		// Commenting section
		
		ControlPanel commenting = new ControlPanel() {

			ControlPanel fbCommenting = new ControlPanel("Facebook") {

				JCheckBox facebookCommenting = new JCheckBox(getText("ui.facebookCommenting"));
				JTextField facebookCommentingPosts = new JSmartTextField("3", 2);

				{
					facebookCommentingPosts.setToolTipText(getText("ui.facebookCommentingPostsInfo"));

					add(facebookCommenting);
					add("br", new JLabel(getText("ui.facebookCommentingPosts")));
					add("tab", facebookCommentingPosts);
					add("br", new JLabel(infoIcon));
					add(new JLabel("<html><i>" + getText("ui.fbAppIdHint") + "</i></html>"));
				}

			};

			ControlPanel disqCommenting = new ControlPanel("Disqus") {

				JCheckBox disqusCommenting = new JCheckBox(getText("ui.disqusCommenting"));
				JTextField disqusAppId = new JSmartTextField("");
				JLabel mandatoryInfo = new JLabel(mandatory);

				{
					disqusAppId.setToolTipText(getText("ui.disqusAppIdInfo"));
					mandatoryInfo.setToolTipText(getText("ui.mandatory"));
					ComponentUtilities.whenSelectedEnable(disqusCommenting, new JComponent[]{disqusAppId});

					add(disqusCommenting);
					add("br", new JLabel(getText("ui.disqusAppId")));
					add(mandatoryInfo);
					add("tab hfill", disqusAppId);
					add("", new JLinkLabel("https://disqus.com/admin/create/", getText("ui.signUp")));
				}

			};

			{
				add("hfill", fbCommenting);
				add("br hfill", disqCommenting);

				putClientProperty("helpPage", helpRoot + "/Sections/Commenting");
			}
		};
		
		// Custom content section
		
		ControlPanel customSection = new ControlPanel() {

			JSmartTextArea customSectionHook = new JSmartTextArea(7, 20);
			JScrollPane customSectionPane = new JScrollPane(customSectionHook);
			JCheckBox customSectionTopLevelOnly = new JCheckBox(getText("ui.topLevelOnly"), false);

			{
				customSectionHook.setEditable(true);
				customSectionHook.setLineWrap(false);
				customSectionHook.setWrapStyleWord(true);
				customSectionHook.setFont(mono);
				customSectionHook.setTabSize(2);
				allowHTMLEditing(customSectionHook);
				customSectionPane.setBorder(BorderFactory.createTitledBorder(getText("ui.customContentHint")));

				add("hfill vfill", customSectionPane);
				add("br", customSectionTopLevelOnly);
				
				putClientProperty("helpPage", helpRoot + "/Sections/Custom_content");
			}
		};

		/*	---------------------------------------------------------------
									Lightbox
			--------------------------------------------------------------- */
		
		ControlPanel lightboxDesign = new ControlPanel() {
		
			JCheckBox lightboxFullscreen = new JCheckBox(getText("ui.lightboxFullscreen"));
			JCheckBox slideshowFullscreen = new JCheckBox(getText("ui.slideshowFullscreen"));
			JCheckBox rightClickProtect = new JCheckBox(getText("ui.rightClickProtect"));
			JComboBox mouseWheelAction = new JComboBox(new Object[] {
				new Item("", getText("ui.default")), 
				new Item("navigation", getText("ui.navigation")),
				new Item("zoom", getText("ui.zoom"))
			});
			JCheckBox enableKeyboard = new JCheckBox(getText("ui.enableKeyboard"), true);
			JComboBox clickAction = new JComboBox(new Object[] {
				new Item("donothing", getText("ui.doNothing")), 
				new Item("nextimage", getText("ui.nextImage")),
				new Item("togglecontrols", getText("ui.toggleControls")),
				new Item("toggleall", getText("ui.toggleAllPanels"))
			});
			JCheckBox clickBesideForIndex = new JCheckBox(getText("ui.clickBesideForIndex"));
						
			{
				lightboxDesignLink = safeLinkLabel(lightboxBackgroundColor, helpRoot + "/Lightbox/Settings", getText("ui.lightboxDesign"));
				
				lightboxUseMainBg.setToolTipText(getText("ui.lightboxUseMainBackgroundInfo"));
				lightboxBackgroundColor.getTextComponent().setFont(mono);
				lightboxBackgroundColor.setToolTipText(getText("ui.backgroundColorInfo"));
				lightboxBorderColor.getTextComponent().setFont(mono);
				lightboxBorderColor.setToolTipText(getText("ui.borderColorInfo"));
				rightClickProtect.setToolTipText(getText("ui.rightClickProtectInfo"));
				mouseWheelAction.setToolTipText(getText("ui.mouseWheelActionInfo"));
				enableKeyboard.setToolTipText(getText("ui.enableKeyboardInfo"));
				clickBesideForIndex.setToolTipText(getText("ui.clickBesideForIndexInfo"));

				add("", lightboxUseMainBg);
				add("br", new JLabelFor(getText("ui.backgroundColor"), lightboxBackgroundColor));
				add("tab", lightboxBackgroundColor);
				add("", lightboxBackgroundColor.getTextComponent());
				add("br", new JLabelFor(getText("ui.borderColor"), lightboxBorderColor));
				add("tab", lightboxBorderColor);
				add("", lightboxBorderColor.getTextComponent());
				add("br", new JLabelFor(getText("ui.borderWidth"), lightboxBorderWidth));
				add("tab", lightboxBorderWidth);
				add("", new JLabel("px"));
				add("br", new JLabel(getText("ui.cornerRadius")));
				add("tab", imgCornerRadius);
				add("", new JLabel("px"));
				add("br", new JLabelFor(getText("ui.padding"), fitPadding));
				add("tab", fitPadding);
				add(new JLabel("px"));
				add("br", lightboxFullscreen);
				add("br", slideshowFullscreen);
				add("br", rightClickProtect);
				add("br", enableKeyboard);
				add("br", new JLabelFor(getText("ui.mouseWheelAction"), mouseWheelAction));
				add("tab", mouseWheelAction);
				add("br", new JLabelFor(getText("ui.clickAction"), clickAction));
				add("tab", clickAction);
				add("br", clickBesideForIndex);

				putClientProperty("helpPage", helpRoot + "/Lightbox/Settings");
			}
		};

		ControlPanel lightboxControls = new ControlPanel() {

			JLabel thumbStripHeightLabel = new JLabel(getText("ui.thumbstripHeight"));
			private JLabel iconThumbstrip = new JLabel(icon("hide-top-panel"));
			private JLabel iconFit = new JLabel(icon("expand"));
			private JLabel iconPlay = new JLabel(icon("play"));
			private JLabel iconFullscr = new JLabel(icon("fullscreen"));
			JCheckBox slideshowTimingControl = new JCheckBox(getText("ui.canChangeSlideshowTiming"));
			
			ControlPanel thumbStripSettings = new ControlPanel() {
				
				{
					add("", thumbnailsVisible);
					add("br", thumbStripHeightLabel);
					add("", thumbstripHeight);
					
				}
			};
			
			ControlPanel fitImageSettings = new ControlPanel() {
			
				{
					displayOriginals.setToolTipText(getText("ui.displayOriginalsInfo"));
					//dontStretchBehind.setToolTipText(getText("ui.dontStretchBehindInfo"));
					fitImagesBetween.setToolTipText(getText("ui.fitImagesBetweenInfo"));
					new StateMonitor() {
						public void onChange() {
							if (fitImagesBetween.getSelectedIndex() == 2 && autohideControls.isSelected()) {
								JOptionPane.showMessageDialog(window, getText("ui.fitImagesBetweenWarning"), getText("ui.warning"), JOptionPane.WARNING_MESSAGE);
							}
						}
					}.add(fitImagesBetween).done();

					add(fitImage);
					add("br", new JLabel(getText("ui.maxZoom")));
					add(" ", maxZoom);
					add(" ", new JLabel("<html><i>1.0: " + getText("ui.neverScaleUp") + "</i></html>"));
					add("br", displayOriginals);
					add("br", new JLabel(getText("ui.fitImagesBetween")));
					add("", fitImagesBetween);
					
				}
			};
			
			{
				autohideControls.setToolTipText(getText("ui.autohideControlsInfo"));
				thumbnailsVisible.setToolTipText(getText("ui.thumbnailsVisibleInfo"));
				showFitToggle.setToolTipText(getText("ui.showFitToggleInfo"));
				showStartStop.setToolTipText(getText("ui.showStartStopInfo"));
				ComponentUtilities.whenSelectedEnable(useThumbnailStrip, new JComponent[]{iconThumbstrip, thumbStripSettings});
				fitImage.setToolTipText(getText("ui.fitImagesInfo"));
				((JSpinner.DefaultEditor)maxZoom.getEditor()).getTextField().setColumns(4);
				ComponentUtilities.whenSelectedEnable(showFitToggle, new JComponent[]{iconFit, zoomSlider});
				zoomSlider.addActionListener((ActionEvent e) -> {
					if (zoomSlider.isSelected()) {
						iconFit.setIcon(icon("zoom-level"));
					} else {
						iconFit.setIcon(icon("expand"));
					}
				});
				infoPanelVisible.setToolTipText(getText("ui.infoPanelVisibleInfo"));
				ComponentUtilities.whenSelectedEnable(showStartStop, new JComponent[]{iconPlay, slideshowTimingControl});
				ComponentUtilities.whenSelectedEnable(showFullscreen, new JComponent[]{iconFullscr});
				slideshowTimingControl.setToolTipText(getText("ui.slideshowTimingControlInfo"));
					
				thumbStripSettings.setBorder(emptyBorder);
				fitImageSettings.setBorder(emptyBorder);
				
				add("tab", controlsUseText);
				add("br tab", autohideControls);
				add("br", iconThumbstrip);
				add("tab", useThumbnailStrip);
				add("br tab", thumbStripSettings);
				add("br", iconFit);
				add("tab", showFitToggle);
				add("tab", zoomSlider);
				add("br tab", fitImageSettings);
				add("br", new JLabel(icon("footer")));
				add("tab", infoPanelVisible);
				add("br", iconPlay);
				add("tab", showStartStop);
				add(" ", slideshowTimingControl);
				add("br", iconFullscr);
				add("tab", showFullscreen);

				putClientProperty("helpPage", helpRoot + "/Lightbox/Control_bar");
			}
		};

		ControlPanel mainImage = new ControlPanel() {

			@Override
			public void importVariables(Map<String, Object> vars) {
				Map<String, Object> skinVars = engine.getSkinVariables();
				Object sdo = skinVars.get("slideshowDelay");
				if (sdo instanceof Double && (Double)sdo > 0.0 && (Double)sdo < 20.0) {
					vars.put("slideshowDelay", (int)((Double)sdo * 1000));
				} else if (sdo instanceof Integer && (int)sdo < 50) {
					vars.put("slideshowDelay", Math.min(20000, (int)sdo * 1000));
				}
			}
			JComboBox transitionType = new JComboBox(new Object[] {
				new Item("crossFade", getText("ui.crossFade")),
				new Item("crossFadeAndSlide", getText("ui.crossFadeAndSlide")),
				new Item("crossFadeAndZoom", getText("ui.crossFadeAndZoom"))
			});
			JSpinner transitionSpeed = new JSpinner(new SpinnerNumberModel(800, 100, 5000, 100));
			JSpinner slideshowDelay = new JSpinner(new SpinnerNumberModel(4000, 50, 20000, 50));
			//JCheckBox slideshowLoop = new JCheckBox(getText("ui.loop"));
			JCheckBox slideshowAuto = new JCheckBox(getText("ui.autoStart"));
			JComboBox afterLast = new JComboBox(new Object[] {
				new Item("donothing", getText("ui.doNothing")), 
				new Item("startover", getText("startOver")),
				new Item("onelevelup", getText("upOneLevel")),
				new Item("backtoindex", getText("backToIndex")),
				new Item("nextfolder", getText("nextFoldersFirstImage")),
				new Item("nextindex", getText("nextIndex")),
				new Item("ask", getText("ui.ask"))
			});
			JCheckBox videoAutoPlay = new JCheckBox(getText("ui.startVideo"));
			JLabel videoAutoPlayInfo = new JLabel(infoIcon);
			JCheckBox videoLoop = new JCheckBox(getText("ui.loopVideos"));
			JCheckBox useAutoPano = new JCheckBox(getText("ui.enableAutoPano"));
			JLabel autoPanoInfo = new JLabel(infoIcon);
			JLabel autoPanoStartLabel = new JLabel(getText("ui.startAt"));
			JComboBox autoPanoStart = new JComboBox(new Object[] {
				new Item("left", getText("ui.leftOrTop")), 
				new Item("center", getText("ui.center")),
				new Item("right", getText("ui.rightOrBottom"))
			});
			JLabel autoPanoSpeedLabel = new JLabel(getText("ui.speed"));
			JComboBox autoPanoSpeed = new JComboBox(new Object[] {
				new Item("veryslow", getText("ui.verySlow")),
				new Item("slow", getText("ui.slow")), 
				new Item("moderate", getText("ui.moderate")),
				new Item("medium", getText("ui.medium")),
				new Item("quick", getText("ui.quick")),
				new Item("fast", getText("ui.fast"))
			});
			JCheckBox use360Player = new JCheckBox(getText("ui.use360Player"));
			JCheckBox autoRotate360 = new JCheckBox(getText("ui.autoRotate"));
			JSpinner rotateSpeed360 = new JSpinner(new SpinnerNumberModel(-2, -8, 8, 1));
			JLabel rpm = new JLabel(getText("ui.rpm"));
			JCheckBox useGoogleDocs = new JCheckBox(getText("ui.useGoogleDocsViewer"));
			ControlPanel autoPanoSettings = new ControlPanel() {
				{
					add("", autoPanoStartLabel);
					add("tab", autoPanoStart);
					add(" ", autoPanoSpeedLabel);
					add("tab", autoPanoSpeed);
				}
			};
			
			{
				transitionSpeed.setToolTipText(getText("ui.transitionSpeedInfo"));
				slideshowDelay.setToolTipText(getText("ui.slideshowDelayInfo"));
				afterLast.setToolTipText(getText("ui.afterLastInfo"));
				videoAutoPlayInfo.addMouseListener(new MouseAdapter() {  
					@Override
					public void mouseReleased(MouseEvent e) {
						JOptionPane.showMessageDialog(window, getText("ui.videoAutoplayInfo1"), getText("ui.warning"), JOptionPane.WARNING_MESSAGE);
				}});
				useAutoPano.setToolTipText(getText("ui.enableAutoPanoInfo"));
				useAutoPano.addItemListener(e -> {
					if (isSkinReady() && e.getStateChange() == ItemEvent.SELECTED) {
						String idim = engine.getImageSize();
						int iw = Integer.parseInt(idim.split("x")[0]);
						int ih = Integer.parseInt(idim.split("x")[1]);

						if (iw < 3000) {
							Object[] options = { 
								getText("yes"),
								getText("no")
							};
							int n = JOptionPane.showOptionDialog(window, 
								getText("ui.autoPanoInfo") + "\n\n" + getText("ui.changeImageDimensionsQuestion"), 
								getText("ui.imageDimensionsTooSmall"), 
								JOptionPane.YES_NO_OPTION, 
								JOptionPane.INFORMATION_MESSAGE,
								null,
								options,
								options[0]
							);
							if (n == 0) {
								try {
									window.ui2Engine();
									engine.setImageSize("4000x"+ih);
									JAlbum.logger.log(Level.FINE, "Image width has changed from {0}px to {1}px.", new Object[]{iw, "4000"});
									window.engine2UI();
								} catch (ParameterException ex) {
									throw new RuntimeException(ex);
								}
								useAutoPano.setSelected(true);
							}
						}
					}
				});
				autoPanoInfo.addMouseListener(new MouseAdapter() {  
					@Override
					public void mouseReleased(MouseEvent e) {
						JOptionPane.showMessageDialog(window, getText("ui.autoPanoInfo"), "Warning", JOptionPane.WARNING_MESSAGE);
				}});
				ComponentUtilities.whenSelectedEnable(useAutoPano, new JComponent[]{autoPanoSettings});
				use360Player.setToolTipText(getText("ui.use360PlayerInfo"));
				ComponentUtilities.whenSelectedEnable(use360Player, new JComponent[]{autoRotate360, rotateSpeed360, rpm});
				useGoogleDocs.setToolTipText(getText("ui.useGoogleDocsInfo"));
				
				add("", new JLabelFor(getText("ui.transition"), transitionType));
				add("tab", transitionType);
				add("br", new JLabelFor(getText("ui.transitionSpeed"), transitionSpeed));
				add("tab", transitionSpeed);
				add(new JLabel("ms"));
				add("br", new JLabel(getText("ui.slideshowDelay")));
				add("tab", slideshowDelay);
				add(new JLabel("ms"));
				add("br tab", slideshowAuto);
				add("br", new JLabelFor(getText("ui.afterLast"), afterLast));
				add("tab", afterLast);
				add("br", videoAutoPlay);
				add("", videoAutoPlayInfo);
				add("br", videoLoop);
				add("br", useAutoPano);
				add("", autoPanoInfo);
				add("br", autoPanoSettings);
				add("br", use360Player);
				add("tab", autoRotate360);
				//add("tab", new JLabelFor(getText("ui.speed"), rotateSpeed360));
				add("tab", rotateSpeed360);
				add(rpm);
				add("br", useGoogleDocs);

				putClientProperty("helpPage", helpRoot + "/Lightbox/Main_image");
			}
		};

		ControlPanel lightboxCaption = new ControlPanel() {
			
			JScrollPane imageCaptionPane = new JScrollPane(imageCaptionTemplate);

			{
				imageCaptionLink = safeLinkLabel(imageCaptionTemplate, helpRoot + "/Lightbox/Caption", getText("ui.lightboxCaption"));
				imageCaptionPresets.addItemListener(listener -> {
					if (isSkinReady()) {
						String s = getSelectedValue(imageCaptionPresets);
						if (!s.equals(CUSTOM)) {
							imageCaptionTemplate.setText(s);
						}
					}
				});	
				new StateMonitor() {
					public void onChange() {
						String s = getSelectedValue(imageCaptionPresets);
						imageCaptionPane.setVisible(s.equals(CUSTOM));
					}
				}.add(imageCaptionPresets).done();
				
				imageCaptionTemplate.setEditable(true);
				imageCaptionTemplate.setLineWrap(true);
				imageCaptionTemplate.setWrapStyleWord(true);
				imageCaptionTemplate.setFont(mono);
				imageCaptionTemplate.setTabSize(2);
				imageCaptionPane.setBorder(BorderFactory.createTitledBorder(getText("ui.captionTemplate")));					
				showImageNumbers.setToolTipText(getText("ui.showImageNumbersInfo"));
				infoPanelAdapt.setToolTipText(getText("ui.adaptToContentInfo"));

				add("", new JLabelFor(getText("ui.caption"), imageCaptionPresets));
				add("tab", imageCaptionPresets);
				add("br hfill", imageCaptionPane);
				add("br", showImageNumbers);
				add("br", infoPanelAdapt);
				
				putClientProperty("helpPage", helpRoot + "/Lightbox/Caption");
			}
		};

		ControlPanel lightboxButtons = new ControlPanel() {
			
			JCheckBox regionsVisible = new JCheckBox(getText("ui.visibleByDefault"));
			JTextField regionsBtnText = new JTextField(getText("regionsBtn"), 8);
			JCheckBox regionsSkipEmpty = new JCheckBox(getText("ui.skipEmptyTags"));
			JCheckBox downloadNonImages = new JCheckBox(getText("ui.enableOnNonImages"));
			JCheckBox downloadScaled = new JCheckBox(getText("ui.enableOnScaled"));
			JLabel regionsLabel = new JLabel(getText("ui.label"));
			JLabel mandatoryInfo = new JLabel(mandatory);
			JCheckBox useFotomoto = new JCheckBox("Fotomoto");
			JTextField fotomotoID = new JSmartTextField(20);

			ControlPanel photoData = new ControlPanel() {

				JCheckBox showPhotoDataInTheCaption = new JCheckBox(getText("ui.inTheCaption"), false);
				JCheckBox showPhotoDataLabel = new JCheckBox(getText("ui.showLabel"), true);
				JTextArea photoDataTemplate = new JSmartTextArea(12, 24);
				JScrollPane photoDataPane = new JScrollPane(photoDataTemplate);
				JButton reset = new JButton(getText("ui.resetToDefaults"));
				WrappableJLabel listMetadataHint = new WrappableJLabel("<html><i>" + getText("ui.listMetadataHint") + "</i></html>");

				{
					showPhotoDataInTheCaption.setToolTipText(getText("ui.inTheCaptionInfo"));
					ComponentUtilities.whenSelectedEnable(showPhotoDataInTheCaption, showPhotoDataLabel);				
					showPhotoDataLabel.setToolTipText(getText("ui.showLabelInfo"));
					photoDataTemplate.setEditable(true);
					photoDataTemplate.setLineWrap(true);
					photoDataTemplate.setWrapStyleWord(true);
					photoDataTemplate.setFont(mono);
					listMetadataHint.setPreferredWidth(uiWidth - 160);
					reset.addMouseListener(new MouseAdapter() {  
						@Override
						public void mouseReleased(MouseEvent e) {
							String s = new SkinModel().photoDataTemplate;
							if (s != null && s.length() > 0) {
								photoDataTemplate.setText(s);
							}
					}});				

					add(showPhotoDataInTheCaption);
					add(" ", showPhotoDataLabel);
					add(" ", reset);
					add("br hfill vfill", photoDataPane);
					add("br", listMetadataHint);
				}
			};

			{
				buttonLabelsVisible.setToolTipText(getText("ui.buttonLabelsVisibleInfo"));
				showMap.setToolTipText(getText("ui.showMapInfo"));
				showShopBtn.setToolTipText(getText("ui.showShopInfo"));
				showFeedbackBtn.setToolTipText(getText("ui.showFeedbackInfo"));
				showRegions.setToolTipText(getText("ui.showRegionsInfo"));
				regionsVisible.setToolTipText(getText("ui.regionsVisibleInfo"));
				ComponentUtilities.whenSelectedEnable(showRegions, new JComponent[]{regionsVisible, regionsLabel, regionsBtnText, regionsSkipEmpty});
				showShare.setToolTipText(getText("ui.showShareInfo"));
				printImageButton.setToolTipText(getText("ui.printImageButtonInfo"));
				downloadNonImages.setToolTipText(getText("ui.enableDownloadNonImagesInfo"));
				ComponentUtilities.whenSelectedEnable(downloadBtn, new JComponent[]{downloadNonImages, downloadScaled});
				fotomotoID.setToolTipText(getText("ui.fotomotoIDInfo"));
				mandatoryInfo.setToolTipText(getText("ui.mandatory"));
				ComponentUtilities.whenSelectedEnable(useFotomoto, new JComponent[]{fotomotoID});
				showPhotoData.setToolTipText(getText("ui.showPhotoDataInfo"));
				ComponentUtilities.whenSelectedEnable(showPhotoData, new JComponent[]{photoData});

				//setLayout(new RiverLayout(4, 5));
				add("", buttonLabelsVisible);
				add("br", new JLabel(icon("location")));
				add("tab", showMap);
				add("br", new JLabel(icon("shopping-cart")));
				add("tab", showShopBtn);
				add("br", new JLabel(icon("email-send")));
				add("tab", showFeedbackBtn);
				add("br", new JLabel(icon("facetag")));
				add("tab", showRegions);
				add("tab", regionsVisible);
				add("br tab", regionsLabel);
				add(regionsBtnText);
				add("tab", regionsSkipEmpty);
				add("br", new JLabel(icon("connect")));
				add("tab", showShare);
				add("br", new JLabel(icon("printer")));
				add("tab", printImageButton);
				add("br", new JLabel(icon("download")));
				add("tab", downloadBtn);
				add("br tab", downloadNonImages);
				add("tab", downloadScaled);
				add("br", new JLabel(icon("fotomoto")));
				add("tab", useFotomoto);
				add("br tab", new JLabelFor(getText("ui.storeId"), fotomotoID));
				add(mandatoryInfo);
				add("", fotomotoID);
				add(new JLinkLabel("https://my.fotomoto.com/signup", getText("ui.signUp")));
				add("br", new JLabel(icon("camera")));
				add("tab", showPhotoData);
				add("br hfill vfill", photoData);
				
				putClientProperty("helpPage", helpRoot + "/Lightbox/Buttons");
			}
		};
											
		/*	---------------------------------------------------------------
									Advanced
			--------------------------------------------------------------- */

		//	Custom code
		
		ControlPanel customCodePanel = new ControlPanel() {

			ControlPanel headHookTab = new ControlPanel() {

				JTextArea headHook = new JSmartTextArea(7, 20);
				JScrollPane headHookPane = new JScrollPane(headHook);

				{
					headHook.setEditable(true);
					headHook.setLineWrap(false);
					headHook.setFont(mono);
					headHook.setTabSize(2);
					headHookPane.setBorder(BorderFactory.createTitledBorder(getText("ui.headText")));

					add("hfill vfill", headHookPane);
				}
			};

			ControlPanel bodyHookTab = new ControlPanel() {

				JTextArea bodyHook = new JSmartTextArea(7, 20);
				JScrollPane bodyHookPane = new JScrollPane(bodyHook);

				{
					bodyHook.setEditable(true);
					bodyHook.setLineWrap(false);
					bodyHook.setFont(mono);
					bodyHook.setTabSize(2);
					bodyHookPane.setBorder(BorderFactory.createTitledBorder(getText("ui.bodyText")));

					add("hfill vfill", bodyHookPane);
				}
			};

			ControlPanel cssHookTab = new ControlPanel() {

				JTextArea cssHook = new JSmartTextArea(15, 20);
				JScrollPane cssHookPane = new JScrollPane(cssHook);
				WrappableJLabel info = new WrappableJLabel("<html><i>" + getText("ui.cssText") + "</i></html>");
				
				{
					cssHook.setEditable(true);
					cssHook.setLineWrap(false);
					cssHook.setFont(mono);
					cssHook.setTabSize(2);
					cssHookPane.setBorder(BorderFactory.createTitledBorder(getText("ui.cssTitle")));
					info.setPreferredWidth(700);
					
					add("hfill vfill", cssHookPane);
					add("br", info);
				}
			};

			ControlPanel jsHookTab = new ControlPanel() {

				JTextArea jsHook = new JSmartTextArea(15, 20);
				JScrollPane jsHookPane = new JScrollPane(jsHook);
				WrappableJLabel info = new WrappableJLabel("<html><i>" + getText("ui.javascriptText") + "</i></html>");
				
				{
					jsHook.setEditable(true);
					jsHook.setLineWrap(false);
					jsHook.setFont(mono);
					jsHook.setTabSize(2);
					jsHookPane.setBorder(BorderFactory.createTitledBorder(getText("ui.javascriptTitle")));
					info.setPreferredWidth(700);
					
					add("hfill vfill", jsHookPane);
					add("br", info);
				}
			};

			JTabbedPane customCodeTabs = new JTabbedPane() {				

				{
					addTab("<HEAD>", icon("code"), headHookTab);
					addTab("<BODY>", icon("code"), bodyHookTab);
					addTab("CSS", icon("css"), cssHookTab);
					addTab("JavaScript", icon("javascript"), jsHookTab);
				}
			};

			{				
				add("hfill vfill", customCodeTabs);

				putClientProperty("helpPage", helpRoot + "/Advanced/Custom_code");
			}
		};
		
		//	Site admin
		
		ControlPanel siteAdminPanel = new ControlPanel() {

			ControlPanel serverRelatedPanel = new ControlPanel(getText("ui.serverRelated")) {
				
				JTextField uploadPath = new JSmartTextField(20);
				JCheckBox useFavicon = new JCheckBox(getText("ui.useTheSkinsFavicon"), true);				
				JCheckBox useMsServer = new JCheckBox(getText("ui.useMsServer"));
				JCheckBox useExpiry = new JCheckBox(getText("ui.useExpiry"), false);
				JCheckBox useRobotsTxt = new JCheckBox(getText("ui.useRobotsTxt"), false);
				JCheckBox avoidCDNs = new JCheckBox(getText("ui.avoidCDNs"), false);
				JCheckBox copyGoogleFonts = new JCheckBox(getText("ui.copyGoogleFonts"), false);
				JCheckBox addLatinExt = new JCheckBox(getText("ui.addLatinExtended"), true);
				WrappableJLabel uploadPathInfo = new WrappableJLabel("<html><i>" + getText("ui.copyPasteAlbumURLHere") + "</i></html>");
				
				{
					uploadPath.setToolTipText(getText("ui.uploadPathInfo"));
					uploadPathInfo.setPreferredWidth(360);
					useFavicon.setToolTipText(getText("ui.useFaviconInfo"));
					useMsServer.setToolTipText(getText("ui.useMsServerInfo"));
					useExpiry.setToolTipText(getText("ui.useExpiryInfo"));
					useRobotsTxt.setToolTipText(getText("ui.useRobotsTxtInfo"));
					avoidCDNs.setToolTipText(getText("ui.avoidCDNsInfo"));
					copyGoogleFonts.setToolTipText(getText("ui.copyGoogleFontsInfo"));
					addLatinExt.setToolTipText(getText("ui.addLatinExtendedInfo"));
					
					add(new JLabel(getText("ui.uploadPath")));
					add("", uploadPath);
					add("br", new JLabel(infoIcon));
					add("", uploadPathInfo);
					add("br", useFavicon);
					add(" ", new JLabel(svgIcon("favicon-320x320", new Dimension(20, 20))));
					add("br", useMsServer);
					add("br", useExpiry);
					add("br", useRobotsTxt);
					add("br", avoidCDNs);
					add("br", copyGoogleFonts);
					add("br", addLatinExt);
				}
			};			
			
			ControlPanel imageRelatedPanel = new ControlPanel(getText("ui.images")) {
				
				JCheckBox hiDpiThemeImage = new JCheckBox(getText("ui.hiDpiThemeImage"));
				JComboBox shareImageDims = new JComboBox(new Object[] {
					"640x480",
					"1024x768",
					"1200x900",
					"1600x1200"
				});
				
				{
					shareImageDims.setEditable(true);
					
					add(hiDpiThemeImage);
					add("br", new JLabel(getText("ui.shareImageDimensions")));
					add("", shareImageDims);
				}
			};
			
			ControlPanel googlePanel = new ControlPanel(getText("ui.googleAnalytics")) {

				JComboBox googleAnalytics = new JComboBox(new Object[] {
					new Item("none", getText("ui.none")), 
					new Item("gtag", "Global Site Tag (gtag.js)"),
					new Item("universal", "Universal (analytics.js)"),
					new Item("classic", "Classic (ga.js) [Legacy]")
				});
				JTextField googleSiteID = new JSmartTextField(20);
				JLabel mandatoryInfo = new JLabel(mandatory);
				JCheckBox supportDoubleclick = new JCheckBox(getText("ui.supportDoubleclick"), false);

				{
					googleSiteID.setToolTipText(getText("ui.googleSiteIDInfo"));
					mandatoryInfo.setToolTipText(getText("ui.mandatory"));

					add("", new JLabelFor(getText("ui.version"), googleAnalytics));
					add(" ", googleAnalytics);
					add(" ", new JLinkLabel("https://www.google.com/analytics/", getText("ui.signUp")));
					add("br", new JLabelFor(getText("ui.googleSiteID"), googleSiteID));
					add(mandatoryInfo);
					add("", googleSiteID);
					add("br", supportDoubleclick);
				}
			};
			
			ControlPanel seoPanel = new ControlPanel(getText("ui.searchEngineOptimization")) {
				
				JTextField titleSEOText = new JSmartTextField(20);
				JTextField descriptionSEOText = new JSmartTextField(20);
				JSpinner preloadThumbs = new JSpinner(new SpinnerNumberModel(20, 0, 1000, 10));
				JCheckBox addAltTags = new JCheckBox(getText("ui.addAltTags"), true);
				JCheckBox slideRedirect = new JCheckBox(getText("ui.redirectSlidePages"), true);
			
				{
					titleSEOText.setToolTipText(getText("ui.titleSEOTextInfo"));
					descriptionSEOText.setToolTipText(getText("ui.descriptionSEOTextInfo"));
					//preloadThumbs.setToolTipText(getText("ui.preloadThumbsInfo"));
					addAltTags.setToolTipText(getText("ui.addAltTagsInfo"));
					writeSitemapXml.setToolTipText(getText("ui.createSitemapXmlInfo"));
					ComponentUtilities.whenSelectedEnable(writeSitemapXml, new JComponent[]{sitemapIncludeSlides});
					sitemapIncludeSlides.setToolTipText(getText("ui.sitemapIncludeSlidesInfo"));
					slideRedirect.setToolTipText(getText("ui.redirectSlidePagesInfo"));
					
					add("", new JLabelFor(getText("ui.titleSEOText"), titleSEOText));
					add("tab", titleSEOText);
					add("br", new JLabelFor(getText("ui.descriptionSEOText"), descriptionSEOText));
					add("tab", descriptionSEOText);
					add("br", new JLabel(getText("ui.preloadThumbs")));
					add("", preloadThumbs);
					add("br", addAltTags);
					add("br", writeSitemapXml);
					add("", sitemapIncludeSlides);
					add("br", slideRedirect);
				}
			};

			ControlPanel cookiePolicyPanel = new ControlPanel(getText("ui.trackingConsentAndCookiePolicy")) {

				JCheckBox askTrackingConsent = new JCheckBox(getText("ui.askTrackingConsent"), false);
				JCheckBox showCookiePolicy = new JCheckBox(getText("ui.showCookiePolicy"), false);				
				JTextField cookiePolicyStay = new JSmartTextField("8", 3);
				JTextField cookiePolicyUrl = new JSmartTextField(20);
				WrappableJLabel cookiePolicyUrlInfo = new WrappableJLabel("<html><i>" + getText("ui.cookiePolicyUrlInfo") + " </i></html>");

				{
					askTrackingConsent.setToolTipText(getText("ui.askTrackingConsentInfo"));
					showCookiePolicy.setToolTipText(getText("ui.showCookiePolicyInfo"));
					cookiePolicyUrlInfo.setPreferredWidth(320);
					
					add(askTrackingConsent);
					add("br", showCookiePolicy);
					add(" ", new JLabel(getText("ui.stay")));
					add("", cookiePolicyStay);
					add("", new JLabel("s"));
					add("br", new JLabel("URL"));
					add("tab", cookiePolicyUrl);
					add("br", new JLabel(infoIcon));
					add("tab", cookiePolicyUrlInfo);
				}
			};

			JCheckBox debugMode = new JCheckBox(getText("ui.debugMode"));

			ControlPanel leftPanel = new ControlPanel() {
				
				{
					add("hfill", serverRelatedPanel);
					add("br hfill", googlePanel);
					add("br", debugMode);					
				}
			};
			
			ControlPanel rightPanel = new ControlPanel() {
				
				{
					add("hfill", imageRelatedPanel);
					add("br hfill", seoPanel);
					add("br hfill", cookiePolicyPanel);
				}
			};
			
			{
				add(leftPanel);
				add(rightPanel);
				
				putClientProperty("helpPage", helpRoot + "/Advanced/Site_admin");
			}
		};
					
		//	Custom Keys
		
		ControlPanel customKeysPanel = new ControlPanel() {
		
			JTextArea customKeys = new JSmartTextArea(7, 20);
			JScrollPane customKeysPane = new JScrollPane(customKeys);

			{
				customKeys.setEditable(true);
				customKeys.setLineWrap(false);
				customKeys.setFont(mono);
				customKeys.setTabSize(2);
				customKeysPane.setBorder(BorderFactory.createTitledBorder(getText("ui.customKeys")));

				add("hfill vfill", customKeysPane);
				add("br", new JLabel(getText("ui.customKeysInfo")));
			}
		};
		
		/*	---------------------------------------------------------------
									Info
			--------------------------------------------------------------- */
						
		ControlPanel info = new ControlPanel() {

			{
				add("center", new JLabel(svgIcon("tiger-logo", new Dimension(144, 144), true)));
				add("br center", new JLabel("<html><h2>" + skin + "</h2></html>"));
				add("br center", new JLabel("Jalbum " + internalVersion));
				add(new JLinkLabel("https://jalbum.net/software/download", getText("ui.upgrade"), getText("ui.downloadJalbum")));
				add("br center", new JLabel(skin + " skin " + skinVer));
				add(new JLinkLabel("https://jalbum.net/skins/skin/" + skin, getText("ui.upgrade"), getText("ui.downloadSkin")));
				//add("br center", new JLabel(getText("ui.latestStableMajorVersion")));
				//add(new JLinkLabel("https://jalbum.net/web/GetSkinFile?versionId=4401&type=jaskin", "2.14.5"));
				add("br center", new JLinkLabel(helpRoot, getText("help")));
				add("br center", new JLinkLabel(supportForum, getText("ui.support")));
			}
		};
			
		/*	---------------------------------------------------------------
									Glossary
			--------------------------------------------------------------- */
						
		ControlPanel glossary = new ControlPanel() {

			private WrappableJLabel customzationDescription = new WrappableJLabel("<html><p>" + getText("ui.customizationDescription") + "</p></html>");
			private WrappableJLabel captionsDescription = new WrappableJLabel("<html><p>" + getText("ui.captionsDescription") + "</p></html>");
			private WrappableJLabel captionVariables = new WrappableJLabel("<html><p>" + getText("ui.captionVariables") + "</p></html>");
			private WrappableJLabel newImagesDescription = new WrappableJLabel("<html><p>" + getText("ui.newImagesDescription") + "</p></html>");
			private WrappableJLabel pagesDescription = new WrappableJLabel("<html><p>" + getText("ui.pagesDescription") + "</p></html>");
			private WrappableJLabel customContentDescription = new WrappableJLabel("<html><p>" + getText("ui.customContentDescription") + "</p></html>");
			private WrappableJLabel customCodeDescription = new WrappableJLabel("<html><p>" + getText("ui.customCodeDescription") + "</p></html>");
			private WrappableJLabel customKeysDescription = new WrappableJLabel("<html><p>" + getText("ui.customKeysDescription") + "</p></html>");
			private int w = uiWidth + previewWidth - 500;
			private String hr = " style=\"padding:6px;border:1px solid #888888;\"";
			{
				customzationDescription.setPreferredWidth(w);
				captionsDescription.setPreferredWidth(w);
				captionVariables.setPreferredWidth(w);
				customContentDescription.setPreferredWidth(w);
				newImagesDescription.setPreferredWidth(w);
				pagesDescription.setPreferredWidth(w);
				customCodeDescription.setPreferredWidth(w);
				customKeysDescription.setPreferredWidth(w);
				
				albumInfoLink = safeLinkLabel(albumInfo, helpRoot + "/Site/Album_info_window", getText("ui.albumInfo"));
				newImagesSectionLink = safeLinkLabel(searchNew, helpRoot + "/Sections/Search_new_images", getText("ui.searchNew"));
				markNewImagesLink = safeLinkLabel(markNew, helpRoot + "/Site/Mark_new_files", getText("ui.markNewFiles"));
				pagesLink = safeLinkLabel(showPages, helpRoot + "/Sections/Pages", getText("ui.pages"));
				customSectionLink = safeLinkLabel(customSection, helpRoot + "/Sections/Custom_content", getText("ui.customSection"));
				customCodeLink = safeLinkLabel(customCodePanel, helpRoot + "/Advanced/Custom_code", getText("ui.customCode"));
				customKeysLink = safeLinkLabel(customKeysPanel, helpRoot + "/Advanced", getText("ui.customKeys"));
					
				add(new JLabel("<html><h3" + hr + ">" + getText("ui.customization") + "</h3></html>"));
				add("br", customzationDescription);
				add("br", siteDesignLink);
				add(" ", typographyLink);
				add(" ", lightboxDesignLink);
				add("br", new JLabel("<html><h3" + hr + ">" + getText("ui.captions") + "</h3></html>"));
				add("br", captionsDescription);
				add("br", titleCaptionLink);
				add(" ", folderCaptionLink);
				add(" ", thumbCaptionLink);
				add(" ", imageCaptionLink);
				add("br", captionVariables);
				add("br", new JLabel("<html><h3" + hr + ">" + getText("ui.newImages") + "</h3></html>"));
				add("br", newImagesDescription);
				add("br", markNewImagesLink);
				add(" ", newImagesSectionLink);
				add("br", new JLabel("<html><h3" + hr + ">" + getText("ui.pages") + "</h3></html>"));
				add("br", pagesDescription);
				add("br", pagesLink);
				add(" ", topNavPagesLink);
				add("br", new JLabel("<html><h3" + hr + ">" + getText("ui.customContent") + "</h3></html>"));
				add("br", customContentDescription);
				add("br", topBarCCLink);
				add(" ", footerCCLink);
				add(" ", albumInfoLink);
				add(" ", customSectionLink);
				add("br", customCodeDescription);
				add("br", customCodeLink);
				add("br", customKeysDescription);
				add("br", customKeysLink);
			}
		};	
		
		/******************************************************************************
		 * 
		 *								GUI main tabs
		 * 
		 ******************************************************************************/
		
		/*	---------------------------------------------------------------
									Site
			--------------------------------------------------------------- */
		
		ControlPanel site = new ControlPanel(new BorderLayout(20, 0)) {
			
			JTabbedPane siteTabs = new JTabbedPane() {
				
				JScrollPane designPanel = new JScrollPane(design, JScrollPane.VERTICAL_SCROLLBAR_AS_NEEDED, JScrollPane.HORIZONTAL_SCROLLBAR_AS_NEEDED);
				JScrollPane topbarPanel = new JScrollPane(topbar, JScrollPane.VERTICAL_SCROLLBAR_AS_NEEDED, JScrollPane.HORIZONTAL_SCROLLBAR_AS_NEEDED);
				JScrollPane heroPanel = new JScrollPane(hero, JScrollPane.VERTICAL_SCROLLBAR_AS_NEEDED, JScrollPane.HORIZONTAL_SCROLLBAR_AS_NEEDED);
				
				{					
					designPanel.setBorder(emptyBorder);
					topbarPanel.setBorder(emptyBorder);
					heroPanel.setBorder(emptyBorder);
					this.setTabPlacement(SwingConstants.LEFT);
					
					addTab("<html><h4 " + tabStyle + ">" + getText("ui.design") + "</h4></html>", icon("design"), designPanel);
					addTab("<html><h4 " + tabStyle + ">" + getText("ui.typography") + "</h4></html>", icon("font"), typography);
					addTab("<html><h4 " + tabStyle + ">" + getText("ui.topBar") + "</h4></html>", icon("header"), topbarPanel);
					addTab("<html><h4 " + tabStyle + ">" + getText("ui.hero") + "</h4></html>", icon("hero"), heroPanel);
					addTab("<html><h4 " + tabStyle + ">" + getText("ui.footer") + "</h4></html>", icon("footer"), footerSettings);
					addTab("<html><h4 " + tabStyle + ">" + getText("ui.markNewFiles") + "</h4></html>", icon("new"), markNew);
					addTab("<html><h4 " + tabStyle + ">" + getText("ui.rating") + "</h4></html>", icon("star"), rating);
					addTab("<html><h4 " + tabStyle + ">" + getText("ui.download") + "</h4></html>", icon("download"), download);
					addTab("<html><h4 " + tabStyle + ">" + getText("ui.social") + "</h4></html>", icon("connect"), social);
					addTab("<html><h4 " + tabStyle + ">" + getText("ui.map") + "</h4></html>", icon("location"), mapSettings);
					addTab("<html><h4 " + tabStyle + ">" + getText("ui.audioClips") + "</h4></html>", icon("volume-up"), audioClips);
					addTab("<html><h4 " + tabStyle + ">" + getText("ui.backgroundMusic") + "</h4></html>", icon("audio"), backgroundMusic);
					addTab("<html><h4 " + tabStyle + ">" + getText("ui.albumInfo") + "</h4></html>", icon("info"), albumInfo);
				}
			};
			
			JPanel sitePreviewPanel = new JPanel(new BorderLayout(0, 0)) {

				{
					sitePreview.getView().setPreferredSize(new Dimension(previewWidth, uiHeight));

					add(sitePreview.getView(), BorderLayout.CENTER);
				}
			};
			
			{				
				((BorderLayout)(getLayout())).setVgap(0);
				((BorderLayout)(getLayout())).setHgap(0);
				siteTabs.setBorder(emptyBorder);
				
				siteTabs.setPreferredSize(new Dimension(uiWidth, uiHeight));
				siteTabs.setMaximumSize(new Dimension(uiWidth + 100, uiHeight));
				sitePreviewPanel.setBorder(BorderFactory.createLineBorder(JAlbumColor.imageBorder, 2));
				
				add(siteTabs);
				add(sitePreviewPanel, BorderLayout.EAST);				
			}
		};
			
		/*	---------------------------------------------------------------
									Sections
			--------------------------------------------------------------- */

		ControlPanel sections = new ControlPanel(new BorderLayout(20, 0)) {
			
			ControlPanel sectionOrdering = new ControlPanel() {

				JDraggableList sectionsOrder = new JDraggableList(new Object[] {
					new JLabelFor("<html><p style=\"padding:3px 0;\">" + getText("ui.pages") + "</p></html>", icon("page")).name("pages"),
					new JLabelFor("<html><p style=\"padding:3px 0;\">" + getText("ui.folders") + "</p></html>", icon("folder")).name("folders"),
					new JLabelFor("<html><p style=\"padding:3px 0;\">" + getText("ui.map") + "</p></html>", icon("location")).name("map"),
					new JLabelFor("<html><p style=\"padding:3px 0;\">" + getText("ui.filtering") + "</p></html>", icon("filter")).name("filtering"),
					new JLabelFor("<html><p style=\"padding:3px 0;\">" + getText("ui.shoppingCart") + "</p></html>", icon("shopping-cart")).name("shoppingCart"),
					new JLabelFor("<html><p style=\"padding:3px 0;\">" + getText("ui.feedback") + "</p></html>", icon("email")).name("feedback"),
					new JLabelFor("<html><p style=\"padding:3px 0;\">" + getText("ui.images") + "</p></html>", icon("thumbnails")).name("images"),
					new JLabelFor("<html><p style=\"padding:3px 0;\">" + getText("ui.tags") + "</p></html>", icon("tag")).name("tags"),
					new JLabelFor("<html><p style=\"padding:3px 0;\">" + getText("ui.searchNew") + "</p></html>", icon("new")).name("newImages"),
					new JLabelFor("<html><p style=\"padding:3px 0;\">" + getText("ui.neighboringFolders") + "</p></html>", icon("prev-next-page")).name("neighboringFolders"),
					new JLabelFor("<html><p style=\"padding:3px 0;\">" + getText("ui.commenting") + "</p></html>", icon("chat")).name("comments"),
					new JLabelFor("<html><p style=\"padding:3px 0;\">" + getText("ui.customContent") + "</p></html>", icon("edit-document")).name("customContent")
				});

				{						
					add("center", new JLabel("<html><h4>" + getText("ui.ordering")+ "</h4></html>"));
					add("br", sectionsOrder);
					add("br center", new JLabel("<html><i>" + getText("ui.dragToReorder") + "</i></html>"));
				}
			};
				
			JTabbedPane sectionTabs = new JTabbedPane() {

				JScrollPane shoppingCartPanel = new JScrollPane(shoppingCart, JScrollPane.VERTICAL_SCROLLBAR_AS_NEEDED, JScrollPane.HORIZONTAL_SCROLLBAR_NEVER);
				JScrollPane filteringAndSortPanel = new JScrollPane(filteringAndSort, JScrollPane.VERTICAL_SCROLLBAR_AS_NEEDED, JScrollPane.HORIZONTAL_SCROLLBAR_NEVER);

				{
					this.setTabPlacement(SwingConstants.LEFT);

					shoppingCartPanel.setBorder(emptyBorder);
					filteringAndSortPanel.setBorder(emptyBorder);

					addTab("<html><h4 " + tabStyle + ">" + getText("ui.folders") + "</h4></html>", icon("folder"), folders);
					addTab("<html><h4 " + tabStyle + ">" + getText("ui.pages") + "</h4></html>", icon("page"), pages);
					addTab("<html><h4 " + tabStyle + ">" + getText("ui.images") + "</h4></html>", icon("thumbnails"), images);
					addTab("<html><h4 " + tabStyle + ">" + getText("ui.map") + "</h4></html>", icon("location"), map);
					addTab("<html><h4 " + tabStyle + ">" + getText("ui.filteringAndSort") + "</h4></html>", icon("filter"), filteringAndSortPanel);
					addTab("<html><h4 " + tabStyle + ">" + getText("ui.shoppingCart") + "</h4></html>", icon("shopping-cart"), shoppingCartPanel);
					addTab("<html><h4 " + tabStyle + ">" + getText("ui.feedback") + "</h4></html>", icon("email-send"), feedback);
					addTab("<html><h4 " + tabStyle + ">" + getText("ui.tagCloud") + "</h4></html>", icon("tag"), tagCloud);
					addTab("<html><h4 " + tabStyle + ">" + getText("ui.searchNew") + "</h4></html>", icon("new"), searchNew);
					addTab("<html><h4 " + tabStyle + ">" + getText("ui.neighboringFolders") + "</h4></html>", icon("prev-next-page"), neighboringFolders);
					addTab("<html><h4 " + tabStyle + ">" + getText("ui.commenting") + "</h4></html>", icon("chat"), commenting);
					addTab("<html><h4 " + tabStyle + ">" + getText("ui.customContent") + "</h4></html>", icon("edit-document"), customSection);

				}
			};
				
			{
				((BorderLayout)(getLayout())).setVgap(0);
				((BorderLayout)(getLayout())).setHgap(0);
				
				sectionTabs.setBorder(emptyBorder);
				
				add(sectionTabs);
				add(sectionOrdering, BorderLayout.EAST);				
			}
		};
						

		/*	---------------------------------------------------------------
									Lightbox
			--------------------------------------------------------------- */
		
		ControlPanel lightbox = new ControlPanel(new BorderLayout(0, 0)) {

			JTabbedPane lightboxTabs = new JTabbedPane() {
				
				{	
					this.setTabPlacement(SwingConstants.LEFT);
										
					addTab("<html><h4 " + tabStyle + ">" + getText("ui.settings") + "</h4></html>", icon("settings"), lightboxDesign);
					addTab("<html><h4 " + tabStyle + ">" + getText("ui.controlBar") + "</h4></html>", icon("embed"), lightboxControls);
					addTab("<html><h4 " + tabStyle + ">" + getText("ui.mainImage") + "</h4></html>", icon("image"), mainImage);
					addTab("<html><h4 " + tabStyle + ">" + getText("ui.caption") + "</h4></html>", icon("text"), lightboxCaption);
					addTab("<html><h4 " + tabStyle + ">" + getText("ui.buttons") + "</h4></html>", icon("button"), lightboxButtons);
				}
			};
			
			ControlPanel lightboxPreviewPanel = new ControlPanel(new BorderLayout(0, 0)) {
				
				JPanel lbPreview = new JPanel(new BorderLayout(0, 0)) {
					
					{
						add(lightboxPreview.getView());
					}
				};
						
				{
					lbPreview.setPreferredSize(new Dimension(previewWidth, (int)Math.round(previewWidth * .8)));
					lbPreview.setBorder(BorderFactory.createLineBorder(JAlbumColor.imageBorder, 2));
					//lightboxPreview.getView().setMaximumSize(new Dimension(previewWidth, previewWidth));
					
					add(lbPreview, BorderLayout.NORTH);
				}
			};
			
			{
				((BorderLayout)(getLayout())).setVgap(0);
				((BorderLayout)(getLayout())).setHgap(0);
				
				lightboxTabs.setBorder(emptyBorder);
				lightboxTabs.setPreferredSize(new Dimension(uiWidth, uiHeight));
				lightboxTabs.setMaximumSize(new Dimension(uiWidth + 100, uiHeight));
				
				add(lightboxTabs);
				add(lightboxPreviewPanel, BorderLayout.EAST);				
			}
		};
		
		/*	---------------------------------------------------------------
									Advanced
			--------------------------------------------------------------- */
		
		ControlPanel advanced = new ControlPanel(new BorderLayout(0, 0)) {
			
			JTabbedPane advancedTabs = new JTabbedPane() {
								
				JScrollPane siteAdminPane = new JScrollPane(siteAdminPanel, JScrollPane.VERTICAL_SCROLLBAR_AS_NEEDED, JScrollPane.HORIZONTAL_SCROLLBAR_NEVER);
				
				{
					//customContentPanel.setBorder(emptyBorder);
					siteAdminPane.setBorder(emptyBorder);
					customCodePanel.setBorder(emptyBorder);
					customKeysPanel.setBorder(emptyBorder);
					this.setTabPlacement(SwingConstants.LEFT);
					
					addTab("<html><h4 " + tabStyle + ">" + getText("ui.siteAdmin") + "</h4></html>", icon("wrench"), siteAdminPane);
					addTab("<html><h4 " + tabStyle + ">" + getText("ui.customCode") + "</h4></html>", icon("code"), customCodePanel);
					addTab("<html><h4 " + tabStyle + ">" + getText("ui.customKeys") + "</h4></html>", icon("rename"), customKeysPanel);
				}
			};
			
			{
				((BorderLayout)(getLayout())).setVgap(0);
				((BorderLayout)(getLayout())).setHgap(0);
				
				advancedTabs.setBorder(emptyBorder);
				
				add(advancedTabs);
			}
		};
		
		/*	---------------------------------------------------------------
									Help
			--------------------------------------------------------------- */

		ControlPanel help = new ControlPanel(new BorderLayout(20, 0)) {
			
			private JTextArea readme = new JSmartTextArea( getFileContents("readme.txt"), 20, 30 );
			private JScrollPane readmePane = new JScrollPane(readme, JScrollPane.VERTICAL_SCROLLBAR_ALWAYS, JScrollPane.HORIZONTAL_SCROLLBAR_NEVER);
			private JScrollPane glossaryPane = new JScrollPane(glossary, JScrollPane.VERTICAL_SCROLLBAR_ALWAYS, JScrollPane.HORIZONTAL_SCROLLBAR_NEVER);
							
			JTabbedPane helpTabs = new JTabbedPane() {
			
				{
					readme.setLineWrap(true);
					readme.setWrapStyleWord(true);
					readme.setEditable(false);
					readme.setFont(mono);
					readmePane.setBorder(emptyBorder);
					glossaryPane.setBorder(emptyBorder);
					glossaryPane.setPreferredSize(new Dimension(uiWidth + previewWidth - 480, uiHeight));
					glossaryPane.setMaximumSize(new Dimension(uiWidth + previewWidth - 480, uiHeight));
					this.setTabPlacement(SwingConstants.LEFT);
					
					addTab("<html><h4 " + tabStyle + ">" + getText("ui.glossary") + "</h4></html>", icon("book-help", 27), glossaryPane);
					addTab("<html><h4 " + tabStyle + ">" + getText("ui.releaseNotes") + "</h4></html>", icon("list", 27), readmePane);
				}
			};
				
			{
				((BorderLayout)(getLayout())).setVgap(0);
				((BorderLayout)(getLayout())).setHgap(0);
				
				info.setPreferredSize(new Dimension(240, uiHeight));
				helpTabs.setMaximumSize(new Dimension(uiWidth + previewWidth - 480, uiHeight));
				
				add(info, BorderLayout.EAST);
				add(helpTabs);
				
			}
		};
		
		/*	---------------------------------------------------------------
									Main tabs
			--------------------------------------------------------------- */
		
		JTabbedPane tabs = new JTabbedPane() {
			
			{		
				site.setBorder(emptyBorder);
				sections.setBorder(emptyBorder);
				lightbox.setBorder(emptyBorder);
				advanced.setBorder(emptyBorder);
				glossary.setBorder(emptyBorder);
				help.setBorder(emptyBorder);
								
				addTab("<html><h4 style='padding:4px 6px;margin:4px;'>" + getText("ui.site") + "</h4></html>", icon("site", 27), site);
				addTab("<html><h4 style='padding:4px 6px;margin:4px;'>" + getText("ui.sections") + "</h4></html>", icon("menu", 27), sections);
				addTab("<html><h4 style='padding:4px 6px;margin:4px;'>" + getText("ui.lightbox") + "</h4></html>", icon("lightbox", 27), lightbox);
				addTab("<html><h4 style='padding:4px 6px;margin:4px;'>" + getText("ui.advanced") + "</h4></html>", icon("wrench", 27), advanced);
				addTab("<html><h4 style='padding:4px 6px;margin:4px;'>" + getText("ui.help") + "</h4></html>", icon("info", 27), help);
			}
		};
		
		private final String indexTemplate = getFileContents("lib/index-template.html");
		private final String lightboxTemplate = getFileContents("lib/lightbox-template.html");
	
		{	
			// Adding UI tabs
			
			((RiverLayout)(getLayout())).setVgap(0);
			((RiverLayout)(getLayout())).setHgap(0);
			tabs.setBorder(emptyBorder);
			
			add(tabs);
			
			/*	-----------------------------------------------------------
										Listeners
				----------------------------------------------------------- */

			window.addJAlbumListener(new JAlbumAdapter() {
				@Override
				public void styleChanged(JAlbumEvent e) {
					styleName.setText(getStyleName());
				}
			});

			window.addPropertyChangeListener(JAlbumFrame.SKIN_LOADED_PROPERTY, setupMonitors);

			window.addJAlbumListener(new JAlbumAdapter() {
				@Override
				public void skinChanged(JAlbumEvent e) {
					//JAlbum.logger.log(Level.FINE, "Skin changed, removing listeners.");
					window.removePropertyChangeListener(JAlbumFrame.SKIN_LOADED_PROPERTY, setupMonitors);
				}
			});
			
			try {
				window.albumExplorer.onContextMenu(menu -> {
					AlbumObject[] sel = window.albumExplorer.explorer.getSelectedAlbumObjects();

					if (sel.length > 0) {
						
						//  Hero slider selector
						createMenuItemForBooleanProperty(menu, sel, "useInSlider", getText("ui.useInHeroSlider"));
						
						//  Panorama selector
						createMenuItemForBooleanProperty(menu, sel, "panorama", getText("ui.panorama"));
						
						//  Hide location selector
						createMenuItemForBooleanProperty(menu, sel, "hideLocation", getText("ui.hideLocation"));
					}
				});
			} catch (Throwable ex) {
				JAlbum.logger.log(Level.INFO, "Right-click extension menu is available only in jAlbum v32 and newer!");
			};
			
			Map<String,Object> vars = engine.getSkinVariables();
			//System.out.println("titleCaptionTemplate = \"" + vars.get("titleCaptionTemplate") + "\"");
			selectComboBoxItem(titleCaptionPresets, (String)vars.get("titleCaptionTemplate"));
			//System.out.println("folderCaptionTemplate = \"" + vars.get("folderCaptionTemplate") + "\"");
			selectComboBoxItem(folderCaptionPresets, (String)vars.get("folderCaptionTemplate"));
			//System.out.println("thumbCaptionTemplate = \"" + vars.get("thumbCaptionTemplate") + "\"");
			selectComboBoxItem(thumbCaptionPresets, (String)vars.get("thumbCaptionTemplate"));
			//System.out.println("imageCaptionTemplate = \"" + vars.get("imageCaptionTemplate") + "\"");
			selectComboBoxItem(imageCaptionPresets, (String)vars.get("imageCaptionTemplate"));
			
			putClientProperty("helpPage", helpRoot);
		}
		
	};

	public Gui() {
		this(JAlbumContext.getInstance());
	}
   
	public Gui(JAlbumContext context) {
		
		super(context);
		PluginContext pc = context.getPluginContext();
		EditPanel editPanel = pc.getEditPanel();
		
		editPanel.addCustomTab(getText("ui.imageData"), new ImageDataUI(context, texts));
		editPanel.addCustomTab(getText("ui.external"), new ExternalContentUI(context, texts));

		skinUi.setBorder(emptyBorder);
		((RiverLayout)(getLayout())).setVgap(0);
		((RiverLayout)(getLayout())).setHgap(0);
		
		window.setSkinUI(skinUi);		
		skinReadyAt = System.currentTimeMillis();
		
	}
	
}
