/* 
 *	main.js - the skin's javascript functions
 */
	
;(function($, $window, $document, $body, undefined) {
	'use strict';
				
	/********************************************************
	 *
	 *		Implementing select functions on images
	 *
	 ********************************************************/
	
	$.fn.selectable = function(options) {
		
			options = $.extend({
								cardClass:				'card',
								thumbClass:				'thumb',
								checkboxClass: 			'checkbox',
								selectedClass:			'checked',
								checkmarkClass:			'icon-checkmark',
								hasSelectedClass:		'has-selected',
								allSelectedClass:		'all-selected',
								selectionChange:		null
							}, options);
			
			var overlay = $(this).eq(0),
				cards,
				
				update = function(n) {
					
						if (typeof options.selectionChange === FUNCTION) {
							options.selectionChange.call(overlay[0], n);
						}
						
						$body
							.toggleClass(options.hasSelectedClass, n > 0)
							.toggleClass(options.allSelectedClass, n === cards.length);
					},
					
				selectedCnt = function() {
						return cards.filter('.' + options.selectedClass).length;
					},
			
				getCards = function() {
						return options.selector? overlay.find(options.selector) : overlay.find('.' + options.cardClass);
					},
			
				selectCard = function(e) {
						var card = (e instanceof $)? e : $(this);
						
						card.addClass(options.selectedClass)
							.find('.' + options.checkboxClass)
							.addClass(options.checkmarkClass);
					},
					
				unselectCard = function(e) {
						var card = (e instanceof $)? e : $(this);
						
						card.removeClass(options.selectedClass)
							.find('.' + options.checkboxClass)
							.removeClass(options.checkmarkClass);
					},
					
				toggleCard = function(e) {
						var card = (e instanceof $)? e : $(this);
						
						if (card.hasClass(options.selectedClass)) {
							unselectCard(card);
						} else {
							selectCard(card);
						}
					},
			
				clicked = function(e) {
						var card = $(e.target).closest('.' + options.cardClass),
							n = selectedCnt();
						
						if (card.hasClass(options.selectedClass)) {
							unselectCard(card);
							n -= 1;
							
							if (typeof options.selectionChange === FUNCTION) {
								options.selectionChange.call(overlay[0], n);
							}
							
						} else {
							
							selectCard(card);
							n += 1;
							
							if (typeof options.selectionChange === FUNCTION) {
								options.selectionChange.call(overlay[0], n);
							}
						}	
							
						update(n);
						
						return false;
					},
			
				selectAll = function() {
					
						if (overlay.is(':visible')) {
							var n = selectedCnt();
							
							if (n < cards.length) {
								n = cards.length;
								cards.not('.' + options.selectedClass).each(function() {
										selectCard($(this));
									});
										
								if (typeof options.selectionChange === FUNCTION) {
									options.selectionChange.call(overlay[0], n);
								}
							}
							
							update(n);
						}
					},
			
				selectNone = function() {
					
						if (overlay.is(':visible')) {
							var n = selectedCnt();
							
							if (n > 0) {
								n = 0;
								cards.filter('.' + options.selectedClass).each(function() {
										unselectCard($(this));
									});
										
								if (typeof options.selectionChange === FUNCTION) {
									options.selectionChange.call(overlay[0], n);
								}
							}
							
							update(n);
						}
					},
					
				selectInverse = function() {
					
						if (overlay.is(':visible')) {
							var n = cards.length - selectedCnt();
							
							if (n === 0) {
								// All selected
								cards.each(function() {
										unselectCard($(this));
									});
								
								if (typeof options.selectionChange === FUNCTION) {
									options.selectionChange.call(overlay[0], 0);
								}
								
							} else {
								// Some selected
								cards.each(function() {
										toggleCard($(this));
									});
								
								if (typeof options.selectionChange === FUNCTION) {
									options.selectionChange.call(overlay[0], n);
								}
							} 
							
							update(n);
						}
					},
				
				selectRefresh = function() {
						var th, 
							cb;
							
						cards = getCards();
						
						cards.each(function() {
								
								if (!($(this).find('span.' + options.checkboxClass).length)) {
									
									th = $(this).children('.' + options.thumbClass).eq(0);
									
									if (th.length) {
										cb = $('<span>', {
												'class': 	options.checkboxClass
											}).appendTo(th);
									} else {
										cb = $('<span>', {
												'class': 	options.checkboxClass
											}).prependTo($(this));
									}
									
									cb.on({
											click: 			clicked
										});
									
									$(this).on({
											selectCard:		selectCard,
											unselectCard:	unselectCard,
											toggleCard:		toggleCard
										});
								}
								
							});
						
						if (typeof options.selectionChange === FUNCTION) {
							options.selectionChange.call(overlay[0], selectedCnt());
						}
						
					};
			
			selectRefresh();
			
			overlay.on({
					selectAll: 			selectAll,
					selectNone: 		selectNone,
					selectInverse: 		selectInverse,
					selectRefresh: 		selectRefresh
				});
					
			return this;
		};
			
		
	/********************************************************
	 *
	 *					The main skin code
	 *
	 ********************************************************/
	
	$.fn.skin = function(settings) {
		
			settings = $.extend({}, $.fn.skin.defaults, settings);
			
			var main					= 	$(this),								// Main SECTION
				content					=	main.find('#main-cont'),				// Main container
				text 					= 	getTranslations({						// Translated texts
													'return':						"return",
													addCart:						"Add to Cart",
													fotomotoBtn:					"Buy / Share",
													fotomotoTooltip:				"Buy prints or digital files, share, send free eCards",
													audioClipLabel:					"Play / pause the attached audio clip",
													selectedItems:					"Selected items",
													label:							"Label",
													search:							"Search",
													searchNew:						"Search New",
													foundNTimes:					"{0} image(s)",
													notFound:						"not found",
													searchPlaceholder:				"Search...",
													more:							"more",
													less:							"less",
													imagesTaken:					"Images taken",
													imagesModified:					"Images modified",
													imagesAdded:					"Images added",
													betweenDays:					"between {0} and {1}",
													onDay:							"on {0}",
													afterDay:						"after {0}",
													beforeDay:						"before {0}",
													sinceMyLastVisit:				"since my last visit",
													results:						"Results",
													sortedBy:						"Sorted by",
													ascending:						"Ascending",
													descending:						"Descending",
													noRating:						"No rating",
													multipleSelectHint:				"Use SHIFT to select range, CTRL for multiple entries",
													select:							"Select",
													reset:							"Reset",
													sortBy:							"Sort by",
													image:							"image",
													images:							"images",
													audio:							"audio",
													audios:							"audios",
													video:							"video",
													videos:							"videos",
													other:							"other",
													others:							"others",
													fatalError:						"Fatal error!"
												}),
			
				shopRoot 				=	$(),										// Root element for cart
				shopBox 				= 	$(),										// Whole Shopping section
				feedbackRoot 			=	$(),										// Root element for the feedback tool
				feedbackBox 			=	$(),										// Whole Feedback section
				mapRoot,																// Root element for map
				album,																	// The database
				ns 						= 	'tiger_skin',								// Namespace
				lastHash 				= 	'',											// Storing last state
				hashChangedByJs 		= 	false,										// Tracking for internally or externally triggered hash changes
				pageExt					= 	settings.indexName.split('.')[1] || 'html',	// Index page extension
				initByHash 				= 	window.location.hash !== '',				// Initial URL contains hash
				nows 					=	new Date() / 1000,							// Now in seconds
				currency 				=	'EUR',
				scrollTop 				= 	$window.scrollTop(),						// initial scroll top
				lazyloadFoldersTimeout 	= 	null,										// Lazyload tracking
				lazyloadFoldersInProgress 	= 	false,									// lock to avoid multiple calls
				
				// Determine card type
				
				cardType = function(card) {
						var s = card.closest('section');
						if (!s.length) {
							return 'image';
						} else if (s.hasClass(settings.folderClass)) {
							return 'folder';
						} else if (s.hasClass(settings.thumbnailsClass)) {
							return 'thumb';
						}
						
						return '';
					},
												
				/********************************************************
				 *
				 *					Managing overlays
				 *				e.g. after search or filter
				 *
				 ********************************************************/
				/*
				
					<section id="main" class="content"> => main
						
						// Base
						<article id="main-cont"> => content
							<section class="thumbnails">
								<div class="cont horizontal caption-over hover-zoom caption-on-hover">
								</div>
							</section>
							<section id="tag-cloud" class="row cont-box"><div class="cont box"><h4 class="icon-label"> Labels</h4><div class="tag-cloud"></div><form class="tag-search" data-depth="tree" data-fields="creator,keywords,folder:title,folder:name,webLocation:title,webPage:title" wtx-context="94581790-1AAD-4D31-91BA-67ADA9D1EC6F"><div class="search-input"><input type="text" placeholder="Search..." aria-label="Search" wtx-context="46A4B41B-684F-4DFB-8E45-6D8FAC4C5691"></div><a class="search-btn icon-search"></a></form></div></section>
							<section class="prev-next-links"><div class="cont"><div class="next-link has-bg" style="background-image:url(../People/thumbs/Newborn.jpg)"><a rel="next" href="../People/index.html"><span class="caption">People <span class="icon-caret-right"></span></span></a></div></div></section>
						</article>
						
						// Base + overlay
						<article id="main-cont" class="has-overlay"> => content
							<section class="thumbnails base">
								<div class="cont horizontal caption-over hover-zoom caption-on-hover">
								</div>
							</section>
							<section class="thumbnails overlay" data-overlay="oly_NNNN">
								<header class="cont overlay-title icon-filter">
									<h4>Title</h4>
									<a class="close">Return</a>
								</header>
								<div class="cont horizontal caption-over hover-zoom caption-on-hover">
								</div>
							</section>
							<section id="tag-cloud" class="row cont-box"><div class="cont box"><h4 class="icon-label"> Labels</h4><div class="tag-cloud"></div><form class="tag-search" data-depth="tree" data-fields="creator,keywords,folder:title,folder:name,webLocation:title,webPage:title" wtx-context="94581790-1AAD-4D31-91BA-67ADA9D1EC6F"><div class="search-input"><input type="text" placeholder="Search..." aria-label="Search" wtx-context="46A4B41B-684F-4DFB-8E45-6D8FAC4C5691"></div><a class="search-btn icon-search"></a></form></div></section>
							<section class="prev-next-links"><div class="cont"><div class="next-link has-bg" style="background-image:url(../People/thumbs/Newborn.jpg)"><a rel="next" href="../People/index.html"><span class="caption">People <span class="icon-caret-right"></span></span></a></div></div></section>
						</article>
						
					</section>
				*/
				/*
					<article class="overlay">
						<section class="thumbnails">
							<div class="cont horizontal caption-over hover-zoom caption-on-hover">
								<div class="card preload hascaption">
									<a class="thumb" href="slides/Dusk.jpg">
										<img src="thumbs/Dusk.jpg" width="244" height="170">
									</a>
									<div class="caption">
										<h6>Title</h6>
										<div class="comment">Comment</div>
									</div>
								</div>
							</div>
						</section>
					</article>
				*/
				
				OLYTAG = 'section',
				
				// Content has overlay?
				
				hasOverlay = function() {
					
						return content.find('.' + settings.overlayClass).length > 0;
					},
					
				// Getting NameSpace, creating new if not exists
				
				getOverlayNS = function() {
						var oly = getOverlay();
						
						if (oly.length) {
							var ns = oly.data('overlay');
						
							if (ns) {
								return ns;
							}
						
							ns = oly.hasClass(settings.baseOverlayClass)? 'base' : getUniqueId();
							
							oly.attr('data-overlay', ns);
					
							return ns;
						}
							
						return  '';
					},
					
				// Getting unique ID for overlays
				
				getUniqueId = function() {
						return 'oly_' + Math.floor(Math.random() * 10000);
					},
					
				// Setting NameSpace
				
				setOverlayNS = function(ns, oly) {
						var oly = (typeof oly === UNDEF)? getOverlay() : oly,
							ns = (typeof ns === UNDEF)? (oly.hasClass(settings.baseOverlayClass)? 'base' : getUniqueId()) : ns;
						
						oly.attr('data-overlay', ns);
						
						return ns;
					},
					
				// Getting active overlay; creating one if missing
				
				getOverlay = function(create) {
						var oly = content.find('.' + settings.thumbnailsClass).last();
						
						if (!oly.length && typeof create !== UNDEF && create) {
							oly = createOverlay();
						}
						
						return oly;
					},
					
				// Getting base overlay
				
				getBaseOverlay = function() {
				
						return content.find('.' + settings.thumbnailsClass).eq(0);
					},
					
				// Scroll to overlay
					
				focusOverlay = function() {
						var oly = getOverlay();
						
						if (oly.length && oly.offset().top < $window.scrollTop() || oly.offset().top > ($window.scrollTop() + $window.height() - 100)) {
							$window.scrollTop(oly.offset().top);
						}
					},
					
				// Creating a new overlay
				
				createOverlay = function() {
						var ns = getUniqueId(),
							oly = $('<' + OLYTAG + '>', {
										'class':		settings.thumbnailsClass + ' ' + settings.overlayClass
									}).attr({
										id:				ns,
										'data-overlay':	ns
									}),
							boly = getBaseOverlay();									// Base overlay
							
						if (boly.length) {
							// Has base: create after it
							boly.after(oly);
						} else {
							// No base: create after the folders if exists
							boly = content.find('.' + settings.foldersClass);
							if (boly.length) {
								boly.after(oly);
							} else {
								// No folders: create at first position
								content.prepend(oly);
							}
						}
						
						createImageContainer(oly);
						
						content.addClass(settings.hasOverlayClass);
						
						return oly;
					},
					
				// Creating header
				
				createOverlayHeader = function(options, oly) {
						var h = $('<header>', {
									 'class':	[	settings.contClass,
									 	 			settings.overlayTitleClass,
									 	 			(options && options.hasOwnProperty('icon'))? options.icon : ''
									 	 		].filter(Boolean).join(' ')
								}).prependTo(oly || getOverlay());
						
						if (options && options.hasOwnProperty('title')) {
							h.append($('<h' + settings.overlayHeadLevel + '>', {
									html: 	options.title
								}));
						}			
								
						h.append($('<a>', {
									'class': 	'close',
									href: 		'',
									text: 		text.return
								})
							.on('click', function() {
									removeOverlay();
									return false;
								}));
						
						return h;
					},
					
				// Getting overlay header
				
				getOverlayHeader = function() {
						
						return getOverlay().children('header').eq(0);
					},
					
				// Getting overlay title
				
				getOverlayTitle = function() {
						
						return getOverlayHeader().find('h' + settings.overlayHeadLevel).eq(0);
					},
					
				// Append text to overlay title
				
				appendToOverlayTitle = function(cont) {
						var t = getOverlayTitle();
						
						return t.length? cont.appendTo(t) : $();
					},
					
				// Adding progressbar
				
				appendProgressbar = function() {
					
						return $('<div>', {
									'class': 	settings.progressbarClass
								}).insertAfter(getOverlayHeader());
					},
					
				// Remove progressbar
				
				removeProgressbar = function(target) {
					
						getOverlay().find('.' + settings.progressbarClass).remove();
					},

				// Associate elements with the current overlay
				
				attachToOverlay = function(el) {
					
						if (el.length) {
							var ns = getOverlayNS();
							
							el.each(function() {
									$(this).attr('data-rel', ns);
								});
						}
					},
					
				// Remove overlay-attached elements	
					
				removeAttachedElementsFromOverlay = function(ns) {
						var ns = ns || getOverlayNS();
						
						if (ns) {
							content.find('[data-rel=' + ns + ']').remove();
						}
					},
				
				// Creating image container within target overlay
				
				createImageContainer = function(oly) {
						var oly = oly || getOverlay();
							
						return $('<div>', {
									'class': 	[
													settings.contClass,
													settings.thumbLayout, 
													'caption-' + settings.captionPlacement,
													(settings.captionPlacement !== 'below' && settings.captionShowOnHover)? 'caption-on-hover' : ''
												].filter(Boolean).join(' ')
									}).appendTo(oly);
					},
					
				// Getting base thumbnail container 
				
				getBaseImageContainer = function() {
				
						return content
								.find('.' + settings.thumbnailsClass + 
									':not(.' + settings.overlayClass + ')')
								.first()
								.children('div.' + settings.contClass);
					},
					
				// Returns the current thumbnail container if exists
				
				getImageContainer = function(create) {
						var ic = content
									.find('.' + settings.thumbnailsClass)
									.last()
									.children('div.' + settings.contClass);
						
						if (!ic.length && create) {
							ic = createImageContainer(getOverlay(true));
						}
						
						return ic; 
					},
					
				// Restoring base overlay's attached elements
				
				restoreBaseOverlay = function() {
						var bo = getBaseOverlay();
						
						if (bo.length) {
							// Attached elements
							$('[data-rel=' + bo.data('overlay') + ']').not('.hidden').show();
							// Refreshing thumbnails
							bo.trigger('refresh');
						}
					},
					
				// Getting all base cards
				
				getBaseCards = function(selector) {
						
						return getBaseImageContainer()
								.find('.' + settings.cardClass + (selector || ''));
					},
					
				// Returns the current cards (base or overlay)
				
				getCards = function(selector) {
					
						return getImageContainer()
								.find('.' + settings.cardClass + (selector || ''));
					},
					
				// Returns base thumbnails
				
				getBaseImages = function(selector) {
					
						return getBaseImageContainer()
								.find('.' + settings.cardClass + 
									' .' +	settings.thumbClass + 
									(selector || ''));
					},
					
				// Returns the current thumbnails (base or overlay)
				
				getImages = function(selector) {
					
						return getImageContainer()
								.find('.' + settings.cardClass + 
									' .' + settings.thumbClass + 
									(selector || ''));
					},
					
				// Returns all videos (base or overlay)
				
				getVideos = function(selector) {
					
						return getOverlay()
								.find('.' + settings.cardClass + ' video' + (selector || ''));
					},
					
				// Returns the current cards (base or overlay)
				
				getSelectedCards = function() {
					
						return getImageContainer()
								.find('.' + settings.cardClass + 
									'.' + settings.selectedClass);
					},
					
				// Getting current card
				
				getActiveCard = function(o) {
						
						return getImageContainer()
								.find('.' + settings.cardClass + 
									'.' + settings.activeClass);
					},
					
				// Returns the current cards (base or overlay)
				
				getSelectableCards = function() {
					
						return getImageContainer()
								.find('.' + settings.cardClass + 
									'.' + settings.selectableClass);
					},
					
				// Removing overlay
				
				removeOverlay = function() {
						var oly = getOverlay();
						
						if (oly.length && oly.hasClass(settings.overlayClass)) {
							var ns = oly.data('overlay');
							
							$window.off('.' + ns);
							
							oly.trigger('removeLightbox');
							
							removeAttachedElementsFromOverlay(ns);
							
							oly.trigger('overlayRemoved');
							
							oly.remove();
							
							content.removeClass(settings.hasOverlayClass);
							
							restoreBaseOverlay();
							
							if (settings.hasOwnProperty('search')) {
								emptySearchBox();
							}
							
							setHash();
							
							updateShares();
							
							if (!getSelectableCards().length) {
								$('.hide-on-empty').hide();
							}
							
							content.hideAllTooltips();
							content.find('[data-sticky-rel]').trigger('stickyRefresh');
						}
					},
					
				// Adding new overlay
			
				addOverlay = function(options) {
						
						// Removing existing overlay first
						if (hasOverlay()) {
							removeOverlay();
						}
						
						content.hideAllTooltips();
						
						// Creating overlay (section.thumbnails)
						var oly = createOverlay();
						
						// Creating header
						createOverlayHeader(options, oly);
							
						// Triggering ready event
						content.trigger('overlayReady', oly);
						
						content.find('[data-sticky-rel]').trigger('stickyRefresh');
						
						return oly;
					},
				
				// Getting file name from thumbnail
				
				getImageFileName = function(t) {
						var l = t.data('href') || t.attr('href');
								
						return l? l.substring(l.lastIndexOf('/') + 1).replace('#img=', '') : '';
					},
				
				// Checking if file is a custom page and matches base name: "name"
				
				checkPageName = function(fn, name) {
						
						if (fn.slice(-pageExt.length) === pageExt) {
							return fn.substring(0, fn.lastIndexOf('.')) === name.substring(0, name.lastIndexOf('.'));
						}
						
						return false;
					},
					
				// Getting image by file name, name is mandatory
				
				getCardByName = function(name, base) {
						var cards,
							t = $();
						
						cards = (typeof base === UNDEF || !base)?
							getCards()
							:
							getBaseCards();
						
						name = decodeURIComponent(name);
						
						cards.find('.' + settings.thumbClass).each(function() {
								var  fn = $(this).data('name') || getImageFileName($(this));
								
								if (fn === name || checkPageName(fn, name)) {
									t = $(this).closest('.' + settings.cardClass);
									return false;
								}
							});
						
						return t;
					},
				
				// Getting folders container
				
				getFolderContainer = function() {
					
						return content.find('.' + settings.foldersClass + ' .' + settings.contClass);
					},
										
				// Getting all folder cards
				
				getFolders = function(sel) {
						var fc = getFolderContainer();
						
						return fc.length? fc.find('.' + settings.cardClass + (sel || '')) : $();
					},
					
				// Getting map
				
				getMapRoot = function(create) {
						var mapbox =  content.find(settings.mapBoxHook);
						
						if (settings.pageType !== 'page') {
							// Finding only the map related to the current overlay 
							var ns = getOverlayNS();
							mapbox = mapbox.filter(function() { return $(this).data('rel') === ns; });
						}
						
						if (mapbox.length && mapbox.is(':visible')) {
							
							return mapbox.find('.' + settings.mapRootClass);
							
						} else if (create) {
							
							var newMap = $('<section class="map show-on-overlay">' +
												'<div class="cont box">' +
													'<div class="' + settings.mapRootClass + '" data-map-root></div>' +
												'</div>' +
											'</section>');
						
							if (mapbox.length) {
								mapbox.after(newMap);
							} else {
								getOverlay().before(newMap);
							}
							
							attachToOverlay(newMap);
							
							return newMap.find('.' + settings.mapRootClass);
						}
						
						return $();
					},
													
				// Getting attached lightbox
				
				getLightbox = function() {
					
						return getImageContainer().find('.' + settings.lightboxClass);
					},
					
				// Loading an image in lightbox
				
				lightboxLoad = function(card, auto) {
					
						pausePlayer(getImages(' .playing'));
						getImageContainer().trigger('lightboxLoad', (typeof auto === UNDEF)? card : [ card, auto ]);
					},
					
				// Leaving lightbox
				
				lightboxQuit = function() {
					
						getImageContainer().trigger('lightboxQuit');
					},
					
				// Getting selected items from current overlay
				
				getSelectedItems = function() {
						var r = [];
								
						getSelectedCards().each(function() {
								r.push($(this).data(J.OBJ));
							});
						
						return r;
					},
				
				// Getting all items from current overlay
				
				getAllItems = function() {
						var r = [];
								
						getCards().each(function() {
								r.push($(this).data(J.OBJ));
							});
						
						return r;
					},
								
				// Getting all items from base overlay
				
				getBaseItems = function() {
						var r = [];
						
						getBaseCards().each(function() {
								r.push($(this).data(J.OBJ));
							});
						
						return r;
					},
					
				// Getting active element's data
				
				getCurrentItem = function() {
						var card = getActiveCard();
						
						if (card.length) {
							return card.data(J.OBJ);
						}
						
						return null;
					},
					
				// Adding one folder's thumb
				
				loadFolderThumb = function(card) {
						
						card.find('[data-src]').each(function() {
								var t = $(this),
									img = card.find('img');
							
								if (!img.length) {
									img = $('<img>', {
											'class': 	settings.hideImageClass,
											alt: 		t.data('alt') || ''
										}).appendTo(t);
										
									img[0].width = t.data('width');
									img[0].height = t.data('height');
								}
								
								img.one('load', function() {
										card.removeClass(settings.lazyloadClass);
										$(this).addClass(settings.showImageClass).removeClass(settings.hideImageClass);
									})
									.attr('src', t.data('src'));
									
								t.removeData('src');
								
								card.find('.mosaic').children().each(function() {
										var m =  $(this),
											src = m.data('thumb'),
											mi = m.children('img').eq(0);
											
										if (src.length) {
											if (!mi.length) {
												mi = $('<img>', { 
														'class': settings.hideImageClass 
													}).appendTo(m);
											}
											
											mi.one('load', function() {
													mi.addClass(settings.showImageClass).removeClass(settings.hideImageClass);
												})
												.attr('src', src);
										}
									});
							});
					},
						
				// Loading visible folder thumbs :: only on base overlay
				
				lazyloadFolders = function(doneFn) {
					
						if (lazyloadFoldersInProgress || getLightbox().is(':visible') || hasOverlay()) {
							clearTimeout(lazyloadFoldersTimeout);
							return;
						}
						
						if (lazyloadFoldersTimeout) {
							clearTimeout(lazyloadFoldersTimeout);
							lazyloadFoldersTimeout = null;
						}
						
						var st = $window.scrollTop();
						
						if (Math.abs(st - scrollTop) > 20) {
							// Scrolling in progress: wait until finishes
							scrollTop = st;
							lazyloadFoldersTimeout = setTimeout(lazyloadFolders, settings.scrollCheckInterval);
							return;
						}
						
						// Getting folders to load
						var folders = getFolders('.' + settings.lazyloadClass);
						
						if (folders.length) {
							var wh = $window.height();
						
							lazyloadFoldersInProgress = true;
							
							folders.each(function() {
									var card = $(this),
										tt = card.offset().top,
										th = card.height();
										
									if ((tt + th) > (scrollTop - wh)) {
										// Below 1 screens above scroll pos
										if (tt < (scrollTop + wh)) {
											loadFolderThumb(card);
										} else {
											// Far below: stop loading more
											return false;
										}
									}
								});
							
							// perpetrating lazyload
							clearTimeout(lazyloadFoldersTimeout);
							lazyloadFoldersTimeout = setTimeout(lazyloadFolders, settings.scrollCheckInterval);
							
						} else {
							
							$window.off('.lazy');
						}
						
						lazyloadFoldersInProgress = false;
						scrollTop = st;
						
						if (typeof doneFn === FUNCTION) {
							doneFn.call(this);
						}
					},
					
				
				/********************************************************
				 *
				 *				Managing buttons and panels
				 *
				 ********************************************************/
				 
				// Toggle select buttons: selected count, max count
				
				toggleSelectButtons = function(sc, mc) {
						var p = shopRoot.add(feedbackRoot);
						
						if (typeof mc === UNDEF || !mc) {
							p.find('.' + settings.selectBoxClass).hide();					
						} else {
							p.find('.' + settings.selectBoxClass).show();
							p.find('.' + settings.selectNoneClass + 
									',.' + settings.addSelectedClass + 
									',.' + settings.keepSelectedClass).toggleClass(settings.disabledClass, typeof sc === UNDEF || sc === 0);
							
							p.find('.' + settings.selectAllClass).toggleClass(settings.disabledClass, typeof mc !== UNDEF && sc === mc);
						}
					},
						
				// Toggle boxes according to item count
				
				toggleBoxes = function(mc) {
					
						feedbackRoot.add(shopRoot).toggleClass(settings.disabledClass, !mc);
					},
					
				// Toggling 'has-images' class
				
				toggleHasImages = function(n) {
						
						main.toggleClass(settings.hasImagesClass, n).toggleClass(settings.noImagesClass, !n);
					},
						
				// Emptying the Search box
				
				emptySearchBox = function() {
						
						$(settings.search.hook).find('input').val('');
					},
					
				/********************************************************
				 *
				 *					Audio clip handling
				 *
				 ********************************************************/
				
				// Creating the AUDIO element
				
				createPlayer = function(btn) {
						var	player = $('<audio>', {
									'class':		'audioclip-player',
									src:			btn.attr('href'),
									id:				btn.attr('rel'),			
									controlsList:	'nofullscreen nodownload noremote noplaybackrate'
								}).hide().insertAfter(btn);
								
						player[0].volume = settings.audioClipVolume;
						
						return player;		
					},
					
				// Audio
					
				togglePlayer = function(e) {
						var btn = $(this),
							pl,
							p;
						
						e.preventDefault();
						
						// Initialized?
						if (btn[0].hasAttribute('rel')) {
							pl = $('#' + btn.attr('rel'));
						}
						
						// Has AUDIO element?
						if (!pl || !pl.length) {
							// not yet created
							btn.attr('rel', 'PL_' +  Math.floor(Math.random() * 10000));
							pl = createPlayer(btn);
						}
						
						if (pl[0].paused) {
							
							pausePlayers();
							
							// Paused: trying to start
							pl[0].play().then(function() {
									btn.removeClass('icon-volume-up paused').addClass('icon-pause playing');
									pl.on('ended', function() {
											btn.removeClass('icon-pause playing').addClass('icon-volume-up paused');
										});
								}, function(err) {
									btn.removeClass('icon-pause playing').addClass('icon-volume-up paused');
									console.log(err);
								});
						} else {
							pl[0].pause();
							btn.removeClass('icon-pause playing').addClass('icon-volume-up paused');
						}
						
						return false;
					},
					
				pausePlayers = function() {
					
						pausePlayer(getImages(' .playing'));
					},
					
				pausePlayer = function(el) {
						
						el.each(function() {
								var pl = $('#' + $(this).attr('rel'));
								
								if (pl.length) {
									pl[0].pause();
									$(this).removeClass('icon-pause playing').addClass('icon-volume-up paused');
								}
							});
					},
					
				/********************************************************
				 *
				 *						Rating
				 *
				 ********************************************************/
				
				// Reads ratings from local storage
				
				loadRatings = function(items) {
						var r = $.cookie(REL_PATH + PAGE_NAME + ':ratings');
						
						if (r) {
							if (typeof r === STRING) {
								r = r.split(':');
							}
							
							for (var i = 0; i < items.length && i < r.length; i++) {
								if (r[i] !== -1) {
									items[i][J.VISITORRATING] = r[i];
								}
							}
						}
						
						return items; 
					},
					
				// Saves visitor ratings to local storage
				
				saveRatings = function(items) {
						
						if (items && items.length) {
							var r = [];
							
							for (var i = 0; i < items.length; i++) {
								r.push(items[i].hasOwnProperty(J.VISITORRATING)? items[i][J.VISITORRATING] : -1);
							}
							
							$.cookie(REL_PATH + PAGE_NAME + ':ratings', r, 604800);
						}
					},
					
				/********************************************************
				 *
				 *						Folder tree
				 *
				 ********************************************************/
				
				getFolderTree = function() {
					
						var getFolderObjects = function(folder) {
								if (folder.hasOwnProperty(J.FOLDERS) && folder[J.FOLDERS].length) {
									var ul, li, i, p, f = folder[J.FOLDERS];
									ul = $('<ul>');
									for (i = 0; i < f.length; i++) {
										p = album.getPath(f[i]);
										li = $('<li>').appendTo(ul);
										li.append($('<a>', {
												href: 	(p? (p + '/') : '') + settings.indexName,
												html: 	f[i][J.TITLE] || f[i][J.NAME] || ''
											})).append(getFolderObjects(f[i]));
										if (!p) {
											li.addClass('actual');
										}
									}
									return ul;
								}
								return $();
							},
							
							getHomeLink = function() {
								var p = settings.rootPath;
								return $('<a>', {
										'class': 	'home-link icon-home',
										href:		(p? (p + '/') : '') + settings.indexName,
										text:		' ' + album.getAlbumTitle()
									});
							};
							
						if (settings.folderTree.type === 'tree') {
							if (settings.level > 0) {
								return getHomeLink().add(getFolderObjects(album.getTree()));
							} else {
								return getFolderObjects(album.getTree());
							}
						} else {
							getFolderObjects(album.getCurrentFolder());
						}
					};
			
				/********************************************************
				 *
				 *					Image overlay rendering 
				 *			or initializing images in the base overlay
				 *
				 ********************************************************/

				$.fn.renderImages = function(items, complete, baseSet) {
					
						if (typeof items === UNDEF || !items.length) {
							return this;
						}
					
						if (!Array.isArray(items)) {
							items = [ items ];
						}
						
						var	overlay				= $(this),												// Overlay
							ns 					= getOverlayNS(),										// Namespace
							imageContainer		= getImageContainer(true),								// image container (for overlays)
							cols 				= 0,													// Columns
							scrollTop 			= $window.scrollTop(),									// Scroll timeout
							lazyloadTimeout 	= null,													// Lazy load timeout
							fixTimeout 			= null,													// Layout fixing timeout
							
							// Rendering images
									
							initImages = function() {
									var cards = getCards(), 
										card,
										ar,
										image,
										img,
										isFolder,
										isLbable,
										isMedia,
										isIcon,
										caption,
										s,
										d,
										cd = settings.hasOwnProperty('markNew')? (nows - settings.markNew.days * ONEDAY_S) : null,
										item,
										shopCnt = 0;
									
									// Initializing images
									for (var i = 0; i < items.length; i++) {
										
										item = items[i];
										isFolder = item[J.CATEGORY] === 'folder';
										isLbable = !isFolder && album.isLightboxable(item);
										isMedia = item[J.CATEGORY] === 'video' || item[J.CATEGORY] === 'audio';
										isIcon = item.hasOwnProperty(J.THUMB)? /res\/\w+\.png$/.test(item[J.THUMB][J.PATH]) : false;
										ar = item.hasOwnProperty(J.THUMB)? item[J.THUMB][J.WIDTH] / item[J.THUMB][J.HEIGHT] : 1;
										s = item[J.THUMBCAPTION];
										
										if (i < cards.length) {
											// Already exists in the HTML source
											card = cards.eq(i);
										} else {
											// Not yet exists in the HTML source: happens only on overlays
											card = $('<div>', {
														'class': 	[ 
																	settings.cardClass, 
																	settings.lazyloadClass, 
																	item[J.CATEGORY],
																	s? 'hascaption' : '',
																	(item.hasOwnProperty(J.THUMB) && item[J.THUMB][J.WIDTH] < settings.thumbDims[0] && item[J.THUMB][J.HEIGHT] < settings.thumbDims[1])? 'tiny' : '',
																	isIcon? 'icon' : '',
																	(ar >= 1.25)? 'landscape' : ((ar <= 0.8)? 'portrait' : 'square')
																].filter(Boolean).join(' ')
															})
														.appendTo(imageContainer);
										}
										
										if (isLbable) {
											card.addClass(settings.lbableClass);
										}
										
										// Data pointer
										card.data(J.OBJ, item);
										
										// Finding/creating image's <a> wrap
										image = card.children('.' + settings.thumbClass);
										
										if (!image.length) {
											image = $('<a>', {
													'class':	settings.thumbClass
												})
												.appendTo(card);
										}
										
										// Creating structure
										
										// Finding/creating image
										img = image.find('img').eq(0);
										
										if (!img.length) {
											img = $('<img>', {
													alt:		album.getAlt(item)
												}).appendTo(image);
										}
										
										if (isIcon) {
											image.addClass('iconthumb');
										} else {
											
											if (item[J.THUMB].hasOwnProperty(J.RENDITIONS)) {
												// Has renditions, wait if larger image is needed
												img.data('hasrenditions', true);
											} else {
												// Removing preload class upon img is loaded
												img.one('load.' + ns, function() {
														$(this).parents('.' + settings.cardClass).removeClass(settings.preloadClass);
													});
											}
									
											// Onload event
											img.addClass(settings.hideImageClass)
												.one('load.' + ns, function() {
													$(this).removeClass(settings.hideImageClass)
														.addClass(settings.showImageClass)
														.parents('.' + settings.cardClass).removeClass(settings.preloadClass);
												});
											
											if (img[0].hasAttribute('src') && img[0].complete) {
												img.trigger('load');
											} else {
												img.addClass(settings.hideImageClass);
											}
											
											if (!isMedia) {
												// Adding width, height
												img.attr({
														width:		item[J.THUMB][J.WIDTH],
														height:		item[J.THUMB][J.HEIGHT]
													});
												
												// Adding pano class
												if ((item[J.THUMB][J.HEIGHT] / item[J.THUMB][J.WIDTH]) < 0.5) {
													card.addClass(settings.panoClass);
												}
											}
										}
										
										// Masonry: adding width
										if (settings.thumbLayout === 'horizontal') {
											card.css({
													flexBasis:		Math.round(item[J.THUMB][J.WIDTH] * 0.9)
												});
										} else if (settings.thumbLayout === 'justified') {
											// Justified: Saving width as data
											card.data('width', item[J.THUMB][J.WIDTH]);
										}
										
										// Save image name: for later
										if (!image.data('name')) {
											image.attr('data-name', album.getItemName(item));
										}
									
										// Caption
										if (s) {
			
											caption = card.children('.' + settings.captionClass);
											
											if (!caption.length) {
												// No caption yet
												caption = $('<div>', {
														'class': 		settings.captionClass,
														html: 			s
													}).appendTo(card);
											}
											
											card.addClass(settings.hascaptionClass);
										}
												
										// Preparing for Lightbox
										if (isLbable) {
												
											// Click event
											image.on('click.' + ns, imageClicked);
											
											// Set href
											if (!image.data('href')) {
												image.attr('data-href', image.attr('href') || album.getOptimalImagePath(item, [ image[0].offsetWidth, image[0].offsetHeight ]));
											}
											
											// Removing link (to avoid unwanted navigation)
											// Links are included to help robots finding the slide images
											if (image[0].hasAttribute('href')) {
												image.attr('href', '');
											}
											
											var hasShop = shopRoot.length && album.hasShop(item);
											
											// Adding select box
											if (hasShop || feedbackRoot.length && settings.feedback) {
												card.addClass(settings.selectableClass);
											}
											
											// Adding shop button
											if (hasShop) {
												d = $('<a>', {
															'class': 	'icon-shopping-cart add-cart',
															text: 		' ' + text.addCart
														})
													.on('click.' + ns, function(e) {
															var card = $(e.target).closest('.' + settings.cardClass);
															if (card.length) {
																shopRoot.trigger('addItems', card.data(J.OBJ));
															}
															return false;
														})
													.appendTo(image);
													
												d.addTooltip(album.getPriceRange(item));
												
												shopCnt += 1;
												
												card.addClass(settings.shoppableClass);
											}
											
											// Adding feedback button
											if (!hasShop && feedbackRoot.length && settings.feedback) {
												d = $('<a>', {
															'class': 	'icon-email-send add-cart',
															text: 		' ' + settings.feedback.addBtnLabel
														})
													.on('click.' + ns, function(e) {
															var card = $(e.target).closest('.' + settings.cardClass);
															if (card.length) {
																feedbackRoot.trigger('addItems', card.data(J.OBJ));
															}
															return false;
														})
													.appendTo(image);
											}
											
											// Adding Fotomoto button (if no shop button)
											if ((!LOCAL || DEBUG) && !hasShop &&
												settings.hasOwnProperty('fotomoto') && !album.hideFotomoto(item)) {
											
												d = $('<a>', {
															'class': 	'icon-shopping-cart add-cart fotomoto-btn',
															text: 		' ' + text.fotomotoBtn
														})
													.on('click.' + ns, function(e) {
															e.preventDefault();
															if (typeof FOTOMOTO !== UNDEF) {
																var card = $(e.target).closest('.' + settings.cardClass);
																if (card.length) {
																	FOTOMOTO.API.showWindow(10, 
																			album.getOriginalPath(card.data(J.OBJ)) || album.getItemPath(card.data(J.OBJ))
																		);
																}
															}
															return false;
														})
													.appendTo(image);
													
												d.addTooltip(text.fotomotoTooltip);
											}
											
											if (!baseSet && settings.linkToFolder) {
												// Folder link
												var p = album.getPath(item);
												
												if (p) {
													image.append($('<a>', {
															'class': 	'icon-folder folder-link',
															href:		 p + settings.indexName
														}));
												}
											}
											
											// Adding audio clip button
											if (settings.useAudioClipButton && item.hasOwnProperty(J.AUDIOCLIP)) {
												s = $('<a>', {
															'class':	'icon-volume-up audioclip',
															href:		item[J.AUDIOCLIP][J.PATH]
														}).on('click', togglePlayer)
														.appendTo(image);
														
												if (!TOUCHENABLED) {
													s.data('tooltip', text.audioClipLabel).addTooltip();
												}
											}
											/*
											if (baseSet) {
												image.attr('href', '#img=' + item[J.PATH]);
											}
											*/
										} else {
											// Not lightboxable, e.g. webLocation or folder: using the link
											card.addClass(item[J.CATEGORY]);
											if (!image[0].href) {
												image.attr('href', album.getItemPath(item));
											}
										}
										
										// Mark new images
										if (cd != null) {
											
											d = item[J.DATES];
											
											if (d && (d = d[settings.markNew.reference]) && d >= cd) {
												s = settings.markNew.text?
													$('<span>', {
															'class':	'new-image',
															text:		settings.markNew.text
														})
													:
													$('<span>', {
														'class':	'icon-new-fill new-image'
														});
												
												image.append(s);
													
												if (!TOUCHENABLED) {
													s.data('tooltip', (new Date(d * 1000)).toLocaleDateString(LOCALE)).addTooltip();
												}
											}
										}
										
										// Rating
										if (settings.useRating && isLbable) {
											
											var r = item[J.RATING] || 0;
											
											if (settings.visitorRating || r) {
												
												s = $('<span>', {
														'class':	'rating',
														html:		STARS
													})
													.appendTo(image);
													
												if (settings.visitorRating) {
													
													if (item.hasOwnProperty(J.VISITORRATING)) {
														// Already exists
														r = item[J.VISITORRATING];
													} else {
														if (!settings.useJalbumRating) {
															// Ignore jAlbum rating
															r = 0;
														}
														// Save the default
														item[J.VISITORRATING] = r;
													}
													
													s.addClass('r' + r + (settings.useJalbumRating? '' : ' icon-close'))
														.on('click.' + ns, function(e) {
																var el = $(this),
																	r = Math.minMax(0, Math.floor((getCoords(e).x - el.offset().left - parseFloat(el.css('paddingLeft'))) / 16), 5),
																	item = el.closest('.' + settings.cardClass).data(J.OBJ),
																	extDir = false;
																	//Math.minMax(0, Math.floor((getCoords(e).x - el.offset().left + 4) / el.width()) * 5, 5);
																	
																el.removeClass('r0 r1 r2 r3 r4 r5').addClass('r' + r);
																item[J.VISITORRATING] = r;
																
																if (!baseSet) {
																	// Propagate to base overlay
																	el = getCardByName(item[J.NAME], true);
																	if (el.length) {
																		el.find('.rating')
																			.removeClass('r0 r1 r2 r3 r4 r5')
																			.addClass('r' + r);
																		el.data(J.OBJ)[J.VISITORRATING] = r;
																		saveRatings(getBaseItems());
																	}
																} else {
																	saveRatings(items);
																}
																
																return false;
															});
												} else {
													// Unalterable rating from jAlbum
													s.addClass('r' + r);
												}
											}
										}
										
										// Prevent copy
										if (settings.rightClickProtect) {
											image.on('contextmenu', function(e) {
													e.preventDefault()
													return false;
												});
										}
									}
	
									// Fixing layout only on screen size change
									$window.on('resize.' + ns, function(e) {
											clearTimeout(fixTimeout);
											fixTimeout = setTimeout(fixLayout, 500);
											return true;
										});
										
									fixLayout();
	
									main.toggleClass(settings.hasShopClass, shopCnt > 0);
									main.toggleClass(settings.noShopClass, !shopCnt);
									
									window.requestAnimationFrame((typeof complete === FUNCTION)? 
											function() { finishInit(complete); }
											:
											finishInit
										);
									
								},
								
							// Finishing initialization
								
							finishInit = function(complete) {
								
									// Moving the active element into view
									overlay.on({
											setactivecard:		function(e, card) {
																		setActiveCard(card);
																	},
									
											refreshRating:		refreshRating,
											
											refresh:			fixLayout
										});
									
									//console.log('renderImages took ' + Math.round((new Date() - timer) / 1000) + 'ms');
									// Done function
									if (typeof complete === FUNCTION) {
										if (typeof baseSet !== UNDEF) {
											complete.call(undefined, overlay, baseSet);
										} else {
											complete.call(undefined, overlay);
										}
									}
								},
								
							// Justifying rows
														
							justifyRows = function(doneFn) {
									var oly = getOverlay();
									
									clearTimeout(fixTimeout);
									
									// No thumbnails section?
									if (!oly.length) {
										return;
									}
									
									// Visible?
									if (!oly.is(':visible')) {
										// Retry until imagecontiner gets visible
										fixTimeout = setTimeout(justifyRows, 100);
										return;
									}
									
									var	cw = oly.children('.' + settings.contClass).width(),
										mcw = 'calc(100% - ' + (2 * settings.thumbGap + 10) + 'px)';
									
									if (oly.innerWidth() < settings.pageWidth && oly.hasClass('overlay')) {
										// Fixing fantom scrollbars
										cw += 32;
									}
									
									// Is it already fixed for this width?
									if (oly.data('fixedfor') === cw) {
										if (typeof doneFn === FUNCTION) {
											doneFn.call();
										}
										return;
									}
									
										// Justifying rows									
									var	justifyRow = function(cards, f) {
												
												if (f > 1.6) {
													f = 1.6;
												}
												
												var h = (cards.length > 1)? Math.floor(settings.thumbDims[1] * f) : 'auto';
												
												cards.each(function() {
														var el = this.querySelector('.' + settings.thumbClass);
														if (el) {
															this.style.width = $(this).data('width') * f + 'px';
															el.style.height = h + 'px';
															el = this.querySelector('.' + settings.captionClass);
															if (el) {
																el.style.removeProperty("max-width");
															}
														}
													});
											},
											
										justifySection = function(cards) {
												var i = 0, 
													rw = 0,
													w;
												
												while (cards.length && i < cards.length) {
													
													w = cards.eq(i).data('width') + 2 * settings.thumbGap;
													
													if ((rw + w) >= cw) {
														// Overflown
														if (rw + (w * settings.justifyWrapThreshold) >= cw) {
															// Wrap this item
															justifyRow(cards.slice(0, i), (cw - 2 * i * settings.thumbGap) * .9999 / (rw - 2 * i * settings.thumbGap));
															cards = cards.slice(i);
															
														} else {
															// With this item
															justifyRow(cards.slice(0, i + 1), (cw - 2 * (i + 1) * settings.thumbGap) * .9999 / (rw + w - 2 * (i + 1) * settings.thumbGap));
															cards = cards.slice(i + 1);
														}
														
														i = 0;
														rw = 0;
														
													} else {
														
														i++;
														rw += w;
													}
												} 
												
												if (cards.length > 1) {
													// Remainder: the last row(s)
													var f = (cw - 2 * cards.length * settings.thumbGap) * .9999 / (rw - 2 * cards.length * settings.thumbGap);
													// Max. 20% zoom
													justifyRow(cards, (f > 1.2)? 1 : f);
												} else {
													// Single image
													justifyRow(cards, 1);
												}
												
												// Remembering last fixed width
												oly.data('fixedfor', cw);
											};
									
									// Starts here
									requestAnimationFrame(function() {
											justifySection(oly.find('.'  + settings.cardClass));
										});
									
									if (typeof doneFn === FUNCTION) {
										doneFn.call();
									}
									
								},
																										
							// Fixing layout for vertical and justified layouts and lazy loading thumbs
							
							fixLayout = function() {
								
									clearTimeout(fixTimeout);
									
									// Fixing layout, starting lazyload on complete
									if (settings.thumbLayout === 'justified') {
										fixTimeout = setTimeout(justifyRows, 100, lazyloadThumbs);
									} else {
										fixTimeout = setTimeout(lazyloadThumbs, settings.scrollCheckInterval / 2);
									}
								},
								
							// Loading visible thumbs
							
							lazyloadThumbs = function() {
								
									if (getLightbox().is(':visible')) {
										return;
									}
									
									clearTimeout(lazyloadTimeout);
									lazyloadTimeout = null;
										
									var st = $window.scrollTop(),
										wh = $window.height();
									
									if (Math.abs(st - scrollTop) > 20) {
										// Scrolling in progress
										scrollTop = st;
										lazyloadTimeout = setTimeout(lazyloadThumbs, settings.scrollCheckInterval);
										return;
									}
									
									// Lazyload images
									var cards = getCards('.' + settings.lazyloadClass).not(':empty');
									
									// Adding preloaded images with too small thumbs and having renditions
									cards = cards.add(getCards('.' + settings.preloadClass).not(':empty').not('.icon').filter(function() {
											var thumb = $(this).find('.' + settings.thumbClass), 
												img = thumb.find('img').eq(0);
												
											if (	!settings.fixedShapeThumbs &&
													img.data('hasrenditions') && 
													(img[0].width * 1.1 < thumb.width() || img[0].height * 1.1 < thumb.height())
												) {
												return true;
											}
											
											$(this).removeClass(settings.preloadClass);
											return false;
										}));
									
									if (cards.length) {
									
										cards.each(function() {
												var card = $(this),
													ct = card.offset().top,
													ch = card.height();
													
												if ((ct + ch) > (scrollTop - wh)) {
													
													if (ct < (scrollTop + 2 * wh)) {
														// 2 screens ahead
														loadThumb(card);
													} else {
														return false;
													}
												}
											});
									
										// perpetrating lazyload
										scrollTop = st;
										lazyloadTimeout = setTimeout(lazyloadThumbs, settings.scrollCheckInterval);
										
									} else {
										// No more left
										$window.off('scroll.' + ns);
									}
								},
						
							// Clicked image
							
							imageClicked = function(e) {
									if (e.target.nodeName === 'A' && ($(e.target).closest('.' + settings.commentClass).length || $(e.target).hasClass('folder-link'))) {
										// Link in comment or link to folder
										return true;
									}
									
									$body.hideAllTooltips();
									
									lightboxLoad($(e.target).closest('.' + settings.cardClass));
									
									return false;
								},
							
							// Adding one thumb
							
							loadThumb = function(card) {
									
									if (card.length) {
										var thumb = card.children('.' + settings.thumbClass).eq(0),
											img = thumb.children('img');
											
										if (!img.length) {
											// Create IMG element if missing
											thumb.append($('<img>', {
													'class': 	settings.hideImageClass
												}));
										}
										
										if (card.hasClass(settings.lazyloadClass) && typeof img.attr('src') !== UNDEF && img.attr('src') !== false) {
											// IMG already has source and lazyloaded
											thumb.removeClass(settings.hideImageClass).addClass(settings.showImageClass);
										} else {
											// Need to load because either missing or low res.
											if (card.hasClass(settings.lazyloadClass)) {
												// Show animation only for lazyloaded images
												img.one('load', function() {
														// Showing image on load
														$(this).removeClass(settings.hideImageClass).addClass(settings.showImageClass);
													});
											}
											// Set source: optimal thumbnail if fixedShape, can fall back to images otherwise  
											img.attr('src', album.getOptimalThumbPath(card.data(J.OBJ), [ thumb.outerWidth(), thumb.outerHeight() ], !settings.fixedShapeThumbs));
										}
										
										// Remove classes
										card.removeClass(settings.lazyloadClass + ' ' + settings.preloadClass);
									}
								},
					
							// Marking active card (el is DOM element)
						
							setActiveCard = function(el) {
								
									if (el && el instanceof Element) {
										var card		= $(el),
											item 		= card.data(J.OBJ),
											wh 			= $window.height(),
											scrollTop 	= $window.scrollTop(),
											ct 			= card.offset().top,
											ch 			= card.outerHeight();
										
										getCards('.' + settings.activeClass).removeClass(settings.activeClass);
										card.addClass(settings.activeClass);
										
										// Triggering active placemark
										if (settings.hasOwnProperty('map')) {
											if (album.hasLocation(item)) {
												getMapRoot().trigger('setActive', item[J.NAME]);
											} else {
												getMapRoot().trigger('resetMarkers');
											}
										}
										
										// Moving into view
										if (ct + ch > scrollTop + wh) {
											window.scrollTo({ 
												top: 		Math.round(ct - wh + ch + 30), 
												behavior: 	'smooth'
											});
										} else if (ct < scrollTop) {
											window.scrollTo({ 
												top: 		Math.round(Math.max(ct - 30, 0)), 
												behavior: 	'smooth'
											});
										}
									}
								},
								
							// Refreshing the rating
							
							refreshRating = function(e, el, rating) {
								
									if (el) {
										var card	= $(el),
											o 		= card.data(J.OBJ),
											r 		= ((typeof rating !== UNDEF)? 
															rating 
															: 
															(settings.visitorRating? o[J.VISITORRATING] : o[J.RATING])
														) || 0;
											
										card.find('.rating')
											.removeClass('r0 r1 r2 r3 r4 r5')
											.addClass('r' + r);
											
										if (!baseSet) {
											getCardByName(o[J.NAME], true).find('.rating')
												.removeClass('r0 r1 r2 r3 r4 r5')
												.addClass('r' + r);
										} else {
											saveRatings(items);
										}
									}
									
								};
						
						// Creating new Unique ID if not yet exists
						if (!ns) {
							ns = setOverlayNS();
						}
						
						if (baseSet && settings.useRating && settings.visitorRating) {
							loadRatings(items);
						}
						
						window.requestAnimationFrame(initImages);
								
						return this;
					};
		
		
			var	
				/********************************************************
				 *
				 *					Preparing images
				 *
				 ********************************************************/
					
				prepareImages = function(overlay, baseSet) {
					
						if (!album) {
							return;
						}
						
						var cards = getCards();
							
						if (cards.length > 1 || settings.showEmptyFilters) {
							
							// Filtering and sort only if there are at least 2 items
							
							if ($('.filter-cont').length) {
								// Initializing Filters / Sort
								prepareFilterBox(baseSet);
							}
							
						}
						
						if (!cards.filter('.' + settings.lbableClass).length) {
							
							$('.' + settings.startshowClass).fadeOut();
							
						} else {
							
							if (album.hasShop() || settings.feedback) {
								
								var ci = getSelectableCards();
								
								toggleSelectButtons(0, ci.length);
								
								toggleBoxes(ci.length);
								
								if (ci.length) {
									// Make items selectable
									overlay.selectable({
											cardClass:			settings.selectableClass,
											selectedClass:		settings.selectedClass,
											selectionChange:	function(n) {
																		toggleSelectButtons(n, ci.length);
																		shopRoot.add(feedbackRoot).trigger('itemsSelected', n);
																	}
										});
									
									
									shopBox.add(feedbackBox)
										.find('.' + settings.keepSelectedClass).on('click.' + ns, function() {
											keepSelectedItems(text.selectedItems, function() {
													setHash();
												});
											
											return false;
										});
										
									// Initialized
									content.find('[data-sticky-rel]').trigger('stickyRefresh');
									// Uninited
									content.find('[data-sticky]').not('[data-sticky-rel]').sticky();
									
								}
							}
						
							// Initializing map
							
							if (settings.hasOwnProperty('map')) {
								
								// Pass defaults to lightbox
								if (settings.map.hasOwnProperty('apiKey')) {
									$.fn.lightbox.defaults.mapApiKey = settings.map.apiKey;
								}
							
								if (settings.map.hasOwnProperty('type')) {
									$.fn.lightbox.defaults.mapType = settings.map.type;
								}
							
								if (settings.map.hasOwnProperty('zoom')) {
									$.fn.lightbox.defaults.mapZoom = settings.map.zoom;
								}

								if (settings.map['index']) {
									prepareMap(getAllItems(), settings.map);
								}
							}
							
							/* Waiting for Derek's answer
							if (!LOCAL && typeof FOTOMOTO !== UNDEF && settings.fotomoto) {
								FOTOMOTO.API.checkinImages(getFotomotoImages(cards), function(images) {
										
									});
							}
							*/
							
							overlay.trigger('lightboxRemove');
							
							overlay.lightbox(album, {
									
									baseSet:		(typeof baseSet === UNDEF)? false : baseSet,
									
									onLoadStart:	function(card) {
															var item = card.data(J.OBJ);
															
															if (item) {
																addParam(settings.pageName || settings.indexName, {
																		img: 		album.getItemName(item)
																	}, item[J.TITLE]);
															}						
														},
														
									onLoadEnd: 		function(card) {
															if (typeof card !== UNDEF) {
																overlay.trigger('setactivecard', card);
															}
															setTimeout(updateShares, 100);
														},
													
									onClose:		function(card) {
															if (settings.slideshowFullscreen && !settings.slideshowFullscreenOnlyMobiles) {
																// Exit fullscreen only on desktops
																exitFullscreen();
															}
															if (typeof card !== UNDEF) {
																overlay.trigger('setactivecard', card);
															}
															updateShares();
															overlay.trigger('refresh');
															removeParam(settings.pageName || settings.indexName, 'img');
														},
														
									onVolumeChange:	function(v) {
															settings.audioClipVolume = v;
														},
													
									onReady:		function() {
															if (initByHash && window.location.hash) {
																initByHash = false;
																var param = readParam();
																
																if (param.hasOwnProperty('label') && param.label !== null) {
																	// Label filter
																	getItemsByLabel(param.label, function() {
																		addParam(settings.indexName, {
																				label:			param.label + '',
																				search:			null,
																				newimages:		null,
																				date:			null
																			}, text.label + ': ' + param.label);
																	});
																} else if (param.hasOwnProperty('search') && param.search !== null) {
																	// Search
																	getItemsBySearch(param.search, function() {
																		addParam(settings.indexName, {
																				search:			param.search + '',
																				label:			null,
																				newimages: 		null,
																				date:			null
																			}, text.search + ': ' + param.search);
																	});
																} else if (param.hasOwnProperty('newimages')) {
																	// Search new images
																	getItemsByDate(getSearchNewOptions(param.newimages), function() {
																		addParam(settings.indexName, {
																				newimages:		param.newimages,
																				search:			null,
																				label:			null,
																				date:			null
																			}, text.searchNew + ': ' + param.newimages);
																	});
																} else if (param.hasOwnProperty('date')) {
																	var options = getDateRangeOptions(param.date);
																	// Search images in date range
																	getItemsByDate(options, function() {
																		addParam(settings.indexName, {
																				date:			param.date,
																				newimages:		null,
																				search:			null,
																				label:			null
																			}, getDateReference(options.reference) + ': ' + param.date);
																	});
																} else if (param.hasOwnProperty('img') && param.img !== null) {
																	var card = getCardByName(param.img);
																	
																	if (card.length) {
																		if (param.hasOwnProperty('slideshow')) {
																			lightboxLoad(card, 2000);
																		} else {
																			lightboxLoad(card);
																		}
																	} else {
																		removeParam(settings.pageName || settings.indexName, 'img');
																	}
																} else if (param.hasOwnProperty('slideshow')) {
																	lightboxLoad(0, 2000);
																}
															}
															
															$('.' + settings.startshowClass).fadeIn().on('click', function() {
																	$(this).trigger('removeTooltip');
																	startSlideshow();
																	return false;
																});
														}
								});
							
						}
					},
				
				// Sets the URL by JS
				
				setHash = function(params, title) {
						var hash = (typeof params === UNDEF)? '' : '#' + paramize(params);
					
						if (hash !== window.location.hash) {
							hashChangedByJs = true;
							if (params) {
								setParam(settings.pageName || settings.indexName, params, title || '');
							} else {
								removeParam(settings.pageName || settings.indexName);
							}
							setTimeout(function() {
									// reseting hashByJs
									hashChangedByJs = false;
								}, 1000);
						}
					},
				
				// Hash change listener :: when the browser's back / forward buttons were pressed
				
				stateChange = function() {
						if (window.location.hash === lastHash || hashChangedByJs) {
							// Same hash or changed by JS (not on external trigger)
							return;
						}
						
						lastHash = window.location.hash;
						hashChangedByJs = false;
						var param = readParam();
						
						if (param.hasOwnProperty('label') && param.label !== null) {
							// Label filter
							getItemsByLabel(param.label, function() {
									updateShares();
								});
						} else if (param.hasOwnProperty('search') && param.search !== null) {
							// Search
							getItemsBySearch(param.search, function() {
									updateShares();
								});
						} else if (param.hasOwnProperty('newimages') && param.newimages !== null) {
							// New images
							getItemsByDate(getSearchNewOptions(param.newimages), function() {
									updateShares();	
								});
						} else if (param.hasOwnProperty('date') && param.date !== null) {
							// Date range
							var options = getDateRangeOptions(param.date);
							getItemsByDate(options, function() {
									updateShares();	
								});
						} else if (param.hasOwnProperty('img') && param.img !== null) {
							// Image (no search or label)
							var card = getCardByName(param.img);
							if (card.length) {
								if (param.hasOwnProperty('slideshow')) {
									lightboxLoad(card, 2000);
								} else {
									lightboxLoad(card);
								}
							} else {
								removeParam(settings.indexName, 'img');
							}
						} else if (param.hasOwnProperty('slideshow')) {
							lightboxLoad(0, 2000);
						} else {
							// Plain index page
							lightboxQuit()
							removeOverlay();
						}
						
					},
				
				// Starts slideshow
				
				startSlideshow = function(e) {
						var card = $();
						
						if (!settings.restartSlideshow) {
							card = getCards('.' + settings.activeClass);
						}
						
						if (!card.length) {
							// Fall back to first
							card = getCards().eq(0);
						}
					
						if (settings.slideshowFullscreen && 
							(!settings.slideshowFullscreenOnlyMobiles || SMALLSCREEN)) { 
							requestFullscreen();
						}
						
						lightboxLoad(card, 2000);
						
						return false;
					},
					
				/********************************************************
				 *
				 *		Functionalities that results in new overlay
				 *
				 ********************************************************/
				
				// Preparing the overlay before the items arrive
				prepareOverlay = function(attr) {
						var oly = addOverlay(attr);
						
						appendProgressbar();
						focusOverlay();
						
						return oly;
					},
					
				// Finishing the overlay when the items are ready
				renderOverlay = function(oly, items, ready) {
					
						oly.renderImages(items, prepareImages);
						
						removeProgressbar();
						
						toggleSelectButtons(0, items.length);
						toggleBoxes(items.length);
						toggleHasImages(items.length);
						
						if (typeof ready === FUNCTION) {
							ready.call();
						}
					},
					
				/********************************************************
				 *
				 *				Keeping only selected items
				 *
				 ********************************************************/
				
				keepSelectedItems = function(tx, ready) {
						var items = getSelectedItems();
						
						if (!items.length || items.length === getCards('.' + settings.lbableClass).length) {
							return;
						}
						
						var oly = prepareOverlay({
									className:	'selected-items',
									icon:		'icon-checkbox-checked',
									title:		'&ldquo;' + tx + '&rdquo;'
								});
							
						setTimeout(function() {
							
								appendToOverlayTitle($('<small>', {
										text: 		text.foundNTimes.replace('{0}', items.length)
									}));
								
								renderOverlay(oly, items, ready);
								
							}, 50);
					},
				
				/********************************************************
				 *
				 *					Search and tag search
				 *
				 ********************************************************/
				 
				// Executing Search
				
				getItemsBySearch = function(tx, ready) {
						var oly = prepareOverlay({
									className:	'search-results',
									icon:		'icon-search',
									title:		'&ldquo;' + tx + '&rdquo;'
								});
							
						album.collectItems($.extend(settings.search, {
								terms:		tx,
								depth:		'tree',
								ready:		function() {
													
													appendToOverlayTitle($('<small>', {
															text: 	this.length? text.foundNTimes.replace('{0}', this.length) : text.notFound
														}));
													
													renderOverlay(oly, this, ready);
												}
							}));
					},
					
				// Preparing Search functionality
				
				prepareSearch = function(f) {
					
						if (!f.length || !album) {
							return;
						}
						
						// Adding defaults if missing
						
						settings.search = $.extend({
								depth:		'tree',
								fields:		'creator,keywords,title,comment'
							}, settings.search);
						
						var startSearch = function(f) {
								
								var tx = f.find('input').eq(0).val();
								
								if (tx && tx.length > 1) {
									getItemsBySearch(tx, function() {
											setHash({
													search:		tx
												}, text.search + ': ' + tx);
											updateShares();
										});
								}
								return false;
							};
							
						f.find('.search-btn').on('click', function() {
								if (f.hasClass(settings.activeClass)) {
									startSearch(f);
									f.removeClass(settings.activeClass);
								} else {
									f.find('input').eq(0).focus();
									f.addClass(settings.activeClass);
								}
								return false;
							});
						
						f.on('submit', function(e) {
								e.preventDefault();
								startSearch(f);
								f.removeClass(settings.activeClass);
								return false;
							});
						
						f.find('input').on('focus',	function() { 
								f.addClass(settings.activeClass); 
							});
					},
				
				/********************************************************
				 *
				 *						Tag cloud
				 *
				 ********************************************************/
				
				// Preparing Tag Search functionality
				
				prepareTagSearch = function(target) {
						var f = target.find('.' + settings.tagSearchClass),
							// Start tag search		
							startSearch = function(f) {
									var tx = f.find('input').eq(0).val();
									
									if (tx) {
										getItemsByLabel(tx, function() {
											setHash({
													label: 	tx
												}, text.label + ': ' + tx);
											updateShares();
										});
									}
									return false;
								};
						
						if (!f.length) {
							f = $('<form>', {
									'class': 		settings.tagSearchClass
								}).appendTo(target);
							
							f.append($('<div>', {
									'class':		'search-input'
								})
								.append($('<input>', {
									'type':			'text',
									placeholder:	text.searchPlaceholder,
									ariaLabel:		text.search
								})));
								
							f.append($('<a>', {
									'class':		'search-btn icon-search'
								}));
						}
						
						f.find('.search-btn').on('click', function() {
								var f = $(this).closest('form');
								f.find('input').eq(0).focus();
								startSearch(f);
								return false;
							});
						
						f.on('submit', function(e) {
								e.preventDefault();
								startSearch($(this));
								return false;
							});
						
					},
				
				// Search labels
			
				getItemsByLabel = function(tx, caseSensitive, ready) {
						var oly = prepareOverlay({
									className:		'tags-found',
									icon:			'icon-tag',
									title:			'&ldquo;' + tx + '&rdquo;'
								});
							
						if (typeof caseSensitive === UNDEF) {
							caseSensitive = false;
						} else if (typeof caseSensitive === FUNCTION) {
							ready = caseSensitive;
							caseSensitive = false;
						}
						
						album.collectItems($.extend(settings.tagCloud, {
								caseSensitive:	caseSensitive,
								exact:			true,
								terms:			tx,
								ready: 			function() {
									
														appendToOverlayTitle($('<small>', {
																text: 	this.length? text.foundNTimes.replace('{0}', this.length) : text.notFound
															}));
														
														renderOverlay(oly, this, ready);
													}
							}));
					},
					
				// Collect items from a path array
				
				getItemsByPaths = function(tx, paths, ready) {
						var oly = prepareOverlay({
									className:	'tags-found',
									icon:		'icon-tag',
									title:		'&ldquo;' + tx + '&rdquo;'
								});
							
						album.collectByPath({
								paths:		paths,
								ready:		function() {
									
														appendToOverlayTitle($('<small>', {
																text: 		this.length? text.foundNTimes.replace('{0}', this.length) : text.notFound
															}));
														
														renderOverlay(oly, this, ready);
													}
							});
					},
					
				// Add toggle button
				
				addToggleButton = function(t) {
						var	btn;

						t.css({
								maxHeight: 	150,
								overflow: 	'hidden'
							});
						
						btn = $('<div>', {
								'class': 	'toggle-height'
							}).insertAfter(t);
						
						btn.append($('<a>', {
								'class': 	'icon-caret-down',
								text: 		text.more
							})
							.on('click', function() {
									var b = t.parent(),
										toggleHeight = function() {
												var h = t.prop('scrollHeight');
												
												b.css({
														height: 	'auto',
														overflow: 	'visible'
													});
												t.css({
														maxHeight: 	150,
													});
												setTimeout(function() {
														t.css({
																maxHeight: 	h + 20
															});
													}, 25);
												btn.addClass(settings.activeClass);
											};
									
									b.css({
											height: 	b.height(),
											overflow: 	'hidden'
										});
									
									t.css({
											maxHeight: 	'none'
										});
									
									setTimeout(toggleHeight, 50);
								
								}
						))
						.append($('<a>', {
								'class': 	'icon-caret-up',
								text: 		text.less
							}).on('click', function() {
								t.css({
										maxHeight: 	150
									});
								btn.removeClass(settings.activeClass);
							}));
					
					},
				
				// Collecting and displaying tags
			
				prepareTagCloud = function(tags) {
						var tc = $(settings.tagCloud['hook'] || '.tag-cloud-cont');
						
						if (!tc.length) {
							return;
						}
						
						if (!tags.length) {
							// Nothing found
							tc.parents('section').eq(0).hide();//remove();
							return;
						}
						
						var t = tc.children('.' + settings.tagCloudClass),
							// Click function
							tagClicked = function() {
									var tx = $(this).children('span').eq(0).text();
									
									if (tx) {
										getItemsByPaths(tx, $(this).data('paths'), function() {
											setHash({
														label: 		tx
													}, text.label + ': ' + tx);
												updateShares();
											});
									}
									
									return false;
								};
						
						// Creating wrap el if not exists
						if (!t.length) {
							t = $('<div>', {
									'class':		settings.tagCloudClass
								}).appendTo(tc);
						}
							
						// Adding tags
						for (var a, r, cnt, i = 0, l = tags.length; i < l; i++) {
							
							// Creating link element
							a = $('<a>', {
										'class': 	'tag'
									})
								.on('click', tagClicked)
								.data('paths', tags[i][1])
								.append($('<span>', {
										html: 		tags[i][0]
									}));
							
							// Adding count
							if (tags[i][1].length > 1) {
								
								if (settings.tagCloud.fontVaries) {
									r = 0;
									cnt = Math.min(64, tags[i][1].length);
									// calculate weight
									while (cnt > 1) {
										r++;
										cnt = cnt >> 1;
									}
									a.addClass('size-' + r);
								}
								
								a.append('&nbsp;').append($('<sup>', {
										text: 		tags[i][1].length
									}));
							}
							
							t.append(a);
						}
						
						// Adding toggle button if list is too high
						if (t.outerHeight() > 175) {
							addToggleButton(t);
						}
						
						// Adding search box
						if (settings.tagCloud['useSearch']) {
							prepareTagSearch(tc.parent());
						}
						
					},
				
				/********************************************************
				 *
				 *						Search new images
				 *
				 ********************************************************/
				
				getSearchNewOptions = function(days) {
					
						return settings.hasOwnProperty('searchNew')?
							{
								reference:		settings.searchNew['reference'] || 'dateTaken',
								depth:			settings.searchNew['depth'] || 'subfolders',
								start:			nows / ONEDAY_S - days
							}
							:
							{
								reference:		'dateTaken',
								depth:			'subfolders',
								start:			nows / ONEDAY_S - days
							};
						
					},
				
				// Search for new images
				
				getDateRangeOptions = function(range) {
						var m = range.match(/^([amt])?([cts])?\:?(\d+)\-?(\d+)?/i),
							o = {
									range:		0,
									reference:	'dateTaken',
									depth:		'current'
								};
								
						if (m) {
							if (m[1]) {
								// Reference
								if (m[1] === 'a') {
									o.added = 'added';
								} else if (m[1] === 'm') {
									o.added = 'fileModified';
								} else if (m[1] === 't') {
									o.added = 'dateTaken';
								}
							}
							
							if (m[2]) {
								// Depth
								if (m[2] === 'c') {
									o.depth = 'current';
								} else if (m[2] === 't') {
									o.depth = 'tree';
								} else if (m[2] === 's') {
									o.depth = 'subfolders';
								}
							}
							
							if (m[4]) {
								// Range
								o.start = parseInt(m[3]) || 0;
								o.end = parseInt(m[4]);
							} else if (m[3]) {
								// Single day
								o.start = parseInt(m[3]);
								o.range = 1;
							}
						}
						
						return o;
					},
				
				getDateReference = function(r) {
					
						if (r === 'dateTaken') { 
							return text.imagesTaken;
						} else if (r === 'fileModified') {
							return text.imagesModified;
						}
						
						return text.imagesAdded;
					},
					
				getDateRangeTitle = function(o) {
						var t1 = getDateReference(o['reference']),
							t2,
							start,
							end,  
							today = Math.floor(new Date() / ONEDAY_MS),
							date = function(d) {
									return new Date(d * ONEDAY_MS).toLocaleDateString(LOCALE);
								};
						
						if (!o.hasOwnProperty('end') && !o.hasOwnProperty('start')) {
							t2 = getTimespan(o.range);
						} else if (!o.hasOwnProperty('end')) {
							if (o.hasOwnProperty('start')) {
								if (o.hasOwnProperty('range')) {
									if (o['range'] > 1) {
										t2 = text.betweenDays.template(date(o.start), date(o.start + o.range));
									} else {
										t2 = text.onDay.template(date(o.start));
									}
								} else {
									t2 = getTimespan(nows / ONEDAY_S - o.start);
									//t2 = text.afterDay.template(date(o.start));
								}
							}
						} else {
							t2 = text.beforeDay.template(date(o.end));
						}	
						
						return t1 + ' ' + t2;
					},
				
				getItemsByDate = function(options, ready) {
						var oly = prepareOverlay({
									className:		'search-results',
									icon:			'icon-calendar',
									title:			getDateRangeTitle(options)
								});
						
						album.collectByDate($.extend(options, {
									ready:		function() {
													
														appendToOverlayTitle($('<small>', {
																text: 	this.length? text.foundNTimes.replace('{0}', this.length) : text.notFound
															}));
														
														renderOverlay(oly, this, ready);
													}
							}));
					},
				
				// Search date range
				
				searchDateRange = function(range) {
					
						if (range) {
							var options = getDateRangeOptions(range);
							
							getItemsByDate(options, function() {
									setHash({
											date:		range
										}, getDateRangeTitle(options));
									updateShares();
								});
						}
						
						return false;
					},
				
				// Search new images
				
				searchNew = function(days) {
						
						if (days) {
							
							getItemsByDate($.extend(settings.searchNew, { 
									start: 		nows / ONEDAY_S - days 
								}), function() {
									setHash({
											newimages:		days
										}, getDateReference(settings.searchNew['reference']));
									updateShares();
								});
						}
						
						return false;
					},
				
				// Prepare Search new functionality
				
				prepareSearchNew = function(f) {
					
						if (!f.length || !album) {
							return;
						}
						
						// Setting up defaults if something is missing
						
						settings.searchNew = $.extend({
								days: 				'3,30,90',
								depth: 				'folder',
								reference: 			'dateTaken',
								sinceLastVisit:		true
							}, settings.searchNew);
						
						var	days = settings.searchNew.days.split(',');
						
						f.append($('<label>', {
								text: 	getDateReference(settings.searchNew.reference)
							}));
								
						f = $('<div>', {
								'class':	'buttons'
							}).appendTo(f);
						
						for (var i = 0; i < days.length; i++) {
							f.append($('<a>', {
									'class': 	'small button',
									text: 		getTimespan(days[i])
								})
								.data('days', parseInt(days[i], 10))
								.on('click', function() {
										searchNew($(this).data('days'));
									}));
						}
								
						if (settings.searchNew.sinceLastVisit) {
							
							var start = $.cookie('lastVisit'),
								now = Math.round(new Date() / 1000);
							
							if (start && start < (now - ONEDAY_S)) {
								f.append($('<a>', {
										'class': 	'small button',
										text: 		text.sinceMyLastVisit
									})
									.on('click', function() {
											searchNew((now - start) / ONEDAY_S);
										}));
							}
							
							// Storing last visit date for 10 years
							$.cookie('lastVisit', now, 315360000);
						}
						
					},
								
				/********************************************************
				 *
				 *					Filtering & Sort
				 *
				 ********************************************************/
				
				// Dates are objects
				
				getDate = function(item, name) {
						if (item.hasOwnProperty(J.DATES)) {
							if (name === 'addedDate') {
								return item[J.DATES]['added'];
							} else if (name === 'originalDate') {
								return item[J.DATES]['dateTaken'];
							} else {
								return item[J.DATES]['fileModified'];
							}
						}
						
						return null;
					},
					
				// Some variables get remapped by the skin
				
				getMappedVal = function(item, name) {
					
						if (item.hasOwnProperty(name)) {
							// Generic case
							return item[name];
						}
						
						return null;
					},
					
				// Do filtering on current folder
				
				doFilter = function(e, ready) {
						var form = $(e.target).closest('form'),
							items = getAllItems(),
							cnt = items.length,
							sbd = {},
							tx = [];
						
						form.find('select:visible:not(:disabled),input[type=range],input[type=date]').each(function() {
								var name = this.name,
									label = form.find('label[for=' + name + ']').text(),
									type = $(this).data('type'),
									value = $(this).val(),
									display = function(val) {
											if (name ===  'rating') {
												return Math.round(val);
											} else if (name === 'fileSize') {
												return niceByte(val);
											} else if (type === 'date') {
												return (new Date(val * 1000)).toLocaleDateString(LOCALE);
											} else if (type === 'numeric') {
												if ($(this).data('unit') === 'seconds') {
													return niceTime(val);
												} else if ($(this).data('unit') === 'byte') {
													return niceByte(val);
												} else if (val >= 10) {
													return Math.round(val);
												} else if (val >= 1) {
													return val.toFixed(1);
												} else if (val >= 0.1) {
													return val.toFixed(2);
												}
												return val;
											}
											return toCurrency(val, currency);
										};
									
								if (type === 'numeric' || type === 'date' || type === 'financial') {
										
									var low = $(this).data('low'),
										high = $(this).data('high');
										
									if (low > this.min || high < this.max) {
										// Only if the range is narrower than the whole range
										
										if (name.endsWith('Date')) {
											// Date
											for (var i = items.length - 1, val; i >= 0; i--) {
												if ((val = getDate(items[i], name)) !== null && 
													(val >= low && val <= high)) {
													continue;
												}
												items.splice(i, 1);
											}
										} else if (JCAMERAFIELDS.indexOf(name) !== -1) {
											// Camera data fields
											if (name === 'exposureTime') {
												for (var i = items.length - 1, val; i >= 0; i--) {
													if (items[i].hasOwnProperty(J.CAMERA) &&
														items[i][J.CAMERA].hasOwnProperty(name) &&
														(val = getExposure(items[i][J.CAMERA][name])) &&
														(val >= low && val <= high)) {
														continue;
													}
													items.splice(i, 1);
												}
											} else if (name.startsWith('focalLength')) {
												for (var i = items.length - 1, val; i >= 0; i--) {
													if (items[i].hasOwnProperty(J.CAMERA) &&
														items[i][J.CAMERA].hasOwnProperty(name) &&
														!isNaN(val = parseFloat(items[i][J.CAMERA][name])) && 
														(val >= low && val <= high)) {
														continue;
													}
													items.splice(i, 1);
												}
											} else {
												for (var i = items.length - 1, val; i >= 0; i--) {
													if (items[i].hasOwnProperty(J.CAMERA) &&
														items[i][J.CAMERA].hasOwnProperty(name) &&
														!isNaN(val = parseFloat(items[i][J.CAMERA][name])) &&
														(val >= low && val <= high)) {
														continue;
													}
													items.splice(i, 1);
												}
											}
										} else if (name === 'rating') {
											// Rating
											for (var i = items.length - 1, val, r = settings.visitorRating? J.VISITORRATING : J.RATING; i >= 0; i--) {
												val = items[i][r] || 0;
												if (val >= low && val <= high) {
													continue;
												}
												items.splice(i, 1);
											}
										} else {
											// Generic
											for (var i = items.length - 1, val; i >= 0; i--) {
												if (!isNaN(val = parseFloat(items[i][name])) && 
													(val >= low && val <= high)) {
													continue;
												}
												items.splice(i, 1);
											}
										}
									
										tx.push(label + '=' + display(low) + '&ndash;' + display(high));
									}
									
								} else if (this.nodeName === 'SELECT' && value !== null && value.length) {
									
									// Only if at least one option is selected
																		
									if (!Array.isArray(value)) {
										value = [ value ];
									}
									
									if (JCAMERAFIELDS.indexOf(name) !== -1) {
										// Camera data :: no arrays expected
										for (var i = items.length - 1, val; i >= 0; i--) {
											if (items[i].hasOwnProperty(J.CAMERA) &&
												items[i][J.CAMERA].hasOwnProperty(name) &&
												(val = items[i][J.CAMERA][name]) !== null &&
												value.includes(val)) {
												continue;
											}
											items.splice(i, 1);
										}
										
									} else {
										// Generic case
										for (var i = items.length - 1, found, val; i >= 0; i--) {
											if (items[i].hasOwnProperty(name)) {
												val = items[i][name];
												if (Array.isArray(val)) {
													// Array e.g. keywords
													found = false;
													for (var j = 0; j < value.length; j++) {
														if (val.includes(value[j])) {
															//console.log(items[i]['name'] + ': ' + val);  
															found = true;
															break;
														}
													}
													if (found) {
														continue;
													}
												} else if (value.includes(val)) {
													continue;
												}
											}
											items.splice(i, 1);
										}
									}
									
									sbd[name] = value;
									
									// Header text addition
									if (this['multiple']) {
										tx.push(name + '=[' + value.join(',') + ']');  
									} else {
										tx.push(label + '=' + value);  
									}
								}
							});
						
						if (items.length < cnt) {
							
							// Only show overlay if at least one thumb was filtered out
							
							var oly = prepareOverlay({
										className:	'filtered-items',
										icon:		'icon-filter',
										title:		text.results
									});
								
							if (!isEmpty(sbd)) {
								oly.data('sbd', sbd);
							}
							
							setTimeout(function() {
									
									appendToOverlayTitle($('<small>', {
											html: 		tx.join(', ') + ': ' + text.foundNTimes.replace('{0}', items.length)
										}));
									
									oly.on('overlayRemoved', resetBaseFilters);
									
									renderOverlay(oly, items, ready);
									
								}, 50);
						}
						
						return false;
					},
				
				// Do sorting on current set of thumbs
				
				doSort = function(e, ready) {
						var el = this.selectedOptions[0],
							name = this.value;
						
						if (!el || !name) {
							return true;
						}
						
						var items = getAllItems(),
							cnt = items.length,
							data = $(el).data('sort'),
							type = data['type'],
							asc = data['ascending'],
							tx = [],
							v1,
							v2;
							
						if (type === 'numeric' || type === 'financial' || type === 'date') {
							// Number
							
							if (name.endsWith('Date')) {
								// Date
								items = items.sort(function(i1, i2) {
										v1 = getDate(i1, name);
										v2 = getDate(i2, name);
										return asc? (v1 || 0) - (v2 || 0)
													:
													(v2 || 0) - (v1 || 0);
									});
								
							} else if (JCAMERAFIELDS.indexOf(name) !== -1) {
								// Camera data
								if (name === 'exposureTime') {
									items = items.sort(function(i1, i2) {
											v1 = (i1.hasOwnProperty(J.CAMERA) && i1[J.CAMERA].hasOwnProperty(name))? 
													i1[J.CAMERA][name] : 0;
											v2 = (i2.hasOwnProperty(J.CAMERA) && i2[J.CAMERA].hasOwnProperty(name))? 
													i2[J.CAMERA][name] : 0;
											return asc? (getExposure(v1 || 0) - getExposure(v2 || 0)) 
														: 
														(getExposure(v2 || 0) - getExposure(v1 || 0));
										});
								} else {
									items = items.sort(function(i1, i2) {
											v1 = (i1.hasOwnProperty(J.CAMERA) && i1[J.CAMERA].hasOwnProperty(name))? 
													parseFloat(i1[J.CAMERA][name]) : 0;
											v2 = (i2.hasOwnProperty(J.CAMERA) && i2[J.CAMERA].hasOwnProperty(name))? 
													parseFloat(i2[J.CAMERA][name]) : 0;
											return asc? ((v1 || 0) - (v2 || 0)) 
														: 
														((v2 || 0) - (v1 || 0));
										});
								}
								
							} else if (name === 'rating') {
								// Rating
								var r = settings.visitorRating? J.VISITORRATING : J.RATING;
								items = items.sort(function(i1, i2) {
										return asc? (i1[r] || 0) - (i2[r] || 0)
													:
													(i2[r] || 0) - (i1[r] || 0);
									});
								
							} else {
								// Every other numeric
								items = items.sort(function(i1, i2) {
										if (i1.hasOwnProperty(name) && i2.hasOwnProperty(name)) {
											return asc?	parseFloat(i1[name]) - parseFloat(i2[name])
														:
														parseFloat(i2[name]) - parseFloat(i1[name]);
										}
										return asc? (i1.hasOwnProperty(name)? 1 : 0)
													:
													(i2.hasOwnProperty(name)? -1 : 0);
									});
							}
							
						} else {
							// String
							
							items = items.sort(function(i1, i2) {
								if (i1.hasOwnProperty(name) && i2.hasOwnProperty(name)) {
									return asc?	(i1[name] + '').localeCompare(i2[name] + '')
												:
												(i2[name] + '').localeCompare(i1[name] + '');
								}
								return asc? (i1.hasOwnProperty(name)? 1 : 0)
											:
											(i2.hasOwnProperty(name)? -1 : 0);
							});
							
						}
						
						var oly = prepareOverlay({
									className:	'sorted-items',
									icon:		'icon-ordering',
									title:		text.sortedBy + '<small>' + data.label + ' (' + (data.ascending? text.ascending : text.descending) + ')</small>' 
								});
							
						setTimeout(function() {
								
								oly.on('overlayRemoved', resetBaseSort);
								
								renderOverlay(oly, items, ready);
							
							}, 50);
						
						return false;
					},
					
				// Read filter ranges from current overlay
				
				getFilterRanges = function(items, filter) {
						
						if (!filter || !items.length) {
							return;
						}
						
						var setMinMax = function(f, val) {
									if (f.hasOwnProperty('min') && f.hasOwnProperty('min')) {
										if (f.min > val) {
											f.min = val;
										} else if (f.max < val) {
											f.max = val;
										}
									} else {
										f.min = f.max = val;
									}
								};
								
						for (var name in filter) {
							
							var f = filter[name], 
								type = f['type'];
							
							if (type === 'numeric' || type === 'date' || type === 'financial') {
								
								if (name.endsWith('Date')) {
									// Date
									for (var i = items.length - 1, val; i >= 0; i--) {
										if ((val = getDate(items[i], name)) !== null) {
											setMinMax(f, val);
										}
									}
								} else if (JCAMERAFIELDS.indexOf(name) !== -1) {
									// Camera data fields
									if (name === 'exposureTime') {
										for (var i = items.length - 1, val; i >= 0; i--) {
											if (items[i].hasOwnProperty(J.CAMERA) &&
												items[i][J.CAMERA].hasOwnProperty(name) &&
												(val = getExposure(items[i][J.CAMERA][name]))) {
												setMinMax(f, val);
											}
										}
									} else if (name.startsWith('focalLength')) {
										for (var i = items.length - 1, val; i >= 0; i--) {
											if (items[i].hasOwnProperty(J.CAMERA) &&
												items[i][J.CAMERA].hasOwnProperty(name) &&
												!isNaN(val = parseFloat(items[i][J.CAMERA][name]))) {
												setMinMax(f, val);
											}
										}
									} else {
										for (var i = items.length - 1, val; i >= 0; i--) {
											if (items[i].hasOwnProperty(J.CAMERA) &&
												items[i][J.CAMERA].hasOwnProperty(name) &&
												!isNaN(val = parseFloat(items[i][J.CAMERA][name]))) {
												setMinMax(f, val);
											}
										}
									}
								} else if (name === 'rating') {
									// Rating
									for (var i = items.length - 1, val, r = settings.visitorRating? J.VISITORRATING : J.RATING; i >= 0; i--) {
										val = items[i][r] || 0;
										setMinMax(f, val);
									}
								} else {
									// Generic
									for (var i = items.length - 1, val; i >= 0; i--) {
										if (!isNaN(val = parseFloat(items[i][name]))) {
											setMinMax(f, val);
										}
									}
								}
								
							} else {
								// String
								if (!f.hasOwnProperty('values')) {
									f.values = [];
								}
								
								if (JCAMERAFIELDS.indexOf(name) !== -1) {
									// Camera data :: no arrays expected
									for (var i = 0, val; i < items.length; i++) {
										if (items[i].hasOwnProperty(J.CAMERA) &&
											items[i][J.CAMERA].hasOwnProperty(name)) {
											pushNew(f.values, (items[i][J.CAMERA][name] + '').trim());
										}
									}
								} else {
									for (var i = 0, val; i < items.length; i++) {
										if (items[i].hasOwnProperty(name)) {
											pushNew(f.values, items[i][name]);
										}
									}
									if (f.values.length > 1) {
										f.values = f.values.sort(function(a, b) { return a.localeCompare(b); });
									}
								}
							}
						}
						
						return filter;
					},
					
				// Enabling / disabling the reset button
				
				enableResetBtn = function(e) {
						var form = (typeof e === UNDEF || e.hasOwnProperty('target'))? $(this).closest('form') : e;
						form.find('button[type=reset]').removeClass('disabled');
					},
					
				disableResetBtn = function(e) {
						var form = (typeof e === UNDEF || e.hasOwnProperty('target'))? $(this).closest('form') : e;
						form.find('button[type=reset]').addClass('disabled');
					},
					
				// Reseting filters to defaults
				
				_resetFilters = function(form) {
					
						form.find('select').each(function() {
								$(this).prop('selectedIndex', this['multiple']? -1 : 0);
							});
						
						form.find('input[type=range]').each(function() {
								$(this).trigger('resetRange');
							});
						
						disableResetBtn(form);
						
					},
					
				// Reset filters after a form elemnt changed
				
				resetFilters = function(e) {
						_resetFilters($(e.target).closest('form'));
						return false;
					},
					
				// Reseting base filters to defaults
				
				resetBaseFilters = function() {
						resetBaseSort();
						_resetFilters($('#filters form.filter'));
					},
					
				// Reseting base filters to defaults
				
				resetBaseSort = function() {
						$('#filters form.sort select').prop('selectedIndex', 0);
					},
					
				// Preparing filters / sort
				
				prepareFilterBox = function(baseSet) {
					
						var items = getAllItems(),
							sbd = getOverlay().data('sbd'),
							filters = album.getPropertyObject(album.getCurrentFolder(), J.FILTERS, true),
							sort = album.getPropertyObject(album.getCurrentFolder(), J.SORT, true),
							filterSection = $('#filters');
							
						if ((!filters && !sort) || !items.length) {
							if (baseSet) {
								filterSection.hide();
							}
							
							return;
						}
						
						var	box = filterSection.find('.' + settings.boxClass),
							filterForm = box.find('form.filter'),
							sortForm = box.find('form.sort'),
							group,
							inp,
							lbl,
							type,
							buttons,
							btn;
							
						if (!baseSet) {
							// Preparing filtering on overlay
							
							box = $('<div>', {
									'class':		settings.contClass + ' ' + settings.boxClass + ' small-column',
									id:				'filters_' + getOverlayNS()
								}).appendTo(filterSection);
								
							box.append($('<h4>', {
									'class':		'icon-' + (filters? 'filter' : 'ordering'),
									text:			filterSection.find('h4').text()
								}));	
							
							if (filters) {
								
								filterForm = $('<form>', {
										'class':		'filter'
									}).appendTo(box);
							}
							
							if (sort) {
								sortForm = $('<form>', {
										'class':		'sort'
									}).appendTo(box);
							}
							
							// Attaching box to current overlay
							attachToOverlay(box);
							
						} else {
							
							box.addClass(settings.hideOnOverlayClass);
						}	
						
						buttons = $('<div>', {
								'class':	'buttons'
							}).appendTo(filterForm);
						
						if (filters) {
							
							// Getting filter range
							filters = getFilterRanges(items, filters);
							
							var m;
							
							for (var f in filters) {
								
								type = filters[f]['type'];
								
								group = $('<div>', {
										'class':		'group ' + ((type === 'multiple_string' || type === 'single_string')? 'select' : 'range') 
									}).insertBefore(buttons);
								
								if (filters[f]['label']) {
									lbl = $('<label>', {
											'for':			f,
											text:			filters[f]['label']
										}).appendTo(group);
								}
									
								if (type === 'multiple_string' || type === 'single_string') {
									
									if (f === 'rating') {
										filters[f].values = [ text.noRating, 1, 2, 3, 4, 5 ];
									}
									
									inp = $('<select>', {
											id:				f,
											name:			f
										}).appendTo(group);
									
									if (type === 'multiple_string') {
										inp.attr('multiple', true);
										inp.addTooltip(text.multipleSelectHint, { delay: 600, stay: 3000 });
									}
									
									if (!filters[f].hasOwnProperty('values') || !filters[f].values.length) { // || filters[f].values.length < 2) {
										
										group.remove();
										
									} else {
									
										// Select hint if more than one option
										if (baseSet || type === 'single_string' && filters[f].values.length > 1) {
											inp.prepend($('<option>', {
													text:			text.select,		// + ' ' + filters[f]['label'],
													disabled:		'',
													selected:		''
												}));
										}
										
										// Adding options
										if (filters[f].values) {
											for (var i = 0; i < filters[f].values.length; i++) {
												inp.append($('<option>', {
														text:			filters[f].values[i]
													}));
											}
										}
										
										if (!baseSet) {
											
											if (filters[f].values.length === 1) {
												// Disabling box if there's only one option
												inp.prop('disabled', true);
												inp.children().first().prop('selected', true);
											} else if (sbd && sbd.hasOwnProperty(f)) {
												// Preselect the previous filtering value(s)
												inp.children('option:enabled').each(function(i) {
														if (sbd[f].includes(filters[f].values[i])) {
															$(this).prop('selected', true);
														}
													});
											}
										}
										
										inp.on('change', enableResetBtn);
									}
									
								} else {
									
									inp = $('<input>', {
											id:			f,
											name:		f,
											type:		'range'
										})
										.data('type', type)
										.prependTo(group);
									
									if (f === 'rating') {
										filters[f]['step'] = 1;
										filters[f]['min'] = 0;
										filters[f]['max'] = 5;
									}
									
									if (!filters[f].hasOwnProperty('min') || !filters[f].hasOwnProperty('max') || filters[f]['min'] === filters[f]['max']) {
										
										group.remove();
										
									} else {
										
										if (type === 'financial' && (m = filters[f]['label'].match(/\S+\s+[\(\[](\S+)[\)\]]/))) {
										
											filters[f]['currency'] = m[1];
										
										} else if (type === 'numeric') {
											
											if ('exposureTime.shutterSpeed'.indexOf(f) >= 0) {
												filters[f]['unit'] = 'seconds';
											} else if (f === 'fileSize') {
												filters[f]['unit'] = 'byte';
												filters[f]['step'] = 1;
											}
										}
										filters[f].onChanged = enableResetBtn;
										filters[f].onReseted = disableResetBtn;
										
										inp.rangeSlider(filters[f]);
									}
								}
									
								inp.attr('data-type', type || 'single_string');
							}
							
							// Adding buttons
							
							if ((inp = filterForm.find('input,select')).length) {
								
								filterForm.data('single', inp.length === 1 && inp[0].nodeName === 'SELECT' && !inp.attr('multiple'));
								
								if (!(baseSet && filterForm.data('single'))) {
									// Reset button
									btn = $('<button>', {
											type:			'reset',
											'class':		(baseSet? 'disabled secondary' : 'alert') + ' button',
											text:			' ' + text.reset
										}).appendTo(buttons);
									
										if (baseSet) {
											btn.on('click.' + ns, resetFilters);
										} else {
											btn.on('click.' + ns, function() {
													resetBaseFilters();
													removeOverlay();
												});
										}
								}
									
								if (filterForm.data('single')) {
									// Single combo box -> Update instantly on change
									inp.on('change.' + ns, doFilter);
									
								} else {
									// Multiple inputs or not single combo box -> Use 'Search' button
									btn = filterForm.find('button[type=submit]');
									
									if (!btn.length) {
										btn = $('<button>', {
												type:			'submit',
												'class':		'button',
												text:			' ' + text.search
											}).appendTo(buttons);
									}
									
									btn.on('click.' + ns, doFilter);
								}
								
								filterForm.on('submit.' + ns, doFilter);
							}
						}
						
						if (buttons.is(':empty')) {
							buttons.remove();
						}
						
						if (sort) {
														
							lbl = $('<label>', {
									'class':		'icon-ordering',
									'for':			'sortby',
									text:			' '
								}).appendTo(sortForm);
								
							inp = $('<select>', {
									'class':		'small',
									id:				'sortby',
									name:			'sortby'
								}).appendTo(sortForm);
							
							inp.prepend($('<option>', {
									text:			text.sortBy,
									disabled:		'',
									selected:		''
								}));
							
							for (var s in sort) {
								
								inp.append($('<option>', {
										html:			'&#8613; ' + sort[s].label,
										value:			s
									}).data('sort', {
										type:			sort[s].type,
										label:			sort[s].label,
										ascending:		true
									}));
								
								inp.append($('<option>', {
										html:			'&#8615; ' + sort[s].label,
										value:			s
									}).data('sort', {
										type:			sort[s].type,
										label:			sort[s].label,
										asc:			false
									}));
							}
						
							inp.on('change.' + ns, doSort);
						}
						
						if ((!sortForm || !sortForm.length || sortForm.is(':empty')) && 
							(!filterForm || !filterForm.length || filterForm.is(':empty'))) {
							filterSection.hide();
						}
					},
				
				/********************************************************
				 *
				 *						Controls
				 *
				 ********************************************************/
				
				// Initializing ZIP download button
				
				initZip = function() {
				
						panels.find('.zip-cont a.button').each(function() {
								var btn = $(this);
								
								getFileSize(btn.attr('href'), function(size) {
										if (size) {
											btn.append('<span class="badge">' + niceByte(size) + '</span>');
										}
									});
							});
					},

				/********************************************************
				 *
				 *			Initialization of misc functions
				 *
				 ********************************************************/
										
				// Right click blocking
				
				avoidRightClick = function() {
						getImages().add(getFolders()).on('contextmenu', 'img', function(e) {
								e.preventDefault()
								return false;
							});
					},
					
				// Initializing background music
				
				prepareBackgroundMusic = function(player) {
					
						if (player.length) {
							player.audioPlayer({
									rootPath: 	settings.rootPath
								});
						}
					},
					
				// Updating the share buttons after a state change
				
				updateShares = function() {
					
						if (settings.hasOwnProperty('share')) {
							
							if (getLightbox().is(':visible')) {
								// Lightbox: sharing actual image
								var item = getCurrentItem();
								
								if (item) {
									$(settings.share.hook).trigger('updateLinks', {
											sites: 			settings.share,
											title: 			item[J.TITLE] || item[J.NAME].stripExt().replace(/[-_]/g, ' ').capitalize(),
											description: 	item[J.COMMENT] || '',
											image: 			album.getOptimalImagePath(item, [ 1200, 800 ]),
											href: 			album.getAbsolutePath(item)
										});
								}
							} else {
								// Index page: reset to page sharing
								$(settings.share.hook).trigger('updateLinks', { href: window.location.href });
							}
						}	
					},
				
				// Initializing Social sharing
				
				prepareShares = function(opt) {
						
						$.fn.renderShares.defaults.buttonTheme = opt['buttonTheme'] || 'dark';
						$.fn.renderShares.defaults.indexName = settings.indexName;
						$.fn.renderShares.defaults.facebookAppId = opt['facebookAppId'] || '';
						$.fn.renderShares.defaults.albumTitle = album.getAlbumTitle(); 
						
						if (opt.hasOwnProperty('emailSubject')) {
							$.fn.renderShares.defaults.emailSubject = opt.emailSubject;
						}
						
						if (opt.hasOwnProperty('emailBody')) {
							$.fn.renderShares.defaults.emailBody = opt.emailBody;
						}
						
						$(opt['hook']).renderShares(opt);
						
						if (opt.hasOwnProperty('sites')) {
							$.fn.lightbox.defaults.share = opt.sites;
							$.fn.lightbox.defaults.shareHook = opt['hook'];
						}
					},
				
				// Reading out map locations
				
				getMarkers = function(images) {
						// Reading out map locations
						var markers = [];
						
						for (var i = 0, l; i < images.length; i++) {
							if (images[i] && (l = album.getLocation(images[i]))) {
								markers.push({
									title:		[((i + 1) + '.'), images[i]['title'], images[i]['comment']].filter(Boolean).join(' '),
									pos: 		l,
									link:		images[i].path
								});
							}
						}
						
						return markers;
					},
						
				// Preparing map
				
				prepareMap = function(images, opt) {
					
						if (images && images.length) {
							opt.markers = getMarkers(images);
						}
							
						if (opt.markers.length || opt.hasOwnProperty('overlay')) {
							var	mapRoot = getMapRoot(true);
						
							if (!mapRoot.data('inited')) {
								// Add map only once
								opt.resPath = settings.resPath;
								
								opt.onMarkerClick = function() {
										if (this.hasOwnProperty('link')) {
											var el = getCardByName(this.link);
												
											if (el.length) {
												if (isFullscreen()) {
													exitFullscreen();
												}
												getOverlay().trigger('setactivecard', el);
												lightboxLoad(el);
											}
										}
									};
									
								if (opt.hasOwnProperty('apiKey')) {
									// Google Maps
									opt.streetViewControl = opt.typeControl = (window.innerWidth >= 640) && (window.innerHeight >= 640);
								}
									
								// Unhide Map section
								mapRoot.closest('section').removeClass('hidden');
								// Calling Map library
								mapRoot.addMap(opt);
								// Set inited flag
								mapRoot.data('inited', true);
							}
							
							main.removeClass(settings.noMapClass).addClass(settings.hasMapClass);
						} else {
							main.removeClass(settings.hasMapClass).addClass(settings.noMapClass);
						}
					},
				
				// Preparing shopping cart
				
				prepareShoppingCart = function() {
					
						// Initializing the cart
						if (shopRoot.length) {
							// Using the paypal cart's currency as global currency
							currency = album.getCurrency();
							
							// Set rangeslider's default currency
							if ($.fn.hasOwnProperty('rangeSlider') && currency) {
								$.fn.rangeSlider.defaults.currency = currency;
							}
						
							// Calling Paypal module
							shopRoot.paypal(album, {
									resPath:				settings.resPath,
									relPath:				settings.relPath,
									rootPath:				settings.rootPath,
									currency:				currency,
									getSelected: 			getSelectedItems,
									selectNone:				function() {
																getBaseOverlay().trigger('selectNone');
															},
									getCurrent:				getCurrentItem,
									useFloatButton:			!(shopRoot.parents('.shop-cont[data-sticky]').length || document.body.clientWidth >= 640)
								});
							
							// "Add selected" button
							shopRoot.find('.' + settings.addSelectedClass).on('click', function() {
									shopRoot.trigger('addItems', getSelectedItems());
									getBaseOverlay().trigger('selectNone');
									return false;
								});
							
							// Ensuring lightbox has 'shop' property
							if (!$.fn.lightbox.defaults.hasOwnProperty('shop')) {
								$.fn.lightbox.defaults.shop = {};
							}
							
							// Setting lightbox's shop root variable
							$.fn.lightbox.defaults.shop.root = shopRoot;
						}
					
					},
					
				// Preparing feedback
				
				prepareFeedback = function() {
					
						// Initializing the feedback cart
						if (feedbackRoot.length) {
								
							// Calling Feedback module
							feedbackRoot.feedback(album, $.extend({}, settings.feedback, {
									resPath:				settings.resPath,
									relPath:				settings.relPath,
									rootPath:				settings.rootPath,
									getSelected:			getSelectedItems,
									selectNone:				function() {
																getBaseOverlay().trigger('selectNone');
															},
									getCurrent:				getCurrentItem,
									useFloatButton:			!(feedbackRoot.parents('.feedback-cont[data-sticky]').length || document.body.clientWidth >= 640)			
								}));
							
							// "Add selected" button
							feedbackRoot.find('.' + settings.addSelectedClass).on('click', function() {
									feedbackRoot.trigger('addItems', getSelectedItems());
									getBaseOverlay().trigger('selectNone');
									return false;
								});
							
							// Ensuring lightbox has 'feedback' property
							if (!$.fn.lightbox.defaults.hasOwnProperty('feedback')) {
								$.fn.lightbox.defaults.feedback = {};
							}
							// Setting lightbox's feedback root variable
							$.fn.lightbox.defaults.feedback.root = feedbackRoot;
							$.fn.lightbox.defaults.feedbackBtnLabel = settings.feedback['viewBtnLabel'];
							$.fn.lightbox.defaults.feedbackAddBtnLabel = settings.feedback['addBtnLabel'];
							$.fn.lightbox.defaults.feedbackBtnTooltip = settings.feedback['addBtnTooltip'];
						}
					},
					
				/********************************************************
				 *
				 *					Index page initialization
				 *
				 ********************************************************/
				
				prepareIndex = function() {
					
						if (!album) {
							return;
						}
						
						var images = album.getImages();
						
						// Removing thumbnail links right away
						getImages().each(function() {
								var a = $(this).find('a');
								a.data('href', a.attr('href'));
								a.attr('href', '');
							});
			
						// Preapring folders
						prepareFolders();
			
						// Shopping cart
						if ($.fn.hasOwnProperty('paypal') && album.hasShop()) {
							shopRoot = $('#shop-root');
							shopBox = $('#shop .box');
							prepareShoppingCart();
						}
						
						// Feedback tool
						if (settings.hasOwnProperty('feedback')) {
							feedbackRoot = $('#feedback-root');
							feedbackBox = $('#feedback .box');
							prepareFeedback();
						}
						
						// "Select all" button
						shopBox.add(feedbackBox).find('.' + settings.selectAllClass).on('click', function() {
								getOverlay().trigger('selectAll');
								return false;
							});
						
						// "Select none" button
						shopBox.add(feedbackBox).find('.' + settings.selectNoneClass).on('click', function() {
								getOverlay().trigger('selectNone');
								return false;
							});
								
						// Rendering thumbnails
						if (images.length) {
							
							// Setting has-images class
							main.data('baseHasImages', true);
							toggleHasImages(true);
							
							// Preparing images on the base overlay
							getBaseOverlay().renderImages(images, prepareImages, true);
							
							// Start slideshow
							$(settings.startSlideshowHook).on('click', startSlideshow);
							                                           
						} else {
							
							// Setting has-images class
							main.data('baseHasImages', false);
							toggleHasImages(false);
							
							// no thumbnails, only folders
							if (window.location.hash) {
								// we should check the search / tags / newimages filter
								stateChange();
							}
							
							$(settings.startSlideshowHook).hide();
						}
												
						// Collecting tags if gathered only from this folder
						if (settings.hasOwnProperty('tagCloud') && settings.tagCloud['depth'] === 'current') {
							album.collectTags($.extend(settings.tagCloud, {
									ready:		function() {
														prepareTagCloud(this);
													}
								}));
						}
						
						// Search 
						if (settings.hasOwnProperty('search')) {
							prepareSearch($(settings.search.hook));
						}
						
						// Search new
						if (settings.hasOwnProperty('searchNew')) {
							prepareSearchNew($(settings.searchNew.hook));
						}
						
						// Preparing share buttons
						if (settings.hasOwnProperty('share')) {
							prepareShares(settings.share);
						}
						
						var slider = $('[data-slider]');
						
						if (slider.length) {
							var sls = slider.data('slider') || {},
								dim = [ slider.innerWidth(), slider.innerHeight() ];
							
							if (sls.hasOwnProperty('names')) {
								
								for (var i = 0; i < sls.names.length; i++) {
									sls.names[i] = [ settings.relPath, sls.names[i] ].filter(Boolean).join('/');
								}
								
								// Selected images
								album.collectByPath({
										folder:		settings.relPath,
										paths:		sls.names,
										levels:		0,
										ready:		function() {
															sls.src = [];
															for (var i in this) {
																sls.src.push(album.getOptimalImagePath(this[i], dim, true, true));
															}
															slider.easySlider(sls);
														}
									});
								
							} else {
								
								album.collectNItem({
										folder:		settings.relPath,
										max:		50,
										quick:		true,
										levels:		2,
										include:	'images',
										sortBy:		'random',
										arRange:	[ Math.min(1, 0.5 * dim[0] / dim[1]), Math.max(1, 2 * dim[0] / dim[1]) ],
										ready:		function() {
															sls.src = [];
															for (var i in this) {
																sls.src.push(album.getOptimalImagePath(this[i], dim, true, true));
															}
															slider.easySlider(sls);
														}
									});
							}
						}
						
						// Adding map on embedded custom pages
						var customMap = $('.pages-cont [data-map]');
						
						if (customMap.length && settings.hasOwnProperty('map')) {
							customMap.addMap($.extend(settings.map, customMap.data('map')));
						}
						
						// Custom query (eg. new images sample box)
						var queryTarget = $('[data-jalbum-query]');
						
						if (queryTarget.length) {
							// Reding settings from data attribute
							var qsettings = queryTarget.data('jalbum-query');
							
							if (qsettings.hasOwnProperty('fn')) {
								// Calling the jalbum.album function
								album[qsettings.fn]($.extend(qsettings, {
											ready:		function(opt) {
												
																for (var i = 0, a, item, mx = Math.min(this.length, 10); i < mx; i++) {
																	
																	item = this[i];
																	
																	a = $('<a>', {
																			href:	album.getLink(item)
																		}).appendTo(queryTarget);
																	
																	a.append($('<img>', {
																			src:		album.getThumbPath(item),
																			width:		item[J.THUMB][J.WIDTH],
																			height:		item[J.THUMB][J.HEIGHT],
																			alt:		item[J.TITLE] || item[J.NAME]
																		}));
																}
																
																if (this.length > 10) {
																	queryTarget.append($('<a>', {
																			'class':	'more',
																			href:		qsettings.linkMore,
																			text:		'+' + (this.length - 10)
																		}));
																}
																
																if (settings.rightClickProtect) {
																	queryTarget.on('contextmenu', 'img', function(e) {
																			e.preventDefault()
																			return false;
																		});
																}
															}
								
									}));
							}
						}
					},
				
				/********************************************************
				 *
				 *					Custom page initialization
				 *
				 ********************************************************/
				
				preparePage = function() {
						// Page :: initializing functions
						// Search 
						if (settings.hasOwnProperty('search')) {
							prepareSearch($(settings.search.hook));
						}
						
						// Preparing share buttons
						if (settings.hasOwnProperty('share')) {
							prepareShares(settings.share);
						}
						
						settings.pageName = window.location.pathname.getFile();
						
						var paths = [];
						
						getCards().each(function() {
								var a = $(this).find('.' + settings.thumbClass);
								a.data('href', a.attr('href'));
								a.attr('href', '');
								// Collecting paths from thumbnails
								paths.push($(this).data(J.PATH));
							});
						
						if (paths.length) {
						
							setOverlayNS();
							
							album.collectByPath({
									paths:		paths,
									ready:		function() {
														// Rendering thumbnails
														if (this.length) {
															
															// Setting has-images class
															main.data('baseHasImages', true);
															toggleHasImages(true);
															
															// Preparing images on the base overlay
															getBaseOverlay().renderImages(this, prepareImages, true);
															
														} else {
															// Setting has-images class
															toggleHasImages(false);
														}
													}
								});
						} else {
							var map = $('[data-map]');
							
							if (map.length) {
								prepareMap(null, extend(settings.map, map.data('map')));
							}
							// Setting has-images class
							main.data('baseHasImages', false);
							toggleHasImages(false);
						}
					},
				
				/********************************************************
				 *
				 *	 Preparing functions that need the full album tree
				 *
				 ********************************************************/
				
				prepareDeepFunctions = function() {
					
						// Collecting tags if gathered from deep tree
						if (settings.hasOwnProperty('tagCloud') && settings.tagCloud['depth'] !== 'current') {
							album.collectTags($.extend(settings.tagCloud, {
									ready:		function() {
														prepareTagCloud(this);
													}
								}));
						}
					},
					
				/********************************************************
				 *
				 *					Preparing folders
				 *
				 ********************************************************/
				
				// Marking new folders
				
				markFoldersNew = function() {
						
						if (!album) {
							return;
						}
						
						var folders = album.getFoldersAndWeblocs(),
							secs 	= settings.markNew['days'] * ONEDAY_S,
							s,
							d;
						
						getFolders().each(function(i) {
								
								if (i < folders.length) {
									d = folders[i][J.DATES];
									
									if (d) {
										d = d[settings.markNew['reference']] || (d[J.DATERANGE] && (d[J.DATERANGE][1] || d[J.DATERANGE][0]));
										
										if (d && (nows - d) < secs) {
											s = settings.markNew.text?
												$('<span>', {
														'class':	'new-image',
														text:		settings.markNew.text
													})
												:
												$('<span>', {
														'class':	'icon-new-fill new-image'
													});
											
											if (!TOUCHENABLED) {
												s.data('tooltip', (new Date(d * 1000)).toLocaleDateString(LOCALE)).addTooltip();
											}
											
											$(this).find('.' + settings.thumbClass).eq(0).append(s);
										}
									}
								}
							});
					},
				
				// Reading counts from a <span class="image">N images</span><span class="audio">N audios</span><span class="video">N videos</span>
				
				readCounts = function(el) {
						var cnt = {
									images:		0,
									audios:		0,
									videos:		0,
									others:		0
								};
								
						el.children().each(function(i) {
								var c = this.innerText.split(/\W/),
									t = this.classList[0];
								
								if (!t && c.length > 1) {
									switch (c[1]) {
										case text.image:
										case text.images:
											t = 'images';
											break;
										case text.audio:
										case text.audios:
											t = 'audios';
											break;
										case text.video:
										case text.videos:
											t = 'videos';
											break;
										case text.other:
										case text.others:
											t = 'others';
											break;
										default:
											t = 'images';
									}
								}
										
								cnt[t] += parseInt(c[0]) || 0;
							});
						
						return cnt;
					},
					
				// Calculates the difference between two count objects
				
				getCountDiff = function(c1, c2) {
						var diff = {
									images:		0,
									audios:		0,
									videos:		0,
									others:		0
								};
								
						for (var c in c1) {
							if (c2.hasOwnProperty(c) && c1[c] !== c2[c]) {
								diff[c] = c1[c] - c2[c];
							}
						}
						
						return diff;
					},
					
				// CHeck if any difference was found
				
				allZero = function(diff) {
					
						for (var c in diff) {
							if (diff[c] !== 0) {
								return false;
							}
						}
						
						return true;
					},
					
				// Creates HTML formatted counts: <span class="image">N images</span><span class="audio">N audios</span><span class="video">N videos</span>
				
				getFormattedCounts = function(cnt) {
						var c = '';
						
						if (cnt['images']) {
							c += '<span class="image">' + cnt['images'] + '&nbsp;' + translate(cnt['images'] > 1? 'images':'image') + '</span>';
						}
						if (cnt['audios']) {
							c += '<span class="audio">' + cnt['audios'] + '&nbsp;' + translate(cnt['audios'] > 1? 'audios':'audio') + '</span>';
						}
						if (cnt['videos']) {
							c += '<span class="video">' + cnt['videos'] + '&nbsp;' + translate(cnt['videos'] > 1? 'videos':'video') + '</span>';
						}
						if (cnt['others']) {
							c += '<span class="other">' + cnt['others'] + '&nbsp;' + translate(cnt['others'] > 1? 'others':'other') + '</span>';
						}
						
						return c;
					},
					
				// Modifying the total 
					
				fixTotalImages = function(diff) {
						var el = $('.footer .counts'),
							cnt = readCounts(el);
							
						for (var c in cnt) {
							if (diff[c] !== 0) {
								cnt[c] += diff[c];
							}
						}
						
						//console.log('Diff: ' + JSON.stringify(diff) + ' cnt: ' + JSON.stringify(cnt));
						
						el.empty().append(getFormattedCounts(cnt));
					},
					
				// Preparing folders
			
				prepareFolders = function() { 
					
						var folders = getFolders();
						
						if (!folders.length) {
							return;
						}
						
						if (settings.hasOwnProperty('markNew')) {
							// Marking new folders
							setTimeout(markFoldersNew, 300);
						}
						
						setTimeout(function() {
								
								// Start lazy load 
								lazyloadFolders();
								
								// Removing prelaod class on folders already loaded
								folders.filter('.' + settings.preloadClass + ':first-child img')
									.one('load', function() {
											$(this).parents('.' + settings.preloadClass).removeClass(settings.preloadClass);
										});
								
								if ((!LOCAL || DEBUG) && settings.hasOwnProperty('weblocationCounts') && settings.weblocationCounts) {
									// Adding counts to weblocations
									folders.filter('.webloc.has-count').each(function() {
											var caption = $(this).children('.caption'),					// Tha caption
												counts = readCounts(caption.find('.counts')),			// Saving original counts
												diff;
											
											loadExternal($(this).children('.thumb').attr('href'), function() {
													
													if (typeof this === OBJECT && this.hasOwnProperty(J.DEEPCOUNTERS)) {
														diff = getCountDiff(this[J.DEEPCOUNTERS], counts);
														
														if (!allZero(diff)) {
															
															caption.find('.counts')
																.empty()
																.append(getFormattedCounts(this[J.DEEPCOUNTERS]));
																	
															fixTotalImages(diff);
														}
													}
												});
										});
								}
								
								folders.find('.mosaic').each(function() {
										var th = $(this).siblings('.thumb');
											
										$(this).children().each(function(i) {
											var src = $(this).children('img').eq(0).attr('src') || $(this).data('thumb'),
												mi = src? 
														$('<img>', {
															'class':	'mosimg',
															src:		src.replace('/thumbs/', '/slides/'),
															id:			'mi' + i
														})
														:
														$();
												
											$(this).on('mouseenter', function(e) {
												mi.removeClass('show-image').addClass('hide-image');
												th.append(mi);
												setTimeout(function() {
														mi.removeClass('hide-image').addClass('show-image');
													}, 100);
											}).on('mouseleave', function(e) {
												setTimeout(function() {
														mi.addClass('hide-image').removeClass('show-image');
														setTimeout(function() {
																mi.remove();
															}, 500);
													}, 200);
											});
										});
									});
								
							}, 500);
					},
				
				/********************************************************
				 *
				 *					Initializing album
				 *
				 ********************************************************/
				
				initAlbum = function(readyFn) {
					
						if (typeof Album === UNDEF) {
							console.log('Critical Error: Missing jalbum.album.js library!');
						}
						
						var needsDeep = settings.hasOwnProperty('tagCloud') && settings.tagCloud['depth'] !== 'current' || 
										settings.pageType === 'page' && settings.pageName === 'newimages',
							params = {
									makeDate:			settings['makeDate'],
									rootPath:			settings['rootPath'],
									relPath:			settings['relPath'],
									hiDpiImages:		settings['hiDpiImages'] || false,
									hiDpiThumbs:		settings['hiDpiThumbs'] || false,
									thumbDims:			settings['thumbDims'] || [ 418, 209 ],
									thumbsFillFrame:	settings['thumbsFill'] || false,
									ready: 				(settings.pageType === 'index')? prepareIndex : preparePage,
									loadDeep:			needsDeep,
									fatalError:			function(txt, attr) {
																$body.modal((typeof attr === UNDEF)? translate(txt) : translate(txt).replace(/\{0\}/g, attr), {
																		title:		text.fatalError,
																		icon:		'error',
																		'class':	'small alert'
																	});
															},
									locale:				settings['locale'] || 'en-US'
								};
							
						if (settings.indexName !== $.fn.skin.defaults.indexName) {
							params.indexName = settings['indexName'];
						}
						
						if (settings.audioPoster !== $.fn.skin.defaults.audioPoster) {
							params.audioPoster = settings['audioPoster'];
						}
							
						if (settings.videoPoster !== $.fn.skin.defaults.videoPoster) {
							params.videoPoster = settings['videoPoster'];
						}
						
						if (settings.useOriginals) {
							params.useOriginals = true;
						}
						
						if (needsDeep) {
							params.deepReady = prepareDeepFunctions;
						}
						
						album = new Album($, params);
					},
					
				// Main initializing function
				
				init = function() {
						
						// Removing search string, e.g. FB data
						if (window.location.search) {
							removeSearch();
						}
						
						// Restoring scroll position, saving on unload
						restoreScrollPosition();
						$(window).on('beforeunload.' + ns, saveScrollPosition);
						
						loadPrefs(settings, [ 'audioClipVolume' ], 'lb_pref');
						
						// Passing along lightbox defaults
						passDefaults(settings, $.fn.lightbox.defaults, 'indexName,level,resPath,useOriginals,previousFoldersLast,previousFolderPath,nextFoldersFirst,nextFolderPath,extraSizes,hiDpiThumbs,hiDpiImages,audioClipVolume,lightboxFullscreen,slideshowFullscreen,rightClickProtect,enableKeyboard,mouseWheelAction,fotomoto,useRating,visitorRating,lbableClass');
						passDefaults(settings.lightbox, $.fn.lightbox.defaults);
						
						// Creating new Album instance
						initAlbum();
			
						// Right click protection on folders, slider
						if (settings.rightClickProtect) {
							avoidRightClick();
						}
						
						// Background music
						prepareBackgroundMusic($('[data-audioPlayer]'));
						
						// Hash change listener
						$window.on('hashchange.' + ns, stateChange);
						
						// Focusing overlay
						main.on('overlayReady.' + ns, function(e, o) {
								moveInView(o);
							});
						
						// Menu button wiring (small screens)
						$('[data-topnav-toggle]').on({
								'click': 		function() {
														$body.toggleClass('menu-on');
														return false;
													},
								'selectstart':	function(e) {
														e.preventDefault();
														return false;
													}
							});
						
						// Scroll to top btn
						if (settings.scrollToTopButton) {
							$body.scrollToTopBtn();
						}
						
						if (hasFullscreen()) {
							
							// Fullscreen change listener
							onFullscreenChange(function(on) {
									$body.toggleClass(settings.fullscreenClass, on);
								});
							
							$('.hero .controls .gofullscreen').on('click', requestFullscreen);
							$('.hero .controls .exitfullscreen').on('click', exitFullscreen);
							
							if (window.innerWidth < 640) {
								// Change to full screen on mobiles on orientation change
								goFullscreenOnLandscapeChange();
							}
						}
					};
					
						
			/********************************************************
			 *
			 *					Execution starts here
			 *
			 ********************************************************/
				
			init();
			
			// Utility function for debugging
			this.getAlbum = function() {
					return album;
				};
			
			return this;
		};
	
	/********************************************************
	 *
	 *						Defaults
	 *
	 ********************************************************/
				
	$.fn.skin.defaults = {
			locale:							'en-US',
			contentClass: 					'content',
			contClass:						'cont',
			startSlideshowHook:				'.hero .start-show',
			hasOverlayClass:				'has-overlay',
			baseOverlayClass:				'base-overlay',
			hideOnOverlayClass:				'hide-on-overlay',
			overlayClass:					'overlay',
			overlayTitleClass:				'title',
			overlayHeadLevel:				3,
			hasImagesClass:					'has-images',
			noImagesClass:					'no-images',
			hasMapClass:					'has-map',
			noMapClass:						'no-map',
			hasShopClass:					'has-shop',
			noShopClass:					'no-shop',
			shoppableClass:					'shoppable',
			progressbarClass:				'progressbar',
			lightboxClass:					'lightbox',
			sliderClass:					'slider',
			belowfoldClass:					'below-fold',
			foldersClass:					'folders',
			folderClass:					'folder',
			thumbnailsClass:				'thumbnails',
			thumbClass:						'thumb',
			boxClass:						'box',
			mapBoxHook:						'section.map,.map-cont',
			mapRootClass:					'map-root',
			tagCloudClass:					'tag-cloud',
			tagSearchClass:					'tag-search',
			thumbLayout:					'fixgrid',
			thumbsFill:						true,
			useAudioClipButton:				false,
			columns:						5,
			thumbDims:						[ 418, 209 ],
			thumbGap:						6,
			fixedShapeThumbs:				false,
			justifyWrapThreshold:			0.333333,
			imageClass:						'image',
			captionClasses:					'over bottom hover',
			cardClass:						'card',
			folderCardClass:				'foldercard',
			mosaicClass:					'mosaic',
			captionPlacement:				'below',
			lbableClass:					'lbable',
			captionClass:					'caption',
			commentClass:					'comment',
			hascaptionClass:				'hascaption',
			selectableClass:				'selectable',
			selectedClass:					'checked',
			activeClass:					'active',
			preloadClass:					'preload',
			lazyloadClass:					'lazyload',
			hideImageClass:					'hide-image',
			showImageClass:					'show-image',
			startshowClass:					'start-btn',
			fullscreenClass:				'fullscreen-on',
			panoClass:						'pano',
			selectBoxClass:					'select-box',
			selectAllClass:					'select-all',
			selectNoneClass:				'select-none',
			keepSelectedClass:				'keep-selected',
			addSelectedClass:				'add-selected',
			scrollCheckInterval:			120,
			indexName:						'index.html',
			audioPoster:					'audio.poster.png',
			videoPoster:					'video.poster.png',
			dateFormat:						'M/d/yyyy',
			audioClipVolume:				0.75,
			lightboxFullscreen:				false,
			slideshowFullscreen:			false,
			rightClickProtect: 				false,
			useRating:						false,
			jalbumRating:					false,
			visitorRating:					false,
			showEmptyFilters:				false,
			enableKeyboard:					true,
			videoAuto:						true,
			linkToFolder:					true,
			useOriginals:					false,
			scrollToTopButton:				true
		};
		
})(jQuery, jQuery(window), jQuery(document), jQuery('body'));
