package story;

/*
	Gui.java - Story skin User Interface
*/

import java.io.*;
import java.util.*;
import java.awt.Color;
import java.awt.event.*;
import java.awt.BorderLayout;
import java.awt.Component;
import java.awt.Dimension;
import java.awt.Font;
import java.beans.PropertyChangeListener;
import java.net.MalformedURLException;
import java.util.logging.Level;
import java.util.logging.Logger;
import javax.swing.*;
import javax.swing.BorderFactory;
import javax.swing.JButton;
import javax.swing.border.Border;
import javax.swing.filechooser.FileNameExtensionFilter;
import edu.stanford.ejalbert.BrowserLauncher;
import com.kitfox.svg.app.beans.SVGIcon;

import icons.Lazaicon;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import net.jalbum.component.JDraggableList;

import se.datadosen.util.Item;
import se.datadosen.util.IO;
import se.datadosen.component.*;
import se.datadosen.component.DeferredSVGIcon;
import se.datadosen.jalbum.AlbumObject;
import se.datadosen.jalbum.JAlbumContext;
import se.datadosen.jalbum.PluginContext;
import se.datadosen.jalbum.EditPanel;
import se.datadosen.jalbum.SkinProperties;
import se.datadosen.jalbum.AccountManager;
import se.datadosen.jalbum.SignInManager;
import se.datadosen.jalbum.JAlbumSite;
import se.datadosen.jalbum.AlbumBean;
import se.datadosen.jalbum.Icons;
import se.datadosen.jalbum.JAlbum;
import se.datadosen.jalbum.JAlbumColor;
import se.datadosen.jalbum.JAlbumFrame;
import se.datadosen.jalbum.ParameterException;
import se.datadosen.jalbum.event.JAlbumAdapter;
import se.datadosen.jalbum.event.JAlbumEvent;
import se.datadosen.util.Languages;
import se.datadosen.util.VersionNumber;

import net.jalbum.browser.WebViewBrowser;

/*************************************************
 * 
 *			Story GUI
 * 
 */

public class Gui extends GuiBase {
		
	private long skinReadyAt = Long.MAX_VALUE;
	private final VersionNumber jalbumVersion = new VersionNumber(AlbumBean.getInternalVersion());
	private final String skinVer = new SkinProperties(skinDirectory).getProperty(SkinProperties.VERSION);
	private final String supportForum = new SkinProperties(skinDirectory).getProperty(SkinProperties.SUPPORT_FORUM);
	private final String helpRoot = "http://jalbum.net/help/en/Skin/" + skin + "/";
	
	private boolean isSkinReady() {
		// Within 1s of launch
		//System.out.println("skinReady()? " + (System.currentTimeMillis() - skinReadyAt));
		return (System.currentTimeMillis() - skinReadyAt) > 1000L;
	}
	
	private String getStyleName() {
		String style = engine.getStyle();
		return style.substring(0, style.indexOf("."));
	}
	
	private String getSkinName() {
		return engine.getSkin();
	}
	
	private String getLastSkinName() {
		String ls = null;
		
		try {
			ls = engine.getLastSkin();
		} catch(Throwable t) {
			JAlbum.logger.log(Level.FINER, "Last skin is unknown.");
		}
		
		return ls;
	}	
	
	private String unCamelCase(String s) {
		
		if (s.length() > 0) {
			s = s.replaceAll("([a-z])([A-Z]+)", "$1 $2");
			return Character.toUpperCase(s.charAt(0)) + s.substring(1);
		}
		
		return "";
	}
	
	private String getText(String t) {
		String s;
		
		try {
			s = texts.getString(t);
		} catch (java.util.MissingResourceException e) {
			JAlbum.logger.log(Level.FINE, "Missing text: \"{0}\"", t);
			if (t.startsWith("ui.")) {
				t = t.substring(3);
			}
			s = unCamelCase(t);
		}
		
		return s;
	}
	
	private final int uiHeight = 600;
	private final int uiWidth = 680;
	private final int previewWidth = 500;
	private final String tabStyle = "style='padding:2px 4px;margin:2px 4px;'";
	private final String CUSTOM = "<!--custom-->";

	private String getFonts(String html, String ff, String hf) {
		String	gf = "",
				fw = "",
				bw = "bold",
				hw = "";
		String[] w;
		
		if (ff.contains(":")) {
			// Google font
			gf = ff;
			fw = ff.split(":")[1];
			if (fw.length() > 0) {
				w = fw.split(",");
				bw = (w.length > 2)? w[2] : w[1];
				fw = w[0];
			}
			
			ff = ff.split(":")[0];
			if (ff.contains(" ")) {
				ff = "\"" + ff + "\"";
			}
			ff += ",sans-serif";
		}

		if (!hf.equals("")) {
			if (!hf.equals(gf)) {
				gf = (gf.length() > 0)? (gf + "|" + hf) : hf;
			}
			if (hf.contains(":")) {
				// Has weight info
				hw = hf.split(":")[1];
				if (hw.length() > 0) {
					// Get base weight
					hw = hw.split(",")[0];
				}
				hf = hf.split(":")[0];
			}
			if (hf.contains(" ")) {
				hf = "\"" + hf + "\"";
			}
			hf += ",sans-serif";
		}
		
		//System.out.println("Google font: " + gf);
		return html.replace("{googleFontsLink}", (gf.length() == 0)? "" : ("\t<link rel=\"stylesheet\" href=\"http://fonts.googleapis.com/css?family=" + gf.replaceAll(" ", "+") + "&display=swap\">\n"))
			.replace("{fontFamily}", ff)
			.replace("{fontWeight}", (fw.equals("")? "300" : fw))
			.replace("{boldWeight}", bw)
			.replace("{headlineFont}", (hf.equals(""))? "inherit" : hf)
			.replace("{headlineWeight}", (hw.equals("")? "400" : hw));
		
	}
				
	private String attemptSignIn() {
		SignInManager mgr = SignInManager.getInstance();
		if ( mgr != null && mgr.isSignedIn() ) {
			return "&cid=" + AccountManager.getCid(mgr.getUserName(), mgr.getPassword());
		}
		return "";
	}
	
	//private String licensee = licenseManager.isLicenseValid()? licenseManager.getUserName() : "";
	
	private String license = licenseManager.isLicenseValid()? licenseManager.getLicenseType() : "";
	
	private boolean commercialUseAllowed = license != null && license.length() > 0 && "pro".equals(license);
	
	private Font mono = new Font("monospaced", Font.PLAIN, 12);
	
	private Border emptyBorder = BorderFactory.createEmptyBorder();
	
	private String getFileContents(String name) {
		StringBuilder cont = new StringBuilder();
		String line;
		String nl = System.getProperty("line.separator");
		File f = new File(skinDirectory, name);
		
		if (f.exists()) {
			try {
				try (java.io.BufferedReader in = new java.io.BufferedReader(new java.io.FileReader(f))) {
					while ((line = in.readLine()) != null) {
						cont.append(line);
						cont.append(nl);
					}
				}
			} catch (FileNotFoundException e) {
				return null;
			} catch (IOException e) {
				return null;
			}
		}
		return cont.toString();
	}
	
	private void writeFile(String fileName, String cont) {
		File out = new File(fileName);
		
		try {
			FileWriter writer = new FileWriter(out, false);
			writer.write(cont);
			writer.close();
		} catch (IOException e) {
			JAlbum.logger.log(Level.FINE, "Error writing \"{0}\".", fileName);
		}
	}
	
	private final Double FULLY_TRANSP = 0.0001D;
	private final Double FULLY_OPAQUE = 0.9999D;
	
	private Color toColor(JColorSelector cs) {
		return toColor(cs.toString());
	}
	
	private static final Pattern RGBA_PATTERN = Pattern.compile("rgba?\\((\\d+),(\\d+),(\\d+)(,([\\d\\.]+))?\\)");
	
	private Color toColor(String c) {
		int a = 255;
		Color clr = Color.GRAY;
		
		//System.out.print("toColor(" + c + ") -> ");
		if (c.startsWith("rgb")) {
			Matcher matcher = RGBA_PATTERN.matcher(c);
			if (matcher.matches()) {
				// Get the color components from the matched groups
				if (matcher.groupCount() > 5) {
					a = Math.min(255, (int)Math.round(Double.parseDouble(matcher.group(5)) * 255.0));
				}
				
				if (a == 255) {
					clr = new Color(
						Math.min(255, Integer.parseInt(matcher.group(1))),
						Math.min(255, Integer.parseInt(matcher.group(2))),
						Math.min(255, Integer.parseInt(matcher.group(3)))
					);
				} else {
					clr = new Color(
						Math.min(255, Integer.parseInt(matcher.group(1))),
						Math.min(255, Integer.parseInt(matcher.group(2))),
						Math.min(255, Integer.parseInt(matcher.group(3))),
						a
					);
				}
			}
			
			return clr;
		}
		
		if (c.charAt(0) == '#') {
			c = c.substring(1);
		}
		
		if (c.length() > 7) {
			a = Integer.parseInt(c.substring(0, 2), 16);
			c = c.substring(2);
		}

		if (c.length() == 3) {
			clr = new Color(
					Math.min(255, Integer.parseInt(c.substring(0,1), 16) * 17),
					Math.min(255, Integer.parseInt(c.substring(1,2), 16) * 17),
					Math.min(255, Integer.parseInt(c.substring(2), 16) * 17)
				);
		}

		if (c.length() == 6) {
			clr = new Color(
					Math.min(255, Integer.parseInt(c.substring(0,2), 16)),
					Math.min(255, Integer.parseInt(c.substring(2,4), 16)),
					Math.min(255, Integer.parseInt(c.substring(4), 16)),
					a
				);
			//System.out.println(clr.getAlpha() + " alpha=" + a);
		}
		
		return clr;
	}
	
	private String getCssColor(JColorSelector cs) {
		
		return getCssColor(toColor(cs));
	}
	
	private String getCssColor(String c) {
		
		return getCssColor(toColor(c));
	}
	
	private String getCssColor(Color c) {
		
		if (c == null) {
			return "transparent";
		}
		
		Double a = c.getAlpha() / 255.0;
		
		if (a < FULLY_TRANSP) {
			// Transparent
			return "transparent";
		} else if (a < FULLY_OPAQUE) {
			// Semi transparent
			return "rgba(" + c.getRed() + "," + c.getGreen() + "," + c.getBlue() + "," + (Math.round(a * 1000) / 1000.0) + ")";
		}
		
		// Opaque
		return "rgba(" + c.getRed() + "," + c.getGreen() + "," + c.getBlue() + ",1)";
	}
	
	private Double getAlpha(JColorSelector cs) {
		Color c = toColor(cs);
		
		return c.getAlpha() / 255.0;
	}
	
	private Double getAlpha(Color c) {
		return c.getAlpha() / 255.0;
	}
	
	private Double getAlpha(String clr) {
		Color c = toColor(clr);
		
		return c.getAlpha() / 255.0;
	}
	
	private String setAlpha(String clr, Double a) {
		Color c = toColor(clr);
		
		return "rgba(" + c.getRed() + "," + c.getGreen() + "," + c.getBlue() + "," + a + ")";
	}
	
	private Color flatColor(JColorSelector bc, JColorSelector fc) {
		
		return flatColor(toColor(bc), toColor(fc));
	}
	
	private Color flatColor(Color bc, Color fc) {
		Double a = fc.getAlpha() / 255.0;
			
		if (a < FULLY_TRANSP) {
			return bc;
		} else if (a > FULLY_OPAQUE) {
			return fc;
		}
		
		return new Color(
			(int)Math.max(0, Math.min(Math.round(a * fc.getRed() + (1 - a) * bc.getRed()), 255)),
			(int)Math.max(0, Math.min(Math.round(a * fc.getGreen() + (1 - a) * bc.getGreen()), 255)),
			(int)Math.max(0, Math.min(Math.round(a * fc.getBlue() + (1 - a) * bc.getBlue()), 255))
		);
	}
	
	private String getMiddleColor(String c1, String c2) {
		Color c = getMiddleColor(toColor(c1), toColor(c2));
		
		return "rgba(" + c.getRed() + "," + c.getGreen() + "," + c.getBlue() + "," + (Double.valueOf(c.getAlpha()) / 255.0) + ")";
	}
	
	private Color getMiddleColor(Color c1, Color c2) {
					
		return new Color(
				(int)Math.round((c1.getRed() + c2.getRed()) / 2),
				(int)Math.round((c1.getGreen() + c2.getGreen()) / 2),
				(int)Math.round((c1.getBlue() + c2.getBlue()) / 2),
				(int)Math.round((c1.getAlpha() + c2.getAlpha()) / 2)
			);
	}

	
	private String getFlatColor(JColorSelector bc, JColorSelector fc) {
		
		return getCssColor(flatColor(toColor(bc), toColor(fc)));
	}
	
	private String getFlatColor(String bc, String fc) {
		
		return getCssColor(flatColor(toColor(bc), toColor(fc)));
	}
	
	private String getFlatColor(Color bc, Color fc) {
		
		return getCssColor(flatColor(bc, fc));
	}
	
	private Double getLuminosity(JColorSelector bg, JColorSelector fg) {
		
		return getLuminosity(flatColor(bg, fg));
	}
	
	private Double getLuminosity(JColorSelector cs) {
		
		return getLuminosity(toColor(cs));
	}
	
	private Double getLuminosity(String c) {
		
		return getLuminosity(toColor(c));
	}
	
	private Double getLuminosity(Color c) {
		
		return	0.0011725490196078 * c.getRed() + 
				0.0023019607843137 * c.getGreen() +
				0.0004470588235294118 * c.getBlue();
	}
	
	private boolean isLightColor(JColorSelector cs) {
		
		return getLuminosity(cs) > 0.5;
	}
	
	private boolean isLightColor(String fc) {
		
		return getLuminosity(fc) > 0.5;
	}
	
	private boolean isLightColor(JColorSelector fc, double threshold) {
		
		return getLuminosity(fc) > threshold;
	}
	
	private boolean isLightColor(String fc, double threshold) {
		
		return getLuminosity(fc) > threshold;
	}
	
	private boolean isLightColor(JColorSelector bc, JColorSelector fc) {
		
		return isLightColor(toColor(bc), toColor(fc), 0.5);
	}
	
	private boolean isLightColor(String bc, String fc) {
		
		return isLightColor(toColor(bc), toColor(fc), 0.5);
	}
	
	private boolean isLightColor(JColorSelector bc, JColorSelector fc, Double threshold) {
		
		return isLightColor(toColor(bc), toColor(fc), threshold);
	}
	
	private boolean isLightColor(String bc, String fc, Double threshold) {
		
		return isLightColor(toColor(bc), toColor(fc), threshold);
	}
	
	private boolean isLightColor(Color bc, Color fc, Double threshold) {
		
		Double a = fc.getAlpha() / 255.0;
		
		if (a > FULLY_OPAQUE) {
			return getLuminosity(fc) >= threshold;
		}
		
		return getLuminosity(flatColor(bc, fc)) >= threshold;
	}
	
	private String getLegibleColor(Color bc1, Color bc2, Color fc, Double f) {
		Color bc = flatColor(bc1, bc2);
		
		return getLegibleColor(bc, fc, f);
	}
		
	private String getLegibleColor(JColorSelector bc, JColorSelector fc) {
		return getLegibleColor(toColor(bc), toColor(fc), 0.6);
	}
	
	private String getLegibleColor(JColorSelector bc, JColorSelector fc, Double f) {
		return getLegibleColor(toColor(bc), toColor(fc), f);
	}
	
	private String getLegibleColor(JColorSelector bc1, JColorSelector bc2, JColorSelector fc) {
		return getLegibleColor(flatColor(bc1, bc2), toColor(fc), 0.6);
	}
	
	private String getLegibleColor(JColorSelector bc1, JColorSelector bc2, JColorSelector fc, Double f) {
		return getLegibleColor(flatColor(bc1, bc2), toColor(fc), f);
	}
		
	private String getLegibleColor(Color bc, Color fc, Double f) {
		int r, g, b;
		
		if (Math.abs(getLuminosity(bc) - getLuminosity(fc)) >= f) {
			return getCssColor(fc);
		}
		
		f *= 255.0;
		
		if (getLuminosity(bc) > 0.5) {
			// Darken
			r = (int)Math.round(Math.max(fc.getRed() - f, 0));
			g = (int)Math.round(Math.max(fc.getGreen() - f, 0));
			b = (int)Math.round(Math.max(fc.getBlue() - f, 0));
		} else {
			// Lighten
			r = (int)Math.round(Math.min(fc.getRed() + f, 255));
			g = (int)Math.round(Math.min(fc.getGreen() + f, 255));
			b = (int)Math.round(Math.min(fc.getBlue() + f, 255));
		}
		
		return getCssColor(new Color(r, g, b));
	}
	
	private final String[] imageFiles = new String[] { "jpg", "jpeg", "png", "gif", "svg" };
	private final FileChooser fc = ChooserFactory.createFileChooser(window);

	private void getFileToRes(String[] extensions, JTextField name, Component c) {
		fc.setFileFilter(new FileNameExtensionFilter(getText("ui.select"), extensions));
		int returnVal = fc.showOpenDialog(c);
		
		if (returnVal == JFileChooser.APPROVE_OPTION) {
			String fn = fc.getSelectedFile().toString();
			
			if (!fn.trim().equals("")) {
				File src = new File(fn);
				File dst = new File(context.getPluginContext().getRootFolder().getFile(), "res");
				if (!dst.exists()) {
					dst.mkdir();
				}
				if (src.exists() &&  dst.exists()) {
					try {
						IO.copyFile(src, dst);
						name.setText(src.getName());
					} catch (IOException e) {
						System.out.println(e);
					}
				}
			}
		}
	}
	
	private void backupProjectFile(String ver) {
		
		try {
			File projectFile = window.projectChooser.getSelectedFile();
			IO.copyFile(projectFile, new File(projectFile.getParentFile(), IO.baseName(projectFile) + "-" + ((ver == null)? "old" : ver) + ".jap"));
		} catch (IOException ex) {
			JAlbum.logger.log(Level.FINE, "Error backing up project file: {0}", window.projectChooser.getSelectedFile());
		}
	}
	
	private StateMonitor commercialMonitor = new StateMonitor() {
		@Override
		public void onChange() {
			
			if (((JCheckBox)source).isSelected() && !commercialUseAllowed) {
				if (isSkinReady()) {
					Object[] options = { 
						getText("ui.signUp"),
						getText("ui.noThanks")
					};
					int n = JOptionPane.showOptionDialog(window, 
						getText("ui.licenseWarningText"), 
						getText("ui.licenseWarningTitle"), 
						JOptionPane.YES_NO_OPTION, 
						JOptionPane.INFORMATION_MESSAGE,
						null,
						options,
						options[1]
					);
					if (n == 0) {
						try {
							BrowserLauncher.openURL(JAlbumSite.getTrueInstance().getMyJAlbumUpgradeUrl() + "/?referrer=" + skin + attemptSignIn());
						} catch (se.datadosen.tags.ElementException | IOException x) {
						}
					}
				}
				((JCheckBox)source).setSelected(false);
			}
		}
	};
		
	private static Icon icon(String basename) {
		return Icons.get(Lazaicon.class, "svg/" + basename + ".svg", 18);
	}
		
	private static Icon icon(String basename, int size) {
		return Icons.get(Lazaicon.class, "svg/" + basename + ".svg", size);
	}
		
	private static Icon svgIcon(String basename, Dimension dim) {
		return svgIcon(basename, dim, false);
	}
	
	private static Icon svgIcon(String basename, Dimension dim, boolean adapt) {
		DeferredSVGIcon icon = new DeferredSVGIcon(Gui.class, "graphics/" + basename + ".svg");
		
		icon.setAntiAlias(true);
		icon.setAdaptColors(adapt);
		icon.setPreferredSize(dim);
		icon.setAutosize(SVGIcon.AUTOSIZE_STRETCH);
		
		return icon;
	}
	
	private Object[] getPosition() {
		return new Object[] {
			new Item("left top", getText("ui.left_top")), 
			new Item("center top", getText("ui.center_top")), 
			new Item("right top", getText("ui.right_top")), 
			new Item("left center", getText("ui.left_middle")), 
			new Item("center center", getText("ui.center_middle")), 
			new Item("right center", getText("ui.right_middle")), 
			new Item("left bottom", getText("ui.left_bottom")), 
			new Item("center bottom", getText("ui.center_bottom")), 
			new Item("right bottom", getText("ui.right_bottom"))
		};
	}

	private int setSelectedValue(JComboBox<Item> box, Object val) {
		int		size = box.getItemCount();
		Item	it;
		
		for (int i = 0; i < size; i++) {
			it = box.getItemAt(i);
			if (it.item.equals(val)) {
				box.setSelectedIndex(i);
				return i;
			}
		}
		
		return -1;
	}
	
	private String getSelectedValue(Object o) {
		if (o instanceof JComboBox) {
			Object so = ((JComboBox)o).getSelectedItem();
			if (so instanceof Item) {
				return ((Item)so).value.toString();
			}
			return (so == null)? null : so.toString();
		}
		return (o == null)? null : o.toString();
	}
	
	private int getSpinnerValueAsInt(JSpinner o) {
		return (Integer)o.getValue();
	}
	
	private Double getSpinnerValueAsDouble(JSpinner o) {
		return (Double)o.getValue();
	}
	
	private Double getSelectedValueAsDouble(Object o) {
		if (o instanceof JComboBox) {
			Object so = ((JComboBox)o).getSelectedItem();
			if (so instanceof Item) {
				String s = ((Item)so).value.toString();
				return ((s != null) && (s.length() > 0))? Double.parseDouble(s) : null;
			}
			return null;
		}
		return (o == null)? null : Double.parseDouble(o.toString());
	}
	
	private Integer getSelectedValueAsInt(JComboBox o) {
		Object so = ((JComboBox)o).getSelectedItem();
		if (so instanceof Item) {
			return Integer.parseInt(((Item)so).value.toString());
		}
		return null;
	}
	
	private Double getThumbAspectRatio() {
		String[] cdim = engine.getThumbSize().split("x");
		Double	w = Double.parseDouble(cdim[0]),
				h = Double.parseDouble(cdim[1]);
		
		if (w != null && h != null) {
			return w / h;
		}
		
		JAlbum.logger.log(Level.FINE, "Invalid thumbnail dimensions: {0} x {1}", new Object[]{ w.toString(), h.toString() });
		return null;
	}
	
	private Date skinLoadedOn = new Date();
		
	private void allowHTMLEditing(JSmartTextArea ta) {
		try {
			ta.setAllowHTMLEditing(true).setFullHTMLEditing(true);
		} catch (Throwable e) {
		}
	}
	
	private void allowSpelling(JSmartTextArea ta) {
		try {
			ta.spelling();
		} catch (Throwable e) {
		}
	}
	
	private String majorVersion(String s) {
		int i = s.indexOf(".");
		if (i > 0) {
			return s.substring(0, i);
		}
		return s;
	}
	
	private int getMajorVersion(Object v) {
		
		if (v == null) {
			return -1;
		} else if (v instanceof Integer) {
			return (int)v;
		} else if (v instanceof String) {
			try {
				return Integer.valueOf(majorVersion((String)v));
			} catch(NumberFormatException ex) {
				return -1;
			}
		} else if (v instanceof Double) {
			return (int)Math.floor((Double)v);
		}
		return -1;
	}
		
	Boolean checkBooleanProperty(AlbumObject[] sel, String pn) {
		
		if (sel.length > 1) {
			boolean v = sel[0].getProperties().get(pn, false);
			
			for (AlbumObject ao : sel) {
				if (ao.getProperties().get(pn, false) != v) {
					return null;
				}
			}
			return v;
		} else if (sel.length == 1) {
			return sel[0].getProperties().get(pn, false);
		} else {
			return null;
		}
	}
	
	// Checking for String property, where non-defined equals ""
	String checkStringProperty(AlbumObject[] sel, String pn) {
		
		if (sel.length == 0) {
			return null;
		} 
		String fv = (String)(sel[0].getProperties().get(pn));
		if (fv == null) {
			fv = "";
		}
		if (sel.length == 1) {
			return fv;
		}			
		String v;
		for (AlbumObject ao : sel) {
			v = (String)(ao.getProperties().get(pn));
			if (v == null && fv.length() > 0 || v != null && !v.equals(fv)) {
				return null;
			}
		}
		return fv;
	}
	
	void createMenuItemForBooleanProperty(JMenu menu, String pn, String title) {
		
		createMenuItemForBooleanProperty(menu, window.albumExplorer.explorer.getSelectedAlbumObjects(), pn, title);
	}
	
	void createMenuItemForBooleanProperty(JMenu menu, AlbumObject[] sel, String pn, String title) {		
		Boolean cv = checkBooleanProperty(sel, pn);

		try {
			if (cv == null) {
				// Indefinite value -> Add submenu
				JMenu subMenu = new JMenu(title);
				JMenuItem all = new JMenuItem(getText("ui.all"));
				all.addActionListener(ae -> {
					for (AlbumObject ao : sel) {
						ao.getProperties().put(pn, true);
						ao.getProperties().save();
					}				
				});
				JMenuItem none = new JMenuItem(getText("ui.none"));
				none.addActionListener(ae -> {
					for (AlbumObject ao : sel) {
						ao.getProperties().put(pn, false);
						ao.getProperties().save();
					}				
				});
				subMenu.add(all);
				subMenu.add(none);
				menu.add(subMenu);

			} else {
				// All items has the same value -> invert them when clicked
				JCheckBoxMenuItem menuItem = new JCheckBoxMenuItem(title, null, cv);

				menuItem.addActionListener(ae -> {
					for (AlbumObject ao : sel) {
						ao.getProperties().put(pn, !cv);
						ao.getProperties().save();
					}
				});

				menu.add(menuItem);
			}
			
		} catch (Throwable ex) {
			JAlbum.logger.log(Level.INFO, "Right-click menu is available only in jAlbum v32 and newer!");
			return;
		};
	}
	
	void createComboSubmenu(JMenu menu, AlbumObject[] sel, String pn, Item[] items, String title) {		
		String cv = checkStringProperty(sel, pn);
		JComboMenu subMenu;
		
		try {
			
			subMenu = new JComboMenu(title, items);
			
			if (cv != null) {
				if (cv.length() == 0) {
					subMenu.setSelectedItem(items[0]);
				} else {
					for (Item it : items) {
						if (it.value.equals(cv)) {
							subMenu.setSelectedItem(it);
							break;
						};
					}
				}
			}

			subMenu.addActionListener(ae -> { 
				String si = (String)subMenu.getSelectedItem().value;
				if (si.length() > 0) {
					for (AlbumObject ao : sel) {
						ao.getProperties().put(pn, si);
						ao.getProperties().save();
					}
				} else {
					for (AlbumObject ao : sel) {
						ao.getProperties().remove(pn);
						ao.getProperties().save();
					}
				}
			});

			menu.add(subMenu);
			
		} catch (Throwable ex) {
			JAlbum.logger.log(Level.INFO, "Right-click combo menu is available only in jAlbum v33 and newer!");
			return;
		};
		
	}
	
	JLinkLabel safeLinkLabel(javax.swing.JComponent comp, String url, String txt) {
		
		if (jalbumVersion.compareTo(new VersionNumber("32.1.4")) >= 0) {
			try {
				return new JLinkLabel(comp, txt);
			} catch (Throwable ex) {
				
			}
		}
		
		return new JLinkLabel(url, txt);
	}
	
	/*****************************************
	 *
	 *			Skin UI starts here
	 *
	 *****************************************/
	
	ControlPanel skinUi = new ControlPanel() {
			
		// UI-wide variables
		
		// Icons used multple places
		private Icon infoIcon = icon("info");
		private Icon mandatory = svgIcon("asterisk", new Dimension(16, 16));
		
		// JLinkLabels need to be difned globally 
		/*
		private JLabel siteDesignLink;
		private JLabel typographyLink;
		private JLabel lightboxDesignLink;
		private JLabel titleCaptionLink;
		private JLabel folderCaptionLink;
		private JLabel thumbCaptionLink;
		private JLabel imageCaptionLink;
		private JLabel markNewImagesLink;
		private JLabel newImagesSectionLink;
		private JLabel pagesLink;
		private JLabel topNavPagesLink;
		private JLabel topBarCCLink;
		private JLabel footerCCLink;
		private JLabel albumInfoLink;
		private JLabel customSectionLink;
		private JLabel customCodeLink;
		private JLabel customKeysLink;
		*/
		// silence further notifications
		private boolean skinChangeReported = false;
				
		private void selectComboBoxItem(JComboBox<Item> cb, JTextArea ta) {
			selectComboBoxItem(cb, ta.getText());
		}
		
		private void selectComboBoxItem(JComboBox<Item> cb, String val) {
			int size = cb.getModel().getSize();
			
			if (size == 0) {
				return;
			}
			
			ComboBoxModel<Item> model = cb.getModel();
			
			if (val != null) {
				for (int i = 0; i < size; i++) {
					if (model.getElementAt(i).value.equals(val)) {
						//System.out.println("Found preset " + i + " = \"" + val + "\"");
						cb.setSelectedIndex(i);
						return;
					}
				}
			}
			
			//System.out.println("Not found preset \"" + val + "\"");
			cb.setSelectedIndex(model.getSize() - 1);
		}
		
		@Override
		public void importVariables(Map<String, Object> vars) {
			
			//JAlbum.logger.log(Level.FINE, "Project file loaded: {0}", window.projectChooser.getSelectedFile());
			//JAlbum.logger.log(Level.FINE, "{0}", vars.keySet());
			/*
			if (window.projectChooser.getSelectedFile() == null) {
				return;
			}
			*/
			if (vars.get("maxHeroHeight") != null) {
				vars.put("heroHeight", vars.get("maxHeroHeight"));
			}
			
			String hit = (String)vars.get("heroImageType");
			if (hit != null && hit.equals("none")) {
				vars.put("heroImageType", "");
			}
			
			String tl = (String)vars.get("thumbLayout");
			if (tl != null && tl.equals("masonry")) {
				vars.put("thumbLayout", "justified");
			}
			
		};

		private JTextField styleName = new JTextField(getStyleName());

		JTextField facebookAppId = new JSmartTextField("", 24);

		JComboBox <Item> maxPageWidth = new JComboBox(new Object[] {
			new Item("none", getText("ui.noLimit")),
			new Item("47.5em", "760px"),
			new Item("52.5em", "840px"),
			new Item("57.5em", "920px"),
			new Item("62.5em", "1000px"),
			new Item("67.5em", "1080px"),
			new Item("72.5em", "1160px"),
			new Item("77.5em", "1240px"),
			new Item("82.5em", "1320px"),
			new Item("87.5em", "1400px"),
			new Item("92.5em", "1480px"),
			new Item("97.5em", "1560px")
		});			

		JComboBox <Item> heroImageType = new JComboBox(new Object[] {
			new Item("", getText("ui.none")),
			new Item("folderImage", getText("ui.folderThumbnail")),
			new Item("albumImage", getText("ui.albumThumbnail")),
			new Item("randomImage", getText("ui.randomImage")),
			new Item("selected", getText("ui.selectedImages"))
		});

		JComboBox <Item> heroOverlayType = new JComboBox(new Object[] {
			new Item("", getText("ui.none")),
			new Item("flat", getText("ui.flat")),
			new Item("linear-gradient", getText("ui.linearGradient")),
			new Item("radial-gradient", getText("ui.radialGradient")),
			new Item("blocks", "Blocks"),
			new Item("bokeh", "Bokeh"),
			new Item("crossdark", "Cross Dark"),
			new Item("crosslight", "Cross Light"),
			new Item("dots", "Dots"),
			new Item("cubes", "Cubes"),
			new Item("mosaic", "Mosaic"),
			new Item("stripes", "Stripes"),
			new Item("zebra", "Zebra")
		});
		
		JSpinner heroGradientAngle = new JSpinner(new SpinnerNumberModel(90, 0, 355, 5));			
		
		JSpinner heroHeight = new JSpinner(new SpinnerNumberModel(100, 30, 100, 5));			

		JComboBox buttonbarPlacement = new JComboBox(new Object[] {
			new Item("above", getText("ui.above")), 
			new Item("below", getText("ui.below"))
		});
		
		JCheckBox topNavigationFull = new JCheckBox(getText("ui.expandOnLargeScreens"));
		
		//JSpinner columns =  new JSpinner(new SpinnerNumberModel(4, 2, 6, 1));
		
		JComboBox columns = new JComboBox(new Object[]{
			2,
			3,
			4,
			5,
			6,
			7,
			8
		});
		
		JSpinner thumbGap =  new JSpinner(new SpinnerNumberModel(1, 0, 20, 1));
		
		JSpinner cornerRadius = new JSpinner(new SpinnerNumberModel(0, 0, 20, 1));
		
		JComboBox thumbThreshold = new JComboBox(new Object[] {
			new Item("0.0", getText("ui.noThumbnails")), 
			new Item("0.5", "1 : 2"), 
			new Item("0.6667", "2 : 3"), 
			new Item("0.75", "3 : 4"), 
			new Item("0.8", "4 : 5"), 
			new Item("0.8667", "5 : 6"), 
			new Item("1.0", "1 : 1"), 
			new Item("1.2", "6 : 5"), 
			new Item("1.25", "5 : 4"), 
			new Item("1.3333", "4 : 3"), 
			new Item("1.5", "3 : 2"), 
			new Item("1.6667", "5 : 3"), 
			new Item("1.75", "7 : 4"), 
			new Item("2.0", "2 : 1"), 
			new Item("100.0", getText("ui.noWideImages"))
		});

		JComboBox wideThreshold = new JComboBox(new Object[] {
			new Item("1.5", "3 : 2"), 
			new Item("1.6667", "5 : 3"), 
			new Item("1.75", "7 : 4"), 
			new Item("2.0", "2 : 1"), 
			new Item("2.5", "5 : 2"), 
			new Item("100.0", getText("ui.noFullWidthImages"))
		});

		JComboBox thumbLayout = new JComboBox(new Object[] {
			new Item("grid", getText("ui.fixedGrid")), 
			//new Item("masonry", getText("ui.masonry")),
			new Item("justified", getText("ui.justified")),
			new Item("patchwork", getText("ui.patchwork"))
		});
		/*
		JComboBox gridVariance = new JComboBox(new Object[] {
			new Item("1", getText("ui.small")), 
			new Item("2", getText("ui.medium")), 
			new Item("3", getText("ui.large"))
		});
		*/
		JComboBox thumbAspectRatio = new JComboBox(new Object[] {
			new Item("0.5", "1 : 2"), 
			new Item("0.6667", "2 : 3"), 
			new Item("0.75", "3 : 4"), 
			new Item("0.8", "4 : 5"), 
			new Item("0.8667", "5 : 6"), 
			new Item("1.0", "1 : 1"), 
			new Item("1.2", "6 : 5"), 
			new Item("1.25", "5 : 4"), 
			new Item("1.3333", "4 : 3"), 
			new Item("1.5", "3 : 2")
		});

		JComboBox captionPlacement = new JComboBox(new Object[]{
			new Item("below", getText("ui.below")),
			new Item("over top", getText("ui.over") + " " + getText("ui.top")), 
			new Item("over middle", getText("ui.over") + " " + getText("ui.middle")), 
			new Item("over bottom", getText("ui.over") + " " + getText("ui.bottom"))
		});
		
		JComboBox captionStyle = new JComboBox(new Object[] {
			new Item("white", getText("ui.white")), 
			new Item("light", getText("ui.light")), 
			new Item("transparent", getText("ui.transparent")), 
			new Item("dark", getText("ui.dark")),
			new Item("black", getText("ui.black"))
		});
		
		JComboBox hoverEffect = new JComboBox(new Object[] {
			new Item("", getText("ui.none")),
			new Item("hover-zoom", getText("ui.zoom")),
			new Item("hover-saturate", getText("ui.blackAndWhiteToColor")),
			new Item("hover-color", getText("ui.colorOverlay")),
			new Item("hover-zoom hover-color", getText("ui.zoom") + " + " + getText("ui.colorOverlay"))
		});
		
		JCheckBox heroImageDesaturate = new JCheckBox(getText("ui.blackAndWhite"), false);
		
		JComboBox<Item> titleCaptionPresets = new JComboBox(new Object[] {
			new Item("", getText("ui.empty")),
			new Item("<h1>${title}</h1>", getText("ui.title")),
			new Item("<h1>${title}</h1>\n<div class=\"description\">${description}</div>", getText("ui.title") + " + " + getText("ui.description") + " (" + getText("ui.default") + ")"),
			new Item("<h1>${title} <span class=\"date\">${folderModDate}</span></h1>\n<div class=\"description\">${description}</div>", getText("ui.title") + " + " + getText("ui.date") + " + " + getText("ui.description")),
			new Item(CUSTOM, "[ " + getText("ui.custom") + " ]")
		});
		JTextArea titleCaptionTemplate = new JSmartTextArea(4, 28);

		JColorSelector heroOverlayColor = new JAlphaColorSelector(getText("ui.heroOverlay"), new JSmartTextField("#b2002366", 10));
		JColorSelector heroOverlayColor2 = new JAlphaColorSelector(getText("ui.heroOverlay"), new JSmartTextField("#b266003f", 10));
		JColorSelector backgroundColor = new JAlphaColorSelector(getText("ui.backgroundColor"), new JSmartTextField("#ff111111", 10));
		JColorSelector differentBackground = new JAlphaColorSelector(getText("ui.differentBackground"), new JSmartTextField("#ff181818", 10));
		JComboBox differentBackgroundFor = new JComboBox(new Object[] {
			new Item("", getText("ui.none")),
			new Item("evenodd", getText("ui.everySecndSection")),
			new Item("folders", getText("ui.folders")),
			new Item("thumbnails", getText("ui.thumbnails")),
			new Item("custompages", getText("ui.customPages")),
			new Item("content", getText("ui.contentBackground"))
		});
		JColorSelector textColor = new JColorSelector(getText("ui.textColor"), new JSmartTextField("#dddddd", 10));
		JColorSelector buttonColor = new JAlphaColorSelector(getText("ui.button"), new JSmartTextField("#ff0099cc", 10));
		JCheckBox coloredButtons = new JCheckBox(getText("ui.useOnButtonBar"));
		JColorSelector buttonTextColor = new JColorSelector(getText("ui.buttonTextColor"), new JSmartTextField("#ffffff", 10));
		JColorSelector panelBackgroundColor = new JAlphaColorSelector(getText("ui.panelBackgroundColor"), new JSmartTextField("#ff222222", 10));
		JColorSelector panelTextColor = new JColorSelector(getText("ui.panelTextColor"), new JSmartTextField("#cccccc", 10));
		
		// Typography
		private final JComboBox fontSuggestions = new JComboBox(new Object[] {
			"[" + getText("ui.suggestedFonts") +  "]",
			"Barlow Semi Condensed Light / Barlow",
			"Cinzel Decorative / Raleway",
			"Dosis Bold / Montserrat",
			"Exo 2 / Exo 2",
			"Fjalla One / Roboto",
			"IBM Plex Serif / IBM Plex Sans",
			"Jura / Roboto Condensed",
			"Martel / Roboto",
			"Merriweather / Merriweather Sans",
			"Oswald / Open Sans",
			"Raleway / Open Sans",
			"Roboto Condensed / Roboto",
			"Roboto Slab / Montserrat",
			"Special Elite / Yanone Kaffeesatz",
			"Unica One / Lato",
			"Yanone Kaffeesatz Bold / Muli"
		});
						
		private void setFontBoxes() {
			
			if (fontSuggestions.getSelectedIndex() == 0) {
				return;
			}
			String	s = fontSuggestions.getSelectedItem().toString(),
					hf = s.split("/")[0].trim(),
					ff = s.split("/")[1].trim();

			setSelectedValue(fontFamily, ff);
			setSelectedValue(headlineFont, hf);
			fontSuggestions.setSelectedIndex(0);
		};
		
		JComboBox fontFamily = new JComboBox(new Object[] {
			new Item("Arial, \"Helvetica Neue\", Helvetica, sans-serif", "Arial"),
			new Item("Baskerville, \"Baskerville Old Face\", \"Hoefler Text\", Garamond, \"Times New Roman\", serif", "Baskerville"),
			new Item("Calibri, Candara, Segoe, \"Segoe UI\", Optima, Arial, sans-serif", "Calibri"),
			new Item("Cambria, Georgia, Times, \"Times New Roman\", serif", "Cambria"),
			new Item("\"Century Gothic\", \"Apple Gothic\", \"Goudy Old Style\", sans-serif", "Century Gothic"),
			new Item("\"Comic Sans MS\", cursive", "Comic Sans"),
			new Item("Consolas, \"Lucida Console\", Monaco, monospace", "Consolas"),
			new Item("Constantia, Palatino, \"Palatino Linotype\", \"Palatino LT STD\", Georgia, serif", "Constantia"),
			new Item("\"Copperplate Light\", \"Copperplate Gothic Light\", serif", "Copperplate Light"),
			new Item("\"Courier New\", Courier, monospace", "Courier New"),
			new Item("\"Franklin Gothic Medium\", \"Arial Narrow Bold\", Arial, sans-serif", "Franklin Gothic"),
			new Item("Futura, \"Century Gothic\", AppleGothic, sans-serif", "Futura"),
			new Item("Garamond, \"Hoefler Text\", Times New Roman, Times, serif", "Garamond"),
			new Item("Geneva, \"Lucida Sans\", \"Lucida Grande\", \"Lucida Sans Unicode\", Verdana, sans-serif", "Geneva"),
			new Item("Georgia, Palatino, \"Palatino Linotype\", Times, \"Times New Roman\", serif", "Georgia"),
			new Item("\"Gill Sans\", \"Gill Sans MT\", Calibri, \"Trebuchet MS\", sans-serif", "Gill Sans"),
			new Item("\"Goudy Old Style\", Garamond, \"Big Caslon\", \"Times New Roman\", serif", "Goudy Old Style"),
			new Item("\"Helvetica Neue\", Helvetica, Arial, sans-serif", "Helvetica Neue"),
			new Item("\"Hoefler Text\", Constantia, Palatino, \"Palatino Linotype\", \"Book Antiqua\", Georgia, serif", "Hoefler Text"),
			new Item("Impact, Haettenschweiler, \"Arial Narrow Bold\", sans-serif", "Impact"),
			new Item("\"Lucida Sans\", \"Lucida Grande\", \"Lucida Sans Unicode\", Verdana, sans-serif", "Lucida Sans"),
			new Item("\"Lucida Bright\", Georgia, serif", "Lucida Bright"),
			new Item("Palatino, \"Palatino Linotype\", \"Book Antiqua\", Georgia, serif", "Palatino"),
			new Item("Segoe, \"Segoe UI\", Tahoma, Geneva, \"Nimbus Sans L\", sans-serif", "Segoe"),
			new Item("Tahoma, Geneva, Verdana, sans-serif", "Tahoma"),
			new Item("Times, \"Times New Roman\", Georgia, serif", "Times"),
			new Item("\"Trebuchet MS\", \"Lucida Sans Unicode\", \"Lucida Grande\", \"Lucida Sans\", Tahoma, sans-serif", "Trebuchet MS"),
			new Item("Verdana, Geneva, Tahoma, sans-serif", "Verdana"),
			new Item("Alegreya Sans:300,300i,500,500i", "Alegreya Sans Light"),
			new Item("Barlow:300,300i,600,600i", "Barlow Light"),
			new Item("Barlow:400,400i,700,700i", "Barlow"),
			new Item("Barlow Semi Condensed:300,300i,500,500i", "Barlow Semi Condensed"),
			new Item("Exo 2:300,300i,600,600i", "Exo 2"),
			new Item("Fira Sans:300,300i,600,600i", "Fira Sans"),
			new Item("IBM Plex Sans:300,300i,600,600i", "IBM Plex Sans"),
			new Item("Josefin Sans:300,300i,600,600i", "Josefin Sans"),
			new Item("Josefin Slab:300,300i,600,600i", "Josefin Slab"),
			new Item("Lato:300,300i,700,700i", "Lato"),
			new Item("Merriweather Sans:300,300i,700,700i", "Merriweather Sans"),
			new Item("Merriweather:300,300i,700,700i", "Merriweather Serif"),
			new Item("Montserrat:300,300i,600,600i", "Montserrat"),
			new Item("Muli:300,300i,700,700i", "Muli"),
			new Item("Open Sans:300,300i,600,600i", "Open Sans"),
			new Item("Raleway:300,300i,600,600i", "Raleway"),
			new Item("Roboto:300,300i,500,500i", "Roboto"),
			new Item("Roboto Condensed:300,300i,700,700i", "Roboto Condensed"),
			new Item("Saira:300,300i,600,600i", "Saira"),
			new Item("Source Sans Pro:300,300i,600,600i", "Source Sans Pro"),
			new Item("Work Sans:300,300i,600,600i", "Work Sans Light"),
			new Item("Yanone Kaffeesatz:300,500", "Yanone Kaffeesatz")
		});
		
		JComboBox headlineFont = new JComboBox(new Object[] {
			new Item("", "[ " + getText("ui.sameAsBaseFont") + " ]"),
			new Item("Abril Fatface", "Abril Fatface"),
			new Item("Alex Brush", "Alex Brush"),
			new Item("Amaranth", "Amaranth"),
			new Item("Amatic SC:700", "Amatic SC Bold"),
			new Item("Anton", "Anton"),
			new Item("Arapey", "Arapey"),
			new Item("Barlow:300", "Barlow Light"),
			new Item("Barlow:500", "Barlow Medium"),
			new Item("Barlow Semi Condensed:300", "Barlow Semi Condensed Light"),
			new Item("Barlow Condensed", "Barlow Condensed"),
			new Item("Cantata One", "Cantata One"),
			new Item("Cinzel Decorative", "Cinzel Decorative"),
			new Item("Cookie", "Cookie"),
			new Item("Dancing Script", "Dancing Script"),
			new Item("Dynalight", "Dynalight"),
			new Item("Dosis:600", "Dosis Bold"),
			new Item("Economica", "Economica"),
			new Item("Emilys Candy", "Emilys Candy"),
			new Item("Euphoria Script", "Euphoria Script"),
			new Item("Exo 2:300", "Exo 2"),
			new Item("Fauna One", "Fauna One"),
			new Item("Fjalla One", "Fjalla One"),
			new Item("Fredericka the Great", "Fredericka the Great"),
			new Item("Geo", "Geo"),
			new Item("Gilda Display", "Gilda Display"),
			new Item("Grand Hotel", "Grand Hotel"),
			new Item("Great Vibes", "Great Vibes"),
			new Item("Gruppo", "Gruppo"),
			new Item("Handlee", "Handlee"),
			new Item("IBM Plex Serif:300", "IBM Plex Serif"),
			new Item("IM Fell English", "IM Fell English"),
			new Item("Italiana", "Italiana"),
			new Item("Josefin Slab", "Josefin Slab"),
			new Item("Julius Sans One", "Julius Sans One"),
			new Item("Jura", "Jura"),
			new Item("La Belle Aurore", "La Belle Aurore"),
			new Item("Libre Baskerville", "Libre Baskerville"),
			new Item("Lobster", "Lobster"),
			new Item("Lobster Two", "Lobster Two"),
			new Item("Lora", "Lora"),
			new Item("Maiden Orange", "Maiden Orange"),
			new Item("Martel:300", "Martel"),
			new Item("Marvel:700", "Marvel Bold"),
			new Item("Medula One", "Medula One"),
			new Item("Merriweather:300", "Merriweather"),
			new Item("Mountains of Christmas", "Mountains of Christmas"),
			new Item("Noticia Text", "Noticia Text"),
			new Item("Noto Serif Display:300", "Noto Serif Light"),
			new Item("Old Standard TT", "Old Standard TT"),
			new Item("Oranienbaum", "Oranienbaum"),
			new Item("Oswald", "Oswald"),
			new Item("Philosopher", "Philosopher"),
			new Item("Poiret One", "Poiret One"),
			new Item("Prata", "Prata"),
			new Item("Princess Sofia", "Princess Sofia"),
			new Item("PT Mono", "PT Mono"),
			new Item("PT Sans Narrow", "PT Sans Narrow"),
			new Item("Raleway:300", "Raleway"),
			new Item("Raleway:600", "Raleway Bold"),
			new Item("Roboto Condensed", "Roboto Condensed"),
			new Item("Roboto Slab:300", "Roboto Slab"),
			new Item("Roboto Slab:600", "Roboto Slab Bold"),
			new Item("Rochester", "Rochester"),
			new Item("Shadows Into Light Two", "Shadows Into Light Two"),
			new Item("Scope One", "Scope One"),
			new Item("Six Caps", "Six Caps"),
			new Item("Sofia", "Sofia"),
			new Item("Sorts Mill Goudy", "Sorts Mill Goudy"),
			new Item("Special Elite", "Special Elite"),
			new Item("Squada One", "Squada One"),
			new Item("Strait", "Strait"),
			new Item("Unica One", "Unica One"),
			new Item("Vidaloka", "Vidaloka"),
			new Item("Work Sans", "Work Sans"),
			new Item("Yanone Kaffeesatz:300", "Yanone Kaffeesatz"),
			new Item("Yanone Kaffeesatz:500", "Yanone Kaffeesatz Bold")
		});

		// Folders
		JSpinner folderColumns = new JSpinner(new SpinnerNumberModel(1, 1, 3, 1));
		
		JCheckBox folderMosaic = new JCheckBox(getText("ui.useMosaic"));

		JCheckBox showFolderImageCount = new JCheckBox(getText("ui.showFolderImageCount"), false);
		
		JCheckBox useContinueBtn = new JCheckBox(getText("ui.continueButton"));
		JCheckBox useSlideshowBtn = new JCheckBox(getText("ui.slideshowButton"));
		JComboBox heroTitleDecoration = new JComboBox(new Object[]{
			new Item("transparent", getText("ui.none")),
			new Item("lighten", getText("ui.lighten")),
			new Item("darken", getText("ui.darken")),
			new Item("panel", getText("ui.panelColors")),
			new Item("border", getText("ui.border"))
		});
		JComboBox<Item> folderCaptionPresets = new JComboBox(new Object[] {
			new Item("", getText("ui.empty")),
			new Item("<h3>${title}</h3>", getText("ui.title")),
			new Item("<h3>${title}</h3>\n<span class=\"date\">${folderModDate}</span>", getText("ui.title") + " + " + getText("ui.date")),
			new Item("<h3>${title}</h3>\n<div class=\"comment\">${comment}</div>", getText("ui.title") + " + " + getText("ui.description")),
			new Item("<h3>${title}</h3>\n<div class=\"comment\">${commentShort}</div>", getText("ui.title") + " + " + getText("ui.descriptionShort") + " (" + getText("ui.default") + ")"),
			new Item("<h3>${title}</h3>\n<span class=\"date\">${folderModDate}</span>\n<div class=\"comment\">${comment}</div>", getText("ui.title") + " + " + getText("ui.date") + " + " + getText("ui.description")),
			new Item("<h3>${title}</h3>\n<span class=\"date\">${folderModDate}</span>\n<div class=\"comment\">${commentShort}</div>", getText("ui.title") + " + " + getText("ui.date") + " + " + getText("ui.descriptionShort")),
			new Item("<h3>${fileLabel}</h3>", getText("ui.fileLabel")),
			new Item("<h3>${fileLabel}</h3><span class=\"date\">${folderModDate}</span>", getText("ui.fileLabel") + " + " + getText("ui.date")),
			new Item("<h3>${fileLabel}</h3>\n<div class=\"comment\">${comment}</div>", getText("ui.fileLabel") + " + " + getText("ui.description")),
			new Item("<h3>${fileLabel}</h3>\n<span class=\"date\">${folderModDate}</span>\n<div class=\"comment\">${comment}</div>", getText("ui.fileLabel") + " + " + getText("ui.date") + " + " + getText("ui.description")),
			new Item(CUSTOM, "[ " + getText("ui.custom") + " ]")
		});
		JTextArea folderCaptionTemplate = new JSmartTextArea(4, 30);
		
		// Pages
		JComboBox showPages = new JComboBox(new Object[]{
			new Item("none", getText("ui.none")), 
			new Item("link", getText("ui.linkOnly")), 
			new Item("excerpt", getText("ui.withExcerpt")), 
			new Item("embedCustom", getText("ui.embedCustom")),
			new Item("embed", getText("ui.embedAll"))
		});

		// Weblocations
		JComboBox showWeblocations = new JComboBox(new Object[]{
			new Item("none", getText("ui.none")), 
			new Item("link", getText("ui.linkOnly")),
			new Item("smallThumb", getText("ui.smallThumbnail")),
			new Item("thumb", getText("ui.thumbnail")),
			new Item("asFolder", getText("ui.asFolder")),
			new Item("embed", getText("ui.embedFull"))
		});
		
		JComboBox thumbCaptionPresets = new JComboBox(new Object[] {
			new Item("", getText("ui.empty")),
			new Item("<div class=\"title\">${fileTitle}</div>", getText("ui.title")),
			new Item("<div class=\"title\">${fileTitle}</div>\n<div class=\"comment\">${comment}</div>", getText("ui.title") + " + " + getText("ui.comment")),
			new Item("<div class=\"title\">${fileTitle}</div>\n<div class=\"comment\">${commentShort}</div>", getText("ui.title") + " + " + getText("ui.commentShort") + " (" + getText("ui.default") + ")"),
			new Item("<div class=\"date\">${originalDate}</div>\n<div class=\"title\">${fileTitle}</div>\n<div class=\"comment\">${commentShort}</div>", getText("ui.date") + " + " + getText("ui.title") + " + " + getText("ui.commentShort")),
			new Item("<div class=\"date\">${originalTime}</div>\n<div class=\"title\">${fileTitle}</div>\n<div class=\"comment\">${commentShort}</div>", getText("ui.time") + " + " + getText("ui.title") + " + " + getText("ui.commentShort")),
			new Item("<span class=\"nr\">${imageNum}</span>\n<div class=\"title\">${fileTitle}</div>\n<div class=\"comment\">${commentShort}</div>", getText("ui.number") + " + " + getText("ui.title") + " + " + getText("ui.commentShort")),
			new Item("<div class=\"title\">${fileLabel}</div>", getText("ui.fileLabel")),
			new Item("<div class=\"title\">${fileLabel}</div>\n<div class=\"comment\">${comment}</div>", getText("ui.fileLabel") + " + " + getText("ui.comment")),
			new Item("<div class=\"title\">${fileLabel}</div>\n<div class=\"comment\">${commentShort}</div>", getText("ui.fileLabel") + " + " + getText("ui.commentShort")),
			new Item("<span class=\"nr\">${imageNum}</span>\n<div class=\"title\">${fileLabel}</div>\n<div class=\"comment\">${commentShort}</div>", getText("ui.number") + " + " + getText("ui.fileLabel") + " + " + getText("ui.commentShort")),
			new Item(CUSTOM, "[ " + getText("ui.custom") + " ]")
		});
		JTextArea thumbCaptionTemplate = new JSmartTextArea(4, 30);
		
		JComboBox singleCaptionPresets = new JComboBox(new Object[] {
			new Item("", getText("ui.empty")),
			new Item("<h4>${fileTitle}</h4>", getText("ui.title")),
			new Item("<h4>${fileTitle}</h4>\n<p class=\"comment\">${comment}</p>", getText("ui.title") + " + " + getText("ui.comment") + " (" + getText("ui.default") + ")"),
			new Item("<h4>${fileLabel}</h4>", getText("ui.fileLabel")),
			new Item("<h4>${fileLabel}</h4>\n<p class=\"comment\">${comment}</p>", getText("ui.fileLabel") + " + " + getText("ui.comment")),
			new Item(CUSTOM, "[ " + getText("ui.custom") + " ]")
		});
		JTextArea singleCaptionTemplate = new JSmartTextArea(4, 30);

		JComboBox fullCaptionPresets = new JComboBox(new Object[] {
			new Item("", getText("ui.empty")),
			new Item("", getText("ui.empty")),
			new Item("<h2 class=\"title\">${fileTitle}</h2>", getText("ui.title")),
			new Item("<h2>${fileTitle}</h2>\n<p class=\"comment\">${comment}</p>", getText("ui.title") + " + " + getText("ui.comment") + " (" + getText("ui.default") + ")"),
			new Item("<h2 class=\"title\">${fileLabel}</h2>", getText("ui.fileLabel")),
			new Item("<h2>${fileLabel}</h2>\n<p class=\"comment\">${comment}</p>", getText("ui.fileLabel") + " + " + getText("ui.comment")),
			new Item(CUSTOM, "[ " + getText("ui.custom") + " ]")
		});
		JTextArea fullCaptionTemplate = new JSmartTextArea(4, 30);

		JComboBox imageCaptionPresets = new JComboBox(new Object[] {
			new Item("", getText("ui.empty")),
			new Item("${originalTime}", getText("ui.time")),
			new Item("<div class=\"title\">${fileTitle}</div>", getText("ui.title")),
			new Item("<div class=\"title\">${fileTitle}</div>\n<div class=\"comment\">${comment}</div>", getText("ui.title") + " + " + getText("ui.comment")),
			new Item("<div class=\"title\">${fileTitle}</div>\n<div class=\"comment\">${commentShort}</div>", getText("ui.title") + " + " + getText("ui.commentShort")),
			new Item("<span>${originalTime}</span>\n<div class=\"title\">${fileTitle}</div>\n<div class=\"comment\">${comment}</div>", getText("ui.time") + " + " + getText("ui.title") + " + " + getText("ui.comment")),
			new Item("<div class=\"title\">${fileLabel}</div>", getText("ui.fileLabel")),
			new Item("<div class=\"title\">${fileLabel}</div>\n<div class=\"comment\">${comment}</div>", getText("ui.fileLabel") + " + " + getText("ui.comment") + " (" + getText("ui.default") + ")"),
			new Item("<div class=\"title\">${fileLabel}</div>\n<div class=\"comment\">${commentShort}</div>", getText("ui.fileLabel") + " + " + getText("ui.commentShort")),
			new Item(CUSTOM, "[ " + getText("ui.custom") + " ]")	
		});
		JTextArea imageCaptionTemplate = new JSmartTextArea(6, 30);

		
				
		private void setMakeSlides() {
			boolean ov = engine.isSlides(),
					nv = (shareFacebook.isSelected() || shareThreads.isSelected() || shareBluesky.isSelected()) 
						|| 
						(writeSitemapXml.isSelected() && sitemapIncludeSlides.isSelected());
			
			if (ov != nv) {
				try {
					window.ui2Engine();
					engine.setSlides(nv);
					window.engine2UI();
				} catch (Exception ex) {
					throw new RuntimeException(ex);
				}
			}
		};
		
		private String lastThumbDimSuggestion = "";
		
		private int getPageWidth() {
			String	pw = ((Item)maxPageWidth.getSelectedItem()).value.toString();
			return pw.equals("none")? 1600 : Math.round(Float.parseFloat(pw.replace("em","")) * 16);
		}
		
		// Returns the thumb width
		private int getMaxThumbWidth() {
			
			return Integer.parseInt(engine.getThumbSize().split("x")[0]);
		}
		
		// Returns the thumb height
		private int getMaxThumbHeight() {
			
			return Integer.parseInt(engine.getThumbSize().split("x")[1]);
		}
		
		// Checks if number is within range of ref
		private boolean inRange(double n, double ref, double range) {
			
			return (n >= ref * (1.0 - range)) && (n < ref * (1.0 + range));
		}
		
		// Calculating proper thumb dimensions for the different thumb layouts
		private void setThumbDimensions() {
			
			if (!isSkinReady()) {
				return;
			}
			
			String	layout = ((Item)thumbLayout.getSelectedItem()).value.toString(),
					cdim = engine.getThumbSize(),
					ndim;
					
			int		cw = getPageWidth() - 40,
					cols = Integer.parseInt(columns.getSelectedItem().toString()),
					tw,
					w,
					h,
					mtw = getMaxThumbWidth(),
					mth = getMaxThumbHeight(),
					tg = (int)thumbGap.getValue();
					
			double	tar;
			
			// Thumb width
			tw = (int)Math.round((cw - tg * 2 * (cols - 1)) / cols);
			 
			switch (layout) {
				case "grid":
					// Grid
					tar = Float.parseFloat((((Item)thumbAspectRatio.getSelectedItem()).value).toString());
					w = tw;
					h = (int)Math.round(tw / tar);
					break;
					
				case "patchwork":
					// Patchwork
					w = (int)Math.round(.62 * cw);
					h = (int)Math.round(.46 * cw);
					break;
					
				default:
					// Justified
					
					tar = Math.min(Float.parseFloat((((Item)thumbThreshold.getSelectedItem()).value).toString()), 3.0);
					if (tar == 0) {
						return;
					}
					
					tar = Math.sqrt(tar);

					w = (int)Math.round(1.6 * tw * tar);
					h = (int)Math.round(1.6 * tw / tar);
			}
			
			ndim = w + "x" + h;
			
			if (!lastThumbDimSuggestion.equals(ndim) &&					// Already set
				!(inRange(w, mtw, 0.05) && inRange(h, mth, 0.05) &&		// Width and Height is out of 5% range
				inRange(w / h, mtw / mth, 0.01))) {						// Aspect ratio is out of 1% range
				
				try {
					window.ui2Engine();
					engine.setThumbSize(ndim);
					JAlbum.logger.log(Level.FINE, "Thumb size has changed from {0}px to {1}px.", new Object[]{cdim, ndim});
					window.engine2UI();
				} catch (ParameterException ex) {
					throw new RuntimeException(ex);
				}
				
				lastThumbDimSuggestion = ndim;
			} else {
				JAlbum.logger.log(Level.FINE, "Thumb size change from {0}px to {1}px is within range.", new Object[]{cdim, ndim});
			}
		};
		
		// These need to be global to watch independent slide pages requirement
		
		JCheckBox shareFacebook = new JCheckBox("Facebook");
		JCheckBox shareThreads = new JCheckBox("Threads");
		JCheckBox shareBluesky = new JCheckBox("Bluesky");
		JCheckBox writeSitemapXml = new JCheckBox(getText("ui.createSitemapXml"));
		JCheckBox sitemapIncludeSlides = new JCheckBox(getText("ui.includeSlides"), true);
		
		// Refreshing preview
		
		private WebViewBrowser designPreview = new WebViewBrowser();
		
		private void refreshPreview() {
			String		html,
						patternDir,
						panelBgColor = getCssColor(panelBackgroundColor.toString()),
						heroBgImage = getSelectedValue(heroImageType),
						heroOlyType = getSelectedValue(heroOverlayType),
						heroBgColor = getCssColor(heroOverlayColor.toString()),
						heroBgColor2 = getCssColor(heroOverlayColor2.toString()),
						heroDecoration = getSelectedValue(heroTitleDecoration),
						placement = getSelectedValue(captionPlacement);
			Color		bgCol = heroOlyType.equals("")? toColor(backgroundColor) : Color.GRAY,
						mc;
			int			cols = Integer.parseInt(getSelectedValue(columns));
			double		ar = Float.parseFloat(getSelectedValue(thumbAspectRatio)),
						bgLum,
						bgAlph,
						heroH = Integer.parseInt(heroHeight.getValue().toString()) * 0.6;
			
			try {
				patternDir = (new File(skinDirectory, "res/pattern").toURI().toURL()).toString();
			} catch (MalformedURLException ex) {
				Logger.getLogger(Gui.class.getName()).log(Level.SEVERE, null, ex);
				patternDir = "";
			}
			
			switch (heroOlyType) {
				case "flat":
					bgLum = getLuminosity(getFlatColor(backgroundColor.toString(), heroOverlayColor.toString()));
					bgAlph = getAlpha(heroOverlayColor.toString());
					break;
				case "linear-gradient":
				case "radial-gradient":
					mc = getMiddleColor(toColor(heroOverlayColor), toColor(heroOverlayColor2));
					bgLum = getLuminosity(getFlatColor(bgCol, mc));
					bgAlph = getAlpha(mc);
					break;
				default:
					bgLum = getLuminosity(bgCol);
					bgAlph = heroBgImage.equals("none")? 1 : 0;
					//System.out.println(bgCol+" Lum="+bgLum+" Alpha="+bgAlph);
					break;
			}
						
			html = designTemplate
				.replace("{patternDir}", patternDir)
				.replace("{backgroundColor}", getCssColor(backgroundColor.toString()))
				.replace("{textColor}", textColor.toString())
				.replace("{buttonbarPlacement}", getSelectedValue(buttonbarPlacement))
				.replace("{maxHeroHeight}", String.valueOf(heroH))
				.replace("{hasHeroBg}", heroBgImage.equals("none")? "no-bg" : ("has-bg" + (heroImageDesaturate.isSelected()? " desaturate" : "")))
				.replace("{heroOverlayType}", heroOlyType)
				.replace("{heroDarkness}", (bgLum > 0.6)? "light" : "dark")
				.replace("{heroGradientAngle}", heroGradientAngle.getValue() + "")
				.replace("{heroOverlayColor}", heroBgColor)
				.replace("{heroOverlayColor2}", heroBgColor2)
				.replace("{useContinueBtn}", useContinueBtn.isSelected()? "" : "hide")
				.replace("{useSlideshowBtn}", useSlideshowBtn.isSelected()? "" : "hide")
				.replace("{heroDecoration}", heroDecoration)
				.replace("{cornerRadius}", cornerRadius.getValue() + "")
				.replace("{differentBackground}", getCssColor(differentBackground.toString()))
				.replace("{differentFor}", (differentBackgroundFor.getSelectedIndex() == 0)? "" : ("different-" + getSelectedValue(differentBackgroundFor)))
				.replace("{buttonColor}", getCssColor(buttonColor.toString()))
				.replace("{buttonTextColor}", buttonTextColor.toString())
				.replace("{panelTextColor}", panelTextColor.toString())
				.replace("{panelBgColor}", panelBgColor)
				.replace("{coloredButtons}", coloredButtons.isSelected()? "colored":"normal")
				.replace("{footerBgColor}", heroBgColor.equals("transparent")? "rgba(255,255,255,.1)" : heroBgColor)
				.replace("{folderCounts}", showFolderImageCount.isSelected()? "" : "hide")
				.replace("{folderColumns}", getSpinnerValueAsInt(folderColumns) + "")
				.replace("{folderMosaic}", folderMosaic.isSelected()? "hasmosaic" : "nomosaic")
				.replace("{folderCaptionBg}", isLightColor(textColor)? "rgba(0,0,0,.8)" : "rgba(255,255,255,.9)")
				.replace("{thumbGap}", thumbGap.getValue().toString())
				.replace("{thumbHeight}", Math.floor(300.0 / cols) + "")
				.replace("{thumbWidth}", Math.floor(100.0 / (cols + 1)) + "")
				.replace("{thumbAr}", Math.round(100 / ar) + "%")
				.replace("{hoverEffect}", getSelectedValue(hoverEffect))
				.replace("{cols}", getSelectedValue(columns))
				.replace("{captionClass}", "caption-" + placement + (placement.equals("below")? "" : (" caption-" + getSelectedValue(captionStyle))))
				.replace("{thumbLayout}", getSelectedValue(thumbLayout))
				.replace("{showPages}", getSelectedValue(showPages));
							
			html = getFonts(html, getSelectedValue(fontFamily), (headlineFont.getSelectedIndex() == 0)? "" : getSelectedValue(headlineFont));
			//System.out.println("\nPreview reload triggered by: " + src + "\n\n" + html);
			
			final String html1 = html;
			
			SwingUtilities.invokeLater(() -> {
				designPreview.loadContent(html1);
			});
		
			//writeFile("D:/Temp/index-preview.html", html);
		}

		PropertyChangeListener setupMonitors = pce -> {
			
			StateMonitor.monitoring(fontSuggestions).onChange(src -> {
				if (isSkinReady() && src != null) {
					setFontBoxes();
				}
			});

			// Thumb size change
			StateMonitor.monitoring(
					styleName,
					maxPageWidth, 
					thumbThreshold, 
					columns, 
					thumbLayout, 
					thumbAspectRatio, 
					thumbGap).onChange(src -> {
				
				if (isSkinReady() && src != null) {
					//JAlbum.logger.log(Level.FINE, "stateMonitor: \"{0}\" changed", getSelectedValue(src));
					setThumbDimensions();
				}
				
			});

			// Separate slide page requirement
			StateMonitor.monitoring(
					shareFacebook,
					shareThreads,
					shareBluesky,
					writeSitemapXml,
					sitemapIncludeSlides).onChange(src -> {
				setMakeSlides();
			});

			// Preview update
			StateMonitor.monitoring(styleName,
					buttonbarPlacement,
					heroHeight,
					backgroundColor,
					textColor, 
					buttonColor, 
					buttonTextColor, 
					coloredButtons,
					fontFamily, 
					headlineFont,
					heroImageType,
					heroImageDesaturate,
					heroOverlayType,
					heroOverlayColor,
					heroOverlayColor2,
					heroGradientAngle,
					heroTitleDecoration,
					useContinueBtn,
					useSlideshowBtn,
					differentBackground, 
					differentBackgroundFor, 
					cornerRadius,
					thumbGap, 
					thumbLayout, 
					thumbAspectRatio, 
					captionPlacement,
					captionStyle,
					hoverEffect,
					columns,
					folderColumns,
					folderMosaic,
					showFolderImageCount,
					showPages, 
					panelTextColor, 
					panelBackgroundColor).onChange((Object src) -> {

				if (isSkinReady() && src != null) {
					refreshPreview();
				}
			});

			// Initial preview rendering
			
			refreshPreview();
		};
				
		
		/*	---------------------------------------------------------------
									Design
			--------------------------------------------------------------- */
		
		// Page design
		
		ControlPanel pageDesign = new ControlPanel(new BorderLayout(20, 0)) {
			
			JCheckBox preferThemeImage = new JCheckBox(getText("ui.preferThemeImage"), false);
			JSpinner heroSlideshowDelay = new JSpinner(new SpinnerNumberModel(4000, 100, 10000, 100));
			JLabel heroSlideshowDelayLabel = new JLabelFor(getText("ui.timing"), heroSlideshowDelay);
			
			ControlPanel colors = new ControlPanel() {
								
				{
					new StateMonitor() {
						public void onChange() {
							boolean imageBg = heroImageType.getSelectedIndex() > 0,
								ssBg = heroImageType.getSelectedIndex() > 2;
							heroImageDesaturate.setVisible(imageBg);
							preferThemeImage.setVisible(imageBg && !ssBg);
							heroSlideshowDelayLabel.setVisible(ssBg);
							heroSlideshowDelay.setVisible(ssBg);
						}
					}.add(heroImageType).done();
					heroOverlayType.addActionListener((ActionEvent e) -> {
						int i = heroOverlayType.getSelectedIndex();
						boolean ssBg = heroImageType.getSelectedIndex() > 2;
						heroGradientAngle.setEnabled(i == 2);
						heroOverlayColor.setEnabled(i > 0);
						heroOverlayColor2.setEnabled(i >= 2 && i <= 3);
						heroSlideshowDelayLabel.setVisible(ssBg);
						heroSlideshowDelay.setVisible(ssBg);
					});
					heroSlideshowDelay.setToolTipText(getText("ui.heroSlideshowDelayInfo"));
					preferThemeImage.setToolTipText(getText("ui.preferThemeImageInfo"));
					textColor.setToolTipText(getText("ui.textColorInfo"));
					textColor.getTextComponent().setFont(mono);
					buttonColor.getTextComponent().setFont(mono);	
					buttonTextColor.getTextComponent().setFont(mono);	
					heroOverlayColor.getTextComponent().setFont(mono);
					heroOverlayColor.setToolTipText(getText("ui.heroOverlayColorInfo"));
					heroOverlayColor2.getTextComponent().setFont(mono);
					//heroOverlayColor2.setToolTipText(getText("ui.heroOverlayColorInfo"));
					differentBackground.getTextComponent().setFont(mono);
					differentBackground.setToolTipText(getText("ui.differentBackgroundInfo"));
					panelBackgroundColor.getTextComponent().setFont(mono);
					panelBackgroundColor.setToolTipText(getText("ui.panelBackgroundColorInfo"));
					panelTextColor.getTextComponent().setFont(mono);
					panelTextColor.setToolTipText(getText("ui.panelTextColorInfo"));
					
					add("", new JLabelFor(getText("ui.heroBackgroundImage"), heroImageType));
					add("tab", heroImageType);
					add("", heroImageDesaturate);
					add("br", heroSlideshowDelayLabel);
					add("tab", heroSlideshowDelay);
					add("br tab", preferThemeImage);
					add("br", new JLabelFor(getText("ui.overlayType"), heroOverlayType));
					add("tab", heroOverlayType);
					add(" ", new JLabelFor(getText("ui.angle"), heroGradientAngle));
					add("", heroGradientAngle);
					add("br", new JLabelFor(getText("ui.heroOverlayColor"), heroOverlayColor));
					add("tab", heroOverlayColor);
					add("tab", heroOverlayColor.getTextComponent());
					add("tab", heroOverlayColor2);
					add("", heroOverlayColor2.getTextComponent());
					add("br", new JLabelFor(getText("ui.titleDecoration"), heroTitleDecoration));
					add("tab", heroTitleDecoration);
					add("br", new JLabelFor(getText("ui.backgroundColor"), backgroundColor));
					add("tab", backgroundColor);
					add("tab", backgroundColor.getTextComponent());
					add("br", new JLabelFor(getText("ui.differentBackground"), differentBackground));
					//add("br", new JLabel(getText("ui.forElements")));
					add("tab", differentBackgroundFor);
					add("br tab", differentBackground);
					add("tab", differentBackground.getTextComponent());
					add("br", new JLabelFor(getText("ui.textColor"), textColor));
					add("tab", textColor);
					add("tab", textColor.getTextComponent());
					add("br", new JLabelFor(getText("ui.accentColor"), buttonColor));
					add("tab", buttonColor);
					add("tab", buttonColor.getTextComponent());
					add("tab", coloredButtons);
					add("br", new JLabelFor(getText("ui.buttonTextColor"), buttonTextColor));
					add("tab", buttonTextColor);
					add("tab", buttonTextColor.getTextComponent());
					add("br", new JLabelFor(getText("ui.panelBackgroundColor"), panelBackgroundColor));
					add("tab", panelBackgroundColor);
					add("tab", panelBackgroundColor.getTextComponent());
					add("br", new JLabelFor(getText("ui.panelTextColor"), panelTextColor));
					add("tab", panelTextColor);
					add("tab", panelTextColor.getTextComponent());
				}
			};
					
			ControlPanel layout = new ControlPanel() {
				
				ControlPanel backgroundImagePanel = new ControlPanel(getText("ui.backgroundImage")) {

					JTextField backgroundImageName = new JSmartTextField(20);
					JButton selectImage = new JButton();
					JComboBox backgroundPos = new JComboBox(getPosition());
					JComboBox backgroundRepeat = new JComboBox(new Object[] {
						new Item("no-repeat", getText("ui.no_repeat")),
						new Item("repeat-x", getText("ui.repeat_x")),
						new Item("repeat-y", getText("ui.repeat_y")),
						new Item("repeat", getText("ui.repeat_both")),
						new Item("stretch", getText("ui.stretch")),
					});

					{
						selectImage.setText(getText("ui.select"));
						selectImage.addActionListener((ActionEvent e) -> {
							getFileToRes(imageFiles, backgroundImageName, skinUi);
						});
						backgroundColor.setToolTipText(getText("ui.backgroundColorInfo"));
						backgroundColor.getTextComponent().setFont(mono);

						add("", backgroundImageName);
						add("", selectImage);
						add("br", new JLabelFor(getText("ui.position"), backgroundPos));
						add("tab", backgroundPos);
						add("br tab", backgroundRepeat);

					}
				};
				
				JComboBox modalWindowsTheme = new JComboBox(new Object[] {
					new Item("auto", getText("ui.auto")),
					new Item("light", getText("ui.light")),
					new Item("dark", getText("ui.dark"))
				});
				
				JComboBox iconStyle = new JComboBox(new Object[] {
					new Item("thin", getText("ui.thin")),
					new Item("fat", getText("ui.fat"))
				});
				
				JCheckBox scrollToTopButton = new JCheckBox(getText("ui.scrollToTopButton"));	
				
				JCheckBox autoclosePanels = new JCheckBox(getText("ui.autoClosePanels"));
				JSpinner autocloseDelay = new JSpinner(new SpinnerNumberModel(20, 3, 120, 1)); 
				
				{
					buttonbarPlacement.setToolTipText(getText("ui.buttonbarPlacementInfo"));
					ComponentUtilities.whenSelectedEnable(autoclosePanels, autocloseDelay);
					autocloseDelay.setToolTipText(getText("ui.autocloseDelayInfo"));
					scrollToTopButton.setToolTipText(getText("ui.scrollToTopButtonInfo"));
					
					add("", new JLabelFor(getText("ui.maxPageWidth"), maxPageWidth));
					add("tab", maxPageWidth);
					add("br", new JLabel(getText("ui.buttonbarPlacement")));
					add("tab", buttonbarPlacement);
					add("br", new JLabelFor(getText("ui.heroHeight"), heroHeight));
					add("tab", heroHeight);
					add(new JLabel("%"));
					add("br", autoclosePanels);
					add("tab", autocloseDelay);
					add("", new JLabel("s"));
					add("br hfill", backgroundImagePanel);
					add("br", new JLabelFor(getText("ui.modalWindowsTheme"), modalWindowsTheme));
					add("tab", modalWindowsTheme);
					add("br", new JLabel(getText("ui.iconStyle")));
					add("tab", iconStyle);
					add("br", new JLabel(getText("ui.cornerRadius")));
					add("tab", cornerRadius);
					add("br", scrollToTopButton);
				}
			};
			
			// Lightbox design

			ControlPanel lightboxDesign = new ControlPanel() {
				
				JColorSelector lightboxColor = new JAlphaColorSelector(getText("ui.backgroundColor"), new JSmartTextField(10));
				JColorSelector lightboxBorderColor = new JAlphaColorSelector(getText("ui.borderColor"), new JSmartTextField(10));
				JCheckBox lightboxDropShadow = new JCheckBox(getText("ui.dropShadow"), false);
				JSpinner lightboxBorderWidth = new JSpinner(new SpinnerNumberModel(0, 0, 40, 1));
				JCheckBox lightboxButtonsBW = new JCheckBox(getText("ui.blackAndWhiteButtons"), false);

				{		
					lightboxColor.getTextComponent().setFont(mono);
					lightboxBorderColor.getTextComponent().setFont(mono);
					lightboxButtonsBW.setToolTipText(getText("ui.blackAndWhiteButtonsInfo"));
					
					add("br", new JLabelFor(getText("ui.backgroundColor"), lightboxColor));
					add("tab", lightboxColor);
					add("tab", lightboxColor.getTextComponent());
					add("br", new JLabelFor(getText("ui.borderColor"), lightboxBorderColor));
					add("tab", lightboxBorderColor);
					add("tab", lightboxBorderColor.getTextComponent());
					add("br", new JLabelFor(getText("ui.borderWidth"), lightboxBorderWidth));
					add("tab", lightboxBorderWidth);
					add("", lightboxDropShadow);
					add("br", lightboxButtonsBW);
				}
			};
			
			ControlPanel languagePanel = new ControlPanel() {
				
				JComboBox<Item> language = new JComboBox<Item>() {
					{
						setModel(Languages.modelFrom(new File(skinDirectory, "texts")));
						insertItemAt(new Item("jalbum", "[ " + getText("ui.jAlbumPreference") +  " ]"), 0);
						setSelectedIndex(0);
					}
				};
				
				{
					language.setToolTipText(getText("ui.languageInfo"));

					add("", new JLabelFor(getText("ui.language"), language));
					add("tab", language);
				}
			};
			
			JTabbedPane pageDesignTabs = new JTabbedPane() {
				
				{
					addTab(getText("ui.colors"), colors);
					addTab(getText("ui.layout"), layout);
					addTab(getText("ui.lightbox"), lightboxDesign);
					addTab(getText("ui.language"), languagePanel);
				}
			};
					
			{
				((BorderLayout)(getLayout())).setVgap(0);
				((BorderLayout)(getLayout())).setHgap(0);
				
				pageDesignTabs.setBorder(emptyBorder);
				
				add(pageDesignTabs);
			}
		};
		
		// Typography

		ControlPanel typography = new ControlPanel() {

			JComboBox baseFontSize = new JComboBox(new Object[] {
				new Item("11", getText("ui.smallest") + " (11px)"),
				new Item("12", getText("ui.tiny") + " (12px)"),
				new Item("13", getText("ui.small") + " (13px)"),
				new Item("14", getText("ui.smaller") + " (14px)"),
				new Item("15", getText("ui.normal") + " (15px)"),
				new Item("16", getText("ui.normal") + " (16px)"),
				new Item("17", getText("ui.larger") + " (17px)"),
				new Item("18", getText("ui.large") + " (18px)"),
				new Item("20", getText("ui.largest") + " (20px)"),
			});
								
			{
				//fontFamily.setEditable(true);
				headlineFont.setEditable(true);

				add(new JLabelFor(getText("ui.pairingSuggestions"), fontSuggestions));
				add("tab", fontSuggestions);
				add("br", new JLabelFor(getText("ui.fontFamily"), fontFamily));
				add("tab", fontFamily);
				add("br", new JLabelFor(getText("ui.baseFontSize"), baseFontSize));
				add("tab", baseFontSize);
				add("br", new JLabelFor(getText("ui.headlineFamily"), headlineFont));
				add("tab", headlineFont);
			}
		};
			
		//	Hero
		
		ControlPanel hero = new ControlPanel(new BorderLayout(20, 0)) {

			ControlPanel heroSettings = new ControlPanel() {
				JTextField logoName = new JSmartTextField(20);
				JButton selectLogo = new JButton(getText("ui.select"));
				JCheckBox showBreadcrumbPath = new JCheckBox(getText("ui.showBreadcrumbPath"));

				JScrollPane titleCaptionPane = new JScrollPane(titleCaptionTemplate);
				JComboBox folderDateSource = new JComboBox(new Object[]{
					new Item("none", getText("ui.none")),
					new Item("fileDate", getText("ui.fileDate")),
					new Item("folderModDate", getText("ui.folderModDate")),
					new Item("lastCameraDate", getText("ui.lastCameraDate")),
					new Item("cameraDateRange", getText("ui.cameraDateRange"))
				});
				JLabel folderDateSourceLabel = new JLabelFor(getText("ui.dateSource"), folderDateSource);
				JCheckBox alwaysRestartSlideshow = new JCheckBox(getText("ui.alwaysRestart"), false);
				JCheckBox slideshowFullscreen = new JCheckBox(getText("ui.goFullScreen"));
				JCheckBox slideshowFullscreenOnlyMobiles = new JCheckBox(getText("ui.onlyOnMobiles"));
				WrappableJLabel heroDesc = new WrappableJLabel("<html><i>" + getText("ui.heroDescription") + "</i></html>");
				
				ControlPanel heroButtons = new ControlPanel(getText("ui.buttons")) {
					
					{
						ComponentUtilities.whenSelectedEnable(useSlideshowBtn, new JComponent[]{ alwaysRestartSlideshow, slideshowFullscreen, slideshowFullscreenOnlyMobiles });
						ComponentUtilities.whenSelectedEnable(slideshowFullscreen, slideshowFullscreenOnlyMobiles);
						
						add("", new JLabel(icon("arrow-down")));
						add("tab", useContinueBtn);
						add("br", new JLabel(icon("play")));
						add("tab", useSlideshowBtn);
						add("br tab", alwaysRestartSlideshow);
						add("br tab", slideshowFullscreen);
						add("br tab", slideshowFullscreenOnlyMobiles);
						
					}
				};
				
				{
					selectLogo.setText(getText("ui.select"));
					selectLogo.addActionListener(new ActionListener() { 
						@Override
						public void actionPerformed(ActionEvent e) {
							getFileToRes(imageFiles, logoName, skinUi);
					}});
					titleCaptionPresets.addItemListener(listener -> {
						String s = getSelectedValue(titleCaptionPresets);
						if (!s.equals(CUSTOM)) {
							titleCaptionTemplate.setText(s);
						}
					});	
					new StateMonitor() {
						public void onChange() {
							String s = getSelectedValue(titleCaptionPresets);
							boolean custom = s.equals(CUSTOM),
								hasDate = custom || s.contains("${folderModDate}");
							titleCaptionPane.setVisible(custom);
							folderDateSourceLabel.setVisible(hasDate);
							folderDateSource.setVisible(hasDate);
						}
					}.add(titleCaptionPresets).done();						
					titleCaptionTemplate.setEditable(true);
					titleCaptionTemplate.setLineWrap(true);
					titleCaptionTemplate.setWrapStyleWord(true);
					titleCaptionTemplate.setFont(mono);
					titleCaptionTemplate.setTabSize(2);
					titleCaptionPane.setBorder(BorderFactory.createTitledBorder(getText("ui.customCode")));
					showBreadcrumbPath.setToolTipText(getText("ui.showBreadcrumbPathInfo"));
					heroDesc.setPreferredWidth(uiWidth - 260);

					add("", new JLabel(getText("ui.logo")));
					add("", logoName);
					add("", selectLogo);
					add("br", showBreadcrumbPath);
					add("br", new JLabelFor(getText("ui.caption"), titleCaptionPresets));
					add("tab", titleCaptionPresets);
					add("br hfill", titleCaptionPane);
					add("br", folderDateSourceLabel);
					add("tab", folderDateSource);
					add("br hfill", heroButtons);
					add("br", new JLabel(infoIcon));
					add("", heroDesc);
				}
			};
					
			ControlPanel customContent = new ControlPanel() {
				
				JSmartTextArea header = new JSmartTextArea(4, 20);
				JScrollPane headerPane = new JScrollPane(header);
				JCheckBox headerFitWidth = new JCheckBox(getText("ui.constrainPageWidth"), true);
				
				{
					allowHTMLEditing(header);
					allowSpelling(header);
					header.setEditable(true);
					header.setLineWrap(true);
					header.setWrapStyleWord(true);
					header.setFont(mono);
					header.setTabSize(2);
					header.setToolTipText(getText("ui.customContentInfo"));
					headerPane.setBorder(BorderFactory.createTitledBorder(getText("ui.customContent")));
					
					add("hfill vfill", headerPane);
					add("br", headerFitWidth);
					add("br", new JLabel(infoIcon));
					add(new JLabel("<html><i>" + getText("ui.addedAboveHero") + "</i></html>"));
				}
			};
			
			JTabbedPane heroTabs = new JTabbedPane() {
				
				{
					addTab(getText("ui.settings"), heroSettings);
					addTab(getText("ui.customContent"), customContent);
				}
			};
			
			{
				((BorderLayout)(getLayout())).setVgap(0);
				((BorderLayout)(getLayout())).setHgap(0);

				heroTabs.setBorder(emptyBorder);
				
				add(heroTabs);
			}

		};
		
		// Images
		
		ControlPanel images = new ControlPanel(new BorderLayout(20, 0)) {

			ControlPanel thumbs = new ControlPanel() {
				
				JLabel thumbThresholdInfo = new JLabel(infoIcon);
				JLabel columnsInfo = new JLabel(infoIcon);
				//JLabel varianceLabel = new JLabelFor(getText("ui.variance"), gridVariance);
				
				JScrollPane thumbCaptionPane = new JScrollPane(thumbCaptionTemplate);
				JCheckBox captionShowOnHover = new JCheckBox(getText("ui.showOnlyOnMouseOver"), true);
				
				{
					thumbThresholdInfo.addMouseListener(new MouseAdapter() {  
						@Override
						public void mouseReleased(MouseEvent e) {
							JOptionPane.showMessageDialog(window, getText("ui.thumbnailThresholdInfo"), "Info", JOptionPane.INFORMATION_MESSAGE);
					}});

					columnsInfo.addMouseListener(new MouseAdapter() {  
						@Override
						public void mouseReleased(MouseEvent e) {
							JOptionPane.showMessageDialog(window, getText("ui.columnsInfo") + "\n\n" + getText("ui.masonryColumnsMaxInfo"), "Info", JOptionPane.INFORMATION_MESSAGE);
					}});

					captionPlacement.addActionListener((ActionEvent e) -> {
						int i = captionPlacement.getSelectedIndex();
						captionStyle.setEnabled(i <= 2);
						captionShowOnHover.setEnabled(i <= 2);
					});
					
					thumbLayout.addActionListener((ActionEvent e) -> {
						thumbAspectRatio.setEnabled(thumbLayout.getSelectedIndex() == 0);
						//gridVariance.setVisible(thumbLayout.getSelectedIndex() == 2);
						//varianceLabel.setVisible(thumbLayout.getSelectedIndex() == 2);
					});
					
					thumbCaptionPresets.addItemListener(listener -> {
						String s = getSelectedValue(thumbCaptionPresets);
						if (!s.equals(CUSTOM)) {
							thumbCaptionTemplate.setText(s);
						}
					});	
					new StateMonitor() {
						public void onChange() {
							String s = getSelectedValue(thumbCaptionPresets);
							thumbCaptionPane.setVisible(s.equals(CUSTOM));
						}
					}.add(thumbCaptionPresets).done();						
					thumbCaptionPane.setBorder(BorderFactory.createTitledBorder(getText("ui.customCode")));
					thumbCaptionTemplate.setEditable(true);
					thumbCaptionTemplate.setLineWrap(true);
					thumbCaptionTemplate.setWrapStyleWord(true);
					thumbCaptionTemplate.setFont(mono);
					
					add("", new JLabelFor(getText("ui.thumbnailThreshold"), thumbThreshold));
					add("tab", thumbThreshold);
					add(thumbThresholdInfo);
					add("br", new JLabelFor(getText("ui.thumbnailLayout"), thumbLayout));
					add("tab", thumbLayout);
					//add("tab", varianceLabel);
					//add("", gridVariance);
					add("br", new JLabelFor(getText("ui.aspectRatio"), thumbAspectRatio));
					add("tab", thumbAspectRatio);
					add("br", new JLabelFor(getText("ui.columns"), columns));
					add("tab", columns);
					add(columnsInfo);
					add("br", new JLabelFor(getText("ui.thumbGap"), thumbGap));
					add("tab", thumbGap);
					add("br", new JLabelFor(getText("ui.captionPlacement"), captionPlacement));
					add("tab", captionPlacement);
					add("br", new JLabelFor(getText("ui.style"), captionStyle));
					add("tab", captionStyle);
					add("br", captionShowOnHover);
					add("br", new JLabelFor(getText("ui.caption"), thumbCaptionPresets));
					add("tab", thumbCaptionPresets);
					add("br hfill", thumbCaptionPane);
				}
			};
			
			ControlPanel single = new ControlPanel() {
				
				JLabel wideThresholdInfo = new JLabel(infoIcon);
				
				JScrollPane singleCaptionPane = new JScrollPane(singleCaptionTemplate);
				
				{
					wideThresholdInfo.addMouseListener(new MouseAdapter() {  
						@Override
						public void mouseReleased(MouseEvent e) {
							JOptionPane.showMessageDialog(window, getText("ui.wideThresholdInfo"), "Info", JOptionPane.INFORMATION_MESSAGE);
					}});
					
					singleCaptionPresets.addItemListener(listener -> {
						String s = getSelectedValue(singleCaptionPresets);
						if (!s.equals(CUSTOM)) {
							singleCaptionTemplate.setText(s);
						}
					});	
					new StateMonitor() {
						public void onChange() {
							String s = getSelectedValue(singleCaptionPresets);
							singleCaptionPane.setVisible(s.equals(CUSTOM));
						}
					}.add(singleCaptionPresets).done();						
					singleCaptionPane.setBorder(BorderFactory.createTitledBorder(getText("ui.customCode")));
					singleCaptionTemplate.setEditable(true);
					singleCaptionTemplate.setLineWrap(true);
					singleCaptionTemplate.setWrapStyleWord(true);
					singleCaptionTemplate.setFont(mono);
					
					add("", new JLabelFor(getText("ui.wideThreshold"), wideThreshold));
					add("tab", wideThreshold);
					add(wideThresholdInfo);
					add("br", new JLabelFor(getText("ui.caption"), singleCaptionPresets));
					add("tab", singleCaptionPresets);
					add("br hfill", singleCaptionPane);
				}
			};

			ControlPanel fullWidth = new ControlPanel() {
				
				JScrollPane fullCaptionPane = new JScrollPane(fullCaptionTemplate);
				
				{
					fullCaptionPresets.addItemListener(listener -> {
						String s = getSelectedValue(fullCaptionPresets);
						if (!s.equals(CUSTOM)) {
							fullCaptionTemplate.setText(s);
						}
					});	
					new StateMonitor() {
						public void onChange() {
							String s = getSelectedValue(fullCaptionPresets);
							fullCaptionPane.setVisible(s.equals(CUSTOM));
						}
					}.add(fullCaptionPresets).done();						
					fullCaptionPane.setBorder(BorderFactory.createTitledBorder(getText("ui.customCode")));
					fullCaptionTemplate.setEditable(true);
					fullCaptionTemplate.setLineWrap(true);
					fullCaptionTemplate.setWrapStyleWord(true);
					fullCaptionTemplate.setFont(mono);
					
					add("", new JLabelFor(getText("ui.caption"), fullCaptionPresets));
					add("tab", fullCaptionPresets);
					add("br hfill", fullCaptionPane);
				}
			};
			
			ControlPanel extra = new ControlPanel() {
				
				WrappableJLabel treatInfo = new WrappableJLabel("<html><i>" + getText("ui.treatImageInfo") + "</i></html>");
				JCheckBox rightClickProtect = new JCheckBox(getText("ui.rightClickProtect"));
				JCheckBox autoplayVideoInView = new JCheckBox(getText("ui.autoplayVideoInView"));
				
				{
					treatInfo.setPreferredWidth(uiWidth - 230);
					rightClickProtect.setToolTipText(getText("ui.rightClickProtectInfo"));

					add("br", new JLabel(infoIcon));
					add("", treatInfo);
					add("br", rightClickProtect);
					add("br", autoplayVideoInView);
					add("br", new JLabel(getText("ui.hoverEffect")));
					add("tab", hoverEffect);
				}
			};

			JTabbedPane imageTabs = new JTabbedPane() {
				
				{
					addTab(getText("ui.thumbnail"), thumbs);
					addTab(getText("ui.singleImage"), single);
					addTab(getText("ui.fullWidth"), fullWidth);
				}
			};
			
			{
				((BorderLayout)(getLayout())).setVgap(0);
				((BorderLayout)(getLayout())).setHgap(0);
				
				imageTabs.setBorder(emptyBorder);
				extra.setBorder(emptyBorder);
				
				{
					add(imageTabs, BorderLayout.NORTH);
					add(extra, BorderLayout.SOUTH);
				}
			}
		};
		
		// Rating
		
		ControlPanel rating = new ControlPanel() {

			JCheckBox useRating = new JCheckBox(getText("ui.displayRating"), false);
			JCheckBox visitorRating = new JCheckBox(getText("ui.allowVisitorRatings"), false);
			JCheckBox useJalbumRating = new JCheckBox(getText("ui.useJalbumRatings"), false);
			
			{
				useRating.setToolTipText(getText("ui.displayRatingInfo"));
				visitorRating.setToolTipText(getText("ui.allowVisitorRatingsInfo"));
				useJalbumRating.setToolTipText(getText("ui.useJalbumRatingsInfo"));
				ComponentUtilities.whenSelectedEnable(useRating, new JComponent[]{ visitorRating, useJalbumRating});
				
				add("br", useRating);
				add("br", useJalbumRating);
				add("br", visitorRating);
			}
		};
				
		// Mark new files
		
		ControlPanel markNew = new ControlPanel() {

			JCheckBox markFilesNew = new JCheckBox(getText("ui.markFilesNew"));	
			JTextField newDaysCount = new JSmartTextField("60", 3);
			JComboBox newDaysRef = new JComboBox(new Object[] {
				new Item("dateTaken", getText("ui.dateTaken")),
				new Item("fileModified", getText("ui.fileModified")),
				new Item("added", getText("ui.addedToAlbum"))
			});
			JComboBox newDaysMark = new JComboBox(new Object[] {
				new Item("icon", getText("ui.icon")),
				new Item("text", getText("ui.text"))
			});
			JTextField newDaysText = new JSmartTextField(getText("new"), 6);
			JLabel newIcon = new JLabel(icon("new-fill"));
			
			{
				newDaysCount.setToolTipText(getText("ui.newDaysCountInfo"));
				newDaysMark.addItemListener(listener -> {
					int i = newDaysMark.getSelectedIndex();
					newDaysText.setVisible(i == 1);
					newIcon.setVisible(i == 0);
				});
				newDaysText.setVisible(false);
				ComponentUtilities.whenSelectedEnable(markFilesNew, new JComponent[]{ newDaysCount, newDaysRef, newDaysMark, newDaysText, newIcon });
				
				add("", markFilesNew);
				add("", newDaysCount);
				add(new JLabel(getText("ui.daysOld")));
				add("br", new JLabel(getText("ui.reference")));
				add("tab", newDaysRef);
				add("br", new JLabel(getText("ui.marker")));
				add("tab", newDaysMark);
				add(newIcon);
				add(newDaysText);
			}
		};
				
		// Folders
		
		ControlPanel folders = new ControlPanel() {
			
			ButtonGroup defaultFolderIcon = new ButtonGroup();
			private JRadioButton defaultFolderIconBook = new JRadioButton("");
			private JRadioButton defaultFolderIconYellow = new JRadioButton("");
			private JRadioButton defaultFolderIconBlue = new JRadioButton("");
			private JRadioButton defaultFolderIconHollow = new JRadioButton("");
			JScrollPane folderCaptionPane = new JScrollPane(folderCaptionTemplate);

			{
				defaultFolderIcon.add(defaultFolderIconBook);
				defaultFolderIcon.add(defaultFolderIconYellow);
				defaultFolderIcon.add(defaultFolderIconBlue);
				defaultFolderIcon.add(defaultFolderIconHollow);
				defaultFolderIconBook.setActionCommand("book");
				defaultFolderIconYellow.setActionCommand("yellow");
				defaultFolderIconBlue.setActionCommand("blue");
				defaultFolderIconHollow.setActionCommand("hollow");
				defaultFolderIconBook.setSelected(true);
				
				folderCaptionPresets.addItemListener(listener -> {
					String s = getSelectedValue(folderCaptionPresets);
					if (!s.equals(CUSTOM)) {
						folderCaptionTemplate.setText(s);
					}
				});	
				new StateMonitor() {
					public void onChange() {
						String s = getSelectedValue(folderCaptionPresets);
						folderCaptionPane.setVisible(s.equals(CUSTOM));
					}
				}.add(folderCaptionPresets).done();						
				folderCaptionTemplate.setEditable(true);
				folderCaptionTemplate.setLineWrap(true);
				folderCaptionTemplate.setWrapStyleWord(true);
				folderCaptionTemplate.setFont(mono);
				folderCaptionPane.setBorder(BorderFactory.createTitledBorder(getText("ui.customCode")));
				showFolderImageCount.setToolTipText(getText("ui.showFolderImageCountInfo"));

				add(new JLabel(getText("ui.columns")));
				add("tab", folderColumns);
				add("tab", folderMosaic);
				add("br", new JLabel(getText("ui.defaultFolderIcon")));
				add("tab", defaultFolderIconBook);
				add(new JLabel(svgIcon("folder-s-book", new Dimension(32, 32))));
				add("", defaultFolderIconYellow);
				add(new JLabel(svgIcon("folder-s-yellow", new Dimension(32, 32))));
				add("", defaultFolderIconBlue);
				add(new JLabel(svgIcon("folder-s-blue", new Dimension(32, 32))));
				add("", defaultFolderIconHollow);
				add(new JLabel(svgIcon("folder-s-hollow", new Dimension(32, 32))));
				add("br", new JLabelFor(getText("ui.caption"), folderCaptionPresets));
				add("tab", folderCaptionPresets);
				add("br hfill", folderCaptionPane);
				add("br", showFolderImageCount);

			}
		};

		// Pages
		
		ControlPanel pages = new ControlPanel() {

			{
				add(new JLabel(getText("ui.showPagesInStory")));
				add("tab", showPages);
			}
		};

		// Pages
		
		ControlPanel weblocations = new ControlPanel() {

			JCheckBox webLocationOpenNew = new JCheckBox(getText("ui.webLocationsOpenInNewWindow"));
			JComboBox webLocationCaptionPlacement = new JComboBox(new Object[] {
				new Item("hide", getText("ui.hidden")),
				new Item("beside", getText("ui.beside")),
				new Item("over", getText("ui.over"))
			});
			JCheckBox gatherWeblocationInfo = new JCheckBox(getText("ui.gatherWeblocationInfo"), true);

			ControlPanel webLocationOptions = new ControlPanel() {
				
				{
					gatherWeblocationInfo.setToolTipText(getText("ui.gatherWeblocationInfoInfo"));
					
					add("", new JLabel(getText("ui.captionPlacement")));
					add("tab", webLocationCaptionPlacement);
					add("br", gatherWeblocationInfo);
					add("br", webLocationOpenNew);
				}
			};
			
			{
				showWeblocations.addActionListener((ActionEvent e) -> {
						int i = showWeblocations.getSelectedIndex();
						webLocationOptions.setEnabled(i > 0 && i < 5);
						gatherWeblocationInfo.setEnabled(i > 1 && i < 5);
						webLocationCaptionPlacement.setEnabled(i == 3);
					});
				
				add(new JLabel(getText("ui.showWeblocationsInStory")));
				add("tab", showWeblocations);
				add("br hfill", webLocationOptions);
			}
		};

		// Neighboring folders
		
		ControlPanel neighboringFolders = new ControlPanel() {
			
			JCheckBox linkNeighboringFolders = new JCheckBox(getText("ui.linkNeighboringFolders"), true);
			JCheckBox neighboringFolderBgImage = new JCheckBox(getText("ui.neighboringFolderBgImage"), true);
			JCheckBox neighboringFolderSkipLevels = new JCheckBox(getText("ui.skipFolderLevels"), false);
			JCheckBox neighboringFolderLoop = new JCheckBox(getText("ui.neighboringFolderLoop"), false);

			{
				linkNeighboringFolders.setToolTipText(getText("ui.linkNeighboringFoldersInfo"));
				neighboringFolderBgImage.setToolTipText(getText("ui.neighboringFolderBgImageInfo"));
				neighboringFolderSkipLevels.setToolTipText(getText("ui.skipFolderLevelsInfo"));
				neighboringFolderLoop.setToolTipText(getText("ui.neighboringFolderLoopInfo"));
				ComponentUtilities.whenSelectedEnable(linkNeighboringFolders, new JComponent[]{ neighboringFolderBgImage, neighboringFolderSkipLevels, neighboringFolderLoop });

				add(linkNeighboringFolders);
				add("br", neighboringFolderSkipLevels);
				add("br", neighboringFolderBgImage);
				add("br", neighboringFolderLoop);
				
				//putClientProperty("helpPage", helpRoot + "/Sections/Neighboring_folders");
			}
		};
								
		// Commenting section
		/*
		ControlPanel commenting = new ControlPanel() {

			ControlPanel fbCommenting = new ControlPanel("Facebook") {

				JCheckBox facebookCommenting = new JCheckBox(getText("ui.facebookCommenting"));
				JTextField facebookCommentingPosts = new JSmartTextField("3", 2);

				{
					facebookCommentingPosts.setToolTipText(getText("ui.facebookCommentingPostsInfo"));

					add(facebookCommenting);
					add("br", new JLabel(getText("ui.facebookCommentingPosts")));
					add("tab", facebookCommentingPosts);
					add("br", new JLabel(infoIcon));
					add(new JLabel("<html><i>" + getText("ui.fbAppIdHint") + "</i></html>"));
				}

			};

			ControlPanel disqCommenting = new ControlPanel("Disqus") {

				JCheckBox disqusCommenting = new JCheckBox(getText("ui.disqusCommenting"));
				JTextField disqusAppId = new JSmartTextField("");
				JLabel mandatoryInfo = new JLabel(mandatory);
				
				{
					disqusAppId.setToolTipText(getText("ui.disqusAppIdInfo"));
					mandatoryInfo.setToolTipText(getText("ui.mandatory"));
					ComponentUtilities.whenSelectedEnable(disqusCommenting, new JComponent[]{disqusAppId, mandatoryInfo});

					add(disqusCommenting);
					add("br", new JLabelFor(getText("ui.disqusAppId"), disqusAppId));
					add(mandatoryInfo);
					add("tab hfill", disqusAppId);
					add("", new JLinkLabel("https://disqus.com/admin/create/", getText("ui.signUp")));
				}

			};

			{
				add("hfill", fbCommenting);
				add("br hfill", disqCommenting);

				//putClientProperty("helpPage", helpRoot + "Sections/Commenting");
			}
		};
		*/
		
		// Footer
		
		ControlPanel footer = new ControlPanel(new BorderLayout(20, 0)) {

			ControlPanel bottomNavigationPanel = new ControlPanel(getText("ui.bottomNavigation")) {
				JCheckBox bottomNavigationIncludeFolders = new JCheckBox(getText("ui.includeFolders"), false);
				JCheckBox bottomNavigationIncludePages = new JCheckBox(getText("ui.includePages"), false);
				JCheckBox bottomNavigationIncludeWebLocations = new JCheckBox(getText("ui.includeWebLocations"), false);
				
				{
					//showBottomNavigation.setToolTipText(getText("ui.showBottomNavigationInfo"));
					//add("", showBottomNavigation);
					add("", bottomNavigationIncludeFolders);
					add("br", bottomNavigationIncludePages);
					add("br", bottomNavigationIncludeWebLocations);
				}
			};

			ControlPanel customLinkPanel = new ControlPanel(getText("ui.customLink")) {

				JTextField customLink = new JSmartTextField(24);
				JTextField customLinkText = new JSmartTextField(24);

				{
					customLink.setToolTipText(getText("ui.customLinkInfo"));
					customLinkText.setToolTipText(getText("ui.customLinkTextInfo"));

					add(new JLabel("URL"));
					add("tab", customLink);
					add("br", new JLabel(getText("ui.customLinkText")));
					add("tab", customLinkText);
				}
			};

			JCheckBox showModifiedDate = new JCheckBox(getText("ui.showModifiedDate"));	
			JCheckBox showImageCount = new JCheckBox(getText("ui.showFolderImageCount"));	

			ControlPanel fbBox = new ControlPanel() {

				JCheckBox useFacebookBox = new JCheckBox(getText("ui.useFacebookBox"));
				JTextField facebookPageId = new JSmartTextField(16);
				JLabel mandatoryInfo = new JLabel(mandatory);
				JCheckBox facebookFaces = new JCheckBox(getText("ui.showFaces"));
				JCheckBox facebookHeader = new JCheckBox(getText("ui.showHeader"));
				WrappableJLabel fbInfo = new WrappableJLabel("<html><i>" + getText("ui.facebookBoxInfo") + "</i></html>");

				{
					mandatoryInfo.setToolTipText(getText("ui.mandatory"));
					ComponentUtilities.whenSelectedEnable(useFacebookBox, new JComponent[] {facebookPageId, facebookFaces, facebookHeader});
					fbInfo.setPreferredWidth(uiWidth - 250);
					
					add(useFacebookBox);
					add("br", new JLabelFor(getText("ui.facebookPageId"), facebookPageId));
					add(mandatoryInfo);
					add("tab", facebookPageId);
					add("br", facebookHeader);
					add("", facebookFaces);
					add("br", new JLabel(infoIcon));
					add(fbInfo);
				}

			};
			
			ControlPanel customFooter = new ControlPanel() {

				JSmartTextArea footer = new JSmartTextArea(4, 20);
				JScrollPane footerPane = new JScrollPane(footer);
				JCheckBox footerFitWidth = new JCheckBox(getText("ui.constrainPageWidth"), true);
				JCheckBox footerTopLevelOnly = new JCheckBox(getText("ui.topLevelOnly"), false);

				{
					allowHTMLEditing(footer);
					allowSpelling(footer);
					
					footer.setEditable(true);
					footer.setLineWrap(true);
					footer.setWrapStyleWord(true);
					footer.setFont(mono);
					footer.setTabSize(2);
					footer.setToolTipText(getText("ui.customContentInfo"));
					footerPane.setBorder(BorderFactory.createTitledBorder(getText("ui.pageBottomText")));

					add("hfill vfill", footerPane);
					add("br", footerFitWidth);
					add("tab", footerTopLevelOnly);
				}
			};
			
			ControlPanel footerSettings = new ControlPanel() {
				
				{
					showImageCount.setToolTipText(getText("ui.showFolderImageCountInfo"));

					add("hfill", bottomNavigationPanel);
					add("br hfill", customLinkPanel);				
					add("br", showModifiedDate);
					add("br", showImageCount);
				}
			};
			
			JTabbedPane footerTabs = new JTabbedPane() {
				
				{
					addTab(getText("ui.settings"), footerSettings);
					addTab(getText("ui.facebookBox"), fbBox);
					addTab(getText("ui.customContent"), customFooter);
				}
			};
			
			{
				((BorderLayout)(getLayout())).setVgap(0);
				((BorderLayout)(getLayout())).setHgap(0);

				footerTabs.setBorder(emptyBorder);
				
				add(footerTabs);
				
				//putClientProperty("helpPage", helpRoot + "Footer/Settings");
			}

		};
										
		/*	---------------------------------------------------------------
									Lightbox
			--------------------------------------------------------------- */
		
		ControlPanel lightboxNavigation = new ControlPanel(getText("ui.navigation")) {
			
			JCheckBox enableKeyboard = new JCheckBox(getText("ui.enableKeyboard"), true);
			JComboBox mouseWheelAction = new JComboBox(new Object[] {
				new Item("", getText("ui.default")), 
				new Item("navigation", getText("ui.navigation")),
				new Item("zoom", getText("ui.zoom"))
			});
			JComboBox clickAction = new JComboBox(new Object[] {
				new Item("donothing", getText("ui.doNothing")), 
				new Item("nextimage", getText("ui.nextImage")),
				new Item("togglecontrols", getText("ui.toggleControls")),
				new Item("toggleall", getText("ui.toggleAllPanels"))
			});
			JCheckBox clickBesideForIndex = new JCheckBox(getText("ui.clickBesideForIndex"));
			JComboBox afterLast = new JComboBox(new Object[] {
				new Item("donothing", getText("ui.doNothing")), 
				new Item("startover", getText("startOver")),
				new Item("onelevelup", getText("upOneLevel")),
				new Item("backtoindex", getText("backToIndex")),
				new Item("nextfolder", getText("ui.nextFoldersFirstImage")),
				new Item("nextindex", getText("nextIndex")),
				new Item("ask", getText("ui.ask"))
			});

			{
				afterLast.setToolTipText(getText("ui.afterLastInfo"));
				enableKeyboard.setToolTipText(getText("ui.enableKeyboardInfo"));
				mouseWheelAction.setToolTipText(getText("ui.mouseWheelActionInfo"));
				clickBesideForIndex.setToolTipText(getText("ui.clickBesideForIndexInfo"));
				add("", enableKeyboard);
				add("br", new JLabelFor(getText("ui.mouseWheelAction"), mouseWheelAction));
				add("tab", mouseWheelAction);
				add("br", new JLabelFor(getText("ui.clickAction"), clickAction));
				add("tab", clickAction);
				add("tab", clickBesideForIndex);
				add("br", new JLabelFor(getText("ui.afterLast"), afterLast));
				add("tab", afterLast);
			}
		};
		
		ControlPanel lightboxSettings = new ControlPanel() {

			@Override
			public void importVariables(Map<String, Object> vars) {
				Map<String, Object> skinVars = engine.getSkinVariables();
				Object sdo = skinVars.get("slideshowDelay");
				if (sdo != null && sdo instanceof Double) {
					if ((Double)sdo > 0.0 && (Double)sdo < 20.0) {
						vars.put("slideshowDelay", (int)((Double)sdo * 1000));
					}
				}
			}
			JCheckBox lightboxFullscreen = new JCheckBox(getText("ui.lightboxFullscreen"));
			JCheckBox videoAutoPlay = new JCheckBox(getText("ui.autoPlayVideos"));
			JLabel videoAutoPlayInfo = new JLabel(infoIcon);
			JCheckBox videoLoop = new JCheckBox(getText("ui.loopVideos"));
			JCheckBox autohideControls = new JCheckBox(getText("ui.autohideControls"));
			JCheckBox autohideCaption = new JCheckBox(getText("ui.autohideCaption"));
			JSpinner transitionSpeed = new JSpinner(new SpinnerNumberModel(1000, 100, 10000, 100));
			JSpinner slideshowDelay = new JSpinner(new SpinnerNumberModel(4000, 100, 20000, 100));
			JCheckBox slideshowTimingControl = new JCheckBox(getText("ui.canChangeSlideshowTiming"));
			JCheckBox useAutoPano = new JCheckBox(getText("ui.autoPanPanoramas"));
			JLabel autoPanoInfo = new JLabel(infoIcon);
			JLabel autoPanoStartLabel = new JLabel(getText("ui.startAt"));
			JComboBox autoPanoStart = new JComboBox(new Object[] {
				new Item("left", getText("ui.leftOrTop")), 
				new Item("center", getText("ui.center")),
				new Item("right", getText("ui.rightOrBottom"))
			});
			JLabel autoPanoSpeedLabel = new JLabel(getText("ui.speed"));
			JComboBox autoPanoSpeed = new JComboBox(new Object[] {
				new Item("veryslow", getText("ui.verySlow")),
				new Item("slow", getText("ui.slow")), 
				new Item("moderate", getText("ui.moderate")),
				new Item("medium", getText("ui.medium")),
				new Item("quick", getText("ui.quick")),
				new Item("fast", getText("ui.fast"))
			});
			JCheckBox use360Player = new JCheckBox(getText("ui.use360Player"));
			JCheckBox autoRotate360 = new JCheckBox(getText("ui.autoRotate"));
			JSpinner rotateSpeed360 = new JSpinner(new SpinnerNumberModel(-2, -8, 8, 1));
			JLabel rpm = new JLabel(getText("ui.rpm"));
			JCheckBox useGoogleDocs = new JCheckBox(getText("ui.useGoogleDocsViewer"));
			ControlPanel autoPanoSettings = new ControlPanel() {
				{
					add("", autoPanoStartLabel);
					add("tab", autoPanoStart);
					add(" ", autoPanoSpeedLabel);
					add("tab", autoPanoSpeed);
				}
			};

			{
				transitionSpeed.setToolTipText(getText("ui.transitionSpeedInfo"));
				slideshowDelay.setToolTipText(getText("ui.slideshowDelayInfo"));
				slideshowTimingControl.setToolTipText(getText("ui.slideshowTimingControlInfo"));
				videoAutoPlayInfo.addMouseListener(new MouseAdapter() {  
					@Override
					public void mouseReleased(MouseEvent e) {
						JOptionPane.showMessageDialog(window, getText("ui.videoAutoplayInfo1"), "Warning", JOptionPane.WARNING_MESSAGE);
				}});
				autohideControls.setToolTipText(getText("ui.autohideControlsInfo"));
				ComponentUtilities.whenSelectedEnable(autohideControls, autohideCaption);
				useAutoPano.setToolTipText(getText("ui.enableAutoPanoInfo"));
				useAutoPano.addItemListener(e -> {
					if (isSkinReady() && e.getStateChange() == ItemEvent.SELECTED) {
						String idim = engine.getImageSize();
						int iw = Integer.parseInt(idim.split("x")[0]);
						int ih = Integer.parseInt(idim.split("x")[1]);

						if (iw < 3000) {
							Object[] options = { 
								getText("yes"),
								getText("no")
							};
							int n = JOptionPane.showOptionDialog(window, 
								getText("ui.autoPanoInfo") + "\n\n" + getText("ui.changeImageDimensionsQuestion"), 
								getText("ui.imageDimensionsTooSmall"), 
								JOptionPane.YES_NO_OPTION, 
								JOptionPane.INFORMATION_MESSAGE,
								null,
								options,
								options[0]
							);
							if (n == 0) {
								try {
									window.ui2Engine();
									engine.setImageSize("4000x"+ih);
									JAlbum.logger.log(Level.FINE, "Image width has changed from {0}px to {1}px.", new Object[]{iw, "4000"});
									window.engine2UI();
								} catch (ParameterException ex) {
									throw new RuntimeException(ex);
								}
								useAutoPano.setSelected(true);
							}
						}
					}
				});
				autoPanoInfo.addMouseListener(new MouseAdapter() {  
					@Override
					public void mouseReleased(MouseEvent e) {
						JOptionPane.showMessageDialog(window, getText("ui.autoPanoInfo"), "Warning", JOptionPane.WARNING_MESSAGE);
				}});
				ComponentUtilities.whenSelectedEnable(useAutoPano, new JComponent[]{autoPanoSettings});
				use360Player.setToolTipText(getText("ui.use360PlayerInfo"));
				ComponentUtilities.whenSelectedEnable(use360Player, new JComponent[]{autoRotate360, rotateSpeed360, rpm});
				useGoogleDocs.setToolTipText(getText("ui.useGoogleDocsInfo"));

				add("br", new JLabelFor(getText("ui.transitionSpeed"), transitionSpeed));
				add("tab", transitionSpeed);
				add(new JLabel("ms"));
				add("br", new JLabel(getText("ui.slideshowDelay")));
				add("tab", slideshowDelay);
				add(new JLabel("ms"));
				add("tab", slideshowTimingControl);
				add("br", lightboxFullscreen);
				add("br", videoAutoPlay);
				add("", videoAutoPlayInfo);
				add("tab", videoLoop);
				add("br", autohideControls);
				add("tab", autohideCaption);
				add("br", useAutoPano);
				add("", autoPanoInfo);
				add("tab", autoPanoSettings);
				add("br", use360Player);
				add("tab", autoRotate360);
				add("", rotateSpeed360);
				add(rpm);
				add("br", useGoogleDocs);
				add("br hfill", lightboxNavigation);
				//add("tab", slideshowAuto);
			}
		};

		ControlPanel lightboxMainImage = new ControlPanel() {

			JCheckBox fitImage = new JCheckBox(getText("ui.fitImages"));
			JCheckBox neverScaleUp = new JCheckBox(getText("ui.neverScaleUp"));
			JLabel maxZoomLabel = new JLabel(getText("ui.maxZoom"));
			JSpinner maxZoom = new JSpinner(new SpinnerNumberModel(1.4, 0.99, 3.0, 0.1));
			JSpinner fitPadding = new JSpinner(new SpinnerNumberModel(0, 0, 100, 1));
			JCheckBox fitBetween = new JCheckBox(getText("ui.fitBetween"), false);
			JCheckBox zoomSlider = new JCheckBox(getText("ui.useZoomSlider"), true);
			JCheckBox displayOriginals = new JCheckBox(getText("ui.displayOriginals"));

			{
				neverScaleUp.setToolTipText(getText("ui.neverScaleUpInfo"));
				((JSpinner.DefaultEditor)maxZoom.getEditor()).getTextField().setColumns(4);
				fitImage.setToolTipText(getText("ui.fitImagesInfo"));
				//ComponentUtilities.whenSelectedEnable(fitImages, new JComponent[]{neverScaleUp});
				//ComponentUtilities.whenSelectedDisable(neverScaleUp, new JComponent[]{maxZoomLabel, maxZoom});
				displayOriginals.setToolTipText(getText("ui.displayOriginalsInfo"));

				add("", new JLabelFor(getText("ui.padding"), fitPadding));
				add("tab", fitPadding);
				add(new JLabel("px"));
				add("br", fitImage);
				add("tab", fitBetween);
				add("br", zoomSlider);
				add("br", maxZoomLabel);
				add("tab", maxZoom);
				add(" ", new JLabel("<html><i>1.0: " + getText("ui.neverScaleUp") + "</i></html>"));
				add("br", displayOriginals);
			}
		};

		/*
		ControlPanel lightboxSettings = new ControlPanel() {
			
			{
				add("hfill", lightboxBehavior);
				add("br hfill", lightboxMainImage);
				add("br hfill", lightboxNavigation);
			}
		};
		*/
		ControlPanel lightboxCaption = new ControlPanel() {

			JScrollPane imageCaptionPane = new JScrollPane(imageCaptionTemplate);
			JCheckBox showImageNumbers = new JCheckBox(getText("ui.showImageNumbers"));

			{
				imageCaptionPresets.addItemListener(listener -> {
					String s = getSelectedValue(imageCaptionPresets);
					if (!s.equals(CUSTOM)) {
						imageCaptionTemplate.setText(s);
					}
				});	
				new StateMonitor() {
					public void onChange() {
						String s = getSelectedValue(imageCaptionPresets);
						imageCaptionPane.setVisible(s.equals(CUSTOM));
					}
				}.add(imageCaptionPresets).done();						
				imageCaptionTemplate.setEditable(true);
				imageCaptionTemplate.setLineWrap(true);
				imageCaptionTemplate.setWrapStyleWord(true);
				imageCaptionTemplate.setFont(mono);
				imageCaptionPane.setBorder(BorderFactory.createTitledBorder(getText("ui.customCode")));
				showImageNumbers.setToolTipText(getText("ui.showImageNumbersInfo"));

				add("", new JLabelFor(getText("ui.caption"), imageCaptionPresets));
				add("tab", imageCaptionPresets);
				add("br hfill", imageCaptionPane);
				add("br", showImageNumbers);
			}
		};
			
		/*	---------------------------------------------------------------
									Functions
			--------------------------------------------------------------- */
						
		// Full screen
		
		ControlPanel fullScreenPanel = new ControlPanel() {
		
			JCheckBox showFullscreen = new JCheckBox(getText("ui.showFullScreenButton"), false);
			
			{
				add(showFullscreen);
			}
		};
												
		// Navigation
		
		ControlPanel navigationPanel = new ControlPanel() {

			JCheckBox topNavigationIncludeFolders = new JCheckBox(getText("ui.includeFolders"));
			JCheckBox topNavigationIncludePages = new JCheckBox(getText("ui.includePages"));
			JCheckBox topNavigationIncludeWebLocations = new JCheckBox(getText("ui.includeWebLocations"));
			JSpinner topNavigationDepth = new JSpinner(new SpinnerNumberModel(3, 1, 6, 1));
			JCheckBox topNavigationCollapsible = new JCheckBox(getText("ui.collapsible"));
			JSpinner maxTitleLength = new JSpinner(new SpinnerNumberModel(48, 24, 160, 8));

			{
				add("", topNavigationIncludeFolders);
				add("br", topNavigationIncludePages);
				add("br", topNavigationIncludeWebLocations);
				add("br", new JLabel(getText("ui.depth")));
				add("tab", topNavigationDepth);
				add("", new JLabel(getText("ui.levels")));
				add("br", new JLabel(getText("ui.maxTitleLength")));
				add("tab", maxTitleLength);
				add("", new JLabel(getText("ui.characters")));
				add("br", topNavigationFull);
				add("br", topNavigationCollapsible);
			}
		};

		// Album info
		
		ControlPanel albumInfoPanel = new ControlPanel() {

			JCheckBox albumInfoTopLevelOnly = new JCheckBox(getText("ui.topLevelOnly"), true);
			JSmartTextArea albumInfo = new JSmartTextArea(6, 20);
			JScrollPane albumInfoPane = new JScrollPane(albumInfo);

			{
				allowHTMLEditing(albumInfo);
				allowSpelling(albumInfo);
				
				albumInfo.setEditable(true);
				albumInfo.setLineWrap(false);
				albumInfo.setFont(mono);
				albumInfo.setTabSize(2);
				albumInfo.setToolTipText(getText("ui.customContentInfo"));
				albumInfoPane.setBorder(BorderFactory.createTitledBorder(getText("ui.albumInfoText")));
				
				add("hfill vfill", albumInfoPane);
				add("br", albumInfoTopLevelOnly);
				
				//putClientProperty("helpPage", helpRoot + "Site/Album_info");
			}
		};

		// Social
		
		ControlPanel socialPanel = new ControlPanel() {
			
			/*
			ControlPanel buttons = new ControlPanel(getText("ui.buttons")) {
				JCheckBox facebookLike = new JCheckBox(getText("ui.facebookLikeButton"));
				JCheckBox twitterTweet = new JCheckBox(getText("ui.tweetButton"));
				JCheckBox pinItButton = new JCheckBox(getText("ui.pinItButton"));

				{
					facebookLike.setToolTipText(getText("ui.facebookLikeButtonInfo"));
					twitterTweet.setToolTipText(getText("ui.tweetButtonInfo"));
					pinItButton.setToolTipText(getText("ui.pinItButtonInfo"));

					add(facebookLike);
					add("br", twitterTweet);
					add("br", pinItButton);
				}
			};
			*/
			ControlPanel shares = new ControlPanel(getText("shareOn")) {

				JCheckBox shareTwitter = new JCheckBox("X");
				JCheckBox sharePinterest = new JCheckBox("Pinterest");
				JCheckBox shareLinkedin = new JCheckBox("LinkedIn");
				JCheckBox shareDigg = new JCheckBox("Digg");
				JCheckBox shareReddit = new JCheckBox("Reddit");
				JCheckBox shareTumblr = new JCheckBox("Tumblr");

				{
					setLayout(new RiverLayout(4, 5));
					add(new JLabel(icon("facebook")));
					add(shareFacebook);
					add("br", new JLabel(icon("threads")));
					add(shareThreads);
					add("br", new JLabel(icon("bluesky")));
					add(shareBluesky);
					add("br", new JLabel(icon("x")));
					add(shareTwitter);
					add("br", new JLabel(icon("reddit")));
					add(shareReddit);
					add("br", new JLabel(icon("pinterest")));
					add(sharePinterest);
					add("br", new JLabel(icon("tumblr")));
					add(shareTumblr);
					add("br", new JLabel(icon("linkedin")));
					add(shareLinkedin);
					add("br", new JLabel(icon("digg")));
					add(shareDigg);
				}
			};
			
			ControlPanel links = new ControlPanel() {
				
				JCheckBox shareEmail = new JCheckBox(getText("ui.email"));
				JLabel emailSubjectLabel = new JLabel(getText("ui.subject"));
				JTextField emailSubject = new JSmartTextField(18);
				JTextArea emailBody = new JSmartTextArea(4,22);
				JScrollPane emailBodyPane = new JScrollPane(emailBody);
				JCheckBox shareLink = new JCheckBox(getText("ui.link"));
				
				{
					emailBody.setEditable(true);
					emailBody.setLineWrap(true);
					emailBody.setWrapStyleWord(true);
					emailBody.setToolTipText(getText("ui.emailBodyInfo"));
					emailBodyPane.setBorder(BorderFactory.createTitledBorder(getText("ui.body")));
					ComponentUtilities.whenSelectedEnable(shareEmail, new JComponent[]{emailSubjectLabel, emailSubject, emailBodyPane});
					
					add("", new JLabel(icon("email")));
					add("tab", shareEmail);
					add("br tab", emailSubjectLabel);
					add("", emailSubject);
					add("br tab", emailBodyPane);
					add("br", new JLabel(icon("link")));
					add("tab", shareLink);
				}
			};
			
			JLabel mandatoryInfo = new JLabel(mandatory);
			
			{
				mandatoryInfo.setToolTipText(getText("ui.mandatory"));
				links.setBorder(emptyBorder);
				
				add("", shares);
				add("hfill", links);
				//add("hfill", buttons);
				add("br", new JLabelFor(getText("ui.facebookAppId"), facebookAppId));
				add(mandatoryInfo);
				add("tab", facebookAppId);
				add("", new JLinkLabel("https://developers.facebook.com/apps", getText("ui.signUp")));

				putClientProperty("helpPage", helpRoot + "Site/Social");
			}
		};

		//	Background music
		
		ControlPanel backgroundMusicPanel = new ControlPanel() {
			

			JPlaylist backgroundAudio = new JPlaylist();
			JCheckBox backgroundAudioAutoPlay = new JCheckBox(getText("ui.autoStart"));
			JSpinner backgroundAudioVolume = new JSpinner(new SpinnerNumberModel(25, 1, 100, 1));
			JCheckBox backgroundAudioSlideshowControl = new JCheckBox(getText("ui.slideshowControl"));
			JCheckBox muteBackgroundAudio = new JCheckBox(getText("ui.muteBackgroundAudio"), true);
			JCheckBox backgroundAudioLoop = new JCheckBox(getText("ui.loop"));
			JCheckBox backgroundAudioRetainPosition = new JCheckBox(getText("ui.retainPosition"), true);
			WrappableJLabel autostartNotice = new WrappableJLabel("<html><i>" + getText("ui.autostartNotice") + "</i></html>");
			
			{
				backgroundAudioSlideshowControl.setToolTipText(getText("ui.slideshowControlInfo"));
				muteBackgroundAudio.setToolTipText(getText("ui.muteBackgroundAudioInfo"));
				backgroundAudioRetainPosition.setToolTipText(getText("ui.retainPositionInfo"));
				autostartNotice.setPreferredWidth(580);
				
				add("br hfill", backgroundAudio);
				add("br", new JLabel(getText("ui.initialVolume")));
				add("", backgroundAudioVolume);
				add("", new JLabel("%"));
				add("br", backgroundAudioAutoPlay);
				add("tab", backgroundAudioLoop);
				add("br", backgroundAudioRetainPosition);
				add("br", backgroundAudioSlideshowControl);
				add("br", muteBackgroundAudio);
				add("br", new JLabel(infoIcon));
				add("", autostartNotice);

				//putClientProperty("helpPage", helpRoot + "Site/Background_music");
			}
		};
		
		// Search
		
		ControlPanel searchPanel = new ControlPanel() {

			JCheckBox useSearch = new JCheckBox(getText("ui.useSearch"), false);
			
			ControlPanel searchOptions = new ControlPanel() {
				
				JCheckBox searchOnHero = new JCheckBox(getText("ui.searchboxOnHero"));
				JTextField searchFields = new JSmartTextField(32);
				WrappableJLabel fieldsInfo = new WrappableJLabel("<html><i>" + getText("ui.searchFieldsInfo") + "</i></html>");
				
				{
					searchOnHero.setToolTipText(getText("ui.searchboxOnHeroInfo"));
					fieldsInfo.setPreferredWidth(580);
					
					add("br", searchOnHero);
					add("br", new JLabelFor(getText("ui.fields"), searchFields));
					add("tab hfill", searchFields);
					add("br", new JLabel(""));
					add("br", new JLabel(infoIcon));
					add(fieldsInfo);
				}
			};
					
			{
				ComponentUtilities.whenSelectedEnable(useSearch, searchOptions);
				searchOptions.setBorder(emptyBorder);
				
				add("", useSearch);
				add("br hfill", searchOptions);
			}
		};
		
		// Tag cloud 
		
		ControlPanel tagCloudPanel = new ControlPanel() {

			JCheckBox showTagCloud = new JCheckBox(getText("ui.showTagCloud"));
			
			ControlPanel tagCloudSettigns = new ControlPanel() {
				JComboBox tagCloudSource = new JComboBox(new Object[] {
					//new Item("none", getText("ui.noTagCloud")),			
					new Item("current", getText("ui.currentFolder")),			
					new Item("subfolders", getText("ui.subfolders")),
					new Item("tree", getText("ui.wholeAlbum"))
				});
				JCheckBox tagCloudUseFolders = new JCheckBox(getText("ui.folders"), true);
				JCheckBox tagCloudUsePages = new JCheckBox(getText("ui.pages"));
				JCheckBox tagCloudUseWebLocations = new JCheckBox(getText("ui.webLocations"));
				JLabel tagCloudSkipLevelsLabel = new JLabel(getText("ui.skipLevels"));
				JSpinner tagCloudSkipLevels = new JSpinner(new SpinnerNumberModel(1, 0, 5, 1));
				JTextField tagCloudLabel = new JSmartTextField(12);
				JTextField tagCloudFields = new JSmartTextField();
				JComboBox tagCloudSort = new JComboBox(new Object[] {
					new Item("none", getText("ui.unsorted")),			
					new Item("name", getText("ui.name")),			
					new Item("frequency", getText("ui.frequency"))
				});
				JCheckBox tagCloudFontVaries = new JCheckBox(getText("ui.tagCloudFontVaries"));
				JComboBox tagCloudSearch = new JComboBox(new Object[] {
					new Item("none", getText("ui.noSearchBox")),	
					new Item("above", getText("ui.above")),	
					new Item("below", getText("ui.below"))
				});

				{
					tagCloudLabel.setText(getText("ui.labels"));
					tagCloudSource.addActionListener(new ActionListener() {
						@Override
						public void actionPerformed(ActionEvent e) {
							int tsi = tagCloudSource.getSelectedIndex();
							tagCloudSkipLevels.setVisible(tsi == 1);
							tagCloudSkipLevelsLabel.setVisible(tsi == 1);
					}});
					tagCloudSkipLevels.setToolTipText(getText("ui.skipLevelsInfo"));
					tagCloudFields.setToolTipText("<html>" + getText("ui.searchFieldsInfo") + "</html>");
					tagCloudFontVaries.setToolTipText(getText("ui.tagCloudFontVariesInfo"));

					add(new JLabelFor(getText("ui.collectFrom"), tagCloudSource));
					add("tab", tagCloudSource);
					add("tab", tagCloudSkipLevelsLabel);
					add("", tagCloudSkipLevels);
					add("", tagCloudUseFolders);
					add("", tagCloudUsePages);
					add("", tagCloudUseWebLocations);
					add("br", new JLabelFor(getText("ui.boxTitle"), tagCloudLabel));
					add("tab hfill", tagCloudLabel);
					add("br", new JLabelFor(getText("ui.fields"), tagCloudFields));
					add("tab hfill", tagCloudFields);
					add("br", new JLabelFor(getText("ui.sortBy"), tagCloudFields));
					add("tab", tagCloudSort);
					add("tab", tagCloudFontVaries);
					add("br", new JLabelFor(getText("ui.addSearchBox"), tagCloudSearch));
					add("tab", tagCloudSearch);

					putClientProperty("helpPage", helpRoot + "Sections/Tag_cloud_box");
				}
			};
			
			{
				tagCloudSettigns.setBorder(emptyBorder);
				ComponentUtilities.whenSelectedEnable(showTagCloud, tagCloudSettigns);
				
				add(showTagCloud);
				add("br hfill", tagCloudSettigns);
				
			}
		};
		
		// Filtering panel
		
		ControlPanel filteringPanel = new ControlPanel() {
						
			
			ControlPanel filtering = new ControlPanel() {
				
				JCheckBox useFilters = new JCheckBox(getText("ui.useFilters"));
				JTextField filterLabel = new JSmartTextField(getText("ui.filter"), 16);
				JCheckBox showEmptyFilters = new JCheckBox(getText("ui.stayVisibleIfNothingToFilter"));
				JVariablesPanel filtersPanel = new JVariablesPanel(texts);
				JTable filterData = filtersPanel.getTable();
				
				{
					showEmptyFilters.setToolTipText(getText("ui.stayVisibleIfNothingToFilterInfo"));
					filtersPanel.setDimensions(new Dimension(320, 160));
					ComponentUtilities.whenSelectedEnable(useFilters, new JComponent[]{ showEmptyFilters, filterLabel, filtersPanel });
					
					add("", useFilters);
					add("tab", new JLabelFor(getText("ui.boxTitle"), filterLabel));
					add("tab", filterLabel);
					add("tab", showEmptyFilters);
					add("br hfill", filtersPanel);
				}
			};
			
			ControlPanel sorting = new ControlPanel() {
				
				JCheckBox useSort = new JCheckBox(getText("ui.useSort"));
				JTextField sortLabel = new JSmartTextField(getText("ui.sortBy"), 16);
				JVariablesPanel sortPanel = new JVariablesPanel(texts);
				JTable sortData = sortPanel.getTable();
				
				{
					sortPanel.setDimensions(new Dimension(320, 80));
					ComponentUtilities.whenSelectedEnable(useSort, new JComponent[]{ sortLabel, sortPanel });
					
					add("", useSort);
					add("tab", new JLabelFor(getText("ui.boxTitle"), sortLabel));
					add("tab", sortLabel);
					add("br hfill", sortPanel);
				}
			};
			
			{
				//ComponentUtilities.whenSelectedEnable(useFilters, new JComponent[]{ filtering });
				//ComponentUtilities.whenSelectedEnable(useSort, new JComponent[]{ sorting });
				filtering.setBorder(emptyBorder);
				sorting.setBorder(emptyBorder);
				
				add("br hfill", filtering);
				add("br hfill", sorting);
				if (jalbumVersion.compareTo(new VersionNumber("18.3b4")) < 0) {
					add("br", new JLabel(getText("ui.cantSaveThisWithVersion").replace("{0}", "18.3")));
				}
				
				//putClientProperty("helpPage", helpRoot + "Sections/Filtering");
			}
		};
		
		// Shopping cart
		
		ControlPanel shopPanel = new ControlPanel() {

			JCheckBox showShop = new JCheckBox(getText("ui.usePaypal"));

			JCheckBox showShopOnImagePagesOnly = new JCheckBox(getText("ui.showShopOnImagePagesOnly"), true);
			JTextField shopLabel = new JTextField(24);
			JTextField shopId = new JTextField(24);
			JLabel mandatoryInfo = new JLabel(mandatory);
			/*JComboBox shopPlacement = new JComboBox(new Object[]{ 
				new Item("aboveThumbs", getText("ui.aboveThumbs")),
				new Item("sidebar", getText("ui.inTheSidebar"))
			});*/
			JComboBox shopCurrency = new JComboBox(new Object[]{ 
				new Item("USD", "United States Dollar"),
				new Item("EUR", "Euro"),
				new Item("GBP", "British Pound"),
				new Item("CAD", "Canadian Dollar"),
				new Item("AUD", "Australian Dollar"),
				//new Item("RUB", "Russia Rubles"),
				new Item("JPY", "Japanese Yen"),
				//new Item("INR", "India Rupees"),
				new Item("NZD", "New Zealand Dollar"),
				new Item("CHF", "Swiss Franc"),
				//new Item("ARS", "Argentina Pesos"),
				//new Item("BHD", "Bahrain Dinars"),
				//new Item("BYR", "Belarus Rubles"),
				//new Item("BAM", "Bosnia & Herzegovina C.Marka"),
				new Item("BRL", "Brazilian Real"),
				//new Item("BGN", "Bulgaria Leva"),
				//new Item("CLP", "Chile Pesos"),
				//new Item("CNY", "China Yuan Renminbi"),
				//new Item("COP", "Colombia Pesos"),
				//new Item("CRC", "Costa Rica Colones"),
				//new Item("HRK", "Croatia Kuna"),
				new Item("CZK", "Czech Koruna"),
				new Item("DKK", "Danish Krone"),
				//new Item("EGP", "Egypt Pounds"),
				//new Item("EEK", "Estonia Krooni"),
				//new Item("GTQ", "Guatemala Quetzales"),
				new Item("HKD", "Hong Kong Dollar"),
				new Item("HUF", "Hungary Forint"),
				//new Item("ISK", "Iceland Kronur"),
				//new Item("IDR", "Indonesia Rupiahs"),
				//new Item("IQD", "Iraq Dinars"),
				new Item("ILS", "Israel New Shekel"),
				//new Item("JMD", "Jamaica Dollars"),
				//new Item("JOD", "Jordan Dinars"),
				//new Item("KWD", "Kuwait Dinars"),
				//new Item("LVL", "Latvia Lati"),
				//new Item("LBP", "Lebanon Pounds"),
				//new Item("LTL", "Lithuania Litai"),
				//new Item("MKD", "Macedonia Denars"),
				new Item("MYR", "Malaysian Ringgit"),
				new Item("MXN", "Mexican Peso"),
				//new Item("MDL", "Moldova Lei"),
				//new Item("MAD", "Morocco Dirhams"),
				new Item("NOK", "Norway Krone"),
				//new Item("PEN", "Peru Nuevos Soles"),
				new Item("PHP", "Philippines Peso"),
				new Item("PLN", "Poland Zlotych"),
				//new Item("RON", "Romania New Lei"),
				new Item("RUB", "Russian Ruble"),
				//new Item("SAR", "Saudi Arabia Riyals"),
				//new Item("RSD", "Serbia Dinars"),
				new Item("SGD", "Singapore Dollar"),
				//new Item("ZAR", "South Africa Rand"),
				//new Item("KRW", "South Korea Won"),
				new Item("SEK", "Sweden Krona"),
				new Item("TWD", "New Taiwan Dollar"),
				new Item("THB", "Thailand Baht"),
				new Item("TRY", "Turkish Lira"),
				//new Item("UAH", "Ukraine Hryvnia"),
				//new Item("AED", "United Arab Emirates Dirhams"),
				//new Item("UYU", "Uruguay Pesos"),
				//new Item("VND", "Vietnam Dong")
			});
			JTextField shopSuccessUrl = new JSmartTextField(24);
			JCheckBox shopSameWindowCheckout = new JCheckBox(getText("ui.sameWindowCheckout"));
			JCheckBox shopSendAlbumName = new JCheckBox(getText("ui.sendAlbumName"));
			JCheckBox shopSendFolderPath = new JCheckBox(getText("ui.sendFolderPath"));
			JComboBox shopSendItemName = new JComboBox(new Object[]{ 
				new Item("name", getText("ui.filename")),
				new Item("title", getText("ui.titleOrFilename")),
				new Item("comment", getText("ui.commentOrFilename"))
			});
			JCheckBox shopAskPermissionToEmpty = new JCheckBox(getText("ui.askPermisssionToEmptyCart"));
			
			JComboBox showPriceRange = new JComboBox(new Object[]{ 
				new Item("", getText("ui.none")),
				new Item("lowest", getText("ui.lowestPriceOnly")),
				new Item("minmax", getText("ui.minMax"))
			});
			JCheckBox usePriceAsSingleOption = new JCheckBox(getText("ui.usePriceAsSingleOption"));
			JComboBox shopOptionSelectMethod = new JComboBox(new Object[]{ 
				new Item("combo", getText("ui.dropdownBox")),
				new Item("radio", getText("ui.radioButtons"))
			});
			JTextArea shopOptions = new JSmartTextArea(8, 28);
			JScrollPane shopOptionsPane = new JScrollPane(shopOptions);
			JLabel shopOptionsInfo = new JLabel(infoIcon);

			JTextField shopMinimalAmount = new JSmartTextField(8);
			JTextField shopHandling = new JSmartTextField(8);
			JLabel shopHandlingInfo = new JLabel(infoIcon);
			JTextField shopTax = new JSmartTextField(8);
			JTextField shopQuantityCap = new JSmartTextField(8);

			JTextArea shopCoupons = new JSmartTextArea(4, 16);
			JScrollPane shopCouponsPane = new JScrollPane(shopCoupons);
			JLabel shopCouponsInfo = new JLabel(infoIcon);

			JTextField shopDiscountRate = new JSmartTextField(8);
			JTextField shopDiscountMinQuantity = new JSmartTextField(8);
			JTextField shopDiscountMinAmount = new JSmartTextField(8);

			JSmartTextArea shopInstructions = new JSmartTextArea(6, 24);	
			JScrollPane shopInstructionsPane = new JScrollPane(shopInstructions);
			JSmartTextArea shopInstructionsBox = new JSmartTextArea(6, 24);	
			JScrollPane shopInstructionsBoxPane = new JScrollPane(shopInstructionsBox);
				
			ControlPanel shoppingCartOptions = new ControlPanel() {

				{
					shopId.setToolTipText(getText("ui.sellerIdInfo"));
					mandatoryInfo.setToolTipText(getText("ui.mandatory"));
					shopLabel.setToolTipText(getText("ui.default") + ": " + getText("ui.shoppingCart"));
					shopCurrency.setEditable(false);
					shopCurrency.setToolTipText(getText("ui.currencyInfo"));
					shopSuccessUrl.setToolTipText(getText("ui.shopSuccessUrlInfo"));
					usePriceAsSingleOption.setToolTipText(getText("ui.usePriceAsSingleOptionInfo"));
					shopSameWindowCheckout.setToolTipText(getText("ui.sameWindowCheckoutInfo"));
					shopSendAlbumName.setToolTipText(getText("ui.sendAlbumNameInfo"));

					add(showShopOnImagePagesOnly);
					add("br", new JLabelFor(getText("ui.boxTitle"), shopLabel));
					add("tab", shopLabel);
					add("br", new JLabelFor(getText("ui.sellerId"), shopId));
					add(mandatoryInfo);
					add("tab", shopId);
					add("br tab", new JLinkLabel("https://www.paypal.com/signup/account", getText("ui.signUp")));
					add("br", new JLabelFor(getText("ui.currency"), shopCurrency));
					add("tab", shopCurrency);
					add("br", new JLabelFor(getText("ui.shopSuccessUrl"), shopSuccessUrl));
					add("tab", shopSuccessUrl);
					add("br", new JLabel(getText("ui.showPriceRange")));
					add("tab", showPriceRange);
					add("br", usePriceAsSingleOption);
					add("br", shopSendAlbumName);
					add("tab", shopSendFolderPath);
					add("tab", shopSendItemName);
					add("", new JLabel(getText("ui.max128Chars")));
					add("br", shopSameWindowCheckout);
					add("br", shopAskPermissionToEmpty);
				}
			};

			ControlPanel shoppingCartPricing = new ControlPanel() {

				{
					shopOptionsInfo.addMouseListener(new MouseAdapter() {  
						@Override
						public void mouseReleased(MouseEvent e) {
							String s = "<html><b><kbd>" + getText("ui.shopOptionsFormat") + "</kbd></b><br>" +
									"<br><b>" + getText("ui.examples") + ":</b><br>" +
									"<br><kbd>" + "Print A4=2.0" + "</kbd><br>" +
									"<i>" + getText("ui.optionExample1Hint") + "</i><br>" +
									"<br><kbd>" + "Print A2=10.0+0.5" + "</kbd><br>" +
									"<i>" + getText("ui.optionExample2Hint") + "</i><br>" +
									"<br><kbd>" + "Print A2=10.0+0.5+0" + "</kbd><br>" +
									"<i>" + getText("ui.optionExample3Hint") + "</i><br>" +
									"<br><i>" + getText("ui.noCurrencySign") + "</i></html>";
							JOptionPane.showMessageDialog(window, s, getText("ui.format"), JOptionPane.INFORMATION_MESSAGE);
					}});
					shopOptions.setToolTipText(getText("ui.shopOptionsInfo"));
					shopOptions.setEditable(true);
					shopOptions.setLineWrap(false);
					shopOptions.setFont(mono);
					shopOptionsPane.setBorder(BorderFactory.createTitledBorder(getText("ui.shopOptions")));
					shopHandlingInfo.addMouseListener(new MouseAdapter() {  
						@Override
						public void mouseReleased(MouseEvent e) {
							String s = "<html><b><kbd>" + getText("ui.handlingFormat") + "</kbd></b><br>" +
									"<br><b>" + getText("ui.examples") + ":</b><br>" +
									"<br><kbd>" + "2.0" + "</kbd><br>" +
									"<i>" + getText("ui.handlingExample1Hint") + "</i><br>" +
									"<br><kbd>" + "0.0+0.1" + "</kbd><br>" +
									"<i>" + getText("ui.handlingExample2Hint") + "</i><br>" +
									"<br><kbd>" + "1.0+0+0.1" + "</kbd><br>" +
									"<i>" + getText("ui.handlingExample3Hint") + "</i><br>" +
									"<br><i>" + getText("ui.noCurrencySign") + "</i></html>";
							JOptionPane.showMessageDialog(window, s, getText("ui.format"), JOptionPane.INFORMATION_MESSAGE);
					}});
					shopHandling.setToolTipText(getText("ui.handlingInfo") + " " + getText("ui.leaveEmpty"));
					shopTax.setToolTipText(getText("ui.taxInfo") + " " + getText("ui.leaveEmpty"));
					shopQuantityCap.setToolTipText(getText("ui.shopQuantityCapInfo") + " " + getText("ui.leaveEmpty"));
					shopMinimalAmount.setToolTipText(getText("ui.shopMinimalCartValueInfo"));
					shopDiscountRate.setToolTipText(getText("ui.shopDiscountRateInfo") + " " + getText("ui.leaveEmpty"));
					shopDiscountMinQuantity.setToolTipText(getText("ui.shopDiscountMinQuantityInfo"));
					shopDiscountMinAmount.setToolTipText(getText("ui.shopDiscountMinAmountInfo"));
					shopCoupons.setToolTipText(getText("ui.shopCouponsInfo"));
					shopCoupons.setEditable(true);
					shopCoupons.setLineWrap(false);
					shopCoupons.setFont(mono);
					shopCouponsInfo.addMouseListener(new MouseAdapter() {  
						@Override
						public void mouseReleased(MouseEvent e) {
							String s = "<html><b><kbd>" + getText("ui.shopCouponsFormat") + "</kbd></b><br>" +
									"<br><b>" + getText("ui.examples") + ":</b><br>" +
									"<br><kbd>" + "5OFF=5.0" + "</kbd><br>" +
									"<i>" + getText("ui.couponExample1Hint") + "</i><br>" +
									"<br><kbd>" + "XMAS=20% &lt;2016-12-24" + "</kbd><br>" +
									"<i>" + getText("ui.couponExample2Hint") + "</i><br>" +
									"<br><kbd>" + "MIN10=30% 10+" + "</kbd><br>" +
									"<i>" + getText("ui.couponExample3Hint") + "</i><br>" +
									"<br><i>" + getText("ui.noCurrencySign") + "</i></html>";
							JOptionPane.showMessageDialog(window, s, getText("ui.format"), JOptionPane.INFORMATION_MESSAGE);
					}});
					shopCouponsPane.setBorder(BorderFactory.createTitledBorder(getText("ui.shopCoupons")));

					add("", new JLabelFor(getText("ui.handling"), shopHandling));
					add(shopHandlingInfo);
					add("tab", shopHandling);
					add("tab", new JLabelFor(getText("ui.tax") + " (%)", shopTax));
					add("tab", shopTax);
					add("br", new JLabelFor(getText("ui.shopQuantityCap"), shopQuantityCap));
					add("tab", shopQuantityCap);
					add("tab", new JLabelFor(getText("ui.shopMinimalCartValue"), shopMinimalAmount));
					add("tab", shopMinimalAmount);
					add("br", new JLabelFor(getText("ui.shopDiscountRate") + " (%)", shopDiscountRate));
					add("tab", shopDiscountRate);
					add("tab", new JLabelFor(getText("ui.minQuantity"), shopDiscountMinQuantity));
					add("tab", shopDiscountMinQuantity);
					add("tab", new JLabelFor(getText("ui.minAmount"), shopDiscountMinAmount));
					add("tab", shopDiscountMinAmount);
					add("br hfill", shopOptionsPane);
					add(shopOptionsInfo);
					add("br hfill", shopCouponsPane);
					add(shopCouponsInfo);
					add("br", new JLabelFor(getText("ui.selectMethod"), shopOptionSelectMethod));
					add("tab", shopOptionSelectMethod);

				}
			};

			ControlPanel shoppingCartInstructions = new ControlPanel() {

				{
					shopInstructions.setToolTipText(getText("ui.shopInstructionsInfo"));
					shopInstructions.setEditable(true);
					shopInstructions.setLineWrap(true);
					shopInstructions.setWrapStyleWord(true);
					shopInstructions.setFont(mono);
					allowHTMLEditing(shopInstructions);
					shopInstructionsPane.setBorder(BorderFactory.createTitledBorder(getText("ui.shopInstructions")));
					shopInstructionsBox.setToolTipText(getText("ui.shopInstructionsBoxInfo"));
					shopInstructionsBox.setEditable(true);
					shopInstructionsBox.setLineWrap(true);
					shopInstructionsBox.setWrapStyleWord(true);
					shopInstructionsBox.setFont(mono);
					shopInstructionsBox.setTabSize(2);
					allowHTMLEditing(shopInstructionsBox);
					shopInstructionsBoxPane.setBorder(BorderFactory.createTitledBorder(getText("ui.shopInstructionsBox")));
					
					add("hfill", shopInstructionsPane);
					add("br hfill", shopInstructionsBoxPane);
				}
			};

			JTabbedPane shopTabs = new JTabbedPane() {
								
				{
					shoppingCartOptions.setBorder(emptyBorder);
					shoppingCartPricing.setBorder(emptyBorder);
					shoppingCartInstructions.setBorder(emptyBorder);
					
					addTab(getText("ui.settings"), shoppingCartOptions);
					addTab(getText("ui.pricing"), shoppingCartPricing);
					addTab(getText("ui.instructions"), shoppingCartInstructions);
				}
			};

			{
				if (!commercialUseAllowed) {
					showShop.setSelected(false);
					commercialMonitor.add(showShop);
				}
				showShop.setToolTipText(getText("ui.sellPhotosInfo"));
				ComponentUtilities.whenSelectedEnable(showShop, new JComponent[]{ shoppingCartOptions, shoppingCartPricing, shoppingCartInstructions } );
				
				add(showShop);
				add("br hfill", shopTabs);

				//putClientProperty("helpPage", helpRoot + "Sections/Shopping_cart");
			}
		};
		
		// Fotomoto
		
		ControlPanel fotomotoPanel = new ControlPanel() {
			
			JCheckBox useFotomoto = new JCheckBox(getText("ui.useFotomoto"));
			JTextField fotomotoID = new JSmartTextField(24);
			JLabel mandatoryInfo = new JLabel(mandatory);
			
			{
				fotomotoID.setToolTipText(getText("ui.fotomotoIDInfo"));
				mandatoryInfo.setToolTipText(getText("ui.mandatory"));
				ComponentUtilities.whenSelectedEnable(useFotomoto, new JComponent[]{fotomotoID, mandatoryInfo});

				//add("", new JLabel(icon("fotomoto")));
				add("tab", useFotomoto);
				add("br", new JLabelFor(getText("ui.storeId"), fotomotoID));
				add(mandatoryInfo);
				add("", fotomotoID);
				add(new JLinkLabel("https://my.fotomoto.com/signup", getText("ui.signUp")));
			}
		};
		
		// Feedback section
		
		ControlPanel feedbackPanel = new ControlPanel() {

			JCheckBox showFeedback = new JCheckBox(getText("ui.useFeedback"));

			ControlPanel feedbackOptions = new ControlPanel() {

				JCheckBox showFeedbackOnImagePagesOnly = new JCheckBox(getText("ui.showOnImagePagesOnly"), true);
				JCheckBox feedbackBoxSticky = new JCheckBox(getText("ui.alwaysVisible"));
				JCheckBox feedbackAddMultiple = new JCheckBox(getText("ui.canAddMultipleTimes"));
				JTextField feedbackLabel = new JTextField(24);
				JTextField feedbackEmail = new JTextField(24);
				JLabel mandatoryInfo = new JLabel(mandatory);
				JComboBox feedbackFormatting = new JComboBox(new Object[] {
					new Item("human", getText("ui.humanReadable")),			
					new Item("serialized", getText("ui.serialized"))
				});
				JTextField feedbackViewButtonLabel = new JTextField(24);
				JTextField feedbackAddButtonLabel = new JTextField(24);
				JTextField feedbackAddButtonTooltip = new JTextField(24);
				JTextField feedbackModalTitle = new JTextField(24);
				JTextField feedbackCopyButtonLabel = new JTextField(24);
				JCheckBox useFeedbackSendButton = new JCheckBox(getText("ui.useSendButton"), true);
				JTextField feedbackSendButtonLabel = new JTextField(24);
				
				{
					feedbackLabel.setToolTipText(getText("ui.feedbackLabelInfo"));
					//ComponentUtilities.whenSelectedDisable(feedbackBoxSticky, feedbackFloatButton);
					mandatoryInfo.setToolTipText(getText("ui.mandatory"));
					feedbackViewButtonLabel.setToolTipText(getText("ui.feedbackViewButtonLabelInfo"));
					feedbackAddButtonLabel.setToolTipText(getText("ui.feedbackAddButtonLabelInfo"));
					feedbackAddButtonTooltip.setToolTipText(getText("ui.feedbackActionButtonTooltipInfo"));
					feedbackModalTitle.setToolTipText(getText("ui.feedbackModalTitleInfo"));
					feedbackCopyButtonLabel.setToolTipText(getText("ui.feedbackCopyButtonLabelInfo"));
					feedbackSendButtonLabel.setToolTipText(getText("ui.feedbackSendButtonLabelInfo"));
					feedbackEmail.setToolTipText(getText("ui.feedbackEmailInfo"));
					
					ComponentUtilities.whenSelectedEnable(useFeedbackSendButton, feedbackSendButtonLabel);
					
					add(showFeedbackOnImagePagesOnly);
					add("br", feedbackBoxSticky);
					add("br", feedbackAddMultiple);
					add("br", new JLabelFor(getText("ui.boxTitle"), feedbackLabel));
					add("tab", feedbackLabel);
					add("tab", new JLabel("<html><i>" + getText("ui.default") + ": \"" + getText("feedback") + "\"</i></html>"));
					add("br", new JLabelFor(getText("ui.feedbackEmail"), feedbackEmail));
					add(mandatoryInfo);
					add("tab", feedbackEmail);
					add("br", new JLabelFor(getText("ui.formatting"), feedbackFormatting));
					add("tab", feedbackFormatting);
					add("br", new JLabelFor(getText("ui.viewButtonLabel"), feedbackViewButtonLabel));
					add("tab", feedbackViewButtonLabel);
					add("tab", new JLabel("<html><i>" + getText("ui.default") + ": \"" + getText("editFeedback") + "\"</i></html>"));
					add("br", new JLabelFor(getText("ui.addButtonLabel"), feedbackAddButtonLabel));
					add("tab", feedbackAddButtonLabel);
					add("tab", new JLabel("<html><i>" + getText("ui.default") + ": \"" + getText("writeFeedback") + "\"</i></html>"));
					add("br", new JLabelFor(getText("ui.addButtonTooltip"), feedbackAddButtonTooltip));
					add("tab", feedbackAddButtonTooltip);
					add("tab", new JLabel("<html><i>" + getText("ui.default") + ": \"" + getText("provideFeedbackOnSelectedItems") + "\"</i></html>"));
					add("br", new JLabelFor(getText("ui.modalTitle"), feedbackModalTitle));
					add("tab", feedbackModalTitle);
					add("tab", new JLabel("<html><i>" + getText("ui.default") + ": \"" + getText("feedbackOnAlbum") + "\"</i></html>"));
					add("br", new JLabelFor(getText("ui.copyButtonLabel"), feedbackCopyButtonLabel));
					add("tab", feedbackCopyButtonLabel);
					add("tab", new JLabel("<html><i>" + getText("ui.default") + ": \"" + getText("copyFeedback") + "\"</i></html>"));
					add("br", useFeedbackSendButton);
					add("br", new JLabelFor(getText("ui.sendButtonLabel"), feedbackSendButtonLabel));
					add("tab", feedbackSendButtonLabel);
					add("tab", new JLabel("<html><i>" + getText("ui.default") + ": \"" + getText("sendFeedback") + "\"</i></html>"));
				}
			};
			
			ControlPanel feedbackForm = new ControlPanel() {
				
				JTextArea feedbackTemplate = new JSmartTextArea(5, 24);
				JScrollPane feedbackTemplatePane = new JScrollPane(feedbackTemplate);
				//JLabel feedbackTemplateInfo = new JLabel(infoIcon);
				
				{
					feedbackTemplate.setToolTipText(getText("ui.feedbackTemplateInfo"));
					feedbackTemplate.setEditable(true);
					feedbackTemplate.setLineWrap(true);
					feedbackTemplate.setWrapStyleWord(true);
					feedbackTemplate.setFont(mono);
					feedbackTemplate.setTabSize(2);
					feedbackTemplatePane.setBorder(BorderFactory.createTitledBorder(getText("ui.feedbackTemplate")));
					
					add("hfill vfill", feedbackTemplatePane);
					
				}
			};
			
			ControlPanel feedbackInstruction = new ControlPanel() {
				
				JSmartTextArea feedbackInstructions = new JSmartTextArea(6, 24);	
				JScrollPane feedbackInstructionsPane = new JScrollPane(feedbackInstructions);
				JSmartTextArea feedbackInstructionsBox = new JSmartTextArea(6, 24);	
				JScrollPane feedbackInstructionsBoxPane = new JScrollPane(feedbackInstructionsBox);
				
				{
					
					feedbackInstructions.setToolTipText(getText("ui.feedbackInstructionsInfo"));
					feedbackInstructions.setEditable(true);
					feedbackInstructions.setLineWrap(true);
					feedbackInstructions.setWrapStyleWord(true);
					feedbackInstructions.setFont(mono);
					feedbackInstructions.setTabSize(2);
					allowHTMLEditing(feedbackInstructions);
					feedbackInstructionsPane.setBorder(BorderFactory.createTitledBorder(getText("ui.feedbackInstructions")));
					
					feedbackInstructionsBox.setToolTipText(getText("ui.feedbackInstructionsBoxInfo"));
					feedbackInstructionsBox.setEditable(true);
					feedbackInstructionsBox.setLineWrap(true);
					feedbackInstructionsBox.setWrapStyleWord(true);
					feedbackInstructionsBox.setFont(mono);
					feedbackInstructionsBox.setTabSize(2);
					allowHTMLEditing(feedbackInstructionsBox);
					feedbackInstructionsBoxPane.setBorder(BorderFactory.createTitledBorder(getText("ui.feedbackInstructionsBox")));

					add("hfill", feedbackInstructionsPane);
					add("br hfill", feedbackInstructionsBoxPane);
				}
			};
			
			JTabbedPane feedbackTabs = new JTabbedPane() {
								
				{
					feedbackOptions.setBorder(emptyBorder);
					feedbackForm.setBorder(emptyBorder);
					feedbackInstruction.setBorder(emptyBorder);
					
					addTab(getText("ui.settings"), feedbackOptions);
					addTab(getText("ui.feedbackForm"), feedbackForm);
					addTab(getText("ui.instructions"), feedbackInstruction);
				}
			};
			
			{
				if (!commercialUseAllowed) {
					showFeedback.setSelected(false);
					commercialMonitor.add(showFeedback);
				}
				showFeedback.setToolTipText(getText("ui.useFeedbackInfo"));
				ComponentUtilities.whenSelectedEnable(showFeedback, new JComponent[] {feedbackOptions,feedbackForm,feedbackInstruction});
				feedbackOptions.setBorder(emptyBorder);

				add(showFeedback);
				add("br vfill hfill", feedbackTabs);

				//putClientProperty("helpPage", helpRoot + "Sections/Feedback");
			}

		};
		
		// Download
		
		ControlPanel downloadPanel = new ControlPanel() {
			
			ControlPanel dl_index = new ControlPanel(getText("ui.story")) {
				
				JComboBox zipImages = new JComboBox(new Object[] {
					new Item("none", getText("ui.nothing")),
					new Item("slides", getText("ui.scaledDown")),
					new Item("originals", getText("ui.originals")),
					new Item("included", getText("ui.includedOriginals"))//,
					//new Item("hires", getText("ui.includedHires")),
					//new Item("album", getText("ui.wholeAlbum"))
				}); 
				JLabel zipInfo = new JLabel(infoIcon);

				{

					zipInfo.addMouseListener(new MouseAdapter() {  
						@Override
						public void mouseReleased(MouseEvent e) {
							JOptionPane.showMessageDialog(window, getText("ui.nonAsciiWarning"), "Warning", JOptionPane.WARNING_MESSAGE);
					}});

					add("", new JLabelFor(getText("ui.offerDownload"), zipImages));
					add("tab", zipImages);
					add("", zipInfo);
				}
			};
					
			ControlPanel dl_lightbox = new ControlPanel(getText("ui.lightbox")) {
				
				JCheckBox showDownload = new JCheckBox(getText("ui.showDownloadButton"));
				JCheckBox downloadNonImages = new JCheckBox(getText("ui.enableDownloadNonImages"));
				JCheckBox downloadScaled = new JCheckBox(getText("ui.enableDownloadScaledImages"));
				
				{
					showDownload.setToolTipText(getText("ui.showDownloadInfo"));
					ComponentUtilities.whenSelectedEnable(showDownload, new JComponent[]{downloadScaled, downloadNonImages});
					downloadNonImages.setToolTipText(getText("ui.enableDownloadNonImagesInfo"));
					downloadNonImages.setToolTipText(getText("ui.enableDownloadNonImagesInfo"));

					//add("", new JLabel(icon("download")));
					add("", showDownload);
					add("br", downloadScaled);
					add("br", downloadNonImages);
				}
			};
			
			{
				add("hfill", dl_index);
				add("br hfill", dl_lightbox);
			}
		};
		
		// Print
		
		ControlPanel printPanel = new ControlPanel() {
			
			JCheckBox printImageButton = new JCheckBox(getText("ui.printImageButton"));
			
			{
				printImageButton.setToolTipText(getText("ui.printImageButtonInfo"));

				add("", printImageButton);
			}
		};
		
		// Map
		
		ControlPanel mapPanel = new ControlPanel() {

			JCheckBox showMap = new JCheckBox(getText("ui.showMap"), true);

			ControlPanel mapOptions = new ControlPanel() {

				JComboBox mapType = new JComboBox(new Object[]{
					new Item("roadmap", getText("ui.roadmap")),
					new Item("satellite", getText("ui.satellite")),
					new Item("hybrid", getText("ui.hybrid")),
					new Item("terrain", getText("ui.terrain"))
				});
				JSlider mapZoom = new JSlider(JSlider.HORIZONTAL, 1, 20, 18);
				JTextField googleApiKey = new JSmartTextField(32);
				JLabel googleApiInfo = new JLabel(infoIcon);
				WrappableJLabel note = new WrappableJLabel("<html><i>" + getText("ui.mapApiKeyInfo") + "</i></html>");
				
				{
					mapZoom.setOrientation(JSlider.HORIZONTAL);
					mapZoom.setMinimum(0);
					mapZoom.setMaximum(20);
					mapZoom.setValue(18);
					mapZoom.setMajorTickSpacing(10);
					mapZoom.setMinorTickSpacing(1);
					mapZoom.setPaintTicks(true);
					mapZoom.setPaintLabels(true);
					mapZoom.setSnapToTicks(true);
					note.setPreferredWidth(500);
					googleApiInfo.addMouseListener(new MouseAdapter() {  
						@Override
						public void mouseReleased(MouseEvent e) {
							JOptionPane.showMessageDialog(window, "<html><i>" + getText("ui.gooleMapsApiNote") + "</i></html>", getText("ui.format"), JOptionPane.INFORMATION_MESSAGE);
					}});
					
					add("br", new JLabelFor(getText("ui.initialView"), mapType));
					add("tab", mapType);
					add("br", new JLabelFor(getText("ui.initialZoom"), mapZoom));
					add("tab", mapZoom);
					add("br", new JLabelFor(getText("ui.googleApiKey"), googleApiKey));
					add("tab", googleApiKey);
					add("", new JLinkLabel("https://console.developers.google.com/apis/credentials", getText("ui.createNew")));
					add("", googleApiInfo);
					add("br tab", note);
					//add("br tab", new JLinkLabel("https://developers.google.com/maps/documentation/javascript/tutorial", getText("ui.readMore")));
				}
			};
				
			{
				ComponentUtilities.whenSelectedEnable(showMap, mapOptions);
				
				add(showMap);
				add("br hfill", mapOptions);

				//putClientProperty("helpPage", helpRoot + "ui/map.html");
			}
		};
				
		// Photodata
		
		ControlPanel photodataPanel = new ControlPanel() {

			JCheckBox showPhotoData = new JCheckBox(getText("ui.showPhotoData"));
			
			ControlPanel photoDataOptions = new ControlPanel() {
				
			
				JTextArea photoDataTemplate = new JSmartTextArea(10, 20);
				JScrollPane photoDataPane = new JScrollPane(photoDataTemplate);
				JButton reset = new JButton(getText("ui.resetToDefaults"));
				JLabel listMetadataHint = new JLabel("<html><i>" + getText("ui.listMetadataHint") + "</i></html>");
				
				{
					photoDataTemplate.setEditable(true);
					photoDataTemplate.setLineWrap(true);
					photoDataTemplate.setWrapStyleWord(true);
					photoDataTemplate.setFont(mono);
					reset.addMouseListener(new MouseAdapter() {  
						@Override
						public void mouseReleased(MouseEvent e) {
							String s = new SkinModel().photoDataTemplate;
							if (s != null && s.length() > 0) {
								photoDataTemplate.setText(s);
							}
					}});				
					
					add("", reset);
					add("br hfill vfill", photoDataPane);
					add("br", listMetadataHint);
				}
			};
			
			{
				showPhotoData.setToolTipText(getText("ui.showPhotoDataInfo"));
				photoDataOptions.setBorder(emptyBorder);
				ComponentUtilities.whenSelectedEnable(showPhotoData, photoDataOptions);

				add("tab", showPhotoData);
				add("br hfill vfill", photoDataOptions);

			}
		};
		
		// Regions
		
		ControlPanel regionsPanel = new ControlPanel() {
			
			JCheckBox showRegions = new JCheckBox(getText("ui.showRegions"), true);
			JCheckBox regionsVisible = new JCheckBox(getText("ui.visibleByDefault"));
			private JLabel buttonLabel = new JLabel(getText("ui.buttonLabel"));
			JTextField regionsBtnText = new JTextField(getText("regionsBtn"), 8);
			JCheckBox regionsSkipEmpty = new JCheckBox(getText("ui.skipEmptyTags"));
			
			{
				showRegions.setToolTipText(getText("ui.showRegionsInfo"));
				regionsVisible.setToolTipText(getText("ui.regionsVisibleInfo"));
				ComponentUtilities.whenSelectedEnable(showRegions, new JComponent[]{ regionsVisible, buttonLabel, regionsBtnText, regionsSkipEmpty });
				
				add("", showRegions);
				add("br", regionsVisible);
				add("br", buttonLabel);
				add("", regionsBtnText);
				add("br", regionsSkipEmpty);
				
			}
		};
		
		// Audio clips

		ControlPanel audioClips = new ControlPanel() {

			JCheckBox useAudioClipButton = new JCheckBox(getText("ui.playAudioClips"), false);
			JCheckBox useAudioClips = new JCheckBox(getText("ui.playAudioClipsInLightbox"), false);
			JSpinner audioClipVolume = new JSpinner(new SpinnerNumberModel(75, 1, 100, 1));
			JCheckBox copyAudioClips = new JCheckBox(getText("ui.copyAudioClips"));

			{
				useAudioClipButton.setToolTipText(getText("ui.playAudioClipsInfo"));
				useAudioClips.setToolTipText(getText("ui.playAudioClipsInLightboxInfo"));
				audioClipVolume.setToolTipText(getText("ui.audioClipVolumeInfo"));
				copyAudioClips.setToolTipText(getText("ui.copyAudioClipsInfo") + " (" + getText("ui.oldMethod") + ")");
				
				add("", useAudioClipButton);
				add("br", useAudioClips);
				add("br", new JLabel(getText("ui.initialVolume")));
				add("", audioClipVolume);
				add("", new JLabel("%"));
				add("br", copyAudioClips);
				
				putClientProperty("helpPage", helpRoot + "/Site/Audio_clips");
			}
		};
		
		
		/*	---------------------------------------------------------------
									Advanced
			--------------------------------------------------------------- */

		//	Custom code
		
		ControlPanel customCodePanel = new ControlPanel() {

			ControlPanel headHookTab = new ControlPanel() {

				JTextArea headHook = new JSmartTextArea(7, 20);
				JScrollPane headHookPane = new JScrollPane(headHook);

				{
					headHook.setEditable(true);
					headHook.setLineWrap(false);
					headHook.setFont(mono);
					headHook.setTabSize(2);
					headHookPane.setBorder(BorderFactory.createTitledBorder(getText("ui.headText")));

					add("hfill vfill", headHookPane);
				}
			};

			ControlPanel bodyHookTab = new ControlPanel() {

				JTextArea bodyHook = new JSmartTextArea(7, 20);
				JScrollPane bodyHookPane = new JScrollPane(bodyHook);

				{
					bodyHook.setEditable(true);
					bodyHook.setLineWrap(false);
					bodyHook.setFont(mono);
					bodyHook.setTabSize(2);
					bodyHookPane.setBorder(BorderFactory.createTitledBorder(getText("ui.bodyText")));

					add("hfill vfill", bodyHookPane);
				}
			};

			ControlPanel cssHookTab = new ControlPanel() {

				JTextArea cssHook = new JSmartTextArea(15, 20);
				JScrollPane cssHookPane = new JScrollPane(cssHook);
				WrappableJLabel info = new WrappableJLabel("<html><i>" + getText("ui.cssText") + "</i></html>");
				
				{
					cssHook.setEditable(true);
					cssHook.setLineWrap(false);
					cssHook.setFont(mono);
					cssHook.setTabSize(2);
					cssHookPane.setBorder(BorderFactory.createTitledBorder(getText("ui.cssTitle")));
					info.setPreferredWidth(700);
					
					add("hfill vfill", cssHookPane);
					add("br", info);
				}
			};

			ControlPanel jsHookTab = new ControlPanel() {

				JTextArea jsHook = new JSmartTextArea(15, 20);
				JScrollPane jsHookPane = new JScrollPane(jsHook);
				WrappableJLabel info = new WrappableJLabel("<html><i>" + getText("ui.javascriptText") + "</i></html>");
				
				{
					jsHook.setEditable(true);
					jsHook.setLineWrap(false);
					jsHook.setFont(mono);
					jsHook.setTabSize(2);
					jsHookPane.setBorder(BorderFactory.createTitledBorder(getText("ui.javascriptTitle")));
					info.setPreferredWidth(700);
					
					add("hfill vfill", jsHookPane);
					add("br", info);
				}
			};

			JTabbedPane customCodeTabs = new JTabbedPane() {				

				{
					addTab("<HEAD>", icon("code"), headHookTab);
					addTab("<BODY>", icon("code"), bodyHookTab);
					addTab("CSS", icon("css"), cssHookTab);
					addTab("JavaScript", icon("javascript"), jsHookTab);
				}
			};

			{				
				add("hfill vfill", customCodeTabs);

				//putClientProperty("helpPage", helpRoot + "ui/site.html");
			}
		};
								
		ControlPanel customContentPanel = new ControlPanel() {

			JSmartTextArea customContent = new JSmartTextArea(6, 20);
			JScrollPane customContentPane = new JScrollPane(customContent);
			JCheckBox customContentFitWidth = new JCheckBox(getText("ui.constrainPageWidth"), true);
			JCheckBox customContentTopLevelOnly = new JCheckBox(getText("ui.topLevelOnly"), false);

			{
				allowHTMLEditing(customContent);
				allowSpelling(customContent);
				
				customContent.setEditable(true);
				customContent.setLineWrap(true);
				customContent.setWrapStyleWord(true);
				customContent.setFont(mono);
				customContent.setTabSize(2);
				customContent.setToolTipText(getText("ui.customContentInfo"));
				customContentPane.setBorder(BorderFactory.createTitledBorder(getText("ui.pageBottomText")));

				add("hfill vfill", customContentPane);
				add("br", customContentFitWidth);
				add("tab", customContentTopLevelOnly);
				
				//putClientProperty("helpPage", helpRoot + "Footer/Custom_footer");

			}
		};

		//	Site admin
		
		ControlPanel siteAdminPanel = new ControlPanel() {

			ControlPanel serverRelatedPanel = new ControlPanel(getText("ui.serverRelated")) {
				
				JTextField uploadPath = new JSmartTextField(20);
				JCheckBox useFavicon = new JCheckBox(getText("ui.useTheSkinsFavicon"), true);				
				JCheckBox useMsServer = new JCheckBox(getText("ui.useMsServer"));
				JCheckBox useExpiry = new JCheckBox(getText("ui.useExpiry"), false);
				JCheckBox useRobotsTxt = new JCheckBox(getText("ui.useRobotsTxt"), false);
				JCheckBox avoidCDNs = new JCheckBox(getText("ui.avoidCDNs"), false);
				JCheckBox copyGoogleFonts = new JCheckBox(getText("ui.copyGoogleFonts"), false);
				JCheckBox addLatinExt = new JCheckBox(getText("ui.addLatinExtended"), true);
				WrappableJLabel uploadPathInfo = new WrappableJLabel("<html><i>" + getText("ui.copyPasteAlbumURLHere") + "</i></html>");
				
				{
					uploadPath.setToolTipText(getText("ui.uploadPathInfo"));
					uploadPathInfo.setPreferredWidth(360);
					useFavicon.setToolTipText(getText("ui.useFaviconInfo"));
					useMsServer.setToolTipText(getText("ui.useMsServerInfo"));
					useExpiry.setToolTipText(getText("ui.useExpiryInfo"));
					useRobotsTxt.setToolTipText(getText("ui.useRobotsTxtInfo"));
					avoidCDNs.setToolTipText(getText("ui.avoidCDNsInfo"));
					copyGoogleFonts.setToolTipText(getText("ui.copyGoogleFontsInfo"));
					addLatinExt.setToolTipText(getText("ui.addLatinExtendedInfo"));
					
					add(new JLabel(getText("ui.uploadPath")));
					add("", uploadPath);
					add("br", new JLabel(infoIcon));
					add("", uploadPathInfo);
					add("br", useFavicon);
					add(" ", new JLabel(svgIcon("favicon", new Dimension(30, 30))));
					add("br", useMsServer);
					add("br", useExpiry);
					add("br", useRobotsTxt);
					add("br", avoidCDNs);
					add("br", copyGoogleFonts);
					add("br", addLatinExt);
				}
			};			
			
			ControlPanel imageRelatedPanel = new ControlPanel(getText("ui.images")) {
				
				JCheckBox hiDpiThemeImage = new JCheckBox(getText("ui.hiDpiThemeImage"));
				JComboBox shareImageDims = new JComboBox(new Object[] {
					"640x480",
					"1024x768",
					"1200x900",
					"1600x1200"
				});
				
				{
					shareImageDims.setEditable(true);
					
					add(hiDpiThemeImage);
					add("br", new JLabel(getText("ui.shareImageDimensions")));
					add("", shareImageDims);
				}
			};
			
			ControlPanel googlePanel = new ControlPanel(getText("ui.googleAnalytics")) {

				JComboBox googleAnalytics = new JComboBox(new Object[] {
					new Item("none", getText("ui.none")), 
					new Item("gtag", "Global Site Tag (gtag.js)"),
					new Item("universal", "Universal (analytics.js)"),
					new Item("classic", "Classic (ga.js) [Legacy]")
				});
				JTextField googleSiteID = new JSmartTextField(20);
				JLabel mandatoryInfo = new JLabel(mandatory);
				JCheckBox supportDoubleclick = new JCheckBox(getText("ui.supportDoubleclick"), false);

				{
					googleSiteID.setToolTipText(getText("ui.googleSiteIDInfo"));
					mandatoryInfo.setToolTipText(getText("ui.mandatory"));

					add("", new JLabelFor(getText("ui.version"), googleAnalytics));
					add(" ", googleAnalytics);
					add(" ", new JLinkLabel("https://www.google.com/analytics/", getText("ui.signUp")));
					add("br", new JLabelFor(getText("ui.googleSiteID"), googleSiteID));
					add(mandatoryInfo);
					add("", googleSiteID);
					add("br", supportDoubleclick);

					//putClientProperty("helpPage", helpRoot + "ui/advanced.html#google-analytics");
				}
			};
			
			ControlPanel seoPanel = new ControlPanel(getText("ui.searchEngineOptimization")) {
				
				JTextField titleSEOText = new JSmartTextField(20);
				JTextField descriptionSEOText = new JSmartTextField(20);
				JSpinner preloadThumbs = new JSpinner(new SpinnerNumberModel(20, 0, 1000, 10));
				JCheckBox addAltTags = new JCheckBox(getText("ui.addAltTags"), true);
				JCheckBox slideRedirect = new JCheckBox(getText("ui.redirectSlidePages"), true);
			
				{
					titleSEOText.setToolTipText(getText("ui.titleSEOTextInfo"));
					descriptionSEOText.setToolTipText(getText("ui.descriptionSEOTextInfo"));
					//preloadThumbs.setToolTipText(getText("ui.preloadThumbsInfo"));
					addAltTags.setToolTipText(getText("ui.addAltTagsInfo"));
					writeSitemapXml.setToolTipText(getText("ui.createSitemapXmlInfo"));
					ComponentUtilities.whenSelectedEnable(writeSitemapXml, new JComponent[]{sitemapIncludeSlides});
					sitemapIncludeSlides.setToolTipText(getText("ui.sitemapIncludeSlidesInfo"));
					slideRedirect.setToolTipText(getText("ui.redirectSlidePagesInfo"));
					
					add("", new JLabelFor(getText("ui.titleSEOText"), titleSEOText));
					add("tab", titleSEOText);
					add("br", new JLabelFor(getText("ui.descriptionSEOText"), descriptionSEOText));
					add("tab", descriptionSEOText);
					add("br", new JLabel(getText("ui.preloadThumbs")));
					add("", preloadThumbs);
					add("br", addAltTags);
					add("br", writeSitemapXml);
					add("", sitemapIncludeSlides);
					add("br", slideRedirect);
				}
			};

			ControlPanel cookiePolicyPanel = new ControlPanel(getText("ui.trackingConsentAndCookiePolicy")) {

				JCheckBox askTrackingConsent = new JCheckBox(getText("ui.askTrackingConsent"), false);
				JCheckBox showCookiePolicy = new JCheckBox(getText("ui.showCookiePolicy"), false);				
				JTextField cookiePolicyStay = new JSmartTextField("8", 3);
				JTextField cookiePolicyUrl = new JSmartTextField(20);
				WrappableJLabel cookiePolicyUrlInfo = new WrappableJLabel("<html><i>" + getText("ui.cookiePolicyUrlInfo") + " </i></html>");

				{
					askTrackingConsent.setToolTipText(getText("ui.askTrackingConsentInfo"));
					showCookiePolicy.setToolTipText(getText("ui.showCookiePolicyInfo"));
					cookiePolicyUrlInfo.setPreferredWidth(320);
					
					add(askTrackingConsent);
					add("br", showCookiePolicy);
					add(" ", new JLabel(getText("ui.stay")));
					add("", cookiePolicyStay);
					add("", new JLabel("s"));
					add("br", new JLabel("URL"));
					add("tab", cookiePolicyUrl);
					add("br", new JLabel(infoIcon));
					add("tab", cookiePolicyUrlInfo);
				}
			};

			JCheckBox debugMode = new JCheckBox(getText("ui.debugMode"));
			
			ControlPanel leftPanel = new ControlPanel() {
				
				{
					add("hfill", serverRelatedPanel);
					add("br hfill", googlePanel);
					add("br", debugMode);					
				}
			};
			
			ControlPanel rightPanel = new ControlPanel() {
				
				{
					add("hfill", imageRelatedPanel);
					add("br hfill", seoPanel);
					add("br hfill", cookiePolicyPanel);
				}
			};
			
			{
				add(leftPanel);
				add(rightPanel);
				
				//putClientProperty("helpPage", helpRoot + "Site/Site_admin");
			}
		};

		//	Custom Keys
		
		ControlPanel customKeysPanel = new ControlPanel() {
		
			JTextArea customKeys = new JSmartTextArea(7, 20);
			JScrollPane customKeysPane = new JScrollPane(customKeys);

			{
				customKeys.setEditable(true);
				customKeys.setLineWrap(false);
				customKeys.setFont(mono);
				customKeys.setTabSize(2);
				customKeysPane.setBorder(BorderFactory.createTitledBorder(getText("ui.customKeys")));

				add("hfill vfill", customKeysPane);
				add("br", new JLabel(getText("ui.customKeysInfo")));
			}
		};
		
		
		/******************************************************************************
		 * 
		 *								GUI main tabs
		 * 
		 ******************************************************************************/				
		
		/*	---------------------------------------------------------------
									Site
			--------------------------------------------------------------- */

		ControlPanel design = new ControlPanel(new BorderLayout(20, 0)) {
			
			JTabbedPane designTabs = new JTabbedPane() {

				
				JScrollPane pageDesignPane = new JScrollPane(pageDesign, JScrollPane.VERTICAL_SCROLLBAR_AS_NEEDED, JScrollPane.HORIZONTAL_SCROLLBAR_NEVER);
				//JScrollPane typographyPane = new JScrollPane(typography, JScrollPane.VERTICAL_SCROLLBAR_AS_NEEDED, JScrollPane.HORIZONTAL_SCROLLBAR_NEVER);
				//JScrollPane heroPane = new JScrollPane(hero, JScrollPane.VERTICAL_SCROLLBAR_AS_NEEDED, JScrollPane.HORIZONTAL_SCROLLBAR_NEVER);
				JScrollPane foldersPane = new JScrollPane(folders, JScrollPane.VERTICAL_SCROLLBAR_AS_NEEDED, JScrollPane.HORIZONTAL_SCROLLBAR_NEVER);
				JScrollPane imagesPane = new JScrollPane(images, JScrollPane.VERTICAL_SCROLLBAR_AS_NEEDED, JScrollPane.HORIZONTAL_SCROLLBAR_NEVER);
				JScrollPane footerPane = new JScrollPane(footer, JScrollPane.VERTICAL_SCROLLBAR_AS_NEEDED, JScrollPane.HORIZONTAL_SCROLLBAR_NEVER);
				
				{
					pageDesignPane.setBorder(emptyBorder);
					//typographyPane.setBorder(emptyBorder);
					//heroPane.setBorder(emptyBorder);
					foldersPane.setBorder(emptyBorder);
					imagesPane.setBorder(emptyBorder);
					footerPane.setBorder(emptyBorder);
					this.setTabPlacement(SwingConstants.LEFT);
					
					addTab("<html><h4 " + tabStyle + ">" + getText("ui.pageDesign") + "</h4></html>", icon("color"), pageDesignPane);
					addTab("<html><h4 " + tabStyle + ">" + getText("ui.typography") + "</h4></html>", icon("type"), typography);
					addTab("<html><h4 " + tabStyle + ">" + getText("ui.hero") + "</h4></html>", icon("hero"), hero);
					addTab("<html><h4 " + tabStyle + ">" + getText("ui.images") + "</h4></html>", icon("thumbnails"), imagesPane);
					addTab("<html><h4 " + tabStyle + ">" + getText("ui.rating") + "</h4></html>", icon("star"), rating);
					addTab("<html><h4 " + tabStyle + ">" + getText("ui.markNew") + "</h4></html>", icon("new"), markNew);
					addTab("<html><h4 " + tabStyle + ">" + getText("ui.folders") + "</h4></html>", icon("folder"), foldersPane);
					addTab("<html><h4 " + tabStyle + ">" + getText("ui.pages") + "</h4></html>", icon("page"), pages);
					addTab("<html><h4 " + tabStyle + ">" + getText("ui.webLocations") + "</h4></html>", icon("globe"), weblocations);
					addTab("<html><h4 " + tabStyle + ">" + getText("ui.neighboringFolders") + "</h4></html>", icon("prev-next-page"), neighboringFolders);
					addTab("<html><h4 " + tabStyle + ">" + getText("ui.footer") + "</h4></html>", icon("footer"), footerPane);
				}
			};
			
			JPanel designPreviewPanel = new JPanel(new BorderLayout(0, 0)) {

				{
					designPreview.getView().setPreferredSize(new Dimension(previewWidth, uiHeight));

					add(designPreview.getView(), BorderLayout.CENTER);
				}
			};
			
			{		
				((BorderLayout)(getLayout())).setVgap(0);
				((BorderLayout)(getLayout())).setHgap(0);
				
				designTabs.setBorder(emptyBorder);
				
				designTabs.setPreferredSize(new Dimension(uiWidth, uiHeight));
				//designTabs.setMaximumSize(new Dimension(uiWidth + 100, uiHeight));
				
				designPreviewPanel.setBorder(BorderFactory.createLineBorder(JAlbumColor.imageBorder, 2));
				
				add(designTabs);
				add(designPreviewPanel, BorderLayout.EAST);
				
				//putClientProperty("helpPage", helpRoot + "ui/site.html");
			}
		};
						
		/*	---------------------------------------------------------------
									Functions
			--------------------------------------------------------------- */

		ControlPanel functions = new ControlPanel(new BorderLayout(20, 0)) {
			
			JTabbedPane functionTabs = new JTabbedPane() {
				
				{
					this.setTabPlacement(SwingConstants.LEFT);

					addTab("<html><h4 " + tabStyle + ">" + getText("ui.fullScreen") + "</h4></html>", icon("fullscreen"), fullScreenPanel);
					addTab("<html><h4 " + tabStyle + ">" + getText("ui.navigation") + "</h4></html>", icon("folders"), navigationPanel);
					addTab("<html><h4 " + tabStyle + ">" + getText("ui.albumInfo") + "</h4></html>", icon("info"), albumInfoPanel);
					addTab("<html><h4 " + tabStyle + ">" + getText("ui.social") + "</h4></html>", icon("connect"), socialPanel);
					addTab("<html><h4 " + tabStyle + ">" + getText("ui.backgroundMusic") + "</h4></html>", icon("audio"), backgroundMusicPanel);
					addTab("<html><h4 " + tabStyle + ">" + getText("ui.search") + "</h4></html>", icon("search"), searchPanel);
					addTab("<html><h4 " + tabStyle + ">" + getText("ui.tagCloud") + "</h4></html>", icon("tag"), tagCloudPanel);
					addTab("<html><h4 " + tabStyle + ">" + getText("ui.filtering") + "</h4></html>", icon("filter"), filteringPanel);
					addTab("<html><h4 " + tabStyle + ">" + getText("ui.shoppingCart") + "</h4></html>", icon("shopping-cart"), shopPanel);
					addTab("<html><h4 " + tabStyle + ">" + "Fotomoto" + "</h4></html>", icon("cart-fotomoto"), fotomotoPanel);
					addTab("<html><h4 " + tabStyle + ">" + getText("ui.feedback") + "</h4></html>", icon("email-send"), feedbackPanel);
					addTab("<html><h4 " + tabStyle + ">" + getText("ui.download") + "</h4></html>", icon("download"), downloadPanel);
					addTab("<html><h4 " + tabStyle + ">" + getText("ui.print") + "</h4></html>", icon("printer"), printPanel);
					addTab("<html><h4 " + tabStyle + ">" + getText("ui.map") + "</h4></html>", icon("location"), mapPanel);
					addTab("<html><h4 " + tabStyle + ">" + getText("ui.photoData") + "</h4></html>", icon("camera"), photodataPanel);
					addTab("<html><h4 " + tabStyle + ">" + getText("ui.regions") + "</h4></html>", icon("facetag"), regionsPanel);
					addTab("<html><h4 " + tabStyle + ">" + getText("ui.audioClips") + "</h4></html>", icon("volume-up"), audioClips);

				}
			};
							
			ControlPanel buttonsOrdering = new ControlPanel() {

				JList buttonsOrder = new JDraggableList(new Object[] {
					new JLabelFor("<html><p style=\"padding:2px 0;\">" + getText("ui.fullScreen") + "</p></html>", icon("fullscreen")).name("fullscreen"),
					new JLabelFor("<html><p style=\"padding:2px 0;\">" + getText("ui.navigation") + "</p></html>", icon("folders")).name("navigation"),
					new JLabelFor("<html><p style=\"padding:2px 0;\">" + getText("ui.albumInfo") + "</p></html>", icon("info")).name("albumInfo"),
					new JLabelFor("<html><p style=\"padding:2px 0;\">" + getText("ui.social") + "</p></html>", icon("connect")).name("social"),
					new JLabelFor("<html><p style=\"padding:2px 0;\">" + getText("ui.backgroundMusic") + "</p></html>", icon("audio")).name("music"),
					new JLabelFor("<html><p style=\"padding:2px 0;\">" + getText("ui.search") + "</p></html>", icon("search")).name("search"),
					new JLabelFor("<html><p style=\"padding:2px 0;\">" + getText("ui.tags") + "</p></html>", icon("tag")).name("tags"),
					new JLabelFor("<html><p style=\"padding:2px 0;\">" + getText("ui.filtering") + "</p></html>", icon("filter")).name("filtering"),
					new JLabelFor("<html><p style=\"padding:2px 0;\">" + getText("ui.shoppingCart") + "</p></html>", icon("shopping-cart")).name("shoppingCart"),
					new JLabelFor("<html><p style=\"padding:2px 0;\">" + "Fotomoto" + "</p></html>", icon("cart-fotomoto")).name("fotomoto"),
					new JLabelFor("<html><p style=\"padding:2px 0;\">" + getText("ui.feedback") + "</p></html>", icon("email-send")).name("feedback"),
					new JLabelFor("<html><p style=\"padding:2px 0;\">" + getText("ui.download") + "</p></html>", icon("download")).name("download"),
					new JLabelFor("<html><p style=\"padding:2px 0;\">" + getText("ui.print") + "</p></html>", icon("printer")).name("print"),
					new JLabelFor("<html><p style=\"padding:2px 0;\">" + getText("ui.map") + "</p></html>", icon("location")).name("map"),
					new JLabelFor("<html><p style=\"padding:2px 0;\">" + getText("ui.photoData") + "</p></html>", icon("camera")).name("photodata"),
					new JLabelFor("<html><p style=\"padding:2px 0;\">" + getText("ui.regions") + "</p></html>", icon("facetag")).name("regions"),
					new JLabelFor("<html><p style=\"padding:2px 0;\">" + getText("ui.audioClips") + "</p></html>", icon("volume-up")).name("audioClip")
				});

				{						
					add("", buttonsOrder);
					add("br center", new JLabel("<html><i>" + getText("ui.dragToReorder") + "</i></html>"));
				}
			};
				
			{
				((BorderLayout)(getLayout())).setVgap(0);
				((BorderLayout)(getLayout())).setHgap(0);
				
				functionTabs.setBorder(emptyBorder);
				functionTabs.setPreferredSize(new Dimension(uiWidth + 200, uiHeight));
				//functionTabs.setMaximumSize(new Dimension(uiWidth + 300, uiHeight));
				
				add(functionTabs);
				add(buttonsOrdering, BorderLayout.EAST);
				
				//putClientProperty("helpPage", helpRoot + "ui/sections.html");
			}
		};
				

		/*	---------------------------------------------------------------
									Functions
			--------------------------------------------------------------- */

		ControlPanel lightbox = new ControlPanel(new BorderLayout(0, 0)) {
			
			JTabbedPane lightboxTabs = new JTabbedPane() {
				
				{
					this.setTabPlacement(SwingConstants.LEFT);

					addTab("<html><h4 " + tabStyle + ">" + getText("ui.settings") + "</h4></html>", icon("settings"), lightboxSettings);
					addTab("<html><h4 " + tabStyle + ">" + getText("ui.mainImage") + "</h4></html>", icon("image"), lightboxMainImage);
					addTab("<html><h4 " + tabStyle + ">" + getText("ui.caption") + "</h4></html>", icon("caption"), lightboxCaption);
				}
			};
					
			{
				((BorderLayout)(getLayout())).setVgap(0);
				((BorderLayout)(getLayout())).setHgap(0);
				
				lightboxTabs.setBorder(emptyBorder);
				
				add(lightboxTabs);
			}
		};
		
		/*	---------------------------------------------------------------
									Advanced
			--------------------------------------------------------------- */
		
		ControlPanel advanced = new ControlPanel(new BorderLayout(0, 0)) {
			
			JTabbedPane advancedTabs = new JTabbedPane() {
								
				JScrollPane siteAdminPane = new JScrollPane(siteAdminPanel, JScrollPane.VERTICAL_SCROLLBAR_AS_NEEDED, JScrollPane.HORIZONTAL_SCROLLBAR_NEVER);
				
				{
					customCodePanel.setBorder(emptyBorder);
					customContentPanel.setBorder(emptyBorder);
					siteAdminPane.setBorder(emptyBorder);
					customKeysPanel.setBorder(emptyBorder);
					this.setTabPlacement(SwingConstants.LEFT);
					
					addTab("<html><h4 " + tabStyle + ">" + getText("ui.siteAdmin") + "</h4></html>", icon("wrench"), siteAdminPane);
					addTab("<html><h4 " + tabStyle + ">" + getText("ui.customCode") + "</h4></html>", icon("code"), customCodePanel);
					addTab("<html><h4 " + tabStyle + ">" + getText("ui.customContent") + "</h4></html>", icon("edit"), customContentPanel);
					addTab("<html><h4 " + tabStyle + ">" + getText("ui.customKeys") + "</h4></html>", icon("rename"), customKeysPanel);
				}
			};
			
			{
				((BorderLayout)(getLayout())).setVgap(0);
				((BorderLayout)(getLayout())).setHgap(0);
				
				advancedTabs.setBorder(emptyBorder);
				
				add(advancedTabs);
				
				//putClientProperty("helpPage", helpRoot + "ui/lightbox.html");
			}
		};
		
		/*	---------------------------------------------------------------
									About
			--------------------------------------------------------------- */

		ControlPanel about = new ControlPanel(new BorderLayout(20, 0)) {
			
			private JTextArea readme = new JSmartTextArea( getFileContents("readme.txt"), 20, 30 );
			private JScrollPane aboutPane = new JScrollPane(readme, JScrollPane.VERTICAL_SCROLLBAR_ALWAYS, JScrollPane.HORIZONTAL_SCROLLBAR_NEVER);
			
			ControlPanel info = new ControlPanel() {
				
				{
					add("center", new JLabel(svgIcon("story-logo", new Dimension(130, 130))));
					add("br center", new JLabel("<html><h2>" + skin + "</h2></html>"));
					add("br center", new JLabel("Jalbum " + internalVersion));
					add(new JLinkLabel("https://jalbum.net/software/download", getText("ui.upgrade"), getText("ui.downloadJalbum")));
					add("br center", new JLabel(skin + " skin " + skinVer));
					add(new JLinkLabel("https://jalbum.net/skins/skin/" + skin, getText("ui.upgrade"), getText("ui.downloadSkin")));
					//add("br center", new JLinkLabel(helpRoot + "index.html", getText("help")));
					add("br center", new JLinkLabel(supportForum, getText("ui.support")));
				}
			};
			
			{								
				((BorderLayout)(getLayout())).setVgap(0);
				((BorderLayout)(getLayout())).setHgap(0);
				
				readme.setLineWrap(true);
				readme.setWrapStyleWord(true);
				readme.setEditable(false);
				readme.setFont(mono);
				
				info.setPreferredSize(new Dimension(240, uiHeight));
				aboutPane.setPreferredSize(new Dimension(400, 320));
				
				add(info, BorderLayout.WEST);
				add(aboutPane);
				
				//putClientProperty("helpPage", helpRoot + "index.html");
			}
		};
		
		/*	---------------------------------------------------------------
									Main tabs
			--------------------------------------------------------------- */
		
		JTabbedPane tabs = new JTabbedPane() {
			
			{
				design.setBorder(emptyBorder);
				functions.setBorder(emptyBorder);
				lightbox.setBorder(emptyBorder);
				advanced.setBorder(emptyBorder);
				about.setBorder(emptyBorder);
				
				addTab("<html><h4 style='padding:4px 6px;margin:4px;'>" + getText("ui.design") + "</h4></html>", icon("design", 27), design);
				addTab("<html><h4 style='padding:4px 6px;margin:4px;'>" + getText("ui.functions") + "</h4></html>", icon("button-clicked", 27), functions);
				addTab("<html><h4 style='padding:4px 6px;margin:4px;'>" + getText("ui.lightbox") + "</h4></html>", icon("lightbox", 27), lightbox);
				addTab("<html><h4 style='padding:4px 6px;margin:4px;'>" + getText("ui.advanced") + "</h4></html>", icon("wrench", 27), advanced);
				addTab("<html><h4 style='padding:4px 6px;margin:4px;'>" + getText("ui.about") + "</h4></html>", icon("info", 27), about);

			}
		};
		
		private final String designTemplate = getFileContents("lib/design-template.html");
	
		{
			// Adding UI tabs
			
			((RiverLayout)(getLayout())).setVgap(0);
			((RiverLayout)(getLayout())).setHgap(0);
			tabs.setBorder(emptyBorder);
			
			add(tabs);
			
			/*	-----------------------------------------------------------
										Listeners
				----------------------------------------------------------- */
			
			window.addJAlbumListener(new JAlbumAdapter() {
				@Override
				public void styleChanged(JAlbumEvent e) {
					styleName.setText(getStyleName());
				}
			});
			
			window.addPropertyChangeListener(JAlbumFrame.SKIN_LOADED_PROPERTY, setupMonitors);

			window.addJAlbumListener(new JAlbumAdapter() {
				@Override
				public void skinChanged(JAlbumEvent e) {
					//JAlbum.logger.log(Level.FINE, "Skin changed, removing listeners.");
					window.removePropertyChangeListener(JAlbumFrame.SKIN_LOADED_PROPERTY, setupMonitors);
				}
			});
						
			try {
				window.albumExplorer.onContextMenu(menu -> {
					AlbumObject[] sel = window.albumExplorer.explorer.getSelectedAlbumObjects();
	
					if (sel.length > 0) {
						//  Treat image as... submenu
						createComboSubmenu(menu, sel, "treatImageAs", new Item[] {
								new Item("", texts.getString("ui.default")),
								new Item("thumbnail", texts.getString("ui.thumbnail")),
								new Item("single", texts.getString("ui.single")),
								new Item("fullwidth", texts.getString("ui.fullWidth"))
							}, getText("ui.treatImageAs"));
						
						//  Hero slider selector
						createMenuItemForBooleanProperty(menu, sel, "useInSlider", getText("ui.useInHeroSlider"));
						
						//  Panorama selector
						createMenuItemForBooleanProperty(menu, sel, "panorama", getText("ui.panorama"));
						
						//  Hide location selector
						createMenuItemForBooleanProperty(menu, sel, "hideLocation", getText("ui.hideLocation"));
					}
				});
			} catch (Throwable ex) {
				JAlbum.logger.log(Level.INFO, "Right-click extension menu is available only in jAlbum v32 and newer!");
			};
			
			Map<String,Object> vars = engine.getSkinVariables();
			selectComboBoxItem(titleCaptionPresets, (String)vars.get("titleCaptionTemplate"));
			selectComboBoxItem(folderCaptionPresets, (String)vars.get("folderCaptionTemplate"));
			selectComboBoxItem(thumbCaptionPresets, (String)vars.get("thumbCaptionTemplate"));
			selectComboBoxItem(singleCaptionPresets, (String)vars.get("singleCaptionTemplate"));
			selectComboBoxItem(fullCaptionPresets, (String)vars.get("fullCaptionTemplate"));
			selectComboBoxItem(imageCaptionPresets, (String)vars.get("imageCaptionTemplate"));
			
			putClientProperty("helpPage", helpRoot);
		}
		
	};
			
	public Gui() {
		this(JAlbumContext.getInstance());
	}
   
	public Gui(JAlbumContext context) {
		
		super(context);
		PluginContext pc = context.getPluginContext();
		EditPanel editPanel = pc.getEditPanel();
		
		editPanel.addCustomTab(getText("ui.imageData"), new ImageDataUI(context, texts));
		editPanel.addCustomTab(getText("ui.external"), new ExternalContentUI(context, texts));

		skinUi.setBorder(emptyBorder);
		((RiverLayout)(getLayout())).setVgap(0);
		((RiverLayout)(getLayout())).setHgap(0);
		
		window.setSkinUI(skinUi);
		skinReadyAt = System.currentTimeMillis();
								
	}
	
}
