/*	
 *	swipe() :: Swipe action using CSS transforms with jQuery position fallback
 *
 *	Copyright by Lazaworx
 *	http://www.lazaworx.com
 *	Author: Laszlo Molnar
 *
 *	Dual licensed under the MIT and GPL licenses.
 *	- http://www.opensource.org/licenses/mit-license.php
 *	- http://www.gnu.org/copyleft/gpl.html
 *
 *	Usage: $(element).swipe( options );
 */
 
;(function($, $window, $document, undefined) {
	'use strict';
	
	$.fn.swipe = function(options) {
		
		options = $.extend({
						onSwipedLeft:			false,			// Swiped left event
						onSwipedRight:			false,			// Swiped right event
						onSwipedUp:				false,			// Swiped up event
						onSwipedDown:			false,			// Swiped down event
						onSmallSwipe:			false,			// Small swipe (< threshold px)
						onSwipeStart:			false,			// Start swiping event
						onPinchZoom:			false,			// Pinch zoomed event
						onClick:				false,			// Clicked
						onDoubleClick:			false,			// Doble-clicked
						onFinished:				false,			// Swiped out - can remove
						threshold:				20,				// Small swipe threshold
						clickThreshold:			150,			// Timespan to tell apart single and double click
						doubleClickThreshold:	350,			// Two clicks come in shorter time treated as double-click 
						noSwipeInside:			'.caption',		// Avoid swipe inside this container
						stopAutoPano:			true,			// Stops autopano move, and doesn't trigger click if done so
						swipeoutDuration:		300,			// Swipe out speed in ms
						overThreshold:			0.2				// Oversized pictures must moved beyond (this * screen width) to trigger swipe 
					}, options);
		
		// Getting coordinates of a touch or mouse event
		var	getDim = function(el) {
					return [ el.outerWidth(), el.outerHeight() ];
				},

			getPos = function(e) {
					if (e.touches) {
						if (e.touches.length > 1) {
							// Two finger event
							return [ e.touches[0].clientX, e.touches[0].clientY, e.touches[1].clientX, e.touches[1].clientY ];
						}
						// Single finger event
						return [ e.touches[0].clientX, e.touches[0].clientY ];
					}
					// mouse event
					return (e.clientX === null)? null : [ e.clientX, e.clientY ];
				};
				
		return this.each(function() {
				
				var self = $(this),							// target element
					cont = self.parent(),					// container or clipping window
					ns = self.data('lsw_ns') ||				// namespace 
						('lsw_' + Math.floor(Math.random() * 10000)),
					rto,									// window resize update timeout
					// Dimensions
					selfDim,								// item dimensions
					clipDim,								// clip dimensions
					scale,									// scaling
					// Event timestamps
					startTime,								// start time
					lastTime,								// previous event time (to calculate speed)
					endTime,								// end time
					// Event coordinates (screen-relative)
					startPos,								// initial position Array[2..4]
					currPos,								// Current position
					// Etc
					speed,									// move speed [ vx, vy ] in px/ms
					startD,									// initial distance of two finger events
					currD,
					mx,										// original matrix
					// Flags
					hasStarted = false,						// flag to avoid double start on touch devices
					hasEnded = true,						// ended swiping
					swipeHandled = true,					// swipe event already fired
					dontSwipeUp = true,						// limit vertical swipe
					dontSwipeDown = true,
					canConstrainHorz = false,				// Can constrain to horizontal
					nonSwipeable = true,					// target element swipeable?
					clickedOnce = false,					// Check for double click
					hadAutoPano = false,					// Remembering if autopano was on
					chkDoubleClick = null,					// Checking double-click (timeout)
					
					// Handling swipe start event
					handleSwipeStart = function(e) {
							if (typeof options.onSwipeStart === FUNCTION) {
								options.onSwipeStart.call(self[0], e);
							}
						},
						
					// Handling swipeout movement finished event
					handleFinished = function(e) {
							//log('handleFinished');
							self.data('scrolling', false);
							if (typeof options.onFinished === FUNCTION) {
								options.onFinished.call(self[0], e);
							}
						},
						
					// Handling small swipe
					handleSmallSwipe = function(e) {
							//log('handleSmallSwipe');
							unregisterEvents(e);
							self.data('scrolling', false);
							if (typeof options.onSmallSwipe === FUNCTION) {
								options.onSmallSwipe.call(self[0], e);
							}
						},
						
					// Handling click event
					handleClick = function(e) {
							//log('handleClick');
							unregisterEvents(e);
							self.data('scrolling', false);
							if (!hadAutoPano && typeof options.onClick === FUNCTION) {
								// Avoid click when stopping panomove consumed the event
								options.onClick.call(self[0], e);
							}
						},
						
					// Handling doubleclick event
					handleDoubleClick = function(e) {
							//log('handleDoubleClick');
							unregisterEvents(e);
							self.data('scrolling', false);
							if (typeof options.onDoubleClick === FUNCTION) {
								options.onDoubleClick.call(self[0], e);
							}
						},
						
					// Handling swipe		
					handleSwipedDown = function(e) {
							//log('handleSwipe:'+(right?'right':'left'));
							if (!swipeHandled) {
								unregisterEvents(e);
								swipeHandled = true;
								// Swiped down
								options.onSwipedDown.call(self[0], e);
							}
						},
							
					// Handling swipe		
					handleSwipedUp = function(e) {
							//log('handleSwipe:'+(right?'right':'left'));
							if (!swipeHandled) {
								unregisterEvents(e);
								swipeHandled = true;
								// Swiped up
								options.onSwipedUp.call(self[0], e);
							}
						},
							
					// Handling swipe		
					handleSwipedRight = function(e) {
							//log('handleSwipe:'+(right?'right':'left'));
							if (!swipeHandled) {
								unregisterEvents(e);
								swipeHandled = true;
								// Swiped right
								options.onSwipedRight.call(self[0], e);
							}
						},
							
					// Handling swipe		
					handleSwipedLeft = function(e) {
							//log('handleSwipe:'+(right?'right':'left'));
							if (!swipeHandled) {
								unregisterEvents(e);
								swipeHandled = true;
								// Swiped left
								options.onSwipedLeft.call(self[0], e);
							}
						},
						
					// Handling pinch zoom
					handlePinchZoom = function(e) {
							//log('handlePinchZoom:'+(right?'right':'left'));
							if (!swipeHandled) {
								unregisterEvents(e);
								swipeHandled = true;
								// Swiped left
								options.onPinchZoom.call(self[0], e);
							}
						},
							
					// Handling right click event
					handleRightClick = function(e) {
							//log('handleRightClick');
							unregisterEvents(e);
							if (typeof options.onRightClick === FUNCTION) {
								options.onRightClick.call(self[0], e);
							}
						},
					
					// Registering events
					registerEvents = function(e) { 
							//log('registerEvents');
							if (e.type === 'mousedown') {
								self.on('mouseup.' + ns + ' mouseout.' + ns, swipeEnd);
								$document.on('mousemove.' + ns, swipeMove);
							} else if (e.type === TOUCH.START) {
								self.on(TOUCH.END + '.' + ns, swipeEnd);
							}
							clickedOnce = false;
						},
						
					unregisterEvents = function(e) {
							//log('unregisterEvents');
							if (e.type === 'mouseup' || e.type === 'mouseout') {
								self.off('mouseup.' + ns + ' mouseout.' + ns);
								$document.off('mousemove.' + ns);
							} else if (e.type === TOUCH.END) {
								self.off(TOUCH.END + '.' + ns);
							}
						},
						
					// Getting distance of two finger events
					getDistance = function(p) {
							return Math.hypot(p[2] - p[0], p[3] - p[1]);
						},

					// Move to
					// p = event coordinates (2 = one finger, 4 = 2 finger gesture)
					// op = opacity (null = don't care
					// speed = transition speed (ms)
					// doneFn = callback
					transformTo = function(p, op, speed, doneFn) {
							var css = {},
								dx,
								dy;
							
							if (p.length === 4) {
								// Move plus scale; 2 finger move
								scale = getDistance(p) / startD;
								// Midpoint
								dx = (p[0] - startPos[0] + p[2] - startPos[2]) / 2;
								dy = (p[1] - startPos[1] + p[3] - startPos[3]) / 2;
							} else {
								// Move only
								dx = p[0] - startPos[0];
								dy = p[1] - startPos[1];
							}
							
							css.transform = 'matrix(' + (mx[0] * scale) + ',' + mx[1] + ',' + mx[2] + ',' + (mx[3] * scale) + ',' + (mx[4] + dx) + ',' + (mx[5] + dy) + ')';
							
							if (typeof speed !== UNDEF && speed) {
								css.transition = 'transform ' + speed + 'ms ease-in-out';
								if (op !== null) {
									css.transition += ',opacity ' + speed + 'ms ease-in-out';
									css.opacity = op;
								}
							} else {
								css.transition = 'none';
							}
							
							if (typeof doneFn === FUNCTION) {
								self.one('transitionend', doneFn);
							}
							
							self.css(css);
						},
					
					// Starting swipe
					swipeStart = function(e) {
							//log('swipeStart:' + e.type);
							
							//log('Touch start, fingers: ' + (e.originalEvent.touches? e.originalEvent.touches.length : 0));
							if (pageZoomed() ||
								e.originalEvent.touches && e.originalEvent.touches.length > 2 ||
								(options.noSwipeInside && $(e.target).closest(options.noSwipeInside).length) ||
								e.target.nodeName === 'A' ||
								e.target.nodeName === 'INPUT' ||
								e.target.nodeName === 'BUTTON' ||
								e.target.nodeName === 'AUDIO' || 
								(e.target.nodeName === 'VIDEO' && 
									(!e.target.paused ||
										e.target.controls && 
										(ISIOSDEVICE || e.offsetY > ($(e.target).height() - 60))
									)
								)
							) {
								// Don't swipe buttons and links and other non-swipeable elements
								// No swiping on iOS video player
								nonSwipeable = true;
								return true;
							}
							
							if (e.type === 'mousedown' && e.which !== 1) {									
								// Right click
								if (options.stopAutoPano && cont.data('panomove')) {
									self.trigger('autopanoStop');
								}
								handleRightClick(e);
								return true;
							}
							
							if (hasStarted || clickedOnce) {
								// No mouseup / touchend registered yet
								return false;
							}
							
							if (e.type !== 'touchstart') {
								// Cancel bubbling, but not break long tap
								e.preventDefault();
							}
							
							hasStarted = true;
							nonSwipeable = false;
							hasEnded = false;
							clickedOnce = false;
							
							if (options.stopAutoPano && cont.data('panomove')) {
								self.trigger('autopanoStop');
								hadAutoPano = true;
							} else {
								hadAutoPano = false;
							}
							
							self.data({
									scrolling: 		true,
									swipeEnded:		false,
									taplength:		0
								});
							
							startTime = lastTime = new Date().getTime();
							currPos = startPos = getPos(e.originalEvent);
							speed = [ 0, 0 ];
							if (startPos.length === 4) {
								startD = getDistance(startPos);
							}
							clipDim = getDim(cont);
							selfDim = getDim(self);
							mx = self.getMatrix();
							scale = 1;
							canConstrainHorz = hadAutoPano || clipDim[1] >= selfDim[1] * mx[3] && clipDim[0] >= selfDim[0] * mx[0];
							//log('wh: ' + clipDim[1] + ' ih: ' + selfDim[1] + ' sc: ' + mx[3].toFixed(2));
							
							if (canConstrainHorz) {
								dontSwipeUp = typeof options.onSwipedUp !== FUNCTION;
								dontSwipeDown = typeof options.onSwipedDown !== FUNCTION;
							} else {
								dontSwipeUp = dontSwipeDown = false;
							}
							
							registerEvents(e);
							handleSwipeStart(e);
							swipeHandled = false;
							// Do not cancel long tap
							return e.type === 'touchstart';
						},
					
					// Moving in progress
					swipeMove = function(e) {
							//log('swipeMove:' + e.type + ' [' + e.originalEvent.touches.length + ']');
							
							if (hasEnded || nonSwipeable ||
								e.originalEvent.touches && e.originalEvent.touches.length > 2) {
								// not active, prohibited area or three finger touch
								return true;
							}
							
							e.preventDefault();
							
							var lastPos = currPos,				// Saving previous position
								t = new Date().getTime(),
								dt = (t - lastTime) || 1;
														
							currPos = getPos(e.originalEvent);
							
							if (startPos === null || startPos.length < currPos.length) {
								// No swipestart happened: init startPos / startD now 
								for (var i = startPos.length || 0; i < currPos.length; i++) {
									startPos[i] = currPos[i];
								}
								startD = getDistance(currPos);
							}
							
							if (dontSwipeUp && dontSwipeDown) {
								// Horizontal-only
								currPos[1] = startPos[1];
								// Speed: Smoothing with 2/3 : 1/3
								speed[0] = (speed[0] + 2 * (currPos[0] - lastPos[0]) / dt) / 3;
								lastPos[0] = currPos[0];
								
							} else {
								var d = Math.abs(currPos[0] - startPos[0]),
									a = Math.abs(currPos[0] - startPos[0] / (currPos[1] - startPos[1]));
									
								// Can move both axes
								if (canConstrainHorz && (d > 30 || d > 8 && a > 0.25)) {
									// Change to horizontal-only swipe
									dontSwipeUp = dontSwipeDown = true;
								}
								
								if (dontSwipeUp && currPos[1] < startPos[1] ||
									dontSwipeDown && currPos[1] > startPos[1]) {
									currPos[1] = startPos[1];
								}
								
								speed = [
										(speed[0] + 2 * (currPos[0] - lastPos[0]) / dt) / 3,
										(speed[1] + 2 * (currPos[1] - lastPos[1]) / dt) / 3
									];
									
								//log('speed: ' + speed[0].toFixed(1) + ', ' + speed[1].toFixed(1));
								lastPos = currPos;
							}
							
							lastTime = t; 
							transformTo(currPos);
							return false;
						},
					
					// Cancel swipe (e.g. moved out of the screen)
					swipeCancel = function(e) {
							//log('swipeCancel:' + e.type);
							
							hasEnded = true;
							hasStarted = false;
							
							unregisterEvents(e);
							
							setTimeout(function() { 
									self.data({
											scrolling:	 	false,
											swipeEnded:		true
										});
								}, 100);
							
							transformTo(startPos, null, 100);
							
							return false;
						},
						
					// Stopped moving or clicked : mouse/touch up
					swipeEnd = function(e) {
							//log('swipeEnd:' + e.type);
							if (nonSwipeable) {
								// Prohibited area
								return true;
							}
							
							e.preventDefault();
							
							if (chkDoubleClick) {
								clearTimeout(chkDoubleClick);
								chkDoubleClick = null;
							}
							
							hasStarted = false;
							hasEnded = true;
							
							//currPos = getPos(e.originalEvent);
							
							if (currPos.length === 4) {
								handlePinchZoom(e);
							} else {
								var dx = currPos[0] - startPos[0],
									dy = currPos[1] - startPos[1],
									dt = new Date().getTime() - startTime,
									w = selfDim[0] * scale * mx[0],
									h = selfDim[1] * scale * mx[3];
								
								if ((dt > 50) && (Math.hypot(dx, dy) > options.threshold)) {
									// Not a click, because it took longer or the distance is larger than the threshold
									
									// Setting data flags
									self.data({
											scrolling:	 	true,
											swipeEnded:		true,
											taplength:		dt
										});
									
									if (dontSwipeUp && dontSwipeDown) {
										// Constrain horizontally
										//var targetDx = dx * speed;
										var extra = speed[0] * options.swipeoutDuration / 2;		// extra = extra travel after end event		
										// Linear slowing: dx = v * t / 2;
										
										//log('speed: ' + speed[0].toFixed(1));
										//log('extra: ' + extra.toFixed(1) + 'px');
										//console.log('mx[4]='+mx[4].toFixed(1)+'\ncurrPos[0]='+currPos[0].toFixed(1)+'\ndx='+dx.toFixed(1)+'\ntargetDx='+targetDx.toFixed(1)+'\nclipDim[0]='+clipDim[0]+'\nselfDim[0]='+selfDim[0]);
										if (	(clipDim[0] >= w) ||
												dx && 
												(
													((mx[4] + dx + extra) > clipDim[0] * (options.overThreshold - 0.5)) ||
													((mx[4] + dx + extra) < clipDim[0] * (0.5 - options.overThreshold) - w)
												)
											) {
											// Swiped => move and fade out
											transformTo([ currPos[0] + extra, startPos[1] ], 0, options.swipeoutDuration, handleFinished);
											
											if (dx > 0) {
												handleSwipedRight(e);
											} else {
												handleSwipedLeft(e);
											}
											
										} else {
											// Panoramic :: Hasn't reached the edge yet
											transformTo([ currPos[0] + extra, startPos[1] ], null, options.swipeoutDuration);
											unregisterEvents(e);
										}
									
									} else {
										// No constrain
										var vertical = Math.abs(dy) > 1.5 * Math.abs(dx), 				// max 30 degrees
											extra = [													// extra = extra travel after end event
													speed[0] * options.swipeoutDuration / 2,
													speed[1] * options.swipeoutDuration / 2
												];
												
										if (w < clipDim[0] * 1.1 || h < clipDim[1] * 1.1) {
											// Smaller than screen at least in one direction and swiped over the boundaries
											if (typeof options.onSwipedDown === FUNCTION && vertical && dy > 0) {
												// Swiped down
												transformTo([ startPos[0], currPos[1] + extra[1] ], 0, options.swipeoutDuration, handleFinished);
												handleSwipedDown(e);
											} else if (typeof options.onSwipedUp === FUNCTION && vertical && dy < 0) {
												// Swiped up
												transformTo([ startPos[0], currPos[1] + extra[1] ], 0, options.swipeoutDuration, handleFinished);
												handleSwipedUp(e);
											} else {
												// Horizontal swipe or vertical but no up or down swipe handler
												transformTo([ currPos[0] + extra[0], startPos[1] ], 0, options.swipeoutDuration, handleFinished);
												if (dx > 0) {
													handleSwipedRight(e);
												} else {
													handleSwipedLeft(e);
												}
											}
										} else {
											// Check for boundary excess
											var off = [ 0, 0 ];											// Amount of moving over the boundaries
											
											if (dx > 0) {
												// Move right check left
												off[0] = Math.max(0, (mx[4] + dx + extra[0]) - clipDim[0] * (options.overThreshold - 0.5));
											} else if (dx < 0) {
												// Check right
												off[0] = Math.min(0, (mx[4] + dx + extra[0]) + w - clipDim[0] * (0.5 - options.overThreshold));
											}
											
											if (dy > 0) {
												// Check top
												off[1] = Math.max(0, (mx[5] + dy + extra[1]) - clipDim[1] * (options.overThreshold - 0.5));
											} else if (dy < 0) {
												// Check bottom
												off[1] = Math.min(0, (mx[5] + dy + extra[1]) + h - clipDim[1] * (0.5 - options.overThreshold));
											}
											
											if (off[0] || off[1]) {
												// Moved over the boundary; check for at least half screen momentum
												if (typeof options.onSwipedDown === FUNCTION && vertical && off[1] > clipDim[1] / 2) {
													// Swiped down
													transformTo([ startPos[0], currPos[1] + extra[1] ], 0, options.swipeoutDuration, handleFinished);
													handleSwipedDown(e);
												} else if (typeof options.onSwipedUp === FUNCTION && vertical && (-off[1]) > clipDim[1] / 2) {
													// Swiped up
													transformTo([ startPos[0], currPos[1] + extra[1] ], 0, options.swipeoutDuration, handleFinished);
													handleSwipedUp(e);
												} else if (Math.abs(off[0]) > clipDim[0] / 2) {
													// Horizontal swipe or vertical but no up or down swipe handler
													transformTo([ currPos[0] + extra[0], startPos[1] ], 0, options.swipeoutDuration, handleFinished);
													if (dx > 0) {
														handleSwipedRight(e);
													} else {
														handleSwipedLeft(e);
													}
												} else {
													// Rebounce to boundary
													transformTo([ currPos[0] + extra[0] - off[0], currPos[1] + extra[1] - off[1] ], null, options.swipeoutDuration);
													unregisterEvents(e);
												}
												
											} else {
												// No need to limit; continue swiping
												transformTo([ currPos[0] + extra[0], currPos[1] + extra[1] ], null, options.swipeoutDuration);
												unregisterEvents(e);
											}
										}
									}	
								
									clickedOnce = false;
								
								} else if (dx || dy) {
									// Small swipe: no click triggered
									handleSmallSwipe(e);
									transformTo(startPos, null, 100);
									clickedOnce = false;
									
								} else if (options.onDoubleClick) {
									// Has doubleclick handler event
									if (clickedOnce) {
										// One click already registered
										if (dt < options.doubleClickThreshold) {
											// Within threshold: Double click
											handleDoubleClick(e);
										} else {
											// Over threshold
											handleClick(e);
										}
										clickedOnce = false;
											
									} else { 
										// Wait if second click happens
										clickedOnce = true;
										// Triggering single click if no second click comes
										chkDoubleClick = setTimeout(function(e) {
												handleClick(e);
												clickedOnce = false;
											}, options.doubleClickThreshold + 10, e);
									}
									
								} else {
									// Handling single clicks only if no autopano was on
									handleClick(e);
								}
							}
							
							return false;
						},
					
					noAction = function(e) {
							e.preventDefault();
							return false;
						},
					
					// Tearing off the swipe handler
					removeSwipe = function() {
							setTimeout(function() { 
									self.data('scrolling', false);
								}, 20);
							self.removeAttr('draggable');
							self.add($document).off('.' + ns);
						};
					
					
				// clean up events that may hang around (in theory this should never happen)
				self.add($document).off('.' + ns);
				
				// Storing namespace
				self.data('lsw_ns', ns);
				
				// Need to be updated about the window size
				$window.on('resize.' + ns, function() {
						clearTimeout(rto);
						rto = setTimeout(function() {
							clipDim = getDim(cont);		
						}, 50);
					});
				
				self.attr('draggable', 'false')
					.on(TOUCH.START + '.' + ns + ' dragstart.' + ns + ' mousedown.' + ns, swipeStart)
					//.on(TOUCH.CANCEL + '.' + ns + ' dragcancel.' + ns, swipeCancel)
					.on(TOUCH.MOVE + '.' + ns + ' drag.' + ns, swipeMove)
					// External remove request
					.on('removeSwipe.' + ns, removeSwipe)
					// Do not select
					.on('selectstart.' + ns, noAction);
				
				//self.logEvents();
		
			});
		};
			
})(jQuery, jQuery(window), jQuery(document));
