/*
 * 	lightbox.js - lightbox for Story skin
 *	jslint browser: true
 */
 
;(function($, $window, $document, $body, undefined) {	
	'use strict';
	// Requires laza.util, laza.swipe and laza.transform
	
	$.fn.lightbox = function(album, settings) {
		
			if (typeof album === UNDEF) {
				return this;
			}
			
			settings = $.extend({}, $.fn.lightbox.defaults, settings);
	
			var self = $(this),
				text = getTranslations($.fn.lightbox.text),
				thumbs = $([]),
				thumb = $(),
				main = $(),
				oldMain = $(),
				image = $(),
				media = $(),
				caption = $(),
				oldCaption = $(),
				controls = $(),
				panels = $(),
				audioClip = $(),
				el,
				isImage = true,
				isVideo = false,
				isAudio = false,
				isOther = false,
				isExternal = false,
				isVr = false,
				isPdf = false,
				isPanorama = false,
				dontRemove = false,
				slideshow = null,
				slideshowSuspended = false,
				slideshowDelay = settings.slideshowDelay,
				transitionSpeed = settings.transitionSpeed,
				audioClipTimeout = null,
				fitTimeout = null,
				controlsTimeout = null,
				resizeTimeout = null,
				autoPanoTimeout = null,
				progressTo = null,
				listening = false,
				
				maxZoom = settings.maxZoom,
				fitImages = settings.fitImages,
				zoomLevel,
				
				curr = -1,
				loadCounter = 0,
				loadTimer,
				connectionSpeed = 0,
				direction = 0,
				imgPadding,
				lbPadding = settings['fitPadding'], 
				
				// Elements
				lightbox,
				lb_overlay,
				lb_activity,
				
				// Controls
				lb_btn_left,
				lb_close,
				lb_zoom,
				lb_zoomin,
				lb_zoomout,
				lb_zoom_slider,
				lb_fullscr,
				lb_exitfullscr,
				lb_play_pause,
				lb_progress,
				lb_timing_value,
				lb_btn_right,
				
				// Caption
				lb_caption,
				
				ns = self.data('llb_ns'),
				extraSizes = settings.hasOwnProperty('extraSizes')? settings.extraSizes.split(/,\s*/) : false,
				
				backgroundAudioRoot = $('[data-audioplayer]'),
				backgroundAudioOn = backgroundAudioRoot.length && !backgroundAudioRoot.data('paused'),
	
				cleanup = function(ns) {	
						// cleaning up the old event handlers
						$window.add(document).off('.' + ns);
						// Removing the lightbox :: you can only have one lightbox per thumb set
						$('#' + ns).remove();
					},
				
				closeModal = function(id) {
					
						lightbox.find('.modal').trigger('destroy');
					},
				
				getControlsHeight = function() {
						
						return controlsHidden()? 0 : Math.ceil(controls.outerHeight());
					},
	
				getCaptionHeight = function() {
					
						if (lightbox.hasClass(settings.captionVisibleClass)
							&& caption && caption.length) {
							return caption.outerHeight();
						}
						
						return 0;
					},
				
				// Fixing the zoom button after pinch
				fixZoom = function(zoom, coords) {
					
						if (settings.useZoom && settings.zoomSlider) {
							if (zoom === 1) {
								return;
							}
							
							var mnr = main.data('minRatio') || 1,
								mxr = main.data('maxRatio') || 1.4,
								z = mnr + zoomLevel * (mxr - mnr) / 100; 
							
							z = Math.minMax(mnr, z * zoom, mxr);
							zoomLevel = (z - mnr) / (mxr - mnr) * 100;
						
							setZoomSlider(zoom);
							setZoomBtn(zoomLevel < 50, zoomLevel >= 50);
						}
					},
				
				// Toggling zoom in simple zomm toggling
				toggleZoom = function(coords) {
					
						if (settings.useZoom && settings.zoomSlider) {
							fitImages = true;
							zoomLevel = (zoomLevel > 50)? 0 : 100;
							setZoomSlider(zoomLevel);
						} else {
							// Change fit strategy
							fitImages = !fitImages;
						}
						//savePrefs();
						if (typeof coords === UNDEF) {
							fitImage(true);
						} else {
							fitImage(true, coords);
						}
					},
				
				// Sets zoom (0 .. 100), where 100 means 1:1 size
				setZoom = function(zoom, coords) {
						//console.log('Set zoom:' + zoom);
						zoomLevel = zoom;
						setZoomSlider(zoom);
						
						if (typeof coords === UNDEF) {
							fitImage(true);
						} else {
							fitImage(true, coords);
						}
					},
					
				increaseZoom = function(incr, coords) {
						var v = settings.zoomSlider? getZoomSlider() : zoomLevel;
						
						if (v < 100) {
							v = Math.min(100, v + ((typeof incr !== UNDEF)? incr : 25));
						}
						
						if (typeof coords === UNDEF) {
							setZoom(v);
						} else {
							setZoom(v, coords);
						}
					},
						
				decreaseZoom = function(incr, coords) {
						var v = settings.zoomSlider? getZoomSlider() : zoomLevel;
						
						if (v > 0) {
							v = Math.max(0, v - ((typeof incr !== UNDEF)? incr : 25));
						}
						
						if (typeof coords === UNDEF) {
							setZoom(v);
						} else {
							setZoom(v, coords);
						}
					},
						
				getZoomSlider = function() {
						return parseInt(lb_zoom_slider.val());
					},
						
				setZoomSlider = function(zoom) {
						lb_zoom_slider.val(zoom);
					},
					
				setZoomBtn = function(canZoomIn, canZoomOut) {
					
						if (settings.zoomSlider) {
							if (canZoomIn || canZoomOut) {
								lightbox.addClass('show-zoom-level');
								if (zoomLevel === 0) {
									lb_zoom.removeClass('icon-zoom-out icon-zoom-level').addClass('icon-zoom-in');
								} else if (zoomLevel === 100) {
									lb_zoom.removeClass('icon-zoom-in icon-zoom-level').addClass('icon-zoom-out');
								} else {
									lb_zoom.removeClass('icon-zoom-in icon-zoom-out').addClass('icon-zoom-level');
								}
						} else {
								lightbox.removeClass('show-zoom-level');
							}
						} else {
							lightbox.toggleClass('show-zoom-out', canZoomOut);
							// Zoom out takes precedence if both possible
							lightbox.toggleClass('show-zoom-in', canZoomIn && !canZoomOut);
						}
						
					},
					
				// Slideshow timing controls
				
				setTiming = function(t) {
						lb_timing_value.html((t >= 1000)? (Math.round(t / 1000) + 's') : (t + 'ms'));
					},
					
				increaseTiming = function() {
					
						if (settings.slideshowDelay > 1000) {
							settings.slideshowDelay += 1000;
						} else {
							settings.slideshowDelay += 100;
						}
						
						setTiming(settings.slideshowDelay);
					},
						
				decreaseTiming = function() {
					
						if (settings.slideshowDelay > 500) {
							if (settings.slideshowDelay > 1000) {
								settings.slideshowDelay -= 1000;
							} else {
								settings.slideshowDelay -= 100;
							}
						}
						
						setTiming(settings.slideshowDelay);
					},
					
				// Animating progress bar
				
				animateProgress = function(duration) {
						var d = 20 / duration,
						
							setProgress = function(to) {
								
									//progress.css('transform', 'rotate(' + to * 360 + 'deg)');
									lb_progress.css('width', to * 100 + '%');

									if (to < 1) {
										to = Math.min(1, to + d);
										progressTo = setTimeout(setProgress, 20, to); 
									}
								};
							
						clearTimeout(progressTo);
						progressTo = setTimeout(setProgress, 20, 0);
					},
						
				showProgress = function() {
						if (lb_progress) {
							lb_progress.show();
							animateProgress(slideshowDelay);
						}
					},
				
				hideProgress = function() {
						if (lb_progress) {
							clearTimeout(progressTo);
							lb_progress.hide();
						}
					},
					
				getLightboxDims = function() {
						var lbDims = [ lightbox.width(), lightbox.height() ],
							controlsHeight = settings.fitBetween? getControlsHeight() : 0,
							captionHeight = settings.fitBetween? getCaptionHeight() : 0;
							
						return [ lbDims[0] - 2 * lbPadding - 2 * imgPadding, lbDims[1] - Math.max(controlsHeight, lbPadding) - Math.max(captionHeight, lbPadding) - 2 * imgPadding ];
					},
					
				// Reacalculating image position and size upon window resize
				
				fitImage = function(animate, center, doneFn) {
					
						if (!main || !main.length) {
							return;
						}
						
						var o = thumbs.eq(curr).data(J.OBJ),
							lbDims = [ lightbox.width(), lightbox.height() ],
							imgDims = album.getDimensions(o),
							maxDims = album.getMaxDimensions(o),
							controlsHeight,
							captionHeight,
							
							checkPano = function(dims) {
									return settings.autoPano && (dims[0] / dims[1] > settings.horzPanoThreshold || dims[0] / dims[1] < settings.vertPanoThreshold);
								},
								
							checkPanoDims = function(dims) {
									return 	dims[0] / lbDims[0] >= settings.panoramaAutoDetect	 			// hangs over the edges least 1.3333 times the width or height
											||
											dims[1] / lbDims[1] >= settings.panoramaAutoDetect;
								},
								
							getTargetDims = function(dims, f) {
									return [ dims[0] * f, dims[1] * f ];
								},
								
							fitPano = function(dims) {
									return Math.max(
											(lbDims[0] - 2 * lbPadding - 2 * imgPadding) / dims[0],
											(lbDims[1] - Math.max(controlsHeight, lbPadding) - Math.max(captionHeight, lbPadding) - 2 * imgPadding) / dims[1]
										);
								},
								
							fitToLightbox = function(dims) {
									return Math.min(	
											(lbDims[0] - 2 * lbPadding - 2 * imgPadding) / dims[0], 
											(lbDims[1] - Math.max(controlsHeight, lbPadding) - Math.max(captionHeight, lbPadding) - 2 * imgPadding) / dims[1]
										);
								};
							
						// Picture padding: border
						if (typeof imgPadding === UNDEF) {
							imgPadding = parseInt(main.css('paddingTop'));
						}
											
						// Lightbox padding
						if (typeof lbPadding === UNDEF) {
							if (settings.hasOwnProperty('fitRatio')) {
								// Image margin defined by "fitRatio" - Photoblogger
								lbPadding = Math.round((1 - settings.fitRatio) * Math.min(lbDims[0], lbDims[1]) / 2);
							} else {
								lbPadding = 0;
							}
						}
						
						controlsHeight = settings.fitBetween? getControlsHeight() : 0;
						captionHeight = settings.fitBetween? getCaptionHeight() : 0;
						
						// Set zoom level if undef
						if (typeof zoomLevel === UNDEF || zoomLevel === null) {
							zoomLevel = fitImages? 0 : 100;
							if (settings.useZoom && settings.zoomSlider) {
								setZoomSlider(zoomLevel);
							}
						}
						
						
						if (isVr || isOther || isExternal || isAudio && image.hasClass('default')) {
							// Non-zoomable types
							var th 	= Math.max(controlsHeight, lbPadding),
								bh 	= Math.max(captionHeight, lbPadding),
								w	= lbDims[0] - 2 * lbPadding,
								h 	= lbDims[1] - (th + bh);
								
							if (isExternal || isAudio) {
								// Might be smaller than viewport
								if (imgDims[0] < w && imgDims[1] < h) {
									// Smaller than available space?
									w = Math.min(imgDims[0], w) + 2 * imgPadding;
									h = Math.min(imgDims[1], h) + 2 * imgPadding;
								}
							}
								
							main.data('fitted', true)
								.css({
									display:		'block',
									width:			w,
									height:			h,
									transition:		'none',
									transform:		'matrix(1,0,0,1,' + Math.round((0 - w) / 2) + ',' + Math.round((th - bh - h) / 2) + ')',
									opacity:		1
								});
								
							if (settings.useZoom) {
								setZoomBtn(false, false);
							}
							
							// Done Function
							if (typeof doneFn === FUNCTION) {
								doneFn.call(null);
							}
							
						} else {
							// Zoomable types
							var minRatio,
								maxRatio,
								targetRatio,
								targetDims,
								matrix = main.getMatrix(),
								optSrc,
								isPano,
								fitted = false;
								
							// Stop pano move
							if (lightbox.data('panomove')) {
								main.trigger('autopanoStop');
							}
							
							// Minimal size	
							minRatio = Math.min(1, fitToLightbox(imgDims));
							
							// Maximum size
							if (isVideo || isAudio) {
								// Audio or Video: constrain to lightbox
								maxRatio =	Math.min(maxZoom, fitToLightbox(imgDims));
							} else if (maxDims) {
								// Has larger variants
								maxRatio = Math.max(maxZoom, 
											Math.min(maxDims[0] / imgDims[0], maxDims[1] / imgDims[1]));
							} else {
								// Normal image
								maxRatio = Math.min(maxZoom, fitToLightbox(imgDims));
							}
							
							if (minRatio > maxRatio) {
								// Swap if reversed
								var r = minRatio;
								minRatio = maxRatio;
								maxRatio = r;
							}
							
							// Target size (fit image taken into account)
							if (settings.useZoom && settings.zoomSlider) {
								// Determining initial size for zoom slider
								if (main.data('inited')) {
									// Already painted
									targetRatio = minRatio + zoomLevel * (maxRatio - minRatio) / 100;
									targetDims = getTargetDims(imgDims, targetRatio);
									fitted = targetRatio == minRatio;
									isPano = !!lightbox.data('hasAutoPano');
								} else {
									// First paint
									if (isPanorama || checkPano(imgDims)) {
										// Panorama: fit to height
										targetRatio = Math.min(maxRatio, fitPano(imgDims));
										targetDims = getTargetDims(imgDims, targetRatio);
										isPano = checkPanoDims(targetDims);
										fitted = true;
									} else if (settings.fitImages) {
										// Fit to screen
										targetRatio = Math.min(maxRatio, fitToLightbox(imgDims));
										targetDims = getTargetDims(imgDims, targetRatio);
										isPano = false;
										fitted = true;
									} else {
										// Show in 1:1 size
										targetRatio = 1;
										targetDims = imgDims;
										isPano = false;
										fitted = targetRatio == minRatio;
									}
									zoomLevel = 100 * Math.minMax(0, (targetRatio - minRatio) / (maxRatio - minRatio), 1);
									setZoomSlider(zoomLevel);
								}
							} else {
								// Fit toggle
								if (isPanorama || checkPano(imgDims)) {
									targetRatio = Math.min(maxZoom, fitImages? fitPano(imgDims) : fitToLightbox(imgDims));
									targetDims = getTargetDims(imgDims, targetRatio);
									isPano = checkPanoDims(targetDims);
									fitted = fitImages;
								} else {
									if (!main.data('inited')) {
										fitImages = settings.fitImages;
										// Initial: fit if larger, 1:1 if smaller
										targetRatio = fitImages? Math.min(maxZoom, fitToLightbox(imgDims)) : 1;
									} else {
										if (fitImages) {
											// Zoom out
											targetRatio = Math.min(1, fitToLightbox(imgDims));
										} else {
											// Zoom in
											targetRatio = Math.max(maxZoom, maxRatio);
										}
									}
									fitted = targetRatio == minRatio;
									targetDims = getTargetDims(imgDims, targetRatio);
								}
							}
							
							// Checking if high res image needed
							if (	isImage && !fitted	&&															// It's an image
									((maxDims[0] > imgDims[0]) || (maxDims[1] > imgDims[1])) && 					// Has higher resolution version
									((targetDims[0] > imgDims[0]) || (targetDims[1] > imgDims[1])) && 				// The target size is larger than the base size
									(!main.data('lastDims') || main.data('lastDims')[0] < targetDims[0])			// The higher resolution version hasn't yet been loaded
								) {
								// Saving optimized image source 
								main.data('optSource', album.getOptimalImagePath(o, targetDims, settings.displayOriginals));
								main.data('lastDims', targetDims);
							}
								
							// Adding padding (border)
							targetDims = [  targetDims[0] + 2 * imgPadding, targetDims[1] + 2 * imgPadding ];
																					
							// Reset scale
							matrix[0] = matrix[3] = 1;
							// Setting position
							if (!main.data('inited') || (!isPano && fitted)) {
								// First paint or fit
								main.data('inited', true);
								matrix[4] = 0 - targetDims[0] / 2;
								matrix[5] = 0 - targetDims[1] / 2  + 
										(fitted?
											(Math.max(controlsHeight, lbPadding) - Math.max(captionHeight, lbPadding)) / 2
											:
											0
										)
							} else {
								// Already painted
								var z = (targetDims[0] - 2 * imgPadding) / main.width(),
									h = main.height();
									
								if (typeof center !== UNDEF && center) {
									// Center around event coords
									var	cx = center.x - lbDims[0] / 2,
										cy = center.y - lbDims[1] / 2; 
									matrix[4] = cx - (cx - matrix[4]) * z;
									matrix[5] = cy - (cy - matrix[5]) * z;
								} else {
									// Use previous center
									matrix[4] *= z;
									matrix[5] *= z;
								}
							}
							
							// Prepare panoramas
							if (isImage) {
								if (isPano) {
									// Pamorama
									if (!main.data('hasAutoPano')) {
										if (fitted && (targetDims[0] > (lbDims[0] + 10) || targetDims[1] > (lbDims[1] + 10))) {
											// Applying panorama player
											if (targetDims[0] / targetDims[1] > lbDims[0] / lbDims[1]) {
												// Horizontal
												matrix[4] -= settings.autoPanoStart * (targetDims[0] - lbDims[0]) / 2;
											} else {
												// Vertical
												matrix[5] -= settings.autoPanoStart * (targetDims[1] - lbDims[1]) / 2;
											}
										}
									}
								} /*else if (targetDims[0] <= lbDims[0] && targetDims[1] <= lbDims[1]) {
									// Fits screen => center it
									matrix[4] = 0 - targetDims[0] / 2;
									matrix[5] = 0 - targetDims[1] / 2;
								}*/
							}
							
							// Functions to execute on transition end or immediately if no animation
							var fto = null,
								finish = function() {
										// Replacing image source if not optimal
										if (main.data('optSource')) {
											var img = main.children('img').eq(0);
											if (img.length) {
												img.attr('src', main.data('optSource'));
												console.log('src replaced: ' + img.attr('src'));
												main.removeData('optSource');
											}
										}
										// Adjust or attach pano player
										if (isImage && isPano && (typeof center === UNDEF || !center)) {
											
											if (main.data('hasAutoPano')) {
												main.trigger('autopanoAdjust');
											} else {
												//console.log('Applying autopano');
												main.autopano({ 
													direction: 	-settings.autoPanoStart || direction,
													speed:		settings.autoPanoSpeed
												});
											}
										}
									
										if (isVideo && settings.videoAuto && media[0].paused) { 
											// Play video only after the transition
											playVideo(media[0]);
										}
									},
								
								endFitImage = function() {
										clearTimeout(fto);
										// Done Function
										if (typeof doneFn === FUNCTION) {
											doneFn(finish);
										} else {
											finish();
										}
									};
						
							main.css({
									display:		'block',
									transition:		animate? ('transform ' + settings.transitionProps + ', width ' + settings.transitionProps + ', height ' + settings.transitionProps) : 'none'
								});

							// Apply transforms
							
							if (animate) {
								//console.log('Scheduling finish()');
								main.one('transitionend', endFitImage);
								fto = setTimeout(endFitImage, 320);
							}
							
							main.data({
									fitted: 		fitted,
									minRatio:		minRatio,
									maxRatio:		maxRatio
								})
								.css({
									transform:		'matrix(' + matrix.join(',') + ')',		
									width: 			targetDims[0],
									height: 		targetDims[1]
								});
								
							// Set zoom button
							if (settings.useZoom) {
								if (maxRatio > targetRatio) {
									setZoomBtn(true, false);
								} else if (minRatio < targetRatio) {
									setZoomBtn(false, true);
								} else {
									setZoomBtn(false, false);
								}
							}
							
							if (!animate) {
								//console.log('Calling finish()');
								fto = setTimeout(endFitImage, 20);
							}
						}
						
					},
									
				// Reading image type
				
				getImageType = function(item) {
						
						isVideo = isAudio = isOther = isImage = isVr = isPdf = isPanorama = false;
						
						isExternal = item.hasOwnProperty(J.EXTERNAL);
						
						isVr = !isExternal && settings.use360Player && (
								(item.hasOwnProperty(J.PROJECTIONTYPE) && item[J.PROJECTIONTYPE] === 'equirectangular') 
								||
								(item[J.CATEGORY] === 'video' && item.hasOwnProperty(J.PANORAMA) && item[J.PANORAMA])
							);
						
						if (!isExternal && !isVr) {
							
							switch (item[J.CATEGORY]) {
								
								case 'video':
									isVideo = true;
									break;
								
								case 'audio':
									isAudio = true;
									break;
								
								case 'other':
									isOther = true;
									isPdf = album.getExtension(item).toLowerCase() === 'pdf';
									break;
								
								default:
									isImage = true;
									isPanorama =
										(item.hasOwnProperty(J.PANORAMA) && item[J.PANORAMA]) 
										|| 
										(	settings.autoPano 
											&& 
											(	item[J.IMAGE][J.WIDTH] > lightbox.width()
												||
												item[J.IMAGE][J.HEIGHT] > lightbox.height()
											)
											&&
											(	(item[J.IMAGE][J.WIDTH] / item[J.IMAGE][J.HEIGHT] > settings.horzPanoThreshold) 
												|| 
												(item[J.IMAGE][J.WIDTH] / item[J.IMAGE][J.HEIGHT] < settings.vertPanoThreshold)
											)
										);
							}
						}
					},
				
				// Gets the custom slideshow interval (if exist)
				
				getSlideshowTiming = function(item) {
					
						if (item.hasOwnProperty(J.SLIDESHOWINTERVAL)) {
							slideshowDelay = item[J.SLIDESHOWINTERVAL];
						} else {
							slideshowDelay = settings.slideshowDelay;
						}
						if (item.hasOwnProperty(J.TRANSITIONSPEED)) {
							transitionSpeed = item[J.TRANSITIONSPEED];
						} else {
							transitionSpeed = settings.transitionSpeed;
						}
					},
					
				// Updates rating (if changed externally)
					
				updateRating = function(r) {
						
						if (curr >= 0) {
							if (typeof r === UNDEF) {
								r = thumbs.eq(curr).data(J.OBJ)[J.VISITORRATING] || 0;
							}
						
							lightbox.find('.' + settings.lightboxCaptionClass + ' .ratingbox span')
								.removeClass('r0 r1 r2 r3 r4 r5')
								.addClass('r' + r);
						}
					},
											
				// Paint regions
				
				paintRegions = function(regions, visible) {
						var el = $('<div>', {
									'class':	settings.regionsClass
								}).appendTo(main);
						
						for (var i = 0, r; i < regions.length; i++) {
							r = regions[i].split(';');
							el.append($('<div>').css({
									left:		(100 * parseFloat(r[1])) + '%',
									top:		(100 * parseFloat(r[2])) + '%',
									width:		(100 * parseFloat(r[3])) + '%',
									height:		(100 * parseFloat(r[4])) + '%'
								}).append('<span>' + r[0] + '</span>'));
						}
						
						el.toggle(typeof visible !== UNDEF && visible);
					},
					
				// Show/hide regions
				
				toggleRegions = function() {
						var el = lightbox.find('.' + settings.regionsClass);
					
						if (el.length) {
							settings.regionsVisible = !settings.regionsVisible;
							if (el.is(':visible')) {
								el.fadeOut();
							} else {
								el.fadeIn();
							}
						}
						
						return false;
					},
											
				// Creating caption and buttons
				
				prepareCaption = function(item) {
						
						caption = $('<div>', {
										'class': 	settings.lightboxCaptionClass + (slideshow? ' slideshow' : '')
									});
						
						var buttons = $('<div>', {
									'class': 	'buttons'
								});
						
						if (settings.useRating) {
							
							var r = item[J.RATING] || 0;
							
							if (settings.visitorRating || r) {
								var s;
							
								el = $('<div>', {
										'class':	'ratingbox'
									}).appendTo(buttons);
									
								el.append($('<label>', {
										html:		text.rating
									}));
									
								s = $('<span>', {
										'class':	'rating',
										html:		STARS
									})
									.appendTo(el);
									
								if (settings.visitorRating) {
									
									if (item.hasOwnProperty(J.VISITORRATING)) {
										// Already exists
										r = item[J.VISITORRATING];
									} else {
										if (!settings.useJalbumRating) {
											// Ignore jAlbum rating
											r = 0;
										}
										// Save the default
										item[J.VISITORRATING] = r;
									}
											
									s.addClass('r' + r + (settings.useJalbumRating? '' : ' icon-close'))
										.on('click.' + ns, function(e) {
												var el = $(this),
													r = Math.minMax(0, Math.floor((getCoords(e).x - el.offset().left - parseFloat(el.css('paddingLeft'))) / 16), 5); 
													//Math.minMax(0, Math.floor((getCoords(e).x - el.offset().left + 4) / el.width()) * 5, 5);
													
												el.removeClass('r0 r1 r2 r3 r4 r5').addClass('r' + r);
												item[J.VISITORRATING] = r;
												
												// Propagate to thumbnails
												self.trigger('refreshRating', thumbs[curr]);
													
												return false;
											});
								} else {
									// Unalterable rating from jAlbum
									s.addClass('r' + r);
								}
							}
						}
						
						// Extra sizes
						
						if (!isExternal && extraSizes && item[J.CATEGORY] === 'image') {
							var p = album.getImagePath(item);
							
							el = ($('<div>', {
										'class': 	'download icon-download'
									})).appendTo(buttons);
							
							for (var i = 0; i < extraSizes.length; i++) {
								el.append($('<a>', {
										text: 		extraSizes[i],
										href:  		p.replace('slides/', 'dl/' + extraSizes[i] + '/'),
										download: 	''
									}));
							}
						}
						
						// Print button
						
						if (settings.printImage && item[J.CATEGORY] === 'image' && !settings.rightClickProtect) {
							controls.find('.print-btn').removeClass('hidden').off('.' + ns).on('click.' + ns, function(e) {
									printImage((item[J.ORIGINAL] && !settings.hideDownload)? item[J.ORIGINAL][J.PATH] : item[J.IMAGE][J.PATH],
										item[J.TITLE] || '',
										item[J.THUMBCAPTION] || ''
									);
								});
						} else {
							controls.find('.print-btn').addClass('hidden');
						}
						
						// Download button
						
						if (!isExternal && settings.showDownload) {
							var link;
							
							if ((isImage || isPdf || settings.allowDownloadOthers) &&
								((link = album.getOriginalPath(item)) ||
								(settings.allowDownloadScaled && (link = album.getImagePath(item))))) {
								controls.find('.download-btn').removeClass('hidden').attr('href', link);
							} else {
								controls.find('.download-btn').addClass('hidden').removeAttr('href');
							}
						}
											
						// Fotomoto button
						
						if ((!LOCAL || DEBUG) && settings.hasOwnProperty('fotomoto') && !album.hideFotomoto(item)) {
							controls.find('.fotomoto-btn').off('.' + ns).on('click.' + ns, function(e) {
									if (typeof FOTOMOTO !== UNDEF) {
										FOTOMOTO.API.showWindow(10, 
												album.getOriginalPath(item) || album.getItemPath(item)
											);
									}
								});
						}
						
						// Photo data
						
						if (item[J.PHOTODATA]) {
							
							controls.find('.photodata-btn').removeClass('hidden');
							panels.find('.photodata').html(item[J.PHOTODATA]);
						
						} else {
							
							controls.find('.photodata-btn').addClass('hidden');
							panels.find('.photodata').empty();
						}
						
						// Regions
						
						if (settings.showRegions) {
							
							if (item[J.REGIONS]) {
								
								controls.find('.regions-btn').removeClass('hidden').off('.' + ns).on('click.' + ns, toggleRegions);
								paintRegions(JSON.parse(item[J.REGIONS]), settings.regionsVisible); 
								
							} else {
								
								controls.find('.regions-btn').addClass('hidden')
							}
						}
						
						// Map
						/*
						if (settings.showMap) {
							
							if (album.hasLocation(item)) {
								setTimeout(function() {
										self.find('[data-map-root]').trigger('setActive', item[J.NAME]);
									}, 100);
							} else {
								self.find('[data-map-root]').trigger('resetMarkers');
							}
						}
						*/
						// Sound clip
						
						var btn = controls.find('.audioclip-btn');
						
						if (!isAudio && item[J.AUDIOCLIP] && btn.length) {
							
							btn.removeClass('hidden')
								.off('.' + ns)
								.on('click.' + ns, function() {
										if (btn.hasClass('icon-pause')) {
											pauseAudioClip();
										} else {
											startAudioClip()
										}
										
										return true;
									});
							
							el = panels.find('.audioclip-cont');
							
							if (el.length) {
								
								el.empty().addClass('paused');
								
								// Audio element
								audioClip = $('<audio>', {
											'class':		'audioclip',
											src:			album.getAudioClipPath(item),
											controls:		'',
											controlsList:	'nofullscreen nodownload noremote noplaybackrate'
										}).appendTo(el);
								
								// Type attribute saving and duration
								if (item[J.AUDIOCLIP].hasOwnProperty(J.TYPE)) {
									audioClip[0]['type'] = item[J.AUDIOCLIP][J.TYPE];
									if (item[J.AUDIOCLIP].hasOwnProperty(J.DURATION)) {
										audioClip.data('duration', item[J.AUDIOCLIP][J.DURATION]);
									}
								}
								
								audioClip.on('volumechange', function(e) {
										
										if (typeof this.volume !== UNDEF) {
											
											settings.volume = this.volume;
											savePrefs(settings, settings.prefsKept, 'lb_pref');
											
											if (typeof settings['onVolumeChange'] === FUNCTION) {
												settings.onVolumeChange(settings.volume);
											}
										}
									});
								
								// Setting loudness
								audioClip[0]['volume'] = settings.volume;
								
								if (settings.autoplayAudioClips) {
									// Trying to autoplay
									startAudioClip();
								}
							}
							
						} else {
							
							btn.addClass('hidden').off('.' + ns);
							
							panels.find('.audioclip-cont').empty();
							audioClip = $();
						}
						
						// Mostphotos
						
						if (item[J.IMAGECAPTION]) {
							caption.append($('<div>', {
									'class': 	'caption scrollable',
									html: 		item[J.IMAGECAPTION]
								}));
							caption.find('[data-tooltip]').addTooltip();
						}
						
						if (!buttons.is(':empty')) {
							caption.addClass(settings.hasbuttonsClass);
							caption.append(buttons);
							buttons.children('a').not('[download]').addTooltip();
						}
						
						if (caption.is(':empty')) {
							caption.remove();
							//lightbox.removeClass(settings.captionVisibleClass);
							lightbox.addClass('no-caption');
						} else {
							//caption.toggle(!slideshow || caption.children('.caption').length > 0);		// ????
							caption.find('[data-tooltip]').addTooltip();
							//lightbox.addClass(settings.captionVisibleClass);
							lightbox.removeClass('no-caption');
							lightbox.append(caption);
						}
							
					},	
				
				// Preloading the next image (direction sensitive)
				
				preloadNext = function() {
						var nextItem;
						
						if (direction < 0 && curr > 0) {
							nextItem = thumbs.eq(curr - 1).data(J.OBJ);
						} else if (curr < thumbs.length - 2) {
							nextItem = thumbs.eq(curr + 1).data(J.OBJ);
						}
						
						if (nextItem) {
							if (nextItem[J.CATEGORY] === 'image') {
								var next = new Image();
								next.src = album.getOptimalImagePath(nextItem, getLightboxDims(), settings.displayOriginals);
							}
						}
					},
				
				// Image clicked :: toggle controls or audio/video controls
				
				imageClicked = function(e) {
						
						if ($(e.target).closest('.lightbox-caption').length) {
							return true;
						}
						
						var touched = $('html').data('whatinput') === 'touch' || e.type === 'touchend';
						
						if (isVideo || isAudio) {
							
							var m = media[0];
								
							if (m.paused) {
								m.play();
								if (VEND !== 'ms') {
									m.controls = false;
								}
								if (touched) {
									hideControls();
									if (lightbox.hasClass(settings.captionVisibleClass)) {
										hideCaption();
									}
								}
							} else {
								m.pause();
								if (VEND !== 'ms') {
									m.controls = true;
								}
								if (touched) {
									if (settings.clickAction === 'togglecontrols') {
										showControls();
										if (!lightbox.hasClass(settings.captionVisibleClass)) {
											showCaption();
										}
									}
								}
							}
							return false;
							
						} else if (isImage) {
							
							if (settings.clickAction === 'nextimage') {
								// image navigation
								if ((getCoords(e).x - main.position().left) > (main.width() / 2)) {
									nextImage();
								} else {
									previousImage();
								}
								if (touched) {
									showControls();
								}
							} else if (settings.clickAction === 'togglecontrols') {
								toggleControls();
							} else if (settings.clickAction === 'toggleall') {
								toggleAll();
							}
						}
					},
					
				removeOld = function() {
						if (oldMain.length) {
							oldMain.remove();
							oldMain = $();
						}
					},
					
				// Doing all the animations in one timeframe
				
				doTransitions = function(doneFn) {
					// The image is loaded
						var	animateOld = function() {
									var omx = oldMain.getMatrix();
											
									omx[4] -= 100 * direction;
									
									oldMain.one('transitionend', removeOld)
										.css({
												transition:		'transform ' + (transitionSpeed / 2) + 'ms ease-out, opacity ' + (transitionSpeed / 2) + 'ms ease-out',
												transform:		'matrix(' + omx.join(',') + ')',
												opacity:		0
											});
										
									// Ensuring the old image is always removed
									setTimeout(removeOld, transitionSpeed / 2 + 50);
									setTimeout(animateNew, 50);
								},
								
							animateNew = function() {
									var mx = main.getMatrix(),
										ox = mx[4];
									
									if (direction) {
										// Offset by 100 pixels
										mx[4] += 100 * direction;
									
										main.css({
												display:		main.hasClass('iframe')? 'flex' : 'block',
												transition:		'none'
											}).css({
												opacity: 		0,
												transform:		'matrix(' +  mx.join(',') + ')'
											});
									} else {
										main.css({
												display:		main.hasClass('iframe')? 'flex' : 'block',
												transition:		'none'
											}).css({
												opacity: 		0
											});
									}
									
									setTimeout(function() {
											window.requestAnimationFrame(function() {
													main.one('transitionend', function() {
															afterTransition(doneFn);
														});
													
													if (direction) {
														// Restore final pos
														mx[4] = ox;
														// Animate to the final position and opacity
														main.css({
																transition:		'transform ' + transitionSpeed + 'ms ease-out, opacity ' + transitionSpeed + 'ms ease-out'
															}).css({
																transform:		'matrix(' + mx.join(',') + ')',
																opacity:		1
															});
													} else {
														main.css({
																transition:		'transform ' + transitionSpeed + 'ms ease-out, opacity ' + transitionSpeed + 'ms ease-out'
															}).css({
																opacity:		1
															});
													}
												});
										}, TOUCHENABLED? 350 : 50);
								};
							
						if (oldMain.length && oldMain.data('swipe') !== 'swipe') {
							animateOld();
						} else {
							animateNew();
						}
						
					},
					
				// Transition has completed
					
				afterTransition = function(doneFn) {
						
						if (typeof doneFn === FUNCTION) {
							doneFn.call();
						}
						
						if (typeof settings.onLoadEnd === FUNCTION) {
							settings.onLoadEnd(thumb);
						}
						
						if (settings.preloadNext) {
							preloadNext();
						}
					},
					
				// The image is loaded
				
				imageReady = function() {
					
						loadCounter--;
						
						if (loadCounter > 0) {
							// Still has to load another image: (poster image) 
							return;
						}
										
						lb_activity.hide();
						
						// Reporting image load in Debug mode
						if (DEBUG) {
							var d = new Date();
							if (isExternal || isVr) {
								console.log((isExternal? 'External content' : '360 player') + ' loaded: ' + (d - loadTimer) + 'ms');
							} else if (image.length) {
								console.log(((isVideo || isAudio)? 'Media' : 'Image') + ' [' + curr + '] loaded: ' + (d - loadTimer) + 'ms src="' + image[0].src + '"' + (connectionSpeed? (' avg. speed: ' + (connectionSpeed / 1000).toFixed(2) + 'MB/s') : ''));
								if (typeof image[0] === UNDEF || typeof media[0] === UNDEF) {
									console.log('Premature ' + (typeof image[0] === UNDEF)? ('loadImage.done(' + image[0].src + ')') : ('loadMedia.done('+ media[0].src + ')'));
								}
							}
						}
						
						// Right click protect
						if ((isImage || isVideo || isAudio) && settings.rightClickProtect) {
							media.on('contextmenu', function(e) {
									e.preventDefault()
									return false;
								});
						}
													
						// Audio / video initilazion
						if (isVideo || isAudio) {
							
							media.attr({
									autoplay: 	settings.videoAuto,
									loop:		settings.videoLoop
								});
							media[0].volume = settings.volume;
							
							media.on('volumechange.' + ns, function() {
									settings.volume = this.volume;
								});
						}
						
						if (oldCaption.length) {
							
							// Removing the old caption
							oldCaption.find('.buttons a').trigger('removeTooltip');
							
							if (oldCaption.length > 1) {
								oldCaption.eq(-1).prevAll('.' + settings.lightboxCaptionClass).remove();
								oldCaption = lightbox.find('.' + settings.lightboxCaptionClass);
							}
							
							if (lightbox.hasClass(settings.captionVisibleClass)) {
								oldCaption.one('transitionend', function() {
										oldCaption.remove();
										oldCaption = $();
									}).css({
										transition:		'opacity ' + (transitionSpeed / 2) + 'ms ease-out',
										opacity:		0
									});
							} else {
								oldCaption.remove();
							}
						}
						
						if (!isVr && !isPdf && !isVideo && (!isExternal || !main.children('iframe').length)) {
							// Swipe handler
							main.swipe({
									onSwipeStart:	function() {
															$(this).trigger('autopanoStop');
															$(this).hideAllTooltips();
														},
									
									onSwipedLeft: 	function() {
															main.data('swipe', 'swipe');
															nextImage();
														},
													
									onSwipedRight: 	function() {
															main.data('swipe', 'swipe');
															previousImage();
														},
														
									onSwipedDown:	function() {
															main.data('swipe', 'down');
															quitLightbox();
														},
														
									onPinchZoom:	function(e) {
															main.data('swipe', 'pinch');
															if (lightbox.data('panomove')) {
																e.preventDefault();
																$(this).trigger('autopanoStop');
															}
															fixZoom(main.getScale()[0], getCoords(e));
														},
													
									onFinished: 	function() {
															if (!dontRemove) {
																$(this).trigger('removeSwipe');
																$(this).remove();
															}
														},
													
									onClick:		function(e) {
															if (lightbox.data('panomove')) {
																main.data('swipe', 'panostop');
																e.preventDefault();
																$(this).trigger('autopanoStop');
															} else {
																main.data('swipe', 'click');
																imageClicked(e);
															}
														},
														
									onRightClick:	function(e) {
															stopAuto(true);
														},
														
									onDoubleClick:	(settings.useZoom && settings.doubleClickZoom)? 
														function(e) {
																var c = getCoords(e);
																main.data('swipe', 'doubleclick');
																if (settings.zoomSlider) {
																	if (zoomLevel < 100) {
																		increaseZoom(50, c);
																	} else {
																		setZoom(0);
																	}
																} else {
																	toggleZoom(c);
																}
															} 
														: 
														null
								});
						}
							
						// Placing and centering the new image
						fitImage(false, null, function(doneFn) {
								// Running transitions
								doTransitions(doneFn);
							});
						
						// Check if video has started 
						if (isVideo && settings.videoAuto && media[0].paused) {
							playVideo(media[0]);
						}
						
						// Continuing slideshow
						if (slideshow) {
							if ((isAudio || isVideo) && settings.videoAuto) {
								suspendAuto();
								media[0].onended = function() {
									resumeAuto();
								};
							} else {
								clearTimeout(slideshow);
								slideshow = setTimeout(nextImage, slideshowDelay);
								showProgress();
							}
						} else {
							//stopAuto(true);
							if (settings.autohideControls) {
								hideControlsLater();
							}
						}
						
					},
				
				// Recentering after swipe for example
				
				recenter = function(el) {
						var el = el || main;
						if (el.data('swipe') === 'swipe') {
							var matrix = main.getMatrix();
							matrix[4] = -main.width() / 2;
							el.one('transitionend', function() {
									el.data('swipe', null);
								}).css({
									opacity: 		1,
									transition:		'opacity ' + (settings.transitionSpeed * 2) + 'ms ease-out, transform ' + (settings.transitionSpeed * 2) + 'ms ease-out',
									transform: 		'matrix(' + matrix.join(',') + ')'
								});
						}
					},
				
				// Main entry point to load a new image
				
				loadImage = function(n) {
					
						if (typeof n !== 'number') {
							n = thumbs.index(n);
						}
						
						if (n < 0 || n >= thumbs.length) {
							// Out of bounds move
							
							dontRemove = true;
							
							if (n < 0) {
								
								switch (settings.afterLast) {
									case 'donothing':
										n = 0;
										if (main.data('swipe') === 'swipe') {
											recenter();
										}
										break;
										
									case 'startover':
										n = thumbs.length - 1;
										break;
										
									case 'onelevelup': 
										if (settings.level) {
											window.location.href = '../' + settings.indexName;
										}
										n = 0;
										break;
										
									case 'nextfolder':
										if (settings.baseSet && settings.previousFoldersLast) {
											window.location.href = settings.previousFoldersLast;
										} else {
											n = 0;
											if (main.data('swipe') === 'swipe') {
												recenter();
											}
										}
										break;
										
									case 'nextindex':
										if (settings.baseSet && settings.previousFolderPath) {
											window.location.href = settings.previousFolderPath;
										} else {
											n = 0;
											if (main.data('swipe') === 'swipe') {
												recenter();
											}
										}
										break;
										
									default:
										n = 0;
										quitLightbox();
										return;
								}
								
							} else {
								
								switch (settings.afterLast) {
									
									case 'donothing':
										dontRemove = true;
										n = thumbs.length - 1;
										stopAuto();
										if (main.data('swipe') === 'swipe') {
											recenter();
										}
										break;
										
									case 'startover':
										n = 0;
										break;
										
									case 'onelevelup': 
										if (settings.level) {
											window.location.href = '../' + settings.indexName;
										}
										n = thumbs.length - 1;
										stopAuto();
										break;
										
									case 'nextfolder':
										if (settings.baseSet && settings.nextFoldersFirst) {
											window.location.href = settings.nextFoldersFirst + (slideshow? '&slideshow' : '');
										} else {
											n = thumbs.length - 1;
											stopAuto();
											if (main.data('swipe') === 'swipe') {
												recenter();
											}
										}
										break;
										
									case 'nextindex':
										if (settings.baseSet && settings.nextFolderPath) {
											window.location.href = settings.nextFolderPath;
										} else {
											n = thumbs.length - 1;
											stopAuto();
											if (main.data('swipe') === 'swipe') {
												recenter();
											}
										}
										break;
										
									case 'ask':
										
										dontRemove = true;
										
										var sr = stopAuto(),
											buttons = new Array();
										
										if (main.data('swipe') === 'swipe') {
											recenter();
										}
										
										n = thumbs.length - 1;
										
										if (thumbs.length > 1) {
											// Start over
											buttons.push({
													t: 	text.startOver,
													c:	'icon-loop',
													h: 	function(lb) { 
															loadImage(0);
														}
												});
										}
										
										if (settings.level) {
											// Up one level
											buttons.push({
													t: 	settings.level? text.upOneLevel : (settings.homepageLinkText || text.backToHome), 
													c:	'icon-one-level-up',	
													h: 	function() { 
															window.location.href = '../' + settings.indexName;
														}
												});
										}
										
										// Back to thumbnails
										buttons.push({
												t: 	text.backToIndex,
												c:	'icon-thumbnails',
												h: 	function() { 
														quitLightbox();
													}
											});
										
										if (settings.baseSet) {
											
											if (settings.nextFoldersFirst) {
												// Go to next folder
												buttons.push({
														t: 	text.nextFolder,
														c:	'icon-arrow-right',
														h: 	function() {
																window.location.href = settings.nextFoldersFirst + (sr? '&slideshow' : '');
															}
													});
											}
											
											if (settings.nextFolderPath) {
												// Go to next index
												buttons.push({
														t: 	text.nextIndex,
														c:	'icon-thumbnails',
														h: 	function() {
																window.location.href = settings.nextFolderPath;
															}
													});
											}
										}
								
										lightbox.modal($('<p>', {
												'class':	'text-center',
												text: 		text.atLastPageQuestion
											}), buttons, {
												'class': 	'secondary',
												icon:		'info',
												title: 		text.atLastPage
											});
										
										return;
											
									default:
										// Back to index
										n = slideshow? 0 : (thumbs.length - 1);
										stopAuto();
										// Resetting play position to 0
										quitLightbox(thumbs[0]);
										return;
								}
							}
						} else {
							dontRemove = false;
						}
						
						// Direction is calculated if the current thumb exists
						if (lightbox.is(':visible')) {
							// Lightbox is on
							if ((Math.abs(n - curr) >= thumbs.length - 1) && settings.afterLast === 'startover') {
								// Flip over
								direction = 1;
							} else {
								direction = (curr > n)? -1 : ((curr < n)? 1 : 0);
							}
							
							if (curr === n) {
								// the requested image is already on screen :: nothing to do
								return;
							}
							
							controls.hideAllTooltips();
							
							if (settings.autoHideControls) {
								hideControlsLater();
							}
							
						} else {
							// We're on the index page :: show lightbox
							listening = true;
							//$body.css('overflow', 'hidden');
							$body.addClass(settings.lightboxOnClass);
							direction = 0;
							
							if (settings.lightboxFullscreen) {
								requestFullscreen();
							}
							
							if (settings.useRating) {
								updateRating();
							}
							
							lightbox.one('transitionend', function() {
									if (settings.autoStart) {
										startAuto();
									}
								});
							
							lightbox.css({
									display:		'block',
									transition:		'opacity 500ms ease-in-out',
									opacity:		1
								});
							backgroundAudioOn = backgroundAudioRoot.length && !backgroundAudioRoot.data('paused');
							
						}
						
						closeModal();
						
						var item;
						
						curr = n;
						thumb = thumbs.eq(curr);
						item = thumb.data(J.OBJ);
						
						if (!item) {
							console.log('Fatal error: image (' + curr + ') is missing from the database! (Upload data1.json again!)');
							return;
						}
						
						lb_activity.show();
						hideProgress();
						getImageType(item);
						getSlideshowTiming(item);
						
						setTimeout(function() {
							
							oldMain = lightbox.find('.' + settings.lightboxMainClass);
							oldCaption = lightbox.find('.' + settings.lightboxCaptionClass);
								
							if (oldMain.length) {
								
								oldMain.trigger('autopanoRemove').off('.' + ns).removeClass(settings.currClass);
								
								if (settings.muteBackgroundAudio && backgroundAudioOn) {
									if (isAudio || isVideo) {
										backgroundAudioRoot.trigger('fadeOutPlayer');
									} else {
										backgroundAudioRoot.trigger('fadeInPlayer');
									}
								}
								
								fadeMedia(oldMain);
							}
							
							// Delayed image loading
							
							loadCounter = 1;
							loadTimer = new Date();
							
							main = $('<div>', {
									'class': 	'lightbox-main ' + settings.currClass + ' ' + (isExternal? 'external' : (isVr? 'vr' : (isPdf? 'pdf' : item[J.CATEGORY])))
								})
								.css({
									opacity: 	0,
									transform:	'matrix(1,0,0,1,-50%,-50%)'
								})
								.appendTo(lightbox);
								
							if (isVr) {
								
								var w = Math.round(lightbox.width() * settings.fitRatio),
									h = Math.round(lightbox.height() * settings.fitRatio);
									
								main.css({
											width:		w,
											height:		h
										})
									.attr('id', 'vr' + curr)
									.addClass('vr')
									.show()
									.one('DOMReady', imageReady);
								
								// Photosphere viewer by Jeremy Heleine
								if (typeof PhotoSphereViewer !== UNDEF) {
									setTimeout(function() {
											var opt = { container: 	main[0] },
												psv,
												gp;
												
											if (item[J.CATEGORY] === 'video') {
												opt.panorama = album.getVideoPath(item);
												opt.adapter = PhotoSphereViewer.EquirectangularVideoAdapter;
												opt.plugins = [ PhotoSphereViewer.VideoPlugin, {} ];
												opt.navbar = [ 'zoom', 'fullscreen' ];
											} else {
												opt.panorama = album.getAbsoluteImagePath(item);
												opt.navbar = [ 'autorotate', 'zoom', 'fullscreen' ];
												if (settings.autoRotate360 && settings.rotateSpeed360) {
													opt.autorotateDelay = 1000;
												}
												opt.autorotateSpeed = settings.rotateSpeed360 + 'rpm';
											}
											
											if (gp = item['gpano']) {
												if (gp['fov']) {
													opt.defaultZoomLvl = gp.fov;
												}
												if (gp['pan'] || gp['tilt'] || gp['roll']) {
													opt.sphereCorrection = { 
																pan:		Math.PI * gp['pan'] / 180, 
																tilt:		Math.PI * gp['tilt'] / 180, 
																roll:		Math.PI * gp['roll'] / 180
														};
												}
											}
											
											psv = new PhotoSphereViewer.Viewer(opt);
											
										}, transitionSpeed);
								} else {
									console.log('Fatal Error: Missing "photo-sphere-viewer.min.js"!');
								}
								
								// Considered ready after 200ms or the "ready" event, whatever comes first 
								setTimeout(imageReady, 200);
								
							} else if (isImage) {
								
								// Image
								// Creating
								media = image = $('<img>').appendTo(main);
								
								if (DEBUG) {
									console.log('Loading image [' + curr + '] src="' + album.getOptimalImagePath(item, getLightboxDims(), settings.displayOriginals) + '"');
								}
								
								if (image[0].complete && image[0].naturalWidth > 0) {
									// From cache
									imageReady();
										
								} else {
									// Loading
									image.one('load.' + ns, function(e) {
											// Triggered on load event
											imageReady();
										});
								}
								
								// Applying the source
								image[0].src = album.getOptimalImagePath(item, getLightboxDims(), settings.displayOriginals);
										
							} else if (isAudio || isVideo) {
								
								if (isVideo) {
									
									var	dur = album.getVideoDuration(item);
									
									// Video
									media = image = $('<video>', {
												preload: 		'auto',
												controlsList:	'nodownload',
												poster: 		album.getPosterPath(item)
											});
									
									if (dur === null || dur >= 2000) {
										// Normal video
										media[0]['controls'] = 'true';
									} else {
										// Short video < 2s
										//media[0].controls = false;
										media[0]['loop'] = 'true';
									}
									
								} else {
									
									var src = album.getPosterPath(item);
										
									// Audio
									loadCounter = 2;
									
									image = $('<img>', {
											'class': 	'poster' + (src.endsWith('poster.png')? ' default' : '')
										}).one('load', function(e) {
											imageReady();
										})
										.attr('src', src)
										.appendTo(main);
									
									media = $('<audio>', {
											preload: 		'auto',
											controls: 		'true',
											controlsList:	'nofullscreen nodownload'
										});
								}
								
								media.one('loadedmetadata', function(e) {
											imageReady();
										})
									.attr('src', album.getSourcePath(item))
									.appendTo(main);
								
								media.on({
										playing: 	mediaPlaying,
										pause: 		mediaPaused,
										ended:		mediaEnded
									});
										
								if (settings.muteBackgroundAudio && backgroundAudioRoot.length) {
									backgroundAudioOn = !backgroundAudioRoot.data('paused');
								}
								
							} else if (isExternal) {
								
								var cont = item[J.EXTERNAL]['cont'],
									size = item[J.EXTERNAL]['size'];
									
								if (size) {
									size = size.split('x');
									main.data({
										oWidth:		parseInt(size[0]),
										oHeight:	parseInt(size[1])
									})
								}
									
								main.one('DOMReady', function(e) {
										imageReady();
									});
								
								if (cont.match(/^https?\:\/\//i) || cont.match(/\.html?$/i) || cont.match(/^\.\.\//)) {
									main.addClass('iframe')
										.append($('<iframe>', { 
												width: 				'100%',
												height: 			'100%',
												src: 				cont,
												frameborder: 		0,
												allowfullscreen: 	'allowfullscreen'
											}));
								} else {
									if (cont.match(/^<iframe/i)) {
										main.addClass('iframe');
									}
									main.append(cont);
								}
								
								// Considered ready after 200ms or the "ready" event, whatever comes first 
								setTimeout(imageReady, 200);
								
							} else {
								// Other
								var appendIcon = function(target, item) {
														target.append($('<div>', {
																'class':	'icon',
																html:		'<a href="' + album.getSourcePath(item) + '" target="_blank"><img class="other" src="' + album.getImagePath(item) + '"></a><p class="icon-external"> ' + text.clickToOpen + '</p></div>'
															}));
													};
								// Other
								if (isPdf && HASPDFVIEWER && !ISIOSDEVICE) {
									// using the PDF viewer plugin
									var o = $('<object>', {
												type:		'application/pdf'
											})
											.attr('data', album.getSourcePath(item));
									
									appendIcon(o, item);
									main.append(o);
									setTimeout(imageReady, 400);
										
								} else if (settings.useGoogleDocs && !LOCAL && 
									'.docx.xlsx.txt.pdf'.indexOf('.' + (item.hasOwnProperty(J.ORIGINAL)? item[J.ORIGINAL][J.PATH] : item[J.NAME]).getExt()) >= 0) {
									// Fallback to Google Doc Viewer
									main.addClass('iframe');
									var target = $('<iframe>', {
												width: 				'100%',
												height: 			'100%',
												frameborder: 		0,
												allowfullscreen: 	'allowfullscreen'
										}).appendTo(main);
									
									setTimeout(function(target, item) {
											target.attr('src', 'https://docs.google.com/viewer?url=' + window.location.href.getDir() + album.getSourcePath(item) + '&embedded=true');
											setTimeout(imageReady, 400);
										}, 100, target, item);
									
								} else {
									// Generic case with a download button
									appendIcon(main, item);
									imageReady();
								}
							}	
							
							prepareCaption(item);
																	
							if (typeof settings.onLoadStart === FUNCTION) {
								settings.onLoadStart(thumb);
							}
						
						}, 30);
					},
				
				// Hiding the lightbox overlay
				
				quitLightbox = function(thumb) {
						
						stopAuto(true);
						
						self.hideAllTooltips();
						
						listening = false;
						
						$body.removeClass(settings.lightboxOnClass);
						$body.hideAllTooltips();
						//$body.add($('html')).css('overflow', '');
						audioClip.remove();
						
						if (settings.muteBackgroundAudio && backgroundAudioOn) {
							backgroundAudioRoot.trigger('fadeInPlayer');
						}
						
						if (!main.length) {
							
							if(!SMALLSCREEN) {
								exitFullscreen();
							}
							
							if (typeof settings.onClose === FUNCTION) {
								if (typeof thumb !== UNDEF) {
									settings.onClose(thumb);
								} else {
									settings.onClose();
								}
							}
							
						} else {
							
							if (lightbox.data('panomove')) {
								// Panorama playing
								main.trigger('autopanoStop');
							}
							
							if ((isAudio || isVideo) && media) {
								var sv = media[0].volume,
									tm = transitionSpeed / (sv * 50),
									fade = function() {
											if (media) {
												var v = Math.max(media[0].volume - 0.02, 0);
												if (v > 0.005) {
													media[0].volume = v;
													setTimeout(fade, tm);
												} else {
													media[0].pause();
												}
											}
										};
								
								media.off('.' + ns);
								fade();
							}
							
							window.requestAnimationFrame(function() {
									
									lightbox.one('transitionend', function() {
											
											if(!SMALLSCREEN) {
												exitFullscreen();
											}
												
											if (main.length) {
												main.remove();
												main = $();
											}
												
											if (caption.length) {
												caption.remove();
												caption = $();
											}
												
											if (settings.onClose !== false) {
												if (thumb !== UNDEF) {
													settings.onClose(thumb);
												} else {
													setting.onClose();
												}
											}
											
											lightbox.css({
													display:	'none'
												});
											
											showControls();
										});
									
									lightbox.css({
											transition:	'opacity 500ms ease-in-out',
											opacity:	0
										});
								});
							
						}
					},
				
				// Removing the lightbox completely
				
				removeLightbox = function() {
						var remove = function() {
									if (settings.lightboxFullscreen) {
										exitFullscreen();
									}
									lightbox.remove();
								};
						// Stop slideshow		
						stopAuto(true);
						listening = false;
						// removing event handlers
						thumbs.off('.' + ns);
						$document.add($window).add(lightbox).off('.' + ns);
						$body.add($('html')).css('overflow', '');
						$body.removeClass(settings.lightboxOnClass);
						
						if (!main.length) {
							// No image; remove immediately
							remove();
						} else {
							// Fade out
							main.css('transition', 'opacity ' + settings.transitionSpeed + 'ms ease-out')
								.one('transitionend', remove)
								.css('opacity', 0);
						}
					},
				
				// Set play/pause button
				
				setPlayPause = function(on) {
					
						if (lb_play_pause) {
							if (on) {
								lb_play_pause.removeClass('icon-play').addClass('icon-pause');
							} else {
								lb_play_pause.removeClass('icon-pause').addClass('icon-play').removeClass('show-progress');
							}
						}
					},
				
				// Toggling auto slideshow
				
				toggleAuto = function() {
					
						if (slideshow) {
							stopAuto();
						} else {
							startAuto();
						}
						
					},
					
				// Starting auto slideshow
				
				startAuto = function(keepMusic) {
						
						clearTimeout(slideshow);
						
						setPlayPause(true);
						
						if (settings.autohideControls) {
							hideControlsLater();
						}
						
						caption.addClass('slideshow');
						
						// Resetting zoom
						fitImages = settings.fitImages;
						
						// Starting background music
						if (!(keepMusic === true) && settings.backgroundAudioSlideshowControl) {
							backgroundAudioRoot.trigger('fadeInPlayer');
						}
						
						// Full screen mode
						if (settings.slideshowFullscreen) {
							requestFullscreen();
						}
						
						// Wake lock
						requestWakeLock();
						
						if (audioClipTimeout) {
							// Wait until audioclip ends
							slideshow = null;
							slideshowSuspended = true;
							onAudioClipEnded(nextImage);
						} else {
							// Immediately start
							slideshowSuspended = false;
							slideshow = setTimeout(nextImage, settings.slideshowDelay / 4);
						}
						
						if (settings.onSlideshowStart !== false) {
							settings.onSlideshowStart(thumb);
						}
					},
				
				resumeAuto = function() {
						
						setPlayPause(true);
						
						if (settings.autohideControls) {
							hideControlsLater();
						}
						
						$body.add(lightbox).addClass(settings.immerseClass);
						slideshow = setTimeout(nextImage, settings.slideshowDelay / 4);
						slideshowSuspended = false;
						
						if (settings.backgroundAudioSlideshowControl) {
							backgroundAudioRoot.trigger('fadeInPlayer');
						}
					},
		
				stopAuto = function(keepMusic) {
						var sr = slideshow !== null;
					
						if (settings.onSlideshowPause !== false) {
							settings.onSlideshowPause(thumb);
						}
						
						if (!(keepMusic === true) && settings.backgroundAudioSlideshowControl) {
							backgroundAudioRoot.trigger('fadeOutPlayer');
						}
						
						setPlayPause(false);
						
						clearTimeout(slideshow);
						slideshow = null;
						slideshowSuspended = false;
						caption.show().removeClass('slideshow');
						$body.add(lightbox).removeClass(settings.immerseClass);
						
						showControls();
						
						if (settings.slideshowFullscreen && !settings.lightboxFullscreen) {
							exitFullscreen();
						}
						
						releaseWakeLock();
						
						if (settings.onSlideshowPause !== false) {
							settings.onSlideshowPause(thumb);
						}
						
						return sr;
					},
				
				suspendAuto = function() {
					
						if (settings.backgroundAudioSlideshowControl) {
							backgroundAudioRoot.trigger('fadeOutPlayer');
						}
						
						clearTimeout(slideshow);
						slideshow = null;
						slideshowSuspended = true;
					},
					
				updateAudioClipProgress = function(pb) {
					
						if (audioClip.length && pb.length) {
							pb.css('width', (100 * audioClip[0].currentTime / audioClip[0].duration) + '%');
						}
					},
					
				startAudioClip = function() {
					
						if (audioClip.length) {
							
							audioClip[0].play().then(function() {
									
									var btn = controls.find('.audioclip-btn'),
										pb = btn.find('.progress');
										
									btn.removeClass('icon-volume-up').addClass('icon-pause');
									audioClip.parent().removeClass('paused').addClass('playing');
									
									if (slideshow) {
										suspendAuto();
									}
									
									if (!pb.length) {
										pb = $('<div>', {
												'class':	'progress'
											}).appendTo(btn);
									}
									
									updateAudioClipProgress(pb);
									
									audioClipTimeout = setInterval(updateAudioClipProgress, 100, pb);
									
									audioClip.on('ended', function() {
										
											btn.removeClass('icon-pause').addClass('icon-volume-up');
											audioClip.parent().removeClass('playing').addClass('paused');
											
											if (slideshowSuspended) {
												resumeAuto();
											}
											
											pb.remove();
											
											clearInterval(audioClipTimeout);
										});
										
								}, function(err) {
									
									btn.removeClass('icon-pause').addClass('icon-volume-up');
									console.log(err);
								});
						}
					},
					
				pauseAudioClip = function() {
							
						if (audioClip.length) {
							var btn = controls.find('.audioclip-btn');
							
							audioClip[0].pause();
							clearInterval(audioClipTimeout);
							
							btn.removeClass('icon-pause').addClass('icon-volume-up');
							audioClip.parent().removeClass('playing').addClass('paused');
							
							if (slideshowSuspended) {
								resumeAuto();
							}
						}
					},	
					
				onAudioClipEnded = function(fn, params) {
					
						if (audioClip.length && typeof fn === FUNCTION) {
							if (audioClip[0].paused) {
								if (typeof params !== UNDEF) {
									fn.apply(null, params);
								} else {
									fn.call(null);
								}
							} else {
								audioClip.on('ended', fn, params);
							}
						}
					},
					
				mediaPlaying = function() {
					
						if (settings.muteBackgroundAudio) {
							if (backgroundAudioOn = !backgroundAudioRoot.data('paused')) {
								backgroundAudioRoot.trigger('fadeOutPlayer');
							}
						}
						startAudioClip();
						main.addClass(settings.playingClass);
					},
					
				mediaPaused = function() {
						
						if (settings.muteBackgroundAudio && backgroundAudioOn) {
							backgroundAudioRoot.trigger('fadeInPlayer');
						}
						pauseAudioClip();
						main.removeClass(settings.playingClass);
					},
					
				mediaEnded = function() {
					
						if (settings.muteBackgroundAudio) {
							backgroundAudioOn = false;
						}
						pauseAudioClip();
						main.removeClass(settings.playingClass);
					},
				
				fadeMedia = function(el) {
						var media = (el || lightbox).find('audio,video').eq(0);
						
						if (media.length) {
							var sv = media[0].volume,
								tm = settings.transitionSpeed / (sv * 50),
								fade = function() {
										if (media) {
											var v = Math.max(media[0].volume - 0.02, 0);
											if (v > 0.005) {
												media[0].volume = v;
												setTimeout(fade, tm);
											} else {
												media[0].pause();
											}
										}
									};
							
							media.off('volumechange.' + ns);
							fade();
						}
					},
					
				stopMedia = function(el) {
						var m = (el || lightbox).find('audio,video').eq(0);
						
						if (m.length) {
							m[0].pause();
						}
					},
					
				showControls = function() {
						clearTimeout(controlsTimeout);
						$body.add(lightbox).removeClass(settings.immerseClass);
						if (settings.autohideCaption) {
							showCaption();
						}
					},
				
				hideControls = function() {
						var hel;
						clearTimeout(controlsTimeout);
						// Check if mouse is over a panel 
						if (hel = document.querySelectorAll(':hover')) {
							for (var i in hel) {
								if (hel[i].id === 'panels') {
									hideControlsLater();
									return;
								}
							}
						}
						
						$body.hideAllTooltips();
						$body.add(lightbox).addClass(settings.immerseClass);
						if (settings.autohideCaption) {
							hideCaption();
						}
					},
					
				hideControlsLater = function() {
						clearTimeout(controlsTimeout);
						controlsTimeout = setTimeout(function() {
								//hideArrows();
								hideControls();
							}, settings.hideControlsDelay);
					},
					
				toggleControls = function() {
						$body.add(lightbox).toggleClass(settings.immerseClass, !controlsHidden());
					},
				
				controlsHidden = function() {
						return $body.hasClass(settings.immerseClass);
					},
				
				showCaption = function() {
						lightbox.addClass(settings.captionVisibleClass);
					},
					
				hideCaption = function() {
						lightbox.removeClass(settings.captionVisibleClass);
					},
				
				toggleAll = function() {
						if (controlsHidden()) {
							showControls();
							// Show items only to be visible by default
							if (!lightbox.hasClass(settings.captionVisibleClass)) {
								showCaption();
							} 
						} else {
							// Hide everything
							hideControls();
							hideCaption();
						}
						savePrefs(settings, settings.prefsKept, 'lb_pref');
						
						if (settings.fitBetween) {
							// Refit only if controls affect the position or height
							if (main.data('fitted')) {
								// re-adjusting position
								fitImage(true);
							}
						}
					},
					
				previousImage = function() {
						loadImage(curr - 1);
					},
				
				nextImage = function() {
						loadImage(curr + 1);
					},
									
				// Setting up the structure
				
				initLightbox = function(ns) {
					
						lightbox = $('<div>', {
									id: 		ns,
									'class': 	'lightbox ' + settings.captionVisibleClass
								})
							.hide()
							.appendTo(self);
						
						// Darken background
						lb_overlay = $('<div>', {
									'class': 	'lightbox-overlay'
								})
							.appendTo(lightbox);
						
						// Activity indicator
						lb_activity = $('<div>', {
									'class': 	'lightbox-loading'
								})
							.append('<div>')
							.append('<div>')
							.append('<div>')
							.appendTo(lightbox);
						
						// Controls group
						var lb_controls = $('<div>', {
									'class':	'controls'
								}).appendTo(lightbox);
							
						// Zoom button(s)
						if (settings.zoomSlider) {
							// Slider
							var zoom_panel= $('<div>', {
										'class':	'zoom-control'
									}),
								zoom_in_btn,
								zoom_out_btn;
								
							zoom_out_btn = $('<a>', {
										'class':	'zoom-out button'
									})
								.on('click', function() {
										decreaseZoom();
										return false;
									})
								.appendTo(zoom_panel);
									
							lb_zoom_slider = $('<input>', {
										name:		'zoom-level',
										type:		'range'
									})
								.on('change', function() {
										setZoom(parseInt(lb_zoom_slider.val()));
										return true;
									})
								.appendTo(zoom_panel);
									
							zoom_in_btn = $('<a>', {
										'class':	'zoom-in button'
									})
								.on('click', function() {
										increaseZoom();
										return false;
									})
								.appendTo(zoom_panel);
								
							// Zoom toggle button
							lb_zoom = $('<button>', {
										type: 		'button',
										'class': 	'zoom-level icon-zoom-level button'
									})
								.on('click.' + ns, function(){
										if (listening) {
											setTimeout(toggleZoom, 50);
											if ($('html').data('whatinput') === 'touch') {
												// Touch toggle
												return true;
											}
										}
										return false; 
									})
								.addTooltip($('<div>', {
										'class':		'zoom-panel'
									}).append(zoom_panel), {
										touchToggle:	true,
										pos:			[1,0,1,2]
									})
								.appendTo(lb_controls);
															
						} else {
							// Zoom toggle button
							lb_zoomin = $('<button>', {
										type: 		'button',
										'class': 	'button zoom-in icon-expand',
										'title': 	text.toggleZoom
									})
								.on('click', function() {
										toggleZoom();
										return false;
									})
								.appendTo(lb_controls);
							
							lb_zoomout = $('<button>', {
										type: 		'button',
										'class': 	'button zoom-out icon-contract',
										'title': 	text.toggleZoom
									})
								.on('click', function() {
										toggleZoom();
										return false;
									})
								.appendTo(lb_controls);
								
							lb_zoom_slider = $();
						}
						
						// Buttons for more than 1 items
						
						if (thumbs.length > 1) {
							
							// Play/Pause button
							lb_play_pause = $('<button>', {
									type: 		'button',
									'class': 	'play-pause icon-play button',
									'title': 	text.slideshow
								})
								.appendTo(lb_controls);
							
							lb_progress = $('<span>', {
									'class': 	'progress'
								})
								.appendTo(lb_play_pause);
								
							lb_play_pause.on('click.' + ns, function(){ 
									if (listening) {
										toggleAuto();
										if ($('html').data('whatinput') === 'touch') {
											// Touch toggle
											return true;
										}
									}
									return false; 
								});
									
							if (settings.timingControl) {
								// Slideshow timing control
								var timing_panel= $('<div>', {
											id:			'timing-control',
											'class':	'timing-control'
										}),
									decr_time_btn = $('<a>', {
											'class':	'decr-time'
										}).appendTo(timing_panel);
										
								lb_timing_value = $('<span>').appendTo(timing_panel);
										
								var incr_time_btn = $('<a>', {
											'class':	'incr-time'
										}).appendTo(timing_panel);
									
								lb_play_pause.addTooltip($('<div>', {
											'class':	'timing-panel'
										}).append($('<div>', {
											'class':	'label',
											html:		text.timing
										})).append(timing_panel), {
											touchToggle:	true,
											delay:			600,
											pos:			[1,0,1,2]
										});
									
								incr_time_btn.on('click', function() {
										increaseTiming();
										return false;
									});
								
								decr_time_btn.on('click', function() {
										decreaseTiming();
										return false;
									});
								
								setTimeout(function() { 
										setTiming(settings.slideshowDelay);
									}, 500);
								
							} else {
								// No timing control
								lb_play_pause.attr('title', text.startStopSlideshow);
							}
							
							// Previous button
							lb_btn_left = $('<button>', {
										type: 		'button',
										'class': 	'arrow-left button'
									})
									.on('click.' + ns, function(){
										if (listening) {
											previousImage();
										}
										return false;
									})
									.appendTo(lightbox);
								
							// Next button
							lb_btn_right = $('<button>', {
										type: 		'button',
										'class': 	'arrow-right button'
									})
									.on('click.' + ns, function(){
										if (listening) {
											nextImage();
										}
										return false;
									})
									.appendTo(lightbox);
						}
						
						// Slideshow delay and transition speed
						if (!settings.hasOwnProperty('slideshowDelay')) {
							slideshowDelay = settings.slideshowDelay = 3000;
						}
						if (!settings.hasOwnProperty('transitionSpeed')) {
							transitionSpeed = settings.transitionSpeed = 400;
						}
						
						// Close button
						lb_close = $('<button>', {
								type: 		'button',
								'class': 	'close icon-close button',
								'title': 	text.close
							})
							.on('click.' + ns, function() {
								if (listening) {
									quitLightbox();
								}
								return false; 
							})
							.appendTo(lb_controls);
						
						// Adding tooltips
						lightbox.find('button').not(lb_zoom).addTooltip();
			
					};
				
			// Initializing Lightbox
						
			if (ns) {
				// should never happen, but still living lightbox attached to thumb elements
				cleanup(ns);
			}
			
			// Creating new namespace
			self.data('llb_ns', ns = 'llb_' + Math.floor(Math.random() * 10000));		
	
			// Finding thumbs
			thumbs = self.find(settings.delegate);
			
			loadPrefs(settings, settings.prefsKept, 'lb_pref');
			
			// Control strip
			controls = $('#controls');
			
			panels = $('#panels');
			
			// Initializing controls
			initLightbox();	
			
			// Setting up events
					
			// Resize
			$window.on('resize.' + ns /* + ' orientationchange.' + ns*/, function() {
					clearTimeout(resizeTimeout);
					// Executed only if 50ms has passed since the last resize event
					resizeTimeout = setTimeout(function() { 
						clearTimeout(resizeTimeout);
						if (main && main.length && !(isVr && isFullscreen())) {
							fitImage();
						}
					}, 50);
				});
			
			// Quit on document touch
			if (settings.quitOnDocClick) {
				
				lb_overlay.on('click.' + ns, function(e) {
						
						if (listening) {
							if (settings.autohideControls && controlsHidden()) {
								// Toggle controls back
								showControls();
								return false;
							}
							
							if (main.length && !$(e.target).is(main)) {
								// Quit
								quitLightbox();
								return false;
							}
						}
						
						return true;
					});
			}
			
			// On mouse-driven devices
			if (settings.autohideControls) {
				
				// Auto hide controls
				$document.on('mousemove.' + ns, function() {
						showControls();
						hideControlsLater();
					});
			}
							
			// Keyboard
			if (settings.enableKeyboard) {
				
				$document.on('keyup.' + ns, function(e) {
					if (!listening || !main.length || lightbox.children('.modal:visible').length || document.activeElement && (document.activeElement.nodeName === 'INPUT' || document.activeElement.nodeName === 'TEXTAREA')) {
						return true;
					}
					
					e.preventDefault();
					
					switch (e.keyCode) {
						
						case 27:
							quitLightbox();
							break;
						
						case 33:
						case 37:
							previousImage();
							break;
							
						case 34:
						case 39:
							nextImage();
							break;
							
						case 97:
						case 35:
							loadImage(thumbs.length - 1);
							break;
							
						case 103:
						case 36:
							loadImage(0);
							break;
						
						case 106:
						case 179:
							toggleAuto();
							break;
							
						case 107:
							if (settings.useZoom) {
								if (settings.zoomSlider) {
									increaseZoom();
								} else {
									toggleZoom();
								}
							} else {
								toggleCaption();
							}
							break;
							
						case 109:
							if (settings.useZoom) {
								if (settings.zoomSlider) {
									decreaseZoom();
								} else {
									toggleZoom();
								}
							} else {
								togglePanels();
							}
							break;
							
						case 32:
							if (isVideo) {
								if (media[0].paused) {
									playVideo(media[0]);
								} else {
									media[0].pause();
								}
							} else if (isAudio) {
								if (media[0].paused) {
									media[0].play();
								} else {
									media[0].pause();
								}
							} else {
								toggleAuto();
							}
							break;
							
						default:
							return true;
											
					}
					
					return false;
				});
			}
			
			// Mouse wheel
			if (settings.mouseWheelAction) {
				var wheelTimeout = null;
				
				lightbox.on('mousewheel.' + ns + ' DOMMouseScroll.' + ns, function(e) {
						if (listening) {
							clearTimeout(wheelTimeout);
							
							if (	!main.length || 
									$(e.target).closest('.' + settings.lightboxCaptionClass).length ||
									main.hasClass('pdf') || 
									main.hasClass('external') ||
									lightbox.children('.modal:visible').length
								) {
								return true;
							}
							
							if (e.ctrlKey == true || settings.mouseWheelAction === 'zoom') {
								if (settings.useZoom) {
									if (settings.zoomSlider) {
										wheelTimeout = setTimeout((e.originalEvent.wheelDelta > 0 || e.originalEvent.detail < 0)?
											function(e) {
												increaseZoom();
											}
											: 
											function(e) {
												decreaseZoom();
											}, 100);
									} else {
										wheelTimeout = setTimeout((e.originalEvent.wheelDelta > 0 || e.originalEvent.detail < 0)?
											function(e) {
												if (!fitImages) {
													fitImages = true;
													fitImage(true);
												}
											}
											: 
											function(e) {
												if (fitImage) {
													fitImages = false;
													fitImage(true);
												}
											}, 100);
									}
								}
							} else if (settings.mouseWheelAction === 'navigation') {
								var d = new Date();
								if (typeof lightbox.data('lastWheel') === UNDEF || (d - lightbox.data('lastWheel')) > 250) {
									// Only if 250ms apart from the last wheel event
									wheelTimeout = setTimeout(function(dir) {
											lightbox.data('lastWheel', new Date());
											if (dir) {
												previousImage();
											} else {
												nextImage();
											}
									 	}, 100, (e.originalEvent.wheelDelta > 0 || e.originalEvent.detail < 0));
								}
							}
						}
						return false;
					});
			}
			
			// Events that can be triggered from outside
			
			// External trigger to switch image
			self.on('lightboxLoad.' + ns, function(e, n, auto) {
						stopAuto(true);
						
						if (typeof n === NUMBER) {
							n = Math.minMax(0, n, thumbs.length);
						} else if (typeof n === UNDEF) {
							n = 0;
						} else {
							n = thumbs.index(n);
						}
						
						if (settings.onStart !== false) {
							settings.onStart((typeof n === NUMBER)? thumbs.eq(n) : th);
						}
						
						loadImage(n);
						
						if (slideshow) {
							stopAuto(true);
						}
					
						if (auto) {
							if (typeof auto === NUMBER) {
								setTimeout(startAuto, auto);
							} else {
								startAuto();
							}
						}
					})
				// External trigger to quit = does not remove, just hides the lightbox
				.on('lightboxQuit.' + ns, function() {
						quitLightbox();
						return false;
					})
				// Removing the lightbox completely
				.on('lightboxRemove.' + ns, function() {
						removeLightbox();
						return false;
					})
				// Stopping auto hide controls
				.on('stopAutoHide.' + ns, function() {
						clearTimeout(controlsTimeout);
					})
				// Returning the current lightbox container
				.on('lightboxContainer.' + ns, function() {
						return lightbox;
					})
				// External trigger to go to the previous image
				.on('lightboxNext.' + ns, function() {
						stopAuto();
						nextImage();
						return false;
					})
				// External trigger to go to next image
				.on('lightboxPrevious.' + ns, function() {
						stopAuto();
						previousImage();
						return false;
					});
					
			// Ready event
			
			if (settings.hasOwnProperty('onReady') && typeof settings.onReady === FUNCTION) {
				settings.onReady(thumb);
			}
			
			// Auto start
			
			if (settings.autoStart && settings.baseSet) {
				startAuto();
			}
	
			return this;
		};
	
	$.fn.lightbox.defaults = {
			delegate:						'.card.lbable',
			lightboxMainClass:				'lightbox-main',
			currClass:						'curr',
			lightboxCaptionClass:			'lightbox-caption',
			lightboxOnClass:				'lightbox-on',
			controlsClass:					'controls',
			buttonClass:					'secondary button small',
			activeClass:					'active',
			scrollClass:					'scroll',
			regionsClass:					'regions',
			captionVisibleClass:			'caption-visible',
			thumbsVisibleClass:				'thumbs-visible',
			hasbuttonsClass:				'hasbuttons',
			immerseClass:					'immerse',
			playingClass:					'playing',
			transitionProps:				'300ms ease-out',
			slideshowDelay:					3000,
			timingControl:					false,
			hideControlsDelay:				2500,
			prefsKept: 						[ 
												'audioClipVolume'
											],
			captionVisible:					true,
			fitImages:						true,
			fitBetween:						false,
			fitBelowControls:				true,
			fitRatio:						1,
			maxZoom:						1.4,
			horzPanoThreshold:				1.8,
			vertPanoThreshold:				0.5,
			displayOriginals:				true,
			useZoom:						true,
			zoomSlider:						false,
			useOriginals:					false,
			hiDpiThumbs:					false,
			hiDpiImages:					false,
			showFullscreen:					false,
			indexName:						'index.html',
			baseSet:						true,
			autohideControls:				false,
			autohideCaption:				false,
			autoStart:						false,
			clickAction:					'donothing',
			mouseWheelAction:				'navigation',
			useSlideshow:					true,
			autoplayAudioClips:				true,
			backgroundAudioSlideshowControl:false,
			muteBackgroundAudio:			true,
			use360Player:					true,
			autoRotate360:					true,
			rotateSpeed360:					-2,
			useGoogleDocs:					true,
			lightboxFullscreen:				false,
			slideshowFullscreen:			false,
			slideshowFullscreenOnlyMobiles:	true,
			afterLast:						'donothing',
			doubleClickZoom:				true,
			mapHeight:						0.8,
			mapApiKey:						'',
			mapType:						'hybrid',
			mapZoom:						15,
			showShopBtn:					false,
			showFeedback:					false,
			showLowestPrice:				false,
			videoAuto:						false,
			videoLoop:						false,
			autoPano:						false,
			autoPanoStart:					0,
			autoPanoSpeed:					80,
			volume:							0.5,
			audioClipVolume:				0.75,
			rightClickProtect: 				false,
			useRating:						false,
			jalbumRating:					false,
			visitorRating:					false,
			buttonLabels:					true,
			showNumbers:					false,
			showShare:						false,
			showDownload:					false,
			allowDownloadScaled:			false,
			allowDownloadOthers:			false,
			showMap:						false,
			showRegions:					true,
			regionsVisible:					false,
			printImage:						false,
			transitionType:					'crossFadeAndSlide',
			transitionSpeed:				400,
			panoramaAutoDetect:				1.3333,
			panoramaTreshold:				[ 2.5, 2.5 ],
			defaultPosterSize:				[ 628, 360 ],
			preloadNext:					true,
			enableKeyboard: 				true,
			enableMouseWheel:				true,
			quitOnEnd:						true,
			quitOnDocClick: 				true,
			onStart:						false,
			onClose:						false,
			onLoadStart:					false,
			onLoadEnd:						false,
			onAfterLast:					false,
			onBeforeFirst:					false,
			onReady:						false,
			onSlideshowStart:				false,
			onSlideshowPause:				false,
			onVolumeChange:					false,
			feedbackBtnLabel:				'Write feedback',
			feedbackBtnTooltip:				'View feedback window',
			feedback:						{}
		};
	
	$.fn.lightbox.text = {
			rating:							"Rating",
			startOver:						"Start over",
			upOneLevel:						"Up one level",
			backToHome:						"Back to home",
			backToIndex:					"Back to index page",
			nextFolder:						"Next folder",
			nextIndex:						"Next index page",
			atLastPageQuestion:				"Where to go next?",
			atLastPage:						"At last page",
			clickToOpen:					"Click to open this document with the associated viewer",
			toggleZoom:						"Toggle Zoom",
			slideshow:						"Slideshow",
			timing:							"Timing",
			startStopSlideshow:				"Start / stop slideshow",
			close:							"Close"
		};

	
})(jQuery, jQuery(window), jQuery(document), jQuery('body'));
