/*	
 *	feedback() :: setting up the shopping cart with Feedback
 *
 *	Usage: $(targetElement).feedback( [items,] options );
 *
 *	Dual licensed under the MIT and GPL licenses.
 *	- http://www.opensource.org/licenses/mit-license.php
 *	- http://www.gnu.org/copyleft/gpl.html
 *
 *	Items: jQuery element (array) holding the elements
 *
 *	Options:
		to: 			(the email address),
		cc:				(cc email),
		subject:		(default subject),
		template: 		(template),
*/

;(function($, $window, $body) {
	'use strict';
	
	$.fn.feedback = function(album, settings) {
		
			if (!album) {
				return this;
			}
			
			settings = $.extend({}, $.fn.feedback.defaults, album.getRootProperty(J.FEEDBACK), settings);
			
			if (!settings.to) {
				console.log('feedback.js: no target email found');
				return this;
			}
			
			var // global vars
				self = 		$(this).eq(0),							// parent element					
				ns = 		self.data('lfb_ns'),					// event namespace
				text = 		getTranslations($.fn.feedback.text),	// texts
				to = 		xDecrypt(settings.to),					// send target
				itemTemplate,										// Item template
				
				LBR =		'%0D%0A',								// line break
				TAB =		'%09',									// Tab
				SPACE = 	'%20',									// Space
				AMP =		'%26',									// Ampersand
				
				// jQuery elements:
				cart,												// the cart parent
				popup,												// modal window
				cont,												// modal window content
				itemCont,											// item container
				copyBtn = $(),										// 'Copy' button
				sendBtn = $(),										// 'Send' button
				emptyCartBtn = $(),									// 'Empty' button
				buttons,											// Add and View buttons
							
				id = {
							cart:			'feedback-cart',
							shortcut:		'feedback-shortcut',
							window:			'feedback-window',
							cont:			'cont',
							beforeItems:	'before-items',
							summary:		'summary',
							summaryTxt:		'summary-txt',
							details:		'details',
							newItems:		'new-items',
							items:			'items',
							afterItems:		'after-items',
							buttons:		'buttons',
							item:			'item',
							path:			'path',
							link:			'link',
							file:			'file',
							info:			'info',
							thumb:			'thumb',
							thumbExt:		'thumbext',
							icon:			'icon',
							title:			'title',
							data:			'data',
							options:		'options'
						},
				
				cookie = {
							data:			'feedback.cart.data',
							date:			'feedback.cart.date'
						},
				
				PATH = DIR_PATH || '/',
					
				setStorage = function(key, value) {
						localStorage.setItem(PATH + key, value); 
					},
					
				getStorage = function(key) {
						return localStorage.getItem(PATH + key);
					},
					
				removeStorage = function(key) {
						localStorage.removeItem(PATH + key);
					},
					
				wipeData = function() {
						if (LOCALSTORAGE) {
							for (var c in cookie) {
								removeStorage(cookie[c]);     
							}
						}
					},
				
				// Returns one item's data: path and serialized values
				
				getFormData = function(f) {
						var d = {};
						
						d[id.path] = f.data(id.path);
						d[id.file] = f.data(id.file);
						d[id.title] = f.find('.' + id.title).text();
						d[id.data] = f.find('.' + id.data + ' form').serialize();
						
						if (typeof f.data(id.link) !== UNDEF) {
							d[id.link] = f.data(id.link);
						}
						
						if (typeof f.data(id.icon) !== UNDEF) {
							d[id.icon] = f.data(id.icon);
						}
						
						if (typeof f.data(id.thumbExt) !== UNDEF) {
							d[id.thumbExt] = f.data(id.thumbExt);
						}
							
						return d;
					},
					
				// Saving the cart to the localStorage
				
				saveData = function(doneFn) {
					
						if (LOCALSTORAGE) {
							var d = {},
								el;
							
							el = cart.find('.' + id.beforeItems + ' form');
							if (el.length) {
								d.before = el.serialize();
							}
							
							el = cart.find('.' + id.item);
							if (el.length) {
								d.items = new Array();
								el.each(function() {
										d.items.push(getFormData($(this)));
									});
							}
							
							el = cart.find('.' + id.afterItems + ' form');
							if (el.length) {
								d.after = el.serialize();
							}
							
							if (!isEmpty(d)) {
								d = JSON.stringify(d);
								setStorage(cookie.data, d);
								// Saving current time
								setStorage(cookie.date, (new Date()).getTime());
							} else {
								wipeData();
							}
						}
						
						if (typeof doneFn === FUNCTION) {
							doneFn.call(this);
						}
					},
				
				// Loading the cart from the localStorage
				
				loadData = function(doneFn) {
					
						if (LOCALSTORAGE) {
							var date = (new Date()).getTime(),
								d;
								
							if (d = getStorage(cookie.date)) {
								if ((date - parseInt(d)) > settings.expiry) {
									wipeData();
									return;
								} else {
									setStorage(cookie.date, date);
								}
							}
								
							if (d = getStorage(cookie.data)) {
								
								var data = JSON.parse(d);
								
								if (!isEmpty(data)) {
									
									if (data['before']) {
										cart.find('.' + id.beforeItems + ' form').deserialize(data.before);
									}
									
									if (data['items']) {
										for (var i = 0; i < data.items.length; i++) {
											addItem(data.items[i]);
										}
									}
									
									if (data['after']) {
										cart.find('.' + id.afterItems + ' form').deserialize(data.after);
									}
								}
							}
						}
						
						updateCart(true);
						
						if (typeof doneFn === FUNCTION) {
							doneFn.call(this);
						}
					},
				
				// Checking if the item to be added is already present
					
				checkUnique = function(data) {
						var it = $();
						
						cont.find('.' + id.item).each(function() {
								var el = $(this);
								if (el.data(id.file) === data[id.file] && el.data(id.path) === data[id.path]) {
									it = el;
									return false;
								}
							});
						
						return it;
					},
					
				// Adding an item
				
				addItem = function(data) {
					
						if (!data || !data.hasOwnProperty(id.file)) {
							return;
						}
						
						var it;
						
						if (!settings.allowMultiple) {	
							it = checkUnique(data);
						
							if (it.length) {
								return it;
							}
						}
						
						// Creating the item
						var	lb,
							tp;
							
						if (data.hasOwnProperty(id.icon)) {
							tp = settings.resPath + '/' + data[id.icon];
						} else {
							if (data.hasOwnProperty(id.path)) {
								if (settings.hasOwnProperty('relPath')) {
									tp = getRelativePath(settings.relPath || '/', data[id.path]);
								} else {
									tp = data[id.path];
								}
							} else {
								tp = '';
							}
							
							if (data.hasOwnProperty(id.link)) {
								tp += data[id.link] + '/';
							}
						
							tp += settings.thumbsDir + '/';
							
							if (data.hasOwnProperty(id.thumbExt)) {
								tp += data[id.file].replaceExt(data[id.thumbExt]);
							} else {
								tp += data[id.file];
							}
						}
							
						it = $('<div>', {
								'class':	id.item
							});
						
						lb = $('<div>', {
								'class':	id.info
							}).appendTo(it);
							
						// Remove button
						lb.append($('<a>', {
								'class': 	'button icon-close'
							}).on('click.' + ns, function() {
								$(this).parents('.' + id.item).remove();
							}).on('selectstart', function(e) {
								e.preventDefault();
								return false;
							}));
							
						// Thumb image
						lb.append($('<img>', {
								'class': 	id.thumb,
								src: 		tp
							}));
						
						// Title
						lb.append($('<span>', {
								'class': 	id.title,
								html: 		data.title
							}));
						
						// Data from template
						it.append($('<div>', {
								'class': 	id.data,
								html:		album.processTemplate(itemTemplate)
							}));
							
						itemCont.append(it);
						
						if (data.data) {
							it.find('form').deserialize(data.data, true);
						}
						
						it.data({
								path:		data[id.path],
								file:		data[id.file]
							});
						
						if (data.hasOwnProperty(id.link)) {
							it.data(id.link, data[id.link]);
						}
						
						if (data.hasOwnProperty(id.icon)) {
							it.data(id.icon, data[id.icon]);
						}
						
						if (data.hasOwnProperty(id.thumbExt)) {
							it.data(id.thumbExt, data[id.thumbExt]);
						}
						
						return it;
					},
					
				// Converting jalbum.album item into data Object
				
				getData = function(item) {
					
						var file = item.hasOwnProperty(J.ORIGINAL)?	item[J.ORIGINAL][J.PATH] : 
									item.hasOwnProperty(J.IMAGE)? item[J.IMAGE][J.PATH].getFile() : 
									encodeURI(item[J.NAME]),
							data = {
									title: 		item[J.TITLE] || item[J.NAME] || '',
									path: 		album.getFolderPath(item), 
									file: 		file
								},
							icon = album.getIconPath(item);
								
						if (!item[J.THUMB][J.PATH].startsWith(settings.thumbsDir + '/')) {
							data[id.link] = item[J.THUMB][J.PATH].split('/' + settings.thumbsDir + '/')[0];
						}

						if (icon) {
							data[id.icon] = icon.getFile();
						} else {
							// Thumb extension (save if different from image ext)
							var tx = item[J.THUMB][J.PATH].getExt();
							if (file.getExt() !== tx) { 
								data[id.thumbExt] = tx;
							}
						}
						
						return data;
						//data.items[i][id.path], data.items[i][id.file], data.items[i][id.title], data.items[i]['data']);
					},
				
				// adding jAlbum.Album objects
				
				addItems = function(items) {
						
						if (!items) {
							return;
						}
						
						var fai = $();			// First added item 
						
						if (Array.isArray(items)) {
							for (var i = 0, it; i < items.length; i++) {
								it = addItem(getData(items[i]));
								if (!i) {
									fai = it;
								}
							}
						} else {
							fai = addItem(getData(items));
						}
						
						updateCart();
						
						showCart(function() {
								// Settings the focus on the first added item
								if (fai.length) {
									fai = fai.find('input,textarea,select');
									if (fai.length) {
										fai[0].focus();
									}
								}
							});
					},
				
				// Number of items
				
				itemCount = function() {
						return cont.find('.' + id.item).length;
					},
									
				// Something has Changed
				
				updateCart = function(initial) {
						var cnt = itemCount();
						
						//sendBtn.add(copyBtn).add(emptyCartBtn).toggleClass('disabled', cnt === 0);
						
						updateItemCount(cnt);
						
						if (!initial) {
							saveData();
						}
					},
				
				// Updating the item count on the Add cart button
	
				updateItemCount = function(count) {
						var btn = buttons.find('.view-cart');
						
						if (btn.length) {
							var b = btn.children('.badge');
							
							if (!b.length) {
								b = $('<span>', {
										'class':	'badge',
									}).appendTo(btn);
							}
							
							if (count) {
								b.text(count).show();
							} else {
								b.hide();
							}
						}
						
						if (settings.useFloatButton) {
							// Toggling visibility
							buttons.removeClass('show-add').addClass('show-view');
						}
					},
					
				// Updating the selection buttons
					
				updateSelectedCount = function(count) {
						var btn = buttons.find('.add-cart.global');
						
						if (btn.length) {
							var b = btn.children('.badge');
							
							if (!b.length) {
								b = $('<span>', {
										'class':	'badge red'
									}).appendTo(btn);
							}
							
							if (count) {
								b.text(count).show();
								btn.removeClass('disabled');
							} else {
								b.hide();
								btn.addClass('disabled');
							}							
						}	
						
						if (settings.useFloatButton) {
							// Toggling visibility
							if (count) {
								buttons.removeClass('show-view').addClass('show-add');
							} else {
								buttons.removeClass('show-add').addClass('show-view');
							}
						}
					},
					
				// Reading form
					
				readForm = function(f) {
						
						if (f) {
							var v,
								b = [];
								
							f.find('input,textarea,select').each(function() {
									if ((v = $(this).val()) !== null) {
										b.push((this.name || this.id || this.nodeName) + ': ' + v);
									}
								});
							
							return b.join(', ');
						}
						
						return '';
					},
					
				// Formatting form
				
				format = function(f) {
						
						if (f) {
							switch (settings.formatting) {
								case 'human':
									return f.serialize().replace(/\+/g, SPACE).replace(/=/g, ': ').replace(/\&/g, LBR);
								case 'serialized':
									return (f.attr('name') || f.attr('id') || 'form') + '="' + encodeURIComponent(f.serialize()) + '";';
								default:
									return encodeURIComponent(readForm(f));
							}
						}
						
						return '';
					},
					
				// Reading content before items
				
				getBeforeItems = function() {
						var f;
							
						if ((f = cont.find('.' + id.beforeItems + ' form')).length) {
							return format(f);
						}
						
						return '';
					},
				
				// Reading items
					
				getItems = function() {
						var items = [];
						
						if (settings.formatting === 'serialized') {
							items.push('items={');
						} else {
							items.push(LBR + '----------------------');
						}
						
						cont.find('.' + id.item).each(function(i) {
								var self = $(this),
									path = self.data(id.path) + self.data(id.file),
									form = self.find('form').eq(0);
								
								switch (settings.formatting) {
									
									case 'human':
										items.push(
												(i? LBR : '') + (String)(i + 1) + '.' + TAB + 
												(encodeURI(path)).replace(/%/g, '%25') + 
												(form? (LBR + TAB + form.serialize().replace(/\+/g, SPACE).replace(/=/g, ': ').replace(/\&/g, ',' + SPACE)) : '') 
											);
										break;
										
									case 'serialized':
										items.push(
												'{file:"' + encodeURI(path).replace(/%/g, '%25') + '"' +
												(form? (',' + (form.attr('name') || form.attr('id') || 'form') + ':"' + encodeURIComponent(self.find('form').serialize()) + '"') : '') + 
												'},'
											);
										break;
										
									default:
										items.push(
												(String)(i + 1) + '.' + TAB + 
												encodeURI(path).replace(/%/g, '%25') + 
												(form? (':' + TAB + encodeURIComponent(readForm(form))) : '') 
											);
										break;
								}
							});
					
						if (settings.formatting === 'serialized') {
							items[items.length - 1] = items[items.length - 1].replace(/\,$/, '');
							items.push('};');
						} else {
							items.push('----------------------' + LBR);
						}
						
						return items.join(LBR);
					},
				
				// Reading content after items
				
				getAfterItems = function() {
						var f;
						
						if ((f = cont.find('.' + id.afterItems + ' form')).length) {
							return format(f);
						}
						
						return '';
					},
					
				// Copying the cart to the clipboard
				
				copyCart = function(tooLarge) {
						var body = getBeforeItems() + LBR + getItems() + LBR + getAfterItems(),
							warn = typeof tooLarge !== UNDEF && tooLarge,
							msg = $('<div>', {
											html: 	(tooLarge? ('<h4 class="icon-warning"> ' + text.tooLong + '</h4>') : '') +
													'<p><textarea name="body" rows="8">' + text.to + ': ' + to + '\n' +
														text.subject + ': ' + popup.find('header').text() + '\n\n' +
														decodeURIComponent(body) + '</textarea></p>' +
													'<p class="icon-info instructions"> ' + text.copyInstructions + '</p>'
										});
						
						$body.modal(msg, 
								[{
									t: 	text.okButton,
									c:	'icon-ok',
									h: 	function() {
											emptyCart();
										}
								}],
								{
									title:		tooLarge? text.warning : text.copiedToClipboard,
									icon:		tooLarge? 'warning' : 'info',
									'class': 	(tooLarge? 'warning' : 'success') + ' small'
								});
						
						msg.find('[name=body]')[0].select();
						document.execCommand('copy');
					
					},
							
				// Sending the cart
				
				sendCart = function() {
					
						var message,
							head = 	'mailto:' + encodeURIComponent(to) +
									'?subject=' + encodeURIComponent(popup.find('header').text()),
							body = 	getBeforeItems() + LBR + getItems() + LBR + getAfterItems();
						
						if (DEBUG) {
							try {
								console.log(decodeURIComponent(body));
							} catch(err) {
								console.log(err);
							}
						}
						
						message = head + '&body=' + body;
							
						if (message.length > 2048) {
							
							copyCart(true);
							
						} else {
						
							window.location.href = message;
							
							emptyCart();
						}
						
						hideCart();
						
					},
	
				// Creating the cart
				
				init = function(doneFn) {
					
						// Cart exists? Yes: clean old event handlers and div's
						if (ns !== undefined) {
							$(popup).off('.' + ns);
							$('#cart_' + ns).remove();
							$('#cart_shortcut_' + ns).remove();
						}
						
						// Creating new namespace
						self.data('lfb_ns', ns = 'lfb_' + Math.floor(Math.random() * 10000));
						
						// Creating the modal base (full screen)
						cart = $('<div>', {
								id: 		'cart_' + ns,
								'class': 	'modal ' + id.cart,
								'role':		'modal'
							}).hide().appendTo($body);
						
						// Creating the buttons
						buttons = $('<div>', {
									id:			'cart_shortcut_' + ns,
									'class': 	settings.useFloatButton? id.shortcut : 'buttons'
								})
							.append($('<a>', {
									'class':	'button view-cart ' + (settings.useFloatButton? 'large icon-email-send' : 'icon-edit secondary'),
									html:		' ' + settings.viewBtnLabel
								})
								.on('click.' + ns, function() {
										showCart();
									})
								)	
							.append($('<a>', {
									'class':	'global button add-cart icon-email-send' + (settings.useFloatButton? ' large' : ''),
									html:		' ' + settings.addBtnLabel
								})
								.on('click.' + ns, function() {
										if (typeof settings.getSelected === FUNCTION) {
											addItems(settings.getSelected.call());
											if (typeof settings.selectNone === FUNCTION) {
												settings.selectNone.call();
											}
										}
									})
								)
							.appendTo(self);
						
						if (settings.addCurrentButton) {
							buttons.append($('<a>', {
									'class':	'local button add-cart icon-edit',
									html:		' ' + settings.addBtnLabel
								})
								.on('click.' + ns, function() {
										if (typeof settings.getCurrent === 'function') {
											addItems(settings.getCurrent.call());
										}
									}));
						}
						
						buttons.find('.add-cart.global').addTooltip(settings.addBtnTooltip, { pos: settings.useFloatButton? [2,1,0,1] : settings.addCartTooltipPos });
						buttons.find('.add-cart.local').addTooltip(settings.addCurrentTooltip, { pos: settings.addCartTooltipPos });
								
						// The inside popup
						popup = $('<div>', {
								'class': 	'window has-header ' + id.window,
								role: 		'dialog'
							}).appendTo(cart);
						
						// Header
						popup.append($('<header>', {
								'class': 	'icon-email-send',
								html: 		' <strong> ' + settings.modalTitle.template(album.getAlbumTitle()) + '</strong>' 
							}));
						
						// Close button
						popup.append($('<a>', {
								'class': 	'btn close',
							}).on('click.' + ns, hideCart));
						
						cart.on('click.' + ns, function(e) {
								if (e.target.id === 'cart_' + ns) {
									hideCart();
									return false;
								};
								return true;
							});
						
						// Modal window content
						cont = $('<div>', {
								'class': 	'content'
							}).appendTo(popup);
						
						var cnt = settings.template.split(settings.itemsStart)[0].trim();
						
						if (cnt) {
							cont.append($('<div>', {
									'class':	id.beforeItems,
									html:		album.processTemplate(cnt)
								}));
						}
											
						// Details container
						itemCont = $('<div>', {
								'class': 	id.items
							}).appendTo(cont);
												
						cnt = settings.template.split(settings.itemsStop);
						
						if (cnt.length && (cnt = cnt[1].trim())) {
							cont.append($('<div>', {
									'class':	id.afterItems,
									html:		album.processTemplate(cnt)
								}));
						}
										
						// Buttons
						var btns = $('<div>', {
								'class': 	id.buttons
							}).appendTo(cont);
						
						btns.append($('<a>', {
								'class': 	'secondary button icon-arrow-left',
								text: 		' ' + text.continueBrowsing
							}).on('click.' + ns, function() {
									if (!$(this).hasClass('disabled')) {
										hideCart();
									}
							}));
						
						emptyCartBtn = $('<a>', {
								'class': 	'alert button icon-trash',
								text: 		' ' + text.removeAll
							}).on('click.' + ns, function() {
									if (!$(this).hasClass('disabled')) {
										emptyCart();
									}
							}).appendTo(btns);
							
						copyBtn = $('<a>', {
								'class': 	'success button icon-external',
								html: 		' ' + settings.copyBtnLabel
							}).on('click.' + ns, function() {
									if (!$(this).hasClass('disabled')) {
										copyCart(false); 
									}
									return false; 
							}).appendTo(btns);
						
						if (settings.useSendButton) {
							sendBtn = $('<a>', {
										'class': 	'button icon-email-send',
										html: 		' ' + settings.sendBtnLabel
								}).on('click.' + ns, function() {
									if (!$(this).hasClass('disabled')) {
										sendCart();
									}
								}).appendTo(btns);
								
							btns.after($('<p>', {
									'class':	'fineprint',
									html:		text.feedbackButtonExplanation
								}));
						}
							
											
						// Instructions
						if (settings.hasOwnProperty('instructions') && settings.instructions) {
							cont.append($('<div>', {
									'class': 	'instructions icon-info'
								})
								.append($('<div>')
								.append(settings.instructions.fixjAlbumPaths(settings.resPath, settings.rootPath, settings.relPath))));
							if (settings.instructions.indexOf('data-tooltip') > 0) {
								cont.find('.instructions [data-tooltip]').addTooltip();
							}
						
						}
						
						// Adding tooltip texts
						popup.find('a.close').addTooltip(text.continueBrowsing);
						
						if (settings.template.indexOf(settings.itemsStart) >= 0) {
							cnt = settings.template.split(settings.itemsStart)[1].trim();
							if (cnt) {
								cnt = cnt.split(settings.itemsStop)[0].trim();
								if (cnt) {
									itemTemplate = cnt.replace(/\$\{fileName\}/g, '\$\{name\}');
								}
							}
						}
						
						if (typeof doneFn === FUNCTION) {
							doneFn.call(this);
						}
					},
				
				// Emptying cart
				
				emptyCart = function(doneFn) {
						$body.modal($('<h3>', {
								'class':	'text-center icon-warning',	
								html:		' ' + text.removeAllItems
							}), [{
								t: 	text.yes,
								c:	'alert icon-trash',
								h: 	function() {
										cont.find('.item').remove();
										updateCart();
										hideCart();
										if (typeof doneFn === FUNCTION) {
												doneFn.call(this);
										}
									}
							}, {	
								t: 	text.no,
								h:	function() {}
							}], {
								icon:		'warning',
								'class':	'alert small'
							});
					},
				
				// Scrolling to last added item
				
				scrollToFocused = function() {
						var focused = $(document.activeElement);
						if (focused.length && $.contains(cont, focused)) {
							focused[0].scrollIntoView({
								behavior: 	'smooth',
								block: 		'nearest'
							});
						}
						/*
						cont.animate({ 
								scrollTop: 	cont[0].scrollHeight - Math.round(cont.outerHeight())  
							}, 400);
						*/
					},
					
				// Showing cart
				
				showCart = function(doneFn) {			
						
						if (!cart.is(':visible')) {
							
							$body.addClass('no-scroll');
							$body.addClass('has-modal');
							
							cart.fadeIn(400, function() {
								if (typeof doneFn === FUNCTION) {
									doneFn.call(this);
								}
								scrollToFocused();
							});
						}
						
						if (settings.useFloatButton) {
							buttons.hide();
						}
					},
				
				// Hiding cart
				
				hideCart = function(doneFn) {
						var c = itemCount();
						
						if (cart.is(':visible')) {
							
							$('body,html').removeClass('no-scroll');
							$('body').removeClass('has-modal');
							
							if (typeof settings['onClose'] === FUNCTION) {
								settings.onClose.call(this);
							}
							
							cart.fadeOut(400, function() {
								if (settings.useFloatButton) {
									buttons.show();
								}
								
								if (typeof doneFn === FUNCTION) {
									doneFn.call(this);
								}
							});
							
							saveData();
						}
					};
				
			
			// Initializing
			
			init(loadData);
			
			// Handlers: external triggers
			
				// Adding new items
			self.on('addItems', function(e) {
						var items = Array.prototype.slice.call(arguments, 1);
						addItems(items);
					})
				// Emptying cart
				.on('emptyCart', function(e, doneFn) {
						emptyCart(doneFn);
					})
				// Showing add cart button
				.on('itemsSelected', function(e, count) {
						updateSelectedCount(count);
					})
				// Showing cart
				.on('showCart', function(e, doneFn) {
						showCart(doneFn);
					})
				// Hiding the cart
				.on('hideCart', function(e, doneFn) {
						hideCart(doneFn);
					})
				// Adding onclose event
				.on('onClose', function(e, doneFn) {
						settings.onClose = doneFn;
					});
			
			//$window.on('unload.' + ns, saveData);
			
			// Ready event
			
			if (settings.hasOwnProperty('onReady')) {
				settings.onReady(this);
			}
	
			return this;
		};
	
	$.fn.feedback.defaults = {
			rootPath:						'',
			relPath:						'',
			expiry:							60 * 60 * 1000,		// 1 hour
			useFloatButton:					false,
			addCurrentButton:				false,
			allowMultiple:					false,
			addCartTooltipPos:				[1,2,1,0],
			formatting:						'human', 			// 'serialized' || 'human' || 'text'
			thumbsDir:						'thumbs',
			boxTitle:						'Feedback',
			viewBtnLabel:					'Feedback',
			addBtnLabel:					'Write feedback',
			addBtnTooltip:					'Provide feedback on the selected items',
			addCurrentTooltip:				'Provide feedback on this item',
			floatBtnLabel:					'View feedback cart',
			modalTitle:						'Feedback on album "{0}"',
			copyBtnLabel:					'Copy to clipboard',
			sendBtnLabel:					'Send feedback',
			useSendButton:					true,
			itemsStart:						'<!-- items:start -->',
			itemsStop:						'<!-- items:stop -->'
		};
			
	$.fn.feedback.text = {
			tooLong:						"This is too long to pass to the email application directly.",
			to:								"To",
			subject:						"Subject",
			copyInstructions:				"The text has been copied to your clipboard. Now switch to the email application (or webmail) and paste it. Move \"To\" and \"Subject\" into the appropriate boxes!",
			okButton:						"OK",
			warning:						"Warning",
			copiedToClipboard:				"Copied to clipboard!",
			continueBrowsing:				"Continue browsing",
			removeAll:						"Remove all",
			feedbackButtonExplanation:		"Try \"Send\" if you have an email application installed, use \"Copy\" for web mail!",
			removeAllItems:					"Remove all items?",
			yes:							"Yes",
			no:								"No"
		};
	
})(jQuery, $(window), $('body'));