package split;

/*
	Gui.java - User Interface
*/

import com.kitfox.svg.app.beans.SVGIcon;
import icons.Lazaicon;
import javax.swing.*;
import java.util.logging.Level;
import java.awt.*;
import java.beans.PropertyChangeListener;
import java.io.*;
import java.net.URLEncoder;
import java.util.logging.Logger;
import javax.swing.border.Border;
import javax.swing.filechooser.FileNameExtensionFilter;
import se.datadosen.component.*;
import net.jalbum.component.*;
import se.datadosen.jalbum.*;
import se.datadosen.util.Item;
import se.datadosen.util.Languages;
import se.datadosen.jalbum.event.JAlbumAdapter;
import se.datadosen.jalbum.event.JAlbumEvent;
import se.datadosen.util.IO;
import se.datadosen.util.VersionNumber;

/*************************************************
 * 
 *			Split skin GUI
 *			author David and Laza
 */

public class Gui extends GuiBase {
	
	private long skinReadyAt = Long.MAX_VALUE;
	private final VersionNumber jalbumVersion = new VersionNumber(AlbumBean.getInternalVersion());
	private final SkinProperties skinProps = new SkinProperties(skinDirectory);
	private final String skinVersion = skinProps.getProperty(SkinProperties.VERSION);
	private final String skinAuthor = skinProps.getProperty(SkinProperties.AUTHOR);
	private final String supportForum = skinProps.getProperty(SkinProperties.SUPPORT_FORUM);
	private final String helpRoot = "http://jalbum.net/help/en/Skin/Split";
	
	private final Border emptyBorder = BorderFactory.createEmptyBorder();
	private final String tabStyle = "style='padding:3px 2px;margin:0'";
	private Font mono = new Font("monospaced", Font.PLAIN, 12);
	
	private static Icon icon(String basename) {
		return Icons.get(Lazaicon.class, "svg/" + basename + ".svg", 18);
	}
		
	private static Icon icon(String basename, int size) {
		return Icons.get(Lazaicon.class, "svg/" + basename + ".svg", size);
	}
	
	private static Icon svgIcon(String basename, Dimension dim) {
		return svgIcon(basename, dim, false);
	}
	
	private static Icon svgIcon(String basename, Dimension dim, boolean adapt) {
		DeferredSVGIcon icon = new DeferredSVGIcon(Gui.class, "graphics/" + basename + ".svg");
		
		icon.setAntiAlias(true);
		icon.setAdaptColors(adapt);
		icon.setPreferredSize(dim);
		icon.setAutosize(SVGIcon.AUTOSIZE_STRETCH);
		
		return icon;
	}
	
	private Object[] getPosition() {
		return new Object[] {
			new Item("left top", getText("ui.left_top")), 
			new Item("center top", getText("ui.center_top")), 
			new Item("right top", getText("ui.right_top")), 
			new Item("left center", getText("ui.left_middle")), 
			new Item("center center", getText("ui.center_middle")), 
			new Item("right center", getText("ui.right_middle")), 
			new Item("left bottom", getText("ui.left_bottom")), 
			new Item("center bottom", getText("ui.center_bottom")), 
			new Item("right bottom", getText("ui.right_bottom"))
		};
	}
	
	private void allowHTMLEditing(JSmartTextArea ta) {
		try {
			ta.setAllowHTMLEditing(true).setFullHTMLEditing(true);
		} catch (Throwable e) {
		}
	}
	
	private void allowSpelling(JSmartTextArea ta) {
		try {
			ta.spelling();
		} catch (Throwable e) {
		}
	}
	
	private String majorVersion(String s) {
		int i = s.indexOf(".");
		if (i > 0) {
			return s.substring(0, i);
		}
		return s;
	}
	
	private int getMajorVersion(Object v) {
		
		if (v == null) {
			return -1;
		} else if (v instanceof Integer) {
			return (int)v;
		} else if (v instanceof String) {
			try {
				return Integer.valueOf((String)v);
			} catch(NumberFormatException ex) {
				return -1;
			}
		} else if (v instanceof Double) {
			return (int)Math.floor((Double)v);
		}
		return -1;
	}

	Boolean checkBooleanProperty(AlbumObject[] sel, String pn) {
		
		if (sel.length > 1) {
			boolean v = sel[0].getProperties().get(pn, false);
			
			for (AlbumObject ao : sel) {
				if (ao.getProperties().get(pn, false) != v) {
					return null;
				}
			}
			return v;
		} else if (sel.length == 1) {
			return sel[0].getProperties().get(pn, false);
		} else {
			return null;
		}
	}
	
	void createMenuItemForBooleanProperty(JMenu menu, String pn, String title) {
		
		createMenuItemForBooleanProperty(menu, window.albumExplorer.explorer.getSelectedAlbumObjects(), pn, title);
	}
	
	void createMenuItemForBooleanProperty(JMenu menu, AlbumObject[] sel, String pn, String title) {		
		Boolean cv = checkBooleanProperty(sel, pn);

		try {
			if (cv == null) {
				// Indefinite value -> Add submenu
				JMenu subMenu = new JMenu(title);
				JMenuItem all = new JMenuItem(getText("ui.all"));
				all.addActionListener(ae -> {
					for (AlbumObject ao : sel) {
						ao.getProperties().put(pn, true);
						ao.getProperties().save();
					}				
				});
				JMenuItem none = new JMenuItem(getText("ui.none"));
				none.addActionListener(ae -> {
					for (AlbumObject ao : sel) {
						ao.getProperties().put(pn, false);
						ao.getProperties().save();
					}				
				});
				subMenu.add(all);
				subMenu.add(none);
				menu.add(subMenu);
	
			} else {
				// All items has the same value -> invert them when clicked
				JCheckBoxMenuItem menuItem = new JCheckBoxMenuItem(title, null, cv);
	
				menuItem.addActionListener(ae -> {
					for (AlbumObject ao : sel) {
						ao.getProperties().put(pn, !cv);
						ao.getProperties().save();
					}
				});
	
				menu.add(menuItem);
			}
			
		} catch (Throwable ex) {
			JAlbum.logger.log(Level.INFO, "Right-click menu is available only in jAlbum v32 and newer!");
		}
	}
	
	private boolean isSkinReady() {
		// Within 1s of launch
		//System.out.println("skinReady()? " + (System.currentTimeMillis() - skinReadyAt));
		return (System.currentTimeMillis() - skinReadyAt) > 1000L;
	}
	
	private String getStyleName() {
		String style = engine.getStyle();
		return style.substring(0, style.indexOf("."));
	}
	
	private String getSkinName() {
		return engine.getSkin();
	}
	
	private String getLastSkinName() {
		String ls = null;
		
		try {
			ls = engine.getLastSkin();
		} catch(Throwable t) {
			JAlbum.logger.log(Level.FINER, "Last skin is unknown.");
		}
		
		return ls;
	}	
	
	private final String unCamelCase(String s) {
		
		if (s.length() > 0) {
			s = s.replaceAll("([a-z])([A-Z]+)", "$1 $2");
			return Character.toUpperCase(s.charAt(0)) + s.substring(1);
		}
		
		return "";
	}

	private final String getText(String t) {
		String s;
		
		try {
			s = texts.getString(t);
		} catch (java.util.MissingResourceException e) {
			JAlbum.logger.log(Level.FINE, "Missing text: \"{0}\"", t);
			if (t.startsWith("ui.")) {
				t = t.substring(3);
			}
			s = unCamelCase(t);
		}
		
		return s;
	}
	
	private final String[] imageFiles = new String[] { "jpg", "jpeg", "png", "gif", "svg" };
	private final FileChooser fc = ChooserFactory.createFileChooser(window);
	
	private void getFileToRes(String[] extensions, JTextField name, Component c) {
		fc.setFileFilter(new FileNameExtensionFilter(getText("ui.select"), extensions));
		int returnVal = fc.showOpenDialog(c);
		
		if (returnVal == JFileChooser.APPROVE_OPTION) {
			String fn = fc.getSelectedFile().toString();
			
			if (!fn.trim().equals("")) {
				File src = new File(fn);
				File dst = new File(context.getPluginContext().getRootFolder().getFile(), "res");
				if (!dst.exists()) {
					dst.mkdir();
				}
				if (src.exists() &&  dst.exists()) {
					try {
						IO.copyFile(src, dst);
						name.setText(src.getName());
					} catch (IOException e) {
						System.out.println(e);
					}
				}
			}
		}
	}
			
	private String getFileContents(File f) {
		StringBuilder cont = new StringBuilder();
		String line;
		String nl = System.getProperty("line.separator");
		
		if (f.exists()) {
			try {
				try (java.io.BufferedReader in = new java.io.BufferedReader(new java.io.FileReader(f))) {
					while ((line = in.readLine()) != null) {
						cont.append(line);
						cont.append(nl);
					}
				}
			} catch (FileNotFoundException e) {
				return null;
			} catch (IOException e) {
				return null;
			}
		}
		return cont.toString();
	}
		
	private int setSelectedValue(JComboBox<Item> box, Object val) {
		int		size = box.getItemCount();
		Item	it;
		
		for (int i = 0; i < size; i++) {
			it = box.getItemAt(i);
			if (it.item.equals(val)) {
				box.setSelectedIndex(i);
				return i;
			}
		}
		
		return -1;
	}
	
	private String getSelectedValue(Object o) {
		if (o instanceof JComboBox) {
			Object so = ((JComboBox)o).getSelectedItem();
			if (so instanceof Item) {
				return ((Item)so).value.toString();
			}
			return (so == null)? null : so.toString();
		}
		return (o == null)? null : o.toString();
	}
	
	/*****************************************
	 *
	 *			Skin UI starts here
	 *
	 *****************************************/

	ControlPanel skinUi = new ControlPanel() {
		
		// Icons used multple places
		private Icon infoIcon = icon("info");
		private Icon mandatory = svgIcon("asterisk", new Dimension(16, 16));
		
		public JTextField themeImageSize = new JTextField("1600x1600");
		
		// styleName is stored as skin variable in order to check style change later
		private JTextField styleName = new JTextField(getStyleName());
		
		// skinVersion is stored as skin variable in order to check major skin version change later
		JTextField majorSkinVersion = new JTextField(majorVersion(skinVersion));
		
		// Typography
		
		JComboBox<Item> fontFamily = new JComboBox<>(new Item[]{
			new Item("Arial, \"Helvetica Neue\", Helvetica, sans-serif", "Arial"),
			new Item("Baskerville, \"Baskerville Old Face\", \"Hoefler Text\", Garamond, \"Times New Roman\", serif", "Baskerville"),
			new Item("Calibri, Candara, Segoe, \"Segoe UI\", Optima, Arial, sans-serif", "Calibri"),
			new Item("Cambria, Georgia, Times, \"Times New Roman\", serif", "Cambria"),
			new Item("\"Century Gothic\", \"Apple Gothic\", \"Goudy Old Style\", sans-serif", "Century Gothic"),
			new Item("\"Comic Sans MS\", cursive", "Comic Sans"),
			new Item("Consolas, \"Lucida Console\", Monaco, monospace", "Consolas"),
			new Item("Constantia, Palatino, \"Palatino Linotype\", \"Palatino LT STD\", Georgia, serif", "Constantia"),
			new Item("\"Copperplate Light\", \"Copperplate Gothic Light\", serif", "Copperplate Light"),
			new Item("\"Courier New\", Courier, monospace", "Courier New"),
			new Item("\"Franklin Gothic Medium\", \"Arial Narrow Bold\", Arial, sans-serif", "Franklin Gothic"),
			new Item("Futura, \"Century Gothic\", AppleGothic, sans-serif", "Futura"),
			new Item("Garamond, \"Hoefler Text\", Times New Roman, Times, serif", "Garamond"),
			new Item("Geneva, \"Lucida Sans\", \"Lucida Grande\", \"Lucida Sans Unicode\", Verdana, sans-serif", "Geneva"),
			new Item("Georgia, Palatino, \"Palatino Linotype\", Times, \"Times New Roman\", serif", "Georgia"),
			new Item("\"Gill Sans\", \"Gill Sans MT\", Calibri, \"Trebuchet MS\", sans-serif", "Gill Sans"),
			new Item("\"Goudy Old Style\", Garamond, \"Big Caslon\", \"Times New Roman\", serif", "Goudy Old Style"),
			new Item("\"Helvetica Neue\", Helvetica, Arial, sans-serif", "Helvetica Neue"),
			new Item("\"Hoefler Text\", Constantia, Palatino, \"Palatino Linotype\", \"Book Antiqua\", Georgia, serif", "Hoefler Text"),
			new Item("Impact, Haettenschweiler, \"Arial Narrow Bold\", sans-serif", "Impact"),
			new Item("\"Lucida Sans\", \"Lucida Grande\", \"Lucida Sans Unicode\", Verdana, sans-serif", "Lucida Sans"),
			new Item("\"Lucida Bright\", Georgia, serif", "Lucida Bright"),
			new Item("Palatino, \"Palatino Linotype\", \"Book Antiqua\", Georgia, serif", "Palatino"),
			new Item("Segoe, \"Segoe UI\", Tahoma, Geneva, \"Nimbus Sans L\", sans-serif", "Segoe"),
			new Item("Tahoma, Geneva, Verdana, sans-serif", "Tahoma"),
			new Item("Times, \"Times New Roman\", Georgia, serif", "Times"),
			new Item("\"Trebuchet MS\", \"Lucida Sans Unicode\", \"Lucida Grande\", \"Lucida Sans\", Tahoma, sans-serif", "Trebuchet MS"),
			new Item("Verdana, Geneva, Tahoma, sans-serif", "Verdana"),
			new Item("", "[   Google Fonts   ]"),
			new Item("_Barlow", "Barlow"),
			new Item("_Barlow Semi Condensed", "Barlow Semi Condensed"),
			new Item("_Cormorant", "Cormorant"),
			new Item("_Dosis", "Dosis"),
			new Item("_Exo 2", "Exo 2"),
			new Item("_IBM Plex Sans", "IBM Plex Sans"),
			new Item("_IBM Plex Serif", "IBM Plex Serif"),
			new Item("_Josefin Sans", "Josefin Sans"),
			new Item("_Josefin Slab", "Josefin Slab"),
			new Item("_Lato", "Lato"),
			new Item("_League Spartan", "League Spartan"),
			new Item("_Montserrat", "Montserrat"),
			new Item("_Noto Serif", "Noto Serif"),
			new Item("_Open Sans", "Open Sans"),
			new Item("_Oswald", "Oswald"),
			new Item("_Quicksand", "Quicksand"),
			new Item("_Raleway", "Raleway"),
			new Item("_Roboto", "Roboto"),
			new Item("_Roboto Condensed", "Roboto Condensed"),
			new Item("_Roboto Slab", "Roboto Slab"),
			new Item("_Saira", "Saira"),
			new Item("_Sofia Sans", "Sofia Sans"),
			new Item("_Sofia Sans Condensed", "Sofia Sans Condensed"),
			new Item("_Source Sans 3", "Source Sans 3"),
			new Item("_Source Serif 4", "Source Serif 4"),
			new Item("_Ubuntu", "Ubuntu"),
			new Item("_Yanone Kaffeesatz", "Yanone Kaffeesatz"),
			new Item("_Work Sans", "Work Sans"),
			new Item("_Zilla Slab", "Zilla Slab")
		});
		JComboBox<Item> fontWeight = new JComboBox<>(new Item[] {
			new Item("100", "Thin"),
			new Item("200", "Extra Light"),
			new Item("300", "Light"),
			new Item("400", "Regular"),
			new Item("500", "Medium"),
			new Item("600", "Semi Bold"),
			new Item("700", "Bold"),
			new Item("800", "Extra Bold"),
			new Item("900", "Black")
		});
		JComboBox headlineFontFamily = new JComboBox(new Object[] {
			new Item("", "[   " + getText("ui.sameAsBaseFont") + "   ]"),
			new Item("Abril Fatface", "Abril Fatface"),
			new Item("Amatic SC:700", "Amatic SC Bold"),
			new Item("Arapey", "Arapey"),
			new Item("Barlow:300", "Barlow Light"),
			new Item("Barlow:500", "Barlow Medium"),
			new Item("Barlow Semi Condensed:400", "Barlow Semi Condensed"),
			new Item("Barlow Condensed", "Barlow Condensed"),
			new Item("Cantata One", "Cantata One"),
			new Item("Cinzel Decorative", "Cinzel Decorative"),
			new Item("Cormorant", "Cormorant"),
			new Item("Dancing Script", "Dancing Script"),
			new Item("Economica", "Economica"),
			new Item("Exo 2:300", "Exo 2"),
			new Item("Fauna One", "Fauna One"),
			new Item("Fjalla One", "Fjalla One"),
			new Item("Geo", "Geo"),
			new Item("Gilda Display", "Gilda Display"),
			new Item("Grand Hotel", "Grand Hotel"),
			new Item("Gruppo", "Gruppo"),
			new Item("Handlee", "Handlee"),
			new Item("IBM Plex Serif:300", "IBM Plex Serif"),
			new Item("Josefin Slab", "Josefin Slab"),
			new Item("Julius Sans One", "Julius Sans One"),
			new Item("Jura", "Jura"),
			new Item("Libre Baskerville", "Libre Baskerville"),
			new Item("Lobster", "Lobster"),
			new Item("Lobster Two", "Lobster Two"),
			new Item("Lora", "Lora"),
			new Item("Martel:300", "Martel Light"),
			new Item("Marvel:700", "Marvel Bold"),
			new Item("Medula One", "Medula One"),
			new Item("Merriweather:300", "Merriweather"),
			new Item("Noticia Text", "Noticia Text"),
			new Item("Noto Serif Display:300", "Noto Serif Light"),
			new Item("Oranienbaum", "Oranienbaum"),
			new Item("Oswald:300", "Oswald Light"),
			new Item("Philosopher", "Philosopher"),
			new Item("Poiret One", "Poiret One"),
			new Item("Prata", "Prata"),
			new Item("PT Mono", "PT Mono"),
			new Item("Raleway:300", "Raleway"),
			new Item("Raleway:600", "Raleway Bold"),
			new Item("Roboto Condensed", "Roboto Condensed"),
			new Item("Roboto Slab:300", "Roboto Slab"),
			new Item("Roboto Slab:600", "Roboto Slab Bold"),
			new Item("Rochester", "Rochester"),
			new Item("Saira", "Saira"),
			new Item("Scope One", "Scope One"),
			new Item("Smooch Sans", "Smooch Sans"),
			new Item("Sofia", "Sofia"),
			new Item("Sofia Sans Condensed:200", "Sofia Sans Condensed ExtraLight"),
			new Item("Sofia Sans Condensed:600", "Sofia Sans Condensed SemiBold"),
			new Item("Special Elite", "Special Elite"),
			new Item("Squada One", "Squada One"),
			new Item("Strait", "Strait"),
			new Item("Unica One", "Unica One"),
			new Item("Vidaloka", "Vidaloka"),
			new Item("Work Sans", "Work Sans"),
			new Item("Yanone Kaffeesatz:300", "Yanone Kaffeesatz"),
			new Item("Yanone Kaffeesatz:500", "Yanone Kaffeesatz Bold"),
			new Item("Zilla Slab", "Zilla Slab")
		});

		private final JComboBox fontSuggestions = new JComboBox(new Object[] {
			"[" + getText("ui.suggestedFonts") +  "]",
			"Barlow Semi Condensed / Barlow",
			"Cormorant / Zilla Slab",
			"Exo 2 / Exo 2",
			"Barlow Condensed / Roboto",
			"IBM Plex Serif / IBM Plex Sans",
			"Jura / Roboto Condensed",
			"Martel Light / Roboto",
			"Oswald Light / Sofia Sans Condensed",
			"Raleway / Lato",
			"Roboto Condensed / Roboto",
			"Roboto Slab / Source Serif 4",
			"Special Elite / Zilla Slab",
			"Unica One / Barlow Semi Condensed",
			"Yanone Kaffeesatz Bold / Barlow"
		});

		JCheckBox shareFacebook = new JCheckBox("Facebook", false);
		JCheckBox shareThreads = new JCheckBox("Threads", false);
		JCheckBox shareBlueSky = new JCheckBox("BlueSky", false);
		JCheckBox shareTwitter = new JCheckBox("Twitter / X", false);
		JCheckBox shareLinkedIn = new JCheckBox("LinkedIn", false);
		JCheckBox sharePinterest = new JCheckBox("Pinterest", false);
		JCheckBox shareReddit = new JCheckBox("Reddit", false);
		JCheckBox shareEmail = new JCheckBox(getText("ui.email"), false);
		JCheckBox shareLink = new JCheckBox(getText("ui.link"), false);
		
		private void setFontBoxes() {
			
			if (fontSuggestions.getSelectedIndex() == 0) {
				return;
			}
			String	s = fontSuggestions.getSelectedItem().toString(),
					hf = s.split("/")[0].trim(),
					ff = s.split("/")[1].trim();

			setSelectedValue(fontFamily, ff);
			setSelectedValue(headlineFontFamily, hf);
			fontSuggestions.setSelectedIndex(0);
		};
		
		private void setMakeSlides() {
			boolean ov = engine.isSlides(),
					nv = shareFacebook.isSelected();
			
			if (ov != nv) {
				try {
					window.ui2Engine();
					engine.setSlides(nv);
					window.engine2UI();
				} catch (Exception ex) {
					throw new RuntimeException(ex);
				}
			}
		};
		
		private PropertyChangeListener setupMonitors = pce -> {
			
			JAlbum.logger.log(Level.FINE, "Setting up state monitors");
			
			StateMonitor.monitoring(fontSuggestions).onChange(src -> {
				if (isSkinReady() && src != null) {
					setFontBoxes();
				}
			});
			/*
			StateMonitor.monitoring(extraSizes).onChange(src -> {
				extraSizesChanged.setSelected(lastExtraSizes == null || !lastExtraSizes.equals(extraSizes.getText()));
			});
			*/
			
			// Separate slide page requirement
			StateMonitor.monitoring(
					shareFacebook,
					shareTwitter
					/*writeSitemapXml,
					sitemapIncludeSlides*/).onChange(src -> {
				setMakeSlides();
			});
		};
		
		/*	---------------------------------------------------------------
									Site
			--------------------------------------------------------------- */
		
		ControlPanel site = new ControlPanel() {

			JComboBox<Item> language = new JComboBox<Item>() {
				{
					setModel(Languages.modelFrom(new File(context.getSkinDir(), "texts")));
					insertItemAt(new Item("jalbum", "[ " + getText("ui.jalbumPreference") + " ]"), 0);
					setSelectedIndex(0);
				}
			};
			
			// Typography
			
			JSpinner albumFontSize = new JSpinner(new SpinnerNumberModel(16, 12, 20, 1));
		
			ControlPanel typography = new ControlPanel() {
				
				JLinkLabel samplesLink = new JLinkLabel("file://" + skinDirectory.toString().replace("\\", "/") + "/font-samples.html", getText("ui.fontSamples"));
				
				{
					setTitle(getText("ui.typography"));
					
					add("", new JLabelFor(getText("ui.pairingSuggestions"), fontSuggestions));
					add("tab", fontSuggestions);
					add("br", new JLabelFor(getText("ui.font"), fontFamily));
					add("tab", fontFamily);
					add("tab", fontWeight);
					add(" ", albumFontSize);
					add(new JLabel("px"));
					add("br", new JLabelFor(getText("ui.headlineFont"), headlineFontFamily));
					add("tab", headlineFontFamily);
					add("br tab", samplesLink);
				}
			};
			
			JColorSelector sidebarBackgroundColor = new JColorSelector(getText("ui.sidebarBackground"), new JSmartTextField(10));
			JColorSelector sidebarTextColor = new JColorSelector(getText("ui.sidebarTextColor"), new JSmartTextField(10));
			JColorSelector thumbnailsBackgroundColor = new JColorSelector(getText("ui.thumbnailsBackground"), new JSmartTextField(10));
			JColorSelector thumbnailsTextColor = new JColorSelector(getText("ui.thumbnailsText"), new JSmartTextField(10));
			JColorSelector lightboxBackgroundColor = new JColorSelector(getText("ui.lightboxBackground"), new JSmartTextField(10));
			JColorSelector lightboxTextColor = new JColorSelector(getText("ui.lightboxText"), new JSmartTextField(10));
			JColorSelector buttonBackgroundColor = new JColorSelector(getText("ui.buttonColor"), new JSmartTextField(10));
			JColorSelector buttonTextColor = new JColorSelector(getText("ui.buttonText"), new JSmartTextField(10));
			
			ControlPanel design = new ControlPanel() {
				
				{ 
					setTitle(getText("ui.design"));
					
					sidebarBackgroundColor.getTextComponent().setFont(mono);
					sidebarTextColor.getTextComponent().setFont(mono);
					thumbnailsBackgroundColor.getTextComponent().setFont(mono);
					thumbnailsTextColor.getTextComponent().setFont(mono);
					lightboxBackgroundColor.getTextComponent().setFont(mono);
					lightboxTextColor.getTextComponent().setFont(mono);
					buttonBackgroundColor.getTextComponent().setFont(mono);
					buttonTextColor.getTextComponent().setFont(mono);

					add("", new JLabelFor(getText("ui.sidebarBackground"), sidebarBackgroundColor));
					add("tab", sidebarBackgroundColor);
					add("tab", sidebarBackgroundColor.getTextComponent());
					add("tab", new JLabelFor(getText("ui.sidebarTextColor"), sidebarTextColor));
					add("tab", sidebarTextColor);
					add("tab", sidebarTextColor.getTextComponent());
					add("br", new JLabelFor(getText("ui.thumbnailsBackground"), thumbnailsBackgroundColor));
					add("tab", thumbnailsBackgroundColor);
					add("tab", thumbnailsBackgroundColor.getTextComponent());
					add("tab", new JLabelFor(getText("ui.thumbnailsText"), thumbnailsTextColor));
					add("tab", thumbnailsTextColor);
					add("tab", thumbnailsTextColor.getTextComponent());
					add("br", new JLabelFor(getText("ui.lightboxBackground"), lightboxBackgroundColor));
					add("tab", lightboxBackgroundColor);
					add("tab", lightboxBackgroundColor.getTextComponent());
					add("tab", new JLabelFor(getText("ui.lightboxText"), lightboxTextColor));
					add("tab", lightboxTextColor);
					add("tab", lightboxTextColor.getTextComponent());
					add("br", new JLabelFor(getText("ui.buttonColor"), buttonBackgroundColor));
					add("tab", buttonBackgroundColor);
					add("tab", buttonBackgroundColor.getTextComponent());
					add("tab", new JLabelFor(getText("ui.buttonText"), buttonTextColor));
					add("tab", buttonTextColor);
					add("tab", buttonTextColor.getTextComponent());
				}
			};
			
			{
				add("", new JLabel(getText("ui.language")));
				add("tab", language);
				add("br hfill", typography);
				add("br hfill", design);
			}
		};
			
		/*	---------------------------------------------------------------
									Sidebar
			--------------------------------------------------------------- */

		ControlPanel sidebar = new ControlPanel() {
			
			ControlPanel navigationPanel = new ControlPanel() {
				
				JComboBox<Item> navigationType = new JComboBox<>(new Item[] {
					new Item("subfolders", getText("ui.subfoldersOnly")),
					new Item("tree", getText("ui.folderTree")),
					new Item("collapsible", getText("ui.collapsibleTree")),
				});
				
				{
					setTitle(getText("ui.navigation"));
					
					add(new JLabel(getText("ui.type")));
					add("tab", navigationType);
				}
			};
			
			ControlPanel sharingPanel = new ControlPanel() {
				{
					setTitle(getText("ui.sharing"));
					
					add(shareFacebook);
					add(shareThreads);
					add(shareBlueSky);
					add(shareTwitter);
					add(shareLinkedIn);
					add("br", sharePinterest);
					add(shareReddit);
					add(shareEmail);
					add(shareLink);
				}
			};
			
			{
				add("hfill", navigationPanel);
				add("br hfill", sharingPanel);
			}
		};
		
		/*	---------------------------------------------------------------
									Thumbnails
			--------------------------------------------------------------- */
		
		ControlPanel thumbnails = new ControlPanel() {
			
			JComboBox<Item> thumbGap = new JComboBox<>(new Item[] {
				new Item("", getText("ui.none")),
				new Item("small", getText("ui.small")),
				new Item("medium", getText("ui.medium")),
				new Item("large", getText("ui.large"))
			});
			
			JComboBox<Item> markActualThumb = new JComboBox<>(new Item[] {
				new Item("", getText("ui.none")),
				new Item("zoomin", getText("ui.zoomIn")),
				new Item("zoomout", getText("ui.zoomOut")),
				new Item("border", getText("ui.border"))
			});
			
			JComboBox<Item> markVisitedThumbs = new JComboBox<>(new Item[] {
				new Item("", getText("ui.none")),
				new Item("grayscale", getText("ui.grayscale")),
				new Item("fade", getText("ui.fade"))
			});
			
			/*
			JComboBox<Item> thumbnailsPadding = new JComboBox<>(new Item[] {
				new Item("minimal", getText("ui.minimal")),
				new Item("small", getText("ui.small")),
				new Item("large", getText("ui.large"))
			});
			*/
			{
				add(new JLabel(getText("ui.gapBetweenThumbnails")));
				add("tab", thumbGap);
				add("br", new JLabel(getText("ui.markActualThumbnail")));
				add("tab", markActualThumb);
				add("br", new JLabel(getText("ui.markVisitedThumbnails")));
				add("tab", markVisitedThumbs);
			}
		};
		
		/*	---------------------------------------------------------------
									Lightbox
			--------------------------------------------------------------- */

		ControlPanel lightbox = new ControlPanel() {
			
			JSpinner transitionLength = new JSpinner(new SpinnerNumberModel(500, 100, 2000, 100));
			JSpinner slideshowDelay = new JSpinner(new SpinnerNumberModel(3000, 500, 9500, 500));
			JLabel slideshowDelayLabel = new JLabel(getText("ui.slideshowInterval"));
			//JLabel slideshowDelayMs = new JLabel("ms");
			JCheckBox autoplayVideos = new JCheckBox(getText("ui.autoStartVideos"), false);
			JCheckBox rightClickProtect = new JCheckBox(getText("ui.rightClickProtect"), false);
			
			JComboBox<Item> lightboxBackground = new JComboBox<>(new Item[] {
				new Item("", getText("ui.empty")),
				new Item("firstImage", getText("ui.firstImage")),
				new Item("themeImage", getText("ui.themeImage"))
			});
			JCheckBox showTitle = new JCheckBox(getText("ui.showTitleAndDescription"), true);
			JCheckBox showDateRange = new JCheckBox(getText("ui.showDateRange"), false);
			JCheckBox showItemCount = new JCheckBox(getText("ui.showItemCount"), false);
			JCheckBox showStartSlideshow = new JCheckBox(getText("ui.showStartSlideshow"), true);
			
			JDraggableList lightboxItems = new JDraggableList(new Object[] {
				new Item("fileTitle", getText("ui.title")),
				new Item("label", getText("ui.baseName")),
				new Item("fileName", getText("ui.fileName")),
				new Item("comment", getText("ui.comment")),
				new Item("keywords", getText("ui.keywords")),
				new Item("photodata", getText("ui.photodata"))
			}, new String[] { "fileTitle", "comment" } );
			JCheckBox lightboxTitleFirstFound = new JCheckBox(getText("ui.displayFirstFound"), true);
			JCheckBox captionCanBePlacedBeside = new JCheckBox(getText("ui.canBePlacedBeside"), false);
				
			JCheckBox showMapBtn = new JCheckBox(getText("ui.showMap"), false);
			JCheckBox showDownloadBtn = new JCheckBox(getText("ui.showDownloadButton"), true);
			JComboBox mouseWheelAction = new JComboBox(new Object[] {
				new Item("", getText("ui.default")), 
				new Item("navigation", getText("ui.navigation")),
				new Item("zoom", getText("ui.zoom"))
			});
			JCheckBox enableKeyboard = new JCheckBox(getText("ui.enableKeyboard"), true);
			
			ControlPanel behaviorPanel = new ControlPanel() {
				{
					setTitle(getText("ui.image"));
					rightClickProtect.setToolTipText(getText("ui.rightClickProtectInfo"));
					mouseWheelAction.setToolTipText(getText("ui.mouseWheelActionInfo"));
					enableKeyboard.setToolTipText(getText("ui.enableKeyboardInfo"));
					add(new JLabel(getText("ui.transitionLength")));
					add("tab", transitionLength);
					add(new JLabel("ms"));
					add("br", slideshowDelayLabel);
					add("tab", slideshowDelay);
					add(new JLabel("ms"));
					add("br", autoplayVideos);
					add("br", rightClickProtect);
					add("br", enableKeyboard);
					add("br", new JLabelFor(getText("ui.mouseWheelAction"), mouseWheelAction));
					add("tab", mouseWheelAction);
				}
			};
			
			ControlPanel titlePanel = new ControlPanel() {
				{
					setTitle(getText("ui.titlePage"));
					add(new JLabel(getText("ui.initialBackground")));
					add("tab", lightboxBackground);
					add("br", showTitle);
					add("br", showItemCount);
					add("br", showDateRange);
					add("br", showStartSlideshow);
					
				}
			};

			ControlPanel captionPanel = new ControlPanel() {
				
				{
					setTitle(getText("ui.caption"));
					lightboxTitleFirstFound.setToolTipText(getText("ui.displayFirstInfo"));
					add(new JLabel(getText("ui.available")));
					add("tab", new JLabel(getText("ui.used")));
					add("br", new JScrollPane(lightboxItems.getSecondaryList()));
					add("tab", new JScrollPane(lightboxItems));
					add("br", lightboxTitleFirstFound);
					add("br", captionCanBePlacedBeside);
				}
			};
			
			ControlPanel buttonsPanel = new ControlPanel() {
				{
					setTitle(getText("ui.buttons"));
					showMapBtn.setToolTipText(getText("ui.mapInfo"));
					showDownloadBtn.setToolTipText(getText("ui.downloadInfo"));
					
					add("", showMapBtn);
					add("br", showDownloadBtn);
				}
			};
			
			ControlPanel leftPanel = new ControlPanel() {
			
				{
					((RiverLayout)(getLayout())).setVgap(0);
					((RiverLayout)(getLayout())).setHgap(0);
					
					add("hfill", titlePanel);
					add("br hfill", behaviorPanel);
				}
			};
			
			ControlPanel rightPanel = new ControlPanel() {
				
				{
					((RiverLayout)(getLayout())).setVgap(0);
					((RiverLayout)(getLayout())).setHgap(0);
					
					add("hfill", captionPanel);
					add("br hfill", buttonsPanel);
					
				}
				
			};
			
			{
				add("vtop", leftPanel);
				add("tab hfill", rightPanel);
			}
		};
		
		/*	---------------------------------------------------------------
									Advanced
			--------------------------------------------------------------- */
		
		//	Custom code
		
		ControlPanel advanced = new ControlPanel() {

			ControlPanel customKeysPanel = new ControlPanel() {

				JTextArea customKeys = new JSmartTextArea(7, 20);
				JScrollPane customKeysPane = new JScrollPane(customKeys);

				{
					customKeys.setEditable(true);
					customKeys.setLineWrap(false);
					customKeys.setFont(mono);
					customKeys.setTabSize(2);
					customKeysPane.setBorder(BorderFactory.createTitledBorder(getText("ui.customKeys")));

					add("hfill vfill", customKeysPane);
					add("br", new JLabel(getText("ui.customKeysInfo")));
				}
			};
			
			ControlPanel headHookTab = new ControlPanel() {

				JTextArea headHook = new JSmartTextArea(7, 20);
				JScrollPane headHookPane = new JScrollPane(headHook);

				{
					headHook.setEditable(true);
					headHook.setLineWrap(false);
					headHook.setFont(mono);
					headHook.setTabSize(2);
					headHookPane.setBorder(BorderFactory.createTitledBorder(getText("ui.headText")));

					add("hfill vfill", headHookPane);
				}
			};

			ControlPanel bodyHookTab = new ControlPanel() {

				JTextArea bodyHook = new JSmartTextArea(7, 20);
				JScrollPane bodyHookPane = new JScrollPane(bodyHook);

				{
					bodyHook.setEditable(true);
					bodyHook.setLineWrap(false);
					bodyHook.setFont(mono);
					bodyHook.setTabSize(2);
					bodyHookPane.setBorder(BorderFactory.createTitledBorder(getText("ui.bodyText")));

					add("hfill vfill", bodyHookPane);
				}
			};

			ControlPanel cssHookTab = new ControlPanel() {

				JTextArea cssHook = new JSmartTextArea(15, 20);
				JScrollPane cssHookPane = new JScrollPane(cssHook);
				WrappableJLabel info = new WrappableJLabel("<html><i>" + getText("ui.cssText") + "</i></html>");
				
				{
					cssHook.setEditable(true);
					cssHook.setLineWrap(false);
					cssHook.setFont(mono);
					cssHook.setTabSize(2);
					cssHookPane.setBorder(BorderFactory.createTitledBorder(getText("ui.cssTitle")));
					info.setPreferredWidth(700);
					
					add("hfill vfill", cssHookPane);
					add("br", info);
				}
			};

			ControlPanel jsHookTab = new ControlPanel() {

				JTextArea jsHook = new JSmartTextArea(15, 20);
				JScrollPane jsHookPane = new JScrollPane(jsHook);
				WrappableJLabel info = new WrappableJLabel("<html><i>" + getText("ui.javascriptText") + "</i></html>");
				
				{
					jsHook.setEditable(true);
					jsHook.setLineWrap(false);
					jsHook.setFont(mono);
					jsHook.setTabSize(2);
					jsHookPane.setBorder(BorderFactory.createTitledBorder(getText("ui.javascriptTitle")));
					info.setPreferredWidth(700);
					
					add("hfill vfill", jsHookPane);
					add("br", info);
				}
			};

			JTabbedPane customCodeTabs = new JTabbedPane() {				

				{
					addTab(getText("ui.customKeys"), icon("rename"), customKeysPanel);
					addTab("<HEAD>", icon("code"), headHookTab);
					addTab("<BODY>", icon("code"), bodyHookTab);
					addTab("CSS", icon("css"), cssHookTab);
					addTab("JavaScript", icon("javascript"), jsHookTab);
				}
			};

			{				
				add("hfill vfill", customCodeTabs);

				putClientProperty("helpPage", helpRoot + "/Advanced/Custom_code");
			}
		};
				
		/*	---------------------------------------------------------------
								About tab
			--------------------------------------------------------------- */
		
		ControlPanel about = new ControlPanel(new BorderLayout(20, 0)) {
			private final JTextArea readme = new JSmartTextArea(getFileContents(new File(skinDirectory, "readme.txt")), 20, 30);
			private final JScrollPane readmePane = new JScrollPane(readme, JScrollPane.VERTICAL_SCROLLBAR_ALWAYS, JScrollPane.HORIZONTAL_SCROLLBAR_NEVER);
			
			ControlPanel info = new ControlPanel() {
				
				{
					add("center", new JLabel(svgIcon("split-logo", new Dimension(128, 128), false)));
					add("br center", new JLabel("<html><h2>" + skin + "</h2></html>"));
					add("br center", new JLabel("Jalbum " + internalVersion));
					add(new JLinkLabel("https://jalbum.net/software/download", getText("ui.upgrade"), getText("ui.downloadJalbum")));
					add("br center", new JLabel(skin + " skin " + skinVersion));
					add(new JLinkLabel("https://jalbum.net/skins/skin/" + skin, getText("ui.upgrade"), getText("ui.downloadSkin")));
					//add("br center", new JLinkLabel(helpRoot, getText("help")));
					add("br center", new JLinkLabel(supportForum, getText("ui.support")));
				}
			};
							
			{
				((BorderLayout)(getLayout())).setVgap(0);
				((BorderLayout)(getLayout())).setHgap(0);
				
				readme.setLineWrap(true);
				readme.setWrapStyleWord(true);
				readme.setEditable(false);
				readme.setFont(mono);
					
				info.setPreferredSize(new Dimension(240, 400));
				readmePane.setPreferredSize(new Dimension(400, 320));
				
				add(info, BorderLayout.WEST);
				add(readmePane);
				
			}
		};

		/*	---------------------------------------------------------------
								Main tabs
			--------------------------------------------------------------- */
		
		JTabbedPane tabs = new JTabbedPane() {
			
			{
				site.setBorder(BorderFactory.createEmptyBorder(10, 10, 10, 10));
				sidebar.setBorder(BorderFactory.createEmptyBorder(10, 10, 10, 10));
				thumbnails.setBorder(BorderFactory.createEmptyBorder(10, 10, 10, 10));
				lightbox.setBorder(BorderFactory.createEmptyBorder(10, 10, 10, 10));
				advanced.setBorder(BorderFactory.createEmptyBorder(10, 10, 10, 10));
				about.setBorder(BorderFactory.createEmptyBorder(10, 10, 10, 10));
				
				addTab("<html><h4 " + tabStyle + ">" + getText("ui.site") + "</h4></html>", icon("settings"), site);
				addTab("<html><h4 " + tabStyle + ">" + getText("ui.sidebar") + "</h4></html>", icon("sidebar-left"), sidebar);
				addTab("<html><h4 " + tabStyle + ">" + getText("ui.thumbnails") + "</h4></html>", icon("thumbnails"), thumbnails);
				addTab("<html><h4 " + tabStyle + ">" + getText("ui.lightbox") + "</h4></html>", icon("lightbox"), lightbox);
				addTab("<html><h4 " + tabStyle + ">" + getText("ui.advanced") + "</h4></html>", icon("wrench"), advanced);
				addTab("<html><h4 " + tabStyle + ">" + getText("ui.about") + "</h4></html>", icon("skin"), about);
				
			}
		};

			
		{
			// Adding UI tabs
			
			((RiverLayout)(getLayout())).setVgap(0);
			((RiverLayout)(getLayout())).setHgap(0);
			tabs.setBorder(emptyBorder);
			
			add("vfill", tabs);
			
			/*	-----------------------------------------------------------
										Listeners
				----------------------------------------------------------- */

			window.addJAlbumListener(new JAlbumAdapter() {
				@Override
				public void styleChanged(JAlbumEvent e) {
					styleName.setText(getStyleName());
				}
			});

			window.addPropertyChangeListener(JAlbumFrame.SKIN_LOADED_PROPERTY, setupMonitors);
			
			window.addJAlbumListener(new JAlbumAdapter() {
				@Override
				public void skinChanged(JAlbumEvent e) {
					//JAlbum.logger.log(Level.FINE, "Skin changed, removing listeners.");
					window.removePropertyChangeListener(JAlbumFrame.SKIN_LOADED_PROPERTY, setupMonitors);
				}
			});
				
			try {
				window.albumExplorer.onContextMenu(menu -> {
					AlbumObject[] sel = window.albumExplorer.explorer.getSelectedAlbumObjects();

					if (sel.length > 0) {
												
						//  Hide location selector
						createMenuItemForBooleanProperty(menu, sel, "hideLocation", getText("ui.hideLocation"));
					}
				});
			} catch (Throwable ex) {
				JAlbum.logger.log(Level.INFO, "Right-click extension menu is available only in jAlbum v32 and newer!");
			};
			
			putClientProperty("helpPage", helpRoot);
			
		}
		
	};
	
	public Gui() {
		this(JAlbumContext.getInstance());
	}
   
	public Gui(JAlbumContext context) {
		
		super(context);
		PluginContext pc = context.getPluginContext();
		//EditPanel editPanel = pc.getEditPanel();
			
		skinUi.setBorder(emptyBorder);
		((RiverLayout)(getLayout())).setVgap(0);
		((RiverLayout)(getLayout())).setHgap(0);
		
		window.setSkinUI(skinUi);		
		skinReadyAt = System.currentTimeMillis();
		
	}

}
		
