	/************************************************
	 *				Common utils
	 ************************************************/
	 
	// Touch event names by browser sniffing 
	const TOUCH_EVENT = (() => {
			if (/Trident|Edge/.test(navigator.userAgent)) {
				// Setting MS events
				if (window.navigator.pointerEnabled) {
					return {
						'START': 	'pointerdown',
						'MOVE':		'pointermove',
						'END':		'pointerup',
						'CANCEL':	'pointercancel'
					};
				}
				return {
					'START': 	'MSPointerDown',
					'MOVE':		'MSPointerMove',
					'END':		'MSPointerUp',
					'CANCEL':	'MSPointerCancel'
				};
			}
			return {
				'START': 	'touchstart',
				'MOVE':		'touchmove',
				'END':		'touchend',
				'CANCEL':	'touchcancel'
			};
		})();
		
	const playVideo = (el, doneFn) => {
		if (el && el.nodeName === 'VIDEO') {
			if (el.paused) {
				let playPromise = el.play();
				if (typeof playPromise !== 'undefine') {
					playPromise.then(() => {
							// Plays immediately
							addClass(el, 'playing');
							if (typeof doneFn === 'function') {
								doneFn.call(null);
							}
						}).catch(err => {
							if (err.name === 'NotAllowedError') {
								// Trying muted
								el.muted = true;
								el.play().then(() => {
										// Works
										console.log('Falling back to muted autoplay! Unmute the video manually!');
										addClass(el, 'playing');
										if (typeof doneFn === 'function') {
											doneFn.call(null);
										}
									}).catch(err => {
										// Rejected
										console.log('Autoplay rejected. Try starting the video manually!');
									});
							} else {
								// Other error
								console.log('Video format not supported!');
							}
						});
				}
			} else {
				if (typeof doneFn === 'function') {
					doneFn.call(null);
				}
			}
		}	
	}
		
	// Merging Objects
	const mergeOptions = (defaults, user) => {
		let merged = { ...defaults };
		for (let key in user) {
			if (user.hasOwnProperty(key) && user[key] !== undefined) {
				merged[key] = user[key];
			}
		}
		return merged;
	}
	
	const getMatrix = (el) => {
		const style = window.getComputedStyle(el);
		const transform = style.transform || style.webkitTransform || style.mozTransform;
		
		if (transform && transform !== 'none') {
			if (transform.startsWith('matrix3d')) {
				const mx3d = transform.substring(9, transform.length - 1).split(',').map(s => parseFloat(s.trim()));
				return [mx3d[0], mx3d[1], mx3d[4], mx3d[5], mx3d[12], mx3d[13]];
			} else if (transform.startsWith('matrix')) {
				return transform.substring(7, transform.length - 1).split(',').map(s => parseFloat(s.trim()));
			}
		}
		
		return [1, 0, 0, 1, 0, 0]; // Default matrix
	}
	
	/************************************************
	 *				Thumbnail scroller
	 ************************************************/
	 
	function ThumbScroller(thumbnails, userOptions) {
		const scroller = thumbnails.querySelector('.scroller');
		const prevBtn = thumbnails.querySelector('.prev-page');
		const nextBtn = thumbnails.querySelector('.next-page');
		const originalCards = [...scroller.querySelectorAll('.card')];
		const totalCards = originalCards.length;
		
		let options = mergeOptions({
			onClick: 				false,
			threshold: 				20							// Tell apart click and swipe (> 20px move)
		}, userOptions);
		
		let thumbnailsPadding;
		let thumbnailsFlex;
					
		if (!thumbnails || !scroller || !prevBtn || !nextBtn || totalCards === 0) {
			if (scroller) scroller.innerHTML = '';
			if (prevBtn) prevBtn.disabled = true;
			if (nextBtn) nextBtn.disabled = true;
			return;
		}
	
		let currentPage = 0;
		let itemsPerPage = 1;
		let totalPages = 1;
		let isPortrait = window.matchMedia("(orientation: portrait)").matches;
		let cardDims;
		let cardMargin;
		let cardsPerRow = 0;
		let cardsPerCol = 0;
		let containerWidth = 0;
		let containerHeight = 0;
		let scrollAmount = 0;
			
		// --- Swipe State Variables ---
		let touchStartX = 0;
		let touchStartY = 0;
		let touchMoveX = 0;
		let touchMoveY = 0;
		let isSwiping = false;
		let initialScrollOffset = 0;
		let scrollerBaseTransition = 'transform 0.5s ease-in-out'; // Store base transition
		const swipeThreshold = 50; // Minimum pixels to be considered a swipe
    
    	function updateCardDimensions() {
			const test = originalCards[0].cloneNode(true);
			
			test.style.visibility = 'hidden';
			test.style.position = 'absolute';
			thumbnails.appendChild(test);
			
			const cardStyle = window.getComputedStyle(test);
			
			cardMargin = {
				left:	parseFloat(cardStyle.marginLeft),
				right:	parseFloat(cardStyle.marginRight),
				top:	parseFloat(cardStyle.marginTop),
				bottom:	parseFloat(cardStyle.marginBottom)
			};
			
			cardDims = { 
				width:  test.offsetWidth + cardMargin.left + cardMargin.right,
				height:	test.offsetHeight + cardMargin.top + cardMargin.bottom
			};
			
			thumbnails.removeChild(test);
		}

		function setupPages() {
			
			scroller.innerHTML = '';
			scroller.style.visibility = 'visible';
			
			if (containerWidth <= 0 || containerHeight <= 0) {
				//console.warn("Container dimensions are zero. Cannot calculate layout.");
				totalPages = 0;
				itemsPerPage = 0;
				scroller.innerHTML = '';
				return;
			}
			
			updateCardDimensions();
	
			if (cardDims.width <= 0 || cardDims.height <= 0) {
				//console.warn("Card size is zero, layout might be incorrect.");
				itemsPerPage = Math.max(1, totalCards);
				setTimeout(setupPages, 200);
				return;
			} else {
				if (isPortrait) { 
					cardsPerRow = Math.max(1, Math.floor((containerWidth - 2 * thumbnailsPadding) / cardDims.width));
					cardsPerCol = Math.max(1, Math.round((containerHeight - 2 * thumbnailsPadding) / cardDims.height));
				} else {
					cardsPerRow = Math.max(1, Math.round((containerWidth - 2 * thumbnailsPadding) / cardDims.width));
					cardsPerCol = Math.max(1, Math.floor((containerHeight - 2 * thumbnailsPadding) / cardDims.height));
				}
				itemsPerPage = cardsPerRow * cardsPerCol;
			}
	
			totalPages = Math.ceil(totalCards / itemsPerPage);
			
			if (totalPages < 2) {
				// Adjust rows / cols to the available space if only one page
				if (isPortrait) {
					while (cardsPerCol >= 1) {
						if (Math.ceil(totalCards / cardsPerCol) * cardDims.width > (containerWidth - 2 * thumbnailsPadding)) {
							break;
						}
						cardsPerCol -= 1;
					}
					cardsPerCol += 1;
					cardsPerRow = Math.ceil(totalCards / cardsPerCol);
				} else {
					while (cardsPerRow >= 1) {
						if (Math.ceil(totalCards / cardsPerRow) * cardDims.height > (containerHeight - 2 * thumbnailsPadding)) {
							break;
						}
						cardsPerRow -= 1;
					}
					cardsPerRow += 1;
					cardsPerCol = Math.ceil(totalCards / cardsPerRow);
				}
				itemsPerPage = cardsPerRow * cardsPerCol;
			}
			
			if (isPortrait) {
				containerHeight = cardsPerCol * cardDims.height + 2 * thumbnailsPadding;
				thumbnails.style.height = containerHeight + 'px';
			} else {
				containerWidth = cardsPerRow * cardDims.width + 2 * thumbnailsPadding;
				thumbnails.style.width = containerWidth + 'px';
			}
			
			window.requestAnimationFrame(() => {
				if (isPortrait) {
					thumbnails.style.width = '100%';
					thumbnails.style.height = containerHeight + 'px';
				} else {
					thumbnails.style.width = containerWidth + 'px';
					thumbnails.style.height = '100%';
				}
					
				for (let i = 0; i < totalPages; i++) {
					const page = document.createElement('div'),
						start = i * itemsPerPage,
						end = Math.min(start + itemsPerPage, totalCards),
						rows = Math.ceil((end - start) / cardsPerRow),
						cols = (rows > 1)? cardsPerRow : (end - start);
		
					page.className = 'page';
					
					if (i === 0) {
						page.classList.add('first-page');
					}
					
					if (i === totalPages - 1) {
						page.classList.add('last-page');
					}
					
					if (rows < 2) {
						page.classList.add('single-row');
					}
					
					page.style.width = containerWidth + 'px';
					page.style.height = containerHeight + 'px';
					
					const padding = (totalPages === 1 && rows === 1)?
						Math.round((containerWidth - end * cardDims.width) / 2)
						:
						Math.round((containerWidth - cardsPerRow * cardDims.width) / 2);
						
					page.style.paddingLeft = padding  + 'px';
					page.style.paddingRight = (padding - 2)  + 'px';
					
					for (let j = start; j < end; j++) {
						page.appendChild(originalCards[j]);
					}
					
					scroller.appendChild(page);
				}
				
				gotoActual();
			});
		}
	
		function gotoActual() {
			const i = thumbnails.getIndex() || 0;
			currentPage = Math.floor(i / itemsPerPage);
			updateScroll();
		}
		
		function calculateLayout() {
			const cw = document.body.clientWidth;
			const ch = document.body.clientHeight;
			
			if (containerWidth === cw && containerHeight === ch) {
				return;
			}
				
			const isLarge = (cw >= 640 && ch >= 480);
			isPortrait = cw < ch;
			thumbnailsPadding = isLarge? options.thumbsPadding : Math.round(options.thumbsPadding * 0.75);
			
			if (isPortrait) {
				containerWidth = cw;
				containerHeight = Math.round(ch * (isLarge? 0.25 : 0.3));
				scrollAmount = containerWidth;
			} else {
				containerWidth = Math.round(cw * (isLarge? 0.25 : 0.3));
				containerHeight = ch;
				scrollAmount = containerHeight;
			}
			
			currentPage = 0;
			setupPages();
		}
	
		function updateScroll(animate = true) {
			
			if (scrollAmount <= 0 && currentPage !== 0) {
				if (currentPage === 0) scroller.style.transform = 'translate(0, 0)';
				return;
			}
			const offset = -currentPage * scrollAmount;
			
			if (animate) {
				scroller.style.transition = scrollerBaseTransition;
			} else {
				scroller.style.transition = 'none';
			}

			if (isPortrait) {
				scroller.style.transform = 'translateX(' + offset + 'px)';
			} else {
				scroller.style.transform = 'translateY(' + offset + 'px)';
			}
			
			prevBtn.disabled = currentPage === 0 || totalPages === 0;
			nextBtn.disabled = currentPage >= totalPages - 1 || totalPages <= 1; // Also check totalPages <= 1
		}
	
		function swipeStart(event) {
			if (totalPages <= 1 || event.target.nodeName === 'BUTTON') return; // No swiping if not enough pages
	
			isSwiping = true;
			touchMoveX = touchStartX = event.touches[0].clientX;
			touchMoveY = touchStartY = event.touches[0].clientY;
			initialScrollOffset = -currentPage * scrollAmount; // Current visual offset
			scroller.style.transition = 'none'; // Disable transition for direct drag
		}
	
		function swipeMove(event) {
			if (!isSwiping || totalPages <= 1) return;
	
			touchMoveX = event.touches[0].clientX;
			touchMoveY = event.touches[0].clientY;
	
			const deltaX = touchMoveX - touchStartX;
			const deltaY = touchMoveY - touchStartY;
			let currentDragOffset = 0;
	
			if (isPortrait) {
				// Prioritize horizontal swipe in portrait
				if (Math.abs(deltaX) > Math.abs(deltaY)) {
					event.preventDefault(); // Prevent vertical page scroll
					currentDragOffset = initialScrollOffset + deltaX;
					scroller.style.transform = 'translateX(' + currentDragOffset + 'px)';
				}
			} else {
				// Prioritize vertical swipe in landscape
				if (Math.abs(deltaY) > Math.abs(deltaX)) {
					event.preventDefault(); // Prevent horizontal page scroll (if any)
					currentDragOffset = initialScrollOffset + deltaY;
					scroller.style.transform = 'translateY(' + currentDragOffset + 'px)';
				}
			}
		}
	
		function swipeEnd(event) {
			if (!isSwiping || totalPages <= 1) return;
			
			event.preventDefault();
			
			isSwiping = false;
			scroller.style.transition = scrollerBaseTransition;
	
			const deltaX = touchMoveX - touchStartX;
			const deltaY = touchMoveY - touchStartY;
			let swipeDistance = 0;
	
			if (isPortrait) {
				swipeDistance = deltaX;
			} else {
				swipeDistance = deltaY;
			}
			
			// The current actual position of the scroller is initialScrollOffset + swipeDistance
			const currentVisualOffset = initialScrollOffset + swipeDistance;
	
			// Determine the closest page based on the final position
			let targetPage = Math.round(-currentVisualOffset / scrollAmount);
	
			// If the swipe wasn't strong enough, snap back to original page
			if (Math.abs(swipeDistance) < swipeThreshold) {
				targetPage = currentPage; // Snap back to original current page before swipe attempt
			} else {
				// If swipe was strong enough, determine direction to confirm page change
				if (swipeDistance < -swipeThreshold) { // Swiped left / up
					targetPage = Math.min(currentPage + 1, totalPages - 1);
				} else if (swipeDistance > swipeThreshold) { // Swiped right / down
					targetPage = Math.max(currentPage - 1, 0);
				} else {
					// Not enough to change page, but might have been dragged a bit
					targetPage = Math.round(-currentVisualOffset / scrollAmount); // Re-evaluate if just dragged
				}
			}
	
			currentPage = Math.max(0, Math.min(targetPage, totalPages - 1)); // Clamp page
	
			// It's a click!
			if (Math.abs(swipeDistance) < options.threshold && typeof options.onClick === 'function') {
				options.onClick.call(thumbnails, event);
			}
			
			updateScroll(); // Animate to the new/snapped page
	
			// Reset touch move positions
			touchMoveX = 0;
			touchMoveY = 0;
		}
	
	
		// Click events on the buttons
    	nextBtn.addEventListener('click', () => {
			if (currentPage < totalPages - 1) {
				currentPage++;
				updateScroll(true);
			}
		});
		prevBtn.addEventListener('click', () => {
			if (currentPage > 0) {
				currentPage--;
				updateScroll(true);
			}
		});
	
		// Touch Listeners
		thumbnails.addEventListener('touchstart', swipeStart, { passive: false });
		thumbnails.addEventListener('touchmove', swipeMove, { passive: false });
		thumbnails.addEventListener('touchend', swipeEnd);
		thumbnails.addEventListener('touchcancel', swipeEnd);
		
		let resizeTimer;
		window.addEventListener('resize', () => {
			clearTimeout(resizeTimer);
			resizeTimer = setTimeout(calculateLayout, 250);
		});
		
		calculateLayout();
		
		thumbnails.refresh = () => {
			clearTimeout(resizeTimer);
			calculateLayout();
		}
		
		thumbnails.gotoPage = n => {
			if (totalPages > 1) {
				if (typeof n === 'undefined') {
					const i = thumbnails.getIndex() || 0;
					n = Math.floor(i / itemsPerPage);
				}
				if (currentPage !== n) {
					currentPage = n;
					updateScroll();
				}
			}
		}
		
		thumbnails.itemsPerPage = () => {
			return itemsPerPage;
		}
	}

	/************************************************
	 *				Swipe handler
	 ************************************************/
	 
	function Swipe(element, userOptions) {
		
		if (!element) {
			return;
		}
	
		const swipeTarget = element;
		
		let options = {
			onSwipeStart:			false,
			onSwipedLeft: 			false,						// Event handlers
			onSwipedRight: 			false,
			onSwipedUp: 			false,
			onSwipedDown: 			false,
			onSmallSwipe: 			false,
			onSwipeStart: 			false,
			onPinchZoom: 			false,
			onClick: 				false,
			onDoubleClick: 			false,
			onFinished: 			false,
			threshold: 				20,							// Tell apart click and swipe (> 20px move)
			clickThreshold: 		150,						// If longer it's a swipe or doubleclick
			doubleClickThreshold: 	350,						// Max time for doubleclick
			noSwipeInside: 			'.caption',					// Avoid swiping inside this element
			swipeoutDuration: 		300,						// Duration of momentum
			overThreshold: 			0.2,						// x screensize
			preventDefaultEvents: 	true,						// Preventing default action
			zoomMin: 				0.2, 						// Min pinch zoom
			zoomMax: 				5,   						// Max pinch zoom
			momentumDecay: 			0.5, 						// Factor for momentum distance
			minMomentumSpeed: 		0.1, 						// Min speed (px/ms) to trigger momentum
		};
	
		// --- State Variables ---
		let cont = swipeTarget.parentElement || document.body,	// Container element
			// Dimensions
			selfDim, 											// Item dimensions
			clipDim, 											// Clip (container) dimensions
			// Scale
			startScale 				= 1, 						// Start scale from CSS transform
			currentScale 			= 1, 						// Following scale
			// Event times
			startTime,											// Saving gesture start time 
			lastTime, 											// Last event time
			endTime,											// Gesture ended time
			// Gesture event coords
			startPosArr,										// Start event coordinates 
			currPosArr, 										// Current event coords [x,y] or [x1,y1,x2,y2]
			speed 					= { x: 0, y: 0 },			// Current speed
			pinchStartDistance,									// Initial finger distance
			// Transform matrix of the element
			startMatrix 			= [1, 0, 0, 1, 0, 0], 		// Start matrix
			currentMatrix 			= [1, 0, 0, 1, 0, 0],		// Current matrix
			startMidPoint,										// Midpoint for multi-finger move
			currentMidPoint,
			// Flags
			hasStarted 				= false, 					// Started?
			hasEnded 				= true, 
			swipeHandled 			= true,
			isTap 					= true, 					// To differentiate move from tap for preventDefault
			constrainHorizontal 	= false, 					// Only horizontal move
			constrainVertical 		= false, 					// Only vertical ???
			canConstrainToHorizontalOnly = false,
			nonSwipeable 			= true,						// Clicked into a restricted element
			clickedOnce 			= false,					// Double-click?
			doubleClickTimeout 		= null,						// Timeout
			resizeTimeout 			= null,						// 
			ongoingTransition 		= false,
			momentumCancelled		= false,
			transitionEndCallback 	= null;
	
	
		// --- Helper Functions ---
	
		const getElementDimensions = (el) => {
			return [el.offsetWidth, el.offsetHeight];
		};
			
		const getEventCoordinates = (e) => {
			if (e.touches) {
				if (e.touches.length > 1) {
					return [e.touches[0].clientX, e.touches[0].clientY, e.touches[1].clientX, e.touches[1].clientY];
				}
				if (e.touches.length === 1) {
					return [e.touches[0].clientX, e.touches[0].clientY];
				}
			}
			if (e.clientX !== undefined && e.clientY !== undefined) { // mouse event
				return [e.clientX, e.clientY];
			}
			return null; // Should not happen if event is valid
		};
	
		const applyTransform = (targetMatrix, duration = 0, newOpacity = 1, callback = null) => {
			if (ongoingTransition && transitionEndCallback) {
				// If a transition was ongoing, remove its specific end listener to avoid premature call
				swipeTarget.removeEventListener('transitionend', transitionEndCallback);
			}
			
			ongoingTransition = false; 							// Reset flag
			currentMatrix = [...targetMatrix]; 					// Update internal state
			currentScale = Math.hypot(targetMatrix[0], targetMatrix[1]); 					// More robust scale calc
	
			let transitionStyles = [];
			if (duration > 0) {
				transitionStyles.push('transform ' + duration + 'ms ease-out');
				ongoingTransition = true;
			} else {
				swipeTarget.style.transition = 'none'; // Ensure immediate application if duration is 0
			}
	
			if (newOpacity !== null) {
				if (duration > 0) {
					transitionStyles.push('opacity ' + duration + 'ms ease-out');
				}
				swipeTarget.style.opacity = newOpacity;
			}
	
			if (duration > 0) {
				swipeTarget.style.transition = transitionStyles.join(', ');
			}
	
			swipeTarget.style.transform = 'matrix(' + targetMatrix.join(',') + ')';
			
			if (duration > 0 && typeof callback === 'function') {
				transitionEndCallback = function(event) {
					// Ensure the event is for the transform and on the swipeTarget itself
					if (event.target === swipeTarget && (event.propertyName.includes('transform') || (newOpacity !== null && event.propertyName.includes('opacity')))) {
						swipeTarget.removeEventListener('transitionend', transitionEndCallback);
						ongoingTransition = false;
						transitionEndCallback = null; // Clear stored callback
						callback();
					}
				};
				swipeTarget.addEventListener('transitionend', transitionEndCallback);
			} else if (typeof callback === 'function') {
				 // If duration is 0, execute callback immediately (async to mimic transitionend somewhat)
				setTimeout(callback, 0);
			}
		};
			
		// Getting finger distance
		const getPinchDistance = coords => {
			return Math.hypot(coords[2] - coords[0], coords[3] - coords[1]);
		};
			
		// Getting midpoint
		const getMidPoint = coords => {
			return (coords.length === 4)? [ coords[0] + (coords[2] - coords[0]) / 2, coords[0] + (coords[2] - coords[0]) / 2 ] : [ coords[0], coords[1] ];  
		};
			
		// Recalculating dimensions (and constraints) on window resize
		const updateAllDimensions = () => {
			clipDim = getElementDimensions(cont); // Assuming cont is the clipping parent
			if (cont === document.body) { // Use window for body
				clipDim = [window.innerWidth, window.innerHeight];
			}
			selfDim = getElementDimensions(swipeTarget);
			canConstrainToHorizontalOnly = clipDim[1] >= selfDim[1] * currentScale && clipDim[0] >= selfDim[0] * currentScale;
			if (canConstrainToHorizontalOnly) {
				constrainVertical = (typeof options.onSwipedUp !== 'function' && typeof options.onSwipedDown !== 'function');
			} else {
				constrainVertical = false; // Cannot constrain if element is scrollable vertically
			}
		};
	
	
		// Event Handlers
		// Swipe start
		function swipeStart(e) {
			
			if (hasStarted || (e.type === 'mousedown' && e.button !== 0)) {
				return;
			}
			
			// Check noSwipeInside (simplified)
			if (options.noSwipeInside && e.target.closest(options.noSwipeInside)) {
				nonSwipeable = true;
				return;
			}
			
			// Disabled elements
			const targetNodeName = e.target.nodeName.toUpperCase();
			if (['A', 'BUTTON', 'INPUT', 'TEXTAREA', 'SELECT', 'AUDIO', 'VIDEO'].includes(targetNodeName)) {
				if (targetNodeName === 'VIDEO') {
					const video = e.target;
					if (!video.paused || (video.controls /*&& check iOS specific offset*/)) {
						nonSwipeable = true; 
						return;
					}
				} else {
					 nonSwipeable = true; 
					 return;
				}
			}
			
			if (typeof options.onSwipeStart === 'function') { 
				options.onSwipeStart.call(swipeTarget, e);
			}
	
			// If there's an ongoing CSS transition (momentum), stop it.
			if (ongoingTransition) {
				applyTransform(getMatrix(swipeTarget), 0, 1); 				// Apply current visual state immediately
				if (transitionEndCallback) {
					swipeTarget.removeEventListener('transitionend', transitionEndCallback);
					transitionEndCallback = null;
				}
				ongoingTransition = false;
			}
			
			 if (e.type === 'mousedown') {
				if (options.preventDefaultEvents !== false) { 				// Assuming you have this option
					e.preventDefault(); 									// Prevent text selection, native browser drag, etc.
				}
			}
			
			e.target.addEventListener('selectstart', (e) => {				// Avoiding the browser's default actions 
				e.preventDefault();
			});
			
			// Setting up Flags
			hasStarted = true;
			hasEnded = false;
			swipeHandled = false;
			nonSwipeable = false;
			isTap = true; 													// Assume it's a tap until significant movement
	
			startTime = lastTime = Date.now();
			startPosArr = currPosArr = getEventCoordinates(e);
	
			if (!startPosArr) { 											// Should not happen
				hasStarted = false; 
				return;
			}
	
			startMatrix = getMatrix(swipeTarget);
			currentMatrix = [...startMatrix];								// Duplicating start matrix
			startScale = Math.hypot(startMatrix[0], startMatrix[1]); 		// Initial scale from matrix
			currentScale = startScale;
	
			speed = { x: 0, y: 0 };
	
			if (startPosArr.length === 4) { 								// Two finger gesture
				pinchStartDistance = getPinchDistance(startPosArr);
			} else {
				pinchStartDistance = 0;
			}
	
			updateAllDimensions(); 											// Update dimensions and constraints
	
			if (e.type === 'mousedown') {									// mousedown
				document.addEventListener('mousemove', swipeMove);
				document.addEventListener('mouseup', swipeEnd);
			} else { 														// touchstart
				// For touch, preventDefault is handled in swipeMove conditionally
				document.addEventListener('touchmove', swipeMove, { passive: false });
				document.addEventListener('touchend', swipeEnd);
				document.addEventListener('touchcancel', swipeEnd);
			}
	
			if (typeof options.onSwipeStart === 'function') {
				options.onSwipeStart.call(swipeTarget, e, { x: currentMatrix[4], y: currentMatrix[5], scale: currentScale });
			}
		}
	
		// Swipe move
		function swipeMove(e) {
			
			if (!hasStarted || hasEnded || nonSwipeable) {
				return;
			}
	
			const newPosArr = getEventCoordinates(e);
			if (!newPosArr) {
				return;
			}
			
			const deltaTime = (Date.now() - lastTime) || 1; 				// avoid division by zero
			let dx = newPosArr[0] - currPosArr[0],
				dy = newPosArr[1] - currPosArr[1];
			
			// Conditionally prevent default if it's a drag and not a tap, and option is set
			if (!isTap && options.preventDefaultEvents !== false) {
				e.preventDefault();
			}
			
			currPosArr = [...newPosArr]; 									// Update current position
	
			let targetMatrix = [...currentMatrix]; 							// Start with current visual matrix
			
			momentumCancelled = false;
	
			if (currPosArr.length === 4/* && startPosArr.length === 4*/) { 		// Pinch Zoom
				const currentPinchDistance = getPinchDistance(currPosArr);
				if (!pinchStartDistance) {
					pinchStartDistance = currentPinchDistance;
					startPosArr[2] = currPosArr[2];
					startPosArr[3] = currPosArr[3];
					return;
				} else {
					let scaleDelta = currentPinchDistance / pinchStartDistance;
					let newCalculatedScale = startScale * scaleDelta;
	
					// Clamp scale
					newCalculatedScale = Math.max(options.zoomMin, Math.min(options.zoomMax, newCalculatedScale));
					scaleDelta = newCalculatedScale / startScale; // Recalculate delta based on clamped scale
	
					// Calculate midpoints for zoom origin
					const startMidX = (startPosArr[0] + startPosArr[2]) / 2;
					const startMidY = (startPosArr[1] + startPosArr[3]) / 2;
					const currentMidX = (currPosArr[0] + currPosArr[2]) / 2;
					const currentMidY = (currPosArr[1] + currPosArr[3]) / 2;
	
					// Pan caused by fingers moving
					const panX = currentMidX - startMidX;
					const panY = currentMidY - startMidY;
	
					// Scale around the pinch center:
					targetMatrix[4] = startMatrix[4] + panX;
					targetMatrix[5] = startMatrix[5] + panY;
					
					// Apply scale
					targetMatrix[0] = startMatrix[0] * scaleDelta; // Assuming initial matrix b,c are 0 for simple scaling
					targetMatrix[3] = startMatrix[3] * scaleDelta; // Scale Y
	
					currentScale = newCalculatedScale; // Store the absolute current scale
				}
			} else if (currPosArr.length === 2 && startPosArr.length === 2) { 	// Single finger drag
				let moveX = currPosArr[0] - startPosArr[0],
					moveY = currPosArr[1] - startPosArr[1];
					
				//console.log('dx=' + moveX + ', dy=' + moveY);
				// Apply constraints
				if (canConstrainToHorizontalOnly) {
					const totalMoveX = Math.abs(currPosArr[0] - startPosArr[0]),
						totalMoveY = Math.abs(currPosArr[1] - startPosArr[1]);
						
					if (totalMoveX > 30 || (totalMoveX > 8 && totalMoveY > 0 && (totalMoveX / totalMoveY) > 4)) { // Strong horizontal intent
						constrainVertical = (typeof options.onSwipedUp !== 'function' && typeof options.onSwipedDown !== 'function');
					}
				}
	
				if (constrainVertical) {
					moveY = 0;
				}
				// if (constrainHorizontal) moveX = 0; // Less common
	
				targetMatrix[4] = startMatrix[4] + moveX;
				targetMatrix[5] = startMatrix[5] + moveY;
			}
	
			// Boundary checks (simplified, needs robust porting from original)
			const finalScaledWidth = selfDim[0] * (targetMatrix[0] / startMatrix[0]) * startScale; // effective scale from start
			const finalScaledHeight = selfDim[1] * (targetMatrix[3] / startMatrix[3]) * startScale;
	
			// If element is wider than container, don't let left edge go past 0, or right edge go past container width
			/*
			if (finalScaledWidth > clipDim[0]) {
				if (targetMatrix[4] > 0) targetMatrix[4] = 0;
				if (targetMatrix[4] + finalScaledWidth < clipDim[0]) targetMatrix[4] = clipDim[0] - finalScaledWidth;
			} else { // Element is narrower than container
				if (targetMatrix[4] < 0) targetMatrix[4] = 0;
				if (targetMatrix[4] + finalScaledWidth > clipDim[0]) targetMatrix[4] = clipDim[0] - finalScaledWidth;
			}
			// Similar for Y
			if (finalScaledHeight > clipDim[1]) {
				if (targetMatrix[5] > 0) targetMatrix[5] = 0;
				if (targetMatrix[5] + finalScaledHeight < clipDim[1]) targetMatrix[5] = clipDim[1] - finalScaledHeight;
			} else {
				if (targetMatrix[5] < 0) targetMatrix[5] = 0;
				if (targetMatrix[5] + finalScaledHeight > clipDim[1]) targetMatrix[5] = clipDim[1] - finalScaledHeight;
			}
			*/
			//console.log('dx=' + targetMatrix[4] + ', dy=' + targetMatrix[5]);
			applyTransform(targetMatrix);
	
			// Calculate speed (smoothed)
			speed.x = (speed.x * 2 + (dx / deltaTime)) / 3;
			speed.y = (speed.y * 2 + (dy / deltaTime)) / 3;
			lastTime = Date.now();
		}
		
		// Swipe end
		function swipeEnd(e) {
			
			if (!hasStarted || hasEnded) {							// Already ended or never started properly
				return;
			}
			
			hasStarted = false;
			hasEnded = true; // Mark as ended first
			endTime = Date.now();
			const duration = endTime - startTime;
	
			//console.log('Swipe end: dx=' + (currPosArr[0] - startPosArr[0]) + 'px dy=' + (currPosArr[1] - startPosArr[1]) + 'px dt=' + duration + 'ms');
			// Cleanup document listeners
			if (e.type === 'mouseup') {
				document.removeEventListener('mousemove', swipeMove);
				document.removeEventListener('mouseup', swipeEnd);
			} else { // touchend or touchcancel
				document.removeEventListener('touchmove', swipeMove);
				document.removeEventListener('touchend', swipeEnd);
				document.removeEventListener('touchcancel', swipeEnd);
			}
	
			const dx = currPosArr[0] - startPosArr[0],
				dy = currPosArr[1] - startPosArr[1],
				distance = Math.hypot(dx, dy),
				currentPosition = { 
					x: 		currentMatrix[4], 
					y: 		currentMatrix[5], 
					scale: 	currentScale 
				};
	
			if (currPosArr.length === 4) { // Pinch ended
				// Final state is already applied.
				// startMatrix = [...currentMatrix]; // Solidify matrix for next gesture
				// swipeHandled = true; // Reset for next discrete swipe event
				if (typeof options.onPinchZoom === 'function') { // Custom event
					 options.onPinchZoomEnd.call(swipeTarget, e, currentPosition);
				}
	
			} else if (distance > options.threshold && duration > 50) { // Swipe
				swipeHandled = true;
				let eventFired = false;
	
				// Momentum calculation (simplified, needs porting of boundary logic for swipe-out)
				let momentumX = speed.x * options.swipeoutDuration * options.momentumDecay;
				let momentumY = speed.y * options.swipeoutDuration * options.momentumDecay;
	
				if (Math.abs(speed.x) < options.minMomentumSpeed) momentumX = 0;
				if (Math.abs(speed.y) < options.minMomentumSpeed) momentumY = 0;
	
				if (constrainVertical) momentumY = 0;
				// if (constrainHorizontal) momentumX = 0;
	
				let targetX = currentMatrix[4] + momentumX;
				let targetY = currentMatrix[5] + momentumY;
				let targetOpacity = 1; // Don't fade out by default
				let swipeOut = false;
	
				// --- Complex Swipe Out / Boundary Logic from jQuery plugin needs to be ported here ---
				// This involves:
				// 1. Checking if element is smaller than container (clipDim[0] >= selfDim[0] * currentScale)
				// 2. Checking options.overThreshold against how far the element *would* travel with momentum
				// 3. Determining if it's a swipe-out (then targetOpacity = 0) or a bounce/constrained momentum.
				// 4. Original code has different logic for horizontal-only (`dontSwipeUp && dontSwipeDown`)
				//	vs. two-axis movement.
	
				// Placeholder for swipe direction callbacks (BEFORE momentum starts)
				const absDx = Math.abs(dx);
				const absDy = Math.abs(dy);
	
				// Simplified swipe out check (if it goes beyond half the container width/height with momentum)
				const finalScaledWidth = selfDim[0] * currentScale;
				const finalScaledHeight = selfDim[1] * currentScale;
	
				let callbackAfterMomentum = function() {
					 startMatrix = getMatrix(swipeTarget); // Update for next interaction
					 if (swipeOut && typeof options.onFinished === 'function') {
						 options.onFinished.call(swipeTarget, e, { x: targetX, y: targetY, scale: currentScale });
					 }
				};
	
				// Horizontal swipe check
				if (absDx > absDy * 1.5 && absDx > options.threshold) { // More horizontal
					if (constrainVertical || (typeof options.onSwipedUp !== 'function' && typeof options.onSwipedDown !== 'function')) {
						// If horizontally swiped out
						if 	(
							(dx > 0 && (currentMatrix[4] + dx + momentumX - finalScaledWidth / 2) > clipDim[0] * (-.5 + options.overThreshold)) ||
							(dx < 0 && (currentMatrix[4] + dx + momentumX + finalScaledWidth / 2) < clipDim[0] * (.5 - options.overThreshold))
							) {
							targetOpacity = 0; 
							swipeOut = true;
							if (dx > 0 && typeof options.onSwipedRight === 'function') {
								options.onSwipedRight.call(swipeTarget, e, { x: targetX, y: targetY, scale: currentScale }); 
								eventFired = true;
							} else if (dx < 0 && typeof options.onSwipedLeft === 'function') { 
								options.onSwipedLeft.call(swipeTarget, e, { x: targetX, y: targetY, scale: currentScale }); 
								eventFired = true; 
							}
						}
					}
				}
				// Vertical swipe check (only if not primarily horizontal and not constrained)
				else if (absDy > absDx * 1.5 && absDy > options.threshold && !constrainVertical &&
						(dy > 0 && typeof options.onSwipedDown === 'function' 
						||
						dy < 0 && typeof options.onSwipedUp === 'function')
					) {
					// If vertically swiped out
					if 	(
						(dy > 0 && (currentMatrix[5] + dy + momentumY - finalScaledHeight / 2) > clipDim[1] * (-.5 + options.overThreshold)) ||
						(dy < 0 && (currentMatrix[5] + dy + momentumY + finalScaledHeight / 2) < clipDim[1] * (.5 - options.overThreshold))
						) {
						targetOpacity = 0; 
						swipeOut = true;
						if (dy > 0 && typeof options.onSwipedDown === 'function') { 
							options.onSwipedDown.call(swipeTarget, e, { x: targetX, y: targetY, scale: currentScale }); 
							eventFired = true; 
						} else if (dy < 0 && typeof options.onSwipedUp === 'function') {
							options.onSwipedUp.call(swipeTarget, e, { x: targetX, y: targetY, scale: currentScale }); 
							eventFired = true; 
						}
					}
				} else {
					targetOpacity = 1;
				}
	
				// If it was a swipe but no specific direction matched or swipe-out condition, it's still a general swipe with momentum.
				// The boundary checks in swipeMove/applyTransform should constrain it.
				// The original has more nuanced bounce logic if not swiped out.
				if (!momentumCancelled) {
					// Apply momentum (or swipe-out animation)
					// The `targetX, targetY` still need to be clamped by the boundary logic in applyTransform
					let finalMatrix = [...currentMatrix];
					finalMatrix[4] = targetX;
					finalMatrix[5] = targetY;
					/*
					// Re-apply boundary constraints to the momentum target
					if (finalScaledWidth > clipDim[0]) {
						if (finalMatrix[4] > 0) finalMatrix[4] = 0;
						if (finalMatrix[4] + finalScaledWidth < clipDim[0]) finalMatrix[4] = clipDim[0] - finalScaledWidth;
					} else {
						if (finalMatrix[4] < 0) finalMatrix[4] = 0;
						if (finalMatrix[4] + finalScaledWidth > clipDim[0]) finalMatrix[4] = clipDim[0] - finalScaledWidth;
					}
					if (finalScaledHeight > clipDim[1]) {
						if (finalMatrix[5] > 0) finalMatrix[5] = 0;
						if (finalMatrix[5] + finalScaledHeight < clipDim[1]) finalMatrix[5] = clipDim[1] - finalScaledHeight;
					} else {
						if (finalMatrix[5] < 0) finalMatrix[5] = 0;
						if (finalMatrix[5] + finalScaledHeight > clipDim[1]) finalMatrix[5] = clipDim[1] - finalScaledHeight;
					}
					*/
					applyTransform(finalMatrix, options.swipeoutDuration, targetOpacity, callbackAfterMomentum);
				}
				
			} else if (distance <= options.threshold && duration < options.doubleClickThreshold) { // Potential click or double click
				if (options.onDoubleClick) {
					if (clickedOnce) { // Double click
						clearTimeout(doubleClickTimeout);
						doubleClickTimeout = null;
						clickedOnce = false;
						swipeHandled = true;
						if (typeof options.onDoubleClick === 'function') {
							options.onDoubleClick.call(swipeTarget, e, currentPosition);
						}
					} else { // First click
						clickedOnce = true;
						doubleClickTimeout = setTimeout(() => {
							if (clickedOnce) { // Timeout, so it's a single click
								clickedOnce = false;
								swipeHandled = true;
								if (typeof options.onClick === 'function') {
									options.onClick.call(swipeTarget, e, currentPosition);
								}
							}
							doubleClickTimeout = null;
						}, options.doubleClickThreshold + 10); // +10 from original
					}
				} else if (typeof options.onClick === 'function') { // No double click, just single click
					swipeHandled = true;
					options.onClick.call(swipeTarget, e, currentPosition);
				}
				 // If it was a tap, ensure element returns to pre-tap precise position if any micro-drag occurred.
				if (isTap) applyTransform(startMatrix, 100, 1);
	
	
			} else { // Small swipe or long press without movement (not a click by duration)
				swipeHandled = true;
				if (typeof options.onSmallSwipe === 'function') {
					options.onSmallSwipe.call(swipeTarget, e, currentPosition);
				}
				// Animate back to start of gesture position
				applyTransform(startMatrix, 100, null, () => {
					startMatrix = getMatrix(swipeTarget); // update after animation
				});
			}
	
			// Reset flags for next interaction (some are reset above)
			// hasStarted = false; // done at start of function or implicitly by hasEnded=true
			constrainVertical = false; // Reset constraint
		}
	
		// --- Teardown and Control ---
		swipeTarget.destroy = () => {
			swipeTarget.removeEventListener(TOUCH_EVENT.START, swipeStart);
			swipeTarget.removeEventListener('mousedown', swipeStart);
			
			// Remove document listeners if any are still attached (e.g., if destroyed mid-gesture)
			document.removeEventListener('mousemove', swipeMove);
			document.removeEventListener('mouseup', swipeEnd);
			document.removeEventListener('touchmove', swipeMove);
			document.removeEventListener('touchend', swipeEnd);
			document.removeEventListener('touchcancel', swipeEnd);
	
			window.removeEventListener('resize', handleResize);
			clearTimeout(resizeTimeout);
			clearTimeout(doubleClickTimeout);
			// Reset styles?
			swipeTarget.style.transform = ''; // Or restore to an initial state if captured
			swipeTarget.style.opacity = '';
			swipeTarget.style.transition = '';
		};
	
		swipeTarget.cancelMomentum = () => {
			
			momentumCancelled = true;
			
			if (ongoingTransition) {
				const computedMatrix = getMatrix(swipeTarget);
				const computedOpacity = parseFloat(window.getComputedStyle(swipeTarget).opacity);
				applyTransform(computedMatrix, 0, computedOpacity); // Apply current visual state immediately
				if (transitionEndCallback) {
					swipeTarget.removeEventListener('transitionend', transitionEndCallback);
					transitionEndCallback = null;
				}
				ongoingTransition = false;
				startMatrix = [...computedMatrix]; // Solidify
			}
				
			hasEnded = true; // Ensure gesture state is fully ended
		};
	
		swipeTarget.recenter = () => {
			
			momentumCancelled = true;
			
			if (ongoingTransition) {
				// console.log('Reset/Recenter: Momentum from previous swipe was active, cancelling.');
				swipeTarget.cancelMomentum(); // This should stop the transition, update currentTransformMatrix/currentCssScale to the point of cancellation, and set ongoingTransition = false.
			}
			let targetMatrix;
			// Use the default animation duration from your swipe options, or a fallback
			const animationDuration = options.swipeoutDuration || 300;
			applyTransform(startMatrix, animationDuration, 1, () => {
				// Callback after animation:
				// Update internal state to reflect the new resting position.
				//startMatrix = [...targetMatrix];
				currentMatrix = [...startMatrix]; 						// Next user gesture should start from this state
				currentScale = Math.hypot(startMatrix[0], startMatrix[1]); // Update scale based on applied matrix
				hasEnded = true; 										// Ensure interaction state is fully ended
				// console.log(`Reset/Recenter type '${resetType}' complete.`);
			});
		};
	
		function handleResize() {
			clearTimeout(resizeTimeout);
			resizeTimeout = setTimeout(updateAllDimensions, 100);
		}
	
		function init() {
			options = mergeOptions(options, userOptions);
	
			swipeTarget.style.userSelect = 'none';
			swipeTarget.style.webkitUserSelect = 'none'; 				// Safari
			swipeTarget.style.touchAction = 'none'; 					// Critical for preventing default scroll/zoom on element
			swipeTarget.setAttribute('draggable', 'false');
	
			swipeTarget.addEventListener('touchstart', swipeStart, { passive: true }); // passive:true for start, only preventDefault in move if needed
			swipeTarget.addEventListener('mousedown', swipeStart);
	
			window.addEventListener('resize', handleResize);
			updateAllDimensions(); 										// Initial check
			startMatrix = getMatrix(swipeTarget); 						// Store initial matrix
			currentMatrix = [...startMatrix];							// Duplication
			startMatrix[4] = 0;											// Ensure the initial transtion offset is not stored 
			currentScale = Math.hypot(startMatrix[0], startMatrix[1]);
		}
	
		init();
	
		return element;
	}
	
	/************************************************
	 *				Lightbox
	 ************************************************/
	 
	function Lightbox(lightbox, thumbnails, settings) {
		// Buttons
		const prevBtn = lightbox.querySelector('.prev-img');
		const nextBtn = lightbox.querySelector('.next-img');
		
		let curr = -1;
		let direction = 0;
		let slideshowTimeout = null;
		
		const transition = ' ' + settings.transTime + 'ms ease-in-out';
		const transition2 = ' ' + settings.transTime + 'ms ease-in-out ' + (settings.transTime / 4) + 'ms';
		const moveDist = 60;
		
		const setDirection = n => { direction = n === curr? 0 : n > curr? 1 : -1; }
		
		// Returns the title element
		lightbox.getTitle = () => {
			return lightbox.querySelector('.page-title');
		}
		
		// Returns the container / main element
		lightbox.getCont = () => {
			return lightbox.querySelector('.cont');
		}
		
		// Gets the current category
		lightbox.getCategory = () => {
			const cont = lightbox.getCont();
			if (cont) return cont.cat || 'image';
			return null;
		}
		
		// Previous image
		lightbox.previousImage = () => {
			const th = thumbnails.getPreviousImage();
			if (th) {
				direction = -1;
				lightbox.loadImage(th, false);
			} else {
				lightbox.recenter();
			}
		}
		
		// Next image
		lightbox.nextImage = () => {
			const th = thumbnails.getNextImage();
			if (th) {
				direction = 1;
				lightbox.loadImage(th, false);
			} else {
				lightbox.recenter();
			}
		}
		
		// Recnetering the image after e.g. a small swipe
		lightbox.recenter = () => {
			lightbox.querySelector('.cont').recenter();
		}
		
		// Toggling controls
		lightbox.toggleControls = () => {
			lightbox.classList.toggle('show-controls');
		}
		
		// Toggling zoom
		lightbox.toggleZoom = z => {
			const cont = lightbox.getCont();
			const img = cont.querySelector('.wrap img');
			const cap = cont.querySelector('.wrap .caption');
			if (img) {
				const nw = img.naturalWidth;
				const nh = img.naturalHeight;
				const ch = cap? cap.clientHeight : 0; 
				const zoom = Math.min(lightbox.clientHeight / nw, lightbox.clientHeight / (nh + ch));
				let mx = getMatrix(cont);
				if (mx[0] === 1) {
					// 1:1 -> Zoomed
					if (typeof z !== 'undefined' && !z) return;
					mx[0] /= zoom;
					mx[3] /= zoom;
				} else {
					// Zoomed -> 1:1
					if (typeof z !== 'undefined' && z) return;
					mx[0] = 1;
					mx[3] = 1;
				}
				cont.style.transition = 'transform 500ms ease-out';
				cont.style.transform = 'matrix(' + mx.join(',') + ')';
				//console.log(mx);
			}
		}
		
		// Image click listener
		lightbox.imageClick = () => {
			lightbox.toggleControls();
		}
		
		// Loading the Nth image
		lightbox.loadNthImage = n => {
			swiped = false;
			setDirection(n);
			if (thumbnails) lightbox.loadImage(thumbnails.getNth(n), false);
		}
		
		// Loading the first image
		lightbox.loadFirstImage = n => {
			if (thumbnails) lightbox.loadImage(thumbnails.getNth(0), true);
		}
		
		// Loading an image after thumbnail click
		lightbox.loadThumb = el => {
			if (el) {
				swiped = false;
				setDirection(thumbnails.getIndex(el));
				lightbox.loadImage(el, false);
			}
		}
		
		// Is the title visible?
		lightbox.isTitleOn = () => {
			const titleEl = lightbox.getTitle();
			if (titleEl) {
				const style = window.getComputedStyle(titleEl);
				return style.display !== 'none' && style.opacity > 0.5;
			}
			return false;
		}
		
		// Showing the title
		lightbox.showTitle = () => {
			const titleEl = lightbox.getTitle();
			
			if (titleEl) {
				const oldEl = lightbox.getCont();
				
				titleEl.classList.remove('hidden');
				titleEl.style.display = 'flex';
				
				requestAnimationFrame(function() { 
					titleEl.style.transition = 'opacity' + transition;
					if (oldEl && settings.backgroundType !== 'firstImage') {
						oldEl.style.opacity = 0;
					}
					titleEl.style.opacity = 1;
					prevBtn.disabled = nextBtn.disabled = true;
				});
				// No active thumb
				thumbnails.setActive();
			}
		}
		
		// Hiding the title
		lightbox.hideTitle = () => {
			const titleEl = lightbox.getTitle();
			const newEl = lightbox.getCont();
			
			if (titleEl && newEl) {
				
				newEl.classList.remove('hidden');
				newEl.style.display = 'flex';
				titleEl.style.transition = 'opacity' + transition;
				newEl.style.transition = 'opacity' + transition;
				
				requestAnimationFrame(function() {
					newEl.addEventListener('transitionend', function() {
						titleEl.classList.add('hidden');
						titleEl.style.display = 'none';
					}, { once: true });
					newEl.style.opacity = 1;
					titleEl.style.opacity = 0;
				});
			}
		}
		
		// Showing locaiton on the map
		lightbox.showLocation = (loc) => {
			const mapOly = document.createElement('div');
			const closeBtn = document.createElement('a');
			const mapCont = document.createElement('div');
			mapOly.className = 'map overlay';
			closeBtn.className = 'close icon-close';
			mapCont.className = 'map-cont';
			mapOly.appendChild(mapCont);
			mapOly.appendChild(closeBtn);
			lightbox.appendChild(mapOly);
			mapOly.style.transition = 'opacity' + transition;
			requestAnimationFrame(function() {
				mapOly.addEventListener('transitionend', function() {
					//lightbox.removeChild(mapOly);
				}, { once: true });
				mapOly.style.opacity = 1;
			});
			closeBtn.addEventListener('click', function() {
				requestAnimationFrame(function() {
					mapOly.addEventListener('transitionend', function() {
						lightbox.removeChild(mapOly);
					}, { once: true });
					mapOly.style.opacity = 0;
				});	
			});
			setTimeout(function() {
				const map = L.map(mapCont).setView(loc, 16); 
				L.tileLayer('https://{s}.tile.openstreetmap.org/{z}/{x}/{y}.png', {
					attribution: '&copy; <a href="https://openstreetmap.org/copyright">OpenStreetMap</a>'
				}).addTo(map);
				L.control.scale({
					imperial: 	true, 
					metric: 	true
				}).addTo(map); 
				L.marker(loc).addTo(map);
			}, 200);
		}

		// Loading one image
		lightbox.loadImage = (el, behind, interval) => {
			let n;
			let max = thumbnails.getMax(); 
			
			if (typeof el === 'number') {
				// Loaded by index
				n = el;
				el = thumbnails.getNth(el);
			} else if (el) {
				// Getting index
				n = parseInt(el.dataset.index || 0);
			}
			
			if (!el || n > max || !lightbox.isTitleOn() && n === curr) {
				// No such image -> Show title
				lightbox.showTitle();
				return;
			} else if (lightbox.isTitleOn() && n === curr) {
				// Showing the image already loaded behind the title 
				swiped = false;
				direction = 0;
				lightbox.hideTitle();
				curr = n;
			} else {
				// Creating new lightbox element
				const thumb = el.querySelector('.thumb');
				if (!thumb) return;
				
				const oldEls = lightbox.querySelectorAll('.cont');
				// More than one old element (a glitch) -> remove the excess
				if (oldEls.length > 1) {
					const olds = Array.from(oldEls);
					for (let i = 0; i < olds.length - 1; i++) {
						const n = olds[i];
						if (n && n.parentNode) n.parentNode.removeChild(n);
					}
				}
				
				const titleEl = lightbox.getTitle();
				const oldEl = lightbox.getCont();
				const newEl = document.createElement('div');
				const wrap = document.createElement('div');
				const cat = thumb.getAttribute('data-category') || 'image';
				const src = thumb.getAttribute('data-src');
				const cap = thumb.getAttribute('data-caption');
				const loc = thumb.getAttribute('data-location');
				const orig = thumb.getAttribute('data-original');
				let oldTo, newTo;
				let main;
				let caption;
				
				if (lightbox.isTitleOn()) {
					swiped = false;
				}
				
				direction = (curr === -1 || n === curr)? 0 : (n > curr)? 1 : -1;
				curr = n;
				
				// Transitions finished, perpetrate slideshow
				const transitionReady = function() {
					if (typeof interval !== 'undefined' && interval > 0) {
						slideshowTimeout = setTimeout(function() {
							clearTimeout(slideshowTimeout);
							lightbox.loadImage(curr + 1, false, interval);
						}, interval);
					}
				}
				
				const removeTitle = function() {
					if (oldTo) {
						clearTimeout(oldTo);
						oldTo = null;
					}
					titleEl.classList.add('hidden');
					titleEl.style.display = 'none';
				}
				
				const removeOld = function() {
					if (oldTo) {
						clearTimeout(oldTo);
						oldTo = null;
					}
					const pn = oldEl.parentNode;
					if (pn) pn.removeChild(oldEl);
				}
				
				// Image is ready -> prepare transitions, attach event handlers, call doTransitions
				const imageReady = function() {
					lightbox.classList.remove('wait');
					
					// Animating the old element
					if (oldEl) {
						// Transition from the previous image
						oldEl.addEventListener('transitionend', removeOld, { once: true });
						oldTo = setTimeout(removeOld, settings.transTime + 50);
						oldEl.style.zIndex = 0;
						if (!swiped) {
							if (direction) {
								oldEl.style.transition = 'transform' + transition + ', opacity' + transition;
								oldEl.style.transform = 'translateX(' + (0 - moveDist * direction) + 'px)';
							} else {
								oldEl.style.transition = 'opacity' + transition;
							}
						}
						oldEl.style.opacity = 0;
					}
					
					// Animating the title
					if (titleEl && !titleEl.classList.contains('hidden') && !behind) {
						// Transition from the title
						titleEl.addEventListener('transitionend', removeTitle, { once: true });
						oldTo = setTimeout(removeTitle, settings.transTime + 50);
						titleEl.style.transition = 'opacity' + transition;
						titleEl.style.opacity = 0;
					}
					
					// Applying the vertical / horizontal class
					if (newEl.querySelector('.caption')) {
						if (settings.adaptivePlacement) {
							let nw, nh;
							const lw = lightbox.clientWidth;
							const lh = lightbox.clientHeight;
							let main = newEl.querySelector('.wrap img');
							if (main) {
								nw = main.naturalWidth;
								nh = main.naturalHeight;
							} else if (main = newEl.querySelector('.wrap video, .wrap audio')) {
								const br = main.getBoundingClientRect();
								nw = br.width;
								nh = br.height;
							} else if (main = newEl.querySelector('.wrap')) {
								nw = main.clientWidth;
								nh = main.clientHeight;
							}
							
							const zoom = Math.min(lw / nw,  lh / nh);
							nw *= zoom;
							nh *= zoom;
							newEl.classList.add((lw - nw) > 120 && (lw - nw > lh - nh)? 'beside' : 'below');
						} else {
							newEl.classList.add('below');
						}
					}

					// Animating the new element
					newEl.addEventListener('transitionend', event => {
						clearTimeout(newTo);
						transitionReady();
					}, { once: true });
					newTo = setTimeout(transitionReady, settings.transTime * 1.25 + 50);
					if (direction) {
						newEl.style.transform = 'translateX(0)';
					}
					newEl.style.opacity = 1;
				}
				
				// Set active thumbnail
				if (!behind) thumbnails.setActive(el);
				
				wrap.className = 'wrap';
				newEl.appendChild(wrap);
				// Pointer to the thumbnail
				newEl.card = el;
				newEl.className = 'cont ' + cat;
				newEl.cat = cat;
				newEl.style.opacity = 0;
				newEl.style.zIndex = 1;
				newEl.style.transition = 'none'; 
				
				if (direction) {
					newEl.style.transform = 'translateX(' + (moveDist * direction) + 'px)';
				}
				if (direction) {
					newEl.style.transition = 'transform' + transition2 + ', opacity' + transition2;
				} else {
					newEl.style.transition = 'opacity' + transition2;
				}
				
				// Caption
				if (cap !== null && cap.length || loc !== null && loc || orig !== null && orig) {
					caption = document.createElement('div');
					caption.className = 'caption';
					if (cap) {
						caption.innerHTML = cap;
					}
					if (loc) {
						const btn = document.createElement('a');
						btn.className = 'btn icon-location';
						btn.innerHTML = ' ' + settings.texts['map'];
						btn.addEventListener('click', function() {
							lightbox.showLocation(loc.split(','))
						});
						caption.appendChild(btn);
					}
					if (orig) {
						const btn = document.createElement('a');
						btn.className = 'btn icon-download';
						btn.setAttribute('href', orig);
						btn.innerHTML = ' ' + settings.texts['download'];
						btn.download = true;
						caption.appendChild(btn);
					}
					newEl.appendChild(caption);
				}
				
				// Adding image element
				if (cat === 'image') {
					main = document.createElement('img');
					main.setAttribute('src', src);
					if (settings.rightClickProtect) main.addEventListener('contextmenu', e => { e.preventDefault(); });
				} else 	if (cat === 'video') {
					// ... or video
					const poster = thumb.getAttribute('data-poster');
					main = document.createElement('video');
					main.setAttribute('src', src);
					main.setAttribute('controls', true);
					main.setAttribute('preload', 'auto');
					main.setAttribute('controlsList', 'nodownload');
					if (poster) {
						main.setAttribute('poster', poster);
					}
					if (settings.autoplayVideos) {
						playVideo(main);
					}
				} else if (cat === 'audio') {
					// ... or audio
					main = document.createElement('audio');
					main.setAttribute('src', src);
					main.setAttribute('controls', true);
					main.setAttribute('preload', 'auto');
					main.setAttribute('controlsList', 'nodownload');
				} else if (cat === 'other') {
					// ... or other
					const op = el.href;
					if (op && '.docx.xlsx.txt.pdf'.indexOf('.' + op.getExt()) >= 0) {
						// Google Docs viewer
						main = document.createElement('iframe');
						main.style.width = '100%';
						main.style.height = '100%',
						main.setAttribute('frameborder', 0);
						main.setAttribute('allowfullscreen', 'allowfullscreen');
						setTimeout(function(main, el) {
							main.setAttribute('src', 'https://docs.google.com/viewer?url=' + window.location.href.getDir() + el.href + '&embedded=true');
						}, 100, main, el);
					} else {
						// Display icon with a link to the original
						main = document.createElement('div');
						main.classList.add('download');
						const link = document.createElement('a');
						link.setAttribute('src', window.location.href);
						link.setAttribute('taget', '_blank');
						const img = document.createElement('img');
						img.setAttribute('src', el.querySelector('img').src);
						link.appendChild(img);
						main.appendChild(link);			
						//...
					}
				}
				
				wrap.appendChild(main);				
				lightbox.appendChild(newEl);
				lightbox.classList.add('wait');
							
				Swipe(newEl, {
					onSwipeStart:	function(e) {
										lightbox.stopSlideshow();
									},
					onSwipedLeft:	function(e, pos) {
										swiped = true;
										direction = -1;
										lightbox.nextImage(); 
									},
					onSwipedRight:	function(e, pos) {
										swiped = true;
										direction = 1;
										lightbox.previousImage(); 
									},
					onClick:		function(e, pos) { 
										lightbox.imageClick(); 
									},
					onDoubleClick:	function(e, pos) { 
										setTimeout(function() { lightbox.toggleZoom(); }, 100); 
									}
				});
					
				switch (cat) {
					case 'image':	main.addEventListener('load', function() {
										requestAnimationFrame(imageReady);
									}); 
									break;
					case 'video':	main.addEventListener('loadedmetadata',  function() {
										requestAnimationFrame(imageReady);
									}); 
									break;
					case 'audio':	
					default:		setTimeout( function() {
										requestAnimationFrame(imageReady);
									}, 400);
				}
			}
			
			// Setting up prev/next button visibility 
			prevBtn.disabled = curr === 0 || max === 0;
			nextBtn.disabled = curr >= max - 1 || max <= 1;
		}
		
		// Starting slideshow
		lightbox.startSlideshow = function(delay) {
			if (typeof delay === 'undefined') {
				delay = settings.slideshowDelay;
			}
			if (slideshowTimeout) clearTimeout(slideshowTimeout);
			lightbox.classList.add('slideshow');
			lightbox.loadImage(0, false, delay);
		}
		
		// Stopping slideshow
		lightbox.stopSlideshow = function() {
			if (slideshowTimeout) clearTimeout(slideshowTimeout);
			lightbox.classList.remove('slideshow');
		}
		
		// Toggling slideshow
		lightbox.toggleSlideshow = function() {
			if (lightbox.classList.contains('slideshow')) {
				lightbox.stopSlideshow();
			} else {
				lightbox.startSlideshow();
			}
		}
		
		if (!lightbox.getTitle()) {
			// Loading the first image if no title page at all
			lightbox.loadNthImage(0);
		} else if (settings.backgroundType === 'firstImage') {
			// Loading the first image as background for the title
			lightbox.loadFirstImage();
		}
		
		// Click event listeners on the buttons
    	nextBtn.addEventListener('click', () => {
			if (curr < thumbnails.getMax() - 1) {
				lightbox.nextImage();
			}
		});
		prevBtn.addEventListener('click', () => {
			if (curr > 0) {
				lightbox.previousImage();
			}
		});
				
	}
	
	/************************************************
	 *				Sidebar
	 ************************************************/
	 
	function Sidebar(sidebar, thumbnails) {
		// Hamburger menu button wiring
		const navButton = sidebar.querySelector('#nav-toggle');
		if (navButton) {
			navButton.addEventListener('click', () => {
				document.body.classList.remove('share-on');
				document.body.classList.toggle('menu-on');
				if (thumbnails) {
					thumbnails.refresh();
				}
			});
		}
		// Share buttons
		const shareButton = sidebar.querySelector('#share-toggle');
		if (shareButton) {
			const buttons = sidebar.querySelectorAll('#share a.btn');
			const uri = encodeURI(window.location.href);
			for (let i = 0, hr; i < buttons.length; i++) {
				hr = buttons[i].href;
				if (hr) buttons[i].href = hr.replaceAll('$PAGE_URL', uri);
			}
				
			shareButton.addEventListener('click', () => {
				document.body.classList.remove('menu-on');
				document.body.classList.toggle('share-on');
				if (thumbnails) {
					thumbnails.refresh();
				}
			});
		}
		// Collapsible folder tree
		const nav = document.querySelector('nav.collapsible');
		if (nav) {
			const cm = [ ...nav.querySelectorAll('.has-submenu') ];
			if (cm) {
				cm.forEach(sm => {
					const cb = document.createElement('b');
					cb.classList.add('collbtn');
					cb.innerHTML = '\u203A'; //&#8250;
					sm.appendChild(cb);
					cb.addEventListener('click', e => {
						e.stopPropagation();
						const el = e.currentTarget.parentNode;
						if (el.classList.contains('opened')) el.classList.remove('opened');
						else el.classList.add('opened');
					});
					cb.addEventListener('selectstart', e => {
						e.preventDefault();
					});
				});
				cm.filter(sm => sm.classList.contains('actual-branch')).forEach(el => {
					el.classList.add('opened');
				});
			}
		}
	}
	
	/************************************************
	 *				OSM map
	 ************************************************/
	 
	function initMap(canvas, options) {
		if (typeof options === 'undefined' || !Array.isArray(options.markers)) {
			return;
		}
		const pos = options.markers[0]['pos'] || [ 0, 0 ];
		const map = L.map(canvas).setView(pos, options['zoom'] || 16); 
		L.tileLayer('https://{s}.tile.openstreetmap.org/{z}/{x}/{y}.png', {
			attribution: '&copy; <a href="https://openstreetmap.org/copyright">OpenStreetMap</a>'
		}).addTo(map);
		const marker = L.marker(pos).addTo(map);
		if (options.markers[0]['title']) {
			marker.bindPopup(options.markers[0]['title']).openPopup();
		}
		L.control.scale({
			imperial: 	true, 
			metric: 	true
		}).addTo(map); 
	}
	
	/************************************************
	 *				Initializing
	 ************************************************/
	 
	function initPage(settings) {
		const sidebar = document.getElementById('sidebar');
		const thumbnails = document.getElementById('thumbnails');
		const lightbox = document.getElementById('lightbox');
		
		if (sidebar) {
			Sidebar(sidebar, thumbnails);
		}
		
		if (thumbnails) {
			const cards = thumbnails.querySelectorAll('.card');
			
			thumbnails.getMax = () => {
				return cards.length;
			}
			
			thumbnails.getActive = () => {
				return thumbnails.querySelector('.active');
			}
			
			thumbnails.getIndex = el => {
				const ae = (typeof el === 'undefined' || el === null)? thumbnails.getActive() : el;
				return ae? [...cards].indexOf(ae) : 0;
			}
			
			thumbnails.getNth = n => {
				return [...cards][(typeof n === 'number' && !isNaN(n))? n : 0];
			}
				
			thumbnails.setActive = el => {
				const ae = thumbnails.getActive();
				if (ae) {
					ae.classList.remove('active');
					ae.classList.add('visited');
				}
				if (typeof el !== 'undefined') {
					el.classList.add('active');
					thumbnails.gotoPage();
				}
			}
			
			thumbnails.getPreviousImage = () => {
				const curr = thumbnails.getIndex();
				return (curr > 0)? thumbnails.getNth(curr - 1) : null; 
			}
			
			thumbnails.getNextImage = () => {
				const curr = thumbnails.getIndex();
				return (curr < cards.length - 1)? thumbnails.getNth(curr + 1) : null; 
			}
			
			const handleClick = e => {
				e.preventDefault();
				const el = e.target.closest('.card');
				lightbox.loadThumb(el);
				return false;
			}
		
			[...cards].forEach(el => {
				const thumb = el.firstChild;
				thumb.addEventListener('click', handleClick, false);
				if (settings.rightClickProtect) thumb.addEventListener('contextmenu', e => { e.preventDefault(); });
			});
			
			thumbnails.addEventListener('selectstart', e => {
				e.preventDefault();
				return false;
			});
			
			ThumbScroller(thumbnails, {
				onClick:		handleClick,
				thumbsPadding:	settings.thumbsPadding
			});
			
			const slideshowBtn = document.getElementById('slideshow');
			if (slideshowBtn) {
				slideshowBtn.addEventListener('click', e => {
					e.preventDefault();
					lightbox.startSlideshow();
				});
			}
		}
		
		if (lightbox && !lightbox.classList.contains('page-wrap')) {
			Lightbox(lightbox, thumbnails, settings);
			// Keyboard handling
			if (settings.enableKeyboard) {
				let lastKeydown = 0;
				document.addEventListener('keydown', function(e) {
					const d = Date.now();
					if (d - lastKeydown < 500) {
					 	 return;
					}
					lastKeydown = d;
					switch (e.keyCode) {
						case 27:
							e.preventDefault();
							lightbox.showTitle();
							break;
						case 33:
							e.preventDefault();
							lightbox.loadNthImage(Math.max(0, thumbnails.getIndex() - thumbnails.itemsPerPage()));
							break;
						case 34:
							e.preventDefault();
							lightbox.loadNthImage(Math.min(thumbnails.getMax() - 1, thumbnails.getIndex() + thumbnails.itemsPerPage()));
							break;
						case 37:
							e.preventDefault();
							lightbox.previousImage();
							break;
						case 39:
							e.preventDefault();
							lightbox.nextImage();
							break;
						case 97:
						case 35:
							e.preventDefault();
							lightbox.loadNthImage(thumbnails.getMax() - 1);
							break;
						case 103:
						case 36:
							e.preventDefault();
							lightbox.loadNthImage(0);
							break;
						case 106:
						case 179:
							e.preventDefault();
							lightbox.toggleSlideshow();
							break;
						case 107:
							e.preventDefault();
							lightbox.toggleZoom();
							break;
					}
				});
			}
			// Mouse wheel handling
			if (settings.wheelAction) {
				let wheelTimeout = null;
				
				lightbox.addEventListener('wheel', function(e) {
					if (wheelTimeout) clearTimeout(wheelTimeout);
					if (!e.deltaY || lightbox.getCategory() === 'other') return;
					if (!!e.ctrlKey || settings.wheelAction === 'zoom') {
						wheelTimeout = setTimeout(lightbox.toggleZoom, 100, e.deltaY <= 0);
					} else {
						if (!lightbox.lastWheel || (Date.now() - lightbox.lastWheel) > 250) {
							// Only if 250 has passed since the last wheeling
							wheelTimeout = setTimeout(function(dir) {
								lightbox.lastWheel = Date.now();
								if (dir) {
									lightbox.previousImage();
								} else {
									lightbox.nextImage();
								}
							}, 100, e.deltaY <= 0);
						}
					}
				}, { passive: true });
			}
		}
		
		const map = document.getElementById('map_canvas');
		if (map) {
			initMap(map, JSON.parse(map.dataset.map));
		}
	}

	initPage(window.skin || {});