package projector;

/*
	Gui.java - Projector User Interface
*/

import java.io.*;
import java.util.Date;
import java.util.logging.Level;
import java.util.logging.Logger;
import java.net.MalformedURLException;
import java.awt.Color;
import java.awt.event.*;
import java.awt.BorderLayout;
import java.awt.Component;
import java.awt.Dimension;
import java.awt.Font;
import java.beans.PropertyChangeListener;
import javax.swing.*;
import javax.swing.border.Border;
import javax.swing.BorderFactory;
import javax.swing.JButton;
import javax.swing.filechooser.FileNameExtensionFilter;
import edu.stanford.ejalbert.BrowserLauncher;
import com.kitfox.svg.app.beans.SVGIcon;

import icons.Lazaicon;
import java.util.Map;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import net.jalbum.component.JDraggableList;

import se.datadosen.util.Item;
import se.datadosen.util.IO;
import se.datadosen.component.*;
import se.datadosen.jalbum.JAlbumContext;
import se.datadosen.jalbum.SkinProperties;
import se.datadosen.jalbum.AccountManager;
import se.datadosen.jalbum.SignInManager;
import se.datadosen.jalbum.JAlbumSite;
import se.datadosen.component.DeferredSVGIcon;
import se.datadosen.jalbum.Icons;
import se.datadosen.jalbum.AlbumBean;
import se.datadosen.jalbum.EditPanel;
import se.datadosen.jalbum.JAlbum;
import se.datadosen.jalbum.JAlbumColor;
import se.datadosen.jalbum.JAlbumFrame;
import se.datadosen.jalbum.PluginContext;
import se.datadosen.jalbum.event.JAlbumAdapter;
import se.datadosen.jalbum.event.JAlbumEvent;
import se.datadosen.util.NameValue;
import se.datadosen.util.VersionNumber;

import net.jalbum.browser.WebViewBrowser;
import se.datadosen.jalbum.AlbumObject;
import se.datadosen.util.Languages;

/*************************************************
 * 
 *					Projector GUI
 * 
 */

public class Gui extends GuiBase {
		
	private long skinReadyAt = Long.MAX_VALUE;
	private final VersionNumber jalbumVersion = new VersionNumber(AlbumBean.getInternalVersion());
	private final String skinVer = new SkinProperties(skinDirectory).getProperty(SkinProperties.VERSION);
	private final String supportForum = new SkinProperties(skinDirectory).getProperty(SkinProperties.SUPPORT_FORUM);
	private final String helpRoot = "http://jalbum.net/help/en/Skin/" + skin + "/";
	
	private boolean isSkinReady() {
		// Within 1s of launch
		//System.out.println("skinReady()? " + (System.currentTimeMillis() - skinReadyAt));
		return (System.currentTimeMillis() - skinReadyAt) > 1000L;
	}
	
	private String getStyleName() {
		String style = engine.getStyle();
		return style.substring(0, style.indexOf("."));
	}
	
	private boolean isLightStyle() {
		String s = getStyleName();
		return "Cardboard,Creme,Exhibition,Hemp,Leather Light,Retro,Ricepaper,Shine,Stonewall,Wallpaper,White,Wood light,Yellow".indexOf(s) >= 0;
	}
	
	private String getSkinName() {
		return engine.getSkin();
	}
	
	private String getLastSkinName() {
		String ls = null;
		
		try {
			ls = engine.getLastSkin();
		} catch(Throwable t) {
			JAlbum.logger.log(Level.FINER, "Last skin is unknown.");
		}
		
		return ls;
	}	
	
	private final String unCamelCase(String s) {
		
		if (s.length() > 0) {
			s = s.replaceAll("([a-z])([A-Z]+)", "$1 $2");
			return Character.toUpperCase(s.charAt(0)) + s.substring(1);
		}
		
		return "";
	}
	
	private String getText(String t) {
		String s;
		
		try {
			s = texts.getString(t);
		} catch (java.util.MissingResourceException e) {
			JAlbum.logger.log(Level.FINE, "Missing text: \"{0}\"", t);
			if (t.startsWith("ui.")) {
				t = t.substring(3);
			}
			s = unCamelCase(t);
		}
		
		return s;
	}
	
	private final int uiHeight = 560;
	private final int uiWidth = 600;
	private final int previewWidth = 520;
	private final String tabStyle = "style='padding:3px 4px;margin:3px 4px;'";
	private final String CUSTOM = "<!--custom-->";
	
	private String getFonts(String html, String ff, String hf) {
		String	gf = "",
				fw = "",
				bw = "bold",
				hw = "";
		String[] w;
		
		if (ff.contains(":")) {
			// Google font
			gf = ff;
			fw = ff.split(":")[1];
			if (fw.length() > 0) {
				w = fw.split(",");
				bw = (w.length > 2)? w[2] : w[1];
				fw = w[0];
			}
			
			ff = ff.split(":")[0];
			if (ff.contains(" ")) {
				ff = "\"" + ff + "\"";
			}
			ff += ",sans-serif";
		}

		if (!hf.equals("")) {
			if (!hf.equals(gf)) {
				gf = (gf.length() > 0)? (gf + "|" + hf) : hf;
			}
			if (hf.contains(":")) {
				// Has weight info
				hw = hf.split(":")[1];
				if (hw.length() > 0) {
					// Get base weight
					hw = hw.split(",")[0];
				}
				hf = hf.split(":")[0];
			}
			if (hf.contains(" ")) {
				hf = "\"" + hf + "\"";
			}
			hf += ",sans-serif";
		}
		
		//System.out.println("Google font: " + gf);
		return html.replace("{googleFontsLink}", (gf.length() == 0)? "" : ("\t<link rel=\"stylesheet\" href=\"http://fonts.googleapis.com/css?family=" + gf.replaceAll(" ", "+") + "&display=swap\">\n"))
			.replace("{fontFamily}", ff)
			.replace("{fontWeight}", (fw.equals("")? "300" : fw))
			.replace("{boldWeight}", bw)
			.replace("{headlineFont}", (hf.equals(""))? "inherit" : hf)
			.replace("{headlineWeight}", (hw.equals("")? "400" : hw));
		
	}
	
	private String attemptSignIn() {
		SignInManager mgr = SignInManager.getInstance();
		if ( mgr != null && mgr.isSignedIn() ) {
			return "&cid=" + AccountManager.getCid(mgr.getUserName(), mgr.getPassword());
		}
		return "";
	}
	
	//private String licensee = licenseManager.isLicenseValid()? licenseManager.getUserName() : "";
	
	private String license = licenseManager.isLicenseValid()? licenseManager.getLicenseType() : "";

	private boolean commercialUseAllowed = license != null && license.length() > 0 && "pro".equals(license);
	
	private Font mono = new Font("monospaced", Font.PLAIN, 12);
	
	private Border emptyBorder = BorderFactory.createEmptyBorder();
	
	private String getFileContents(String name) {
		StringBuilder cont = new StringBuilder();
		String line;
		String nl = System.getProperty("line.separator");
		File f = new File(skinDirectory, name);
		
		if (f.exists()) {
			try {
				try (java.io.BufferedReader in = new java.io.BufferedReader(new java.io.FileReader(f))) {
					while ((line = in.readLine()) != null) {
						cont.append(line);
						cont.append(nl);
					}
				}
			} catch (FileNotFoundException e) {
				return null;
			} catch (IOException e) {
				return null;
			}
		}
		return cont.toString();
	}
	
	private void writeFile(String fileName, String cont) {
		File out = new File(fileName);
		
		try {
			FileWriter writer = new FileWriter(out, false);
			writer.write(cont);
			writer.close();
		} catch (IOException e) {
			JAlbum.logger.log(Level.FINE, "Error writing \"{0}\".", fileName);
		}
	}
 	
	private final Double FULLY_TRANSP = 0.0001D;
	private final Double FULLY_OPAQUE = 0.9999D;
	
	private Color toColor(JColorSelector cs) {
		return toColor(cs.toString());
	}
	
	private static final Pattern RGBA_PATTERN = Pattern.compile("rgba?\\((\\d+),(\\d+),(\\d+)(,([\\d\\.]+))?\\)");
	
	private Color toColor(String c) {
		int a = 255;
		Color clr = Color.GRAY;
		
		//System.out.print("toColor(" + c + ") -> ");
		if (c.startsWith("rgb")) {
			Matcher matcher = RGBA_PATTERN.matcher(c);
			if (matcher.matches()) {
				// Get the color components from the matched groups
				if (matcher.groupCount() > 5) {
					a = Math.min(255, (int)Math.round(Double.parseDouble(matcher.group(5)) * 255.0));
				}
				
				if (a == 255) {
					clr = new Color(
						Math.min(255, Integer.parseInt(matcher.group(1))),
						Math.min(255, Integer.parseInt(matcher.group(2))),
						Math.min(255, Integer.parseInt(matcher.group(3)))
					);
				} else {
					clr = new Color(
						Math.min(255, Integer.parseInt(matcher.group(1))),
						Math.min(255, Integer.parseInt(matcher.group(2))),
						Math.min(255, Integer.parseInt(matcher.group(3))),
						a
					);
				}
			}
			
			return clr;
		}
		
		if (c.charAt(0) == '#') {
			c = c.substring(1);
		}
		
		if (c.length() > 7) {
			a = Integer.parseInt(c.substring(0, 2), 16);
			c = c.substring(2);
		}
		
		if (c.length() == 3) {
			clr = new Color(
					Math.min(255, Integer.parseInt(c.substring(0,1), 16) * 17),
					Math.min(255, Integer.parseInt(c.substring(1,2), 16) * 17),
					Math.min(255, Integer.parseInt(c.substring(2), 16) * 17)
				);
		}
		
		if (c.length() == 6) {
			clr = new Color(
					Math.min(255, Integer.parseInt(c.substring(0,2), 16)),
					Math.min(255, Integer.parseInt(c.substring(2,4), 16)),
					Math.min(255, Integer.parseInt(c.substring(4), 16)),
					a
				);
			//System.out.println(clr.getAlpha() + " alpha=" + a);
		}
		
		return clr;
	}
	
	private String getCssColor(JColorSelector cs) {
		
		return getCssColor(toColor(cs));
	}
	
	private String getCssColor(String c) {
		
		return getCssColor(toColor(c));
	}
	
	private String getCssColor(Color c) {
		
		if (c == null) {
			return "transparent";
		}
		
		Double a = c.getAlpha() / 255.0;
		
		if (a < FULLY_TRANSP) {
			// Transparent
			return "transparent";
		} else if (a < FULLY_OPAQUE) {
			// Semi transparent
			return "rgba(" + c.getRed() + "," + c.getGreen() + "," + c.getBlue() + "," + (Math.round(a * 1000) / 1000.0) + ")";
		}
		
		// Opaque
		return "rgba(" + c.getRed() + "," + c.getGreen() + "," + c.getBlue() + ",1)";
	}
	
	private Double getAlpha(JColorSelector cs) {
		Color c = toColor(cs);
		
		return c.getAlpha() / 255.0;
	}
	
	private Double getAlpha(Color c) {
		return c.getAlpha() / 255.0;
	}
	
	private Double getAlpha(String clr) {
		Color c = toColor(clr);
		
		return c.getAlpha() / 255.0;
	}
	
	private String setAlpha(String clr, Double a) {
		Color c = toColor(clr);
		
		return "rgba(" + c.getRed() + "," + c.getGreen() + "," + c.getBlue() + "," + a + ")";
	}
	
	private Color flatColor(JColorSelector bc, JColorSelector fc) {
		
		return flatColor(toColor(bc), toColor(fc));
	}
	
	private Color flatColor(Color bc, Color fc) {
		Double a = fc.getAlpha() / 255.0;
			
		if (a < FULLY_TRANSP) {
			return bc;
		} else if (a > FULLY_OPAQUE) {
			return fc;
		}
		
		return new Color(
			(int)Math.max(0, Math.min(Math.round(a * fc.getRed() + (1 - a) * bc.getRed()), 255)),
			(int)Math.max(0, Math.min(Math.round(a * fc.getGreen() + (1 - a) * bc.getGreen()), 255)),
			(int)Math.max(0, Math.min(Math.round(a * fc.getBlue() + (1 - a) * bc.getBlue()), 255))
		);
	}
	
	private String getMiddleColor(String c1, String c2) {
		Color c = getMiddleColor(toColor(c1), toColor(c2));
		
		return "rgba(" + c.getRed() + "," + c.getGreen() + "," + c.getBlue() + "," + (Double.valueOf(c.getAlpha()) / 255.0) + ")";
	}
	
	private Color getMiddleColor(Color c1, Color c2) {
					
		return new Color(
				(int)Math.round((c1.getRed() + c2.getRed()) / 2),
				(int)Math.round((c1.getGreen() + c2.getGreen()) / 2),
				(int)Math.round((c1.getBlue() + c2.getBlue()) / 2),
				(int)Math.round((c1.getAlpha() + c2.getAlpha()) / 2)
			);
	}

	
	private String getFlatColor(JColorSelector bc, JColorSelector fc) {
		
		return getCssColor(flatColor(toColor(bc), toColor(fc)));
	}
	
	private String getFlatColor(String bc, String fc) {
		
		return getCssColor(flatColor(toColor(bc), toColor(fc)));
	}
	
	private String getFlatColor(Color bc, Color fc) {
		
		return getCssColor(flatColor(bc, fc));
	}
	
	private Double getLuminosity(JColorSelector bg, JColorSelector fg) {
		
		return getLuminosity(flatColor(bg, fg));
	}
	
	private Double getLuminosity(JColorSelector cs) {
		
		return getLuminosity(toColor(cs));
	}
	
	private Double getLuminosity(String c) {
		
		return getLuminosity(toColor(c));
	}
	
	private Double getLuminosity(Color c) {
		
		return	0.0011725490196078 * c.getRed() + 
				0.0023019607843137 * c.getGreen() +
				0.0004470588235294118 * c.getBlue();
	}
	
	private boolean isLightColor(JColorSelector cs) {
		
		return getLuminosity(cs) > 0.5;
	}
	
	private boolean isLightColor(String fc) {
		
		return getLuminosity(fc) > 0.5;
	}
	
	private boolean isLightColor(JColorSelector fc, double threshold) {
		
		return getLuminosity(fc) > threshold;
	}
	
	private boolean isLightColor(String fc, double threshold) {
		
		return getLuminosity(fc) > threshold;
	}
	
	private boolean isLightColor(JColorSelector bc, JColorSelector fc) {
		
		return isLightColor(toColor(bc), toColor(fc), 0.5);
	}
	
	private boolean isLightColor(String bc, String fc) {
		
		return isLightColor(toColor(bc), toColor(fc), 0.5);
	}
	
	private boolean isLightColor(JColorSelector bc, JColorSelector fc, Double threshold) {
		
		return isLightColor(toColor(bc), toColor(fc), threshold);
	}
	
	private boolean isLightColor(String bc, String fc, Double threshold) {
		
		return isLightColor(toColor(bc), toColor(fc), threshold);
	}
	
	private boolean isLightColor(Color bc, Color fc, Double threshold) {
		
		Double a = fc.getAlpha() / 255.0;
		
		if (a > FULLY_OPAQUE) {
			return getLuminosity(fc) >= threshold;
		}
		
		return getLuminosity(flatColor(bc, fc)) >= threshold;
	}
	
	private String getLegibleColor(Color bc1, Color bc2, Color fc, Double f) {
		Color bc = flatColor(bc1, bc2);
		
		return getLegibleColor(bc, fc, f);
	}
		
	private String getLegibleColor(JColorSelector bc, JColorSelector fc) {
		return getLegibleColor(toColor(bc), toColor(fc), 0.6);
	}
	
	private String getLegibleColor(JColorSelector bc, JColorSelector fc, Double f) {
		return getLegibleColor(toColor(bc), toColor(fc), f);
	}
	
	private String getLegibleColor(JColorSelector bc1, JColorSelector bc2, JColorSelector fc) {
		return getLegibleColor(flatColor(bc1, bc2), toColor(fc), 0.6);
	}
	
	private String getLegibleColor(JColorSelector bc1, JColorSelector bc2, JColorSelector fc, Double f) {
		return getLegibleColor(flatColor(bc1, bc2), toColor(fc), f);
	}
	
	private String getLegibleColor(Color bc, Color fc, Double f) {
		int r, g, b;
		
		if (Math.abs(getLuminosity(bc) - getLuminosity(fc)) >= f) {
			return getCssColor(fc);
		}
		
		f *= 255.0;
		
		if (getLuminosity(bc) > 0.5) {
			// Darken
			r = (int)Math.round(Math.max(fc.getRed() - f, 0));
			g = (int)Math.round(Math.max(fc.getGreen() - f, 0));
			b = (int)Math.round(Math.max(fc.getBlue() - f, 0));
		} else {
			// Lighten
			r = (int)Math.round(Math.min(fc.getRed() + f, 255));
			g = (int)Math.round(Math.min(fc.getGreen() + f, 255));
			b = (int)Math.round(Math.min(fc.getBlue() + f, 255));
		}
		
		return getCssColor(new Color(r, g, b));
	}
	
	private final String[] imageFiles = new String[] { "jpg", "jpeg", "png", "gif", "svg" };
	private final FileChooser fc = ChooserFactory.createFileChooser(window);

	private void getFileToRes(String[] extensions, JTextField name, Component c) {
		fc.setFileFilter(new FileNameExtensionFilter(getText("ui.select"), extensions));
		int returnVal = fc.showOpenDialog(c);
		
		if (returnVal == JFileChooser.APPROVE_OPTION) {
			String fn = fc.getSelectedFile().toString();
			
			if (!fn.trim().equals("")) {
				File src = new File(fn);
				File dst = new File(context.getPluginContext().getRootFolder().getFile(), "res");
				if (!dst.exists()) {
					dst.mkdir();
				}
				if (src.exists() &&  dst.exists()) {
					try {
						IO.copyFile(src, dst);
						name.setText(src.getName());
					} catch (IOException e) {
						System.out.println(e);
					}
				}
			}
		}
	}
	
	private void backupProjectFile(String ver) {
		
		try {
			File projectFile = window.projectChooser.getSelectedFile();
			IO.copyFile(projectFile, new File(projectFile.getParentFile(), IO.baseName(projectFile) + "-" + ((ver == null)? "old" : ver) + ".jap"));
		} catch (IOException ex) {
			JAlbum.logger.log(Level.FINE, "Error backing up project file: {0}", window.projectChooser.getSelectedFile());
		}
	}
	
	private StateMonitor commercialMonitor = new StateMonitor() {
		@Override
		public void onChange() {
			
			if (((JCheckBox)source).isSelected() && !commercialUseAllowed) {
				if (isSkinReady()) {
					Object[] options = { 
						getText("ui.signUp"),
						getText("ui.noThanks")
					};
					int n = JOptionPane.showOptionDialog(window, 
						getText("ui.licenseWarningText"), 
						getText("ui.licenseWarningTitle"), 
						JOptionPane.YES_NO_OPTION, 
						JOptionPane.INFORMATION_MESSAGE,
						null,
						options,
						options[1]
					);
					if (n == 0) {
						try {
							BrowserLauncher.openURL(JAlbumSite.getTrueInstance().getMyJAlbumUpgradeUrl() + "/?referrer=" + skin + attemptSignIn());
						} catch (se.datadosen.tags.ElementException | IOException x) {
						}
					}
				}
				((JCheckBox)source).setSelected(false);
			}
		}
	};
		
	private static Icon icon(String basename) {
		return Icons.get(Lazaicon.class, "svg/" + basename + ".svg", 18);
    }
		
	private static Icon icon(String basename, int size) {
		return Icons.get(Lazaicon.class, "svg/" + basename + ".svg", size);
	}
		
	private static Icon svgIcon(String basename, Dimension dim) {
		return svgIcon(basename, dim, false);
	}
	
	private static Icon svgIcon(String basename, Dimension dim, boolean adapt) {
		DeferredSVGIcon icon = new DeferredSVGIcon(Gui.class, "graphics/" + basename + ".svg");
		
		icon.setAntiAlias(true);
		icon.setAdaptColors(adapt);
		icon.setPreferredSize(dim);
		icon.setAutosize(SVGIcon.AUTOSIZE_STRETCH);
		
		return icon;
    }
	
	private Object[] getPosition() {
		return new Object[] {
			new Item("left top", getText("ui.left_top")), 
			new Item("center top", getText("ui.center_top")), 
			new Item("right top", getText("ui.right_top")), 
			new Item("left center", getText("ui.left_middle")), 
			new Item("center center", getText("ui.center_middle")), 
			new Item("right center", getText("ui.right_middle")), 
			new Item("left bottom", getText("ui.left_bottom")), 
			new Item("center bottom", getText("ui.center_bottom")), 
			new Item("right bottom", getText("ui.right_bottom"))
		};
	}

	private int setSelectedValue(JComboBox<Item> box, Object val) {
		int		size = box.getItemCount();
		Item	it;
		
		for (int i = 0; i < size; i++) {
			it = box.getItemAt(i);
			if (it.item.equals(val)) {
				box.setSelectedIndex(i);
				return i;
			}
		}
		
		return -1;
	}
	
	private String getSelectedValue(Object o) {
		if (o instanceof JComboBox) {
			Object so = ((JComboBox)o).getSelectedItem();
			if (so instanceof Item) {
				return ((Item)so).value.toString();
			}
			return (so == null)? null : so.toString();
		}
		return (o == null)? null : o.toString();
	}
	
	private int getSpinnerValueAsInt(JSpinner o) {
		return (Integer)o.getValue();
	}
	
	private Double getSpinnerValueAsDouble(JSpinner o) {
		return (Double)o.getValue();
	}
	
	private Double getSelectedValueAsDouble(Object o) {
		if (o instanceof JComboBox) {
			Object so = ((JComboBox)o).getSelectedItem();
			if (so instanceof Item) {
				String s = ((Item)so).value.toString();
				return ((s != null) && (s.length() > 0))? Double.parseDouble(s) : null;
			}
			return null;
		}
		return (o == null)? null : Double.parseDouble(o.toString());
	}
	
	private Integer getSelectedValueAsInt(JComboBox o) {
		Object so = ((JComboBox)o).getSelectedItem();
		if (so instanceof Item) {
			return Integer.parseInt(((Item)so).value.toString());
		}
		return null;
	}
	
	private Double getThumbAspectRatio() {
		String[] cdim = engine.getThumbSize().split("x");
		Double	w = Double.parseDouble(cdim[0]),
				h = Double.parseDouble(cdim[1]);
		
		if (w != null && h != null) {
			return w / h;
		}
		
		JAlbum.logger.log(Level.FINE, "Invalid thumbnail dimensions: {0} x {1}", new Object[]{ w.toString(), h.toString() });
		return null;
	}
	
	private Date skinLoadedOn = new Date();
	
	// Detects skin load (within two seconds of initing?)
	// isSynching and currentThread methods are unreliable :(
	
	/*
	private boolean isSynchingUI() {
		try {
			java.lang.reflect.Method m = se.datadosen.util.PropertyBinder.class.getMethod("isSynchingUI");
			
			Object is = m.invoke(null);
			String tn = Thread.currentThread().getName();
			JAlbum.logger.log(Level.FINE, "isSynchingUI: {0}, Thread: {1}", new Object[]{is.toString(), tn});
			return !is.equals(Boolean.FALSE) && !tn.equals("Skin onload");
			
		} catch (IllegalAccessException | IllegalArgumentException | InvocationTargetException | NoSuchMethodException | SecurityException ex) {
			JAlbum.logger.log(Level.FINE, "isSynchingUI error. {0}", ex);
		}
		
		return false;
	}
	*/
	
	private void allowHTMLEditing(JSmartTextArea ta) {
		try {
			ta.setAllowHTMLEditing(true).setFullHTMLEditing(true);
		} catch (Throwable e) {
		}
	}
	
	private void allowSpelling(JSmartTextArea ta) {
		try {
			ta.spelling();
		} catch (Throwable e) {
		}
	}
	
	private String majorVersion(String s) {
		int i = s.indexOf(".");
		if (i > 0) {
			return s.substring(0, i);
		}
		return s;
	}
	
	private int getMajorVersion(Object v) {
		
		if (v == null) {
			return -1;
		} else if (v instanceof Integer) {
			return (int)v;
		} else if (v instanceof String) {
			try {
				return Integer.valueOf(majorVersion((String)v));
			} catch(NumberFormatException ex) {
				return -1;
			}
		} else if (v instanceof Double) {
			return (int)Math.floor((Double)v);
		}
		return -1;
	}
	
	Boolean checkBooleanProperty(AlbumObject[] sel, String pn) {
		
		if (sel.length > 1) {
			boolean v = sel[0].getProperties().get(pn, false);
			
			for (AlbumObject ao : sel) {
				if (ao.getProperties().get(pn, false) != v) {
					return null;
				}
			}
			return v;
		} else if (sel.length == 1) {
			return sel[0].getProperties().get(pn, false);
		} else {
			return null;
		}
	}
	
	void createMenuItemForBooleanProperty(JMenu menu, String pn, String title) {
		
		createMenuItemForBooleanProperty(menu, window.albumExplorer.explorer.getSelectedAlbumObjects(), pn, title);
	}
	
	void createMenuItemForBooleanProperty(JMenu menu, AlbumObject[] sel, String pn, String title) {		
		Boolean cv = checkBooleanProperty(sel, pn);

		try {
			if (cv == null) {
				// Indefinite value -> Add submenu
				JMenu subMenu = new JMenu(title);
				JMenuItem all = new JMenuItem(getText("ui.all"));
				all.addActionListener(ae -> {
					for (AlbumObject ao : sel) {
						ao.getProperties().put(pn, true);
						ao.getProperties().save();
					}				
				});
				JMenuItem none = new JMenuItem(getText("ui.none"));
				none.addActionListener(ae -> {
					for (AlbumObject ao : sel) {
						ao.getProperties().put(pn, false);
						ao.getProperties().save();
					}				
				});
				subMenu.add(all);
				subMenu.add(none);
				menu.add(subMenu);
	
			} else {
				// All items has the same value -> invert them when clicked
				JCheckBoxMenuItem menuItem = new JCheckBoxMenuItem(title, null, cv);
	
				menuItem.addActionListener(ae -> {
					for (AlbumObject ao : sel) {
						ao.getProperties().put(pn, !cv);
						ao.getProperties().save();
					}
				});
	
				menu.add(menuItem);
			}
			
		} catch (Throwable ex) {
			JAlbum.logger.log(Level.INFO, "Right-click menu is available only in jAlbum v32 and newer!");
		}
	}
	
	/*****************************************
	 *
	 *			Skin UI starts here
	 *
	 *****************************************/
	
	ControlPanel skinUi = new ControlPanel() {
		
		// UI-wide variables
		
		// Icons used multple places
		private Icon infoIcon = icon("info");
		private Icon mandatory = svgIcon("asterisk", new Dimension(16, 16));
		
		// silence further notifications
		private boolean skinChangeReported = false;
		
		// styleName is stored as skin variable in order to check style change later
		private JTextField styleName = new JTextField(getStyleName());
		
		// skinVersion is stored as skin variable in order to check major skin version change later
		JTextField majorSkinVersion = new JTextField(majorVersion(skinVer));
		
		JTextField facebookAppId = new JSmartTextField("", 24);
		
		JColorSelector backgroundColor = new JAlphaColorSelector(getText("ui.backgroundColor"), new JSmartTextField("#222222", 10));
		JTextField backgroundImageName = new JSmartTextField(16);
		JComboBox backgroundPos = new JComboBox(getPosition());
		JComboBox backgroundRepeat = new JComboBox(new Object[] {
			new Item("no-repeat", getText("ui.no_repeat")),
			new Item("repeat-x", getText("ui.repeat_x")),
			new Item("repeat-y", getText("ui.repeat_y")),
			new Item("repeat", getText("ui.repeat_both")),
			new Item("stretch", getText("ui.stretch")),
		});
		
		JTextField sidebarBackgroundImageName = new JSmartTextField(16);
		JColorSelector sidebarBackgroundColor = new JAlphaColorSelector(getText("ui.sidebar"), new JSmartTextField("#ff111111", 10));
		
		JColorSelector heroBackgroundColor = new JAlphaColorSelector(getText("ui.coverPage"), new JSmartTextField("#88111111", 10));
		JColorSelector heroTextColor = new JColorSelector(getText("ui.textColor"), new JSmartTextField("#bbbbbb", 10));
		JTextField heroBackgroundImageName = new JSmartTextField(16);
		JComboBox heroBackgroundPos = new JComboBox(getPosition());
		JComboBox heroBackgroundRepeat = new JComboBox(new Object[] {
			new Item("no-repeat", getText("ui.no_repeat")),
			new Item("repeat-x", getText("ui.repeat_x")),
			new Item("repeat-y", getText("ui.repeat_y")),
			new Item("repeat", getText("ui.repeat_both")),
			new Item("stretch", getText("ui.stretch")),
		});
		
		JColorSelector linkColor = new JColorSelector(getText("ui.linkColor"), new JSmartTextField("#dddddd", 10));
		JColorSelector hoverColor = new JColorSelector(getText("ui.hoverColor"), new JSmartTextField("#ffffff", 10));
		JColorSelector buttonColor = new JAlphaColorSelector(getText("ui.button"), new JSmartTextField("#2199e8", 10));

		// Typography
		private final JComboBox fontSuggestions = new JComboBox(new Object[] {
			"[" + getText("ui.suggestedFonts") +  "]",
			"Oswald / Helvetica Neue",
			"Barlow Semi Condensed Light / Barlow Light",
			"Cinzel Decorative / Raleway",
			"Dosis Bold / Montserrat",
			"Exo 2 / Exo 2",
			"Fjalla One / Roboto",
			"IBM Plex Serif / IBM Plex Sans",
			"Jura / Roboto Condensed",
			"Martel / Roboto",
			"Merriweather / Merriweather Sans",
			"Oswald / Open Sans",
			"Raleway / Open Sans",
			"Roboto Condensed / Roboto",
			"Roboto Slab / Montserrat",
			"Special Elite / Yanone Kaffeesatz",
			"Unica One / Lato",
			"Yanone Kaffeesatz Bold / Muli"
		});
						
		private void setFontBoxes() {
			
			if (fontSuggestions.getSelectedIndex() == 0) {
				return;
			}
			String	s = fontSuggestions.getSelectedItem().toString(),
					hf = s.split("/")[0].trim(),
					ff = s.split("/")[1].trim();

			setSelectedValue(fontFamily, ff);
			setSelectedValue(headlineFont, hf);
			fontSuggestions.setSelectedIndex(0);
		};
		
		JComboBox fontFamily = new JComboBox(new Object[] {
			new Item("Arial, \"Helvetica Neue\", Helvetica, sans-serif", "Arial"),
			new Item("Baskerville, \"Baskerville Old Face\", \"Hoefler Text\", Garamond, \"Times New Roman\", serif", "Baskerville"),
			new Item("Calibri, Candara, Segoe, \"Segoe UI\", Optima, Arial, sans-serif", "Calibri"),
			new Item("Cambria, Georgia, Times, \"Times New Roman\", serif", "Cambria"),
			new Item("\"Century Gothic\", \"Apple Gothic\", \"Goudy Old Style\", sans-serif", "Century Gothic"),
			new Item("\"Comic Sans MS\", cursive", "Comic Sans"),
			new Item("Consolas, \"Lucida Console\", Monaco, monospace", "Consolas"),
			new Item("Constantia, Palatino, \"Palatino Linotype\", \"Palatino LT STD\", Georgia, serif", "Constantia"),
			new Item("\"Copperplate Light\", \"Copperplate Gothic Light\", serif", "Copperplate Light"),
			new Item("\"Courier New\", Courier, monospace", "Courier New"),
			new Item("\"Franklin Gothic Medium\", \"Arial Narrow Bold\", Arial, sans-serif", "Franklin Gothic"),
			new Item("Futura, \"Century Gothic\", AppleGothic, sans-serif", "Futura"),
			new Item("Garamond, \"Hoefler Text\", Times New Roman, Times, serif", "Garamond"),
			new Item("Geneva, \"Lucida Sans\", \"Lucida Grande\", \"Lucida Sans Unicode\", Verdana, sans-serif", "Geneva"),
			new Item("Georgia, Palatino, \"Palatino Linotype\", Times, \"Times New Roman\", serif", "Georgia"),
			new Item("\"Gill Sans\", \"Gill Sans MT\", Calibri, \"Trebuchet MS\", sans-serif", "Gill Sans"),
			new Item("\"Goudy Old Style\", Garamond, \"Big Caslon\", \"Times New Roman\", serif", "Goudy Old Style"),
			new Item("\"Helvetica Neue\", Helvetica, Arial, sans-serif", "Helvetica Neue"),
			new Item("\"Hoefler Text\", Constantia, Palatino, \"Palatino Linotype\", \"Book Antiqua\", Georgia, serif", "Hoefler Text"),
			new Item("Impact, Haettenschweiler, \"Arial Narrow Bold\", sans-serif", "Impact"),
			new Item("\"Lucida Sans\", \"Lucida Grande\", \"Lucida Sans Unicode\", Verdana, sans-serif", "Lucida Sans"),
			new Item("\"Lucida Bright\", Georgia, serif", "Lucida Bright"),
			new Item("Palatino, \"Palatino Linotype\", \"Book Antiqua\", Georgia, serif", "Palatino"),
			new Item("Segoe, \"Segoe UI\", Tahoma, Geneva, \"Nimbus Sans L\", sans-serif", "Segoe"),
			new Item("Tahoma, Geneva, Verdana, sans-serif", "Tahoma"),
			new Item("Times, \"Times New Roman\", Georgia, serif", "Times"),
			new Item("\"Trebuchet MS\", \"Lucida Sans Unicode\", \"Lucida Grande\", \"Lucida Sans\", Tahoma, sans-serif", "Trebuchet MS"),
			new Item("Verdana, Geneva, Tahoma, sans-serif", "Verdana"),
			new Item("Alegreya Sans:300,300i,500,500i", "Alegreya Sans Light"),
			new Item("Barlow:300,300i,600,600i", "Barlow Light"),
			new Item("Barlow:400,400i,700,700i", "Barlow"),
			new Item("Barlow Semi Condensed:300,300i,500,500i", "Barlow Semi Condensed"),
			new Item("Exo 2:300,300i,600,600i", "Exo 2"),
			new Item("Fira Sans:300,300i,600,600i", "Fira Sans"),
			new Item("IBM Plex Sans:300,300i,600,600i", "IBM Plex Sans"),
			new Item("Josefin Sans:300,300i,600,600i", "Josefin Sans"),
			new Item("Josefin Slab:300,300i,600,600i", "Josefin Slab"),
			new Item("Lato:300,300i,700,700i", "Lato"),
			new Item("Merriweather Sans:300,300i,700,700i", "Merriweather Sans"),
			new Item("Merriweather:300,300i,700,700i", "Merriweather Serif"),
			new Item("Montserrat:300,300i,600,600i", "Montserrat"),
			new Item("Muli:300,300i,700,700i", "Muli"),
			new Item("Open Sans:300,300i,600,600i", "Open Sans"),
			new Item("Raleway:300,300i,600,600i", "Raleway"),
			new Item("Roboto:300,300i,500,500i", "Roboto"),
			new Item("Roboto Condensed:300,300i,700,700i", "Roboto Condensed"),
			new Item("Source Sans Pro:300,300i,600,600i", "Source Sans Pro"),
			new Item("Yanone Kaffeesatz:300,500", "Yanone Kaffeesatz")
		});
		
		JComboBox headlineFont = new JComboBox(new Object[] {
			new Item("", "[ " + getText("ui.sameAsBaseFont") + " ]"),
			new Item("Abril Fatface", "Abril Fatface"),
			new Item("Alex Brush", "Alex Brush"),
			new Item("Amaranth", "Amaranth"),
			new Item("Amatic SC:700", "Amatic SC Bold"),
			new Item("Anton", "Anton"),
			new Item("Arapey", "Arapey"),
			new Item("Barlow:300", "Barlow Light"),
			new Item("Barlow:500", "Barlow Medium"),
			new Item("Barlow Semi Condensed:300", "Barlow Semi Condensed Light"),
			new Item("Barlow Condensed", "Barlow Condensed"),
			new Item("Cantata One", "Cantata One"),
			new Item("Cinzel Decorative", "Cinzel Decorative"),
			new Item("Cookie", "Cookie"),
			new Item("Dancing Script", "Dancing Script"),
			new Item("Dynalight", "Dynalight"),
			new Item("Dosis:600", "Dosis Bold"),
			new Item("Economica", "Economica"),
			new Item("Emilys Candy", "Emilys Candy"),
			new Item("Euphoria Script", "Euphoria Script"),
			new Item("Exo 2:300", "Exo 2"),
			new Item("Fauna One", "Fauna One"),
			new Item("Fjalla One", "Fjalla One"),
			new Item("Fredericka the Great", "Fredericka the Great"),
			new Item("Geo", "Geo"),
			new Item("Gilda Display", "Gilda Display"),
			new Item("Grand Hotel", "Grand Hotel"),
			new Item("Great Vibes", "Great Vibes"),
			new Item("Gruppo", "Gruppo"),
			new Item("Handlee", "Handlee"),
			new Item("IBM Plex Serif:300", "IBM Plex Serif"),
			new Item("IM Fell English", "IM Fell English"),
			new Item("Italiana", "Italiana"),
			new Item("Josefin Slab", "Josefin Slab"),
			new Item("Julius Sans One", "Julius Sans One"),
			new Item("Jura", "Jura"),
			new Item("La Belle Aurore", "La Belle Aurore"),
			new Item("Libre Baskerville", "Libre Baskerville"),
			new Item("Lobster", "Lobster"),
			new Item("Lobster Two", "Lobster Two"),
			new Item("Lora", "Lora"),
			new Item("Maiden Orange", "Maiden Orange"),
			new Item("Martel:300", "Martel"),
			new Item("Marvel:700", "Marvel Bold"),
			new Item("Medula One", "Medula One"),
			new Item("Merriweather:300", "Merriweather"),
			new Item("Mountains of Christmas", "Mountains of Christmas"),
			new Item("Noticia Text", "Noticia Text"),
			new Item("Noto Serif Display:300", "Noto Serif Light"),
			new Item("Old Standard TT", "Old Standard TT"),
			new Item("Oranienbaum", "Oranienbaum"),
			new Item("Oswald", "Oswald"),
			new Item("Philosopher", "Philosopher"),
			new Item("Poiret One", "Poiret One"),
			new Item("Prata", "Prata"),
			new Item("Princess Sofia", "Princess Sofia"),
			new Item("PT Mono", "PT Mono"),
			new Item("PT Sans Narrow", "PT Sans Narrow"),
			new Item("Raleway:300", "Raleway"),
			new Item("Raleway:600", "Raleway Bold"),
			new Item("Roboto Condensed", "Roboto Condensed"),
			new Item("Roboto Slab:300", "Roboto Slab"),
			new Item("Roboto Slab:600", "Roboto Slab Bold"),
			new Item("Rochester", "Rochester"),
			new Item("Shadows Into Light Two", "Shadows Into Light Two"),
			new Item("Scope One", "Scope One"),
			new Item("Six Caps", "Six Caps"),
			new Item("Sofia", "Sofia"),
			new Item("Sorts Mill Goudy", "Sorts Mill Goudy"),
			new Item("Special Elite", "Special Elite"),
			new Item("Squada One", "Squada One"),
			new Item("Strait", "Strait"),
			new Item("Unica One", "Unica One"),
			new Item("Vidaloka", "Vidaloka"),
			new Item("Yanone Kaffeesatz:300", "Yanone Kaffeesatz"),
			new Item("Yanone Kaffeesatz:500", "Yanone Kaffeesatz Bold")
		});
		
		JCheckBox folderTitleSameAsHeadline = new JCheckBox(getText("ui.sameAsHeadline"), true);
		JComboBox heroType = new JComboBox(new Object[] {
			new Item("fullwidth", getText("ui.fullWidth")),
			new Item("card", getText("ui.card")),
			new Item("turntable", getText("ui.turntable"))
		});
		
		JComboBox folderThumbType = new JComboBox(new Object[] {
			new Item("circle", getText("ui.circle")),
			new Item("rounded", getText("ui.rounded")),
			new Item("square", getText("ui.square"))
		});
		
		private void setMakeSlides() {
			boolean ov = engine.isSlides(),
					nv = (showShare.isSelected() &&				// Share individual slides?
							(shareFacebook.isSelected() || shareThreads.isSelected() || shareBluesky.isSelected())) 
						|| 
						(writeSitemapXml.isSelected() && sitemapIncludeSlides.isSelected());
			
			if (ov != nv) {
				try {
					window.ui2Engine();
					engine.setSlides(nv);
					window.engine2UI();
				} catch (Exception ex) {
					throw new RuntimeException(ex);
				}
			}
		};
		
		// These need to be global to watch independent slide pages requirement
		
		JCheckBox showShare = new JCheckBox(getText("ui.share"), true);
		JCheckBox shareFacebook = new JCheckBox("Facebook");
		JCheckBox shareThreads = new JCheckBox("Threads");
		JCheckBox shareBluesky = new JCheckBox("Bluesky");
		JCheckBox writeSitemapXml = new JCheckBox(getText("ui.createSitemapXml"));
		JCheckBox sitemapIncludeSlides = new JCheckBox(getText("ui.includeSlides"), true);
		
		// Refreshing design preview
		
		private WebViewBrowser designPreview = new WebViewBrowser();

		private void refreshDesignPreview() {
			String		html,
						sName = styleName.getText(),
						stylePath,
						txtColor = heroTextColor.toString();
			Double		sidebarAlpha = getAlpha(sidebarBackgroundColor.toString()),
						textLuminosity = getLuminosity(txtColor);
			String 		bgRepeat = getSelectedValue(backgroundRepeat),
						hbgRepeat = getSelectedValue(heroBackgroundRepeat),
						bgColor = backgroundColor.toString(),
						sidebarBgColor = sidebarBackgroundColor.toString(),
						lightText = ((textLuminosity > 0.86)? txtColor : "#eeeeee"),
						darkText = ((textLuminosity < 0.3)? txtColor : "#333333"),
						sidebarTextColor = isLightColor((sidebarAlpha > 0.5)? sidebarBgColor : bgColor)?
												darkText : lightText,
						buttonTextColor = (getLuminosity(buttonColor.toString()) > 0.75)?
												darkText : lightText,
						panelBgColor = (sidebarAlpha < 0.5)? getCssColor(bgColor) :
										((sidebarAlpha >= 0.8)? 
											getCssColor(sidebarBgColor)
											: 
											setAlpha(sidebarBackgroundColor.toString(), 0.9)
										);
			
			try {
				stylePath = (new File(skinDirectory, "styles/" + sName).toURI().toURL()).toString();
			} catch (MalformedURLException ex) {
				Logger.getLogger(Gui.class.getName()).log(Level.SEVERE, null, ex);
				stylePath = "";
			}
			
			html = designTemplate
				.replace("{styleName}", sName)
				.replace("{backgroundColor}", getCssColor(bgColor))
				.replace("{heroBackgroundColor}", getCssColor(heroBackgroundColor.toString()))
				.replace("{sidebarBackgroundColor}", getCssColor(sidebarBgColor))
				.replace("{sidebarTextColor}", sidebarTextColor)
				.replace("{panelBackgroundColor}", panelBgColor)
				.replace("{heroTextColor}", heroTextColor.toString())
				.replace("{linkColor}", linkColor.toString())
				.replace("{hoverColor}", hoverColor.toString())
				.replace("{buttonColor}", getCssColor(buttonColor.toString()))
				.replace("{buttonTextColor}", buttonTextColor)
				.replace("{heroType}", getSelectedValue(heroType))
				.replace("{folderThumbType}", getSelectedValue(folderThumbType))
				.replace("{backgroundImage}", "url(" + stylePath + "/" + backgroundImageName.getText() + ")")
				.replace("{backgroundPosition}", getSelectedValue(backgroundPos))
				.replace("{backgroundRepeat}",  bgRepeat.equals("stretch")? "background-size: cover" : ("background-repeat: " + bgRepeat))
				.replace("{sidebarBackgroundImage}", "url(" + stylePath + "/" + sidebarBackgroundImageName.getText() + ")")
				.replace("{heroBackgroundImage}", "url(" + stylePath + "/" + heroBackgroundImageName.getText() + ")")
				.replace("{heroBackgroundPosition}", getSelectedValue(heroBackgroundPos))
				.replace("{heroBackgroundRepeat}", hbgRepeat.equals("stretch")? "background-size: cover" : ("background-repeat: " + hbgRepeat));
				
			html = getFonts(html, getSelectedValue(fontFamily), (headlineFont.getSelectedIndex() == 0)? "" : getSelectedValue(headlineFont));
			//System.out.println("\nPreview reload triggered by: " + src + "\n\n" + html + "\nStyle = " + sName);
			
			final String html1 = html;
			
			SwingUtilities.invokeLater(() -> {
				designPreview.loadContent(html1);
			});
			//writeFile("D:/Temp/index-preview.html", html);
		};
		
		private PropertyChangeListener setupMonitors = pce -> {
			
			StateMonitor.monitoring(fontSuggestions).onChange(src -> {
				if (isSkinReady() && src != null) {
					setFontBoxes();
				}
			});

			// Separate slide page requirement
			StateMonitor.monitoring(
					shareFacebook,
					shareThreads,
					shareBluesky,
					showShare,
					writeSitemapXml,
					sitemapIncludeSlides).onChange(src -> {
				setMakeSlides();
			});

			// Index preview update
			StateMonitor.monitoring(
					styleName,
					backgroundColor, 
					backgroundPos,
					backgroundRepeat,
					heroBackgroundColor,
					heroBackgroundPos,
					heroBackgroundRepeat,
					sidebarBackgroundColor, 
					heroTextColor, 
					linkColor, 
					hoverColor,
					buttonColor,
					fontFamily, 
					headlineFont, 
					folderTitleSameAsHeadline, 
					heroType,
					folderThumbType
				).onChange((Object src) -> {

				if (isSkinReady() && src != null) {
					refreshDesignPreview();
				}
			});
				
			// Initial preview rendering
			
			refreshDesignPreview();
		};
		
		/*	---------------------------------------------------------------
									Site
			--------------------------------------------------------------- */
		
		// Design
		
		ControlPanel pageDesign = new ControlPanel() {

			JComboBox<Item> language = new JComboBox<Item>() {
				{
					setModel(Languages.modelFrom(new File(skinDirectory, "texts")));
					insertItemAt(new Item("jalbum", "[ " + getText("ui.jAlbumPreference") + " ]"), 0);
					setSelectedIndex(0);
				}
			};
				
			JComboBox modalWindowsTheme = new JComboBox(new Object[] {
				new Item("auto", getText("ui.auto")),
				new Item("light", getText("ui.light")),
				new Item("dark", getText("ui.dark"))
			});

			JComboBox iconStyle = new JComboBox(new Object[] {
				new Item("thin", getText("ui.thin")),
				new Item("fat", getText("ui.fat"))
			});
				
			ControlPanel pageBackground = new ControlPanel() {
				
				JButton selectImage = new JButton();
				
				{
					selectImage.setText(getText("ui.select"));
					selectImage.addActionListener((ActionEvent e) -> {
							getFileToRes(imageFiles, backgroundImageName, skinUi);
					});
					backgroundColor.setToolTipText(getText("ui.backgroundColorInfo"));
					backgroundColor.getTextComponent().setFont(mono);
					
					add("", new JLabelFor(getText("ui.backgroundColor"), backgroundColor));
					add("tab", backgroundColor);
					add("tab", backgroundColor.getTextComponent());
					add("br", new JLabelFor(getText("ui.backgroundImage"), backgroundImageName));
					add("tab", backgroundImageName);
					add("", selectImage);
					add("br", new JLabelFor(getText("ui.position"), backgroundPos));
					add("tab", backgroundPos);
					add(" ", backgroundRepeat);
				}
			};
			
			ControlPanel sidebarBackground = new ControlPanel() {
			
				JButton selectImage = new JButton();
				
				{
					selectImage.setText(getText("ui.select"));
					selectImage.addActionListener((ActionEvent e) -> {
							getFileToRes(imageFiles, sidebarBackgroundImageName, skinUi);
					});
					sidebarBackgroundColor.getTextComponent().setFont(mono);
					sidebarBackgroundColor.setToolTipText(getText("ui.sidebarColorInfo"));
					
					add("", new JLabelFor(getText("ui.backgroundColor"), sidebarBackgroundColor));
					add("tab", sidebarBackgroundColor);
					add("tab", sidebarBackgroundColor.getTextComponent());
					add("br", new JLabelFor(getText("ui.backgroundImage"), sidebarBackgroundImageName));
					add("tab", sidebarBackgroundImageName);
					add("", selectImage);
				}
			};
			
			ControlPanel heroBackground = new ControlPanel() {
				
				JButton selectImage = new JButton();
				
				{
					heroBackgroundColor.setToolTipText(getText("ui.coverColorInfo"));
					heroBackgroundColor.getTextComponent().setFont(mono);
					heroTextColor.getTextComponent().setFont(mono);
					heroTextColor.setToolTipText(getText("ui.textColorInfo"));
					selectImage.setText(getText("ui.select"));
					selectImage.addActionListener((ActionEvent e) -> {
							getFileToRes(imageFiles, heroBackgroundImageName, skinUi);
					});
					
					add("", new JLabelFor(getText("ui.backgroundColor"), heroBackgroundColor));
					add("tab", heroBackgroundColor);
					add("tab", heroBackgroundColor.getTextComponent());
					add("br", new JLabelFor(getText("ui.textColor"), heroTextColor));
					add("tab", heroTextColor);
					add("tab", heroTextColor.getTextComponent());
					add("br", new JLabelFor(getText("ui.backgroundImage"), heroBackgroundImageName));
					add("tab", heroBackgroundImageName);
					add("", selectImage);
					add("br", new JLabelFor(getText("ui.position"), heroBackgroundPos));
					add("tab", heroBackgroundPos);
					add(" ", heroBackgroundRepeat);
				}
			};
			
			JTabbedPane backgroundTabs = new JTabbedPane() {

				{
					addTab("<html><h4 " + tabStyle + ">" + getText("ui.coverPage") + "</h4></html>", icon("hero"), heroBackground);
					addTab("<html><h4 " + tabStyle + ">" + getText("ui.sidebar") + "</h4></html>", icon("sidebar-left"), sidebarBackground);
					addTab("<html><h4 " + tabStyle + ">" + getText("ui.lightbox") + "</h4></html>", icon("lightbox"), pageBackground);
				}
			};
			
			ControlPanel siteSettings = new ControlPanel() {
				
				{
					language.setToolTipText(getText("ui.languageInfo"));
					buttonColor.getTextComponent().setFont(mono);	
					linkColor.setToolTipText(getText("ui.linkColorInfo"));
					linkColor.getTextComponent().setFont(mono);
					hoverColor.setToolTipText(getText("ui.hoverColorInfo"));
					hoverColor.getTextComponent().setFont(mono);
					
					add("br", new JLabelFor(getText("ui.language"), language));
					add("tab", language);
					add("br", new JLabelFor(getText("ui.linkColor"), linkColor));
					add("tab", linkColor);
					add("tab", linkColor.getTextComponent());
					add("br", new JLabelFor(getText("ui.hoverColor"), hoverColor));
					add("tab", hoverColor);
					add("tab", hoverColor.getTextComponent());
					add("br", new JLabelFor(getText("ui.button"), buttonColor));
					add("tab", buttonColor);
					add("tab", buttonColor.getTextComponent());
					add("br", new JLabelFor(getText("ui.modalWindowsTheme"), modalWindowsTheme));
					add("tab", modalWindowsTheme);
					add("br", new JLabel(getText("ui.iconStyle")));
					add("tab", iconStyle);
					
				}
			};
				
			{
				((RiverLayout)(getLayout())).setVgap(0);
				((RiverLayout)(getLayout())).setHgap(0);
				add("hfill", siteSettings);
				add("br hfill", backgroundTabs);
				
			}
		};

		// Typography
		
		ControlPanel typography = new ControlPanel() {

			{
				headlineFont.setEditable(true);

				add("", new JLabelFor(getText("ui.headlineFamily"), headlineFont));
				add("tab", headlineFont);
				add("br", new JLabelFor(getText("ui.fontFamily"), fontFamily));
				add("tab", fontFamily);
				add("br", new JLabelFor(getText("ui.pairingSuggestions"), fontSuggestions));
				add("tab", fontSuggestions);
			}
		};

		// Cover header

		ControlPanel coverHeader = new ControlPanel() {
			
			
			JTextField logoName = new JSmartTextField(16);
			JButton selectLogo = new JButton(getText("ui.select"));
			JCheckBox showBreadcrumbPath = new JCheckBox(getText("ui.showBreadcrumbPath"));
			
			ControlPanel topMenu = new ControlPanel(getText("ui.topMenu")) {
				JCheckBox topMenuIncludeFolders = new JCheckBox(getText("ui.includeFolders"));
				JCheckBox topMenuIncludePages = new JCheckBox(getText("ui.includePages"));
				JCheckBox topMenuIncludeWebLocations = new JCheckBox(getText("ui.includeWebLocations"));
						
				{
					add("", topMenuIncludeFolders);
					add("br", topMenuIncludePages);
					add("br", topMenuIncludeWebLocations);
				}
			};
			
			JSmartTextArea header = new JSmartTextArea(4, 20);
			JScrollPane headerPane = new JScrollPane(header);
			
			{
				selectLogo.setText(getText("ui.select"));
				selectLogo.addActionListener(new ActionListener() { 
					@Override
					public void actionPerformed(ActionEvent e) {
						getFileToRes(imageFiles, logoName, skinUi);
				}});
				showBreadcrumbPath.setToolTipText(getText("ui.showBreadcrumbPathInfo"));
				allowHTMLEditing(header);
				//allowSpelling(header);
				header.setEditable(true);
				header.setLineWrap(true);
				header.setWrapStyleWord(true);
				header.setFont(mono);
				header.setTabSize(2);
				header.setToolTipText(getText("ui.customContentInfo"));
				headerPane.setBorder(BorderFactory.createTitledBorder(getText("ui.customContent") + ": " + getText("ui.addedAtCoverPageTop")));
				
				add("", new JLabel(getText("ui.logo")));
				add("tab", logoName);
				add("", selectLogo);
				add("br hfill", topMenu);
				add("br", showBreadcrumbPath);
				add("br hfill vfill", headerPane);
			}
		};
		
		// Cover title
		
		ControlPanel coverTitle = new ControlPanel() {

			JComboBox<NameValue<String>> presets = new JComboBox<>(new NameValue[] {
				new NameValue(getText("ui.empty"), ""),
				new NameValue(getText("ui.title"), "<h1>${title}</h1>"),
				new NameValue(getText("ui.title") + " + " + getText("ui.description"), "<h1>${title}</h1>\n<div class=\"description\">${description}</div>"),
				new NameValue(getText("ui.title") + " + " + getText("ui.date") + " + " + getText("ui.description") + " (" + getText("ui.default") + ")", "<h1>${title} <span class=\"date\">${folderModDate}</span></h1>\n<div class=\"description\">${description}</div>")
			});
			JTextArea titleCaptionTemplate = new JSmartTextArea(4, 20);
			JScrollPane captionPane = new JScrollPane(titleCaptionTemplate);
			JComboBox folderDateSource = new JComboBox(new Object[]{
				new Item("none", getText("ui.none")),
				new Item("fileDate", getText("ui.fileDate")),
				new Item("folderModDate", getText("ui.folderModDate")),
				new Item("lastCameraDate", getText("ui.lastCameraDate")),
				new Item("cameraDateRange", getText("ui.cameraDateRange"))
			});
			JTextField folderImageSize = new JTextField("1800x800");
			private void setFolderImageSize() {
				
				if (!isSkinReady()) {
					return;
				}
				
				int i = heroType.getSelectedIndex();
				
				if (i == 0) {
					// Full width
					folderImageSize.setText("1800x800");
				} else if (i == 1) {
					// Card
					folderImageSize.setText("1000x600");
				} else {
					// Turntable
					folderImageSize.setText("320x320");
				}
				
				
			};
			
			{
				heroType.addActionListener((ActionEvent e) -> {
					setFolderImageSize();
				});
				folderImageSize.setEnabled(false);
				titleCaptionTemplate.setEditable(true);
				titleCaptionTemplate.setLineWrap(true);
				titleCaptionTemplate.setWrapStyleWord(true);
				titleCaptionTemplate.setFont(mono);
				titleCaptionTemplate.setTabSize(2);
				captionPane.setBorder(BorderFactory.createTitledBorder(getText("ui.captionTemplate")));
				presets.addItemListener(listener -> {
					titleCaptionTemplate.setText(((NameValue<String>)presets.getSelectedItem()).value);
				});
				
				add("", new JLabel(getText("ui.type")));
				add("tab", heroType);
				add("tab", folderImageSize);
				add("br", new JLabelFor(getText("ui.presets"), presets));
				add("tab", presets);
				add("br hfill", captionPane);
				add("br", new JLabelFor("${folderModDate} =", folderDateSource));
				add("tab", folderDateSource);
			}

		};

		// Folders 
		
		ControlPanel folders = new ControlPanel() {

			JComboBox<NameValue<String>> presets = new JComboBox<>(new NameValue[] {
				new NameValue(getText("ui.empty"), ""),
				new NameValue(getText("ui.title"), "<h3>${title}</h3>"),
				new NameValue(getText("ui.title") + " + " + getText("ui.comment"), "<h3>${title}</h3>\n<div class=\"comment\">${comment}</div>"),
				new NameValue(getText("ui.title") + " + " + getText("ui.commentShort") + " (" + getText("ui.default") + ")", "<h3>${title}</h3>\n<div class=\"comment\">${commentShort}</div>"),
				new NameValue(getText("ui.fileLabel"), "<h3>${fileLabel}</h3>"),
				new NameValue(getText("ui.fileLabel") + " + " + getText("ui.comment"), "<h3>${fileLabel}</h3>\n<div class=\"comment\">${comment}</div>")
			});
			JTextArea folderCaptionTemplate = new JSmartTextArea(4, 30);
			JScrollPane captionPane = new JScrollPane(folderCaptionTemplate);
			JCheckBox showFolderImageCount = new JCheckBox(getText("ui.showFolderImageCount"), true);
			JCheckBox gatherWeblocationInfo = new JCheckBox(getText("ui.gatherWeblocationInfo"), true);
			JCheckBox webLocationOpenNew = new JCheckBox(getText("ui.webLocationsOpenInNewWindow"));

			{					
				presets.addItemListener(listener -> {
					if (!isSkinReady()) {
						return;
					}
					folderCaptionTemplate.setText(((NameValue<String>)presets.getSelectedItem()).value);
				});	
				folderCaptionTemplate.setEditable(true);
				folderCaptionTemplate.setLineWrap(true);
				folderCaptionTemplate.setWrapStyleWord(true);
				folderCaptionTemplate.setFont(mono);
				folderCaptionTemplate.setTabSize(2);
				captionPane.setBorder(BorderFactory.createTitledBorder(getText("ui.captionTemplate")));
				showFolderImageCount.setToolTipText(getText("ui.showFolderImageCountInfo"));
				gatherWeblocationInfo.setToolTipText(getText("ui.gatherWeblocationInfoInfo"));

				add(new JLabel(getText("ui.type")));
				add("tab", folderThumbType);
				add("br", new JLabelFor(getText("ui.presets"), presets));
				add("tab", presets);
				add("br hfill", captionPane);
				add("br", showFolderImageCount);
				add("br", gatherWeblocationInfo);
				add("br", webLocationOpenNew);
			}
		};
		
		// Footer
		
		ControlPanel footerTab = new ControlPanel() {

			JCheckBox showModifiedDate = new JCheckBox(getText("ui.showModifiedDate"));	
			JCheckBox showImageCount = new JCheckBox(getText("ui.showFolderImageCount"));	
			JSmartTextArea footer = new JSmartTextArea(8, 20);
			JScrollPane footerPane = new JScrollPane(footer);
			
			ControlPanel customLinkPanel = new ControlPanel(getText("ui.customLink")) {

				JTextField customLink = new JSmartTextField(16);
				JTextField customLinkText = new JSmartTextField(16);

				{
					customLink.setToolTipText(getText("ui.customLinkInfo"));
					customLinkText.setToolTipText(getText("ui.customLinkTextInfo"));
					allowHTMLEditing(footer);
					
					add(new JLabel("URL"));
					add("tab hfill", customLink);
					add("br", new JLabel(getText("ui.customLinkText")));
					add("tab hfill", customLinkText);
				}
			};

			{
				showImageCount.setToolTipText(getText("ui.showFolderImageCountInfo"));
				footer.setEditable(true);
				footer.setLineWrap(true);
				footer.setWrapStyleWord(true);
				footer.setFont(mono);
				footer.setTabSize(2);
				footer.setToolTipText(getText("ui.customContentInfo"));
				footerPane.setBorder(BorderFactory.createTitledBorder(getText("ui.customFooterText")));

				add("", showModifiedDate);
				add("br", showImageCount);
				add("br hfill", customLinkPanel);				
				add("br hfill vfill", footerPane);
			}

		};
		
		/*	---------------------------------------------------------------
									Sidebar
			--------------------------------------------------------------- */
			
		// Top navigation
		
		ControlPanel topNavigation = new ControlPanel() {
			
			JComboBox topNavigationDisplay = new JComboBox(new Object[] {
				new Item("button", getText("ui.onlyButton")),
				new Item("auto", getText("ui.dependsOnWindowSize")),
				new Item("always", getText("ui.alwaysShow"))
			});
			JCheckBox topNavigationIncludeFolders = new JCheckBox(getText("ui.includeFolders"), true);
			JCheckBox topNavigationIncludePages = new JCheckBox(getText("ui.includePages"));
			JCheckBox topNavigationIncludeWebLocations = new JCheckBox(getText("ui.includeWebLocations"));
			JSpinner topNavigationDepth = new JSpinner(new SpinnerNumberModel(4, 1, 6, 1));

			{
				add(new JLabelFor(getText("ui.folderTreeVisibility"), topNavigationDisplay));
				add(" ", topNavigationDisplay);
				add("br", topNavigationIncludeFolders);
				add("br", topNavigationIncludePages);
				add("br", topNavigationIncludeWebLocations);
				add("br", new JLabel(getText("ui.depth")));
				add("", topNavigationDepth);
				add("", new JLabel(getText("ui.levels")));
			}
		};
		
		// Search
		
		ControlPanel search = new ControlPanel() {

			JCheckBox useSearch = new JCheckBox(getText("ui.useSearch"));
			JTextField searchFields = new JSmartTextField(16);
			JLabel fieldsInfo = new JLabel("<html><i>" + getText("ui.searchFieldsInfo") + "</i></html>");

			{
				ComponentUtilities.whenSelectedEnable(useSearch, new JComponent[]{searchFields, fieldsInfo});

				add("", useSearch);
				add("br", new JLabelFor(getText("ui.fields"), searchFields));
				add("tab hfill", searchFields);
				add("br", new JLabel(""));
				add("tab", fieldsInfo);
			}
		};
				
		// Tag cloud
		
		ControlPanel tagCloud = new ControlPanel() {

			JComboBox tagCloudSource = new JComboBox(new Object[] {
				new Item("none", getText("ui.noTagCloud")),			
				new Item("current", getText("ui.currentFolder")),			
				new Item("subfolders", getText("ui.subfolders")),
				new Item("tree", getText("ui.wholeAlbum"))
			});
			JCheckBox tagCloudUseFolders = new JCheckBox(getText("ui.folders"), true);
			JCheckBox tagCloudUsePages = new JCheckBox(getText("ui.pages"));
			JCheckBox tagCloudUseWebLocations = new JCheckBox(getText("ui.webLocations"));
			JLabel tagCloudSkipLevelsLabel = new JLabel(getText("ui.skipLevels"));
			JSpinner tagCloudSkipLevels = new JSpinner(new SpinnerNumberModel(1, 0, 5, 1));
			JTextField tagCloudLabel = new JSmartTextField(12);
			JTextField tagCloudFields = new JSmartTextField(16);
			JComboBox tagCloudSort = new JComboBox(new Object[] {
				new Item("none", getText("ui.unsorted")),			
				new Item("name", getText("ui.name")),			
				new Item("frequency", getText("ui.frequency"))
			});
			JCheckBox tagCloudFontVaries = new JCheckBox(getText("ui.tagCloudFontVaries"));
			JCheckBox tagCloudSearch = new JCheckBox(getText("ui.addSearchBox"));

			{
				tagCloudLabel.setText(getText("ui.labels"));
				tagCloudSource.addActionListener(new ActionListener() {
					@Override
					public void actionPerformed(ActionEvent e) {
						int tsi = tagCloudSource.getSelectedIndex();
						tagCloudUseFolders.setEnabled(tsi > 0);
						tagCloudUsePages.setEnabled(tsi > 0);
						tagCloudUseWebLocations.setEnabled(tsi > 0);
						tagCloudLabel.setEnabled(tsi > 0);
						tagCloudFields.setEnabled(tsi > 0);
						tagCloudSort.setEnabled(tsi > 0);
						tagCloudFontVaries.setEnabled(tsi > 0);
						tagCloudSearch.setEnabled(tsi > 0);
						tagCloudSkipLevels.setVisible(tsi == 2);
						tagCloudSkipLevelsLabel.setVisible(tsi == 2);
				}});
				tagCloudSkipLevels.setToolTipText(getText("ui.skipLevelsInfo"));
				tagCloudFields.setToolTipText("<html>" + getText("ui.searchFieldsInfo") + "</html>");
				tagCloudFontVaries.setToolTipText(getText("ui.tagCloudFontVariesInfo"));

				add(new JLabelFor(getText("ui.collectFrom"), tagCloudSource));
				add("tab", tagCloudSource);
				add("tab", tagCloudSkipLevelsLabel);
				add("", tagCloudSkipLevels);
				add("", tagCloudUseFolders);
				add("", tagCloudUsePages);
				add("", tagCloudUseWebLocations);
				add("br", new JLabelFor(getText("ui.boxTitle"), tagCloudLabel));
				add("tab hfill", tagCloudLabel);
				add("br", new JLabelFor(getText("ui.fields"), tagCloudFields));
				add("tab hfill", tagCloudFields);
				add("br", new JLabelFor(getText("ui.sortBy"), tagCloudSort));
				add("tab", tagCloudSort);
				add("tab", tagCloudFontVaries);
				add("br tab", tagCloudSearch);
			}
		};

		// Search new
		
		ControlPanel searchNew = new ControlPanel() {

			JComboBox searchNewSource = new JComboBox(new Object[] {
				new Item("none", getText("ui.noSearchNew")),			
				new Item("current", getText("ui.currentFolder")),			
				new Item("subfolders", getText("ui.subfolders")),
				new Item("tree", getText("ui.wholeAlbum"))
			});
			//JTextField searchNewLabel = new JSmartTextField(12);
			JTextField searchNewDays = new JSmartTextField(12);
			JComboBox searchNewReference = new JComboBox(new Object[] {
				new Item("dateTaken", getText("ui.dateTaken")),
				new Item("fileModified", getText("ui.fileModified")),
				new Item("added", getText("ui.addedToAlbum"))
			});
			JCheckBox searchNewSinceLastVisit = new JCheckBox(getText("ui.searchNewSinceLastVisit"));

			{
				searchNewSource.addActionListener(new ActionListener() {
					@Override
					public void actionPerformed(ActionEvent e) {
						boolean on = searchNewSource.getSelectedIndex() > 0;
						//searchNewLabel.setEnabled(on);
						searchNewDays.setEnabled(on);
						searchNewReference.setEnabled(on);
						searchNewSinceLastVisit.setEnabled(on);
				}});
				//searchNewLabel.setText(getText("ui.newImages"));
				searchNewDays.setToolTipText(getText("ui.searchNewDaysInfo"));
				searchNewSinceLastVisit.setToolTipText(getText("ui.searchNewSinceLastVisitInfo"));

				add(new JLabelFor(getText("ui.collectFrom"), searchNewSource));
				add("tab", searchNewSource);
				//add("br", new JLabelFor(getText("ui.boxTitle"), searchNewLabel));
				//add("tab hfill", searchNewLabel);
				add("br", new JLabelFor(getText("ui.days"), searchNewDays));
				add("tab hfill", searchNewDays);
				add("br", new JLabelFor(getText("ui.reference"), searchNewReference));
				add("tab", searchNewReference);
				add("br tab", searchNewSinceLastVisit);
				
				putClientProperty("helpPage", helpRoot + "Sections/Search_new_images");
			}
		};

		// Map
		
		ControlPanel map = new ControlPanel() {

			JCheckBox showMapSection = new JCheckBox(getText("ui.showMap"), true);
			JComboBox mapType = new JComboBox(new Object[]{
				new Item("roadmap", getText("ui.roadmap")),
				new Item("satellite", getText("ui.satellite")),
				new Item("hybrid", getText("ui.hybrid")),
				new Item("terrain", getText("ui.terrain"))
			});
			JSlider mapZoom = new JSlider(JSlider.HORIZONTAL, 1, 20, 18);
			JTextField googleApiKey = new JSmartTextField(32);
			JLabel mandatoryInfo = new JLabel(mandatory);
					
			{
				showMapSection.setToolTipText(getText("ui.showMapSectionInfo"));
				mapZoom.setOrientation(JSlider.HORIZONTAL);
				mapZoom.setMinimum(0);
				mapZoom.setMaximum(20);
				mapZoom.setValue(18);
				mapZoom.setMajorTickSpacing(10);
				mapZoom.setMinorTickSpacing(1);
				mapZoom.setPaintTicks(true);
				mapZoom.setPaintLabels(true);
				mapZoom.setSnapToTicks(true);
				mandatoryInfo.setToolTipText(getText("ui.mandatory"));

				//ComponentUtilities.whenSelectedEnable(showMap, new JComponent[]{ mapAll, mapType, mapZoom });
				add(showMapSection);
				add("br", new JLabelFor(getText("ui.initialView"), mapType));
				add("tab", mapType);
				add("br", new JLabelFor(getText("ui.initialZoom"), mapZoom));
				add("tab", mapZoom);
				add("br", new JLabelFor(getText("ui.googleApiKey"), googleApiKey));
				add(mandatoryInfo);
				add("tab", googleApiKey);
				add("br tab", new JLinkLabel("https://console.developers.google.com/apis/credentials", getText("ui.createNew")));
				add("br tab", new JLabel("<html><i>" + getText("ui.googleApiKeyInfo") + "</i></html>"));
				add("br tab", new JLinkLabel("https://developers.google.com/maps/documentation/javascript/tutorial", getText("ui.readMore")));

				//putClientProperty("helpPage", helpRoot + "ui/map.html");
			}
		};
		
		//	Social
		
		ControlPanel social = new ControlPanel() {
			
			ControlPanel shares = new ControlPanel(getText("ui.shareButtonFor")) {

				JCheckBox shareTwitter = new JCheckBox("X");
				JCheckBox sharePinterest = new JCheckBox("Pinterest");
				JCheckBox shareLinkedin = new JCheckBox("LinkedIn");
				JCheckBox shareDigg = new JCheckBox("Digg");
				JCheckBox shareReddit = new JCheckBox("Reddit");
				JCheckBox shareTumblr = new JCheckBox("Tumblr");

				{
					setLayout(new RiverLayout(4, 5));
					add(new JLabel(icon("facebook")));
					add(shareFacebook);
					add("br", new JLabel(icon("threads")));
					add(shareThreads);
					add("br", new JLabel(icon("bluesky")));
					add(shareBluesky);
					add("br", new JLabel(icon("x")));
					add(shareTwitter);
					add("br", new JLabel(icon("reddit")));
					add(shareReddit);
					add("br", new JLabel(icon("pinterest")));
					add(sharePinterest);
					add("br", new JLabel(icon("tumblr")));
					add(shareTumblr);
					add("br", new JLabel(icon("linkedin")));
					add(shareLinkedin);
					add("br", new JLabel(icon("digg")));
					add(shareDigg);
				}
			};

			ControlPanel links = new ControlPanel() {
				
				JCheckBox shareEmail = new JCheckBox("Email");
				JLabel emailSubjectLabel = new JLabel(getText("ui.subject"));
				JTextField emailSubject = new JSmartTextField(18);
				JTextArea emailBody = new JSmartTextArea(4,20);
				JScrollPane emailBodyPane = new JScrollPane(emailBody);
				JCheckBox shareLink = new JCheckBox(getText("ui.link"));
				
				{
					emailBody.setEditable(true);
					emailBody.setLineWrap(true);
					emailBody.setWrapStyleWord(true);
					emailBody.setToolTipText(getText("ui.emailBodyInfo"));
					emailBodyPane.setBorder(BorderFactory.createTitledBorder(getText("ui.body")));
					ComponentUtilities.whenSelectedEnable(shareEmail, new JComponent[]{emailSubjectLabel, emailSubject, emailBodyPane});
					
					add("", new JLabel(icon("email")));
					add("tab", shareEmail);
					add("br tab", emailSubjectLabel);
					add("", emailSubject);
					add("br tab", emailBodyPane);
					add("br", new JLabel(icon("link")));
					add("tab", shareLink);
				}
			};
			
			JTextField facebookAppId = new JSmartTextField("");
			JLabel mandatoryInfo = new JLabel(mandatory);
			
			{
				mandatoryInfo.setToolTipText(getText("ui.mandatory"));
				links.setBorder(emptyBorder);
				
				add("", shares);
				add("hfill", links);
				add("br", new JLabelFor(getText("ui.facebookAppId"), facebookAppId));
				add(mandatoryInfo);
				add("tab hfill", facebookAppId);
				add(new JLinkLabel("https://developers.facebook.com/apps", getText("ui.signUp")));
			}

		};

		// Full screen button
		
		ControlPanel fullScreen = new ControlPanel() {
			
			JCheckBox showFullscreen = new JCheckBox(getText("ui.showFullScreenButton"), false);
			
			{
				add(showFullscreen);
			}
		};
			
		// Background music

		ControlPanel backgroundMusic = new ControlPanel() {
			
			JPlaylist backgroundAudio = new JPlaylist();
			JCheckBox backgroundAudioAutoPlay = new JCheckBox(getText("ui.autoStart"));
			JSpinner backgroundAudioVolume = new JSpinner(new SpinnerNumberModel(25, 1, 100, 1));
			JCheckBox backgroundAudioSlideshowControl = new JCheckBox(getText("ui.slideshowControl"));
			JCheckBox muteBackgroundAudio = new JCheckBox(getText("ui.muteBackgroundAudio"), true);
			JCheckBox backgroundAudioLoop = new JCheckBox(getText("ui.loop"));
			JCheckBox backgroundAudioRetainPosition = new JCheckBox(getText("ui.retainPosition"), true);
			WrappableJLabel autostartNotice = new WrappableJLabel("<html><i>" + getText("ui.autostartNotice") + "</i></html>");

			{
				backgroundAudioSlideshowControl.setToolTipText(getText("ui.slideshowControlInfo"));
				muteBackgroundAudio.setToolTipText(getText("ui.muteBackgroundAudioInfo"));
				backgroundAudioRetainPosition.setToolTipText(getText("ui.retainPositionInfo"));
				autostartNotice.setPreferredWidth(uiWidth - 260);
				
				add("br hfill", backgroundAudio);
				add("br", new JLabel(getText("ui.initialVolume")));
				add("", backgroundAudioVolume);
				add("", new JLabel("%"));
				add("br", backgroundAudioAutoPlay);
				add("tab", backgroundAudioLoop);
				add("br", backgroundAudioRetainPosition);
				add("br", backgroundAudioSlideshowControl);
				add("br", muteBackgroundAudio);
				add("br", new JLabel(infoIcon));
				add("", autostartNotice);
				
				//putClientProperty("helpPage", helpRoot + "ui/background-audio.html");
			}
		};
		
		/*	---------------------------------------------------------------
									Lightbox
			--------------------------------------------------------------- */

		// Thumbnail strip
		
		ControlPanel thumbnailStrip = new ControlPanel() {

			JComboBox thumbstripVisibility = new JComboBox(new Object[] {
				new Item("hidden", getText("ui.alwaysHidden")), 
				new Item("auto", getText("ui.hiddenDuringSlideshow")), 
				new Item("visible", getText("ui.alwaysVisible"))
			});
			JComboBox thumbstripPosition = new JComboBox(new Object[] {
				new Item("top", getText("ui.top")), 
				new Item("right", getText("ui.right")), 
				new Item("bottom", getText("ui.bottom")), 
				new Item("left", getText("ui.left"))
			});
			JCheckBox fixedShapeThumbs = new JCheckBox(getText("ui.fixedShapeThumbs"), false);
			JTextArea thumbCaptionTemplate = new JSmartTextArea(4, 30);
			JComboBox<NameValue<String>> presets = new JComboBox<>(new NameValue[] {
				new NameValue(getText("ui.empty"), ""),
				new NameValue(getText("ui.title"), "<h6>${fileTitle}</h6>"),
				new NameValue(getText("ui.title") + " + " + getText("ui.comment"), "<h6>${fileTitle}</h6>\n<div class=\"comment\">${comment}</div>"),
				new NameValue(getText("ui.title") + " + " + getText("ui.commentShort") + " (" + getText("ui.default") + ")", "<h6>${fileTitle}</h6>\n<div class=\"comment\">${commentShort}</div>"),
				new NameValue(getText("ui.number") + " + " + getText("ui.title") + " + " + getText("ui.commentShort"), "<span class=\"nr\">${imageNum}</span>\n<h6>${fileTitle}</h6>\n<div class=\"comment\">${commentShort}</div>"),
				new NameValue(getText("ui.fileLabel"), "<h6>${fileLabel}</h6>"),
				new NameValue(getText("ui.fileLabel") + " + " + getText("ui.comment"), "<h6>${fileLabel}</h6>\n<div class=\"comment\">${comment}</div>"),
				new NameValue(getText("ui.fileLabel") + " + " + getText("ui.commentShort"), "<h6>${fileLabel}</h6>\n<div class=\"comment\">${commentShort}</div>"),
				new NameValue(getText("ui.number") + " + " + getText("ui.fileLabel") + " + " + getText("ui.commentShort"), "<span class=\"nr\">${imageNum}</span>\n<h6>${fileLabel}</h6>\n<div class=\"comment\">${commentShort}</div>")
			});
			JScrollPane captionPane = new JScrollPane(thumbCaptionTemplate);

			{
				fixedShapeThumbs.setToolTipText(getText("ui.fixedShapeThumbsInfo"));
				presets.addItemListener(listener -> {
					if (!isSkinReady()) {
						return;
					}
					thumbCaptionTemplate.setText(((NameValue<String>)presets.getSelectedItem()).value);
				});	
				captionPane.setBorder(BorderFactory.createTitledBorder(getText("ui.captionTemplate")));
				thumbCaptionTemplate.setEditable(true);
				thumbCaptionTemplate.setLineWrap(true);
				thumbCaptionTemplate.setWrapStyleWord(true);
				thumbCaptionTemplate.setFont(mono);
				thumbCaptionTemplate.setTabSize(2);
				
				add(new JLabel(getText("ui.thumbstripVisibility")));
				add("tab", thumbstripVisibility);
				add("br", new JLabelFor(getText("ui.thumbstripPosition"), thumbstripPosition));
				add("tab", thumbstripPosition);
				add("br", fixedShapeThumbs);
				add("br", new JLabelFor(getText("ui.presets"), presets));
				add("tab", presets);
				add("br hfill", captionPane);

			}
		};

		// Mark new files
		
		ControlPanel markNew = new ControlPanel() {

			JCheckBox markFilesNew = new JCheckBox(getText("ui.markFilesNew"));	
			JTextField newDaysCount = new JSmartTextField("60", 3);
			JComboBox newDaysRef = new JComboBox(new Object[] {
				new Item("dateTaken", getText("ui.dateTaken")),
				new Item("fileModified", getText("ui.fileModified")),
				new Item("added", getText("ui.addedToAlbum"))
			});
			JComboBox newDaysMark = new JComboBox(new Object[] {
				new Item("icon", getText("ui.icon")),
				new Item("text", getText("ui.text"))
			});
			JTextField newDaysText = new JSmartTextField(getText("new"), 6);
			JLabel newIcon = new JLabel(icon("new-fill"));

			{
				newDaysCount.setToolTipText(getText("ui.newDaysCountInfo"));
				newDaysMark.addItemListener(listener -> {
					int i = newDaysMark.getSelectedIndex();
					newDaysText.setVisible(i == 1);
					newIcon.setVisible(i == 0);
				});
				newDaysText.setVisible(false);
				ComponentUtilities.whenSelectedEnable(markFilesNew, new JComponent[]{ newDaysCount, newDaysRef, newDaysMark, newDaysText, newIcon });

				add("", markFilesNew);
				add("", newDaysCount);
				add(new JLabel(getText("ui.daysOld")));
				add("br", new JLabel(getText("ui.reference")));
				add(newDaysRef);
				add("br", new JLabel(getText("ui.marker")));
				add("tab", newDaysMark);
				add(newIcon);
				add(newDaysText);
			}
		};

		// Main image
		
		ControlPanel mainImage = new ControlPanel() {

			@Override
			public void importVariables(Map<String, Object> vars) {
				Map<String, Object> skinVars = engine.getSkinVariables();
				Object sdo = skinVars.get("slideshowDelay");
				if (sdo != null && sdo instanceof Double) {
					if ((Double)sdo > 0.0 && (Double)sdo < 20.0) {
						vars.put("slideshowDelay", (int)((Double)sdo * 1000));
					}
				}
			}
			JCheckBox useFullScreen = new JCheckBox(getText("ui.fullscreenOnMobiles"));
			JCheckBox videoAutoPlay = new JCheckBox(getText("ui.startVideo"));
			JLabel videoAutoPlayInfo = new JLabel(infoIcon);
			JCheckBox hideSidebarDuringSlideshow = new JCheckBox(getText("ui.hideSidebarDuringSlideshow"));
			//JCheckBox use360Player = new JCheckBox(getText("ui.use360Player"));
			JComboBox<Item> transitionType = new JComboBox(new Object[] {
				new Item("crossfade", getText("ui.crossFade")),
				new Item("zoom", getText("ui.zoom")),
				new Item("kenburns", getText("ui.kenBurns")),
				new Item("stack", getText("ui.stack")),
				new Item("slide", getText("ui.slide")),
				new Item("swap", getText("ui.swap")),
				//new Item("carousel", getText("ui.carousel")),
				new Item("flip", getText("ui.flip")),
				new Item("book", getText("ui.book")),
				new Item("cube", getText("ui.cube"))
				//new Item("coverflow", getText("ui.coverflow")),
				//new Item("random", getText("ui.random"))
			});
			//JTextField transitionSpeed = new JSmartTextField("800", 8);
			JSpinner transitionSpeed = new JSpinner(new SpinnerNumberModel(1000, 50, 10000, 50));
			JSpinner slideshowDelay = new JSpinner(new SpinnerNumberModel(3000, 0, 20000, 50));
			//JCheckBox slideshowLoop = new JCheckBox(getText("ui.loop"));
			JCheckBox slideshowAuto = new JCheckBox(getText("ui.autoStartSlideshow"));
			JComboBox afterLast = new JComboBox(new Object[] {
				new Item("donothing", getText("ui.doNothing")), 
				new Item("startover", getText("startOver")),
				new Item("onelevelup", getText("upOneLevel")),
				new Item("backtocover", getText("backToCover")),
				new Item("nextfolder", getText("nextFoldersFirstImage")),
				new Item("ask", getText("ui.ask"))
			});
			JCheckBox neighboringFolderSkipLevels = new JCheckBox(getText("ui.skipFolderLevels"), false);
			JCheckBox neighboringFolderLoop = new JCheckBox(getText("ui.neighboringFolderLoop"), false);
			
			JComboBox fitImages = new JComboBox(new Object[] {
				new Item("no", getText("ui.originalSize")), 
				new Item("vertically", getText("ui.fitVertically")),
				new Item("both", getText("ui.fitBoth")),
				new Item("cover", getText("ui.cover"))
			});

			JCheckBox neverScaleUp = new JCheckBox(getText("ui.neverScaleUp"));
			JSpinner fitPadding = new JSpinner(new SpinnerNumberModel(10, 0, 100, 1));
			JCheckBox panoramaPlayer = new JCheckBox(getText("ui.panoramaPlayer"), false);
			JCheckBox panoramaPlayerAuto = new JCheckBox(getText("ui.autoStart"), true);

			JCheckBox rightClickProtect = new JCheckBox(getText("ui.rightClickProtect"));
				
			ControlPanel behaviorPanel = new ControlPanel(getText("ui.behavior")) {

				{
					useFullScreen.setToolTipText(getText("ui.fullscreenOnMobilesInfo"));
					//use360Player.setToolTipText(getText("ui.use360PlayerInfo"));
					afterLast.setToolTipText(getText("ui.afterLastInfo"));
					videoAutoPlayInfo.addMouseListener(new MouseAdapter() {  
						@Override
						public void mouseReleased(MouseEvent e) {
							JOptionPane.showMessageDialog(window, getText("ui.videoAutoplayInfo1"), "Warning", JOptionPane.WARNING_MESSAGE);
					}});
					neighboringFolderSkipLevels.setToolTipText(getText("ui.skipFolderLevelsInfo"));
					neighboringFolderLoop.setToolTipText(getText("ui.neighboringFolderLoopInfo"));
					transitionSpeed.setToolTipText(getText("ui.transitionSpeedInfo"));
					slideshowDelay.setToolTipText(getText("ui.slideshowDelayInfo"));
									
					//add("tab", use360Player);
					add("", slideshowAuto);
					add("br", new JLabelFor(getText("ui.transition"), transitionType));
					add("tab", transitionType);
					add("tab", new JLabelFor(getText("ui.transitionSpeed"), transitionSpeed));
					add(transitionSpeed);
					add(new JLabel("ms"));
					add("br", new JLabel(getText("ui.slideshowDelay")));
					add("tab", slideshowDelay);
					add(new JLabel("ms"));
					add("br", new JLabelFor(getText("ui.afterLast"), afterLast));
					add("tab", afterLast);
					add("br", neighboringFolderSkipLevels);
					add("tab", neighboringFolderLoop);
					add("br", useFullScreen);
					add("br", hideSidebarDuringSlideshow);
					add("br", videoAutoPlay);
					add("", videoAutoPlayInfo);
				}
			};

			ControlPanel mainImagePanel = new ControlPanel(getText("ui.images")) {

				{
					fitImages.setToolTipText(getText("ui.fitImagesInfo"));
					neverScaleUp.setToolTipText(getText("ui.neverScaleUpInfo"));
					ComponentUtilities.whenSelectedEnable(panoramaPlayer, new JComponent[]{panoramaPlayerAuto});
					rightClickProtect.setToolTipText(getText("ui.rightClickProtectInfo"));
				
					add(new JLabelFor(getText("ui.scaling"), fitImages));
					add("", fitImages);
					add(" ", new JLabelFor(getText("ui.padding"), fitPadding));
					add("", fitPadding);
					add(new JLabel("px"));
					add("br", neverScaleUp);
					add("br", rightClickProtect);
					//add("br", dontStretchBehind);
				}
			};


			ControlPanel navigation = new ControlPanel(getText("ui.navigation")) {
				JCheckBox enableMouseWheel = new JCheckBox(getText("ui.enableMouseWheel"), true);
				JCheckBox enableKeyboard = new JCheckBox(getText("ui.enableKeyboard"), true);
				JComboBox clickAction = new JComboBox(new Object[] {
					new Item("donothing", getText("ui.doNothing")), 
					new Item("nextimage", getText("ui.nextImage")),
					new Item("togglecontrols", getText("ui.toggleThumbstrip"))
				});

				{
					enableMouseWheel.setToolTipText(getText("ui.enableMouseWheelInfo"));
					enableKeyboard.setToolTipText(getText("ui.enableKeyboardInfo"));
					
					add("", enableMouseWheel);
					add("tab", enableKeyboard);
					add("br", new JLabel(getText("ui.clickAction")));
					add("tab", clickAction);
				}
			};
			
			{
				transitionType.addItemListener(listener -> {
					
					//Thread.dumpStack();
					if (!isSkinReady()) {
						return;
					}
					
					double d;
					int t, p;
					Item i = (Item)transitionType.getSelectedItem();

					switch ((String)i.value) {
						case 	"crossfade":	d = 3;		t = 1000;	p = 16;		break;
						case	"zoom":			d = 3;		t = 1000;	p = 16;		break;
						case 	"kenburns":		d = 0;		t = 4000;	p = 0;		break;
						case 	"stack":		d = 2.8;	t = 1200;	p = 32;		break;
						case 	"slide":		d = 3.2;	t = 800;	p = 16;		break;
						case 	"swap":			d = 2;		t = 2000;	p = 48;		break;
						case 	"carousel":		d = 3;		t = 1000;	p = 16;		break;
						case 	"flip":			d = 3;		t = 1000;	p = 64;		break;
						case 	"book":			d = 3;		t = 1000;	p = 64;		break;
						case 	"cube":			d = 3;		t = 1000;	p = 64;		break;
						case 	"coverflow":	d = 3;		t = 1000;	p = 0;		break;
						default:	return;
					}
					transitionSpeed.setValue(t);
					slideshowDelay.setValue(d);
					fitPadding.setValue(p);
				});	
				
				add("hfill", behaviorPanel);
				add("br hfill", mainImagePanel);
				add("br hfill", navigation);
			}
		};

		// Caption / Buttons
		
		ControlPanel imageCaption = new ControlPanel(getText("ui.caption")) {
			
			JComboBox captionPlacement = new JComboBox(new Object[] {
				new Item("left top", getText("ui.left_top")), 
				new Item("center top", getText("ui.center_top")), 
				new Item("right top", getText("ui.right_top")), 
				new Item("left middle", getText("ui.left_middle")), 
				new Item("center middle", getText("ui.center_middle")), 
				new Item("right middle", getText("ui.right_middle")), 
				new Item("left bottom", getText("ui.left_bottom")), 
				new Item("center bottom", getText("ui.center_bottom")), 
				new Item("right bottom", getText("ui.right_bottom"))
			});
			JComboBox captionStyle = new JComboBox(new Object[] {
				new Item("white", getText("ui.white")), 
				new Item("light", getText("ui.light")), 
				new Item("transparent", getText("ui.transparent")), 
				new Item("dark", getText("ui.dark")),
				new Item("black", getText("ui.black"))
			});
			
			JComboBox<NameValue<String>> presets = new JComboBox<>(new NameValue[] {
				new NameValue(getText("ui.empty"), ""),
				new NameValue(getText("ui.title"), "<h6>${fileTitle}</h6>"),
				new NameValue(getText("ui.title") + " + " + getText("ui.comment"), "<h5>${fileTitle}</h5>\n<div class=\"comment\">${comment}</div>"),
				new NameValue(getText("ui.title") + " + " + getText("ui.commentShort"), "<h6>${fileTitle}</h6>\n<div class=\"comment\">${commentShort}</div>"),
				new NameValue(getText("ui.fileLabel"), "<h5>${fileLabel}</h5>"),
				new NameValue(getText("ui.fileLabel") + " + " + getText("ui.comment") + " (" + getText("ui.default") + ")", "<h5>${fileLabel}</h5>\n<div class=\"comment\">${comment}</div>"),
				new NameValue(getText("ui.fileLabel") + " + " + getText("ui.commentShort"), "<h5>${fileLabel}</h5>\n<div class=\"comment\">${commentShort}</div>"),
				new NameValue(getText("ui.title") + " + " + getText("ui.comment") + " + " + getText("author") + " + " + getText("keywords"), "<h5>${fileLabel}</h5>\n<div class=\"comment\">${comment}</div>\n<div class=\"comment\"><small class=\"icon-user\"> ${author}</small> <small class=\"icon-label\"> ${keywords}</small></div>")
			});
			JTextArea imageCaptionTemplate = new JSmartTextArea(3, 30);
			JScrollPane imageCaptionPane = new JScrollPane(imageCaptionTemplate);
			JCheckBox showImageNumbers = new JCheckBox(getText("ui.showImageNumbers"));
						
			{
				presets.addItemListener(listener -> {
					if (!isSkinReady()) {
						return;
					}
					imageCaptionTemplate.setText(((NameValue<String>)presets.getSelectedItem()).value);
				});	
				imageCaptionTemplate.setEditable(true);
				imageCaptionTemplate.setLineWrap(true);
				imageCaptionTemplate.setWrapStyleWord(true);
				imageCaptionTemplate.setFont(mono);
				imageCaptionTemplate.setTabSize(2);
				imageCaptionPane.setBorder(BorderFactory.createTitledBorder(getText("ui.captionTemplate")));					
				showImageNumbers.setToolTipText(getText("ui.showImageNumbersInfo"));
				
				add("", new JLabelFor(getText("ui.position"), captionPlacement));
				add("tab", captionPlacement);
				add("", new JLabelFor(getText("ui.style"), captionStyle));
				add("tab", captionStyle);
				add("br", new JLabelFor(getText("ui.presets"), presets));
				add("tab", presets);
				add("br hfill", imageCaptionPane);
				add("br", showImageNumbers);
			}
		};
			
		ControlPanel buttons = new ControlPanel(getText("ui.buttons")) {

			JCheckBox showMap = new JCheckBox(getText("ui.map"), true);
			JCheckBox showRegions = new JCheckBox(getText("ui.regions"), true);
			JCheckBox regionsVisible = new JCheckBox(getText("ui.visibleByDefault"));
			private JLabel buttonLabel = new JLabel(getText("ui.buttonLabel"));
			JTextField regionsBtnText = new JTextField(getText("regions"), 8);
			JCheckBox regionsSkipEmpty = new JCheckBox(getText("ui.skipEmptyTags"));
			JCheckBox printImageButton = new JCheckBox(getText("ui.print"));
			JCheckBox showDownload = new JCheckBox(getText("ui.download"));
			JCheckBox useAudioClips = new JCheckBox(getText("ui.audioClip"));
			JLabel volumeLabel = new JLabel(getText("ui.initialVolume"));
			JSpinner audioClipVolume = new JSpinner(new SpinnerNumberModel(75, 1, 100, 1));
			JCheckBox copyAudioClips = new JCheckBox(getText("ui.copyAudioClips"));
			JCheckBox downloadNonImages = new JCheckBox(getText("ui.enableDownloadNonImages"));
			JCheckBox downloadScaled = new JCheckBox(getText("ui.enableDownloadScaledImages"));
			JCheckBox useFotomoto = new JCheckBox(getText("ui.fotomoto"));
			JTextField fotomotoID = new JSmartTextField(16);
			JLabel mandatoryInfo = new JLabel(mandatory);
			JCheckBox showPhotoData = new JCheckBox(getText("ui.showPhotoData"));

			// Photodata
			
			ControlPanel photoData = new ControlPanel() {
	
				JCheckBox showPhotoDataLabel = new JCheckBox(getText("ui.showLabel"), true);
				JTextArea photoDataTemplate = new JSmartTextArea(12, 30);
				JScrollPane photoDataPane = new JScrollPane(photoDataTemplate);
				JButton reset = new JButton(getText("ui.resetToDefaults"));
				JLabel listMetadataHint = new JLabel("<html><i>" + getText("ui.listMetadataHint") + "</i></html>");
	
				{
					showPhotoDataLabel.setToolTipText(getText("ui.showLabelInfo"));
					photoDataTemplate.setEditable(true);
					photoDataTemplate.setLineWrap(true);
					photoDataTemplate.setWrapStyleWord(true);
					photoDataTemplate.setFont(mono);
					reset.addMouseListener(new MouseAdapter() {  
						@Override
						public void mouseReleased(MouseEvent e) {
							if (reset.isEnabled()) {
								String s = new SkinModel().photoDataTemplate;
								if (s != null && s.length() > 0) {
									photoDataTemplate.setText(s);
								}
							}
					}});				
	
					add("", showPhotoDataLabel);
					add(" ", reset);
					add("br hfill vfill", photoDataPane);
					add("br", listMetadataHint);
	
					//putClientProperty("helpPage", helpRoot + "ui/photo-data.html");
				}
			};
			
			{
				showMap.setToolTipText(getText("ui.mapInfo"));
				showRegions.setToolTipText(getText("ui.regionsInfo"));
				regionsVisible.setToolTipText(getText("ui.regionsVisibleInfo"));
				ComponentUtilities.whenSelectedEnable(showRegions, new JComponent[]{ regionsVisible, buttonLabel, regionsBtnText, regionsSkipEmpty });
				showShare.setToolTipText(getText("ui.shareInfo"));
				printImageButton.setToolTipText(getText("ui.printInfo"));
				showDownload.setToolTipText(getText("ui.downloadInfo"));
				ComponentUtilities.whenSelectedEnable(showDownload, new JComponent[]{downloadScaled, downloadNonImages});
				downloadNonImages.setToolTipText(getText("ui.enableDownloadNonImagesInfo"));
				useAudioClips.setToolTipText(getText("ui.audioClipInfo"));
				ComponentUtilities.whenSelectedEnable(useAudioClips, new JComponent[]{volumeLabel, audioClipVolume, copyAudioClips});
				audioClipVolume.setToolTipText(getText("ui.audioClipVolumeInfo"));
				copyAudioClips.setToolTipText(getText("ui.copyAudioClipsInfo") + " (" + getText("ui.oldMethod") + ")");
				fotomotoID.setToolTipText(getText("ui.fotomotoIDInfo"));
				mandatoryInfo.setToolTipText(getText("ui.mandatory"));
				ComponentUtilities.whenSelectedEnable(useFotomoto, new JComponent[]{fotomotoID,mandatoryInfo});
				showPhotoData.setToolTipText(getText("ui.showPhotoDataInfo"));
				ComponentUtilities.whenSelectedEnable(showPhotoData, new JComponent[]{photoData});

				add("", new JLabel(icon("location")));
				add("tab", showMap);
				add("br", new JLabel(icon("facetag")));
				add("tab", showRegions);
				add("tab", regionsVisible);
				add(" ", buttonLabel);
				add("", regionsBtnText);
				add(" ", regionsSkipEmpty);
				add("br", new JLabel(icon("connect")));
				add("tab", showShare);
				add("br", new JLabel(icon("location")));
				add("tab", printImageButton);
				add("br", new JLabel(icon("download")));
				add("tab", showDownload);
				add("tab", downloadScaled);
				add(" ", downloadNonImages);
				add("br", new JLabel(icon("volume-up")));
				add("tab", useAudioClips);
				add("tab", volumeLabel);
				add("", audioClipVolume);
				add("", copyAudioClips);
				add("br", new JLabel(icon("fotomoto")));
				add("tab", useFotomoto);
				add(" ", new JLabelFor(getText("ui.storeId"), fotomotoID));
				add(mandatoryInfo);
				add("", fotomotoID);
				add(new JLinkLabel("https://my.fotomoto.com/signup", getText("ui.signUp")));
				add("br", new JLabel(icon("camera")));
				add("tab", showPhotoData);
				add("br hfill", photoData);
			}
		};
		
		/*	---------------------------------------------------------------
									Advanced
			--------------------------------------------------------------- */

		//	Custom code
		
		ControlPanel customCodePanel = new ControlPanel() {

			ControlPanel headHookTab = new ControlPanel() {

				JTextArea headHook = new JSmartTextArea(7, 20);
				JScrollPane headHookPane = new JScrollPane(headHook);

				{
					headHook.setEditable(true);
					headHook.setLineWrap(false);
					headHook.setFont(mono);
					headHook.setTabSize(2);
					headHookPane.setBorder(BorderFactory.createTitledBorder(getText("ui.headText")));

					add("hfill vfill", headHookPane);
				}
			};

			ControlPanel bodyHookTab = new ControlPanel() {

				JTextArea bodyHook = new JSmartTextArea(7, 20);
				JScrollPane bodyHookPane = new JScrollPane(bodyHook);

				{
					bodyHook.setEditable(true);
					bodyHook.setLineWrap(false);
					bodyHook.setFont(mono);
					bodyHook.setTabSize(2);
					bodyHookPane.setBorder(BorderFactory.createTitledBorder(getText("ui.bodyText")));

					add("hfill vfill", bodyHookPane);
				}
			};

			ControlPanel cssHookTab = new ControlPanel() {

				JTextArea cssHook = new JSmartTextArea(15, 20);
				JScrollPane cssHookPane = new JScrollPane(cssHook);
				WrappableJLabel info = new WrappableJLabel("<html><i>" + getText("ui.cssText") + "</i></html>");

				{
					cssHook.setEditable(true);
					cssHook.setLineWrap(false);
					cssHook.setFont(mono);
					cssHook.setTabSize(2);
					cssHookPane.setBorder(BorderFactory.createTitledBorder(getText("ui.cssTitle")));
					info.setPreferredWidth(700);

					add("hfill vfill", cssHookPane);
					add("br", info);
				}
			};

			ControlPanel jsHookTab = new ControlPanel() {

				JTextArea jsHook = new JSmartTextArea(15, 20);
				JScrollPane jsHookPane = new JScrollPane(jsHook);
				WrappableJLabel info = new WrappableJLabel("<html><i>" + getText("ui.javascriptText") + "</i></html>");

				{
					jsHook.setEditable(true);
					jsHook.setLineWrap(false);
					jsHook.setFont(mono);
					jsHook.setTabSize(2);
					jsHookPane.setBorder(BorderFactory.createTitledBorder(getText("ui.javascriptTitle")));
					info.setPreferredWidth(700);

					add("hfill vfill", jsHookPane);
					add("br", info);
				}
			};

			JTabbedPane customCodeTabs = new JTabbedPane() {				

				{
					addTab("<HEAD>", icon("code"), headHookTab);
					addTab("<BODY>", icon("code"), bodyHookTab);
					addTab("CSS", icon("css"), cssHookTab);
					addTab("JavaScript", icon("javascript"), jsHookTab);
				}
			};

			{				
				add("hfill vfill", customCodeTabs);

				//putClientProperty("helpPage", helpRoot + "ui/site.html");
			}
		};
		
		//	Site admin
		
		ControlPanel siteAdminPanel = new ControlPanel() {

			ControlPanel serverRelatedPanel = new ControlPanel(getText("ui.serverRelated")) {
				
				JTextField uploadPath = new JSmartTextField(20);
				JCheckBox useFavicon = new JCheckBox(getText("ui.useTheSkinsFavicon"), true);				
				JCheckBox useMsServer = new JCheckBox(getText("ui.useMsServer"));
				JCheckBox useExpiry = new JCheckBox(getText("ui.useExpiry"), false);
				JCheckBox useRobotsTxt = new JCheckBox(getText("ui.useRobotsTxt"), false);
				JCheckBox avoidCDNs = new JCheckBox(getText("ui.avoidCDNs"), false);
				JCheckBox copyGoogleFonts = new JCheckBox(getText("ui.copyGoogleFonts"), false);
				JCheckBox addLatinExt = new JCheckBox(getText("ui.addLatinExtended"), true);
				WrappableJLabel uploadPathInfo = new WrappableJLabel("<html><i>" + getText("ui.copyPasteAlbumURLHere") + "</i></html>");
				
				{
					uploadPath.setToolTipText(getText("ui.uploadPathInfo"));
					uploadPathInfo.setPreferredWidth(360);
					useFavicon.setToolTipText(getText("ui.useFaviconInfo"));
					useMsServer.setToolTipText(getText("ui.useMsServerInfo"));
					useExpiry.setToolTipText(getText("ui.useExpiryInfo"));
					useRobotsTxt.setToolTipText(getText("ui.useRobotsTxtInfo"));
					avoidCDNs.setToolTipText(getText("ui.avoidCDNsInfo"));
					copyGoogleFonts.setToolTipText(getText("ui.copyGoogleFontsInfo"));
					addLatinExt.setToolTipText(getText("ui.addLatinExtendedInfo"));
					
					add(new JLabel(getText("ui.uploadPath")));
					add("", uploadPath);
					add("br", new JLabel(infoIcon));
					add("", uploadPathInfo);
					add("br", useFavicon);
					add(" ", new JLabel(svgIcon("favicon", new Dimension(24, 24))));
					add("br", useMsServer);
					add("br", useExpiry);
					add("br", useRobotsTxt);
					add("br", avoidCDNs);
					add("br", copyGoogleFonts);
					add("br", addLatinExt);
				}
			};			
			
			ControlPanel imageRelatedPanel = new ControlPanel(getText("ui.images")) {
				
				JCheckBox hiDpiThemeImage = new JCheckBox(getText("ui.hiDpiThemeImage"));
				JComboBox shareImageDims = new JComboBox(new Object[] {
					"640x480",
					"1024x768",
					"1200x900",
					"1600x1200"
				});
				
				{
					shareImageDims.setEditable(true);
					
					add(hiDpiThemeImage);
					add("br", new JLabel(getText("ui.shareImageDimensions")));
					add("", shareImageDims);
				}
			};
			
			ControlPanel googlePanel = new ControlPanel(getText("ui.googleAnalytics")) {

				JComboBox googleAnalytics = new JComboBox(new Object[] {
					new Item("none", getText("ui.none")), 
					new Item("gtag", "Global Site Tag (gtag.js)"),
					new Item("universal", "Universal (analytics.js)"),
					new Item("classic", "Classic (ga.js) [Legacy]")
				});
				JTextField googleSiteID = new JSmartTextField(20);
				JLabel mandatoryInfo = new JLabel(mandatory);
				JCheckBox supportDoubleclick = new JCheckBox(getText("ui.supportDoubleclick"), false);

				{
					googleSiteID.setToolTipText(getText("ui.googleSiteIDInfo"));
					mandatoryInfo.setToolTipText(getText("ui.mandatory"));

					add("", new JLabelFor(getText("ui.version"), googleAnalytics));
					add(" ", googleAnalytics);
					add(" ", new JLinkLabel("https://www.google.com/analytics/", getText("ui.signUp")));
					add("br", new JLabelFor(getText("ui.googleSiteID"), googleSiteID));
					add(mandatoryInfo);
					add("", googleSiteID);
					add("br", supportDoubleclick);

					//putClientProperty("helpPage", helpRoot + "ui/advanced.html#google-analytics");
				}
			};
			
			ControlPanel seoPanel = new ControlPanel(getText("ui.searchEngineOptimization")) {
				
				JTextField titleSEOText = new JSmartTextField(20);
				JTextField descriptionSEOText = new JSmartTextField(20);
				JSpinner preloadThumbs = new JSpinner(new SpinnerNumberModel(20, 0, 1000, 10));
				JCheckBox addAltTags = new JCheckBox(getText("ui.addAltTags"), true);
				JCheckBox slideRedirect = new JCheckBox(getText("ui.redirectSlidePages"), true);
			
				{
					titleSEOText.setToolTipText(getText("ui.titleSEOTextInfo"));
					descriptionSEOText.setToolTipText(getText("ui.descriptionSEOTextInfo"));
					//preloadThumbs.setToolTipText(getText("ui.preloadThumbsInfo"));
					addAltTags.setToolTipText(getText("ui.addAltTagsInfo"));
					writeSitemapXml.setToolTipText(getText("ui.createSitemapXmlInfo"));
					ComponentUtilities.whenSelectedEnable(writeSitemapXml, new JComponent[]{sitemapIncludeSlides});
					sitemapIncludeSlides.setToolTipText(getText("ui.sitemapIncludeSlidesInfo"));
					slideRedirect.setToolTipText(getText("ui.redirectSlidePagesInfo"));
					
					add("", new JLabelFor(getText("ui.titleSEOText"), titleSEOText));
					add("tab", titleSEOText);
					add("br", new JLabelFor(getText("ui.descriptionSEOText"), descriptionSEOText));
					add("tab", descriptionSEOText);
					add("br", new JLabel(getText("ui.preloadThumbs")));
					add("", preloadThumbs);
					add("br", addAltTags);
					add("br", writeSitemapXml);
					add("", sitemapIncludeSlides);
					add("br", slideRedirect);
				}
			};

			ControlPanel cookiePolicyPanel = new ControlPanel(getText("ui.trackingConsentAndCookiePolicy")) {

				JCheckBox askTrackingConsent = new JCheckBox(getText("ui.askTrackingConsent"), false);
				JCheckBox showCookiePolicy = new JCheckBox(getText("ui.showCookiePolicy"), false);				
				JTextField cookiePolicyStay = new JSmartTextField("8", 3);
				JTextField cookiePolicyUrl = new JSmartTextField(20);
				WrappableJLabel cookiePolicyUrlInfo = new WrappableJLabel("<html><i>" + getText("ui.cookiePolicyUrlInfo") + " </i></html>");

				{
					askTrackingConsent.setToolTipText(getText("ui.askTrackingConsentInfo"));
					showCookiePolicy.setToolTipText(getText("ui.showCookiePolicyInfo"));
					cookiePolicyUrlInfo.setPreferredWidth(320);
					
					add(askTrackingConsent);
					add("br", showCookiePolicy);
					add(" ", new JLabel(getText("ui.stay")));
					add("", cookiePolicyStay);
					add("", new JLabel("s"));
					add("br", new JLabel("URL"));
					add("tab", cookiePolicyUrl);
					add("br", new JLabel(infoIcon));
					add("tab", cookiePolicyUrlInfo);
				}
			};
			
			JCheckBox debugMode = new JCheckBox(getText("ui.debugMode"));

			ControlPanel leftPanel = new ControlPanel() {
				
				{
					add("hfill", serverRelatedPanel);
					add("br hfill", googlePanel);					
					add("br", debugMode);					
				}
			};
			
			ControlPanel rightPanel = new ControlPanel() {
				
				{
					add("hfill", imageRelatedPanel);
					add("br hfill", seoPanel);
					add("br hfill", cookiePolicyPanel);
				}
			};
			
			{
				add(leftPanel);
				add(rightPanel);
				
				//putClientProperty("helpPage", helpRoot + "Site/Site_admin");
			}
		};
		
		//	Custom Keys
		
		ControlPanel customKeysPanel = new ControlPanel() {
		
			JTextArea customKeys = new JSmartTextArea(7, 20);
			JScrollPane customKeysPane = new JScrollPane(customKeys);

			{
				customKeys.setEditable(true);
				customKeys.setLineWrap(false);
				customKeys.setFont(mono);
				customKeys.setTabSize(2);
				customKeysPane.setBorder(BorderFactory.createTitledBorder(getText("ui.customKeys")));

				add("hfill vfill", customKeysPane);
				add("br", new JLabel(getText("ui.customKeysInfo")));
			}
		};
		
		
		/******************************************************************************
		 * 
		 *								GUI main tabs
		 * 
		 ******************************************************************************/				
		
		/*	---------------------------------------------------------------
									Site
			--------------------------------------------------------------- */
		
		ControlPanel site = new ControlPanel(new BorderLayout(20, 0)) {
			
			JTabbedPane siteTabs = new JTabbedPane() {

				JScrollPane pageDesignPane = new JScrollPane(pageDesign, JScrollPane.VERTICAL_SCROLLBAR_AS_NEEDED, JScrollPane.HORIZONTAL_SCROLLBAR_NEVER);
				
				{
					pageDesignPane.setBorder(emptyBorder);
					this.setTabPlacement(SwingConstants.LEFT);
					
					addTab("<html><h4 " + tabStyle + ">" + getText("ui.design") + "</h4></html>", icon("design"), pageDesignPane);
					addTab("<html><h4 " + tabStyle + ">" + getText("ui.typography") + "</h4></html>", icon("font"), typography);
					addTab("<html><h4 " + tabStyle + ">" + getText("ui.header") + "</h4></html>", icon("header"), coverHeader);
					addTab("<html><h4 " + tabStyle + ">" + getText("ui.title") + "</h4></html>", icon("hero"), coverTitle);
					addTab("<html><h4 " + tabStyle + ">" + getText("ui.folders") + "</h4></html>", icon("folder"), folders);
					addTab("<html><h4 " + tabStyle + ">" + getText("ui.footer") + "</h4></html>", icon("footer"), footerTab);
				}
			};
			
			//JScrollPane siteTabsPane = new JScrollPane(siteTabs, JScrollPane.VERTICAL_SCROLLBAR_AS_NEEDED, JScrollPane.HORIZONTAL_SCROLLBAR_NEVER);

			JPanel designPreviewPanel = new JPanel(new BorderLayout(0, 0)) {

				{
					designPreview.getView().setPreferredSize(new Dimension(previewWidth, uiHeight));
					
					add(designPreview.getView(), BorderLayout.CENTER);
				}
			};
			
			{	
				((BorderLayout)(getLayout())).setVgap(0);
				((BorderLayout)(getLayout())).setHgap(0);
				siteTabs.setBorder(emptyBorder);
				
				siteTabs.setPreferredSize(new Dimension(uiWidth, uiHeight));
				designPreviewPanel.setBorder(BorderFactory.createLineBorder(JAlbumColor.imageBorder, 2));
				
				add(siteTabs);
				add(designPreviewPanel, BorderLayout.EAST);
				
				//putClientProperty("helpPage", helpRoot + "ui/site.html");
			}
		};
					
		/*	---------------------------------------------------------------
									Sidebar
			--------------------------------------------------------------- */

		ControlPanel sidebar = new ControlPanel(new BorderLayout(0, 0)) {
			
			ControlPanel ordering = new ControlPanel() {

				JList sidebarOrder = new JDraggableList(new Object[] {
					new JLabelFor("<html><p style=\"padding:3px 0;\">" + getText("ui.navigation") + "</p></html>", icon("folders")).name("navigation"),
					new JLabelFor("<html><p style=\"padding:3px 0;\">" + getText("ui.search") + "</p></html>", icon("search")).name("search"),
					new JLabelFor("<html><p style=\"padding:3px 0;\">" + getText("ui.tags") + "</p></html>", icon("tag")).name("tags"),
					new JLabelFor("<html><p style=\"padding:3px 0;\">" + getText("ui.searchNew") + "</p></html>", icon("new")).name("newImages"),
					new JLabelFor("<html><p style=\"padding:3px 0;\">" + getText("ui.map") + "</p></html>", icon("location")).name("map"),
					new JLabelFor("<html><p style=\"padding:3px 0;\">" + getText("ui.social") + "</p></html>", icon("connect")).name("social"),
					new JLabelFor("<html><p style=\"padding:3px 0;\">" + getText("ui.fullScreen") + "</p></html>", icon("fullscreen")).name("fullscreen"),
					new JLabelFor("<html><p style=\"padding:3px 0;\">" + getText("ui.backgroundMusic") + "</p></html>", icon("audio")).name("music")
				});

				{						
					add("center", new JLabel("<html><h4>" + getText("ui.ordering")+ "</h4></html>"));
					add("br", sidebarOrder);
					add("br center", new JLabel("<html><i>" + getText("ui.dragToReorder") + "</i></html>"));
				}
			};
				
			JTabbedPane sidebarTabs = new JTabbedPane() {

				JScrollPane socialPanel = new JScrollPane(social, JScrollPane.VERTICAL_SCROLLBAR_AS_NEEDED, JScrollPane.HORIZONTAL_SCROLLBAR_NEVER);

				{
					this.setTabPlacement(SwingConstants.LEFT);
					
					socialPanel.setBorder(emptyBorder);
					search.setBorder(emptyBorder);

					addTab("<html><h4 " + tabStyle + ">" + getText("ui.navigation") + "</h4></html>", icon("folders"), topNavigation);
					addTab("<html><h4 " + tabStyle + ">" + getText("ui.search") + "</h4></html>", icon("search"), search);
					addTab("<html><h4 " + tabStyle + ">" + getText("ui.tagCloud") + "</h4></html>", icon("tag"), tagCloud);
					addTab("<html><h4 " + tabStyle + ">" + getText("ui.searchNew") + "</h4></html>", icon("new"), searchNew);
					addTab("<html><h4 " + tabStyle + ">" + getText("ui.map") + "</h4></html>", icon("location"), map);
					addTab("<html><h4 " + tabStyle + ">" + getText("ui.social") + "</h4></html>", icon("connect"), socialPanel);
					addTab("<html><h4 " + tabStyle + ">" + getText("ui.fullScreen") + "</h4></html>", icon("fullscreen"), fullScreen);
					addTab("<html><h4 " + tabStyle + ">" + getText("ui.backgroundMusic") + "</h4></html>", icon("audio"), backgroundMusic);

				}
			};
							
			{
				((BorderLayout)(getLayout())).setVgap(0);
				((BorderLayout)(getLayout())).setHgap(0);
				
				sidebarTabs.setBorder(emptyBorder);
				sidebarTabs.setPreferredSize(new Dimension(uiWidth, uiHeight));
				
				add(sidebarTabs);
				add(ordering, BorderLayout.EAST);
				
				//putClientProperty("helpPage", helpRoot + "ui/sections.html");
			}
		};
		
		/*	---------------------------------------------------------------
									Lightbox
			--------------------------------------------------------------- */
		
		ControlPanel lightbox = new ControlPanel(new BorderLayout(0, 0)) {
			
			JTabbedPane lightboxTabs = new JTabbedPane() {
				
				JScrollPane markNewPane = new JScrollPane(markNew, JScrollPane.VERTICAL_SCROLLBAR_AS_NEEDED, JScrollPane.HORIZONTAL_SCROLLBAR_NEVER);
				JScrollPane mainImagePanel = new JScrollPane(mainImage, JScrollPane.VERTICAL_SCROLLBAR_AS_NEEDED, JScrollPane.HORIZONTAL_SCROLLBAR_NEVER);
				
				{
					this.setTabPlacement(SwingConstants.LEFT);
					
					thumbnailStrip.setBorder(emptyBorder);
					markNewPane.setBorder(emptyBorder);
					mainImagePanel.setBorder(emptyBorder);
					imageCaption.setBorder(emptyBorder);
					buttons.setBorder(emptyBorder);
					
					addTab("<html><h4 " + tabStyle + ">" + getText("ui.thumbnailStrip") + "</h4></html>", icon("thumbnails"), thumbnailStrip);
					addTab("<html><h4 " + tabStyle + ">" + getText("ui.markNew") + "</h4></html>", icon("new"), markNewPane);
					addTab("<html><h4 " + tabStyle + ">" + getText("ui.mainImage") + "</h4></html>", icon("image"), mainImagePanel);
					addTab("<html><h4 " + tabStyle + ">" + getText("ui.caption") + "</h4></html>", icon("caption"), imageCaption);					
					addTab("<html><h4 " + tabStyle + ">" + getText("ui.buttons") + "</h4></html>", icon("button"), buttons);					
					//addTab("<html><h4>" + getText("ui.customFunction") + "</h4></html>", icon("wrench"), imgHookTab);					
				}
			};
			
			{
				((BorderLayout)(getLayout())).setVgap(0);
				((BorderLayout)(getLayout())).setHgap(0);

				lightboxTabs.setBorder(emptyBorder);
				lightboxTabs.setPreferredSize(new Dimension(uiWidth, uiHeight));
				
				add(lightboxTabs);
				
				//putClientProperty("helpPage", helpRoot + "ui/lightbox.html");
			}
		};
		
		/*	---------------------------------------------------------------
									Advanced
			--------------------------------------------------------------- */
		
		ControlPanel advanced = new ControlPanel(new BorderLayout(0, 0)) {
			
			JTabbedPane advancedTabs = new JTabbedPane() {
								
				JScrollPane siteAdminPane = new JScrollPane(siteAdminPanel, JScrollPane.VERTICAL_SCROLLBAR_AS_NEEDED, JScrollPane.HORIZONTAL_SCROLLBAR_NEVER);
				
				{
					siteAdminPane.setBorder(emptyBorder);
					customCodePanel.setBorder(emptyBorder);
					customKeysPanel.setBorder(emptyBorder);
					this.setTabPlacement(SwingConstants.LEFT);
					
					addTab("<html><h4 " + tabStyle + ">" + getText("ui.siteAdmin") + "</h4></html>", icon("wrench"), siteAdminPane);
					addTab("<html><h4 " + tabStyle + ">" + getText("ui.customCode") + "</h4></html>", icon("code"), customCodePanel);
					addTab("<html><h4 " + tabStyle + ">" + getText("ui.customKeys") + "</h4></html>", icon("rename"), customKeysPanel);
				}
			};
			
			{
				((BorderLayout)(getLayout())).setVgap(0);
				((BorderLayout)(getLayout())).setHgap(0);
				
				advancedTabs.setBorder(emptyBorder);
				
				add(advancedTabs);
				
				//putClientProperty("helpPage", helpRoot + "ui/lightbox.html");
			}
		};
				
		/*	---------------------------------------------------------------
									About
			--------------------------------------------------------------- */

		ControlPanel about = new ControlPanel(new BorderLayout(20, 0)) {
			
			private JTextArea readme = new JSmartTextArea( getFileContents("readme.txt"), 20, 30 );
			private JScrollPane aboutPane = new JScrollPane(readme, JScrollPane.VERTICAL_SCROLLBAR_ALWAYS, JScrollPane.HORIZONTAL_SCROLLBAR_NEVER);
			
			ControlPanel info = new ControlPanel() {
				
				{
					add("center", new JLabel(svgIcon("logo-projector", new Dimension(120, 120))));
					add("br center", new JLabel("<html><h2>" + skin + "</h2></html>"));
					add("br center", new JLabel("Jalbum " + internalVersion));
					add(new JLinkLabel("https://jalbum.net/software/download", getText("ui.upgrade"), getText("ui.downloadJalbum")));
					add("br center", new JLabel(skin + " skin " + skinVer));
					add(new JLinkLabel("https://jalbum.net/skins/skin/" + skin, getText("ui.upgrade"), getText("ui.downloadSkin")));
					//add("br center", new JLinkLabel(helpRoot + "index.html", getText("help")));
					add("br center", new JLinkLabel(supportForum, getText("ui.support")));
				}
			};
			
			{								
				((BorderLayout)(getLayout())).setVgap(0);
				((BorderLayout)(getLayout())).setHgap(0);
				
				readme.setLineWrap(true);
				readme.setWrapStyleWord(true);
				readme.setEditable(false);
				readme.setFont(mono);
				
				info.setPreferredSize(new Dimension(240, uiHeight));
				aboutPane.setPreferredSize(new Dimension(uiWidth, uiHeight));
				
				add(info, BorderLayout.WEST);
				add(aboutPane);
				
				//putClientProperty("helpPage", helpRoot + "index.html");
			}
		};
		
		/*	---------------------------------------------------------------
									Main tabs
			--------------------------------------------------------------- */
		
		JTabbedPane tabs = new JTabbedPane() {
			
			{
				site.setBorder(emptyBorder);
				sidebar.setBorder(emptyBorder);
				lightbox.setBorder(emptyBorder);
				advanced.setBorder(emptyBorder);
				about.setBorder(emptyBorder);
				
				addTab("<html><h4 style='padding:4px 6px;margin:4px;'>" + getText("ui.site") + "</h4></html>", icon("site", 27), site);
				addTab("<html><h4 style='padding:4px 6px;margin:4px;'>" + getText("ui.sidebar") + "</h4></html>", icon("sidebar-left", 27), sidebar);
				addTab("<html><h4 style='padding:4px 6px;margin:4px;'>" + getText("ui.lightbox") + "</h4></html>", icon("lightbox", 27), lightbox);
				addTab("<html><h4 style='padding:4px 6px;margin:4px;'>" + getText("ui.advanced") + "</h4></html>", icon("wrench", 27), advanced);
				addTab("<html><h4 style='padding:4px 6px;margin:4px;'>" + getText("ui.about") + "</h4></html>", icon("info", 27), about);
			} 
		};
			
		private final String designTemplate = getFileContents("lib/design-template.html");
	
		{	
			// Adding UI tabs
			
			((RiverLayout)(getLayout())).setVgap(0);
			((RiverLayout)(getLayout())).setHgap(0);
			tabs.setBorder(emptyBorder);
			
			add(tabs);
									
			/*	-----------------------------------------------------------
										Listeners
				----------------------------------------------------------- */
			
			window.addJAlbumListener(new JAlbumAdapter() {
				@Override
				public void styleChanged(JAlbumEvent e) {
					styleName.setText(getStyleName());
				}
			});

			window.addPropertyChangeListener(JAlbumFrame.SKIN_LOADED_PROPERTY, setupMonitors);

			window.addJAlbumListener(new JAlbumAdapter() {
				@Override
				public void skinChanged(JAlbumEvent e) {
					//JAlbum.logger.log(Level.FINE, "Skin changed, removing listeners.");
					window.removePropertyChangeListener(JAlbumFrame.SKIN_LOADED_PROPERTY, setupMonitors);
				}
			});
		
			try {
				window.albumExplorer.onContextMenu(menu -> {
					AlbumObject[] sel = window.albumExplorer.explorer.getSelectedAlbumObjects();

					if (sel.length > 0) {
						
						//  Panorama selector
						createMenuItemForBooleanProperty(menu, sel, "panorama", getText("ui.panorama"));
						
						//  Hide location selector
						createMenuItemForBooleanProperty(menu, sel, "hideLocation", getText("ui.hideLocation"));
					}
				});
			} catch (Throwable ex) {
				JAlbum.logger.log(Level.INFO, "Right-click extension menu is available only in jAlbum v32 and newer!");
			};
		}
		
	};

	public Gui() {
		this(JAlbumContext.getInstance());
	}
   
	public Gui(JAlbumContext context) {
		
		super(context);
		PluginContext pc = context.getPluginContext();
		EditPanel editPanel = pc.getEditPanel();
		
		editPanel.addCustomTab(getText("ui.imageData"), new ImageDataUI(context, texts));
		
		skinUi.setBorder(emptyBorder);
		((RiverLayout)(getLayout())).setVgap(0);
		((RiverLayout)(getLayout())).setHgap(0);
				
		window.setSkinUI(skinUi);
		skinReadyAt = System.currentTimeMillis();
		
	}
	
}
