/*	
 *	addTooltip() :: little Popup displaying 'title' text, or passed text (can be HTML)
 *
 *	Copyright by Lazaworx
 *	http://www.lazaworx.com
 *	Author: Laszlo Molnar
 *
 *	Dual licensed under the MIT and GPL licenses.
 *	- http://www.opensource.org/licenses/mit-license.php
 *	- http://www.gnu.org/copyleft/gpl.html
 *
 *	Usage: $(element).addTooltip( [txt,] options );
 *	options:
		id: 	'tooltip',
		stay: 	3000,
		posX: 	ALIGN_CENTER,
		posY: 	ALIGN_BOTTOM,
		toX: 	ALIGN_CENTER,
		toY: 	ALIGN_TOP
 */

;(function($, $body) {
	'use strict';
			
	$.fn.hideAllTooltips = function(except) {
		
			return this.each(function() {
					
					//log('Hiding all tooltips');
					$(this).data('suppressTooltips', true).find('[data-tooltip-id]').each(function() {
							if (!$(this).data('tooltip-keep')) { 
								var id = $(this).attr('data-tooltip-id');
								
								if (id && id.length && id !== except) {
									$('#' + id).hide();
								}
							}
						});
				});
		};
	
	$.fn.destroyAllTooltips = function(except) {
		
			return this.each(function() {
					
					//log('Removing ' + $(this).find('[data-tooltip-id]').length + ' tooltip(s).');
					$(this).data('suppressTooltips', true).find('[data-tooltip-id]').each(function() {
							var id = $(this).attr('data-tooltip-id');
							//log(id);
							if (id && id.length && id !== except) {
								$('#' + id).remove();
							}
						});
				});
		};
	
	// Hiding all tooltips on mouse leaving the document
	$(document).on('mouseleave._ltt_', function() {
			$body.hideAllTooltips();
		});
	
	$.fn.addTooltip = function(content, settings) {
		
			if (typeof content !== UNDEF && typeof content !== 'string' && !content.jquery) {
				settings = content;
				content = null;
			}
			
			settings = $.extend({}, $.fn.addTooltip.defaults, settings);
			
			//console.log(content || $(this).data('tooltip') || $(this).attr('title'));
			
			var isVisible = function(el) {
						
						if (typeof el === UNDEF || !el || !el.length) {
							return true;
						} else if (el.is(':hidden') || el.css('opacity') < .25) {
							//console.log(el[0].id + ' is hidden or transparent!');
							return false;
						} else {
							var visible = true;
							el.parents().each(function() {
									if ($(this).is(':hidden') || $(this).css('opacity') < 0.25) {
										visible = false;
										return false;
									}
								});
							
							return visible;
						}
					},
					
				// Creating new tooltip
				createNew = function(el, cont) {
						var tt;
										
						if (!cont) {
							
							if (cont = el.data('tooltip')) {
								if (cont.jquery) {
									// jQuery element
									cont.removeClass('hidden');
								} else if (cont.charAt(0) === '.') {
									// read from related layer
									cont = el.find(cont).eq(0);
								} else if (cont.charAt(0) === '#') {
									cont = $(cont);
								}
							} else {
								// read from data or title attr
								cont = el.attr('title');
								el.removeAttr('title');
							}
							
							if (!cont || !cont.length) {
								return $();
							}
							
							tt = $('<div>', {
									html: 	cont
								}).appendTo('body');
							
						} else if (typeof cont === STRING) {
							
							// passed directly :: html structure as string
							tt = $('<div>', {
									html: 	cont
								}).appendTo('body');
							
						} else if (cont.jquery) {
							
							// jQuery element
							if (!$.contains(document.body, cont[0])) {
								tt = cont.appendTo('body');
							} else {
								tt = cont;
							}
							
						} else {
							return $();
						}
						
						if (tt.is(':empty')) {
							return null;
						}
						
						tt	.attr('id', el.attr('data-tooltip-id'))
							.addClass(settings.className)
							.attr('role', 'tooltip')
							.attr('aria-hidden', true)
							.hide()
							.append($('<span>', {
								'class': 	settings.nub
							}));
						
						return tt;
						
					};
			
			return this.each(function() {
				
					if (this['data-tooltip-id']) {
						// already exists
						return true;
					}
					
					var self = $(this), 			// trigger element
						tooltip = $(),				// tooltip layer
						to, 						// show timeout
						hto,						// hide timeout
						//over = false,				// over the tooltip 
						focus = false,				// tooltip <input> got focus
						offs,						// last offset to detect if moving
						start,						// event start
						//events = '',				// cumulating events
						wasOn = false,				// was on at the beginning of the event?
						ns = '_ltt_' + Math.floor(Math.random() * 10000),
						//isFireFox = /(Firefox\/\d+)/.test(navigator.userAgent),
							
						// Create
						createTooltip = function() {
							
								//log('Creating tooltip...');
								
								if (self.data('suppressTooltips')) {
									return false;
								}
								
								var tt = createNew(self, content);
								
								if (tt.length) {
									// Keep the popup live while the mouse is over
									tt	
										.on('mouseover.' + ns, overTooltip)
										.on('mouseleave.' + ns, leaveTooltip);
											
									// ... or an input box has focus
									tt	
										.find('input, textarea')
										.on('focus.' + ns, function(e) {
												focus = true;
												clearTimeout(hto);
												hto = null;
											})
										.on('blur.' + ns, function() {
												focus = false;
											});
								}
								
								return tt;
							},
						
						// Tooltip is visible?
						tooltipVisible = function() {
							
								return !!tooltip && tooltip.is(':visible') && (tooltip.css('opacity') > 0.2);
							},
						
						// Entering tooltip
						overTooltip =  function(e) {
								
								//log(e.type + ' :: Over tooltip');
								clearTimeout(hto);
								//hto = null;
								//over = true;
							},
						
						// Leaving tooltip
						leaveTooltip = function(e) {
		
								//log(e.type + ' :: Leaving tooltip');
								if (!focus) {
									// No action when in an input box
									//over = false;
									hideTooltipLater();
									//clearTimeout(hto);
									//hto = setTimeout(hideTooltip, Math.max(settings.stay, 500));
									return;
								}
							},
							
						// Hiding the popup
						hideTooltip = function() {
							
								//if (!over) {
									//wasOn = false; 
									clearTimeout(to);
									clearTimeout(hto);
									//to = hto = null; 
									//over = false;
									//events = '';
									//log('Hiding tooltip');
									
									tooltip	
										.css({
												opacity:	0
											})
										.one('transitionend', function() {
												$(this).css({
														display:	'none'
													});
											});
								//}
							},
						
						// Hide later :: automatically on touch devices
						hideTooltipLater = function() {
							
								//log('Hiding tooltip later (' + Math.max(settings.stay, 500) + 'ms)');
								clearTimeout(hto);
								hto = setTimeout(hideTooltip, Math.max(settings.stay, 500));
							},
						
						
						// Showing, aligning and fading in
						showTooltip = function() {
								var o = self.offset();
								
								if (self.data('suppressTooltips')) {
									return;
								}
								
								clearTimeout(hto);
								//hto = null;
								//log('Showing tooltip'); //, pos: ' + o.left.toFixed(3) + ',' + o.top.toFixed(3));
								
								if (!offs || (Math.abs(o.top - offs.top) < 1 && Math.abs(o.left - offs.left) < 1)) {
									// only if target layer not in move
									if (settings.exclusive) {
										self.data('tooltip-keep', true);
										$body.hideAllTooltips(self.data('tooltip-id'));
										self.data('tooltip-keep', null);
									}
									
									tooltip	
										.css({
												opacity: 	0,
												display:	'block'
											})
										.alignTo(self, {
												gap: 		settings.gap,
												pos: 		settings.pos
											})
										.css({
												opacity: 	1
											})
										.one('transitionend', function() {
												$(this).css({
														display:	'block'
													});
												if (whatInput.ask('intent') !== 'mouse') {
													hideTooltipLater();
												}
											});
										
									//over = true;
									//wasOn = true;
										
								}
								
								offs = o;
							},
						
						// Avoid Click
						avoidClick = function(e) {
							
								e.preventDefault();
								clearTimeout(to);
								clearTimeout(hto);
								//to = hto = null;
							},
						
						// Test for link
						hotspotHasLink = function(el) {
								var	a = el.closest('a');
								
								return a.length && a.attr('href') && !a.attr('href').startsWith('javascript');
							},
						
						// The hotspot clicked
						hotspotClicked = function(e) {
								//log(e.type + ' :: Clicked spot');
								//log(whatInput.ask());
								//log(events);
								
								if (isVisible($(e.target).closest('[data-tooltip-id]'))) {
									
									clearTimeout(to);
									//to = null;
									
									if (settings.touchToggle || whatInput.ask('intent') !== 'mouse') {
									//if (events.indexOf(TOUCH.START) !== -1 || isFireFox) {
										// touched
										// Firefox by default emulates touch events with mouse events, 
										// no way you can tell the difference, so it's safer to treat like touch
										
										var now = new Date();
										
										if (settings.touchToggle || (now - start) > 1000) {
											// touch toggle or long touch
											//log('Touched for ' + (now-start) + 'ms');
											//log('wasOn='+wasOn+' hotspotHasLink()='+hotspotHasLink($(e.target))+' visible='+tooltipVisible());
											if (hotspotHasLink($(e.target)) && tooltipVisible()) {
												// Link and tt is visible :: let click it
												return true;
											} else {
												// No link or need to toggle on first
												avoidClick(e);
												
												if (tooltipVisible()) {
													hideTooltip();
												} else {
													showTooltip();
												}
												
												return false;
											}
										}		
									}
									
									if (wasOn) {
										clearTimeout(hto);
										hto = null;
										hideTooltip();
									}
									
									//events = '';
									//over = false;
								}
								
								return true;
							},
						
						// Leaving the trigger element
						leaveHotspot = function(e) {
								
								clearTimeout(hto);
								clearTimeout(to);
								
								if (isVisible($(e.target).closest('[data-tooltip-id]'))) {
									
									//hto = null;
									//log(e.type + ' :: Leaving spot');
									
									hto = setTimeout(hideTooltip, (whatInput.ask('intent') === 'mouse')? 500 : 3000);
									//over = false;
								}
							},
						
						// Entering the hotspot :: hover or focus
						enterHotspot = function(e) {
								//log(e.type + ' :: Entering spot');
								
								clearTimeout(hto);
								
								if (!self.data('suppressTooltips') && isVisible($(e.target).closest('[data-tooltip-id]'))) {
									
									//wasOn = tooltipVisible();
									start = new Date();
									
									offs = self.offset();
									tooltip = $('#' + self.data('tooltip-id'));
									//log('pos: ' + offs.left.toFixed(3) + ',' + offs.top.toFixed(3));
									
									if (!tooltip.length) {
										
										tooltip = createTooltip();
										
										if (!tooltip.length) {
											destroyTooltip();
											return true;
										}
										
									} else {
										
										
										//hto = null;
									}
									
									clearTimeout(to);
									//to = null;
									
									if (whatInput.ask('intent') === 'mouse') {
										// Show on hover after a delay
										to = setTimeout(showTooltip, settings.delay);
									}
								}
								
								return true;
							},
						
						// Force removing the tooltip
						destroyTooltip = function(e) {
								self.off('.' + ns);
								to = clearTimeout(to);
								hto = clearTimeout(hto);
								self.data('suppressTooltips', true);
								$('#' + self.attr('data-tooltip-id')).remove();
								self.attr('data-tooltip-id', null);
							};
							
					
					self.attr('data-tooltip-id', ns)
						.data('suppressTooltips', false)
						.on('destroyTooltip', destroyTooltip)
						.on('removeTooltip', hideTooltip)
						.on('focus.' + ns + ' mouseover.' + ns /*+ ' ' + TOUCH.START + '.' + ns*/, enterHotspot)
						.on('blur.' + ns + ' mouseleave.' + ns, leaveHotspot)
						.on('click.' + ns, hotspotClicked);
					
				});
		};
	
	/*	pos:
		ALIGN_LEFT = ALIGN_TOP = 0
		ALIGN_CENTER = ALIGN_MIDDLE = 1
		ALIGN_RIGHT = ALIGN_BOTTOM = 2
	*/
	$.fn.addTooltip.defaults = {
			delay: 				50,
			className: 			'tooltip',
			nub: 				'nub',
			stay: 				2000,
			exclusive:			true,
			touchToggle:		false,
			pos: 				[1,2,1,0],
			gap:				6
		};
	
})(jQuery, $('body'));
