/*	
 *	swipe() :: Swipe action using CSS transforms with jQuery position fallback
 *
 *	Copyright by Lazaworx
 *	http://www.lazaworx.com
 *	Author: Laszlo Molnar
 *
 *	Dual licensed under the MIT and GPL licenses.
 *	- http://www.opensource.org/licenses/mit-license.php
 *	- http://www.gnu.org/copyleft/gpl.html
 *
 *	Usage: $(element).swipe( options );
 */
 
;(function($, $window, $document, undefined) {
	'use strict';
	
	$.fn.swipe = function(options) {
		
		options = $.extend({
						onSwipedLeft:			false,			// Swiped left event
						onSwipedRight:			false,			// Swiped right event
						onSwipedUp:				false,			// Swiped up event
						onSwipedDown:			false,			// Swiped down event
						onSmallSwipe:			false,			// Small swipe (< threshold px)
						onSwipeStart:			false,			// Start swiping event
						onClick:				false,			// Clicked
						onDoubleClick:			false,			// Doble-clicked
						onFinished:				false,			// Swiped out - can remove
						threshold:				20,				// Small swipe threshold
						clickThreshold:			150,			// Timespan to tell apart single and double click
						doubleClickThreshold:	200,
						noSwipeInside:			'.caption',		// Avoid swipe inside this container
						swipeoutSpeed:			300,			// Swipe out speed in ms
						overThreshold:			0.25			// Oversized pictures must moved beyond (this * screen width) to trigger swipe 
					}, options);
		
		// Getting coordinates of a touch or mouse event
		var	getDim = function(el) {
					return [ el.outerWidth(), el.outerHeight() ];
				},
				
			getDistance = function(diff) {
					return Array.isArray(diff)?
						Math.sqrt(Math.pow(diff[0], 2) + Math.pow(diff[1], 2))
						:
						Math.abs(diff);
				},
				
			// Getting coordinates of a touch or mouse event	
			getXY = function(e) {
					if (e.touches && e.touches.length === 1) {
						return [ 
							Math.round(e.touches[0].pageX),
							Math.round(e.touches[0].pageY)
						];
					} else if (e.clientX !== null) {
						return [
							Math.round(e.pageX),
							Math.round(e.pageY)
						];
					}
					return null;
				},
			
			getX = function(e) {
					if (e.touches && e.touches.length === 1) {
						return Math.round(e.touches[0].pageX);
					} else if (e.pageX !== null) {
						return Math.round(e.pageX);
					}
					return null;
				};
				
		return this.each(function() {
				
				var self = $(this),							// target element
					clip = self.parent(),					// clipping window
					ns = self.data('lsw_ns') ||				// namespace 
						('lsw_' + Math.floor(Math.random() * 10000)),
					selfDim,								// item dimensions
					clipDim,								// clip dimensions
					rto,									// window resize update timeout
					startTime,								// start time
					endTime,								// end time
					startPos,								// start position
					diff,									// move difference
					pos,									// current position
					mtx,									// matrix(3d)
					started = false,						// ongoing swipe
					ended = true,							// ended swiping
					swipeHandled = true,					// swipe event already fired
					dontSwipeUp = true,						// limit vertical swipe
					dontSwipeDown = true,
					nonSwipeable = true,					// target element swipeable?
					clickedOnce = false,					// Check for double click
					chkDoubleClick = null,					// Checking double-click (timeout)
					timer = null,
					media = self.hasClass('audio') || self.hasClass('video'), 
					
					// Handling swipe start event
					handleSwipeStart = function(e) {
							if (typeof options.onSwipeStart === FUNCTION) {
								options.onSwipeStart.call(self[0], e);
							}
						},
						
					// Handling swipeout movement finished event
					handleFinished = function(e) {
							//log('handleFinished');
							clearTimeout(timer);
							if (typeof options.onFinished === FUNCTION) {
								options.onFinished.call(self[0], e);
							}
						},
						
					// Handling small swipe
					handleSmallSwipe = function(e) {
							//log('handleSmallSwipe');
							unregisterEvents(e);
							if (typeof options.onSmallSwipe === FUNCTION) {
								options.onSmallSwipe.call(self[0], e);
							}
						},
						
					// Handling click event
					handleClick = function(e) {
							//log('handleClick');
							unregisterEvents(e);
							if (typeof options.onClick === FUNCTION) {
								options.onClick.call(self[0], e);
							}
						},
						
					// Handling doubleclick event
					handleDoubleClick = function(e) {
							//log('handleDoubleClick');
							unregisterEvents(e);
							if (typeof options.onDoubleClick === FUNCTION) {
								options.onDoubleClick.call(self[0], e);
							}
						},
						
					// Handling swiped down	
					handleSwipedDown = function(e) {
							//log('handleSwipe:'+(right?'right':'left'));
							if (!swipeHandled) {
								unregisterEvents(e);
								swipeHandled = true;
								// Swiped down
								options.onSwipedDown.call(self[0], e);
							}
						},
							
					// Handling swiped up		
					handleSwipedUp = function(e) {
							//log('handleSwipe:'+(right?'right':'left'));
							if (!swipeHandled) {
								unregisterEvents(e);
								swipeHandled = true;
								// Swiped up
								options.onSwipedUp.call(self[0], e);
							}
						},
							
					// Handling swiped right	
					handleSwipedRight = function(e) {
							//log('handleSwipe:'+(right?'right':'left'));
							if (!swipeHandled) {
								unregisterEvents(e);
								swipeHandled = true;
								// Swiped right
								options.onSwipedRight.call(self[0], e);
							}
						},
							
					// Handling swiped left
					handleSwipedLeft = function(e) {
							//log('handleSwipe:'+(right?'right':'left'));
							if (!swipeHandled) {
								unregisterEvents(e);
								swipeHandled = true;
								// Swiped left
								options.onSwipedLeft.call(self[0], e);
							}
						},
							
					// Handling right click event
					handleRightClick = function(e) {
							//log('handleRightClick');
							unregisterEvents(e);
							if (typeof options.onRightClick === FUNCTION) {
								options.onRightClick.call(self[0], e);
							}
						},
					
					// Registering events
					registerEvents = function(e) { 
							//log('registerEvents');
							if (e.type === 'mousedown') {
								self.on('mouseup.' + ns + ' mouseout.' + ns, swipeEnd);
								$document.on('mousemove.' + ns, swipeMove);
							} else if (e.type === TOUCH.START) {
								self.on(TOUCH.END + '.' + ns, swipeEnd);
							}
							clickedOnce = false;
						},
					
					// Un-registering events
					unregisterEvents = function(e) {
							//log('unregisterEvents');
							if (e.type === 'mouseup' || e.type === 'mouseout') {
								self.off('mouseup.' + ns + ' mouseout.' + ns);
								$document.off('mousemove.' + ns);
							} else if (e.type === TOUCH.END) {
								self.off(TOUCH.END + '.' + ns);
							}
						},
						
					// Setting position
					setPos = function(pos, timing, opacity, doneFn) {
							var m = Array.from(mtx || (mtx = self.getMatrix())),
								tr,
								
								applyTransform = function() {
										//log(self[0].id + tr);
										if (timing) {
											if (typeof opacity !== UNDEF) {
												self.css({
														transition:		'transform ' + timing + 'ms ease-out, opacity ' + timing + 'ms linear',
														opacity:		opacity,
														transform:		tr
													});
											} else {
												self.css({
														transition:		'transform ' + timing + 'ms ease-out',
														transform:		tr
													});
											}
										} else {
											self.css({
														transition:		'none',
														transform:		tr
													});
										}
									};
							
							if (self.data.rolling) {
								
							}
							
							if (!m.length) {
								return;
							}
							
							if (m.length === 16) {
								if (pos[0]) {
									m[12] += pos[0];
								}
								if (pos[1]) {
									m[13] += pos[1];
								}
								tr = 'matrix3d(' + m.join(',') + ')';
							} else {
								tr = 'translateX(' + (m[4] + pos[0]) + 'px)';
								if (pos[1]) {
									tr += ' translateY(' + (m[5] + pos[1]) + 'px)';
								}
								if (m[0] !== 1 || m[3] !== 1) {
									tr += ' scale(' + m[0] + ',' + m[3] + ')';
								}
							}
							
							if (typeof doneFn === FUNCTION) {
								self.one('transitionend', doneFn);
							}
							
							window.requestAnimationFrame(applyTransform);

						},
					
					// Read position from matrix
					getPosFromMtx = function(mtx) {
							
							return (mtx.length === 16)? [ mtx[12], mtx[13] ] : [ mtx[4], mtx[5] ];
						},
						
					// Starting swipe
					swipeStart = function(e) {
							//log('swipeStart:' + e.type);
							
							if (clickedOnce) {
								// No mouseup / touchend registered yet
								return false;
							}
							
							if (e.type !== 'touchstart') {
								// Cancel bubbling, but not break long tap
								e.preventDefault();
							}
							
							if (e.type === 'mousedown' && e.which !== 1) {									
								// Right click
								handleRightClick(e);
								return true;
							}
							
							//log('Touch start, fingers: ' + (e.originalEvent.touches? e.originalEvent.touches.length : 0));
							if (pageZoomed() ||
								e.originalEvent.touches && e.originalEvent.touches.length > 1 ||			// multi finger touch
								(options.noSwipeInside && $(e.target).closest(options.noSwipeInside).length) ||
								e.target.nodeName === 'A' ||
								e.target.nodeName === 'INPUT' ||
								e.target.nodeName === 'BUTTON' ||
								e.target.nodeName === 'AUDIO' || 
								(e.target.nodeName === 'VIDEO' && 
									e.target.controls && 
									(ISIOSDEVICE || e.offsetY > ($(e.target).height() - 60)))
								) {
								// Don't swipe buttons and links and other non-swipeable elements
								// No swiping on iOS video player
								nonSwipeable = true;
								return true;
							}
							
							nonSwipeable = false;
							ended = false;
							clickedOnce = false;
							
							//self.stop(true, false);
							self.data({
									scrolling: 		true,
									swipeEnded:		false,
									taplength:		0
								});
							
							startTime = new Date().getTime();
							startPos = getXY(e.originalEvent);
							
							clipDim = getDim(clip);
							selfDim = getDim(self);
							
							if (clipDim[1] >= selfDim[1]) {
								dontSwipeUp = typeof options.onSwipedUp !== FUNCTION;
								dontSwipeDown = typeof options.onSwipedDown !== FUNCTION;
							} else {
								dontSwipeUp = dontSwipeDown = false;
							}
							
							diff = [0, 0];
							mtx = self.getMatrix();
							if (!mtx) {
								// No transform applied so far
								mtx = [ 1, 0, 0, 1, 0, 0 ];
								
								self.css({
										transform:	'matrix(1, 0, 0, 1, 0, 0)'
									});
							}
							
							pos = getPosFromMtx(mtx);
							registerEvents(e);
							handleSwipeStart(e);
							swipeHandled = false;
							// Do not cancel long tap
							return e.type === 'touchstart';
						},
					
					// Moving in progress
					swipeMove = function(e) {
							//log('swipeMove:' + e.type);
							
							if (ended || nonSwipeable || 
								e.originalEvent.touches && e.originalEvent.touches.length > 1) {
								// not active, prohibited area or multi finger touch
								return true;
							}
							
							e.preventDefault();
							
							diff = getXY(e.originalEvent);
							diff[0] -= startPos[0];			// Relative distance from start
							
							if (dontSwipeUp && dontSwipeDown) {
								// Horizontal-only
								diff[1] = 0;
							} else {
								diff[1] -= startPos[1];
								// Can move both axes
								if (Math.abs(diff[0]) < 100 && Math.abs(diff[0]) > 15 && Math.abs(diff[0] / diff[1]) > 0.667) {
									// Change to horizontal-only swipe
									dontSwipeUp = dontSwipeDown = true;
								}
								
								if (dontSwipeUp) {
									diff[1] = Math.max(0, diff[1]);
								} else if (dontSwipeDown) {
									diff[1] = Math.min(0, diff[1]);
								}
							}
							
							setPos(diff);		//[/*pos[0] + */diff[0], /*pos[1] + */diff[1]]);
							
							return false;
						},
					
					// Reset swipe (e.g. moved out of the screen)
					swipeReset = function(e) {
							//log('swipeCancel:' + e.type);
							
							ended = true;
							
							if (self.data('scrolling')) {
								setPos([0, 0], 100, 1, function() {
										self.data({
											scrolling:	 	false,
											swipeEnded:		true
										});
									});
							}
						},
						
					// Stopped moving or clicked : mouse/touch up
					swipeEnd = function(e) {
							//log('swipeEnd:' + e.type);
							if (nonSwipeable) {
								// Prohibited area
								return true;
							}
							
							e.preventDefault();
							
							if (chkDoubleClick) {
								clearTimeout(chkDoubleClick);
								chkDoubleClick = null;
							}
							
							ended = true;
							endTime = new Date().getTime();
							
							if ((endTime - startTime) > 50 && (getDistance(diff) > options.threshold)) {
								//log('swiped: ' + (endTime - startTime) + 'ms ' + getDistance(diff) + 'px');
								// Not a click, because it took longer or the distance is larger than the threshold
								var speed = 1 + options.swipeoutSpeed / (endTime - startTime),
									canceled = !!self.data.swipeEnded;
								
								self.data({
										scrolling:	 	true,
										swipeEnded:		true,
										taplength:		endTime - startTime
									});
								
								if (dontSwipeUp && dontSwipeDown) {
									var targetPos = canceled? 0 : diff[0] * speed;
									
									if ((clipDim[0] >= selfDim[0]) ||
											Math.abs(diff[0]) > 0 && (
												((targetPos - selfDim[0] / 2) > clipDim[0] * (options.overThreshold - 0.5)) ||
												((targetPos + selfDim[0] / 2) < clipDim[0] * (0.5 - options.overThreshold))
											)
										) {
										// Swiped
										setPos([ targetPos, 0 ], options.swipeoutSpeed, canceled? 1 : 0);
										
										if (diff[0] > 0) {
											handleSwipedRight(e);
										} else {
											handleSwipedLeft(e);
										}
										
									} else {
										// Panoramic :: Hasn't reached the edge yet
										setPos([ targetPos, 0 ], options.swipeoutSpeed);
										unregisterEvents(e);
									}
									
								} else {
									var vertical = Math.abs(diff[1]) > 1.5 * Math.abs(diff[0]); // max 30 degrees
										targetPos = canceled?
											[ 0, 0 ]
											:
											[
												pos[0] + diff[0] * speed,
												pos[1] + diff[1] * speed
											];
									
									if (vertical && (
											selfDim[1] > clipDim[1] && ( 
												(diff[1] > 0 && (targetPos[1] - selfDim[1] / 2) <= clipDim[1] * (options.overThreshold - 0.5)) ||
												(diff[1] < 0 && (targetPos[1] + selfDim[1] / 2) >= clipDim[1] * (0.5 - options.overThreshold))
											)
										) ||
										!vertical && (
											selfDim[0] > clipDim[0] && ( 
												(diff[0] > 0 && (targetPos[0] - selfDim[0] / 2) <= clipDim[0] * (options.overThreshold - 0.5)) ||
												(diff[0] < 0 && (targetPos[0] + selfDim[0] / 2) >= clipDim[0] * (0.5 - options.overThreshold))
											)
										)) { 
										// Hasn't reached the edge yet; continue scrolling
										setPos(targetPos, options.swipeoutSpeed, canceled? 1 : 0);
										timer = setTimeout(handleFinished, options.swipeoutSpeed + 20, e);
										unregisterEvents(e);
									} else {
										// Swiped over the boundaries										
										if (typeof options.onSwipedDown === FUNCTION &&
											vertical && diff[1] > 0) {
											// Swiped down
											targetPos[0] = pos[0];
											setPos(targetPos, options.swipeoutSpeed, 0, handleFinished);
											handleSwipedDown(e);
										} else if (typeof options.onSwipedUp === FUNCTION && 
											vertical && diff[1] < 0) {
											// Swiped up
											targetPos[0] = pos[0];
											setPos(targetPos, options.swipeoutSpeed, 0, handleFinished);
											handleSwipedUp(e);
										} else {
											// Horizontal swipe or vertical but no up or down swipe handler
											setPos(targetPos, options.swipeoutSpeed, 0, handleFinished);
											if (diff[0] > 0) {
												handleSwipedRight(e);
											} else {
												handleSwipedLeft(e);
											}
										}
									}
								}	
								
								clickedOnce = false;
								
							} else if (diff[0] || diff[1]) {
								// Small swipe: no click triggered
								handleSmallSwipe(e);
								setPos([ 0, 0 ], 100);
															
								clickedOnce = false;
								
							} else if (options.onDoubleClick) {
								
								// Has doubleclick handler event
								if (clickedOnce) {
									// One click already registered
									if ((new Date().getTime() - endTime) < options.doubleClickThreshold) {
										// Within threshold: Double click
										handleDoubleClick(e);
									} else {
										// Over threshold
										handleClick(e);
									}
									clickedOnce = false;
										
								} else { 
									// Wait if second click happens
									clickedOnce = true;
									// Triggering single click if no second click comes
									chkDoubleClick = setTimeout(function(e) {
											handleClick(e);
											clickedOnce = false;
										}, options.doubleClickThreshold + 10, e);
								}
								
							} else {
								// Handling only single clicks
								handleClick(e);
							}
							
							return false;
						},
					
					noAction = function(e) {
							e.preventDefault();
							return false;
						},
					
					// Tearing off the swipe handler
					removeSwipe = function() {
							setTimeout(function() { 
									self.data('scrolling', false);
								}, 20);
							self.removeAttr('draggable');
							self.add($document).off('.' + ns);
						};
					
				// clean up events that may hang around (in theory this should never happen)
				self.add($document).off('.' + ns);
				
				// Storing namespace
				self.data('lsw_ns', ns);
				
				// Need to be updated about the window size
				$window.on('resize.' + ns, function() {
						clearTimeout(rto);
						rto = setTimeout(function() {
							clipDim = getDim(clip);		
						}, 50);
					});
				
				self.attr('draggable', 'false')
					.on(TOUCH.START + '.' + ns + ' dragstart.' + ns + ' mousedown.' + ns, swipeStart)
					.on(TOUCH.MOVE + '.' + ns + ' drag.' + ns, swipeMove)
					// External recentering request 
					.on('resetSwipe.' + ns, function() { 
						swipeReset(); 
					})
					// External remove request
					.on('removeSwipe.' + ns, removeSwipe)
					// Do not select
					.on('selectstart.' + ns, noAction);
				
				//self.logEvents();
		
			});
		};
			
})(jQuery, jQuery(window), jQuery(document));
