;
/* 
 *	The Album model
 * 
 *	(c) Laszlo Molnar, 2015, 2017, 2020, 2021
 *	Licensed under Creative Commons Attribution-NonCommercial-ShareAlike 
 *	<http://creativecommons.org/licenses/by-nc-sa/3.0/>
 *
 *	requires: jQuery, laza.util
 */

 
/*
 * 	Constants
 */
 
const J = {
			// jAlbum variables
			ALBUM:					'album',				// jAlbum's album data 
			FOLDERS: 				'folders',				// Folders array
			NAME: 					'name',					// FileName of the item
			PATH: 					'path',					// Path to file
			THUMB: 					'thumb',				// Thumbnail properties object
			IMAGE: 					'image',				// Image properties object
			WIDTH: 					'width',				// Width
			HEIGHT: 				'height',				// Height
			RENDITIONS:				'renditions',			// Variant renditions
			ORIGINAL: 				'original',				// Original object
			OBJECTS: 				'objects',				// Objects array (not folders)
			FILEDATE: 				'fileDate',				// File modified date
			COMMENT: 				'comment',				// Comment
			TITLE: 					'title',				// Title
			KEYWORDS:				'keywords',				// Keywords
			COUNTERS: 				'counters',				// Counters object
			DEEPCOUNTERS: 			'deepCounters',			// Deepcounters object
			FILESIZE: 				'fileSize',				// File size (int)
			CATEGORY: 				'category',				// Category
			RATING:					'rating',				// Rating
			CAMERA: 				'camera',				// Camera data object
			VIDEO: 					'video',				// Video data object
			DURATION:				'duration',				// Video duration
			FPS:					'fps',					// Video frame per sec.
			HIDDEN:					'hidden',				// Hidden
			// extra vars
			LEVEL: 					'level',				// Folder depth level
			PATHREF: 				'pathRef',				// Path from root
			PARENTREF: 				'parentRef',			// Path to parent from root
			RELPATH: 				'relPath',				// Relative path from currentFolder to this folder 
			FOLDERCAPTION:			'folderCaption',		// Folder caption derived from template
			IMAGECAPTION: 			'imageCaption',			// Image caption derived from template
			THUMBCAPTION: 			'thumbCaption',			// Thumbnail caption derived from template
			PHOTODATA: 				'photodata',			// Formatted photo data
			LOCATION: 				'location',				// GPS location
			REGIONS:				'regions',				// Regions (face tags)
			SHOP:					'shop',					// Global or individual shop options
			EXTERNAL:				'external',				// External (link or content)
			PROJECTIONTYPE:			'projectionType',		// 3D image projection type (equirectangular)
			GPANO:					'gpano',				// Globe panorama (360°)
			ORIGINALFILE:			'originalFile',			// Path to the original file in case of GIF's
			SLIDESHOWINTERVAL:		'slideshowInterval',	// Custom slideshow delay
			TRANSITIONSPEED:		'transitionSpeed',		// Custom transotoin speed
			DATES:					'dates',				// Dates object
			ADDED:					'added',				// Added date
			DATETAKEN:				'dateTaken',			// Date taken
			FILEMODIFIED:			'fileModified',			// File modified
			DATERANGE:				'dateRange',			// Date range (folders)
			MOSTPHOTOS: 			'mostphotos',			// Mostphotos ID
			HIDEFOTOMOTO:			'hideFotomoto',			// Hide from Fotomoto
			FOTOMOTOCOLLECTION:		'fotomotoCollection',	// Fotomoto collection type
			AUDIOCLIP:				'audioClip',			// Attached audio clip (mp3)
			PANORAMA:				'panorama',				// To be treated as panorama?
			FILTERS:				'filters',				// Filters array (global or folders)
			SORT:					'sort',					// Sort array (global or folders)
			VISITORRATING:			'visitorRating',		// Ratings from jAlbum
			OBJ: 					'obj',					// Store item as data attr of an element: el.data(J.OBJ)  
			LOADCOUNTER:			'loadcounter',			// Load counter array by category
			TOTAL:					'total',				// Total items loaded
			FOLDERINDEX:			'folderindex',			// Numbering folders: 0 ... N
			READY:					'ready',				// Data has been loaded
			ONREADY:				'onReady',				// Cached functions for data loaded event
			DEEP:					'deep',					// Using tree or deep-data structure?
			SIZE:					'size',					// External content embed size 
			CONT:					'cont',					// External content
			TYPE:					'type'					// E.g. for audioClip.type
		},
	
	JCAMERAFIELDS = [
			'aperture',
			'exposureTime',
			'originalDate',
			'cameraModel',
			'location',
			'focusDistance',
			'focalLength35mm',
			'cameraMake',
			'resolution',
			'isoEquivalent',
			'flash',
			'focalLength'
		]; 

/*
 *	Album object :: use 
 *		myAlbum = new Album({settings});
 *		or
 *		myAlbum = new Album();
 *		myAlbum.init({settings});
 */

var Album = function($, options) {
	
	let instance = null,
	
		settings = {
				// Name of the tree file
				treeFile: 				'tree.json',
				// Name of the folder data
				dataFile: 				'data1.json',
				// Name of the master file
				deepDataFile:			'deep-data.json',
				// index file name
				indexName: 				'index.html',
				//Folder image file name
				folderImageFile:		'folderimage.jpg',
				// Folder image dimensions
				folderImageDims:		[ 1200, 800 ],
				// Folder thumbnail file name 
				folderThumbFile:		'folderthumb.jpg',
				// Folder thumbnail dimanesions
				folderThumbDims:		[ 1024, 768 ],
				// Thumbnail dimensions
				thumbDims:				[ 240, 180 ],
				// Name of the thumbs folder
				thumbsDir: 				'thumbs',
				// Name of the slides folder
				slidesDir: 				'slides',
				// Name of the hires folder
				hiresDir:				'hi-res',
				// High DPI thumbnails
				hiDpiThumbs:			false,
				// High DPI images
				hiDpiImages:			false,
				// Use originals?
				useOriginals:			false,
				// Do thumbs fill the frame or fit 
				thumbsFillFrame:		true,
				// Default poster images
				audioPoster:			'audio.poster.png',
				defaultAudioPosterSize:	[ 628, 360 ],
				videoPoster:			'video.poster.png',
				defaultVideoPosterSize:	[ 628, 360 ],
				// Path to root from current folder (eg. "../../")
				rootPath: 				'',
				// Relative path to the current folder (eg. "folder/subfolder") 
				relPath: 				'',
				// Loading the whole data tree
				loadDeep:				false,
				// Lazy load :: loads folder data only when requested or at the time of initialization
				lazy: 					true,
				// Fatal error handling
				fatalError:				(txt, attr) => log('Error: ' + ((typeof attr === UNDEF)? translate(txt) : translate(txt).replace(/\{0\}/g, attr))),
				// Possible object types
				possibleTypes:			[ 'folder', 'webPage', 'webLocation', 'image', 'video', 'audio', 'other' ],
				locale:					'en-US'
			},
			
		// Texts translated
		text = getTranslations({
				and:						'and',
				from:						'From {0}'
			}),
		// Creating RegExp with localized AND
		andRegExp = new RegExp(`\\s+(?:&|${text.and})\\s+`, 'gu'),
		// Global variables
		// URL of to top level album page (null => we're in the album's subfolder, using the relative "rootPath") 
		albumPath = null,
		// Absolute URL of the top level page
		absolutePath,
		// Cache buster
		cacheBuster = '',
		// The container for the entire tree
		tree,
		// Collection of album paths from root in order to store only the references
		paths = [],
		// Collection of relative paths in order to store only the references
		relPaths = [],
		// Path to current folder
		currentFolder,
		// Collection the JSON promises
		defer = [],
		// Album ready state: tree.json and data1.json is loaded
		ready = false,
		// Deep ready: all the data structure is ready
		deepReady = false,
		// Collecting deep ready callbacks to execute on deep ready event
		deepReadyFunctions = [],
		// Is deep ready?
		isDeep = () => (tree && tree.hasOwnProperty(J.DEEP) && tree[J.DEEP]),
		
		/***********************************************
		 *					Debug
		 */
		 
		// Logging
		scriptName = 'jalbum-album.js',
	
		log = txt => {
			
				if (console && typeof txt !== UNDEF) {
					
					if (txt.match(/^Error\:/i)) {
						console.error(scriptName + ' ' + txt);
					}
					
					if (typeof DEBUG !== UNDEF && DEBUG) {
						if (txt.match(/^Warning\:/i)) {
							console.warn(scriptName + ' ' + txt);
						} else if (txt.match(/^Info\:/i)) {
							console.info(scriptName + ' ' + txt);
						} else {
							console.log(scriptName + ' ' + txt);
						}
					}
				}
			},
				
		// Utility functions
		getFilenameFromURL = n => decodeURIComponent(n.slice(n.lastIndexOf('/') + 1)),
		
		// File name without extension
		getBasename = o => { 
				var m = getItemName(o).match(/^(.+)\./); 
				return m? m[1] : '';
			},
				
		// File extension
		getExtension = o => { 
				var m = getItemName(o).match(/\.(\w+)$/); 
				return m? m[1] : '';
			},
				
		// Can the browser display this image type?
		isBrowserFriendly = n => {
				var m = n.match(/\.\w+$/);
				return m? ('.jpg.jpeg.png.webp.webm.gif'.indexOf(m[0].toLowerCase()) !== -1) : false;
			},
			
		/***********************************************
		 *					Type check
		 */
		 
		// Image?
		isImage = o => o.hasOwnProperty(J.CATEGORY) && o[J.CATEGORY] === 'image',
			
		// Audio?
		isAudio = o => o.hasOwnProperty(J.CATEGORY) && o[J.CATEGORY] === 'audio',
			
		// Video?
		isVideo = o => o.hasOwnProperty(J.CATEGORY) && o[J.CATEGORY] === 'video',
			
		// Folder?
		isFolder = o => o.hasOwnProperty(J.FOLDERINDEX) || o.hasOwnProperty(J.LEVEL),
			
		// Ordinary object?
		isLightboxable = o => (o.hasOwnProperty(J.CATEGORY) && 'image.video.audio.other'.indexOf(o[J.CATEGORY]) !== -1),
	
		// Is this the current folder?
		isCurrentFolder = o => o === currentFolder,
			
		/***********************************************
		 *					Access
		 */
		 
		// Returns path separated with separator 
		/*	'': 			current, 
			'/': 			album root, 
			'folderName': 	subfolder,
			'name1/name2':	deep folder
		*/
		
		makePath = function() {
			
				if (arguments.length) {
					let p = [];
					
					for (let i = 0, a; i < arguments.length; i++) {
						a = arguments[i];
						
						if (!a.length) {
							// Can be empty
							continue;
						}
						
						if (a === '/') {
							// Return to root
							p = [];
						} else {
							// Removing trailing slashes if any
							if (a[0] === '/') {
								a = a.slice(1);
							}
							if (a.slice(-1) === '/') {
								a = a.slice(0, -1);
							}
							
							if (a.length) {
								// Adding segment
								p.push(a);
							}
						}
					}
						
					return p.join('/');
				}
				
				return '';
			},
			
		// Get absolute path to folder from relative folder path
		
		getAbsoluteFolderPath = path => {
			
				if (typeof path === UNDEF) {
					return window.location.href.substring(0, window.location.href.lastIndexOf('/'));
				} else if (path.match(/^https?\:\/\//i)) {
					// Absolute
					return path;
				} else if (path[0] === '/') {
					// Site root relative
					return window.location.origin + path;
				} else {
					let url = window.location.href;
					// Find last '/'
					url = url.substring(0, url.lastIndexOf('/'));
					// Go back until ../
					if (path.endsWith('..')) {
						// Ensure to avoid ".." ending
						path += '/';
					}
					while (path.startsWith('../')) {
						url = url.substring(0, url.lastIndexOf('/', url.length - 2));
						path = path.slice(3);
					}
					return url + path;
				}
			},
			
		// Getting path reference either by searching or adding as new
		// 0 => top level folder
		
		getPathRef = p => {
			
				if (typeof p === UNDEF || !p) {
					// Top level folder
					return 0;
				}
			
				if (p.slice(-1) !== '/') {
					// Sanitizing paths: adding extra slash at the end 
					p += '/';
				}
				
				let idx = paths.indexOf(p);
				
				if (idx >= 0) {
					// Found => return
					return idx + 1;
				}
				
				// Add new
				return paths.push(p);
			},
			
		// Reverse function
		
		toPath = o => (o && o.hasOwnProperty(J.PATHREF) && o[J.PATHREF])? paths[o[J.PATHREF] - 1] : '',
		
		// Getting path reference either by searching or adding as new
		// 0 => current folder
		
		getRelPathRef = p => {
			
				if (typeof p === UNDEF || !p) {
					// Current folder
					return 0;
				}
			
				if (p.slice(-1) !== '/') {
					// Sanitizing paths: adding extra slash at the end 
					p += '/';
				}
				
				let idx = relPaths.indexOf(p);
				
				if (idx >= 0) {
					// Found => return
					return idx + 1;
				}
				
				// Add new
				return relPaths.push(p);
			},
		
		// Reverse function
		
		toRelPath = o => (o && o.hasOwnProperty(J.RELPATH) && o[J.RELPATH])? relPaths[o[J.RELPATH] - 1] : '',	
		
		// Fixing path containing ../ part(s)
		
		fixUrl = p => {
				let i, 
					j, 
					t = p + '';
					
				while ((i = t.indexOf('../')) > 0) {
					if (i === 1 || (j = t.lastIndexOf('/', i - 2)) === -1) {
						return t.substring(i + 3);
					}
					t = t.substring(0, j) + t.substring(i + 2);
				}
				
				return t;
			},
			
		// Get absolute URL to album's root folder
		
		getAlbumPath = () => {
			
				if (albumPath !== null) {
					return albumPath;
				}
				
				let p = window.location.pathname,
					l = currentFolder[J.LEVEL];
				
				do {
					p = p.substring(0, p.lastIndexOf('/'));
					l = l - 1;
				} while (l >= 0);
				
				return p;
			},
			
		// Get album root folder
		
		getAlbumRootPath = () => absolutePath,
			
		// Getting folder path for an object (absolute or relative)
		
		getPath = o => {
			
				if (typeof o !== UNDEF) {
					return (albumPath !== null)? makePath(albumPath, toPath(o)) : toRelPath(o);
				}
				// Top level or external reference to top level
				return albumPath || '';
			},
			
		// Getting path from album root to the item's folder
		
		getFolderPath = o => (typeof o !== UNDEF)? toPath(o) : '',
		
		// Getting absolute path for an object
		
		getAbsolutePath = o => (
				o.hasOwnProperty(J.LEVEL)?
					// Folder
					makePath(albumPath || absolutePath, getFolderPath(o))
					:
					// Object
					makePath(albumPath || absolutePath, getFolderPath(o), settings.indexName + '#img=' + o[J.PATH])
			),
			
		// Gets the path to item: absolute or relative
		
		getItemPath = o => {
				if (typeof o !== UNDEF) {
					let p = getPath(o),
						c = o[J.CATEGORY] || 'folder';
					
					if (c === 'folder') {
						return p;
					} else if (c === 'video') {
						return makePath(p, o[J.VIDEO][J.PATH]);
					} else if (c === 'audio' || c === 'other' || o.hasOwnProperty(J.ORIGINAL)) {
						return makePath(p, o[J.ORIGINAL][J.PATH]);
					} else if (c === 'image') {
						return makePath(p, o[J.IMAGE][J.PATH]);
					} else if (c === 'webPage') {
						return makePath(p, o[J.PATH]);
					} else {
						// webLocation: absolute
						return o[J.PATH];
					}
				}
				
				return null;
			},
			
		// Gets the audioclip path
		
		getAudioClipPath = o => 
				(typeof o !== UNDEF && o.hasOwnProperty(J.AUDIOCLIP))?
					makePath(getPath(o), o[J.AUDIOCLIP][J.PATH]) 
					: 
					null,
			
		// Getting path to object from album root
		
		getObjectPath = o => 
				(typeof o !== UNDEF)? 
					(toPath(o) + ((o[J.CATEGORY] !== 'folder')? o[J.PATH] : '')) 
					: 
					null,
			
		// Tries to figure out the items dimensions
		
		getDimensions = o => {
				
				if (typeof o === UNDEF || o[J.CATEGORY] === 'other' && getExtension(o).toLowerCase() === 'pdf') {
					// No (valid) object or has no dimensions
					return null;
				} else if (o.hasOwnProperty(J.EXTERNAL)) {
					// External content
					let s = o[J.EXTERNAL][J.SIZE];
					
					if (s) {
						// Has manually specified size
						s = s.split('x');
						return [ s[0], s[1] || Math.round(s[0] *.75)];
						
					} else {
						s = guessDimensions(o[J.EXTERNAL][J.CONT]);
						if (s) {
							// Has explicitly given size in the HTML code
							return s;
						} else {
							// No info
							if (o[J.EXTERNAL][J.CONT].includes('vimeo.com') ||
								o[J.EXTERNAL][J.CONT].includes('youtube.com') ||
								o[J.EXTERNAL][J.CONT].includes('youtu.be')) {
								// In case of YouTube or Vimeo, set 720p as default
								w = 1280;
								if ((s = o[J.EXTERNAL][J.CONT].match(/.*style="padding(-bottom)?:\s?([\d\.]+)%/)) && s && (s.length > 1)) {
									// Responsive embedding
									return [ 1280, Math.round((w * parseFloat(s[2])) / 100) ];
								} else {
									return [ 1280, 720 ];
								}
							} else {
								return [ o[J.IMAGE][J.WIDTH], o[J.IMAGE][J.HEIGHT] ];
							}
						}
					}
				} else if (o[J.CATEGORY] === 'audio' && o[J.IMAGE][J.PATH].endsWith('res/audio.png')) {
					// Audio with poster image
					return [ settings.defaultAudioPosterSize[0], settings.defaultAudioPosterSize[1] ];
					
				} else if (o[J.CATEGORY] === 'video') {
					// Video
					return [ o[J.VIDEO][J.WIDTH], o[J.VIDEO][J.HEIGHT] ];
					
				}
				
				if (settings.useOriginals && settings.hiDpiImages) {
					return [ o[J.IMAGE][J.WIDTH] / 2, o[J.IMAGE][J.HEIGHT] / 2 ];
				}
				
				return [ o[J.IMAGE][J.WIDTH], o[J.IMAGE][J.HEIGHT] ];
			},
			
		// Tries to figure out the original image dimensions
		
		getOriginalDimensions = o => {
			
				if (typeof o !== UNDEF) {
				
					if (o[J.CATEGORY] === 'video') {
						// Video
						return [ o[J.VIDEO][J.WIDTH], o[J.VIDEO][J.HEIGHT] ];
					}
					
					if (o.hasOwnProperty(J.ORIGINAL)) {
						return [ o[J.ORIGINAL][J.WIDTH], o[J.ORIGINAL][J.HEIGHT] ];
					}
				}
				
				return null;
			},
			
		// Tries to figure out the original image dimensions
		
		getMaxDimensions = o => {
			
				if (typeof o !== UNDEF) {
					let dim;
					
					if (settings.useOriginals && (dim = getOriginalDimensions(o))) {
						// Has original
						return dim;
					} else if (o.hasOwnProperty(J.IMAGE)) {
						// Has image
						let img = o[J.IMAGE];
						
						if (img.hasOwnProperty(J.RENDITIONS)) {
							// Has renditions (variants)
							dim = [ img[J.RENDITIONS][0][J.WIDTH], img[J.RENDITIONS][0][J.HEIGHT] ];
						
							for (let i = 1; i < img[J.RENDITIONS].length; i++) {
								if (img[J.RENDITIONS][i][J.WIDTH] > dim[0]) {
									dim = 	[ img[J.RENDITIONS][i][J.WIDTH], img[J.RENDITIONS][i][J.HEIGHT] ];
								}
							}
							
							return dim;
							
						}
							
						return settings.hiDpiImages?
								[ 2 * img[J.WIDTH], 2 * img[J.HEIGHT] ]
								:
								[ img[J.WIDTH], img[J.HEIGHT] ];
					}
				}
				
				return null;
			},
			
		// Returns the parsed regions array
		
		getRegions = function(o) {
			
				if (typeof o !== UNDEF && o.hasOwnProperty(J.REGIONS)) {
					return JSON.parse(o[J.REGIONS]);
				}
				
				return [];
			},
			
		// Returns absolute or relative link to object in the real album
		
		getLink = function(o) {
			
				if (typeof o !== UNDEF) {
					
					switch (o[J.CATEGORY]) {
						
						case 'folder':
							
							return getPath(o);
							
						case 'webLocation':
							
							return o[J.PATH];
							
						case 'webPage':
							
							return makePath(getPath(o), o[J.PATH]);
							
						default:
							
							// Image or other lightboxable item
							return makePath(getPath(o), '#img=' + o[J.PATH]);
					}
				}
				
				return null;
			},
							
		// Get path to item from root
		
		getRootPath = o => (typeof o !== UNDEF)? makePath(getPath(o), o[J.PATH]) : null,
			
		// Get the pointer to a folder in a tree from path reference number
		
		getPointer = n => {
			
				if (typeof n !== NUMBER || n <= 0) {
					// root
					return tree;
				}
				
				n--;
				
				if (n > paths.length) {
					log('Error: out of bounds path reference (' + n + ')!');
					return null;
				}
				
				return getFolder(paths[n]);
			},
				
		// Gets relative folder path from the current folder (unnecessary)
		
		getRelativeFolderPath = o => (typeof o !== UNDEF)? fixUrl(settings.rootPath + toPath(o)) : null,
					
		// Returns folder object by full path
		
		getFolder = path => {
				
				if (typeof path === UNDEF) {
					// Invalid
					return null;
				} else if (!path.length) {
					// Top level
					return tree;
				}
				
				if (path.endsWith('/')) {
					// Sanitize
					path = path.slice(0, -1);
				}
				if (path.startsWith('/')) {
					// Sanitize
					path = path.slice(1);
				}
				
				let f = tree,
					p = path.split('/'),
					level;
					
				for (level = 0; level < p.length; level++) {
					if (isDeep()) {
						if (f.hasOwnProperty(J.OBJECTS) && 
							(f = f[J.OBJECTS].find(o => (o[J.CATEGORY] === 'folder' && o[J.PATH] === p[level])))) {
							continue;
						}
					} else if (f.hasOwnProperty(J.FOLDERS) &&
						(f = f[J.FOLDERS].find(o => o[J.PATH] === p[level]))) { 
						continue;
					}
					// Not found: return null 
					return null;
				}
				
				// Matched full path
				return (level === p.length)? f : null;
			},
			
		// Returns current folder
		
		getCurrentFolder = () => currentFolder,
		
		// Getting the parent
		
		getParent = o => {
			
				if (typeof o === UNDEF) {
					// Invalid
					o = currentFolder;
				}
				
				if (o === tree) {
					// At top level
					return null;
				}
				
				// Getting current folder or the parent in case of a folder
				let p = o.hasOwnProperty(J.PARENTREF)? getPointer(o[J.PARENTREF]) : getPointer(o[J.PATHREF]);
				
				// Avoid endless loops
				return (p === o)? null : p;
			},
			
		// Getting an item from full path, waits for folder to load if not present
		// Optionally passing an extra parameter
		
		getItem = function(path, doneFn, p) {
				
				if (typeof doneFn !== FUNCTION) {
					return;
				}
				
				let _getItem = function(folder, args) {
							let name = args[0],
								doneFn = args[1];
								
							if (folder.hasOwnProperty(J.OBJECTS)) {
								let o;
								
								if (name.endsWith('.webloc')) {
									// Filtering for "slides/NAME.ext"
									name = name.slice(0, -7);
									o = folder[J.OBJECTS].find(function(o) {
											return o[J.CATEGORY] === 'webLocation' && o[J.IMAGE][J.PATH].slice(7).stripExt() === name;
										});
								} else {
									o = folder[J.OBJECTS].find(function(o) {
											return o.hasOwnProperty(J.ORIGINAL) && o[J.ORIGINAL][J.PATH] === name || o[J.PATH] === name;
										});
								}
								
								if (typeof o === UNDEF) {
									//log('Missing object "' + name +  '"');
									o = null;
								}
								
								if (typeof p !== UNDEF) {
									doneFn.call(o, p);
								} else {
									doneFn.call(o);
								}
							} else {
								//log('No "Objects" in folder "' + folder[J.NAME] + '"');
							}
						};
						
				if (!path) {
					// Tree
					if (typeof p !== UNDEF) {
						doneFn.call(tree, p);
					} else {
						doneFn.call(tree);
					}
				} else if (path.endsWith('/')) {
					// Folder
					if (typeof p !== UNDEF) {
						doneFn.call(getFolder(path), p);
					} else {
						doneFn.call(getFolder(path));
					}
				} else {
					// An object
					let i = path.lastIndexOf('/'),
						folder = (i === -1)? tree : getFolder(path.substring(0, i)),
						name = path.substring(i + 1);
					
					if (folder) {
						// Exists
						if (deepReady || folder.hasOwnProperty(J.READY) && folder.ready) {
							// Objects already loaded
							//log('Accessing "' + name + '" in folder "' + folder[J.NAME] + '"');
							_getItem(folder, [ name, doneFn ]);
							
						} else if (isDeep()) {
							// Loads deep, but database hasn't been loaded yet
							//log('Not yet deep ready when accessing "' + name + '" in folder "' + folder[J.NAME] + '"');
							onDeepReady(folder, _getItem, [ name, doneFn ]);
						} else {
							// Have to wait until objects get loaded
							//log('Folder "' + folder[J.NAME] + '" is not yet ready when accessing "' + name + '"');
							onFolderReady(folder, _getItem, [ name, doneFn ]);
						}
					}
				}
				
				return null;
			},
		
		// Returns all objects in a folder but subfolders
		
		getObjects = f => {
				let folder = (typeof f === UNDEF)? currentFolder : f,
					objects = [];
					
				if (folder.hasOwnProperty(J.OBJECTS)) {
					folder[J.OBJECTS].forEach(o => {
							if (o[J.CATEGORY] !== 'folder') {
								objects.push(o);
							}
						});
				}
				
				return objects;
			},
			
		// Returns all objects in a folder
		
		getAllObjects = f => {
				let folder = (typeof f === UNDEF)? currentFolder : f,
					objects = [];
					
				if (folder.hasOwnProperty(J.OBJECTS)) {
					folder[J.OBJECTS].forEach(o => {
							if (o.hasOwnProperty(J.FOLDERINDEX)) {
								objects.push(folder[J.FOLDERS][o[J.FOLDERINDEX]]);
							} else {
								objects.push(o);
							}
						});
				}
				
				return objects;
			},
			
		// Returns only the lightboxable items
		
		getImages = f => {
				let folder = (typeof f === UNDEF)? currentFolder : f,
					objects = [];
			
				if (folder.hasOwnProperty(J.OBJECTS)) {
					folder[J.OBJECTS].forEach(o => {
							if (isLightboxable(o)) {
								objects.push(o);
							}
						});
				}
							
				return objects;
			},
		
		// Returns the folders as an array. No hidden folders. Avoid copying the child folders.
		
		getFolders = f => {
				let folder = (typeof f === UNDEF)? currentFolder : f,
					folders = [],
					
					// Avoid copying deep structure; only the immediate properties
					copyProps = f => {
							let o = {};
							
							for (let prop in f) {
								// Copying folder variables besides OBJECTS and ALBUM (only those missing)
								if (prop !== J.OBJECTS && prop !== J.ALBUM && prop !== J.OBJECTS) {
									o[prop] = f[prop];
								}
							}
							
							return o;
						};	
							
				if (folder) {
					if (isDeep()) {
						if (folder.hasOwnProperty(J.OBJECTS)) {
							folder[J.OBJECTS].forEach(o => {
									if (o[J.CATEGORY] === 'folder' && !(o.hasOwnProperty(J.HIDDEN) && o.hidden)) {
										folders.push(copyProps(o));
									}
								});
						}
					} else if (folder.hasOwnProperty(J.FOLDERS)) {
						folder[J.FOLDERS].forEach(f => {
								if (!(f.hasOwnProperty(J.HIDDEN) && f.hidden)) {
									folders.push(copyProps(f));
								}
							});
					}
				}
							
				return folders;
			},
			
		/***********************************************
		 *
		 *				Properties
		 *
		 */
		 
		// Album make date/time in UTC
		
		getMakeDate = () => new Date(tree[J.FILEDATE]),
			
		// Album title
		
		getAlbumTitle = () => tree[J.TITLE] || tree[J.NAME],
			
		// Filename for images, originalname for other
		
		getItemName = o => (
				(o[J.CATEGORY] === 'video')?
					getFilenameFromURL(o[J.VIDEO][J.PATH])
					:
					(o.hasOwnProperty(J.ORIGINAL)? 
						getFilenameFromURL(o[J.ORIGINAL][J.PATH])
						:
						o[J.NAME]
					)
			),
		
		// Level?
		
		getLevel = o => {
				o = o || currentFolder;
				return o.hasOwnProperty(J.LEVEL)? o[J.LEVEL] : getLevel(getParent(o));
			},
						
		// Title
		
		getTitle = o => (o || currentFolder)[J.TITLE] || '',
			
		// Name
		
		getName = o => (o || currentFolder)[J.NAME] || '',
			
		// Label: underscores replaced with spaces, file extension removed
		
		getLabel = o => ((o || currentFolder)[J.NAME] || '').replace(/\.\w+$/, '').replace(/_/g, ' '),
			
		// Title or name for ALT tags
		
		getAlt = o => getTitle(o) || getLabel(o),
		
		// Comment
		
		getComment = o => (o || currentFolder)[J.COMMENT] || '',
			
		// Thumbnail path (folder thumbs located one level up!)
		
		getThumbPath = o => makePath(getPath(o), 
				o.hasOwnProperty(J.LEVEL)?
					// Folder
					o[J.THUMB][J.PATH].replace(o[J.PATH] + '/', '')
					:
					o[J.THUMB][J.PATH]
			),
			
		// Icon path?
		
		getIconPath = o => o[J.THUMB][J.PATH].match(/res\/\w+\.png$/)? o[J.THUMB][J.PATH] : '',
		
		// Image path (falls back to thumb path, e.g. PDF files)
		
		getImagePath = o => makePath(getPath(o), 
				o.hasOwnProperty(J.LEVEL)?
					// Folder: grab the same image as thumb from slides
					o[J.THUMB][J.PATH].replace(settings.thumbsDir + '/', settings.slidesDir + '/') 
					:
					// Image or other
					(o.hasOwnProperty(J.ORIGINALFILE)?
						o[J.ORIGINALFILE]
						:
						(o.hasOwnProperty(J.IMAGE)? 
							o[J.IMAGE][J.PATH] 
							: 
							o[J.THUMB][J.PATH]
						)
					)
			),
		
		// Image path from root
		
		getImagePathFromRoot = o => makePath(toPath(o), 
				o.hasOwnProperty(J.ORIGINALFILE)?
					o[J.ORIGINALFILE]
					:
					(o.hasOwnProperty(J.IMAGE)? 
						o[J.IMAGE][J.PATH] 
						: 
						o[J.THUMB][J.PATH]
					)
			),
		
		// Absolute image path
		
		getAbsoluteImagePath = o => makePath(absolutePath, getImagePathFromRoot(o)),
			
		// Theme image path
		
		getThemeImagePath = o => makePath(getPath(o), settings.folderImageFile),
			
		// Original path
		
		getOriginalPath = o => (o.hasOwnProperty(J.ORIGINALFILE)?
				makePath(getPath(o), o[J.ORIGINALFILE]) 
				:
				(o.hasOwnProperty(J.ORIGINAL)? 
					makePath(getPath(o), o[J.ORIGINAL][J.PATH]) 
					: 
					null
				)
			),
			
		// Video path
		
		getVideoPath = o => (o.hasOwnProperty(J.VIDEO)?
				makePath(getPath(o), o[J.VIDEO][J.PATH]) 
				: 
				null
			),
			
		// Poster path for audio and video files
		
		getPosterPath = o => {
				let c = o[J.CATEGORY],
					ip = o[J.IMAGE][J.PATH];
				
				// Audio and video: using default poster graphics if iconpath was used
				if ((c === 'audio' || c === 'video') &&	!ip.startsWith(settings.slidesDir + '/')) {
					return makePath(settings.rootPath, 'res', settings[c + 'Poster']);
				}
				
				return makePath(getPath(o), o[J.IMAGE][J.PATH]);
			},
			
		// Get optimum sized representing image
		
		getOptimalImage = (o, dim) => makePath(getPath(o), 
				o.hasOwnProperty(J.LEVEL)?
				(
					// Folder
					(dim[0] > settings.folderThumbDims[0] || 
					 dim[1] > settings.folderThumbDims[1])? 
						settings.folderImageFile 
						: 
						settings.folderThumbFile
				)
				:
				(
					// Not folder
					(o.hasOwnProperty(J.ORIGINALFILE)?
						o[J.ORIGINALFILE]
						:
						(
							(o.hasOwnProperty(J.IMAGE) && 
								(dim[0] > settings.thumbDims[0] || 
								dim[1] > settings.thumbDims[1])
							)? 
								o[J.IMAGE][J.PATH] 
								: 
								o[J.THUMB][J.PATH]
						)
					)
				)
			),
			
		// Finding closest rendition
		
		getClosestRendition = (r, dim, fill) => {
				
				if (!WEBP_LOSSY) {
					// No webp support: filtering out
					r = r.filter(o => !o.name.endsWith('.webp'));
				}
				
				if (r.length > 1) {
					let w = dim[0] * PIXELRATIO,
						h = dim[1] * PIXELRATIO;
						
					// Variants: calculate scale ratio
					r.forEach(o => {
							let s = fill? Math.max(w / o.width, h / o.height) : Math.min(w / o.width, h / o.height);
							
							o.match = (s > 1)? 
									// Upscaling penalized by factor 3: 1x -> 120% == 2x -> 50%
									(3 * (1 - 1 / s))
									:
									// Downscaling
									(1 - s)
						});
						
					// Sort by scale
					r.sort((o1, o2) => o1.match - o2.match);
				}
				
				return r[0];
			},
			
		// Get path to optimum image. If no dim specified the image is optimized for the screen size.
		
		getOptimalImagePath = (o, dim, useOriginal, fill) => {
				var fill = (typeof fill === UNDEF)? false : fill;
				
				if (o.hasOwnProperty(J.LEVEL)) {
					// A folder
					return toRelPath(o) + '/' + settings.folderImageFile;
				}
				
				if (o[J.CATEGORY] === 'webLocation') {
					// Web location
					return makePath(getPath(o), o[J.IMAGE][J.PATH]);
				}
			
				if (o.hasOwnProperty(J.ORIGINALFILE)) {
					return makePath(getPath(o), o[J.ORIGINALFILE]);
				}
			
				let r = o[J.IMAGE].hasOwnProperty(J.RENDITIONS)?
							getClosestRendition(o[J.IMAGE][J.RENDITIONS], dim || [ window.outerWidth, window.outerHeight ], fill)
							:
							(settings.hiDpiImages?
								{
									width:		2 * o[J.IMAGE][J.WIDTH],
									height:		2 * o[J.IMAGE][J.HEIGHT]
								}
								:
								{
									width:		o[J.IMAGE][J.WIDTH],
									height:		o[J.IMAGE][J.HEIGHT]
								}
							);
							
				if (typeof useOriginal !== UNDEF && useOriginal &&
					o.hasOwnProperty(J.ORIGINAL) && dim[0] > r[J.WIDTH] &&
					o[J.ORIGINAL][J.WIDTH] > o[J.IMAGE][J.WIDTH] &&
					isBrowserFriendly(o[J.ORIGINAL][J.PATH])) {
					// Using the original
					return makePath(getPath(o), o[J.ORIGINAL][J.PATH]);
				}
				
				// No original: Using the closest rendition
				return makePath(getPath(o), r.hasOwnProperty(J.NAME)? (settings.slidesDir + '/' + r[J.NAME]) : o[J.IMAGE][J.PATH]);
			},
		
		// Replace file name in a path
		
		replaceFile = (p, fn) => (p.slice(0, p.lastIndexOf('/') + 1) + fn),
		
		// Check if image dimensions are large enough to fit or fill the target
		
		hasEnoughPixels = (tdim, idim, mxz, fill) => (fill? 
				idim[0] * mxz >= tdim[0] && idim[1] * mxz >= tdim[1] 
				:
				idim[0] * mxz >= tdim[0] || idim[1] * mxz >= tdim[1]
			),
			
		// Get path to optimum thumb
		
		getOptimalThumbPath = (o, dim, useSlides) => {
			
				if (o[J.THUMB].hasOwnProperty(J.RENDITIONS)) {
					// Has renditions
					let r = getClosestRendition(o[J.THUMB][J.RENDITIONS], dim, settings.thumbsFillFrame);
					
					if (o[J.CATEGORY] === 'folder') {
						// Folder: only thumb path exists
						return makePath(getPath(o), replaceFile(o[J.THUMB][J.PATH], r[J.NAME]));
						
					} else if (o[J.CATEGORY] === 'video' || hasEnoughPixels(dim, [r[J.WIDTH], r[J.HEIGHT]], 1.07, settings.thumbsFillFrame)) {
						// Thumb is large enough: return thumb path
						return makePath(getPath(o),  settings.thumbsDir, r[J.NAME]);
					} else if (typeof useSlides !== UNDEF && useSlides && o[J.IMAGE].hasOwnProperty(J.RENDITIONS)) {
						// Try with image renditions
						r = getClosestRendition(o[J.IMAGE][J.RENDITIONS], dim, settings.thumbsFillFrame);
						
						return makePath(getPath(o), settings.slidesDir, r[J.NAME]);
					}
					
				}
				
				// No renditions
				let idim = [ o[J.THUMB][J.WIDTH], o[J.THUMB][J.HEIGHT] ];
					
				if (settings.hiDpiTHumbs) {
					idim[0] *= 2;
					idim[1] *= 2;
				}
				
				if (typeof useSlides !== UNDEF && useSlides && 
					(o[J.CATEGORY] !== 'video' && o.hasOwnProperty(J.IMAGE)) && 
					!hasEnoughPixels(dim, idim, 1.15, settings.thumbsFillFrame)) {
					// The thumb is too small, use the image
					return makePath(getPath(o), o[J.IMAGE][J.PATH]);
				}
				
				// Returning the thumb
				return makePath(getPath(o), o[J.THUMB][J.PATH]);
			},
		
		// Original or source path
		
		getSourcePath = (o) => makePath(getPath(o), 
				(o.hasOwnProperty(J.ORIGINAL)? 
					o[J.ORIGINAL][J.PATH] 
					: 
					(o.hasOwnProperty(J.IMAGE)? 
						o[J.IMAGE][J.PATH]
						:
						o[J.THUMB][J.PATH]
					)
				)
			),
			
		// Absolute path to an object as HTML page
		
		getAbsoluteItemPath = (o) => makePath(absolutePath, getItemPath(o)),
						
		// Get video duration in ms
		
		getVideoDuration = o => {
				let v = o[J.VIDEO];
				
				if (v && v.hasOwnProperty(J.DURATION)) {
					return videoDurationMs(v[J.DURATION]);
				}
				
				return null;
			},
			
		// Has shop options?
		
		hasShop = o => {
				if (typeof o === UNDEF) {
					return tree.hasOwnProperty(J.SHOP);
				}
				
				let p = getInheritedPropertyObject(o, J.SHOP);
				
				return p && (
							p.hasOwnProperty('usePrice') && p.usePrice 
							|| 
							p.hasOwnProperty('options') && p.options && p.options !== '-'
						);
			},
			
		// Has map location?
		
		hasLocation = o => o.hasOwnProperty(J.LOCATION) ||
				(o.hasOwnProperty(J.CAMERA) && o[J.CAMERA].hasOwnProperty(J.LOCATION)),
			
		// Get location in NN.NNN,NN.NNN format (Lat,long)
		
		getLocation = o => o.hasOwnProperty(J.LOCATION)? 
				o[J.LOCATION]
				:
				(
					(o.hasOwnProperty(J.CAMERA) && o[J.CAMERA].hasOwnProperty(J.LOCATION))?
						(o[J.CAMERA][J.LOCATION]['lat'] + ',' + o[J.CAMERA][J.LOCATION]['long']) 
						:
						null
				),
				
		// Hide from Fotomoto?
		
		hideFotomoto = o => o.hasOwnProperty(J.HIDEFOTOMOTO)? 
				o[J.HIDEFOTOMOTO]
				:
				getInheritedProperty(o, J.HIDEFOTOMOTO) || false
			,		
			
		// Get the lowest price of an item
		
		getPriceRange = o => {
				let p = getInheritedPropertyObject(o || tree, J.SHOP);
				
				if (p && p['options'] !== '-' && p['showPriceRange']) { 
					let	opt = p.options.split('::'),
						min = Number.MAX_VALUE,
						max = Number.MIN_VALUE;
					
					if (opt.length > 1) {
						for (let i = 0; i < opt.length; i++) {
							min = Math.min(parseFloat(opt[i].split('=')[1].split('+')[0]), min);
						}
						if (p.showPriceRange === 'minmax') {
							for (let i = 0; i < opt.length; i++) {
								max = Math.max(parseFloat(opt[i].split('=')[1].split('+')[0]), max);
							}
							return toCurrency(min, p['currency']) + '&ndash;' + toCurrency(max, p['currency']);
						}
						return text.from.template(toCurrency(min, p['currency']));
					} else {
						return toCurrency(opt[0].split('=')[1].split('+')[0], p['currency']);
					}
				}
				
				return '';
			},
			
		// Get shopping cart's currency, falls back to EUR
			
		getCurrency = () => getRootProperty(J.SHOP)['currency'] || 'EUR',
			
		// Counting folders recursively (current folder included)
		
		getDeepFolderCount = f => {
				let c = 0,
					folder = (typeof f === UNDEF)? currentFolder : f;
				
				if (folder.hasOwnProperty(J.DEEPCOUNTERS) && folder[J.DEEPCOUNTERS].hasOwnProperty(J.FOLDERS)) {
					// Has deep folder count
					return folder[J.DEEPCOUNTERS][J.FOLDERS] + 1;
				}
				
				if (isDeep()) {
					if (folder.hasOwnProperty(J.OBJECTS)) {
						folder[J.OBJECTS].forEach(o => {
								if (o[J.CATEGORY] === 'folder' && !(o.hasOwnProperty(J.HIDDEN) && o[J.HIDDEN])) {
									c += getDeepFolderCount(o);
								}
							});
					}
				} else if (folder.hasOwnProperty(J.FOLDERS)) {
					folder[J.FOLDERS].forEach(o => {
							if (!(o.hasOwnProperty(J.HIDDEN) && o[J.HIDDEN])) {
								c += getDeepFolderCount(o);
							}
						});
				}
				
				// Adding missing deep counter
				if (!folder.hasOwnProperty(J.DEEPCOUNTERS)) {
					folder[J.DEEPCOUNTERS] = {};
				}
				
				// Saving deep folder count
				folder[J.DEEPCOUNTERS][J.FOLDERS] = c;
				
				return c + 1;
			},
			
		// Getting folder count with limited folder depth
			
		getFolderCount = (f, levels) => {
				let folder = (typeof f === UNDEF)? currentFolder : f,
					maxLevel = folder[J.LEVEL] + ((typeof levels === UNDEF)? 0 : levels),
					
					getCount = folder => {
							let c = 1;
							
							if (folder[J.LEVEL] <= maxLevel) {
								// Count only within max levels
								if (isDeep()) {
									if (folder.hasOwnProperty(J.OBJECTS)) {
										folder[J.OBJECTS].forEach(o => { 
												if (o[J.CATEGORY] === 'folder' && !(o.hasOwnProperty(J.HIDDEN) && o[J.HIDDEN])) {
													c += getCount(o);
												}
											});
									}
								} else if (folder.hasOwnProperty(J.FOLDERS)) { 
									folder[J.FOLDERS].forEach(o => { 
											if (!(o.hasOwnProperty(J.HIDDEN) && o[J.HIDDEN])) {
												c += getCount(o); 
											}
										});
								}
							}
							
							return c;
						};
						
				return getCount(folder);
			},
			
		/********************************************
		 *
		 *		  Generic property functions
		 *
		 */
		 
		// Returns a property from the root level
		
		getRootProperty = a => tree.hasOwnProperty(a)? tree[a] : null,

		// Retrieving an Object property of an element with fallback to upper level folders
		
		getInheritedPropertyObject = (o, a) => {
				let p = {};
				
				do {
					if (o.hasOwnProperty(a)) {
						p = $.extend(true, {}, o[a], p);
					}
				} while (o = getParent(o));
				
				return (Object.getOwnPropertyNames(p)).length? p : null;
			},
						
		// Retrieving a single property of an element with fallback to upper level folders
		
		getInheritedProperty = (o, a) => {
				
				if (a.indexOf('.') >= 0) {
					a = a.split('.');
					
					if (a[0] === 'album') {
						return getRootProperty(a[1]);
					}
					
					do {
						if (o.hasOwnProperty(a[0])) {
							return $.extend(true, {}, o[a[0]][a[1]]);
						}
					} while (o = getParent(o));
					
					return null;
				}
					
				do {
					if (o.hasOwnProperty(a)) {
						return $.extend(true, {}, o[a]);
					}
				} while (o = getParent(o));
				
				return null;
			},
			
		// Returns a property (normal or inherited way)
		
		getProperty = (o, a, inherit) => {
				let r;
			
				if (inherit) {
					r = getInheritedProperty(o, a);
				} else if (a.indexOf('.') > 0) {
					a = a.split('.');
					r = (o.hasOwnProperty(a[0]))? o[a[0]][a[1]] : null;
				} else if (o.hasOwnProperty(a)) {
					r = o[a];
				}
				
				return $.extend(true, {}, r);
			},
		
		// Returns an Object property (normal or inherited way)
		
		getPropertyObject = (o, a, inherit) => ( 
				inherit? getInheritedPropertyObject(o, a) : (o.hasOwnProperty(a)? $.etxend(true, {}, o[a]) : null)
			),
		
		// Getting folder's index in Folders array
		
		_getFolderIndex = (parent, f) => {
				let p = f[J.PATH],
					idx = -1;
					
				if (isDeep()) {
					if (parent.hasOwnProperty(J.OBJECTS)) {
						idx = parent[J.OBJECTS].findIndex(o => (o[J.CATEGORY] === 'folder' && o[J.PATH] === p));
					}
				} else {
					if (parent.hasOwnProperty(J.FOLDERS)) {
						idx = parent[J.FOLDERS].findIndex(f => f[J.PATH] === p);
					}
				}
				
				return idx;
			},	
					
		// Returns the next folder
		
		_getNextFolder = function(f) {
				let folder = f || currentFolder,
					parent = getParent(folder),
					idx = _getFolderIndex(parent, folder[J.PATH]);
				
				if (idx >= 0) {
					if (settings.loadDeep) {
						if (parent.hasOwnProperty(J.OBJECTS)) {
							for (let i = idx + 1; i < parent[J.OBJECTS].length; i++) {
								if (parent[J.OBJECTS][i][J.CATEGORY] === 'folder' && 
									!(parent[J.OBJECTS][i].hasOwnProperty(J.HIDDEN) && parent[J.OBJECTS][i][J.HIDDEN])) {
									return parent[J.OBJECTS][i];
								}
							}
						}
					} else {
						if (parent.hasOwnProperty(J.FOLDERS)) {
							for (let i = idx + 1; i < parent[J.FOLDERS].length; i++) {
								if (!(parent[J.FOLDERS][i].hasOwnProperty(J.HIDDEN) && parent[J.OBJECTS][i][J.HIDDEN])) {
									return parent[J.FOLDERS][i];
								}
							}
						}
					}
				}
				
				return null;
			},
		
		// Returns the previous folder
		
		_getPreviousFolder = function(f) {
				let folder = f || currentFolder,
					parent = getParent(folder),
					idx = _getFolderIndex(parent, folder[J.PATH]);
				
				if (idx >= 0) {
					if (settings.loadDeep) {
						if (parent.hasOwnProperty(J.OBJECTS)) {
							for (let i = idx - 1; i >= 0; i--) {
								if (parent[J.OBJECTS][i][J.CATEGORY] === 'folder' &&
									!(parent[J.OBJECTS][i].hasOwnProperty(J.HIDDEN) && parent[J.OBJECTS][i][J.HIDDEN])) {
									return parent[J.OBJECTS][i];
								}
							}
						}
					} else {
						if (parent.hasOwnProperty(J.FOLDERS)) {
							for (let i = idx - 1; i >= 0; i--) {
								if (!(parent[J.FOLDERS][i].hasOwnProperty(J.HIDDEN) && parent[J.OBJECTS][i][J.HIDDEN])) {
									return parent[J.FOLDERS][i];
								}
							}
						}
					}
				}
				
				return null;
			},
				
		// Get next folder's first image
		
		getNextFoldersFirstImage = function(readyFn) {
				let folder = _getNextFolder(currentFolder),
					
					_getFirstImage = function(folder) {
							if (folder.hasOwnProperty(J.OBJECTS)) {
								folder[J.OBJECTS].forEach(o => {
										if (isLightboxable(o)) {
											return o;
										}
									});
							}
						};
				
				if (folder) {
					
					if (folder.hasOwnProperty(J.READY) && folder.ready) {
						readyFn(_getFirstImage(folder));
					} else {
						onFolderReady(folder, function(folder, args) {
								args[0](_getFirstImage(folder));
							}, [ readyFn ]);
					}
				}
			},
		
		// Get previous folder's last image
		
		getPreviousFoldersLastImage = function(readyFn) {
				let folder = _getPreviousFolder(currentFolder),
					
					_getLastImage = function(folder) {
							if (folder.hasOwnProperty(J.OBJECTS)) {
								for (let o = folder[J.OBJECTS], i = o.length - 1; i >= 0; i--) {
									if (isLightboxable(o[i])) {
										return o[i];
									}
								}
							}
						};
			
				
				if (folder) {
					
					if (folder.hasOwnProperty(J.READY) && folder.ready) {
						readyFn(_getLastImage(folder));
					} else {
						onFolderReady(folder, function(folder, args) {
								args[0](_getLastImage(folder));
							}, [ readyFn ]);
					}
				}
				
				return null;
			},
			
		/***************************************************************
		 *
		 *	Sort items: images, folders
		 *		sortBy:			sort criteria ('original' | 'random' | 'date' | 'name' | 'fileSize')
		 *		reference:		if sort by date ('added' | 'fileModified' | 'dateTaken')
		 *		reverse:		false = normal, true = reversed
		 *		foldersFirst:	put folders first
		 */
		
		sortItems = function(images, folders, opt) {
			
				if (!Array.isArray(images)) {
					// Images array is mandatory
					return null;
				}
				
				if (!Array.isArray(folders)) {
					// No folders provided
					opt = folders || {};
					folders = null;
				}
				
				let options = $.extend({
							sortBy:				'original',			// No change
							reference:			J.DATETAKEN,		// Taken date
							reverse:			false,				// Not reversed
							foldersFirst:		true				// Put folders first
						}, opt);
				
				// Ordering items
				switch (options.sortBy) {
					
					case 'random':
						
						// Random
						
						images.sort(() => (0.5 - Math.random()));
						
						if (folders) {
							folders.sort(() => (0.5 - Math.random()));
						}
						
						break;
					
					case 'date': 
						
						// Date
						
						let ref = options['reference'];
						
						if (options.reverse) {
							
							images.sort((a, b) => (
									(a.hasOwnProperty(J.DATES)? a[J.DATES][ref] : a[J.FILEDATE]) - 
									(b.hasOwnProperty(J.DATES)? b[J.DATES][ref] : b[J.FILEDATE])
								));
							
							if (folders) {
								folders.sort((a, b) => (a[J.FILEDATE] - b[J.FILEDATE]));
							}
							
						} else {
							
							images.sort((a, b) => (
									(b.hasOwnProperty(J.DATES)? b[J.DATES][ref] : b[J.FILEDATE]) - 
									(a.hasOwnProperty(J.DATES)? a[J.DATES][ref] : a[J.FILEDATE])
								));
							
							if (folders) {
								folders.sort((a, b) => (b[J.FILEDATE] - a[J.FILEDATE]));
							}
						}
						
						break;
						
					case J.NAME:
						
						// File name
						
						if (options.reverse) {
							
							images.sort((a, b) => ('' + a[J.NAME]).localeCompare('' + b[J.NAME]));
							
							if (folders) {
								folders.sort((a, b) => ('' + a[J.NAME]).localeCompare('' + b[J.NAME])); 
							}
							
						} else {
							
							images.sort((a, b) => ('' + b[J.NAME]).localeCompare('' + a[J.NAME]));
							
							if (folders) {
								folders.sort((a, b) => ('' + b[J.NAME]).localeCompare('' + a[J.NAME]));
							}
							
						}
						
						break;
						
					case J.FILESIZE:
						
						// File size: only images
						
						if (options.reverse) {
							images.sort((a, b) => (a[J.FILESIZE] - b[J.FILESIZE]));
						} else {
							images.sort((a, b) => (b[J.FILESIZE] - a[J.FILESIZE]));
						}
						
						break;
						
					default:
						
						// Descending?
						if (options.reverse) {
							
							images.reverse();
							
							if (folders) {
								folders.reverse();
							}
						}
						
						break;
						
				}
				
				// Concatenating folders and images arrays
				
				if (folders) {
					if (options.foldersFirst) {
						return folders.concat(images);
					} else {
						return images.concat(folders);
					}
				}
				
				return images;
			
			},

		/***************************************************************
		 *
		 *	collectByPath: Collect items by their paths within the album structure
		 *		paths:			array of root paths
		 *		sortBy:			sort criteria ('original' | 'random' | 'date' | 'name' | 'fileSize')
		 *		reference:		if sort by date ('added' | 'fileModified' | 'dateTaken')
		 *		reverse:		false = normal, true = reversed
		 *		ready:			the function to call after items ready
		 */
		
		collectByPath = function(opt) {
				
				//log('collectByPath(' + JSON.stringify(opt) + ')');
				if (typeof opt === UNDEF || !opt.hasOwnProperty('paths') || !Array.isArray(opt.paths) || typeof opt['ready'] !== FUNCTION) {
					return [];
				}
				
				let	items 			= [],
					options 		= $.extend({
												folder:			'',				// Root folder
												levels:			0,				// Current folder only
												sortBy:			'original',		// No sort
												reference:		'dateTaken',	// Reference
												reverse:		false			// Ascending
											}, opt),
					max				= options.paths.length,
					counter			= 0,
					fto,
					folder			= getFolder(options.folder),
					
					_skipEmpty = function(items) {
							let arr = [];
							
							for (let i of items) {
								i && arr.push(i);
							}
							
							return arr;
						},
					
					_finished = function() {
						
							clearTimeout(fto);
							
							if (counter < max) {
								log('Error: Timeout collecting items. Image set is incomplete (' + counter + '/' + max + ')!');
								items = _skipEmpty(items)
							}
							
							items = sortItems(items, {
									sortBy:			options['sortBy'],
									reference:		options['reference'],
									reverse:		options['reverse']
								});
							
							options.ready.call(items, options);
						};
				
				// Allowing 25ms / item to collect, so it won't run infinitely if some items missing
				fto = setTimeout(_finished, max * 25);
				
				for (let n = 0; n < max; n++) {
					
					getItem(options.paths[n], function(i) {
							// Non-linear
							if (this && typeof i !== UNDEF && this !== window) {
								items[i] = this;
								if (++counter === max) {
									// All items got collected
									clearTimeout(fto);
									_finished();
								}
							}
						}, n);
				}
				
				
			},
			
		/***************************************************************
		 *
		 *	collectNItem: Collect items from folder
		 *		folder:			start folder
		 *		levels:			depth below the start folder
		 *		max: 			maximum number
		 *		arRange:		[ min, max ] - aspect ratio range
		 *		sortBy:			sort criteria ('original' | 'random' | 'date' | 'name' | 'fileSize')
		 *		reference:		if sort by date ('added' | 'fileModified' | 'dateTaken')
		 *		sortOrder:		1: ascending 0: descending
		 *		quick:			true: stops when enough element has been gathered 
		 *						false: loads all folders before it selects the "max" elements based on sort
		 */
		
		collectNItem = function(opt) {
				
				//log('collectNItem(' + JSON.stringify(opt) + ')');
				if (typeof opt === UNDEF || typeof opt['ready'] !== FUNCTION) {
					return;
				}
				
				let options 		= $.extend({
											folder:			'',				// Root folder
											levels:			0,				// Current folder only
											include:		'images',		// Items to include
											max:			0,				// No limit
											sortBy:			'original',		// No sort			
											sortOrder:		0				// Ascending
										}, opt),
					items 			= [],
					folders			= [],
					folder			= getFolder(options.folder),
					foldersToLoad 	= options.levels? getFolderCount(folder, options.levels) : 1,
					foldersLoaded	= 0,
					completed		= false,
					needsDeepData	= false, //options.levels > 0 && folder.hasOwnProperty(J.FOLDERS),
					useImages		= options.include.indexOf('images') !== -1,
					useFolders		= options.include.indexOf('folders') !== -1,
				
					_addItems = function(folder) {
							// Adding images in a folder
							folder[J.OBJECTS].forEach(o => {
									if (o[J.CATEGORY] === 'image' && 
											(
												!options['arRange'] || 
												(o[J.IMAGE][J.WIDTH] / o[J.IMAGE][J.HEIGHT] >= options.arRange[0] &&
												o[J.IMAGE][J.WIDTH] / o[J.IMAGE][J.HEIGHT] <= options.arRange[1])
											)
										) {
										items.push(o);
									}
								});
						},
					
					_addFolder = function(folder) {
						
							if (useFolders) {
								folders.push(folder);
							}
							
							if (useImages) {
								// Adding objects of one folder
								if (folder.hasOwnProperty(J.OBJECTS)) {
									// Already loaded
									_addItems(folder);
								} else if (!folder[J.READY]) {
									// Wait to be loaded
									onFolderReady(folder, _addItems);
									//loadFolder(folder);
								}
							}
							
							if (folder[J.LEVEL] <= maxLevel) {
								// recursive to subfolders
								if (isDeep()) {
									if (folder.hasOwnProperty(J.OBJECTS)) {
										folder[J.OBJECTS].forEach(o => {
												if (o[J.CATEGORY] === 'folder' && !(o.hasOwnProperty(J.HIDDEN) && o[J.HIDDEN])) {
													_addFolder(o);
												}
											});
									}
								} else if (folder.hasOwnProperty(J.FOLDERS)) {
									folder[J.FOLDERS].forEach(o => {
											if (!(o.hasOwnProperty(J.HIDDEN) && o[J.HIDDEN])) {
												_addFolder(o);
											}
										});
								}
							}
							
							foldersLoaded++;
						},
						
					// Using deep-data	
					_readyDeep = function() {
						
							_addFolder(folder);
							
							setTimeout(_finished, 20);
						},
						
					// Loading data1.json files one-by-one
					_loadAll = function() {
						
							// Deep data already loaded or data1.json is enough
							_addFolder(folder);
							
							// Allow some time to add the first defer
							setTimeout(function() {
									if (defer.length) {
										$.when.apply($, defer).done(_finished);
									} else {
										_finished();
									}
								}, 20);						
						},
						
					_finished = function() {
						
							if (completed) {
								return;
							}
							
							if (options.max && options.quick && (items.length + folders.length) >= options.max ||  
								!defer.length || (foldersLoaded >= foldersToLoad)) { 
							
								items = sortItems(items, folders, {
										sortBy:			(options['sortOrder'] === -1)? 'random' : (options['sortBy'] || 'original'),
										reference:		options['reference'] || J.FILEDATE,
										reverse:		options['sortOrder'] === 0,
										foldersFirst:	options['include'].startsWith('folders')
									});
									
								// Chopping the extra items
								if (options.max && options.max < items.length) {
									items = items.slice(0, options.max);
								}
								
								completed = true;
								options.ready.call(items, options);
								
							} else {
								setTimeout(_finished, 50);
								return;
							}
						};
						
				// Starting a new promise collect
				defer = [];
				completed = false;
				
				if (!options.hasOwnProperty('quick')) {
					options.quick = options.max && options.sortBy !== 'original';
				}
				
				maxLevel = folder[J.LEVEL] + options.levels;
				
				random = options.sortOrder === -1;
				if (random) {
					settings.sortBy = 'original';
				}
					
				if (needsDeepData && !deepReady) {
					// Loading deep data, falling back to recursive data1.json on error
					loadDeep(_readyDeep, _loadAll);
					
				} else {
					// No deep data needed, loading current and its subtree (if needed)
					_loadAll();
				}
			},
			
		/***************************************************************
		 *
		 *	collectByDate: Collect items by date
		 *		depth:			where to collect ('tree' | 'current' | 'subfolders')
		 *		sortBy:			sort criteria ('original' | 'random' | 'date' | 'name' | 'fileSize')
		 *		reference:		if sort by date ('added' | 'fileModified' | 'dateTaken')
		 *		reverse:		false = normal, true = reversed
		 *		range:			time range
		 *							range = now - range ... now
		 *							range, start = start ... start + range
		 *							range, end = end - range ... end
		 *							Where start and end are days since 1900-01-01
		 *		Returns only lightboxable items
		 */
		
		collectByDate = function(opt) {
			
				//log('collectByDate(' + JSON.stringify(opt) + ')');
				if (typeof opt === UNDEF || typeof opt['ready'] !== FUNCTION) {
					return;
				}
				
				let options 		= $.extend({
											sort:			true,
											reverse:		false,
											reference:		J.DATETAKEN,
											depth: 			'current' 		// 'tree' | 'current' | 'subfolders'
										}, opt),
					items 			= [],
					start,
					end, 
					foldersToLoad 	= (options.depth === 'current')? 1 : getDeepFolderCount((options.depth === 'tree')? tree : currentFolder),
					foldersLoaded	= 0,
					completed		= false,
					needsDeepData	= options.depth === 'tree' && tree.hasOwnProperty(J.FOLDERS) ||
									  options.depth === 'subfolders' && currentFolder.hasOwnProperty(J.FOLDERS) && currentFolder[J.LEVEL] < 3,
				
					
					_findByDate = function(folder) {
							
							// Find images that fall into the date range
							
							if (!folder || !folder.hasOwnProperty(J.OBJECTS)) {
								return;
							}
							
							folder[J.OBJECTS].forEach(o => {
									if (isLightboxable(o) &&
										(d = o[J.DATES]) && 
										(d = d[options.reference]) && 
										(d >= start) && (d <= end)) {
										items.push(o);
									}
								});
														
							foldersLoaded++;
						},
					
					_addFolder = function(folder) {
							// Adding one folder
							
							if (!folder || (folder.hasOwnProperty(J.HIDDEN) && folder.hidden)) {
								return;
							}
							
							// Adding objects of one folder
							if (folder.hasOwnProperty(J.OBJECTS)) {
								// Already loaded
								_findByDate(folder);
							} else if (!folder[J.READY]) {
								// Wait to be loaded
								onFolderReady(folder, _findByDate);
								//loadFolder(folder);
							}
							
							if (isDeep()) {
								if (folder.hasOwnProperty(J.OBJECTS)) {
									folder[J.OBJECTS].forEach(o => {
											if (o[J.CATEGORY] === 'folder' && !(o.hasOwnProperty(J.HIDDEN) && o[J.HIDDEN])) {
												_addFolder(o);
											}
										});
								}
							} else if (folder.hasOwnProperty(J.FOLDERS)) {
								folder[J.FOLDERS].forEach(o => {
										if (!(o.hasOwnProperty(J.HIDDEN) && o[J.HIDDEN])) {
											_addFolder(o);
										}
									});
							}
						},
						
					_readyDeep = function() {
						
							_addFolder((options.depth === 'tree')? tree : currentFolder);
							
							setTimeout(_finished, 20);
						},
						
					_loadAll = function() {
						
							// Deep data already loaded or data1.json is enough
							_addFolder((options.depth === 'tree')? tree : currentFolder);
							
							// Allow some time to add the first defer
							setTimeout(function() {
									if (defer.length) {
										$.when.apply($, defer).done(_finished);
									} else {
										_finished();
									}
								}, 20);						
						},
						
					_finished = function() {
						
							if (completed) {
								return;
							}
							
							//console.log('_finished: toLoad=' + foldersToLoad + ' loaded=' + foldersLoaded);
							if (defer.length && foldersToLoad > foldersLoaded) {
								setTimeout(_finished, 20);
								return;
							}
							
							// Ordering items
							
							if (options.sort) {
								items = sortItems(items, {
									sortBy:			options['sortBy'] || 'date',
									reference:		options['reference'] || J.DATETAKEN,
									reverse:		options['reverse'] || false
								});
							}
							
							if (options.max && options.max < items.length) {
								items = items.slice(0, options.max);
							}
							
							completed = true;
							options.ready.call(items, options);
						};
				
				// Starting a new promise collect
				defer = [];
				completed = false;
				
				// options.start and options.end are days since 1970-01-01, range is number of days
				if (options.hasOwnProperty('end')) {
					end = options.end * ONEDAY_S;
				}
				
				if (options.hasOwnProperty('start')) {
					start = options.start * ONEDAY_S;
				}
				
				if (options.hasOwnProperty('range')) {
					if (start !== null) {
						end = start + options.range * ONEDAY_S;
					} else if (end !== null) {
						start = end - options.range * ONEDAY_S;
					} else {
						// Up to now
						end = Math.round(new Date() / 1000);
						start = end - options.range * ONEDAY_S;
					}
				}
				
				if (typeof start === UNDEF) {
					start = 0;
				}
				
				if (typeof end === UNDEF) {
					end = Math.round(new Date() / 1000);
				}
								
				if (needsDeepData && !deepReady) {
					
					// Loading deep data, falling back to recursive data1.json on error
					loadDeep(_readyDeep, _loadAll);
					
				} else {
					
					_loadAll();
				}
			},
		
		/***************************************************************
		 *
		 *	collectItems: Collecting search results
		 *
		 *		fields: 		fields to watch
		 *		types:			'all' or comma separated list of allowed types ('image' | 'audio' | 'video' | ...)
		 *		depth:			where to collect ('tree' | 'current' | 'subfolders')
		 *		exact:			exact search (or conjunctive)
		 *  	caseSensitive:	case sensitivity: (false by default)
		 *		max:			maximum number of results
		 */
		
		collectItems = function(opt) {
			
				//log('collectItems(' + JSON.stringify(opt) + ')');
				if (typeof opt === UNDEF || !opt.hasOwnProperty('terms') || typeof opt['ready'] !== FUNCTION) {
					return;
				}
				
				let options 		= $.extend({
												fields: 		'creator,keywords,name,title,comment,regions',
												types:			'all',
												depth: 			'current', 		// 'tree' | 'current' | 'subfolders'
												divisible:		'name,title,comment,photodata',
												exact: 			false,
												caseSensitive:	false	
											}, opt),
					items 			= [], 																	// Found items
					terms,																					// Terms to look for - used for undivisible fields
					termsArr,																				// Terms to look for (Array) - used for divisible fields
					fields 			= Array.isArray(options.fields)? options.fields : options.fields.split(/,\s?/), // fields to check
					fieldslength 	= fields.length,														// Fields length
					divisible		= new Array(fieldslength),												// Fields that can be divided
					exact			= false,																// Exact search: ""foo boo""
					conjunctive 	= false,																// Conjunctive search: "foo and boo"
					splitRegExp,																			// RegExp that works as word delimiter
					allTypes		= options.types === 'all',												// Every item type to check
					types 			= {},																	// Item types to check
					foldersToLoad 	= (options.depth === 'current')? 
											1 
											: 
											getDeepFolderCount((options.depth === 'tree')? tree : currentFolder),
					foldersLoaded	= 0,
					completed		= false,
					needsDeepData	= options.depth === 'tree' && tree.hasOwnProperty(J.FOLDERS) ||
									  options.depth === 'subfolders' && currentFolder.hasOwnProperty(J.FOLDERS) && currentFolder[J.LEVEL] < 3,
										
					_searchItem = function(o, cat) {
						
							let found = new Array(termsArr.length).fill(false),
								
								foundAny = () => found.some(el => !!el),
								
								foundAll = () => found.every(el => !!el),
									
								//hasFound = () => conjunctive? allTrue() : anyTrue(),
								
								// Finds 
								find = function(term, s, div) {
									
										if (!options.caseSensitive) {
											s = s.toLowerCase();
										}
										
										if (div) {
											// Divisible field
											s = s.replace(splitRegExp, ' ');
											return (' ' + s).includes(' ' + term);
										}
										
										if (exact) {
											// Exact search
											return term === s;
										}	
										
										// Not exact but must start with the search string
										return s.startsWith(term);
									},
									
								// Search for a single term
								searchTerm = function(term, s, div) {
									
										if (!Array.isArray(s)) {
											// The value is single
											if (exact) {
												s = s.toLowerCase();
												if (div) {
													return s.slice(0, term.length) === term || s.indexOf(' ' + term) !== -1;
												}
												return s === term;
											}
											
											return find(term, s, div);
										}
										
										// The value is an Array
										let m = false;
										
										for (let i = 0; i < s.length; i++) {
											if (find(term, s[i], div)) {
												m = true;
											}
										}
										
										return m;
									},
								
								// Search for terms: single or multiple terms 
								searchTerms = function(terms, s, div) {
										let m = false;
									
										if (exact && s.includes(';')) {
											// Treat like an array
											s = s.split(';');
										}
										
										if (Array.isArray(terms)) {
											// Multi term search
											for (let i = 0; i < terms.length; i++) {
												if (searchTerm(terms[i], s, div)) {
													m = found[i] = true;
													if (!conjunctive) {
														return true;
													}
												}
											}
											//log(o[J.NAME] + ' searchTerms("' + terms + '", "' + s + '", ' + exact + ') ==> ' + m);
										} else {
											// One term search
											if (searchTerm(terms, s, div)) {
												m = found[0] = true;
											}
										}
										
										return m; 
									};
								
							for (let i = 0, f, p; i < fieldslength; i++) {
								
								// Category specific field?
								if (fields[i].length > 1) {
									// e.g. "folder:title"
									if (fields[i][0] !== cat) {
										// Skip this type
										continue;
									}
									f = fields[i][1];
								} else {
									f = fields[i][0];
								}
								
								if (JCAMERAFIELDS.indexOf(f) >= 0 && o.hasOwnProperty(J.CAMERA)) {
									// Camera data
									p = o[J.CAMERA][f];
									if (typeof p === UNDEF) {
										// not found: retry on plain attribute, e.g. "aperture"
										p = o[f];
									}
									
								} else if (f === J.NAME) {
									// File name
									if (o.hasOwnProperty(J.ORIGINAL)) {
										// Has original: use that's file name
										p = decodeURIComponent(o[J.ORIGINAL][J.PATH].getFile());
									} else {
										// Slide image name
										p = o[J.NAME];
									}
									// Searching for the whole name and parts broken by word boundaries 
									p = p + ' ' + p.replace(/[\.\-_]/g, ' ');
									
								} else if (f === J.REGIONS) {
									// Regions
									if (o.hasOwnProperty(J.REGIONS)) {
										// Get names
										p = [];
										JSON.parse(o[J.REGIONS]).forEach(a => p.push(a.split(';')[0] + ''));
										p = p.filter(Boolean);
									} else {
										p = null;
									}
								} else {
									// All other fields
									p = o[f];
								}
									
								// Has such a property?
								if (typeof p !== UNDEF && p !== null && p.length) {
									if (Array.isArray(p) && !divisible[i]) {
										// The property is an Array and undivisible
										for (let j = 0; j < p.length; j++) {
											if (searchTerms((conjunctive || !exact)? termsArr : terms, p[j], divisible[i])) {
												if (!conjunctive) {
													// Not conjunctive: return on the first found
													return true;
												}
											}
										}
										
									} else {
										// Not exact or not array
										if (f === J.COMMENT || f.endsWith('Caption')) {
											// Strip HTML on fields might contain it
											p = p.stripHTML();
										} else if (Array.isArray(p)) {
											// Array
											p = p.join(' ');
										} else {
											// Otherwise ensure it's a string
											p = p + '';
										}
										
										if (searchTerms((conjunctive || !exact)? termsArr : terms, p, divisible[i])) {
											if (!conjunctive) {
												// Found one: don't care about other fields
												return true;
											}
										}
									}
								}
							}
							
							// all terms found (conjunctive) | any term found otherwise
							return conjunctive? foundAll() : foundAny();
						},
						
					_searchFolder = function(folder) {
							/*
							if (typeof DEBUG !== UNDEF && DEBUG) {
								log('Searching folder "' + folder[J.NAME] + '" ' + (folder[J.OBJECTS]? folder[J.OBJECTS].length : 0) + ' items');
							}
							*/
							if (!folder) {
								return;
							}
							if (folder.hasOwnProperty(J.OBJECTS)) {
								// Objects
								folder[J.OBJECTS].forEach(o => {
										if (o.hasOwnProperty(J.FOLDERINDEX)) {
											// it's just a reference to Folders[] (when using data1.json)
											if (allTypes || types['folder']) {
												if (_searchItem(folder[J.FOLDERS][o[J.FOLDERINDEX]], 'folder')) {
													items.push(o);
												}
											}
										} else {
											// Normal tree
											if (o.hasOwnProperty(J.CATEGORY) && (allTypes || types[o[J.CATEGORY]])) {
												if (_searchItem(o, o[J.CATEGORY])) {
													items.push(o);
												}
											}
										}
									});
							}
							
							foldersLoaded++;
						},
				
					_addFolder = function(folder) {
							// Adding one folder
							
							if (!folder || (folder.hasOwnProperty(J.HIDDEN) && folder.hidden)) {
								return;
							}
							
							// Adding objects of one folder
							if (folder.hasOwnProperty(J.OBJECTS)) {
								// Already loaded
								_searchFolder(folder);
							} else if (!folder[J.READY]) {
								// Wait to be loaded
								onFolderReady(folder, _searchFolder);
								//loadFolder(folder);
							}
							
							if (isDeep()) {
								if (folder.hasOwnProperty(J.OBJECTS)) {
									folder[J.OBJECTS].forEach(o => {
											if (o[J.CATEGORY] === 'folder' && !(o.hasOwnProperty(J.HIDDEN) && o[J.HIDDEN])) {
												_addFolder(o);
											}
										});
								}
							} else if (options.depth !== 'current' && folder.hasOwnProperty(J.FOLDERS)) {
								folder[J.FOLDERS].forEach(o => {
										if (!(o.hasOwnProperty(J.HIDDEN) && o[J.HIDDEN])) {
											_addFolder(o);
										}
									});
							}
						},
						
					_readyDeep = function() {
						
							_addFolder((options.depth === 'tree')? tree : currentFolder);
							setTimeout(_finished, 20);
						},
						
					_loadAll = function() {
						
							// Deep data already loaded or data1.json is enough
							_addFolder((options.depth === 'tree')? tree : currentFolder);
							
							// Allow some time to add the first defer
							setTimeout(function() {
									if (defer.length) {
										$.when.apply($, defer).done(_finished);
									} else {
										_finished();
									}
								}, 20);						
						},
						
					_finished = function() {
							
							if (completed) {
								return;
							}
							
							//console.log('_finished: toLoad=' + foldersToLoad + ' loaded=' + foldersLoaded);
							if (defer.length && foldersToLoad > foldersLoaded) {
								// Premature call for finished
								setTimeout(_finished, 20);
								return;
							}
							
							if (options.max && options.max < items.length) {
								items = items.slice(0, options.max);
							}
							
							completed = true;
							options.ready.call(items, options);
						};
				
				// Starting a new promise collect
				defer = [];
				completed = false;
				terms = options.terms.trim(); 
				if (!options.caseSensitive) {
					terms = terms.toLowerCase();
				}
				
				if (options.terms[0] === '"' && options.terms[options.terms.length - 1] === '"') {
					// Exact search with quotes: "something exact"
					terms = terms.slice(1, -1).trim();
					termsArr = [ terms ];
					exact = !opt.hasOwnProperty('exact')? true : opt.exact;
					
				} else {
					// Any word or conjunctive
					let lct = terms.toLowerCase();
						
					if (andRegExp.test(lct)) {
						// Conjunctive search
						termsArr = terms.split(andRegExp);
						terms = terms.replace(andRegExp, ' ');
						conjunctive = true;
					} else {
						// Normal search
						termsArr = terms.split(' ');
						conjunctive = false;	
					}
					// Removing empty elements
					termsArr = termsArr.filter(Boolean);
					splitRegExp = new RegExp('[,\\\|\\\:' + ((terms.includes('(') || terms.includes(')'))? '' : '\\\(\\\)') + (terms.includes('-')? '' : '\\\-') + (terms.includes('.')? '' : '\\\.') + (terms.includes('_')? '' : '_') + ']', 'g');
					exact = opt.hasOwnProperty('exact')? opt.exact : false;
				}
				
				// Determining divisible fields
				for (let i = 0, f, d = options.divisible.split(','); i < fieldslength; i++) {
					fields[i] = fields[i].split(':');
					divisible[i] = d.indexOf(fields[i][1] || fields[i][0]) >= 0;
				}
				
				if (!allTypes) {
					if (options.types.charAt(0) === '-') {
						// Negative
						settings.possibleTypes.forEach(t => {
								if (options.types.indexOf(t) === -1) {
									types[t] = true;
								}
							});
					} else {
						// Positive
						options.types.split(/,\s?/).forEach(t => { types[t] = true; });
					}
				}
				
				if (needsDeepData) {
					
					if (!deepReady) {
						// Loading deep data, falling back to recursive data1.json on error
						loadDeep(_readyDeep, _loadAll);
					} else {
						_readyDeep();
					}
					
				} else {
					
					_loadAll();
				}
									
			},
		
		/***************************************************************
		 *
		 *	collectTags:	Collecting tags for Tag cloud
		 *		fields: 		fields to watch
		 *		types:			'all' or comma separated list of allowed types ('image' | 'audio' | 'video' | ...)
		 *		depth:			where to collect ('tree' | 'current' | 'subfolders')
		 *		divisible:		those fields that can be splitted around spaces or dashes
		 */
		
		collectTags = function(opt) {
			
				//log('collectTags(' + JSON.stringify(opt) + ')');
				if (typeof opt === UNDEF || typeof opt['ready'] !== FUNCTION) {
					return;
				}
					
				let options 		= $.extend({
											fields: 		'creator,keywords,folder:title,webLocation:title,regions',
											types:			'all',	
											depth: 			'current', 			// 'tree' | 'current' | 'subfolders'
											minLength:		1,
											divisible:		'comment,title'
										}, opt),
					tags 			= [], 
					fields 			= Array.isArray(options.fields)? options.fields : options.fields.split(/,\s?/), 
					fieldslength 	= fields.length,
					divisible		= new Array(fieldslength),
					allTypes 		= options.types === 'all',
					types			= {},
					//exact			= {},
					
					// Add tags collected from an item
					// tags = [ 'tag', paths[], 'TAG' ]
					
					_addTags = function(o, newTags) {
							let nt = newTags.split('^').filter(t => t.length >= options.minLength)
							
							for (let i = 0, idx, tag, p; i < nt.length; i++) {
								
								tag = nt[i].toUpperCase();
								p = getObjectPath(o);
								
								if (p !== null) {
									if (!tags || !tags.length) {
										// Empty
										tags = [[ nt[i], [ p ], tag ]];
									} else { 
										// Does this tag exist?
										if ((idx = tags.findIndex(t => t[2] === tag)) >= 0) {
											// Yes: add path
											if (tags[idx][1].indexOf(p) === -1) {
												// This items has been added already?
												tags[idx][1].push(p);
											}
										} else {
											// No: add new
											tags.push([ nt[i], [ p ], tag ]);
										}
									}
								}
							}
						},
					
					// Collects tags from an item
					
					_collectTags = function(o, cat) {
							let ctags = '^',			// Collected tags = 'tag1^tag2^tag3'
								ctagsuc = '^',			// Same in uppercase for comparison
							
								add = function(tag, field, divisible) {
										
										if (!tag) {
											return;
										}
										
										let t, 
											ta;
											
										if (divisible) {
											if (field === 'comment' || field.endsWith('Caption')) {
												tag = tag.stripHTML();
											}
											//ta = tag.split(/\W+/);
											ta = tag.split(/[\s,_\.\?\!\-\(\)\[\]]/);
											ta = removeEmpty(ta);
										} else {
											ta = tag.toString().split(';');
										}
										
										for (let i = 0, l = ta.length, fnd = false; i < l; i++) {
										
											t = ta[i].trim();
											
											if (t.length < options.minLength ||
												t.length === 1 && '!@#$%^&*()-_=+[{]};:\'",<.>/?\\|'.indexOf(t.charCodeAt(0))) {
												// Empty, too short, or special char
												continue;
											}
											
											if (ctagsuc.indexOf('^' + t.toUpperCase() + '^') === -1) {
												ctags += t + '^';
												ctagsuc += t.toUpperCase() + '^';
											}
										}
									};
							
							for (let i = 0, f, p; i < fieldslength; i++) {
								
								if (fields[i].length > 1) {
									if (fields[i][0] !== cat) {
										continue;
									}
									f = fields[i][1];
								} else {
									f = fields[i][0];
								}
								
								
								if (JCAMERAFIELDS.indexOf(f) >= 0 && o.hasOwnProperty(J.CAMERA)) {
									// camera data
									p = o[J.CAMERA][f];
									if (typeof p === UNDEF) {
										p = o[f];
									}
								} else if (f === J.REGIONS) {
									// Regions
									p = o.hasOwnProperty(J.REGIONS)? JSON.parse(o[J.REGIONS]) : null;
									
								} else {
									p = o[f];
								}
									
								if (typeof p !== UNDEF && p != null) {
									//log(o['name'] + '[' + f + '] = ' + o[f] + ' (' + (Array.isArray(o[f])? 'array':(typeof o[f])) + ')');
									if (f === J.REGIONS) {
										// Checking for the names only
										for (let j = 0; j < p.length; j++) {
											add(p[j].split(';')[0], f, divisible[i]);
										}
									} else if (Array.isArray(p)) {
										// Array type field
										for (let j = 0; j < p.length; j++) {
											add(p[j], f, divisible[i]);
										}
									} else {
										add(p, f, divisible[i]);
									}
								}
							}
							
							//log(ctags);
							if (ctags.length > 1) {
								_addTags(o, ctags);
							}
						},
					
					// Collect tags from all objects in a folder
					
					_addItems = function(folder) {
					
							// Adds fields from objects array
							if (!folder) {
								return;
							}
							
							if (folder !== tree && (allTypes || types['folder'])) {
								// Current folder
								_collectTags(folder, 'folder');
							}
							
							if (folder.hasOwnProperty(J.OBJECTS)) {
								// Ordinary objects
								for (let i = 0, o = folder[J.OBJECTS], cat; i < o.length; i++) {
									if (o[i].hasOwnProperty(J.CATEGORY)) {
										cat = o[i][J.CATEGORY];
										if (allTypes || types[cat]) {
											_collectTags(o[i], cat);
										}
									}
								}
							}
						},
				
					// Queues one folder to collect tags  
					
					_addFolder = function(folder) {
							// Adding one folder
							
							if (!folder || (folder.hasOwnProperty(J.HIDDEN) && folder.hidden)) {
								return;
							}
							
							onFolderReady(folder, _addItems);
							
							if (isDeep()) {
								if (folder.hasOwnProperty(J.OBJECTS)) {
									folder[J.OBJECTS].forEach(o => {
											if (o[J.CATEGORY] === 'folder' && !(o.hasOwnProperty(J.HIDDEN) && o[J.HIDDEN])) {
												_addFolder(o);
											}
										});
								}
							} else if (options.depth !== 'current' && folder.hasOwnProperty(J.FOLDERS)) {
								folder[J.FOLDERS].forEach(o => {
										if (!(o.hasOwnProperty(J.HIDDEN) && o[J.HIDDEN])) {
											_addFolder(o);
										}
									});
							}
						},
						
					// Arrange the tags when ready
					
					_finally = function() {
							// Sort
							if (options.sort === 'name') {
								tags.sort((a, b) => ('' + a[2]).localeCompare('' + b[2]));	
							} else if (options.sort === 'frequency') {
								tags.sort((a, b) => (b[1] - a[1]));
							}
							// Max.
							if (options.max && options.max < tags.length) {
								tags = tags.slice(0, options.max);
							}
							// Done functions
							options.ready.call(tags, options);
						};
				
				// Starting a new promise collect
				
				defer = [];
				
				// Determining exact/divisible fields
				
				for (let i = 0, f, d = options.divisible.split(','); i < fieldslength; i++) {
					fields[i] = fields[i].split(':');
					divisible[i] = d.indexOf(fields[i][1] || fields[i][0]) !== -1;
				}
				
				// Creating object types array too look for
				
				if (!allTypes) {
					if (options.types.charAt(0) === '-') {
						// Negative
						settings.possibleTypes.forEach(t => {
								if (options.types.indexOf(t) === -1) {
									types[t] = true;
								}
							});
					} else {
						// Positive
						options.types.split(/,\s?/).forEach(t => { types[t] = true; });
					}
				}
				
				// Adding folder(s)
				
				_addFolder((options.depth === 'tree')? tree : currentFolder);
				
				if (defer.length) {
					$.when.apply($, defer).done(_finally);
				} else {
					_finally();
				}
						
			},
		
		/*
		 * Getting one key from an object in a processed form, e.g. fileLabel variables are handled
		 */
		
		getKey = function(o, k) {
				
				if (typeof o !== UNDEF && typeof k === STRING) {
					if (k === 'label') {
						return getLabel(o);
					} else if (k === 'fileLabel') {
						return o.hasOwnProperty(J.TITLE)? o[J.TITLE] : getLabel(o);
					} else if (k === 'originalTime' &&
							o.hasOwnProperty(J.DATES) && 
							o[J.DATES].hasOwnProperty(J.DATETAKEN)
						) {
						return new Date(o[J.DATES][J.DATETAKEN] * 1000).toLocaleString(settings.locale, {month: '2-digit', day: '2-digit', hour12: false, hour: '2-digit', minute:'2-digit'});
					} else if (o.hasOwnProperty(k)) {
						return o[k];
					}
				}
				
				return ''; 
			},
			
		/***************************************************************
		 *
		 *	processTemplate:	Processing template for an object
		 *		template:		string with variables in format ${name} or in ${var1|var2} fallback format
		 *		co:				current object
		 *		removeEmpty:	removing empty tags around missing/empty variables
		 */
		
		processTemplate = function(template, co, removeEmpty) {
			
				let remove = (typeof removeEmpty !== UNDEF)? removeEmpty : false,
					o = co || currentFolder,
					i0,
					i1,
					m,
					v;
				
				if (template && template.indexOf('${') > 0) {
				
					while (m = template.match(/\$\{([\w\.|]+)\}/)) {
						if (m[1].indexOf('|') > 0) {
							// ${var1|var2} fallback format
							for (let i = 0, k = m[1].split('|'); i < k.length; i++) {
								if (v = getKey(k[i])) {
									// Found
									break; 
								}
							}
						} else {
							// Single variable
							v = getKey(o, m[1]);
						}
						
						if (v === null && remove) {
							// Remove empty HTML tags
							i0 = m.index - 1;
							i1 = i0 + m[0].length;
							
							if (i0 > 0 && template[i0] === '>' && i1 < (sb.length - 1) && template[i1] === '<') {
								
								i0 = template.lastIndexOf('<', i0);
								i1 = template.indexOf('>', i1);
								
								if (i0 >= 0 && i1 >= 0) {
									template = template.slice(0, i0) + template.slice(i1);
									continue;
								}
							}
						}
						// Replacing or removing variable
						template = template.slice(0, m.index) + (v || '') + template.slice(m.index + m[0].length);
					}
				}
		
				return template;
			},
									
		
		/***************************************************************
		 *
		 * 				Initial processing of the album
		 *
		 */
		
		_up = '../../../../../../../../../../../../../../../../../../../../',
		
		relativePath = function(from, to) {
				if (typeof from === UNDEF || !from.length || from === '/') {
					// From root
					return to || '';
				} else if (typeof to === UNDEF || !to.length || to === '/') {
					// To root
					return _up.slice(0, from.split('/').length * 3);
				} else if (from === to) {
					// Same
					return '';
				}
				
				from = from.split('/');
				to = to.split('/');
				
				while (from.length && to.length && from[0] === to[0]) {
					from.shift();
					to.shift();
				}
				
				return _up.slice(0, from.length * 3) + to.join('/');
			},
			
		/*
		 *	Adding level, parent pointers and relative paths for the whole tree
		 */
		 
		addExtras = function() {
				let addFolder = function(f, level, path, parentRef) {
						let fp = level? makePath(path, f[J.PATH]) : '',		// folder path: subfolder/subfolder/
							pr = level? getPathRef(fp) : 0,					// path reference number
							rp = (albumPath === null)? getRelPathRef(relativePath(settings.relPath, fp)) : -1;	// Relative path
						
						// Enhancing folders with extra variables
						
						// Level
						f[J.LEVEL] = level;
						
						// Folder path reference from root
						f[J.PATHREF] = pr;
						
						// Relative path from currentFolder to this folder
						if (albumPath === null) {
							// if we're inside the album (providing simpler paths)
							f[J.RELPATH] = rp;
						}
						
						// Check for missing Category
						if (!f.hasOwnProperty(J.CATEGORY)) {
							f[J.CATEGORY] = 'folder';
						}
						
						// Parent reference
						if (level) {
							f[J.PARENTREF] = parentRef;
						}
						
						// Fixing tree.json anomaly:
						// thumb path is different in deep-data.json and data1.json
						if (f[J.THUMB][J.PATH].startsWith(f[J.PATH] + '/' + settings.thumbsDir)) {
							f[J.THUMB][J.PATH] = f[J.THUMB][J.PATH].slice(f[J.PATH].length + 1);
						}
						
						// Recursive to subfolders
						if (settings.loadDeep) {
							// Adding path references in case of deep-data.json
							f[J.OBJECTS].filter(o => o[J.CATEGORY] !== 'folder').forEach((o) => {
									o[J.PATHREF] = pr;
									if (albumPath === null) {
										o[J.RELPATH] = rp;
									}
								});
							
							// Collecting folders' indexes into FOLDERS array for easier access later 
							let idx = 0;
							f[J.OBJECTS].filter(o => o[J.CATEGORY] === 'folder').forEach((o, i) => {
									if (!idx) {
										f[J.FOLDERS] = [];
									}	
									f[J.FOLDERS].push(i++);
									// Recursive call for subfolders
									addFolder(o, level + 1, fp, pr);
								});
						} else {
							// Iterating over folders
							if (f.hasOwnProperty(J.FOLDERS)) {
								f[J.FOLDERS].forEach(o => addFolder(o, level + 1, fp, pr));
							}
						}
					};
				
				addFolder(tree, 0, '', 0);
			},
			
		/*
		 *	callOnReady: Calling cached onready functions and removing them
		 */
		 
		callOnReady = function(folder) {
				// firing all cached onFolderReady functions
				//log('Firing ' + folder[J.ONREADY].length + ' cached onReady function(s) in folder "' + folder[J.NAME] + '"');
				if (folder.hasOwnProperty(J.ONREADY) && Array.isArray(folder[J.ONREADY])) {
					folder[J.ONREADY].forEach(rfn => {
							if (rfn.length > 1) {
								// With arguments
								rfn[0](folder, rfn[1]);
							} else {
								// No arguments
								rfn[0](folder);
							}
						});
					folder[J.ONREADY] = [];
				}
			},
			
		/*
		 *	addObjects: Adding Objects array to tree when using data1.json
		 *		executed after the folder's data file has loaded
		 *		d = JSON data loaded
		 *		folder = target folder
		 */				
		
		addObjects = function(folder, d) {
				let j = 0,			// Counting folders
					f;
					
				// Ensure it exists
				folder[J.OBJECTS] = [];
				
				//log('Adding objects in folder "' + (folder[J.NAME] || folder[J.PATH]) + '"');
					
				d[J.OBJECTS].forEach(o => {
						// Counting loaded object categories
						tree[J.LOADCOUNTER][o[J.CATEGORY]]++;
						tree[J.LOADCOUNTER][J.TOTAL]++;
						
						if (o[J.CATEGORY] === 'folder') {
							// Folder
							// Proper way:
							//o = { folderIndex: folder[J.FOLDERS].findIndex(f => o[J.PATH] === f[J.PATH]) };
							// Heuristic: (provided tree.json has the same folder ordering as data1.json)
							if (j < folder[J.FOLDERS].length) {
								
								f = folder[J.FOLDERS][j];
								
								// Copying missing folder variables but OBJECTS and ALBUM
								for (let prop in o) {
									if (prop !== J.OBJECTS && prop !== J.ALBUM && !f.hasOwnProperty(prop)) {
										f[prop] = o[prop];
									}
								}
								
								// J.FOLDERS array already holds the full info, storing only a pointer
								o = {};
								o[J.FOLDERINDEX] = j++;
								
							} else {
							
								log('Database error: Folder count inconsistency in folder "' + folder[J.NAME] + '". tree.json <> data1.json');
							}
							
						} else {
							// Not a folder
							// Adding absolute and relative paths
							o[J.PATHREF] = folder[J.PATHREF];
							o[J.RELPATH] = folder[J.RELPATH];
						}
						// Storing object in the tree
						folder[J.OBJECTS].push(o);
					});
				
				folder[J.READY] = true;
				
				callOnReady(folder);
			},
			
		/*
		 *	onFolderReady: Checks if folder data has been loaded 
		 *		if loaded executes the function immediately 
		 *		if not then saves the function for later execution 
		 *		fn - is the folder to be called on ready
		 *		params - the parameters array (optional)
		 */
		
		onFolderReady = function(folder, fn, params) {
			
				if (folder && typeof fn === FUNCTION) {
					
					if (deepReady || (folder.hasOwnProperty(J.READY) && folder[J.READY])) {
						// Already loaded: fire immediately
						if (typeof params !== UNDEF) {
							fn(folder, params);
						} else {
							fn(folder);
						}
					} else {
						// Need to cache
						//log('Caching onReady function for folder "' + folder[J.NAME] + '" ' + (folder.hasOwnProperty(J.ONREADY)? ('[' + (folder[J.ONREADY].length + 1) + ']') : '(onReady array is missing)'));
						if (!folder.hasOwnProperty(J.ONREADY)) {
							// Loading hasn't been initiated, let's load it!
							loadFolder(folder);
						}
						
						if (typeof params !== UNDEF) {
							folder[J.ONREADY].push([ fn, Array.isArray(params)? params : [ params ] ]);
						} else {
							folder[J.ONREADY].push([ fn ]);
						}
					}
				}
			},
			
		/*
		 *	loadFolder: Loading data for a single folder, optionally recursively
		 */
		
		loadFolder = function(folder, deep) {
			
				//log('LoadFolder("' + folder[J.NAME] + '")');
				
				if (!folder.hasOwnProperty(J.ONREADY)) {
					// Creating an empty array for onready functions, also signaling that the load has been initiated
					folder[J.ONREADY] = [];
				}
				
				if (!folder.hasOwnProperty(J.OBJECTS) || !folder[J.OBJECTS].length) {
					// we need to load data1.json
					let src = makePath(getPath(folder), settings.dataFile) + cacheBuster;
					
					// building defer array to be able to check the full load
					if (!defer) {
						defer = [];
					}
					//log('Loading objects for folder "' + folder[J.NAME] + '"');
					// Cache buster with ?makeDate
					defer.push($.getJSON(src)
							.done(function(d) {
								//log('Objects loaded for folder "' + folder[J.NAME] + '"');
								// Copying objects
								addObjects(folder, d);
								
							}).fail(function(jqxhr, status, error) {
								log('Error: Objects could not be loaded for "' + src + '": ' + status + ', ' + error);
							})
						);
					
					if (deep && folder.hasOwnProperty(J.FOLDERS)) {
						folder[J.FOLDERS].forEach(f => loadFolder(f, true));
					}
					
				} else {
					callOnReady(folder);
				}
			},
			
		/*
		 *	initCounters: Initializing tree-level counters
		 */
		
		initCounters = function() {
			
				if (tree) {
					// Initializing the load counters
					tree[J.LOADCOUNTER] = {};
					tree[J.LOADCOUNTER][J.TOTAL] = 0;
					
					settings.possibleTypes.forEach(t => {
							tree[J.LOADCOUNTER][t] = 0;
						});
				}
			},
			
		/*
		 *	setCurrentFolder: Sets currentFolder variable as a shortcut
		 */
		
		setCurrentFolder = function() {
				// Getting the pointer to the current folder
				currentFolder = getFolder(settings.relPath);
				
				if (currentFolder === null) {
					if (typeof settings.fatalError === FUNCTION) {
						settings.fatalError.call(this, 'noSuchFolder', settings.relPath);
					}
					
					return false;
				}
				
				return true;
			},
							
		/*
		 *	loadNeededFolders: executed when tree.json has loaded
		 */
		
		loadNeededFolders = function() {
					
				// Collecting promises for data1.json loads
				defer = [];
				
				// Loading current folder (+ deep folders?)
				loadFolder(settings.lazy? currentFolder : tree, !settings.lazy);
				
				// Waiting for all requested folders be loaded
				$.when.apply($, defer)
					.done(function() {
							let d = new Date();
							
							if (typeof DEBUG !== UNDEF && DEBUG) {
								log(defer.length + ' folder(s) loaded: ' + (d - instance) + 'ms');
							}
							
							ready = true;
							defer = null;
							
							if (typeof settings['ready'] === FUNCTION) {
								settings.ready.call(this);
								settings.ready = null;
							} else if (!tree.hasOwnProperty(J.FOLDERS)) {
								// Flat album: calling deep ready immediately
								if (typeof settings['deepReady'] === FUNCTION) {
									settings.deepReady.call(this);
									settings.deepReady = null;
								}
							}
						})
					
					.fail(function(jqxhr, status, error) {
							
							if (typeof settings['fatalError'] === FUNCTION) {
								settings.fatalError.call(this, 'cantLoadDataForFolder', currentFolder[J.PATH]);
							}
							
						});
			},
			
		/*
		 *	initFolderReady: initializing ready state and ready events in a recursive manner (in case tree.json was loaded)
		 */
		
		initFolderReady = function(folder) {
			
				// Sets READY property
				if (!folder.hasOwnProperty(J.READY)) {
					folder[J.READY] = false;
				}
				
				// Go deep recursively into subfolders
				if (settings.loadDeep) {
					if (folder.hasOwnProperty(J.OBJECTS)) {
						folder[J.OBJECTS].forEach(o => { if (o[J.CATEGORY] === 'folder') initFolderReady(o); });
					}
				} else if (folder.hasOwnProperty(J.FOLDERS)) {
					folder[J.FOLDERS].forEach(f => initFolderReady(f));
				}
			},
						
		/*
		 *	loadTree: Loading tree.json from the top level folder
		 */
		
		loadTree = function(doneFn) {
			
				let ins = new Date(),
					src = makePath(albumPath || settings.rootPath, settings.treeFile) + cacheBuster;
				
				// Set loadDeep to false whatever it was. Means tree.json is used, not deep-data.json
				settings.loadDeep = false;
				//log('loadTree("' + src + '")');
				
				// Loading tree.json
				return $.getJSON(src)
					.done(function(d) {
							// tree.json loaded
							
							tree = d;
							
							// Adding counters
							initCounters();
							
							// Initializing ready state and onready events in every folder
							initFolderReady(tree);
							
							if (typeof DEBUG !== UNDEF && DEBUG) {
								log('Tree loaded: ' + ((new Date()) - ins) + 'ms');
								ins = new Date();
							}
							
							if (setCurrentFolder()) {
							
								// Adding extra variables
								addExtras();
								
								// Calling "done" function
								if (typeof doneFn === FUNCTION) {
									doneFn.call(this);
								}
							}
							
						})
					
					.fail(function(jqxhr, status, error) {
								
							if (typeof settings['fatalError'] === FUNCTION) {
								settings.fatalError.call(this, 'localAccessBlocked', src);
							}
							
						});
			},
			
		/*
		 *	migrateTree: moves data from tree to the new deep-data structure
		 *		used when deep data is loaded after tree used initially 
		 */
		 
		initDeep = function(d) {
				
				// Set loadDeep to true whatever it was. Means deep-data.json is used, not tree.json
				settings.loadDeep = true;
				
				if (typeof tree === UNDEF) {
					
					// First load
					tree = d;
					
					tree[J.DEEP] = true;
					
					// Adding counters
					initCounters();
					
					// Enhancing all the folders with PATH and RELPATH variables, counting object types
					addExtras();
					
				} else {
					
					// Migrating from tree.json
					let	oldTree = tree,
						migrateFolder = function(oldFolder, newFolder) {
								//log('Migrating folder "' + oldFolder[J.NAME] + '"');
								// Copying missing properties (e.g. addExtras)
								for (let prop in oldFolder) {
									if (prop !== J.FOLDERS && prop !== J.ALBUM && 
										prop !== J.OBJECTS && prop !== J.READY && 
										prop !== J.ONREADY && !newFolder.hasOwnProperty(prop)) {
										//log('Copying property "' + prop + '".');
										newFolder[prop] = oldFolder[prop];
									}
								}
								
								// Firing ready functions attached to old folder
								if (oldFolder.hasOwnProperty(J.ONREADY)) {
									//log('Firing ' + oldFolder[J.ONREADY].length + ' ready events.');
									oldFolder[J.ONREADY].forEach(rfn => (rfn.length > 1)? rfn[0](newFolder, rfn[1]) : rfn[0](newFolder));
								}
								
								if (newFolder.hasOwnProperty(J.OBJECTS)) {
									let pr = oldFolder[J.PATHREF],
										rp = oldFolder[J.RELPATH] || null,
										idx = 0;
										
									//log('Adding extras to "' + oldFolder[J.NAME] + '".');	
									// Adding path references in case of deep-data.json
									// Collecting folders' indexes into FOLDERS array for easier access later 
									newFolder[J.OBJECTS].forEach(o => {
											if (o[J.CATEGORY] !== 'folder') {
												o[J.PATHREF] = pr;
												if (rp) {
													o[J.RELPATH] = rp;
												}
											} else {
												if (!idx) {
													newFolder[J.FOLDERS] = [];
												}	
												newFolder[J.FOLDERS].push(idx++);
											}
										});
								
									if (oldFolder.hasOwnProperty(J.FOLDERS)) {
										oldFolder[J.FOLDERS].forEach(f => {
												// Recursive call for subfolders
												const nf = newFolder[J.OBJECTS].find(o => (o[J.CATEGORY] === 'folder' && o[J.PATH] === f[J.PATH]));
												if (nf) {
													migrateFolder(f, nf);
												} else {
													log('Database inconsistency when accessing folder "' + oldFolder[J.NAME] + '"!');
												}
											});
									}
								}
								
							};
							
					// Replacing tree with deep data 
					tree = d;
					
					// Deep structure flag
					tree[J.DEEP] = true;
					
					// Migrating the missing properties
					migrateFolder(oldTree, tree);
				}					
			},
			
		/*
		 *	onDeepReady: Checks if all data has been loaded 
		 *		if loaded executes the function immediately 
		 *		if not then saves the function for later execution 
		 *		fn - is the folder to be called on ready
		 *		params - the parameters array (optional)
		 */
		
		onDeepReady = function(folder, fn, params) {
			
				if (folder && typeof fn === FUNCTION) {
					
					if (deepReady) {
						// Already loaded: fire immediately
						if (typeof params !== UNDEF) {
							fn(folder, params);
						} else {
							fn(folder);
						}
					} else {
						// Need to cache
						//log('Caching onReady function for folder "' + folder[J.NAME] + '" ' + (folder.hasOwnProperty(J.ONREADY)? ('[' + (folder[J.ONREADY].length + 1) + ']') : '(onReady array is missing)'));
						
						if (typeof params !== UNDEF) {
							deepReadyFunctions.push([ folder, fn, Array.isArray(params)? params : [ params ] ]);
						} else {
							deepReadyFunctions.push([ folder, fn ]);
						}
					}
				}
			},
			
		/*
		 *	loadDeep: Loads deep data structure
		 */
		
		loadDeep = function(doneFn, failFn) {
				
				if (deepReady) {
					// Already loaded 
					//log('loadDeep(): already in "deepReady" state');
					doneFn.call(tree);
					return;
				}
				
				let ins = new Date(),
					src = makePath(albumPath || settings.rootPath, settings.deepDataFile) + cacheBuster;
				
				//log('loadDeep("' + src + '")');
				
				// Loading deep-data.json
				return $.getJSON(src)
					.done(function(d) {
							// deep-data.json loaded
							
							if (typeof DEBUG !== UNDEF && DEBUG) {
								log('Deep data loaded: ' + ((new Date()) - ins) + 'ms');
								ins = new Date();
							}
							
							initDeep(d);
							
							// Setting current folder variable
							if (setCurrentFolder()) {
							
								deepReady = true;
								/*
								if (typeof DEBUG !== UNDEF && DEBUG) {
									log('Deep data objects are ready: ' + ((new Date()) - ins) + 'ms' + ' total: ' + tree[J.LOADCOUNTER][J.TOTAL] + ' objects');
								}
								*/
								if (typeof doneFn === FUNCTION) {
									doneFn.call(this);
								} else {
									// Calling folder ready function
									if (typeof settings['ready'] === FUNCTION) {
										settings.ready.call(this);
										settings.ready = null;
									}
									
									// Calling deep ready function 
									if (typeof settings['deepReady'] === FUNCTION) {
										settings.deepReady.call(this);
										settings.deepReady = null;
									}
								}
								
								if (deepReadyFunctions.length) {
									// [ folder, readyFn, arguments ]
									deepReadyFunctions.forEach(rfn => (rfn.length > 2)? rfn[1](rfn[0], rfn[2]) : rfn[1](rfn[0]));
									deepReadyFunctions = [];
								}
							}
							
						})
					.fail(function(jqxhr, status, error) {
						
							if (typeof settings['fatalError'] === FUNCTION) {
								settings.fatalError.call(this, 'databaseAccessDenied', src);
							}
							
							if (typeof failFn === FUNCTION) {
								failFn.call(this);
							}
							
						});
			},
			
		/***************************************************************
		 *
		 *						Initializing
		 *
		 */
		
		init = opt => {
			
				if (instance) {
					// Already initialized
					return instance;
				}
				
				// Unique ID
				instance = new Date();
				
				if (typeof opt !== UNDEF) {
					// Passed options
					$.extend(settings, opt);
				}
				
				// Resetting ready state
				ready = deepReady = false;
				
				// using albumPath: calling Album from outside
				if (settings.hasOwnProperty('albumPath')) {
					
					// Initializing by absolute URL
					albumPath = settings.albumPath;
					
					// Sanitizing URL
					if (albumPath.slice(-1) !== '/') {
						albumPath += '/';
					}
				}
				
				// Assigning absolute path
				absolutePath = getAbsoluteFolderPath(albumPath || settings.rootPath);
				
				// Using makeDate to get rid of the old version of JSON files
				if (settings.hasOwnProperty('makeDate')) {
					cacheBuster = '?' + settings.makeDate;
				}
				
				// Loading album's JSON file
				if (settings.loadDeep || settings.hasOwnProperty('deepReady')) {
					// Loading deep-data.json 
					loadDeep();
				} else {
					// Loading tree.jsonm when ready loading the current folder
					loadTree(loadNeededFolders);
				}
						
			};
		
	
	if (options) {
		
		if (typeof DEBUG !== UNDEF && DEBUG) {
			log('new Album(' + JSON.stringify(options) + ');');
		}
		
		init(options);
	}
	
	// Exposing internal functions
	
	return {
			//init: 							init,
			//isReady:						isReady,
			//isDeepReady:					isDeepReady,
			
			// Debug
			//getTree: 						getTree,
			//getPaths: 						getPaths,
			
			// Type checking
			isImage: 						isImage,
			isAudio: 						isAudio,
			isVideo: 						isVideo,
			isFolder:						isFolder,
			isLightboxable: 				isLightboxable,
			isCurrentFolder: 				isCurrentFolder,
			
			// Access
			getAlbumPath:					getAlbumPath,
			getAlbumRootPath:				getAlbumRootPath,
			getPath:						getPath,
			getAbsolutePath:				getAbsolutePath,
			getItemPath:					getItemPath,
			getAudioClipPath:				getAudioClipPath,
			getDimensions:					getDimensions,
			getOriginalDimensions:			getOriginalDimensions,
			getMaxDimensions:				getMaxDimensions,
			getRegions:						getRegions,
			getLink:						getLink,
			getRootPath:					getRootPath,
			getFolderPath:					getFolderPath,
			getRelativeFolderPath:			getRelativeFolderPath,
			getFolder: 						getFolder,
			getCurrentFolder:				getCurrentFolder,
			getParent: 						getParent,
			getItem:						getItem,
			getObjects: 					getObjects,
			getImages: 						getImages,
			getFolders:						getFolders,
			getAllObjects:					getAllObjects,
			
			// Properties
			getMakeDate: 					getMakeDate,
			getAlbumTitle: 					getAlbumTitle,
			getItemName:					getItemName,
			getExtension:					getExtension,
			getLevel: 						getLevel,
			getTitle: 						getTitle,
			getName: 						getName,
			getLabel: 						getLabel,
			getAlt:							getAlt,
			getComment: 					getComment,
			getThumbPath: 					getThumbPath,
			getIconPath: 					getIconPath,
			getImagePath: 					getImagePath,
			getAbsoluteImagePath:			getAbsoluteImagePath,
			getThemeImagePath:				getThemeImagePath,
			getOriginalPath:				getOriginalPath,
			getVideoPath:					getVideoPath,
			getPosterPath: 					getPosterPath,
			getOptimalImage:				getOptimalImage,
			getOptimalImagePath:			getOptimalImagePath,
			getOptimalThumbPath:			getOptimalThumbPath,
			getSourcePath: 					getSourcePath,
			getAbsoluteItemPath: 			getAbsoluteItemPath,
			getVideoDuration:				getVideoDuration,
			hasShop: 						hasShop,
			hasLocation: 					hasLocation,
			getLocation: 					getLocation,
			hideFotomoto:					hideFotomoto,
			getPriceRange:					getPriceRange,
			getCurrency:					getCurrency,
			getDeepFolderCount:				getDeepFolderCount,
			
			// Generic property
			getRootProperty: 				getRootProperty,
			getInheritedPropertyObject:		getInheritedPropertyObject,
			getInheritedProperty:			getInheritedProperty,
			getProperty: 					getProperty,
			getPropertyObject:				getPropertyObject,
			
			getNextFoldersFirstImage:		getNextFoldersFirstImage,
			getPreviousFoldersLastImage:	getPreviousFoldersLastImage,
			
			sortItems:						sortItems,
			
			// Search
			collectByPath:					collectByPath,
			collectNItem:					collectNItem,
			collectByDate: 					collectByDate,
			collectItems: 					collectItems,
			collectTags: 					collectTags,
			
			processTemplate:				processTemplate
				
		};
		
};