package plain;

/*
	Gui.java - User Interface
*/

import com.kitfox.svg.app.beans.SVGIcon;
import icons.Lazaicon;
import javax.swing.*;
import java.util.logging.Level;
import java.awt.*;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.awt.event.MouseAdapter;
import java.awt.event.MouseEvent;
import java.beans.PropertyChangeListener;
import java.io.*;
import javax.swing.border.Border;
import javax.swing.filechooser.FileNameExtensionFilter;
import se.datadosen.component.*;
import net.jalbum.component.*;
import se.datadosen.jalbum.*;
import se.datadosen.util.Item;
import se.datadosen.util.Languages;
import static se.datadosen.component.StateMonitor.enable;
import se.datadosen.jalbum.event.JAlbumAdapter;
import se.datadosen.jalbum.event.JAlbumEvent;
import se.datadosen.util.IO;
import se.datadosen.util.VersionNumber;

/*************************************************
 * 
 *			Plain skin GUI
 *			author David and Laza
 */

public class Gui extends GuiBase {
	
	private long skinReadyAt = Long.MAX_VALUE;
	private final VersionNumber jalbumVersion = new VersionNumber(AlbumBean.getInternalVersion());
	private final SkinProperties skinProps = new SkinProperties(skinDirectory);
	private final String skinVersion = skinProps.getProperty(SkinProperties.VERSION);
	private final String skinAuthor = skinProps.getProperty(SkinProperties.AUTHOR);
	private final String supportForum = skinProps.getProperty(SkinProperties.SUPPORT_FORUM);
	private final String helpRoot = "http://jalbum.net/help/en/Skin/Plain";
	
	private final Border emptyBorder = BorderFactory.createEmptyBorder();
	private final String tabStyle = "style='padding:3px 2px;margin:0'";
	private Font mono = new Font("monospaced", Font.PLAIN, 12);
	
	private static Icon icon(String basename) {
		return Icons.get(Lazaicon.class, "svg/" + basename + ".svg", 18);
	}
		
	private static Icon icon(String basename, int size) {
		return Icons.get(Lazaicon.class, "svg/" + basename + ".svg", size);
	}
	
	private static Icon svgIcon(String basename, Dimension dim) {
		DeferredSVGIcon icon = new DeferredSVGIcon(Gui.class, "graphics/" + basename + ".svg");

		icon.setAntiAlias(true);
		icon.setAdaptColors(true);
		icon.setPreferredSize(dim);
		icon.setAutosize(SVGIcon.AUTOSIZE_STRETCH);

		return icon;
	}
	
	private Object[] getPosition() {
		return new Object[] {
			new Item("left top", getText("ui.left_top")), 
			new Item("center top", getText("ui.center_top")), 
			new Item("right top", getText("ui.right_top")), 
			new Item("left center", getText("ui.left_middle")), 
			new Item("center center", getText("ui.center_middle")), 
			new Item("right center", getText("ui.right_middle")), 
			new Item("left bottom", getText("ui.left_bottom")), 
			new Item("center bottom", getText("ui.center_bottom")), 
			new Item("right bottom", getText("ui.right_bottom"))
		};
	}
	
	private void allowHTMLEditing(JSmartTextArea ta) {
		try {
			ta.setAllowHTMLEditing(true).setFullHTMLEditing(true);
		} catch (Throwable e) {
		}
	}
	
	private void allowSpelling(JSmartTextArea ta) {
		try {
			ta.spelling();
		} catch (Throwable e) {
		}
	}
	
	private String majorVersion(String s) {
		int i = s.indexOf(".");
		if (i > 0) {
			return s.substring(0, i);
		}
		return s;
	}
	
	private int getMajorVersion(Object v) {
		
		if (v == null) {
			return -1;
		} else if (v instanceof Integer) {
			return (int)v;
		} else if (v instanceof String) {
			try {
				return Integer.valueOf(majorVersion((String)v));
			} catch(NumberFormatException ex) {
				return -1;
			}
		} else if (v instanceof Double) {
			return (int)Math.floor((Double)v);
		}
		return -1;
	}
	
	private boolean isSkinReady() {
		// Within 1s of launch
		//System.out.println("skinReady()? " + (System.currentTimeMillis() - skinReadyAt));
		return (System.currentTimeMillis() - skinReadyAt) > 1000L;
	}
	
	private String getStyleName() {
		String style = engine.getStyle();
		return style.substring(0, style.indexOf("."));
	}
	
	private String getSkinName() {
		return engine.getSkin();
	}
	
	private String getLastSkinName() {
		String ls = null;
		
		try {
			ls = engine.getLastSkin();
		} catch(Throwable t) {
			JAlbum.logger.log(Level.FINER, "Last skin is unknown.");
		}
		
		return ls;
	}	
	
	private final String unCamelCase(String s) {
		
		if (s.length() > 0) {
			s = s.replaceAll("([a-z])([A-Z]+)", "$1 $2");
			return Character.toUpperCase(s.charAt(0)) + s.substring(1);
		}
		
		return "";
	}

	private final String getText(String t) {
		String s;
		
		try {
			s = texts.getString(t);
		} catch (java.util.MissingResourceException e) {
			JAlbum.logger.log(Level.FINE, "Missing text: \"{0}\"", t);
			if (t.startsWith("ui.")) {
				t = t.substring(3);
			}
			s = unCamelCase(t);
		}
		
		return s;
	}
	
	private final String[] imageFiles = new String[] { "jpg", "jpeg", "png", "gif", "svg" };
	private final FileChooser fc = ChooserFactory.createFileChooser(window);
	
	private void getFileToRes(String[] extensions, JTextField name, Component c) {
		fc.setFileFilter(new FileNameExtensionFilter(getText("ui.select"), extensions));
		int returnVal = fc.showOpenDialog(c);
		
		if (returnVal == JFileChooser.APPROVE_OPTION) {
			String fn = fc.getSelectedFile().toString();
			
			if (!fn.trim().equals("")) {
				File src = new File(fn);
				File dst = new File(context.getPluginContext().getRootFolder().getFile(), "res");
				if (!dst.exists()) {
					dst.mkdir();
				}
				if (src.exists() &&  dst.exists()) {
					try {
						IO.copyFile(src, dst);
						name.setText(src.getName());
					} catch (IOException e) {
						System.out.println(e);
					}
				}
			}
		}
	}
			
	private String getFileContents(File f) {
		StringBuilder cont = new StringBuilder();
		String line;
		String nl = System.getProperty("line.separator");
		
		if (f.exists()) {
			try {
				try (java.io.BufferedReader in = new java.io.BufferedReader(new java.io.FileReader(f))) {
					while ((line = in.readLine()) != null) {
						cont.append(line);
						cont.append(nl);
					}
				}
			} catch (FileNotFoundException e) {
				return null;
			} catch (IOException e) {
				return null;
			}
		}
		return cont.toString();
	}
		
	private int setSelectedValue(JComboBox<Item> box, Object val) {
		int		size = box.getItemCount();
		Item	it;
		
		for (int i = 0; i < size; i++) {
			it = box.getItemAt(i);
			if (it.item.equals(val)) {
				box.setSelectedIndex(i);
				return i;
			}
		}
		
		return -1;
	}
	
	private String getSelectedValue(Object o) {
		if (o instanceof JComboBox) {
			Object so = ((JComboBox)o).getSelectedItem();
			if (so instanceof Item) {
				return ((Item)so).value.toString();
			}
			return (so == null)? null : so.toString();
		}
		return (o == null)? null : o.toString();
	}
	
	/*****************************************
	 *
	 *			Skin UI starts here
	 *
	 *****************************************/

	ControlPanel skinUi = new ControlPanel() {
		
		// Icons used multple places
		private Icon infoIcon = icon("info");
		private Icon mandatory = svgIcon("asterisk", new Dimension(16, 16));
		
		// styleName is stored as skin variable in order to check style change later
		private JTextField styleName = new JTextField(getStyleName());
		
		// skinVersion is stored as skin variable in order to check major skin version change later
		JTextField majorSkinVersion = new JTextField(majorVersion(skinVersion));
		
		// Typography
		
		JComboBox<Item> fontFamily = new JComboBox<>(new Item[]{
			new Item("Arial, \"Helvetica Neue\", Helvetica, sans-serif", "Arial"),
			new Item("Baskerville, \"Baskerville Old Face\", \"Hoefler Text\", Garamond, \"Times New Roman\", serif", "Baskerville"),
			new Item("Calibri, Candara, Segoe, \"Segoe UI\", Optima, Arial, sans-serif", "Calibri"),
			new Item("Cambria, Georgia, Times, \"Times New Roman\", serif", "Cambria"),
			new Item("\"Century Gothic\", \"Apple Gothic\", \"Goudy Old Style\", sans-serif", "Century Gothic"),
			new Item("\"Comic Sans MS\", cursive", "Comic Sans"),
			new Item("Consolas, \"Lucida Console\", Monaco, monospace", "Consolas"),
			new Item("Constantia, Palatino, \"Palatino Linotype\", \"Palatino LT STD\", Georgia, serif", "Constantia"),
			new Item("\"Copperplate Light\", \"Copperplate Gothic Light\", serif", "Copperplate Light"),
			new Item("\"Courier New\", Courier, monospace", "Courier New"),
			new Item("\"Franklin Gothic Medium\", \"Arial Narrow Bold\", Arial, sans-serif", "Franklin Gothic"),
			new Item("Futura, \"Century Gothic\", AppleGothic, sans-serif", "Futura"),
			new Item("Garamond, \"Hoefler Text\", Times New Roman, Times, serif", "Garamond"),
			new Item("Geneva, \"Lucida Sans\", \"Lucida Grande\", \"Lucida Sans Unicode\", Verdana, sans-serif", "Geneva"),
			new Item("Georgia, Palatino, \"Palatino Linotype\", Times, \"Times New Roman\", serif", "Georgia"),
			new Item("\"Gill Sans\", \"Gill Sans MT\", Calibri, \"Trebuchet MS\", sans-serif", "Gill Sans"),
			new Item("\"Goudy Old Style\", Garamond, \"Big Caslon\", \"Times New Roman\", serif", "Goudy Old Style"),
			new Item("\"Helvetica Neue\", Helvetica, Arial, sans-serif", "Helvetica Neue"),
			new Item("\"Hoefler Text\", Constantia, Palatino, \"Palatino Linotype\", \"Book Antiqua\", Georgia, serif", "Hoefler Text"),
			new Item("Impact, Haettenschweiler, \"Arial Narrow Bold\", sans-serif", "Impact"),
			new Item("\"Lucida Sans\", \"Lucida Grande\", \"Lucida Sans Unicode\", Verdana, sans-serif", "Lucida Sans"),
			new Item("\"Lucida Bright\", Georgia, serif", "Lucida Bright"),
			new Item("Palatino, \"Palatino Linotype\", \"Book Antiqua\", Georgia, serif", "Palatino"),
			new Item("Segoe, \"Segoe UI\", Tahoma, Geneva, \"Nimbus Sans L\", sans-serif", "Segoe"),
			new Item("Tahoma, Geneva, Verdana, sans-serif", "Tahoma"),
			new Item("Times, \"Times New Roman\", Georgia, serif", "Times"),
			new Item("\"Trebuchet MS\", \"Lucida Sans Unicode\", \"Lucida Grande\", \"Lucida Sans\", Tahoma, sans-serif", "Trebuchet MS"),
			new Item("Verdana, Geneva, Tahoma, sans-serif", "Verdana"),
			new Item("", "[   Google Fonts   ]"),
			new Item("_Barlow", "Barlow"),
			new Item("_Barlow Semi Condensed", "Barlow Semi Condensed"),
			new Item("_Cinzel", "Cinzel"),
			new Item("_Cormorant", "Cormorant"),
			new Item("_Dosis", "Dosis"),
			new Item("_Exo 2", "Exo 2"),
			new Item("_IBM Plex Sans", "IBM Plex Sans"),
			new Item("_IBM Plex Serif", "IBM Plex Serif"),
			new Item("_Josefin Sans", "Josefin Sans"),
			new Item("_Josefin Slab", "Josefin Slab"),
			new Item("_Lato", "Lato"),
			new Item("_League Spartan", "League Spartan"),
			new Item("_Montserrat", "Montserrat"),
			new Item("_Noto Serif", "Noto Serif"),
			new Item("_Open Sans", "Open Sans"),
			new Item("_Oswald", "Oswald"),
			new Item("_Quicksand", "Quicksand"),
			new Item("_Raleway", "Raleway"),
			new Item("_Roboto", "Roboto"),
			new Item("_Roboto Condensed", "Roboto Condensed"),
			new Item("_Roboto Slab", "Roboto Slab"),
			new Item("_Saira", "Saira"),
			new Item("_Sofia Sans", "Sofia Sans"),
			new Item("_Sofia Sans Condensed", "Sofia Sans Condensed"),
			new Item("_Source Sans 3", "Source Sans 3"),
			new Item("_Source Serif 4", "Source Serif 4"),
			new Item("_Ubuntu", "Ubuntu"),
			new Item("_Yanone Kaffeesatz", "Yanone Kaffeesatz"),
			new Item("_Work Sans", "Work Sans"),
			new Item("_Zilla Slab", "Zilla Slab")
		});
		JComboBox<Item> fontWeight = new JComboBox<>(new Item[] {
			new Item("100", "Thin"),
			new Item("200", "Extra Light"),
			new Item("300", "Light"),
			new Item("400", "Regular"),
			new Item("500", "Medium"),
			new Item("600", "Semi Bold"),
			new Item("700", "Bold"),
			new Item("800", "Extra Bold"),
			new Item("900", "Black")
		});
		JComboBox headlineFontFamily = new JComboBox(new Object[] {
			new Item("", "[   " + getText("ui.sameAsBaseFont") + "   ]"),
			new Item("Abril Fatface", "Abril Fatface"),
			new Item("Amatic SC:700", "Amatic SC Bold"),
			new Item("Arapey", "Arapey"),
			new Item("Barlow:300", "Barlow Light"),
			new Item("Barlow:500", "Barlow Medium"),
			new Item("Barlow Semi Condensed:400", "Barlow Semi Condensed"),
			new Item("Barlow Condensed", "Barlow Condensed"),
			new Item("Cantata One", "Cantata One"),
			new Item("Cookie", "Cookie"),
			new Item("Cormorant", "Cormorant"),
			new Item("Dancing Script", "Dancing Script"),
			new Item("Economica", "Economica"),
			new Item("Exo 2:300", "Exo 2"),
			new Item("Fauna One", "Fauna One"),
			new Item("Fjalla One", "Fjalla One"),
			new Item("Geo", "Geo"),
			new Item("Gilda Display", "Gilda Display"),
			new Item("Grand Hotel", "Grand Hotel"),
			new Item("Gruppo", "Gruppo"),
			new Item("Handlee", "Handlee"),
			new Item("IBM Plex Serif:300", "IBM Plex Serif"),
			new Item("Josefin Slab", "Josefin Slab"),
			new Item("Julius Sans One", "Julius Sans One"),
			new Item("Jura", "Jura"),
			new Item("Libre Baskerville", "Libre Baskerville"),
			new Item("Lobster", "Lobster"),
			new Item("Lobster Two", "Lobster Two"),
			new Item("Lora", "Lora"),
			new Item("Martel:300", "Martel Light"),
			new Item("Marvel:700", "Marvel Bold"),
			new Item("Medula One", "Medula One"),
			new Item("Merriweather:300", "Merriweather"),
			new Item("Noticia Text", "Noticia Text"),
			new Item("Noto Serif Display:300", "Noto Serif Light"),
			new Item("Oranienbaum", "Oranienbaum"),
			new Item("Oswald:300", "Oswald Light"),
			new Item("Philosopher", "Philosopher"),
			new Item("Poiret One", "Poiret One"),
			new Item("Prata", "Prata"),
			new Item("PT Mono", "PT Mono"),
			new Item("Raleway:300", "Raleway"),
			new Item("Raleway:600", "Raleway Bold"),
			new Item("Roboto Condensed", "Roboto Condensed"),
			new Item("Roboto Slab:300", "Roboto Slab"),
			new Item("Roboto Slab:600", "Roboto Slab Bold"),
			new Item("Rochester", "Rochester"),
			new Item("Saira", "Saira"),
			new Item("Scope One", "Scope One"),
			new Item("Smooch Sans", "Smooch Sans"),
			new Item("Sofia", "Sofia"),
			new Item("Sofia Sans Condensed:200", "Sofia Sans Condensed ExtraLight"),
			new Item("Sofia Sans Condensed:600", "Sofia Sans Condensed SemiBold"),
			new Item("Special Elite", "Special Elite"),
			new Item("Squada One", "Squada One"),
			new Item("Strait", "Strait"),
			new Item("Unica One", "Unica One"),
			new Item("Vidaloka", "Vidaloka"),
			new Item("Work Sans", "Work Sans"),
			new Item("Yanone Kaffeesatz:300", "Yanone Kaffeesatz"),
			new Item("Yanone Kaffeesatz:500", "Yanone Kaffeesatz Bold"),
			new Item("Zilla Slab", "Zilla Slab")
		});

		private final JComboBox fontSuggestions = new JComboBox(new Object[] {
			"[" + getText("ui.suggestedFonts") +  "]",
			"Barlow Semi Condensed / Barlow",
			"Cormorant / Zilla Slab",
			"Exo 2 / Exo 2",
			"Barlow Condensed / Roboto",
			"IBM Plex Serif / IBM Plex Sans",
			"Jura / Roboto Condensed",
			"Martel Light / Roboto",
			"Oswald Light / Sofia Sans Condensed",
			"Raleway / Lato",
			"Roboto Condensed / Roboto",
			"Roboto Slab / Source Serif 4",
			"Special Elite / Zilla Slab",
			"Unica One / Barlow Semi Condensed",
			"Yanone Kaffeesatz Bold / Barlow"
		});

		JCheckBox shareFacebook = new JCheckBox("Facebook", false);
		JCheckBox shareTwitter = new JCheckBox("Twitter / X", false);
		JCheckBox shareLinkedIn = new JCheckBox("LinkedIn", false);
		JCheckBox sharePinterest = new JCheckBox("Pinterest", false);
		JCheckBox shareReddit = new JCheckBox("Reddit", false);
		JCheckBox shareEmail = new JCheckBox(getText("ui.email"), false);
		JCheckBox shareLink = new JCheckBox(getText("ui.link"), false);
		
		private void setFontBoxes() {
			
			if (fontSuggestions.getSelectedIndex() == 0) {
				return;
			}
			String	s = fontSuggestions.getSelectedItem().toString(),
					hf = s.split("/")[0].trim(),
					ff = s.split("/")[1].trim();

			setSelectedValue(fontFamily, ff);
			setSelectedValue(headlineFontFamily, hf);
			fontSuggestions.setSelectedIndex(0);
		};
		
		private void setMakeSlides() {
			boolean ov = engine.isSlides(),
					nv = shareFacebook.isSelected();
			
			if (ov != nv) {
				try {
					window.ui2Engine();
					engine.setSlides(nv);
					window.engine2UI();
				} catch (Exception ex) {
					throw new RuntimeException(ex);
				}
			}
		};
		
		private PropertyChangeListener setupMonitors = pce -> {
			
			JAlbum.logger.log(Level.FINE, "Setting up state monitors");
			
			StateMonitor.monitoring(fontSuggestions).onChange(src -> {
				if (isSkinReady() && src != null) {
					setFontBoxes();
				}
			});
			/*
			StateMonitor.monitoring(extraSizes).onChange(src -> {
				extraSizesChanged.setSelected(lastExtraSizes == null || !lastExtraSizes.equals(extraSizes.getText()));
			});
			*/
			
			// Separate slide page requirement
			StateMonitor.monitoring(
					shareFacebook,
					shareTwitter
					/*writeSitemapXml,
					sitemapIncludeSlides*/).onChange(src -> {
				setMakeSlides();
			});
		};
		
		/*	---------------------------------------------------------------
									Site
			--------------------------------------------------------------- */
		
		ControlPanel site = new ControlPanel() {

			JComboBox<Item> language = new JComboBox<Item>() {
				{
					setModel(Languages.modelFrom(new File(context.getSkinDir(), "texts")));
					insertItemAt(new Item("jalbum", "[ " + getText("ui.jalbumPreference") + " ]"), 0);
					setSelectedIndex(0);
				}
			};
			
			// Typography
			
			JSpinner albumFontSize = new JSpinner(new SpinnerNumberModel(16, 12, 20, 1));
		
			ControlPanel typography = new ControlPanel() {
				
				{
					setTitle(getText("ui.typography"));
					
					add("", new JLabelFor(getText("ui.pairingSuggestions"), fontSuggestions));
					add("tab", fontSuggestions);
					add("br", new JLabelFor(getText("ui.font"), fontFamily));
					add("tab", fontFamily);
					add("tab", fontWeight);
					add(" ", albumFontSize);
					add(new JLabel("px"));
					add("br", new JLabelFor(getText("ui.headlineFont"), headlineFontFamily));
					add("tab", headlineFontFamily);
				}
			};
			
			// Design
			
			JComboBox contentPadding = new JComboBox<>(new Item[] {
				new Item("12", getText("ui.thin")),
				new Item("24", getText("ui.narrow")),
				new Item("36", getText("ui.normal")),
				new Item("48", getText("ui.wide")),
				new Item("60", getText("ui.huge"))
			});
			
			JComboBox zipImages = new JComboBox(new Object[] {
				new Item("none", getText("ui.noDownloadButton")),
				new Item("slides", getText("ui.scaledDown")),
				new Item("originals", getText("ui.originals")),
				new Item("included", getText("ui.includedOriginals"))
			}); 
			JLabel zipInfo = new JLabel(infoIcon);
			
			JColorSelector pageBgColor = new JColorSelector(getText("ui.pageBackground"), new JSmartTextField(10));
			JTextField backgroundImageName = new JSmartTextField(16);
			JComboBox backgroundPos = new JComboBox(getPosition());
			JComboBox backgroundRepeat = new JComboBox(new Object[] {
				new Item("no-repeat", getText("ui.no_repeat")),
				new Item("repeat-x", getText("ui.repeat_x")),
				new Item("repeat-y", getText("ui.repeat_y")),
				new Item("repeat", getText("ui.repeat_both")),
				new Item("stretch", getText("ui.stretch")),
			});
			JColorSelector textColor = new JColorSelector(getText("ui.textColor"), new JSmartTextField(10));
			JColorSelector accentColor = new JColorSelector(getText("ui.accentColor"), new JSmartTextField(10));
			JColorSelector borderColor = new JColorSelector(getText("ui.borderColor"), new JSmartTextField(10));
			JColorSelector secondaryBgColor = new JColorSelector(getText("ui.aboveThumbsBackground"), new JSmartTextField(10));
			JColorSelector secondaryTextColor = new JColorSelector(getText("ui.aboveThumbsTextColor"), new JSmartTextField(10));
			JColorSelector sidebarBgColor = new JColorSelector(getText("ui.sidebarBackground"), new JSmartTextField(10));
			JColorSelector sidebarTextColor = new JColorSelector(getText("ui.sidebarTextColor"), new JSmartTextField(10));
			JColorSelector lightboxBgColor = new JAlphaColorSelector(getText("ui.lightboxBackground"), new JSmartTextField(10));
			JSpinner lightboxBorderWidth = new JSpinner(new SpinnerNumberModel(0, 0, 30, 1));
			JColorSelector lightboxBorderColor = new JAlphaColorSelector(getText("ui.imageBorder"), new JSmartTextField(10));
			ControlPanel design = new ControlPanel() {

				private JButton selectImage = new JButton();

				{
					setTitle(getText("ui.design"));
					
					contentPadding.setToolTipText(getText("ui.whiteSpaceInfo"));
					selectImage.setText(getText("ui.select"));
					selectImage.addActionListener(new ActionListener() { 
						@Override
						public void actionPerformed(ActionEvent e) {
							getFileToRes(imageFiles, backgroundImageName, skinUi);
					}});
					
					secondaryBgColor.setToolTipText(getText("ui.aboveThumbsInfo"));
					secondaryBgColor.getTextComponent().setToolTipText(getText("ui.aboveThumbsInfo"));
					secondaryTextColor.setToolTipText(getText("ui.aboveThumbsInfo"));
					secondaryTextColor.getTextComponent().setToolTipText(getText("ui.aboveThumbsInfo"));
						
					pageBgColor.setColor("#f4f4f4");
					pageBgColor.getTextComponent().setFont(mono);
					textColor.setColor("#222222");
					textColor.getTextComponent().setFont(mono);
					accentColor.setColor("#0099cc");
					accentColor.getTextComponent().setFont(mono);
					borderColor.setColor("#222222");
					borderColor.getTextComponent().setFont(mono);
					secondaryBgColor.setColor("#333333");
					secondaryBgColor.getTextComponent().setFont(mono);
					secondaryTextColor.setColor("#dddddd");
					secondaryTextColor.getTextComponent().setFont(mono);
					sidebarBgColor.setColor("#333333");
					sidebarBgColor.getTextComponent().setFont(mono);
					sidebarTextColor.setColor("#dddddd");
					sidebarTextColor.getTextComponent().setFont(mono);
					pageBgColor.setColor("#f4f4f4");
					pageBgColor.getTextComponent().setFont(mono);
					lightboxBgColor.setColor("#fa111111");
					lightboxBgColor.getTextComponent().setFont(mono);
					lightboxBorderColor.setColor("#ffeeeeee");
					lightboxBorderColor.getTextComponent().setFont(mono);

					add("", new JLabel(getText("ui.whiteSpace")));
					add("tab", contentPadding);
					add("br", new JLabelFor(getText("ui.backgroundImage"), backgroundImageName));
					add("tab", backgroundImageName);
					add(" ", selectImage);
					add("br", new JLabelFor(getText("ui.position"), backgroundPos));
					add("tab", backgroundPos);
					add(" ", backgroundRepeat);
					add("br", new JLabelFor(getText("ui.pageBackground"), pageBgColor));
					add("tab", pageBgColor);
					add("tab", pageBgColor.getTextComponent());
					add("tab", new JLabelFor(getText("ui.textColor"), textColor));
					add("tab", textColor);
					add("tab", textColor.getTextComponent());
					add("br", new JLabelFor(getText("ui.accentColor"), accentColor));
					add("tab", accentColor);
					add("tab", accentColor.getTextComponent());
					add("tab", new JLabelFor(getText("ui.borderColor"), borderColor));
					add("tab", borderColor);
					add("tab", borderColor.getTextComponent());
					add("br", new JLabelFor(getText("ui.aboveThumbsBackground"), secondaryBgColor));
					add("tab", secondaryBgColor);
					add("tab", secondaryBgColor.getTextComponent());
					add("tab", new JLabelFor(getText("ui.aboveThumbsTextColor"), secondaryTextColor));
					add("tab", secondaryTextColor);
					add("tab", secondaryTextColor.getTextComponent());
					add("br", new JLabelFor(getText("ui.sidebarBackground"), sidebarBgColor));
					add("tab", sidebarBgColor);
					add("tab", sidebarBgColor.getTextComponent());
					add("tab", new JLabelFor(getText("ui.sidebarTextColor"), sidebarTextColor));
					add("tab", sidebarTextColor);
					add("tab", sidebarTextColor.getTextComponent());
					add("br", new JLabelFor(getText("ui.lightboxBackground"), lightboxBgColor));
					add("tab", lightboxBgColor);
					add("tab", lightboxBgColor.getTextComponent());
					add("tab", new JLabelFor(getText("ui.border"), lightboxBorderWidth));
					add("", lightboxBorderWidth);
					add("", new JLabel("px"));
					add("tab", lightboxBorderColor);
					add("tab", lightboxBorderColor.getTextComponent());
				}
			};

			ControlPanel sharingPanel = new ControlPanel() {
				{
					setTitle(getText("ui.sharing"));
					
					add(shareFacebook);
					add(shareTwitter);
					add(shareLinkedIn);
					add(sharePinterest);
					add(shareReddit);
					add("br", shareEmail);
					add(shareLink);
				}
			};
	
			{
				zipInfo.addMouseListener(new MouseAdapter() {  
					@Override
					public void mouseReleased(MouseEvent e) {
						JOptionPane.showMessageDialog(window, getText("ui.nonAsciiWarning"), getText("ui.warning"), JOptionPane.WARNING_MESSAGE);
				}});
				
				add("", new JLabel(getText("ui.language")));
				add("tab", language);
				add("br", new JLabelFor(getText("ui.offerDownload"), zipImages));
				add("tab", zipImages);
				add("", zipInfo);
				add("br hfill", typography);
				add("br hfill", design);
				add("br hfill", sharingPanel);
			}
		};

		/*	---------------------------------------------------------------
									Layout
			--------------------------------------------------------------- */

		ControlPanel layout = new ControlPanel() {

			JComboBox<Item> folderPlacement = new JComboBox<>(new Item[]{
				new Item("sidebar", getText("ui.sidebar")),
				new Item("top", getText("ui.pageTop")),
				new Item("above", getText("ui.aboveTheThumbnails")),
				new Item("mixed", getText("ui.mixed")),
				new Item("section", getText("ui.newSection"))
			});
			JComboBox<Item> pagePlacement = new JComboBox<>(new Item[] {
				new Item("sidebar", getText("ui.sidebar")),
				new Item("top", getText("ui.pageTop")),
				new Item("above", getText("ui.aboveTheThumbnails")),
				new Item("mixed", getText("ui.mixed")),
				new Item("embed", getText("ui.embed"))
			});
			JComboBox<Item> webLocationPlacement = new JComboBox<>(new Item[] {
				new Item("sidebar", getText("ui.sidebar")),
				new Item("top", getText("ui.pageTop")),
				new Item("above", getText("ui.aboveTheThumbnails")),
				new Item("mixed", getText("ui.mixed"))
			});
				
			JCheckBox navigationThumbs = new JCheckBox(getText("ui.showThumbnails"), false);
			JCheckBox navigationCollapsed = new JCheckBox(getText("ui.collapsed"), false);
			JCheckBox foldersAboveIfNoImages = new JCheckBox(getText("ui.showFoldersIfNoImages"), true);
			JCheckBox folderMosaic = new JCheckBox(getText("ui.useFolderMosaic"), false);
			
			ControlPanel placementPanel = new ControlPanel() {
				
				ControlPanel navigationPanel = new ControlPanel() {
					{
						setTitle(getText("ui.sidebarNavigation"));
						navigationThumbs.setToolTipText(getText("ui.navigationThumbsInfo"));
						navigationCollapsed.setToolTipText(getText("ui.navigationCollapsedInfo"));
						
						add(navigationThumbs);
						add("tab", navigationCollapsed);
					}
				};
				
				ControlPanel placementSettings = new ControlPanel() {

					{
						add("", new JLabel(getText("ui.folders")));
						add("tab", folderPlacement);
						add("br", new JLabel(getText("ui.pages")));
						add("tab", pagePlacement);
						add("br", new JLabel(getText("ui.webLocations")));
						add("tab", webLocationPlacement);
						add("br hfill", navigationPanel);
					}
				};
				
				JLabel placementGraphics = new JLabel(svgIcon("placement", new Dimension(220, 150)));
				
				ControlPanel placementLegend = new ControlPanel() {					
					{
						add("br", new JLabel("<html><span style=\"display:inline;height:12px;background-color:#eb3b87;\"> &nbsp; </span>&nbsp;" + getText("ui.sidebar") + "</html>"));
						add("br", new JLabel("<html><span style=\"display:inline;height:12px;background-color:#1ba243;\"> &nbsp; </span>&nbsp;" + getText("ui.pageTop") + "</html>"));
						add("br", new JLabel("<html><span style=\"display:inline;height:12px;background-color:#f2aa00;\"> &nbsp; </span>&nbsp;" + getText("ui.aboveTheThumbnails") + "</html>"));
						add("br", new JLabel("<html><span style=\"display:inline;height:12px;background-color:#00acd3;\"> &nbsp; </span>&nbsp;" + getText("ui.mixed") + "</html>"));
						add("br", new JLabel("<html><span style=\"display:inline;height:12px;background-color:#595eb8;\"> &nbsp; </span>&nbsp;" + getText("ui.newSection") + " / " + getText("ui.embed") + "</html>"));

					}
				};
				
				ControlPanel placement = new ControlPanel(new BorderLayout()) {
					{
						((BorderLayout)(getLayout())).setVgap(0);
						((BorderLayout)(getLayout())).setHgap(10);
						
						add(placementSettings, BorderLayout.WEST);
						add(placementGraphics, BorderLayout.CENTER);
						add(placementLegend, BorderLayout.EAST);
					}
				};
				

				{
					setTitle(getText("ui.placement"));
					foldersAboveIfNoImages.setToolTipText(getText("ui.foldersAboveIfNoImagesInfo"));
					folderMosaic.setToolTipText(getText("ui.useFolderMosaicInfo"));
					
					new StateMonitor() {
						@Override
						public void onChange() {
							String fp = ((Item)folderPlacement.getSelectedItem()).value.toString();
							String pp = ((Item)pagePlacement.getSelectedItem()).value.toString();
							String wp = ((Item)webLocationPlacement.getSelectedItem()).value.toString();
							foldersAboveIfNoImages.setEnabled(fp.equals("sidebar") || fp.equals("top"));
							folderMosaic.setEnabled(fp.equals("sidebar") || fp.equals("above"));
							navigationPanel.setEnabled(fp.equals("sidebar") || pp.equals("sidebar") || wp.equals("sidebar"));
						}
					}.add(folderPlacement).add(pagePlacement).add(webLocationPlacement).done();
									
					add("br hfill", placement);
					add("br", foldersAboveIfNoImages);
					add("br", folderMosaic);
				}
			};
			
			JLabel layoutGraphicsFixed = new JLabel(svgIcon("fixed", new Dimension(220, 122)));
			JLabel layoutGraphicsGrid = new JLabel(svgIcon("grid", new Dimension(220, 122)));
			JLabel layoutGraphicsJustified = new JLabel(svgIcon("justified", new Dimension(220, 122)));
			WrappableJLabel layoutJustifiedInfo = new WrappableJLabel("<html><p style=\"text-align:center;\"><i>" + getText("ui.justifiedInfo") + "</i></p></html>");

		/*	---------------------------------------------------------------
									Thumbnails
			--------------------------------------------------------------- */
		
			ControlPanel thumbnails = new ControlPanel(new BorderLayout()) {

				JCheckBox fixedShapeThumbs = new JCheckBox(getText("ui.fixedShapeThumbnails"), false);
				JCheckBox justifyThumbs = new JCheckBox(getText("ui.justifyThumbnails"), true);
				JSpinner thumbSpacing = new JSpinner(new SpinnerNumberModel(8, 0, 30, 1));
				JCheckBox dropShadow = new JCheckBox(getText("ui.dropShadow"), false);
				JComboBox borderRounding = new JComboBox<>(new Item[] {
					new Item("0", getText("ui.none")),
					new Item("3", getText("ui.small")),
					new Item("6", getText("ui.medium")),
					new Item("10", getText("ui.large"))
				});
				JCheckBox useLineborder = new JCheckBox(getText("ui.lineBorder"), false);
				JSpinner borderWidth = new JSpinner(new SpinnerNumberModel(1, 1, 16, 1));
				JSpinner borderPadding = new JSpinner(new SpinnerNumberModel(2, 0, 16, 1));
				
				ControlPanel thumbSettings = new ControlPanel() {
					{
						fixedShapeThumbs.setToolTipText(getText("ui.fixedShapeThumbnailsInfo"));
						justifyThumbs.setToolTipText(getText("ui.justifyThumbnailsInfo"));
						borderPadding.setToolTipText(getText("ui.borderPaddingInfo"));
						
						add("", fixedShapeThumbs);
						add("br", justifyThumbs);
						add("br", new JLabelFor(getText("ui.thumbnailGap"), thumbSpacing));
						add("", thumbSpacing);
						add(new JLabelFor("px", thumbSpacing));
						add("br", dropShadow);
						add("tab", new JLabelFor(getText("ui.cornerRounding"), borderRounding));
						add(borderRounding);
						add("br", useLineborder);
						add("tab", new JLabelFor(getText("ui.width"), borderWidth));
						add("", borderWidth);
						add(new JLabelFor("px", borderWidth));
						add("tab", new JLabelFor(getText("ui.padding"), borderPadding));
						add("", borderPadding);
						add(new JLabelFor("px", borderPadding));
					}
				};
				
				ControlPanel thumbDemo = new ControlPanel() {
					{
						layoutJustifiedInfo.setPreferredWidth(220);
						
						add(layoutGraphicsFixed);
						add(layoutGraphicsGrid);
						add(layoutGraphicsJustified);
						add("br center", layoutJustifiedInfo);
					}
				};

				{
					setTitle(getText("ui.thumbnails"));
					new StateMonitor() {
						@Override
						public void onChange() {
							enable(useLineborder.isSelected(), borderWidth, borderPadding);
						}
					}.add(useLineborder).done();
					
					ActionListener thumbLayoutListener = new ActionListener() {
						@Override
						public void actionPerformed(ActionEvent e) {
							String[] tds = context.getEngine().getThumbSize().split("x");
							int[] td = { Integer.parseInt(tds[0]), Integer.parseInt(tds[1]) };
							int option;

							if (fixedShapeThumbs.isSelected()) {
								if (td[0] > 1.5 * td[1]) {
									td[0] = (int)Math.round(td[1] * 1.33333);
									option = JOptionPane.showConfirmDialog(context.getFrame(), getText("ui.fixedShapeDimensionsMismatch").replace("{0}", td[0]+"x"+td[1]), getText("ui.changeThumbnailBounds"), JOptionPane.OK_CANCEL_OPTION, JOptionPane.WARNING_MESSAGE);
									if (option != JOptionPane.YES_OPTION) {
										return;
									}
								} else {
									return;
								}
							} else if (justifyThumbs.isSelected()) {
								if (td[0] < 2 * td[1]) {
									td[0] = (int)Math.round(td[1] * 2);
									option = JOptionPane.showConfirmDialog(context.getFrame(), getText("ui.justifiedDimensionsMismatch").replace("{0}", td[0]+"x"+td[1]), getText("ui.changeThumbnailBounds"), JOptionPane.OK_CANCEL_OPTION, JOptionPane.WARNING_MESSAGE);
									if (option != JOptionPane.YES_OPTION) {
										return;
									}
								} else {
									return;
								}
							}

							try {
								context.getFrame().ui2Engine();
								context.getEngine().setThumbSize(td[0] + "x" + td[1]);
								JAlbum.logger.log(Level.FINE, "Thumb size has changed to {0}x{1}px.", td);
								context.getFrame().engine2UI();
							} catch (ParameterException ex) {
								throw new RuntimeException(ex);
							}
						}
					};
	
					fixedShapeThumbs.addActionListener(thumbLayoutListener);
					justifyThumbs.addActionListener(thumbLayoutListener);

					new StateMonitor() {
						@Override
						public void onChange() {
							if (fixedShapeThumbs.isSelected()) {
								layoutGraphicsFixed.setVisible(true);
								layoutGraphicsGrid.setVisible(false);
								layoutGraphicsJustified.setVisible(false);
								layoutJustifiedInfo.setVisible(false);
							} else if (justifyThumbs.isSelected()) {
								layoutGraphicsFixed.setVisible(false);
								layoutGraphicsGrid.setVisible(false);
								layoutGraphicsJustified.setVisible(true);
								layoutJustifiedInfo.setVisible(true);
							} else {
								layoutGraphicsFixed.setVisible(false);
								layoutGraphicsGrid.setVisible(true);
								layoutGraphicsJustified.setVisible(false);								
								layoutJustifiedInfo.setVisible(false);
							}
						}
					}.add(fixedShapeThumbs).add(justifyThumbs).done();
					
					ComponentUtilities.whenSelectedDisable(fixedShapeThumbs, justifyThumbs);
					
					((BorderLayout)(getLayout())).setVgap(0);
					((BorderLayout)(getLayout())).setHgap(10);

					add(thumbSettings, BorderLayout.WEST);
					add(thumbDemo, BorderLayout.EAST);
				}
			};

			{
				
				add("br hfill", placementPanel);
				add("br hfill", thumbnails);
				
			}
		};
		
		/*	---------------------------------------------------------------
								Captions tab
			--------------------------------------------------------------- */
		
		ControlPanel captions = new ControlPanel(new BorderLayout(20, 20)) {

			JCheckBox showAlbumTitle = new JCheckBox(getText("ui.showAlbumTitleInTheSubfolders"), false);
			JCheckBox showAlbumDescription = new JCheckBox(getText("ui.showAlbumDescriptionInTheSubfolders"), false);
			
			ControlPanel pageTitlePanel = new ControlPanel() {
				{
					setTitle(getText("ui.pageTitle"));
					
					showAlbumTitle.setToolTipText(getText("ui.showAlbumTitleInfo"));
					showAlbumDescription.setToolTipText(getText("ui.showAlbumDescriptionInfo"));
					
					add(showAlbumTitle);
					add("br", showAlbumDescription);
				}
			};

			JDraggableList thumbnailItems = new JDraggableList(new Object[] {
				new Item("fileTitle", getText("ui.title")),
				new Item("comment", getText("ui.comment")),
				new Item("keywords", getText("ui.keywords")),
				new Item("label", getText("ui.baseName")),
				new Item("fileName", getText("ui.fileName"))
			}, new String[] { "fileTitle" } );
			JComboBox thumbnailItemsDisplay = new JComboBox(new Object[] { 
				new Item("firstFound", getText("ui.showFirstFound")), 
				new Item("all", getText("ui.showAll")) 
			});
			JCheckBox showFolderInfo = new JCheckBox(getText("ui.showFolderInfo"));
			
			ControlPanel thumbnailCaptionPanel = new ControlPanel() {
				{					
					setTitle(getText("ui.thumbnails"));
					add(new JLabel(getText("ui.available")));
					add("tab", new JLabel(getText("ui.used")));
					add("br", new JScrollPane(thumbnailItems.getSecondaryList()));
					add("tab", new JScrollPane(thumbnailItems));
					add("br tab", thumbnailItemsDisplay);
					add("br", showFolderInfo);
				}
			};
			
			JDraggableList lightboxItems = new JDraggableList(new Object[] {
				new Item("fileTitle", getText("ui.title")),
				new Item("comment", getText("ui.comment")),
				new Item("keywords", getText("ui.keywords")),
				new Item("label", getText("ui.baseName")),
				new Item("fileName", getText("ui.fileName"))
			}, new String[] { "fileTitle", "comment" } );
			JComboBox lightboxItemsDisplay = new JComboBox(new Object[] { new Item("firstFound", getText("ui.showFirstFound")), new Item("all", getText("ui.showAll")) });
			
			ControlPanel lightboxCaptionPanel = new ControlPanel() {
				{
					lightboxItemsDisplay.setSelectedIndex(1);
					
					setTitle(getText("ui.lightbox"));
					add(new JLabel(getText("ui.available")));
					add("tab", new JLabel(getText("ui.used")));
					add("br", new JScrollPane(lightboxItems.getSecondaryList()));
					add("tab", new JScrollPane(lightboxItems));
					add("br tab", lightboxItemsDisplay);
				}
			};
			
			{					
				((BorderLayout)(getLayout())).setVgap(10);
				((BorderLayout)(getLayout())).setHgap(10);
				
				add(pageTitlePanel, BorderLayout.NORTH);
				add(thumbnailCaptionPanel, BorderLayout.WEST);
				add(lightboxCaptionPanel);
			}
		};
		
		/*	---------------------------------------------------------------
								Lightbox tab
			--------------------------------------------------------------- */
		
		ControlPanel lightbox = new ControlPanel() {

			JCheckBox fullScreenLightbox = new JCheckBox(getText("ui.lightboxTriggersFullScreen"), false);
			JCheckBox fullScreenLightboxMobilesOnly = new JCheckBox(getText("ui.onMobilesOnly"), true);
			JComboBox afterLast = new JComboBox<>(new Item[] {
				new Item("loop", getText("ui.loop")),
				new Item("ask", getText("ui.ask"))
			});
			JCheckBox useSlideshow = new JCheckBox(getText("ui.allowSlideshow"), false);
			JLabel slideshowDelayLabel = new JLabel(getText("ui.interval"));
			JSpinner slideshowDelay = new JSpinner(new SpinnerNumberModel(3000, 500, 20000, 100));
			JLabel msLabel = new JLabel("ms");
			JCheckBox useLocation = new JCheckBox(getText("ui.showLocationOnMap"), false);
			
			ControlPanel mapsPanel = new ControlPanel() {
				
				JTextField mapsApiKey = new JSmartTextField(24);
				JComboBox mapType = new JComboBox(new Object[]{
					new Item("hybrid", getText("ui.hybrid")),
					new Item("roadmap", getText("ui.roadMap")),
					new Item("satellite", getText("ui.satellite")),
					new Item("terrain", getText("ui.terrain"))
				});
				JSpinner mapZoom = new JSpinner(new SpinnerNumberModel(16, 1, 21, 1));
				
				{	
					setTitle(getText("ui.mapSettings"));
					add(new JLabel(getText("ui.type")));
					add("tab", mapType);
					add("tab", new JLabel(getText("ui.zoom")));
					add("tab", mapZoom);
					add("br", new JLabel(getText("ui.apiKey")));
					add("tab", mapsApiKey);
					add("", new JLinkLabel("https://console.cloud.google.com/apis/credentials", "Google Console"));
					add("br tab", new JLabel("<html><i>" + getText("ui.mapApiKeyInfo") + "</i></html>"));
				}
			};
			
			JCheckBox usePhotodata = new JCheckBox(getText("ui.showPhotoData"), false);
			JCheckBox rightClickProtect = new JCheckBox(getText("ui.rightClickProtect"));
			
			{
				ComponentUtilities.whenSelectedEnable(fullScreenLightbox, fullScreenLightboxMobilesOnly);
				ComponentUtilities.whenSelectedEnable(useSlideshow, new JComponent[]{slideshowDelayLabel, slideshowDelay, msLabel});
				ComponentUtilities.whenSelectedEnable(useLocation, mapsPanel);
				rightClickProtect.setToolTipText(getText("ui.rightClickProtectInfo"));

				add("", fullScreenLightbox);
				add("br", new JLabel("   "));
				add("", fullScreenLightboxMobilesOnly);
				add("br", new JLabel(getText("ui.whatToDoAfterTheLastItem")));
				add("tab", afterLast);
				add("br", useSlideshow);
				add("tab", slideshowDelayLabel);
				add("", slideshowDelay);
				add("", msLabel);
				add("br", useLocation);
				add("br", new JLabel("   "));
				add("", mapsPanel);
				add("br", usePhotodata);
				add("br", rightClickProtect);
			}
		};

		/*	---------------------------------------------------------------
									Advanced
			--------------------------------------------------------------- */
		
		//	Custom code
		
		ControlPanel advanced = new ControlPanel() {

			ControlPanel customKeysPanel = new ControlPanel() {

				JTextArea customKeys = new JSmartTextArea(7, 20);
				JScrollPane customKeysPane = new JScrollPane(customKeys);

				{
					customKeys.setEditable(true);
					customKeys.setLineWrap(false);
					customKeys.setFont(mono);
					customKeys.setTabSize(2);
					customKeysPane.setBorder(BorderFactory.createTitledBorder(getText("ui.customKeys")));

					add("hfill vfill", customKeysPane);
					add("br", new JLabel(getText("ui.customKeysInfo")));
				}
			};
			
			ControlPanel headHookTab = new ControlPanel() {

				JTextArea headHook = new JSmartTextArea(7, 20);
				JScrollPane headHookPane = new JScrollPane(headHook);

				{
					headHook.setEditable(true);
					headHook.setLineWrap(false);
					headHook.setFont(mono);
					headHook.setTabSize(2);
					headHookPane.setBorder(BorderFactory.createTitledBorder(getText("ui.headText")));

					add("hfill vfill", headHookPane);
				}
			};

			ControlPanel bodyHookTab = new ControlPanel() {

				JTextArea bodyHook = new JSmartTextArea(7, 20);
				JScrollPane bodyHookPane = new JScrollPane(bodyHook);

				{
					bodyHook.setEditable(true);
					bodyHook.setLineWrap(false);
					bodyHook.setFont(mono);
					bodyHook.setTabSize(2);
					bodyHookPane.setBorder(BorderFactory.createTitledBorder(getText("ui.bodyText")));

					add("hfill vfill", bodyHookPane);
				}
			};

			ControlPanel cssHookTab = new ControlPanel() {

				JTextArea cssHook = new JSmartTextArea(15, 20);
				JScrollPane cssHookPane = new JScrollPane(cssHook);
				WrappableJLabel info = new WrappableJLabel("<html><i>" + getText("ui.cssText") + "</i></html>");
				
				{
					cssHook.setEditable(true);
					cssHook.setLineWrap(false);
					cssHook.setFont(mono);
					cssHook.setTabSize(2);
					cssHookPane.setBorder(BorderFactory.createTitledBorder(getText("ui.cssTitle")));
					info.setPreferredWidth(700);
					
					add("hfill vfill", cssHookPane);
					add("br", info);
				}
			};

			ControlPanel jsHookTab = new ControlPanel() {

				JTextArea jsHook = new JSmartTextArea(15, 20);
				JScrollPane jsHookPane = new JScrollPane(jsHook);
				WrappableJLabel info = new WrappableJLabel("<html><i>" + getText("ui.javascriptText") + "</i></html>");
				
				{
					jsHook.setEditable(true);
					jsHook.setLineWrap(false);
					jsHook.setFont(mono);
					jsHook.setTabSize(2);
					jsHookPane.setBorder(BorderFactory.createTitledBorder(getText("ui.javascriptTitle")));
					info.setPreferredWidth(700);
					
					add("hfill vfill", jsHookPane);
					add("br", info);
				}
			};

			JTabbedPane customCodeTabs = new JTabbedPane() {				

				{
					addTab(getText("ui.customKeys"), icon("rename"), customKeysPanel);
					addTab("<HEAD>", icon("code"), headHookTab);
					addTab("<BODY>", icon("code"), bodyHookTab);
					addTab("CSS", icon("css"), cssHookTab);
					addTab("JavaScript", icon("javascript"), jsHookTab);
				}
			};

			{				
				add("hfill vfill", customCodeTabs);

				putClientProperty("helpPage", helpRoot + "/Advanced/Custom_code");
			}
		};
				
		/*	---------------------------------------------------------------
								About tab
			--------------------------------------------------------------- */
		
		ControlPanel about = new ControlPanel() {
			private final JTextArea readme = new JSmartTextArea(getFileContents(new File(skinDirectory, "readme.txt")), 20, 30);
			private final JScrollPane readmePane = new JScrollPane(readme, JScrollPane.VERTICAL_SCROLLBAR_ALWAYS, JScrollPane.HORIZONTAL_SCROLLBAR_NEVER);
			ControlPanel links = new ControlPanel() {
				{
					add("right", new JLinkLabel("https://photoswipe.com/", "PhotoSwipe Lightbox"));
					add("", new JLinkLabel("https://jalbum.net/forum/forum.jspa?forumID=88", "Plain skin forum"));
					
				}
			};
			
			{
				readme.setLineWrap(true);
				readme.setWrapStyleWord(true);
				readme.setEditable(false);
				readme.setFont(mono);
				
				add(new JLabel("<html><h2>" + context.getEngine().getSkin() + " skin</h2><p>version " + skinVersion + " ©2023 " + skinAuthor + "</p></html>"));
				add("hfill", links);
				add("br hfill vfill", readmePane);
			}
		};

		/*	---------------------------------------------------------------
								Main tabs
			--------------------------------------------------------------- */
		
		JTabbedPane tabs = new JTabbedPane() {
			
			{
				site.setBorder(BorderFactory.createEmptyBorder(10, 10, 10, 10));
				captions.setBorder(BorderFactory.createEmptyBorder(10, 10, 10, 10));
				lightbox.setBorder(BorderFactory.createEmptyBorder(10, 10, 10, 10));
				
				addTab("<html><h4 " + tabStyle + ">" + getText("ui.site") + "</h4></html>", icon("settings"), site);
				addTab("<html><h4 " + tabStyle + ">" + getText("ui.layout") + "</h4></html>", icon("story"), layout);
				addTab("<html><h4 " + tabStyle + ">" + getText("ui.captions") + "</h4></html>", icon("caption"), captions);
				addTab("<html><h4 " + tabStyle + ">" + getText("ui.lightbox") + "</h4></html>", icon("lightbox"), lightbox);
				addTab("<html><h4 " + tabStyle + ">" + getText("ui.advanced") + "</h4></html>", icon("wrench"), advanced);
				addTab("<html><h4 " + tabStyle + ">" + getText("ui.about") + "</h4></html>", icon("skin"), about);
			}
		};

		{
			// Adding UI tabs
			
			((RiverLayout)(getLayout())).setVgap(0);
			((RiverLayout)(getLayout())).setHgap(0);
			tabs.setBorder(emptyBorder);
			
			add(tabs);
			
			/*	-----------------------------------------------------------
										Listeners
				----------------------------------------------------------- */

			window.addJAlbumListener(new JAlbumAdapter() {
				@Override
				public void styleChanged(JAlbumEvent e) {
					styleName.setText(getStyleName());
				}
			});

			window.addPropertyChangeListener(JAlbumFrame.SKIN_LOADED_PROPERTY, setupMonitors);
			
			window.addJAlbumListener(new JAlbumAdapter() {
				@Override
				public void skinChanged(JAlbumEvent e) {
					//JAlbum.logger.log(Level.FINE, "Skin changed, removing listeners.");
					window.removePropertyChangeListener(JAlbumFrame.SKIN_LOADED_PROPERTY, setupMonitors);
				}
			});
				
			putClientProperty("helpPage", helpRoot);
			
		}
		
	};
	
	public Gui() {
		this(JAlbumContext.getInstance());
	}
   
	public Gui(JAlbumContext context) {
		
		super(context);
		PluginContext pc = context.getPluginContext();
		//EditPanel editPanel = pc.getEditPanel();
			
		skinUi.setBorder(emptyBorder);
		((RiverLayout)(getLayout())).setVgap(0);
		((RiverLayout)(getLayout())).setHgap(0);
		
		window.setSkinUI(skinUi);		
		skinReadyAt = System.currentTimeMillis();
		
	}

}
