/*
 *	Zip routines
 */

package photoblogger;

import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.UnsupportedEncodingException;
import java.net.URLDecoder;
import java.util.ArrayList;
import java.util.Map;
import java.util.logging.Level;
import java.util.zip.ZipEntry;
import java.util.zip.ZipOutputStream;
import se.datadosen.jalbum.AlbumBean;
import se.datadosen.jalbum.AlbumObject;
import se.datadosen.jalbum.Category;
import se.datadosen.jalbum.CompiledBase;
import se.datadosen.jalbum.JAlbum;
import se.datadosen.jalbum.JAlbumUtilities;

/**
 *
 * @author Laszlo Molnar
 */

public class Zip extends CompiledBase {
	
	public Zip(AlbumBean engine) {
		super(engine);
	}
	
	/* 
	 * Handling the status text in the client
	 */
	
	private final ArrayList<String> statusStack = new ArrayList<>();
	
	private void updateStatus(String s) {
		
		if (window != null) {
			statusStack.add(window.statusBar.getText());
			window.statusBar.setText(s);
		}
	}
	
	private void revertStatus() {
		
		if (window != null) {
			if (statusStack.isEmpty()) {
				window.statusBar.setText("");
			} else {
				window.statusBar.setText(statusStack.get(statusStack.size() - 1));
				statusStack.remove(statusStack.size() - 1);
			}
		}
	}
	
	private String urlDecode(String s) {
		
		if (s == null)
			return s;
		
		try {
			s = URLDecoder.decode(s, "UTF-8");
		} catch (UnsupportedEncodingException e) {
			JAlbum.logger.log(Level.WARNING, "Unsupported encoding exception \"{0}\"", s); 
		}
		return s;
	}
	
	private boolean isEmpty(Object o) {
		return o == null || o.toString().trim().equals("");
	}
	
	/* 
	 * zipFolder: zipping all images in a folder 
	 * type = originals | included | hires | images
	 */
	
	private int zipFolder(AlbumObject folder, ZipOutputStream out, String type) throws IOException {
		byte[] buff = new byte[2048];
		boolean orig = type.equals("originals");
		boolean incl = type.equals("included") || type.equals("hires");
		File outDir = new File(rootOutputDirectory, folder.getPathFromRoot()), file = null;
		FileInputStream in;
		Category cat;
		int len, added = 0;
		Map vars;
		String f, fp;

		for (AlbumObject ao : folder.getChildren()) {
			if (ao.isIncluded()) {
				cat = ao.getCategory();

				if ((cat != Category.webPage) && (cat != Category.folder) && (cat != Category.webLocation)) {
					if (orig) { // using the original file from the original place (not included too)
						file = ao.getFile();
					} else {
						vars = (Map)ao.getVars();
						
						if (vars != null) {
							
							if (incl) {
								fp = "originalPath";
							} else if (cat == Category.video) {
								fp = "videoPath";
							} else if (cat == Category.image) {
								fp = "imagePath";
							} else {
								fp = "closeupPath";
							}
							
							f = urlDecode((String)vars.get(fp));
							
							JAlbum.logger.log(Level.FINER, "zipFolder: Adding {0} ({1})", new Object[]{ao.getName(), f});
							
							if (!isEmpty(f)) {
							   file = new File(outDir, f);
							}
						}
					}
					
					if (file != null && file.exists()) {
						in = null;
						
						try {
							in = new FileInputStream(file);
						} catch (FileNotFoundException e) {
							JAlbum.logger.log(Level.WARNING, "zipFolder: File could not be opened: {0}", file.toString());
						}
						
						if (in != null) {
							try {
								out.putNextEntry(new ZipEntry(file.getName()));
								while ((len = in.read(buff)) > 0 ) {
									out.write(buff, 0, len);
								}
							} catch(IOException e) {
								JAlbum.logger.log(Level.WARNING, "IO Exception {0}", e);
							}
							out.closeEntry();
							in.close();
							added++;
						}
						
					} else if (!incl) {
						JAlbum.logger.log(Level.WARNING, "zipFolder: File does not exists: {0}", ((file == null)? ao.getName() : file.toString()));
					}
				}
			}
		}

		return added;
	}
	
	/*
	 * zipImages: zip all images in an album, crawls through the structure
	 * creates one zip file per folder
	 */
	
	private void zipImages(AlbumObject folder, String type) {
		// Adding images in the current folder
		String zipName = folder.getWebName() + ".zip";
		int added;
		
		//JAlbum.logger.log(Level.FINER, "zipImages: {0}, {1}", new Object[]{folder.getName(), type});
		updateStatus("Zipping files in folder: " + folder.toString());
		
		// Top level
		try {
			
			File zip = new File(rootOutputDirectory, folder.getPathFromRoot());
			
			if (zip.exists()) {
				
				zip = new File(zip, zipName);
				
				if (!zip.exists() || zip.lastModified() < folder.getLastModified()) {
					if (zip.exists() && zip.canWrite()) {
						zip.delete();
					}
					
					try (ZipOutputStream zout = new ZipOutputStream(new FileOutputStream(zip))) {
						added = zipFolder(folder, zout, type);
						zout.flush();
					}
					
					if (added == 0) {
						zip.delete();
					}
				}
			}
		} catch (IOException e) {
			JAlbum.logger.log(Level.WARNING, "IO Error:", e);
		}
		
		revertStatus();
		
		// Recursive for subfolders
		for (AlbumObject ao : folder.getChildren()) {
			if ((ao.getCategory() == Category.folder) && ao.isIncluded() && !ao.isHidden()) {
				zipImages(ao, type);
			}
		}
		
	}
	
	/*
	 * zipAll: zips all files in a folder plus the subdirectories
	 */
	
	private int zipAll(File folder, ZipOutputStream out, String base, String zipName) throws IOException {
		byte[] buff = new byte[2048];
		FileInputStream in;
		String name;
		int len, baselen = base.length() + 1, added = 0;

		updateStatus("Zipping folder: " + folder.toString());
		
		for (File file : folder.listFiles()) {
			if (file.isDirectory()) {
				// Recursive call
				added += zipAll(file, out, base, zipName);
			} else {
				name = file.getName();
				if (name.equals(zipName))
					continue;
				name = file.getAbsolutePath();
				if (baselen > 0)
					name = name.substring(baselen);
				in = null;
				try {
					in = new FileInputStream(file);
				} catch (IOException e) {
					JAlbum.logger.log(Level.WARNING, "File error: {0}, {1}", new Object[]{ file.getName(), e });
				}
				if (in != null) {
					try {
						out.putNextEntry(new ZipEntry(name));
						while ((len = in.read(buff)) > 0 ) {
							out.write(buff, 0, len);
						}
					} catch(IOException e) {
						JAlbum.logger.log(Level.WARNING, "IO Exception {0}", e);
					}
					out.closeEntry();
					in.close();
					added++;
				}
			}
		}
		
		revertStatus();
		return added;
	}
	
	/*
	 * zipOutputDirectory: zips the album
	 * creates one zip file per album
	 */
	
	private void zipOutputDirectory() {
		String zipName = rootFolder.getWebName() + ".zip";
		int added;
		
		try {
			
			File zip = new File(rootOutputDirectory, zipName);
			
			if (zip.lastModified() >= JAlbumUtilities.deepLastModified(rootFolder)) {
				return;
			}
			
			if (zip.exists() && zip.canWrite()) {
				zip.delete();
			}
			
			try (ZipOutputStream zout = new ZipOutputStream(new FileOutputStream(zip))) {
				added = zipAll(rootOutputDirectory, zout, rootOutputDirectory.getAbsolutePath(), zipName);
				zout.flush();
			}
			
			if (added == 0) {
				zip.delete();
			}
			
		} catch (IOException e) {
			JAlbum.logger.log(Level.WARNING, "IO Error:", e);
		}
	}
	
	/*
	 * createZip: the main routine, routes to zipOutputDirectory or zipImages based on type
	 */
	
	public void createZip(String type) {
				
		if (type.equals("album")) {
			zipOutputDirectory();
		} else {
			zipImages(rootFolder, type);
		}
	}
	
}
