package photoblogger;

/*
	Gui.java - User Interface
*/

import java.io.*;
import java.util.*;
import java.awt.Color;
import java.awt.event.*;
import java.awt.BorderLayout;
import java.awt.Component;
import java.awt.Desktop;
import java.awt.Dimension;
import java.awt.Font;
import java.beans.PropertyChangeListener;
import java.net.MalformedURLException;
import java.util.logging.Level;
import java.util.logging.Logger;
import java.net.URISyntaxException;
import java.net.URL;
import java.util.Map;
import javax.swing.*;
import javax.swing.BorderFactory;
import javax.swing.JButton;
import javax.swing.border.Border;
import javax.swing.filechooser.FileNameExtensionFilter;
import edu.stanford.ejalbert.BrowserLauncher;
import com.kitfox.svg.app.beans.SVGIcon;

import icons.Lazaicon;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import net.jalbum.component.JDraggableList;

import se.datadosen.util.Item;
import se.datadosen.util.IO;
import se.datadosen.component.*;
import se.datadosen.component.DeferredSVGIcon;
import se.datadosen.jalbum.AlbumObject;
import se.datadosen.jalbum.JAlbumContext;
import se.datadosen.jalbum.PluginContext;
import se.datadosen.jalbum.EditPanel;
import se.datadosen.jalbum.SkinProperties;
import se.datadosen.jalbum.AccountManager;
import se.datadosen.jalbum.SignInManager;
import se.datadosen.jalbum.JAlbumSite;
import se.datadosen.jalbum.AlbumBean;
import se.datadosen.jalbum.Icons;
import se.datadosen.jalbum.JAlbum;
import se.datadosen.jalbum.JAlbumColor;
import se.datadosen.jalbum.JAlbumFrame;
import se.datadosen.jalbum.ParameterException;
import se.datadosen.jalbum.event.JAlbumAdapter;
import se.datadosen.jalbum.event.JAlbumEvent;
import se.datadosen.util.Languages;
import se.datadosen.util.VersionNumber;

import net.jalbum.browser.WebViewBrowser;

/*************************************************
 * 
 *			Photoblogger GUI
 * 
 */

public class Gui extends GuiBase {
		
	private long skinReadyAt = Long.MAX_VALUE;
	private final VersionNumber jalbumVersion = new VersionNumber(AlbumBean.getInternalVersion());
	private final String skinVer = new SkinProperties(skinDirectory).getProperty(SkinProperties.VERSION);
	private final String supportForum = new SkinProperties(skinDirectory).getProperty(SkinProperties.SUPPORT_FORUM);
	//private final String helpRoot = "http://jalbum.net/help/en/Skin/" + skin + "/";
	
	private boolean isSkinReady() {
		// Within 1s of launch
		//System.out.println("skinReady()? " + (System.currentTimeMillis() - skinReadyAt));
		return (System.currentTimeMillis() - skinReadyAt) > 1000L;
	}
	
	private String getStyleName() {
		String style = engine.getStyle();
		return style.substring(0, style.indexOf("."));
	}
	
	private String getSkinName() {
		return engine.getSkin();
	}
	
	private String getLastSkinName() {
		String ls = null;
		
		try {
			ls = engine.getLastSkin();
		} catch(Throwable t) {
			JAlbum.logger.log(Level.FINER, "Last skin is unknown.");
		}
		
		return ls;
	}	
	
	private String unCamelCase(String s) {
		
		if (s.length() > 0) {
			s = s.replaceAll("([a-z])([A-Z]+)", "$1 $2");
			return Character.toUpperCase(s.charAt(0)) + s.substring(1);
		}
		
		return "";
	}
	
	private String getText(String t) {
		String s;
		
		try {
			s = texts.getString(t);
		} catch (java.util.MissingResourceException e) {
			JAlbum.logger.log(Level.FINE, "Missing text: \"{0}\"", t);
			if (t.startsWith("ui.")) {
				t = t.substring(3);
			}
			s = unCamelCase(t);
		}
		
		return s;
	}
	
	private final int uiHeight = 580;
	private final int uiWidth = 520;
	private final int tabWidth = 200;
	private final int previewWidth = 520;
	private final String tabStyle = "style='padding:3px 4px;margin:3px 4px;'";
	private final String CUSTOM = "<!--custom-->";
	
	private String getFonts(String html, String ff, String hf) {
		String	gf = "",
				fw = "",
				bw = "bold",
				hw = "";
		String[] w;
		
		if (ff.contains(":")) {
			// Google font
			gf = ff;
			fw = ff.split(":")[1];
			if (fw.length() > 0) {
				w = fw.split(",");
				bw = (w.length > 2)? w[2] : w[1];
				fw = w[0];
			}
			
			ff = ff.split(":")[0];
			if (ff.contains(" ")) {
				ff = "\"" + ff + "\"";
			}
			ff += ",sans-serif";
		}

		if (!hf.equals("")) {
			if (!hf.equals(gf)) {
				gf = (gf.length() > 0)? (gf + "|" + hf) : hf;
			}
			if (hf.contains(":")) {
				// Has weight info
				hw = hf.split(":")[1];
				if (hw.length() > 0) {
					// Get base weight
					hw = hw.split(",")[0];
				}
				hf = hf.split(":")[0];
			}
			if (hf.contains(" ")) {
				hf = "\"" + hf + "\"";
			}
			hf += ",sans-serif";
		}
		
		//System.out.println("Google font: " + gf);
		return html.replace("{googleFontsLink}", (gf.length() == 0)? "" : ("\t<link rel=\"stylesheet\" href=\"http://fonts.googleapis.com/css?family=" + gf.replaceAll(" ", "+") + "&display=swap\">\n"))
			.replace("{fontFamily}", ff)
			.replace("{fontWeight}", (fw.equals("")? "300" : fw))
			.replace("{boldWeight}", bw)
			.replace("{headlineFont}", (hf.equals(""))? "inherit" : hf)
			.replace("{headlineWeight}", (hw.equals("")? "400" : hw));
		
	}
				
	private String attemptSignIn() {
		SignInManager mgr = SignInManager.getInstance();
		if ( mgr != null && mgr.isSignedIn() ) {
			return "&cid=" + AccountManager.getCid(mgr.getUserName(), mgr.getPassword());
		}
		return "";
	}
	
	//private String licensee = licenseManager.isLicenseValid()? licenseManager.getUserName() : "";
	
	private String license = licenseManager.isLicenseValid()? licenseManager.getLicenseType() : "";
	
	private boolean commercialUseAllowed = license != null && license.length() > 0 && "pro".equals(license);
	
	private Font mono = new Font("monospaced", Font.PLAIN, 12);
	
	private Border emptyBorder = BorderFactory.createEmptyBorder();
	
	private String getFileContents(String name) {
		StringBuilder cont = new StringBuilder();
		String line;
		String nl = System.getProperty("line.separator");
		File f = new File(skinDirectory, name);
		
		if (f.exists()) {
			try {
				try (java.io.BufferedReader in = new java.io.BufferedReader(new java.io.FileReader(f))) {
					while ((line = in.readLine()) != null) {
						cont.append(line);
						cont.append(nl);
					}
				}
			} catch (FileNotFoundException e) {
				return null;
			} catch (IOException e) {
				return null;
			}
		}
		return cont.toString();
	}
	
	private void writeFile(String fileName, String cont) {
		File out = new File(fileName);
		
		try {
			FileWriter writer = new FileWriter(out, false);
			writer.write(cont);
			writer.close();
		} catch (IOException e) {
			JAlbum.logger.log(Level.FINE, "Error writing \"{0}\".", fileName);
		}
	}
	
	private final Double FULLY_TRANSP = 0.0001D;
	private final Double FULLY_OPAQUE = 0.9999D;
	
	private Color toColor(JColorSelector cs) {
		return toColor(cs.toString());
	}
	
	private static final Pattern RGBA_PATTERN = Pattern.compile("rgba?\\((\\d+),(\\d+),(\\d+)(,([\\d\\.]+))?\\)");
	
	private Color toColor(String c) {
		int a = 255;
		Color clr = Color.GRAY;
		
		//System.out.print("toColor(" + c + ") -> ");
		if (c.startsWith("rgb")) {
			Matcher matcher = RGBA_PATTERN.matcher(c);
			if (matcher.matches()) {
				// Get the color components from the matched groups
				if (matcher.groupCount() > 5) {
					a = Math.min(255, (int)Math.round(Double.parseDouble(matcher.group(5)) * 255.0));
				}
				
				if (a == 255) {
					clr = new Color(
						Math.min(255, Integer.parseInt(matcher.group(1))),
						Math.min(255, Integer.parseInt(matcher.group(2))),
						Math.min(255, Integer.parseInt(matcher.group(3)))
					);
				} else {
					clr = new Color(
						Math.min(255, Integer.parseInt(matcher.group(1))),
						Math.min(255, Integer.parseInt(matcher.group(2))),
						Math.min(255, Integer.parseInt(matcher.group(3))),
						a
					);
				}
			}
			
			return clr;
		}
		
		if (c.charAt(0) == '#') {
			c = c.substring(1);
		}
		
		if (c.length() > 7) {
			a = Integer.parseInt(c.substring(0, 2), 16);
			c = c.substring(2);
		}
		
		if (c.length() == 3) {
			clr = new Color(
					Math.min(255, Integer.parseInt(c.substring(0,1), 16) * 17),
					Math.min(255, Integer.parseInt(c.substring(1,2), 16) * 17),
					Math.min(255, Integer.parseInt(c.substring(2), 16) * 17)
				);
		}
		
		if (c.length() == 6) {
			clr = new Color(
					Math.min(255, Integer.parseInt(c.substring(0,2), 16)),
					Math.min(255, Integer.parseInt(c.substring(2,4), 16)),
					Math.min(255, Integer.parseInt(c.substring(4), 16)),
					a
				);
			//System.out.println(clr.getAlpha() + " alpha=" + a);
		}
		
		return clr;
	}
	
	private String getCssColor(JColorSelector cs) {
		
		return getCssColor(toColor(cs));
	}
	
	private String getCssColor(String c) {
		
		return getCssColor(toColor(c));
	}
	
	private String getCssColor(Color c) {
		
		if (c == null) {
			return "transparent";
		}
		
		Double a = c.getAlpha() / 255.0;
		
		if (a < FULLY_TRANSP) {
			// Transparent
			return "transparent";
		} else if (a < FULLY_OPAQUE) {
			// Semi transparent
			return "rgba(" + c.getRed() + "," + c.getGreen() + "," + c.getBlue() + "," + (Math.round(a * 1000) / 1000.0) + ")";
		}
		
		// Opaque
		return "rgba(" + c.getRed() + "," + c.getGreen() + "," + c.getBlue() + ",1)";
	}
	
	private Double getAlpha(JColorSelector cs) {
		Color c = toColor(cs);
		
		return c.getAlpha() / 255.0;
	}
	
	private Double getAlpha(Color c) {
		return c.getAlpha() / 255.0;
	}
	
	private Double getAlpha(String clr) {
		Color c = toColor(clr);
		
		return c.getAlpha() / 255.0;
	}
	
	private String setAlpha(String clr, Double a) {
		Color c = toColor(clr);
		
		return "rgba(" + c.getRed() + "," + c.getGreen() + "," + c.getBlue() + "," + a + ")";
	}
	
	private Color flatColor(JColorSelector bc, JColorSelector fc) {
		
		return flatColor(toColor(bc), toColor(fc));
	}
	
	private Color flatColor(Color bc, Color fc) {
		Double a = fc.getAlpha() / 255.0;
			
		if (a < FULLY_TRANSP) {
			return bc;
		} else if (a > FULLY_OPAQUE) {
			return fc;
		}
		
		return new Color(
			(int)Math.max(0, Math.min(Math.round(a * fc.getRed() + (1 - a) * bc.getRed()), 255)),
			(int)Math.max(0, Math.min(Math.round(a * fc.getGreen() + (1 - a) * bc.getGreen()), 255)),
			(int)Math.max(0, Math.min(Math.round(a * fc.getBlue() + (1 - a) * bc.getBlue()), 255))
		);
	}
		
	private String getMiddleColor(String c1, String c2) {
		Color c = getMiddleColor(toColor(c1), toColor(c2));
		
		return "rgba(" + c.getRed() + "," + c.getGreen() + "," + c.getBlue() + "," + (Double.valueOf(c.getAlpha()) / 255.0) + ")";
	}
	
	private Color getMiddleColor(Color c1, Color c2) {
					
		return new Color(
				(int)Math.round((c1.getRed() + c2.getRed()) / 2),
				(int)Math.round((c1.getGreen() + c2.getGreen()) / 2),
				(int)Math.round((c1.getBlue() + c2.getBlue()) / 2),
				(int)Math.round((c1.getAlpha() + c2.getAlpha()) / 2)
			);
	}

	
	private String getFlatColor(JColorSelector bc, JColorSelector fc) {
		
		return getCssColor(flatColor(toColor(bc), toColor(fc)));
	}
	
	private String getFlatColor(String bc, String fc) {
		
		return getCssColor(flatColor(toColor(bc), toColor(fc)));
	}
	
	private String getFlatColor(Color bc, Color fc) {
		
		return getCssColor(flatColor(bc, fc));
	}
	
	private Double getLuminosity(JColorSelector bg, JColorSelector fg) {
		
		return getLuminosity(flatColor(bg, fg));
	}
	
	private Double getLuminosity(JColorSelector cs) {
		
		return getLuminosity(toColor(cs));
	}
	
	private Double getLuminosity(String c) {
		
		return getLuminosity(toColor(c));
	}
	
	private Double getLuminosity(Color c) {
		
		return	0.0011725490196078 * c.getRed() + 
				0.0023019607843137 * c.getGreen() +
				0.0004470588235294118 * c.getBlue();
	}
	
	private boolean isLightColor(JColorSelector cs) {
		
		return getLuminosity(cs) > 0.5;
	}
	
	private boolean isLightColor(String fc) {
		
		return getLuminosity(fc) > 0.5;
	}
	
	private boolean isLightColor(JColorSelector fc, double threshold) {
		
		return getLuminosity(fc) > threshold;
	}
	
	private boolean isLightColor(String fc, double threshold) {
		
		return getLuminosity(fc) > threshold;
	}
	
	private boolean isLightColor(JColorSelector bc, JColorSelector fc) {
		
		return isLightColor(toColor(bc), toColor(fc), 0.5);
	}
	
	private boolean isLightColor(String bc, String fc) {
		
		return isLightColor(toColor(bc), toColor(fc), 0.5);
	}
	
	private boolean isLightColor(JColorSelector bc, JColorSelector fc, Double threshold) {
		
		return isLightColor(toColor(bc), toColor(fc), threshold);
	}
	
	private boolean isLightColor(String bc, String fc, Double threshold) {
		
		return isLightColor(toColor(bc), toColor(fc), threshold);
	}
	
	private boolean isLightColor(Color bc, Color fc, Double threshold) {
		
		Double a = fc.getAlpha() / 255.0;
		
		if (a > FULLY_OPAQUE) {
			return getLuminosity(fc) >= threshold;
		}
		
		return getLuminosity(flatColor(bc, fc)) >= threshold;
	}

	private String getLegibleColor(Color bc1, Color bc2, Color fc, Double f) {
		Color bc = flatColor(bc1, bc2);
		
		return getLegibleColor(bc, fc, f);
	}
		
	private String getLegibleColor(JColorSelector bc, JColorSelector fc) {
		return getLegibleColor(toColor(bc), toColor(fc), 0.6);
	}
	
	private String getLegibleColor(JColorSelector bc, JColorSelector fc, Double f) {
		return getLegibleColor(toColor(bc), toColor(fc), f);
	}
	
	private String getLegibleColor(JColorSelector bc1, JColorSelector bc2, JColorSelector fc) {
		return getLegibleColor(flatColor(bc1, bc2), toColor(fc), 0.6);
	}
	
	private String getLegibleColor(JColorSelector bc1, JColorSelector bc2, JColorSelector fc, Double f) {
		return getLegibleColor(flatColor(bc1, bc2), toColor(fc), f);
	}
	
	private String getLegibleColor(Color bc, Color fc, Double f) {
		int r, g, b;
		
		if (Math.abs(getLuminosity(bc) - getLuminosity(fc)) >= f) {
			return getCssColor(fc);
		}
		
		f *= 255.0;
		
		if (getLuminosity(bc) > 0.5) {
			// Darken
			r = (int)Math.round(Math.max(fc.getRed() - f, 0));
			g = (int)Math.round(Math.max(fc.getGreen() - f, 0));
			b = (int)Math.round(Math.max(fc.getBlue() - f, 0));
		} else {
			// Lighten
			r = (int)Math.round(Math.min(fc.getRed() + f, 255));
			g = (int)Math.round(Math.min(fc.getGreen() + f, 255));
			b = (int)Math.round(Math.min(fc.getBlue() + f, 255));
		}
		
		return getCssColor(new Color(r, g, b));
	}
	
	private final String[] imageFiles = new String[] { "jpg", "jpeg", "png", "gif", "svg" };
	private final FileChooser fc = ChooserFactory.createFileChooser(window);

	private void getFileToRes(String[] extensions, JTextField name, Component c) {
		fc.setFileFilter(new FileNameExtensionFilter(getText("ui.select"), extensions));
		int returnVal = fc.showOpenDialog(c);
		
		if (returnVal == JFileChooser.APPROVE_OPTION) {
			String fn = fc.getSelectedFile().toString();
			
			if (!fn.trim().equals("")) {
				File src = new File(fn);
				File dst = new File(context.getPluginContext().getRootFolder().getFile(), "res");
				if (!dst.exists()) {
					dst.mkdir();
				}
				if (src.exists() &&  dst.exists()) {
					try {
						IO.copyFile(src, dst);
						name.setText(src.getName());
					} catch (IOException e) {
						System.out.println(e);
					}
				}
			}
		}
	}
	
	private void backupProjectFile(String ver) {
		
		try {
			File projectFile = window.projectChooser.getSelectedFile();
			IO.copyFile(projectFile, new File(projectFile.getParentFile(), IO.baseName(projectFile) + "-" + ((ver == null)? "old" : ver) + ".jap"));
		} catch (IOException ex) {
			JAlbum.logger.log(Level.FINE, "Error backing up project file: {0}", window.projectChooser.getSelectedFile());
		}
	}
	
	private StateMonitor commercialMonitor = new StateMonitor() {
		@Override
		public void onChange() {
			
			if (((JCheckBox)source).isSelected() && !commercialUseAllowed) {
				if (isSkinReady()) {
					Object[] options = { 
						getText("ui.signUp"),
						getText("ui.noThanks")
					};
					int n = JOptionPane.showOptionDialog(window, 
						getText("ui.licenseWarningText"), 
						getText("ui.licenseWarningTitle"), 
						JOptionPane.YES_NO_OPTION, 
						JOptionPane.INFORMATION_MESSAGE,
						null,
						options,
						options[1]
					);
					if (n == 0) {
						try {
							BrowserLauncher.openURL(JAlbumSite.getTrueInstance().getMyJAlbumUpgradeUrl() + "/?referrer=" + skin + attemptSignIn());
						} catch (se.datadosen.tags.ElementException | IOException x) {
						}
					}
				}
				((JCheckBox)source).setSelected(false);
			}
		}
	};
		
	private static Icon icon(String basename) {
		return Icons.get(Lazaicon.class, "svg/" + basename + ".svg", 18);
	}
		
	private static Icon icon(String basename, int size) {
		return Icons.get(Lazaicon.class, "svg/" + basename + ".svg", size);
	}
		
	private static Icon svgIcon(String basename, Dimension dim) {
		return svgIcon(basename, dim, false);
	}
	
	private static Icon svgIcon(String basename, Dimension dim, boolean adapt) {
		DeferredSVGIcon icon = new DeferredSVGIcon(Gui.class, "graphics/" + basename + ".svg");
		
		icon.setAntiAlias(true);
		icon.setAdaptColors(adapt);
		icon.setPreferredSize(dim);
		icon.setAutosize(SVGIcon.AUTOSIZE_STRETCH);
		
		return icon;
	}
	
	private Object[] getPosition() {
		return new Object[] {
			new Item("left top", getText("ui.left_top")), 
			new Item("center top", getText("ui.center_top")), 
			new Item("right top", getText("ui.right_top")), 
			new Item("left center", getText("ui.left_middle")), 
			new Item("center center", getText("ui.center_middle")), 
			new Item("right center", getText("ui.right_middle")), 
			new Item("left bottom", getText("ui.left_bottom")), 
			new Item("center bottom", getText("ui.center_bottom")), 
			new Item("right bottom", getText("ui.right_bottom"))
		};
	}

	private int setSelectedValue(JComboBox<Item> box, Object val) {
		int		size = box.getItemCount();
		Item	it;
		
		for (int i = 0; i < size; i++) {
			it = box.getItemAt(i);
			if (it.item.equals(val)) {
				box.setSelectedIndex(i);
				return i;
			}
		}
		
		return -1;
	}
	
	private String getSelectedValue(Object o) {
		if (o instanceof JComboBox) {
			Object so = ((JComboBox)o).getSelectedItem();
			if (so instanceof Item) {
				return ((Item)so).value.toString();
			}
			return (so == null)? null : so.toString();
		}
		return (o == null)? null : o.toString();
	}
	
	private int getSpinnerValueAsInt(JSpinner o) {
		return (Integer)o.getValue();
	}
	
	private Double getSpinnerValueAsDouble(JSpinner o) {
		return (Double)o.getValue();
	}
	
	private Double getSelectedValueAsDouble(Object o) {
		if (o instanceof JComboBox) {
			Object so = ((JComboBox)o).getSelectedItem();
			if (so instanceof Item) {
				String s = ((Item)so).value.toString();
				return ((s != null) && (s.length() > 0))? Double.parseDouble(s) : null;
			}
			return null;
		}
		return (o == null)? null : Double.parseDouble(o.toString());
	}
	
	private Integer getSelectedValueAsInt(JComboBox o) {
		Object so = ((JComboBox)o).getSelectedItem();
		if (so instanceof Item) {
			return Integer.parseInt(((Item)so).value.toString());
		}
		return null;
	}
	
	private Double getThumbAspectRatio() {
		String[] cdim = engine.getThumbSize().split("x");
		Double	w = Double.parseDouble(cdim[0]),
				h = Double.parseDouble(cdim[1]);
		
		if (w != null && h != null) {
			return w / h;
		}
		
		JAlbum.logger.log(Level.FINE, "Invalid thumbnail dimensions: {0} x {1}", new Object[]{ w.toString(), h.toString() });
		return null;
	}
	
	private Date skinLoadedOn = new Date();
		
	private void allowHTMLEditing(JSmartTextArea ta) {
		try {
			ta.setAllowHTMLEditing(true).setFullHTMLEditing(true);
		} catch (Throwable e) {
		}
	}
	
	private void allowSpelling(JSmartTextArea ta) {
		try {
			ta.spelling();
		} catch (Throwable e) {
		}
	}
	
	private String majorVersion(String s) {
		int i = s.indexOf(".");
		if (i > 0) {
			return s.substring(0, i);
		}
		return s;
	}
	
	private int getMajorVersion(Object v) {
		
		if (v == null) {
			return -1;
		} else if (v instanceof Integer) {
			return (int)v;
		} else if (v instanceof String) {
			try {
				return Integer.valueOf(majorVersion((String)v));
			} catch(NumberFormatException ex) {
				return -1;
			}
		} else if (v instanceof Double) {
			return (int)Math.floor((Double)v);
		}
		return -1;
	}
	
	Boolean checkBooleanProperty(AlbumObject[] sel, String pn) {
		
		if (sel.length > 1) {
			boolean v = sel[0].getProperties().get(pn, false);
			
			for (AlbumObject ao : sel) {
				if (ao.getProperties().get(pn, false) != v) {
					return null;
				}
			}
			return v;
		} else if (sel.length == 1) {
			return sel[0].getProperties().get(pn, false);
		} else {
			return null;
		}
	}
	/*
	void createMenuItemForBooleanProperty(JMenu menu, String pn, String title) {
		
		createMenuItemForBooleanProperty(menu, window.albumExplorer.explorer.getSelectedAlbumObjects(), pn, title);
	}
	
	void createMenuItemForBooleanProperty(JMenu menu, AlbumObject[] sel, String pn, String title) {		
		Boolean cv = checkBooleanProperty(sel, pn);

		try {
			if (cv == null) {
				// Indefinite value -> Add submenu
				JMenu subMenu = new JMenu(title);
				JMenuItem all = new JMenuItem(getText("ui.all"));
				all.addActionListener(ae -> {
					for (AlbumObject ao : sel) {
						ao.getProperties().put(pn, true);
						ao.getProperties().save();
					}				
				});
				JMenuItem none = new JMenuItem(getText("ui.none"));
				none.addActionListener(ae -> {
					for (AlbumObject ao : sel) {
						ao.getProperties().put(pn, false);
						ao.getProperties().save();
					}				
				});
				subMenu.add(all);
				subMenu.add(none);
				menu.add(subMenu);
	
			} else {
				// All items has the same value -> invert them when clicked
				JCheckBoxMenuItem menuItem = new JCheckBoxMenuItem(title, null, cv);
	
				menuItem.addActionListener(ae -> {
					for (AlbumObject ao : sel) {
						ao.getProperties().put(pn, !cv);
						ao.getProperties().save();
					}
				});
	
				menu.add(menuItem);
			}
			
		} catch (Throwable ex) {
			JAlbum.logger.log(Level.INFO, "Right-click menu is available only in jAlbum v32 and newer!");
		}
	}
	*/
	JLinkLabel safeLinkLabel(javax.swing.JComponent comp, String url, String txt) {
		
		if (jalbumVersion.compareTo(new VersionNumber("32.1.4")) >= 0) {
			try {
				return new JLinkLabel(comp, txt);
			} catch (Throwable ex) {
				
			}
		}
		
		return new JLinkLabel(url, txt);
	}
	
	/*****************************************
	 *
	 *			Skin UI starts here
	 *
	 *****************************************/
	
	ControlPanel skinUi = new ControlPanel() {
			
		// UI-wide variables
		
		// Icons used multple places
		private Icon infoIcon = icon("info");
		private Icon mandatory = svgIcon("asterisk", new Dimension(16, 16));
		
		// silence further notifications
		private boolean skinChangeReported = false;
		
		private void selectComboBoxItem(JComboBox<Item> cb, JTextArea ta) {
			selectComboBoxItem(cb, ta.getText());
		}
		
		private void selectComboBoxItem(JComboBox<Item> cb, String val) {
			int size = cb.getModel().getSize();
			
			if (size == 0) {
				return;
			}
			
			ComboBoxModel<Item> model = cb.getModel();
			
			if (val != null) {
				for (int i = 0; i < size; i++) {
					if (model.getElementAt(i).value.equals(val)) {
						//System.out.println("Found preset " + i + " = \"" + val + "\"");
						cb.setSelectedIndex(i);
						return;
					}
				}
			}
			
			//System.out.println("Not found preset \"" + val + "\"");
			cb.setSelectedIndex(model.getSize() - 1);
		}
		
		@Override
		public void importVariables(Map<String, Object> vars) {
			
			//JAlbum.logger.log(Level.FINE, "Project file loaded: {0}", window.projectChooser.getSelectedFile());
			//JAlbum.logger.log(Level.FINE, "{0}", vars.keySet());
			
			if (skinChangeReported || window.projectChooser.getSelectedFile() == null) {
				return;
			}
			
			Map<String, Object> skinVars = engine.getSkinVariables();
			int	omv = getMajorVersion(skinVars.get("majorSkinVersion")),
				cmv = getMajorVersion(skinVer);
			
			if (omv == -1) {
				// Fallback to 3.0.x variable
				omv = getMajorVersion(skinVars.get("skinVersion"));
				if (omv == -1 && skinVars.get("useAutoPano") != null) {
					// Guess if v3+
					omv = 3;
				}
			}
			
			//JAlbum.logger.log(Level.FINE, "skinVars = {" + skinVars.toString() + "}");
			
			// Old and current major versions
			
			// Old skin name
			String	lastSkin = getLastSkinName();
			boolean skinChanged = lastSkin != null && !getSkinName().equals(lastSkin);
			
			// Attempt writing out the current version
			if (cmv != -1) {
				skinVars.put("majorSkinVersion", cmv);
			}
			//System.out.println("skinVer="+skinVer+", majorSkinVersion="+cmv);
			
			if (skinChanged || cmv > omv) {
				JNotification jn;
			
				if (skinChanged) {
					// Other skin
					JAlbum.logger.log(Level.FINE, "Skin changed: {0} -> {1} v{2}", new Object[]{lastSkin, skin, cmv});
					jn = new JNotification(getText("ui.madeWithAnotherSkin") + "\n" + 
							getText("ui.backupSaved"), JNotification.Type.SKIN);
					backupProjectFile((lastSkin != null)? lastSkin : "old");					
					jn.setExpiration(3);
					jn.setRememberDismissed(true);
					window.showNotification(jn);
				} else {
					// Major version change
					JAlbum.logger.log(Level.FINE, "Major version changed: {0} v{1} -> v{2}", new Object[]{skin, omv, cmv});
					/*
					jn = new JNotification(getText("ui.madeWithOldVersion").replaceAll("\\{0\\}", skin) + 
							((omv == -1)? "" : ("(" + omv + ")"))+ "\n" +
							"\n" + getText("ui.backupSaved"), JNotification.Type.SKIN);
					backupProjectFile((omv >= 0)? String.valueOf(omv) : "old");
					*/
				}
				
				skinChangeReported = true;
				// Have to process all subdirectories after a skin or major version change
				engine.setUpdatedDirsOnly(false);
			}
			
			// Fixing old maxPageWidth syntax
			String mpw = (String)vars.get("maxPageWidth");
			if (mpw != null && !mpw.equals("none") && !mpw.endsWith("rem")) {
				vars.put("maxPageWidth", mpw.replace("em", "rem"));
			}
			
			Double zoom = (Double)vars.get("zoom");
			if (zoom != null) {
				vars.put("baseFontSize", Math.max(11, Math.min(18, (int)Math.round(zoom * 16))));
			}
			
			Object sdo = skinVars.get("slideshowDelay");
			if (sdo instanceof Double && (Double)sdo > 0.0 && (Double)sdo < 20.0) {
				vars.put("slideshowDelay", (int)((Double)sdo * 1000));
			} else if (sdo instanceof Integer && (int)sdo < 50) {
				vars.put("slideshowDelay", Math.min(20000, (int)sdo * 1000));
			}
			
			// Using Justified instead of Masonry / Horizontal
			String tlo = (String)vars.get("thumbLayout");
			if (tlo != null && tlo.equals("horizontal")) {
				vars.put("thumbLayout", "justified");
			}

			if (vars.get("topNavigationHideLong") == null || omv < 6) {
				// Version < 6.1
				// Variable omv is unreliable
				String gak = (String)vars.get("googleApiKey");
				if (gak == null || gak.trim().length() == 0) {
					vars.put("_showMapBtn", false);
					vars.put("_mapPlacement", "");
					vars.put("mapInSections", false);
					vars.put("mapInSidebar", false);
				}
			}			
		}
		/*
		private String typeOf(Object o) {
			Class c = o.getClass();
			return c.toString().replaceAll("class java\\.lang\\.", "").replaceAll("Boolean", "boolean").replaceAll("Integer", "int").replaceAll("Double", "double");
		}
		private String add_tab(String s, int tl) {
			return s + "\t\t\t\t\t\t\t\t\t\t\t\t\t\t\t\t\t\t\t\t\t\t\t".substring(0, 1 + Math.max(0, tl - s.length() - 1) / 4);
		}
		
		@Override
		public void importVariables(Map<String, Object> vars) {
			
			JAlbum.logger.log(Level.FINE, "Mapping skin variables");

			Map<String, Object> skinVars = engine.getSkinVariables();
			Map<String, Object> skinVarsSorted = new TreeMap<>(skinVars);
			String	key;
			String	ic,
					dc,
					fv,
					comment;
			Object	iv,
					dv;

			System.out.println("public class SkinModel\n{");
			for (Map.Entry<String, Object> entry : skinVarsSorted.entrySet()) {
				key = entry.getKey();
				dv = entry.getValue();
				dc = typeOf(dv);
				iv = vars.get(key);
				ic = (iv == null)? "" : typeOf(iv);
				
				if (iv == null || !dc.equals(ic)) {
					// No imported value or of different type
					fv = (dv == null)? "null" : (dc.equals("String")? ("\"" + dv.toString().replaceAll("\n", "\\n") + "\"") : dv.toString());
					comment = (iv == null)? "Missing key" : ("Incompatible types: " + dc + " =/= " + ic);
				} else {
					// Importing
					fv = ic.equals("String")? ("\"" + iv.toString().replaceAll("\"", "\\\\\"").replaceAll("\n", "\\\\n").replaceAll("\t", "\\\\t") + "\"") : iv.toString();
					comment = "";
				}
				System.out.println("\tpublic " + add_tab(dc, 12) + add_tab(key, 40) + "= " + ((comment.length() > 0)? (add_tab(fv + ";", 32) + "// " + comment) : (fv + ";")));
			}
			System.out.println("}");		
		}
		*/
		
		// styleName is stored as skin variable in order to check style change later
		private JTextField styleName = new JTextField(getStyleName());
		
		// skinVersion is stored as skin variable in order to check major skin version change later
		JTextField majorSkinVersion = new JTextField(majorVersion(skinVer));

		// Tracking extra size changes in order to be able to remove the stale files
		JTextField extraSizes = new JTextField(24);
		String lastExtraSizes = (String)engine.getSkinVariables().get("extraSizes");
		JCheckBox extraSizesChanged = new JCheckBox("", false);
		
		JTextField facebookAppId = new JSmartTextField("");
		
		JComboBox maxPageWidth = new JComboBox(new Object[] {
			new Item("none", getText("ui.noLimit")),
			new Item("47.5rem", "760px"),
			new Item("52.5rem", "840px"),
			new Item("57.5rem", "920px"),
			new Item("62.5rem", "1000px"),
			new Item("67.5rem", "1080px"),
			new Item("72.5rem", "1160px"),
			new Item("77.5rem", "1240px"),
			new Item("82.5rem", "1320px"),
			new Item("87.5rem", "1400px"),
			new Item("92.5rem", "1480px"),
			new Item("97.5rem", "1560px")
		});	
		
		JComboBox sidebarPosition = new JComboBox(new Object[] {
			new Item("left", getText("ui.left")), 
			new Item("right", getText("ui.right"))
		});
		
		JCheckBox sidebarSticky = new JCheckBox(getText("ui.sticky"));
		JCheckBox fullWidth = new JCheckBox(getText("ui.fullWidth"));

		JColorSelector pageColor = new JAlphaColorSelector(getText("ui.pageColor"), new JSmartTextField(10));
		
		JComboBox backgroundPattern = new JComboBox(new Object[] {
			new Item("", "none"),
			new Item("cardboard.png", "Cardboard"),
			new Item("circles.png", "Circles"),
			new Item("dots.png", "Dots"),
			new Item("gplay.png", "GPlay"),
			new Item("material.png", "Material"),
			new Item("linen.png", "Linen"),
			new Item("pinstripe.png", "Pinstripe"),
			new Item("tiles.png", "Tiles"),
			new Item("wood.png", "Wood")
		});	
		
		JColorSelector topmenuColor = new JAlphaColorSelector(getText("ui.topmenu"), new JSmartTextField(10));
		JColorSelector headerColor = new JAlphaColorSelector(getText("ui.header"), new JSmartTextField(10));
		JColorSelector backgroundColor = new JAlphaColorSelector(getText("ui.mainContent"), new JSmartTextField(10));
		JColorSelector footerColor = new JAlphaColorSelector(getText("ui.footer"), new JSmartTextField(10));
		
		JColorSelector textColor = new JColorSelector(getText("ui.textColor"), new JSmartTextField(10));
		JColorSelector reverseTextColor = new JColorSelector(getText("ui.reverseTextColor"), new JSmartTextField(10));
		JColorSelector linkColor = new JColorSelector(getText("ui.linkColor"), new JSmartTextField(10));
		JColorSelector hoverColor = new JColorSelector(getText("ui.hoverColor"), new JSmartTextField(10));
		
		JColorSelector boxBackgroundColor = new JAlphaColorSelector(getText("ui.backgroundColor"), new JSmartTextField(10));
		JColorSelector boxBorderColor = new JAlphaColorSelector(getText("ui.borderColor"), new JSmartTextField(10));
		JCheckBox boxDropShadow = new JCheckBox(getText("ui.dropShadow"));
		JSpinner boxBorderWidth = new JSpinner(new SpinnerNumberModel(0, 0, 10, 1));
		JComboBox boxPadding = new JComboBox(new Object[] {
			new Item("none", getText("ui.none")),
			new Item("small", getText("ui.small")),
			new Item("medium", getText("ui.medium")),
			new Item("large", getText("ui.large")),
			new Item("x-large", getText("ui.xlarge"))
		});
		JColorSelector folderColor = new JAlphaColorSelector(getText("ui.folderColor"), new JSmartTextField("#ff222222", 10));
		
		// Top bar
		JCheckBox topbarSticky = new JCheckBox(getText("ui.stickToTop"));

		JComboBox heroType = new JComboBox(new Object[] {
			new Item("flatColor", getText("ui.flatColor")),
			new Item("gradient", getText("ui.gradient")),
			new Item("folderImage", getText("ui.folderThumbnail")),
			new Item("albumImage", getText("ui.albumThumbnail"))/*,
			new Item("randomImage", getText("ui.randomImage"))*/
		});

		JCheckBox heroImageDesaturate = new JCheckBox(getText("ui.blackAndWhite"), false);
		
		JComboBox titlePlacement = new JComboBox(new Object[] {
			new Item("centered", getText("ui.center")),
			new Item("left top", getText("ui.left") + " " + getText("ui.top")),
			new Item("left middle", getText("ui.left") + " " + getText("ui.middle")),
			new Item("left bottom", getText("ui.left") + " " + getText("ui.bottom"))
		});

		JComboBox sliderType = new JComboBox(new Object[] {
			new Item("none", getText("ui.noSlider")),			
			new Item("small", getText("ui.small")),
			new Item("fullWidth", getText("ui.fullWidth"))
		});
		
		JComboBox<Item> titleCaptionPresets = new JComboBox(new Object[] {
			new Item("<h1>${title}</h1>", getText("ui.title")),
			new Item("<h1>${title}</h1>\n<div class=\"description\">${description}</div>", getText("ui.title") + " + " + getText("ui.description")),
			new Item("<h1>${title}</h1>\n<span class=\"date\">${folderModDate}</span>\n<div class=\"description\">${description}</div>", getText("ui.title") + " + " + getText("ui.date") + " + " + getText("ui.description") + " (" + getText("ui.default") + ")"),
			new Item(CUSTOM, "[ " + getText("ui.custom") + " ]")
		});
		JTextArea titleCaptionTemplate = new JSmartTextArea(4, 28);
		
		// Typography
		private final JComboBox fontSuggestions = new JComboBox(new Object[] {
			"[" + getText("ui.suggestedFonts") +  "]",
			"Barlow Semi Condensed / Barlow",
			"Cinzel Decorative / Raleway",
			"Dosis Bold / Montserrat",
			"Exo 2 / Exo 2",
			"Fjalla One / Roboto",
			"IBM Plex Serif / IBM Plex Sans",
			"Jura / Roboto Condensed",
			"Martel / Roboto",
			"Merriweather / Merriweather Sans",
			"Oswald / Open Sans",
			"Raleway / Open Sans",
			"Roboto Condensed / Roboto",
			"Roboto Slab / Montserrat",
			"Special Elite / Yanone Kaffeesatz",
			"Unica One / Lato",
			"Yanone Kaffeesatz Bold / Muli"
		});
						
		private void setFontBoxes() {
			
			if (fontSuggestions.getSelectedIndex() == 0) {
				return;
			}
			String	s = fontSuggestions.getSelectedItem().toString(),
					hf = s.split("/")[0].trim(),
					ff = s.split("/")[1].trim();

			setSelectedValue(fontFamily, ff);
			setSelectedValue(headlineFont, hf);
			fontSuggestions.setSelectedIndex(0);
		};
		
		JComboBox fontFamily = new JComboBox(new Object[] {
			new Item("Arial, \"Helvetica Neue\", Helvetica, sans-serif", "Arial"),
			new Item("Baskerville, \"Baskerville Old Face\", \"Hoefler Text\", Garamond, \"Times New Roman\", serif", "Baskerville"),
			new Item("Calibri, Candara, Segoe, \"Segoe UI\", Optima, Arial, sans-serif", "Calibri"),
			new Item("Cambria, Georgia, Times, \"Times New Roman\", serif", "Cambria"),
			new Item("\"Century Gothic\", \"Apple Gothic\", \"Goudy Old Style\", sans-serif", "Century Gothic"),
			new Item("\"Comic Sans MS\", cursive", "Comic Sans"),
			new Item("Consolas, \"Lucida Console\", Monaco, monospace", "Consolas"),
			new Item("Constantia, Palatino, \"Palatino Linotype\", \"Palatino LT STD\", Georgia, serif", "Constantia"),
			new Item("\"Copperplate Light\", \"Copperplate Gothic Light\", serif", "Copperplate Light"),
			new Item("\"Courier New\", Courier, monospace", "Courier New"),
			new Item("\"Franklin Gothic Medium\", \"Arial Narrow Bold\", Arial, sans-serif", "Franklin Gothic"),
			new Item("Futura, \"Century Gothic\", AppleGothic, sans-serif", "Futura"),
			new Item("Garamond, \"Hoefler Text\", Times New Roman, Times, serif", "Garamond"),
			new Item("Geneva, \"Lucida Sans\", \"Lucida Grande\", \"Lucida Sans Unicode\", Verdana, sans-serif", "Geneva"),
			new Item("Georgia, Palatino, \"Palatino Linotype\", Times, \"Times New Roman\", serif", "Georgia"),
			new Item("\"Gill Sans\", \"Gill Sans MT\", Calibri, \"Trebuchet MS\", sans-serif", "Gill Sans"),
			new Item("\"Goudy Old Style\", Garamond, \"Big Caslon\", \"Times New Roman\", serif", "Goudy Old Style"),
			new Item("\"Helvetica Neue\", Helvetica, Arial, sans-serif", "Helvetica Neue"),
			new Item("\"Hoefler Text\", Constantia, Palatino, \"Palatino Linotype\", \"Book Antiqua\", Georgia, serif", "Hoefler Text"),
			new Item("Impact, Haettenschweiler, \"Arial Narrow Bold\", sans-serif", "Impact"),
			new Item("\"Lucida Sans\", \"Lucida Grande\", \"Lucida Sans Unicode\", Verdana, sans-serif", "Lucida Sans"),
			new Item("\"Lucida Bright\", Georgia, serif", "Lucida Bright"),
			new Item("Palatino, \"Palatino Linotype\", \"Book Antiqua\", Georgia, serif", "Palatino"),
			new Item("Segoe, \"Segoe UI\", Tahoma, Geneva, \"Nimbus Sans L\", sans-serif", "Segoe"),
			new Item("Tahoma, Geneva, Verdana, sans-serif", "Tahoma"),
			new Item("Times, \"Times New Roman\", Georgia, serif", "Times"),
			new Item("\"Trebuchet MS\", \"Lucida Sans Unicode\", \"Lucida Grande\", \"Lucida Sans\", Tahoma, sans-serif", "Trebuchet MS"),
			new Item("Verdana, Geneva, Tahoma, sans-serif", "Verdana"),
			new Item("Alegreya Sans:300,300i,500,500i", "Alegreya Sans Light"),
			new Item("Barlow:300,300i,600,600i", "Barlow Light"),
			new Item("Barlow:400,400i,700,700i", "Barlow"),
			new Item("Barlow Semi Condensed:300,300i,500,500i", "Barlow Semi Condensed"),
			new Item("Exo 2:300,300i,600,600i", "Exo 2"),
			new Item("Fira Sans:300,300i,600,600i", "Fira Sans"),
			new Item("IBM Plex Sans:300,300i,600,600i", "IBM Plex Sans"),
			new Item("Josefin Sans:300,300i,600,600i", "Josefin Sans"),
			new Item("Josefin Slab:300,300i,600,600i", "Josefin Slab"),
			new Item("Lato:300,300i,700,700i", "Lato"),
			new Item("Merriweather Sans:300,300i,700,700i", "Merriweather Sans"),
			new Item("Merriweather:300,300i,700,700i", "Merriweather Serif"),
			new Item("Montserrat:300,300i,600,600i", "Montserrat"),
			new Item("Muli:300,300i,700,700i", "Muli"),
			new Item("Open Sans:300,300i,600,600i", "Open Sans"),
			new Item("Raleway:300,300i,600,600i", "Raleway"),
			new Item("Roboto:300,300i,500,500i", "Roboto"),
			new Item("Roboto Condensed:300,300i,700,700i", "Roboto Condensed"),
			new Item("Saira:300,300i,600,600i", "Saira"),
			new Item("Source Sans Pro:300,300i,600,600i", "Source Sans Pro"),
			new Item("Work Sans:300,300i,600,600i", "Work Sans Light"),
			new Item("Yanone Kaffeesatz:300,500", "Yanone Kaffeesatz")
		});
		
		JComboBox headlineFont = new JComboBox(new Object[] {
			new Item("", "[ " + getText("ui.sameAsBaseFont") + " ]"),
			new Item("Abril Fatface", "Abril Fatface"),
			new Item("Alex Brush", "Alex Brush"),
			new Item("Amaranth", "Amaranth"),
			new Item("Amatic SC:700", "Amatic SC Bold"),
			new Item("Anton", "Anton"),
			new Item("Arapey", "Arapey"),
			new Item("Barlow:300", "Barlow Light"),
			new Item("Barlow:500", "Barlow Medium"),
			new Item("Barlow Semi Condensed:300", "Barlow Semi Condensed Light"),
			new Item("Barlow Condensed", "Barlow Condensed"),
			new Item("Cantata One", "Cantata One"),
			new Item("Cinzel Decorative", "Cinzel Decorative"),
			new Item("Cookie", "Cookie"),
			new Item("Dancing Script", "Dancing Script"),
			new Item("Dynalight", "Dynalight"),
			new Item("Dosis:600", "Dosis Bold"),
			new Item("Economica", "Economica"),
			new Item("Emilys Candy", "Emilys Candy"),
			new Item("Euphoria Script", "Euphoria Script"),
			new Item("Exo 2:300", "Exo 2"),
			new Item("Fauna One", "Fauna One"),
			new Item("Fjalla One", "Fjalla One"),
			new Item("Fredericka the Great", "Fredericka the Great"),
			new Item("Geo", "Geo"),
			new Item("Gilda Display", "Gilda Display"),
			new Item("Grand Hotel", "Grand Hotel"),
			new Item("Great Vibes", "Great Vibes"),
			new Item("Gruppo", "Gruppo"),
			new Item("Handlee", "Handlee"),
			new Item("IBM Plex Serif:300", "IBM Plex Serif"),
			new Item("IM Fell English", "IM Fell English"),
			new Item("Italiana", "Italiana"),
			new Item("Josefin Slab", "Josefin Slab"),
			new Item("Julius Sans One", "Julius Sans One"),
			new Item("Jura", "Jura"),
			new Item("La Belle Aurore", "La Belle Aurore"),
			new Item("Libre Baskerville", "Libre Baskerville"),
			new Item("Lobster", "Lobster"),
			new Item("Lobster Two", "Lobster Two"),
			new Item("Lora", "Lora"),
			new Item("Maiden Orange", "Maiden Orange"),
			new Item("Martel:300", "Martel"),
			new Item("Marvel:700", "Marvel Bold"),
			new Item("Medula One", "Medula One"),
			new Item("Merriweather:300", "Merriweather"),
			new Item("Mountains of Christmas", "Mountains of Christmas"),
			new Item("Noticia Text", "Noticia Text"),
			new Item("Noto Serif Display:300", "Noto Serif Light"),
			new Item("Old Standard TT", "Old Standard TT"),
			new Item("Oranienbaum", "Oranienbaum"),
			new Item("Oswald", "Oswald"),
			new Item("Philosopher", "Philosopher"),
			new Item("Poiret One", "Poiret One"),
			new Item("Prata", "Prata"),
			new Item("Princess Sofia", "Princess Sofia"),
			new Item("PT Mono", "PT Mono"),
			new Item("PT Sans Narrow", "PT Sans Narrow"),
			new Item("Raleway:300", "Raleway"),
			new Item("Raleway:600", "Raleway Bold"),
			new Item("Roboto Condensed", "Roboto Condensed"),
			new Item("Roboto Slab:300", "Roboto Slab"),
			new Item("Roboto Slab:600", "Roboto Slab Bold"),
			new Item("Rochester", "Rochester"),
			new Item("Shadows Into Light Two", "Shadows Into Light Two"),
			new Item("Scope One", "Scope One"),
			new Item("Six Caps", "Six Caps"),
			new Item("Sofia", "Sofia"),
			new Item("Sorts Mill Goudy", "Sorts Mill Goudy"),
			new Item("Special Elite", "Special Elite"),
			new Item("Squada One", "Squada One"),
			new Item("Strait", "Strait"),
			new Item("Unica One", "Unica One"),
			new Item("Vidaloka", "Vidaloka"),
			new Item("Work Sans", "Work Sans"),
			new Item("Yanone Kaffeesatz:300", "Yanone Kaffeesatz"),
			new Item("Yanone Kaffeesatz:500", "Yanone Kaffeesatz Bold")
		});

		JComboBox<Item> folderCaptionPresets = new JComboBox(new Object[] {
			new Item("", getText("ui.empty")),
			new Item("<h3>${title}</h3>", getText("ui.title")),
			new Item("<h3>${title}</h3>\n<div class=\"comment\">${comment}</div>", getText("ui.title") + " + " + getText("ui.description")),
			new Item("<h3>${title}</h3>\n<div class=\"comment\">${commentShort}</div>", getText("ui.title") + " + " + getText("ui.descriptionShort") + " (" + getText("ui.default") + ")"),
			new Item("<h3>${fileLabel}</h3>", getText("ui.fileLabel")),
			new Item("<h3>${fileLabel}</h3>\n<div class=\"comment\">${comment}</div>", getText("ui.fileLabel") + " + " + getText("ui.description")),
			new Item("<h3>${title}</h3>\n<span class=\"date\">${folderModDate}</span>", getText("ui.title") + " + " + getText("ui.date")),
			new Item("<h3>${title}</h3>\n<span class=\"date\">${folderModDate}</span>\n<div class=\"comment\">${comment}</div>", getText("ui.title") + " + " + getText("ui.date") + " + " + getText("ui.description")),
			new Item("<h3>${title}</h3>\n<span class=\"date\">${folderModDate}</span>\n<div class=\"comment\">${commentShort}</div>", getText("ui.title") + " + " + getText("ui.date") + " + " + getText("ui.descriptionShort")),
			new Item("<h3>${fileLabel}</h3>\n<span class=\"date\">${folderModDate}</span>", getText("ui.fileLabel") + " + " + getText("ui.date")),
			new Item("<h3>${fileLabel}</h3>\n<span class=\"date\">${folderModDate}</span>\n<div class=\"comment\">${comment}</div>", getText("ui.fileLabel") + " + " + getText("ui.date") + " + " + getText("ui.description")),
			new Item(CUSTOM, "[ " + getText("ui.custom") + " ]")
		});
		JTextArea folderCaptionTemplate = new JSmartTextArea(4, 30);
		
		JComboBox folderColumns = new JComboBox(new Object[]{
			1,
			2,
			3,
			4
		});
		JCheckBox fixedShapeFolderThumbs = new JCheckBox(getText("ui.fixedShapeThumbs"), false);
		JComboBox folderThumbAspectRatio = new JComboBox(new Object[]{
			new Item("0", getText("ui.sameAsThumbnails")),
			new Item("0.3333", "1:3"),
			new Item("0.5", "1:2"),
			new Item("0.5625", "9:16"),
			new Item("0.6667", "2:3"),
			new Item("0.75", "3:4"),
			new Item("0.9333", "5:6"),
			new Item("1", "1:1"),
			new Item("1.2", "6:5"),
			new Item("1.3333", "4:3"),
			new Item("1.5", "3:2"),
			new Item("1.7778", "16:9"),
			new Item("2", "2:1"),
			new Item("3", "3:1"),
			new Item("4", "4:1")
		});
		
		// Folder image size
		
		JTextField folderImageSize = new JTextField("1800x600");
		
		private void setFolderImageSize() {
			if (!isSkinReady()) {
				return;
			}
			
			int pw = (fullWidth.isSelected()? 1880 : getPageWidth()) + 40;
			
			folderImageSize.setText(pw + "x" + Math.round(pw / 3));
			//JAlbum.logger.log(Level.FINE, "Folder image size is set to {0}px", folderImageSize.getText());
		};
																		
		// Folder thumb size
		
		JTextField folderThumbSize = new JTextField("600x420");
		
		private void setFolderThumbSize() {
			String	tl = getSelectedValue(thumbLayout);
			int		pw = (int)Math.round(getPageWidth() * 1.15);
			Integer	cols = getSelectedValueAsInt(folderColumns);
			Double	ar = getSelectedValueAsDouble(folderThumbAspectRatio);
			
			if (cols == null || ar == null) {
				// Error
				return;
			}
			
			if (ar < 0.1) {
				// "Same as thumb..."
				if (!tl.equals("fixgrid")) {
					ar = 1.333333;
				} else {
					ar = getThumbAspectRatio();
					if (ar == null) {
						return;
					}
				}
			}
			
			folderThumbSize.setText(Math.round(pw / cols) + "x" + Math.round(pw / ar / cols));

		};
		JComboBox showPages = new JComboBox(new Object[]{
			new Item("none", getText("ui.none")), 
			new Item("link", getText("ui.linkOnly")), 
			new Item("excerpt", getText("ui.withExcerpt")), 
			new Item("embedCustom", getText("ui.embedCustom")),
			new Item("embed", getText("ui.embedAll"))
		});

		// Weblocaitons
		JComboBox showWeblocations = new JComboBox(new Object[]{
			new Item("none", getText("ui.none")), 
			new Item("link", getText("ui.linkOnly")), 
			new Item("thumb", getText("ui.showThumbnail")), 
			new Item("embed", getText("ui.embedFull"))
		});

		// Thumbnails
		JComboBox thumbLayout = new JComboBox(new Object[]{
			new Item("fixgrid", getText("ui.fixedShapeGrid")),
			new Item("grid", getText("ui.grid")),
			new Item("justified", getText("ui.justified")),
			new Item("vertical", getText("ui.cards") + " (" + getText("ui.vertical") + ")")
		});

		JComboBox thumbAspectRatio = new JComboBox(new Object[] {
			new Item("0.5", "1 : 2"), 
			new Item("0.6667", "2 : 3"), 
			new Item("0.75", "3 : 4"), 
			new Item("0.8", "4 : 5"), 
			new Item("0.8667", "5 : 6"), 
			new Item("1.0", "1 : 1"), 
			new Item("1.2", "6 : 5"), 
			new Item("1.25", "5 : 4"), 
			new Item("1.3333", "4 : 3"), 
			new Item("1.5", "3 : 2"),
			new Item("1.7778", "16 : 9"),
			new Item("2.0", "2 : 1"),
			new Item("3.0", "3 : 1")
		});

		JComboBox thumbGap = new JComboBox(new Object[] {
			new Item("none", getText("ui.none")),
			new Item("tiny", getText("ui.thin")),
			new Item("small", getText("ui.small")),
			new Item("medium", getText("ui.medium")),
			new Item("large", getText("ui.large"))
		});
		
		JComboBox columns = new JComboBox(new Object[]{
			2,
			3,
			4,
			5,
			6
		});
		
		JComboBox thumbCaptionPresets = new JComboBox(new Object[] {
			new Item("", getText("ui.empty")),
			new Item("<div class=\"title\">${fileTitle}</div>", getText("ui.title")),
			new Item("<div class=\"title\">${fileTitle}</div>\n<div class=\"comment\">${comment}</div>", getText("ui.title") + " + " + getText("ui.comment")),
			new Item("<div class=\"title\">${fileTitle}</div>\n<div class=\"comment\">${commentShort}</div>", getText("ui.title") + " + " + getText("ui.commentShort") + " (" + getText("ui.default") + ")"),
			new Item("<div class=\"date\">${originalDate}</div>\n<div class=\"title\">${fileTitle}</div>\n<div class=\"comment\">${commentShort}</div>", getText("ui.date") + " + " + getText("ui.title") + " + " + getText("ui.commentShort")),
			new Item("<div class=\"date\">${originalTime}</div>\n<div class=\"title\">${fileTitle}</div>\n<div class=\"comment\">${commentShort}</div>", getText("ui.time") + " + " + getText("ui.title") + " + " + getText("ui.commentShort")),
			new Item("<span class=\"nr\">${imageNum}</span>\n<div class=\"title\">${fileTitle}</div>\n<div class=\"comment\">${commentShort}</div>", getText("ui.number") + " + " + getText("ui.title") + " + " + getText("ui.commentShort")),
			new Item("<div class=\"title\">${fileLabel}</div>", getText("ui.fileLabel")),
			new Item("<div class=\"title\">${fileLabel}</div>\n<div class=\"comment\">${comment}</div>", getText("ui.fileLabel") + " + " + getText("ui.comment")),
			new Item("<div class=\"title\">${fileLabel}</div>\n<div class=\"comment\">${commentShort}</div>", getText("ui.fileLabel") + " + " + getText("ui.commentShort")),
			new Item("<span class=\"nr\">${imageNum}</span>\n<div class=\"title\">${fileLabel}</div>\n<div class=\"comment\">${commentShort}</div>", getText("ui.number") + " + " + getText("ui.fileLabel") + " + " + getText("ui.commentShort")),
			new Item(CUSTOM, "[ " + getText("ui.custom") + " ]")
		});
		JTextArea thumbCaptionTemplate = new JSmartTextArea(4, 30);
			
		// These need to be global to watch independent slide pages requirement
		JCheckBox shareFacebook = new JCheckBox("Facebook");
		JCheckBox shareThreads = new JCheckBox("Threads");
		JCheckBox shareBluesky = new JCheckBox("Bluesky");
		JCheckBox writeSitemapXml = new JCheckBox(getText("ui.createSitemapXml"));
		JCheckBox sitemapIncludeSlides = new JCheckBox(getText("ui.includeSlides"), true);
		
		JCheckBox mapInSections = new JCheckBox(getText("ui.showMapInMainContent"), false);
		JCheckBox mapInSidebar = new JCheckBox(getText("ui.showMapInSidebar"), false);
		JComboBox _mapPlacement = new JComboBox(new Object[] {
			new Item("", getText("ui.noMapOnIndexPage")),
			new Item("sections", getText("ui.mainContent")),
			new Item("sidebar", getText("ui.sidebar"))
		});
		JCheckBox showMapBtn = new JCheckBox(getText("ui.map"));
		private JCheckBox _showMapBtn = new JCheckBox(getText("ui.lightbox"));
		
		private void setMakeSlides() {
			boolean ov = engine.isSlides(),
					nv = shareFacebook.isSelected() || (writeSitemapXml.isSelected() && sitemapIncludeSlides.isSelected());
			
			if (ov != nv) {
				try {
					window.ui2Engine();
					engine.setSlides(nv);
					window.engine2UI();
				} catch (Exception ex) {
					throw new RuntimeException(ex);
				}
			}
		};
		
		private String lastThumbDimSuggestion = "";
		
		private int getPageWidth() {
			String	pw = ((Item)maxPageWidth.getSelectedItem()).value.toString();
			return pw.equals("none")? 1880 : Math.round(Float.parseFloat(pw.replace("rem","")) * 16);
		}
		
		// Calculating proper thumb dimensions for the different thumb layouts
		
		private int getThumbGap() {
			switch (((Item)thumbGap.getSelectedItem()).value.toString()) {
				case "tiny":
					return 2;
				case "small":
					return 6;
				case "medium":
					return 12;
				case "large":
					return 20;
			}
			return 0;
		}
		
		private int getThumbPad() {
			switch (((Item)boxPadding.getSelectedItem()).value.toString()) {	
				case "small":
					return 4;
				case "medium":
					return 8;
				case "large":
					return 12;
				case "x-large":
					return 20;
			}
			return 0;
		}
		
		// Returns the thumb width
		private int getMaxThumbWidth() {
			
			return Integer.parseInt(engine.getThumbSize().split("x")[0]);
		}
		
		// Returns the thumb height
		private int getMaxThumbHeight() {
			
			return Integer.parseInt(engine.getThumbSize().split("x")[1]);
		}
		
		// Checks if number is within range of ref
		private boolean inRange(double n, double ref, double range) {
			
			return (n >= ref * (1.0 - range)) && (n < ref * (1.0 + range));
		}
		
		// Calculating proper thumb dimensions for the different thumb layouts
		private void setThumbDimensions() {
			
			if (!isSkinReady()) {
				return;
			}
			
			String	layout = ((Item)thumbLayout.getSelectedItem()).value.toString(),
					cdim = engine.getThumbSize(),
					ndim;

			int		cw = getPageWidth(),
					cols = Integer.parseInt(columns.getSelectedItem().toString()),
					mtw = getMaxThumbWidth(),
					mth = getMaxThumbHeight(),
					tg = getThumbGap(),
					tp = getThumbPad(),
					tb = (int)boxBorderWidth.getValue(),
					tw,
					w,
					h;
			
			double	tar;

			//w = (int) Math.ceil((pageWidth - 30) / (layout.equals("horizontal")? Math.max(cols - 1, 1.5) : cols));
			tw = (int)Math.round(((cw - tg * (cols - 1)) / cols) - 2 * (tp + tb));
			//System.out.println("Math.round(((" + cw + " - " + tg + " * (" + cols + " - 1)) / " + cols + ") - 2 * (" + tp + " + " + tb + ")) = " + tw);

			switch (layout) {
				case "horizontal":
					w = (int)Math.round(tw * 1.66667);
					h = (int)Math.round(tw * 0.75);
					break;
					
				case "justified":
					w = (int)Math.round(tw * 1.6);
					h = (int)Math.round(tw * 0.8);
					/*
					w = (int)Math.round(tw * 1.4142);
					h = (int)Math.round(tw * 0.7071);
					*/
					break;
					
				case "vertical":
					w = tw;
					h = (int)Math.round(tw * 1.5);
					break;
					
				default:
					w = tw;
					h = (int)Math.round(tw / Float.parseFloat((((Item)thumbAspectRatio.getSelectedItem()).value).toString()));
			}
			
			ndim = w + "x" + h;
			
			if (!lastThumbDimSuggestion.equals(ndim) &&					// Already set
				!(inRange(w, mtw, 0.05) && inRange(h, mth, 0.05) &&		// Width and Height is out of 5% range
				inRange(w / h, mtw / mth, 0.01))) {						// Aspect ratio is out of 1% range
				
				try {
					window.ui2Engine();
					engine.setThumbSize(ndim);
					JAlbum.logger.log(Level.FINE, "Thumb size has changed from {0}px to {1}px.", new Object[]{cdim, ndim});
					window.engine2UI();
				} catch (ParameterException ex) {
					throw new RuntimeException(ex);
				}
				
				lastThumbDimSuggestion = ndim;
			} else {
				JAlbum.logger.log(Level.FINE, "Thumb size change from {0}px to {1}px is within range.", new Object[]{cdim, ndim});
			}
		};
				
		JComboBox imageCaptionPresets = new JComboBox(new Object[] {
			new Item("", getText("ui.empty")),
			new Item("${originalTime}", getText("ui.time")),
			new Item("<div class=\"title\">${fileTitle}</div>", getText("ui.title")),
			new Item("<div class=\"title\">${fileTitle}</div>\n<div class=\"comment\">${comment}</div>", getText("ui.title") + " + " + getText("ui.comment") + " (" + getText("ui.default") + ")"),
			new Item("<div class=\"title\">${fileTitle}</div>\n<div class=\"comment\">${commentShort}</div>", getText("ui.title") + " + " + getText("ui.commentShort")),
			new Item("<span>${originalTime}</span>\n<div class=\"title\">${fileTitle}</div>\n<div class=\"comment\">${comment}</div>", getText("ui.time") + " + " + getText("ui.title") + " + " + getText("ui.comment")),
			new Item("<div class=\"title\">${fileLabel}</div>", getText("ui.fileLabel")),
			new Item("<div class=\"title\">${fileLabel}</div>\n<div class=\"comment\">${comment}</div>", getText("ui.fileLabel") + " + " + getText("ui.comment")),
			new Item("<div class=\"title\">${fileLabel}</div>\n<div class=\"comment\">${commentShort}</div>", getText("ui.fileLabel") + " + " + getText("ui.commentShort")),
			new Item(CUSTOM, "[ " + getText("ui.custom") + " ]")	
		});
		JTextArea imageCaptionTemplate = new JSmartTextArea(4, 30);
						
		// Refreshing preview
		
		private WebViewBrowser sitePreview = new WebViewBrowser();
		
		private void refreshPreview() {
			String		html,
						patternPath,
						_heroType = getSelectedValue(heroType),
						_pageColor = getCssColor(pageColor),
						_topmenuColor = getCssColor(topmenuColor),
						_headerColor = getCssColor(headerColor),
						_backgroundColor = getCssColor(backgroundColor),
						_footerColor = getCssColor(footerColor),
						_headlineFont = getSelectedValue(headlineFont),
						topmenuTextColor = getLegibleColor(topmenuColor, textColor, 0.7),
						buttonTextColor = getLegibleColor(linkColor, textColor, 0.7),
						headerTextColor = "#ffffff",
						topmenuBorder = "none",
						footerBorder = "none",
						_thumbLayout = getSelectedValue(thumbLayout),
						_boxBorderColor = getCssColor(boxBorderColor),
						_boxShadow = "none";
			int			_thumbGap = thumbGap.getSelectedIndex(),
						_boxBorderWidth = (int)boxBorderWidth.getValue(),
						_boxPadding = boxPadding.getSelectedIndex() * 2;
			boolean		_fullWidth = fullWidth.isSelected(),
						isLightBg = isLightColor(pageColor, backgroundColor),
						isScript =	_headlineFont.contains("Script") || 
									_headlineFont.contains("Decor") || 
									"Grand Hotel,Great Vibes,Lobster,Lobster Two,Sofia".contains(_headlineFont);
			
			try {
				patternPath = (new File(skinDirectory, "patterns/background/" + (isLightBg? "light" : "dark")).toURI().toURL()).toString();
			} catch (MalformedURLException ex) {
				Logger.getLogger(Gui.class.getName()).log(Level.SEVERE, null, ex);
				patternPath = null;
			}				
			
			if (boxDropShadow.isSelected()) {
				_boxShadow = isLightBg? "1px 2px 6px rgba(0,0,0,0.2)" : "1px 2px 6px rgba(0,0,0,0.6)";
			}
			
			if (_headerColor.equals(_topmenuColor) && _heroType.equals("flatColor")) {
				topmenuBorder = _boxBorderWidth + "px solid " + (_pageColor.equals(_topmenuColor)? _boxBorderColor : _pageColor);
			}
			
			if (_footerColor.equals(_backgroundColor)) {
				footerBorder = _boxBorderWidth + "px solid " + (_footerColor.equals(_pageColor)? _boxBorderColor : _pageColor);
			}
							
			if (_heroType.equals("flatColor") || _heroType.equals("gradient")) {
				if (!_fullWidth) {
					headerTextColor = getLegibleColor(backgroundColor, headerColor, textColor);
				} else {
					if (_heroType.equals("gradient")) {
						headerTextColor = getLegibleColor(pageColor, backgroundColor, textColor);
					} else {
						//System.out.println("flatColor(" + getCssColor(pageColor) + ", " + getCssColor(headerColor) + ") => " + getCssColor(flatColor(pageColor, headerColor)));
						//System.out.println("getLegibleColor(" + getCssColor(pageColor) + ", " + getCssColor(headerColor) + ", " + getCssColor(textColor) + ") => " + getLegibleColor(pageColor, headerColor, textColor));
						headerTextColor = getLegibleColor(pageColor, headerColor, textColor);
					}
				}
			}
			
			//System.out.println("\nTemplate:\n\n" + designTemplate);
			
			html = designTemplate
				.replace("{fullWidth}", fullWidth.isSelected()? "full-width" : "page-wide")
				.replace("{sidebarOrder}", sidebarPosition.getSelectedIndex() * 2 + "")
				.replace("{pageBgColor}", _pageColor)
				.replace("{mainBgColor}", _backgroundColor)
				.replace("{patternPath}", (patternPath != null)? ("url(\"" + patternPath + "/" + getSelectedValue(backgroundPattern) + "\")") : "none")
				.replace("{topmenuBgColor}", _topmenuColor)
				.replace("{topmenuTextColor}", topmenuTextColor)
				.replace("{headerTextColor}", headerTextColor)
				.replace("{headerBgColor}", _headerColor)
				.replace("{footerTextColor}", getLegibleColor(footerColor, textColor, 0.5))
				.replace("{footerBgColor}", _footerColor)
				.replace("{textColor}", textColor.toString())
				.replace("{buttonTextColor}", buttonTextColor)
				.replace("{linkColor}", linkColor.toString())
				.replace("{hoverColor}", hoverColor.toString())
				.replace("{thumbLayout}", _thumbLayout.equals("justified")? "horizontal" : _thumbLayout)
				.replace("{thumbPadding}", _boxPadding + "")
				.replace("{thumbGap}", _thumbGap + "")
				.replace("{heroType}", _heroType + (heroImageDesaturate.isSelected()? " desaturate" : ""))
				.replace("{headClass}", getSelectedValue(titlePlacement) + ((_heroType.equals("albumImage") || _heroType.equals("folderImage"))? " hasbgimage" : ""))
				.replace("{sliderType}", getSelectedValue(sliderType))
				.replace("{boxHeadTT}", isScript? "none" : "uppercase")
				.replace("{boxPadding}", 2 + _boxPadding * 2 + "")
				.replace("{boxBorderWidth}", _boxBorderWidth + "")
				.replace("{boxTextColor}", getLegibleColor(boxBackgroundColor, textColor))
				.replace("{boxBgColor}", getCssColor(boxBackgroundColor))
				.replace("{boxBorderColor}", _boxBorderColor)
				.replace("{boxShadow}", _boxShadow)
				.replace("{topmenuBorder}", topmenuBorder)
				.replace("{footerBorder}", footerBorder)
				.replace("{folderTextColor}", getLegibleColor(folderColor, textColor))
				.replace("{folderBgColor}", getCssColor(folderColor));
							
			html = getFonts(html, getSelectedValue(fontFamily), (headlineFont.getSelectedIndex() == 0)? "" : getSelectedValue(headlineFont));
			//System.out.println("\nPreview reload triggered by: " + src + "\n\n" + html);
			//System.out.println("\nPage color: " + _pageColor + "\nTop menu color: " + _topmenuColor + "\nHeader color: " + _headerColor + "\nFooter color: " + _footerColor);
			
			final String html1 = html;
			
			SwingUtilities.invokeLater(() -> {
				sitePreview.loadContent(html1);
			});
			
			//writeFile("D:/Temp/design-preview.html", html);
		}
		
		private PropertyChangeListener setupMonitors = pce -> {
			
			StateMonitor.monitoring(fontSuggestions).onChange(src -> {
				if (isSkinReady() && src != null) {
					setFontBoxes();
				}
			});

			StateMonitor.monitoring(extraSizes).onChange(src -> {
				//System.out.println("Extra sizes changed: " + lastExtraSizes + " -> " + extraSizes.getText());
				extraSizesChanged.setSelected(lastExtraSizes == null || !lastExtraSizes.equals(extraSizes.getText()));
			});
			
			StateMonitor.monitoring(
					styleName,
					maxPageWidth, 
					columns,
					boxBorderWidth,
					thumbGap,
					thumbLayout,
					thumbAspectRatio).onChange(src -> {
				
				if (isSkinReady() && src != null) {
					JAlbum.logger.log(Level.FINE, "stateMonitor: \"{0}\" changed", getSelectedValue(src));
					setThumbDimensions();
				}
				
			});
		
			StateMonitor.monitoring(maxPageWidth).onChange(w -> {
				setFolderImageSize();
				setFolderThumbSize();
			});

			StateMonitor.monitoring(fullWidth).onChange(w -> {
				setFolderImageSize();
			});
			
			// Separate slide page requirement
			StateMonitor.monitoring(
					shareFacebook,
					shareThreads,
					shareBluesky,
					writeSitemapXml,
					sitemapIncludeSlides).onChange(src -> {
				setMakeSlides();
			});

			// Preview update
			StateMonitor.monitoring(
					styleName,
					sidebarPosition,
					fullWidth,
					pageColor,
					backgroundPattern,
					topmenuColor,
					headerColor,
					backgroundColor,
					footerColor,
					textColor,
					linkColor,
					hoverColor,
					boxBackgroundColor,
					boxBorderColor,
					boxDropShadow,
					boxBorderWidth,
					boxPadding,
					folderColor,
					fontFamily, 
					headlineFont,
					heroType,
					heroImageDesaturate,
					titlePlacement,
					sliderType,
					thumbLayout,
					thumbGap
				).onChange((Object src) -> {

				if (isSkinReady() && src != null) {
					refreshPreview();
				}
			});
				
			// Initial setup
			setThumbDimensions();
			setFolderImageSize();
			setFolderThumbSize();
			
			// Initial preview rendering
			
			refreshPreview();
		};
		
		
		/*	---------------------------------------------------------------
									Design
			--------------------------------------------------------------- */
		
		ControlPanel colors = new ControlPanel(new BorderLayout(0, 0)) {
			
			//	Colors

			ControlPanel pageColors = new ControlPanel() {

				{
					pageColor.getTextComponent().setFont(mono);
					topmenuColor.getTextComponent().setFont(mono);
					headerColor.getTextComponent().setFont(mono);
					backgroundColor.getTextComponent().setFont(mono);
					footerColor.getTextComponent().setFont(mono);
					textColor.getTextComponent().setFont(mono);
					reverseTextColor.getTextComponent().setFont(mono);
					linkColor.getTextComponent().setFont(mono);
					hoverColor.getTextComponent().setFont(mono);
					/*maxPageWidth.addActionListener((ActionEvent e) -> {
						setThumbDimensions();
						setFolderImageSize();
						setFolderThumbSize();
					});
					*/

					add("", new JLabelFor(getText("ui.pageColor"), pageColor));
					add("tab", pageColor);
					add("tab", pageColor.getTextComponent());
					add("br", new JLabelFor(getText("ui.backgroundPattern"), backgroundPattern));
					add("tab", backgroundPattern);
					add("br", new JLabelFor(getText("ui.topmenu"), topmenuColor));
					add("tab", topmenuColor);
					add("tab", topmenuColor.getTextComponent());
					add("br", new JLabelFor(getText("ui.header"), headerColor));
					add("tab", headerColor);
					add("tab", headerColor.getTextComponent());
					add("br", new JLabelFor(getText("ui.mainContent"), backgroundColor));
					add("tab", backgroundColor);
					add("tab", backgroundColor.getTextComponent());
					add("br", new JLabelFor(getText("ui.footer"), footerColor));
					add("tab", footerColor);
					add("tab", footerColor.getTextComponent());
					add("br", new JLabelFor(getText("ui.textColor"), textColor));
					add("tab", textColor);
					add("tab", textColor.getTextComponent());
					add("br", new JLabelFor(getText("ui.reverseTextColor"), reverseTextColor));
					add("tab", reverseTextColor);
					add("tab", reverseTextColor.getTextComponent());
					add("br", new JLabelFor(getText("ui.linkColor"), linkColor));
					add("tab", linkColor);
					add("tab", linkColor.getTextComponent());
					add("br", new JLabelFor(getText("ui.hoverColor"), hoverColor));
					add("tab", hoverColor);
					add("tab", hoverColor.getTextComponent());
				}
			};
				
			// Boxes

			ControlPanel boxes = new ControlPanel() {

				{
					boxBackgroundColor.getTextComponent().setFont(mono);
					boxBorderColor.getTextComponent().setFont(mono);
					folderColor.getTextComponent().setFont(mono);

					add("", new JLabelFor(getText("ui.backgroundColor"), boxBackgroundColor));
					add("tab", boxBackgroundColor);
					add("", boxBackgroundColor.getTextComponent());
					add("br", new JLabelFor(getText("ui.padding"), boxPadding));
					add("tab", boxPadding);
					add("br", new JLabelFor(getText("ui.borderColor"), boxBorderColor));
					add("tab", boxBorderColor);
					add("", boxBorderColor.getTextComponent());
					add("br", new JLabelFor(getText("ui.borderWidth"), boxBorderWidth));
					add("tab", boxBorderWidth);
					add("tab", boxDropShadow);
					add("br", new JLabelFor(getText("ui.folderColor"), folderColor));
					add("tab", folderColor);
					add("", folderColor.getTextComponent());
				}
			};
			
			// Lightbox design
			
			ControlPanel lightboxDesign = new ControlPanel() {
				
				JColorSelector lightboxColor = new JAlphaColorSelector(getText("ui.backgroundColor"), new JSmartTextField(10));
				JColorSelector lightboxBorderColor = new JAlphaColorSelector(getText("ui.borderColor"), new JSmartTextField(10));
				JCheckBox lightboxDropShadow = new JCheckBox(getText("ui.dropShadow"), false);
				JSpinner lightboxBorderWidth = new JSpinner(new SpinnerNumberModel(0, 0, 40, 1));
				
				{		
					lightboxColor.getTextComponent().setFont(mono);
					lightboxBorderColor.getTextComponent().setFont(mono);
					
					add("br", new JLabelFor(getText("ui.backgroundColor"), lightboxColor));
					add("tab", lightboxColor);
					add("tab", lightboxColor.getTextComponent());
					add("br", new JLabelFor(getText("ui.borderColor"), lightboxBorderColor));
					add("tab", lightboxBorderColor);
					add("tab", lightboxBorderColor.getTextComponent());
					add("br", new JLabelFor(getText("ui.borderWidth"), lightboxBorderWidth));
					add("tab", lightboxBorderWidth);
					add("", lightboxDropShadow);					
				}
			};
			
			JTabbedPane pageDesignTabs = new JTabbedPane() {
				
				{
					addTab(getText("ui.site"), pageColors);
					addTab(getText("ui.boxes"), boxes);
					addTab(getText("ui.lightbox"), lightboxDesign);
				}
			};
					
			{
				
				((BorderLayout)(getLayout())).setVgap(0);
				((BorderLayout)(getLayout())).setHgap(0);
				
				pageDesignTabs.setPreferredSize(new Dimension(uiWidth, uiHeight));
				pageDesignTabs.setMaximumSize(new Dimension(uiWidth + 80, uiHeight));
				
				add(pageDesignTabs);
			}
		};
		
		//	Layout panel
		
		ControlPanel layout = new ControlPanel() {
			
			JComboBox<Item> language = new JComboBox<Item>() {
				{
					setModel(Languages.modelFrom(new File(skinDirectory, "texts")));
					insertItemAt(new Item("jalbum", "[ " + getText("ui.jAlbumPreference") + " ]"), 0);
					setSelectedIndex(0);
				}
			};
					
			JComboBox blogDepth = new JComboBox(new Object[] {
				new Item("0", getText("ui.noBlogFeatures")),
				new Item("1", getText("ui.topLevelOnly")),
				new Item("2", "2 " + getText("ui.levels")),
				new Item("3", "3 " + getText("ui.levels")),
				new Item("4", "4 " + getText("ui.levels")),
				new Item("5", "5 " + getText("ui.levels"))
			});
			
			JLabel blogInfo = new JLabel(infoIcon);

			JCheckBox scrollToTopButton = new JCheckBox(getText("ui.scrollToTopButton"));	

			JComboBox modalWindowsTheme = new JComboBox(new Object[] {
				new Item("auto", getText("ui.auto")),
				new Item("light", getText("ui.light")),
				new Item("dark", getText("ui.dark"))
			});

			JComboBox iconStyle = new JComboBox(new Object[] {
				new Item("thin", getText("ui.thin")),
				new Item("fat", getText("ui.fat"))
			});
			
			{
				language.setToolTipText(getText("ui.languageInfo"));
				blogInfo.addMouseListener(new MouseAdapter() {  
					@Override
					public void mouseReleased(MouseEvent e) {
						JOptionPane.showMessageDialog(window, getText("ui.blogStyleInfo"), "Info", JOptionPane.INFORMATION_MESSAGE);
				}});
				
				add("", new JLabelFor(getText("ui.language"), language));
				add("tab", language);
				add("br", new JLabelFor(getText("ui.blogDepth"), blogDepth));
				add("tab", blogDepth);
				add(" ", blogInfo);
				add("br", new JLabelFor(getText("ui.maxPageWidth"), maxPageWidth));
				add("tab", maxPageWidth);
				add("br", new JLabel(getText("ui.sidebarPosition")));
				add("tab", sidebarPosition);
				add("tab", sidebarSticky);
				add("br", new JLabelFor(getText("ui.modalWindowsTheme"), modalWindowsTheme));
				add("tab", modalWindowsTheme);
				add("br", new JLabel(getText("ui.iconStyle")));
				add("tab", iconStyle);
				add("br", fullWidth);
				add("br", scrollToTopButton);

			}
		};
		
		// Typography
		
		ControlPanel typography = new ControlPanel() {
			/*
			JComboBox zoom = new JComboBox(new Object[] {
				new Item("0.75", "75%"),
				new Item("0.8125", "81%"),
				new Item("0.875", "88%"),
				new Item("0.9375", "94%"),
				new Item("1.0", "100%"),
				new Item("1.0625", "106%"),
				new Item("1.125", "112%"),
				new Item("1.1875", "118%"),
				new Item("1.25", "125%")					
			});
			*/
			JComboBox baseFontSize = new JComboBox(new Object[] {
				new Item("11", getText("ui.smallest") + " (11px)"),
				new Item("12", getText("ui.tiny") + " (12px)"),
				new Item("13", getText("ui.small") + " (13px)"),
				new Item("14", getText("ui.smaller") + " (14px)"),
				new Item("15", getText("ui.normal") + " (15px)"),
				new Item("16", getText("ui.normal") + " (16px)"),
				new Item("17", getText("ui.larger") + " (17px)"),
				new Item("18", getText("ui.large") + " (18px)"),
				new Item("20", getText("ui.largest") + " (20px)"),
			});
								
			JComboBox headlineSize = new JComboBox(new Object[] {
				new Item("1.2", "120%"),
				new Item("1.6", "160%"),
				new Item("2.0", "200%"),
				new Item("2.4", "240%"),
				new Item("2.8", "280%"),
				new Item("3.2", "320%"),
				new Item("3.6", "360%"),
				new Item("4.0", "400%"),
				new Item("5.0", "500%")
			});
				
			{
				headlineFont.setEditable(true);

				add(new JLabelFor(getText("ui.fontFamily"), fontFamily));
				add("tab", fontFamily);
				add("br", new JLabelFor(getText("ui.baseFontSize"), baseFontSize));
				add("tab", baseFontSize);
				add("br", new JLabelFor(getText("ui.headlineFont"), headlineFont));
				add("tab", headlineFont);
				add("br", new JLabelFor(getText("ui.headlineFontSize"), headlineSize));
				add("tab", headlineSize);
				add("br", new JLabelFor(getText("ui.pairingSuggestions"), fontSuggestions));
				add("tab", fontSuggestions);
				//add("tab", new JLinkLabel(skinDirectory + File.separator + "font-samples.html", getText("ui.fontSamples")));
			}
		};
			
		// Rating
		
		ControlPanel rating = new ControlPanel() {

			JCheckBox useRating = new JCheckBox(getText("ui.displayRating"));
			JCheckBox visitorRating = new JCheckBox(getText("ui.allowVisitorRatings"));
			JCheckBox useJalbumRating = new JCheckBox(getText("ui.useJalbumRatings"));
			
			{
				useRating.setToolTipText(getText("ui.displayRatingInfo"));
				visitorRating.setToolTipText(getText("ui.allowVisitorRatingsInfo"));
				useJalbumRating.setToolTipText(getText("ui.useJalbumRatingsInfo"));
				ComponentUtilities.whenSelectedEnable(useRating, new JComponent[]{ visitorRating, useJalbumRating});
				
				add("br", useRating);
				add("br", useJalbumRating);
				add("br", visitorRating);
			}
		};
				
		// Mark new files
		
		ControlPanel markNew = new ControlPanel() {

			JCheckBox markFilesNew = new JCheckBox(getText("ui.markFilesNew"));	
			JTextField newDaysCount = new JSmartTextField("60", 3);
			JComboBox newDaysRef = new JComboBox(new Object[] {
				new Item("dateTaken", getText("ui.dateTaken")),
				new Item("fileModified", getText("ui.fileModified")),
				new Item("added", getText("ui.addedToAlbum"))
			});
			JComboBox newDaysMark = new JComboBox(new Object[] {
				new Item("icon", getText("ui.icon")),
				new Item("text", getText("ui.text"))
			});
			JTextField newDaysText = new JSmartTextField(getText("new"), 6);
			JLabel newIcon = new JLabel(icon("new-fill"));
			
			{
				newDaysCount.setToolTipText(getText("ui.newDaysCountInfo"));
				newDaysMark.addItemListener(listener -> {
					int i = newDaysMark.getSelectedIndex();
					newDaysText.setVisible(i == 1);
					newIcon.setVisible(i == 0);
				});
				newDaysText.setVisible(false);
				ComponentUtilities.whenSelectedEnable(markFilesNew, new JComponent[]{ newDaysCount, newDaysRef, newDaysMark, newDaysText, newIcon });
				
				add("", markFilesNew);
				add("", newDaysCount);
				add(new JLabel(getText("ui.daysOld")));
				add("br", new JLabel(getText("ui.reference")));
				add("tab", newDaysRef);
				add("br", new JLabel(getText("ui.marker")));
				add("tab", newDaysMark);
				add(newIcon);
				add(newDaysText);
			}
		};
		
		//	Extra
		
		ControlPanel extra = new ControlPanel() {
			
			{				
				//extraSizes.setToolTipText(getText("ui.extraSizesInfo"));

				add("br", new JLabel(getText("ui.extraSizes")));
				add("hfill", extraSizes);
				add("br", new JLabel("<html><i>" + getText("ui.extraSizesInfo") + "</i></html>"));
				
				//putClientProperty("helpPage", helpRoot + "Site/Extra");
			}
		};

		//	Social
		
		ControlPanel social = new ControlPanel() {
			
			ControlPanel shares = new ControlPanel(getText("ui.shareButtonFor")) {

				JCheckBox shareTwitter = new JCheckBox("X (Twitter)");
				JCheckBox sharePinterest = new JCheckBox("Pinterest");
				JCheckBox shareLinkedin = new JCheckBox("LinkedIn");
				JCheckBox shareDigg = new JCheckBox("Digg");
				JCheckBox shareReddit = new JCheckBox("Reddit");
				JCheckBox shareTumblr = new JCheckBox("Tumblr");

				{
					setLayout(new RiverLayout(4, 5));
					add(new JLabel(icon("facebook")));
					add(shareFacebook);
					add("br", new JLabel(icon("threads")));
					add(shareThreads);
					add("br", new JLabel(icon("bluesky")));
					add(shareBluesky);
					add("br", new JLabel(icon("x")));
					add(shareTwitter);
					add("br", new JLabel(icon("reddit")));
					add(shareReddit);
					add("br", new JLabel(icon("pinterest")));
					add(sharePinterest);
					add("br", new JLabel(icon("tumblr")));
					add(shareTumblr);
					add("br", new JLabel(icon("linkedin")));
					add(shareLinkedin);
					add("br", new JLabel(icon("digg")));
					add(shareDigg);
				}
			};

			ControlPanel links = new ControlPanel() {
				
				JCheckBox shareEmail = new JCheckBox("Email");
				JLabel emailSubjectLabel = new JLabel(getText("ui.subject"));
				JTextField emailSubject = new JSmartTextField(18);
				JTextArea emailBody = new JSmartTextArea(4,20);
				JScrollPane emailBodyPane = new JScrollPane(emailBody);
				JCheckBox shareLink = new JCheckBox(getText("ui.link"));
				
				{
					emailBody.setEditable(true);
					emailBody.setLineWrap(true);
					emailBody.setWrapStyleWord(true);
					emailBody.setToolTipText(getText("ui.emailBodyInfo"));
					emailBodyPane.setBorder(BorderFactory.createTitledBorder(getText("ui.body")));
					ComponentUtilities.whenSelectedEnable(shareEmail, new JComponent[]{emailSubjectLabel, emailSubject, emailBodyPane});
					
					add("", new JLabel(icon("email")));
					add("tab", shareEmail);
					add("br tab", emailSubjectLabel);
					add("", emailSubject);
					add("br tab", emailBodyPane);
					add("br", new JLabel(icon("link")));
					add("tab", shareLink);
				}
			};
			
			JTextField facebookAppId = new JSmartTextField(20);
			JLabel mandatoryInfo = new JLabel(mandatory);
			WrappableJLabel facebookAppIdInfo = new WrappableJLabel("<html><i>" + getText("ui.facebookAppIdInfo") + "</i></html>");
			
			{
				mandatoryInfo.setToolTipText(getText("ui.mandatory"));
				links.setBorder(emptyBorder);
				facebookAppIdInfo.setPreferredWidth(uiWidth - 140);
				
				add("", shares);
				add("hfill", links);
				add("br", new JLabelFor(getText("ui.facebookAppId"), facebookAppId));
				add(mandatoryInfo);
				add("tab", facebookAppId);
				add(new JLinkLabel("https://developers.facebook.com/apps", getText("ui.signUp")));
				add("br tab", facebookAppIdInfo);
			}

		};
		
		//	Map settings
		
		ControlPanel mapSettings = new ControlPanel() {

			JTextField indexMapLabel = new JTextField(24);
			
			JComboBox mapType = new JComboBox(new Object[]{
				new Item("roadmap", getText("ui.roadmap")),
				new Item("satellite", getText("ui.satellite")),
				new Item("hybrid", getText("ui.hybrid")),
				new Item("terrain", getText("ui.terrain"))
			});
			JSlider mapZoom = new JSlider(JSlider.HORIZONTAL, 1, 20, 18);
			JTextField googleApiKey = new JSmartTextField(32);
			JLabel googleApiInfo = new JLabel(infoIcon);
			WrappableJLabel note = new WrappableJLabel("<html><i>" + getText("ui.mapApiKeyInfo") + "</i></html>");
			//WrappableJLabel instr = new WrappableJLabel("<html><i>" + getText("ui.selectIndexOrLightbox") + "</i></html>");
			
			{
				_mapPlacement.addItemListener(listener -> {
					int mp = _mapPlacement.getSelectedIndex();
					mapInSections.setSelected(mp == 1);
					mapInSidebar.setSelected(mp == 2);
				});
				_showMapBtn.addItemListener(listener -> {
					showMapBtn.setSelected(_showMapBtn.isSelected());
				});
				mapZoom.setOrientation(JSlider.HORIZONTAL);
				mapZoom.setMinimum(0);
				mapZoom.setMaximum(20);
				mapZoom.setValue(18);
				mapZoom.setMajorTickSpacing(10);
				mapZoom.setMinorTickSpacing(1);
				mapZoom.setPaintTicks(true);
				mapZoom.setPaintLabels(true);
				mapZoom.setSnapToTicks(true);
				//instr.setPreferredWidth(uiWidth - 120);
				note.setPreferredWidth(uiWidth - 120);
				googleApiInfo.addMouseListener(new MouseAdapter() {  
					@Override
					public void mouseReleased(MouseEvent e) {
						JOptionPane.showMessageDialog(window, "<html><i>" + getText("ui.gooleMapsApiNote") + "</i></html>", getText("ui.format"), JOptionPane.INFORMATION_MESSAGE);
				}});
				
				add(new JLabel(getText("ui.placement")));
				add("tab", _mapPlacement);
				add("tab", _showMapBtn);
				add("br", new JLabel(getText("ui.boxTitle")));
				add("tab", indexMapLabel);
				add("br", new JLabelFor(getText("ui.initialView"), mapType));
				add("tab", mapType);
				add("br", new JLabelFor(getText("ui.initialZoom"), mapZoom));
				add("tab", mapZoom);
				add("br", new JLabelFor(getText("ui.googleApiKey"), googleApiKey));
				add("tab", googleApiKey);
				add("br tab", new JLinkLabel("https://console.developers.google.com/apis/credentials", getText("ui.createNew")));
				add("", googleApiInfo);
				add("br tab", note);
				//add("br tab", instr);
			}

		};
				
		// Audio clips

		ControlPanel audioClips = new ControlPanel() {

			JCheckBox useAudioClipButton = new JCheckBox(getText("ui.playAudioClips"), false);
			JSpinner audioClipVolume = new JSpinner(new SpinnerNumberModel(75, 1, 100, 1));
			JCheckBox copyAudioClips = new JCheckBox(getText("ui.copyAudioClips"));

			{
				useAudioClipButton.setToolTipText(getText("ui.playAudioClipsInfo"));
				audioClipVolume.setToolTipText(getText("ui.audioClipVolumeInfo"));
				copyAudioClips.setToolTipText(getText("ui.copyAudioClipsInfo") + " (" + getText("ui.oldMethod") + ")");
				
				add("", useAudioClipButton);
				add("br", new JLabel(getText("ui.initialVolume")));
				add("", audioClipVolume);
				add("", new JLabel("%"));
				add("br", copyAudioClips);
				
				//putClientProperty("helpPage", helpRoot + "/Site/Sound_clips");
			}
		};
		
		//	Background music
		
		ControlPanel backgroundMusic = new ControlPanel() {
			
			JPlaylist backgroundAudio = new JPlaylist();
			JCheckBox backgroundAudioAutoPlay = new JCheckBox(getText("ui.autoStart"));
			JSpinner backgroundAudioVolume = new JSpinner(new SpinnerNumberModel(25, 1, 100, 1));
			JCheckBox backgroundAudioSlideshowControl = new JCheckBox(getText("ui.slideshowControl"));
			JCheckBox muteBackgroundAudio = new JCheckBox(getText("ui.muteBackgroundAudio"), true);
			JCheckBox backgroundAudioLoop = new JCheckBox(getText("ui.loop"));
			JCheckBox backgroundAudioRetainPosition = new JCheckBox(getText("ui.retainPosition"), true);
			WrappableJLabel autostartNotice = new WrappableJLabel("<html><i>" + getText("ui.autostartNotice") + "</i></html>");
			
			{
				backgroundAudioSlideshowControl.setToolTipText(getText("ui.slideshowControlInfo"));
				muteBackgroundAudio.setToolTipText(getText("ui.muteBackgroundAudioInfo"));
				backgroundAudioRetainPosition.setToolTipText(getText("ui.retainPositionInfo"));
				autostartNotice.setPreferredWidth(uiWidth - 40);
				
				add("br hfill", backgroundAudio);
				add("br", new JLabel(getText("ui.initialVolume")));
				add("", backgroundAudioVolume);
				add("", new JLabel("%"));
				add("br", backgroundAudioAutoPlay);
				add("tab", backgroundAudioLoop);
				add("br", backgroundAudioRetainPosition);
				add("br", backgroundAudioSlideshowControl);
				add("br", muteBackgroundAudio);
				add("br", new JLabel(infoIcon));
				add("", autostartNotice);
				
				//putClientProperty("helpPage", helpRoot + "Site/Background_music");
			}
		};
		

		//	Top bar
		
		ControlPanel topbarTab = new ControlPanel() {
			/*
			@Override
			public void importVariables(Map<String, Object> vars) {
				Object stn = vars.get("showTopNavigation");
				// Upgrading from 1.x
				if (stn != null) {
					if (vars.get("topNavigationIncludeFolders") == null) {
						vars.put("topNavigationIncludeFolders", stn);
						vars.put("topNavigationIncludePages", stn);
						vars.put("topNavigationIncludeWebLocations", stn);
					}
					vars.put("topNavigationPlacement", "headerRight");
					vars.put("heroType", "flatColor");
				}
			};
			*/
			ControlPanel logo = new ControlPanel(getText("ui.logo")) {

				JTextField logoName = new JSmartTextField(24);
				JButton selectLogo = new JButton(getText("ui.select"));

				{
					selectLogo.addActionListener(new ActionListener() { 
						@Override
						public void actionPerformed(ActionEvent e) {
							getFileToRes(imageFiles, logoName, skinUi);
					}});

					add("", logoName);
					add("", selectLogo);
				}
			};

			ControlPanel topNavigation = new ControlPanel(getText("ui.topNavigation")) {
				
				JComboBox topNavigationPlacement = new JComboBox(new Object[] {
					new Item("topbar", getText("ui.topBar")),
					new Item("headerLeft", getText("ui.header") + " / " + getText("ui.left")),
					new Item("headerCenter", getText("ui.header") + " / " + getText("ui.center")),
					new Item("headerRight", getText("ui.header") + " / " + getText("ui.right")),
					new Item("belowLeft", getText("ui.belowHeader") + " / " + getText("ui.left")),
					new Item("belowCenter", getText("ui.belowHeader") + " / " + getText("ui.center")),
					new Item("belowRight", getText("ui.belowHeader") + " / " + getText("ui.right"))
				});
				JCheckBox topNavigationIncludeFolders = new JCheckBox(getText("ui.includeFolders"));
				JCheckBox topNavigationIncludePages = new JCheckBox(getText("ui.includePages"));
				JCheckBox topNavigationIncludePagesBlogOnly = new JCheckBox(getText("ui.onBlogPagesOnly"));
				JCheckBox topNavigationIncludeWebLocations = new JCheckBox(getText("ui.includeWebLocations"));
				JCheckBox topNavigationIncludeWebLocationsBlogOnly = new JCheckBox(getText("ui.onBlogPagesOnly"));
				JSpinner topNavigationDepth = new JSpinner(new SpinnerNumberModel(4, 1, 6, 1));
				JSpinner topNavigationHideLong = new JSpinner(new SpinnerNumberModel(16, 1, 200, 1));
				JSpinner maxTitleLength = new JSpinner(new SpinnerNumberModel(48, 24, 160, 8));
				
				{
					topNavigationPlacement.setSelectedItem(0);
					topNavigationHideLong.setToolTipText(getText("ui.useButtonAboveInfo"));
					ComponentUtilities.whenSelectedEnable(topNavigationIncludePages, topNavigationIncludePagesBlogOnly);
					ComponentUtilities.whenSelectedEnable(topNavigationIncludeWebLocations, topNavigationIncludeWebLocationsBlogOnly);
					
					add("", new JLabel(getText("ui.placement")));
					add("tab", topNavigationPlacement);
					add("br", new JLabel(getText("ui.depth")));
					add("tab", topNavigationDepth);
					add("", new JLabel(getText("ui.levels")));
					add("br", new JLabel(getText("ui.useButtonAbove")));
					add("tab", topNavigationHideLong);
					add("", new JLabel(getText("ui.elements")));
					add("br", new JLabel(getText("ui.maxTitleLength")));
					add("tab", maxTitleLength);
					add("", new JLabel(getText("ui.characters")));
					add("br", topNavigationIncludeFolders);
					add("br", topNavigationIncludePages);
					add("tab", topNavigationIncludePagesBlogOnly);
					add("br", topNavigationIncludeWebLocations);
					add("tab", topNavigationIncludeWebLocationsBlogOnly);
				}
			};

			ControlPanel search = new ControlPanel(getText("ui.search")) {

				JCheckBox useSearch = new JCheckBox(getText("ui.useSearch"));
				JCheckBox searchBoxVisible = new JCheckBox(getText("ui.searchBoxVisible"));
				JTextField searchFields = new JSmartTextField(24);
				WrappableJLabel fieldsInfo = new WrappableJLabel("<html><i>" + getText("ui.searchFieldsInfo") + "</i></html>");

				{
					searchBoxVisible.setToolTipText(getText("ui.searchBoxVisibleInfo"));
					ComponentUtilities.whenSelectedEnable(useSearch, new JComponent[]{searchBoxVisible, searchFields, fieldsInfo});
					fieldsInfo.setPreferredWidth(uiWidth - 64);
					
					add("", useSearch);
					add("br", searchBoxVisible);
					add("br", new JLabelFor(getText("ui.fields"), searchFields));
					add("tab hfill", searchFields);
					add("br", new JLabel(infoIcon));
					add("", fieldsInfo);
				}
			};

			{
				add(topbarSticky);
				add("br hfill", logo);
				add("br hfill", topNavigation);
				add("br hfill", search);
				
				//putClientProperty("helpPage", helpRoot + "Header/Top_bar");
			}
		};
		
		//	Hero

		ControlPanel heroTab = new ControlPanel() {
			
			JComboBox heroNavigation = new JComboBox(new Object[]{
				new Item("none", getText("ui.none")),
				new Item("breadcrumb", getText("ui.breadcrumb")),
				new Item("uplink", getText("ui.upOneLevelButton")),
				new Item("both", getText("ui.both"))
			});
			
			JCheckBox preferThemeImage = new JCheckBox(getText("ui.preferThemeImage"), false);
			JCheckBox showStartSlideshow = new JCheckBox(getText("ui.showStartSlideshow"));
			JCheckBox alwaysRestartSlideshow = new JCheckBox(getText("ui.alwaysRestart"), false);			
			JScrollPane titleCaptionPane = new JScrollPane(titleCaptionTemplate);
			
			JComboBox folderDateSource = new JComboBox(new Object[]{
				new Item("none", getText("ui.none")),
				new Item("fileDate", getText("ui.fileDate")),
				new Item("folderModDate", getText("ui.folderModDate")),
				new Item("lastCameraDate", getText("ui.lastCameraDate")),
				new Item("cameraDateRange", getText("ui.cameraDateRange"))
			});
			
			//	Custom header

			ControlPanel customHeader = new ControlPanel(getText("ui.customContent")) {

				JSmartTextArea bodyTopHook = new JSmartTextArea(7, 20);
				JScrollPane bodyTopHookPane = new JScrollPane(bodyTopHook);
				JCheckBox bodyTopFitWidth = new JCheckBox(getText("ui.constrainPageWidth"), true);
				JCheckBox bodyTopTopLevelOnly = new JCheckBox(getText("ui.topLevelOnly"), false);

				{
					bodyTopHook.setEditable(true);
					bodyTopHook.setLineWrap(false);
					bodyTopHook.setWrapStyleWord(true);
					bodyTopHook.setFont(mono);
					bodyTopHook.setTabSize(2);
					allowHTMLEditing(bodyTopHook);
					//bodyTopHookPane.setBorder(BorderFactory.createTitledBorder(getText("ui.bodyTopText")));

					add("hfill vfill", bodyTopHookPane);
					add("br", bodyTopFitWidth);
					add("tab", bodyTopTopLevelOnly);

					//putClientProperty("helpPage", helpRoot + "Header/Custom_content");
				}
			};
			
			{
				new StateMonitor() {
					@Override
					public void onChange() {
						boolean imageBg = heroType.getSelectedIndex() > 1;
						heroImageDesaturate.setVisible(imageBg);
						preferThemeImage.setVisible(imageBg);
					}
				}.add(heroType).done();
				showStartSlideshow.setToolTipText(getText("ui.showStartSlideshowInfo"));
				ComponentUtilities.whenSelectedEnable(showStartSlideshow, alwaysRestartSlideshow);
				preferThemeImage.setToolTipText(getText("ui.preferThemeImageInfo"));
				titleCaptionTemplate.setEditable(true);
				titleCaptionTemplate.setLineWrap(true);
				titleCaptionTemplate.setWrapStyleWord(true);
				titleCaptionTemplate.setFont(mono);
				titleCaptionTemplate.setTabSize(2);
				titleCaptionPresets.addItemListener(listener -> {
					String s = getSelectedValue(titleCaptionPresets);
					if (!s.equals(CUSTOM)) {
						titleCaptionTemplate.setText(s);
					}
				});	
				new StateMonitor() {
					public void onChange() {
						String s = getSelectedValue(titleCaptionPresets);
						titleCaptionPane.setVisible(s.equals(CUSTOM));
					}
				}.add(titleCaptionPresets).done();					
				titleCaptionPane.setBorder(BorderFactory.createTitledBorder(getText("ui.captionTemplate")));

				add("", new JLabel(getText("ui.headerBackground")));
				add("tab", heroType);
				add("br tab", heroImageDesaturate);
				add("br tab", preferThemeImage);
				add("br", new JLabel(getText("ui.titlePlacement")));
				add("tab", titlePlacement);
				add("br", new JLabelFor(getText("ui.navigationType"), heroNavigation));
				add("tab", heroNavigation);
				add("br tab", showStartSlideshow);
				add("br tab", alwaysRestartSlideshow);
				add("br", new JLabelFor(getText("ui.caption"), titleCaptionPresets));
				add("tab", titleCaptionPresets);
				add("br hfill", titleCaptionPane);
				add("br", new JLabelFor("${folderModDate} =", folderDateSource));
				add("tab", folderDateSource);
				add("br hfill vfill", customHeader);

				//putClientProperty("helpPage", helpRoot + "Header/Hero");
			}
		};
		
		//	Slider
		
		ControlPanel sliderTab = new ControlPanel() {

			JCheckBox sliderShowCaption = new JCheckBox(getText("ui.showCaption"));
			JSpinner sliderItems = new JSpinner(new SpinnerNumberModel(5, 2, 20, 1));
			JCheckBox sliderThumbsSkip = new JCheckBox(getText("ui.useOnlyForSlider"));
			JComboBox sliderTransition = new JComboBox(new Object[] {
				new Item("crossFade", getText("ui.crossFade")),
				new Item("kenburns", getText("ui.kenBurnsEffect")),
				new Item("slide", getText("ui.horizontalSlide")),
				new Item("slideOver", getText("ui.slideOver"))					
			});
			JSpinner sliderDelay = new JSpinner(new SpinnerNumberModel(2000, 100, 10000, 100));
			JSpinner sliderTransitionSpeed = new JSpinner(new SpinnerNumberModel(400, 0, 10000, 50));

			{					
				add("", new JLabelFor(getText("ui.type"), sliderType));
				add("tab", sliderType);
				add("br tab", sliderShowCaption);
				add("br", new JLabelFor(getText("ui.sliderItems"), sliderItems));
				add("tab", sliderItems);
				add("br tab", sliderThumbsSkip);
				add("br", new JLabelFor(getText("ui.transition"), sliderTransition));
				add("tab", sliderTransition);
				add("br", new JLabelFor(getText("ui.idleLength"), sliderDelay));
				add("tab", sliderDelay);
				add(new JLabel("ms"));
				add("br", new JLabelFor(getText("ui.transitionSpeed"), sliderTransitionSpeed));
				add("tab", sliderTransitionSpeed);
				add(new JLabel("ms"));					
			}
		};
				
		/*	---------------------------------------------------------------
									Sections / Main content
			--------------------------------------------------------------- */
		
		// Folders section
		
		ControlPanel folders = new ControlPanel() {
			
			@Override
			public void importVariables(Map<String, Object> vars) {
				String v;
				
				if ((v = (String)vars.get("folderCaptionTemplate")) != null) {
					if (v.equals("<header><span class=\"date\">${folderModDate}</span><h4>${title}</h4></header><div class=\"comment\">${commentShort}</div>")) {
						vars.put("folderCaptionTemplate", "<h3>${title} <span class=\"date\">${folderModDate}</span></h3>\n<div class=\"comment\">${commentShort}</div>)");
					}
				}
				
				if ((v = (String)vars.get("enhanceFolders")) != null && vars.get("enhanceFolderCnt") == null) {
					if (v.equals("none")) {
						vars.put("enhanceFolderCnt", "0");
					} else if (v.equals("firstOnly")) {
						vars.put("enhanceFolderCnt", "1");
					}
				}
			};


			JLabel columnsInfo = new JLabel(infoIcon);
			JComboBox enhanceFolderCnt = new JComboBox(new Object[] {
				new Item("0", getText("ui.none")),
				new Item("1", getText("ui.firstOnly")),
				new Item("2", 2 + " " + getText("ui.folders")),
				new Item("3", 3 + " " + getText("ui.folders")),
				new Item("999", getText("ui.all"))
			});
			JLabel enhanceFoldersInfo = new JLabel(infoIcon);
			
			ButtonGroup defaultFolderIcon = new ButtonGroup();
			private JRadioButton defaultFolderIconBook = new JRadioButton("");
			private JRadioButton defaultFolderIconYellow = new JRadioButton("");
			private JRadioButton defaultFolderIconBlue = new JRadioButton("");
			private JRadioButton defaultFolderIconHollow = new JRadioButton("");
			
			JTextArea folderCaptionTemplate = new JSmartTextArea(4, 30);
			JScrollPane folderCaptionPane = new JScrollPane(folderCaptionTemplate);
			JComboBox folderCaptionPlacement = new JComboBox(new Object[] {
				new Item("separate", getText("ui.separate") + " (" + getText("ui.below") + ")"),
				new Item("over left top", getText("ui.left") + " " + getText("ui.top")), 
				new Item("over center top", getText("ui.center") + " " + getText("ui.top")), 
				new Item("over right top", getText("ui.right") + " " + getText("ui.top")), 
				new Item("over left middle", getText("ui.left") + " " + getText("ui.middle")), 
				new Item("over center middle", getText("ui.center") + " " + getText("ui.middle")), 
				new Item("over right middle", getText("ui.right") + " " + getText("ui.middle")), 
				new Item("over left bottom", getText("ui.left") + " " + getText("ui.bottom")), 
				new Item("over center bottom", getText("ui.center") + " " + getText("ui.bottom")), 
				new Item("over right bottom", getText("ui.right") + " " + getText("ui.bottom")),
			});
			JCheckBox showFolderImageCount = new JCheckBox(getText("ui.showFolderImageCount"), true);
			JCheckBox gatherWeblocationInfo = new JCheckBox(getText("ui.gatherWeblocationInfo"), true);
			JCheckBox webLocationOpenNew = new JCheckBox(getText("ui.webLocationsOpenInNewWindow"));
			JCheckBox hideWeblocations = new  JCheckBox(getText("ui.hideWeblocations"), false);

			{
				new StateMonitor() {
					@Override
					public void onChange() {
						setFolderThumbSize();
					}
				}.add(folderColumns).add(folderThumbAspectRatio);
				
				columnsInfo.addMouseListener(new MouseAdapter() {  
					@Override
					public void mouseReleased(MouseEvent e) {
						JOptionPane.showMessageDialog(window, getText("ui.columnsInfo"), "Info", JOptionPane.INFORMATION_MESSAGE);
				}});
				
				fixedShapeFolderThumbs.setToolTipText(getText("ui.fixedShapeThumbsInfo"));
				
				enhanceFoldersInfo.addMouseListener(new MouseAdapter() {  
					@Override
					public void mouseReleased(MouseEvent e) {
						JOptionPane.showMessageDialog(window, getText("ui.enhanceFoldersInfo"), "Info", JOptionPane.INFORMATION_MESSAGE);
				}});
				
				defaultFolderIcon.add(defaultFolderIconBook);
				defaultFolderIcon.add(defaultFolderIconYellow);
				defaultFolderIcon.add(defaultFolderIconBlue);
				defaultFolderIcon.add(defaultFolderIconHollow);
				defaultFolderIconBook.setSelected(true);
				defaultFolderIconBook.setActionCommand("book");
				defaultFolderIconYellow.setActionCommand("yellow");
				defaultFolderIconBlue.setActionCommand("blue");
				defaultFolderIconHollow.setActionCommand("hollow");
				
				folderCaptionPresets.addItemListener(listener -> {
					String s = getSelectedValue(folderCaptionPresets);
					if (!s.equals(CUSTOM)) {
						folderCaptionTemplate.setText(s);
					}
				});	
				new StateMonitor() {
					public void onChange() {
						String s = getSelectedValue(folderCaptionPresets);
						folderCaptionPane.setVisible(s.equals(CUSTOM));
					}
				}.add(folderCaptionPresets).done();						
				folderCaptionTemplate.setEditable(true);
				folderCaptionTemplate.setLineWrap(true);
				folderCaptionTemplate.setWrapStyleWord(true);
				folderCaptionTemplate.setFont(mono);
				folderCaptionPane.setBorder(BorderFactory.createTitledBorder(getText("ui.captionTemplate")));
				ComponentUtilities.whenSelectedEnable(fixedShapeFolderThumbs, new JComponent[]{folderThumbAspectRatio});
				gatherWeblocationInfo.setToolTipText(getText("ui.gatherWeblocationInfoInfo"));

				add(new JLabelFor(getText("ui.columns"), folderColumns));
				add("tab", folderColumns);
				add(columnsInfo);
				add("br tab", fixedShapeFolderThumbs);
				add("br", new JLabelFor(getText("ui.aspectRatio"), folderThumbAspectRatio));
				add("tab", folderThumbAspectRatio);
				add("br", new JLabel(getText("ui.enhanceFolders")));
				add("tab", enhanceFolderCnt);
				add(enhanceFoldersInfo);
				add("br", new JLabel(getText("ui.defaultFolderIcon")));
				add("tab", defaultFolderIconBook);
				add(new JLabel(svgIcon("folder-s-book", new Dimension(32, 32))));
				add("", defaultFolderIconYellow);
				add(new JLabel(svgIcon("folder-s-yellow", new Dimension(32, 32))));
				add("", defaultFolderIconBlue);
				add(new JLabel(svgIcon("folder-s-blue", new Dimension(32, 32))));
				add("", defaultFolderIconHollow);
				add(new JLabel(svgIcon("folder-s-hollow", new Dimension(32, 32))));
				add("br", new JLabelFor(getText("ui.captionPlacement"), folderCaptionPlacement));
				add("tab", folderCaptionPlacement);
				add("br", new JLabelFor(getText("ui.caption"), folderCaptionPresets));
				add("tab", folderCaptionPresets);
				add("br hfill", folderCaptionPane);
				add("br", showFolderImageCount);
				add("br", gatherWeblocationInfo);
				add("br", hideWeblocations);
				add("br", webLocationOpenNew);
			}
		};

		// Pages section
		
		ControlPanel pages = new ControlPanel() {

			{
				add(new JLabel(getText("ui.showPagesOnIndex")));
				add("tab", showPages);
			}
		};

		// Images section (thumbnails)
		
		ControlPanel images = new ControlPanel() {

			JCheckBox thumbGrayscaleEffect = new JCheckBox(getText("ui.thumbGrayscaleEffect"), false);
			JCheckBox markActualImage = new JCheckBox(getText("ui.markActualImage"), true);
			JLabel columnsInfo = new JLabel(infoIcon);
			JScrollPane thumbCaptionPane = new JScrollPane(thumbCaptionTemplate);
			JComboBox captionPlacement = new JComboBox(new Object[]{
				new Item("over top", getText("ui.center") + " " + getText("ui.top")), 
				new Item("over middle", getText("ui.center") + " " + getText("ui.middle")), 
				new Item("over bottom", getText("ui.center") + " " + getText("ui.bottom")), 
				new Item("below", getText("ui.below"))
			});
			JCheckBox captionShowOnHover = new JCheckBox(getText("ui.showOnlyOnMouseOver"), true);

			{
				thumbGrayscaleEffect.setToolTipText(getText("ui.thumbGrayscaleEffectInfo"));
				columnsInfo.addMouseListener(new MouseAdapter() {  
					@Override
					public void mouseReleased(MouseEvent e) {
						JOptionPane.showMessageDialog(window, getText("ui.columnsInfo") + "\n\n" + getText("ui.masonryColumnsInfo"), "Info", JOptionPane.INFORMATION_MESSAGE);
				}});
				thumbCaptionPresets.addItemListener(listener -> {
					String s = getSelectedValue(thumbCaptionPresets);
					if (!s.equals(CUSTOM)) {
						thumbCaptionTemplate.setText(s);
					}
				});	
				new StateMonitor() {
					public void onChange() {
						String s = getSelectedValue(thumbCaptionPresets);
						thumbCaptionPane.setVisible(s.equals(CUSTOM));
					}
				}.add(thumbCaptionPresets).done();						
				captionPlacement.addActionListener((ActionEvent e) -> {
					captionShowOnHover.setEnabled(captionPlacement.getSelectedIndex() <= 2);
				});
				thumbLayout.addActionListener((ActionEvent e) -> {
					thumbAspectRatio.setEnabled(thumbLayout.getSelectedIndex() == 0);
				});
				thumbCaptionPane.setBorder(BorderFactory.createTitledBorder(getText("ui.captionTemplate")));
				thumbCaptionTemplate.setEditable(true);
				thumbCaptionTemplate.setLineWrap(true);
				thumbCaptionTemplate.setWrapStyleWord(true);
				thumbCaptionTemplate.setFont(mono);

				add(new JLabelFor(getText("ui.thumbnailLayout"), thumbLayout));
				add("tab", thumbLayout);
				add("tab", new JLabelFor(getText("ui.aspectRatio"), thumbAspectRatio));
				add("tab", thumbAspectRatio);
				add("br", new JLabelFor(getText("ui.columns"), columns));
				add("tab", columns);
				add(columnsInfo);
				add("br", new JLabelFor(getText("ui.thumbGap"), thumbGap));
				add("tab", thumbGap);
				add("br tab", markActualImage);
				add("br tab", thumbGrayscaleEffect);
				add("br", new JLabelFor(getText("ui.captionPlacement"), captionPlacement));
				add("tab", captionPlacement);
				add("br tab", captionShowOnHover);
				add("br", new JLabelFor(getText("ui.caption"), thumbCaptionPresets));
				add("tab", thumbCaptionPresets);
				add("br hfill", thumbCaptionPane);
			}
		};
		
		// Map section
		
		ControlPanel map = new ControlPanel() {
						
			{
				mapInSections.addItemListener(listener -> {
					if (mapInSections.isSelected()) {
						_mapPlacement.setSelectedIndex(1);
						mapInSidebar.setSelected(false);
					} else {
						_mapPlacement.setSelectedIndex(mapInSidebar.isSelected()? 2 : 0);
					}
				});
				
				add("", mapInSections);
				add("br", new JLabel(infoIcon));
				add(new JLabel("<html><i>" + getText("ui.findCorrespondingSettings") + " " + getText("ui.site") + " / " + getText("ui.map") + "</i></html>"));
			}
		};
					
		// Shopping cart section
		/*
		ControlPanel shoppingCart = new ControlPanel() {

			JCheckBox shopInSections = new JCheckBox(getText("ui.showShopInMainContent"));

			{
				if (license.length() == 0) {
					shopInSections.setSelected(false);
					commercialMonitor.add(shopInSections);
				}
				shopInSections.setToolTipText(getText("ui.sellPhotosInfo"));

				add(shopInSections);
				add("br", new JLabel(infoIcon));
				add(new JLabel("<html><i>" + getText("ui.findCorrespondingSettings") + " " + getText("ui.site") + " / " + getText("ui.shoppingCart") + "</i></html>"));

				//putClientProperty("helpPage", helpRoot + "Sections/Shopping_cart");
			}
		};
		*/
		// Neighboring folders
		
		ControlPanel neighboringFolders = new ControlPanel() {
			
			JCheckBox linkNeighboringFolders = new JCheckBox(getText("ui.linkNeighboringFolders"), true);
			JCheckBox neighboringFolderBgImage = new JCheckBox(getText("ui.neighboringFolderBgImage"), true);
			JCheckBox neighboringFolderSkipLevels = new JCheckBox(getText("ui.skipFolderLevels"), false);
			JCheckBox neighboringFolderLoop = new JCheckBox(getText("ui.neighboringFolderLoop"), false);

			{
				linkNeighboringFolders.setToolTipText(getText("ui.linkNeighboringFoldersInfo"));
				neighboringFolderBgImage.setToolTipText(getText("ui.neighboringFolderBgImageInfo"));
				neighboringFolderSkipLevels.setToolTipText(getText("ui.skipFolderLevelsInfo"));
				neighboringFolderLoop.setToolTipText(getText("ui.neighboringFolderLoopInfo"));
				ComponentUtilities.whenSelectedEnable(linkNeighboringFolders, new JComponent[]{ neighboringFolderBgImage, neighboringFolderSkipLevels, neighboringFolderLoop });
				
				add(linkNeighboringFolders);
				add("br", neighboringFolderSkipLevels);
				add("br", neighboringFolderBgImage);
				add("br", neighboringFolderLoop);
				//putClientProperty("helpPage", helpRoot + "Sections/Neighboring_folders");
			}
		};
				
		//	Shopping cart settings
		
		ControlPanel shoppingCart = new ControlPanel() {

			JCheckBox showShop = new JCheckBox(getText("ui.usePaypal"));

			JCheckBox showShopOnImagePagesOnly = new JCheckBox(getText("ui.showShopOnImagePagesOnly"), true);
			JTextField shopLabel = new JTextField(24);
			JTextField shopId = new JTextField(24);
			JLabel mandatoryInfo = new JLabel(mandatory);
			/*JComboBox shopPlacement = new JComboBox(new Object[]{ 
				new Item("aboveThumbs", getText("ui.aboveThumbs")),
				new Item("sidebar", getText("ui.inTheSidebar"))
			});*/
			JComboBox shopCurrency = new JComboBox(new Object[]{ 
				new Item("USD", "United States Dollar"),
				new Item("EUR", "Euro"),
				new Item("GBP", "British Pound"),
				new Item("CAD", "Canadian Dollar"),
				new Item("AUD", "Australian Dollar"),
				//new Item("RUB", "Russia Rubles"),
				new Item("JPY", "Japanese Yen"),
				//new Item("INR", "India Rupees"),
				new Item("NZD", "New Zealand Dollar"),
				new Item("CHF", "Swiss Franc"),
				//new Item("ARS", "Argentina Pesos"),
				//new Item("BHD", "Bahrain Dinars"),
				//new Item("BYR", "Belarus Rubles"),
				//new Item("BAM", "Bosnia & Herzegovina C.Marka"),
				new Item("BRL", "Brazilian Real"),
				//new Item("BGN", "Bulgaria Leva"),
				//new Item("CLP", "Chile Pesos"),
				//new Item("CNY", "China Yuan Renminbi"),
				//new Item("COP", "Colombia Pesos"),
				//new Item("CRC", "Costa Rica Colones"),
				//new Item("HRK", "Croatia Kuna"),
				new Item("CZK", "Czech Koruna"),
				new Item("DKK", "Danish Krone"),
				//new Item("EGP", "Egypt Pounds"),
				//new Item("EEK", "Estonia Krooni"),
				//new Item("GTQ", "Guatemala Quetzales"),
				new Item("HKD", "Hong Kong Dollar"),
				new Item("HUF", "Hungary Forint"),
				//new Item("ISK", "Iceland Kronur"),
				//new Item("IDR", "Indonesia Rupiahs"),
				//new Item("IQD", "Iraq Dinars"),
				new Item("ILS", "Israel New Shekel"),
				//new Item("JMD", "Jamaica Dollars"),
				//new Item("JOD", "Jordan Dinars"),
				//new Item("KWD", "Kuwait Dinars"),
				//new Item("LVL", "Latvia Lati"),
				//new Item("LBP", "Lebanon Pounds"),
				//new Item("LTL", "Lithuania Litai"),
				//new Item("MKD", "Macedonia Denars"),
				new Item("MYR", "Malaysian Ringgit"),
				new Item("MXN", "Mexican Peso"),
				//new Item("MDL", "Moldova Lei"),
				//new Item("MAD", "Morocco Dirhams"),
				new Item("NOK", "Norway Krone"),
				//new Item("PEN", "Peru Nuevos Soles"),
				new Item("PHP", "Philippines Peso"),
				new Item("PLN", "Poland Zlotych"),
				//new Item("RON", "Romania New Lei"),
				new Item("RUB", "Russian Ruble"),
				//new Item("SAR", "Saudi Arabia Riyals"),
				//new Item("RSD", "Serbia Dinars"),
				new Item("SGD", "Singapore Dollar"),
				//new Item("ZAR", "South Africa Rand"),
				//new Item("KRW", "South Korea Won"),
				new Item("SEK", "Sweden Krona"),
				new Item("TWD", "New Taiwan Dollar"),
				new Item("THB", "Thailand Baht"),
				new Item("TRY", "Turkish Lira"),
				//new Item("UAH", "Ukraine Hryvnia"),
				//new Item("AED", "United Arab Emirates Dirhams"),
				//new Item("UYU", "Uruguay Pesos"),
				//new Item("VND", "Vietnam Dong")
			});
			JTextField shopSuccessUrl = new JSmartTextField(24);
			JCheckBox shopSameWindowCheckout = new JCheckBox(getText("ui.sameWindowCheckout"));
			JCheckBox shopSendAlbumName = new JCheckBox(getText("ui.sendAlbumName"));
			JCheckBox shopSendFolderPath = new JCheckBox(getText("ui.sendFolderPath"));
			JComboBox shopSendItemName = new JComboBox(new Object[]{ 
				new Item("name", getText("ui.filename")),
				new Item("title", getText("ui.titleOrFilename")),
				new Item("comment", getText("ui.commentOrFilename"))
			});
			JCheckBox shopAskPermissionToEmpty = new JCheckBox(getText("ui.askPermisssionToEmptyCart"));
			
			JComboBox showPriceRange = new JComboBox(new Object[]{ 
				new Item("", getText("ui.none")),
				new Item("lowest", getText("ui.lowestPriceOnly")),
				new Item("minmax", getText("ui.minMax"))
			});
			JCheckBox usePriceAsSingleOption = new JCheckBox(getText("ui.usePriceAsSingleOption"));
			JComboBox shopOptionSelectMethod = new JComboBox(new Object[]{ 
				new Item("combo", getText("ui.dropdownBox")),
				new Item("radio", getText("ui.radioButtons"))
			});
			JTextArea shopOptions = new JSmartTextArea(8, 28);
			JScrollPane shopOptionsPane = new JScrollPane(shopOptions);
			JLabel shopOptionsInfo = new JLabel(infoIcon);

			JTextField shopMinimalAmount = new JSmartTextField(8);
			JTextField shopHandling = new JSmartTextField(8);
			JLabel shopHandlingInfo = new JLabel(infoIcon);
			JTextField shopTax = new JSmartTextField(8);
			JTextField shopQuantityCap = new JSmartTextField(8);

			JTextArea shopCoupons = new JSmartTextArea(4, 16);
			JScrollPane shopCouponsPane = new JScrollPane(shopCoupons);
			JLabel shopCouponsInfo = new JLabel(infoIcon);

			JTextField shopDiscountRate = new JSmartTextField(8);
			JTextField shopDiscountMinQuantity = new JSmartTextField(8);
			JTextField shopDiscountMinAmount = new JSmartTextField(8);

			JSmartTextArea shopInstructions = new JSmartTextArea(6, 24);	
			JScrollPane shopInstructionsPane = new JScrollPane(shopInstructions);
			JSmartTextArea shopInstructionsBox = new JSmartTextArea(6, 24);	
			JScrollPane shopInstructionsBoxPane = new JScrollPane(shopInstructionsBox);
				
			ControlPanel shoppingCartOptions = new ControlPanel() {

				{
					shopId.setToolTipText(getText("ui.sellerIdInfo"));
					mandatoryInfo.setToolTipText(getText("ui.mandatory"));
					shopLabel.setToolTipText(getText("ui.default") + ": " + getText("ui.shoppingCart"));
					shopCurrency.setEditable(false);
					shopCurrency.setToolTipText(getText("ui.currencyInfo"));
					shopSuccessUrl.setToolTipText(getText("ui.shopSuccessUrlInfo"));
					usePriceAsSingleOption.setToolTipText(getText("ui.usePriceAsSingleOptionInfo"));
					shopSameWindowCheckout.setToolTipText(getText("ui.sameWindowCheckoutInfo"));
					shopSendAlbumName.setToolTipText(getText("ui.sendAlbumNameInfo"));

					add(showShopOnImagePagesOnly);
					add("br", new JLabelFor(getText("ui.boxTitle"), shopLabel));
					add("tab", shopLabel);
					add("br", new JLabelFor(getText("ui.sellerId"), shopId));
					add(mandatoryInfo);
					add("tab", shopId);
					add("br tab", new JLinkLabel("https://www.paypal.com/signup/account", getText("ui.signUp")));
					add("br", new JLabelFor(getText("ui.currency"), shopCurrency));
					add("tab", shopCurrency);
					add("br", new JLabelFor(getText("ui.shopSuccessUrl"), shopSuccessUrl));
					add("tab", shopSuccessUrl);
					add("br", new JLabel(getText("ui.showPriceRange")));
					add("tab", showPriceRange);
					add("br", usePriceAsSingleOption);
					add("br", shopSendAlbumName);
					add("tab", shopSendFolderPath);
					add("tab", shopSendItemName);
					add("", new JLabel(getText("ui.max128Chars")));
					add("br", shopSameWindowCheckout);
					add("br", shopAskPermissionToEmpty);
				}
			};

			ControlPanel shoppingCartPricing = new ControlPanel() {

				{
					shopOptionsInfo.addMouseListener(new MouseAdapter() {  
						@Override
						public void mouseReleased(MouseEvent e) {
							String s = "<html><b><kbd>" + getText("ui.shopOptionsFormat") + "</kbd></b><br>" +
									"<br><b>" + getText("ui.examples") + ":</b><br>" +
									"<br><kbd>" + "Print A4=2.0" + "</kbd><br>" +
									"<i>" + getText("ui.optionExample1Hint") + "</i><br>" +
									"<br><kbd>" + "Print A2=10.0+0.5" + "</kbd><br>" +
									"<i>" + getText("ui.optionExample2Hint") + "</i><br>" +
									"<br><kbd>" + "Print A2=10.0+0.5+0" + "</kbd><br>" +
									"<i>" + getText("ui.optionExample3Hint") + "</i><br>" +
									"<br><i>" + getText("ui.noCurrencySign") + "</i></html>";
							JOptionPane.showMessageDialog(window, s, getText("ui.format"), JOptionPane.INFORMATION_MESSAGE);
					}});
					shopOptions.setToolTipText(getText("ui.shopOptionsInfo"));
					shopOptions.setEditable(true);
					shopOptions.setLineWrap(false);
					shopOptions.setFont(mono);
					shopOptionsPane.setBorder(BorderFactory.createTitledBorder(getText("ui.shopOptions")));
					shopHandlingInfo.addMouseListener(new MouseAdapter() {  
						@Override
						public void mouseReleased(MouseEvent e) {
							String s = "<html><b><kbd>" + getText("ui.handlingFormat") + "</kbd></b><br>" +
									"<br><b>" + getText("ui.examples") + ":</b><br>" +
									"<br><kbd>" + "2.0" + "</kbd><br>" +
									"<i>" + getText("ui.handlingExample1Hint") + "</i><br>" +
									"<br><kbd>" + "0.0+0.1" + "</kbd><br>" +
									"<i>" + getText("ui.handlingExample2Hint") + "</i><br>" +
									"<br><kbd>" + "1.0+0+0.1" + "</kbd><br>" +
									"<i>" + getText("ui.handlingExample3Hint") + "</i><br>" +
									"<br><i>" + getText("ui.noCurrencySign") + "</i></html>";
							JOptionPane.showMessageDialog(window, s, getText("ui.format"), JOptionPane.INFORMATION_MESSAGE);
					}});
					shopHandling.setToolTipText(getText("ui.handlingInfo") + " " + getText("ui.leaveEmpty"));
					shopTax.setToolTipText(getText("ui.taxInfo") + " " + getText("ui.leaveEmpty"));
					shopQuantityCap.setToolTipText(getText("ui.shopQuantityCapInfo") + " " + getText("ui.leaveEmpty"));
					shopMinimalAmount.setToolTipText(getText("ui.shopMinimalCartValueInfo"));
					shopDiscountRate.setToolTipText(getText("ui.shopDiscountRateInfo") + " " + getText("ui.leaveEmpty"));
					shopDiscountMinQuantity.setToolTipText(getText("ui.shopDiscountMinQuantityInfo"));
					shopDiscountMinAmount.setToolTipText(getText("ui.shopDiscountMinAmountInfo"));
					shopCoupons.setToolTipText(getText("ui.shopCouponsInfo"));
					shopCoupons.setEditable(true);
					shopCoupons.setLineWrap(false);
					shopCoupons.setFont(mono);
					shopCouponsInfo.addMouseListener(new MouseAdapter() {  
						@Override
						public void mouseReleased(MouseEvent e) {
							String s = "<html><b><kbd>" + getText("ui.shopCouponsFormat") + "</kbd></b><br>" +
									"<br><b>" + getText("ui.examples") + ":</b><br>" +
									"<br><kbd>" + "5OFF=5.0" + "</kbd><br>" +
									"<i>" + getText("ui.couponExample1Hint") + "</i><br>" +
									"<br><kbd>" + "XMAS=20% &lt;2016-12-24" + "</kbd><br>" +
									"<i>" + getText("ui.couponExample2Hint") + "</i><br>" +
									"<br><kbd>" + "MIN10=30% 10+" + "</kbd><br>" +
									"<i>" + getText("ui.couponExample3Hint") + "</i><br>" +
									"<br><i>" + getText("ui.noCurrencySign") + "</i></html>";
							JOptionPane.showMessageDialog(window, s, getText("ui.format"), JOptionPane.INFORMATION_MESSAGE);
					}});
					shopCouponsPane.setBorder(BorderFactory.createTitledBorder(getText("ui.shopCoupons")));

					add("", new JLabelFor(getText("ui.handling"), shopHandling));
					add(shopHandlingInfo);
					add("tab", shopHandling);
					add("tab", new JLabelFor(getText("ui.tax") + " (%)", shopTax));
					add("tab", shopTax);
					add("br", new JLabelFor(getText("ui.shopQuantityCap"), shopQuantityCap));
					add("tab", shopQuantityCap);
					add("tab", new JLabelFor(getText("ui.shopMinimalCartValue"), shopMinimalAmount));
					add("tab", shopMinimalAmount);
					add("br", new JLabelFor(getText("ui.shopDiscountRate") + " (%)", shopDiscountRate));
					add("tab", shopDiscountRate);
					add("tab", new JLabelFor(getText("ui.minQuantity"), shopDiscountMinQuantity));
					add("tab", shopDiscountMinQuantity);
					add("tab", new JLabelFor(getText("ui.minAmount"), shopDiscountMinAmount));
					add("tab", shopDiscountMinAmount);
					add("br hfill", shopOptionsPane);
					add(shopOptionsInfo);
					add("br hfill", shopCouponsPane);
					add(shopCouponsInfo);
					add("br", new JLabelFor(getText("ui.selectMethod"), shopOptionSelectMethod));
					add("tab", shopOptionSelectMethod);

				}
			};

			ControlPanel shoppingCartInstructions = new ControlPanel() {

				{
					shopInstructions.setToolTipText(getText("ui.shopInstructionsInfo"));
					shopInstructions.setEditable(true);
					shopInstructions.setLineWrap(true);
					shopInstructions.setWrapStyleWord(true);
					shopInstructions.setFont(mono);
					allowHTMLEditing(shopInstructions);
					shopInstructionsPane.setBorder(BorderFactory.createTitledBorder(getText("ui.shopInstructions")));
					shopInstructionsBox.setToolTipText(getText("ui.shopInstructionsBoxInfo"));
					shopInstructionsBox.setEditable(true);
					shopInstructionsBox.setLineWrap(true);
					shopInstructionsBox.setWrapStyleWord(true);
					shopInstructionsBox.setFont(mono);
					shopInstructionsBox.setTabSize(2);
					allowHTMLEditing(shopInstructionsBox);
					shopInstructionsBoxPane.setBorder(BorderFactory.createTitledBorder(getText("ui.shopInstructionsBox")));
					
					add("hfill", shopInstructionsPane);
					add("br hfill", shopInstructionsBoxPane);
				}
			};
			
			JTabbedPane shopTabs = new JTabbedPane() {
								
				{
					shoppingCartOptions.setBorder(emptyBorder);
					shoppingCartPricing.setBorder(emptyBorder);
					shoppingCartInstructions.setBorder(emptyBorder);
					
					addTab(getText("ui.settings"), shoppingCartOptions);
					addTab(getText("ui.pricing"), shoppingCartPricing);
					addTab(getText("ui.instructions"), shoppingCartInstructions);
				}
			};

			{
				if (!commercialUseAllowed) {
					showShop.setSelected(false);
					commercialMonitor.add(showShop);
				}
				showShop.setToolTipText(getText("ui.sellPhotosInfo"));
				ComponentUtilities.whenSelectedEnable(showShop, new JComponent[]{ shoppingCartOptions, shoppingCartPricing, shoppingCartInstructions } );
				
				add("", showShop);
				add("br hfill", shopTabs);

				//putClientProperty("helpPage", helpRoot + "Sections/Shopping_cart");
			}
		};
		
		// Feedback section
		
		ControlPanel feedback = new ControlPanel() {

			JCheckBox showFeedback = new JCheckBox(getText("ui.useFeedback"));

			ControlPanel feedbackOptions = new ControlPanel() {

				JCheckBox showFeedbackOnImagePagesOnly = new JCheckBox(getText("ui.showOnImagePagesOnly"), true);
				JCheckBox feedbackAddMultiple = new JCheckBox(getText("ui.canAddMultipleTimes"));
				JTextField feedbackLabel = new JTextField(20);
				JTextField feedbackEmail = new JTextField(20);
				JLabel mandatoryInfo = new JLabel(mandatory);
				JComboBox feedbackFormatting = new JComboBox(new Object[] {
					new Item("human", getText("ui.humanReadable")),			
					new Item("serialized", getText("ui.serialized"))
				});
				JTextField feedbackViewButtonLabel = new JTextField(20);
				JTextField feedbackAddButtonLabel = new JTextField(20);
				JTextField feedbackAddButtonTooltip = new JTextField(20);
				JTextField feedbackModalTitle = new JTextField(20);
				JTextField feedbackCopyButtonLabel = new JTextField(20);
				JCheckBox useFeedbackSendButton = new JCheckBox(getText("ui.useSendButton"), true);
				JTextField feedbackSendButtonLabel = new JTextField(20);
				
				{
					feedbackLabel.setToolTipText(getText("ui.feedbackLabelInfo"));
					//ComponentUtilities.whenSelectedDisable(feedbackBoxSticky, feedbackFloatButton);
					mandatoryInfo.setToolTipText(getText("ui.mandatory"));
					feedbackViewButtonLabel.setToolTipText(getText("ui.feedbackViewButtonLabelInfo"));
					feedbackAddButtonLabel.setToolTipText(getText("ui.feedbackAddButtonLabelInfo"));
					feedbackAddButtonTooltip.setToolTipText(getText("ui.feedbackActionButtonTooltipInfo"));
					feedbackModalTitle.setToolTipText(getText("ui.feedbackModalTitleInfo"));
					feedbackCopyButtonLabel.setToolTipText(getText("ui.feedbackCopyButtonLabelInfo"));
					feedbackSendButtonLabel.setToolTipText(getText("ui.feedbackSendButtonLabelInfo"));
					feedbackEmail.setToolTipText(getText("ui.feedbackEmailInfo"));
					
					ComponentUtilities.whenSelectedEnable(useFeedbackSendButton, feedbackSendButtonLabel);
					
					add(showFeedbackOnImagePagesOnly);
					add("br", feedbackAddMultiple);
					add("br", new JLabelFor(getText("ui.boxTitle"), feedbackLabel));
					add("tab", feedbackLabel);
					add("tab", new JLabel("<html><i>" + getText("ui.default") + ": \"" + getText("feedback") + "\"</i></html>"));
					add("br", new JLabelFor(getText("ui.feedbackEmail"), feedbackEmail));
					add(mandatoryInfo);
					add("tab", feedbackEmail);
					add("br", new JLabelFor(getText("ui.formatting"), feedbackFormatting));
					add("tab", feedbackFormatting);
					add("br", new JLabelFor(getText("ui.viewButtonLabel"), feedbackViewButtonLabel));
					add("tab", feedbackViewButtonLabel);
					add("tab", new JLabel("<html><i>" + getText("ui.default") + ": \"" + getText("editFeedback") + "\"</i></html>"));
					add("br", new JLabelFor(getText("ui.addButtonLabel"), feedbackAddButtonLabel));
					add("tab", feedbackAddButtonLabel);
					add("tab", new JLabel("<html><i>" + getText("ui.default") + ": \"" + getText("writeFeedback") + "\"</i></html>"));
					add("br", new JLabelFor(getText("ui.addButtonTooltip"), feedbackAddButtonTooltip));
					add("tab", feedbackAddButtonTooltip);
					add("tab", new JLabel("<html><i>" + getText("ui.default") + ": \"" + getText("provideFeedbackOnSelectedItems") + "\"</i></html>"));
					add("br", new JLabelFor(getText("ui.modalTitle"), feedbackModalTitle));
					add("tab", feedbackModalTitle);
					add("tab", new JLabel("<html><i>" + getText("ui.default") + ": \"" + getText("feedbackOnAlbum") + "\"</i></html>"));
					add("br", new JLabelFor(getText("ui.copyButtonLabel"), feedbackCopyButtonLabel));
					add("tab", feedbackCopyButtonLabel);
					add("tab", new JLabel("<html><i>" + getText("ui.default") + ": \"" + getText("copyFeedback") + "\"</i></html>"));
					add("br", useFeedbackSendButton);
					add("br", new JLabelFor(getText("ui.sendButtonLabel"), feedbackSendButtonLabel));
					add("tab", feedbackSendButtonLabel);
					add("tab", new JLabel("<html><i>" + getText("ui.default") + ": \"" + getText("sendFeedback") + "\"</i></html>"));
				}
			};
			
			ControlPanel feedbackForm = new ControlPanel() {
				
				JTextArea feedbackTemplate = new JSmartTextArea(5, 24);
				JScrollPane feedbackTemplatePane = new JScrollPane(feedbackTemplate);
				//JLabel feedbackTemplateInfo = new JLabel(infoIcon);
				
				{
					feedbackTemplate.setToolTipText(getText("ui.feedbackTemplateInfo"));
					feedbackTemplate.setEditable(true);
					feedbackTemplate.setLineWrap(true);
					feedbackTemplate.setWrapStyleWord(true);
					feedbackTemplate.setFont(mono);
					feedbackTemplate.setTabSize(2);
					feedbackTemplatePane.setBorder(BorderFactory.createTitledBorder(getText("ui.feedbackTemplate")));
					
					add("hfill vfill", feedbackTemplatePane);
					
				}
			};
			
			ControlPanel feedbackInstruction = new ControlPanel() {
				
				JSmartTextArea feedbackInstructions = new JSmartTextArea(6, 24);	
				JScrollPane feedbackInstructionsPane = new JScrollPane(feedbackInstructions);
				JSmartTextArea feedbackInstructionsBox = new JSmartTextArea(6, 24);	
				JScrollPane feedbackInstructionsBoxPane = new JScrollPane(feedbackInstructionsBox);
				
				{
					
					feedbackInstructions.setToolTipText(getText("ui.feedbackInstructionsInfo"));
					feedbackInstructions.setEditable(true);
					feedbackInstructions.setLineWrap(true);
					feedbackInstructions.setWrapStyleWord(true);
					feedbackInstructions.setFont(mono);
					feedbackInstructions.setTabSize(2);
					allowHTMLEditing(feedbackInstructions);
					feedbackInstructionsPane.setBorder(BorderFactory.createTitledBorder(getText("ui.feedbackInstructions")));
					
					feedbackInstructionsBox.setToolTipText(getText("ui.feedbackInstructionsBoxInfo"));
					feedbackInstructionsBox.setEditable(true);
					feedbackInstructionsBox.setLineWrap(true);
					feedbackInstructionsBox.setWrapStyleWord(true);
					feedbackInstructionsBox.setFont(mono);
					feedbackInstructionsBox.setTabSize(2);
					allowHTMLEditing(feedbackInstructionsBox);
					feedbackInstructionsBoxPane.setBorder(BorderFactory.createTitledBorder(getText("ui.feedbackInstructionsBox")));

					add("hfill", feedbackInstructionsPane);
					add("br hfill", feedbackInstructionsBoxPane);
				}
			};
			
			JTabbedPane feedbackTabs = new JTabbedPane() {
								
				{
					feedbackOptions.setBorder(emptyBorder);
					feedbackForm.setBorder(emptyBorder);
					feedbackInstruction.setBorder(emptyBorder);
					
					addTab(getText("ui.settings"), feedbackOptions);
					addTab(getText("ui.feedbackForm"), feedbackForm);
					addTab(getText("ui.instructions"), feedbackInstruction);
				}
			};
			
			{
				if (!commercialUseAllowed) {
					showFeedback.setSelected(false);
					commercialMonitor.add(showFeedback);
				}
				showFeedback.setToolTipText(getText("ui.useFeedbackInfo"));
				ComponentUtilities.whenSelectedEnable(showFeedback, new JComponent[] {feedbackOptions,feedbackForm,feedbackInstruction});
				feedbackOptions.setBorder(emptyBorder);

				add(showFeedback);
				add("br vfill hfill", feedbackTabs);

				//putClientProperty("helpPage", helpRoot + "Sections/Feedback");
			}

		};
		
		// Commenting section
		
		ControlPanel commenting = new ControlPanel() {

			ControlPanel fbCommenting = new ControlPanel("Facebook") {

				JCheckBox facebookCommenting = new JCheckBox(getText("ui.facebookCommenting"));
				JTextField facebookCommentingPosts = new JSmartTextField("3", 2);

				{
					facebookCommentingPosts.setToolTipText(getText("ui.facebookCommentingPostsInfo"));

					add(facebookCommenting);
					add("br", new JLabel(getText("ui.facebookCommentingPosts")));
					add("tab", facebookCommentingPosts);
					add("br", new JLabel(infoIcon));
					add(new JLabel("<html><i>" + getText("ui.fbAppIdHint") + "</i></html>"));
				}

			};

			ControlPanel disqCommenting = new ControlPanel("Disqus") {

				JCheckBox disqusCommenting = new JCheckBox(getText("ui.disqusCommenting"));
				JTextField disqusAppId = new JSmartTextField("");
				JLabel mandatoryInfo = new JLabel(mandatory);
				
				{
					disqusAppId.setToolTipText(getText("ui.disqusAppIdInfo"));
					mandatoryInfo.setToolTipText(getText("ui.mandatory"));
					ComponentUtilities.whenSelectedEnable(disqusCommenting, new JComponent[]{disqusAppId, mandatoryInfo});

					add(disqusCommenting);
					add("br", new JLabelFor(getText("ui.disqusAppId"), disqusAppId));
					add(mandatoryInfo);
					add("tab hfill", disqusAppId);
					add("", new JLinkLabel("https://disqus.com/admin/create/", getText("ui.signUp")));
				}

			};

			{
				add("hfill", fbCommenting);
				add("br hfill", disqCommenting);

				//putClientProperty("helpPage", helpRoot + "Sections/Commenting");
			}
		};
		
		// Custom code section
		
		ControlPanel customSection = new ControlPanel() {

			JSmartTextArea customSectionHook = new JSmartTextArea(7, 20);
			JScrollPane customSectionPane = new JScrollPane(customSectionHook);
			JCheckBox customSectionTopLevelOnly = new JCheckBox(getText("ui.topLevelOnly"), false);

			{
				customSectionHook.setEditable(true);
				customSectionHook.setLineWrap(false);
				customSectionHook.setWrapStyleWord(true);
				customSectionHook.setFont(mono);
				customSectionHook.setTabSize(2);
				allowHTMLEditing(customSectionHook);
				//customSectionPane.setBorder(BorderFactory.createTitledBorder(getText("ui.customContentHint")));

				add(new JLabel(getText("ui.customContentHint")));
				add("br hfill vfill", customSectionPane);
				add("br", customSectionTopLevelOnly);
				
				//putClientProperty("helpPage", helpRoot + "Sections/Custom_content");
			}
		};
		
		/*	---------------------------------------------------------------
									Sidebar
			--------------------------------------------------------------- */
			
		ControlPanel tagCloudBox = new ControlPanel() {

			JComboBox tagCloudSource = new JComboBox(new Object[] {
				new Item("none", getText("ui.noTagCloud")),			
				new Item("current", getText("ui.currentFolder")),			
				new Item("subfolders", getText("ui.subfolders")),
				new Item("tree", getText("ui.wholeAlbum"))
			});
			JCheckBox tagCloudUseFolders = new JCheckBox(getText("ui.folders"), true);
			JCheckBox tagCloudUsePages = new JCheckBox(getText("ui.pages"));
			JCheckBox tagCloudUseWebLocations = new JCheckBox(getText("ui.webLocations"));
			JLabel tagCloudSkipLevelsLabel = new JLabel(getText("ui.skipLevels"));
			JSpinner tagCloudSkipLevels = new JSpinner(new SpinnerNumberModel(1, 0, 5, 1));
			JTextField tagCloudLabel = new JSmartTextField(20);
			JTextField tagCloudFields = new JSmartTextField();
			JComboBox tagCloudSort = new JComboBox(new Object[] {
				new Item("none", getText("ui.unsorted")),			
				new Item("name", getText("ui.name")),			
				new Item("frequency", getText("ui.frequency"))
			});
			JComboBox tagCloudSearch = new JComboBox(new Object[] {
				new Item("none", getText("ui.noSearchBox")),	
				new Item("above", getText("ui.above")),	
				new Item("below", getText("ui.below"))
			});
			JCheckBox tagCloudFontVaries = new JCheckBox(getText("ui.tagCloudFontVaries"));

			{
				tagCloudLabel.setText(getText("ui.labels"));
				tagCloudSource.addActionListener(new ActionListener() {
					@Override
					public void actionPerformed(ActionEvent e) {
						int tsi = tagCloudSource.getSelectedIndex();
						tagCloudUseFolders.setEnabled(tsi > 0);
						tagCloudUsePages.setEnabled(tsi > 0);
						tagCloudUseWebLocations.setEnabled(tsi > 0);
						tagCloudLabel.setEnabled(tsi > 0);
						tagCloudFields.setEnabled(tsi > 0);
						tagCloudSort.setEnabled(tsi > 0);
						tagCloudFontVaries.setEnabled(tsi > 0);
						tagCloudSearch.setEnabled(tsi > 0);
						tagCloudSkipLevels.setVisible(tsi == 2);
						tagCloudSkipLevelsLabel.setVisible(tsi == 2);
				}});
				tagCloudSkipLevels.setToolTipText(getText("ui.skipLevelsInfo"));
				tagCloudFields.setToolTipText("<html>" + getText("ui.searchFieldsInfo") + "</html>");
				tagCloudFontVaries.setToolTipText(getText("ui.tagCloudFontVariesInfo"));

				add(new JLabelFor(getText("ui.collectFrom"), tagCloudSource));
				add("tab", tagCloudSource);
				add("tab", tagCloudSkipLevelsLabel);
				add("", tagCloudSkipLevels);
				add("br tab", tagCloudUseFolders);
				add("", tagCloudUsePages);
				add("", tagCloudUseWebLocations);
				add("br", new JLabelFor(getText("ui.boxTitle"), tagCloudLabel));
				add("tab", tagCloudLabel);
				add("br", new JLabelFor(getText("ui.fields"), tagCloudFields));
				add("tab hfill", tagCloudFields);
				add("br", new JLabelFor(getText("ui.sortBy"), tagCloudFields));
				add("tab", tagCloudSort);
				add("tab", tagCloudFontVaries);
				add("br", new JLabelFor(getText("ui.addSearchBox"), tagCloudSearch));
				add("tab", tagCloudSearch);
			}
		};

		ControlPanel folderTreeBox = new ControlPanel() {

			JComboBox folderTreeDepth = new JComboBox(new Object[] {
				new Item("none", getText("ui.noFolderTree")),			
				new Item("all", getText("ui.allPages")),
				new Item("blog", getText("ui.onlyBlogPages"))/*,
				new Item("subfoldersOnly", getText("ui.subfoldersOnly"))*/
			});
			JComboBox folderTreeType = new JComboBox(new Object[] {
				new Item("tree", getText("ui.wholeAlbum")),
				new Item("subfolders", getText("ui.subfoldersOnly"))
			});
			JTextField folderTreeLabel = new JSmartTextField(24);
			JCheckBox folderTreeCollapsible = new JCheckBox(getText("ui.collapsible"));

			{
				//folderTreeLabel.setText(getText("ui.folderTree"));
				folderTreeLabel.setToolTipText(getText("ui.default") + ": " + getText("ui.folderTree"));
				folderTreeDepth.addActionListener(new ActionListener() {
					@Override
					public void actionPerformed(ActionEvent e) {
						boolean on = folderTreeDepth.getSelectedIndex() > 0;
						folderTreeType.setEnabled(on);
						folderTreeLabel.setEnabled(on);
						folderTreeCollapsible.setEnabled(on);
				}});

				add(new JLabelFor(getText("ui.showOn"), folderTreeDepth));
				add("tab", folderTreeDepth);
				add("br", new JLabelFor(getText("ui.collectFrom"), folderTreeType));
				add("tab", folderTreeType);
				add("br", new JLabelFor(getText("ui.boxTitle"), folderTreeLabel));
				add("tab", folderTreeLabel);
				add("br", folderTreeCollapsible);
			}
		};

		ControlPanel searchNewBox = new ControlPanel() {

			JComboBox searchNewSource = new JComboBox(new Object[] {
				new Item("none", getText("ui.noSearchNew")),			
				new Item("current", getText("ui.currentFolder")),			
				new Item("subfolders", getText("ui.subfolders")),
				new Item("tree", getText("ui.wholeAlbum"))
			});
			JTextField searchNewLabel = new JSmartTextField(20);
			JTextField searchNewDays = new JSmartTextField(20);
			JComboBox searchNewReference = new JComboBox(new Object[] {
				new Item("dateTaken", getText("ui.dateTaken")),
				new Item("fileModified", getText("ui.fileModified")),
				new Item("added", getText("ui.addedToAlbum"))
			});
			JCheckBox searchNewSinceLastVisit = new JCheckBox(getText("ui.searchNewSinceLastVisit"));

			{
				searchNewLabel.setToolTipText(getText("ui.default") + ": " + getText("ui.newImages"));
				searchNewSource.addActionListener(new ActionListener() {
					@Override
					public void actionPerformed(ActionEvent e) {
						boolean on = searchNewSource.getSelectedIndex() > 0;
						searchNewLabel.setEnabled(on);
						searchNewDays.setEnabled(on);
						searchNewReference.setEnabled(on);
						searchNewSinceLastVisit.setEnabled(on);
				}});
				searchNewLabel.setText(getText("ui.newImages"));
				searchNewDays.setToolTipText(getText("ui.searchNewDaysInfo"));
				searchNewSinceLastVisit.setToolTipText(getText("ui.searchNewSinceLastVisitInfo"));

				add(new JLabelFor(getText("ui.collectFrom"), searchNewSource));
				add("tab", searchNewSource);
				add("br", new JLabelFor(getText("ui.boxTitle"), searchNewLabel));
				add("tab", searchNewLabel);
				add("br", new JLabelFor(getText("ui.days"), searchNewDays));
				add("tab", searchNewDays);
				add("br", new JLabelFor(getText("ui.reference"), searchNewReference));
				add("tab", searchNewReference);
				add("br tab", searchNewSinceLastVisit);
			}
		};

		// Filtering box
		
		ControlPanel filteringBox = new ControlPanel() {
									
			ControlPanel filtering = new ControlPanel() {
				
				JCheckBox useFilters = new JCheckBox(getText("ui.useFilters"));
				JTextField filterLabel = new JSmartTextField(getText("ui.filter"), 16);
				JCheckBox showEmptyFilters = new JCheckBox(getText("ui.stayVisibleIfNothingToFilter"));
				JVariablesPanel filtersPanel = new JVariablesPanel(texts);
				JTable filterData = filtersPanel.getTable();
				
				{
					showEmptyFilters.setToolTipText(getText("ui.stayVisibleIfNothingToFilterInfo"));
					filtersPanel.setDimensions(new Dimension(320, 160));
					StateMonitor.monitoring(useFilters).onChange(src -> {
						boolean uf = useFilters.isSelected();
						filterLabel.setEnabled(uf);
						showEmptyFilters.setEnabled(uf);
						filtersPanel.setEnabled(uf);
					});
					add("", useFilters);
					add("tab", new JLabelFor(getText("ui.boxTitle"), filterLabel));
					add("tab", filterLabel);
					add("tab", showEmptyFilters);
					add("br hfill", filtersPanel);
				}
			};
			
			ControlPanel sorting = new ControlPanel() {
				
				JCheckBox useSort = new JCheckBox(getText("ui.useSort"));
				JTextField sortLabel = new JSmartTextField(getText("ui.sortBy"), 16);
				JVariablesPanel sortPanel = new JVariablesPanel(texts);
				JTable sortData = sortPanel.getTable();
				
				{
					sortPanel.setDimensions(new Dimension(320, 80));
					StateMonitor.monitoring(useSort).onChange(src -> {
						boolean us = useSort.isSelected();
						sortLabel.setEnabled(us);
						sortPanel.setEnabled(us);
					});
					add("", useSort);
					add("tab", new JLabelFor(getText("ui.boxTitle"), sortLabel));
					add("tab", sortLabel);
					add("br hfill", sortPanel);
				}
			};
			
			{
				filtering.setBorder(emptyBorder);
				sorting.setBorder(emptyBorder);
				
				add("br hfill", filtering);
				add("br hfill", sorting);
				if (jalbumVersion.compareTo(new VersionNumber("18.3b4")) < 0) {
					add("br", new JLabel(getText("ui.cantSaveThisWithVersion").replace("{0}", "18.3")));
				}
				
				//putClientProperty("helpPage", helpRoot + "Sections/Filtering");
			}
		};
		
		ControlPanel downloadBox = new ControlPanel() {

			JTextField downloadLabel = new JSmartTextField(24);
			JComboBox zipImages = new JComboBox(new Object[] {
				new Item("none", getText("ui.nothing")),
				new Item("slides", getText("ui.scaledDown")),
				new Item("originals", getText("ui.originals")),
				new Item("included", getText("ui.includedOriginals"))
			}); 
			JLabel zipInfo = new JLabel(infoIcon);
			
			{
				downloadLabel.setToolTipText(getText("ui.default") + ": " + getText("ui.downloadImages"));
				zipInfo.addMouseListener(new MouseAdapter() {  
					@Override
					public void mouseReleased(MouseEvent e) {
						JOptionPane.showMessageDialog(window, getText("ui.nonAsciiWarning"), "Warning", JOptionPane.WARNING_MESSAGE);
				}});
				
				add(new JLabelFor(getText("ui.boxTitle"), downloadLabel));
				add("tab", downloadLabel);
				add("br", new JLabelFor(getText("ui.offerDownload"), zipImages));
				add("", zipImages);
				add("", zipInfo);

			}
		};

		ControlPanel externalLinksBox = new ControlPanel() {

			JTextField externalLinksLabel = new JSmartTextField(20);
			JTextArea externalLinks = new JSmartTextArea(5, 30);
			JScrollPane externalLinksPane = new JScrollPane(externalLinks);

			{
				externalLinksLabel.setToolTipText(getText("ui.default") + ": " + getText("ui.links"));
				externalLinks.setText("");
				externalLinks.setEditable(true);
				externalLinks.setLineWrap(true);
				externalLinks.setWrapStyleWord(true);
				externalLinks.setFont(mono);
				externalLinksPane.setPreferredSize(new Dimension(uiWidth + previewWidth - tabWidth - 40, uiHeight - 100));
				
				add(new JLabelFor(getText("ui.boxTitle"), externalLinksLabel));
				add("tab", externalLinksLabel);
				add("br", new JLabel(getText("ui.externalLinksInfo")));
				add("br", externalLinksPane);

			}
		};
			
		ControlPanel mapBox = new ControlPanel() {
						
			{
				mapInSidebar.addItemListener(listener -> {
					if (mapInSidebar.isSelected()) {
						_mapPlacement.setSelectedIndex(2);
						mapInSections.setSelected(false);
					} else {
						_mapPlacement.setSelectedIndex(mapInSections.isSelected()? 1 : 0);
					}
				});
				
				add("", mapInSidebar);
				add("br", new JLabel(infoIcon));
				add(new JLabel("<html><i>" + getText("ui.findCorrespondingSettings") + " " + getText("ui.site") + " / " + getText("ui.map") + "</i></html>"));
			}
		};
		
		/*
		ControlPanel shopBox = new ControlPanel() {
			
			JCheckBox shopInSidebar = new JCheckBox(getText("ui.showShopInSidebar"), false);
			
			{
				if (license.length() == 0) {
					shopInSidebar.setSelected(false);
					commercialMonitor.add(shopInSidebar);
				}
				add("", shopInSidebar);
				add("br", new JLabel(infoIcon));
				add(new JLabel("<html><i>" + getText("ui.findCorrespondingSettings") + " " + getText("ui.site") + " / " + getText("ui.shoppingCart") + "</i></html>"));
			}
		};
		*/
		
		ControlPanel facebookBox = new ControlPanel() {

			JCheckBox useFacebookBox = new JCheckBox(getText("ui.useFacebookBox"));
			JTextField facebookLabel = new JSmartTextField("Facebook", 20);
			JTextField facebookPageId = new JSmartTextField(20);
			JLabel mandatoryInfo = new JLabel(mandatory);
			JCheckBox facebookFaces = new JCheckBox(getText("ui.showFaces"));
			JCheckBox facebookHeader = new JCheckBox(getText("ui.showHeader"));

			{
				mandatoryInfo.setToolTipText(getText("ui.mandatory"));
				ComponentUtilities.whenSelectedEnable(useFacebookBox, new JComponent[] {facebookLabel, facebookPageId, mandatoryInfo, facebookFaces, facebookHeader});
				
				add(useFacebookBox);
				add("br", new JLabelFor(getText("ui.boxTitle"), facebookLabel));
				add("tab", facebookLabel);
				add("br", new JLabelFor(getText("ui.facebookPageId"), facebookPageId));
				add(mandatoryInfo);
				add("tab", facebookPageId);
				add("br", facebookHeader);
				add("tab", facebookFaces);
			}

		};

		ControlPanel twitterBox = new ControlPanel() {

			JCheckBox useTwitterBox = new JCheckBox(getText("ui.useTwitterBox"), false);
			JTextField twitterLabel = new JSmartTextField("Twitter", 16);
			//JTextField twitterId = new JSmartTextField(16);
			//JTextField twitterWidgetId = new JSmartTextField(16);
			JTextArea twitterCode = new JSmartTextArea(5, 20);
			JScrollPane twitterCodePane = new JScrollPane(twitterCode);
			JSpinner twitterPostCount = new JSpinner(new SpinnerNumberModel(5, 1, 20, 1));
			JLabel twitterHint = new JLabel(getText("ui.twitterHint"));

			{
				twitterCode.setEditable(true);
				twitterCode.setLineWrap(true);
				twitterCode.setWrapStyleWord(true);
				twitterCode.setFont(mono);
				twitterCode.setTabSize(2);
				ComponentUtilities.whenSelectedEnable(useTwitterBox, new JComponent[] {twitterLabel, twitterCode, twitterHint, twitterPostCount});

				add(useTwitterBox);
				add("br", new JLabelFor(getText("ui.boxTitle"), twitterLabel));
				add("tab", twitterLabel);
				add("br", new JLabelFor(getText("ui.numberOfPosts"), twitterPostCount));
				add("tab", twitterPostCount);
				add("br", new JLinkLabel("https://publish.twitter.com/", getText("ui.createNew")));
				add("br", twitterHint);
				add("br hfill", twitterCodePane);
			}

		};
		
		ControlPanel instagramBox = new ControlPanel() {

			JCheckBox useInstagramBox = new JCheckBox(getText("ui.useInstagramBox"), false);
			JTextField instagramLabel = new JSmartTextField("Instagram", 20);
			JTextField instagramClientId = new JSmartTextField(20);
			JLinkLabel clientIdLink = new JLinkLabel("https://www.instagram.com/developer/clients/manage/", getText("ui.createNew"));
			JTextField instagramAccessToken = new JSmartTextField(20);
			JButton generateToken = new JButton(getText("ui.generateToken"));
			JSpinner instagramPostCount = new JSpinner(new SpinnerNumberModel(4, 2, 20, 1));

			{
				ComponentUtilities.whenSelectedEnable(useInstagramBox, new JComponent[] {instagramLabel, instagramClientId, clientIdLink, instagramAccessToken, generateToken, instagramPostCount});
				generateToken.addActionListener(new ActionListener() { 
					@Override
					public void actionPerformed(ActionEvent e) {
						String cid = instagramClientId.getText();
						if (cid!= null && cid.length() > 12) {
							try {
							   Desktop.getDesktop().browse(new URL("https://instagram.com/oauth/authorize/?client_id=" + cid + "&redirect_uri=http://instafeedjs.com&response_type=token&scope=public_content").toURI());
							} catch (IOException | URISyntaxException ex) {
								ex.printStackTrace();
							}
						}
				}});
				
				add(useInstagramBox);
				add("br", new JLabelFor(getText("ui.boxTitle"), instagramLabel));
				add("tab", instagramLabel);
				add("br", new JLabelFor(getText("ui.instagramClientId"), instagramClientId));
				add("tab", instagramClientId);
				add("", clientIdLink);
				add("br", new JLabelFor(getText("ui.instagramAcessToken"), instagramAccessToken));
				add("tab", instagramAccessToken);
				add("", generateToken);
				//add("", new JLinkLabel("http://instagram.com/developer/clients/register/", getText("ui.createNew")));
				add("br", new JLabelFor(getText("ui.numberOfPosts"), instagramPostCount));
				add("tab", instagramPostCount);
			}

		};
		
		ControlPanel customSidebar = new ControlPanel() {

			JSmartTextArea sidebarHook = new JSmartTextArea(7, 20);
			JScrollPane sidebarHookPane = new JScrollPane(sidebarHook);

			{
				sidebarHook.setEditable(true);
				sidebarHook.setLineWrap(false);
				sidebarHook.setWrapStyleWord(true);
				sidebarHook.setFont(mono);
				sidebarHook.setTabSize(2);
				allowHTMLEditing(sidebarHook);
				//sidebarHook.setBorder(BorderFactory.createTitledBorder(getText("ui.sidebarText")));

				add("", new JLabel(getText("ui.sidebarText")));
				add("br hfill vfill", sidebarHookPane);
				
				//putClientProperty("helpPage", helpRoot + "Sections/Custom_content");
			}
		};
		
		/*	---------------------------------------------------------------
									Footer
			--------------------------------------------------------------- */
		
		ControlPanel footerSettings = new ControlPanel() {

			ControlPanel bottomNavigationPanel = new ControlPanel(getText("ui.bottomNavigation")) {
				//JCheckBox showBottomNavigation = new JCheckBox(getText("ui.showBottomNavigation"), false);	
				JCheckBox bottomNavigationIncludeFolders = new JCheckBox(getText("ui.includeFolders"), true);
				JCheckBox bottomNavigationIncludePages = new JCheckBox(getText("ui.includePages"), true);
				JCheckBox bottomNavigationIncludeWebLocations = new JCheckBox(getText("ui.includeWebLocations"), true);
				
				@Override
				public void importVariables(Map<String, Object> vars) {
					Object sbn = vars.get("showBottomNavigation");
					if (sbn != null && vars.get("bottomNavigationIncludeFolders") == null) {
						vars.put("bottomNavigationIncludeFolders", sbn);
						vars.put("bottomNavigationIncludePages", sbn);
						vars.put("bottomNavigationIncludeWebLocations", sbn);
					}
				};
				
				{
					//showBottomNavigation.setToolTipText(getText("ui.showBottomNavigationInfo"));
					//add("", showBottomNavigation);
					add("", bottomNavigationIncludeFolders);
					add("br", bottomNavigationIncludePages);
					add("br", bottomNavigationIncludeWebLocations);
				}
			};

			ControlPanel customLinkPanel = new ControlPanel(getText("ui.customLink")) {

				JTextField customLink = new JSmartTextField();
				JTextField customLinkText = new JSmartTextField();

				{
					customLink.setToolTipText(getText("ui.customLinkInfo"));
					customLinkText.setToolTipText(getText("ui.customLinkTextInfo"));

					add(new JLabel("URL"));
					add("tab hfill", customLink);
					add("br", new JLabel(getText("ui.customLinkText")));
					add("tab hfill", customLinkText);
				}
			};

			JCheckBox showModifiedDate = new JCheckBox(getText("ui.showModifiedDate"));	
			JCheckBox showImageCount = new JCheckBox(getText("ui.showFolderImageCount"));	

			ControlPanel customContent = new ControlPanel(getText("ui.customContent")) {
				JSmartTextArea footer = new JSmartTextArea(6, 30);
				JScrollPane footerPane = new JScrollPane(footer);
				JCheckBox footerFitWidth = new JCheckBox(getText("ui.constrainPageWidth"), true);
				JCheckBox footerTopLevelOnly = new JCheckBox(getText("ui.topLevelOnly"), false);
			
				{
					footer.setEditable(true);
					footer.setLineWrap(true);
					footer.setWrapStyleWord(true);
					footer.setFont(mono);
					footer.setTabSize(2);
					allowHTMLEditing(footer);
					footer.setToolTipText(getText("ui.customContentInfo"));
					//footerPane.setBorder(BorderFactory.createTitledBorder(getText("ui.pageBottomText")));
					
					add("br hfill vfill", footerPane);
					add("br", footerFitWidth);
					add("tab", footerTopLevelOnly);
					
				}
			};
			
			{
				showImageCount.setToolTipText(getText("ui.showFolderImageCountInfo"));

				add("hfill", bottomNavigationPanel);
				add("br hfill", customLinkPanel);				
				add("br", showModifiedDate);
				add("tab", showImageCount);
				add("br hfill vfill", customContent);
				
				//putClientProperty("helpPage", helpRoot + "Footer/Settings");
			}

		};
			
		/*	---------------------------------------------------------------
									Lightbox
			--------------------------------------------------------------- */
				
		ControlPanel lightboxSettings = new ControlPanel() {
			
			ControlPanel behaviorPanel = new ControlPanel(getText("ui.behavior")) {

				JCheckBox lightboxFullscreen = new JCheckBox(getText("ui.lightboxFullscreen"));
				JCheckBox slideshowFullscreen = new JCheckBox(getText("ui.slideshowFullscreen"));
				JCheckBox showFullscreen = new JCheckBox(getText("ui.showFullscreenButton"));
				JCheckBox videoAutoPlay = new JCheckBox(getText("ui.autoPlayVideos"));
				JLabel videoAutoPlayInfo = new JLabel(infoIcon);
				JCheckBox videoLoop = new JCheckBox(getText("ui.loopVideos"));
				JCheckBox autohideControls = new JCheckBox(getText("ui.autohideControls"));
				JCheckBox useAutoPano = new JCheckBox(getText("ui.enableAutoPano"));
				JLabel autoPanoInfo = new JLabel(infoIcon);
				JLabel autoPanoStartLabel = new JLabel(getText("ui.startAt"));
				JComboBox autoPanoStart = new JComboBox(new Object[] {
					new Item("left", getText("ui.leftOrTop")), 
					new Item("center", getText("ui.center")),
					new Item("right", getText("ui.rightOrBottom"))
				});
				JLabel autoPanoSpeedLabel = new JLabel(getText("ui.speed"));
				JComboBox autoPanoSpeed = new JComboBox(new Object[] {
					new Item("veryslow", getText("ui.verySlow")),
					new Item("slow", getText("ui.slow")), 
					new Item("moderate", getText("ui.moderate")),
					new Item("medium", getText("ui.medium")),
					new Item("quick", getText("ui.quick")),
					new Item("fast", getText("ui.fast"))
				});
				JCheckBox use360Player = new JCheckBox(getText("ui.use360Player"));
				JCheckBox autoRotate360 = new JCheckBox(getText("ui.autoRotate"));
				JSpinner rotateSpeed360 = new JSpinner(new SpinnerNumberModel(-2, -8, 8, 1));
				JLabel rpm = new JLabel(getText("ui.rpm"));
				JCheckBox useGoogleDocs = new JCheckBox(getText("ui.useGoogleDocsViewer"));
				JSpinner transitionSpeed = new JSpinner(new SpinnerNumberModel(400, 100, 5000, 100));
				JSpinner slideshowDelay = new JSpinner(new SpinnerNumberModel(4000, 500, 10000, 100));
				//JCheckBox slideshowAuto = new JCheckBox(getText("ui.autoStart"));
				JCheckBox slideshowTimingControl = new JCheckBox(getText("ui.canChangeSlideshowTiming"));

				ControlPanel autoPanoSettings = new ControlPanel() {
					{
						add("", autoPanoStartLabel);
						add("tab", autoPanoStart);
						add(" ", autoPanoSpeedLabel);
						add("tab", autoPanoSpeed);
					}
				};
			
				{
					ComponentUtilities.whenSelectedDisable(lightboxFullscreen, new JComponent[]{slideshowFullscreen, showFullscreen});
					videoAutoPlayInfo.addMouseListener(new MouseAdapter() {  
						@Override
						public void mouseReleased(MouseEvent e) {
							JOptionPane.showMessageDialog(window, getText("ui.videoAutoplayInfo1"), "Warning", JOptionPane.WARNING_MESSAGE);
					}});
					useAutoPano.addItemListener(e -> {
						if (isSkinReady() && e.getStateChange() == ItemEvent.SELECTED) {
							String idim = engine.getImageSize();
							int iw = Integer.parseInt(idim.split("x")[0]);
							int ih = Integer.parseInt(idim.split("x")[1]);

							if (iw < 3000) {
								Object[] options = { 
									getText("yes"),
									getText("no")
								};
								int n = JOptionPane.showOptionDialog(window, 
									getText("ui.autoPanoInfo") + "\n\n" + getText("ui.changeImageDimensionsQuestion"), 
									getText("ui.imageDimensionsTooSmall"), 
									JOptionPane.YES_NO_OPTION, 
									JOptionPane.INFORMATION_MESSAGE,
									null,
									options,
									options[0]
								);
								if (n == 0) {
									try {
										window.ui2Engine();
										engine.setImageSize("4000x"+ih);
										JAlbum.logger.log(Level.FINE, "Image width has changed from {0}px to {1}px.", new Object[]{iw, "4000"});
										window.engine2UI();
									} catch (ParameterException ex) {
										throw new RuntimeException(ex);
									}
									useAutoPano.setSelected(true);
								}
							}
						}
					});
					autoPanoInfo.addMouseListener(new MouseAdapter() {  
						@Override
						public void mouseReleased(MouseEvent e) {
							JOptionPane.showMessageDialog(window, getText("ui.autoPanoInfo"), "Warning", JOptionPane.WARNING_MESSAGE);
					}});
					slideshowFullscreen.setToolTipText(getText("ui.slideshowFullScreenInfo"));
					autohideControls.setToolTipText(getText("ui.autohideControlsInfo"));
					useAutoPano.setToolTipText(getText("ui.autoDetectPanoramasInfo"));
					ComponentUtilities.whenSelectedEnable(useAutoPano, new JComponent[]{autoPanoSettings});
					use360Player.setToolTipText(getText("ui.use360PlayerInfo"));
					ComponentUtilities.whenSelectedEnable(use360Player, new JComponent[]{autoRotate360, rotateSpeed360, rpm});
					useGoogleDocs.setToolTipText(getText("ui.useGoogleDocsInfo"));
					transitionSpeed.setToolTipText(getText("ui.transitionSpeedInfo"));
					slideshowDelay.setToolTipText(getText("ui.slideshowDelayInfo"));
					slideshowTimingControl.setToolTipText(getText("ui.slideshowTimingControlInfo"));
					
					add("", lightboxFullscreen);
					add("tab", slideshowFullscreen);
					add("br", showFullscreen);
					add("br", videoAutoPlay);
					add("", videoAutoPlayInfo);
					add("tab", videoLoop);
					add("br", autohideControls);
					add("br", useAutoPano);
					add("", autoPanoInfo);
					add("br", autoPanoSettings);
					add("br", use360Player);
					add("tab", autoRotate360);
					add("tab", rotateSpeed360);
					add(rpm);
					add("br", useGoogleDocs);
					add("br", new JLabelFor(getText("ui.transitionSpeed"), transitionSpeed));
					add("tab", transitionSpeed);
					add(new JLabel("ms"));
					add("br", new JLabel(getText("ui.slideshowTiming")));
					add("tab", slideshowDelay);
					add(new JLabel("ms"));
					add(" ", slideshowTimingControl);
					//add(" ", slideshowLoop);
					//add("tab", slideshowAuto);
				}
			};
			
			ControlPanel navigation = new ControlPanel(getText("ui.navigation")) {
				JComboBox mouseWheelAction = new JComboBox(new Object[] {
					new Item("", getText("ui.default")), 
					new Item("navigation", getText("ui.navigation")),
					new Item("zoom", getText("ui.zoom"))
				});
				JCheckBox enableKeyboard = new JCheckBox(getText("ui.enableKeyboard"), true);
				JComboBox clickAction = new JComboBox(new Object[] {
					new Item("donothing", getText("ui.doNothing")), 
					new Item("nextimage", getText("ui.nextImage")),
					new Item("togglecontrols", getText("ui.toggleControls")),
					new Item("toggleall", getText("ui.toggleAllPanels"))
				});
				JCheckBox clickBesideForIndex = new JCheckBox(getText("ui.clickBesideForIndex"));
				JComboBox afterLast = new JComboBox(new Object[] {
					new Item("donothing", getText("ui.doNothing")), 
					new Item("startover", getText("startOver")),
					new Item("onelevelup", getText("upOneLevel")),
					new Item("backtoindex", getText("backToIndex")),
					new Item("nextfolder", getText("nextFoldersFirstImage")),
					new Item("nextindex", getText("nextIndex")),
					new Item("ask", getText("ui.ask"))
				});

				{
					mouseWheelAction.setToolTipText(getText("ui.mouseWheelActionInfo"));
					enableKeyboard.setToolTipText(getText("ui.enableKeyboardInfo"));
					clickBesideForIndex.setToolTipText(getText("ui.clickBesideForIndexInfo"));
					afterLast.setToolTipText(getText("ui.afterLastInfo"));
					
					add("", new JLabelFor(getText("ui.mouseWheelAction"), mouseWheelAction));
					add("tab", mouseWheelAction);
					add("br", enableKeyboard);
					add("br", clickBesideForIndex);
					add("br", new JLabelFor(getText("ui.clickAction"), clickAction));
					add("tab", clickAction);
					add("br", new JLabelFor(getText("ui.afterLast"), afterLast));
					add("tab", afterLast);
				}
			};
			
			{
				add("hfill", behaviorPanel);
				add("br hfill", navigation);
			}
		};
		
		ControlPanel mainImage = new ControlPanel() {

			JCheckBox fitImage = new JCheckBox(getText("ui.fitImages"));
			JLabel fitImageInfo = new JLabel(infoIcon);
			JCheckBox fitShrinkOnly = new JCheckBox(getText("ui.onlyShrink"));
			JLabel maxZoomLabel = new JLabel(getText("ui.maxZoom"));
			JSpinner maxZoom = new JSpinner(new SpinnerNumberModel(1.4, 0.99, 3.0, 0.1));
			JSpinner fitPadding = new JSpinner(new SpinnerNumberModel(0, 0, 100, 1));
			JCheckBox fitAboveCaption = new JCheckBox(getText("ui.fitAboveCaption"));
			JCheckBox zoomSlider = new JCheckBox(getText("ui.useZoomSlider"));
			JCheckBox displayOriginals = new JCheckBox(getText("ui.displayOriginals"));
			JCheckBox rightClickProtect = new JCheckBox(getText("ui.rightClickProtect"));
			
			{
				((JSpinner.DefaultEditor)maxZoom.getEditor()).getTextField().setColumns(4);
				fitImage.setToolTipText(getText("ui.fitImagesInfo"));
				fitImageInfo.addMouseListener(new MouseAdapter() {  
					@Override
					public void mouseReleased(MouseEvent e) {
						JOptionPane.showMessageDialog(window, "<html><i>" + getText("ui.fitImageWarning") + "</i></html>", getText("ui.format"), JOptionPane.INFORMATION_MESSAGE);
				}});
				zoomSlider.setToolTipText(getText("ui.useZoomSliderInfo"));
				displayOriginals.setToolTipText(getText("ui.displayOriginalsInfo"));
				rightClickProtect.setToolTipText(getText("ui.rightClickProtectInfo"));
				new StateMonitor() {
					public void onChange() {
						Boolean fi = fitImage.isSelected();
						fitImageInfo.setVisible(!fi);
						fitShrinkOnly.setVisible(fi);
						maxZoomLabel.setVisible(fi);
						maxZoom.setVisible(fi);
					}
				}.add(fitImage).done();
				ComponentUtilities.whenSelectedDisable(fitShrinkOnly, new JComponent[]{maxZoomLabel, maxZoom});
				
				add("", fitImage);
				add(fitImageInfo);
				add("tab", fitShrinkOnly);
				add("tab", maxZoomLabel);
				add("", maxZoom);
				add("br", new JLabelFor(getText("ui.padding"), fitPadding));
				add("", fitPadding);
				add("", new JLabel("px"));
				add("br", fitAboveCaption);
				add("br", zoomSlider);
				add("br", displayOriginals);
				add("br", rightClickProtect);
			}
						
			//putClientProperty("helpPage", helpRoot + "Lightbox/Main_image");
		};
			
		ControlPanel lightboxCaption  = new ControlPanel() {
			
			JScrollPane imageCaptionPane = new JScrollPane(imageCaptionTemplate);
			JCheckBox showImageNumbers = new JCheckBox(getText("ui.showImageNumbers"));
			JCheckBox infoPanelAdapt = new JCheckBox(getText("ui.adaptToContent"));

			{
				imageCaptionPresets.addItemListener(listener -> {
					String s = getSelectedValue(imageCaptionPresets);
					if (!s.equals(CUSTOM)) {
						imageCaptionTemplate.setText(s);
					}
				});	
				new StateMonitor() {
					public void onChange() {
						String s = getSelectedValue(imageCaptionPresets);
						imageCaptionPane.setVisible(s.equals(CUSTOM));
					}
				}.add(imageCaptionPresets).done();						
				imageCaptionTemplate.setEditable(true);
				imageCaptionTemplate.setLineWrap(true);
				imageCaptionTemplate.setWrapStyleWord(true);
				imageCaptionTemplate.setFont(mono);
				showImageNumbers.setToolTipText(getText("ui.showImageNumbersInfo"));
				infoPanelAdapt.setToolTipText(getText("ui.adaptToContentInfo"));

				add("", new JLabelFor(getText("ui.caption"), imageCaptionPresets));
				add("tab", imageCaptionPresets);
				add("br hfill", imageCaptionPane);
				add("br", showImageNumbers);
				add("br", infoPanelAdapt);
			}
		};

		
		ControlPanel lightboxButtons = new ControlPanel() {
				
			JCheckBox buttonLabelsVisible = new JCheckBox(getText("ui.buttonLabelsVisible"));
			JCheckBox showShopBtn = new JCheckBox(getText("ui.shoppingCart"));
			JCheckBox showFeedbackBtn = new JCheckBox(getText("ui.feedback"));
			JCheckBox showRegions = new JCheckBox(getText("ui.regions"), true);
			private JLabel regionsLabel = new JLabel(getText("ui.buttonLabel"));
			JCheckBox regionsVisible = new JCheckBox(getText("ui.visibleByDefault"));
			JTextField regionsBtnText = new JTextField(getText("regionsBtn"), 8);
			JCheckBox regionsSkipEmpty = new JCheckBox(getText("ui.skipEmptyTags"));
			JCheckBox showShare = new JCheckBox(getText("ui.share"), true);
			JCheckBox printImageButton = new JCheckBox(getText("ui.print"));
			JCheckBox showDownload = new JCheckBox(getText("ui.download"));
			JCheckBox downloadNonImages = new JCheckBox(getText("ui.enableDownloadNonImages"));
			JCheckBox downloadScaled = new JCheckBox(getText("ui.enableDownloadScaledImages"));
			JCheckBox useFotomoto = new JCheckBox("Fotomoto");
			JTextField fotomotoID = new JSmartTextField(24);
			JLabel mandatoryInfo = new JLabel(mandatory);
			JCheckBox showPhotoData = new JCheckBox(getText("ui.showPhotoData"));

			ControlPanel photoData = new ControlPanel() {

				JCheckBox showPhotoDataLabel = new JCheckBox(getText("ui.showLabel"), true);
				JTextArea photoDataTemplate = new JSmartTextArea(10, 120);
				JScrollPane photoDataPane = new JScrollPane(photoDataTemplate);
				JButton reset = new JButton(getText("ui.resetToDefaults"));
				JLabel listMetadataHint = new JLabel("<html><i>" + getText("ui.listMetadataHint") + "</i></html>");

				{
					showPhotoData.setToolTipText(getText("ui.showPhotoDataInfo"));
					showPhotoDataLabel.setToolTipText(getText("ui.showLabelInfo"));
					photoDataTemplate.setEditable(true);
					photoDataTemplate.setLineWrap(false);
					//photoDataTemplate.setWrapStyleWord(true);
					photoDataTemplate.setFont(mono);
					reset.addMouseListener(new MouseAdapter() {  
						@Override
						public void mouseReleased(MouseEvent e) {
							String s = new SkinModel().photoDataTemplate;
							if (s != null && s.length() > 0) {
								photoDataTemplate.setText(s);
							}
					}});				

					ComponentUtilities.whenSelectedEnable(showPhotoData, new JComponent[]{showPhotoDataLabel, photoDataTemplate, reset, listMetadataHint});

					add("", showPhotoDataLabel);
					add(" ", reset);
					add("br vfill", photoDataPane);
					add("br", listMetadataHint);

					//putClientProperty("helpPage", helpRoot + "Lightbox/Photo_data");
				}
			};
			
			{
				buttonLabelsVisible.setToolTipText(getText("ui.buttonLabelsVisibleInfo"));
				printImageButton.setToolTipText(getText("ui.printImageButtonInfo"));
				showMapBtn.setToolTipText(getText("ui.showMapInfo"));
				showMapBtn.addItemListener(listener -> {
					_showMapBtn.setSelected(showMapBtn.isSelected());
				});
				showShopBtn.setToolTipText(getText("ui.showShopInfo"));
				showFeedbackBtn.setToolTipText(getText("ui.showFeedbackInfo"));
				showDownload.setToolTipText(getText("ui.showDownloadInfo"));
				ComponentUtilities.whenSelectedEnable(showDownload, new JComponent[]{downloadScaled, downloadNonImages});
				downloadNonImages.setToolTipText(getText("ui.enableDownloadNonImagesInfo"));
				showRegions.setToolTipText(getText("ui.showRegionsInfo"));
				regionsVisible.setToolTipText(getText("ui.regionsVisibleInfo"));
				ComponentUtilities.whenSelectedEnable(showRegions, new JComponent[]{regionsLabel, regionsBtnText, regionsSkipEmpty});
				regionsBtnText.setToolTipText(getText("ui.default") + ": " + getText("ui.faces"));
				showShare.setToolTipText(getText("ui.showShareInfo"));
				printImageButton.setToolTipText(getText("ui.printImageButtonInfo"));
				downloadNonImages.setToolTipText(getText("ui.enableDownloadNonImagesInfo"));
				fotomotoID.setToolTipText(getText("ui.fotomotoIDInfo"));
				mandatoryInfo.setToolTipText(getText("ui.mandatory"));
				ComponentUtilities.whenSelectedEnable(useFotomoto, new JComponent[]{fotomotoID, mandatoryInfo});
				ComponentUtilities.whenSelectedEnable(showPhotoData, photoData);

				add("", buttonLabelsVisible);
				add("br", new JLabel(icon("location")));
				add("tab", showMapBtn);
				add("br", new JLabel(icon("shopping-cart")));
				add("tab", showShopBtn);
				add("br", new JLabel(icon("email-send")));
				add("tab", showFeedbackBtn);
				add("br", new JLabel(icon("user")));
				add("tab", showRegions);
				add("tab", regionsVisible);
				add("br tab", regionsLabel);
				add(regionsBtnText);
				add(" ", regionsSkipEmpty);
				add("br", new JLabel(icon("connect")));
				add("tab", showShare);
				add("br", new JLabel(icon("printer")));
				add("tab", printImageButton);
				add("br", new JLabel(icon("download")));
				add("tab", showDownload);
				add("tab", downloadScaled);
				add("tab", downloadNonImages);
				add("br", new JLabel(icon("fotomoto")));
				add("tab", useFotomoto);
				add("tab", new JLabelFor(getText("ui.storeId"), fotomotoID));
				add(mandatoryInfo);
				add("", fotomotoID);
				add(new JLinkLabel("https://my.fotomoto.com/signup", getText("ui.signUp")));
				add("br", new JLabel(icon("camera")));
				add("tab", showPhotoData);
				add("br hfill", photoData);

			}
						
		};
		
		/*	---------------------------------------------------------------
									Advanced
			--------------------------------------------------------------- */

		//	Custom code
		
		ControlPanel customCodePanel = new ControlPanel() {

			ControlPanel headHookTab = new ControlPanel() {

				JTextArea headHook = new JSmartTextArea(7, 20);
				JScrollPane headHookPane = new JScrollPane(headHook);

				{
					headHook.setEditable(true);
					headHook.setLineWrap(false);
					headHook.setFont(mono);
					headHook.setTabSize(2);
					headHookPane.setBorder(BorderFactory.createTitledBorder(getText("ui.headText")));

					add("hfill vfill", headHookPane);
				}
			};

			ControlPanel bodyHookTab = new ControlPanel() {

				JTextArea bodyHook = new JSmartTextArea(7, 20);
				JScrollPane bodyHookPane = new JScrollPane(bodyHook);

				{
					bodyHook.setEditable(true);
					bodyHook.setLineWrap(false);
					bodyHook.setFont(mono);
					bodyHook.setTabSize(2);
					bodyHookPane.setBorder(BorderFactory.createTitledBorder(getText("ui.bodyText")));

					add("hfill vfill", bodyHookPane);
				}
			};

			ControlPanel cssHookTab = new ControlPanel() {

				JTextArea cssHook = new JSmartTextArea(15, 20);
				JScrollPane cssHookPane = new JScrollPane(cssHook);
				WrappableJLabel info = new WrappableJLabel("<html><i>" + getText("ui.cssText") + "</i></html>");
				
				{
					cssHook.setEditable(true);
					cssHook.setLineWrap(false);
					cssHook.setFont(mono);
					cssHook.setTabSize(2);
					cssHookPane.setBorder(BorderFactory.createTitledBorder(getText("ui.cssTitle")));
					info.setPreferredWidth(uiWidth + previewWidth - 40);
					
					add("hfill vfill", cssHookPane);
					add("br", info);
				}
			};

			ControlPanel jsHookTab = new ControlPanel() {

				JTextArea jsHook = new JSmartTextArea(15, 20);
				JScrollPane jsHookPane = new JScrollPane(jsHook);
				WrappableJLabel info = new WrappableJLabel("<html><i>" + getText("ui.javascriptText") + "</i></html>");
				
				{
					jsHook.setEditable(true);
					jsHook.setLineWrap(false);
					jsHook.setFont(mono);
					jsHook.setTabSize(2);
					jsHookPane.setBorder(BorderFactory.createTitledBorder(getText("ui.javascriptTitle")));
					info.setPreferredWidth(uiWidth + previewWidth - 40);
					
					add("hfill vfill", jsHookPane);
					add("br", info);
				}
			};

			JTabbedPane customCodeTabs = new JTabbedPane() {				

				{
					addTab("<HEAD>", icon("code"), headHookTab);
					addTab("<BODY>", icon("code"), bodyHookTab);
					addTab("CSS", icon("css"), cssHookTab);
					addTab("JavaScript", icon("javascript"), jsHookTab);
				}
			};

			{				
				add("hfill vfill", customCodeTabs);

				//putClientProperty("helpPage", helpRoot + "ui/site.html");
			}
		};
		/*						
		ControlPanel customContentPanel = new ControlPanel() {

			JSmartTextArea customContent = new JSmartTextArea(6, 20);
			JScrollPane customContentPane = new JScrollPane(customContent);
			JCheckBox customContentFitWidth = new JCheckBox(getText("ui.constrainPageWidth"));
			JCheckBox customContentTopLevelOnly = new JCheckBox(getText("ui.topLevelOnly"));

			{
				allowHTMLEditing(customContent);
				allowSpelling(customContent);
				
				customContent.setEditable(true);
				customContent.setLineWrap(true);
				customContent.setWrapStyleWord(true);
				customContent.setFont(mono);
				customContent.setTabSize(2);
				customContent.setToolTipText(getText("ui.customContentInfo"));
				customContentPane.setBorder(BorderFactory.createTitledBorder(getText("ui.pageBottomText")));

				add("hfill vfill", customContentPane);
				add("br", customContentFitWidth);
				add("tab", customContentTopLevelOnly);
				
				//putClientProperty("helpPage", helpRoot + "Footer/Custom_footer");

			}
		};
		*/
		//	Site admin
		
		ControlPanel siteAdminPanel = new ControlPanel() {

			ControlPanel serverRelatedPanel = new ControlPanel(getText("ui.serverRelated")) {
				
				JTextField uploadPath = new JSmartTextField(20);
				JCheckBox useFavicon = new JCheckBox(getText("ui.useTheSkinsFavicon"), true);				
				JCheckBox useMsServer = new JCheckBox(getText("ui.useMsServer"));
				JCheckBox useExpiry = new JCheckBox(getText("ui.useExpiry"), false);
				JCheckBox useRobotsTxt = new JCheckBox(getText("ui.useRobotsTxt"), false);
				JCheckBox avoidCDNs = new JCheckBox(getText("ui.avoidCDNs"), false);
				JCheckBox copyGoogleFonts = new JCheckBox(getText("ui.copyGoogleFonts"), false);
				JCheckBox addLatinExt = new JCheckBox(getText("ui.addLatinExtended"), true);
				WrappableJLabel uploadPathInfo = new WrappableJLabel("<html><i>" + getText("ui.copyPasteAlbumURLHere") + "</i></html>");
				
				{
					uploadPath.setToolTipText(getText("ui.uploadPathInfo"));
					uploadPathInfo.setPreferredWidth(360);
					useFavicon.setToolTipText(getText("ui.useFaviconInfo"));
					useMsServer.setToolTipText(getText("ui.useMsServerInfo"));
					useExpiry.setToolTipText(getText("ui.useExpiryInfo"));
					useRobotsTxt.setToolTipText(getText("ui.useRobotsTxtInfo"));
					avoidCDNs.setToolTipText(getText("ui.avoidCDNsInfo"));
					copyGoogleFonts.setToolTipText(getText("ui.copyGoogleFontsInfo"));
					addLatinExt.setToolTipText(getText("ui.addLatinExtendedInfo"));
						
					add(new JLabel(getText("ui.uploadPath")));
					add("", uploadPath);
					add("br", new JLabel(infoIcon));
					add("", uploadPathInfo);
					add("br", useFavicon);
					add(" ", new JLabel(svgIcon("logo-photoblogger", new Dimension(32, 32))));
					add("br", useMsServer);
					add("br", useExpiry);
					add("br", useRobotsTxt);
					add("br", avoidCDNs);
					add("br", copyGoogleFonts);
					add("br", addLatinExt);
				}
			};			
			
			ControlPanel imageRelatedPanel = new ControlPanel(getText("ui.images")) {
				
				JCheckBox hiDpiThemeImage = new JCheckBox(getText("ui.hiDpiThemeImage"));
				JComboBox shareImageDims = new JComboBox(new Object[] {
					"640x480",
					"1024x768",
					"1200x900",
					"1600x1200"
				});
				
				{
					shareImageDims.setEditable(true);
					
					add(hiDpiThemeImage);
					add("br", new JLabel(getText("ui.shareImageDimensions")));
					add("", shareImageDims);
				}
			};
			
			ControlPanel googlePanel = new ControlPanel(getText("ui.googleAnalytics")) {

				JComboBox googleAnalytics = new JComboBox(new Object[] {
					new Item("none", getText("ui.none")), 
					new Item("gtag", "Global Site Tag (gtag.js)"),
					new Item("universal", "Universal (analytics.js)"),
					new Item("classic", "Classic (ga.js) [Legacy]")
				});
				JTextField googleSiteID = new JSmartTextField(20);
				JLabel mandatoryInfo = new JLabel(mandatory);
				JCheckBox supportDoubleclick = new JCheckBox(getText("ui.supportDoubleclick"), false);

				{
					googleSiteID.setToolTipText(getText("ui.googleSiteIDInfo"));
					mandatoryInfo.setToolTipText(getText("ui.mandatory"));

					add("", new JLabelFor(getText("ui.version"), googleAnalytics));
					add(" ", googleAnalytics);
					add(" ", new JLinkLabel("https://www.google.com/analytics/", getText("ui.signUp")));
					add("br", new JLabelFor(getText("ui.googleSiteID"), googleSiteID));
					add(mandatoryInfo);
					add("", googleSiteID);
					add("br", supportDoubleclick);
				}
			};
			
			ControlPanel seoPanel = new ControlPanel(getText("ui.searchEngineOptimization")) {
				
				JTextField titleSEOText = new JSmartTextField(20);
				JTextField descriptionSEOText = new JSmartTextField(20);
				JSpinner preloadThumbs = new JSpinner(new SpinnerNumberModel(20, 0, 1000, 10));
				JCheckBox addAltTags = new JCheckBox(getText("ui.addAltTags"), true);
				JCheckBox slideRedirect = new JCheckBox(getText("ui.redirectSlidePages"), true);
			
				{
					titleSEOText.setToolTipText(getText("ui.titleSEOTextInfo"));
					descriptionSEOText.setToolTipText(getText("ui.descriptionSEOTextInfo"));
					//preloadThumbs.setToolTipText(getText("ui.preloadThumbsInfo"));
					addAltTags.setToolTipText(getText("ui.addAltTagsInfo"));
					writeSitemapXml.setToolTipText(getText("ui.createSitemapXmlInfo"));
					ComponentUtilities.whenSelectedEnable(writeSitemapXml, new JComponent[]{sitemapIncludeSlides});
					sitemapIncludeSlides.setToolTipText(getText("ui.sitemapIncludeSlidesInfo"));
					slideRedirect.setToolTipText(getText("ui.redirectSlidePagesInfo"));
					
					add("", new JLabelFor(getText("ui.titleSEOText"), titleSEOText));
					add("tab", titleSEOText);
					add("br", new JLabelFor(getText("ui.descriptionSEOText"), descriptionSEOText));
					add("tab", descriptionSEOText);
					add("br", new JLabel(getText("ui.preloadThumbs")));
					add("", preloadThumbs);
					add("br", addAltTags);
					add("br", writeSitemapXml);
					add("", sitemapIncludeSlides);
					add("br", slideRedirect);
				}
			};

			ControlPanel cookiePolicyPanel = new ControlPanel(getText("ui.trackingConsentAndCookiePolicy")) {

				JCheckBox askTrackingConsent = new JCheckBox(getText("ui.askTrackingConsent"), false);
				JCheckBox showCookiePolicy = new JCheckBox(getText("ui.showCookiePolicy"), false);				
				JTextField cookiePolicyStay = new JSmartTextField("8", 3);
				JTextField cookiePolicyUrl = new JSmartTextField(20);
				WrappableJLabel cookiePolicyUrlInfo = new WrappableJLabel("<html><i>" + getText("ui.cookiePolicyUrlInfo") + " </i></html>");

				{
					askTrackingConsent.setToolTipText(getText("ui.askTrackingConsentInfo"));
					showCookiePolicy.setToolTipText(getText("ui.showCookiePolicyInfo"));
					cookiePolicyUrlInfo.setPreferredWidth(320);
					
					add(askTrackingConsent);
					add("br", showCookiePolicy);
					add(" ", new JLabel(getText("ui.stay")));
					add("", cookiePolicyStay);
					add("", new JLabel("s"));
					add("br", new JLabel("URL"));
					add("tab", cookiePolicyUrl);
					add("br", new JLabel(infoIcon));
					add("tab", cookiePolicyUrlInfo);
				}
			};

			JCheckBox debugMode = new JCheckBox(getText("ui.debugMode"));

			ControlPanel leftPanel = new ControlPanel() {
				
				{
					add("hfill", serverRelatedPanel);
					add("br hfill", googlePanel);
					add("br", debugMode);					
				}
			};
			
			ControlPanel rightPanel = new ControlPanel() {
				
				{
					add("hfill", imageRelatedPanel);
					add("br hfill", seoPanel);
					add("br hfill", cookiePolicyPanel);
				}
			};
			
			{
				add(leftPanel);
				add(rightPanel);
				
				//putClientProperty("helpPage", helpRoot + "Site/Site_admin");
			}
		};
					
		//	Custom Keys
		
		ControlPanel customKeysPanel = new ControlPanel() {
		
			JTextArea customKeys = new JSmartTextArea(7, 20);
			JScrollPane customKeysPane = new JScrollPane(customKeys);

			{
				customKeys.setEditable(true);
				customKeys.setLineWrap(false);
				customKeys.setFont(mono);
				customKeys.setTabSize(2);
				customKeysPane.setBorder(BorderFactory.createTitledBorder(getText("ui.customKeys")));

				add("hfill vfill", customKeysPane);
				add("br", new JLabel(getText("ui.customKeysInfo")));
			}
		};
		
		
		/******************************************************************************
		 * 
		 *								GUI main tabs
		 * 
		 ******************************************************************************/				
		
		/*	---------------------------------------------------------------
									Site
			--------------------------------------------------------------- */

		ControlPanel site = new ControlPanel(new BorderLayout(20, 0)) {
			
			JTabbedPane siteTabs = new JTabbedPane() {
				
				JScrollPane topbarPanel = new JScrollPane(topbarTab, JScrollPane.VERTICAL_SCROLLBAR_AS_NEEDED, JScrollPane.HORIZONTAL_SCROLLBAR_NEVER);
				JScrollPane footerPanel = new JScrollPane(footerSettings, JScrollPane.VERTICAL_SCROLLBAR_AS_NEEDED, JScrollPane.HORIZONTAL_SCROLLBAR_NEVER);
				
				{
					topbarPanel.setBorder(emptyBorder);
					footerPanel.setBorder(emptyBorder);
					this.setTabPlacement(SwingConstants.LEFT);
					
					addTab("<html><h4 " + tabStyle + ">" + getText("ui.colors") + "</h4></html>", icon("color"), colors);
					addTab("<html><h4 " + tabStyle + ">" + getText("ui.design") + "</h4></html>", icon("design"), layout);
					addTab("<html><h4 " + tabStyle + ">" + getText("ui.typography") + "</h4></html>", icon("font"), typography);
					addTab("<html><h4 " + tabStyle + ">" + getText("ui.topBar") + "</h4></html>", icon("header"), topbarPanel);
					addTab("<html><h4 " + tabStyle + ">" + getText("ui.header") + "</h4></html>", icon("title"), heroTab);
					addTab("<html><h4 " + tabStyle + ">" + getText("ui.slider") + "</h4></html>", icon("transition"), sliderTab);
					addTab("<html><h4 " + tabStyle + ">" + getText("ui.newImages") + "</h4></html>", icon("new"), markNew);
					addTab("<html><h4 " + tabStyle + ">" + getText("ui.rating") + "</h4></html>", icon("star"), rating);
					addTab("<html><h4 " + tabStyle + ">" + getText("ui.extraSizes") + "</h4></html>", icon("more"), extra);
					addTab("<html><h4 " + tabStyle + ">" + getText("ui.social") + "</h4></html>", icon("connect"), social);
					addTab("<html><h4 " + tabStyle + ">" + getText("ui.map") + "</h4></html>", icon("location"), mapSettings);
					addTab("<html><h4 " + tabStyle + ">" + getText("ui.audioClips") + "</h4></html>", icon("volume-up"), audioClips);
					addTab("<html><h4 " + tabStyle + ">" + getText("ui.backgroundMusic") + "</h4></html>", icon("audio"), backgroundMusic);
					addTab("<html><h4 " + tabStyle + ">" + getText("ui.footer") + "</h4></html>", icon("footer"), footerPanel);
				}
			};
			
			JPanel sitePreviewPanel = new JPanel(new BorderLayout(0, 0)) {

				{
					sitePreview.getView().setPreferredSize(new Dimension(previewWidth, uiHeight));

					add(sitePreview.getView(), BorderLayout.CENTER);
				}
			};
			
			{	
				((BorderLayout)(getLayout())).setVgap(0);
				((BorderLayout)(getLayout())).setHgap(0);
				siteTabs.setBorder(emptyBorder);
				
				siteTabs.setPreferredSize(new Dimension(uiWidth + tabWidth, uiHeight));
				siteTabs.setMaximumSize(new Dimension(uiWidth + tabWidth + 80, uiHeight));
				sitePreviewPanel.setBorder(BorderFactory.createLineBorder(JAlbumColor.imageBorder, 2));
				
				add(siteTabs);
				add(sitePreviewPanel, BorderLayout.EAST);
				
				//putClientProperty("helpPage", helpRoot + "ui/site.html");
			}
		};
					
		/*	---------------------------------------------------------------
									Sections
			--------------------------------------------------------------- */

		ControlPanel sections = new ControlPanel(new BorderLayout(20, 0)) {
			
			ControlPanel sectionOrdering = new ControlPanel() {

				JDraggableList sectionsOrder = new JDraggableList(new Object[] {
					new JLabelFor("<html><p style=\"padding:3px 0;\">" + getText("ui.pages") + "</p></html>", icon("page")).name("pages"),
					new JLabelFor("<html><p style=\"padding:3px 0;\">" + getText("ui.folders") + "</p></html>", icon("folder")).name("folders"),
					new JLabelFor("<html><p style=\"padding:3px 0;\">" + getText("ui.map") + "</p></html>", icon("location")).name("map"),
					new JLabelFor("<html><p style=\"padding:3px 0;\">" + getText("ui.images") + "</p></html>", icon("thumbnails")).name("images"),
					new JLabelFor("<html><p style=\"padding:3px 0;\">" + getText("ui.neighboringFolders") + "</p></html>", icon("prev-next-page")).name("neighboringFolders"),
					new JLabelFor("<html><p style=\"padding:3px 0;\">" + getText("ui.commenting") + "</p></html>", icon("chat")).name("comments"),
					new JLabelFor("<html><p style=\"padding:3px 0;\">" + getText("ui.customContent") + "</p></html>", icon("edit-document")).name("customContent")
				});

				{						
					add("center", new JLabel("<html><h4>" + getText("ui.ordering")+ "</h4></html>"));
					add("br", sectionsOrder);
					add("br center", new JLabel("<html><i>" + getText("ui.dragToReorder") + "</i></html>"));
				}
			};

			JTabbedPane sectionTabs = new JTabbedPane() {

				{
					this.setTabPlacement(SwingConstants.LEFT);

					addTab("<html><h4 " + tabStyle + ">" + getText("ui.folders") + "</h4></html>", icon("folder"), folders);
					addTab("<html><h4 " + tabStyle + ">" + getText("ui.pages") + "</h4></html>", icon("page"), pages);
					addTab("<html><h4 " + tabStyle + ">" + getText("ui.images") + "</h4></html>", icon("thumbnails"), images);
					addTab("<html><h4 " + tabStyle + ">" + getText("ui.map") + "</h4></html>", icon("location"), map);
					addTab("<html><h4 " + tabStyle + ">" + getText("ui.neighboringFolders") + "</h4></html>", icon("prev-next-page"), neighboringFolders);
					addTab("<html><h4 " + tabStyle + ">" + getText("ui.commenting") + "</h4></html>", icon("chat"), commenting);
					addTab("<html><h4 " + tabStyle + ">" + getText("ui.customContent") + "</h4></html>", icon("edit-document"), customSection);

				}
			};
				
			{
				//sectionTabs.setBorder(BorderFactory.createMatteBorder(0, 0, 0, 3, JAlbumColor.imageBorder));
				
				((BorderLayout)(getLayout())).setVgap(0);
				((BorderLayout)(getLayout())).setHgap(0);
				
				sectionTabs.setBorder(emptyBorder);
				
				add(sectionTabs);
				add(sectionOrdering, BorderLayout.EAST);
				
				//putClientProperty("helpPage", helpRoot + "ui/sections.html");
			}
		};
		
		/*	---------------------------------------------------------------
									Sidebar
			--------------------------------------------------------------- */

		ControlPanel sidebar = new ControlPanel(new BorderLayout(20, 0)) {
			
			ControlPanel sidebarOrdering = new ControlPanel() {

				JDraggableList sidebarOrder = new JDraggableList(new Object[] {
					new JLabelFor("<html><p style=\"padding:3px 0;\">" + getText("ui.tagCloud") + "</h4></html>", icon("tag")).name("tagCloud"),
					new JLabelFor("<html><p style=\"padding:3px 0;\">" + getText("ui.folderTree") + "</h4></html>", icon("site")).name("folderTree"),
					new JLabelFor("<html><p style=\"padding:3px 0;\">" + getText("ui.filtering") + "</h4></html>", icon("filter")).name("filtering"),
					new JLabelFor("<html><p style=\"padding:3px 0;\">" + getText("ui.searchNew") + "</h4></html>", icon("new")).name("searchNew"),
					new JLabelFor("<html><p style=\"padding:3px 0;\">" + getText("ui.downloadZip") + "</h4></html>", icon("download")).name("zip"),
					new JLabelFor("<html><p style=\"padding:3px 0;\">" + getText("ui.externalLinks") + "</h4></html>", icon("external")).name("externalLinks"),
					new JLabelFor("<html><p style=\"padding:3px 0;\">" + getText("ui.map") + "</h4></html>", icon("location")).name("map"),
					new JLabelFor("<html><p style=\"padding:3px 0;\">" + getText("ui.shoppingCart") + "</h4></html>", icon("shopping-cart")).name("shop"),
					new JLabelFor("<html><p style=\"padding:3px 0;\">" + getText("ui.feedback") + "</h4></html>", icon("email-send")).name("feedback"),
					new JLabelFor("<html><p style=\"padding:3px 0;\">" + getText("ui.facebookBox") + "</h4></html>", icon("facebook")).name("facebookBox"),
					new JLabelFor("<html><p style=\"padding:3px 0;\">" + getText("ui.twitterBox") + "</h4></html>", icon("x")).name("twitterBox"),
					new JLabelFor("<html><p style=\"padding:3px 0;\">" + getText("ui.instagramBox") + "</h4></html>", icon("instagram")).name("instagramBox"),
					new JLabelFor("<html><p style=\"padding:3px 0;\">" + getText("ui.customContent") + "</h4></html>", icon("edit-document")).name("customContent")
				});

				{						
					add("center", new JLabel("<html><h4>" + getText("ui.ordering")+ "</h4></html>"));
					add("br", sidebarOrder);
					add("br center", new JLabel("<html><i>" + getText("ui.dragToReorder") + "</i></html>"));
				}
			};

			JTabbedPane sidebarTabs = new JTabbedPane() {

				JScrollPane shoppingCartPanel = new JScrollPane(shoppingCart, JScrollPane.VERTICAL_SCROLLBAR_AS_NEEDED, JScrollPane.HORIZONTAL_SCROLLBAR_NEVER);
				JScrollPane filteringAndSortPanel = new JScrollPane(filteringBox, JScrollPane.VERTICAL_SCROLLBAR_AS_NEEDED, JScrollPane.HORIZONTAL_SCROLLBAR_NEVER);
				
				{
					shoppingCartPanel.setBorder(emptyBorder);
					filteringAndSortPanel.setBorder(emptyBorder);
					this.setTabPlacement(SwingConstants.LEFT);

					addTab("<html><h4 " + tabStyle + ">" + getText("ui.tagCloud") + "</h4></html>", icon("tag"), tagCloudBox);
					addTab("<html><h4 " + tabStyle + ">" + getText("ui.folderTree") + "</h4></html>", icon("site"), folderTreeBox);
					addTab("<html><h4 " + tabStyle + ">" + getText("ui.filteringAndSort") + "</h4></html>", icon("filter"), filteringAndSortPanel);
					addTab("<html><h4 " + tabStyle + ">" + getText("ui.searchNew") + "</h4></html>", icon("new"), searchNewBox);
					addTab("<html><h4 " + tabStyle + ">" + getText("ui.downloadZip") + "</h4></html>", icon("download"), downloadBox);
					addTab("<html><h4 " + tabStyle + ">" + getText("ui.externalLinks") + "</h4></html>", icon("external"), externalLinksBox);
					addTab("<html><h4 " + tabStyle + ">" + getText("ui.map") + "</h4></html>", icon("location"), mapBox);
					addTab("<html><h4 " + tabStyle + ">" + getText("ui.shoppingCart") + "</h4></html>", icon("shopping-cart"), shoppingCartPanel);
					addTab("<html><h4 " + tabStyle + ">" + getText("ui.feedback") + "</h4></html>", icon("email-send"), feedback);
					addTab("<html><h4 " + tabStyle + ">" + getText("ui.facebookBox") + "</h4></html>", icon("facebook"), facebookBox);
					addTab("<html><h4 " + tabStyle + ">" + getText("ui.twitterBox") + "</h4></html>", icon("x"), twitterBox);
					addTab("<html><h4 " + tabStyle + ">" + getText("ui.instagramBox") + "</h4></html>", icon("instagram"), instagramBox);
					addTab("<html><h4 " + tabStyle + ">" + getText("ui.customContent") + "</h4></html>", icon("edit-document"), customSidebar);
				}
			};
				
			{
				((BorderLayout)(getLayout())).setVgap(0);
				((BorderLayout)(getLayout())).setHgap(0);
				
				//sidebarTabs.setBorder(BorderFactory.createMatteBorder(0, 0, 0, 3, JAlbumColor.imageBorder));
				sidebarTabs.setBorder(emptyBorder);
				
				add(sidebarTabs);
				add(sidebarOrdering, BorderLayout.EAST);
				
				//putClientProperty("helpPage", helpRoot + "ui/sections.html");
			}
		};
		
		/*	---------------------------------------------------------------
									Lightbox
			--------------------------------------------------------------- */
		
		ControlPanel lightbox = new ControlPanel(new BorderLayout(0, 0)) {
			
			JTabbedPane lightboxTabs = new JTabbedPane() {
				
				JScrollPane mainImagePanel = new JScrollPane(mainImage, JScrollPane.VERTICAL_SCROLLBAR_AS_NEEDED, JScrollPane.HORIZONTAL_SCROLLBAR_NEVER);
				JScrollPane lightboxButtonsPanel = new JScrollPane(lightboxButtons, JScrollPane.VERTICAL_SCROLLBAR_AS_NEEDED, JScrollPane.HORIZONTAL_SCROLLBAR_NEVER);
				
				{
					mainImagePanel.setBorder(emptyBorder);
					lightboxButtonsPanel.setBorder(emptyBorder);
					this.setTabPlacement(SwingConstants.LEFT);
										
					addTab("<html><h4 " + tabStyle + ">" + getText("ui.settings") + "</h4></html>", icon("settings"), lightboxSettings);
					addTab("<html><h4 " + tabStyle + ">" + getText("ui.mainImage") + "</h4></html>", icon("image"), mainImagePanel);
					addTab("<html><h4 " + tabStyle + ">" + getText("ui.caption") + "</h4></html>", icon("text"), lightboxCaption);
					addTab("<html><h4 " + tabStyle + ">" + getText("ui.buttons") + "</h4></html>", icon("button"), lightboxButtonsPanel);
				}
			};
			
			{
				((BorderLayout)(getLayout())).setVgap(0);
				((BorderLayout)(getLayout())).setHgap(0);
				
				add(lightboxTabs);
				
				//putClientProperty("helpPage", helpRoot + "ui/lightbox.html");
			}
		};
		
		/*	---------------------------------------------------------------
									Advanced
			--------------------------------------------------------------- */
		
		ControlPanel advanced = new ControlPanel(new BorderLayout(0, 0)) {
			
			JTabbedPane advancedTabs = new JTabbedPane() {
								
				JScrollPane siteAdminPane = new JScrollPane(siteAdminPanel, JScrollPane.VERTICAL_SCROLLBAR_AS_NEEDED, JScrollPane.HORIZONTAL_SCROLLBAR_NEVER);
				
				{
					customCodePanel.setBorder(emptyBorder);
					//customContentPanel.setBorder(emptyBorder);
					siteAdminPane.setBorder(emptyBorder);
					customKeysPanel.setBorder(emptyBorder);
					this.setTabPlacement(SwingConstants.LEFT);
					
					addTab("<html><h4 " + tabStyle + ">" + getText("ui.siteAdmin") + "</h4></html>", icon("wrench"), siteAdminPane);
					addTab("<html><h4 " + tabStyle + ">" + getText("ui.customCode") + "</h4></html>", icon("code"), customCodePanel);
					//addTab("<html><h4 " + tabStyle + ">" + getText("ui.customContent") + "</h4></html>", icon("edit"), customContentPanel);
					addTab("<html><h4 " + tabStyle + ">" + getText("ui.customKeys") + "</h4></html>", icon("rename"), customKeysPanel);
				}
			};
			
			{
				((BorderLayout)(getLayout())).setVgap(0);
				((BorderLayout)(getLayout())).setHgap(0);
				
				advancedTabs.setBorder(emptyBorder);
				
				add(advancedTabs);
				//putClientProperty("helpPage", helpRoot + "ui/lightbox.html");
			}
		};
		
	
		/*	---------------------------------------------------------------
									About
			--------------------------------------------------------------- */

		ControlPanel about = new ControlPanel(new BorderLayout(20, 0)) {
			
			private JTextArea readme = new JSmartTextArea( getFileContents("readme.txt"), 20, 30 );
			private JScrollPane aboutPane = new JScrollPane(readme, JScrollPane.VERTICAL_SCROLLBAR_ALWAYS, JScrollPane.HORIZONTAL_SCROLLBAR_NEVER);
			
			ControlPanel info = new ControlPanel() {
				
				{
					add("center", new JLabel(svgIcon("logo-photoblogger", new Dimension(128, 128))));
					add("br center", new JLabel("<html><h2>" + skin + "</h2></html>"));
					add("br center", new JLabel("Jalbum " + internalVersion));
					add(new JLinkLabel("https://jalbum.net/software/download", getText("ui.upgrade"), getText("ui.downloadJalbum")));
					add("br center", new JLabel(skin + " skin " + skinVer));
					add(new JLinkLabel("https://jalbum.net/skins/skin/" + skin, getText("ui.upgrade"), getText("ui.downloadSkin")));
					//add("br center", new JLinkLabel(helpRoot + "index.html", getText("help")));
					add("br center", new JLinkLabel(supportForum, getText("ui.support")));
				}
			};
			
			{								
				((BorderLayout)(getLayout())).setVgap(0);
				((BorderLayout)(getLayout())).setHgap(0);
				
				readme.setLineWrap(true);
				readme.setWrapStyleWord(true);
				readme.setEditable(false);
				readme.setFont(mono);
				
				info.setPreferredSize(new Dimension(240, uiHeight));
				aboutPane.setPreferredSize(new Dimension(400,320));
				
				add(info, BorderLayout.WEST);
				add(aboutPane);
				
				//putClientProperty("helpPage", helpRoot + "index.html");
			}
		};
		
		/*	---------------------------------------------------------------
									Main tabs
			--------------------------------------------------------------- */
		
		JTabbedPane tabs = new JTabbedPane() {
			
			{
				site.setBorder(emptyBorder);
				sections.setBorder(emptyBorder);
				sidebar.setBorder(emptyBorder);
				lightbox.setBorder(emptyBorder);
				advanced.setBorder(emptyBorder);
				about.setBorder(emptyBorder);
				
				addTab("<html><h4 style='padding:4px 6px;margin:4px;'>" + getText("ui.site") + "</h4></html>", icon("site", 27), site);
				addTab("<html><h4 style='padding:4px 6px;margin:4px;'>" + getText("ui.mainContent") + "</h4></html>", icon("sections", 27), sections);
				addTab("<html><h4 style='padding:4px 6px;margin:4px;'>" + getText("ui.sidebar") + "</h4></html>", icon("sidebar-right", 27), sidebar);
				addTab("<html><h4 style='padding:4px 6px;margin:4px;'>" + getText("ui.lightbox") + "</h4></html>", icon("lightbox", 27), lightbox);
				addTab("<html><h4 style='padding:4px 6px;margin:4px;'>" + getText("ui.advanced") + "</h4></html>", icon("wrench", 27), advanced);
				addTab("<html><h4 style='padding:4px 6px;margin:4px;'>" + getText("ui.about") + "</h4></html>", icon("info", 27), about);
			}
		};
		
		private final String designTemplate = getFileContents("lib/index-template.html");
	
		{
			// Adding UI tabs
			
			((RiverLayout)(getLayout())).setVgap(0);
			((RiverLayout)(getLayout())).setHgap(0);
			tabs.setBorder(emptyBorder);
			
			add(tabs);
			
			/*	-----------------------------------------------------------
										Listeners
				----------------------------------------------------------- */
			
			window.addJAlbumListener(new JAlbumAdapter() {
				@Override
				public void styleChanged(JAlbumEvent e) {
					styleName.setText(getStyleName());
				}
			});
			
			window.addPropertyChangeListener(JAlbumFrame.SKIN_LOADED_PROPERTY, setupMonitors);

			window.addJAlbumListener(new JAlbumAdapter() {
				@Override
				public void skinChanged(JAlbumEvent e) {
					//JAlbum.logger.log(Level.FINE, "Skin changed, removing listeners.");
					window.removePropertyChangeListener(JAlbumFrame.SKIN_LOADED_PROPERTY, setupMonitors);
				}
			});
			
			Map<String,Object> vars = engine.getSkinVariables();
			selectComboBoxItem(titleCaptionPresets, (String)vars.get("titleCaptionTemplate"));
			selectComboBoxItem(folderCaptionPresets, (String)vars.get("folderCaptionTemplate"));
			selectComboBoxItem(thumbCaptionPresets, (String)vars.get("thumbCaptionTemplate"));
			selectComboBoxItem(imageCaptionPresets, (String)vars.get("imageCaptionTemplate"));
			
		}
		
	};
	
	public Gui() {
		this(JAlbumContext.getInstance());
	}
   
	public Gui(JAlbumContext context) {
		
		super(context);
		PluginContext pc = context.getPluginContext();
		EditPanel editPanel = pc.getEditPanel();
		
		editPanel.addCustomTab(getText("ui.imageData"), new ImageDataUI(context, texts));
		editPanel.addCustomTab(getText("ui.external"), new ExternalContentUI(context, texts));

		skinUi.setBorder(emptyBorder);
		((RiverLayout)(getLayout())).setVgap(0);
		((RiverLayout)(getLayout())).setHgap(0);
		
		window.setSkinUI(skinUi);
		skinReadyAt = System.currentTimeMillis();
								
	}
	
}