/*
 * 	laza.autopano.js - auto panorama viewer
 *	jslint browser: true
 */
 
;(function($, $window, $document, $body) {	
	'use strict';
	// Requires laza.util

	$.fn.autopano = function(settings) {
		
			settings = $.extend({}, $.fn.autopano.defaults, settings);
	
			var self = $(this),							// Element
				cont = self.parent(),					// Container
				exposedStart = false,					// Left end reached
				exposedEnd = false,						// Right end reached
				direction = 0,							// Current direction
				maxDur = 1600 / settings.speed,			// Max. duration (40s - 20s - 10s)
				minDur = 120 / settings.speed,			// Min. duration (3s - 1.5s - 0.75s)
				size, 									// Element size
				screenSize,								// Container size
				vertical = false,						// Move axis
				rto,									// Resize timeout
				ns = self.data('lap_ns'),				// Name space
					
				// Initializing
				
				init = function() {
						
						// Updating window size
						
						$window.on('resize.' + ns, function() {
								clearTimeout(rto);
								rto = setTimeout(updateDims, 50, false);
								return true;
							});
				
						// External events
						
						self.data('hasAutoPano', true)
							.on('autopanoRemove.' + ns, function() {
								direction = 0;
								remove();
								return false;
							})
							.on('autopanoStart', function() {
								startAuto();
								return false;
							})
							.on('autopanoStop', function() {
								direction = 0;
								stopAuto();
								return false;
							})
							.on('autopanoAdjust', function() {
								updateDims(false);
							});
						
						updateDims(true);
					},
				
				// Update dimensions
				
				updateDims = function(first) {
						var mx = self.getMatrix(), 
							ew = self[0].scrollWidth,
							eh = self[0].scrollHeight,
							cw = cont[0].clientWidth,
							ch = cont[0].clientHeight;
							
						if (ew / cw < eh / ch) {
							// Vertical
							size = eh;
							screenSize = ch;
							if (!vertical) {
								// Changed direction
								vertical = true;
								if (!first) {
									mx[4] = - ew / 2;
									self.css({
											transition: 	'none',
											transform:		'matrix(' + mx.join(',') + ')'
										});
								}
							}
						} else {
							// Horizontal
							size = ew;
							screenSize = cw;
							if (vertical) {
								// Changed direction
								vertical = false;
								if (!first) {
									mx[5] = - eh / 2;
									self.css({
											transition: 	'none',
											transform:		'matrix(' + mx.join(',') + ')'
										});
								}
							}
						}
						
						if (size > screenSize) {
							//console.log('Dimensions updated');
							if (first || !direction) {
								// Start if wider/taller than screen
								setTimeout(startAuto, 100);
							} else if (direction > 0) {
								moveToEnd();
							} else {
								moveToStart();
							}
						} else {
							// Stop if smaller
							stopAuto();
						}
					},
				
				// Remove autopano
				
				remove = function() {
					
						stopMove();
						
						setTimeout(function() { 
								cont.data('panomove', false);
							}, 20);
						
						self.removeAttr('draggable');
						self.add($document).off('.' + ns);
					},
					
				// Reached the start
				
				reachedStart = function(e) {
						//eventLog(e);
						exposedStart = true;
						
						if (direction) {
							if (!exposedEnd || settings.loop) {
								moveToEnd();
							} else {
								stopAuto();
							}
						}
					},
					
				// Reached the end
				
				reachedEnd = function(e) {
						//eventLog(e);
						exposedEnd = true;
						
						if (direction) {
							if (!exposedStart || settings.loop) {
								moveToStart();
							} else {
								stopAuto();
							}
						}
					},
					
				// Move to left end
				
				moveToStart = function() {
						var dis = - screenSize / 2,
							mx = self.getMatrix(),
							dur;
							
						if (vertical) {
							dur = 1000 * Math.minMax(minDur, Math.abs(mx[5] - dis) / settings.speed, maxDur),
							mx[5] = dis;
						} else {
							dur = 1000 * Math.minMax(minDur, Math.abs(mx[4] - dis) / settings.speed, maxDur),
							mx[4] = dis;
						}
						
						exposedEnd = false;
						direction = -1;
						self[0].style.transition = 'none'; 
						self.one('transitionend', reachedStart)
							.css({
								transition:		'transform ' + dur + 'ms ' + settings.easing,
								transform:		'matrix(' + mx.join(',') + ')'
							});	
					},
				
				// Move to right end
				
				moveToEnd = function() {
						var dis = screenSize / 2 - size,
							mx = self.getMatrix(),
							dur;
						
						if (vertical) {
							dur = 1000 * Math.minMax(minDur, Math.abs(mx[5] - dis) / settings.speed, maxDur),
							mx[5] = dis;
						} else {
							dur = 1000 * Math.minMax(minDur, Math.abs(mx[4] - dis) / settings.speed, maxDur),
							mx[4] = dis;
						}
						
						exposedStart = false;
						direction = 1;
						self[0].style.transition = 'none'; 
						self.one('transitionend', reachedEnd)
							.css({
								transition:		'transform ' + dur + 'ms ' + settings.easing,
								transform:		'matrix(' + mx.join(',') + ')'
							});
					},
					
				// Start auto pano
				
				startAuto = function() {
						var mx = self.getMatrix(),
							c = vertical? (self[0].scrollHeight / 2 + mx[5]) : (self[0].scrollWidth / 2 + mx[4]);
						
						if (settings.direction < 0 || (settings.direction === 0 && c > 0)) {
							moveToStart();
						} else {
							moveToEnd();
						}
						
						cont.addClass(settings.panoMoveClass);
						cont.data('panomove', true);	
					},
				
				// Stop moving
				
				stopMove = function() {
						var mx = self.getMatrix();
						
						direction = 0;
						self[0].removeEventListener('transitionend', function(){});
						
						// Move immediately
						self.css({
								transition:		'none',
								transform:		'matrix(' + mx.join(',') + ')'
							});
						
						cont.data('panomove', false);
						cont.removeClass(settings.panoMoveClass);
					},
					
				// Pause auto pano
				
				stopAuto = function() {
						
						stopMove();
						/*
						setTimeout(function() {
								cont.data('panomove', false);
							}, 300);
						*/
					};
				
	
			// Initializing
						
			if (ns) {
				$window.add($document).off('.' + ns);
			}
			
			// Creating new namespace
			
			self.data('lap_ns', ns = 'lap_' + Math.floor(Math.random() * 10000));
			
			// Initializing
			
			init();
			
			// Ready event
			
			if (settings.hasOwnProperty('onReady') && typeof settings.onReady === FUNCTION) {
				settings.onReady(thumb);
			}
			
			return this;
		};
	
	$.fn.autopano.defaults = {
			direction:			-1,									// 1: to right, -1: to left
			speed:				80,									// pixel / s
			easing:				'cubic-bezier(0.2, 0, 0.8, 1)',		// Easing function
			autoStart:			true,								// Auto start
			loop:				true,								// continue after exposed both ends
			panoMoveClass:		'pano-move'							// Class for the container during move
		};
	
})(jQuery, jQuery(window), jQuery(document), jQuery('body'));
