/*jslint browser: true*/
;(function($, window, document, undefined) {
	'use strict';
	
	/*
	 *	Google Maps functions
	 *	canvas is a jQuery element with data('location') and other optional parameters
	 */

			
	$.fn.addMap = function(settings) {
		
		settings = $.extend({}, $.fn.addMap.defaults, settings);
			
		return this.each(function() {

				var canvas 			= 	$(this),								// Map root
					ns				=	canvas.data('map_ns');					// Namespace 
				
				if (typeof ns !== UNDEF) {
					// Already instantitated
					return;
				}
				
				canvas.data('map_ns', ns = 'map_' + Math.floor(Math.random() * 10000));
				if (!this.id) {
					this.id = ns;
				}

				var	options 		= $.extend({}, settings, readData(canvas, 'location,type,zoom,label,markers,overlay,current')),
					map,
					marker 			= [],
					active 			= -1,
					bounds,
					hasBounds		= false,
					//normalPin,
					//activePin,
					boundsApplied 	= false,
					
					// Gets a marker by its name
					
					getMarkerByName = function(n) {
						
							for (var i = 0; i < marker.length; i++) {
								if (marker[i]['link'] === n) {
									return i;
								}
							}
							
							return -1;
						},
					
					// Animates a marker and centers map around this marker
					
					animateMarker = function(m) {
							m.setAnimation(google.maps.Animation.BOUNCE);
							map.setCenter(m.getPosition());
						},
							
					// Sets a marker as active
					
					setActive = function(e, n) {
							var n = (typeof n === UNDEF)? -1 : (typeof n === NUMBER)? n : getMarkerByName(n);
							//console.log('setActive(' + n + ')');
							if (n !== active) {
								
								if (active >= 0) {
									// Stop animating the active
									marker[active].setAnimation(null);
								}
								
								if (n >= 0 && n < marker.length) {
									// Animate the new
									animateMarker(marker[n]);
								}
								
								active = n;
							}
						},
					
					// Refreshing map, if it was rendered on a hidden layer
					
					refresh = function() {
							
							if (options.fitBounds && hasBounds && !boundsApplied) {
								fitBounds();
							}
							
							if (active !== -1) {
								map.setCenter(marker[active].getPosition());
							}
						},
						
					// Fits map into bounds
					
					fitBounds = function() {
							if (canvas.is(':visible')) {
								map.fitBounds(bounds);
								//google.maps.event.trigger(map, 'resize');
								map.panToBounds(bounds);
								boundsApplied = true;
							}
						},
						
					// Reset marker animations
					
					resetMarkers = function() {
						
							if (marker && marker.length) {
								
								if (active >= 0) {
									if (marker[active].getAnimation() !== null) {
										marker[active].setAnimation(null);
									}
									active = -1;
								}
							}
							
							if (options.fitBounds && hasBounds) {
								fitBounds();
							} else if (marker && marker.length) {
								map.setCenter(marker[0].getPosition());
							}
						},
								
					// Marker clicked event
						
					markerClicked = function() {
						
							if (marker && marker.length) {
								
								if (active >= 0) {
									if (marker[active].getAnimation() !== null) {
										marker[active].setAnimation(null);
									}
									active = -1;
								}
								
								animateMarker(this);
							}
							
							if (typeof options.onMarkerClick === FUNCTION) {
								options.onMarkerClick.call(this);
							}
						},
						
					// Loading script

					initMap = async function() {
							const { Map } = await google.maps.importLibrary('maps'),
								{ Marker } = await google.maps.importLibrary('marker');
							
							if (options.autoLoad) {
								renderMap();
							}
							
							canvas.on('render', renderMap);
						},
						
					// Load overlay(s)
					
					loadOverlays = function(overlays, doneFn) {
							var xhr = new XMLHttpRequest();
							
							xhr.open('GET', overlays[0], true);
							
							xhr.onreadystatechange = function () {
								
									if (xhr.readyState === 4 && xhr.status === 200) {
										// Parse GPX data track points
										var points = xhr.responseXML.getElementsByTagName('trkpt'),
											path = [];
										
										for (var i = 0; i < points.length; i++) {
											path.push(new google.maps.LatLng(parseFloat(points[i].getAttribute('lat')), parseFloat(points[i].getAttribute('lon'))));
										}
										
										if (path.length) {
											// Create a polyline and set its path
											var polyline = new google.maps.Polyline({
													path: 			path,
													geodesic: 		true,
													strokeColor: 	'#FF8800', 	// Line color
													strokeOpacity: 	1.0, 		// Line opacity
													strokeWeight: 	2, 			// Line thickness
												});
											
											// Add the polyline to the map
											polyline.setMap(map);
											
											// Fit the map bounds to the polyline
											path.forEach(function (point) {
													bounds.extend(point);
													hasBounds = true;
												});
										}
										
										if (overlays.length > 1) {
											// Next overlay
											loadOverlays(overlays.slice(1), doneFn);
										} else {
											// Overlays rendered, call doneFn
											if (typeof doneFn === FUNCTION) {
												doneFn.call(null);
											}
										}
									}
								};
								
							xhr.send();
						},

					// Rendering map
				
					renderMap = function() {
							
							if (canvas.data('rendered')) {
								return;
							}
							
							var	delay = 10,
								markerTo = null,
							
								// Adding markers one-by-one with delay
								addMarker = function(i) {
										
										clearTimeout(markerTo);
										
										if (i < options.markers.length) {
											
											if (options.markers[i].hasOwnProperty('latLng')) {
												marker[i] = new google.maps.Marker({
														map: 			map,
														position:		options.markers[i]['latLng'],
														animation: 		google.maps.Animation.DROP
													});
												
												if (options.markers[i].hasOwnProperty('title')) {
													marker[i].setTitle(options.markers[i]['title']);
												}
					
												if (options.markers[i].hasOwnProperty('link')) {
													marker[i].link = options.markers[i]['link'];
													google.maps.event.addListener(marker[i], 'click', markerClicked);
												}
											}
											
											// Calling the next with delay
											markerTo = setTimeout(addMarker, delay, i + 1);
											
										} else {
											// All markers added
											// Fit bounds if more than 1 marker
											if (options.markers.length > 1) {
												fitBounds();
											}
										}
									},
									
								// Prepare markers, fit bounds or center map 
								setMarkers = function() {
										if (options.hasOwnProperty('markers')) {
											// Delay: used to delay between dropping pins
											delay = Math.minMax(10, Math.round(1000 / options.markers.length), 100);
											
											if (options.markers.length > 1) {
												// Only if more than 1 marker
												for (var i = 0; i < options.markers.length; i++) {
													bounds.extend(options.markers[i].latLng);
												}
												hasBounds = true;
											} else {
												map.setCenter(options.markers[0]['latLng']);
												map.setZoom(options.zoom);
											}
											
											addMarker(0);
										}
								
									};
							
							// Creating boundaries array
							bounds = new google.maps.LatLngBounds();
							
							// Styling pins
							//normalPin = new google.maps.PinElement({ background: '#0066CC', borderColor: '#0055AA' });
							//activePin = new google.maps.PinElement({ scale: 1.5, background: "#CC6600", borderColor: '#AA5500' });
							
							// Creating Map
							map = new google.maps.Map(canvas[0], {
										mapTypeId: 				options.type,
										mapTypeControl: 		options.typeControl,
										mapTypeControlOptions: 	{
																	position: google.maps.ControlPosition.RIGHT_TOP,
																	style: google.maps.MapTypeControlStyle.DROPDOWN_MENU 
																},
										fullscreenControl: 		options.fullscreenControl,
										scrollwheel: 			options.enableScrollWheel,
										zoomControl: 			options.zoomControl,
										scaleControl: 			options.scaleControl,
										streetViewControl: 		options.streetViewControl
									});
								
							// Initializing markers
							if (options.hasOwnProperty('markers')) {
								// Converting old "pos" property into "latLng"
								for (var i = 0, ll; i < options.markers.length; i++) {
									if (options.markers[i].hasOwnProperty('pos') && !options.markers[i].hasOwnProperty('latLng')) {
										ll = options.markers[i].pos.split(',');
										options.markers[i].latLng = {
												lat:	parseFloat(ll[0]),
												lng:	parseFloat(ll[1])
											};
									}
								}
								
							} else if (options.hasOwnProperty('location')) {
								// No markers: single locations
								var ll = options.location.split(',');
								options.markers = [{ 
										latLng: { 
												lat:	parseFloat(ll[0]),
												lng:	parseFloat(ll[1])
											}
									}];
									
							} else if (!options.hasOwnProperty('overlay')) {
								// No location and no overlay => set center to 0, 0
								console.log('Error in AddMap: Nothing to display on the map!');
								options.markers = [{ 
										latLng: { 
												lat: 	0, 
												lng: 	0 
											}
									}];
							}
							
							if (options.hasOwnProperty('overlay')) {
								// Loading overlay(s), then add markers
								loadOverlays(options.overlay.split(','), setMarkers);
							} else if (options.markers) {
								// Start adding markers immediately
								setMarkers();
							}
							
							// Events
							if (typeof options['onTypeChanged'] === FUNCTION) {
								google.maps.event.addListener(map, 'maptypeid_changed', function() {
										options.onTypeChanged.call(this, map.getMapTypeId());
									});
							}
							
							if (typeof options['onZoomChanged'] === FUNCTION) {
								google.maps.event.addListener(map, 'zoom_changed', function() {
										options.onZoomChanged.call(this, map.getZoom());
									});
							}
							
							canvas.data('rendered', true);
							
							// External triggers
							canvas.on({
									setActive: 		setActive,
									resetMarkers:	resetMarkers,
									refresh:		refresh
								});
							
							if (typeof doneFn === FUNCTION) {
								doneFn.call(null);
							}
						};
				
				// Loading script
				
				initMap();
				
			});
		
		};
					
	$.fn.addMap.defaults = {
			zoom:					15,
			fitBounds:				true,
			type:					'hybrid',
			typeControl:			true,
			fullscreenControl: 		false,
			zoomControl:			true,
			scaleControl:			false,
			streetViewControl:		false,
			enableScrollWheel:		false,
			autoLoad:				true
		};
	
})(jQuery, window, document);
