package lucid;

/*
	Gui.java - Lucid skin User Interface
*/

import com.kitfox.svg.app.beans.SVGIcon;
import edu.stanford.ejalbert.BrowserLauncher;
import icons.Lazaicon;
import java.awt.BorderLayout;
import java.awt.Color;
import java.awt.Component;
import java.awt.Dimension;
import java.awt.Font;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.awt.event.MouseAdapter;
import java.awt.event.MouseEvent;
import java.beans.PropertyChangeListener;
import java.io.File;
import java.io.FileNotFoundException;
import java.io.FileWriter;
import java.io.IOException;
import java.text.DateFormat;
import java.text.SimpleDateFormat;
import java.util.Date;
import java.util.Map;
import java.util.logging.Level;
import javax.swing.BorderFactory;
import javax.swing.Icon;
import javax.swing.JButton;
import javax.swing.JCheckBox;
import javax.swing.JComboBox;
import javax.swing.JComponent;
import javax.swing.JFileChooser;
import javax.swing.JLabel;
import javax.swing.JList;
import javax.swing.JOptionPane;
import javax.swing.JPanel;
import javax.swing.JScrollPane;
import javax.swing.JSlider;
import javax.swing.JSpinner;
import javax.swing.JTabbedPane;
import javax.swing.JTextArea;
import javax.swing.JTextField;
import javax.swing.SpinnerNumberModel;
import javax.swing.SwingConstants;
import javax.swing.SwingUtilities;
import javax.swing.border.Border;
import javax.swing.filechooser.FileNameExtensionFilter;
import net.jalbum.browser.WebViewBrowser;
import net.jalbum.component.JDraggableList;
import se.datadosen.component.ChooserFactory;
import se.datadosen.component.ComponentUtilities;
import se.datadosen.component.ControlPanel;
import se.datadosen.component.DeferredSVGIcon;
import se.datadosen.component.FileChooser;
import se.datadosen.component.JAlphaColorSelector;
import se.datadosen.component.JColorSelector;
import se.datadosen.component.JLabelFor;
import se.datadosen.component.JLinkLabel;
import se.datadosen.component.JNotification;
import se.datadosen.component.JPlaylist;
import se.datadosen.component.JSmartTextArea;
import se.datadosen.component.JSmartTextField;
import se.datadosen.component.RiverLayout;
import se.datadosen.component.StateMonitor;
import se.datadosen.component.WrappableJLabel;
import se.datadosen.jalbum.AccountManager;
import se.datadosen.jalbum.AlbumBean;
import se.datadosen.jalbum.Icons;
import se.datadosen.jalbum.JAlbum;
import se.datadosen.jalbum.JAlbumColor;
import se.datadosen.jalbum.JAlbumContext;
import se.datadosen.jalbum.JAlbumFrame;
import se.datadosen.jalbum.JAlbumSite;
import se.datadosen.jalbum.ParameterException;
import se.datadosen.jalbum.SignInManager;
import se.datadosen.jalbum.SkinProperties;
import se.datadosen.jalbum.event.JAlbumAdapter;
import se.datadosen.jalbum.event.JAlbumEvent;
import se.datadosen.util.IO;
import se.datadosen.util.Item;
import se.datadosen.util.NameValue;
import se.datadosen.util.VersionNumber;

import se.datadosen.util.Languages;

/*************************************************
 * 
 *					Lucid GUI
 * 
 */
 
public class Gui extends GuiBase {
		
	private boolean skinReady = false;
	private final VersionNumber jalbumVersion = new VersionNumber(AlbumBean.getInternalVersion());
	private final String skinVer = new SkinProperties(skinDirectory).getProperty(SkinProperties.VERSION);
	private final String supportForum = new SkinProperties(skinDirectory).getProperty(SkinProperties.SUPPORT_FORUM);
	private final String helpRoot = "http://jalbum.net/help/en/Skin/" + skin + "/";
	
	private String getStyleName() {
		window.ui2Engine();
		String style = engine.getStyle();
		return style.substring(0, style.indexOf("."));
	}
	
	private String getSkinName() {
		return engine.getSkin();
	}
	
	private String getLastSkinName() {
		String ls = null;
		
		try {
			ls = engine.getLastSkin();
		} catch(Throwable t) {
			JAlbum.logger.log(Level.FINER, "Last skin is unknown.");
		}
		
		return ls;
	}	
	
	private final String unCamelCase(String s) {
		
		if (s.length() > 0) {
			s = s.replaceAll("([a-z])([A-Z]+)", "$1 $2");
			return Character.toUpperCase(s.charAt(0)) + s.substring(1);
		}
		
		return "";
	}
	
	private final String getText(String t) {
		String s;
		
		try {
			s = texts.getString(t);
		} catch (java.util.MissingResourceException e) {
			JAlbum.logger.log(Level.FINE, "Missing text: \"{0}\"", t);
			if (t.startsWith("ui.")) {
				t = t.substring(3);
			}
			s = unCamelCase(t);
		}
		
		return s;
	}
	
	private final int uiHeight = 480;
	private final int uiWidth = 400;
	private final int previewWidth = 480;
	private final String tabStyle = "style='padding:3px 4px;margin:3px 4px;'";
	
	private String getFonts(String html, String ff, String hf) {
		String	gf = "",
				fw = "",
				bw = "bold",
				hw = "";
		String[] w;
		
		if (ff.contains(":")) {
			// Google font
			gf = ff;
			fw = ff.split(":")[1];
			if (fw.length() > 0) {
				w = fw.split(",");
				bw = (w.length > 2)? w[2] : w[1];
				fw = w[0];
			}
			
			ff = ff.split(":")[0];
			if (ff.contains(" ")) {
				ff = "\"" + ff + "\"";
			}
			ff += ",sans-serif";
		}

		if (!hf.equals("")) {
			if (!hf.equals(gf)) {
				gf = (gf.length() > 0)? (gf + "|" + hf) : hf;
			}
			if (hf.contains(":")) {
				// Has weight info
				hw = hf.split(":")[1];
				if (hw.length() > 0) {
					// Get base weight
					hw = hw.split(",")[0];
				}
				hf = hf.split(":")[0];
			}
			if (hf.contains(" ")) {
				hf = "\"" + hf + "\"";
			}
			hf += ",sans-serif";
		}
		
		//System.out.println("Google font: " + gf);
		return html.replace("{googleFontsLink}", (gf.length() == 0)? "" : ("\t<link rel=\"stylesheet\" href=\"http://fonts.googleapis.com/css?family=" + gf.replaceAll(" ", "+") + "&display=swap\">\n"))
			.replace("{fontFamily}", ff)
			.replace("{fontWeight}", (fw.equals("")? "300" : fw))
			.replace("{boldWeight}", bw)
			.replace("{headlineFont}", (hf.equals(""))? "inherit" : hf)
			.replace("{headlineWeight}", (hw.equals("")? "400" : hw));
		
	}
	
	private DateFormat getDateFormat() {
		return new SimpleDateFormat("yyyy-MMM-dd");
	}
	
	private String attemptSignIn() {
		SignInManager mgr = SignInManager.getInstance();
		if ( mgr != null && mgr.isSignedIn() ) {
			return "&cid=" + AccountManager.getCid(mgr.getUserName(), mgr.getPassword());
		}
		return "";
	}
	
	private String licensee = licenseManager.isLicenseValid()? licenseManager.getUserName() : "";
	
	private String license = licenseManager.isLicenseValid()? licenseManager.getLicenseType() : "";
	
	private boolean commercialUseAllowed = license.length() > 0 && "pro".equals(license);
	
	private Font mono = new Font("monospaced", Font.PLAIN, 12);
	
	private Border emptyBorder = BorderFactory.createEmptyBorder();
	
	private String getFileContents(String name) {
		StringBuilder cont = new StringBuilder();
		String line;
		String nl = System.getProperty("line.separator");
		File f = new File(skinDirectory, name);
		
		if (f.exists()) {
			try {
				try (java.io.BufferedReader in = new java.io.BufferedReader(new java.io.FileReader(f))) {
					while ((line = in.readLine()) != null) {
						cont.append(line);
						cont.append(nl);
					}
				}
			} catch (FileNotFoundException e) {
				return null;
			} catch (IOException e) {
				return null;
			}
		}
		return cont.toString();
	}
	
	private void writeFile(String fileName, String cont) {
		File out = new File(fileName);
		
		try {
			FileWriter writer = new FileWriter(out, false);
			writer.write(cont);
			writer.close();
		} catch (IOException e) {
			JAlbum.logger.log(Level.FINE, "Error writing \"{0}\".", fileName);
			e.printStackTrace();
		}
	}
	
	private final Double FULLY_TRANSP = 0.0001D;
	private final Double FULLY_OPAQUE = 0.9999D;
	
	private Color toColor(JColorSelector cs) {
		return toColor(cs.toString());
	}
	
	private Color toColor(String c) {
		int a = 255;
		Color clr = Color.GRAY;
		
		//System.out.print("toColor(" + c + ") -> ");
		if (c.charAt(0) == '#') {
			c = c.substring(1);
		}
		
		if (c.length() > 7) {
			a = Integer.parseInt(c.substring(0, 2), 16);
			c = c.substring(2);
		}
		
		if (c.length() == 3) {
			clr = new Color(
					Math.min(255, Integer.parseInt(c.substring(0,1), 16) * 17),
					Math.min(255, Integer.parseInt(c.substring(1,2), 16) * 17),
					Math.min(255, Integer.parseInt(c.substring(2), 16) * 17)
				);
		}
		
		if (c.length() == 6) {
			clr = new Color(
					Math.min(255, Integer.parseInt(c.substring(0,2), 16)),
					Math.min(255, Integer.parseInt(c.substring(2,4), 16)),
					Math.min(255, Integer.parseInt(c.substring(4), 16)),
					a
				);
			//System.out.println(clr.getAlpha() + " alpha=" + a);
		}
		
		return clr;
	}
	
	private String getCssColor(JColorSelector cs) {
		
		return getCssColor(toColor(cs));
	}
	
	private String getCssColor(String c) {
		
		return getCssColor(toColor(c));
	}
	
	private String getCssColor(Color c) {
		
		if (c == null) {
			return "transparent";
		}
		
		Double a = c.getAlpha() / 255.0;
		
		if (a < FULLY_TRANSP) {
			// Transparent
			return "transparent";
		} else if (a < FULLY_OPAQUE) {
			// Semi transparent
			return "rgba(" + c.getRed() + "," + c.getGreen() + "," + c.getBlue() + "," + (Math.round(a * 1000) / 1000.0) + ")";
		}
		
		// Opaque
		return "rgba(" + c.getRed() + "," + c.getGreen() + "," + c.getBlue() + ",1)";
	}
	
	private Double getAlpha(JColorSelector cs) {
		Color c = toColor(cs);
		
		return c.getAlpha() / 255.0;
	}
	
	private Double getAlpha(String clr) {
		Color c = toColor(clr);
		
		return c.getAlpha() / 255.0;
	}
	
	private String setAlpha(String clr, Double a) {
		Color c = toColor(clr);
		
		return "rgba(" + c.getRed() + "," + c.getGreen() + "," + c.getBlue() + "," + a + ")";
	}
	
	private Color flatColor(JColorSelector bc, JColorSelector fc) {
		
		return flatColor(toColor(bc), toColor(fc));
	}
	
	private Color flatColor(Color bc, Color fc) {
		Double a = fc.getAlpha() / 255.0;
			
		if (a < FULLY_TRANSP) {
			return bc;
		} else if (a > FULLY_OPAQUE) {
			return fc;
		}
		
		return new Color(
			(int)Math.max(0, Math.min(Math.round(a * fc.getRed() + (1 - a) * bc.getRed()), 255)),
			(int)Math.max(0, Math.min(Math.round(a * fc.getGreen() + (1 - a) * bc.getGreen()), 255)),
			(int)Math.max(0, Math.min(Math.round(a * fc.getBlue() + (1 - a) * bc.getBlue()), 255))
		);
	}
	
	private String getFlatColor(JColorSelector bc, JColorSelector fc) {
		
		return getCssColor(flatColor(toColor(bc), toColor(fc)));
	}
	
	private String getFlatColor(String bc, String fc) {
		
		return getCssColor(flatColor(toColor(bc), toColor(fc)));
	}
	
	private String getFlatColor(Color bc, Color fc) {
		
		return getCssColor(flatColor(bc, fc));
	}
	
	private Double getLuminosity(JColorSelector bg, JColorSelector fg) {
		
		return getLuminosity(flatColor(bg, fg));
	}
	
	private Double getLuminosity(JColorSelector cs) {
		
		return getLuminosity(toColor(cs));
	}
	
	private Double getLuminosity(String c) {
		
		return getLuminosity(toColor(c));
	}
	
	private Double getLuminosity(Color c) {
		
		return	0.0011725490196078 * c.getRed() + 
				0.0023019607843137 * c.getGreen() +
				0.0004470588235294118 * c.getBlue();
	}
	
	private boolean isLightColor(JColorSelector cs) {
		
		return getLuminosity(cs) > 0.5;
	}
	
	private boolean isLightColor(String fc) {
		
		return getLuminosity(fc) > 0.5;
	}
	
	private boolean isLightColor(JColorSelector fc, double threshold) {
		
		return getLuminosity(fc) > threshold;
	}
	
	private boolean isLightColor(String fc, double threshold) {
		
		return getLuminosity(fc) > threshold;
	}
	
	private boolean isLightColor(JColorSelector bc, JColorSelector fc) {
		
		return isLightColor(toColor(bc), toColor(fc), 0.5);
	}
	
	private boolean isLightColor(String bc, String fc) {
		
		return isLightColor(toColor(bc), toColor(fc), 0.5);
	}
	
	private boolean isLightColor(JColorSelector bc, JColorSelector fc, Double threshold) {
		
		return isLightColor(toColor(bc), toColor(fc), threshold);
	}
	
	private boolean isLightColor(String bc, String fc, Double threshold) {
		
		return isLightColor(toColor(bc), toColor(fc), threshold);
	}
	
	private boolean isLightColor(Color bc, Color fc, Double threshold) {
		
		Double a = fc.getAlpha() / 255.0;
		
		if (a > FULLY_OPAQUE) {
			return getLuminosity(fc) >= threshold;
		}
		
		return getLuminosity(flatColor(bc, fc)) >= threshold;
	}
	
	private String getLegibleColor(Color bc1, Color bc2, Color fc, Double f) {
		Color bc = flatColor(bc1, bc2);
		
		return getLegibleColor(bc, fc, f);
	}
		
	private String getLegibleColor(JColorSelector bc, JColorSelector fc) {
		return getLegibleColor(toColor(bc), toColor(fc), 0.6);
	}
	
	private String getLegibleColor(JColorSelector bc, JColorSelector fc, Double f) {
		return getLegibleColor(toColor(bc), toColor(fc), f);
	}
	
	private String getLegibleColor(JColorSelector bc1, JColorSelector bc2, JColorSelector fc) {
		return getLegibleColor(flatColor(bc1, bc2), toColor(fc), 0.6);
	}
	
	private String getLegibleColor(JColorSelector bc1, JColorSelector bc2, JColorSelector fc, Double f) {
		return getLegibleColor(flatColor(bc1, bc2), toColor(fc), f);
	}
	
	private String getLegibleColor(Color bc, Color fc, Double f) {
		int r, g, b;
		
		if (Math.abs(getLuminosity(bc) - getLuminosity(fc)) >= f) {
			return getCssColor(fc);
		}
		
		f *= 255.0;
		
		if (getLuminosity(bc) > 0.5) {
			// Darken
			r = (int)Math.round(Math.max(fc.getRed() - f, 0));
			g = (int)Math.round(Math.max(fc.getGreen() - f, 0));
			b = (int)Math.round(Math.max(fc.getBlue() - f, 0));
		} else {
			// Lighten
			r = (int)Math.round(Math.min(fc.getRed() + f, 255));
			g = (int)Math.round(Math.min(fc.getGreen() + f, 255));
			b = (int)Math.round(Math.min(fc.getBlue() + f, 255));
		}
		
		return getCssColor(new Color(r, g, b));
	}
	
	private final String[] imageFiles = new String[] { "jpg", "jpeg", "png", "gif", "svg" };
	private final FileChooser fc = ChooserFactory.createFileChooser(window);

	private void getFileToRes(String[] extensions, JTextField name, Component c) {
		fc.setFileFilter(new FileNameExtensionFilter(getText("ui.select"), extensions));
		int returnVal = fc.showOpenDialog(c);
		
		if (returnVal == JFileChooser.APPROVE_OPTION) {
			String fn = fc.getSelectedFile().toString();
			
			if (!fn.trim().equals("")) {
				File src = new File(fn);
				File dst = new File(context.getPluginContext().getRootFolder().getFile(), "res");
				if (!dst.exists()) {
					dst.mkdir();
				}
				if (src.exists() &&  dst.exists()) {
					try {
						IO.copyFile(src, dst);
						name.setText(src.getName());
					} catch (IOException e) {
						System.out.println(e);
					}
				}
			}
		}
	}
	
	private void backupProjectFile(String ver) {
		
		try {
			File projectFile = window.projectChooser.getSelectedFile();
			IO.copyFile(projectFile, new File(projectFile.getParentFile(), IO.baseName(projectFile) + "-" + ((ver == null)? "old" : ver) + ".jap"));
		} catch (IOException ex) {
			JAlbum.logger.log(Level.FINE, "Error backing up project file: {0}", window.projectChooser.getSelectedFile());
		}
	};
	
	private StateMonitor commercialMonitor = new StateMonitor() {
		@Override
		public void onChange() {
			
			if (((JCheckBox)source).isSelected() && !commercialUseAllowed) {
				if (skinReady) {
					Object[] options = { 
						getText("ui.signUp"),
						getText("ui.noThanks")
					};
					int n = JOptionPane.showOptionDialog(window, 
						getText("ui.licenseWarningText"), 
						getText("ui.licenseWarningTitle"), 
						JOptionPane.YES_NO_OPTION, 
						JOptionPane.INFORMATION_MESSAGE,
						null,
						options,
						options[1]
					);
					if (n == 0) {
						try {
							BrowserLauncher.openURL(JAlbumSite.getTrueInstance().getMyJAlbumUpgradeUrl() + "/?referrer=" + skin + attemptSignIn());
						} catch (se.datadosen.tags.ElementException | IOException x) {
						}
					}
				}
				((JCheckBox)source).setSelected(false);
			}
		}
	};
		
	private static Icon icon(String basename) {
		return Icons.get(Lazaicon.class, "svg/" + basename + ".svg", 18);
	}
		
	private static Icon icon(String basename, int size) {
		return Icons.get(Lazaicon.class, "svg/" + basename + ".svg", size);
	}
		
	private static Icon svgIcon(String basename, Dimension dim) {
		DeferredSVGIcon icon = new DeferredSVGIcon(Gui.class, "graphics/" + basename + ".svg");
		
		icon.setAntiAlias(true);
		icon.setAdaptColors(false);
		icon.setPreferredSize(dim);
		icon.setAutosize(SVGIcon.AUTOSIZE_STRETCH);
		
		return icon;
	}
		
	private Object[] getPosition() {
		return new Object[] {
			new Item("left top", getText("ui.left_top")), 
			new Item("center top", getText("ui.center_top")), 
			new Item("right top", getText("ui.right_top")), 
			new Item("left center", getText("ui.left_middle")), 
			new Item("center center", getText("ui.center_middle")), 
			new Item("right center", getText("ui.right_middle")), 
			new Item("left bottom", getText("ui.left_bottom")), 
			new Item("center bottom", getText("ui.center_bottom")), 
			new Item("right bottom", getText("ui.right_bottom"))
		};
	}

	private int setSelectedValue(JComboBox<Item> box, Object val) {
		int		size = box.getItemCount();
		Item	it;
		
		for (int i = 0; i < size; i++) {
			it = box.getItemAt(i);
			if (it.item.equals(val)) {
				box.setSelectedIndex(i);
				return i;
			}
		}
		
		return -1;
	}

	private String getSelectedValue(Object o) {
		if (o instanceof JComboBox) {
			Object so = ((JComboBox)o).getSelectedItem();
			if (so instanceof Item) {
				return ((Item)so).value.toString();
			}
			return (so == null)? null : so.toString();
		}
		return (o == null)? null : o.toString();
	}
	
	private int getSpinnerValueAsInt(JSpinner o) {
		return (Integer)o.getValue();
	}
	
	private Double getSpinnerValueAsDouble(JSpinner o) {
		return (Double)o.getValue();
	}
	
	private Double getSelectedValueAsDouble(Object o) {
		if (o instanceof JComboBox) {
			Object so = ((JComboBox)o).getSelectedItem();
			if (so instanceof Item) {
				String s = ((Item)so).value.toString();
				return ((s != null) && (s.length() > 0))? Double.parseDouble(s) : null;
			}
			return null;
		}
		return (o == null)? null : Double.parseDouble(o.toString());
	}
	
	private Integer getSelectedValueAsInt(JComboBox o) {
		Object so = ((JComboBox)o).getSelectedItem();
		if (so instanceof Item) {
			return Integer.parseInt(((Item)so).value.toString());
		}
		return null;
	}
	
	private Double getThumbAspectRatio() {
		String[] cdim = engine.getThumbSize().split("x");
		Double	w = Double.parseDouble(cdim[0]),
				h = Double.parseDouble(cdim[1]);
		
		if (w != null && h != null) {
			return w / h;
		}
		
		JAlbum.logger.log(Level.FINE, "Invalid thumbnail dimensions: {0} x {1}", new Object[]{ w.toString(), h.toString() });
		return null;
	}
	
	private Date skinLoadedOn = new Date();
	
	// Detects skin load (within two seconds of initing?)
	// isSynching and currentThread methods are unreliable :(
	
	private boolean isSynchingUI() {
		return ((new Date()).getTime() - skinLoadedOn.getTime()) < 2000;
		/*
		try {
			java.lang.reflect.Method m = se.datadosen.util.PropertyBinder.class.getMethod("isSynchingUI");
			
			Object is = m.invoke(null);
			String tn = Thread.currentThread().getName();
			//JAlbum.logger.log(Level.FINE, "isSynchingUI: {0}, Thread: {1}", new Object[]{is.toString(), tn});
			return !is.equals(Boolean.FALSE) && !tn.equals("Skin onload");
			
		} catch (IllegalAccessException | IllegalArgumentException | InvocationTargetException | NoSuchMethodException | SecurityException ex) {
			JAlbum.logger.log(Level.FINE, "isSynchingUI error. {0}", ex);
		}
		return false;
		*/
	}
	
	private void allowHTMLEditing(JSmartTextArea ta) {
		try {
			ta.setAllowHTMLEditing(true).setFullHTMLEditing(true);
		} catch (Throwable e) {
		}
	}
	
	private void allowSpelling(JSmartTextArea ta) {
		try {
			ta.spelling();
		} catch (Throwable e) {
		}
	}
			
	private String majorVersion(String s) {
		int i = s.indexOf(".");
		if (i > 0) {
			return s.substring(0, i);
		}
		return s;
	}
	
	private int getMajorVersion(Object v) {
		
		if (v == null) {
			return -1;
		} else if (v instanceof Integer) {
			return (int)v;
		} else if (v instanceof String) {
			try {
				return Integer.valueOf((String)v);
			} catch(NumberFormatException ex) {
				return -1;
			}
		} else if (v instanceof Double) {
			return (int)Math.floor((Double)v);
		}
		return -1;
	}
	
	/*****************************************
	 *
	 *			Skin UI starts here
	 *
	 *****************************************/
		
	ControlPanel skinUi = new ControlPanel() {
			
		// UI-wide variables

		// styleName is stored as skin variable in order to check style change later
		private JTextField styleName = new JTextField(getStyleName());
		
		// Icons used multple places
		private Icon infoIcon = icon("info");
		private Icon mandatory = svgIcon("asterisk", new Dimension(16, 16));
		
		// skinVersion is stored as skin variable in order to check major skin version change later
		JTextField skinVersion = new JTextField(engine.getSkinProperties().getProperty(SkinProperties.VERSION, "0.0"));
		// silence further notifications
		private boolean skinChangeReported = false;
				
		@Override
		public void importVariables(Map<String, Object> vars) {
			
			//JAlbum.logger.log(Level.FINE, "Project file loaded: {0}", window.projectChooser.getSelectedFile());
			//JAlbum.logger.log(Level.FINE, "{0}", vars.keySet());
			
			if (skinChangeReported || window.projectChooser.getSelectedFile() == null) {
				return;
			}
			
			Map<String, Object> skinVars = engine.getSkinVariables();
			int	omv = getMajorVersion(skinVars.get("majorSkinVersion")),
				cmv = getMajorVersion(skinVer);
			
			if (omv == -1) {
				omv = getMajorVersion(skinVars.get("skinVersion"));
			}
			
			// Old skin name
			String	lastSkin = getLastSkinName();
			boolean skinChanged = lastSkin != null && !getSkinName().equals(lastSkin);
			
			// Attempt writing out the current version
			if (cmv != -1) {
				skinVars.put("majorSkinVersion", cmv);
			}
			
			if (skinChanged || cmv > omv) {
				JNotification jn;
			
				//JAlbum.logger.log(Level.FINE, "Major version change {0} v{1} -> {2} v{3}", new Object[]{oldSkinName, oldVersion, skin, skinVer});
				if (skinChanged) {
					// Other skin
					JAlbum.logger.log(Level.FINE, "Skin changed: {0} -> {1} v{2}", new Object[]{lastSkin, skin, cmv});
					jn = new JNotification(getText("ui.madeWithAnotherSkin") + "\n" + 
							getText("ui.backupSaved"), JNotification.Type.SKIN);
					backupProjectFile((lastSkin != null)? lastSkin : "old");					
				} else {
					// Major version change
					JAlbum.logger.log(Level.FINE, "Major version changed: {0} v{1} -> v{2}", new Object[]{skin, omv, cmv});
					jn = new JNotification(getText("ui.madeWithOldVersion").replaceAll("\\{0\\}", skin) + 
							((omv == -1)? "" : ("(" + omv + ")"))+ "\n" +
							"\n" + getText("ui.backupSaved"), JNotification.Type.SKIN);
					backupProjectFile((omv >= 0)? String.valueOf(omv) : "old");
				}
				
				jn.setExpiration(3);
				jn.setRememberDismissed(true);
				window.showNotification(jn);
				skinChangeReported = true;
				
				// Have to process all subdirectories after a skin or major version change
				engine.setUpdatedDirsOnly(false);
			}
		}
		
		JTextField facebookAppId = new JSmartTextField(16);
		
		JColorSelector backgroundColor = new JAlphaColorSelector(getText("ui.backgroundColor"), new JSmartTextField(10));
		JColorSelector controlsBackgroundColor = new JAlphaColorSelector(getText("ui.controlsBackgroundColor"), new JSmartTextField(10));
		JColorSelector textColor = new JColorSelector(getText("ui.textColor"), new JSmartTextField(10));
		JColorSelector buttonColor = new JAlphaColorSelector(getText("ui.button"), new JSmartTextField(10));
		JColorSelector buttonTextColor = new JColorSelector(getText("ui.buttonText"), new JSmartTextField(10));
		
		// Typography
		private final JComboBox fontSuggestions = new JComboBox(new Object[] {
			"[" + getText("ui.suggestedFonts") +  "]",
			"Cinzel Decorative / Raleway",
			"Dosis Bold / Montserrat",
			"Exo 2 / Exo 2",
			"Fjalla One / Roboto",
			"IBM Plex Serif / IBM Plex Sans",
			"Jura / Roboto Condensed",
			"Martel / Roboto",
			"Merriweather / Merriweather Sans",
			"Oswald / Open Sans",
			"Raleway / Open Sans",
			"Roboto Condensed / Roboto",
			"Roboto Slab / Montserrat",
			"Special Elite / Yanone Kaffeesatz",
			"Unica One / Lato",
			"Yanone Kaffeesatz Bold / Muli"
		});

		private void setFontBoxes() {
			
			if (fontSuggestions.getSelectedIndex() == 0) {
				return;
			}
			String	s = fontSuggestions.getSelectedItem().toString(),
					hf = s.split("/")[0].trim(),
					ff = s.split("/")[1].trim();

			setSelectedValue(fontFamily, ff);
			setSelectedValue(headlineFont, hf);
			fontSuggestions.setSelectedIndex(0);
		};
		
		JComboBox fontFamily = new JComboBox(new Object[] {
			new Item("Arial, \"Helvetica Neue\", Helvetica, sans-serif", "Arial"),
			new Item("Baskerville, \"Baskerville Old Face\", \"Hoefler Text\", Garamond, \"Times New Roman\", serif", "Baskerville"),
			new Item("Calibri, Candara, Segoe, \"Segoe UI\", Optima, Arial, sans-serif", "Calibri"),
			new Item("Cambria, Georgia, Times, \"Times New Roman\", serif", "Cambria"),
			new Item("\"Century Gothic\", \"Apple Gothic\", \"Goudy Old Style\", sans-serif", "Century Gothic"),
			new Item("\"Comic Sans MS\", cursive", "Comic Sans"),
			new Item("Consolas, \"Lucida Console\", Monaco, monospace", "Consolas"),
			new Item("Constantia, Palatino, \"Palatino Linotype\", \"Palatino LT STD\", Georgia, serif", "Constantia"),
			new Item("\"Copperplate Light\", \"Copperplate Gothic Light\", serif", "Copperplate Light"),
			new Item("\"Courier New\", Courier, monospace", "Courier New"),
			new Item("\"Franklin Gothic Medium\", \"Arial Narrow Bold\", Arial, sans-serif", "Franklin Gothic"),
			new Item("Futura, \"Century Gothic\", AppleGothic, sans-serif", "Futura"),
			new Item("Garamond, \"Hoefler Text\", Times New Roman, Times, serif", "Garamond"),
			new Item("Geneva, \"Lucida Sans\", \"Lucida Grande\", \"Lucida Sans Unicode\", Verdana, sans-serif", "Geneva"),
			new Item("Georgia, Palatino, \"Palatino Linotype\", Times, \"Times New Roman\", serif", "Georgia"),
			new Item("\"Gill Sans\", \"Gill Sans MT\", Calibri, \"Trebuchet MS\", sans-serif", "Gill Sans"),
			new Item("\"Goudy Old Style\", Garamond, \"Big Caslon\", \"Times New Roman\", serif", "Goudy Old Style"),
			new Item("\"Helvetica Neue\", Helvetica, Arial, sans-serif", "Helvetica Neue"),
			new Item("\"Hoefler Text\", Constantia, Palatino, \"Palatino Linotype\", \"Book Antiqua\", Georgia, serif", "Hoefler Text"),
			new Item("Impact, Haettenschweiler, \"Arial Narrow Bold\", sans-serif", "Impact"),
			new Item("\"Lucida Sans\", \"Lucida Grande\", \"Lucida Sans Unicode\", Verdana, sans-serif", "Lucida Sans"),
			new Item("\"Lucida Bright\", Georgia, serif", "Lucida Bright"),
			new Item("Palatino, \"Palatino Linotype\", \"Book Antiqua\", Georgia, serif", "Palatino"),
			new Item("Segoe, \"Segoe UI\", Tahoma, Geneva, \"Nimbus Sans L\", sans-serif", "Segoe"),
			new Item("Tahoma, Geneva, Verdana, sans-serif", "Tahoma"),
			new Item("Times, \"Times New Roman\", Georgia, serif", "Times"),
			new Item("\"Trebuchet MS\", \"Lucida Sans Unicode\", \"Lucida Grande\", \"Lucida Sans\", Tahoma, sans-serif", "Trebuchet MS"),
			new Item("Verdana, Geneva, Tahoma, sans-serif", "Verdana"),
			new Item("Alegreya Sans:300,300i,500,500i", "Alegreya Sans Light"),
			new Item("Barlow:300,300i,600,600i", "Barlow Light"),
			new Item("Barlow Semi Condensed:300,300i,500,500i", "Barlow Semi Condensed"),
			new Item("Exo 2:300,300i,600,600i", "Exo 2"),
			new Item("Fira Sans:300,300i,600,600i", "Fira Sans"),
			new Item("IBM Plex Sans:300,300i,600,600i", "IBM Plex Sans"),
			new Item("Josefin Sans:300,300i,600,600i", "Josefin Sans"),
			new Item("Josefin Slab:300,300i,600,600i", "Josefin Slab"),
			new Item("Lato:300,300i,700,700i", "Lato"),
			new Item("Merriweather Sans:300,300i,700,700i", "Merriweather Sans"),
			new Item("Merriweather:300,300i,700,700i", "Merriweather Serif"),
			new Item("Montserrat:300,300i,600,600i", "Montserrat"),
			new Item("Muli:300,300i,700,700i", "Muli"),
			new Item("Open Sans:300,300i,600,600i", "Open Sans"),
			new Item("Raleway:300,300i,600,600i", "Raleway"),
			new Item("Roboto:300,300i,500,500i", "Roboto"),
			new Item("Roboto Condensed:300,300i,700,700i", "Roboto Condensed"),
			new Item("Source Sans Pro:300,300i,600,600i", "Source Sans Pro"),
			new Item("Yanone Kaffeesatz:300,500", "Yanone Kaffeesatz")
		});
		
		JComboBox headlineFont = new JComboBox(new Object[] {
			new Item("", "[ " + getText("ui.sameAsBaseFont") + " ]"),
			new Item("Abril Fatface", "Abril Fatface"),
			new Item("Amaranth", "Amaranth"),
			new Item("Amatic SC:700", "Amatic SC Bold"),
			new Item("Arapey", "Arapey"),
			new Item("Barlow:300", "Barlow Light"),
			new Item("Barlow Semi Condensed:300", "Barlow Semi Condensed Light"),
			new Item("Barlow Condensed", "Barlow Condensed"),
			new Item("Cinzel Decorative", "Cinzel Decorative"),
			new Item("Dosis:600", "Dosis Bold"),
			new Item("Euphoria Script", "Euphoria Script"),
			new Item("Exo 2:300", "Exo 2"),
			new Item("Fauna One", "Fauna One"),
			new Item("Fjalla One", "Fjalla One"),
			new Item("Gilda Display", "Gilda Display"),
			new Item("Grand Hotel", "Grand Hotel"),
			new Item("Great Vibes", "Great Vibes"),
			new Item("IBM Plex Serif:300", "IBM Plex Serif"),
			new Item("Josefin Slab", "Josefin Slab"),
			new Item("Julius Sans One", "Julius Sans One"),
			new Item("Jura", "Jura"),
			new Item("Libre Baskerville", "Libre Baskerville"),
			new Item("Lobster", "Lobster"),
			new Item("Lobster Two", "Lobster Two"),
			new Item("Martel:300", "Martel"),
			new Item("Marvel:700", "Marvel Bold"),
			new Item("Merriweather:300", "Merriweather"),
			new Item("Oswald", "Oswald"),
			new Item("Poiret One", "Poiret One"),
			new Item("PT Sans Narrow", "PT Sans Narrow"),
			new Item("Raleway:300", "Raleway"),
			new Item("Raleway:600", "Raleway Bold"),
			new Item("Roboto Condensed", "Roboto Condensed"),
			new Item("Roboto Slab:300", "Roboto Slab"),
			new Item("Roboto Slab:600", "Roboto Slab Bold"),
			new Item("Scope One", "Scope One"),
			new Item("Six Caps", "Six Caps"),
			new Item("Sofia", "Sofia"),
			new Item("Special Elite", "Special Elite"),
			new Item("Unica One", "Unica One"),
			new Item("Vidaloka", "Vidaloka"),
			new Item("Yanone Kaffeesatz:300", "Yanone Kaffeesatz"),
			new Item("Yanone Kaffeesatz:500", "Yanone Kaffeesatz Bold")
		});
		
		JSpinner thumbGap =  new JSpinner(new SpinnerNumberModel(1, 0, 20, 1));
		
		JComboBox thumbStyle = new JComboBox(new Object[] {
			new Item("grid", getText("ui.grid")), 
			new Item("dots", getText("ui.dots")), 
			new Item("masonry", getText("ui.masonry"))
		});
		
		JSpinner neighboringItemsOpacity = new JSpinner(new SpinnerNumberModel(25, 0, 100, 5));
		
		JSpinner fillHorizontally = new JSpinner(new SpinnerNumberModel(90, 50, 100, 1));
		JSpinner fillVertically = new JSpinner(new SpinnerNumberModel(100, 50, 100, 1));
		
		JCheckBox showFullscreen = new JCheckBox(getText("ui.showFullScreenButton"), true);
		JCheckBox showImageNumbers = new JCheckBox(getText("ui.showNumbers"));
		
		JTextArea imageCaptionTemplate = new JSmartTextArea(5, 12);

		JComboBox captionPlacement = new JComboBox(new Object[] {
			new Item("left top", getText("ui.left_top")), 
			new Item("center top", getText("ui.center_top")), 
			new Item("right top", getText("ui.right_top")), 
			new Item("left middle", getText("ui.left_middle")), 
			new Item("center middle", getText("ui.center_middle")), 
			new Item("right middle", getText("ui.right_middle")), 
			new Item("left bottom", getText("ui.left_bottom")), 
			new Item("center bottom", getText("ui.center_bottom")), 
			new Item("right bottom", getText("ui.right_bottom"))
		});
		
		JComboBox captionStyle = new JComboBox(new Object[] {
			new Item("white", getText("ui.white")), 
			new Item("light", getText("ui.light")), 
			new Item("transparent", getText("ui.transparent")), 
			new Item("dark", getText("ui.dark")),
			new Item("black", getText("ui.black"))
		});
		
		JCheckBox printImageButton = new JCheckBox(getText("ui.printImageButton"));
		JCheckBox showDownload = new JCheckBox(getText("ui.showDownload"));
		JCheckBox showPhotoData = new JCheckBox(getText("ui.showPhotoData"));
		
		/*	---------------------------------------------------------------
									Previews
			--------------------------------------------------------------- */
		
		WebViewBrowser designPreview = new WebViewBrowser();
		WebViewBrowser thumbsPreview = new WebViewBrowser();
						
		private String lastThumbDimSuggestion = "";
		
		// Calculating proper thumb dimensions for the different thumb layouts
		
		private void setThumbDimensions() {
			
			if (!skinReady) {
				return;
			}
			
			String	style = ((Item)thumbStyle.getSelectedItem()).value.toString(),
					cdim = engine.getThumbSize(),
					ndim;

			switch (style) {
				case "grid":
					ndim = "120x120";
					break;
				case "dots":
					ndim = "120x120";
					break;
				default:
					ndim = "200x120";
					break;
			}
			
			if (!cdim.equals(ndim) && !lastThumbDimSuggestion.equals(ndim)) {
				
				// Ask for changing thumb dims
				Object[] options = { 
					getText("ui.yes"),
					getText("ui.no")
				};

				int n = JOptionPane.showOptionDialog(window, 
					getText("ui.changeThumbnailDimensionsQuestion").replace("{0}", ndim), 
					getText("ui.updatingThumbnailDimensions"), 
					JOptionPane.YES_NO_OPTION, 
					JOptionPane.INFORMATION_MESSAGE,
					null,
					options,
					options[0]
				);

				if (n == 0) {
					try {
						JAlbum.logger.log(Level.FINE, "Thumb size is changed to {0}px", ndim);
						window.ui2Engine();
						engine.setThumbSize(ndim);
						window.engine2UI();
					} catch (ParameterException ex) {
						throw new RuntimeException(ex);
					}
					lastThumbDimSuggestion = "";
				} else {
					lastThumbDimSuggestion = ndim;
				}
			}
		};
		
		// Refreshing preview
						
		private void refreshDesignPreview() {
			String	html,
					bc = getCssColor(backgroundColor.toString()),
					cbc = getCssColor(controlsBackgroundColor.toString()),
					caption = imageCaptionTemplate.getText().trim();
			Double	cbo = getAlpha(controlsBackgroundColor.toString());

			if (caption.length() > 0) {
				caption = caption
					.replace("${fileLabel}", "Title or file name")
					.replace("${fileTitle}", "Title")
					.replace("${author}", "Author's Name.")
					.replace("${keywords}", "Keyword 1, Keyword 2")
					.replace("${comment}", "This is the comment.")
					.replace("${commentShort}", "Short comment.");
			}
			
			html = designTemplate
				.replace("{bodyClass}", (cbo > 0.95)? "opaque" : "transp")
				.replace("{backgroundColor}", bc)
				.replace("{controlsBgColor}", cbc)
				.replace("{textColor}", textColor.toString())
				.replace("{buttonColor}", getCssColor(buttonColor.toString()))
				.replace("{buttonTextColor}", getCssColor(buttonTextColor.toString()))
				.replace("{fullscreenVisible}", showFullscreen.isSelected()? "visible" : "hidden")
				.replace("{numbersVisible}", showImageNumbers.isSelected()? "visible" : "hidden")
				.replace("{captionPlacement}", getSelectedValue(captionPlacement))
				.replace("{captionStyle}", getSelectedValue(captionStyle))
				.replace("{captionVisibility}", ((caption.length() > 0) || printImageButton.isSelected() || showDownload.isSelected() || showPhotoData.isSelected())? "visible" : "hidden") 
				.replace("{caption}", caption)
				.replace("{showPrint}", printImageButton.isSelected()? "" : "hidden")
				.replace("{showDownload}", showDownload.isSelected()? "" : "hidden")
				.replace("{showPhotodata}", showPhotoData.isSelected()? "" : "hidden")
				.replace("{fillHorizontal}", getSpinnerValueAsInt(fillHorizontally) + "")
				.replace("{fillVertical}", getSpinnerValueAsInt(fillVertically) + "")
				.replace("{neighboringItemsOpacity}", (getSpinnerValueAsInt(neighboringItemsOpacity) / 100.0) + "");
			
			html = getFonts(html, getSelectedValue(fontFamily), (headlineFont.getSelectedIndex() == 0)? "" : getSelectedValue(headlineFont));
			
			final String html1 = html;

			SwingUtilities.invokeLater(() -> {
				designPreview.loadContent(html1);
			});
			//writeFile("D:/Temp/design-preview.html", html);

		};
		
		private void refreshThumbsPreview() {
			String	html;
			
			html = thumbsTemplate.
				replace("{backgroundColor}", getCssColor(backgroundColor.toString()))
				.replace("{panelBackgroundColor}", getCssColor(controlsBackgroundColor.toString()))
				.replace("{thumbGap}", thumbGap.getValue().toString())
				.replace("{thumbStyle}", ((Item)thumbStyle.getSelectedItem()).value.toString());
							
			//System.out.println("\nPreview reload triggered by: " + src + "\n\n" + html);
			
			final String html1 = html;

			SwingUtilities.invokeLater(() -> {
				thumbsPreview.loadContent(html1);
			});
			//writeFile("D:/Temp/thumbs-preview.html", html);
		};
			
		/*	---------------------------------------------------------------
									Monitors
			--------------------------------------------------------------- */
			
		PropertyChangeListener setupMonitors = pce -> {
			
			StateMonitor.monitoring(fontSuggestions).onChange(src -> {
				if (skinReady && src != null) {
					setFontBoxes();
				}
			});

			StateMonitor.monitoring(
					thumbGap, 
					thumbStyle).onChange(src -> {
			
				if (skinReady && src != null) {
					refreshThumbsPreview();
				}
			});
				
			// Design preview update
			StateMonitor.monitoring(styleName,
					backgroundColor, 
					controlsBackgroundColor, 
					textColor, 
					buttonColor, 
					buttonTextColor, 
					fontFamily, 
					headlineFont, 
					neighboringItemsOpacity, 
					fillHorizontally, 
					fillVertically, 
					showFullscreen, 
					showImageNumbers, 
					captionPlacement, 
					captionStyle, 
					imageCaptionTemplate,
					printImageButton,
					showDownload,
					showPhotoData
				).onChange(src -> {

				if (skinReady && src != null) {
					refreshDesignPreview();
				}
			});

			// Initial preview rendering
			
			refreshDesignPreview();
			refreshThumbsPreview();
		};
		
		/*	---------------------------------------------------------------
									Site
			--------------------------------------------------------------- */
		
		// Design
		
		ControlPanel design = new ControlPanel() {

			JComboBox<Item> language = new JComboBox<Item>() {
				{
					setModel(Languages.modelFrom(new File(skinDirectory, "texts")));
					insertItemAt(new Item("jalbum", "[ " + getText("ui.jAlbumPreference") + " ]"), 0);
					setSelectedIndex(0);
				}
			};
			
			JSmartTextField backgroundImageName = new JSmartTextField(12);
			JButton selectImage = new JButton();
			JComboBox backgroundPos = new JComboBox(getPosition());
			JComboBox backgroundRepeat = new JComboBox(new Object[] {
				new Item("no-repeat", getText("ui.no_repeat")),
				new Item("repeat-x", getText("ui.repeat_x")),
				new Item("repeat-y", getText("ui.repeat_y")),
				new Item("repeat", getText("ui.repeat_both")),
				new Item("stretch", getText("ui.stretch")),
			});
			JComboBox modalWindowsTheme = new JComboBox(new Object[] {
				new Item("auto", getText("ui.auto")),
				new Item("light", getText("ui.light")),
				new Item("dark", getText("ui.dark"))
			});
			JComboBox iconStyle = new JComboBox(new Object[] {
				new Item("thin", getText("ui.thin")),
				new Item("fat", getText("ui.fat"))
			});

			//WrappableJLabel lightboxNote = new WrappableJLabel("<html><i>" + getText("ui.pushesDownLightboxIfOpaque") + "</i></html>");

			{
				language.setToolTipText(getText("ui.languageInfo"));
				selectImage.setText(getText("ui.select"));
				selectImage.addActionListener(new ActionListener() { 
					@Override
					public void actionPerformed(ActionEvent e) {
						getFileToRes(imageFiles, backgroundImageName, skinUi);
				}});
				backgroundColor.setToolTipText(getText("ui.backgroundColorInfo"));
				backgroundColor.getTextComponent().setFont(mono);
				controlsBackgroundColor.setToolTipText(getText("ui.controlsBackgroundColorInfo"));
				//lightboxNote.setPreferredWidth(300);
				textColor.setToolTipText(getText("ui.textColorInfo"));
				controlsBackgroundColor.getTextComponent().setFont(mono);
				textColor.getTextComponent().setFont(mono);
				buttonColor.getTextComponent().setFont(mono);	
				buttonTextColor.getTextComponent().setFont(mono);	

				add("", new JLabelFor(getText("ui.language"), language));
				add("tab", language);
				add("br", new JLabelFor(getText("ui.backgroundImage"), backgroundImageName));
				add("tab", backgroundImageName);
				add("", selectImage);
				add("br", new JLabelFor(getText("ui.position"), backgroundPos));
				add("tab", backgroundPos);
				add("br tab", backgroundRepeat);
				add("br", new JLabelFor(getText("ui.backgroundColor"), backgroundColor));
				add("tab", backgroundColor);
				add("tab", backgroundColor.getTextComponent());
				add("br", new JLabelFor(getText("ui.controlsBackgroundColor"), controlsBackgroundColor));
				add("tab", controlsBackgroundColor);
				add("tab", controlsBackgroundColor.getTextComponent());
				add("br", new JLabelFor(getText("ui.textColor"), textColor));
				add("tab", textColor);
				add("tab", textColor.getTextComponent());
				add("br", new JLabelFor(getText("ui.button") + " / " + getText("ui.highlight"), buttonColor));
				add("tab", buttonColor);
				add("tab", buttonColor.getTextComponent());
				add("br", new JLabelFor(getText("ui.buttonText"), buttonTextColor));
				add("tab", buttonTextColor);
				add("tab", buttonTextColor.getTextComponent());
				add("br", new JLabelFor(getText("ui.modalWindowsTheme"), modalWindowsTheme));
				add("tab", modalWindowsTheme);
				add("br", new JLabel(getText("ui.iconStyle")));
				add("tab", iconStyle);
				
				//putClientProperty("helpPage", helpRoot + "ui/design.html");
			}
		};
				
		// Typography panel
		
		ControlPanel typography = new ControlPanel() {

			{
				headlineFont.setEditable(true);

				add(new JLabelFor(getText("ui.fontFamily"), fontFamily));
				add("tab", fontFamily);
				add("br", new JLabelFor(getText("ui.headlineFont"), headlineFont));
				add("tab", headlineFont);
				add("br", new JLabelFor(getText("ui.pairingSuggestions"), fontSuggestions));
				add("tab", fontSuggestions);
			}
		};
		
		// Title page panel

		ControlPanel titlePage = new ControlPanel(new BorderLayout(0, 0)) {
				
			ControlPanel titlePageSettings = new ControlPanel() {

				JTextField logoName = new JSmartTextField(12);
				JButton selectLogo = new JButton(getText("ui.select"));
				JSmartTextArea headerContent = new JSmartTextArea(6, 20);
				JScrollPane headerContentPane = new JScrollPane(headerContent);
				JSmartTextArea footerContent = new JSmartTextArea(6, 20);
				JScrollPane footerContentPane = new JScrollPane(footerContent);
				
				{
					selectLogo.setText(getText("ui.select"));
					selectLogo.addActionListener(new ActionListener() { 
						@Override
						public void actionPerformed(ActionEvent e) {
							getFileToRes(imageFiles, logoName, skinUi);
					}});
					headerContent.setLineWrap(false);
					headerContent.setFont(mono);
					headerContent.setTabSize(2);
					allowHTMLEditing(headerContent);
					headerContentPane.setBorder(BorderFactory.createTitledBorder(getText("ui.aboveContent")));
					footerContent.setLineWrap(false);
					footerContent.setFont(mono);
					footerContent.setTabSize(2);
					allowHTMLEditing(footerContent);
					footerContentPane.setBorder(BorderFactory.createTitledBorder(getText("ui.belowContent")));
					
					add("", new JLabel(getText("ui.logo")));
					add("tab hfill", logoName);
					add("", selectLogo);
					add("br hfill", headerContentPane);
					add("br hfill", footerContentPane);
				}
			};
				
			ControlPanel folderCaptionSettings = new ControlPanel() {
				
				private JComboBox<NameValue<String>> presets = new JComboBox<>(new NameValue[] {
					new NameValue("[ " + getText("ui.choosePreset") + " ]", null),
					new NameValue(getText("ui.empty"), ""),
					new NameValue(getText("ui.title"), "<h3>${title}</h3>"),
					new NameValue(getText("ui.title") + " + " + getText("ui.comment"), "<h3>${title}</h3>\n<div class=\"comment\">${comment}</div>"),
					new NameValue(getText("ui.title") + " + " + getText("ui.commentShort") + " (" + getText("ui.default") + ")", "<h3>${title}</h3>\n<div class=\"comment\">${commentShort}</div>"),
					new NameValue(getText("ui.title") + " + " + getText("ui.comment") + " + " + getText("ui.date"), "<h3>${title}</h3>\n<div class=\"comment\">${comment}</div>\n<div class=\"date\">${folderModDate}</div>"),
					new NameValue(getText("ui.fileLabel"), "<h3>${fileLabel}</h3>"),
					new NameValue(getText("ui.fileLabel") + " + " + getText("ui.comment"), "<h3>${fileLabel}</h3>\n<div class=\"comment\">${comment}</div>"),
					new NameValue(getText("ui.fileLabel") + " + " + getText("ui.comment") + " + " + getText("ui.date"), "<h3>${fileLabel}</h3>\n<div class=\"comment\">${comment}</div>\n<div class=\"date\">${folderModDate}</div>")
				});
				JTextArea folderCaptionTemplate = new JTextArea(6,20);
				JScrollPane captionPane = new JScrollPane(folderCaptionTemplate);
				JComboBox folderDateSource = new JComboBox(new Object[]{
					new Item("none", getText("ui.none")),
					new Item("fileDate", getText("ui.fileDate")),
					new Item("folderModDate", getText("ui.folderModDate")),
					new Item("lastCameraDate", getText("ui.lastCameraDate")),
					new Item("cameraDateRange", getText("ui.cameraDateRange"))
				});
				JCheckBox showFolderImageCount = new JCheckBox(getText("ui.showFolderImageCount"));	

				{
					folderCaptionTemplate.setEditable(true);
					folderCaptionTemplate.setLineWrap(true);
					folderCaptionTemplate.setWrapStyleWord(true);
					folderCaptionTemplate.setFont(mono);
					folderCaptionTemplate.setTabSize(2);
					captionPane.setBorder(BorderFactory.createTitledBorder(getText("ui.captionTemplate")));
					presets.addItemListener(listener -> {
						if (presets.getSelectedIndex() > 0) {
							folderCaptionTemplate.setText(((NameValue<String>)presets.getSelectedItem()).value);
						}
					});

					add("", new JLabelFor(getText("ui.presets"), presets));
					add("tab", presets);
					add("br hfill", captionPane);
					add("br", new JLabelFor("${folderModDate} = ", folderDateSource));
					add(" ", folderDateSource);
					add("br", showFolderImageCount);

				}
			};
							
			JTabbedPane titlePageTabs = new JTabbedPane() {
				
				{
					titlePageSettings.setBorder(emptyBorder);
					folderCaptionSettings.setBorder(emptyBorder);
					
					addTab("<html><h4 " + tabStyle + ">" + getText("ui.layout") + "</h4></html>", titlePageSettings);
					addTab("<html><h4 " + tabStyle + ">" + getText("ui.folderCaption") + "</h4></html>", folderCaptionSettings);
				}
			};
			
			{
				((BorderLayout)(getLayout())).setVgap(0);
				((BorderLayout)(getLayout())).setHgap(0);
				
				add(titlePageTabs);
			}
		};
		
		// Custom content page panel
		/*
		ControlPanel customContent = new ControlPanel() {
				
			ControlPanel headerPanel = new ControlPanel() {

				JSmartTextArea headerContent = new JSmartTextArea(6, 12);
				JScrollPane headerContentPane = new JScrollPane(headerContent);

				{
					//headerContent.setEditable(true);
					headerContent.setLineWrap(false);
					headerContent.setFont(mono);
					headerContent.setTabSize(2);
					allowHTMLEditing(headerContent);

					add("hfill vfill", headerContentPane);
				}
			};

			ControlPanel footerPanel = new ControlPanel() {

				JSmartTextArea footerContent = new JSmartTextArea(6, 12);
				JScrollPane footerContentPane = new JScrollPane(footerContent);

				{
					//footerContent.setEditable(true);
					footerContent.setLineWrap(false);
					footerContent.setFont(mono);
					footerContent.setTabSize(2);
					allowHTMLEditing(footerContent);

					add("hfill vfill", footerContentPane);
				}
			};
				
			JTabbedPane titlePageTabs = new JTabbedPane() {
				
				{
					headerPanel.setBorder(emptyBorder);
					footerPanel.setBorder(emptyBorder);
					
					addTab("<html><h4 " + tabStyle + ">" + getText("ui.header") + "</h4></html>", headerPanel);
					addTab("<html><h4 " + tabStyle + ">" + getText("ui.footer") + "</h4></html>", footerPanel);
				}
			};
			
			{
				add("hfill vfill", titlePageTabs);
			}

		};
		*/
		// Mark new files
		
		ControlPanel markNew = new ControlPanel() {

			JCheckBox markFilesNew = new JCheckBox(getText("ui.markFilesNew"));	
			JTextField newDaysCount = new JSmartTextField("60", 3);
			JComboBox newDaysRef = new JComboBox(new Object[] {
				new Item("dateTaken", getText("ui.dateTaken")),
				new Item("fileModified", getText("ui.fileModified")),
				new Item("added", getText("ui.addedToAlbum"))
			});
			JComboBox newDaysMark = new JComboBox(new Object[] {
				new Item("icon", getText("ui.icon")),
				new Item("text", getText("ui.text"))
			});
			JTextField newDaysText = new JSmartTextField(getText("new"), 6);
			JLabel newIcon = new JLabel(icon("new-fill"));
			
			{
				newDaysCount.setToolTipText(getText("ui.newDaysCountInfo"));
				newDaysMark.addItemListener(listener -> {
					int i = newDaysMark.getSelectedIndex();
					newDaysText.setVisible(i == 1);
					newIcon.setVisible(i == 0);
				});
				newDaysText.setVisible(false);
				ComponentUtilities.whenSelectedEnable(markFilesNew, new JComponent[]{ newDaysCount, newDaysRef, newDaysMark, newDaysText, newIcon });
				
				add("", markFilesNew);
				add("", newDaysCount);
				add(new JLabel(getText("ui.daysOld")));
				add("br", new JLabel(getText("ui.reference")));
				add("tab", newDaysRef);
				add("br", new JLabel(getText("ui.marker")));
				add("tab", newDaysMark);
				add(newIcon);
				add(newDaysText);
			}
		};
		
		/*	---------------------------------------------------------------
									Control strip
			--------------------------------------------------------------- */
				
		ControlPanel topNavigation = new ControlPanel() {
			
			JCheckBox showUpButton = new JCheckBox(getText("ui.showUpButton"));
			
			ControlPanel folderTree = new ControlPanel(getText("ui.folderTree")) {
				
				JCheckBox topNavigationVisible = new JCheckBox(getText("ui.visibleByDefault"), false);
				JCheckBox topNavigationIncludeFolders = new JCheckBox(getText("ui.includeFolders"), true);
				JCheckBox topNavigationIncludePages = new JCheckBox(getText("ui.includePages"));
				JCheckBox topNavigationIncludeWebLocations = new JCheckBox(getText("ui.includeWebLocations"));
				JSpinner topNavigationDepth = new JSpinner(new SpinnerNumberModel(4, 1, 6, 1));
			
				{
					add("", topNavigationVisible);
					add("br", topNavigationIncludeFolders);
					add("br", topNavigationIncludePages);
					add("br", topNavigationIncludeWebLocations);
					add("br", new JLabel(getText("ui.depth")));
					add("", topNavigationDepth);
					add("", new JLabel(getText("ui.levels")));
				}
			};

			{
				add("br", showUpButton);
				add("br hfill", folderTree);
			}
		};

		ControlPanel thumbnailStrip = new ControlPanel() {

			{
				thumbsPreview.getView().setPreferredSize(new Dimension(480, uiHeight - 120));
				
				new StateMonitor() {
					@Override
					public void onChange() {
						setThumbDimensions();
					}
				}.add(thumbStyle);
				
				//add("br", new JLabelFor(getText("ui.position"), thumbPosition));
				//add("tab", thumbPosition);
				add("br", new JLabelFor(getText("ui.style"), thumbStyle));
				add("tab", thumbStyle);
				add("br", new JLabelFor(getText("ui.gap"), thumbGap));
				add("tab", thumbGap);
				add("br tab", thumbsPreview.getView());
			}
		};
		
		ControlPanel titlePanel = new ControlPanel() {
			
			JCheckBox showTitle = new JCheckBox(getText("ui.showTitle"));
			JCheckBox showDescription = new JCheckBox(getText("ui.showDescription"));
			JCheckBox showDateRange = new JCheckBox(getText("ui.showDateRange"));
			JCheckBox showCounts = new JCheckBox(getText("ui.showCounts"));
			JCheckBox showBreadcrumbPath = new JCheckBox(getText("ui.showBreadcrumbPath"));
			
			ControlPanel titleSubPanel = new ControlPanel() {

				{
					showBreadcrumbPath.setToolTipText(getText("ui.showBreadcrumbPathInfo"));
					
					add(showDescription);
					add("br", showDateRange);
					add("br", showCounts);
					add("br", showBreadcrumbPath);
				}
			};
			
			{
				ComponentUtilities.whenSelectedEnable(showTitle, titleSubPanel);
				
				add(showTitle);
				add("br hfill", titleSubPanel);
			}
		};
		
		//	Social
		
		ControlPanel social = new ControlPanel() {
			
			ControlPanel shares = new ControlPanel(getText("ui.shareButtonFor")) {

				JCheckBox shareFacebook = new JCheckBox("Facebook");
				JCheckBox shareTwitter = new JCheckBox("Twitter");
				JCheckBox sharePinterest = new JCheckBox("Pinterest");
				JCheckBox shareLinkedin = new JCheckBox("LinkedIn");
				JCheckBox shareDigg = new JCheckBox("Digg");
				JCheckBox shareReddit = new JCheckBox("Reddit");
				JCheckBox shareTumblr = new JCheckBox("Tumblr");

				{
					setLayout(new RiverLayout(4, 5));
					add(new JLabel(icon("facebook")));
					add(shareFacebook);
					add("br", new JLabel(icon("twitter")));
					add(shareTwitter);
					add("br", new JLabel(icon("tumblr")));
					add(shareTumblr);
					add("br", new JLabel(icon("pinterest")));
					add(sharePinterest);
					add("br", new JLabel(icon("linkedin")));
					add(shareLinkedin);
					add("br", new JLabel(icon("digg")));
					add(shareDigg);
					add("br", new JLabel(icon("reddit")));
					add(shareReddit);
				}
			};

			ControlPanel links = new ControlPanel() {
				
				JCheckBox shareEmail = new JCheckBox("Email");
				JLabel emailSubjectLabel = new JLabel(getText("ui.subject"));
				JTextField emailSubject = new JSmartTextField(18);
				JCheckBox shareLink = new JCheckBox(getText("ui.link"));
				
				{
					ComponentUtilities.whenSelectedEnable(shareEmail, new JComponent[]{emailSubjectLabel, emailSubject});
					
					add("", new JLabel(icon("email")));
					add("tab", shareEmail);
					add("br tab", emailSubjectLabel);
					add("", emailSubject);
					add("br", new JLabel(icon("link")));
					add("tab", shareLink);
				}
			};
			
			JLabel mandatoryInfo = new JLabel(mandatory);
			
			{
				mandatoryInfo.setToolTipText(getText("ui.mandatory"));
				links.setBorder(emptyBorder);
				
				add("", shares);
				add("hfill", links);
				add("br", new JLabelFor(getText("ui.facebookAppId"), facebookAppId));
				add(mandatoryInfo);
				add("tab", facebookAppId);
				add(new JLinkLabel("https://developers.facebook.com/apps", getText("ui.signUp")));
			}

		};
				
		// Map
		
		ControlPanel map = new ControlPanel() {

			JCheckBox showMap = new JCheckBox(getText("ui.showMap"), true);
			
			JComboBox mapType = new JComboBox(new Object[]{
				new Item("roadmap", getText("ui.roadmap")),
				new Item("satellite", getText("ui.satellite")),
				new Item("hybrid", getText("ui.hybrid")),
				new Item("terrain", getText("ui.terrain"))
			});
			JSlider mapZoom = new JSlider(JSlider.HORIZONTAL, 1, 20, 18);
			JTextField googleApiKey = new JSmartTextField(24);
			JLabel mandatoryInfo = new JLabel(mandatory);
			WrappableJLabel note = new WrappableJLabel("<html><i>" + getText("ui.gooleMapsApiNote") + "</i></html>");

			ControlPanel mapPanel = new ControlPanel() {

				{
					mapZoom.setOrientation(JSlider.HORIZONTAL);
					mapZoom.setMinimum(0);
					mapZoom.setMaximum(20);
					mapZoom.setValue(18);
					mapZoom.setMajorTickSpacing(10);
					mapZoom.setMinorTickSpacing(1);
					mapZoom.setPaintTicks(true);
					mapZoom.setPaintLabels(true);
					mapZoom.setSnapToTicks(true);
					mandatoryInfo.setToolTipText(getText("ui.mandatory"));
					note.setPreferredWidth(uiWidth - 120);

					add("br", new JLabelFor(getText("ui.initialView"), mapType));
					add("tab", mapType);
					add("br", new JLabelFor(getText("ui.initialZoom"), mapZoom));
					add("tab", mapZoom);
					add("br", new JLabelFor(getText("ui.googleApiKey"), googleApiKey));
					add(mandatoryInfo);
					add("tab", googleApiKey);
					add("br tab", new JLinkLabel("https://console.developers.google.com/apis/credentials", getText("ui.createNew")));
					add("br tab", note);
					add("br tab", new JLinkLabel("https://developers.google.com/maps/documentation/javascript/tutorial", getText("ui.readMore")));
				}
			};
				
			{
				ComponentUtilities.whenSelectedEnable(showMap, mapPanel);
				
				add(showMap);
				add("br hfill", mapPanel);

				//putClientProperty("helpPage", helpRoot + "ui/map.html");
			}
		};
		
		ControlPanel backgroundMusic = new ControlPanel() {


			JPlaylist backgroundAudio = new JPlaylist();
			JCheckBox backgroundAudioAutoPlay = new JCheckBox(getText("ui.autoStart"));
			JSpinner backgroundAudioVolume = new JSpinner(new SpinnerNumberModel(25, 1, 100, 1));
			JCheckBox backgroundAudioSlideshowControl = new JCheckBox(getText("ui.slideshowControl"));
			JCheckBox backgroundAudioLoop = new JCheckBox(getText("ui.loop"));
			JCheckBox backgroundAudioRetainPosition = new JCheckBox(getText("ui.retainPosition"), true);

			{
				backgroundAudioSlideshowControl.setToolTipText(getText("ui.slideshowControlInfo"));
				backgroundAudioRetainPosition.setToolTipText(getText("ui.retainPositionInfo"));
				add("br hfill", backgroundAudio);
				add("br", new JLabel(getText("ui.initialVolume")));
				add("", backgroundAudioVolume);
				add("", new JLabel("%"));
				add("br", backgroundAudioAutoPlay);
				add("tab", backgroundAudioSlideshowControl);
				add("br", backgroundAudioLoop);
				add("tab", backgroundAudioRetainPosition);

				//putClientProperty("helpPage", helpRoot + "ui/background-audio.html");
			}
		};

		/*	---------------------------------------------------------------
									Advanced
			--------------------------------------------------------------- */

		//	Custom code
		
		ControlPanel customCodePanel = new ControlPanel() {

			ControlPanel headHookTab = new ControlPanel() {

				JTextArea headHook = new JSmartTextArea(7, 20);
				JScrollPane headHookPane = new JScrollPane(headHook);

				{
					headHook.setEditable(true);
					headHook.setLineWrap(false);
					headHook.setFont(mono);
					headHook.setTabSize(2);
					headHookPane.setBorder(BorderFactory.createTitledBorder(getText("ui.headText")));

					add("hfill vfill", headHookPane);
				}
			};

			ControlPanel bodyHookTab = new ControlPanel() {

				JTextArea bodyHook = new JSmartTextArea(7, 20);
				JScrollPane bodyHookPane = new JScrollPane(bodyHook);

				{
					bodyHook.setEditable(true);
					bodyHook.setLineWrap(false);
					bodyHook.setFont(mono);
					bodyHook.setTabSize(2);
					bodyHookPane.setBorder(BorderFactory.createTitledBorder(getText("ui.bodyText")));

					add("hfill vfill", bodyHookPane);
				}
			};

			ControlPanel cssHookTab = new ControlPanel() {

				JTextArea cssHook = new JSmartTextArea(15, 20);
				JScrollPane cssHookPane = new JScrollPane(cssHook);
				WrappableJLabel info = new WrappableJLabel("<html><i>" + getText("ui.cssText") + "</i></html>");
				
				{
					cssHook.setEditable(true);
					cssHook.setLineWrap(false);
					cssHook.setFont(mono);
					cssHook.setTabSize(2);
					cssHookPane.setBorder(BorderFactory.createTitledBorder(getText("ui.cssTitle")));
					info.setPreferredWidth(700);
					
					add("hfill vfill", cssHookPane);
					add("br", info);
				}
			};

			ControlPanel jsHookTab = new ControlPanel() {

				JTextArea jsHook = new JSmartTextArea(15, 20);
				JScrollPane jsHookPane = new JScrollPane(jsHook);
				WrappableJLabel info = new WrappableJLabel("<html><i>" + getText("ui.javascriptText") + "</i></html>");
				
				{
					jsHook.setEditable(true);
					jsHook.setLineWrap(false);
					jsHook.setFont(mono);
					jsHook.setTabSize(2);
					jsHookPane.setBorder(BorderFactory.createTitledBorder(getText("ui.javascriptTitle")));
					info.setPreferredWidth(700);
					
					add("hfill vfill", jsHookPane);
					add("br", info);
				}
			};

			JTabbedPane customCodeTabs = new JTabbedPane() {				

				{
					addTab("<HEAD>", icon("code"), headHookTab);
					addTab("<BODY>", icon("code"), bodyHookTab);
					addTab("CSS", icon("css"), cssHookTab);
					addTab("JavaScript", icon("javascript"), jsHookTab);
				}
			};

			{				
				add("hfill vfill", customCodeTabs);

				//putClientProperty("helpPage", helpRoot + "ui/site.html");
			}
		};

		//	Site admin
		
		ControlPanel siteAdminPanel = new ControlPanel() {

			ControlPanel serverRelatedPanel = new ControlPanel(getText("ui.serverRelated")) {
				
				JTextField uploadPath = new JSmartTextField(16);
				JCheckBox useFavicon = new JCheckBox(getText("ui.useTheSkinsFavicon"), true);				
				JCheckBox useMsServer = new JCheckBox(getText("ui.useMsServer"));
				JCheckBox useExpiry = new JCheckBox(getText("ui.useExpiry"), false);
				JCheckBox useRobotsTxt = new JCheckBox(getText("ui.useRobotsTxt"), false);
				JCheckBox avoidCDNs = new JCheckBox(getText("ui.avoidCDNs"), true);
				JCheckBox copyGoogleFonts = new JCheckBox(getText("ui.copyGoogleFonts"), true);
				WrappableJLabel uploadPathInfo = new WrappableJLabel("<html><i>" + getText("ui.copyPasteAlbumURLHere") + "</i></html>");
				
				{
					uploadPath.setToolTipText(getText("ui.uploadPathInfo"));
					uploadPathInfo.setPreferredWidth(360);
					useFavicon.setToolTipText(getText("ui.useFaviconInfo"));
					useMsServer.setToolTipText(getText("ui.useMsServerInfo"));
					useExpiry.setToolTipText(getText("ui.useExpiryInfo"));
					useRobotsTxt.setToolTipText(getText("ui.useRobotsTxtInfo"));
					avoidCDNs.setToolTipText(getText("ui.avoidCDNsInfo"));
					copyGoogleFonts.setToolTipText(getText("ui.copyGoogleFontsInfo"));
					
					add(new JLabel(getText("ui.uploadPath")));
					add("tab hfill", uploadPath);
					add("br", new JLabel(infoIcon));
					add("", uploadPathInfo);
					add("br", useFavicon);
					add(" ", new JLabel(svgIcon("logo-lucid", new Dimension(24, 24))));
					add("br", useMsServer);
					add("br", useExpiry);
					add("br", useRobotsTxt);
					add("br", avoidCDNs);
					add("br", copyGoogleFonts);
				}
			};			
			
			ControlPanel googlePanel = new ControlPanel(getText("ui.googleAnalytics")) {

				JComboBox googleAnalytics = new JComboBox(new Object[] {
					new Item("none", getText("ui.none")), 
					new Item("gtag", "Global Site Tag (gtag.js)"),
					new Item("universal", "Universal (analytics.js)"),
					new Item("classic", "Classic (ga.js) [Legacy]")
				});
				JTextField googleSiteID = new JSmartTextField(12);
				JLabel mandatoryInfo = new JLabel(mandatory);
				JCheckBox supportDoubleclick = new JCheckBox(getText("ui.supportDoubleclick"), false);

				{
					googleSiteID.setToolTipText(getText("ui.googleSiteIDInfo"));
					mandatoryInfo.setToolTipText(getText("ui.mandatory"));

					add("", new JLabelFor(getText("ui.version"), googleAnalytics));
					add(" ", googleAnalytics);
					add(" ", new JLinkLabel("https://www.google.com/analytics/", getText("ui.signUp")));
					add("br", new JLabelFor(getText("ui.googleSiteID"), googleSiteID));
					add(mandatoryInfo);
					add("hfill", googleSiteID);
					add("br", supportDoubleclick);

					//putClientProperty("helpPage", helpRoot + "ui/advanced.html#google-analytics");
				}
			};
			
			ControlPanel seoPanel = new ControlPanel(getText("ui.searchEngineOptimization")) {
				
				JTextField titleSEOText = new JSmartTextField(12);
				JTextField descriptionSEOText = new JSmartTextField(12);
				JSpinner preloadThumbs = new JSpinner(new SpinnerNumberModel(20, 0, 1000, 10));
				JCheckBox addAltTags = new JCheckBox(getText("ui.addAltTags"), true);
				JCheckBox writeSitemapXml = new JCheckBox(getText("ui.createSitemapXml"));
				JCheckBox sitemapIncludeSlides = new JCheckBox(getText("ui.includeSlides"), true);
			
				{
					titleSEOText.setToolTipText(getText("ui.titleSEOTextInfo"));
					descriptionSEOText.setToolTipText(getText("ui.descriptionSEOTextInfo"));
					//preloadThumbs.setToolTipText(getText("ui.preloadThumbsInfo"));
					addAltTags.setToolTipText(getText("ui.addAltTagsInfo"));
					writeSitemapXml.setToolTipText(getText("ui.createSitemapXmlInfo"));
					ComponentUtilities.whenSelectedEnable(writeSitemapXml, new JComponent[]{sitemapIncludeSlides});
					sitemapIncludeSlides.setToolTipText(getText("ui.sitemapIncludeSlidesInfo"));
					
					add("", new JLabelFor(getText("ui.titleSEOText"), titleSEOText));
					add("tab hfill", titleSEOText);
					add("br", new JLabelFor(getText("ui.descriptionSEOText"), descriptionSEOText));
					add("tab hfill", descriptionSEOText);
					add("br", new JLabel(getText("ui.preloadThumbs")));
					add("", preloadThumbs);
					add("br", addAltTags);
					add("br", writeSitemapXml);
					add("", sitemapIncludeSlides);
				}
			};

			ControlPanel cookiePolicyPanel = new ControlPanel(getText("ui.trackingConsentAndCookiePolicy")) {

				JCheckBox askTrackingConsent = new JCheckBox(getText("ui.askTrackingConsent"), false);
				JCheckBox showCookiePolicy = new JCheckBox(getText("ui.showCookiePolicy"), false);				
				JTextField cookiePolicyStay = new JSmartTextField("8", 3);
				JTextField cookiePolicyUrl = new JSmartTextField(18);
				WrappableJLabel cookiePolicyUrlInfo = new WrappableJLabel("<html><i>" + getText("ui.cookiePolicyUrlInfo") + " </i></html>");

				{
					askTrackingConsent.setToolTipText(getText("ui.askTrackingConsentInfo"));
					showCookiePolicy.setToolTipText(getText("ui.showCookiePolicyInfo"));
					cookiePolicyUrlInfo.setPreferredWidth(320);
					
					add(askTrackingConsent);
					add("br", showCookiePolicy);
					add(" ", new JLabel(getText("ui.stay")));
					add("", cookiePolicyStay);
					add("", new JLabel("s"));
					add("br", new JLabel("URL"));
					add("tab hfill", cookiePolicyUrl);
					add("br", new JLabel(infoIcon));
					add("tab", cookiePolicyUrlInfo);
				}
			};

			JCheckBox debugMode = new JCheckBox(getText("ui.debugMode"));

			ControlPanel leftPanel = new ControlPanel() {
				
				JComboBox shareImageDims = new JComboBox(new Object[] {
					"640x480",
					"1024x768",
					"1200x900",
					"1600x1200"
				});
				
				{
					shareImageDims.setEditable(true);
					
					add("hfill", serverRelatedPanel);
					add("br hfill", googlePanel);					
					add("br", new JLabel(getText("ui.shareImageDimensions")));
					add("", shareImageDims);
				}
			};
			
			ControlPanel rightPanel = new ControlPanel() {
				
				{
					add("hfill", seoPanel);
					add("br hfill", cookiePolicyPanel);
					add("br", debugMode);					
				}
			};
			
			{
				add(leftPanel);
				add(rightPanel);
				
				//putClientProperty("helpPage", helpRoot + "Site/Site_admin");
			}
		};
		
		/*	---------------------------------------------------------------
									Lightbox
			--------------------------------------------------------------- */

		ControlPanel mainImage = new ControlPanel() {

			JSpinner hideControls = new JSpinner(new SpinnerNumberModel(3, 0, 999, 1));
			private String[] hideControlsLabel = getText("ui.hideControlsAfter").split("\\{0\\}");
			JSpinner transitionSpeed = new JSpinner(new SpinnerNumberModel(1000, 50, 10000, 50));
			JCheckBox slideshowAuto = new JCheckBox(getText("ui.autoStart"));
			JSpinner slideshowDelay = new JSpinner(new SpinnerNumberModel(3, 0, 20, 0.1));
			JComboBox afterLast = new JComboBox(new Object[] {
				new Item("donothing", getText("ui.doNothing")), 
				new Item("startover", getText("startOver")),
				new Item("onelevelup", getText("upOneLevel")),
				new Item("nextfolder", getText("nextFoldersFirstImage")),
				new Item("nextindex", getText("nextIndex")),
				new Item("ask", getText("ui.ask"))
			});
			
			JCheckBox neverScaleUp = new JCheckBox(getText("ui.neverScaleUp"));
			JCheckBox videoAutoPlay = new JCheckBox(getText("ui.startVideo"));
			JLabel videoAutoPlayInfo = new JLabel(infoIcon);
			JCheckBox enableMouseWheel = new JCheckBox(getText("ui.enableMouseWheel"));
			JCheckBox enableKeyboard = new JCheckBox(getText("ui.enableKeyboard"));
			JCheckBox clickForNext = new JCheckBox(getText("ui.clickForNext"));
			JCheckBox rightClickProtect = new JCheckBox(getText("ui.rightClickProtect"));
						
			{				
				showImageNumbers.setToolTipText(getText("ui.showNumbersInfo"));
				afterLast.setToolTipText(getText("ui.afterLastInfo"));
				transitionSpeed.setToolTipText(getText("ui.transitionSpeedInfo"));
				slideshowDelay.setToolTipText(getText("ui.slideshowDelayInfo"));
				neverScaleUp.setToolTipText(getText("ui.neverScaleUpInfo"));
				videoAutoPlayInfo.addMouseListener(new MouseAdapter() {  
						@Override
						public void mouseReleased(MouseEvent e) {
							JOptionPane.showMessageDialog(window, getText("ui.videoAutoplayInfo1"), "Warning", JOptionPane.WARNING_MESSAGE);
					}});
				enableMouseWheel.setToolTipText(getText("ui.enableMouseWheelInfo"));
				enableKeyboard.setToolTipText(getText("ui.enableKeyboardInfo"));
				clickForNext.setToolTipText(getText("ui.clickForNextInfo"));

				add("", new JLabel(hideControlsLabel[0]));
				add("tab", hideControls);
				if (hideControlsLabel.length > 1) {
					add(new JLabel(hideControlsLabel[1]));
				}
				add("br tab", new JLabel("<html><i>" + getText("ui.hideControlsInfo") + "</i></html>"));
				add("br", showImageNumbers);
				add("br", showFullscreen);
				add(new JLabel(icon("fullscreen")));
				add("br", slideshowAuto);
				add("br", new JLabelFor(getText("ui.transitionSpeed"), transitionSpeed));
				add("tab", transitionSpeed);
				add(new JLabel("ms"));
				add("br", new JLabel(getText("ui.slideshowDelay")));
				add("tab", slideshowDelay);
				add(new JLabel("s"));
				add("br", new JLabelFor(getText("ui.afterLast"), afterLast));
				add("tab", afterLast);

				add("br", new JLabelFor(getText("ui.neighboringItemsVisibility"), neighboringItemsOpacity));
				add("tab", neighboringItemsOpacity);
				add("br", new JLabelFor(getText("ui.fillHorizontally"), fillHorizontally));
				add("tab", fillHorizontally);
				add("br", new JLabelFor(getText("ui.fillVertically"), fillVertically));
				add("tab", fillVertically);
				add("br", neverScaleUp);
				
				add("br", videoAutoPlay);
				add("", videoAutoPlayInfo);
				add("br", enableMouseWheel);
				add("br", enableKeyboard);
				add("br", clickForNext);
				add("br", rightClickProtect);
			}
		};

		ControlPanel imageCaption = new ControlPanel() {
						
			private JComboBox<NameValue<String>> presets = new JComboBox<>(new NameValue[] {
				new NameValue("[ " + getText("ui.choosePreset") + " ]", null),
				new NameValue(getText("ui.empty"), ""),
				new NameValue(getText("ui.title"), "<div class=\"title\">${fileTitle}</div>"),
				new NameValue(getText("ui.title") + " + " + getText("ui.comment"), "<div class=\"title\">${fileTitle}</div>\n<div class=\"comment\">${comment}</div>"),
				new NameValue(getText("ui.title") + " + " + getText("ui.commentShort"), "<div class=\"title\">${fileTitle}</div>\n<div class=\"comment\">${commentShort}</div>"),
				new NameValue(getText("ui.fileLabel"), "<div class=\"title\">${fileLabel}</div>"),
				new NameValue(getText("ui.fileLabel") + " + " + getText("ui.comment") + " (" + getText("ui.default") + ")", "<div class=\"title\">${fileLabel}</div>\n<div class=\"comment\">${comment}</div>"),
				new NameValue(getText("ui.fileLabel") + " + " + getText("ui.commentShort"), "<div class=\"title\">${fileLabel}</div>\n<div class=\"comment\">${commentShort}</div>"),
				new NameValue(getText("ui.title") + " + " + getText("ui.comment") + " + " + getText("author") + " + " + getText("keywords"), "<div class=\"title\">${fileLabel}</div>\n<div class=\"comment\">${comment}</div>\n<div class=\"comment\"><small class=\"icon-user\"> ${author}</small> <small class=\"icon-label\"> ${keywords}</small></div>")
			});
			JScrollPane imageCaptionPane = new JScrollPane(imageCaptionTemplate);
						
			{
				presets.addItemListener(listener -> {
					if (presets.getSelectedIndex() > 0) {
						imageCaptionTemplate.setText(((NameValue<String>)presets.getSelectedItem()).value);
					}
				});	
				imageCaptionTemplate.setEditable(true);
				imageCaptionTemplate.setLineWrap(true);
				imageCaptionTemplate.setWrapStyleWord(true);
				imageCaptionTemplate.setFont(mono);
				imageCaptionTemplate.setTabSize(2);
				imageCaptionPane.setBorder(BorderFactory.createTitledBorder(getText("ui.captionTemplate")));					
				
				add("", new JLabelFor(getText("ui.position"), captionPlacement));
				add("tab", captionPlacement);
				add("br", new JLabelFor(getText("ui.style"), captionStyle));
				add("tab", captionStyle);
				add("br", new JLabelFor(getText("ui.presets"), presets));
				add("tab", presets);
				add("br hfill", imageCaptionPane);
			}
		};
			
		ControlPanel buttons = new ControlPanel() {
			
			ControlPanel downloadOptions = new ControlPanel() {
				
				JCheckBox downloadNonImages = new JCheckBox(getText("ui.enableDownloadNonImages"));
				JCheckBox downloadScaled = new JCheckBox(getText("ui.enableDownloadScaledImages"));
				
				{
					downloadNonImages.setToolTipText(getText("ui.enableDownloadNonImagesInfo"));
					add(downloadScaled);
					add("br", downloadNonImages);
				}
			};
			
			JCheckBox useFotomoto = new JCheckBox(getText("ui.useFotomoto"));
			
			ControlPanel photodataOptions = new ControlPanel() {
				
				JCheckBox showPhotoDataLabel = new JCheckBox(getText("ui.showLabel"), true);
				JTextArea photoDataTemplate = new JSmartTextArea(12, 16);
				JScrollPane photoDataPane = new JScrollPane(photoDataTemplate);
				JButton reset = new JButton(getText("ui.resetToDefaults"));
				WrappableJLabel listMetadataHint = new WrappableJLabel("<html><i>" + getText("ui.listMetadataHint") + "</i></html>");
				
				{
					showPhotoDataLabel.setToolTipText(getText("ui.showLabelInfo"));
					photoDataTemplate.setEditable(true);
					photoDataTemplate.setLineWrap(false);
					//photoDataTemplate.setWrapStyleWord(true);
					photoDataTemplate.setFont(mono);
					reset.addMouseListener(new MouseAdapter() {  
						@Override
						public void mouseReleased(MouseEvent e) {
							if (reset.isEnabled()) {
								String s = new SkinModel().photoDataTemplate;
								if (s != null && s.length() > 0) {
									photoDataTemplate.setText(s);
								}
							}
					}});
					photoDataPane.setPreferredSize(new Dimension(380, 240));
					listMetadataHint.setPreferredWidth(380);
					
					add(" ", showPhotoDataLabel);
					add(" ", reset);
					add("br", photoDataPane);
					add("br", listMetadataHint);
				}
			};
			
			{
				printImageButton.setToolTipText(getText("ui.printImageButtonInfo"));
				showDownload.setToolTipText(getText("ui.showDownloadInfo"));
				ComponentUtilities.whenSelectedEnable(showDownload, new JComponent[]{downloadOptions});
				showPhotoData.setToolTipText(getText("ui.showPhotoDataInfo"));
				ComponentUtilities.whenSelectedEnable(showPhotoData, new JComponent[]{photodataOptions});

				add("", new JLabel(icon("printer")));
				add("tab", printImageButton);
				add("br", new JLabel(icon("download")));
				add("tab", showDownload);
				add("br tab hfill", downloadOptions);
				add("br", new JLabel(icon("camera")));
				add("tab", showPhotoData);
				add("br tab hfill vfill", photodataOptions);
			}
		};
				
		/******************************************************************************
		 * 
		 *								GUI main tabs
		 * 
		 ******************************************************************************/				
		
		/*	---------------------------------------------------------------
									Site
			--------------------------------------------------------------- */
		
		ControlPanel site = new ControlPanel(new BorderLayout(0, 0)) {
			
			JTabbedPane siteTabs = new JTabbedPane() {

				JScrollPane mainImagePane = new JScrollPane(mainImage, JScrollPane.VERTICAL_SCROLLBAR_AS_NEEDED, JScrollPane.HORIZONTAL_SCROLLBAR_NEVER);
				{
					/*
					design.setBorder(emptyBorder);
					typography.setBorder(emptyBorder);
					markNew.setBorder(emptyBorder);
					*/
					mainImagePane.setBorder(emptyBorder);
					/*
					imageCaption.setBorder(emptyBorder);
					buttons.setBorder(emptyBorder);
					*/
					this.setTabPlacement(SwingConstants.LEFT);
															
					addTab("<html><h4 " + tabStyle + ">" + getText("ui.design") + "</h4></html>", icon("design"), design);
					addTab("<html><h4 " + tabStyle + ">" + getText("ui.typography") + "</h4></html>", icon("font"), typography);
					addTab("<html><h4 " + tabStyle + ">" + getText("ui.titlePage") + "</h4></html>", icon("title"), titlePage);
					addTab("<html><h4 " + tabStyle + ">" + getText("ui.lightbox") + "</h4></html>", icon("lightbox"), mainImagePane);
					addTab("<html><h4 " + tabStyle + ">" + getText("ui.caption") + "</h4></html>", icon("comment"), imageCaption);					
					addTab("<html><h4 " + tabStyle + ">" + getText("ui.buttons") + "</h4></html>", icon("button"), buttons);					
					addTab("<html><h4 " + tabStyle + ">" + getText("ui.newImages") + "</h4></html>", icon("new"), markNew);

				}
			};
			
			ControlPanel designPreviewPanel = new ControlPanel(new BorderLayout(0, 0)) {

				JPanel desPreview = new JPanel(new BorderLayout(0, 0)) {
					
					{
						add(designPreview.getView());
					}
				};
				
				{
					desPreview.setPreferredSize(new Dimension(previewWidth - 4, previewWidth - 80));
					desPreview.setBorder(BorderFactory.createLineBorder(JAlbumColor.imageBorder, 2));
					
					add(desPreview, BorderLayout.NORTH);
				}
			};
			
			{				
				((BorderLayout)(getLayout())).setVgap(0);
				((BorderLayout)(getLayout())).setHgap(0);
				
				siteTabs.setBorder(emptyBorder);
				
				//designPreviewPanel.setPreferredSize(new Dimension(previewWidth, uiHeight));
							
				add(siteTabs);
				add(designPreviewPanel, BorderLayout.EAST);
				
				//putClientProperty("helpPage", helpRoot + "ui/site.html");
			}
		};
		
		/*	---------------------------------------------------------------
									Controls
			--------------------------------------------------------------- */

		ControlPanel controls = new ControlPanel(new BorderLayout(20, 0)) {
			
			ControlPanel ordering = new ControlPanel() {

				JList controlsOrder = new JDraggableList(new Object[] {
					new JLabelFor("<html><p style=\"padding:3px 0;\">" + getText("ui.navigation") + "</p></html>", icon("menu")).name("navigation"),
					new JLabelFor("<html><p style=\"padding:3px 0;\">" + getText("ui.thumbnails") + "</p></html>", icon("thumbnails")).name("thumbnails"),
					new JLabelFor("<html><p style=\"padding:3px 0;\">" + getText("ui.title") + "</p></html>", icon("type")).name("title"),
					new JLabelFor("<html><p style=\"padding:3px 0;\">" + getText("ui.map") + "</p></html>", icon("location")).name("map"),
					new JLabelFor("<html><p style=\"padding:3px 0;\">" + getText("ui.social") + "</p></html>", icon("connect")).name("social"),
					new JLabelFor("<html><p style=\"padding:3px 0;\">" + getText("ui.backgroundMusic") + "</p></html>", icon("audio")).name("music")
				});

				{						
					add("center", new JLabel("<html><h4>" + getText("ui.ordering")+ "</h4></html>"));
					add("br", controlsOrder);
					add("br center", new JLabel("<html><i>" + getText("ui.dragToReorder") + "</i></html>"));
				}
			};

			JTabbedPane controlTabs = new JTabbedPane() {

				{
					this.setTabPlacement(SwingConstants.LEFT);
					
					addTab("<html><h4 " + tabStyle + ">" + getText("ui.navigation") + "</h4></html>", icon("folders"), topNavigation);
					addTab("<html><h4 " + tabStyle + ">" + getText("ui.thumbnails") + "</h4></html>", icon("thumbnails"), thumbnailStrip);
					addTab("<html><h4 " + tabStyle + ">" + getText("ui.title") + "</h4></html>", icon("type"), titlePanel);
					addTab("<html><h4 " + tabStyle + ">" + getText("ui.map") + "</h4></html>", icon("location"), map);
					addTab("<html><h4 " + tabStyle + ">" + getText("ui.social") + "</h4></html>", icon("connect"), social);
					addTab("<html><h4 " + tabStyle + ">" + getText("ui.backgroundMusic") + "</h4></html>", icon("audio"), backgroundMusic);

				}
			};
				
			{
				((BorderLayout)(getLayout())).setVgap(0);
				((BorderLayout)(getLayout())).setHgap(0);
				controlTabs.setBorder(emptyBorder);
				
				add(controlTabs);
				add(ordering, BorderLayout.EAST);
				
				//putClientProperty("helpPage", helpRoot + "ui/sections.html");
			}
		};
				
		/*	---------------------------------------------------------------
									Advanced
			--------------------------------------------------------------- */
		
		ControlPanel advanced = new ControlPanel(new BorderLayout(0, 0)) {
			
			JTabbedPane advancedTabs = new JTabbedPane() {
								
				JScrollPane siteAdminPane = new JScrollPane(siteAdminPanel, JScrollPane.VERTICAL_SCROLLBAR_AS_NEEDED, JScrollPane.HORIZONTAL_SCROLLBAR_NEVER);
				
				{
					customCodePanel.setBorder(emptyBorder);
					siteAdminPane.setBorder(emptyBorder);
					this.setTabPlacement(SwingConstants.LEFT);
					
					addTab("<html><h4 " + tabStyle + ">" + getText("ui.siteAdmin") + "</h4></html>", icon("wrench"), siteAdminPane);
					addTab("<html><h4 " + tabStyle + ">" + getText("ui.customCode") + "</h4></html>", icon("code"), customCodePanel);
				}
			};
			
			{
				((BorderLayout)(getLayout())).setVgap(0);
				((BorderLayout)(getLayout())).setHgap(0);
				
				add(advancedTabs);
				//putClientProperty("helpPage", helpRoot + "ui/lightbox.html");
			}
		};
		
		/*	---------------------------------------------------------------
									About
			--------------------------------------------------------------- */

		ControlPanel about = new ControlPanel(new BorderLayout(20, 0)) {
			
			private JTextArea readme = new JSmartTextArea( getFileContents("readme.txt"), 20, 30 );
			private JScrollPane aboutPane = new JScrollPane(readme, JScrollPane.VERTICAL_SCROLLBAR_ALWAYS, JScrollPane.HORIZONTAL_SCROLLBAR_NEVER);
			
			ControlPanel info = new ControlPanel() {
				
				{
					add("center", new JLabel(svgIcon("logo-lucid", new Dimension(120, 120))));
					add("br center", new JLabel("<html><h2>" + skin + "</h2></html>"));
					add("br center", new JLabel("Jalbum " + internalVersion));
					add(new JLinkLabel("https://jalbum.net/software/download", getText("ui.upgrade"), getText("ui.downloadJalbum")));
					add("br center", new JLabel(skin + " skin " + skinVer));
					add(new JLinkLabel("https://jalbum.net/skins/skin/" + skin, getText("ui.upgrade"), getText("ui.downloadSkin")));
					//add("br center", new JLinkLabel(helpRoot + "index.html", getText("help")));
					add("br center", new JLinkLabel(supportForum, getText("ui.support")));
				}
			};
			
			{								
				((BorderLayout)(getLayout())).setVgap(0);
				((BorderLayout)(getLayout())).setHgap(0);
				
				readme.setLineWrap(true);
				readme.setWrapStyleWord(true);
				readme.setEditable(false);
				readme.setFont(mono);
				
				info.setPreferredSize(new Dimension(240, uiHeight));
				//aboutPane.setPreferredSize(new Dimension(uiWidth + previewWidth - 240, uiHeight));
				
				add(info, BorderLayout.WEST);
				add(aboutPane);
				
				//putClientProperty("helpPage", helpRoot + "index.html");
			}
		};
		
		JTabbedPane tabs = new JTabbedPane() {
			
			{	
				site.setBorder(emptyBorder);
				controls.setBorder(emptyBorder);
				advanced.setBorder(emptyBorder);
				about.setBorder(emptyBorder);
				
				addTab("<html><h4 style='padding:4px 6px;margin:4px;'>" + getText("ui.site") + "</h4></html>", icon("site", 27), site);
				addTab("<html><h4 style='padding:4px 6px;margin:4px;'>" + getText("ui.controls") + "</h4></html>", icon("header", 27), controls);
				addTab("<html><h4 style='padding:4px 6px;margin:4px;'>" + getText("ui.advanced") + "</h4></html>", icon("wrench", 27), advanced);
				addTab("<html><h4 style='padding:4px 6px;margin:4px;'>" + getText("ui.about") + "</h4></html>", icon("info", 27), about);
			}
		};
		
		private final String designTemplate = getFileContents("lib/design-template.html");
		private final String thumbsTemplate = getFileContents("lib/thumbs-template.html");
	
		{
			// Adding UI tabs
			
			((RiverLayout)(getLayout())).setVgap(0);
			((RiverLayout)(getLayout())).setHgap(0);
			tabs.setBorder(emptyBorder);
			
			add("vfill", tabs);
			
			/*	-----------------------------------------------------------
										Listeners
				----------------------------------------------------------- */
			
			window.addJAlbumListener(new JAlbumAdapter() {
				@Override
				public void styleChanged(JAlbumEvent e) {
					styleName.setText(getStyleName());
				}
			});
			
			window.addPropertyChangeListener(JAlbumFrame.SKIN_LOADED_PROPERTY, setupMonitors);
			
			window.addJAlbumListener(new JAlbumAdapter() {
				@Override
				public void skinChanged(JAlbumEvent e) {
					//JAlbum.logger.log(Level.FINE, "Skin changed, removing listeners.");
					window.removePropertyChangeListener(JAlbumFrame.SKIN_LOADED_PROPERTY, setupMonitors);
				}
			});
				
		}
		
	};
		
	public Gui() {
		this(JAlbumContext.getInstance());
	}
		
	public Gui(JAlbumContext context) {
		
		super(context);

		skinUi.setBorder(emptyBorder);
		((RiverLayout)(getLayout())).setVgap(0);
		((RiverLayout)(getLayout())).setHgap(0);

		window.pack();
								
		window.setSkinUI(skinUi);
		skinReady = true;
	}
	
}