/* 
 *	main.js - the skin's javascript functions
 */
	
;(function($, $window, $document, $body, undefined) {
	'use strict';

	// The main skin code
	
	// Random number between -base and +base 
	var	random = function(base) {
				return base * (1 - 2 * Math.random());
			};
	
	$.fn.skin = function(settings) {
		
			settings 				= $.extend({}, $.fn.skin.defaults, settings);
			
			var main 				= $(this).eq(0),								// Main container
				ns 					= 'lucid_skin',									// Namespace
				backgroundAudioRoot = $('[data-audioplayer]'),
				backgroundAudioOn 	= backgroundAudioRoot.length && !backgroundAudioRoot.data(settings.pausedClass),
				
				text 				= getTranslations({								// Translated texts
											newItem:						'NEW',
											atFirstPage:					'At first page',
											atLastPage:						'At last page',
											atLastPageQuestion:				'Where to go next?',
											startOver:						'Start over',
											backToHome:						'Back to home',
											upOneLevel:						'Up one level',
											previousFolder:					'Previous folder',
											nextFolder:						'Next folder',
											download: 						'Download',
											print:							'Print',
											metaBtn: 						'Photo data',
											clickToOpen:					'Click to open this document with the associated viewer!'
										}),
				// Main sections
				lightbox			= $('#lightbox'),								// Lightbox element
				cards,																// Card container in Lightbox
				thumbnails			= $('#thumbnails'),								// Thumbnails section
				thumbCont			= $('#thumbnails .thumb-cont'),					// Thumbnail container
				thumbs				= thumbCont.find('.thumb'), 					// Collection of thumbs (<a>)
				controls			= $('#controls'),								// Controls overlay
				panels				= $('#panels'),									// Panels
				navigation			= $('#navigation'),								// Folders, up link, etc.
				numbers				= $('#numbers'),								// Numbers
				
				// Buttons
				prevBtn,															// Previous button
				nextBtn,															// Next button
				playBtn,															// Play button
				pauseBtn,															// Pause button
				fullscrBtn,															// Full screen button
				windowedBtn,														// Exit full screen
				
				// State
				curr				= 0,											// Current image
				nextCard			= 0,											// Next card to show
				dir					= 1,											// Direction
				loop				= false,										// Looped lightbox
				swiped				= false,										// Currently swiped element
				transition,															// Transition parameters
				lastChangeDate  	= new Date() - 10000,							// Last image change date for adaptive speed
				lastHash 			= '',											// Storing last state
				initByHash 			= false,										// Initial URL contains hash
				nows 				= new Date() / 1000,							// Now in seconds
				
				// Timeouts
				cleanupTo			= null,											// Clean up cards timeout
				focusTo				= null,											// Focus later timeout
				layoutTo			= null,											// Layout refresh on resize
				scrollTo			= null,											// Thumbnails scrolled timeout
				controlsTo			= null,											// Fade controls timeout
				slideshow			= null,											// slideshow timeout
				suspended			= false,										// suspended slideshow 
				
				mousemoveListener	= false,										// Mouse move listener is engaged
				inited				= false,
				
	
				/********************************************************
				 *
				 *						Utility functions
				 *
				 ********************************************************/
				 
				// Get image number by filename
				
				getImageByName = function(n) {
					
						if (n) {
							for (var i = 0; i < thumbs.length; i++) {
								if (thumbs.eq(i).attr('href').getFile() === n) {
									return i;
								}
							}
						}
						
						return -1;
					},
					
				// Update URL
				
				updateUrl = function(n) {
						var t = thumbs.eq(n),
							f = t.attr('href').getFile();
							
						addParam(settings.indexName, {
								img:	f
							}, t.attr('title') || f.stripExt());
					},
					
				// Hash/history changed
				
				stateChange = function() {
						if (window.location.hash === lastHash) {
							return;
						}
						
						lastHash = window.location.hash;
						var param = readParam();
						
						if (param.hasOwnProperty('img')) {
							var n = getImageByName(param.img);
							
							if (n >= 0) {
								focusCard(n);
							} else {
								removeParam(settings.indexName, 'img');
							}
						
							updateShares();
						}
					},
				
				// Type check
				
				typeCheck = function(query, n) {
						var t = thumbs.eq((typeof n === UNDEF)? curr : n).data('vars'),
							q = query.split(',');
						
						if (t && (t = t['type'])) {
							for (var i = 0; i < q.length; i++) {
								if (q[i] === t) {
									return true;
								}
							}
						}
						
						return false;
					},
												
				/********************************************************
				 *
				 *						Controls
				 *
				 ********************************************************/
				 
				// Initializing background music
				
				initBackgroundMusic = function(player) {
					
						if (player.length) {
							$.fn.audioPlayer.defaults.rootPath = settings.rootPath;
							player.audioPlayer();
						}
					},
					
				// Updating the share buttons after a state change
				
				updateShares = function() {
					
						if (settings.hasOwnProperty('share')) {
							
							if (lightbox.is(':visible')) {
								// Lightbox: sharing actual image
								var th = thumbs.eq(curr),
									vars = th.data('vars');
								
								if (vars) {
									$(settings.share.hook).trigger('updateLinks', {
											sites: 			settings.share,
											title: 			th.attr('title') || th.attr('href').stripExt().replace(/[-_]/g, ' ').capitalize(),
											description: 	vars.caption || '',
											image: 			th.attr('href'),
											href: 			th.attr('href').replaceExt('html')
										});
								}
							} else {
								// Index page: reset to page sharing
								$(settings.share.hook).trigger('updateLinks');
							}
						}	
					},
										
				// Initializing Social sharing
				
				initShares = function(opt) {
						
						// Social sharing
						
						$.fn.renderShares.defaults.buttonTheme = opt['buttonTheme'] || 'dark';
						$.fn.renderShares.defaults.indexName = settings.indexName;
						$.fn.renderShares.defaults.facebookAppId = opt['facebookAppId'] || '';
						
						if (opt.hasOwnProperty('callAction')) {
							$.fn.renderShares.defaults.callAction = opt.callAction;
						}
						
						$(opt['hook']).renderShares(opt);
						
					},

				// Updating current number
				
				updateNumber = function(n) {
						
						if (numbers.length) {
							numbers.html('<big>' + n + '</big><small>' + thumbs.length + '</small>');
						}
					},
					
				// Show controls
				
				showControls = function() {
					
						controlsTo = clearTimeout(controlsTo);
						main.removeClass(settings.controlsHideClass);
					},
					
				// Hide controls
				
				hideControls = function() {
					
						if (settings.hideControls) {
							controlsTo = clearTimeout(controlsTo);
							if (!panels.children('.on').length) {
								// No panels open
								main.addClass(settings.controlsHideClass);
							}
						}
					},
					
				// Toggle controls
				
				toggleControls = function() {
					
						if (settings.hideControls && !panels.children('.on').length) {
							if (main.hasClass(settings.controlsHideClass)) {
								showControls();
							} else {
								hideControls();
							}
						}
					},
					
				// Listening to mouse move to get back control bar
					
				setupMousemoveListener = function() {
						
						if (whatInput.ask('intent') === 'mouse') {
							
							$document.on('mousemove.' + ns, function() {
									if (main.hasClass(settings.controlsHideClass)) {
										showControls();
									}
									
									hideControlsLater();
								});
							
							mousemoveListener = true;
						}
					},
				
				// Hide controls later
				
				hideControlsLater = function() {
						
						if (whatInput.ask('intent') === 'mouse') {
							
							if (!mousemoveListener) {
								setupMousemoveListener();
							}
							
							clearTimeout(controlsTo);
							controlsTo = setTimeout(hideControls, slideshow? Math.min(settings.slideshowDelay / 2, settings.hideControls * 1000) : settings.hideControls * 1000);
						}
					},
					
				// Show panel
				
				showPanel = function(p, doneFn) {
						
						stopAuto();
						
						p.each(function() {
								var p = $(this),
									id = this.id,
									btn = controls.find('[data-rel=' + id + ']:visible');
								
								// Related button
								btn.addClass('on');
								
								// Preset css
								p.addClass('on').css({
										left:			Math.min(btn.position().left, panels.outerWidth() - p.outerWidth()),
										opacity:		0,
										transform:		'translateY(-16px)',
										transition:		'none',
										display: 		'flex'
									});
								
								window.requestAnimationFrame(function() {
										
										p.one('transitionend', function(e) {
											
												$(this).css('display', 'flex');
												
												if (id === 'thumbnails') {
													lazyloadThumbs();
												}
												
												if (typeof doneFn === FUNCTION) {
													doneFn.call(null);
												}
											})
										.css({
												opacity:		1,
												transform:		'translateY(0)',
												transition:		'500ms transform ease-out'
											});
									});
							});
					},
						
				// Hide panel
				
				hidePanel = function(p, doneFn) {
					
						p.each(function() {
								var id = this.id,
									btn = $('#controls [data-rel=' + id + ']');
								
								// Related button
								btn.removeClass('on');
								
								p.removeClass('on')
									.one('transitionend', function(e) {
											
											$(this).css('display', 'none');
											
											if (typeof doneFn === FUNCTION) {
												doneFn.call(null);
											}
										})
									.css({
											opacity:		0,
											transform:		'translateY(-16px)'
										});
							});
					},
					
				hideAllPanels = function() {
					
						hidePanel(panels.children('.on'));
						
					},
					
				// Initializing controls
				
				initControls = function() {
					
						// Fullscreen button
						
						if (settings.showFullscreen) {
								
							fullscrBtn = controls.find('.fullscreen.btn');
							windowedBtn = controls.find('.windowed.btn');
							
							if (hasFullscreen()) {
								
								fullscrBtn.on('click.' + ns, function() {
										requestFullscreen(function() {
												main.addClass(settings.fullscreenClass);
											})
									});
								
								windowedBtn.on('click.' + ns, function() {
										exitFullscreen(function() {
												main.removeClass(settings.fullscreenClass);
											})
									});
								
							} else {
								
								// No API
								fullscrBtn.hide();
								windowedBtn.hide();
							
							}
								
						}
							
						// Toggle buttons
						
						controls
							.find('[data-rel]')
							.each(function() {
									var btn = $(this),
										p = $('#' + btn.data('rel'));
										
									if (p.length) {
										
										if (btn.hasClass('on')) {
											// Visible by default
											showPanel(p);
										}
										
										btn
											.addClass('has-panel')
											.on('selectstart.' + ns, function(e) {
													e.preventDefault();
													return false;
												})
											.on('click.' + ns, function() {
												var btn = $(this),
													ac = btn.data('autoclose'),
													p = $('#' + btn.data('rel'));
													
												if (p.length) {
													
													if (!p.is(':visible')) {
														// Turn ON
														
														// Hide all others
														hidePanel(panels.children(':visible'));
														
														// Show related
														showPanel(p);
														
														// Auto close
														if (ac) {
															var acTo,
																over;
															
															// Do not autoclose while mouse is around
															p.add(btn)
																.on('mouseenter.' + ns, function() {
																		over = true;
																		acTo = clearTimeout(acTo);
																	})
																.on('mouseover.' + ns, function() {
																		over = true;
																	})
																.on('mouseleave.' + ns, function() {
																		clearTimeout(acTo);
																		over = false;
																		acTo = setTimeout(function() {
																				if (!over) {
																					hidePanel(p);
																				}
																			}, ac);
																	});
														}
														
													} else {
														// Turn OFF
														hidePanel(p);
													}
												}
											});

									} else {
										// No related panel
										btn.removeAttr('data-rel');
									}
								});
							
					},
										
				/********************************************************
				 *
				 *						Thumbnails
				 *
				 ********************************************************/
				 
				// Focus thumbnail
				
				focusThumb = function(n) {
					
						thumbs.removeClass(settings.activeClass).eq((typeof n !== UNDEF)? n : curr).addClass(settings.activeClass);
					},
												
				// Lazy loading thumbnails having "lazyload" class
				
				lazyloadThumbs = function(target, doneFn, layoutModFn) {
						var vert = main.hasClass('thumbs-left') || main.hasClass('thumbs-right'),
							pd = (vert? thumbCont.height() : thumbCont.width()),
							scroll,
							t,
							img,
							src,
							imgs = $();
						
						// Set target position if passed 
						if (typeof target === UNDEF || typeof target === FUNCTION) {
							scroll = vert? thumbCont.scrollTop() : thumbCont.scrollLeft();
							if (typeof target === FUNCTION) {
								if (typeof doneFn === FUNCTION) {
									layoutModFn = doneFn;
								}
								doneFn = target;
							}
						} else {
							scroll = target;
						}
						 
						thumbCont
							.find('a.' + settings.lazyloadClass)
							.filter(function() {
									// Visible or near visible items
									var p = $(this).position()[vert? 'top' : 'left'] + scroll;
									
									// Half screen width backwards, full screen width forwards
									return (p + pd / 2) >= 0 && (p - pd * 2) <= 0;
								})
							.each(function() {
									t = $(this);
									
									if (!t.children('img').length) {
										var vars = t.data('vars'),
											tw = vars['thumbWidth'],
											th = vars['thumbHeight'],
											ar = tw / th;
										
										if (src = vars['thumb']) {
											img = $('<img>', {
													'class': 		settings.hideImageClass + ' ' + ((ar >= 1.25)? 'landscape' : ((ar <= 0.8)? 'portrait' : 'square'))
												})
												// Onload action
												.one('load', function() {
														$(this).addClass(settings.showImageClass).removeClass(settings.hideImageClass);
														$(this).parent().removeClass(settings.lazyloadClass);
														if (typeof layoutModFn === FUNCTION) {
															layoutModFn.call();
														}
													})
												.attr({
														width:		tw,
														height:		th,
														alt:		vars['title'] || src.stripExt(),
														src:		getThumbSource(t, [ t.outerWidth(), t.outerHeight() ])
													})
												.appendTo(t);
												
											imgs.append(img);
										}
									}
								});
						
						if (typeof doneFn === FUNCTION) {
							if (imgs.length) {
								imgs.waitAllImg(doneFn);
							} else {
								doneFn.call();
							}
						}
					},
				
				// Initializing thumbnails
				
				initThumbs = function() {
						var d,
							el,
							cd = settings.hasOwnProperty('markNew')? (nows - settings.markNew.days * ONEDAY_S) : null;
						
						// Storing "rel", implementing click
						thumbs.each(function(i) {
								//this.vars = $(this).data('vars');
								$(this)
									.data('rel', i)
									.on('click', function(e) {
											if (e.target.nodeName !== 'SPAN') {
												thumbnails.hideAllTooltips();
												hidePanel(thumbnails);
												focusCard($(this).data('rel'));
											}
											return false;
										});
									
								if (whatInput.ask('intent') === 'mouse') {
									$(this).addTooltip();
								}
								
								if (cd) {
									d = $(this).data('vars')['date'];
									
									if (d && d >= cd) {
										el = $('<span>', {
												'class':	'icon-new new-image'
											}).appendTo($(this));
											
										if (whatInput.ask('intent') === 'mouse') {
											el.data('tooltip', (new Date(d * 1000)).toLocaleDateString()).addTooltip();
										}
									}
								}
							});
						
						// Right click protection on thumbnails
						if (settings.rightClickProtect) {
							thumbCont.on('contextmenu', 'img', function(e) {
									e.preventDefault()
									return false;
								});
						}
						
						// Lazyload 
						thumbCont.on('scroll.' + ns, function() {
								clearTimeout(scrollTo);
								scrollTo = setTimeout(lazyloadThumbs, 100);
							});
					},
					
				/********************************************************
				 *
				 *						Map
				 *
				 ********************************************************/
				 
					
				// Initializing map
				
				initMap = function() {

						// Reading map marker locations
						var getMarkers = function() {
								
									var markers = [];
									
									thumbs.each(function(i) {
										var vars = $(this).data('vars');
										
										if (typeof vars !== UNDEF && vars.hasOwnProperty('location')) {
											markers.push({
													title:		[((i + 1) + '.'), $(this).attr('title')].join(' '),
													pos: 		vars.location,
													link:		i
												});
										}
									});
									
									return markers;
								};
					
						// Collecting markers and initializing map
						if (settings.hasOwnProperty('map')) {
							
							$('.map-root').addMap({
									markers: 			getMarkers(),
									type:				settings.map['type'] || 'roadmap',
									zoom:				settings.map['zoom'] || 16,
									apiKey:				settings.map['apiKey'],
									fitBounds:			true,
									fullscreenControl:	false,
									onMarkerClick:		function() {
																if (this.hasOwnProperty('link')) {
																	focusCard(this.link);
																}
															}
								});
							
							// Showing map only if map button clicked
							$('#map-btn').on('click', function() {
									if (!$('.map-root').data('rendered')) {
										setTimeout(function(btn) {
												if (btn.hasClass('on')) {
													$('.map-root').trigger('render');
												}
											}, 20, $(this));
									}
								});
						}
						
					},
	
				/********************************************************
				 *
				 *						Lightbox
				 *
				 ********************************************************/
				 
				// Gets related thumb by number
				
				getRelatedCard = function(n) {
					
						return cards.children('[data-rel=' + (loop? ((n + thumbs.length) % thumbs.length) : n) + ']');
					},
					
				// Determine direction
				
				setDir = function(next) {
						var next = (typeof next === UNDEF)? nextCard : next;
					
						if (loop) {
							if (curr === thumbs.length - 1 && next === 0) {
								return 1;
							} else if (!curr && next === thumbs.length - 1) {
								return -1;
							}
						}
						
						return (next === curr)? 0 : ((next > curr)? 1 : -1);
					},
					
				// Start auto play
				
				startAuto = function() {
						//log('Start auto');
						
						hideAllPanels();
						main.removeClass(settings.pausedClass).addClass(settings.playingClass);
						if (pauseBtn) pauseBtn.addClass('animate');
						
						if (!settings.auto && settings.backgroundAudioSlideshowControl) {
							backgroundAudioRoot.trigger('fadeInPlayer');
						}
						
						slideshow = clearTimeout(slideshow);
						settings.auto = true;
						slideshow = setTimeout(next, transition.delay / 4);
						
						//swiped = false;
						
					},
										
				// Stop auto play
				
				stopAuto = function() {
					
						//log('Stop auto');
						if (settings.auto && settings.backgroundAudioSlideshowControl) {
							backgroundAudioRoot.trigger('fadeOutPlayer');
						}
						
						main.removeClass(settings.playingClass).addClass(settings.pausedClass);
						if (pauseBtn) pauseBtn.removeClass('animate');
						slideshow = clearTimeout(slideshow);
						settings.auto = false;
						//swiped = false;
						
					},
												
				// Suspending auto during video play
				
				suspendAuto = function() {
						
						main.removeClass(settings.playingClass).addClass(settings.pausedClass);
						
						if (settings.backgroundAudioSlideshowControl) {
							backgroundAudioRoot.trigger('fadeOutPlayer');
						}
						
						suspended = true;
						slideshow = clearTimeout(slideshow);
					},
					
				// Resuming auto after video ended
					
				resumeAuto = function() {
						
						suspended = false;
						slideshow = setTimeout(next, transition.delay / 4);
						
						if (settings.backgroundAudioSlideshowControl) {
							backgroundAudioRoot.trigger('fadeInPlayer');
						}
						
						main.removeClass(settings.pausedClass).addClass(settings.playingClass);
					},
					
				// Media started
				
				mediaPlaying = function(e) {
					
						if (settings.auto) {
							suspendAuto();
						} else if (settings.muteBackgroundAudio) {
							if (backgroundAudioOn = !backgroundAudioRoot.data(settings.pausedClass)) {
								backgroundAudioRoot.trigger('fadeOutPlayer');
							}
						}
						
						$(this).closest('.' + settings.cardClass).addClass(settings.playingClass);
					},
					
				// Media paused
				
				mediaPaused = function(e) {
					
						if (settings.auto) {
							resumeAuto();
						} else if (settings.muteBackgroundAudio && backgroundAudioOn) {
							backgroundAudioRoot.trigger('fadeInPlayer');
						}
						
						$(this).closest('.' + settings.cardClass).removeClass(settings.playingClass);
					},
					
				// Play media
				
				playMedia = function(n) {
						var n = (typeof n === UNDEF)? curr : n,
							c = getRelatedCard(n),
							m = (c.length? c.find('video,audio')[0] : null);
						
						if (m && m.paused) {
							if (m.nodeName === 'VIDEO') {
								playVideo(m);
							} else {
								// Audio autostart might be refused 
								m.play();
							}
						}
					},
					
				// Pause media
				
				pauseMedia = function(n) {
						var n = (typeof n === UNDEF)? curr : n,
							c = getRelatedCard(n),
							m = (c.length? c.find('video,audio')[0] : null);
						
						if (m && !m.paused) {
							m.pause();
						}
					},
					
				// Toggle media
				
				toggleMedia = function() {
						var n = (typeof n === UNDEF)? curr : n,
							c = getRelatedCard(n),
							m = (c.length? c.find('video,audio')[0] : null);
						
						if (m) {
							if (m.paused) {
								var promise = m.play();
								if (promise) {
									// Catching browser refuse
									promise.catch(function(err) {
											console.log('Autoplay has prevented by the browser. ' + err);
											if (!m.muted) {
												// Retry muted
												m.muted = true;
												promise = m.play();
											}
										});
								}
							} else {
								m.pause();
							}
						}
					},
													
				// Media ended
				
				mediaEnded = function() {
					
						backgroundAudioOn = false;
						
						if (settings.auto) {
							resumeAuto();
						}
					},
					
				// Auto starting video?
				
				autoStartMedia = function(n) {
						
						if (typeCheck('video,audio', n)) {
							if (settings.videoAuto) {
								playMedia(nextCard);
							} else {
								pauseMedia(nextCard);
							}
						}
					},
					
				// Settings cards' horizontal position
				
				setPos = function(pos) {
						cards.css({
								left:		pos + 'px'
							});
					},
					
				// Stops current animation
				
				stopTransform = function() {
						var cs = window.getComputedStyle(cards[0], null);
						
						cards[0].style['transform'] = cs['transform'];
						
					},
					
				// Adjusting parent div so card gets centered horizontally
				
				centerCardHorz = function(n) {
						var c = getRelatedCard((typeof n === UNDEF)? curr : n);
							
						if (c.length) {
							var cp = cards.position().left,
								tp = Math.round((lightbox.outerWidth() - c.outerWidth()) / 2 - c.position().left);
							
							//console.log('centerCardHorz: ' + cp + 'px => ' + tp + 'px'); 
							
							if (cp !== tp) {
								
								if (!inited) {
									// Immediate
									cards.css({
											left:			tp + 'px',
											transition:		'none'
										});
									
								} else {
									var speed;
									
									if (swiped && !!(speed = cards.data('currSpeed'))) {
										speed = Math.round(Math.abs(tp - cp) / Math.abs(speed));
									} else {
										speed = transition.adaptiveSpeed;
									}
									
									//console.log('\tspeed: ' + speed + 'ms'); 
									
									// Animate
									cards.css({
											left:			tp + 'px',
											transition:		'left ' + speed + 'ms cubic-bezier(0.15, 0, 0.4, 1)'
										});
								}
							}
							
						}
					},
				
				// Reset position after swipe
				
				resetCardPosition = function() {
						
						if (swiped) {
							centerCardHorz();
							swiped = false;
						}
					},
					
				// Refreshing layout after resize
				
				layoutRefresh = function() {
						var lw = lightbox.width() * transition.fillHorz,
							lh = lightbox.height() * transition.fillVert,
							
							resizeAll = function() {
								
								cards.children().each(function() {
										var card = $(this),
											eq = card.data('rel');
											
										if (eq !== null && eq >= 0 && eq < thumbs.length) {
											var th = thumbs.eq(eq),
												vars;
												
											if (th.length && (vars = th.data('vars'))) {
												var iw = vars['width'],
													ih = vars['height'],
													cat = vars['type'],
													r,
													cw;
													
												if (cat === 'image' || cat === 'video') {
													// Enlargement factor
													r = Math.min(lw / iw, lh / ih);
													
													if (settings.scaleUp) {
														r = Math.min(r, settings.maxScale);
													} else {
														r = Math.min(1, r);
													}
													
													cw = Math.round(iw * r);
													
												} else {
												
													cw = iw;
												}
												
												card.width(cw);
												card.find('img,video').eq(0).css({
														width:		cw,
														height:		Math.round(ih * r),
													});
											}
										}
									});
								
									centerCardHorz(curr);
								};
							
						lightbox.removeClass('landscape portrait')
							.addClass((lw >= lh)? 'landscape' : 'portrait');
						
						stopTransform();
						
						window.requestAnimationFrame(resizeAll);
						
					},
				
				// Cards ready
				
				cardReady = function() {
					
						cards.removeClass('bwd fwd');
						
						if (dir) {
							cards.addClass((dir > 0)? 'fwd' : 'bwd');
						}
						
						$('.map-root').trigger('resetMarkers');
						
						var init = function() {
									var nc = getRelatedCard(nextCard);
									//console.log('cardReady(' + nextCard + ' at:' + nc.data('rel') + ')');
									// Old current card
									cards.children('.' + settings.currClass).removeClass(settings.currClass);
									
									if (nc.length) {
										// New current card
										nc.removeClass(settings.nextClass + ' ' + settings.prevClass).addClass(settings.currClass);
										nc.prevAll().removeClass(settings.nextClass).addClass(settings.prevClass);
										nc.nextAll().removeClass(settings.prevClass).addClass(settings.nextClass);
									}
									
									if (cards.data('offs')) {
										cards.css('left', cards.position().left + cards.data('offs'));
									}
									
									if (dir || swiped) {
										// Animate
										setTimeout(function() {
												window.requestAnimationFrame(start);
												//start();
											}, 50);
									} else {
										// No animation
										transitionCompleted(nextCard);
									}
								},
								
							start = function() {
								
									centerCardHorz(nextCard);
									swiped = false;
									
									cards.one('transitionend', function(e) {
											if (e.originalEvent.target === this) {
												//window.requestAnimationFrame(end);
												end();
											}
										});
								},
								
							end = function() {
									//cleanupTo = setTimeout(cleanupCards, 100);
									transitionCompleted(nextCard);
								};
										
						window.requestAnimationFrame(init);
						//init();
						
						// Next image
						if (settings.auto && !suspended) {
							clearTimeout(slideshow);
							//pauseBtn.addClass('animate');
							slideshow = setTimeout(next, transition.delay + transition.speed);
						} else {
							stopAuto(true);
							if (settings.hideControls) {
								hideControlsLater();
							}
						}
						
						curr = nextCard;
					},
				
				// Prepare cards
				
				prepareCards = function() {
						var	wait = $(),
							first,
							last,
							
							// Creating one card
							
							createCard = function(i) {
									var eq = loop? ((i + thumbs.length) % thumbs.length) : i,
										card = getRelatedCard(eq),
										cw = 0;
									
									// Creating empty card
									if (!card.length) {
											
										card = $('<div>', {
												'data-rel':		eq,
												'class':		settings.cardClass
											});
									
										// Real card? (not empty)
										if (eq >= 0 && eq < thumbs.length) {
											var	img = card.children('img,video,audio,object'),
												thumb,
												img,
												caption,
												buttons,
												btn,
												lw = lightbox.width() * transition.fillHorz,
												lh = lightbox.height() * transition.fillVert,
												vars,
												cat,
												iw,
												ih,
												r;
											
											if (!img.length) {
												// Image is not yet loaded
												
												// Related thumb
												thumb = thumbs.eq(eq);
												vars = thumb.data('vars');
												cat = vars['type'];
												
												// Type helper class											
												card.addClass(cat);
												
												// Set width
												iw = vars['width'];
												ih = vars['height'];
												
												if (cat === 'image' || cat === 'video') {
													// Enlargement factor
													r = Math.min(lw / iw, lh / ih);
													
													if (settings.scaleUp) {
														r = Math.min(r, settings.maxScale);
													} else {
														r = Math.min(1, r);
													}
													
													cw = Math.round(iw * r);
													
												} else {
												
													cw = iw;
												}
												
												card.width(cw);
												
												switch (cat) {
													
													case 'image':
														
														// Creating Img 
														
														img = $('<img>')
															.attr({
																	width:		cw,
																	height:		Math.round(ih * r),
																	alt:		thumb.find('img').attr('alt') || ''
																})
															.one('load', function(e) {
																	$(this).addClass(settings.loadedClass);
																})
															.attr('src', getImageSource(thumb, [ cw, ih * r ]))
															.appendTo(card);
															
														break;
													
													case 'video':
														
														// Creating Video 
														
														img = $('<video>')
															.one('loadedmetadata', function(e) {
																	$(this).addClass(settings.loadedClass);
																})
															.attr({
																	controls:					true,
																	controlslist:				'nodownload nofullscreen',
																	disablePictureInPicture:	true, 
																	width:						cw,
																	height:						Math.round(ih * r),
																	type:						'video/mp4',
																	src:						vars['original']
																})
															.on('play', mediaPlaying)
															.on('pause', mediaPaused)
															.on('ended', mediaEnded)
															.appendTo(card);
														
														break;
														
													default:
														
														// Other
														img = $('<img>')
															.attr({
																	width:		iw,
																	height:		ih
																})
															.one('load', function(e) {
																	$(this).addClass(settings.loadedClass);
																})
															.attr('src', vars['thumb'])
															.appendTo(card);
															
														break;
														
												}
												
												caption = $('<div>', {
														'class':	settings.captionClass + ' ' + settings.captionStyle + ' ' + settings.captionPlacement
													});
												
												if (vars['caption']) {
													// Adding caption
														caption.append($('<div>', {
																'class':	'text',
																html:		vars['caption'].replace(/_#34_/g, '"')
															}));
												}
	
												// Buttons
												
												buttons = $('<div>', {
														'class': 	settings.buttonsClass
													});
												
												// Print button
												if (settings.printImage && cat === 'image' && !settings.rightClickProtect) {
													btn = $('<a>', {
															'class': 	'btn icon-printer',
															title:		text.print
														}).on('click', function(e) {
															printImage((vars['original'] && settings.showDownload)? vars['original'] : thumb.attr('href'),
																	(vars['original'] || thumb.attr('href')).getFile() || '',
																	vars['caption'] || ''
																);
																
															return false;
														}).appendTo(buttons);
												}
												
												// Download button
												if (cat === 'other' || 
													settings.showDownload && (cat === 'image' || settings.allowDownloadOthers)) {
												
													var href = vars['original'];
													
													if (!href && settings.allowDownloadScaled) {
														href = thumb.attr('href');
													}
													
													if (href) {
														btn = $('<a>', {
																'class': 	'btn icon-download',
																download: 	'',
																href: 		href,
																title:		text.download
															}).appendTo(buttons);
													}
												}
	
												// Photo data
												if (vars['photodata'] && cat !== 'video' && cat !== 'audio') {
													btn = $('<a>', {
															'class':	'btn icon-camera meta'
														}).appendTo(buttons);
														
													btn.data('tooltip', '<div class="photodata">' + vars['photodata'] + '</div>');
												}
												
												// Appending buttons if any
												if (!buttons.is(':empty')) {
													caption.append(buttons);
													buttons.children().addTooltip({
															touchToggle:	true
														});
												}
												
												if (!caption.is(':empty')) {
													card.append(caption);
												}
												
												// Right click protect
												if (settings.rightClickProtect) {
													img.on('contextmenu', function(e) {
															e.preventDefault()
															return false;
														});
												}
												
												// We have to wait for images left to target
												if (eq <= nextCard) {
													wait = wait.add(img);
												}
											
											} else {
												// Already loaded
												// Stop video
												if (img[0].nodeName === 'VIDEO' || img[0].nodeName === 'AUDIO') {
													img[0].pause();
												}
												
												cw = card.width();
												
											}
											
											
										} else {
											// Empty card
											card.addClass('blank').show();
											card.find('img').addClass(settings.loadedClass);
											cw = card.width();
										}
									
										// Adding card at the proper position
										
										if (i === nextCard) {
											// Current card
											var rc = getRelatedCard(i + 1);
											
											if (rc.length) {
												rc.before(card);
											} else {
												rc = getRelatedCard(i - 1);
												if (rc.length) {
													rc.after(card);
												} else {
													cards.append(card);
												}
											}
											
										} else if (i < nextCard) {
											// Insert before
											var rc = getRelatedCard(i + 1);
											
											if (rc.length) {
												rc.before(card);
												// before target: fix left position
												//cards.data('offs', cards.data('offs') - cw);
												//console.log('Inserting before card ' + nextCard + ', fixed offset: ' + cw);
												cards.css('left', Math.round(cards.position().left - cw) + 'px');
											} else {
												cards.prepend(card);
											}
											
										} else {
											// Insert after
											var rc = getRelatedCard(i - 1);
											
											if (rc.length) {
												rc.after(card);
											} else {
												cards.append(card);
											}
										}
									}
										
								},
								
							// Add cards before and after target
							
							addCards = function() {
									var	i;
									
									if (!inited) {
										
										for (i = first; i <= last; i++) {
											createCard(i);
										}
										
										// Centering after the first paint
										centerCardHorz();
										inited = true;
										
									} else {
										
										// Adding current card
										createCard(nextCard);
										
										// Adding cards before
										for (i = nextCard - 1; i > first; i--) {
											createCard(i);
										}
										
										// Adding cards after
										for (i = nextCard + 1; i < last; i++) {
											createCard(i);
										}
										
									}
									
									// Has new images to wait?
									if (!wait.length) {
										// No: Firing cardReady immediately
										cardReady();
									} else if (wait.length === 1) {
										// Wait for one image
										wait.one('load loadedmetadata', cardReady);
									} else {
										// Wait for  multiple images to be loaded
										wait.waitAllImg(cardReady);
									}
								},
							
							// Remove frames before range
							
							removeBefore = function(cc) {
									var br = cc.prevAll().filter(function() {
												var n = $(this).data('rel');
									
												return loop? 
													(n < nextCard && n < first) || (n > nextCard && n < (first + thumbs.length))
													:
													n < first;
											});
									
									if (br.length) {
										// Offsetting removed elements 
										br.each(function() {
												var w = $(this).outerWidth();
												//console.log('Removing card ' + $(this).data('rel') + ' [' + w + 'px].');
												//cards.data('offs', cards.data('offs') + $(this).outerWidth());
												$(this).remove();
												cards.css('left', Math.round(cards.position().left + w) + 'px'); 
											});
									}
									
									window.requestAnimationFrame(addCards);
								},
								
							// Start with removing cards not in range
							
							start = function() {
								
									cards.css('transition',	'none').data('offs', 0);
									
									// Initial setup
									if (cards.is(':empty')) {
										addCards();
									} else {
										var cc = getRelatedCard(nextCard);
										
										if (cc.length) {
											// Remove cards after range
											cc.nextAll().filter(function() {
													var n = $(this).data('rel');
													
													return loop? 
														(n > nextCard && n > last) || (n < nextCard && n > (last - thumbs.length))
														:
														n > last;
													
												}).remove();
											
											// Remove card before range in a new anim frame
											window.requestAnimationFrame(function() {
													removeBefore(cc);
												});
											//removeBefore(cc);
											
										} else {
											
											addCards();
										}
									}
								};
						
						// Avoid cleanup
						
						clearTimeout(cleanupTo);
						cleanupTo = null;
						
						// Adding new cards
						
						first = nextCard - transition.before;
						if (dir < 0) {
							first--;
						}
						
						last = nextCard + transition.after;
						if (dir > 0) {
							last++;
						}
						
						window.requestAnimationFrame(start);
						//start();
					},
					
				// Reading the string renditions array
				
				getRenditionsArray = function(s) {
						var a = [];
						
						if (s[0] === '[') {
							s = s.slice(1, -1);
						}
						
						s.split(/,\s?/).forEach(function(r) {
								r = r.match(/(.+) \((\d+)x(\d+)\)/);
								if (r && r.length > 3) {
									a.push([ r[1], r[2], r[3] ]);
								}
							});
						
						return a;
					},
				
				// Gets the closest rendition for a given window
				
				getClosestRendition = function(ra, dim) {
						var dim = (typeof dim === UNDEF)? [ $body.width(), $body.height() ] : dim,
							w = dim[0] * PIXELRATIO,
							h = dim[1] * PIXELRATIO;
						
						if (typeof ra === STRING) {
							ra = getRenditionsArray(ra);
						}
						
						if (ra && ra.length > 1) {
							 
							if (!WEBP_LOSSY) {
								// No webp support: filtering out
								ra = ra.filter(function(r) {
										return !r[0].endsWith('.webp');
									});
							}
							
							ra.forEach(function(r) {
									var s = Math.min(w / r[1], h / r[2]);
							
									r[3] = (s > 1)? 
											// Upscaling penalized by factor 3: 1x -> 120% == 2x -> 50%
											(3 * (1 - 1 / s))
											:
											// Downscaling
											(1 - s)
								});
						
							// Sort by scale
							ra.sort(function(r1, r2) { 
									return r1[3] - r2[3];
								});
						}
						
						return ra[0][0];
					},
					
				// Get optimal source
				
				getImageSource = function(thumb, dim) {
						var vars = thumb.data('vars');
						
						if (vars) {
							
							if (vars.hasOwnProperty('imageRenditions')) {
								return settings.slidesDir + '/' + getClosestRendition(vars['imageRenditions'], (typeof dim === UNDEF)? [ $body.width(), $body.height() ] : dim);
							}
						}	
						
						return thumb.attr('href');
					},
				
				// Get optimal source
				
				getThumbSource = function(thumb, dim) {
						var vars = thumb.data('vars');
						
						if (vars) {
							
							if (vars.hasOwnProperty('thumbRenditions')) {
								return settings.thumbsDir + '/' + getClosestRendition(vars['thumbRenditions'], (typeof dim === UNDEF)? settings.thumbDims : dim);
							}
							
							return settings.thumbsDir + '/' + vars['thumb']
						}	
						
						return thumb.attr('src');
					},
				
				// Function executed on transition complete
				
				transitionCompleted = function(n) {
						var n = (typeof n !== UNDEF)? n : curr;
					
						autoStartMedia(n);
						
						//if ($('.map-root').is(':visible')) {
							$('.map-root').trigger('setActive', n);
						//}
					},
					
				// Load image
				
				focusCard = function(n) {
						var dt = new Date();
						
						main.hideAllTooltips();
						
						if (n < 0 || n >= thumbs.length || ((dt - lastChangeDate) < 200)) {
							return;
						}
						
						//console.log('focusCard(' + n + ')');
						nextCard = n;
						dir = setDir();
						
						transition.adaptiveSpeed = Math.minMax(transition.speed / 5, dt - lastChangeDate, transition.speed);
						lastChangeDate = dt;
						
						pauseMedia(curr);
						
						if (curr !== nextCard || cards.is(':empty')) {
							prepareCards();
						} else {
							cardReady();
						}
						
						focusThumb(nextCard);
						hidePanel(thumbnails);
						/*
						if (settings.videoAuto) {
							setTimeout(function() {
									autoStartMedia(nextCard);
								}, transition.speed);
						}
						*/
						updateUrl(nextCard);
						
						setTimeout(function() {
								if (settings.hasOwnProperty('share') && lightbox.is(':visible')) {
									var vars = thumbs.eq(nextCard).data('vars');
									
									$(settings.share.hook).trigger('updateLinks', {
											//url:			window.location.href,
											image:			thumbs.eq(nextCard).attr('href') || ('thumbs/' + vars['thumb']),
											title:			vars['title'],
											description:	vars['caption']
										});
								}
							}, 300);
						
						if (settings.showNumbers) {
							updateNumber(nextCard + 1);
						}
						/*
						if (pauseBtn) {
							pauseBtn.addClass('animate');
						}
						*/
					},
					
				// Focusing the closest card to lightbox center
				
				loadClosestCard = function(targetPos, dir) {
					
						//stopTransform();
						
						var cc = cards.children('#card_' + curr),
						
							center = lightbox.width() / 2 - ((typeof targetPos === UNDEF)? cards.position().left : targetPos),
							
							found = false,
							
							gotoCard = function(n) {
									//console.log('Focusing closest image ' + (cn + 1) + ', target = ' + targetPos);
									if (n !== null && n !== curr && !isNaN(n)) {
										
										if (n >= thumbs.length) {
											// After last
											if (loop) {
												// Loop to first
												focusCard(0);
											} else if (settings.afterLast === 'nextfolder' && settings.nextFoldersFirst) {
												// Next folder
												stopAuto();
												window.location.href = settings.nextFoldersFirst;
											} else {
												// No change
												resetCardPosition();
											}
											
										} else if (n < 0) {
											// Before first
											if (loop) {
												// Loop to first
												focusCard(thumbs.length - 1);
											} else if (settings.afterLast === 'nextfolder' && settings.previousFoldersLast) {
												// Next folder
												stopAuto();
												window.location.href = settings.previousFoldersLast;
											} else {
												// No change
												resetCardPosition();
											}
											
										} else {
											// Valid card number
											focusCard(n);
										}
										
									} else {
										// No cards match, reset position
										resetCardPosition();
									}
								
								};
						
						if (typeof dir === UNDEF) {
							dir = 0;
						}
						
						if (dir > 0) {
							
							// Check all cards after the current
							cc.nextAll().each(function() {
									if (center < $(this).position().left + $(this).width()) {
										gotoCard($(this).data('rel'));
										found = true;
										return false;
									}
								});
								
							if (!found) {
								// Focus the one after the last
								gotoCard((cards.filter('[data-rel]').last().data('rel') || curr) + 1);
							}
							
						} else if (dir < 0) {
							
							// Check all cards before the current
							cc.prevAll().each(function() {
									if (center > $(this).position().left) {
										gotoCard($(this).data('rel'));
										found = true;
										return false;
									}
								});
							
							if (!found) {
								// Focus the one before the first
								gotoCard((cards.filter('[data-rel]').first().data('rel') || curr) - 1);
							}
							
						} else {
							
							resetCardPosition();
						}
							
					},
							
				// Previous card
				
				previous = function() {
						
						if (settings.hideControls) {
							hideControlsLater();
						}
						
						//swiped = false;
						
						if (curr) {
							// Not first
							focusCard(curr - 1);
							return;
							
						} else {
							
							stopAuto();
							
							// First
							switch (settings.afterLast) {
								
								case 'donothing':
								case 'onelevelup':
								case 'ask':
									
									stopAuto();
									break;
									
								case 'nextfolder':
									
									stopAuto();
									if (settings.previousFoldersLast) {
										window.location.href = settings.previousFoldersLast;
									} else {
										focusCard(0);
									}
									break;
									
								case 'startover':
									
									focusCard(thumbs.length - 1);
									return;
									
							}
							
						}
						
						resetCardPosition();
					},
					
				// Next card
				
				next = function() {
						
						hideAllPanels();
						if (settings.hideControls) {
							hideControlsLater();
						}
						
						//swiped = false;
						
						if (curr < (thumbs.length - 1)) {
							
							// Not at end
							focusCard(curr + 1);
							return;
							
						} else {
							
							// At the end
							switch (settings.afterLast) {
								
								case 'donothing':
									
									stopAuto();
									break;
									
								case 'startover':
									
									focusCard(0);
									return;
									
								case 'onelevelup':
									
									stopAuto();
									
									if (settings.level) {
										window.location.href = '../' + settings.indexName;
									}
									break;
									
								case 'nextfolder':
									
									var ss = settings.auto;
									
									stopAuto();
									
									if (settings.nextFoldersFirst) {
										window.location.href = settings.nextFoldersFirst + (ss? '&slideshow' : '');
									} else {
										focusCard(thumbs.length - 1);
									}
									break;
									
								case 'ask':
							
									stopAuto();
									
									var buttons = new Array();
									
									resetCardPosition();
									
									if (thumbs.length > 1) {
										// Start over
										buttons.push({
												t: 	text.startOver,
												c:	'icon-loop',
												h: 	function(lb) { 
														focusCard(0);
														loop = true;
													}
											});
									}
									
									if (settings.level) {
										// Up one level
										buttons.push({
												t: 	text.upOneLevel, 
												c:	'icon-one-level-up',	
												h: 	function() { 
														window.location.href = '../' + settings.indexName;
													}
											});
									}
																		
									if (settings.nextFoldersFirst) {
										// Go to next folder
										buttons.push({
												t: 	text.nextFolder,
												c:	'icon-arrow-right',
												h: 	function() {
														window.location.href = settings.nextFoldersFirst;
													}
											});
									}
							
									main.modal($('<p>', {
											'class':	'text-center',
											text: 		text.atLastPageQuestion
										}), buttons, {
											onClose:	function() {
															focusCard(thumbs.length - 1);
														},
											'class': 	'secondary',
											title: 		text.atLastPage
										});
									
									return;
									
							}
							
						}
						
						resetCardPosition();
					},
					
				// Next one in the current direction
				
				change = function() {
					
						clearTimeout(slideshow);
						
						if (dir >= 0) {
							next();
						} else {
							previous();
						}
					},
					
				// Initializing lightbox
				
				initLightbox = function() {
					
						// Extending settings with data-settings="" attribute
						
						if (lightbox.data('settings')) {
							$.extend(setting, lightbox.data('settings'));
						}
						
						// Finding / Creating cards container
						
						cards = lightbox.find('.' + settings.cardsClass);
						
						if (!cards.length) {
							cards = $('<div>', {
									'class':	settings.cardsClass
								}).appendTo(lightbox);
						}
						
						// Collecting transition properties
						
						transition = settings.defaults[settings.transitionType || 'conveyor'];
						
						if (settings.hasOwnProperty('transitionSpeed')) {
							transition.speed = settings.transitionSpeed;
						}
						
						if (settings.hasOwnProperty('slideshowDelay')) {
							transition.delay = settings.slideshowDelay;
						}
						
						if (settings.hasOwnProperty('fillHorz')) {
							transition.fillHorz = settings.fillHorz / 100;
						}
						
						if (settings.hasOwnProperty('fillVert')) {
							transition.fillVert = settings.fillVert / 100;
						}
						
						// Looping conveyor
						
						loop = settings.afterLast === 'startover';
						
						// Reducing before/after count if too few items
						
						if (thumbs.length === 1) {
							transition.after = transition.before = 0;
						} else {
							transition.after = Math.min(transition.after, Math.ceil(thumbs.length / 2));
							transition.before = Math.min(transition.before, Math.floor(thumbs.length / 2));
						}
						
						// Mouse wheel
						
						if (settings.enableMouseWheel) {
							var wheelTimeout = null;
							lightbox.on('mousewheel.' + ns + ' DOMMouseScroll.' + ns, function(e) {
									wheelTimeout = clearTimeout(wheelTimeout);
									wheelTimeout = setTimeout((e.originalEvent.wheelDelta > 0 || e.originalEvent.detail < 0)? previous : next, 100); 
									return false;
								});
						}
						
						// Buttons
						
						if (thumbs.length > 1) {
							// Previous button
							prevBtn	= controls.find('.previous');
							prevBtn.on('click.' + ns, previous);
							
							// Play button
							playBtn = controls.find('.play');
							playBtn.on('click.' + ns, startAuto);
							
							// Pause button
							pauseBtn = controls.find('.pause');
							pauseBtn.on('click.' + ns, stopAuto);
							
							// Next button
							nextBtn = controls.find('.next');
							nextBtn.on('click.' + ns, next);
						}
												
						// Keyboard
						
						if (settings.enableKeyboard) {
							
							$document.on('keyup.' + ns, function(e) {
								if (!main.length || 
										lightbox.children('.modal:visible').length || 
										document.activeElement && (document.activeElement.nodeName === 'INPUT' || document.activeElement.nodeName === 'TEXTAREA')
									) {
									return true;
								}
								
								e.preventDefault();
								
								switch (e.keyCode) {
									
									case 10:
									case 13:
										hidePanel(thumbnails);
										break;
										
									case 33:
									case 37:
										previous();
										break;
										
									case 34:
									case 39:
										next();
										break;
										
									case 97:
									case 35:
										focusCard(thumbs.length - 1);
										break;
										
									case 103:
									case 36:
										focusCard(0);
										break;
									
									case 106:
									case 179:
										if (slideshow) {
											stopAuto();
										} else {
											startAuto();
										}
										break;
										
									case 32:
										if (typeCheck('video,audio')) {
											toggleMedia();
										} else {
											if (slideshow) {
												stopAuto();
											} else {
												startAuto();
											}
										}
										break;
										
									default:
										return true;
														
								}
								
								return false;
							});
						}
						
						// Handling clicks on lightbox
						
						var lastClick = 0,
						
							lightboxClick = function(e) {
									var d = new Date();
									
									if ((d - lastClick) > 100) {
										
										// Handle only if separate click 
										lastClick = d;
										
										if (thumbnails.is(':visible')) {
											// Hide thumbnails panel
											hidePanel(thumbnails);
											return false;
											
										} else if (e.target.nodeName !== 'BUTTON' && e.target.nodeName !== 'A') {
											var el = $(e.target).closest('.' + settings.cardClass);
											
											if (el.length && el.data('rel') !== curr) {
												// Focus previous / next card
												focusCard(el.data('rel'));
												return false;
											} else {
												// Handle video start / stop
												if (typeCheck('video,audio')) {
													toggleMedia();
													return false;
												}
											}
										}
									}
									
									return true;
								};
						
						// Swipe function
						
						cards
							.swipe({
									onSwipedRight: 	function(e) {
														swiped = true;
														//pauseMedia(curr);
														loadClosestCard(cards.data('targetPos'), -1);
														//clearTimeout(focusTo);
														//focusTo = setTimeout(loadClosestCard, 50, [ cards.data('targetPos'), -1 ]);
													},
													
									onSwipedLeft: 	function(e) {
														swiped = true;
														//pauseMedia(curr);
														loadClosestCard(cards.data('targetPos'), 1);
														//clearTimeout(focusTo);
														//focusTo = setTimeout(loadClosestCard, 50, [ cards.data('targetPos'), 1]);
													},
													
									onSwipeStart:	function() {
														$body.hideAllTooltips();
														stopAuto();
													},
													
									onClick:		function(e) {
														// Small swipe => click
														if (!swiped) {
															lightboxClick(e);
														}
													}
								})
							//.on('click.' + ns, lightboxClick);

						
						// Lightbox resize event listener 
						
						$window.on('resize.' + ns, function() {
								clearTimeout(layoutTo);
								layoutTo = setTimeout(layoutRefresh, 100);
							});
						
					},
				
				/********************************************************
				 *
				 *						Initialization
				 *
				 ********************************************************/
				 
				init = function() {
					
						// Removing search string
						
						if (window.location.search) {
							removeSearch();
						}
						
						// Background music
						
						initBackgroundMusic(backgroundAudioRoot);
						
						// Preparing share buttons
						
						if (settings.hasOwnProperty('share')) {
							initShares(settings.share);
						}
						
						// Hash change listener
						
						$window.on('hashchange.' + ns, stateChange);
						
						if (window.location.hash) {
							var m = window.location.hash.match(/img=([^&]+)/);
							if (m) {
								curr = getImageByName(m[1]);
								if (curr !== -1) {
									initByHash = true;
									settings.auto = !!window.location.hash.match(/\&slideshow$/);
								} else {
									curr = 0;
								}
							}
						}
						
						// Controls
						
						initControls();
						
						// Thumbnails
						
						if (thumbnails.length) {
							initThumbs();
						}
						
						// Initializing map
						
						if (settings.hasOwnProperty('map')) {
							initMap();
						}
					
						// Lightbox
						
						if (lightbox.length) {
							
							initLightbox();
							focusCard(curr);
							
						}
			
						// Tool tips
						if (whatInput.ask('intent') === 'mouse') {
							$('#controls [data-tooltip]').addTooltip({
									delay:		1000,
									stay:		2000,
									pos:		[0, 1, 2, 1]
								});
						}
						
						// Autoplay
						
						if (settings.auto && lightbox.length && thumbs.length > 1) {
							startAuto();
						}
					};
					
			/********************************************************
			 *
			 *					Execution starts here
			 *
			 ********************************************************/
				
			init();
			
			return this;
		};
		
	/********************************************************
	 *
	 *						Defaults
	 *
	 ********************************************************/
				
	$.fn.skin.defaults = {
			activeClass: 					'active',
			lazyloadClass:					'lazyload',
			thumbsDir:						'thumbs',
			slidesDir:						'slides',
			indexName:						'index.html',
			cardsClass:						'cards',
			cardClass:						'card',
			currClass:						'curr',
			prevClass:						'prev',
			nextClass:						'next',
			loadedClass:					'loaded',
			captionClass:					'caption',
			buttonsClass:					'buttons',
			hideImageClass:					'hide-image',
			showImageClass:					'show-image',
			fullscreenClass:				'fullscreen',
			controlsHideClass:				'hide-controls',
			playingClass:					'playing',
			pausedClass:					'paused',
			
			transTime:						500, 				// Menu transition length
			clickForNext:					true,
			backgroundAudioSlideshowControl:false,
			muteBackgroundAudio:			true,
			videoAuto:						false,
			videoLoop:						false,
			volume:							0.5,
			auto:							false,
			afterLast:						'donothing',
			
			rightClickProtect: 				false,
			allowDownloadOthers:			false,
			
			hideControls:					0,
			
			showNumbers:					true,
			
			transitionType:					'conveyor',
			captionStyle:					'dark',
			captionPlacement:				'center bottom',
			
			showFullscreen:					true,
			enableKeyboard:					true,
			enableMouseWheel:				true,
			
			scaleUp:						false,
			maxScale:						1.4,
			
			fillHorz:						90,
			fillVert:						100,
			
			defaults: 						{
												'conveyor':				{
																			before:		2,
																			after:		2,
																			delay:		1000,
																			speed:		2000,
																			fillHorz:	0.9,
																			fillVert:	1
																		},
												'kenburns':				{
																			before:		1,
																			after:		0,
																			delay:		0,
																			speed:		4000
																		},
												'crossfade':			{
																			before:		1,
																			after:		0,
																			delay:		1000,
																			speed:		2000
																		},
												'zoom':					{
																			before:		1,
																			after:		0,
																			delay:		1000,
																			speed:		2000
																		},
												'slide':				{
																			before:		1,
																			after:		0,
																			delay:		1000,
																			speed:		2000
																		}
											},
											
			fit:							'both'								// 'cover', 'fit'
			
		};
			
})(jQuery, jQuery(window), jQuery(document), jQuery('body'));
