package lizard;

/*
	Gui.java - User Interface
*/

import java.io.*;
import java.util.*;
import java.awt.Color;
import java.awt.event.*;
import java.awt.BorderLayout;
import java.awt.Component;
import java.awt.Dimension;
import java.awt.Font;
import java.beans.PropertyChangeListener;
import java.util.logging.Level;
import javax.swing.*;
import javax.swing.BorderFactory;
import javax.swing.JButton;
import javax.swing.border.Border;
import javax.swing.filechooser.FileNameExtensionFilter;
import edu.stanford.ejalbert.BrowserLauncher;
import com.kitfox.svg.app.beans.SVGIcon;

import icons.Lazaicon;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import se.datadosen.util.Item;
import se.datadosen.util.IO;
import se.datadosen.component.*;
import se.datadosen.component.DeferredSVGIcon;
import se.datadosen.jalbum.AlbumObject;
import se.datadosen.jalbum.JAlbumContext;
import se.datadosen.jalbum.PluginContext;
import se.datadosen.jalbum.EditPanel;
import se.datadosen.jalbum.SkinProperties;
import se.datadosen.jalbum.AccountManager;
import se.datadosen.jalbum.SignInManager;
import se.datadosen.jalbum.JAlbumSite;
import se.datadosen.jalbum.AlbumBean;
import se.datadosen.jalbum.Icons;
import se.datadosen.jalbum.JAlbum;
import se.datadosen.jalbum.JAlbumFrame;
import se.datadosen.jalbum.ParameterException;
import se.datadosen.jalbum.event.JAlbumAdapter;
import se.datadosen.jalbum.event.JAlbumEvent;
import se.datadosen.util.Languages;
import se.datadosen.util.VersionNumber;

/*************************************************
 * 
 *					Lizard GUI
 * 
 */

public class Gui extends GuiBase {
		
	private long skinReadyAt = Long.MAX_VALUE;
	private final VersionNumber jalbumVersion = new VersionNumber(AlbumBean.getInternalVersion());
	private final String skinVer = new SkinProperties(skinDirectory).getProperty(SkinProperties.VERSION);
	private final String supportForum = new SkinProperties(skinDirectory).getProperty(SkinProperties.SUPPORT_FORUM);
	private final String helpRoot = "http://jalbum.net/help/en/Skin/Lizard";
	
	private boolean isSkinReady() {
		// Within 1s of launch
		//System.out.println("skinReady()? " + (System.currentTimeMillis() - skinReadyAt));
		return (System.currentTimeMillis() - skinReadyAt) > 1000L;
	}
	
	private String getStyleName() {
		String style = engine.getStyle();
		return style.substring(0, style.indexOf("."));
	}
	
	private String getSkinName() {
		return engine.getSkin();
	}
	
	private String getLastSkinName() {
		String ls = null;
		
		try {
			ls = engine.getLastSkin();
		} catch(Throwable t) {
			JAlbum.logger.log(Level.FINER, "Last skin is unknown.");
		}
		
		return ls;
	}	
	
	private String unCamelCase(String s) {
		
		if (s.length() > 0) {
			s = s.replaceAll("([a-z])([A-Z]+)", "$1 $2");
			return Character.toUpperCase(s.charAt(0)) + s.substring(1);
		}
		
		return "";
	}
	
	private String getText(String t) {
		String s;
		
		try {
			s = texts.getString(t);
		} catch (java.util.MissingResourceException e) {
			JAlbum.logger.log(Level.FINE, "Missing text: \"{0}\"", t);
			if (t.startsWith("ui.")) {
				t = t.substring(3);
			}
			s = unCamelCase(t);
		}
		
		return s;
	}
	
	private final int uiHeight = 600;
	private final int uiWidth = 600;
	/*
	private final int previewWidth = 560;
	*/
	private final String tabStyle = "style='padding:3px 4px;margin:3px 4px;'";
	private final String CUSTOM = "<!--custom-->";
	
	private String getFonts(String html, String ff, String hf) {
		String	gf = "",
				fw = "",
				bw = "bold",
				hw = "";
		String[] w;
		
		if (ff.contains(":")) {
			// Google font
			gf = ff;
			fw = ff.split(":")[1];
			if (fw.length() > 0) {
				w = fw.split(",");
				bw = (w.length > 2)? w[2] : w[1];
				fw = w[0];
			}
			
			ff = ff.split(":")[0];
			if (ff.contains(" ")) {
				ff = "\"" + ff + "\"";
			}
			ff += ",sans-serif";
		}

		if (!hf.equals("")) {
			if (!hf.equals(gf)) {
				gf = (gf.length() > 0)? (gf + "|" + hf) : hf;
			}
			if (hf.contains(":")) {
				// Has weight info
				hw = hf.split(":")[1];
				if (hw.length() > 0) {
					// Get base weight
					hw = hw.split(",")[0];
				}
				hf = hf.split(":")[0];
			}
			if (hf.contains(" ")) {
				hf = "\"" + hf + "\"";
			}
			hf += ",sans-serif";
		}
		
		//System.out.println("Google font: " + gf);
		return html.replace("{googleFontsLink}", (gf.length() == 0)? "" : ("\t<link rel=\"stylesheet\" href=\"http://fonts.googleapis.com/css?family=" + gf.replaceAll(" ", "+") + "&display=swap\">\n"))
			.replace("{fontFamily}", ff)
			.replace("{fontWeight}", (fw.equals("")? "300" : fw))
			.replace("{boldWeight}", bw)
			.replace("{headlineFont}", (hf.equals(""))? "inherit" : hf)
			.replace("{headlineWeight}", (hw.equals("")? "400" : hw));
		
	}
	
	private String attemptSignIn() {
		SignInManager mgr = SignInManager.getInstance();
		if ( mgr != null && mgr.isSignedIn() ) {
			return "&cid=" + AccountManager.getCid(mgr.getUserName(), mgr.getPassword());
		}
		return "";
	}
		
	private String license = licenseManager.isLicenseValid()? licenseManager.getLicenseType() : "";
	
	private boolean commercialUseAllowed = false;
	
	private Font mono = new Font("monospaced", Font.PLAIN, 12);
	
	private Border emptyBorder = BorderFactory.createEmptyBorder();
	
	private String getFileContents(String name) {
		StringBuilder cont = new StringBuilder();
		String line;
		String nl = System.getProperty("line.separator");
		File f = new File(skinDirectory, name);
		
		if (f.exists()) {
			try {
				try (java.io.BufferedReader in = new java.io.BufferedReader(new java.io.FileReader(f))) {
					while ((line = in.readLine()) != null) {
						cont.append(line);
						cont.append(nl);
					}
				}
			} catch (FileNotFoundException e) {
				return null;
			} catch (IOException e) {
				return null;
			}
		}
		return cont.toString();
	}
	
	private void writeFile(String fileName, String cont) {
		File out = new File(fileName);
		
		try {
			FileWriter writer = new FileWriter(out, false);
			writer.write(cont);
			writer.close();
		} catch (IOException e) {
			JAlbum.logger.log(Level.FINE, "Error writing \"{0}\".", fileName);
		}
	}
	
	private final Double FULLY_TRANSP = 0.0001D;
	private final Double FULLY_OPAQUE = 0.9999D;
	
	private Color toColor(JColorSelector cs) {
		return toColor(cs.toString());
	}
	
	private static final Pattern RGBA_PATTERN = Pattern.compile("rgba?\\((\\d+),(\\d+),(\\d+)(,([\\d\\.]+))?\\)");
	
	private Color toColor(String c) {
		int a = 255;
		Color clr = Color.GRAY;
		
		//System.out.print("toColor(" + c + ") -> ");
		if (c.startsWith("rgb")) {
			Matcher matcher = RGBA_PATTERN.matcher(c);
			if (matcher.matches()) {
				// Get the color components from the matched groups
				if (matcher.groupCount() > 5) {
					a = Math.min(255, (int)Math.round(Double.parseDouble(matcher.group(5)) * 255.0));
				}
				
				if (a == 255) {
					clr = new Color(
						Math.min(255, Integer.parseInt(matcher.group(1))),
						Math.min(255, Integer.parseInt(matcher.group(2))),
						Math.min(255, Integer.parseInt(matcher.group(3)))
					);
				} else {
					clr = new Color(
						Math.min(255, Integer.parseInt(matcher.group(1))),
						Math.min(255, Integer.parseInt(matcher.group(2))),
						Math.min(255, Integer.parseInt(matcher.group(3))),
						a
					);
				}
			}
			
			return clr;
		}
		
		if (c.charAt(0) == '#') {
			c = c.substring(1);
		}
		
		if (c.length() > 7) {
			a = Integer.parseInt(c.substring(0, 2), 16);
			c = c.substring(2);
		}
		
		if (c.length() == 3) {
			clr = new Color(
					Math.min(255, Integer.parseInt(c.substring(0,1), 16) * 17),
					Math.min(255, Integer.parseInt(c.substring(1,2), 16) * 17),
					Math.min(255, Integer.parseInt(c.substring(2), 16) * 17)
				);
		}
		
		if (c.length() == 6) {
			clr = new Color(
					Math.min(255, Integer.parseInt(c.substring(0,2), 16)),
					Math.min(255, Integer.parseInt(c.substring(2,4), 16)),
					Math.min(255, Integer.parseInt(c.substring(4), 16)),
					a
				);
			//System.out.println(clr.getAlpha() + " alpha=" + a);
		}
		
		return clr;
	}
	
	private String getCssColor(JColorSelector cs) {
		
		return getCssColor(toColor(cs));
	}
	
	private String getCssColor(String c) {
		
		return getCssColor(toColor(c));
	}
	
	private String getCssColor(Color c) {
		
		if (c == null) {
			return "transparent";
		}
		
		Double a = c.getAlpha() / 255.0;
		
		if (a < FULLY_TRANSP) {
			// Transparent
			return "transparent";
		} else if (a < FULLY_OPAQUE) {
			// Semi transparent
			return "rgba(" + c.getRed() + "," + c.getGreen() + "," + c.getBlue() + "," + (Math.round(a * 1000) / 1000.0) + ")";
		}
		
		// Opaque
		return "rgba(" + c.getRed() + "," + c.getGreen() + "," + c.getBlue() + ",1)";
	}
	
	private Double getAlpha(JColorSelector cs) {
		Color c = toColor(cs);
		
		return c.getAlpha() / 255.0;
	}
	
	private Double getAlpha(Color c) {
		return c.getAlpha() / 255.0;
	}
	
	private Double getAlpha(String clr) {
		Color c = toColor(clr);
		
		return c.getAlpha() / 255.0;
	}
	
	private String setAlpha(String clr, Double a) {
		Color c = toColor(clr);
		
		return "rgba(" + c.getRed() + "," + c.getGreen() + "," + c.getBlue() + "," + a + ")";
	}
	
	private Color flatColor(JColorSelector bc, JColorSelector fc) {
		
		return flatColor(toColor(bc), toColor(fc));
	}
	
	private Color flatColor(Color bc, Color fc) {
		Double a = fc.getAlpha() / 255.0;
			
		if (a < FULLY_TRANSP) {
			return bc;
		} else if (a > FULLY_OPAQUE) {
			return fc;
		}
		
		return new Color(
			(int)Math.max(0, Math.min(Math.round(a * fc.getRed() + (1 - a) * bc.getRed()), 255)),
			(int)Math.max(0, Math.min(Math.round(a * fc.getGreen() + (1 - a) * bc.getGreen()), 255)),
			(int)Math.max(0, Math.min(Math.round(a * fc.getBlue() + (1 - a) * bc.getBlue()), 255))
		);
	}
	
	private String getMiddleColor(String c1, String c2) {
		Color c = getMiddleColor(toColor(c1), toColor(c2));
		
		return "rgba(" + c.getRed() + "," + c.getGreen() + "," + c.getBlue() + "," + (Double.valueOf(c.getAlpha()) / 255.0) + ")";
	}
	
	private Color getMiddleColor(Color c1, Color c2) {
					
		return new Color(
				(int)Math.round((c1.getRed() + c2.getRed()) / 2),
				(int)Math.round((c1.getGreen() + c2.getGreen()) / 2),
				(int)Math.round((c1.getBlue() + c2.getBlue()) / 2),
				(int)Math.round((c1.getAlpha() + c2.getAlpha()) / 2)
			);
	}

	
	private String getFlatColor(JColorSelector bc, JColorSelector fc) {
		
		return getCssColor(flatColor(toColor(bc), toColor(fc)));
	}
	
	private String getFlatColor(String bc, String fc) {
		
		return getCssColor(flatColor(toColor(bc), toColor(fc)));
	}
	
	private String getFlatColor(Color bc, Color fc) {
		
		return getCssColor(flatColor(bc, fc));
	}
	
	private Double getLuminosity(JColorSelector bg, JColorSelector fg) {
		
		return getLuminosity(flatColor(bg, fg));
	}
			
	private Double getLuminosity(JColorSelector cs) {
		
		return getLuminosity(toColor(cs));
	}
	
	private Double getLuminosity(String c) {
		
		return getLuminosity(toColor(c));
	}
	
	private Double getLuminosity(Color c) {
		
		return	0.0011725490196078 * c.getRed() + 
				0.0023019607843137 * c.getGreen() +
				0.0004470588235294118 * c.getBlue();
	}
	
	private boolean isLightColor(JColorSelector cs) {
		
		return getLuminosity(cs) > 0.5;
	}
	
	private boolean isLightColor(String fc) {
		
		return getLuminosity(fc) > 0.5;
	}
	
	private boolean isLightColor(JColorSelector fc, double threshold) {
		
		return getLuminosity(fc) > threshold;
	}
	
	private boolean isLightColor(String fc, double threshold) {
		
		return getLuminosity(fc) > threshold;
	}
	
	private boolean isLightColor(JColorSelector bc, JColorSelector fc) {
		
		return isLightColor(toColor(bc), toColor(fc), 0.5);
	}
	
	private boolean isLightColor(String bc, String fc) {
		
		return isLightColor(toColor(bc), toColor(fc), 0.5);
	}
	
	private boolean isLightColor(JColorSelector bc, JColorSelector fc, Double threshold) {
		
		return isLightColor(toColor(bc), toColor(fc), threshold);
	}
	
	private boolean isLightColor(String bc, String fc, Double threshold) {
		
		return isLightColor(toColor(bc), toColor(fc), threshold);
	}
	
	private boolean isLightColor(Color bc, Color fc, Double threshold) {
		
		Double a = fc.getAlpha() / 255.0;
		
		if (a > FULLY_OPAQUE) {
			return getLuminosity(fc) >= threshold;
		}
		
		return getLuminosity(flatColor(bc, fc)) >= threshold;
	}
	
	private String getLegibleColor(Color bc1, Color bc2, Color fc, Double f) {
		Color bc = flatColor(bc1, bc2);
		
		return getLegibleColor(bc, fc, f);
	}
		
	private String getLegibleColor(JColorSelector bc, JColorSelector fc) {
		return getLegibleColor(toColor(bc), toColor(fc), 0.6);
	}
	
	private String getLegibleColor(JColorSelector bc, JColorSelector fc, Double f) {
		return getLegibleColor(toColor(bc), toColor(fc), f);
	}
	
	private String getLegibleColor(JColorSelector bc1, JColorSelector bc2, JColorSelector fc) {
		return getLegibleColor(flatColor(bc1, bc2), toColor(fc), 0.6);
	}
	
	private String getLegibleColor(JColorSelector bc1, JColorSelector bc2, JColorSelector fc, Double f) {
		return getLegibleColor(flatColor(bc1, bc2), toColor(fc), f);
	}
	
	private String getLegibleColor(Color bc, Color fc, Double f) {
		int r, g, b;
		
		if (Math.abs(getLuminosity(bc) - getLuminosity(fc)) >= f) {
			return getCssColor(fc);
		}
		
		f *= 255.0;
		
		if (getLuminosity(bc) > 0.5) {
			// Darken
			r = (int)Math.round(Math.max(fc.getRed() - f, 0));
			g = (int)Math.round(Math.max(fc.getGreen() - f, 0));
			b = (int)Math.round(Math.max(fc.getBlue() - f, 0));
		} else {
			// Lighten
			r = (int)Math.round(Math.min(fc.getRed() + f, 255));
			g = (int)Math.round(Math.min(fc.getGreen() + f, 255));
			b = (int)Math.round(Math.min(fc.getBlue() + f, 255));
		}
		
		return getCssColor(new Color(r, g, b));
	}
	
	private final String[] imageFiles = new String[] { "jpg", "jpeg", "png", "gif", "svg" };
	private final FileChooser fc = ChooserFactory.createFileChooser(window);

	private void getFileToRes(String[] extensions, JTextField name, Component c) {
		fc.setFileFilter(new FileNameExtensionFilter(getText("ui.select"), extensions));
		int returnVal = fc.showOpenDialog(c);
		
		if (returnVal == JFileChooser.APPROVE_OPTION) {
			String fn = fc.getSelectedFile().toString();
			
			if (!fn.trim().equals("")) {
				File src = new File(fn);
				File dst = new File(context.getPluginContext().getRootFolder().getFile(), "res");
				if (!dst.exists()) {
					dst.mkdir();
				}
				if (src.exists() &&  dst.exists()) {
					try {
						IO.copyFile(src, dst);
						name.setText(src.getName());
					} catch (IOException e) {
						System.out.println(e);
					}
				}
			}
		}
	}
	
	private void backupProjectFile(String ver) {
		
		try {
			File projectFile = window.projectChooser.getSelectedFile();
			IO.copyFile(projectFile, new File(projectFile.getParentFile(), IO.baseName(projectFile) + "-" + ((ver == null)? "old" : ver) + ".jap"));
		} catch (IOException ex) {
			JAlbum.logger.log(Level.FINE, "Error backing up project file: {0}", window.projectChooser.getSelectedFile());
		}
	}
	
	private StateMonitor commercialMonitor = new StateMonitor() {
		@Override
		public void onChange() {
			
			if (((JCheckBox)source).isSelected() && !commercialUseAllowed) {
				if (isSkinReady()) {
					Object[] options = { 
						getText("ui.signUp"),
						getText("ui.noThanks")
					};
					int n = JOptionPane.showOptionDialog(window, 
						getText("ui.licenseWarningText"), 
						getText("ui.licenseWarningTitle"), 
						JOptionPane.YES_NO_OPTION, 
						JOptionPane.INFORMATION_MESSAGE,
						null,
						options,
						options[1]
					);
					if (n == 0) {
						try {
							BrowserLauncher.openURL(JAlbumSite.getTrueInstance().getMyJAlbumUpgradeUrl() + "/?referrer=" + skin + attemptSignIn());
						} catch (se.datadosen.tags.ElementException | IOException x) {
						}
					}
				}
				((JCheckBox)source).setSelected(false);
			}
		}
	};

	private static Icon icon(String basename) {
		return Icons.get(Lazaicon.class, "svg/" + basename + ".svg", 18);
	}
		
	private static Icon icon(String basename, int size) {
		return Icons.get(Lazaicon.class, "svg/" + basename + ".svg", size);
	}
		
	private static Icon svgIcon(String basename, Dimension dim) {
		return svgIcon(basename, dim, false);
	}
	
	private static Icon svgIcon(String basename, Dimension dim, boolean adapt) {
		DeferredSVGIcon icon = new DeferredSVGIcon(Gui.class, "graphics/" + basename + ".svg");
		
		icon.setAntiAlias(true);
		icon.setAdaptColors(adapt);
		icon.setPreferredSize(dim);
		icon.setAutosize(SVGIcon.AUTOSIZE_STRETCH);
		
		return icon;
	}
	
	private Object[] getPosition() {
		return new Object[] {
			new Item("left top", getText("ui.left_top")), 
			new Item("center top", getText("ui.center_top")), 
			new Item("right top", getText("ui.right_top")), 
			new Item("left center", getText("ui.left_middle")), 
			new Item("center center", getText("ui.center_middle")), 
			new Item("right center", getText("ui.right_middle")), 
			new Item("left bottom", getText("ui.left_bottom")), 
			new Item("center bottom", getText("ui.center_bottom")), 
			new Item("right bottom", getText("ui.right_bottom"))
		};
	}
	
	private int setSelectedValue(JComboBox<Item> box, Object val) {
		int		size = box.getItemCount();
		Item	it;
		
		for (int i = 0; i < size; i++) {
			it = box.getItemAt(i);
			if (it.item.equals(val)) {
				box.setSelectedIndex(i);
				return i;
			}
		}
		
		return -1;
	}
	
	private String getSelectedValue(Object o) {
		if (o instanceof JComboBox) {
			Object so = ((JComboBox)o).getSelectedItem();
			if (so instanceof Item) {
				return ((Item)so).value.toString();
			}
			return (so == null)? null : so.toString();
		}
		return (o == null)? null : o.toString();
	}
	
	private int getSpinnerValueAsInt(JSpinner o) {
		return (Integer)o.getValue();
	}
	
	private Double getSpinnerValueAsDouble(JSpinner o) {
		return (Double)o.getValue();
	}
	
	private Double getSelectedValueAsDouble(Object o) {
		if (o instanceof JComboBox) {
			Object so = ((JComboBox)o).getSelectedItem();
			if (so instanceof Item) {
				String s = ((Item)so).value.toString();
				return ((s != null) && (s.length() > 0))? Double.parseDouble(s) : null;
			}
			return null;
		}
		return (o == null)? null : Double.parseDouble(o.toString());
	}
	
	private Integer getSelectedValueAsInt(JComboBox o) {
		Object so = ((JComboBox)o).getSelectedItem();
		if (so instanceof Item) {
			return Integer.parseInt(((Item)so).value.toString());
		}
		return null;
	}
	
	private Double getThumbAspectRatio() {
		String[] cdim = engine.getThumbSize().split("x");
		Double	w = Double.parseDouble(cdim[0]),
				h = Double.parseDouble(cdim[1]);
		
		if (w != null && h != null) {
			return w / h;
		}
		
		JAlbum.logger.log(Level.FINE, "Invalid thumbnail dimensions: {0} x {1}", new Object[]{ w.toString(), h.toString() });
		return null;
	}
			
	private Date skinLoadedOn = new Date();
		
	private void allowHTMLEditing(JSmartTextArea ta) {
		try {
			ta.setAllowHTMLEditing(true).setFullHTMLEditing(true);
		} catch (Throwable e) {
		}
	}
	
	private void allowSpelling(JSmartTextArea ta) {
		try {
			ta.spelling();
		} catch (Throwable e) {
		}
	}
	
	private String majorVersion(String s) {
		int i = s.indexOf(".");
		if (i > 0) {
			return s.substring(0, i);
		}
		return s;
	}
	
	private int getMajorVersion(Object v) {
		
		if (v == null) {
			return -1;
		} else if (v instanceof Integer) {
			return (int)v;
		} else if (v instanceof String) {
			try {
				return Integer.valueOf(majorVersion((String)v));
			} catch(NumberFormatException ex) {
				return -1;
			}
		} else if (v instanceof Double) {
			return (int)Math.floor((Double)v);
		}
		return -1;
	}
	
	Boolean checkBooleanProperty(AlbumObject[] sel, String pn) {
		
		if (sel.length > 1) {
			boolean v = sel[0].getProperties().get(pn, false);
			
			for (AlbumObject ao : sel) {
				if (ao.getProperties().get(pn, false) != v) {
					return null;
				}
			}
			return v;
		} else if (sel.length == 1) {
			return sel[0].getProperties().get(pn, false);
		} else {
			return null;
		}
	}
	
	void createMenuItemForBooleanProperty(JMenu menu, String pn, String title) {
		
		createMenuItemForBooleanProperty(menu, window.albumExplorer.explorer.getSelectedAlbumObjects(), pn, title);
	}
	
	void createMenuItemForBooleanProperty(JMenu menu, AlbumObject[] sel, String pn, String title) {		
		Boolean cv = checkBooleanProperty(sel, pn);

		try {
			if (cv == null) {
				// Indefinite value -> Add submenu
				JMenu subMenu = new JMenu(title);
				JMenuItem all = new JMenuItem(getText("ui.all"));
				all.addActionListener(ae -> {
					for (AlbumObject ao : sel) {
						ao.getProperties().put(pn, true);
						ao.getProperties().save();
					}				
				});
				JMenuItem none = new JMenuItem(getText("ui.none"));
				none.addActionListener(ae -> {
					for (AlbumObject ao : sel) {
						ao.getProperties().put(pn, false);
						ao.getProperties().save();
					}				
				});
				subMenu.add(all);
				subMenu.add(none);
				menu.add(subMenu);
	
			} else {
				// All items has the same value -> invert them when clicked
				JCheckBoxMenuItem menuItem = new JCheckBoxMenuItem(title, null, cv);
	
				menuItem.addActionListener(ae -> {
					for (AlbumObject ao : sel) {
						ao.getProperties().put(pn, !cv);
						ao.getProperties().save();
					}
				});
	
				menu.add(menuItem);
			}
			
		} catch (Throwable ex) {
			JAlbum.logger.log(Level.INFO, "Right-click menu is available only in jAlbum v32 and newer!");
		}
	}
	
	JLinkLabel safeLinkLabel(javax.swing.JComponent comp, String url, String txt) {
		
		if (jalbumVersion.compareTo(new VersionNumber("32.1.4")) >= 0) {
			try {
				return new JLinkLabel(comp, txt);
			} catch (Throwable ex) {
				
			}
		}
		
		return new JLinkLabel(url, txt);
	}
	
	/*****************************************
	 *
	 *			Skin UI starts here
	 *
	 *****************************************/

	ControlPanel skinUi = new ControlPanel() {
		
		// UI-wide variables
		
		// Icons used multple places
		private Icon infoIcon = icon("info");
		private Icon mandatory = svgIcon("asterisk", new Dimension(16, 16));
		
		// silence further notifications
		private boolean skinChangeReported = false;
						
		private void selectComboBoxItem(JComboBox<Item> cb, JTextArea ta) {
			selectComboBoxItem(cb, ta.getText());
		}
		
		private void selectComboBoxItem(JComboBox<Item> cb, String val) {
			int size = cb.getModel().getSize();
			
			if (size == 0) {
				return;
			}
			
			ComboBoxModel<Item> model = cb.getModel();
			
			if (val != null) {
				for (int i = 0; i < size; i++) {
					if (model.getElementAt(i).value.equals(val)) {
						//System.out.println("Found preset " + i + " = \"" + val + "\"");
						cb.setSelectedIndex(i);
						return;
					}
				}
			}
			
			//System.out.println("Not found preset \"" + val + "\"");
			cb.setSelectedIndex(model.getSize() - 1);
		}
		
		// styleName is stored as skin variable in order to check style change later
		private JTextField styleName = new JTextField(getStyleName());
		
		// skinVersion is stored as skin variable in order to check major skin version change later
		JTextField majorSkinVersion = new JTextField(majorVersion(skinVer));
		
		// Tracking extra size changes in order to be able to remove the stale files
		JTextField extraSizes = new JTextField(24);
		String lastExtraSizes = (String)engine.getSkinVariables().get("extraSizes");
		JCheckBox extraSizesChanged = new JCheckBox("", false);
		/*
		JTextField folderThumbSize = new JTextField("320x240");
		JTextField folderImageSize = new JTextField("1000x240");
		*/
		// Background
		JTextField backgroundImageName = new JSmartTextField(10);
		JComboBox backgroundPos = new JComboBox(getPosition());
		JComboBox backgroundRepeat = new JComboBox(new Object[] {
			new Item("no-repeat", getText("ui.no_repeat")),
			new Item("repeat-x", getText("ui.repeat_x")),
			new Item("repeat-y", getText("ui.repeat_y")),
			new Item("repeat", getText("ui.repeat_both")),
			new Item("stretch", getText("ui.stretch")),
		});
		
		// Text, button
		JColorSelector textColor = new JColorSelector(getText("ui.textColor"), new JSmartTextField(10));
		JColorSelector linkColor = new JColorSelector(getText("ui.linkColor"), new JSmartTextField(10));
		JColorSelector hoverColor = new JColorSelector(getText("ui.hoverColor"), new JSmartTextField(10));
		JColorSelector backgroundColor = new JAlphaColorSelector(getText("ui.backgroundColor"), new JSmartTextField(10));
		JColorSelector footerColor = new JAlphaColorSelector(getText("ui.footer"), new JSmartTextField(10));
		
		// Top bar		
		JColorSelector topbarBackgroundColor = new JAlphaColorSelector(getText("ui.background") + ": " + getText("ui.topBar"), new JSmartTextField(10));
		
		JComboBox topbarDesign = new JComboBox(new Object[] {
			new Item("", getText("ui.none")),
			new Item("gradient", getText("ui.gradient")),
			new Item("recessed", getText("ui.recessed")),
			new Item("barrel", getText("ui.barrel")),
			new Item("pipe", getText("ui.pipe")),
			new Item("shiny", getText("ui.shiny")),
			new Item("metal", getText("ui.metallic")),
		});
		
		JColorSelector titleBackgroundColor = new JAlphaColorSelector(getText("ui.background") + ": " + getText("ui.title"), new JSmartTextField(10));
		
		JComboBox titleDesign = new JComboBox(new Object[] {
			new Item("plain", getText("ui.plain")),
			new Item("border", getText("ui.border")),
			new Item("recessed", getText("ui.recessed")),
			new Item("border-recessed", getText("ui.borderAndRecessed")),
			new Item("embossed", getText("ui.embossed")),
			new Item("shiny", getText("ui.shiny")),
			new Item("shiny-recessed", getText("ui.shinyAndRecessed")),
			new Item("glass", getText("ui.glass")),
			new Item("metal", getText("ui.metallic")),
			new Item("aqua", getText("ui.aqua")),
		});
		
		JComboBox titleRounding = new JComboBox(new Object[] {
			new Item("0", getText("ui.square")),
			new Item("5", getText("ui.slightlyRounded")),
			new Item("10", getText("ui.rounded")),
			new Item("32", getText("ui.round"))
		});
		
		JSpinner titleBorderWidth = new JSpinner(new SpinnerNumberModel(3, 0, 8, 1));
		JColorSelector titleBorderColor = new JAlphaColorSelector(getText("ui.borderColor"), new JSmartTextField(10));
		
		JCheckBox titleTextShadow = new JCheckBox(getText("ui.textShadow"));

		JColorSelector buttonBackgroundColor = new JAlphaColorSelector(getText("ui.background") + ": " + getText("ui.button"), new JSmartTextField(10));
		JColorSelector buttonHoverBackgroundColor = new JAlphaColorSelector(getText("ui.hover") + ": " + getText("ui.button"), new JSmartTextField(10));
		
		JComboBox buttonDesign = new JComboBox(new Object[] {
			new Item("plain", getText("ui.plain")),
			new Item("border", getText("ui.border")),
			new Item("recessed", getText("ui.recessed")),
			new Item("border-recessed", getText("ui.borderAndRecessed")),
			new Item("embossed", getText("ui.embossed")),
			new Item("shiny", getText("ui.shiny")),
			new Item("shiny-recessed", getText("ui.shinyAndRecessed")),
			new Item("shiny-border", getText("ui.shinyAndBorder")),
			new Item("glass", getText("ui.glass")),
			new Item("metal", getText("ui.metallic")),
			new Item("aqua", getText("ui.aqua")),
			new Item("protruding", getText("ui.protruding")),
			new Item("chiseled", getText("ui.chiseled")),
			
		});
		
		JComboBox buttonRounding = new JComboBox(new Object[] {
			new Item("0", getText("ui.square")),
			new Item("5", getText("ui.slightlyRounded")),
			new Item("10", getText("ui.rounded")),
			new Item("32", getText("ui.round")),
		});
		
		JSpinner buttonBorderWidth = new JSpinner(new SpinnerNumberModel(3, 0, 8, 1));
		JColorSelector buttonBorderColor = new JAlphaColorSelector(getText("ui.borderColor"), new JSmartTextField(10));
		
		JCheckBox showStartSlideshow = new JCheckBox(getText("ui.showStartSlideshow"), true);
		JCheckBox useSearch = new JCheckBox(getText("ui.useSearch"));

		// Boxes
		JColorSelector boxBackgroundColor = new JAlphaColorSelector(getText("ui.backgroundColor"), new JSmartTextField(10));
		
		JComboBox boxDesign = new JComboBox(new Object[] {
			new Item("plain", getText("ui.plain")),
			new Item("barrel", getText("ui.barrel")),
			new Item("embossed", getText("ui.embossed")),
			new Item("shiny", getText("ui.shiny")),
			new Item("metal", getText("ui.metallic")),
		});
		
		JCheckBox boxDropShadow = new JCheckBox(getText("ui.dropShadow"));
		JCheckBox fixedShapeThumbs = new JCheckBox(getText("ui.fixedShape"));
		JCheckBox diaFrame = new JCheckBox(getText("ui.diaFrameEffect"));
		JSpinner boxBorderWidth = new JSpinner(new SpinnerNumberModel(1, 0, 12, 1));
		JColorSelector boxBorderColor = new JAlphaColorSelector(getText("ui.borderColor"), new JSmartTextField(10));
		JSpinner cornerRadius = new JSpinner(new SpinnerNumberModel(6, 0, 100, 1));
		
		// Calculating proper thumb dimensions for the diaframe style
		private void checkThumbDimensions() {
			
			if (!isSkinReady()) {
				return;
			}
			
			String[] cdim = engine.getThumbSize().split("x");
			int tw, th;
			
			tw = Integer.parseInt(cdim[0]);
			th = Integer.parseInt(cdim[1]);
			
			if (Math.abs((float)tw/th - 1.33333) >= 0.05) {
				//JAlbum.logger.log(Level.FINE, "Aspect ratio: {0}", (float)tw/th);
				// Not 4:3 thumb aspect ratio
				th = (int)Math.round(tw * 0.75);
				
				// Ask for changing thumb dims
				Object[] options = { 
					getText("ui.yes"),
					getText("ui.no")
				};

				int n = JOptionPane.showOptionDialog(window, 
					getText("ui.changeThumbnailDimensionsQuestion").replace("{0}", tw + "x" + th), 
					getText("ui.updatingThumbnailDimensions"), 
					JOptionPane.YES_NO_OPTION, 
					JOptionPane.INFORMATION_MESSAGE,
					null,
					options,
					options[0]
				);

				if (n == 0) {
					try {
						window.ui2Engine();
						engine.setThumbSize(tw + "x" + th);
						JAlbum.logger.log(Level.FINE, "Thumb size has changed from {0}x{1}px to {2}x{3}px.", new Object[]{cdim[0], cdim[1], tw, th});
						window.engine2UI();
					} catch (ParameterException ex) {
						throw new RuntimeException(ex);
					}
				}
			}
		};

		JComboBox boxPadding = new JComboBox(new Object[] {
			new Item("none", getText("ui.none")),
			new Item("small", getText("ui.small")),
			new Item("medium", getText("ui.medium")),
			new Item("large", getText("ui.large")),
			new Item("x-large", getText("ui.xlarge"))
		});
		
		JColorSelector folderBehindColor = new JAlphaColorSelector(getText("ui.underColor"), new JSmartTextField(10));
		JColorSelector folderFrontColor = new JAlphaColorSelector(getText("ui.frontColor"), new JSmartTextField(10));

		// Caption templates

		JComboBox folderCaptionPresets = new JComboBox(new Object[] {
			new Item("<h3>${title}</h3>", getText("ui.title") + " (" + getText("ui.default") + ")"),
			new Item("<h3>${title}</h3>\n<div class=\"comment\">${comment}</div>", getText("ui.title") + " + " + getText("ui.comment")),
			new Item("<h3>${title}</h3>\n<div class=\"comment\">${commentShort}</div>", getText("ui.title") + " + " + getText("ui.commentShort")),
			new Item("<h3>${fileLabel}</h3>", getText("ui.fileLabel")),
			new Item("<h3>${fileLabel}</h3>\n<div class=\"comment\">${comment}</div>", getText("ui.fileLabel") + " + " + getText("ui.comment")),
			new Item(CUSTOM, "[ " + getText("ui.custom") + " ]")
		});
		JTextArea folderCaptionTemplate = new JSmartTextArea(4, 30);
		
		JComboBox pageCaptionPresets = new JComboBox(new Object[] {
			new Item("<h3>${fileLabel}</h3>", getText("ui.fileLabel") + " (" + getText("ui.default") + ")"),
			new Item("<h3>${fileLabel}</h3>\n<div class=\"comment\">${comment}</div>", getText("ui.fileLabel") + " + " + getText("ui.comment")),
			new Item("<h3>${fileLabel}</h3>\n<div class=\"comment\">${commentShort}</div>", getText("ui.fileLabel") + " + " + getText("ui.commentShort")),
			new Item(CUSTOM, "[ " + getText("ui.custom") + " ]")
		});
		JTextArea pageCaptionTemplate = new JSmartTextArea(4, 30);
		
		JTextArea thumbCaptionTemplate = new JSmartTextArea(4, 30);
		JComboBox thumbCaptionPresets = new JComboBox(new Object[] {
			new Item("", getText("ui.empty")),
			new Item("<div class=\"title\">${fileTitle}</div>", getText("ui.title")),
			new Item("<div class=\"title\">${fileTitle}</div>\n<div class=\"comment\">${comment}</div>", getText("ui.title") + " + " + getText("ui.comment")),
			new Item("<div class=\"title\">${fileTitle}</div>\n<div class=\"comment\">${commentShort}</div>", getText("ui.title") + " + " + getText("ui.commentShort") + " (" + getText("ui.default") + ")"),
			new Item("<span class=\"nr\">${imageNum}</span>\n<div class=\"title\">${fileTitle}</div>\n<div class=\"comment\">${commentShort}</div>", getText("ui.number") + " + " + getText("ui.title") + " + " + getText("ui.commentShort")),
			new Item("<div class=\"title\">${fileLabel}</div>", getText("ui.fileLabel")),
			new Item("<div class=\"title\">${fileLabel}</div>\n<div class=\"comment\">${comment}</div>", getText("ui.fileLabel") + " + " + getText("ui.comment")),
			new Item("<div class=\"title\">${fileLabel}</div>\n<div class=\"comment\">${commentShort}</div>", getText("ui.fileLabel") + " + " + getText("ui.commentShort")),
			new Item("<span class=\"nr\">${imageNum}</span>\n<div class=\"title\">${fileLabel}</div>\n<div class=\"comment\">${commentShort}</div>", getText("ui.number") + " + " + getText("ui.fileLabel") + " + " + getText("ui.commentShort")),
			new Item(CUSTOM, "[ " + getText("ui.custom") + " ]")
		});
		
		JComboBox titleCaptionPresets = new JComboBox(new Object[] {
			new Item("", getText("ui.empty")),
			new Item("<h3>${fileTitle}</h3>", getText("ui.title")),
			new Item("<span class=\"date\">${originalTime}</span>\n<h3>${fileTitle}</h3>", getText("ui.time") + " + " + getText("ui.title")),
			new Item("<h3>${fileLabel}</h3>", getText("ui.fileLabel") + " (" + getText("ui.default") + ")"),
			new Item("<span class=\"date\">${originalTime}</span>\n<h3>${fileLabel}</h3>", getText("ui.time") + " + " + getText("ui.fileLabel")),
			new Item(CUSTOM, "[ " + getText("ui.custom") + " ]")
		});
		JTextArea titleCaptionTemplate = new JSmartTextArea(6, 30);
		
		JComboBox imageCaptionPresets = new JComboBox(new Object[] {
			new Item("", getText("ui.empty")),
			new Item("<div class=\"comment\">${comment}</div>", getText("ui.comment") + " (" + getText("ui.default") + ")"),
			new Item("<span>${originalTime}</span>\n<div class=\"comment\">${comment}</div>", getText("ui.time") + " + " + getText("ui.comment")),
			new Item("<div class=\"title\">${fileTitle}</div>", getText("ui.title")),
			new Item("<div class=\"title\">${fileTitle}</div>\n<div class=\"comment\">${comment}</div>", getText("ui.title") + " + " + getText("ui.comment")),
			new Item("<div class=\"title\">${fileTitle}</div>\n<div class=\"comment\">${commentShort}</div>", getText("ui.title") + " + " + getText("ui.commentShort")),
			new Item("<span>${originalTime}</span>\n<div class=\"title\">${fileTitle}</div>\n<div class=\"comment\">${comment}</div>", getText("ui.time") + " + " + getText("ui.title") + " + " + getText("ui.comment")),
			new Item("<div class=\"title\">${fileLabel}</div>", getText("ui.fileLabel")),
			new Item("<div class=\"title\">${fileLabel}</div>\n<div class=\"comment\">${comment}</div>", getText("ui.fileLabel") + " + " + getText("ui.comment")),
			new Item("<div class=\"title\">${fileLabel}</div>\n<div class=\"comment\">${commentShort}</div>", getText("ui.fileLabel") + " + " + getText("ui.commentShort")),
			new Item(CUSTOM, "[ " + getText("ui.custom") + " ]")
		});
		JTextArea imageCaptionTemplate = new JSmartTextArea(6, 30);
		
		// Typography
		private final JComboBox fontSuggestions = new JComboBox(new Object[] {
			"[" + getText("ui.suggestedFonts") +  "]",
			"Barlow Condensed / Barlow Light",
			"Cinzel Decorative / Raleway",
			"Dosis Bold / Montserrat",
			"Exo 2 / Exo 2",
			"Fjalla One / Roboto",
			"IBM Plex Serif / IBM Plex Sans",
			"Jura / Roboto Condensed",
			"Martel / Roboto",
			"Merriweather / Merriweather Sans",
			"Oswald / Open Sans",
			"Raleway / Open Sans",
			"Roboto Condensed / Roboto",
			"Roboto Slab / Montserrat",
			"Special Elite / Yanone Kaffeesatz",
			"Unica One / Lato",
			"Yanone Kaffeesatz Bold / Muli"
		});

		private void setFontBoxes() {
			
			if (fontSuggestions.getSelectedIndex() == 0) {
				return;
			}
			String	s = fontSuggestions.getSelectedItem().toString(),
					hf = s.split("/")[0].trim(),
					ff = s.split("/")[1].trim();

			setSelectedValue(fontFamily, ff);
			setSelectedValue(headlineFont, hf);
			fontSuggestions.setSelectedIndex(0);
		};
		
		JComboBox fontFamily = new JComboBox(new Object[] {
			new Item("Arial, \"Helvetica Neue\", Helvetica, sans-serif", "Arial"),
			new Item("Baskerville, \"Baskerville Old Face\", \"Hoefler Text\", Garamond, \"Times New Roman\", serif", "Baskerville"),
			new Item("Calibri, Candara, Segoe, \"Segoe UI\", Optima, Arial, sans-serif", "Calibri"),
			new Item("Cambria, Georgia, Times, \"Times New Roman\", serif", "Cambria"),
			new Item("\"Century Gothic\", \"Apple Gothic\", \"Goudy Old Style\", sans-serif", "Century Gothic"),
			new Item("\"Comic Sans MS\", cursive", "Comic Sans"),
			new Item("Consolas, \"Lucida Console\", Monaco, monospace", "Consolas"),
			new Item("Constantia, Palatino, \"Palatino Linotype\", \"Palatino LT STD\", Georgia, serif", "Constantia"),
			new Item("\"Copperplate Light\", \"Copperplate Gothic Light\", serif", "Copperplate Light"),
			new Item("\"Courier New\", Courier, monospace", "Courier New"),
			new Item("\"Franklin Gothic Medium\", \"Arial Narrow Bold\", Arial, sans-serif", "Franklin Gothic"),
			new Item("Futura, \"Century Gothic\", AppleGothic, sans-serif", "Futura"),
			new Item("Garamond, \"Hoefler Text\", Times New Roman, Times, serif", "Garamond"),
			new Item("Geneva, \"Lucida Sans\", \"Lucida Grande\", \"Lucida Sans Unicode\", Verdana, sans-serif", "Geneva"),
			new Item("Georgia, Palatino, \"Palatino Linotype\", Times, \"Times New Roman\", serif", "Georgia"),
			new Item("\"Gill Sans\", \"Gill Sans MT\", Calibri, \"Trebuchet MS\", sans-serif", "Gill Sans"),
			new Item("\"Goudy Old Style\", Garamond, \"Big Caslon\", \"Times New Roman\", serif", "Goudy Old Style"),
			new Item("\"Helvetica Neue\", Helvetica, Arial, sans-serif", "Helvetica Neue"),
			new Item("\"Hoefler Text\", Constantia, Palatino, \"Palatino Linotype\", \"Book Antiqua\", Georgia, serif", "Hoefler Text"),
			new Item("Impact, Haettenschweiler, \"Arial Narrow Bold\", sans-serif", "Impact"),
			new Item("\"Lucida Sans\", \"Lucida Grande\", \"Lucida Sans Unicode\", Verdana, sans-serif", "Lucida Sans"),
			new Item("\"Lucida Bright\", Georgia, serif", "Lucida Bright"),
			new Item("Palatino, \"Palatino Linotype\", \"Book Antiqua\", Georgia, serif", "Palatino"),
			new Item("Segoe, \"Segoe UI\", Tahoma, Geneva, \"Nimbus Sans L\", sans-serif", "Segoe"),
			new Item("Tahoma, Geneva, Verdana, sans-serif", "Tahoma"),
			new Item("Times, \"Times New Roman\", Georgia, serif", "Times"),
			new Item("\"Trebuchet MS\", \"Lucida Sans Unicode\", \"Lucida Grande\", \"Lucida Sans\", Tahoma, sans-serif", "Trebuchet MS"),
			new Item("Verdana, Geneva, Tahoma, sans-serif", "Verdana"),
			new Item("Alegreya Sans:300,300i,500,500i", "Alegreya Sans Light"),
			new Item("Barlow:300,300i,600,600i", "Barlow Light"),
			new Item("Barlow:400,400i,700,700i", "Barlow"),
			new Item("Barlow Semi Condensed:300,300i,500,500i", "Barlow Semi Condensed"),
			new Item("Exo 2:300,300i,600,600i", "Exo 2"),
			new Item("Fira Sans:300,300i,600,600i", "Fira Sans"),
			new Item("IBM Plex Sans:300,300i,600,600i", "IBM Plex Sans"),
			new Item("Josefin Sans:300,300i,600,600i", "Josefin Sans"),
			new Item("Josefin Slab:300,300i,600,600i", "Josefin Slab"),
			new Item("Lato:300,300i,700,700i", "Lato"),
			new Item("Merriweather Sans:300,300i,700,700i", "Merriweather Sans"),
			new Item("Merriweather:300,300i,700,700i", "Merriweather Serif"),
			new Item("Montserrat:300,300i,600,600i", "Montserrat"),
			new Item("Muli:300,300i,700,700i", "Muli"),
			new Item("Open Sans:300,300i,600,600i", "Open Sans"),
			new Item("Raleway:300,300i,600,600i", "Raleway"),
			new Item("Roboto:300,300i,500,500i", "Roboto"),
			new Item("Roboto Condensed:300,300i,700,700i", "Roboto Condensed"),
			new Item("Saira:300,300i,600,600i", "Saira"),
			new Item("Source Sans Pro:300,300i,600,600i", "Source Sans Pro"),
			new Item("Work Sans:300,300i,600,600i", "Work Sans Light"),
			new Item("Yanone Kaffeesatz:300,500", "Yanone Kaffeesatz")
		});
		
		JComboBox headlineFont = new JComboBox(new Object[] {
			new Item("", "[ " + getText("ui.sameAsBaseFont") + " ]"),
			new Item("Abril Fatface", "Abril Fatface"),
			new Item("Amatic SC:700", "Amatic SC Bold"),
			new Item("Arapey", "Arapey"),
			new Item("Barlow:300", "Barlow Light"),
			new Item("Barlow:500", "Barlow Medium"),
			new Item("Barlow Semi Condensed:400", "Barlow Semi Condensed"),
			new Item("Barlow Condensed", "Barlow Condensed"),
			new Item("Bebas Neue", "Bebas Neue"),
			new Item("Cantata One", "Cantata One"),
			new Item("Cookie", "Cookie"),
			new Item("Dancing Script", "Dancing Script"),
			new Item("Economica", "Economica"),
			new Item("Exo 2:300", "Exo 2"),
			new Item("Fauna One", "Fauna One"),
			new Item("Fjalla One", "Fjalla One"),
			new Item("Geo", "Geo"),
			new Item("Gilda Display", "Gilda Display"),
			new Item("Grand Hotel", "Grand Hotel"),
			new Item("Gruppo", "Gruppo"),
			new Item("Handlee", "Handlee"),
			new Item("IBM Plex Serif:300", "IBM Plex Serif"),
			new Item("Italiana", "Italiana"),
			new Item("Josefin Slab", "Josefin Slab"),
			new Item("Julius Sans One", "Julius Sans One"),
			new Item("Jura", "Jura"),
			new Item("Libre Baskerville", "Libre Baskerville"),
			new Item("Lobster", "Lobster"),
			new Item("Lobster Two", "Lobster Two"),
			new Item("Lora", "Lora"),
			new Item("Martel:300", "Martel Light"),
			new Item("Marvel:700", "Marvel Bold"),
			new Item("Medula One", "Medula One"),
			new Item("Merriweather:300", "Merriweather"),
			new Item("Noticia Text", "Noticia Text"),
			new Item("Noto Serif Display:300", "Noto Serif Light"),
			new Item("Oranienbaum", "Oranienbaum"),
			new Item("Oswald", "Oswald"),
			new Item("Philosopher", "Philosopher"),
			new Item("Poiret One", "Poiret One"),
			new Item("Prata", "Prata"),
			new Item("PT Mono", "PT Mono"),
			new Item("Raleway:300", "Raleway"),
			new Item("Raleway:600", "Raleway Bold"),
			new Item("Roboto Condensed", "Roboto Condensed"),
			new Item("Roboto Slab:300", "Roboto Slab"),
			new Item("Roboto Slab:600", "Roboto Slab Bold"),
			new Item("Rochester", "Rochester"),
			new Item("Saira", "Saira"),
			new Item("Scope One", "Scope One"),
			new Item("Smooch Sans", "Smooch Sans"),
			new Item("Sofia", "Sofia"),
			new Item("Special Elite", "Special Elite"),
			new Item("Squada One", "Squada One"),
			new Item("Strait", "Strait"),
			new Item("Unica One", "Unica One"),
			new Item("Vidaloka", "Vidaloka"),
			new Item("Work Sans", "Work Sans"),
			new Item("Yanone Kaffeesatz:300", "Yanone Kaffeesatz"),
			new Item("Yanone Kaffeesatz:500", "Yanone Kaffeesatz Bold")
		});
		
		// Pages
		JCheckBox showPages = new JCheckBox(getText("ui.showPages"));

		// Weblocaitons
		JCheckBox showWeblocations = new JCheckBox(getText("ui.showWebLocations"));

		JComboBox thumbGap = new JComboBox(new Object[] {
			new Item("none", getText("ui.none")),
			new Item("tiny", getText("ui.thin")),
			new Item("small", getText("ui.small")),
			new Item("medium", getText("ui.medium")),
			new Item("large", getText("ui.large"))
		});

		private void setMakeSlides() {
			boolean ov = engine.isSlides(),
					nv = (showShare.isSelected() &&				// Share individual slides?
							(shareFacebook.isSelected() || shareThreads.isSelected() || shareBluesky.isSelected())) 
						|| 
						(writeSitemapXml.isSelected() && sitemapIncludeSlides.isSelected());
			
			if (ov != nv) {
				try {
					window.ui2Engine();
					engine.setSlides(nv);
					window.engine2UI();
				} catch (Exception ex) {
					throw new RuntimeException(ex);
				}
			}
		};
		
		private String lastThumbDimSuggestion = "";
		
		// Functionality-related variables
		
		// Lightbox global values
		
		JCheckBox lightboxUseMainBg = new JCheckBox(getText("ui.lightboxUseMainBackground"));
		JColorSelector lightboxBackgroundColor = new JAlphaColorSelector(getText("ui.backgroundColor"), new JSmartTextField("#e5111111", 10));
		JColorSelector lightboxBorderColor = new JAlphaColorSelector(getText("ui.borderColor"), new JSmartTextField("#eeeeee", 10));
		JSpinner lightboxBorderWidth = new JSpinner(new SpinnerNumberModel(10, 0, 100, 1));
		JSpinner imgCornerRadius = new JSpinner(new SpinnerNumberModel(0, 0, 200, 1));
		JSpinner fitPadding = new JSpinner(new SpinnerNumberModel(15, 0, 100, 1));
		JCheckBox useThumbstrip = new JCheckBox(getText("ui.useThumbstrip"));
		JComboBox thumbstripPosition = new JComboBox(new Object[] {
			new Item("above", getText("ui.above")), 
			new Item("bothsides", getText("ui.bothSides")), 
			new Item("below", getText("ui.below"))
		});
		JCheckBox thumbstripVisible = new JCheckBox(getText("ui.thumbstripVisibleByDefault"));
		JComboBox thumbstripHeight = new JComboBox(new Object[] {
			new Item("60", getText("ui.tiny")), 
			new Item("80", getText("ui.small")), 
			new Item("110", getText("ui.medium")), 
			new Item("140", getText("ui.large"))
		});
		JCheckBox showFitToggle = new JCheckBox(getText("ui.showFitToggle"));
		JCheckBox zoomSlider = new JCheckBox(getText("ui.useZoomSlider"));
		JCheckBox showStartStop = new JCheckBox(getText("ui.showStartStop"));
		JCheckBox showFullscreen = new JCheckBox(getText("ui.showFullScreenButton"));
		
		JCheckBox fitImage = new JCheckBox(getText("ui.fitImagesByDefault"));
		JSpinner maxZoom = new JSpinner(new SpinnerNumberModel(1.4, 1.0, 3.0, 0.1));
		JCheckBox displayOriginals = new JCheckBox(getText("ui.displayOriginals"));
		JCheckBox showImageNumbers = new JCheckBox(getText("ui.showImageNumbers"));
		JCheckBox captionVisible = new JCheckBox(getText("ui.captionVisibleByDefault"));

		JCheckBox showMapBtn = new JCheckBox(getText("ui.map"));
		JCheckBox showShopBtn = new JCheckBox(getText("ui.shoppingCart"));
		JCheckBox showFeedbackBtn = new JCheckBox(getText("ui.feedback"));
		JCheckBox showRegions = new JCheckBox(getText("ui.regions"));
		JCheckBox showShare = new JCheckBox(getText("ui.share"));
		JCheckBox printImageButton = new JCheckBox(getText("ui.print"));
		JCheckBox downloadBtn = new JCheckBox(getText("ui.download"));
		JCheckBox showPhotoData = new JCheckBox(getText("ui.photoData"));
				
		// Advanced tab's global vars
		
		JCheckBox shareFacebook = new JCheckBox("Facebook");
		JCheckBox shareThreads = new JCheckBox("Threads");
		JCheckBox shareBluesky = new JCheckBox("Bluesky");
		JCheckBox writeSitemapXml = new JCheckBox(getText("ui.createSitemapXml"));
		JCheckBox sitemapIncludeSlides = new JCheckBox(getText("ui.includeSlides"), true);
		
		private PropertyChangeListener setupMonitors = pce -> {
			
			JAlbum.logger.log(Level.FINE, "Setting up state monitors");
			
			StateMonitor.monitoring(fontSuggestions).onChange(src -> {
				if (isSkinReady() && src != null) {
					setFontBoxes();
				}
			});
			
			StateMonitor.monitoring(extraSizes).onChange(src -> {
				extraSizesChanged.setSelected(lastExtraSizes == null || !lastExtraSizes.equals(extraSizes.getText()));
			});
			/*
			// Thumb size change
			StateMonitor.monitoring(
					maxPageWidth,
					thumbCols,
					thumbGap,
					boxPadding,
					boxBorderWidth,
					thumbAspectRatio).onChange(src -> {

				JAlbum.logger.log(Level.FINE, "stateMonitor: \"{0}\" changed", src);
				
				if (isSkinReady() && src != null) {
					setThumbDimensions();
				}

			});
			*/
			// Separate slide page requirement
			StateMonitor.monitoring(
					shareFacebook,
					shareThreads,
					shareBluesky,
					showShare,
					writeSitemapXml,
					sitemapIncludeSlides).onChange(src -> {
				setMakeSlides();
			});
			
		};
			
		/*	---------------------------------------------------------------
								Design tab
			--------------------------------------------------------------- */
		
		ControlPanel generalDesign = new ControlPanel() {

			JComboBox<Item> language = new JComboBox<Item>() {
				{
					setModel(Languages.modelFrom(new File(skinDirectory, "texts")));
					insertItemAt(new Item("jalbum", "[ " + getText("ui.jAlbumPreference") + " ]"), 0);
					setSelectedIndex(0);
				}
			};
				
			private JButton selectImage = new JButton();
			
			JComboBox modalWindowsTheme = new JComboBox(new Object[] {
				new Item("auto", getText("ui.auto")),
				new Item("light", getText("ui.light")),
				new Item("dark", getText("ui.dark"))
			});
			
			JCheckBox scrollToTopButton = new JCheckBox(getText("ui.scrollToTopButton"));	
				
			JComboBox iconStyle = new JComboBox(new Object[] {
				new Item("thin", getText("ui.thin")),
				new Item("fat", getText("ui.fat"))
			});
			

			{
				language.setToolTipText(getText("ui.languageInfo"));
				selectImage.setText(getText("ui.select"));
				selectImage.addActionListener(new ActionListener() { 
					@Override
					public void actionPerformed(ActionEvent e) {
						getFileToRes(imageFiles, backgroundImageName, skinUi);
				}});
				backgroundColor.getTextComponent().setFont(mono);
				backgroundColor.setToolTipText(getText("ui.backgroundColorInfo"));
				footerColor.getTextComponent().setFont(mono);
				boxBackgroundColor.getTextComponent().setFont(mono);
				textColor.getTextComponent().setFont(mono);
				linkColor.getTextComponent().setFont(mono);
				hoverColor.getTextComponent().setFont(mono);
				
				scrollToTopButton.setToolTipText(getText("ui.scrollToTopButtonInfo"));

				add("", new JLabelFor(getText("ui.language"), language));
				add("tab", language);
				add("br", new JLabelFor(getText("ui.backgroundImage"), backgroundImageName));
				add("tab", backgroundImageName);
				add(" ", selectImage);
				add("br", new JLabelFor(getText("ui.position"), backgroundPos));
				add("tab", backgroundPos);
				add("br tab", backgroundRepeat);
				add("br", new JLabelFor(getText("ui.backgroundColor"), backgroundColor));
				add("tab", backgroundColor);
				add("tab", backgroundColor.getTextComponent());
				add("br", new JLabelFor(getText("ui.textColor"), textColor));
				add("tab", textColor);
				add("tab", textColor.getTextComponent());
				add("br", new JLabelFor(getText("ui.linkColor"), linkColor));
				add("tab", linkColor);
				add("tab", linkColor.getTextComponent());
				add("br", new JLabelFor(getText("ui.hoverColor"), hoverColor));
				add("tab", hoverColor);
				add("tab", hoverColor.getTextComponent());
				add("br", new JLabelFor(getText("ui.footer"), footerColor));
				add("tab", footerColor);
				add("tab", footerColor.getTextComponent());
				add("br", new JLabelFor(getText("ui.modalWindowsTheme"), modalWindowsTheme));
				add("tab", modalWindowsTheme);
				add("br", new JLabel(getText("ui.iconStyle")));
				add("tab", iconStyle);
				add("br", scrollToTopButton);
					
				putClientProperty("helpPage", helpRoot + "/Site/Design");
			}
		};

		// Typography
		
		ControlPanel typography = new ControlPanel() {

			JComboBox baseFontSize = new JComboBox(new Object[] {
				new Item("11", getText("ui.smallest") + " (11px)"),
				new Item("12", getText("ui.tiny") + " (12px)"),
				new Item("13", getText("ui.small") + " (13px)"),
				new Item("14", getText("ui.smaller") + " (14px)"),
				new Item("15", getText("ui.normal") + " (15px)"),
				new Item("16", getText("ui.normal") + " (16px)"),
				new Item("17", getText("ui.larger") + " (17px)"),
				new Item("18", getText("ui.large") + " (18px)"),
				new Item("20", getText("ui.largest") + " (20px)"),
			});
								
			{
				headlineFont.setEditable(true);

				add(new JLabelFor(getText("ui.pairingSuggestions"), fontSuggestions));
				add("tab", fontSuggestions);
				add("br", new JLabelFor(getText("ui.fontFamily"), fontFamily));
				add("tab", fontFamily);
				add("br", new JLabelFor(getText("ui.baseFontSize"), baseFontSize));
				add("tab", baseFontSize);
				add("br", new JLabelFor(getText("ui.headlineFont"), headlineFont));
				add("tab", headlineFont);
				
				putClientProperty("helpPage", helpRoot + "/Site/Typography");
			}
		};

		// Top bar
		
		ControlPanel topbar = new ControlPanel() {

			JTextField logoName = new JSmartTextField(24);
			private JButton selectLogo = new JButton();
			JComboBox logoLinking = new JComboBox(new Object[] {
				new Item("toplevel", getText("ui.topLevelPage")),
				new Item("external", getText("ui.externalHomePage")),
				new Item("none", getText("ui.none")),
			});
			private JButton selectTopbarBackgroundImage = new JButton();
			JComboBox showBreadcrumbPath = new JComboBox(new Object[]{
				new Item("none", getText("ui.none")),
				new Item("inline", getText("ui.alongWithTitle")),
				new Item("tooltip", getText("ui.asTooltip"))
			});
			JCheckBox alwaysRestartSlideshow = new JCheckBox(getText("ui.alwaysRestart"), false);
				
			ControlPanel titleDesignPanel = new ControlPanel(getText("ui.title")) {
				
				JLabel borderLabel = new JLabel(getText("ui.border"));
				
				{
					titleDesign.addActionListener(new ActionListener() {
						@Override
						public void actionPerformed(ActionEvent e) {
							int tdi = titleDesign.getSelectedIndex();
							boolean hasBorder = tdi == 1 || tdi == 3;
							borderLabel.setVisible(hasBorder);
							titleBorderColor.setVisible(hasBorder);
							titleBorderColor.getTextComponent().setVisible(hasBorder);
							titleBorderWidth.setVisible(hasBorder);
					}});
					titleBackgroundColor.getTextComponent().setFont(mono);
					titleBorderColor.getTextComponent().setFont(mono);
					showBreadcrumbPath.setToolTipText(getText("ui.showBreadcrumbPathInfo"));
					
					add(new JLabel(getText("ui.design")));
					add("tab", titleDesign);				
					add("", titleRounding);
					add("br", new JLabelFor(getText("ui.backgroundColor"), titleBackgroundColor));
					add("tab", titleBackgroundColor);
					add("tab", titleBackgroundColor.getTextComponent());
					add("br", borderLabel);
					add("tab", titleBorderColor);
					add("tab", titleBorderColor.getTextComponent());
					add("", titleBorderWidth);
					add("br", new JLabelFor(getText("ui.showBreadcrumbPath"), showBreadcrumbPath));
					add("tab", showBreadcrumbPath);
				}
			};

			ControlPanel buttonDesignPanel = new ControlPanel(getText("ui.buttons")) {
				
				JLabel borderLabel = new JLabel(getText("ui.border"));
				
				{
					buttonDesign.addActionListener(new ActionListener() {
						@Override
						public void actionPerformed(ActionEvent e) {
							int bdi = buttonDesign.getSelectedIndex();
							boolean hasBorder = bdi == 1 || bdi == 3 || bdi == 7;
							borderLabel.setVisible(hasBorder);
							buttonBorderColor.setVisible(hasBorder);
							buttonBorderColor.getTextComponent().setVisible(hasBorder);
							buttonBorderWidth.setVisible(hasBorder);
					}});
					buttonBackgroundColor.getTextComponent().setFont(mono);
					buttonHoverBackgroundColor.getTextComponent().setFont(mono);
					buttonBorderColor.getTextComponent().setFont(mono);
					showStartSlideshow.setToolTipText(getText("ui.showStartSlideshowInfo"));
					ComponentUtilities.whenSelectedEnable(showStartSlideshow, new JComponent[] { alwaysRestartSlideshow });
					
					add(new JLabel(getText("ui.design")));
					add("tab", buttonDesign);
					add("", buttonRounding);
					add("br", new JLabelFor(getText("ui.backgroundColor"), buttonBackgroundColor));
					add("tab", buttonBackgroundColor);
					add("", buttonBackgroundColor.getTextComponent());
					add("br", borderLabel);
					add("tab", buttonBorderColor);
					add("", buttonBorderColor.getTextComponent());
					add("", buttonBorderWidth);
					add("br", new JLabelFor(getText("ui.hoverColor"), buttonHoverBackgroundColor));
					add("tab", buttonHoverBackgroundColor);
					add("", buttonHoverBackgroundColor.getTextComponent());
					add("br", showStartSlideshow);
					add(" ", alwaysRestartSlideshow);
				}
			};
			
			{
				
				
				selectLogo.setText(getText("ui.select"));
				selectLogo.addActionListener(new ActionListener() { 
					@Override
					public void actionPerformed(ActionEvent e) {
						getFileToRes(imageFiles, logoName, skinUi);
				}});

				topbarBackgroundColor.getTextComponent().setFont(mono);

				add(new JLabel(getText("ui.design")));
				add("tab", topbarDesign);
				add("br", new JLabelFor(getText("ui.backgroundColor"), topbarBackgroundColor));
				add("tab", topbarBackgroundColor);
				add("tab", topbarBackgroundColor.getTextComponent());
				add("tab", titleTextShadow);
				add("br", new JLabelFor(getText("ui.logo"), logoName));
				add("tab", logoName);
				add("", selectLogo);
				add("br", new JLabel(getText("ui.linkTo")));
				add("tab", logoLinking);
				add("br hfill", titleDesignPanel);
				add("br hfill", buttonDesignPanel);
				
				putClientProperty("helpPage", helpRoot + "/Site/Top_bar");
			}
		};

		// Thumbnail design
		
		ControlPanel thumbnails = new ControlPanel() {

			ControlPanel folderDesignPanel = new ControlPanel(getText("ui.folders")) {
				
				{
					folderBehindColor.getTextComponent().setFont(mono);
					folderFrontColor.getTextComponent().setFont(mono);
					
					add(new JLabelFor(getText("ui.frontColor"), folderFrontColor));
					add("tab", folderFrontColor);
					add("tab", folderFrontColor.getTextComponent());
					add("br", new JLabelFor(getText("ui.underColor"), folderBehindColor));
					add("tab", folderBehindColor);
					add("tab", folderBehindColor.getTextComponent());
				}
			};
			/*	
			JLabel arLabel = new JLabel(getText("ui.aspectRatio"));
			
			ControlPanel thumbLayoutPanel = new ControlPanel(getText("ui.layout")) {
				
				JLabel columnsInfo = new JLabel(infoIcon);
				
				{
					columnsInfo.addMouseListener(new MouseAdapter() {  
						@Override
						public void mouseReleased(MouseEvent e) {
							WrappableJLabel lbl = new WrappableJLabel("<html><p>" + getText("ui.columnsInfo") + "</p><br><p>" + getText("ui.masonryColumnsInfo") + "</p></html>");
							lbl.setPreferredWidth(360);
							JOptionPane.showMessageDialog(window, lbl, getText("ui.thumbCols"), JOptionPane.INFORMATION_MESSAGE);
					}});
					
					add("", new JLabelFor(getText("ui.thumbCols"), thumbCols));
					add("tab", thumbCols);
					add("", columnsInfo);
					add("br", arLabel);
					add("tab", thumbAspectRatio);
				}
			};
			*/	
			ControlPanel frameDesignPanel = new ControlPanel(getText("ui.frames")) {
				
				JCheckBox hoverEffectThumbs = new JCheckBox(getText("ui.mouseOverEffect"), true);
				
				{
					boxBorderColor.getTextComponent().setFont(mono);
					//thumbCols.setToolTipText(getText("ui.thumbColsInfo"));
					cornerRadius.setToolTipText(getText("ui.cornerRadiusInfo"));
					fixedShapeThumbs.setToolTipText(getText("ui.fixedShapeThumbsInfo"));
					new StateMonitor() {
						public void onChange() {
							if (fixedShapeThumbs.isSelected()) {								
								diaFrame.setEnabled(true);
								if (diaFrame.isSelected()) {
									checkThumbDimensions();
								}
							} else {
								diaFrame.setEnabled(false);
							}
						}
					}.add(fixedShapeThumbs).add(diaFrame).done();
					diaFrame.setToolTipText(getText("ui.diaFrameInfo"));
					hoverEffectThumbs.setToolTipText(getText("ui.mouseOverEffectInfo"));
					
					add("", new JLabelFor(getText("ui.design")));
					add("tab", boxDesign);
					//add("br tab", bgImageControls);
					add("br", new JLabel(getText("ui.borderWidth")));
					add("tab", boxBorderWidth);
					add("br", new JLabel(getText("ui.borderColor")));
					add("tab", boxBorderColor);
					add("tab", boxBorderColor.getTextComponent());						
					add("br", boxDropShadow);
					add("br", fixedShapeThumbs);
					add("br", diaFrame);
					add("br", new JLabelFor(getText("ui.backgroundColor"), boxBackgroundColor));
					add("tab", boxBackgroundColor);
					add("tab", boxBackgroundColor.getTextComponent());
					add("br", new JLabel(getText("ui.cornerRadius")));
					add("tab", cornerRadius);
					add("", new JLabel("px"));
					add("br", new JLabelFor(getText("ui.padding"), boxPadding));
					add("tab", boxPadding);
					add("br", new JLabelFor(getText("ui.gap"), thumbGap));
					add("tab", thumbGap);
					add("br", hoverEffectThumbs);
				}
			};
			
			/*
			ControlPanel aspectRatioPanel = new ControlPanel(getText("ui.aspectRatio")) {

				JTextField maxHorizontalAR = new JTextField(8);
				JTextField minVerticalAR = new JTextField(8);

				{
					maxHorizontalAR.setToolTipText(getText("ui.aspectRatioInfo") + " (> 1)");
					minVerticalAR.setToolTipText(getText("ui.aspectRatioInfo") + " (< 1)");

					add("br", new JLabelFor(getText("ui.max"), maxHorizontalAR));
					add("", maxHorizontalAR);
					add("tab", new JLabelFor(getText("ui.min"), minVerticalAR));
					add("", minVerticalAR);

				}
			};
			*/
			/*
			ControlPanel bgImageControls = new ControlPanel() {

				private JButton selectBoxBackgroundImage = new JButton();

				{
					selectBoxBackgroundImage.setText(getText("ui.select"));
					selectBoxBackgroundImage.addActionListener(new ActionListener() { 
						@Override
						public void actionPerformed(ActionEvent e) {
							getFileToRes(imageFiles, boxBackgroundImageName, skinUi);
					}});
			
					add("", new JLabelFor(getText("ui.backgroundImage"), boxBackgroundImageName));
					add("tab", boxBackgroundImageName);
					add(" ", selectBoxBackgroundImage);

				}
			};
			*/

			{				
				//add("hfill", thumbLayoutPanel);
				add("hfill", folderDesignPanel);
				add("br hfill", frameDesignPanel);
				//add("br hfill", aspectRatioPanel);
			}
		};
		
		ControlPanel thumbnailCaptions = new ControlPanel() {
			
			ControlPanel folderCaption = new ControlPanel(getText("ui.folders")) {
				/*			
				JComboBox folderDateSource = new JComboBox(new Object[]{
					new Item("none", getText("ui.none")),
					new Item("fileDate", getText("ui.fileDate")),
					new Item("folderModDate", getText("ui.folderModDate")),
					new Item("lastCameraDate", getText("ui.lastCameraDate")),
					new Item("cameraDateRange", getText("ui.cameraDateRange"))
				});
				*/
				JScrollPane captionPane = new JScrollPane(folderCaptionTemplate);
				JCheckBox showFolderImageCount = new JCheckBox(getText("ui.showFolderImageCount"), true);

				{
					folderCaptionPresets.addItemListener(listener -> {
						if (isSkinReady()) {
							String s = getSelectedValue(folderCaptionPresets);
							if (!s.equals(CUSTOM)) {
								folderCaptionTemplate.setText(s);
							}
						}
					});	
					new StateMonitor() {
						public void onChange() {
							String s = getSelectedValue(folderCaptionPresets);
							captionPane.setVisible(s.equals(CUSTOM));
						}
					}.add(folderCaptionPresets).done();

					folderCaptionTemplate.setEditable(true);
					folderCaptionTemplate.setLineWrap(true);
					folderCaptionTemplate.setWrapStyleWord(true);
					folderCaptionTemplate.setFont(mono);
					folderCaptionTemplate.setTabSize(2);
					captionPane.setBorder(BorderFactory.createTitledBorder(getText("ui.captionTemplate")));
					
					add("", new JLabel(getText("ui.caption")));
					add("tab", folderCaptionPresets);
					add("br hfill", captionPane);
					add("br", showFolderImageCount);
				}		
			};
			
			ControlPanel pageCaption = new ControlPanel(getText("ui.pages")) {
				
				JScrollPane captionPane = new JScrollPane(pageCaptionTemplate);
				
				{
					pageCaptionPresets.addItemListener(listener -> {
						if (isSkinReady()) {
							String s = getSelectedValue(pageCaptionPresets);
							if (!s.equals(CUSTOM)) {
								pageCaptionTemplate.setText(s);
							}
						}
					});	
					new StateMonitor() {
						public void onChange() {
							String s = getSelectedValue(pageCaptionPresets);
							captionPane.setVisible(s.equals(CUSTOM));
						}
					}.add(pageCaptionPresets).done();
					
					add("", showPages);
					add("br", new JLabel(getText("ui.caption")));
					add("tab", pageCaptionPresets);
					add("br hfill", captionPane);
				}
			};
			
			ControlPanel webLocationCaption = new ControlPanel(getText("ui.webLocations")) {
				
				JCheckBox webLocationOpenNew = new JCheckBox(getText("ui.webLocationsOpenInNewWindow"));
				
				{
					showWeblocations.setToolTipText(getText("ui.showWeblocationsInfo"));
					ComponentUtilities.whenSelectedEnable(showWeblocations, new JComponent[]{ webLocationOpenNew });					
					
					add("", showWeblocations);
					add("br", webLocationOpenNew);
				}
			};
			
			ControlPanel imageCaption = new ControlPanel(getText("ui.images")) {
				
				JScrollPane captionPane = new JScrollPane(thumbCaptionTemplate);
				JCheckBox focusCurrentThumb = new JCheckBox(getText("ui.focusCurrentThumb"), true);
				
				{
					thumbCaptionPresets.addItemListener(listener -> {
						if (isSkinReady()) {
							String s = getSelectedValue(thumbCaptionPresets);
							if (!s.equals(CUSTOM)) {
								thumbCaptionTemplate.setText(s);
							}
						}
					});	
					new StateMonitor() {
						public void onChange() {
							String s = getSelectedValue(thumbCaptionPresets);
							captionPane.setVisible(s.equals(CUSTOM));
						}
					}.add(thumbCaptionPresets).done();
					
					captionPane.setBorder(BorderFactory.createTitledBorder(getText("ui.captionTemplate")));
					thumbCaptionTemplate.setEditable(true);
					thumbCaptionTemplate.setLineWrap(true);
					thumbCaptionTemplate.setWrapStyleWord(true);
					thumbCaptionTemplate.setFont(mono);
					thumbCaptionTemplate.setTabSize(2);
					focusCurrentThumb.setToolTipText(getText("ui.focusCurrentThumbInfo"));
					//ComponentUtilities.whenSelectedDisable(fixedShapeThumbs, new JComponent[]{maxHorizontalAR, minVerticalAR});

					add("", new JLabel(getText("ui.caption")));
					add("tab", thumbCaptionPresets);
					add("br hfill", captionPane);
					add("br", focusCurrentThumb);
				}
			};

			
			{
				
				add("hfill", folderCaption);
				add("br hfill", pageCaption);
				add("br hfill", webLocationCaption);
				add("br hfill", imageCaption);
				
				putClientProperty("helpPage", helpRoot + "/Site/Thumbdesign");
			}
		};
		
		// Mark new files
		
		ControlPanel markNew = new ControlPanel() {

			JCheckBox markFilesNew = new JCheckBox(getText("ui.markFilesNew"));	
			JTextField newDaysCount = new JSmartTextField("60", 3);
			JComboBox newDaysRef = new JComboBox(new Object[] {
				new Item("dateTaken", getText("ui.dateTaken")),
				new Item("fileModified", getText("ui.fileModified")),
				new Item("added", getText("ui.addedToAlbum"))
			});
			JComboBox newDaysMark = new JComboBox(new Object[] {
				new Item("icon", getText("ui.icon")),
				new Item("text", getText("ui.text"))
			});
			JTextField newDaysText = new JSmartTextField(getText("new"), 6);
			JLabel newIcon = new JLabel(icon("new-fill"));
			
			{
				newDaysCount.setToolTipText(getText("ui.newDaysCountInfo"));
				newDaysMark.addItemListener(listener -> {
					int i = newDaysMark.getSelectedIndex();
					newDaysText.setVisible(i == 1);
					newIcon.setVisible(i == 0);
				});
				newDaysText.setVisible(false);
				ComponentUtilities.whenSelectedEnable(markFilesNew, new JComponent[]{ newDaysCount, newDaysRef, newDaysMark, newDaysText, newIcon });
				
				add("", markFilesNew);
				add("", newDaysCount);
				add(new JLabel(getText("ui.daysOld")));
				add("br", new JLabel(getText("ui.reference")));
				add("tab", newDaysRef);
				add("br", new JLabel(getText("ui.marker")));
				add("tab", newDaysMark);
				add(newIcon);
				add(newDaysText);
				
				putClientProperty("helpPage", helpRoot + "/Site/Mark_new_files");
			}
		};
		
		// Footer
		
		ControlPanel footerSettings = new ControlPanel() {

			ControlPanel bottomNavigationPanel = new ControlPanel(getText("ui.bottomNavigation")) {
				//JCheckBox showBottomNavigation = new JCheckBox(getText("ui.showBottomNavigation"), false);	
				JCheckBox bottomNavigationIncludeFolders = new JCheckBox(getText("ui.includeFolders"), true);
				JCheckBox bottomNavigationIncludePages = new JCheckBox(getText("ui.includePages"), true);
				JCheckBox bottomNavigationIncludeWebLocations = new JCheckBox(getText("ui.includeWebLocations"), true);
				
				@Override
				public void importVariables(Map<String, Object> vars) {
					Object sbn = vars.get("showBottomNavigation");
					if (sbn != null && vars.get("bottomNavigationIncludeFolders") == null) {
						vars.put("bottomNavigationIncludeFolders", sbn);
						vars.put("bottomNavigationIncludePages", sbn);
						vars.put("bottomNavigationIncludeWebLocations", sbn);
					}
				};
				
				{
					//showBottomNavigation.setToolTipText(getText("ui.showBottomNavigationInfo"));
					//add("", showBottomNavigation);
					add("", bottomNavigationIncludeFolders);
					add("br", bottomNavigationIncludePages);
					add("br", bottomNavigationIncludeWebLocations);
				}
			};

			ControlPanel customLinkPanel = new ControlPanel(getText("ui.customLink")) {

				JTextField customLink = new JSmartTextField();
				JTextField customLinkText = new JSmartTextField();

				{
					customLink.setToolTipText(getText("ui.customLinkInfo"));
					customLinkText.setToolTipText(getText("ui.customLinkTextInfo"));

					add(new JLabel("URL"));
					add("tab hfill", customLink);
					add("br", new JLabel(getText("ui.customLinkText")));
					add("tab hfill", customLinkText);
				}
			};

			ControlPanel customFooterContent = new ControlPanel(getText("ui.customContent")) {

				JSmartTextArea footer = new JSmartTextArea(6, 20);
				JScrollPane footerPane = new JScrollPane(footer, JScrollPane.VERTICAL_SCROLLBAR_AS_NEEDED, JScrollPane.HORIZONTAL_SCROLLBAR_AS_NEEDED);
				JCheckBox footerFitWidth = new JCheckBox(getText("ui.constrainPageWidth"), true);
				JCheckBox footerTopLevelOnly = new JCheckBox(getText("ui.topLevelOnly"), false);
				private WrappableJLabel instr = new WrappableJLabel("<html><i>" + getText("ui.pageBottomText") + "</i></html>");
				
				{
					footer.setEditable(true);
					footer.setLineWrap(true);
					footer.setWrapStyleWord(true);
					footer.setFont(mono);
					footer.setTabSize(2);
					footer.setToolTipText(getText("ui.customContentInfo"));
					footerPane.setPreferredSize(new Dimension(uiWidth - 260, 64));
					allowHTMLEditing(footer);
					instr.setPreferredWidth(uiWidth - 260);

					add("hfill vfill", footerPane);
					add("br", footerFitWidth);
					add("br", footerTopLevelOnly);
					add("br", instr);

				}
			};
			
			JCheckBox showModifiedDate = new JCheckBox(getText("ui.showModifiedDate"));	
			JCheckBox showImageCount = new JCheckBox(getText("ui.showFolderImageCount"));	

			{
				showImageCount.setToolTipText(getText("ui.showFolderImageCountInfo"));

				add("hfill", bottomNavigationPanel);
				add("br hfill", customLinkPanel);				
				add("br", showModifiedDate);
				add("br", showImageCount);
				add("br hfill vfill", customFooterContent);
				
				putClientProperty("helpPage", helpRoot + "/Site/Footer");
			}

		};
			
		/*	---------------------------------------------------------------
								Top bar tabs
			--------------------------------------------------------------- */

		// Navigation popup
		
		ControlPanel navigation = new ControlPanel() {

			JCheckBox topNavigationIncludeFolders = new JCheckBox(getText("ui.includeFolders"));
			JCheckBox topNavigationIncludePages = new JCheckBox(getText("ui.includePages"));
			JCheckBox topNavigationIncludeWebLocations = new JCheckBox(getText("ui.includeWebLocations"));
			JSpinner topNavigationDepth = new JSpinner(new SpinnerNumberModel(4, 1, 6, 1));
			JSpinner maxTitleLength = new JSpinner(new SpinnerNumberModel(48, 32, 160, 16));
			
			{
				add("", topNavigationIncludeFolders);
				add("br", topNavigationIncludePages);
				add("br", topNavigationIncludeWebLocations);
				add("br", new JLabel(getText("ui.depth")));
				add("", topNavigationDepth);
				add("", new JLabel(getText("ui.levels")));
				add("br", new JLabel(getText("ui.maxTitleLength")));
				add("tab", maxTitleLength);
				add("", new JLabel(getText("ui.characters")));
			}
		};
		
		//	Download
		
		ControlPanel download = new ControlPanel() {

			private WrappableJLabel instr = new WrappableJLabel("<html><i>" + getText("ui.extraSizesInfo") + "</i></html>");
			JTextField downloadLabel = new JSmartTextField(16);
			JComboBox zipImages = new JComboBox(new Object[] {
				new Item("none", getText("ui.nothing")),
				new Item("slides", getText("ui.scaledDown")),
				new Item("originals", getText("ui.originals")),
				new Item("included", getText("ui.includedOriginals"))
			}); 
			JLabel zipInfo = new JLabel(infoIcon);
			
			{
				instr.setPreferredWidth(uiWidth - 300);
				downloadLabel.setToolTipText(getText("ui.default") + ": " + getText("ui.downloadImages"));
				zipInfo.addMouseListener(new MouseAdapter() {  
					@Override
					public void mouseReleased(MouseEvent e) {
						JOptionPane.showMessageDialog(window, getText("ui.nonAsciiWarning"), getText("ui.warning"), JOptionPane.WARNING_MESSAGE);
				}});
				
				add("br", new JLabel(getText("ui.extraSizes")));
				add("tab", extraSizes);
				add("br tab", instr);
				add("br", new JLabelFor(getText("ui.offerDownload"), zipImages));
				add("", zipImages);
				add("", zipInfo);
				
				putClientProperty("helpPage", helpRoot + "/Site/Download");
			}
		};

		// Album info popup
		
		ControlPanel albumInfo = new ControlPanel() {

			JCheckBox headerTopLevelOnly = new JCheckBox(getText("ui.topLevelOnly"), true);
			JSmartTextArea header = new JSmartTextArea(6, 30);
			JScrollPane headerPane = new JScrollPane(header);

			{
				header.setEditable(true);
				header.setLineWrap(false);
				header.setFont(mono);
				header.setTabSize(2);
				header.setToolTipText(getText("ui.customContentInfo"));
				allowHTMLEditing(header);
				headerPane.setBorder(BorderFactory.createTitledBorder(getText("ui.albumInfoText")));
				
				add("hfill vfill", headerPane);
				add("br", headerTopLevelOnly);
				
				putClientProperty("helpPage", helpRoot + "/Site/Album_info_window");
			}
		};
		
		ControlPanel search = new ControlPanel() {

			JTextField searchFields = new JSmartTextField(18);
			//WrappableJLabel fieldsInfo = new WrappableJLabel("<html><i>" + getText("ui.searchFieldsInfo") + "</i></html>");

			{
				searchFields.setToolTipText("<html>" + getText("ui.searchFieldsInfo") + "</html>");
				//fieldsInfo.setPreferredWidth(uiWidth - 340);
				ComponentUtilities.whenSelectedEnable(useSearch, searchFields);

				add("", useSearch);
				add("br", new JLabelFor(getText("ui.fields"), searchFields));
				add("tab hfill", searchFields);
				add("br", new JLabel(""));
				//add("tab", fieldsInfo);
			}
		};

		// Background music

		ControlPanel backgroundMusic = new ControlPanel() {

			JPlaylist backgroundAudio = new JPlaylist();
			JCheckBox backgroundAudioAutoPlay = new JCheckBox(getText("ui.autoStart"));
			JSpinner backgroundAudioVolume = new JSpinner(new SpinnerNumberModel(25, 1, 100, 1));
			JCheckBox backgroundAudioSlideshowControl = new JCheckBox(getText("ui.slideshowControl"));
			JCheckBox muteBackgroundAudio = new JCheckBox(getText("ui.muteBackgroundAudio"), true);
			JCheckBox backgroundAudioLoop = new JCheckBox(getText("ui.loop"));
			JCheckBox backgroundAudioRetainPosition = new JCheckBox(getText("ui.retainPosition"), true);
			WrappableJLabel autostartNotice = new WrappableJLabel("<html><i>" + getText("ui.autostartNotice") + "</i></html>");

			{
				backgroundAudioSlideshowControl.setToolTipText(getText("ui.slideshowControlInfo"));
				muteBackgroundAudio.setToolTipText(getText("ui.muteBackgroundAudioInfo"));
				backgroundAudioRetainPosition.setToolTipText(getText("ui.retainPositionInfo"));
				autostartNotice.setPreferredWidth(uiWidth - 260);
				
				add("br hfill", backgroundAudio);
				add("br", new JLabel(getText("ui.initialVolume")));
				add("", backgroundAudioVolume);
				add("", new JLabel("%"));
				add("br", backgroundAudioAutoPlay);
				add("tab", backgroundAudioLoop);
				add("br", backgroundAudioRetainPosition);
				add("br", backgroundAudioSlideshowControl);
				add("br", muteBackgroundAudio);
				add("br", new JLabel(infoIcon));
				add("", autostartNotice);
				
				putClientProperty("helpPage", helpRoot + "/Site/Background_music");
			}
		};
		
		
		/*	---------------------------------------------------------------
								Functionality tab
			--------------------------------------------------------------- */
		
		// Social settings
		
		ControlPanel social = new ControlPanel() {

			ControlPanel shares = new ControlPanel(getText("shareOn")) {

				JCheckBox shareTwitter = new JCheckBox("X");
				JCheckBox sharePinterest = new JCheckBox("Pinterest");
				JCheckBox shareLinkedin = new JCheckBox("LinkedIn");
				JCheckBox shareDigg = new JCheckBox("Digg");
				JCheckBox shareReddit = new JCheckBox("Reddit");
				JCheckBox shareTumblr = new JCheckBox("Tumblr");
				
				{
					setLayout(new RiverLayout(4, 5));
					add(new JLabel(icon("facebook")));
					add(shareFacebook);
					add("br", new JLabel(icon("threads")));
					add(shareThreads);
					add("br", new JLabel(icon("bluesky")));
					add(shareBluesky);
					add("br", new JLabel(icon("x")));
					add(shareTwitter);
					add("br", new JLabel(icon("reddit")));
					add(shareReddit);
					add("br", new JLabel(icon("pinterest")));
					add(sharePinterest);
					add("br", new JLabel(icon("tumblr")));
					add(shareTumblr);
					add("br", new JLabel(icon("linkedin")));
					add(shareLinkedin);
					add("br", new JLabel(icon("digg")));
					add(shareDigg);
				}
			};

			JTextField facebookAppId = new JSmartTextField(24);
			JLabel mandatoryInfo = new JLabel(mandatory);
			
			ControlPanel links = new ControlPanel() {
				
				JCheckBox shareEmail = new JCheckBox(getText("ui.email"));
				JLabel emailSubjectLabel = new JLabel(getText("ui.subject"));
				JTextField emailSubject = new JSmartTextField(18);
				JTextArea emailBody = new JSmartTextArea(4,22);
				JScrollPane emailBodyPane = new JScrollPane(emailBody);
				JCheckBox shareLink = new JCheckBox(getText("ui.link"));
				
				{
					emailBody.setEditable(true);
					emailBody.setLineWrap(true);
					emailBody.setWrapStyleWord(true);
					emailBody.setToolTipText(getText("ui.emailBodyInfo"));
					emailBodyPane.setBorder(BorderFactory.createTitledBorder(getText("ui.body")));
					ComponentUtilities.whenSelectedEnable(shareEmail, new JComponent[]{emailSubjectLabel, emailSubject, emailBodyPane});
					
					add("", new JLabel(icon("email")));
					add("tab", shareEmail);
					add("br tab", emailSubjectLabel);
					add("", emailSubject);
					add("br tab", emailBodyPane);
					add("br", new JLabel(icon("link")));
					add("tab", shareLink);
					add("br", new JLabelFor(getText("ui.facebookAppId"), facebookAppId));
					add(mandatoryInfo);
					add("", facebookAppId);
					add("", new JLinkLabel("https://developers.facebook.com/apps", getText("ui.signUp")));
				}
			};
			
			ControlPanel fbBox = new ControlPanel(getText("ui.facebookBox")) {

				JCheckBox useFacebookBox = new JCheckBox(getText("ui.useFacebookBox"));
				JTextField facebookPageId = new JSmartTextField(24);
				JLabel mandatoryInfo = new JLabel(mandatory);
				JCheckBox facebookFaces = new JCheckBox(getText("ui.showFaces"));
				JCheckBox facebookHeader = new JCheckBox(getText("ui.showHeader"));
				WrappableJLabel fbBoxInfo = new WrappableJLabel("<html><i>" + getText("ui.facebookBoxInfo") + " </i></html>");

				{
					mandatoryInfo.setToolTipText(getText("ui.mandatory"));
					fbBoxInfo.setPreferredWidth(uiWidth - 260);
					ComponentUtilities.whenSelectedEnable(useFacebookBox, new JComponent[] {facebookPageId, facebookFaces, facebookHeader});
					
					add(useFacebookBox);
					add("br", new JLabelFor(getText("ui.facebookPageId"), facebookPageId));
					add(mandatoryInfo);
					add("tab", facebookPageId);
					add("br", facebookHeader);
					add("", facebookFaces);
					add("br", fbBoxInfo);
				}

			};

			{
				mandatoryInfo.setToolTipText(getText("ui.mandatory"));
				links.setBorder(emptyBorder);
				
				add(shares);
				add("hfill", links);
				/*
				add("br", new JLabelFor(getText("ui.facebookAppId"), facebookAppId));
				add(mandatoryInfo);
				add("tab hfill", facebookAppId);
				add("", new JLinkLabel("https://developers.facebook.com/apps", getText("ui.signUp")));
				*/
				add("br hfill", fbBox);

				putClientProperty("helpPage", helpRoot + "/Site/Social");
			}
		};
				
		// Search new section
		
		ControlPanel searchNew = new ControlPanel() {

			JComboBox searchNewSource = new JComboBox(new Object[] {
				new Item("none", getText("ui.noSearchNew")),			
				new Item("current", getText("ui.currentFolder")),			
				new Item("subfolders", getText("ui.subfolders")),
				new Item("tree", getText("ui.wholeAlbum"))
			});
			JTextField searchNewLabel = new JSmartTextField(12);
			JTextField searchNewDays = new JSmartTextField(12);
			JComboBox searchNewReference = new JComboBox(new Object[] {
				new Item("dateTaken", getText("ui.dateTaken")),
				new Item("fileModified", getText("ui.fileModified")),
				new Item("added", getText("ui.addedToAlbum"))
			});
			JCheckBox searchNewSinceLastVisit = new JCheckBox(getText("ui.searchNewSinceLastVisit"));

			{
				searchNewSource.addActionListener(new ActionListener() {
					@Override
					public void actionPerformed(ActionEvent e) {
						boolean on = searchNewSource.getSelectedIndex() > 0;
						searchNewLabel.setEnabled(on);
						searchNewDays.setEnabled(on);
						searchNewReference.setEnabled(on);
						searchNewSinceLastVisit.setEnabled(on);
				}});
				searchNewLabel.setText(getText("ui.newImages"));
				searchNewDays.setToolTipText(getText("ui.searchNewDaysInfo"));
				searchNewSinceLastVisit.setToolTipText(getText("ui.searchNewSinceLastVisitInfo"));

				add(new JLabelFor(getText("ui.collectFrom"), searchNewSource));
				add("tab", searchNewSource);
				add("br", new JLabelFor(getText("ui.boxTitle"), searchNewLabel));
				add("tab hfill", searchNewLabel);
				add("br", new JLabelFor(getText("ui.days"), searchNewDays));
				add("tab hfill", searchNewDays);
				add("br", new JLabelFor(getText("ui.reference"), searchNewReference));
				add("tab", searchNewReference);
				add("br tab", searchNewSinceLastVisit);
				
				putClientProperty("helpPage", helpRoot + "/Sections/Search_new_images");
			}
		};
		
		// Rating
		
		ControlPanel rating = new ControlPanel() {

			JCheckBox useRating = new JCheckBox(getText("ui.displayRating"));
			JCheckBox visitorRating = new JCheckBox(getText("ui.allowVisitorRatings"));
			JCheckBox useJalbumRating = new JCheckBox(getText("ui.useJalbumRatings"));
			
			{
				useRating.setToolTipText(getText("ui.displayRatingInfo"));
				visitorRating.setToolTipText(getText("ui.allowVisitorRatingsInfo"));
				useJalbumRating.setToolTipText(getText("ui.useJalbumRatingsInfo"));
				ComponentUtilities.whenSelectedEnable(useRating, new JComponent[]{ visitorRating, useJalbumRating});
				
				add("br", useRating);
				add("br", useJalbumRating);
				add("br", visitorRating);
				
				putClientProperty("helpPage", helpRoot + "/Site/Rating");
			}
		};
				
		//	Map settings
		
		ControlPanel mapSettings = new ControlPanel() {

			JCheckBox showMapSection = new JCheckBox(getText("ui.showMapSection"));
			JComboBox mapType = new JComboBox(new Object[]{
				new Item("roadmap", getText("ui.roadmap")),
				new Item("satellite", getText("ui.satellite")),
				new Item("hybrid", getText("ui.hybrid")),
				new Item("terrain", getText("ui.terrain"))
			});
			JSlider mapZoom = new JSlider(JSlider.HORIZONTAL, 1, 20, 18);
			JTextField googleApiKey = new JSmartTextField(24);
			JLabel googleApiInfo = new JLabel(infoIcon);
			WrappableJLabel note = new WrappableJLabel("<html><i>" + getText("ui.mapApiKeyInfo") + "</i></html>");
			
			{
				showMapSection.setToolTipText(getText("ui.showMapSectionInfo"));
				mapZoom.setOrientation(JSlider.HORIZONTAL);
				mapZoom.setMinimum(0);
				mapZoom.setMaximum(20);
				mapZoom.setValue(18);
				mapZoom.setMajorTickSpacing(10);
				mapZoom.setMinorTickSpacing(1);
				mapZoom.setPaintTicks(true);
				mapZoom.setPaintLabels(true);
				mapZoom.setSnapToTicks(true);
				note.setPreferredWidth(uiWidth - 200);
				googleApiInfo.addMouseListener(new MouseAdapter() {  
					@Override
					public void mouseReleased(MouseEvent e) {
						JOptionPane.showMessageDialog(window, "<html><i>" + getText("ui.gooleMapsApiNote") + "</i></html>", getText("ui.format"), JOptionPane.INFORMATION_MESSAGE);
				}});
				
				add(showMapSection);
				add("br", new JLabelFor(getText("ui.initialView"), mapType));
				add("tab", mapType);
				add("br", new JLabelFor(getText("ui.initialZoom"), mapZoom));
				add("tab", mapZoom);
				add("br", new JLabelFor(getText("ui.googleApiKey"), googleApiKey));
				add("tab", googleApiKey);
				add("br tab", new JLinkLabel("https://console.developers.google.com/apis/credentials", getText("ui.createNew")));
				add("", googleApiInfo);
				add("br tab", note);

				putClientProperty("helpPage", helpRoot + "/Site/Map");
			}

		};
		
		ControlPanel map = new ControlPanel() {
			
			JCheckBox useMap = new JCheckBox(getText("ui.useMap"));
			
			{
				useMap.setToolTipText(getText("ui.sellPhotosInfo"));
				ComponentUtilities.whenSelectedEnable(useMap, mapSettings);
				
				add("", useMap);
				add("br hfill", mapSettings);
			}
		};
				
		// Tag cloud section

		ControlPanel tagCloud = new ControlPanel() {
			
			JComboBox tagCloudPosition = new JComboBox(new Object[] {
				new Item("top", getText("ui.top")),			
				new Item("bottom", getText("ui.bottom"))
			});
			JComboBox tagCloudSource = new JComboBox(new Object[] {
				new Item("none", getText("ui.noTagCloud")),			
				new Item("current", getText("ui.currentFolder")),			
				new Item("subfolders", getText("ui.subfolders")),
				new Item("tree", getText("ui.wholeAlbum"))
			});
			JCheckBox tagCloudUseFolders = new JCheckBox(getText("ui.folders"), true);
			JCheckBox tagCloudUsePages = new JCheckBox(getText("ui.pages"));
			JCheckBox tagCloudUseWebLocations = new JCheckBox(getText("ui.webLocations"));
			JLabel tagCloudSkipLevelsLabel = new JLabel(getText("ui.skipLevels"));
			JSpinner tagCloudSkipLevels = new JSpinner(new SpinnerNumberModel(1, 0, 5, 1));
			JTextField tagCloudLabel = new JSmartTextField(12);
			JTextField tagCloudFields = new JSmartTextField();
			JComboBox tagCloudSort = new JComboBox(new Object[] {
				new Item("none", getText("ui.unsorted")),			
				new Item("name", getText("ui.name")),			
				new Item("frequency", getText("ui.frequency"))
			});
			JCheckBox tagCloudFontVaries = new JCheckBox(getText("ui.tagCloudFontVaries"));
			JCheckBox tagCloudSearch = new JCheckBox(getText("ui.addSearchBox"));

			{
				tagCloudPosition.setToolTipText(getText("ui.tagCloudPositionInfo"));
				tagCloudLabel.setText(getText("ui.labels"));
				tagCloudSource.addActionListener(new ActionListener() {
					@Override
					public void actionPerformed(ActionEvent e) {
						int tsi = tagCloudSource.getSelectedIndex();
						tagCloudUseFolders.setEnabled(tsi > 0);
						tagCloudUsePages.setEnabled(tsi > 0);
						tagCloudUseWebLocations.setEnabled(tsi > 0);
						tagCloudLabel.setEnabled(tsi > 0);
						tagCloudFields.setEnabled(tsi > 0);
						tagCloudSort.setEnabled(tsi > 0);
						tagCloudFontVaries.setEnabled(tsi > 0);
						tagCloudSearch.setEnabled(tsi > 0);
						tagCloudSkipLevels.setVisible(tsi == 2);
						tagCloudSkipLevelsLabel.setVisible(tsi == 2);
				}});
				tagCloudSkipLevels.setToolTipText(getText("ui.skipLevelsInfo"));
				tagCloudFields.setToolTipText("<html>" + getText("ui.searchFieldsInfo") + "</html>");
				tagCloudFontVaries.setToolTipText(getText("ui.tagCloudFontVariesInfo"));

				add(new JLabelFor(getText("ui.position"), tagCloudPosition));
				add("tab", tagCloudPosition);
				add("br", new JLabelFor(getText("ui.collectFrom"), tagCloudSource));
				add("tab", tagCloudSource);
				add("tab", tagCloudSkipLevelsLabel);
				add("", tagCloudSkipLevels);
				add("", tagCloudUseFolders);
				add("", tagCloudUsePages);
				add("", tagCloudUseWebLocations);
				add("br", new JLabelFor(getText("ui.boxTitle"), tagCloudLabel));
				add("tab hfill", tagCloudLabel);
				add("br", new JLabelFor(getText("ui.fields"), tagCloudFields));
				add("tab hfill", tagCloudFields);
				add("br", new JLabelFor(getText("ui.sortBy"), tagCloudFields));
				add("tab", tagCloudSort);
				add("tab", tagCloudFontVaries);
				add("br tab", tagCloudSearch);
				
				putClientProperty("helpPage", helpRoot + "/Sections/Tag_cloud_box");
			}
		};

		//	Shopping cart section
		
		ControlPanel shoppingCart = new ControlPanel() {

			JCheckBox showShop = new JCheckBox(getText("ui.usePaypal"));

			JCheckBox showShopOnImagePagesOnly = new JCheckBox(getText("ui.showShopOnImagePagesOnly"), true);
			JTextField shopLabel = new JTextField(24);
			JTextField shopId = new JTextField(24);
			JLabel mandatoryInfo = new JLabel(mandatory);
			/*JComboBox shopPlacement = new JComboBox(new Object[]{ 
				new Item("aboveThumbs", getText("ui.aboveThumbs")),
				new Item("sidebar", getText("ui.inTheSidebar"))
			});*/
			JComboBox shopCurrency = new JComboBox(new Object[]{ 
				new Item("USD", "United States Dollar"),
				new Item("EUR", "Euro"),
				new Item("GBP", "British Pound"),
				new Item("CAD", "Canadian Dollar"),
				new Item("AUD", "Australian Dollar"),
				//new Item("RUB", "Russia Rubles"),
				new Item("JPY", "Japanese Yen"),
				//new Item("INR", "India Rupees"),
				new Item("NZD", "New Zealand Dollar"),
				new Item("CHF", "Swiss Franc"),
				//new Item("ARS", "Argentina Pesos"),
				//new Item("BHD", "Bahrain Dinars"),
				//new Item("BYR", "Belarus Rubles"),
				//new Item("BAM", "Bosnia & Herzegovina C.Marka"),
				new Item("BRL", "Brazilian Real"),
				//new Item("BGN", "Bulgaria Leva"),
				//new Item("CLP", "Chile Pesos"),
				//new Item("CNY", "China Yuan Renminbi"),
				//new Item("COP", "Colombia Pesos"),
				//new Item("CRC", "Costa Rica Colones"),
				//new Item("HRK", "Croatia Kuna"),
				new Item("CZK", "Czech Koruna"),
				new Item("DKK", "Danish Krone"),
				//new Item("EGP", "Egypt Pounds"),
				//new Item("EEK", "Estonia Krooni"),
				//new Item("GTQ", "Guatemala Quetzales"),
				new Item("HKD", "Hong Kong Dollar"),
				new Item("HUF", "Hungary Forint"),
				//new Item("ISK", "Iceland Kronur"),
				//new Item("IDR", "Indonesia Rupiahs"),
				//new Item("IQD", "Iraq Dinars"),
				new Item("ILS", "Israel New Shekel"),
				//new Item("JMD", "Jamaica Dollars"),
				//new Item("JOD", "Jordan Dinars"),
				//new Item("KWD", "Kuwait Dinars"),
				//new Item("LVL", "Latvia Lati"),
				//new Item("LBP", "Lebanon Pounds"),
				//new Item("LTL", "Lithuania Litai"),
				//new Item("MKD", "Macedonia Denars"),
				new Item("MYR", "Malaysian Ringgit"),
				new Item("MXN", "Mexican Peso"),
				//new Item("MDL", "Moldova Lei"),
				//new Item("MAD", "Morocco Dirhams"),
				new Item("NOK", "Norway Krone"),
				//new Item("PEN", "Peru Nuevos Soles"),
				new Item("PHP", "Philippines Peso"),
				new Item("PLN", "Poland Zlotych"),
				//new Item("RON", "Romania New Lei"),
				new Item("RUB", "Russian Ruble"),
				//new Item("SAR", "Saudi Arabia Riyals"),
				//new Item("RSD", "Serbia Dinars"),
				new Item("SGD", "Singapore Dollar"),
				//new Item("ZAR", "South Africa Rand"),
				//new Item("KRW", "South Korea Won"),
				new Item("SEK", "Sweden Krona"),
				new Item("TWD", "New Taiwan Dollar"),
				new Item("THB", "Thailand Baht"),
				new Item("TRY", "Turkish Lira"),
				//new Item("UAH", "Ukraine Hryvnia"),
				//new Item("AED", "United Arab Emirates Dirhams"),
				//new Item("UYU", "Uruguay Pesos"),
				//new Item("VND", "Vietnam Dong")
			});
			JTextField shopSuccessUrl = new JSmartTextField(24);
			JCheckBox shopSameWindowCheckout = new JCheckBox(getText("ui.sameWindowCheckout"));
			JCheckBox shopSendAlbumName = new JCheckBox(getText("ui.sendAlbumName"));
			JCheckBox shopSendFolderPath = new JCheckBox(getText("ui.sendFolderPath"));
			JComboBox shopSendItemName = new JComboBox(new Object[]{ 
				new Item("name", getText("ui.filename")),
				new Item("title", getText("ui.titleOrFilename")),
				new Item("comment", getText("ui.commentOrFilename"))
			});
			JCheckBox shopAskPermissionToEmpty = new JCheckBox(getText("ui.askPermisssionToEmptyCart"));
			
			JComboBox showPriceRange = new JComboBox(new Object[]{ 
				new Item("", getText("ui.none")),
				new Item("lowest", getText("ui.lowestPriceOnly")),
				new Item("minmax", getText("ui.minMax"))
			});
			JCheckBox usePriceAsSingleOption = new JCheckBox(getText("ui.usePriceAsSingleOption"));
			JComboBox shopOptionSelectMethod = new JComboBox(new Object[]{ 
				new Item("combo", getText("ui.dropdownBox")),
				new Item("radio", getText("ui.radioButtons"))
			});
			JTextArea shopOptions = new JSmartTextArea(8, 28);
			JScrollPane shopOptionsPane = new JScrollPane(shopOptions);
			JLabel shopOptionsInfo = new JLabel(infoIcon);

			JTextField shopMinimalAmount = new JSmartTextField(8);
			JTextField shopHandling = new JSmartTextField(8);
			JLabel shopHandlingInfo = new JLabel(infoIcon);
			JTextField shopTax = new JSmartTextField(8);
			JTextField shopQuantityCap = new JSmartTextField(8);

			JTextArea shopCoupons = new JSmartTextArea(4, 16);
			JScrollPane shopCouponsPane = new JScrollPane(shopCoupons);
			JLabel shopCouponsInfo = new JLabel(infoIcon);

			JTextField shopDiscountRate = new JSmartTextField(8);
			JTextField shopDiscountMinQuantity = new JSmartTextField(8);
			JTextField shopDiscountMinAmount = new JSmartTextField(8);

			JSmartTextArea shopInstructions = new JSmartTextArea(6, 24);	
			JScrollPane shopInstructionsPane = new JScrollPane(shopInstructions);
			JSmartTextArea shopInstructionsBox = new JSmartTextArea(6, 24);	
			JScrollPane shopInstructionsBoxPane = new JScrollPane(shopInstructionsBox);
				
			ControlPanel shoppingCartOptions = new ControlPanel() {

				{
					shopId.setToolTipText(getText("ui.sellerIdInfo"));
					mandatoryInfo.setToolTipText(getText("ui.mandatory"));
					shopLabel.setToolTipText(getText("ui.default") + ": " + getText("ui.shoppingCart"));
					shopCurrency.setEditable(false);
					shopCurrency.setToolTipText(getText("ui.currencyInfo"));
					shopSuccessUrl.setToolTipText(getText("ui.shopSuccessUrlInfo"));
					usePriceAsSingleOption.setToolTipText(getText("ui.usePriceAsSingleOptionInfo"));
					shopSameWindowCheckout.setToolTipText(getText("ui.sameWindowCheckoutInfo"));
					shopSendAlbumName.setToolTipText(getText("ui.sendAlbumNameInfo"));

					add(showShopOnImagePagesOnly);
					add("br", new JLabelFor(getText("ui.boxTitle"), shopLabel));
					add("tab", shopLabel);
					add("br", new JLabelFor(getText("ui.sellerId"), shopId));
					add(mandatoryInfo);
					add("tab", shopId);
					add("br tab", new JLinkLabel("https://www.paypal.com/signup/account", getText("ui.signUp")));
					add("br", new JLabelFor(getText("ui.currency"), shopCurrency));
					add("tab", shopCurrency);
					add("br", new JLabelFor(getText("ui.shopSuccessUrl"), shopSuccessUrl));
					add("tab", shopSuccessUrl);
					add("br", new JLabel(getText("ui.showPriceRange")));
					add("tab", showPriceRange);
					add("br", usePriceAsSingleOption);
					add("br", shopSendAlbumName);
					add("tab", shopSendFolderPath);
					add("tab", shopSendItemName);
					add("", new JLabel(getText("ui.max128Chars")));
					add("br", shopSameWindowCheckout);
					add("br", shopAskPermissionToEmpty);
				}
			};

			ControlPanel shoppingCartPricing = new ControlPanel() {

				{
					shopOptionsInfo.addMouseListener(new MouseAdapter() {  
						@Override
						public void mouseReleased(MouseEvent e) {
							String s = "<html><b><kbd>" + getText("ui.shopOptionsFormat") + "</kbd></b><br>" +
									"<br><b>" + getText("ui.examples") + ":</b><br>" +
									"<br><kbd>" + "Print A4=2.0" + "</kbd><br>" +
									"<i>" + getText("ui.optionExample1Hint") + "</i><br>" +
									"<br><kbd>" + "Print A2=10.0+0.5" + "</kbd><br>" +
									"<i>" + getText("ui.optionExample2Hint") + "</i><br>" +
									"<br><kbd>" + "Print A2=10.0+0.5+0" + "</kbd><br>" +
									"<i>" + getText("ui.optionExample3Hint") + "</i><br>" +
									"<br><i>" + getText("ui.noCurrencySign") + "</i></html>";
							JOptionPane.showMessageDialog(window, s, getText("ui.format"), JOptionPane.INFORMATION_MESSAGE);
					}});
					shopOptions.setToolTipText(getText("ui.shopOptionsInfo"));
					shopOptions.setEditable(true);
					shopOptions.setLineWrap(false);
					shopOptions.setFont(mono);
					shopOptionsPane.setBorder(BorderFactory.createTitledBorder(getText("ui.shopOptions")));
					shopHandlingInfo.addMouseListener(new MouseAdapter() {  
						@Override
						public void mouseReleased(MouseEvent e) {
							String s = "<html><b><kbd>" + getText("ui.handlingFormat") + "</kbd></b><br>" +
									"<br><b>" + getText("ui.examples") + ":</b><br>" +
									"<br><kbd>" + "2.0" + "</kbd><br>" +
									"<i>" + getText("ui.handlingExample1Hint") + "</i><br>" +
									"<br><kbd>" + "0.0+0.1" + "</kbd><br>" +
									"<i>" + getText("ui.handlingExample2Hint") + "</i><br>" +
									"<br><kbd>" + "1.0+0+0.1" + "</kbd><br>" +
									"<i>" + getText("ui.handlingExample3Hint") + "</i><br>" +
									"<br><i>" + getText("ui.noCurrencySign") + "</i></html>";
							JOptionPane.showMessageDialog(window, s, getText("ui.format"), JOptionPane.INFORMATION_MESSAGE);
					}});
					shopHandling.setToolTipText(getText("ui.handlingInfo") + " " + getText("ui.leaveEmpty"));
					shopTax.setToolTipText(getText("ui.taxInfo") + " " + getText("ui.leaveEmpty"));
					shopQuantityCap.setToolTipText(getText("ui.shopQuantityCapInfo") + " " + getText("ui.leaveEmpty"));
					shopMinimalAmount.setToolTipText(getText("ui.shopMinimalCartValueInfo"));
					shopDiscountRate.setToolTipText(getText("ui.shopDiscountRateInfo") + " " + getText("ui.leaveEmpty"));
					shopDiscountMinQuantity.setToolTipText(getText("ui.shopDiscountMinQuantityInfo"));
					shopDiscountMinAmount.setToolTipText(getText("ui.shopDiscountMinAmountInfo"));
					shopCoupons.setToolTipText(getText("ui.shopCouponsInfo"));
					shopCoupons.setEditable(true);
					shopCoupons.setLineWrap(false);
					shopCoupons.setFont(mono);
					shopCouponsInfo.addMouseListener(new MouseAdapter() {  
						@Override
						public void mouseReleased(MouseEvent e) {
							String s = "<html><b><kbd>" + getText("ui.shopCouponsFormat") + "</kbd></b><br>" +
									"<br><b>" + getText("ui.examples") + ":</b><br>" +
									"<br><kbd>" + "5OFF=5.0" + "</kbd><br>" +
									"<i>" + getText("ui.couponExample1Hint") + "</i><br>" +
									"<br><kbd>" + "XMAS=20% &lt;2016-12-24" + "</kbd><br>" +
									"<i>" + getText("ui.couponExample2Hint") + "</i><br>" +
									"<br><kbd>" + "MIN10=30% 10+" + "</kbd><br>" +
									"<i>" + getText("ui.couponExample3Hint") + "</i><br>" +
									"<br><i>" + getText("ui.noCurrencySign") + "</i></html>";
							JOptionPane.showMessageDialog(window, s, getText("ui.format"), JOptionPane.INFORMATION_MESSAGE);
					}});
					shopCouponsPane.setBorder(BorderFactory.createTitledBorder(getText("ui.shopCoupons")));

					add("", new JLabelFor(getText("ui.handling"), shopHandling));
					add(shopHandlingInfo);
					add("tab", shopHandling);
					add("tab", new JLabelFor(getText("ui.tax") + " (%)", shopTax));
					add("tab", shopTax);
					add("br", new JLabelFor(getText("ui.shopQuantityCap"), shopQuantityCap));
					add("tab", shopQuantityCap);
					add("tab", new JLabelFor(getText("ui.shopMinimalCartValue"), shopMinimalAmount));
					add("tab", shopMinimalAmount);
					add("br", new JLabelFor(getText("ui.shopDiscountRate") + " (%)", shopDiscountRate));
					add("tab", shopDiscountRate);
					add("tab", new JLabelFor(getText("ui.minQuantity"), shopDiscountMinQuantity));
					add("tab", shopDiscountMinQuantity);
					add("tab", new JLabelFor(getText("ui.minAmount"), shopDiscountMinAmount));
					add("tab", shopDiscountMinAmount);
					add("br hfill", shopOptionsPane);
					add(shopOptionsInfo);
					add("br hfill", shopCouponsPane);
					add(shopCouponsInfo);
					add("br", new JLabelFor(getText("ui.selectMethod"), shopOptionSelectMethod));
					add("tab", shopOptionSelectMethod);

				}
			};

			ControlPanel shoppingCartInstructions = new ControlPanel() {

				{
					shopInstructions.setToolTipText(getText("ui.shopInstructionsInfo"));
					shopInstructions.setEditable(true);
					shopInstructions.setLineWrap(true);
					shopInstructions.setWrapStyleWord(true);
					shopInstructions.setFont(mono);
					allowHTMLEditing(shopInstructions);
					shopInstructionsPane.setBorder(BorderFactory.createTitledBorder(getText("ui.shopInstructions")));
					shopInstructionsBox.setToolTipText(getText("ui.shopInstructionsBoxInfo"));
					shopInstructionsBox.setEditable(true);
					shopInstructionsBox.setLineWrap(true);
					shopInstructionsBox.setWrapStyleWord(true);
					shopInstructionsBox.setFont(mono);
					shopInstructionsBox.setTabSize(2);
					allowHTMLEditing(shopInstructionsBox);
					shopInstructionsBoxPane.setBorder(BorderFactory.createTitledBorder(getText("ui.shopInstructionsBox")));
					
					add("hfill", shopInstructionsPane);
					add("br hfill", shopInstructionsBoxPane);
				}
			};

			//WrappableJLabel instr = new WrappableJLabel("<html><i>" + getText("ui.selectIndexOrLightbox") + "</i></html>");
			
			JTabbedPane shopTabs = new JTabbedPane() {
								
				{
					shoppingCartOptions.setBorder(emptyBorder);
					shoppingCartPricing.setBorder(emptyBorder);
					shoppingCartInstructions.setBorder(emptyBorder);
					
					addTab(getText("ui.settings"), shoppingCartOptions);
					addTab(getText("ui.pricing"), shoppingCartPricing);
					addTab(getText("ui.instructions"), shoppingCartInstructions);
				}
			};

			{
				if (!commercialUseAllowed) {
					showShop.setSelected(false);
					commercialMonitor.add(showShop);
				}
				//instr.setPreferredSize(new Dimension(uiWidth - 210, 32));
				showShop.setToolTipText(getText("ui.sellPhotosInfo"));
				ComponentUtilities.whenSelectedEnable(showShop, new JComponent[]{ shoppingCartOptions, shoppingCartPricing, shoppingCartInstructions } );
				
				add("", showShop);
				add("br hfill", shopTabs);
				//add("br", new JLabel(infoIcon));
				//add(instr);

				putClientProperty("helpPage", helpRoot + "/Sections/Shopping_cart");
			}
		};
		
		// Feedback section
		
		ControlPanel feedback = new ControlPanel() {

			JCheckBox showFeedback = new JCheckBox(getText("ui.useFeedback"));

			ControlPanel feedbackOptions = new ControlPanel() {

				JCheckBox showFeedbackOnImagePagesOnly = new JCheckBox(getText("ui.showOnImagePagesOnly"), true);
				JCheckBox feedbackAddMultiple = new JCheckBox(getText("ui.canAddMultipleTimes"));
				JTextField feedbackLabel = new JTextField(24);
				JTextField feedbackEmail = new JTextField(24);
				JLabel mandatoryInfo = new JLabel(mandatory);
				JComboBox feedbackFormatting = new JComboBox(new Object[] {
					new Item("human", getText("ui.humanReadable")),			
					new Item("serialized", getText("ui.serialized"))
				});
				JTextField feedbackViewButtonLabel = new JTextField(24);
				JTextField feedbackAddButtonLabel = new JTextField(24);
				JTextField feedbackAddButtonTooltip = new JTextField(24);
				JTextField feedbackModalTitle = new JTextField(24);
				JTextField feedbackCopyButtonLabel = new JTextField(24);
				JCheckBox useFeedbackSendButton = new JCheckBox(getText("ui.useSendButton"), true);
				JTextField feedbackSendButtonLabel = new JTextField(24);
				
				{
					feedbackLabel.setToolTipText(getText("ui.feedbackLabelInfo"));
					mandatoryInfo.setToolTipText(getText("ui.mandatory"));
					feedbackViewButtonLabel.setToolTipText(getText("ui.feedbackViewButtonLabelInfo"));
					feedbackAddButtonLabel.setToolTipText(getText("ui.feedbackAddButtonLabelInfo"));
					feedbackAddButtonTooltip.setToolTipText(getText("ui.feedbackActionButtonTooltipInfo"));
					feedbackModalTitle.setToolTipText(getText("ui.feedbackModalTitleInfo"));
					feedbackCopyButtonLabel.setToolTipText(getText("ui.feedbackCopyButtonLabelInfo"));
					feedbackSendButtonLabel.setToolTipText(getText("ui.feedbackSendButtonLabelInfo"));
					feedbackEmail.setToolTipText(getText("ui.feedbackEmailInfo"));
					
					ComponentUtilities.whenSelectedEnable(useFeedbackSendButton, feedbackSendButtonLabel);
					
					add(showFeedbackOnImagePagesOnly);
					add("br", feedbackAddMultiple);
					add("br", new JLabelFor(getText("ui.boxTitle"), feedbackLabel));
					add("tab", feedbackLabel);
					add("tab", new JLabel("<html><i>" + getText("ui.default") + ": \"" + getText("feedback") + "\"</i></html>"));
					add("br", new JLabelFor(getText("ui.feedbackEmail"), feedbackEmail));
					add(mandatoryInfo);
					add("tab", feedbackEmail);
					add("br", new JLabelFor(getText("ui.formatting"), feedbackFormatting));
					add("tab", feedbackFormatting);
					add("br", new JLabelFor(getText("ui.viewButtonLabel"), feedbackViewButtonLabel));
					add("tab", feedbackViewButtonLabel);
					add("tab", new JLabel("<html><i>" + getText("ui.default") + ": \"" + getText("editFeedback") + "\"</i></html>"));
					add("br", new JLabelFor(getText("ui.addButtonLabel"), feedbackAddButtonLabel));
					add("tab", feedbackAddButtonLabel);
					add("tab", new JLabel("<html><i>" + getText("ui.default") + ": \"" + getText("writeFeedback") + "\"</i></html>"));
					add("br", new JLabelFor(getText("ui.addButtonTooltip"), feedbackAddButtonTooltip));
					add("tab", feedbackAddButtonTooltip);
					add("tab", new JLabel("<html><i>" + getText("ui.default") + ": \"" + getText("provideFeedbackOnSelectedItems") + "\"</i></html>"));
					add("br", new JLabelFor(getText("ui.modalTitle"), feedbackModalTitle));
					add("tab", feedbackModalTitle);
					add("tab", new JLabel("<html><i>" + getText("ui.default") + ": \"" + getText("feedbackOnAlbum") + "\"</i></html>"));
					add("br", new JLabelFor(getText("ui.copyButtonLabel"), feedbackCopyButtonLabel));
					add("tab", feedbackCopyButtonLabel);
					add("tab", new JLabel("<html><i>" + getText("ui.default") + ": \"" + getText("copyFeedback") + "\"</i></html>"));
					add("br", useFeedbackSendButton);
					add("br", new JLabelFor(getText("ui.sendButtonLabel"), feedbackSendButtonLabel));
					add("tab", feedbackSendButtonLabel);
					add("tab", new JLabel("<html><i>" + getText("ui.default") + ": \"" + getText("sendFeedback") + "\"</i></html>"));
				}
			};
			
			ControlPanel feedbackForm = new ControlPanel() {
				
				JTextArea feedbackTemplate = new JSmartTextArea(5, 24);
				JScrollPane feedbackTemplatePane = new JScrollPane(feedbackTemplate);
				//JLabel feedbackTemplateInfo = new JLabel(infoIcon);
				
				{
					feedbackTemplate.setToolTipText(getText("ui.feedbackTemplateInfo"));
					feedbackTemplate.setEditable(true);
					feedbackTemplate.setLineWrap(true);
					feedbackTemplate.setWrapStyleWord(true);
					feedbackTemplate.setFont(mono);
					feedbackTemplate.setTabSize(2);
					feedbackTemplatePane.setBorder(BorderFactory.createTitledBorder(getText("ui.feedbackTemplate")));
					
					add("hfill vfill", feedbackTemplatePane);
					
				}
			};
			
			ControlPanel feedbackInstruction = new ControlPanel() {
				
				JSmartTextArea feedbackInstructions = new JSmartTextArea(6, 24);	
				JScrollPane feedbackInstructionsPane = new JScrollPane(feedbackInstructions);
				JSmartTextArea feedbackInstructionsBox = new JSmartTextArea(6, 24);	
				JScrollPane feedbackInstructionsBoxPane = new JScrollPane(feedbackInstructionsBox);
				
				{
					
					feedbackInstructions.setToolTipText(getText("ui.feedbackInstructionsInfo"));
					feedbackInstructions.setEditable(true);
					feedbackInstructions.setLineWrap(true);
					feedbackInstructions.setWrapStyleWord(true);
					feedbackInstructions.setFont(mono);
					feedbackInstructions.setTabSize(2);
					allowHTMLEditing(feedbackInstructions);
					feedbackInstructionsPane.setBorder(BorderFactory.createTitledBorder(getText("ui.feedbackInstructions")));
					
					feedbackInstructionsBox.setToolTipText(getText("ui.feedbackInstructionsBoxInfo"));
					feedbackInstructionsBox.setEditable(true);
					feedbackInstructionsBox.setLineWrap(true);
					feedbackInstructionsBox.setWrapStyleWord(true);
					feedbackInstructionsBox.setFont(mono);
					feedbackInstructionsBox.setTabSize(2);
					allowHTMLEditing(feedbackInstructionsBox);
					feedbackInstructionsBoxPane.setBorder(BorderFactory.createTitledBorder(getText("ui.feedbackInstructionsBox")));

					add("hfill", feedbackInstructionsPane);
					add("br hfill", feedbackInstructionsBoxPane);
				}
			};
			
			JTabbedPane feedbackTabs = new JTabbedPane() {
								
				{
					feedbackOptions.setBorder(emptyBorder);
					feedbackForm.setBorder(emptyBorder);
					feedbackInstruction.setBorder(emptyBorder);
					
					addTab(getText("ui.settings"), feedbackOptions);
					addTab(getText("ui.feedbackForm"), feedbackForm);
					addTab(getText("ui.instructions"), feedbackInstruction);
				}
			};
			
			{
				if (!commercialUseAllowed) {
					showFeedback.setSelected(false);
					commercialMonitor.add(showFeedback);
				}
				showFeedback.setToolTipText(getText("ui.useFeedbackInfo"));
				ComponentUtilities.whenSelectedEnable(showFeedback, new JComponent[] {feedbackOptions,feedbackForm,feedbackInstruction});
				feedbackOptions.setBorder(emptyBorder);

				add(showFeedback);
				add("br vfill hfill", feedbackTabs);

				putClientProperty("helpPage", helpRoot + "/Sections/Feedback");
			}

		};
		
		/*
		// Audio clips

		ControlPanel audioClips = new ControlPanel() {

			JCheckBox useAudioClipButton = new JCheckBox(getText("ui.playAudioClips"), false);
			JSpinner audioClipVolume = new JSpinner(new SpinnerNumberModel(75, 1, 100, 1));
			JCheckBox copyAudioClips = new JCheckBox(getText("ui.copyAudioClips"));

			{
				useAudioClipButton.setToolTipText(getText("ui.playAudioClipsInfo"));
				audioClipVolume.setToolTipText(getText("ui.audioClipVolumeInfo"));
				copyAudioClips.setToolTipText(getText("ui.copyAudioClipsInfo") + " (" + getText("ui.oldMethod") + ")");
				
				add("", useAudioClipButton);
				add("br", new JLabel(getText("ui.initialVolume")));
				add("", audioClipVolume);
				add("", new JLabel("%"));
				add("br", copyAudioClips);
				
				//putClientProperty("helpPage", helpRoot + "/Site/Sound_clips");
			}
		};
		*/

		/*	---------------------------------------------------------------
								Sections / Main content
			--------------------------------------------------------------- */
		
		// Neighboring folders section

		ControlPanel neighboringFolders = new ControlPanel() {
			
			JCheckBox linkNeighboringFolders = new JCheckBox(getText("ui.linkNeighboringFolders"), true);
			JCheckBox neighboringFolderBgImage = new JCheckBox(getText("ui.neighboringFolderBgImage"), true);
			JCheckBox neighboringFolderSkipLevels = new JCheckBox(getText("ui.skipFolderLevels"), false);
			JCheckBox neighboringFolderLoop = new JCheckBox(getText("ui.neighboringFolderLoop"), false);

			{
				linkNeighboringFolders.setToolTipText(getText("ui.linkNeighboringFoldersInfo"));
				neighboringFolderBgImage.setToolTipText(getText("ui.neighboringFolderBgImageInfo"));
				neighboringFolderSkipLevels.setToolTipText(getText("ui.skipFolderLevelsInfo"));
				neighboringFolderLoop.setToolTipText(getText("ui.neighboringFolderLoopInfo"));
				ComponentUtilities.whenSelectedEnable(linkNeighboringFolders, new JComponent[]{ neighboringFolderBgImage, neighboringFolderSkipLevels, neighboringFolderLoop });

				add(linkNeighboringFolders);
				add("br", neighboringFolderSkipLevels);
				add("br", neighboringFolderBgImage);
				add("br", neighboringFolderLoop);
				
				putClientProperty("helpPage", helpRoot + "/Sections/Neighboring_folders");
			}
		};				
		
		// Custom content section
		
		ControlPanel customSection = new ControlPanel() {

			JSmartTextArea customSection = new JSmartTextArea(7, 20);
			JScrollPane customSectionPane = new JScrollPane(customSection);
			JCheckBox customSectionTopLevelOnly = new JCheckBox(getText("ui.topLevelOnly"), false);

			{
				customSection.setEditable(true);
				customSection.setLineWrap(false);
				customSection.setWrapStyleWord(true);
				customSection.setFont(mono);
				customSection.setTabSize(2);
				allowHTMLEditing(customSection);
				customSectionPane.setBorder(BorderFactory.createTitledBorder(getText("ui.customSectionHint")));

				add("hfill vfill", customSectionPane);
				add("br", customSectionTopLevelOnly);
				
				putClientProperty("helpPage", helpRoot + "/Sections/Custom_content");
			}
		};

		/*	---------------------------------------------------------------
									Lightbox
			--------------------------------------------------------------- */
		
		ControlPanel lightboxDesign = new ControlPanel() {
		
			JCheckBox lightboxFullscreen = new JCheckBox(getText("ui.lightboxFullscreen"));
			JCheckBox slideshowFullscreen = new JCheckBox(getText("ui.slideshowFullscreen"));
			JCheckBox rightClickProtect = new JCheckBox(getText("ui.rightClickProtect"));
			JComboBox mouseWheelAction = new JComboBox(new Object[] {
				new Item("", getText("ui.default")), 
				new Item("navigation", getText("ui.navigation")),
				new Item("zoom", getText("ui.zoom"))
			});
			JCheckBox enableKeyboard = new JCheckBox(getText("ui.enableKeyboard"), true);
			JComboBox clickAction = new JComboBox(new Object[] {
				new Item("donothing", getText("ui.doNothing")), 
				new Item("nextimage", getText("ui.nextImage")),
				new Item("togglecontrols", getText("ui.toggleControls")),
				new Item("toggleall", getText("ui.toggleAllPanels"))
			});
			JCheckBox clickBesideForIndex = new JCheckBox(getText("ui.clickBesideForIndex"));
						
			{
				lightboxUseMainBg.setToolTipText(getText("ui.lightboxUseMainBackgroundInfo"));
				lightboxBackgroundColor.getTextComponent().setFont(mono);
				lightboxBackgroundColor.setToolTipText(getText("ui.backgroundColorInfo"));
				lightboxBorderColor.getTextComponent().setFont(mono);
				lightboxBorderColor.setToolTipText(getText("ui.borderColorInfo"));
				rightClickProtect.setToolTipText(getText("ui.rightClickProtectInfo"));
				mouseWheelAction.setToolTipText(getText("ui.mouseWheelActionInfo"));
				enableKeyboard.setToolTipText(getText("ui.enableKeyboardInfo"));
				clickBesideForIndex.setToolTipText(getText("ui.clickBesideForIndexInfo"));

				add("", lightboxUseMainBg);
				add("br", new JLabelFor(getText("ui.backgroundColor"), lightboxBackgroundColor));
				add("tab", lightboxBackgroundColor);
				add("", lightboxBackgroundColor.getTextComponent());
				add("br", new JLabelFor(getText("ui.borderColor"), lightboxBorderColor));
				add("tab", lightboxBorderColor);
				add("", lightboxBorderColor.getTextComponent());
				add("br", new JLabelFor(getText("ui.borderWidth"), lightboxBorderWidth));
				add("tab", lightboxBorderWidth);
				add("", new JLabel("px"));
				add("br", new JLabel(getText("ui.cornerRadius")));
				add("tab", imgCornerRadius);
				add("", new JLabel("px"));
				add("br", new JLabelFor(getText("ui.padding"), fitPadding));
				add("tab", fitPadding);
				add(new JLabel("px"));
				add("br", lightboxFullscreen);
				add("br", slideshowFullscreen);
				add("br", rightClickProtect);
				add("br", enableKeyboard);
				add("br", new JLabelFor(getText("ui.mouseWheelAction"), mouseWheelAction));
				add("tab", mouseWheelAction);
				add("br", new JLabelFor(getText("ui.clickAction"), clickAction));
				add("tab", clickAction);
				add("br", clickBesideForIndex);

				putClientProperty("helpPage", helpRoot + "/Lightbox/Settings");
			}
		};

		ControlPanel lightboxControls = new ControlPanel() {

			private JLabel iconFit = new JLabel(icon("expand"));
			private JLabel iconPlay = new JLabel(icon("play"));
			private JLabel iconFullscr = new JLabel(icon("fullscreen"));
			JCheckBox slideshowTimingControl = new JCheckBox(getText("ui.canChangeSlideshowTiming"));
			
			ControlPanel fitImageSettings = new ControlPanel() {
			
				{					
					displayOriginals.setToolTipText(getText("ui.displayOriginalsInfo"));
					
					add(fitImage);
					add("br", new JLabel(getText("ui.maxZoom")));
					add(" ", maxZoom);
					add(" ", new JLabel("<html><i>1.0: " + getText("ui.neverScaleUp") + "</i></html>"));
					add("br", displayOriginals);
					
				}
			};
			
			{
				showFitToggle.setToolTipText(getText("ui.showFitToggleInfo"));
				showStartStop.setToolTipText(getText("ui.showStartStopInfo"));
				fitImage.setToolTipText(getText("ui.fitImagesInfo"));
				((JSpinner.DefaultEditor)maxZoom.getEditor()).getTextField().setColumns(4);
				ComponentUtilities.whenSelectedEnable(showFitToggle, new JComponent[]{iconFit, zoomSlider});
				zoomSlider.addActionListener((ActionEvent e) -> {
					if (zoomSlider.isSelected()) {
						iconFit.setIcon(icon("zoom-level"));
					} else {
						iconFit.setIcon(icon("expand"));
					}
				});
				ComponentUtilities.whenSelectedEnable(showStartStop, new JComponent[]{iconPlay, slideshowTimingControl});
				ComponentUtilities.whenSelectedEnable(showFullscreen, new JComponent[]{iconFullscr});
				slideshowTimingControl.setToolTipText(getText("ui.slideshowTimingControlInfo"));
				
				fitImageSettings.setBorder(emptyBorder);
				
				add("", iconFit);
				add("tab", showFitToggle);
				add("tab", zoomSlider);
				add("br tab", fitImageSettings);
				add("br", iconPlay);
				add("tab", showStartStop);
				add(" ", slideshowTimingControl);
				add("br", iconFullscr);
				add("tab", showFullscreen);

				putClientProperty("helpPage", helpRoot + "/Lightbox/Control_bar");
			}
		};

		ControlPanel lightboxButtons = new ControlPanel() {
			
			JCheckBox regionsVisible = new JCheckBox(getText("ui.visibleByDefault"));
			JTextField regionsBtnText = new JTextField(getText("regionsBtn"), 8);
			JCheckBox regionsSkipEmpty = new JCheckBox(getText("ui.skipEmptyTags"));
			JCheckBox downloadNonImages = new JCheckBox(getText("ui.enableOnNonImages"));
			JCheckBox downloadScaled = new JCheckBox(getText("ui.enableOnScaled"));
			JLabel regionsLabel = new JLabel(getText("ui.label"));
			JLabel mandatoryInfo = new JLabel(mandatory);
			JCheckBox useFotomoto = new JCheckBox("Fotomoto");
			JTextField fotomotoID = new JSmartTextField(20);

			ControlPanel photoData = new ControlPanel() {

				JTextArea photoDataTemplate = new JSmartTextArea(12, 24);
				JScrollPane photoDataPane = new JScrollPane(photoDataTemplate);
				JButton reset = new JButton(getText("ui.resetToDefaults"));
				WrappableJLabel listMetadataHint = new WrappableJLabel("<html><i>" + getText("ui.listMetadataHint") + "</i></html>");

				{
					photoDataTemplate.setEditable(true);
					photoDataTemplate.setLineWrap(true);
					photoDataTemplate.setWrapStyleWord(true);
					photoDataTemplate.setFont(mono);
					listMetadataHint.setPreferredWidth(uiWidth - 160);
					reset.addMouseListener(new MouseAdapter() {  
						@Override
						public void mouseReleased(MouseEvent e) {
							String s = new SkinModel().photoDataTemplate;
							if (s != null && s.length() > 0) {
								photoDataTemplate.setText(s);
							}
					}});				

					add(" ", reset);
					add("br hfill vfill", photoDataPane);
					add("br", listMetadataHint);
				}
			};

			{
				showMapBtn.setToolTipText(getText("ui.showMapInfo"));
				showShopBtn.setToolTipText(getText("ui.showShopInfo"));
				showFeedbackBtn.setToolTipText(getText("ui.showFeedbackInfo"));
				showRegions.setToolTipText(getText("ui.showRegionsInfo"));
				regionsVisible.setToolTipText(getText("ui.regionsVisibleInfo"));
				ComponentUtilities.whenSelectedEnable(showRegions, new JComponent[]{regionsVisible, regionsLabel, regionsBtnText, regionsSkipEmpty});
				showShare.setToolTipText(getText("ui.showShareInfo"));
				printImageButton.setToolTipText(getText("ui.printImageButtonInfo"));
				downloadNonImages.setToolTipText(getText("ui.enableDownloadNonImagesInfo"));
				ComponentUtilities.whenSelectedEnable(downloadBtn, new JComponent[]{downloadNonImages, downloadScaled});
				fotomotoID.setToolTipText(getText("ui.fotomotoIDInfo"));
				mandatoryInfo.setToolTipText(getText("ui.mandatory"));
				ComponentUtilities.whenSelectedEnable(useFotomoto, new JComponent[]{fotomotoID});
				showPhotoData.setToolTipText(getText("ui.showPhotoDataInfo"));
				ComponentUtilities.whenSelectedEnable(showPhotoData, new JComponent[]{photoData});

				add("br", new JLabel(icon("location")));
				add("tab", showMapBtn);
				add("br", new JLabel(icon("shopping-cart")));
				add("tab", showShopBtn);
				add("br", new JLabel(icon("email-send")));
				add("tab", showFeedbackBtn);
				add("br", new JLabel(icon("facetag")));
				add("tab", showRegions);
				add("tab", regionsVisible);
				add("br tab", regionsLabel);
				add(regionsBtnText);
				add("tab", regionsSkipEmpty);
				add("br", new JLabel(icon("connect")));
				add("tab", showShare);
				add("br", new JLabel(icon("printer")));
				add("tab", printImageButton);
				add("br", new JLabel(icon("download")));
				add("tab", downloadBtn);
				add("br tab", downloadNonImages);
				add("tab", downloadScaled);
				add("br", new JLabel(icon("fotomoto")));
				add("tab", useFotomoto);
				add("br tab", new JLabelFor(getText("ui.storeId"), fotomotoID));
				add(mandatoryInfo);
				add("", fotomotoID);
				add(new JLinkLabel("https://my.fotomoto.com/signup", getText("ui.signUp")));
				add("br", new JLabel(icon("camera")));
				add("tab", showPhotoData);
				add("br hfill vfill", photoData);
				
				putClientProperty("helpPage", helpRoot + "/Lightbox/Buttons");
			}
		};
											
		ControlPanel mainImage = new ControlPanel() {

			JComboBox transitionType = new JComboBox(new Object[] {
				new Item("crossFade", getText("ui.crossFade")),
				new Item("crossFadeAndSlide", getText("ui.crossFadeAndSlide")),
				new Item("crossFadeAndZoom", getText("ui.crossFadeAndZoom"))
			});
			JSpinner transitionSpeed = new JSpinner(new SpinnerNumberModel(800, 100, 5000, 100));
			JSpinner slideshowDelay = new JSpinner(new SpinnerNumberModel(4000, 50, 20000, 50));
			//JCheckBox slideshowLoop = new JCheckBox(getText("ui.loop"));
			JCheckBox slideshowAuto = new JCheckBox(getText("ui.autoStart"));
			JComboBox afterLast = new JComboBox(new Object[] {
				new Item("donothing", getText("ui.doNothing")), 
				new Item("startover", getText("startOver")),
				new Item("onelevelup", getText("upOneLevel")),
				new Item("backtoindex", getText("backToIndex")),
				new Item("nextfolder", getText("nextFoldersFirstImage")),
				new Item("nextindex", getText("nextIndex")),
				new Item("ask", getText("ui.ask"))
			});
			JCheckBox videoAutoPlay = new JCheckBox(getText("ui.startVideo"));
			JLabel videoAutoPlayInfo = new JLabel(infoIcon);
			JCheckBox videoLoop = new JCheckBox(getText("ui.loopVideos"));
			JCheckBox useAutoPano = new JCheckBox(getText("ui.enableAutoPano"));
			JLabel autoPanoInfo = new JLabel(infoIcon);
			JLabel autoPanoStartLabel = new JLabel(getText("ui.startAt"));
			JComboBox autoPanoStart = new JComboBox(new Object[] {
				new Item("left", getText("ui.leftOrTop")), 
				new Item("center", getText("ui.center")),
				new Item("right", getText("ui.rightOrBottom"))
			});
			JLabel autoPanoSpeedLabel = new JLabel(getText("ui.speed"));
			JComboBox autoPanoSpeed = new JComboBox(new Object[] {
				new Item("veryslow", getText("ui.verySlow")),
				new Item("slow", getText("ui.slow")), 
				new Item("moderate", getText("ui.moderate")),
				new Item("medium", getText("ui.medium")),
				new Item("quick", getText("ui.quick")),
				new Item("fast", getText("ui.fast"))
			});
			JCheckBox use360Player = new JCheckBox(getText("ui.use360Player"));
			JCheckBox autoRotate360 = new JCheckBox(getText("ui.autoRotate"));
			JSpinner rotateSpeed360 = new JSpinner(new SpinnerNumberModel(-2, -8, 8, 1));
			JLabel rpm = new JLabel(getText("ui.rpm"));
			JCheckBox useGoogleDocs = new JCheckBox(getText("ui.useGoogleDocsViewer"));
			ControlPanel autoPanoSettings = new ControlPanel() {
				{
					add("", autoPanoStartLabel);
					add("tab", autoPanoStart);
					add(" ", autoPanoSpeedLabel);
					add("tab", autoPanoSpeed);
				}
			};

			{
				transitionSpeed.setToolTipText(getText("ui.transitionSpeedInfo"));
				slideshowDelay.setToolTipText(getText("ui.slideshowDelayInfo"));
				afterLast.setToolTipText(getText("ui.afterLastInfo"));
				videoAutoPlayInfo.addMouseListener(new MouseAdapter() {  
					@Override
					public void mouseReleased(MouseEvent e) {
						JOptionPane.showMessageDialog(window, getText("ui.videoAutoplayInfo1"), getText("ui.warning"), JOptionPane.WARNING_MESSAGE);
				}});
				useAutoPano.setToolTipText(getText("ui.enableAutoPanoInfo"));
				useAutoPano.addItemListener(e -> {
					if (isSkinReady() && e.getStateChange() == ItemEvent.SELECTED) {
						String idim = engine.getImageSize();
						int iw = Integer.parseInt(idim.split("x")[0]);
						int ih = Integer.parseInt(idim.split("x")[1]);

						if (iw < 3000) {
							Object[] options = { 
								getText("yes"),
								getText("no")
							};
							int n = JOptionPane.showOptionDialog(window, 
								getText("ui.autoPanoInfo") + "\n\n" + getText("ui.changeImageDimensionsQuestion"), 
								getText("ui.imageDimensionsTooSmall"), 
								JOptionPane.YES_NO_OPTION, 
								JOptionPane.INFORMATION_MESSAGE,
								null,
								options,
								options[0]
							);
							if (n == 0) {
								try {
									window.ui2Engine();
									engine.setImageSize("4000x"+ih);
									JAlbum.logger.log(Level.FINE, "Image width has changed from {0}px to {1}px.", new Object[]{iw, "4000"});
									window.engine2UI();
								} catch (ParameterException ex) {
									throw new RuntimeException(ex);
								}
								useAutoPano.setSelected(true);
							}
						}
					}
				});
				autoPanoInfo.addMouseListener(new MouseAdapter() {  
					@Override
					public void mouseReleased(MouseEvent e) {
						JOptionPane.showMessageDialog(window, getText("ui.autoPanoInfo"), "Warning", JOptionPane.WARNING_MESSAGE);
				}});
				ComponentUtilities.whenSelectedEnable(useAutoPano, new JComponent[]{autoPanoSettings});
				use360Player.setToolTipText(getText("ui.use360PlayerInfo"));
				ComponentUtilities.whenSelectedEnable(use360Player, new JComponent[]{autoRotate360, rotateSpeed360, rpm});
				useGoogleDocs.setToolTipText(getText("ui.useGoogleDocsInfo"));

				add("", new JLabelFor(getText("ui.transition"), transitionType));
				add("tab", transitionType);
				add("br", new JLabelFor(getText("ui.transitionSpeed"), transitionSpeed));
				add("tab", transitionSpeed);
				add(new JLabel("ms"));
				add("br", new JLabel(getText("ui.slideshowDelay")));
				add("tab", slideshowDelay);
				add(new JLabel("ms"));
				add("br tab", slideshowAuto);
				add("br", new JLabelFor(getText("ui.afterLast"), afterLast));
				add("tab", afterLast);
				add("br", videoAutoPlay);
				add("", videoAutoPlayInfo);
				add("br", videoLoop);
				add("br", useAutoPano);
				add("", autoPanoInfo);
				add("br", autoPanoSettings);
				add("br", use360Player);
				add("tab", autoRotate360);
				//add("tab", new JLabelFor(getText("ui.speed"), rotateSpeed360));
				add("tab", rotateSpeed360);
				add(rpm);
				add("br", useGoogleDocs);

				putClientProperty("helpPage", helpRoot + "/Lightbox/Main_image");
			}
		};

		ControlPanel lightboxCaption = new ControlPanel() {
			
			ControlPanel titleCaption = new ControlPanel(getText("ui.topBar")) {
				
				JScrollPane titleCaptionPane = new JScrollPane(titleCaptionTemplate);
				
				{
					showImageNumbers.setToolTipText(getText("ui.showImageNumbersInfo"));
					titleCaptionPresets.addItemListener(listener -> {
						if (isSkinReady()) {
							String s = getSelectedValue(titleCaptionPresets);
							if (!s.equals(CUSTOM)) {
								titleCaptionTemplate.setText(s);
							}
						}
					});	
					new StateMonitor() {
						public void onChange() {
							String s = getSelectedValue(titleCaptionPresets);
							titleCaptionPane.setVisible(s.equals(CUSTOM));
						}
					}.add(titleCaptionPresets).done();
					titleCaptionTemplate.setEditable(true);
					titleCaptionTemplate.setLineWrap(true);
					titleCaptionTemplate.setWrapStyleWord(true);
					titleCaptionTemplate.setFont(mono);
					titleCaptionTemplate.setTabSize(2);
					titleCaptionPane.setBorder(BorderFactory.createTitledBorder(getText("ui.captionTemplate")));					
					
					add("", showImageNumbers);
					add("br", new JLabelFor(getText("ui.template"), titleCaptionPresets));
					add("tab", titleCaptionPresets);
					add("br hfill", titleCaptionPane);
					
				}
			};
			
			ControlPanel imageCaption = new ControlPanel(getText("ui.caption")) {
				
				JScrollPane imageCaptionPane = new JScrollPane(imageCaptionTemplate);
				
				{	
					imageCaptionPresets.addItemListener(listener -> {
						String s = getSelectedValue(imageCaptionPresets);
						if (!s.equals(CUSTOM)) {
							imageCaptionTemplate.setText(s);
						}
					});	
					new StateMonitor() {
						public void onChange() {
							String s = getSelectedValue(imageCaptionPresets);
							imageCaptionPane.setVisible(s.equals(CUSTOM));
						}
					}.add(imageCaptionPresets).done();
					imageCaptionTemplate.setEditable(true);
					imageCaptionTemplate.setLineWrap(true);
					imageCaptionTemplate.setWrapStyleWord(true);
					imageCaptionTemplate.setFont(mono);
					imageCaptionTemplate.setTabSize(2);
					imageCaptionPane.setBorder(BorderFactory.createTitledBorder(getText("ui.captionTemplate")));					

					add("", new JLabelFor(getText("ui.template"), imageCaptionPresets));
					add("tab", imageCaptionPresets);
					add("br hfill", imageCaptionPane);
					add("br", captionVisible);
				}
			};

			{
				add("hfill", titleCaption);
				add("br hfill", imageCaption);
				
				putClientProperty("helpPage", helpRoot + "/Lightbox/Caption");
			}
		};

		ControlPanel thumbStripPanel = new ControlPanel() {
			
			ControlPanel thumbStripSettings  = new ControlPanel() {
				
				{
					thumbstripVisible.setToolTipText(getText("ui.thumbstripVisibleInfo"));
					
					add("", thumbstripVisible);
					/*
					add("", new JLabel(getText("ui.position")));
					add("", thumbstripPosition);
					*/
					add("br", new JLabel(getText("ui.size")));
					add("", thumbstripHeight);	
				}
			};
			
			{
				useThumbstrip.setToolTipText(getText("ui.showThumbstripInfo"));
				thumbStripSettings.setBorder(emptyBorder);
				ComponentUtilities.whenSelectedEnable(useThumbstrip, new JComponent[]{thumbStripSettings});
				
				add("", useThumbstrip);
				add("br tab", thumbStripSettings);

			}
		};
			
		/*	---------------------------------------------------------------
									Advanced
			--------------------------------------------------------------- */

		//	Custom code
		
		ControlPanel customCodePanel = new ControlPanel() {

			ControlPanel headHookTab = new ControlPanel() {

				JTextArea headHook = new JSmartTextArea(7, 20);
				JScrollPane headHookPane = new JScrollPane(headHook);

				{
					headHook.setEditable(true);
					headHook.setLineWrap(false);
					headHook.setFont(mono);
					headHook.setTabSize(2);
					headHookPane.setBorder(BorderFactory.createTitledBorder(getText("ui.headText")));

					add("hfill vfill", headHookPane);
				}
			};

			ControlPanel bodyHookTab = new ControlPanel() {

				JTextArea bodyHook = new JSmartTextArea(7, 20);
				JScrollPane bodyHookPane = new JScrollPane(bodyHook);

				{
					bodyHook.setEditable(true);
					bodyHook.setLineWrap(false);
					bodyHook.setFont(mono);
					bodyHook.setTabSize(2);
					bodyHookPane.setBorder(BorderFactory.createTitledBorder(getText("ui.bodyText")));

					add("hfill vfill", bodyHookPane);
				}
			};

			ControlPanel cssHookTab = new ControlPanel() {

				JTextArea cssHook = new JSmartTextArea(15, 20);
				JScrollPane cssHookPane = new JScrollPane(cssHook);
				WrappableJLabel info = new WrappableJLabel("<html><i>" + getText("ui.cssText") + "</i></html>");
				
				{
					cssHook.setEditable(true);
					cssHook.setLineWrap(false);
					cssHook.setFont(mono);
					cssHook.setTabSize(2);
					cssHookPane.setBorder(BorderFactory.createTitledBorder(getText("ui.cssTitle")));
					info.setPreferredWidth(700);
					
					add("hfill vfill", cssHookPane);
					add("br", info);
				}
			};

			ControlPanel jsHookTab = new ControlPanel() {

				JTextArea jsHook = new JSmartTextArea(15, 20);
				JScrollPane jsHookPane = new JScrollPane(jsHook);
				WrappableJLabel info = new WrappableJLabel("<html><i>" + getText("ui.javascriptText") + "</i></html>");
				
				{
					jsHook.setEditable(true);
					jsHook.setLineWrap(false);
					jsHook.setFont(mono);
					jsHook.setTabSize(2);
					jsHookPane.setBorder(BorderFactory.createTitledBorder(getText("ui.javascriptTitle")));
					info.setPreferredWidth(700);
					
					add("hfill vfill", jsHookPane);
					add("br", info);
				}
			};

			JTabbedPane customCodeTabs = new JTabbedPane() {				

				{
					addTab("<HEAD>", icon("code"), headHookTab);
					addTab("<BODY>", icon("code"), bodyHookTab);
					addTab("CSS", icon("css"), cssHookTab);
					addTab("JavaScript", icon("javascript"), jsHookTab);
				}
			};

			{				
				add("hfill vfill", customCodeTabs);

				putClientProperty("helpPage", helpRoot + "/Advanced/Custom_code");
			}
		};
		
		//	Site admin
		
		ControlPanel siteAdminPanel = new ControlPanel() {

			ControlPanel serverRelatedPanel = new ControlPanel(getText("ui.serverRelated")) {
				
				JTextField uploadPath = new JSmartTextField(20);
				JCheckBox useFavicon = new JCheckBox(getText("ui.useTheSkinsFavicon"), true);				
				JCheckBox useMsServer = new JCheckBox(getText("ui.useMsServer"));
				JCheckBox useExpiry = new JCheckBox(getText("ui.useExpiry"), false);
				JCheckBox useRobotsTxt = new JCheckBox(getText("ui.useRobotsTxt"), false);
				JCheckBox avoidCDNs = new JCheckBox(getText("ui.avoidCDNs"), false);
				JCheckBox copyGoogleFonts = new JCheckBox(getText("ui.copyGoogleFonts"), false);
				JCheckBox addLatinExt = new JCheckBox(getText("ui.addLatinExtended"), true);
				WrappableJLabel uploadPathInfo = new WrappableJLabel("<html><i>" + getText("ui.copyPasteAlbumURLHere") + "</i></html>");
				
				{
					uploadPath.setToolTipText(getText("ui.uploadPathInfo"));
					uploadPathInfo.setPreferredWidth(360);
					useFavicon.setToolTipText(getText("ui.useFaviconInfo"));
					useMsServer.setToolTipText(getText("ui.useMsServerInfo"));
					useExpiry.setToolTipText(getText("ui.useExpiryInfo"));
					useRobotsTxt.setToolTipText(getText("ui.useRobotsTxtInfo"));
					avoidCDNs.setToolTipText(getText("ui.avoidCDNsInfo"));
					copyGoogleFonts.setToolTipText(getText("ui.copyGoogleFontsInfo"));
					addLatinExt.setToolTipText(getText("ui.addLatinExtendedInfo"));
					
					add(new JLabel(getText("ui.uploadPath")));
					add("", uploadPath);
					add("br", new JLabel(infoIcon));
					add("", uploadPathInfo);
					add("br", useFavicon);
					add(" ", new JLabel(svgIcon("favicon", new Dimension(20, 20))));
					add("br", useMsServer);
					add("br", useExpiry);
					add("br", useRobotsTxt);
					add("br", avoidCDNs);
					add("br", copyGoogleFonts);
					add("br", addLatinExt);
				}
			};			
			
			ControlPanel imageRelatedPanel = new ControlPanel(getText("ui.images")) {
				
				//JCheckBox hiDpiThemeImage = new JCheckBox(getText("ui.hiDpiThemeImage"));
				JComboBox shareImageDims = new JComboBox(new Object[] {
					"640x480",
					"1024x768",
					"1200x900",
					"1600x1200"
				});
				
				{
					shareImageDims.setEditable(true);
					
					//add(hiDpiThemeImage);
					add(new JLabel(getText("ui.shareImageDimensions")));
					add("", shareImageDims);
				}
			};			
			
			ControlPanel googlePanel = new ControlPanel(getText("ui.googleAnalytics")) {

				JComboBox googleAnalytics = new JComboBox(new Object[] {
					new Item("none", getText("ui.none")), 
					new Item("gtag", "Global Site Tag (gtag.js)"),
					new Item("universal", "Universal (analytics.js)"),
					new Item("classic", "Classic (ga.js) [Legacy]")
				});
				JTextField googleSiteID = new JSmartTextField(20);
				JLabel mandatoryInfo = new JLabel(mandatory);
				JCheckBox supportDoubleclick = new JCheckBox(getText("ui.supportDoubleclick"), false);

				{
					googleSiteID.setToolTipText(getText("ui.googleSiteIDInfo"));
					mandatoryInfo.setToolTipText(getText("ui.mandatory"));

					add("", new JLabelFor(getText("ui.version"), googleAnalytics));
					add(" ", googleAnalytics);
					add(" ", new JLinkLabel("https://www.google.com/analytics/", getText("ui.signUp")));
					add("br", new JLabelFor(getText("ui.googleSiteID"), googleSiteID));
					add(mandatoryInfo);
					add("", googleSiteID);
					add("br", supportDoubleclick);
				}
			};
			
			ControlPanel seoPanel = new ControlPanel(getText("ui.searchEngineOptimization")) {
				
				JTextField titleSEOText = new JSmartTextField(20);
				JTextField descriptionSEOText = new JSmartTextField(20);
				JSpinner preloadThumbs = new JSpinner(new SpinnerNumberModel(20, 0, 1000, 10));
				JCheckBox addAltTags = new JCheckBox(getText("ui.addAltTags"), true);
				JCheckBox slideRedirect = new JCheckBox(getText("ui.redirectSlidePages"), true);
				
				{
					titleSEOText.setToolTipText(getText("ui.titleSEOTextInfo"));
					descriptionSEOText.setToolTipText(getText("ui.descriptionSEOTextInfo"));
					//preloadThumbs.setToolTipText(getText("ui.preloadThumbsInfo"));
					addAltTags.setToolTipText(getText("ui.addAltTagsInfo"));
					writeSitemapXml.setToolTipText(getText("ui.createSitemapXmlInfo"));
					ComponentUtilities.whenSelectedEnable(writeSitemapXml, new JComponent[]{sitemapIncludeSlides});
					sitemapIncludeSlides.setToolTipText(getText("ui.sitemapIncludeSlidesInfo"));
					slideRedirect.setToolTipText(getText("ui.redirectSlidePagesInfo"));
					
					add("", new JLabelFor(getText("ui.titleSEOText"), titleSEOText));
					add("tab", titleSEOText);
					add("br", new JLabelFor(getText("ui.descriptionSEOText"), descriptionSEOText));
					add("tab", descriptionSEOText);
					add("br", new JLabel(getText("ui.preloadThumbs")));
					add("", preloadThumbs);
					add("br", addAltTags);
					add("br", writeSitemapXml);
					add("", sitemapIncludeSlides);
					add("br", slideRedirect);
				}
			};

			ControlPanel cookiePolicyPanel = new ControlPanel(getText("ui.trackingConsentAndCookiePolicy")) {

				JCheckBox askTrackingConsent = new JCheckBox(getText("ui.askTrackingConsent"), false);
				JCheckBox showCookiePolicy = new JCheckBox(getText("ui.showCookiePolicy"), false);				
				JTextField cookiePolicyStay = new JSmartTextField("8", 3);
				JTextField cookiePolicyUrl = new JSmartTextField(20);
				WrappableJLabel cookiePolicyUrlInfo = new WrappableJLabel("<html><i>" + getText("ui.cookiePolicyUrlInfo") + " </i></html>");

				{
					askTrackingConsent.setToolTipText(getText("ui.askTrackingConsentInfo"));
					showCookiePolicy.setToolTipText(getText("ui.showCookiePolicyInfo"));
					cookiePolicyUrlInfo.setPreferredWidth(320);
					
					add(askTrackingConsent);
					add("br", showCookiePolicy);
					add(" ", new JLabel(getText("ui.stay")));
					add("", cookiePolicyStay);
					add("", new JLabel("s"));
					add("br", new JLabel("URL"));
					add("tab", cookiePolicyUrl);
					add("br", new JLabel(infoIcon));
					add("tab", cookiePolicyUrlInfo);
				}
			};

			JCheckBox debugMode = new JCheckBox(getText("ui.debugMode"));

			ControlPanel leftPanel = new ControlPanel() {
				
				{
					add("hfill", serverRelatedPanel);
					add("br hfill", googlePanel);
					add("br", debugMode);					
				}
			};
			
			ControlPanel rightPanel = new ControlPanel() {
				
				{
					add("hfill", imageRelatedPanel);
					add("br hfill", seoPanel);
					add("br hfill", cookiePolicyPanel);
				}
			};
			
			{
				add(leftPanel);
				add(rightPanel);
				
				//putClientProperty("helpPage", helpRoot + "/Advanced/Site_admin");
			}
		};
					
		//	Custom Keys
		
		ControlPanel customKeysPanel = new ControlPanel() {
		
			JTextArea customKeys = new JSmartTextArea(7, 20);
			JScrollPane customKeysPane = new JScrollPane(customKeys);

			{
				customKeys.setEditable(true);
				customKeys.setLineWrap(false);
				customKeys.setFont(mono);
				customKeys.setTabSize(2);
				customKeysPane.setBorder(BorderFactory.createTitledBorder(getText("ui.customKeys")));

				add("hfill vfill", customKeysPane);
				add("br", new JLabel(getText("ui.customKeysInfo")));
			}
		};
		
		
		/******************************************************************************
		 * 
		 *								GUI main tabs
		 * 
		 ******************************************************************************/
		
		/*	---------------------------------------------------------------
									Site
			--------------------------------------------------------------- */
		
		ControlPanel design = new ControlPanel(new BorderLayout(20, 0)) {
			
			JTabbedPane designTabs = new JTabbedPane() {
				
				JScrollPane designPanel = new JScrollPane(generalDesign, JScrollPane.VERTICAL_SCROLLBAR_AS_NEEDED, JScrollPane.HORIZONTAL_SCROLLBAR_AS_NEEDED);
				JScrollPane topbarPanel = new JScrollPane(topbar, JScrollPane.VERTICAL_SCROLLBAR_AS_NEEDED, JScrollPane.HORIZONTAL_SCROLLBAR_AS_NEEDED);
				JScrollPane thumbnailsPanel = new JScrollPane(thumbnails, JScrollPane.VERTICAL_SCROLLBAR_AS_NEEDED, JScrollPane.HORIZONTAL_SCROLLBAR_AS_NEEDED);
				
				{
					designPanel.setBorder(emptyBorder);
					topbarPanel.setBorder(emptyBorder);
					thumbnailsPanel.setBorder(emptyBorder);
					this.setTabPlacement(SwingConstants.LEFT);
					
					addTab("<html><h4 " + tabStyle + ">" + getText("ui.general") + "</h4></html>", icon("settings"), designPanel);
					addTab("<html><h4 " + tabStyle + ">" + getText("ui.typography") + "</h4></html>", icon("font"), typography);
					addTab("<html><h4 " + tabStyle + ">" + getText("ui.topBar") + "</h4></html>", icon("header"), topbarPanel);
					addTab("<html><h4 " + tabStyle + ">" + getText("ui.thumbnails") + "</h4></html>", icon("thumbnails"), thumbnailsPanel);
					addTab("<html><h4 " + tabStyle + ">" + getText("ui.captions") + "</h4></html>", icon("caption"), thumbnailCaptions);
					addTab("<html><h4 " + tabStyle + ">" + getText("ui.markNewFiles") + "</h4></html>", icon("new"), markNew);
					addTab("<html><h4 " + tabStyle + ">" + getText("ui.neighboringFolders") + "</h4></html>", icon("prev-next-page"), neighboringFolders);
					addTab("<html><h4 " + tabStyle + ">" + getText("ui.footer") + "</h4></html>", icon("footer"), footerSettings);
				}
			};
			/*
			JPanel sitePreviewPanel = new JPanel(new BorderLayout(0, 0)) {

				{
					sitePreview.getView().setPreferredSize(new Dimension(previewWidth, uiHeight));

					add(sitePreview.getView(), BorderLayout.CENTER);
				}
			};
			*/
			{				
				((BorderLayout)(getLayout())).setVgap(0);
				((BorderLayout)(getLayout())).setHgap(0);
				designTabs.setBorder(emptyBorder);
				
				designTabs.setPreferredSize(new Dimension(uiWidth, uiHeight));
				designTabs.setMaximumSize(new Dimension(uiWidth + 100, uiHeight));
				//sitePreviewPanel.setBorder(BorderFactory.createLineBorder(JAlbumColor.imageBorder, 2));
				
				add(designTabs);
				//add(sitePreviewPanel, BorderLayout.EAST);				
			}
		};

		/*	---------------------------------------------------------------
									Top bar
			--------------------------------------------------------------- */

		ControlPanel topBar = new ControlPanel(new BorderLayout(20, 0)) {
			
			JTabbedPane topBarTabs = new JTabbedPane() {
				
				{
					this.setTabPlacement(SwingConstants.LEFT);
					
					addTab("<html><h4 " + tabStyle + ">" + getText("ui.navigationMenu") + "</h4></html>", icon("menu"), navigation);
					addTab("<html><h4 " + tabStyle + ">" + getText("ui.sharing") + "</h4></html>", icon("connect"), social);
					addTab("<html><h4 " + tabStyle + ">" + getText("ui.search") + "</h4></html>", icon("search"), search);
					addTab("<html><h4 " + tabStyle + ">" + getText("ui.searchNew") + "</h4></html>", icon("new"), searchNew);
					addTab("<html><h4 " + tabStyle + ">" + getText("ui.download") + "</h4></html>", icon("download"), download);
					addTab("<html><h4 " + tabStyle + ">" + getText("ui.albumInfo") + "</h4></html>", icon("info"), albumInfo);
					addTab("<html><h4 " + tabStyle + ">" + getText("ui.backgroundMusic") + "</h4></html>", icon("audio"), backgroundMusic);
				
				}
			};
			
			{
				((BorderLayout)(getLayout())).setVgap(0);
				((BorderLayout)(getLayout())).setHgap(0);
				
				topBarTabs.setBorder(emptyBorder);
				
				add(topBarTabs);
				//add(sectionOrdering, BorderLayout.EAST);				
			}
			
		};
		
		/*	---------------------------------------------------------------
									Functionality
			--------------------------------------------------------------- */

		ControlPanel functionality = new ControlPanel(new BorderLayout(20, 0)) {
			/*
			ControlPanel sectionOrdering = new ControlPanel() {
				
				JDraggableList sectionsOrder = new JDraggableList(new Object[] {
					new JLabelFor("<html><p style=\"padding:3px 0;\">" + getText("ui.map") + "</p></html>", icon("location")).name("map"),
					new JLabelFor("<html><p style=\"padding:3px 0;\">" + getText("ui.thumbnails") + "</p></html>", icon("thumbnails")).name("items"),
					new JLabelFor("<html><p style=\"padding:3px 0;\">" + getText("ui.neighboringFolders") + "</p></html>", icon("prev-next-page")).name("neighboringFolders"),
					new JLabelFor("<html><p style=\"padding:3px 0;\">" + getText("ui.customContent") + "</p></html>", icon("edit-document")).name("customContent")
				});

				{						
					add("center", new JLabel("<html><h4>" + getText("ui.ordering")+ "</h4></html>"));
					add("br", sectionsOrder);
					add("br center", new JLabel("<html><i>" + getText("ui.dragToReorder") + "</i></html>"));
				}
			};
			*/	
			JTabbedPane functionalityTabs = new JTabbedPane() {

				JScrollPane shoppingCartPanel = new JScrollPane(shoppingCart, JScrollPane.VERTICAL_SCROLLBAR_AS_NEEDED, JScrollPane.HORIZONTAL_SCROLLBAR_NEVER);
				
				{
					this.setTabPlacement(SwingConstants.LEFT);
					
					shoppingCartPanel.setBorder(emptyBorder);

					addTab("<html><h4 " + tabStyle + ">" + getText("ui.rating") + "</h4></html>", icon("star"), rating);
					addTab("<html><h4 " + tabStyle + ">" + getText("ui.tagCloud") + "</h4></html>", icon("tag"), tagCloud);
					addTab("<html><h4 " + tabStyle + ">" + getText("ui.map") + "</h4></html>", icon("location"), map);
					addTab("<html><h4 " + tabStyle + ">" + getText("ui.shoppingCart") + "</h4></html>", icon("shopping-cart"), shoppingCartPanel);
					addTab("<html><h4 " + tabStyle + ">" + getText("ui.feedback") + "</h4></html>", icon("email-send"), feedback);
					//addTab("<html><h4 " + tabStyle + ">" + getText("ui.audioClips") + "</h4></html>", icon("volume-up"), audioClips);
					addTab("<html><h4 " + tabStyle + ">" + getText("ui.customSection") + "</h4></html>", icon("edit-document"), customSection);

				}
			};
				
			{
				((BorderLayout)(getLayout())).setVgap(0);
				((BorderLayout)(getLayout())).setHgap(0);
				
				functionalityTabs.setBorder(emptyBorder);
				
				add(functionalityTabs);
				//add(sectionOrdering, BorderLayout.EAST);				
			}
		};
						

		/*	---------------------------------------------------------------
									Lightbox
			--------------------------------------------------------------- */
		
		ControlPanel lightbox = new ControlPanel(new BorderLayout(0, 0)) {

			JTabbedPane lightboxTabs = new JTabbedPane() {
				
				{
					this.setTabPlacement(SwingConstants.LEFT);
										
					addTab("<html><h4 " + tabStyle + ">" + getText("ui.settings") + "</h4></html>", icon("settings"), lightboxDesign);
					addTab("<html><h4 " + tabStyle + ">" + getText("ui.topBar") + "</h4></html>", icon("header"), lightboxControls);
					addTab("<html><h4 " + tabStyle + ">" + getText("ui.buttons") + "</h4></html>", icon("more-menu"), lightboxButtons);
					addTab("<html><h4 " + tabStyle + ">" + getText("ui.mainImage") + "</h4></html>", icon("image"), mainImage);
					addTab("<html><h4 " + tabStyle + ">" + getText("ui.caption") + "</h4></html>", icon("text"), lightboxCaption);
					addTab("<html><h4 " + tabStyle + ">" + getText("ui.thumbnails") + "</h4></html>", icon("thumbnails"), thumbStripPanel);
				}
			};
			/*
			ControlPanel lightboxPreviewPanel = new ControlPanel(new BorderLayout(0, 0)) {
				
				JPanel lbPreview = new JPanel(new BorderLayout(0, 0)) {
					
					{
						add(lightboxPreview.getView());
					}
				};
						
				{
					lbPreview.setPreferredSize(new Dimension(previewWidth, (int)Math.round(previewWidth * .8)));
					lbPreview.setBorder(BorderFactory.createLineBorder(JAlbumColor.imageBorder, 2));
					//lightboxPreview.getView().setMaximumSize(new Dimension(previewWidth, previewWidth));
					
					add(lbPreview, BorderLayout.NORTH);
				}
			};
			*/
			{
				((BorderLayout)(getLayout())).setVgap(0);
				((BorderLayout)(getLayout())).setHgap(0);
				
				lightboxTabs.setBorder(emptyBorder);
				lightboxTabs.setPreferredSize(new Dimension(uiWidth, uiHeight));
				lightboxTabs.setMaximumSize(new Dimension(uiWidth + 100, uiHeight));
				
				add(lightboxTabs);
				//add(lightboxPreviewPanel, BorderLayout.EAST);				
			}
		};
		
		/*	---------------------------------------------------------------
									Advanced
			--------------------------------------------------------------- */
		
		ControlPanel advanced = new ControlPanel(new BorderLayout(0, 0)) {
			
			JTabbedPane advancedTabs = new JTabbedPane() {
								
				JScrollPane siteAdminPane = new JScrollPane(siteAdminPanel, JScrollPane.VERTICAL_SCROLLBAR_AS_NEEDED, JScrollPane.HORIZONTAL_SCROLLBAR_NEVER);
				
				{
					customCodePanel.setBorder(emptyBorder);
					siteAdminPane.setBorder(emptyBorder);
					customKeysPanel.setBorder(emptyBorder);
					this.setTabPlacement(SwingConstants.LEFT);
					
					addTab("<html><h4 " + tabStyle + ">" + getText("ui.siteAdmin") + "</h4></html>", icon("wrench"), siteAdminPane);
					addTab("<html><h4 " + tabStyle + ">" + getText("ui.customCode") + "</h4></html>", icon("code"), customCodePanel);
					addTab("<html><h4 " + tabStyle + ">" + getText("ui.customKeys") + "</h4></html>", icon("rename"), customKeysPanel);
				}
			};
			
			{
				((BorderLayout)(getLayout())).setVgap(0);
				((BorderLayout)(getLayout())).setHgap(0);
				
				advancedTabs.setBorder(emptyBorder);
				
				add(advancedTabs);
			}
		};
		
	
		/*	---------------------------------------------------------------
									About
			--------------------------------------------------------------- */

		ControlPanel about = new ControlPanel(new BorderLayout(20, 0)) {
			
			private JTextArea readme = new JSmartTextArea( getFileContents("readme.txt"), 20, 30 );
			private JScrollPane aboutPane = new JScrollPane(readme, JScrollPane.VERTICAL_SCROLLBAR_ALWAYS, JScrollPane.HORIZONTAL_SCROLLBAR_NEVER);
	
			ControlPanel info = new ControlPanel() {
				
				{
					add("center", new JLabel(svgIcon("lizard-logo-128", new Dimension(128, 128), true)));
					add("br center", new JLabel("<html><h2>" + skin + "</h2></html>"));
					add("br center", new JLabel("Jalbum " + internalVersion));
					add(new JLinkLabel("https://jalbum.net/software/download", getText("ui.upgrade"), getText("ui.downloadJalbum")));
					add("br center", new JLabel(skin + " skin " + skinVer));
					add(new JLinkLabel("https://jalbum.net/skins/skin/" + skin, getText("ui.upgrade"), getText("ui.downloadSkin")));
					//add("br center", new JLabel(getText("ui.latestStableMajorVersion")));
					//add(new JLinkLabel("https://jalbum.net/web/GetSkinFile?versionId=4401&type=jaskin", "2.14.5"));
					add("br center", new JLinkLabel(helpRoot, getText("help")));
					add("br center", new JLinkLabel(supportForum, getText("ui.support")));
				}
			};
							
			{
				((BorderLayout)(getLayout())).setVgap(0);
				((BorderLayout)(getLayout())).setHgap(0);
				
				readme.setLineWrap(true);
				readme.setWrapStyleWord(true);
				readme.setEditable(false);
				readme.setFont(mono);
					
				info.setPreferredSize(new Dimension(240, uiHeight));
				aboutPane.setPreferredSize(new Dimension(400, 320));
				
				add(info, BorderLayout.WEST);
				add(aboutPane);
				
			}
		};
		
		/*	---------------------------------------------------------------
									Main tabs
			--------------------------------------------------------------- */
		
		JTabbedPane tabs = new JTabbedPane() {
			
			{		
				design.setBorder(emptyBorder);
				topBar.setBorder(emptyBorder);
				functionality.setBorder(emptyBorder);
				lightbox.setBorder(emptyBorder);
				advanced.setBorder(emptyBorder);
				about.setBorder(emptyBorder);
				
				addTab("<html><h4 " + tabStyle + ">" + getText("ui.design") + "</h4></html>", icon("design", 27), design);
				addTab("<html><h4 " + tabStyle + ">" + getText("ui.topBar") + "</h4></html>", icon("header", 27), topBar);
				addTab("<html><h4 " + tabStyle + ">" + getText("ui.functionality") + "</h4></html>", icon("gear", 27), functionality);
				addTab("<html><h4 " + tabStyle + ">" + getText("ui.lightbox") + "</h4></html>", icon("lightbox", 27), lightbox);
				addTab("<html><h4 " + tabStyle + ">" + getText("ui.advanced") + "</h4></html>", icon("wrench", 27), advanced);
				addTab("<html><h4 " + tabStyle + ">" + getText("ui.about") + "</h4></html>", icon("info", 27), about);
			}
		};
		
		private final String indexTemplate = getFileContents("lib/index-template.html");
		private final String lightboxTemplate = getFileContents("lib/lightbox-template.html");
	
		{	
			System.out.println("license = \"" + license + "\"");
			commercialUseAllowed = license != null && license.length() > 0 && "pro".equals(license);
			
			// Adding UI tabs
			
			((RiverLayout)(getLayout())).setVgap(0);
			((RiverLayout)(getLayout())).setHgap(0);
			tabs.setBorder(emptyBorder);
			
			add(tabs);
			
			/*	-----------------------------------------------------------
										Listeners
				----------------------------------------------------------- */

			window.addJAlbumListener(new JAlbumAdapter() {
				@Override
				public void styleChanged(JAlbumEvent e) {
					styleName.setText(getStyleName());
				}
			});

			window.addPropertyChangeListener(JAlbumFrame.SKIN_LOADED_PROPERTY, setupMonitors);
			
			window.addJAlbumListener(new JAlbumAdapter() {
				@Override
				public void skinChanged(JAlbumEvent e) {
					//JAlbum.logger.log(Level.FINE, "Skin changed, removing listeners.");
					window.removePropertyChangeListener(JAlbumFrame.SKIN_LOADED_PROPERTY, setupMonitors);
				}
			});
				
			Map<String,Object> vars = engine.getSkinVariables();
			//System.out.println("folderCaptionTemplate = \"" + vars.get("folderCaptionTemplate") + "\"");
			selectComboBoxItem(folderCaptionPresets, (String)vars.get("folderCaptionTemplate"));
			//System.out.println("pageCaptionTemplate = \"" + vars.get("pageCaptionTemplate") + "\"");
			selectComboBoxItem(pageCaptionPresets, (String)vars.get("pageCaptionTemplate"));
			//System.out.println("thumbCaptionTemplate = \"" + vars.get("thumbCaptionTemplate") + "\"");
			selectComboBoxItem(thumbCaptionPresets, (String)vars.get("thumbCaptionTemplate"));
			//System.out.println("titleCaptionTemplate = \"" + vars.get("titleCaptionTemplate") + "\"");
			selectComboBoxItem(titleCaptionPresets, (String)vars.get("titleCaptionTemplate"));
			//System.out.println("imageCaptionTemplate = \"" + vars.get("imageCaptionTemplate") + "\"");
			selectComboBoxItem(imageCaptionPresets, (String)vars.get("imageCaptionTemplate"));
			
			putClientProperty("helpPage", helpRoot);
			
		}
		
	};

	public Gui() {
		this(JAlbumContext.getInstance());
	}
   
	public Gui(JAlbumContext context) {
		
		super(context);
		PluginContext pc = context.getPluginContext();
		EditPanel editPanel = pc.getEditPanel();
			
		editPanel.addCustomTab(getText("ui.imageData"), new ImageDataUI(context, texts));
		editPanel.addCustomTab(getText("ui.external"), new ExternalContentUI(context, texts));

		skinUi.setBorder(emptyBorder);
		((RiverLayout)(getLayout())).setVgap(0);
		((RiverLayout)(getLayout())).setHgap(0);
		
		window.setSkinUI(skinUi);		
		skinReadyAt = System.currentTimeMillis();
		
	}
	
}
