/*	
 *	addTooltip() :: little Popup displaying 'title' text, or passed text (can be HTML)
 *
 *	Copyright by Lazaworx
 *	http://www.lazaworx.com
 *	Author: Laszlo Molnar
 *
 *	Dual licensed under the MIT and GPL licenses.
 *	- http://www.opensource.org/licenses/mit-license.php
 *	- http://www.gnu.org/copyleft/gpl.html
 *
 *	Usage: $(element).addTooltip( [txt,] options );
 *	options:
		id: 	'tooltip',
		stay: 	3000,
		posX: 	ALIGN_CENTER,
		posY: 	ALIGN_BOTTOM,
		toX: 	ALIGN_CENTER,
		toY: 	ALIGN_TOP
 */

;(function($, $body) {
	'use strict';
			
	$.fn.hideAllTooltips = function(except) {
		
			return this.each(function() {
					
					//log('Hiding all tooltips');
					$(this).data('suppressTooltips', true).find('[data-tooltip-id]').each(function() {
							if (!$(this).data('tooltip-keep')) { 
								var id = $(this).attr('data-tooltip-id');
								
								if (id && id.length && id !== except) {
									$('#' + id).hide();
								}
							}
						});
				});
		};
	
	$.fn.destroyAllTooltips = function(except) {
		
			return this.each(function() {
					
					//log('Removing ' + $(this).find('[data-tooltip-id]').length + ' tooltip(s).');
					$(this).data('suppressTooltips', true).find('[data-tooltip-id]').each(function() {
							var id = $(this).attr('data-tooltip-id');
							//log(id);
							if (id && id.length && id !== except) {
								$('#' + id).remove();
							}
						});
				});
		};
	
	// Hiding all tooltips on mouse leaving the document
	$(document).on('mouseleave._ltt_', function() {
			$body.hideAllTooltips();
		});
	
	$.fn.addTooltip = function(content, settings) {
		
			if (typeof content !== UNDEF && typeof content !== STRING && !content.jquery) {
				settings = content;
				content = null;
			}
			
			settings = $.extend({}, $.fn.addTooltip.defaults, settings);
			
			//console.log(content || $(this).data('tooltip') || $(this).attr('title'));
			
			var isVisible = function(el) {
						
						if (typeof el === UNDEF || !el || !el.length) {
							return true;
						} else if (el.is(':hidden') || el.css('opacity') < .25) {
							//console.log(el[0].id + ' is hidden or transparent!');
							return false;
						} else {
							var visible = true;
							el.parents().each(function() {
									if ($(this).is(':hidden') || $(this).css('opacity') < 0.25) {
										visible = false;
										return false;
									}
								});
							
							return visible;
						}
					},
					
				// Creating new tooltip
				createNew = function(el, cont) {
						var tt;
										
						if (!cont) {
							
							if (cont = el.data('tooltip')) {
								
								if (cont.jquery) {
									// jQuery element
									cont.removeClass('hidden');
								} else {
									
									if (typeof cont === STRING) {
										// Selector
										if (cont.charAt(0) === '.') {
											// read from related layer
											cont = el.find(cont).eq(0);
										} else if (cont.charAt(0) === '#') {
											// ID
											cont = $(cont);
										}
												
									} else {
										// Read settings from data
										if (typeof cont === OBJECT) {
											settings = $.extend({}, settings, cont);
										}
										
										if (settings.hasOwnProperty('html')) {
											// read from settings.html
											cont = settings['html'].replaceAll('\uff02', '"');
										} else {
											// read from title attr
											cont = el.attr('title');
											el.removeAttr('title');
										}
									}
								}
								
							} else {
								// read from data or title attr
								cont = el.attr('title');
								el.removeAttr('title');
							}
							
							if (!cont || !cont.length) {
								return $();
							}
							
							tt = $('<div>', {
									html: 	cont
								}).appendTo('body');
							
						} else if (typeof cont === STRING) {
							
							// passed directly :: html structure as string
							tt = $('<div>', {
									html: 	cont
								}).appendTo('body');
							
						} else if (cont.jquery) {
							
							// jQuery element
							if (!$.contains(document.body, cont[0])) {
								tt = cont.appendTo('body');
							} else {
								tt = cont;
							}
							
						} else {
							return $();
						}
						
						if (tt.is(':empty')) {
							return null;
						}
						
						tt	.attr('id', el.attr('data-tooltip-id'))
							.addClass(settings.className)
							.attr('role', 'tooltip')
							.append($('<span>', {
								'class': 	settings.nub
							}))
							.data('display', (tt.css('display') === 'none')? 'block' : tt.css('display'))
							.attr('aria-hidden', true)
							.hide();
						
						return tt;
						
					};
			
			return this.each(function() {
				
					if ($(this).data('tooltip-id')) {
						// already has tooltip
						return true;
					}
					
					var self = $(this), 	// trigger element
						options = settings,	// initializing with outer scope settings 
						tt,					// tooltip layer
						to, 				// show timeout
						hto,				// hide timeout
						overtt = false,		// over the tooltip 
						focus = false,		// tooltip input got focus
						offs,				// last offset to detect if moving
						start,				// event start
						wasOn = false,		// was on at the beginning of the event?
						ns = '_ltt_' + Math.floor(Math.random() * 10000),
							
						// Create
						create = function() {
								//log('Creating tooltip');
								if (self.data('suppressTooltips') || !(tt = createNew(self, content))) {
									return false;
								}
								
								// Keep the popup live while the mouse is over
								tt	.on('mouseenter.' + ns, getFocus)
									.on('mouseleave.' + ns, lostFocus);
										
								// ... or an input box has focus
								tt	.find('input, textarea')
									.on('focus.' + ns, function() {
											focus = true;
											getFocus(this);
										})
									.on('blur.' + ns, function() {
											focus = false;
										});
									
								// A button that closes the tooltip on click
								tt	.find('[data-closer]')
									.on('click.' + ns, function() {
											setTimeout(hideLater, 1000);
											return true;
										});
									
								return true;
							},
						
						// getFocus
						getFocus =  function(e) {
								
								//log('Getting focus (' + e.type + ')');
								overtt = true;
								if (isVisible($(this))) {
									clearTimeout(hto);
								}
							},
						
						// lostFocus
						lostFocus = function(e) {
		
								//log('Losing focus (' + e.type + ')');
								if (focus) {
									return;
								}
								
								overtt = false;
								clearTimeout(hto);
								hto = setTimeout(hide, Math.max(options.stay, 500));
							},
							
						// Hiding the popup
						hide = function() {
								
								//log('Hiding tooltip (over = ' + overtt + ')');
								clearTimeout(hto);
								
								if (overtt === false) {
									wasOn = false; 
									clearTimeout(to);
									//events = '';
									if (tt) {
										tt	.css({
													opacity:	0
												})
											.one('transitionend', function() {
													$(this).css({
															display:	'none'
														});
												});
									}
								}
							},
						
						// Hide later :: automatically on touch devices
						hideLater = function() {
								//log('Hiding tooltip (stay = ' + options.stay + 'ms)');
								clearTimeout(hto);
								hto = setTimeout(hide, options.stay);
							},
						
						
						// Showing, aligning and fading in
						show = function() {
								var o = self.offset();
								//log('Showing tooltip');
								if (self.data('suppressTooltips')) {
									return;
								}
								
								clearTimeout(hto);
								//log('pos: ' + o.left.toFixed(3) + ',' + o.top.toFixed(3));
								
								if (!offs || (Math.abs(o.top - offs.top) < 1 && Math.abs(o.left - offs.left) < 1)) {
									// only if target layer not in move
									if (options.exclusive) {
										self.data('tooltip-keep', true);
										$body.hideAllTooltips(self.data('tooltip-id'));
										self.data('tooltip-keep', null);
									}
									
									tt	.css({
												opacity: 	0,
												display:	'block'
											})
										.alignTo(self, {
												gap: 		options.gap,
												pos: 		options.pos
											})
										.css({
												opacity: 	1
											})
										.one('transitionend', function() {
												$(this).css({
														display:	'block'
													});
												hideLater();
											});
										
									wasOn = true;
										
								}
								
								offs = o;
							},
						
						// Entering the hotspot :: hover or focus
						enter = function(e) {
								//log('Entering spot (' + e.type + ')');
		
								if (!self.data('suppressTooltips') && isVisible($(e.target).closest('[data-tooltip-id]'))) {
									wasOn = ttVisible();
									start = new Date();
									offs = self.offset();
									tt = $('#' + self.data('tooltip-id'));
									//log('pos: ' + offs.left.toFixed(3) + ',' + offs.top.toFixed(3));
									
									if (!tt.length) {
										if (!create()) {
											destroy();
											return true;
										}
									} else {
										clearTimeout(hto);
									}
									
									clearTimeout(to);
									to = null;
									
									if (whatInput.ask('intent') === 'mouse') {
										to = setTimeout(show, options.delay);
									}
								}
								
								return true;
							},
						
						// Leaving the trigger element
						leave = function(e) {
								
								clearTimeout(to);
								to = null;
								
								if (isVisible($(e.target).closest('[data-tooltip-id]'))) {
									clearTimeout(hto);
									//log('Leaving spot (' + e.target.nodeName + ')');
									if (whatInput.ask('intent') === 'mouse') {
										hto = setTimeout(hide, 500);
									} else {
										hto = setTimeout(hide, 3000);
									}
								}
								
								return false;
							},
						
						// Avoid Click
						avoidClick = function(e) {
								e.preventDefault();
								clearTimeout(to);
								clearTimeout(hto);
							},
						
						ttVisible = function() {
								return !!tt && tt.is(':visible') && (tt.css('opacity') > 0.99);
							},
						
						// Test for link
						hasLink = function(el) {
								var	a = el.closest('a');
								return a.length && a.attr('href') && !a.attr('href').startsWith('javascript');
							},
						
						// The hotspot clicked
						clicked = function(e) {
								//log('Clicking spot + (target: ' + e.target.nodeName + ')');
								//log(whatInput.ask());
								//log(events);
								
								if (isVisible($(e.target).closest('[data-tooltip-id]'))) {
									clearTimeout(to);
									
									if (options.touchToggle || whatInput.ask('intent') !== 'mouse') {
									//if (events.indexOf(TOUCH.START) !== -1 || isFireFox) {
										// touched
										// Firefox by default emulates touch events with mouse events, 
										// no way you can tell the difference, so it's safer to treat like touch
										
										var now = new Date();
										
										if (options.touchToggle || (now - start) > 1000) {
											// touch toggle or long touch
											//log('Touched for ' + (now-start) + 'ms');
											//log('wasOn='+wasOn+' hasLink()='+hasLink($(e.target))+' visible='+ttVisible());
											if (hasLink($(e.target)) && ttVisible()) {
												// Link and tt is visible :: let click it
												return true;
											} else {
												// No link or need to toggle on first
												avoidClick(e);
												
												if (wasOn) {
													hide();
												} else {
													show();
												}
												
												return false;
											}
										}		
										
									} else if (hasLink($(e.target))) {
										return true;
									}
									
									if (wasOn) {
										clearTimeout(hto);
										hide();
									}
									
									//events = '';
									overtt = false;
								}
								
								return true;
							},
						
						// Force removing the tooltip
						destroy = function(e) {
								self.off('.' + ns);
								clearTimeout(to);
								clearTimeout(hto);
								self.data('suppressTooltips', true);
								$('#' + self.attr('data-tooltip-id')).remove();
								self.attr('data-tooltip-id', null);
							};
							
					if (self.attr('data-tooltip') && typeof self.data('tooltip') === OBJECT) {
						options = $.extend({}, options, self.data('tooltip'));
						self.removeAttr('data-tooltip');
					}
					
					self.attr('data-tooltip-id', ns)
						.data('suppressTooltips', false)
						.on('destroyTooltip', destroy)
						.on('removeTooltip', hide)
						.on('focus.' + ns + ' mouseenter.' + ns /*+ ' ' + TOUCH.START + '.' + ns*/, enter)
						.on('blur.' + ns + ' mouseleave.' + ns, leave)
						.on('click.' + ns, clicked);
					
				});
		};
	
	/*	pos:
		ALIGN_LEFT = ALIGN_TOP = 0
		ALIGN_CENTER = ALIGN_MIDDLE = 1
		ALIGN_RIGHT = ALIGN_BOTTOM = 2
	*/
	$.fn.addTooltip.defaults = {
			delay: 				100,
			className: 			'tooltip',
			nub: 				'nub',
			stay: 				2000,
			exclusive:			true,
			touchToggle:		false,
			pos: 				[1,2,1,0],
			gap:				6
		};
	
})(jQuery, $('body'));
