/*	
 *	laza.modal() :: adding modal window to any layer (typically 'body')
 *
 *	Copyright by Lazaworx
 *	http://www.lazaworx.com
 *	Author: Laszlo Molnar
 *
 *	Dual licensed under the MIT and GPL licenses.
 *	- http://www.opensource.org/licenses/mit-license.php
 *	- http://www.gnu.org/copyleft/gpl.html
 *
 *	Usage: $(element).modal( content, buttons, options );
 */
 
;(function($, $window, $body) {
	'use strict';
		
	$.fn.modal = function(content, buttons, settings) {
			
			if (typeof content === 'string') {
				// Content is a String?
				content = $(document.createTextNode(content));
			}
			
			if (!(content instanceof $ && content.length)) {
				// Is jQuery element?
				return;
			}
			
			if (typeof buttons !== UNDEF && !Array.isArray(buttons)) {
				// No buttons? Swap with settings
				settings = buttons; 
				buttons = null;
			}
			
			if (content.data('modal')) {
				// Settings passed as data attribute of the container
				settings = $.extend({}, settings, content.data('modal'));
			}
			
			// Merge settings with defaults
			settings = $.extend({}, $.fn.modal.defaults, settings);
			
			var container = $(this),
				modals = container.find('.' + settings.modalClass),
				lastZ = modals.length? parseInt(modals.last().css('zIndex')) : 300000,
				text = getTranslations($.fn.modal.text),
				
				// Aligning the window to an existing element 
				alignPopup = function(modal, to) {
					
						if (!to.jquery) {
							to = $(to);
						}
						
						var toPos = to.position(),
							toDim = [ to.outerWidth(), to.outerHeight() ],
							wDim = [ modal.outerWidth(), modal.outerHeight() ],
							midPos = [ (toPos.left + toDim[0] / 2) / wDim[0], (toPos.top + toDim[1] / 2) / wDim[1] ],
							css = {},
							ccss = {};

						// Horizontal
						if (midPos[0] > 0.333 && midPos[0] < 0.667) {
							// Center
							//left = '50%';
							css.left = ccss.left = (toPos.left + toDim[0] / 2) + 'px';
							css.transform = ccss.transform = 'translateX(-50%)';
						} else if (midPos[0] <= 0.333) {
							// Left
							css.left = ccss.left = toPos.left + 'px';
							css.right = ccss.right = 'auto';
							css.transform = ccss.transform = 'none';
						} else {
							// Right
							css.right = ccss.right = (wDim[0] - (toPos.left + toDim[0])) + 'px';
							css.left = ccss.left = 'auto';
							css.transform = ccss.transform = 'none';
						}
						
						// Vertical
						if (midPos[1] <= 0.5) {
							// Top
							css.top = (toPos.top + toDim[1] + 4) + 'px';
							ccss.top = (toPos.top + toDim[1] - settings.closeSize) + 'px'; 
						} else {
							// Bottom
							css.bottom = (wDim[1] - toPos.top - 4) + 'px';
							ccss.bottom = (wDim[1] - toPos.top - settings.closeSize) + 'px';
							ccss.top = 'auto';
						}
							
						// Applying the position
						modal.children('.' + settings.windowClass).css(css);
						modal.children('.' + settings.closeClass).css(ccss);
					},
					
				// Does the DOM already contain the element?
				DOMContains = function(content) {
						return $.contains(document.documentElement, content[0]);
					},
					
				// Setting up an existing modal
				setupExistingModal = function(content) {
						var modal = content.parents('.' + settings.modalClass);
						
						if (!modal.length || modal.parent() !== container) {
							// Moving under container
							modal = content.appendTo(container);
						}
						
						modal.data('keep', true);
						
						return modal;
					},
				
				createNewModal = function(content, buttons) {
						// Cceating new modal
						var modal,
							popup,
							closeBtn,
							cont;
						
						modal = $('<div>', {
								'class': 		settings.modalClass,
								zIndex:			lastZ + 1
							}).appendTo(container);
							
						modal.data('modal-ns', 'lmo_' + Math.floor(Math.random() * 10000));
							
						popup = $('<div>', {
								'class': 		settings.windowClass + (settings.hasOwnProperty('class')? (' ' + settings['class']) : ''),
								role: 			'dialog'
							}).appendTo(modal);
							
						closeBtn = $('<a>', {
								'class': 		settings.closeClass,
								'title': 		text.closeWindow
							}).appendTo((settings.alignTo)? modal : popup);
							
						closeBtn.addTooltip();
						
						cont = $('<div>', {
								'class': 		settings.contentClass + ' scrollable'
							}).appendTo(popup);
							
						if (settings.title) {
							popup.addClass(settings.hasHeaderClass).prepend($('<header>', {
									'class':		settings.headerClass + (settings.icon? (' icon-' + settings.icon) : ''),
									html: 			'<h3>' + settings.title + '</h3>'
								}));
						}
						
						cont.append(content.show());
							
						if (buttons) {
							var i, 
								a, 
								
								btnCont = $('<div>', { 
										'class': 		settings.buttonsClass 
									}).appendTo(cont),
								
								clickhandler = function(e) {
									e.preventDefault();
									var a = e.target;
									
									if (typeof a.handler === FUNCTION) {
										if (a.handler.call(this) !== false) {
											close($(a).closest('.' + settings.modalClass));
										}
									}
									return false;
								};
							
							for (i = 0; i < buttons.length; i++) {
								a = $('<button>', {
										'class': 		[ settings.buttonClass, buttons[i].c ].filter(Boolean).join(' '),
										html: 			' ' + buttons[i].t
									}).on('click', clickhandler).appendTo(btnCont);
								
								if (typeof buttons[i].h === FUNCTION) {
									a[0].handler = buttons[i].h;
								}
							}
						}
						//setMaxHeight();
						
						return modal;
					},
					
				init = function(content, buttons) {
						var modal = DOMContains(content)? setupExistingModal(content) : 
									((typeof buttons === UNDEF)? createNewModal(content) : createNewModal(content, buttons));
						
						if (settings.hasOwnProperty('closeOnClick')) {
							$(settings.closeOnClick).on('click', function() {
									close($(this).closest('.' + settings.modalClass));
								});
						}
						
						var at = settings.hasOwnProperty('alignTo')? ((settings.alignTo instanceof jQuery)? settings.alignTo : $(settings.alignTo)) : $();
						
						if (at.length && at.is(':visible')) {
							alignPopup(modal, at);
						}
						
						// Prepare
						modal.hide().css('opacity', 1).fadeIn(settings.speed);
						
						$body.addClass(settings.hasModalClass);
						
						// Externally triggered close
						modal.on('destroy', function() {
								close($(this));
							});
						
						modal.on('close', function() {
								close($(this));
							});
						
						// Close button click handler
						modal.find('.' + settings.closeClass).add(modal).on('click.' + modal.data('modal-ns'), function(e) {
								if (e.target === this) {
									close($(this).closest('.' + settings.modalClass));
									if (typeof settings['onClose'] === FUNCTION) {
										settings.onClose.call(this);
									}
									return false;
								}
							});
						
						if (settings.autoFade) {
							modal.data('to', setTimeout(function() {
									close(modal);
								}, settings.autoFade));
						}

						return modal;
					},
				/*			
				setMaxHeight = function() {
						cont.css({
							overflow:	'auto',
							maxHeight:	Math.round($window.height() * 0.9) - (popup.hasClass('has-header')? closeBtn.height() : 0)
						});
					},
				*/
				close = function(modal) {
					
						if (!modal.length) {
							return;
						}
						
						clearTimeout(modal.data('to'));
						
						modal.destroyAllTooltips();
						//$window.off('.' + modal.data('modal-ns'));
						
						modal.fadeOut(settings.speed, function() {
								if (modal.data('keep')) {
									modal.hide().css('opacity', 1);
								} else {
									modal.remove();
								}
							});
						
						// Still has other modals
						$body.toggleClass(settings.hasModalClass, container.find('.' + settings.modalClass).length);
					};
			
			// Starts here
			return init(content, buttons);
		};
	
	$.fn.modal.defaults = {
			modalClass:			'modal',
			windowClass:		'window',
			headerClass:		'header',
			closeClass:			'close',
			contentClass:		'content',
			buttonsClass:		'buttons',
			buttonClass:		'button',
			hasHeaderClass:		'has-header',
			hasModalClass:		'has-modal',
			speed: 				300,
			autoFade: 			0,
			width: 				400,
			type: 				'normal',
			closeSize:			44
		};
	
	$.fn.modal.text = {
			closeWindow:					"Close window"
		};

})(jQuery, jQuery(window), jQuery('body'));
