/*
 * 	lightbox.js - lightbox for Photoblogger skin
 *	jslint browser: true
 */
 
;(function($, $window, $document, $body, undefined) {	
	'use strict';
	// Requires laza.util, laza.swipe and laza.transform
	
	$.fn.lightbox = function(album, settings) {
		
			if (typeof album === UNDEF) {
				return this;
			}
			
			settings = $.extend({}, $.fn.lightbox.defaults, settings);
	
			var self = $(this),
				text = getTranslations($.fn.lightbox.text),
				thumbs = $([]),
				thumb = $(),
				main = $(),
				oldMain = $(),
				image = $(),
				media = $(),
				audioClip = $(),
				el,
				isImage = true,
				isVideo = false,
				isAudio = false,
				isOther = false,
				isExternal = false,
				isVr = false,
				isPdf = false,
				isPanorama = false,
				dontRemove = false,
				slideshow = null,
				slideshowSuspended = false,
				slideshowDelay = settings.slideshowDelay,
				transitionSpeed = settings.transitionSpeed,
				audioClipTimeout = null,
				fitTimeout = null,
				topbarTimeout = null,
				resizeTimeout = null,
				autoPanoTimeout = null,
				listening = false,
				
				maxZoom = settings.maxZoom,
				fitImages = settings.fitImages,
				zoomLevel,
				
				thumbsVisibleDefault = settings.thumbsVisible,
				captionVisibleDefault = settings.captionVisible,
				
				curr = -1,
				loadCounter = 0,
				loadTimer,
				connectionSpeed = 0,
				direction = 0,
				imgPadding,
				lbPadding = settings['fitPadding'], 
				
				// Elements
				lightbox,
				lb_topbar,
				lb_cont,
				lb_overlay,
				lb_activity,
				lb_menu,
				
				// Thumbstrip
				lb_thumbstrip,
				lb_thumb_cont,
				lb_thumbs,
				
				// Controls
				lb_close,
				lb_title,
				lb_zoom,
				lb_zoomin,
				lb_zoomout,
				lb_zoom_slider,
				lb_fullscr,
				lb_exitfullscr,
				lb_play_pause,
				lb_timing_value,
				lb_btn_left,
				lb_btn_right,
				lb_btn_menu,
				
				// Caption
				lb_caption = $(),
				
				ns = self.data('llb_ns'),
				extraSizes = settings.hasOwnProperty('extraSizes')? settings.extraSizes.split(/,\s*/) : false,
				
				backgroundAudioRoot = $('[data-audioplayer]'),
				backgroundAudioOn = backgroundAudioRoot.length && !backgroundAudioRoot.data('paused'),
	
				cleanup = function(ns) {	
						// cleaning up the old event handlers
						$window.add(document).off('.' + ns);
						// Removing the lightbox :: you can only have one lightbox per thumb set
						$('#' + ns).remove();
					},
				
				closeModal = function(id) {
					
						lightbox.find('.modal').trigger('destroy');
					},
				
				getThumbsHeight = function() {
					
						if (settings.thumbsVisible && thumbs.length > 1
							&& lb_thumb_cont && lb_thumb_cont.length) {
							return lb_thumb_cont.outerHeight();
						}
						
						return 0;
					},
				
				getTopbarHeight = function() {
						
						return topbarHidden()? 0 : Math.ceil(lb_topbar.outerHeight());
					},
	
				getPanelsHeight = function() {
						
						return getThumbsHeight() + getTopbarHeight();
					},
	
				// Fixing the zoom button after pinch
				fixZoom = function(zoom, coords) {
					
						if (settings.useZoom && settings.zoomSlider) {
							if (zoom === 1) {
								return;
							}
							
							var mnr = main.data('minRatio') || 1,
								mxr = main.data('maxRatio') || 1.4,
								z = mnr + zoomLevel * (mxr - mnr) / 100; 
							
							z = Math.minMax(mnr, z * zoom, mxr);
							zoomLevel = (z - mnr) / (mxr - mnr) * 100;
						
							setZoomSlider(zoom);
							setZoomBtn(zoomLevel < 50, zoomLevel >= 50);
						}
					},
				
				// Toggling zoom in simple zomm toggling
				toggleZoom = function(coords) {
					
						if (settings.useZoom && settings.zoomSlider) {
							fitImages = true;
							zoomLevel = (zoomLevel > 50)? 0 : 100;
							setZoomSlider(zoomLevel);
						} else {
							// Change fit strategy
							fitImages = !fitImages;
						}
						//savePrefs();
						if (typeof coords === UNDEF) {
							fitImage(true);
						} else {
							fitImage(true, coords);
						}
					},
				
				// Sets zoom (0 .. 100), where 100 means 1:1 size
				setZoom = function(zoom, coords) {
						//console.log('Set zoom:' + zoom);
						zoomLevel = zoom;
						setZoomSlider(zoom);
						
						if (typeof coords === UNDEF) {
							fitImage(true);
						} else {
							fitImage(true, coords);
						}
					},
					
				increaseZoom = function(incr, coords) {
						var v = settings.zoomSlider? getZoomSlider() : zoomLevel;
						
						if (v < 100) {
							v = Math.min(100, v + ((typeof incr !== UNDEF)? incr : 25));
						}
						
						if (typeof coords === UNDEF) {
							setZoom(v);
						} else {
							setZoom(v, coords);
						}
					},
						
				decreaseZoom = function(incr, coords) {
						var v = settings.zoomSlider? getZoomSlider() : zoomLevel;
						
						if (v > 0) {
							v = Math.max(0, v - ((typeof incr !== UNDEF)? incr : 25));
						}
						
						if (typeof coords === UNDEF) {
							setZoom(v);
						} else {
							setZoom(v, coords);
						}
					},
						
				getZoomSlider = function() {
						return parseInt(lb_zoom_slider.val());
					},
						
				setZoomSlider = function(zoom) {
						lb_zoom_slider.val(zoom);
					},
	
				setZoomBtn = function(canZoomIn, canZoomOut) {
					
						if (settings.zoomSlider) {
							if (canZoomIn || canZoomOut) {
								lightbox.addClass('show-zoom-level');
								if (zoomLevel === 0) {
									lb_zoom.removeClass('icon-zoom-out icon-zoom-level').addClass('icon-zoom-in');
								} else if (zoomLevel === 100) {
									lb_zoom.removeClass('icon-zoom-in icon-zoom-level').addClass('icon-zoom-out');
								} else {
									lb_zoom.removeClass('icon-zoom-in icon-zoom-out').addClass('icon-zoom-level');
								}
							} else {
								lightbox.removeClass('show-zoom-level');
							}
						} else {
							lightbox.toggleClass('show-zoom-out', canZoomOut);
							// Zoom out takes precedence if both possible
							lightbox.toggleClass('show-zoom-in', canZoomIn && !canZoomOut);
						}
						
					},
					
				// Slideshow timing controls
				
				setTiming = function(t) {
						lb_timing_value.html((t >= 1000)? (Math.round(t / 1000) + 's') : (t + 'ms')); 
					},
					
				increaseTiming = function() {
					
						if (settings.slideshowDelay > 1000) {
							settings.slideshowDelay += 1000;
						} else {
							settings.slideshowDelay += 100;
						}
						
						setTiming(settings.slideshowDelay);
					},
						
				decreaseTiming = function() {
					
						if (settings.slideshowDelay > 500) {
							if (settings.slideshowDelay > 1000) {
								settings.slideshowDelay -= 1000;
							} else {
								settings.slideshowDelay -= 100;
							}
						}
						
						setTiming(settings.slideshowDelay);
					},
					
				getLightboxDims = function() {
						var lbDims = [ lightbox.width(), lightbox.height() ];
						
						return [ lbDims[0] - 2 * lbPadding, lbDims[1] - Math.max(getThumbsHeight(), lbPadding) - Math.max(getTopbarHeight(), lbPadding) ];
					},
					
				// Reacalculating image position and size upon window resize
				
				fitImage = function(animate, center, doneFn) {
					
						if (!main || !main.length) {
							return;
						}
						
						var o 				= thumbs.eq(curr).data(J.OBJ),
							lbDims 			= [ lb_cont.width(), lb_cont.height() ],
							imgDims 		= album.getDimensions(o),
							maxDims 		= album.getMaxDimensions(o),
							topbarHeight 	= getTopbarHeight(),
							thumbsHeight 	= getThumbsHeight(),
							
							checkPano = function(dims) {
									return settings.autoPano && (dims[0] / dims[1] > 2 || dims[0] / dims[1] < .5625);
								},
								
							checkPanoDims = function(dims) {
									return 	dims[0] / lbDims[0] >= settings.panoramaAutoDetect	 			// hangs over the edges least 1.3333 times the width or height
											||
											dims[1] / lbDims[1] >= settings.panoramaAutoDetect;
								},
								
							getTargetDims = function(dims, f) {
									return [ dims[0] * f, dims[1] * f ];
								},
								
							fitPano = function(dims) {
									return Math.max(
											(lbDims[0] - 2 * lbPadding - 2 * imgPadding) / dims[0],
											(lbDims[1] - 2 * lbPadding - topbarHeight - thumbsHeight - 2 * imgPadding) / dims[1]
										);
								},
								
							fitToLightbox = function(dims) {
									return Math.min(	
											(lbDims[0] - 2 * lbPadding - 2 * imgPadding) / dims[0], 
											(lbDims[1] - 2 * lbPadding - topbarHeight - thumbsHeight - 2 * imgPadding) / dims[1]
										);
								};
							
						// Image padding = border
						if (typeof imgPadding === UNDEF) {
							imgPadding = parseInt(main.css('paddingTop'));
						}
											
						// Lightbox padding
						if (typeof lbPadding === UNDEF) {
							if (settings.hasOwnProperty('fitRatio')) {
								// Image margin defined by "fitRatio" - Photoblogger
								lbPadding = Math.round((1 - settings.fitRatio) * Math.min(lbDims[0], lbDims[1]) / 2);
							} else {
								lbPadding = 0;
							}
						}
						
						// Set zoom level if undef
						if (typeof zoomLevel === UNDEF || zoomLevel === null) {
							zoomLevel = fitImages? 0 : 100;
							if (settings.useZoom && settings.zoomSlider) {
								setZoomSlider(zoomLevel);
							}
						}
						
						
						if (isVr || isOther || isExternal || isAudio && image.hasClass('default')) {
							// Non-zoomable types
							var th 	= Math.max(topbarHeight, lbPadding),
								bh 	= Math.max(thumbsHeight, lbPadding),
								w	= lbDims[0] - 2 * lbPadding,
								h 	= lbDims[1] - (th + bh);
								
							if (isExternal || isAudio) {
								// Might be smaller than viewport
								if (imgDims[0] < w && imgDims[1] < h) {
									// Smaller than available space?
									w = Math.min(imgDims[0], w) + 2 * imgPadding;
									h = Math.min(imgDims[1], h) + 2 * imgPadding;
								}
							}
								
							main.data('fitted', true)
								.css({
									display:		'block',
									width:			w,
									height:			h,
									transition:		'none',
									transform:		'matrix(1,0,0,1,' + Math.round((0 - w) / 2) + ',' + Math.round((th - bh - h) / 2) + ')',
									opacity:		1
								});
							
							if (settings.useZoom) {
								setZoomBtn(false, false);
							}
							
							// Done Function
							if (typeof doneFn === FUNCTION) {
								doneFn.call(null);
							}
							
						} else {
							// Zoomable types
							var minRatio,
								maxRatio,
								targetRatio,
								targetDims,
								matrix = main.getMatrix(),
								optSrc,
								isPano,
								fitted = false;
								
							// Stop pano move
							if (lightbox.data('panomove')) {
								main.trigger('autopanoStop');
							}
							
							// Minimal size	
							minRatio = Math.min(1, fitToLightbox(imgDims));
							
							// Maximum size
							if (isVideo || isAudio) {
								// Audio or Video: constrain to lightbox
								maxRatio =	Math.min(maxZoom, fitToLightbox(imgDims));
							} else if (maxDims) {
								// Has larger variants
								maxRatio = Math.max(maxZoom, 
											Math.min(maxDims[0] / imgDims[0], maxDims[1] / imgDims[1]));
							} else {
								// Normal image
								maxRatio = Math.min(maxZoom, fitToLightbox(imgDims));
							}
							
							if (minRatio > maxRatio) {
								// Swap if reversed
								var r = minRatio;
								minRatio = maxRatio;
								maxRatio = r;
							}
							
							// Target size (fit image taken into account)
							if (settings.useZoom && settings.zoomSlider) {
								// Determining initial size for zoom slider
								if (main.data('inited')) {
									// Already painted
									targetRatio = minRatio + zoomLevel * (maxRatio - minRatio) / 100;
									targetDims = getTargetDims(imgDims, targetRatio);
									fitted = targetRatio == minRatio;
									isPano = !!lightbox.data('hasAutoPano');
								} else {
									// First paint
									if (isPanorama || checkPano(imgDims)) {
										// Panorama: fit to height
										targetRatio = Math.min(maxRatio, fitPano(imgDims));
										targetDims = getTargetDims(imgDims, targetRatio);
										isPano = checkPanoDims(targetDims);
										fitted = true;
									} else if (settings.fitImages) {
										// Fit to screen
										targetRatio = Math.min(maxRatio, fitToLightbox(imgDims));
										targetDims = getTargetDims(imgDims, targetRatio);
										isPano = false;
										fitted = true;
									} else {
										// Show in 1:1 size
										targetRatio = 1;
										targetDims = imgDims;
										isPano = false;
										fitted = targetRatio == minRatio;
									}
									zoomLevel = 100 * Math.minMax(0, (targetRatio - minRatio) / (maxRatio - minRatio), 1);
									setZoomSlider(zoomLevel);
								}
							} else {
								// Fit toggle
								if (isPanorama || checkPano(imgDims)) {
									targetRatio = Math.min(maxZoom, fitImages? fitPano(imgDims) : fitToLightbox(imgDims));
									targetDims = getTargetDims(imgDims, targetRatio);
									isPano = checkPanoDims(targetDims);
								} else {
									targetRatio = fitImages? Math.min(Math.max(maxZoom, maxRatio), fitToLightbox(imgDims)) : 1;
									targetDims = getTargetDims(imgDims, targetRatio);
								}
								fitted = fitImages;
							}
							
							// Checking if high res image needed
							if (	isImage && 																		// It's an image
									((maxDims[0] > imgDims[0]) || (maxDims[1] > imgDims[1])) && 					// Has higher resolution version
									((targetDims[0] > imgDims[0]) || (targetDims[1] > imgDims[1])) && 				// The target size is larger than the base size
									(!main.data('lastDims') || main.data('lastDims')[0] < targetDims[0])			// The higher resolution version hasn't yet been loaded
								) {
								// Saving optimized image source 
								main.data('optSource', album.getOptimalImagePath(o, targetDims, settings.displayOriginals));
								main.data('lastDims', targetDims);
							}
								
							// Adding padding (border)
							targetDims = [  targetDims[0] + 2 * imgPadding, targetDims[1] + 2 * imgPadding ];
																					
							// Reset scale
							matrix[0] = matrix[3] = 1;
							// Setting position
							if (!main.data('inited') || (!isPano && fitted)) {
								// First paint or fit
								main.data('inited', true);
								matrix[4] = 0 - targetDims[0] / 2;
								matrix[5] = 0 - targetDims[1] / 2  + 
										(fitted?
											(Math.max(topbarHeight, lbPadding) - Math.max(thumbsHeight, lbPadding)) / 2
											:
											0
										)
							} else {
								// Already painted
								var z = (targetDims[0] - 2 * imgPadding) / main.width(),
									h = main.height();
									
								if (typeof center !== UNDEF && center) {
									// Center around event coords
									var	cx = center.x - lbDims[0] / 2,
										cy = center.y - lbDims[1] / 2; 
									matrix[4] = cx - (cx - matrix[4]) * z;
									matrix[5] = cy - (cy - matrix[5]) * z;
								} else {
									// Use previous center
									matrix[4] *= z;
									matrix[5] *= z;
								}
							}
							
							// Prepare panoramas
							if (isImage) {
								if (isPano) {
									// Pamorama
									if (!main.data('hasAutoPano') && fitted && (targetDims[0] > (lbDims[0] + 10) || targetDims[1] > (lbDims[1] + 10))) {
										// Applying panorama player
										if (targetDims[0] / targetDims[1] > lbDims[0] / lbDims[1]) {
											// Horizontal
											matrix[4] -= settings.autoPanoStart * (targetDims[0] - lbDims[0]) / 2;
										} else {
											// Vertical
											matrix[5] -= settings.autoPanoStart * (targetDims[1] - lbDims[1]) / 2;
										}
										
									}
								}
							}
							
							// Functions to execute on transition end or immediately if no animation
							var fto = null,
								finish = function() {
										// Replacing image source if not optimal
										if (main.data('optSource')) {
											var img = main.children('img').eq(0);
											if (img.length) {
												img.attr('src', main.data('optSource'));
												console.log('src replaced: ' + img.attr('src'));
												main.removeData('optSource');
											}
										}
										// Adjust or attach pano player
										if (isImage && isPano && (typeof center === UNDEF || !center)) {
											
											if (main.data('hasAutoPano')) {
												main.trigger('autopanoAdjust');
											} else {
												//console.log('Applying autopano');
												main.autopano({ 
													direction: 	-settings.autoPanoStart || direction,
													speed:		settings.autoPanoSpeed
												});
											}
										}
									
										if (isVideo && settings.videoAuto && media[0].paused) { 
											// Play video only after the transition
											playVideo(media[0]);
										}
									},
								
								endFitImage = function() {
										clearTimeout(fto);
										// Done Function
										if (typeof doneFn === FUNCTION) {
											doneFn(finish);
										} else {
											finish();
										}
									};
						
							main.css({
									display:		'block',
									transition:		animate? ('transform ' + settings.transitionProps + ', width ' + settings.transitionProps + ', height ' + settings.transitionProps) : 'none'
								});

							// Apply transforms
							
							if (animate) {
								//console.log('Scheduling finish()');
								main.one('transitionend', endFitImage);
								fto = setTimeout(endFitImage, 320);
							}
							
							main.data({
									fitted: 		fitted,
									minRatio:		minRatio,
									maxRatio:		maxRatio
								})
								.css({
									transform:		'matrix(' + matrix.join(',') + ')',		
									width: 			targetDims[0],
									height: 		targetDims[1]
								});
								
							// Set zoom button
							if (settings.useZoom) {
								setZoomBtn(targetRatio < maxRatio, targetRatio > minRatio);
							}
							
							if (!animate) {
								//console.log('Calling finish()');
								fto = setTimeout(endFitImage, 20);
							}
						}
						
					},
				
				// Reading image type
				
				getImageType = function(item) {
						
						isVideo = isAudio = isOther = isImage = isVr = isPdf = isPanorama = false;
						
						isExternal = item.hasOwnProperty(J.EXTERNAL);
						
						isVr = !isExternal && settings.use360Player && (
								(item.hasOwnProperty(J.PROJECTIONTYPE) && item[J.PROJECTIONTYPE] === 'equirectangular') 
								||
								(item[J.CATEGORY] === 'video' && item.hasOwnProperty(J.PANORAMA) && item[J.PANORAMA])
							);
						
						if (!isExternal && !isVr) {
							
							switch (item[J.CATEGORY]) {
								
								case 'video':
									isVideo = true;
									break;
								
								case 'audio':
									isAudio = true;
									break;
								
								case 'other':
									isOther = true;
									isPdf = album.getExtension(item).toLowerCase() === 'pdf';
									break;
								
								default:
									isImage = true;
									isPanorama =
										(item.hasOwnProperty(J.PANORAMA) && item[J.PANORAMA]) 
										|| 
										(	settings.autoPano 
											&& 
											(	item[J.IMAGE][J.WIDTH] > lightbox.width()
												||
												item[J.IMAGE][J.HEIGHT] > lightbox.height()
											)
											&&
											(	(item[J.IMAGE][J.WIDTH] / item[J.IMAGE][J.HEIGHT] > settings.horzPanoThreshold) 
												|| 
												(item[J.IMAGE][J.WIDTH] / item[J.IMAGE][J.HEIGHT] < settings.vertPanoThreshold)
											)
										);
							}
						}
					},
				
				// Gets the custom slideshow interval (if exist)
				
				getSlideshowTiming = function(item) {
					
						if (item.hasOwnProperty(J.SLIDESHOWINTERVAL)) {
							slideshowDelay = item[J.SLIDESHOWINTERVAL];
						} else {
							slideshowDelay = settings.slideshowDelay;
						}
						if (item.hasOwnProperty(J.TRANSITIONSPEED)) {
							transitionSpeed = item[J.TRANSITIONSPEED];
						} else {
							transitionSpeed = settings.transitionSpeed;
						}
					},
					
				// Updates rating (if changed externally)
					
				updateRating = function(r) {
						
						if (curr >= 0) {
							if (typeof r === UNDEF) {
								r = thumbs.eq(curr).data(J.OBJ)[J.VISITORRATING] || 0;
							}
						
							lb_cont.find('.' + settings.lightboxCaptionClass + ' .ratingbox span')
								.removeClass('r0 r1 r2 r3 r4 r5')
								.addClass('r' + r);
						}
					},
					
				// Paint regions
				
				paintRegions = function(regions, visible) {
						var el = $('<div>', {
									'class':	settings.regionsClass
								}).appendTo(main);
						
						for (var i = 0, r; i < regions.length; i++) {
							r = regions[i].split(';');
							el.append($('<div>').css({
									left:		(100 * parseFloat(r[1])) + '%',
									top:		(100 * parseFloat(r[2])) + '%',
									width:		(100 * parseFloat(r[3])) + '%',
									height:		(100 * parseFloat(r[4])) + '%'
								}).append('<span>' + r[0] + '</span>'));
						}
						
						el.toggle(typeof visible !== UNDEF && visible);
					},
					
				// Show/hide regions
				
				toggleRegions = function() {
						var el = lightbox.find('.' + settings.regionsClass);
					
						if (el.length) {
							settings.regionsVisible = !settings.regionsVisible;
							if (el.is(':visible')) {
								el.fadeOut();
							} else {
								el.fadeIn();
							}
						}
						
						return false;
					},
											
				// Creating caption and buttons
				
				prepareCaption = function(item) {
						var s = item[J.IMAGECAPTION];
						
						if (s) {
							lightbox.removeClass('no-caption');
							lb_caption = $('<div>', {
									'class': 	settings.lightboxCaptionClass + ' ' + settings.captionPlacement,
									html:		'<div class="scrollable">' + s + '</div>'
								}).appendTo(main);
							lb_caption.append($('<a>', {
									'class':	'caption-toggle icon-more button'
								}).on('click', function(e) {
									lightbox.toggleClass(settings.captionVisibleClass);
								}));
									
						} else {
							lightbox.addClass('no-caption');
							lb_caption = $();
						}
						
						s = settings.titleCaptionTemplate? album.processTemplate(settings.titleCaptionTemplate, item, true) : '';
						
						if (s && s.length) {
							lb_title.empty().append(s);
						} else {
							lb_title.empty();
						}
						
						if (settings.showNumbers) {
							lb_title.prepend($('<span>', {
									'class':	'numbers',
									'html':		(curr + 1) + '<em>' + thumbs.length + '</em>'
								}));
						}
						
						var mi = menuItems(item);
						
						removeExtraButtons();
						
						if (mi > 1 || settings.useRating) {
							// Multiple buttons or rating
							lb_btn_menu.show();
						} else {
							lb_btn_menu.hide();
							if (mi === 1) {
								// Single button, showing directly
								createButtons(false);
							}
						}
					},
					
				// Showing the menu on click
				
				showMenu = function() {
						var menu = createButtons(true);
						
						if (!menu.is(':empty')) {
							menu.children('a').not('[download]').addTooltip({ pos: [2,1,0,1] });
							lb_cont.modal(menu, {
									'class':		'context-menu',
									alignTo:		lb_btn_menu,
									closeOnClick:	'.menu .button'
								});
						} else {
							lb_btn_menu.hide();
						}
					},
					
				// Checking if any button has to be added to the menu
				
				hasMenu = function(item) {
						return settings.useRating || menuItems(item) > 0;
					},
				
				// Number of buttons to be added to the menu
				
				menuItems = function(item) {
						
						return 	[	(!isExternal && extraSizes && item[J.CATEGORY] === 'image'),
									((DEBUG || !LOCAL) && !!settings.share && settings.showShare),
									(settings.printImage && item[J.CATEGORY] === 'image' && !settings.rightClickProtect),
									(!isExternal && !!settings.showDownload && (!!item[J.ORIGINAL] && (isImage || isPdf || settings.allowDownloadOthers) || isImage && settings.allowDownloadScaled)),
									(settings.hasOwnProperty('shop') && album.hasShop(item) && settings.showShop),
									(settings.hasOwnProperty('feedback') && settings.showFeedback),
									((!LOCAL || DEBUG) && typeof FOTOMOTO !== UNDEF && settings.fotomoto && !album.hideFotomoto(item)),
									(!isExternal && !!item[J.PHOTODATA]),
									(!isExternal && settings.showRegions && !!item[J.REGIONS]),
									(settings.showMap && !!item[J.LOCATION]),
									(!LOCAL && !isExternal && !!item[J.MOSTPHOTOS])
								].filter(Boolean).length;
					},
				
				// Removing extra buttons remained from the previous item
				
				removeExtraButtons = function() {
						
						lb_topbar.find('.' + settings.extraBtnClass).remove();
					},
					
				// Creating the menu structure for the current item
				
				createButtons = function(subMenu) {
						var item = thumbs.eq(curr).data(J.OBJ),
							target = subMenu? $('<div>', {
										'class': 	'menu'
									}) : lb_btn_menu;
						
						if (settings.useRating) {
							
							var r = item[J.RATING] || 0;
							
							if (settings.visitorRating || r) {
								var s;
							
								el = $('<div>', {
										'class':	'ratingbox'
									}).appendTo(target);
								/*	
								el.append($('<label>', {
										html:		text.rating
									}));
								*/
								s = $('<span>', {
										'class':	'rating',
										html:		STARS
									})
									.appendTo(el);
									
								if (settings.visitorRating) {
									
									if (item.hasOwnProperty(J.VISITORRATING)) {
										// Already exists
										r = item[J.VISITORRATING];
									} else {
										if (!settings.useJalbumRating) {
											// Ignore jAlbum rating
											r = 0;
										}
										// Save the default
										item[J.VISITORRATING] = r;
									}
											
									s.addClass('r' + r + (settings.useJalbumRating? '' : ' icon-close'))
										.on('click.' + ns, function(e) {
												var el = $(this),
													r = Math.minMax(0, Math.floor((getCoords(e).x - el.offset().left - parseFloat(el.css('paddingLeft'))) / 16), 5); 
													//Math.minMax(0, Math.floor((getCoords(e).x - el.offset().left + 4) / el.width()) * 5, 5);
													
												el.removeClass('r0 r1 r2 r3 r4 r5').addClass('r' + r);
												item[J.VISITORRATING] = r;
												
												// Propagate to thumbnails
												self.trigger('refreshRating', thumbs[curr]);
													
												return false;
											});
								} else {
									// Unalterable rating from jAlbum
									s.addClass('r' + r);
								}
							}
						}
						
						// Extra sizes
						
						if (!isExternal && extraSizes && item[J.CATEGORY] === 'image') {
							var p = album.getImagePath(item);
							
							el = $('<div>', {
									'class': 	'download icon-download'
								});
							
							if (subMenu) {
								target.append(el);
								
								for (var i = 0; i < extraSizes.length; i++) {
									el.append($('<a>', {
											text: 		extraSizes[i],
											href:  		p.replace('slides/', 'dl/' + extraSizes[i] + '/'),
											download: 	''
										}));
								}
							} else {
								el.addClass(settings.extraBtnClass).insertAfter(target);
							}
							
						}
						
						// Share buttons
						
						if ((DEBUG || !LOCAL) && settings.share && settings.showShare) {
							
							el = $('<a>', {
									'class': 	settings.buttonClass + ' icon-connect'
								}).data('tooltip', text.shareLabel);

							
							if (subMenu) {
								target.append(el);
								el.html('<span>' + text.share + '</span>')
							} else {
								el.addClass(settings.extraBtnClass).insertAfter(target);
							}
							
							el.on('click', function() {
									var target = $('<div>', {
												'class': 	'social'
											});
									
									lb_cont.modal(target, {
											title: 		text.shareOn,
											icon:		'connect',
											onClose:	function() {
																if (slideshowSuspended) {
																	resumeAuto();
																}
															}
										});
									
									setTimeout(function() {
											target.renderShares({
													sites: 			settings.share,
													title: 			item[J.TITLE] || item[J.NAME].stripExt().replace(/[-_]/g, ' ').capitalize(),
													description: 	item[J.COMMENT],
													image: 			album.getOptimalThumbPath(item, [ 960, 640 ]),
													href: 			album.getAbsolutePath(item)
												});
										}, 100);
									
									return false;
								});
						}
						
						// Print button
						
						if (settings.printImage && item[J.CATEGORY] === 'image' && !settings.rightClickProtect) {
							el = $('<a>', {
									'class': 	settings.buttonClass + ' icon-printer'
								}).data('tooltip', text.printLabel)
								
							if (subMenu) {
								target.append(el);
								el.html('<span>' + text.print + '</span>');
							} else {
								el.addClass(settings.extraBtnClass).insertAfter(target);
							}
							
							el.on('click', function(e) {
									printImage((item[J.ORIGINAL] && settings.showDownload)? album.getOriginalPath(item) : album.getImagePath(item),
										item[J.TITLE] || '',
										item[J.THUMBCAPTION] || ''
									);
								});
						}
						
						// Download button
						
						if (!isExternal && settings.showDownload) {
							var link;
							
							if (!!item[J.ORIGINAL] && (isImage || isPdf || settings.allowDownloadOthers)) {
								// Original of images and PDF's are allowed to download, others with extra permission
								link = 	album.getOriginalPath(item);
							} else if (isImage && settings.allowDownloadScaled) { 
								link = album.getImagePath(item);
							}
								
							if (link) {
								
								el = $('<a>', {
										'class': 		settings.buttonClass + ' icon-download',
										href: 			link,
										download: 		''
									});
								
								if (subMenu) {
									target.append(el);
									el.html('<span>' + text.download + '</span>');
								} else {
									el.addClass(settings.extraBtnClass).insertAfter(target);
								}
									
								if (!ISSEPARATEHOVER) {
									// Adding size as tooltip
									getFileSize(link, function(size, el) { 
											if (size && el) {
												el.addTooltip(niceByte(size), { pos: subMenu? [2,1,0,1] : [1,2,1,0], delay: 300 });
											}
										}, el);
								}
							}
						}
						
						// PayPal button
						
						if (settings.hasOwnProperty('shop') && album.hasShop(item) && settings.showShop) {
							var txt = album.getPriceRange(item);
							
							txt = text.addCart + (txt? (' <b>' + txt + '</b>') : '');
	
							el = $('<a>', {
									'class': 		settings.buttonClass + ' icon-shopping-cart-add'
								});
							
							if (subMenu) {
								target.append(el);
								el.html('<span>' + txt + '</span>');
								el.data('tooltip', text.shopLabel);
							} else {
								el.addClass(settings.extraBtnClass).insertAfter(target);
								el.data('tooltip', text.shopLabel + ': ' + txt);
							}
							
							el.on('click', function(e) {
										
									if (slideshow) {
										suspendAuto();
										settings.shop.root.trigger('onClose', function() {
												if (slideshowSuspended) {
													resumeAuto();
												}
												settings.shop.root.trigger('onClose', null);
											});
									}
									
									settings.shop.root.trigger('addItems', item);
									return false;
								});
							
							var vb = $('<a>', {
									'class': 		settings.buttonClass + ' secondary icon-shopping-cart'
								}).data('tooltip', text.viewCartLabel);
							
							if (subMenu) {
								target.append(vb);
								vb.html('<span>' + text.viewCart + '</span>');
							} else {
								vb.addClass(settings.extraBtnClass).insertAfter(el);
							}
							
							vb.on('click', function(e) {
									settings.shop.root.trigger('showCart');
								});
						}
						
						// Feedback button
						
						if (settings.hasOwnProperty('feedback') && settings.showFeedback) {
							el = $('<a>', {
										'class': 		settings.buttonClass + ' icon-email-send'
									}).on('click', function(e) {
									
										if (slideshow) {
											suspendAuto();
											settings.feedback.root.trigger('onClose', function() {
													if (slideshowSuspended) {
														resumeAuto();
													}
													settings.feedback.root.trigger('onClose', null);
												});
										}
										
										settings.feedback.root.trigger('addItems', item);
										return false;
									}).appendTo(target);
						
							el.html('<span>' + text.addComment + '</span>')
								.data('tooltip', text.feedbackLabel);
							
						}
						
						// Fotomoto button
						
						if ((!LOCAL || DEBUG) && typeof FOTOMOTO !== UNDEF && !!settings.fotomoto && !album.hideFotomoto(item)) {
							el = $('<a>', {
										'class': 	settings.buttonClass + ' icon-fotomoto'
									}).data('tooltip', text.fotomotoTooltip);
							
							if (subMenu) {
								target.append(el);
								el.html('<span>' + text.fotomotoBtn + '</span>');
							} else {
								el.addClass(settings.extraBtnClass).insertAfter(target);
							}
							
							el.on('click', function(e) {
									e.preventDefault();
									
									if (slideshow) {
										stopAuto();
									}
									
									//console.log('Fotomoto button clicked!');
									if (typeof FOTOMOTO !== UNDEF) {
										FOTOMOTO.API.showWindow(10, 
											album.getOriginalPath(item) || album.getItemPath(item)
										);
									}
									return false;
								});
						}
						
						// Photo data
						
						if (!isExternal && !!item[J.PHOTODATA]) {
							el = $('<a>', {
									'class': 	settings.buttonClass + ' icon-camera'
								}).data('tooltip', text.metaLabel);
								
							if (subMenu) {
								target.append(el);
								el.html('<span>' + text.metaBtn + '</span>');
							} else {
								el.addClass(settings.extraBtnClass).insertAfter(target);
							}
							
							el.on('click', function() {
									
									if (slideshow) {
										suspendAuto();
									}
									
									lb_cont.modal($('<div>', {
											'class':	'photodata',
											html: 		item[J.PHOTODATA]
										}), {
											title: 		text.metaBtn,
											'class':	'photodata',
											alignTo:	subMenu? '' : $(this),
											icon:		'camera',
											onClose:	function() {
																if (slideshowSuspended) {
																	resumeAuto();
																}
															}
										});
									
									return false;
								});
						}
						
						// Regions
						
						if (!isExternal && settings.showRegions && !!item[J.REGIONS]) {
							el = $('<a>', {
									'class': 	settings.buttonClass + ' icon-facetag'
								}).data('tooltip', text.regionsLabel);
							
							if (subMenu) {
								target.append(el);
								el.html('<span>' + (settings.regionsBtn || text.regionsBtn) + '</span>');
							} else {
								el.addClass(settings.extraBtnClass).insertAfter(target);
							}
								
							paintRegions(JSON.parse(item[J.REGIONS]), settings.regionsVisible); 
							
							el.on('click', toggleRegions);
						}
						
						// Map
						
						if (settings.showMap && !!item[J.LOCATION]) {
							
							el = $('<a>', {
									'class': 	settings.buttonClass + ' icon-location'
								}).data('tooltip', text.mapLabel);
							
							if (subMenu) {
								target.append(el);
								el.html('<span>' + text.mapBtn + '</span>');
							} else {
								el.addClass(settings.extraBtnClass).insertAfter(target);
							}
						
							el.on('click', function() {
									var mc = $('<div>', {
												'class': 	'map-cont'
											});
								
									if (slideshow) {
										suspendAuto();
									}
									
									mc.height($window.height() * settings.mapHeight);
									
									lb_cont.modal(mc, {
											'class': 	'no-padding large',
											icon:		'location',
											onClose:	function() {
																if (slideshowSuspended) {
																	resumeAuto();
																}
															}
										});
									
									setTimeout(function() {
											if (settings.hasOwnProperty('mapApiKey') && settings.mapApiKey) {
												// Google
												mc.addMap({
													apiKey:				settings['mapApiKey'],
													type:				settings['mapType'] || 'hybrid',
													zoom:				settings['mapZoom'] || 16,
													location: 			item[J.LOCATION],
													fitBounds:			false,
													fullscreenControl: 	false,
													onTypeChanged:		function(type) {
																				settings.mapType = type;
																			},
													onZoomChanged:		function(zoom) {
																				settings.mapZoom = zoom;
																			}
												});
											} else {
												// OSM
												mc.addMap({
														resPath:			settings.resPath,
														zoom:				settings['mapZoom'] || 16,
														location: 			item[J.LOCATION],
														fitBounds:			false
													});
											}
										}, 100);
									
									return false;
							});
						}
						
						// Mostphotos
						
						if (!LOCAL && !isExternal && !!item[J.MOSTPHOTOS]) {
							el = $('<a>', {
									'class': 		settings.buttonClass + ' icon-shopping-cart',
									target: 		'_blank',
									href: 			'https://mostphotos.com/' + item[J.MOSTPHOTOS]
								}).data('tooltip', text.mostphotosLabel);
							
							if (subMenu) {
								target.append(el);
								el.html('<span>' + text.mostphotosBtn + '</span>');
							} else {
								el.addClass(settings.extraBtnClass).insertAfter(target);
							}
						}
						
						// Sound clip
						
						if (!isAudio && !!item[J.AUDIOCLIP]) {
							audioClip = $('<audio>', {
									'class':		'audioClip',
									src:			item[J.AUDIOCLIP],
									controlsList:	'nofullscreen nodownload noremote'
								});
							el = $('<a>', {
									'class':		'button play-pause icon-play'
								});
							
							if (subMenu) {
								target.append(el);
								el.html('<span>' + text.mostphotosBtn + '</span>');
							} else {
								el.addClass(settings.extraBtnClass).insertAfter(target);
							}
							
							// Setting half loudness
							audioClip[0]['volume'] = 0.5;
							
							el.on('click', function() {
									if ($(this).hasClass('icon-pause')) {
										pauseAudioClip();
									} else {
										startAudioClip()
									}
								});
							
							target.append(el).append(audioClip);
							
							// Trying to autoplay
							startAudioClip();
							
						} else {
							audioClip = $();
						}
						
						if (!subMenu) {
							lb_topbar.find('.' + settings.extraBtnClass).not('.download').addTooltip({ pos: [1,2,1,0], delay: 500 });
						}
						
						return target;							
					},	
				
				// Preloading the next image (direction sensitive)
				
				preloadNext = function() {
						var nextItem;
						
						if (direction < 0 && curr > 0) {
							nextItem = thumbs.eq(curr - 1).data(J.OBJ);
						} else if (curr < thumbs.length - 2) {
							nextItem = thumbs.eq(curr + 1).data(J.OBJ);
						}
						
						if (nextItem) {
							if (nextItem[J.CATEGORY] === 'image') {
								var next = new Image();
								next.src = album.getOptimalImagePath(nextItem, getLightboxDims(), settings.displayOriginals);
							}
						}
					},
				
				// Image clicked :: toggle controls or audio/video controls
				
				imageClicked = function(e) {
						
						if ($(e.target).closest('.lightbox-caption').length) {
							return true;
						}
						
						var touched = $('html').data('whatinput') === 'touch' || e.type === 'touchend';
						
						//log(e.type);
									
						if (isVideo || isAudio) {
							
							var m = media[0];
								
							if (m.paused) {
								m.play();
								if (VEND !== 'ms') {
									m.controls = false;
								}
								if (touched) {
									toggleTopbar();
									if (lightbox.hasClass(settings.thumbsVisibleClass)) {
										hideThumbs();
									}
									if (lightbox.hasClass(settings.captionVisibleClass)) {
										hideCaption();
									}
								}
							} else {
								m.pause();
								if (VEND !== 'ms') {
									m.controls = true;
								}
								if (touched) {
									if (settings.clickAction === 'togglecontrols') {
										toggleTopbar();
									} else if (settings.clickAction === 'toggleall') {
										toggleAll();
									}
								}
							}
							return false;
							
						} else if (isImage) {
							
							if (settings.clickAction === 'nextimage') {
								// image navigation
								if ((getCoords(e).x - main.position().left) > (main.width() / 2)) {
									nextImage();
								} else {
									previousImage();
								}
								if (touched) {
									showTopbar();
								}
							} else if (settings.clickAction === 'togglecontrols') {
								toggleTopbar();
							} else if (settings.clickAction === 'toggleall') {
								toggleAll();
							}
						}
					},
				
				removeOld = function() {
						if (oldMain.length) {
							oldMain.remove();
							oldMain = $();
						}
					},
					
				// Doing all the animations in one timeframe
				
				doTransitions = function(doneFn) {
						
						switch (settings.transitionType) {
							
							case 'crossFadeAndSlide':
								var mx = main.getMatrix(),
									ox = mx[4];
								
								if (direction) {
									// Offset by 100 pixels
									mx[4] += 100 * direction;
								
									main.css({
											display:		'block',
											transition:		'none',
											opacity: 		0,
											transform:		'matrix(' +  mx.join(',') + ')'
										}).show();
								} else {
									main.css({
											display:		'block',
											transition:		'none',
											opacity: 		0
										}).show();
								}
								
								window.requestAnimationFrame(function() {
										
										if (oldMain.length && oldMain !== main) {
											
											if (oldMain.data('swipe') === 'swipe') {
												setTimeout(removeOld, transitionSpeed / 2);
											} else {
												var omx = oldMain.getMatrix();
												
												omx[4] -= 100 * direction;
												
												oldMain.one('transitionend', removeOld).css({
														transition:		'transform ' + (transitionSpeed / 2) + 'ms ease-out, opacity ' + (transitionSpeed / 2) + 'ms ease-out'
													}).css({
														transform:		'matrix(' + omx.join(',') + ')',
														opacity:		0
													});
											}
										}
										
										main.one('transitionend', function() {
												afterTransition(doneFn);
											});
										
										setTimeout(function() {
												if (direction) {
													// Restore final pos
													mx[4] = ox;
													// Animate to the final position and opacity
													main.css({
															transition:		'transform ' + transitionSpeed + 'ms ease-out, opacity ' + transitionSpeed + 'ms ease-out'
														}).css({
															transform:		'matrix(' + mx.join(',') + ')',
															opacity:		1
														});
												} else {
													main.css({
															transition:		'transform ' + transitionSpeed + 'ms ease-out, opacity ' + transitionSpeed + 'ms ease-out'
														}).css({
															opacity:		1
														});
												}
											}, TOUCHENABLED? 350 : 50);
									});
									
								break;
								
							case 'crossFadeAndZoom':
								
								var mx = main.getMatrix(),
									osx = mx[0],
									osy = mx[3];
								
								if (direction) {
									mx[0] *= (1 - direction / 40);
									mx[3] *= (1 - direction / 40);
									
									main.css({
											display:		'block',
											transition:		'none',
											opacity: 		0,
											transform:		'matrix(' + mx.join(',')  + ')'
										});
								} else {
									main.css({
											display:		'block',
											transition:		'none',
											opacity: 		0
										});
								}
								
								window.requestAnimationFrame(function() {
										
										if (oldMain.length && oldMain !== main) {
											
											if (oldMain.data('swipe') === 'swipe') {
												setTimeout(removeOld, transitionSpeed / 2);
											} else {
												var omx = oldMain.getMatrix();
												
												omx[0] *= (1 + direction / 20);
												omx[3] *= (1 + direction / 20);
												
												oldMain.one('transitionend', function() {
														oldMain.remove();
														oldMain = $();
													}).css({
														transition:		'transform ' + (transitionSpeed / 2) + 'ms ease-out, opacity ' + (transitionSpeed / 2) + 'ms ease-out',
														transform:		'matrix(' + omx.join(',') + ')',
														opacity:		0
													});
											}
										}
										
										main.one('transitionend', function() {
												afterTransition(doneFn);
											});
										
										setTimeout(function() {
												if (direction) {
													// Restore final scale
													mx[0] = osx;
													mx[3] = osy;
													// Animate to the final scale and opacity
													main.css({
															transition:		'transform ' + transitionSpeed + 'ms ease-out, opacity ' + transitionSpeed + 'ms ease-out',
															transform:		'matrix(' + mx.join(',') + ')',
															opacity:		1
														});
												} else {
													// No direction -> no scale
													main.css({
															transition:		'transform ' + transitionSpeed + 'ms ease-out, opacity ' + transitionSpeed + 'ms ease-out',
															opacity:		1
														});
												}
											}, TOUCHENABLED? 350 : 50);
									});
								
								break;
								
							default:
								
								main.css({
										display:		'block',
										transition:		'none',
										opacity: 		0
									});
								
								window.requestAnimationFrame(function() {
										
										if (oldMain.length && oldMain !== main) {
											
											if (oldMain.data('swipe') === 'swipe') {
												setTimeout(removeOld, transitionSpeed / 2);
											} else {
												oldMain.on('transitionend', function() {
														oldMain.remove();
														oldMain = $();
													}).css({
														transition:		'opacity ' + (transitionSpeed / 2) + 'ms ease-out', 
														opacity: 		0
													});
											}
										}
										
										main.one('transitionend', function() {
												afterTransition(doneFn);
											}).css({
												transition: 	'opacity ' + transitionSpeed + 'ms ease-out',
												opacity:		1
											});
										
									});
						}
						
					},
					
				// Transition has completed
					
				afterTransition = function(doneFn) {
						
						if (typeof doneFn === FUNCTION) {
							doneFn.call();
						}
						
						if (typeof settings.onLoadEnd === FUNCTION) {
							settings.onLoadEnd(thumb);
						}
						
						if (settings.preloadNext) {
							preloadNext();
						}
					},
				
				// The image is loaded
				
				imageReady = function() {
					
						loadCounter--;
						
						if (loadCounter > 0) {
							// Still has to load another image: (poster image) 
							return;
						}
										
						lb_activity.hide();
						
						/*			
						// Performace metrics
						if (typeof performance !== UNDEF) {
							
							console.log('performance exists');
									
							var pl = performance.getEntriesByType("resource");
							
							if (pl && Array.isArray(pl)) {
								var p = pl[pl.length - 1];
								
								console.log('performance last entry: ' + p);
								
								if (typeof image[0] !== UNDEF && p.name === image[0].src) {
									connectionSpeed = connectionSpeed? ((connectionSpeed + p.decodedBodySize / p.duration) / 2) : (p.decodedBodySize / p.duration);
								}
								
								if (DEBUG) {
									console.log(p.initiatorType + ':' + p.name.substring(p.name.lastIndexOf('/') + 1) + ' [' + Math.round(p.decodedBodySize / 1000) + 'kB] loaded in ' + Math.round(p.duration) + 'ms => ' + (p.decodedBodySize / p.duration / 1000).toFixed(2) + 'MB/s' + ' (avg:' + (connectionSpeed / 1000).toFixed(2) + 'MB/s)');
								}
							}
							
						}
						*/
						
						// Reporting image load in Debug mode
						if (DEBUG) {
							var d = new Date();
							if (isExternal || isVr) {
								console.log((isExternal? 'External content' : '360 player') + ' loaded: ' + (d - loadTimer) + 'ms');
							} else if (image.length) {
								console.log(((isVideo || isAudio)? 'Media' : 'Image') + ' [' + curr + '] loaded: ' + (d - loadTimer) + 'ms src="' + image[0].src + '"' + (connectionSpeed? (' avg. speed: ' + (connectionSpeed / 1000).toFixed(2) + 'MB/s') : ''));
								if (typeof image[0] === UNDEF || typeof media[0] === UNDEF) {
									console.log('Premature ' + (typeof image[0] === UNDEF)? ('loadImage.done(' + image[0].src + ')') : ('loadMedia.done('+ media[0].src + ')'));
								}
							}
						}
						
						// Right click protect
						if ((isImage || isVideo || isAudio) && settings.rightClickProtect) {
							media.on('contextmenu', function(e) {
									e.preventDefault()
									return false;
								});
						}
													
						// Audio / video initilazion
						if (isVideo || isAudio) {
							
							media.attr({
									autoplay: 	settings.videoAuto,
									loop:		settings.videoLoop
								});
							media[0].volume = settings.volume;
							
							media.on('volumechange.' + ns, function() {
									settings.volume = this.volume;
								});
						}
												
						if (!isVr && !isPdf && !isVideo && (!isExternal || !main.children('iframe').length)) {
							// Swipe handler
							main.swipe({
									onSwipeStart:	function() {
															$(this).trigger('autopanoStop');
															$(this).hideAllTooltips();
														},
									
									onSwipedLeft: 	function() {
															main.data('swipe', 'swipe');
															nextImage();
														},
													
									onSwipedRight: 	function() {
															main.data('swipe', 'swipe');
															previousImage();
														},
													
									onSwipedDown:	function() {
															main.data('swipe', 'down');
															quitLightbox();
														},
													
									onPinchZoom:	function(e) {
															main.data('swipe', 'pinch');
															if (lightbox.data('panomove')) {
																e.preventDefault();
																$(this).trigger('autopanoStop');
															}
															fixZoom(main.getScale()[0], getCoords(e));
														},
													
									onFinished: 	function() {
															if (!dontRemove) {
																$(this).trigger('removeSwipe');
																$(this).remove();
															}
														},
													
									onClick:		function(e) {
															if (lightbox.data('panomove')) {
																main.data('swipe', 'panostop');
																e.preventDefault();
																$(this).trigger('autopanoStop');
															} else {
																main.data('swipe', 'click');
																imageClicked(e);
															}
														},
														
									onRightClick:	function(e) {
															stopAuto(true);
														},
														
									onDoubleClick:	(settings.useZoom && settings.doubleClickZoom)? 
														function(e) {
																var c = getCoords(e);
																main.data('swipe', 'doubleclick');
																if (settings.zoomSlider) {
																	if (zoomLevel < 100) {
																		increaseZoom(50, c);
																	} else {
																		setZoom(0);
																	}
																} else {
																	toggleZoom(c);
																}
														}
														: 
														null
								});
						}
								
						// Placing and centering the new image
						fitImage(false, null, function(doneFn) {
								// Running transitions
								doTransitions(doneFn);
							});
						
						// Check if video has started 
						if (isVideo && settings.videoAuto && media[0].paused) {
							playVideo(media[0]);
						}
						
						// Continuing slideshow
						if (slideshow) {
							if ((isAudio || isVideo) && settings.videoAuto) {
								suspendAuto();
								media[0].onended = function() {
									resumeAuto();
								};
							} else {
								clearTimeout(slideshow);
								slideshow = setTimeout(nextImage, slideshowDelay);
							}
						} else {
							//stopAuto(true);
							if (settings.autohideTopbar) {
								hideTopbarLater();
							}
						}
						
					},
				
				// Recentering after swipe for example
				
				recenter = function(el) {
						var el = el || main;
						if (el.data('swipe') === 'swipe') {
							var matrix = main.getMatrix();
							matrix[4] = -main.width() / 2;
							el.one('transitionend', function() {
									el.data('swipe', null);
								}).css({
									opacity: 		1,
									transition:		'opacity ' + (settings.transitionSpeed * 2) + 'ms ease-out, transform ' + (settings.transitionSpeed * 2) + 'ms ease-out',
									transform: 		'matrix(' + matrix.join(',') + ')'
								});
						}
					},
					
				// Showing lightbox
				
				showLightbox = function() {
						
						lightbox.css({
								display:		'flex',
								visibility:		'visible',
								transition:		'opacity ' + settings.transitionSpeed + 'ms ease-out'
							})
							.one('transitionend', function() {
									if (settings.autoStart) {
										startAuto();
									}
								})
							.css('opacity', 1);
						
						lb_overlay.css({
								transition:		'opacity ' + (settings.transitionSpeed + 250) + 'ms ease-out',
								opacity:		1
							});
					},
					
				// Main entry point to load a new image
				
				loadImage = function(n) {
					
						if (typeof n !== 'number') {
							n = thumbs.index(n);
						}
						
						if (n < 0 || n >= thumbs.length) {
							// Out of bounds move
							
							dontRemove = true;
							
							if (n < 0) {
							
								switch (settings.afterLast) {
									case 'donothing':
										n = 0;
										if (main.data('swipe') === 'swipe') {
											recenter();
										}
										break;
										
									case 'startover':
										n = thumbs.length - 1;
										break;
										
									case 'onelevelup': 
										if (settings.level) {
											window.location.href = '../' + settings.indexName;
										}
										n = 0;
										break;
										
									case 'nextfolder':
										if (settings.baseSet && settings.previousFoldersLast) {
											window.location.href = settings.previousFoldersLast;
										} else {
											n = 0;
											if (main.data('swipe') === 'swipe') {
												recenter();
											}
										}
										break;
										
									case 'nextindex':
										if (settings.baseSet && settings.previousFolderPath) {
											window.location.href = settings.previousFolderPath;
										} else {
											n = 0;
											if (main.data('swipe') === 'swipe') {
												recenter();
											}
										}
										break;
										
									default:
										n = 0;
										quitLightbox();
										return;
								}
								
							} else {
								
								switch (settings.afterLast) {
									
									case 'donothing':
										dontRemove = true;
										n = thumbs.length - 1;
										stopAuto();
										if (main.data('swipe') === 'swipe') {
											recenter();
										}
										break;
										
									case 'startover':
										n = 0;
										break;
										
									case 'onelevelup':
										if (settings.level) {
											window.location.href = '../' + settings.indexName;
										}
										n = thumbs.length - 1;
										stopAuto();
										break;
										
									case 'nextfolder':
										if (settings.baseSet && settings.nextFoldersFirst) {
											window.location.href = settings.nextFoldersFirst + (slideshow? '&slideshow' : '');
										} else {
											n = thumbs.length - 1;
											stopAuto();
											if (main.data('swipe') === 'swipe') {
												recenter();
											}
										}
										break;
										
									case 'nextindex':
										if (settings.baseSet && settings.nextFolderPath) {
											window.location.href = settings.nextFolderPath;
										} else {
											n = thumbs.length - 1;
											stopAuto();
											if (main.data('swipe') === 'swipe') {
												recenter();
											}
										}
										break;
										
									case 'ask':
										
										dontRemove = true;
										
										var sr = stopAuto(),
											menu = $('<div>', {
													'class': 	'menu'
												});
										
										if (main.data('swipe') === 'swipe') {
											recenter();
										}
										
										n = thumbs.length - 1;
							
										if (thumbs.length > 1) {
											// Start over
											menu.append($('<a>', {
													'class':	'button icon-loop',
													html:		text.startOver
												}).on('click', 	function() { 
													loadImage(0);
												}));
										}
										
										if (settings.level) {
											// Up one level
											menu.append($('<a>', {
													'class':	'button icon-one-level-up',
													html:		settings.level? text.upOneLevel : (settings.homepageLinkText || text.backToHome)
												}).on('click', 	function() { 
													window.location.href = '../' + settings.indexName;
												}));
										}
										
										// Back to thumbnails
										menu.append($('<a>', {
												'class':	'button icon-arrow-up',
												html:		text.backToIndex
											}).on('click', 	function() { 
												quitLightbox();
											}));
										
										if (settings.baseSet) {
											
											if (settings.nextFoldersFirst) {
												// Go to next folder
												menu.append($('<a>', {
														'class':	'button icon-arrow-right',
														html:		text.nextFolder
													}).on('click', 	function() { 
														window.location.href = settings.nextFoldersFirst + (sr? '&slideshow' : '');
													}));
											}
											
											if (settings.nextFolderPath) {
												// Go to next index
												menu.append($('<a>', {
														'class':	'button icon-thumbnails',
														html:		text.nextIndex
													}).on('click', 	function() { 
														window.location.href = settings.nextFolderPath;
													}));
											}
										}
								
										lb_cont.modal(menu, {
												alignTo:	lb_btn_right,
												'class':	'context-menu',
												onClose:	function() {
																loadImage(thumbs.length - 1);
															}
											});
										
										return;
											
									default:
										// Back to index
										n = slideshow? 0 : (thumbs.length - 1);
										stopAuto();
										// Resetting play position to 0
										quitLightbox(thumbs[0]);
										return;
								}
							}
						} else {
							dontRemove = false;
						}
						
						// Direction is calculated if the current thumb exists
						if (lightbox.is(':visible')) {
							// Lightbox is on
							if ((Math.abs(n - curr) >= thumbs.length - 1) && settings.afterLast === 'startover') {
								// Flip over
								direction = 1;
							} else {
								direction = (curr > n)? -1 : ((curr < n)? 1 : 0);
							}
							
							if (curr === n) {
								// the requested image is already on screen :: nothing to do
								return;
							}
							
							//lb_topbar.hideAllTooltips();
							
							if (settings.autoHideTopbar) {
								hideTopbarLater();
							}
							
						} else {
							// We're on the index page :: show lightbox
							listening = true;
							$body.add($('html')).css('overflow', 'hidden');
							$body.addClass(settings.lightboxOnClass);
							direction = 0;
							
							if (settings.lightboxFullscreen) {
								requestFullscreen();
							}
							
							if (settings.useRating) {
								updateRating();
							}
							
							showLightbox();
							backgroundAudioOn = backgroundAudioRoot.length && !backgroundAudioRoot.data('paused');
							
							lazyloadThumbs(fitThumbstrip);
						}
						
						closeModal();
						
						var item;
						
						curr = n;
						thumb = thumbs.eq(curr);
						item = thumb.data(J.OBJ);
						
						if (settings.useThumbstrip && thumbs.length > 1) {
							setActiveThumb();
						}
						
						if (!item) {
							console.log('Fatal error: image (' + curr + ') is missing from the database! (Upload data1.json again!)');
							return;
						}
						
						lb_activity.show();
						
						getImageType(item);
						getSlideshowTiming(item);
						
						if (typeof settings.onLoadStart === FUNCTION) {
							settings.onLoadStart(thumb);
						}
						
						setTimeout(function() {
							
							oldMain = lb_cont.find('.' + settings.lightboxMainClass);
							
							if (oldMain.length) {
								
								oldMain.trigger('autopanoRemove').off('.' + ns);
								
								if (settings.muteBackgroundAudio && backgroundAudioOn) {
									if (isAudio || isVideo) {
										backgroundAudioRoot.trigger('fadeOutPlayer');
									} else {
										backgroundAudioRoot.trigger('fadeInPlayer');
									}
								}
								
								fadeMedia(oldMain);
							}
				
							// Delayed image loading
		
							loadCounter = 1;
							loadTimer = new Date();
							
							main = $('<div>', {
									'class': 	'lightbox-main ' + (isExternal? 'external' : (isVr? 'vr' : (isPdf? 'pdf' : item[J.CATEGORY])))
								})
								.css({
									opacity: 	0,
									transform:	'matrix(1,0,0,1,-50%,-50%)'
								})
								.appendTo(lb_cont);
								
							if (settings.hasOwnProperty('shop') && album.hasShop(item) ||
								settings.hasOwnProperty('feedback') && settings.showFeedback) {
								main.append($('<a>', {
										'class':		settings.checkboxClass + (thumb.hasClass(settings.selectedClass)? (' ' + settings.checkmarkClass) : '')
									}).on('click', function() {
										$(this).toggleClass(settings.checkmarkClass);
										thumb.trigger('toggleCard');
										return false;
									}));
								if (settings.hasOwnProperty('shop') && album.hasShop(item)) {
									main.append($('<a>', {
											'class': 		'icon-shopping-cart-add add-cart-btn'
										})
										.addTooltip(text.addCart, { pos: [0,1,2,1] })
										.on('click', function(e) {
											if (slideshow) {
												suspendAuto();
												settings.shop.root.trigger('onClose', function() {
														if (slideshowSuspended) {
															resumeAuto();
														}
														settings.shop.root.trigger('onClose', null);
													});
											}
											settings.shop.root.trigger('addItems', item);
											return false;
										}));
								} else {
									main.append($('<a>', {
											'class': 		'icon-email-send add-cart-btn',
											text:			' ' + settings.feedbackAddBtnLabel
										}).on('click', function(e) {
											if (slideshow) {
												suspendAuto();
												settings.feedback.root.trigger('onClose', function() {
														if (slideshowSuspended) {
															resumeAuto();
														}
														settings.feedback.root.trigger('onClose', null);
													});
											}
											settings.feedback.root.trigger('addItems', item);
											return false;
										}));
								}
							}

							if (isVr) {
								
								var w = Math.round(lightbox.width() * settings.fitRatio),
									h = Math.round(lightbox.height() * settings.fitRatio);
									
								main.css({
											width:		w,
											height:		h
										})
									.attr('id', 'vr' + curr)
									.addClass('vr')
									.show()
									.one('DOMReady', imageReady);
								
								// Photosphere viewer by Jeremy Heleine
								if (typeof PhotoSphereViewer !== UNDEF) {
									setTimeout(function() {
											var opt = { container: 	main[0] },
												psv,
												gp;
												
											if (item[J.CATEGORY] === 'video') {
												opt.panorama = album.getVideoPath(item);
												opt.adapter = PhotoSphereViewer.EquirectangularVideoAdapter;
												opt.plugins = [ PhotoSphereViewer.VideoPlugin, {} ];
												opt.navbar = [ 'zoom', 'fullscreen' ];
											} else {
												opt.panorama = album.getAbsoluteImagePath(item);
												opt.navbar = [ 'autorotate', 'zoom', 'fullscreen' ];
												if (settings.autoRotate360 && settings.rotateSpeed360) {
													opt.autorotateDelay = 1000;
												}
												opt.autorotateSpeed = settings.rotateSpeed360 + 'rpm';
											}
											
											if (gp = item['gpano']) {
												if (gp['fov']) {
													opt.defaultZoomLvl = gp.fov;
												}
												if (gp['pan'] || gp['tilt'] || gp['roll']) {
													opt.sphereCorrection = { 
																pan:		Math.PI * gp['pan'] / 180, 
																tilt:		Math.PI * gp['tilt'] / 180, 
																roll:		Math.PI * gp['roll'] / 180
														};
												}
											}
											
											psv = new PhotoSphereViewer.Viewer(opt);
											
										}, transitionSpeed);
								} else {
									console.log('Fatal Error: Missing "photo-sphere-viewer.min.js"!');
								}
								
								// Considered ready after 200ms or the "ready" event, whatever comes first 
								setTimeout(imageReady, 200);
								
							} else if (isImage) {
								
								// Image
								// Creating
								media = image = $('<img>').appendTo(main);
								
								if (DEBUG) {
									console.log('Loading image [' + curr + '] src="' + album.getOptimalImagePath(item, getLightboxDims(), settings.displayOriginals) + '"');
								}
								
								if (image[0].complete && image[0].naturalWidth > 0) {
									// From cache
									imageReady();
									
								} else {
									// Loading
									image.one('load.' + ns, function(e) {
											// Triggered on load event
											imageReady();
										});
								}
							
								// Applying the source
								image[0].src = album.getOptimalImagePath(item, getLightboxDims(), settings.displayOriginals);
									
							} else if (isAudio || isVideo) {
								
								if (isVideo) {
									
									var	dur = album.getVideoDuration(item);
									
									// Video
									media = image = $('<video>', {
												preload: 		'auto',
												controlsList:	'nodownload',
												poster: 		album.getPosterPath(item)
											});
									
									if (dur === null || dur >= 2000) {
										// Normal video
										media[0]['controls'] = 'true';
									} else {
										// Short video < 2s
										//media[0].controls = false;
										media[0]['loop'] = 'true';
									}
									
								} else {
									
									var src = album.getPosterPath(item);
										
									// Audio
									loadCounter = 2;
									
									image = $('<img>', {
											'class': 	'poster' + (src.endsWith('poster.png')? ' default' : '')
										}).one('load', function(e) {
											imageReady();
										})
										.attr('src', src)
										.appendTo(main);
									
									media = $('<audio>', {
											preload: 		'auto',
											controls: 		'true',
											controlsList:	'nofullscreen nodownload'
										});
								}
								
								media.one('loadedmetadata', function(e) {
											imageReady();
										})
									.attr('src', album.getSourcePath(item))
									.appendTo(main);
								
								media.on({
										playing: 	mediaPlaying,
										pause: 		mediaPaused,
										ended:		mediaEnded
									});
										
								if (settings.muteBackgroundAudio && backgroundAudioRoot.length) {
									backgroundAudioOn = !backgroundAudioRoot.data('paused');
								}
								
							} else if (isExternal) {
								
								var cont = item[J.EXTERNAL]['cont'],
									size = item[J.EXTERNAL]['size'];
									
								if (size) {
									size = size.split('x');
									main.data({
										oWidth:		parseInt(size[0]),
										oHeight:	parseInt(size[1])
									})
								}
									
								main.one('DOMReady', function(e) {
										imageReady();
									});
								
								if (cont.match(/^https?\:\/\//i) || cont.match(/\.html?$/i) || cont.match(/^\.\.\//)) {
									main.addClass('iframe')
										.append($('<iframe>', { 
												width: 				'100%',
												height: 			'100%',
												src: 				cont,
												frameborder: 		0,
												allowfullscreen: 	'allowfullscreen'
											}));
								} else {
									if (cont.match(/^<iframe/i)) {
										main.addClass('iframe');
									}
									main.append(cont);
								}
								
								// Considered ready after 200ms or the "ready" event, whatever comes first 
								setTimeout(imageReady, 200);
								
							} else {
								// Other
								var appendIcon = function(target, item) {
														target.append($('<div>', {
																'class':	'icon',
																html:		'<a href="' + album.getSourcePath(item) + '" target="_blank"><img class="other" src="' + album.getImagePath(item) + '"></a><p class="icon-external"> ' + text.clickToOpen + '</p></div>'
															}));
													};
								// Other
								if (isPdf && HASPDFVIEWER && !ISIOSDEVICE) {
									// using the PDF viewer plugin
									var o = $('<object>', {
												type:		'application/pdf'
											})
											.attr('data', album.getSourcePath(item));
									
									appendIcon(o, item);
									main.append(o);
									setTimeout(imageReady, 400);
									
								} else if (settings.useGoogleDocs && !LOCAL && 
									'.docx.xlsx.txt.pdf'.indexOf('.' + (item.hasOwnProperty(J.ORIGINAL)? item[J.ORIGINAL][J.PATH] : item[J.NAME]).getExt()) >= 0) {
									// Fallback to Google Doc Viewer
									main.addClass('iframe');
									var target = $('<iframe>', {
												width: 				'100%',
												height: 			'100%',
												frameborder: 		0,
												allowfullscreen: 	'allowfullscreen'
										}).appendTo(main);
									
									setTimeout(function(target, item) {
											target.attr('src', 'https://docs.google.com/viewer?url=' + window.location.href.getDir() + album.getSourcePath(item) + '&embedded=true');
											setTimeout(imageReady, 400);
										}, 100, target, item);
									
								} else {
									// Generic case with a download button
									appendIcon(main, item);
									imageReady();
								}
							}	
							
							prepareCaption(item);
																	
							if (typeof settings.onLoadStart === FUNCTION) {
								settings.onLoadStart(thumb);
							}
						
						}, 30);
					},
				
				// Hiding the lightbox overlay
				
				quitLightbox = function(thumb) {
						
						stopAuto(true);
						
						self.destroyAllTooltips();
							
						listening = false;
						$body.add($('html')).css('overflow', '');
						$body.removeClass(settings.lightboxOnClass);
						audioClip.remove();
						
						if (settings.muteBackgroundAudio && backgroundAudioOn) {
							backgroundAudioRoot.trigger('fadeInPlayer');
						}
						
						if (!main.length) {
							
							if(!SMALLSCREEN) {
								exitFullscreen();
							}
							
							if (typeof settings.onClose === FUNCTION) {
								if (typeof thumb !== UNDEF) {
									settings.onClose(thumb);
								} else {
									settings.onClose();
								}
							}
							
						} else {
							
							if (lightbox.data('panomove')) {
								// Panorama playing
								main.trigger('autopanoStop');
							}
							
							if ((isAudio || isVideo) && media) {
								var sv = media[0].volume,
									tm = transitionSpeed / (sv * 50),
									fade = function() {
											if (media) {
												var v = Math.max(media[0].volume - 0.02, 0);
												if (v > 0.005) {
													media[0].volume = v;
													setTimeout(fade, tm);
												} else {
													media[0].pause();
												}
											}
										};
								
								media.off('.' + ns);
								fade();
							}
							
							window.requestAnimationFrame(function() {
									
									main.css({
											transition: 	'opacity ' + (settings.transitionSpeed / 2) + 'ms ease-out'
										}).css({
											opacity:		0
										});
									
									lightbox.css({
											transition: 	'opacity ' + settings.transitionSpeed + 'ms ease-out'
										})
										.one('transitionend', function() {
											// Avoid flicker
											lightbox.css('visibility', 'hidden');
											setTimeout(function() {
													lightbox.css('display', 'none');
												}, 20);
											
											if (!SMALLSCREEN) {
												exitFullscreen();
											}
											
											if (main.length) {
												main.remove();
												main = $();
											}
											
											if (lb_caption.length) {
												lb_caption.remove();
												lb_caption = $();
											}
										})
										.css({
											opacity:		0
										});
								});
																
							//inProgress = false;
							if (settings.onClose !== false) {
								if (thumb !== UNDEF) {
									settings.onClose(thumb);
								} else {
									setting.onClose();
								}
							}
								
							showTopbar();
							$body.hideAllTooltips();
						}
					},
				
				// Removing the lightbox completely
				
				removeLightbox = function() {
						var remove = function() {
									if (settings.lightboxFullscreen) {
										exitFullscreen();
									}
									lightbox.remove();
								};
						// Stop slideshow		
						stopAuto(true);
						listening = false;
						// removing event handlers
						thumbs.off('.' + ns);
						$document.add($window).add(lightbox).off('.' + ns);
						$body.add($('html')).css('overflow', '');
						$body.removeClass(settings.lightboxOnClass);
						
						if (!main.length) {
							// No image; remove immediately
							remove();
						} else {
							// Fade out
							main.css('transition', 'opacity ' + settings.transitionSpeed + 'ms ease-out')
								.one('transitionend', remove)
								.css('opacity', 0);
						}
					},
				
				// Set play/pause button
				
				setPlayPause = function(on) {
					
						if (lb_play_pause) {
							if (on) {
								lb_play_pause.removeClass('icon-play').addClass('icon-pause');
							} else {
								lb_play_pause.removeClass('icon-pause').addClass('icon-play');//.removeClass('show-progress');
							}
						}
					},
				
				// Toggling auto slideshow
				
				toggleAuto = function() {
					
						if (slideshow) {
							stopAuto();
						} else {
							startAuto();
						}
						
					},
					
				// Starting auto slideshow
				
				startAuto = function(keepMusic) {
						
						clearTimeout(slideshow);
						
						setPlayPause(true);
						$body.add(lightbox).addClass(settings.immerseClass);
						
						if (settings.autohideTopbar) {
							hideTopbarLater();
						}
						
						lb_caption.addClass('slideshow');
						
						// Resetting zoom
						fitImages = settings.fitImages;
						
						// Starting background music
						if (!(keepMusic === true) && settings.backgroundAudioSlideshowControl) {
							backgroundAudioRoot.trigger('fadeInPlayer');
						}
						
						// Full screen mode
						if (settings.slideshowFullscreen) {
							requestFullscreen();
						}
						
						// Wake lock
						requestWakeLock();
						
						if (audioClipTimeout) {
							// Wait until audioclip ends
							slideshow = null;
							slideshowSuspended = true;
							onAudioClipEnded(nextImage);
						} else {
							// Immediately start
							slideshowSuspended = false;
							slideshow = setTimeout(nextImage, settings.slideshowDelay / 4);
						}
						
						if (settings.onSlideshowStart !== false) {
							settings.onSlideshowStart(thumb);
						}
					},
				
				resumeAuto = function() {
						
						setPlayPause(true);
						
						if (settings.autohideTopbar) {
							hideTopbarLater();
						}
						
						$body.add(lightbox).addClass(settings.immerseClass);
						slideshow = setTimeout(nextImage, settings.slideshowDelay / 4);
						slideshowSuspended = false;
						
						if (settings.backgroundAudioSlideshowControl) {
							backgroundAudioRoot.trigger('fadeInPlayer');
						}
					},
		
				stopAuto = function(keepMusic) {
						var sr = slideshow !== null;
					
						if (settings.onSlideshowPause !== false) {
							settings.onSlideshowPause(thumb);
						}
						
						if (!(keepMusic === true) && settings.backgroundAudioSlideshowControl) {
							backgroundAudioRoot.trigger('fadeOutPlayer');
						}
						
						setPlayPause(false);
						
						clearTimeout(slideshow);
						slideshow = null;
						slideshowSuspended = false;
						lb_caption.show().removeClass('slideshow');
						$body.add(lightbox).removeClass(settings.immerseClass);
						
						showTopbar();
						
						if (settings.slideshowFullscreen && !settings.lightboxFullscreen) {
							exitFullscreen();
						}
						
						releaseWakeLock();
						
						if (settings.onSlideshowPause !== false) {
							settings.onSlideshowPause(thumb);
						}
						
						return sr;
					},
				
				suspendAuto = function() {
					
						if (settings.backgroundAudioSlideshowControl) {
							backgroundAudioRoot.trigger('fadeOutPlayer');
						}
						
						clearTimeout(slideshow);
						slideshow = null;
						slideshowSuspended = true;
					},
					
				updateAudioClipProgress = function(pb) {
					
						if (audioClip.length && pb.length) {
							pb.css('width', (100 * audioClip[0].currentTime / audioClip[0].duration) + '%');
						}
					},
					
				startAudioClip = function() {
					
						if (audioClip.length) {
							
							audioClip[0].play().then(function() {
								
									var btn = audioClip.siblings('.audioclip-btn'),
										pb = btn.children('.progress');
										
									btn.removeClass('icon-volume-up').addClass('icon-pause');
									audioClip.parent().removeClass('paused').addClass('playing');
									
									if (slideshow) {
										suspendAuto();
									}
									
									if (!pb.length) {
										pb = $('<div>', {
												'class':	'progress'
											}).appendTo(btn);
									}
									
									updateAudioClipProgress(pb);
									
									audioClipTimeout = setInterval(updateAudioClipProgress, 100, pb);
									
									audioClip.on('ended', function() {
										
											btn.removeClass('icon-pause').addClass('icon-volume-up');
											audioClip.parent().removeClass('playing').addClass('paused');
											
											if (slideshowSuspended) {
												resumeAuto();
											}
											
											pb.remove();
											
											clearInterval(audioClipTimeout);
										});
											
								}, function(err) {
									
									btn.removeClass('icon-pause').addClass('icon-volume-up');
									console.log(err);
								});
						}
					},
					
				pauseAudioClip = function() {
							
						if (audioClip.length) {
							var btn = audioClip.siblings('.audioclip-btn');
							
							audioClip[0].pause();
							clearInterval(audioClipTimeout);
							
							btn.removeClass('icon-pause').addClass('icon-volume-up');
							audioClip.parent().removeClass('playing').addClass('paused');
							
							if (slideshowSuspended) {
								resumeAuto();
							}
						}
					},	
					
				onAudioClipEnded = function(fn, params) {
					
						if (audioClip.length && typeof fn === FUNCTION) {
							if (audioClip[0].paused) {
								if (typeof params !== UNDEF) {
									fn.apply(null, params);
								} else {
									fn.call(null);
								}
							} else {
								audioClip.on('ended', fn, params);
							}
						}
					},
					
				mediaPlaying = function() {
					
						if (settings.muteBackgroundAudio) {
							if (backgroundAudioOn = !backgroundAudioRoot.data('paused')) {
								backgroundAudioRoot.trigger('fadeOutPlayer');
							}
						}
						startAudioClip();
						main.addClass(settings.playingClass);
					},
					
				mediaPaused = function() {
						
						if (settings.muteBackgroundAudio && backgroundAudioOn) {
							backgroundAudioRoot.trigger('fadeInPlayer');
						}
						pauseAudioClip();
						main.removeClass(settings.playingClass);
					},
					
				mediaEnded = function() {
					
						if (settings.muteBackgroundAudio) {
							backgroundAudioOn = false;
						}
						pauseAudioClip();
						main.removeClass(settings.playingClass);
					},
					
				fadeMedia = function(el) {
						var media = (el || lb_cont).find('audio,video').eq(0);
						
						if (media.length) {
							var sv = media[0].volume,
								tm = settings.transitionSpeed / (sv * 50),
								fade = function() {
										if (media) {
											var v = Math.max(media[0].volume - 0.02, 0);
											if (v > 0.005) {
												media[0].volume = v;
												setTimeout(fade, tm);
											} else {
												media[0].pause();
											}
										}
									};
							
							media.off('volumechange.' + ns);
							fade();
						}
					},
					
				stopMedia = function(el) {
						var m = (el || lb_cont).find('audio,video').eq(0);
						
						if (m.length) {
							m[0].pause();
						}
					},
					
				showTopbar = function() {
						clearTimeout(topbarTimeout);
						lightbox.removeClass(settings.topbarHideClass);
						fitImage(true);
					},
				
				hideTopbar = function() {
						clearTimeout(topbarTimeout);
						lb_topbar.hideAllTooltips();
						lightbox.addClass(settings.topbarHideClass);
						fitImage(true);
					},
				
				hideTopbarLater = function() {
						clearTimeout(topbarTimeout);
						topbarTimeout = setTimeout(function() {
								//hideArrows();
								$body.hideAllTooltips();
								hideTopbar();
							}, slideshow? Math.min(settings.slideshowDelay / 2, settings.hideTopbarDelay) : settings.hideTopbarDelay);
					},
				
				toggleTopbar = function() {
						clearTimeout(topbarTimeout);
						lightbox.toggleClass(settings.topbarHideClass, !topbarHidden());
						fitImage(true);
					},
				
				topbarHidden = function() {
						return lightbox.hasClass(settings.topbarHideClass);
					},
				
				toggleFullscreen = function() {
						if (isFullscreen()) {
							exitFullscreen();
						} else {
							requestFullscreen();
						}
					},
					
				showThumbs = function() {
						lb_topbar.hideAllTooltips();
						lightbox.addClass(settings.thumbsVisibleClass);
						settings.thumbsVisible = thumbsVisibleDefault = true;
						savePrefs(settings, settings.prefsKept, 'lb_pref');
						if (main.data('fitted')) {
							// re-adjusting position
							fitImage(true);
						}
					},
				
				hideThumbs = function() {
						lb_topbar.hideAllTooltips();
						lightbox.removeClass(settings.thumbsVisibleClass);
						settings.thumbsVisible = thumbsVisibleDefault = false;
						savePrefs(settings, settings.prefsKept, 'lb_pref');
						if (main.data('fitted')) {
							// re-adjusting position
							fitImage(true);
						}
					},
				
				toggleThumbs = function() {
						if (settings.thumbsVisible) {
							hideThumbs();
						} else {
							showThumbs();
						}
					},
					
				showCaption = function() {
						lightbox.addClass(settings.captionVisibleClass);
						settings.captionVisible = captionVisibleDefault = true;
						savePrefs(settings, settings.prefsKept, 'lb_pref');
						if (main.data('fitted')) {
							// re-adjusting position
							fitImage(true);
						}
					},
					
				hideCaption = function() {
						lb_topbar.hideAllTooltips();
						lightbox.removeClass(settings.captionVisibleClass);
						settings.captionVisible = captionVisibleDefault = false;
						savePrefs(settings, settings.prefsKept, 'lb_pref');
						if (main.data('fitted')) {
							// re-adjusting position
							fitImage(true);
						}
					},
				
				toggleCaption = function() {
						if (settings.captionVisible) {
							hideCaption();
						} else {
							showCaption();
						}
					},
					
				toggleAll = function() {
						if (settings.captionVisible || settings.thumbsVisible) {
							thumbsVisibleDefault = settings.thumbsVisible;
							captionVisibleDefault = settings.captionVisible;
							// Hide everything
							lb_topbar.hideAllTooltips();
							clearTimeout(topbarTimeout);
							lightbox.addClass(settings.topbarHideClass);
							lightbox.removeClass(settings.thumbsVisibleClass + ' ' + settings.captionVisibleClass);
							settings.thumbsVisible = settings.captionVisible = false;
						} else {
							showTopbar();
							// Show items only to be visible by default
							if ((thumbsVisibleDefault || !captionVisibleDefault) && !settings.thumbsVisible) {
								lightbox.addClass(settings.thumbsVisibleClass);
								settings.thumbsVisible = true;
							}
							if ((captionVisibleDefault || !thumbsVisibleDefault) && !settings.captionVisible) {
								lightbox.addClass(settings.captionVisibleClass);
								settings.captionVisible = true;
							}
						}
						
						savePrefs(settings, settings.prefsKept, 'lb_pref');
						if (!settings.fitImages) {
							if (main.data('fitted')) {
								// re-adjusting position
								fitImage(true);
							}
						}
					},
					
				previousImage = function() {
						loadImage(curr - 1);
					},
				
				nextImage = function() {
						loadImage(curr + 1);
					},
					
				moveThumbs = function(x) {
						
						lb_thumbs
							.css({
								transition:		'transform ' + settings.thumbSpeed + 'ms ease-out',
								transform:		'translateX(' + x + 'px)'
							});
							
						lazyloadThumbs();
					},
				
				setActiveThumb = function() {
					
						if (!settings.useThumbstrip || thumbs.length < 2) {
							return;
						}
						
						var cthumb = lb_thumbs.children().eq((curr < 0)? 0 : curr),
							tl = cthumb.position().left,
							tw = cthumb.width(),
							cl = lb_thumbs.getTranslate()[0],
							cw = lb_thumb_cont.width(),
							tsw = lb_thumbs.width();
							
						lb_thumbs.children().removeClass(settings.activeClass);
						cthumb.addClass(settings.activeClass);
						
						if (tl < -cl) {
							moveThumbs(Math.min(0, settings.thumbOverhead - tl));
						} else  if ((tl + tw) > (cw - cl)) {
							moveThumbs(Math.max(cw - tl - tw - settings.thumbOverhead, cw - tsw));
						} else if ((cw - cl) > tsw) {
							moveThumbs(cw - tsw);
						}
							
					},
				
				// Adding one thumb
				
				loadThumb = function(thumb) {
						
						if (thumb.length) {
							var i = $('<img>', {
									'class': 	'hide-image'
								})
								// Onload action
								.one('load', function() {
									$(this).removeClass('hide-image').addClass('show-image');
									$(this).parent().removeClass(settings.lazyloadClass);
								})
								.attr('src', thumb.attr('href'));
							
							if (settings.rightClickProtect) {
								i.on('contextmenu', function(e) {
										e.preventDefault()
										return false;
									});
							}
							
							thumb.append(i).removeAttr('href');
						}
											
					},
					
				// Loading visible thumbs
				
				lazyloadThumbs = function(callback) {
					
						if (!lightbox.is(':visible') || !settings.useThumbstrip || thumbs.length < 2) {
							return;
						}
						
						var cw = lb_thumb_cont.width(),
							cl = lb_thumbs.getTranslate()[0];
						
						lb_thumbs.children('.' + settings.lazyloadClass).each(function() {
							var t = $(this),
								tl = t.position().left,
								tw = t.width();
								
							if (((tl + cl) < (cw * 7 / 4)) && ((tl + tw + cl) > (0 - cw * 3 / 4))) {
								// In view
								t.removeClass(settings.lazyloadClass);
								loadThumb(t);
								/*
								t.addTooltip({
										delay:	1000
									});
								*/
							} else if ((tl + cl) >= cw) {
								// Right to clip window
								return false;
							}
						});
						
						if (typeof callback === FUNCTION) {
							callback.call(this);
						}
					},
					
				// Fit thumbnail strip if shorter than the original space
				
				fitThumbstrip = function() {
						if (!lightbox.is(':visible')) {
							return;
						}
						
						var lt = lb_thumbs.children(':last-child');
	
						if ((lt.position().left + lt.outerWidth()) > lb_thumb_cont.width()) {
							lb_thumbstrip.addClass(settings.scrollClass);
							setActiveThumb();
						} else {
							lb_thumbstrip.removeClass(settings.scrollClass);
							moveThumbs(0);
						}
					},
					
				// Scrolling the thumb strip
				
				scrollThumbs = function(direction) {
						var cw = lb_thumb_cont.width(),
							cl = lb_thumbs.getTranslate()[0];
					
						if (cw > lb_thumbs.width()) {
							if (cl) {
								moveThumbs(0);
							}
						} else {
							cl = (direction < 0)? 
								Math.min(0, cl + cw)
								:
								Math.max(cw - lb_thumbs.width(), cl - cw);
							moveThumbs(cl);
						}						
					},
									
				// Settings up the thumb strip
				
				initThumbstrip = function() {
					
						if (settings.thumbsVisible) {
							lightbox.addClass(settings.thumbsVisibleClass);
						}
						
						// Thumbsstrip
						lb_thumbstrip = $('<div>', {
								'class':	'thumb-strip'
							})
							.appendTo(lightbox);
							
						lb_thumbstrip.append($('<button>', {
								'class':	'left icon-caret-left'
							}).on('click.' + ns, function(e) {
								scrollThumbs(-1);
								return false;
							}));
							
						lb_thumbstrip.append($('<button>', {
								'class':	'right icon-caret-right'
							}).on('click.' + ns, function(e) {
								scrollThumbs(1);
								return false;
							}));
							
						lb_thumb_cont = $('<div>', {
								'class':	'thumb-cont'
							}).appendTo(lb_thumbstrip);
							
						lb_thumb_cont.on('selectstart.' + ns, function(e) {
								return false;
							});
							
						lb_thumbs = $('<div>', {
								'class':	'thumbs'
							}).appendTo(lb_thumb_cont);
						
						lb_thumbstrip.append($('<a>', {
								'class':	'thumbs-toggle icon-thumbnails button',
								title:		text.showHideThumbnails
							}).on('click', function(e) {
								toggleThumbs();
							}).addTooltip());
							
						var item,
							w,
							h = lb_thumbs.height() || 74,
							a,
							tw = 0;
							
						thumbs.each(function(i) {
								
							item = $(this).data(J.OBJ);
							
							a = $('<a>', {
									'class':	settings.lazyloadClass,
									title:		item[J.THUMBCAPTION] || '',
									href:		album.getThumbPath(item)
								}).appendTo(lb_thumbs);
							
							a.on('click', function(e) {
									var a = $(e.target).closest('a');
									
									loadImage(lb_thumbs.children().index(a));
									return false;
								});
							
							w = Math.floor(Math.max((h - 4) * 3 / 4, Math.min(item[J.THUMB][J.WIDTH], (h - 4) * item[J.THUMB][J.WIDTH] / item[J.THUMB][J.HEIGHT])));
							
							a.width(w);
							
							tw += w;
						});
					},
					
				// Setting up the controls
				
				initTopbar = function() {
					
						// Lightbox top bar
						lb_topbar = $('<div>', {
									'class':	$('body > section.top-bar')[0].classList
								}).appendTo(lightbox);
								
						var topbar_cont,
							topbar_left,
							topbar_right;
						
						topbar_cont = $('<div>', {
									'class':	'bar'
								}).appendTo(lb_topbar);
								
						topbar_cont = $('<div>', {
									'class':	'cont'
								}).appendTo(topbar_cont);
								
						topbar_left = $('<div>', {
									'class':	'left-column'
								}).appendTo(topbar_cont);
								
						lb_title = $('<div>', {
									'class':	'title'
								}).appendTo(topbar_cont);
								
						topbar_right = $('<div>', {
									'class':	'right-column'
								}).appendTo(topbar_cont);
							
						// Close button
						lb_close = $('<a>', {
									'class': 	'button close-btn icon-thumbnails',
									'title': 	text.backToIndex
								})
							.on('click.' + ns, function() {
									if (listening) {
										quitLightbox(); 
									}
									return false; 
								})
							.appendTo(topbar_left);
							
						// Menu button
						lb_btn_menu = $('<a>', {
									'class': 	'button more icon-more-menu'
								})
								.on('click.' + ns, function(){
									if (listening) {
										showMenu();
									}
									return false;
								})
								.hide()
								.appendTo(topbar_right);
								
						// Zoom button
						if (settings.useZoom) {
							
							if (settings.zoomSlider) {
								// Slider
								var zoom_panel= $('<div>', {
											'class':	'zoom-control'
										}),
									zoom_out_btn = $('<a>', {
											'class':	'zoom-out button'
										}).appendTo(zoom_panel);
										
								lb_zoom_slider = $('<input>', {
											name:		'zoom-level',
											type:		'range'
										}).appendTo(zoom_panel);
										
								var zoom_in_btn = $('<a>', {
											'class':	'zoom-in button'
										}).appendTo(zoom_panel);
									
								// Zoom toggle button
								lb_zoom = $('<a>', {
											type: 		'button',
											'class': 	'button zoom-level icon-zoom-level'
										})
									.on('click.' + ns, function(){
											if (listening) {
												// Toggle zoom 
												toggleZoom();
											}
											return false; 
										})
									.addTooltip($('<div>', {
											'class':	'zoom-panel'
										}).append(zoom_panel), {
											touchToggle:	true
										}).appendTo(topbar_right);
										
									lb_zoom_slider.on('change', function() {
											setZoom(parseInt(lb_zoom_slider.val()));
											return true;
										});
									
									zoom_in_btn.on('click', function() {
											increaseZoom();
											return false;
										});
									
									zoom_out_btn.on('click', function() {
											decreaseZoom();
											return false;
										});
									
							} else {
								
								// Zoom toggle button
								lb_zoomin = $('<a>', {
											'class': 	'button zoom-in icon-expand',
											'title': 	text.oneToOneSize
										})
									.on('click.' + ns, function() {
											if (listening) {
												toggleZoom();
											}
											return false;
										})
									.appendTo(topbar_right);
								
								lb_zoomout = $('<a>', {
											'class': 	'button zoom-out icon-contract',
											'title': 	text.fitToScreen
										})
									.on('click.' + ns, function() { 
											if (listening) {
												toggleZoom();
											}
											return false; 
										})
									.appendTo(topbar_right);
									
								lb_zoom_slider = $();
							}
						}
						
						// Buttons for more than 1 items
						
						if (settings.useSlideshow && thumbs.length > 1) {
							
							// Play/Pause button
							lb_play_pause = $('<a>', {
									type: 		'button',
									'class': 	'button play-pause icon-play',
									'title': 	text.slideshow
								})
								.appendTo(topbar_right);
								
							lb_play_pause.on('click.' + ns, function(){ 
									if (listening) {
										toggleAuto();
										if ($('html').data('whatinput') === 'touch') {
											// Touch toggle
											return true;
										}
									}
									return false; 
								});
									
							if (settings.timingControl) {
								// Slideshow timing control
								var timing_panel= $('<div>', {
											id:			'timing-control',
											'class':	'timing-control'
										}),
									decr_time_btn = $('<a>', {
											'class':	'decr-time'
										}).appendTo(timing_panel);
										
								lb_timing_value = $('<span>').appendTo(timing_panel);
										
								var incr_time_btn = $('<a>', {
											'class':	'incr-time'
										}).appendTo(timing_panel);
									
								lb_play_pause.addTooltip($('<div>', {
											'class':	'timing-panel'
										}).append($('<div>', {
											'class':	'label',
											html:		text.timing
										})).append(timing_panel), {
											touchToggle:	true,
											delay:			600,
											pos:			[1,0,1,2]
										});
									
								incr_time_btn.on('click', function() {
										increaseTiming();
										return false;
									});
								
								decr_time_btn.on('click', function() {
										decreaseTiming();
										return false;
									});
								
								setTimeout(function() { 
										setTiming(settings.slideshowDelay);
									}, 500);
								
							} else {
								// No timing control
								lb_play_pause.attr('title', text.startStopSlideshow);
							}
								
							// Previous button
							lb_btn_left = $('<a>', {
										'class': 	'button prev icon-arrow-left'
									})
									.on('click.' + ns, function(){
										if (listening) {
											previousImage();
										}
										return false;
									})
									.appendTo(topbar_right);
								
							// Next button
							lb_btn_right = $('<a>', {
										'class': 	'button next icon-arrow-right'
									})
									.on('click.' + ns, function(){
										if (listening) {
											nextImage();
										}
										return false;
									})
									.appendTo(topbar_right);
									
						}
						
						if (settings.showFullscreen && hasFullscreen()) {
							lb_fullscr = $('<a>', {
										'class': 	'button fullscreen icon-fullscreen',
										title: 		text.fullScreen
									})
								.on('click.' + ns, toggleFullscreen)
								.appendTo(topbar_right);
								
							lb_exitfullscr = $('<a>', {
										'class': 	'button exitfullscreen icon-fullscreen-off',
										title: 		text.exitFullscreen
									})
								.on('click.' + ns, toggleFullscreen)
								.appendTo(topbar_right);
						}
						
						// Adding tooltips
						lb_topbar.find('a').not(lb_zoom).addTooltip();
					},
					
				// Setting up the structure
				
				initLightbox = function(ns) {
					
						lightbox = $('<div>', {
									id: 		ns,
									'class': 	'lightbox' + (captionVisibleDefault? (' ' + settings.captionVisibleClass) : '')
								})
							.hide()
							.appendTo(self);

								
						// Lightbox container
						lb_cont = $('<div>', {
									'class':	'lightbox-cont'
								}).appendTo(lightbox);
								
						// Darken background
						lb_overlay = $('<div>', {
									'class': 	'lightbox-overlay'
								})
							.appendTo(lb_cont);
						
						// Activity indicator
						lb_activity = $('<div>', {
									'class': 	'lightbox-loading'
								})
							.append('<div>')
							.append('<div>')
							.append('<div>')
							.appendTo(lb_cont);
							
						// Apply visibility classes
						lightbox.toggleClass(settings.thumbsVisibleClass, settings.thumbsVisible && thumbs.length > 1);
						lightbox.toggleClass(settings.captionVisibleClass, settings.captionVisible);
						//lightbox.toggleClass('zoomed', zoomed);
						
						// Top bar
						initTopbar();
			
						// Thumb strip
						if (settings.useThumbstrip && thumbs.length > 1) {
							initThumbstrip();
						}
						
						// Slideshow delay and transition speed
						if (!settings.hasOwnProperty('slideshowDelay')) {
							slideshowDelay = settings.slideshowDelay = 4000;
						}
						if (!settings.hasOwnProperty('transitionSpeed')) {
							transitionSpeed = settings.transitionSpeed = 400;
						}
						
						// Quit beside handler
						if (!settings.quitOnDocClick) {
							lb_overlay.on('click.' + ns, function(e) {
									
									if (listening) {
										if (this === e.target) {
											var p = getCoords(e);
											
											if (p.x >= lightbox.width() / 2) {
												nextImage();
											} else {
												previousImage();
											}
										}
										return false;
									}
								
								});
						}
					};
				
					
			// Initializing Lightbox
						
			if (ns) {
				// should never happen, but still living lightbox attached to thumb elements
				cleanup(ns);
			}
			
			// Creating new namespace
			self.data('llb_ns', ns = 'llb_' + Math.floor(Math.random() * 10000));		
	
			// Finding thumbs
			thumbs = self.find(settings.delegate);
			
			loadPrefs(settings, settings.prefsKept, 'lb_pref');
			
			// Initializing controls
			initLightbox();	
			
			// Setting up events
					
			// Resize
			$window.on('resize.' + ns /* + ' orientationchange.' + ns*/, function() {
					clearTimeout(resizeTimeout);
					// Executed only if 50ms has passed since the last resize event
					resizeTimeout = setTimeout(function() { 
						clearTimeout(resizeTimeout);
						if (main && main.length && !(isVr && isFullscreen())) {
							fitImage();
						}
						lazyloadThumbs(fitThumbstrip);
					}, 50);
				});
			
			// Quit on document touch
			if (settings.quitOnDocClick) {
				
				lb_overlay.on('click.' + ns, function(e) {
						
						if (listening) {
							if (settings.autohideTopbar && topbarHidden()) {
								// Toggle controls back
								showTopbar();
								return false;
							}
							
							if (main.length && !$(e.target).is(main)) {
								// Quit
								quitLightbox();
								return false;
							}
						}
						
						return true;
					});
			}
			
			// On mouse-driven devices
			if (settings.autohideTopbar) {
				
				// Auto hide topbar
				$document.on('mousemove.' + ns, function() {
						showTopbar();
						hideTopbarLater();
					});
			}
							
			// Keyboard
			if (settings.enableKeyboard) {
				
				$document.on('keyup.' + ns, function(e) {
					if (!listening || !main.length || lb_cont.children('.modal:visible').length || document.activeElement && (document.activeElement.nodeName === 'INPUT' || document.activeElement.nodeName === 'TEXTAREA')) {
						return true;
					}
					
					e.preventDefault();
					
					switch (e.keyCode) {
						
						case 27:
							quitLightbox();
							break;
						
						case 33:
						case 37:
							previousImage();
							break;
							
						case 34:
						case 39:
							nextImage();
							break;
							
						case 97:
						case 35:
							loadImage(thumbs.length - 1);
							break;
							
						case 103:
						case 36:
							loadImage(0);
							break;
						
						case 106:
						case 179:
							toggleAuto();
							break;
							
						case 107:
							if (settings.useZoom) {
								if (settings.zoomSlider) {
									increaseZoom();
								} else {
									toggleZoom();
								}
							} else {
								toggleCaption();
							}
							break;
							
						case 109:
							if (settings.useZoom && settings.zoomSlider) {
								decreaseZoom();
							} else {
								toggleAll();
							}
							break;
							
						case 32:
							if (isVideo) {
								if (media[0].paused) {
									playVideo(media[0]);
								} else {
									media[0].pause();
								}
							} else if (isAudio) {
								if (media[0].paused) {
									media[0].play();
								} else {
									media[0].pause();
								}
							} else {
								toggleAuto();
							}
							break;
							
						default:
							return true;
											
					}
					
					return false;
				});
			}
			
			// Mouse wheel
			if (settings.mouseWheelAction) {
				var wheelTimeout = null;
				
				lb_cont.on('mousewheel.' + ns + ' DOMMouseScroll.' + ns, function(e) {
						if (listening) {
							clearTimeout(wheelTimeout);
							
							if (	!main.length || 
									$(e.target).closest('.' + settings.lightboxCaptionClass).length ||
									main.hasClass('pdf') || 
									main.hasClass('external') ||
									lb_cont.children('.modal:visible').length
								) {
								return true;
							}
							
							if (e.ctrlKey == true || settings.mouseWheelAction === 'zoom') {
								if (settings.useZoom) {
									if (settings.zoomSlider) {
										wheelTimeout = setTimeout((e.originalEvent.wheelDelta > 0 || e.originalEvent.detail < 0)?
											function(e) {
												increaseZoom();
											}
											: 
											function(e) {
												decreaseZoom();
											}, 100);
									} else {
										wheelTimeout = setTimeout((e.originalEvent.wheelDelta > 0 || e.originalEvent.detail < 0)?
											function(e) {
												if (!fitImages) {
													fitImages = true;
													fitImage(true);
												}
											}
											: 
											function(e) {
												if (fitImage) {
													fitImages = false;
													fitImage(true);
												}
											}, 100);
									}
								}
							} else if (settings.mouseWheelAction === 'navigation') {
								var d = new Date();
								if (typeof lb_cont.data('lastWheel') === UNDEF || (d - lb_cont.data('lastWheel')) > 250) {
									// Only if 250ms apart from the last wheel event
									wheelTimeout = setTimeout(function(dir) {
											lb_cont.data('lastWheel', new Date());
											if (dir) {
												previousImage();
											} else {
												nextImage();
											}
									 	}, 100, (e.originalEvent.wheelDelta > 0 || e.originalEvent.detail < 0));
								}
							}
						}
						return false;
					});
			}
			
			// Events that can be triggered from outside
			
			// External trigger to switch image
			self.on('lightboxLoad.' + ns, function(e, n, auto) {
						stopAuto(true);
						
						if (typeof n === NUMBER) {
							n = Math.minMax(0, n, thumbs.length);
						} else if (typeof n === UNDEF) {
							n = 0;
						} else {
							n = thumbs.index(n);
						}
						
						if (settings.onStart !== false) {
							settings.onStart((typeof n === NUMBER)? thumbs.eq(n) : th);
						}
						
						loadImage(n);
						
						if (slideshow) {
							stopAuto(true);
						}
						
						if (auto) {
							if (typeof auto === NUMBER) {
								setTimeout(startAuto, auto);
							} else {
								startAuto();
							}
						}
					})
				// External trigger to quit = does not remove, just hides the lightbox
				.on('lightboxQuit.' + ns, function() {
						quitLightbox();
						return false;
					})
				// Removing the lightbox completely
				.on('lightboxRemove.' + ns, function() {
						removeLightbox();
						return false;
					})
				// Stopping auto hide topbar
				.on('stopAutoHide.' + ns, function() {
						clearTimeout(topbarTimeout);
					})
				// Returning the current lightbox container
				.on('lightboxContainer.' + ns, function() {
						return lightbox;
					})
				// External trigger to go to the previous image
				.on('lightboxNext.' + ns, function() {
						stopAuto();
						nextImage();
						return false;
					})
				// External trigger to go to next image
				.on('lightboxPrevious.' + ns, function() {
						stopAuto();
						previousImage();
						return false;
					});
					
			// Ready event
			
			if (settings.hasOwnProperty('onReady') && typeof settings.onReady === FUNCTION) {
				settings.onReady(thumb);
			}
			
			// Auto start
			
			if (settings.autoStart && settings.baseSet) {
				startAuto();
			}
	
			return this;
		};
	
	$.fn.lightbox.defaults = {
			delegate:						'.card.lbable',
			lightboxMainClass:				'lightbox-main',
			lightboxCaptionClass:			'lightbox-caption',
			lightboxOnClass:				'lightbox-on',
			topbarClass:					'top-bar',
			buttonClass:					'button',
			extraBtnClass:					'extra',
			lazyloadClass:					'lazyload',
			activeClass:					'active',
			scrollClass:					'scroll',
			regionsClass:					'regions',
			captionVisibleClass:			'caption-visible',
			thumbsVisibleClass:				'thumbs-visible',
			hasbuttonsClass:				'hasbuttons',
			immerseClass:					'immerse',
			topbarVisibleClass:				'topbar-visible',
			topbarHideClass:				'topbar-hide',
			checkboxClass:					'checkbox',
			checkmarkClass:					'icon-checkmark',
			selectedClass:					'checked',
			playingClass:					'playing',
			transitionProps:				'300ms ease-out',
			slideshowDelay:					4000,
			timingControl:					false,
			hideTopbarDelay:				2500,
			prefsKept: 						[ 
												'thumbsVisible',
												'captionVisible',
												'audioClipVolume'
											],
			thumbsVisible:					true,
			useThumbstrip:					false,
			captionVisible:					true,
			titleCaptionTemplate:			'<h3>${fileLabel}</h3>',
			showNumbers:					true,
			fitImages:						true,
			fitAboveCaption:				true,
			fitRatio:						0.94,
			maxZoom:						1.4,
			horzPanoThreshold:				1.8,
			vertPanoThreshold:				0.5,
			displayOriginals:				true,
			useZoom:						true,
			zoomSlider:						false,
			useOriginals:					false,
			hiDpiThumbs:					false,
			hiDpiImages:					false,
			showFullscreen:					false,
			indexName:						'index.html',
			baseSet:						true,
			autohideTopbar:					false,
			autoStart:						false,
			clickAction:					'toggleall',
			mouseWheelAction:				'navigation',
			useSlideshow:					true,
			autoplayAudioClips:				true,
			backgroundAudioSlideshowControl:false,
			muteBackgroundAudio:			true,
			use360Player:					true,
			autoRotate360:					true,
			rotateSpeed360:					-2,
			useGoogleDocs:					true,
			lightboxFullscreen:				false,
			slideshowFullscreen:			false,
			afterLast:						'donothing',
			doubleClickZoom:				true,
			mapHeight:						0.8,
			mapType:						'hybrid',
			mapZoom:						15,
			showShop:						false,
			showFeedback:					false,
			showLowestPrice:				false,
			videoAuto:						false,
			videoLoop:						false,
			autoPano:						false,
			autoPanoStart:					0,
			autoPanoSpeed:					80,
			volume:							0.5,
			audioClipVolume:				0.75,
			rightClickProtect: 				false,
			captionPlacement:				'center bottom',
			useRating:						false,
			jalbumRating:					false,
			visitorRating:					false,
			buttonLabels:					true,
			showNumbers:					true,
			infoPanelAdapt:					false,
			showShare:						false,
			showDownload:					false,
			allowDownloadScaled:			false,
			allowDownloadOthers:			false,
			showMap:						false,
			showRegions:					true,
			regionsVisible:					false,
			printImage:						false,
			transitionType:					'crossFadeAndSlide',
			transitionSpeed: 				400,
			panoramaAutoDetect:				1.3333,
			panoramaTreshold:				[ 2.5, 2.5 ],
			defaultPosterSize:				[ 628, 360 ],
			thumbSpeed:						400,
			thumbOverhead:					40,
			preloadNext:					true,
			enableKeyboard: 				true,
			quitOnEnd:						true,
			quitOnDocClick: 				true,
			onStart:						false,
			onClose:						false,
			onLoadStart:					false,
			onLoadEnd:						false,
			onReady:						false,
			onSlideshowStart:				false,
			onSlideshowPause:				false,
			onVolumeChange:					false,
			feedbackBtnLabel:				'Write feedback',
			feedbackBtnTooltip:				'View feedback window',
			feedback:						{}
		};
	
	$.fn.lightbox.text = {
			rating:							"Rating",
			shareLabel:						"Share this photo over social sites",
			share:							"Share",
			shareOn:						"Share on",
			printLabel:						"Print out this photo on your printer",
			print:							"Print",
			download:						"Download",
			addCart:						"Add to Cart",
			shopLabel:						"Add this item to the shopping cart",
			viewCartLabel:					"View shopping cart",
			viewCart:						"View Cart",
			addComment:						"Add comment",
			feedbackLabel:					"Feedback",
			fotomotoTooltip:				"Buy prints or digital files, share, send free eCards",
			fotomotoBtn:					"Buy / Share",
			metaLabel:						"Display photographic (Exif/Iptc) data",
			metaBtn:						"Photo data",
			regionsLabel:					"Show tagged people",
			regionsBtn:						"People",
			mapLabel:						"Show the photo location on map",
			mapBtn:							"Map",
			mostphotosLabel:				"Download this image from <b>mostphotos.com</b>!",
			mostphotosBtn:					"Purchase",
			startOver:						"Start over",
			upOneLevel:						"Up one level",
			backToHome:						"Back to home",
			backToIndex:					"Back to index page",
			nextFolder:						"Next folder",
			nextIndex:						"Next index page",
			clickToOpen:					"Click to open this document with the associated viewer",
			showHideThumbnails:				"Show / hide thumbnails",
			oneToOneSize:					"1:1 size",
			fitToScreen:					"Fit to screen",
			slideshow:						"Slideshow",
			timing:							"Timing",
			startStopSlideshow:				"Start / stop slideshow",
			fullScreen:						"Full screen",
			exitFullscreen:					"Exit"
		};

	
})(jQuery, jQuery(window), jQuery(document), jQuery('body'));
