/* 
 *	main.js - the skin's javascript functions
 */
	
;(function($, $window, $document, $body, undefined) {
	'use strict';

	// The main skin code
	
	// Random number between -base and +base 
	var	random = function(base) {
				return base * (1 - 2 * Math.random());
			};
	
	$.fn.skin = function(settings) {
		
			settings 				= $.extend({}, $.fn.skin.defaults, settings);
			
			var main 				= $(this).eq(0),								// Main container
				ns 					= 'animatics_skin',									// Namespace
				backgroundAudioRoot = $('[data-audioplayer]'),
				backgroundAudioOn 	= backgroundAudioRoot.length && !backgroundAudioRoot.data(settings.pausedClass),
				
				text 				= getTranslations({								// Translated texts
											newItem:						'NEW',
											atFirstPage:					'At first page',
											atLastPage:						'At last page',
											atLastPageQuestion:				'Where to go next?',
											startOver:						'Start over',
											backToHome:						'Back to home',
											upOneLevel:						'Up one level',
											previousFolder:					'Previous folder',
											nextFolder:						'Next folder',
											download: 						'Download',
											print:							'Print',
											metaBtn: 						'Photo data',
											clickToOpen:					'Click to open this document with the associated viewer!'
										}),
				
				id 					= {
											prev:							'jprev',
											next:							'jnext',
											cards:							'cards',
											card:							'card',
											caption:						'caption',
											title:							'title',
											hasCaption:						'hascaption',
											imageReady:						'imageready',
											prevPage:						'prevpg',
											nextPage:						'nextpg',
											currPage:						'currpg',
											threedee:						'threedee'
										},

				// Main sections
				lightbox			= $('#lightbox'),								// Lightbox element
				cards,																// Card container in Lightbox
				thumbnails			= $('#thumbnails'),								// Thumbnails section
				thumbCont			= $('#thumbnails .thumb-cont'),					// Thumbnail container
				thumbs				= thumbCont.find('.thumb'), 					// Collection of thumbs (<a>)
				controls			= $('#controls'),								// Controls overlay
				lbcontrols			= $('#lbcontrols'),								// Lightbox controls
				panels				= $('#panels'),									// Panels
				navigation			= $('#navigation'),								// Folders, up link, etc.
				numbers				= $('#numbers'),								// Numbers
				
				// Buttons
				prevBtn,															// Previous button
				nextBtn,															// Next button
				playBtn,															// Play button
				pauseBtn,															// Pause button
				fullscrBtn,															// Full screen button
				windowedBtn,														// Exit full screen
				progress,															// Progress bar
				
				// State
				curr				= 0,											// Current image
				nextCard			= 0,											// Next card to show
				dir					= 1,											// Direction
				loop				= false,										// Looped lightbox
				transition,															// Transition parameters
				scaling				= settings.scaling,								// Scaling method (none|fit|fill|full)
				lastChangeDate  	= new Date() - 10000,							// Last image change date for adaptive speed
				lastHash 			= '',											// Storing last state
				initByHash 			= false,										// Initial URL contains hash
				nows 				= new Date() / 1000,							// Now in seconds
				
				// Timeouts
				cleanupTo			= null,											// Clean up cards timeout
				focusTo				= null,											// Focus later timeout
				layoutTo			= null,											// Layout refresh on resize
				scrollTo			= null,											// Thumbnails scrolled timeout
				controlsTo			= null,											// Fade controls timeout
				progressTo			= null,											// Progressbar timeout
				slideshow			= null,											// slideshow timeout
				slideshowSuspended	= false,										// suspended slideshow 
				
				mousemoveListener	= false,										// Mouse move listener is engaged
				
				captionPlacement,
				
	
				/********************************************************
				 *
				 *						Utility functions
				 *
				 ********************************************************/
				 
				// Get image number by filename
				
				getImageByName = function(n) {
					
						if (n) {
							for (var i = 0; i < thumbs.length; i++) {
								if (thumbs.eq(i).attr('href').getFile() === n) {
									return i;
								}
							}
						}
						
						return -1;
					},
					
				// Update URL
				
				updateUrl = function(n) {
						var t = thumbs.eq(n),
							f = t.attr('href').getFile();
							
						addParam(settings.indexName, {
								img:	f
							}, t.attr('title') || f.stripExt());
					},
					
				// Hash/history changed
				
				stateChange = function() {
						if (window.location.hash === lastHash) {
							return;
						}
						
						lastHash = window.location.hash;
						var param = readParam();
						
						if (param.hasOwnProperty('img')) {
							var n = getImageByName(param.img);
							
							if (n >= 0) {
								focus(n);
							} else {
								removeParam(settings.indexName, 'img');
							}
							
							updateShares();
						}
					},
				
				// Type check
				
				typeCheck = function(query, n) {
						var t = thumbs.eq((typeof n === UNDEF)? curr : n).data('vars'),
							q = query.split(',');
						
						if (t && (t = t['type'])) {
							for (var i = 0; i < q.length; i++) {
								if (q[i] === t) {
									return true;
								}
							}
						}
						
						return false;
					},
							
				// Stops current animation
				
				stopTransition = function(el) {
						el.each(function() {
								var cs = window.getComputedStyle(this, null);
								//console.log('Stopping all transforms');
								this.style['transform'] = cs['transform'];
							});
					},
					
				/********************************************************
				 *
				 *						Controls
				 *
				 ********************************************************/
				 
				// Initializing background music
				
				initBackgroundMusic = function(player) {
					
						if (player.length) {
							$.fn.audioPlayer.defaults.rootPath = settings.rootPath;
							player.audioPlayer();
						}
					},
					
				// Getting active element's data
				
				getCurrentItem = function() {
						var card = getActiveCard();
						
						if (card.length) {
							return card.data(J.OBJ);
						}
						
						return null;
					},
					
				// Figuring out the share image path
					
				getShareImage = function(card) {
						var	vars = card.data('vars'),
							ip = card.attr('href');
						
						if (vars['type'] === 'video' || vars['type'] === 'audio') {
							ip = ip.replaceExt('jpg');
						} else if (ip) {
							if (ip.getExt() === 'gif') {
								ip = ip.replaceExt('png');
							}
						} else {
							ip = 'thumbs/' + vars['thumb'];
						}
						
						return ip;
					},
					
				// Updating the share buttons after a state change
				
				updateShares = function() {
					
						if (settings.hasOwnProperty('share')) {
							
							if (lightbox.is(':visible')) {
								// Lightbox: sharing actual image
								var th = thumbs.eq(curr),
									vars = th.data('vars');
								
								if (vars) {
									$(settings.share.hook).trigger('updateLinks', {
											sites: 			settings.share,
											title: 			th.attr('title') || th.attr('href').stripExt().replace(/[-_]/g, ' ').capitalize(),
											description: 	vars.caption || '',
											image: 			getShareImage(th),
											href: 			th.attr('href').replaceExt('html')
										});
								}
							} else {
								// Index page: reset to page sharing
								$(settings.share.hook).trigger('updateLinks');
							}
						}	
					},
										
				// Initializing Social sharing
				
				initShares = function(opt) {
						
						// Social sharing
						
						$.fn.renderShares.defaults.buttonTheme = opt['buttonTheme'] || 'dark';
						$.fn.renderShares.defaults.indexName = settings.indexName;
						$.fn.renderShares.defaults.facebookAppId = opt['facebookAppId'] || '';
						
						if (opt.hasOwnProperty('callAction')) {
							$.fn.renderShares.defaults.callAction = opt.callAction;
						}
						
						$(opt['hook']).renderShares(opt);
						
					},

				// Updating current number
				
				updateNumber = function(n) {
						
						if (numbers.length) {
							numbers.html('<big>' + n + '</big><small>' + thumbs.length + '</small>');
						}
					},
					
				// Fixing Thumbnails top position
				
				fixThumbPos = function() {
					
						var h = thumbnails.outerHeight(),
							t = thumbnails.position().top,
							wh = $window.height() - panels.position().top,
							bt = controls.find('[data-rel=thumbnails]').position().top;
							
						thumbnails.css('top', Math.minMax(0, wh - h, bt));
					},
					
				// Show controls
				
				showControls = function() {
					
						clearTimeout(controlsTo);
						controlsTo = null;
						main.removeClass(settings.hideControlsClass);
					},
					
				// Hide controls
				
				hideControls = function() {
					
						clearTimeout(controlsTo);
						controlsTo = null;
						
						if (!panels.children('.on').length) {
							// No panels open
							controls.add(lbcontrols).hideAllTooltips();
							main.addClass(settings.hideControlsClass);
						}
					},
					
				// Toggle controls
				
				toggleControls = function() {
					
						if (!panels.children('.on').length) {
							if (main.hasClass(settings.hideControlsClass)) {
								showControls();
							} else {
								hideControls();
							}
						}
					},
				
				// Listening to mouse move to get back control bar
					
				setupMousemoveListener = function() {
						
						if (settings.autoHideControls && whatInput.ask('intent') === 'mouse') {
							
							$document.on('mousemove.' + ns, function() {
									if (main.hasClass(settings.hideControlsClass)) {
										showControls();
									}
									
									hideControlsLater();
								});
							
							mousemoveListener = true;
						}
					},
				
				// Hide controls later
				
				hideControlsLater = function() {
					
						if (settings.autoHideControls && whatInput.ask('intent') === 'mouse') {
							
							if (!mousemoveListener) {
								setupMousemoveListener();
							}
							
							clearTimeout(controlsTo);
							controlsTo = setTimeout(hideControls, slideshow? Math.min(settings.slideshowDelay / 2, settings.hideControlsDelay * 1000) : settings.hideControlsDelay * 1000);
						}
					},
					
				// Show panel
				
				showPanel = function(p, doneFn) {
						
						var id = p[0].id,
							btn = controls.add(lbcontrols).find('[data-rel=' + id + ']:visible'),
							
							init = function() {
									// Preset
									var top = btn.position().top;
									
									// Related button
									btn.add(p).addClass('on');
									
									p.css({
											transition:		'none',
											display: 		'flex',
											opacity:		0,
											transform:		'translateX(-16px)',
											maxHeight:		($body.outerHeight() - panels.offset().top - top) + 'px' 		
										});
									
									if (id === 'thumbnails') {
										fixThumbPos();
									} else {
										p.css({
												top:			top,
											});
									}
									
									p.data('finished', false);
									window.requestAnimationFrame(start);
								},
								
							start = function() {
								
									p.one('transitionend', function() {
												if (!p.data('finished')) {
													p.data('finished', true);
													window.requestAnimationFrame(finish);
												}
											})
										.css({
												transition:		'all 500ms ease-out',
												opacity:		1,
												transform:		'translateX(0)'
											});
								
								},
								
							finish = function() {
								
								var ac = btn.data('autoclose');

									if (ac) {
										
										// Do not autoclose while mouse is around
										p.add(btn)
											.on('mouseenter.' + ns, function() {
													p.data('over', true);
													clearTimeout(p.data('fade'));
												})
											.on('mouseover.' + ns, function() {
													p.data('over', true);
												})
											.on('mouseleave.' + ns, function() {
													clearTimeout(p.data('fade'));
													p.data('over', true);
													p.data('fade', setTimeout(function() {
															if (!p.data('over')) {
																clearTimeout(p.data('fade'));
																hidePanel(p);
															}
														}, ac));
												});
									}
									
									if (id === 'thumbnails') {
										lazyloadThumbs(fixThumbPos, fixThumbPos);
									}
									
									if (typeof doneFn === FUNCTION) {
										doneFn.call(null);
									}
								};
						
						clearTimeout(p.data('fade'));
						window.requestAnimationFrame(init);
								
					},
						
				// Hide panel
				
				hidePanel = function(p, doneFn) {
					
						p.each(function() {
								var p = $(this),
									id = this.id,
									btn = $('#controls [data-rel=' + id + ']'),
									finished = false;
								
								// Related button
								btn.removeClass('on');
								p.hideAllTooltips();
								//console.log('Hiding "' + id + '"');
								//stopTransition(p);
								
								p.one('transitionend', function(e) {
											if (!finished) {
												finished = true;
												$(this)
													.removeClass('on')
													.css('display', 'none');
													
												if (typeof doneFn === FUNCTION) {
													doneFn.call(null);
												}
											}
										})
									.css({
											transition:		'all 500ms ease-out',
											opacity:		0,
											transform:		'translateX(-16px)'
										});
							});
					},
					
				hideAllPanels = function() {
						hidePanel(panels.children('.on'));
					},
					
				// Initializing controls
				
				initControls = function() {
						var menuTo = null;
						
						// Fullscreen button
						
						if (settings.showFullscreen) {
							
							fullscrBtn = lbcontrols.find('.fullscreen.btn');
							windowedBtn = lbcontrols.find('.windowed.btn');
							
							if (hasFullscreen()) {
								
								fullscrBtn.on('click.' + ns, function() {
										requestFullscreen(function() {
												main.addClass(settings.fullscreenClass);
											})
									});
								
								windowedBtn.on('click.' + ns, function() {
										exitFullscreen(function() {
												main.removeClass(settings.fullscreenClass);
											})
									});
								
							} else {
								
								// No API
								fullscrBtn.hide();
								windowedBtn.hide();
							
							}
								
						}
						
						// Toggling the whole menu
						
						controls
							.find('.nav-icon')
							.on('click', function() {
									clearTimeout(menuTo);
									if (controls.hasClass('active')) {
										// Hiding the control strip
										// Hide all panels
										hideAllPanels();
										// Remove animation
										controls.addClass('remove').removeClass('active');
										// Reseting
										menuTo = setTimeout(function() {
												controls.removeClass('remove');
											}, settings.transTime);
										main.addClass(settings.hideControlsClass);
									} else {
										// Showing the control strip
										controls.addClass('active').removeClass('remove');
										main.removeClass(settings.hideControlsClass);
									}
								});
						
							
						// Progress bar subelements
						
						progress = lbcontrols.find('.pause>.progress');
					
						// Toggle buttons
						
						controls
							.add(lbcontrols)
							.find('[data-rel]')
							.each(function() {
									var btn = $(this),
										p = $('#' + btn.data('rel'));
										
									if (p.length) {
										
										btn
											.addClass('has-panel')
											.on('selectstart.' + ns, function(e) {
													e.preventDefault();
													return false;
												})
											.on('click.' + ns, function(e) {
													
												if ((!$(e.target).length || !$(e.target).attr('href')) && controls.hasClass('active')) {
													// Only if panels are visible
													var btn = $(this),
														
													p = $('#' + btn.data('rel'));
													
													// Has panel
													if (!p.hasClass('on')) {
														// Turn ON
														
														// Hide all others
														hideAllPanels();
														
														// Show related
														showPanel(p);
														
													} else {
														// Turn OFF
														hidePanel(p);
													}
													
													return false;
												}
													
												return true;
											});
	
									} else {
										// No related panel
										btn.removeAttr('data-rel');
									}
								});
							
					},
					
				// Toggling fit to screen method with full zoom
				
				toggleZoom = function() {
						scaling = (scaling === 'full')? settings.scaling : 'full';
						layoutRefresh();
					},
						
										
				/********************************************************
				 *
				 *						Thumbnails
				 *
				 ********************************************************/
				 
				// Focus thumbnail
				
				focusThumb = function(n) {
					
						thumbs.removeClass(settings.activeClass).eq((typeof n !== UNDEF)? n : curr).addClass(settings.activeClass);
					},
												
				// Lazy loading thumbnails having "lazyload" class
				
				lazyloadThumbs = function(target, doneFn, layoutModFn) {
						var vert = main.hasClass('thumbs-left') || main.hasClass('thumbs-right'),
							pd = (vert? thumbCont.height() : thumbCont.width()),
							scroll,
							t,
							img,
							src,
							imgs = $();
						
						// Set target position if passed 
						if (typeof target === UNDEF || typeof target === FUNCTION) {
							scroll = vert? thumbCont.scrollTop() : thumbCont.scrollLeft();
							if (typeof target === FUNCTION) {
								if (typeof doneFn === FUNCTION) {
									layoutModFn = doneFn;
								}
								doneFn = target;
							}
						} else {
							scroll = target;
						}
						 
						thumbCont
							.find('a.' + settings.lazyloadClass)
							.filter(function() {
									// Visible or near visible items
									var p = $(this).position()[vert? 'top' : 'left'] + scroll;
									
									// Half screen width backwards, full screen width forwards
									return (p + pd / 2) >= 0 && (p - pd * 2) <= 0;
								})
							.each(function() {
									t = $(this);
									
									if (!t.children('img').length) {
										var vars = t.data('vars'),
											tw = vars['thumbWidth'],
											th = vars['thumbHeight'],
											ar = tw / th;
										
										if (src = vars['thumb']) {
											img = $('<img>', {
													'class': 	settings.hideImageClass + ' ' + ((ar >= 1.25)? 'landscape' : ((ar <= 0.8)? 'portrait' : 'square'))
												})
												// Onload action
												.one('load', function() {
													$(this).addClass(settings.showImageClass).removeClass(settings.hideImageClass);
													$(this).parent().removeClass(settings.lazyloadClass);
													if (typeof layoutModFn === FUNCTION) {
														layoutModFn.call();
													}
												})
												.attr({
													width:		tw,
													height:		th,
													alt:		vars['title'] || src.stripExt(),
													src:		getThumbSource(t, [ t.outerWidth(), t.outerHeight() ])
												})
												.appendTo(t);
												
											imgs.append(img);
										}
									}
								});
						
						if (typeof doneFn === FUNCTION) {
							if (imgs.length) {
								imgs.waitAllImg(doneFn);
							} else {
								doneFn.call();
							}
						}
					},
				
				// Initializing thumbnails
				
				initThumbs = function() {
						var d,
							el,
							cd = settings.hasOwnProperty('markNew')? (nows - settings.markNew.days * ONEDAY_S) : null;
						
						// Storing "rel", implementing click
						thumbs.each(function(i) {
								//this.vars = $(this).data('vars');
								$(this)
									.data('rel', i)
									.on('click', function(e) {
											if (e.target.nodeName !== 'SPAN') {
												thumbnails.hideAllTooltips();
												hidePanel(thumbnails);
												focus($(this).data('rel'));
											}
											return false;
										});
									
								if (whatInput.ask('intent') === 'mouse') {
									$(this).addTooltip();
								}
								
								if (cd) {
									d = $(this).data('vars')['date'];
									
									if (d && d >= cd) {
										el = $('<span>', {
												'class':	'icon-new new-image'
											}).appendTo($(this));
											
										if (whatInput.ask('intent') === 'mouse') {
											el.data('tooltip', (new Date(d * 1000)).toLocaleDateString()).addTooltip();
										}
									}
								}
							});
						
						// Right click protection on thumbnails
						if (settings.rightClickProtect) {
							thumbCont.on('contextmenu', 'img', function(e) {
									e.preventDefault()
									return false;
								});
						}
						
						// Lazyload 
						thumbCont.on('scroll.' + ns, function() {
								clearTimeout(scrollTo);
								scrollTo = setTimeout(lazyloadThumbs, 100);
							});
					},
					
				/********************************************************
				 *
				 *						Map
				 *
				 ********************************************************/
				 
					
				// Initializing map
				
				initMap = function() {

						// Reading map marker locations
						var getMarkers = function() {
								
									var markers = [];
									
									thumbs.each(function(i) {
										var vars = $(this).data('vars');
										
										if (typeof vars !== UNDEF && vars.hasOwnProperty('location')) {
											markers.push({
													title:		[((i + 1) + '.'), $(this).attr('title')].join(' '),
													pos: 		vars.location,
													link:		i
												});
										}
									});
									
									return markers;
								};
					
						// Collecting markers and initializing map
						if (settings.hasOwnProperty('map')) {
							
							$('.map-root').addMap({
									markers: 			getMarkers(),
									type:				settings.map['type'] || 'roadmap',
									zoom:				settings.map['zoom'] || 16,
									apiKey:				settings.map['apiKey'],
									fitBounds:			true,
									fullscreenControl:	false,
									onMarkerClick:		function() {
																if (this.hasOwnProperty('link')) {
																	focus(this.link);
																}
															}
								});
							
							// Showing map only if map button clicked
							$('#map-btn').on('click', function() {
									if (!$('.map-root').data('rendered')) {
										setTimeout(function(btn) {
												if (btn.hasClass('on')) {
													$('.map-root').trigger('render');
												}
											}, 20, $(this));
									}
								});
						}
						
					},
	
				/********************************************************
				 *
				 *						Lightbox
				 *
				 ********************************************************/
				 
				// Gets related thumb by number
				
				getRelatedCard = function(n) {
					
						return cards.children('[data-rel=' + n + ']');
					},
					
				// Determine direction
				
				setDir = function(next) {
						var next = (typeof next === UNDEF)? nextCard : next;
						
						if (loop) {
							if (curr === thumbs.length - 1 && next === 0) {
								return 1;
							} else if (!curr && next === thumbs.length - 1) {
								return -1;
							}
						}
						
						return (next === curr)? 0 : ((next > curr)? 1 : -1);
					},
					
				// Animating progress bar
				
				animateProgress = function(duration) {
						var d = 20 / duration,
						
							setProgress = function(to) {
								
									//progress.css('transform', 'rotate(' + to * 360 + 'deg)');
									progress.css('width', to * 100 + '%');

									if (to < 1) {
										to = Math.min(1, to + d);
										progressTo = setTimeout(setProgress, 20, to); 
									}
								};
							
						clearTimeout(progressTo);
						progressTo = setTimeout(setProgress, 20, 0);
					},
							
				//  Starts slideshow (core)
				
				_startAuto = function() {
					
						slideshowSuspended = false;
						
						if (isMediaPlaying()) {
							pauseMedia();
						}
						
						slideshow = setTimeout(function() {
							animateProgress(transition.delay + transition.speed);
							next();
						}, Math.max(transition.delay / 4, 100));
						
						if (settings.backgroundAudioSlideshowControl) {
							backgroundAudioRoot.trigger('fadeInPlayer');
						}
						
						slideshowSuspended = false;
					},
				
				// Start auto play
				
				startAuto = function() {
						//log('Start auto');
						clearTimeout(slideshow);
						settings.auto = true;
						requestWakeLock();
						hideAllPanels();
						main.removeClass(settings.pausedClass).addClass(settings.playingClass);
						
						if (settings.slideshowFullscreen) {
							requestFullscreen(function() {
									_startAuto();
									main.addClass('fullscreen');
								});
						} else {
							_startAuto();
						}
					},
										
				// Resuming auto after video ended
					
				resumeAuto = function() {
						
						slideshowSuspended = false;
						slideshow = setTimeout(function() {
								animateProgress(transition.delay + transition.speed);
								next();
							}, Math.max(transition.delay / 4, 100));
						
						if (settings.backgroundAudioSlideshowControl) {
							backgroundAudioRoot.trigger('fadeInPlayer');
						}
						
						main.removeClass(settings.pausedClass).addClass(settings.playingClass);
					},
					
				// Stop auto play
				
				stopAuto = function() {
						//log('Stop auto');
						
						clearTimeout(slideshow);
						slideshow = null;
						slideshowSuspended = false;
						
						if (settings.auto && settings.backgroundAudioSlideshowControl) {
							backgroundAudioRoot.trigger('fadeOutPlayer');
						}
						
						settings.auto = false;
						
						main.removeClass(settings.playingClass).addClass(settings.pausedClass);
						
						if (settings.slideshowFullscreen && !settings.lightboxFullscreen) {
							exitFullscreen(function() {
									lightbox.removeClass('fullscreen');
								});
						}
						
						releaseWakeLock();
					},
												
				// Suspending auto during video play
				
				suspendAuto = function() {
						
						clearTimeout(slideshow);
						
						if (settings.auto && settings.backgroundAudioSlideshowControl) {
							backgroundAudioRoot.trigger('fadeOutPlayer');
						}
						
						main.removeClass(settings.playingClass).addClass(settings.pausedClass);
						
						slideshowSuspended = !settings.auto;
					},
					
				// Is media playing?
				
				isMediaPlaying = function() {
						var m = getRelatedCard(curr).children('audio,video');
						
						return m.length? !m[0].paused : false;
					},
							
				// Media started
				
				mediaPlaying = function(e) {
					
						if (settings.auto) {
							suspendAuto();
						} else if (settings.muteBackgroundAudio) {
							if (backgroundAudioOn = !backgroundAudioRoot.data(settings.pausedClass)) {
								backgroundAudioRoot.trigger('fadeOutPlayer');
							}
						}
						
						$(this).closest('.' + settings.cardClass).addClass(settings.playingClass);
					},
					
				// Media paused
				
				mediaPaused = function(e) {
					
						if (settings.auto) {
							resumeAuto();
						} else if (settings.muteBackgroundAudio && backgroundAudioOn) {
							backgroundAudioRoot.trigger('fadeInPlayer');
						}
						
						$(this).closest('.' + settings.cardClass).removeClass(settings.playingClass);
					},
					
				// Play media
				
				playMedia = function(n) {
						var n = (typeof n === UNDEF)? curr : n,
							c = getRelatedCard(n),
							m = (c.length? c.find('video,audio')[0] : null);
						
						if (m && m.paused) {
							if (m.nodeName === 'VIDEO') {
								playVideo(m);
							} else {
								// Audio autostart might be refused 
								m.play();
							}
						}
					},
					
				// Pause media
				
				pauseMedia = function(n) {
						var n = (typeof n === UNDEF)? curr : n,
							c = getRelatedCard(n),
							m = (c.length? c.find('video,audio')[0] : null);
						
						if (m && !m.paused) {
							m.pause();
						}
					},
					
				// Toggle media
				
				toggleMedia = function() {
						var n = (typeof n === UNDEF)? curr : n,
							c = getRelatedCard(n),
							m = (c.length? c.find('video,audio')[0] : null);
						
						if (m) {
							if (m.paused) {
								var promise = m.play();
								if (promise) {
									// Catching browser refuse
									promise.catch(function(err) {
											console.log('Autoplay has prevented by the browser. ' + err);
											if (!m.muted) {
												// Retry muted
												m.muted = true;
												promise = m.play();
											}
										});
								}
							} else {
								m.pause();
							}
						}
					},
													
				// Media ended
				
				mediaEnded = function() {
					
						backgroundAudioOn = false;
						
						if (settings.auto && slideshowSuspended) {
							resumeAuto();
						}
					},
					
				// Auto starting video?
				
				autoStartMedia = function(n) {
					
						if (typeCheck('video,audio', n)) {
							if (settings.videoAuto) {
								playMedia(nextCard);
							} else {
								pauseMedia(nextCard);
							}
						}
					},
				
				recenter = function() {
						var card = getRelatedCard(curr);
						
						card.css({
								transition:		'all .2s ease-out'
							}).css({
								opacity:		1,
								transform:		'translateX(0)'
							});
					},
					
				// Refreshing layout after resize
				
				layoutRefresh = function() {
						var lw = lightbox.width() - 2 * settings.fitPadding,
							lh = lightbox.height() - 2 * settings.fitPadding;
							
						lightbox.removeClass('landscape portrait')
							.addClass((lw >= lh)? 'landscape' : 'portrait');
						
						cards.children().each(function() {
								var card = $(this),
									eq = card.data('rel');
									
								if (eq !== null && eq >= 0 && eq < thumbs.length) {
									var th = thumbs.eq(eq),
										vars;
										
									if (th.length && (vars = th.data('vars'))) {
										var cw = vars['width'],
											ch = vars['height'],
											cat = vars['type'],
											r = 1,
											ra;
										
											
										if (scaling === 'full' && vars.hasOwnProperty('imageRenditions')) {
											// Full zoom
											var ra = vars['imageRenditions'];
											if (typeof ra === STRING) {
												ra = getRenditionsArray(ra);
											}
											if (ra.length) {
												card.find('img').eq(0).attr('src', SLIDES_DIR + '/' + ra[0][0]);
												cw = ra[0][1];
												ch = ra[0][2];
											}
										} else {
												
											if (scaling === 'none' || cat === 'other') {
												// No scaling
												if (cw > lw || ch > lh) {
													// Scale down if too large
													r = Math.min(lw / cw, lh / ch);
												}
											} else if (scaling === 'fit' || cat === 'video' || cat === 'audio') {
												// Fit
												r = Math.min(settings.maxScale, Math.min(lw / cw, lh / ch));
											} else if (scaling === 'fill') {
												// Fill
												r = Math.min(settings.maxScale, Math.max(lw / cw, lh / ch));
											}
											
											cw *= r;
											ch *= r;
										}
										
										card.css({
												transition:		'all .2s linear',
												width:			Math.round(cw),
												height:			Math.round(ch),
												left:			Math.round((lw - cw) / 2) + settings.fitPadding,
												top:			Math.round((lh - ch) / 2) + settings.fitPadding
											});
									}
								}
							});
					},
				
				// Prepare cards
				
				prepareCard = function() {
						var	wait = $(),
							
							// Creating one card
							
							createCard = function(i) {
									var card,
										scard,
										thumb,
										img,
										caption,
										buttons,
										btn,
										lw = lightbox.width() - 2 * settings.fitPadding,
										lh = lightbox.height() - 2 * settings.fitPadding,
										iw,
										ih,
										src,
										cw = 0,
										ch,
										vars,
										cat,
										r,
										eq = loop? ((i + thumbs.length) % thumbs.length) : i;
									
									// Already exists?
									card = getRelatedCard(eq);
									
									// Creating empty card
									if (!card.length) {
										card = $('<div>', {
												id:				'card_' + i,
												'data-rel':		eq,
												'class':		settings.cardClass
											});
									}
									
									// Real card? (not empty)
									if (eq >= 0 && eq < thumbs.length) {
										
										// Img already there?
										img = card.children('img,video,object');
										
										if (!img.length) {
											// Related thumb
											thumb = thumbs.eq(eq);
											vars = thumb.data('vars');
											cat = vars['type'];
											
											// Type helper class											
											card.addClass(cat);
											
											// Set width
											iw = cw = vars['width'];
											ih = ch = vars['height'];
											r = 1;
											
											if (scaling === 'full' && vars.hasOwnProperty('imageRenditions')) {
												// Full zoom
												var ra = vars['imageRenditions'];
												if (typeof ra === STRING) {
													ra = getRenditionsArray(ra);
												}
												if (ra.length) {
													cw = ra[0][1];
													ch = ra[0][2];
													src = SLIDES_DIR + '/' + ra[0][0];
												}
											} else {
												
												if (scaling === 'none' || cat === 'other') {
													// No scaling
													if (cw > lw || ch > lh) {
														// Scale down if too large
														r = Math.min(lw / cw, lh / ch);
													}
												} else if (scaling === 'fit' || cat === 'video' || cat === 'audio') {
													// Fit
													r = Math.min(settings.maxScale, Math.min(lw / cw, lh / ch));
												} else if (scaling === 'fill') {
													// Fill
													r = Math.min(settings.maxScale, Math.max(lw / cw, lh / ch));
												} 
												
												cw *= r;
												ch *= r;
												src = (cat === 'image')? getImageSource(thumb, [ cw, ch ]) : vars['original'];
											}
											
											card.css({
													width:		Math.round(cw),
													height:		Math.round(ch),
													left:		Math.round((lw - cw) / 2) + settings.fitPadding,
													top:		Math.round((lh - ch) / 2) + settings.fitPadding
												});
											
											switch (cat) {
												
												case 'image':
													
													// Creating Img 
													img = $('<img>')
														.attr({
																width:		iw,
																height:		ih,
																alt:		thumb.find('img').attr('alt') || ''
															})
														.one('load', function(e) {
																$(this).addClass(settings.loadedClass);
															})
														.attr('src', src)
														.appendTo(card);
														
													break;
												
												case 'video':
													
													// Creating Video 
													img = $('<video>')
														.one('loadedmetadata', function(e) {
																$(this).addClass(settings.loadedClass);
															})
														.attr({
																controls:					true,
																controlslist:				'nodownload nofullscreen',
																disablePictureInPicture:	true, 
																width:						iw,
																height:						ih,
																type:						'video/mp4',
																src:						src
															})
														.on('play', mediaPlaying)
														.on('pause', mediaPaused)
														.on('ended', mediaEnded)
														.appendTo(card);
													
													break;
													
												default:
													
													// Other
													img = $('<img>')
														.attr({
																width:		iw,
																height:		ih
															})
														.one('load', function(e) {
																$(this).addClass(settings.loadedClass);
															})
														.attr('src', vars['thumb'])
														.appendTo(card);
														
													break;
											}
											
											// Buttons
											
											buttons = $('<div>', {
													'class': 	settings.buttonsClass
												});
											
											// Print button
											if (settings.printImage && cat === 'image' && !settings.rightClickProtect) {
												btn = $('<a>', {
														'class': 	'btn icon-printer',
														title:		text.print
													}).on('click', function(e) {
														printImage((vars['original'] && settings.showDownload)? vars['original'] : thumb.attr('href'),
																(vars['original'] || thumb.attr('href')).getFile() || '',
																vars['caption'] || ''
															);
															
														return false;
													}).appendTo(buttons);
											}
											
											// Download button
											if (cat === 'other' || 
												settings.showDownload && (cat === 'image' || settings.allowDownloadOthers)) {
											
												var href = vars['original'];
												
												if (!href && settings.allowDownloadScaled) {
													href = thumb.attr('href');
												}
												
												if (href) {
													btn = $('<a>', {
															'class': 	'btn icon-download',
															download: 	'',
															href: 		href,
															title:		text.download
														}).appendTo(buttons);
												}
											}

											// Photo data
											if (vars['photodata'] && cat !== 'video' && cat !== 'audio') {
												btn = $('<a>', {
														'class':	'btn icon-camera meta'
													}).appendTo(buttons);
													
												btn.data('tooltip', '<div class="photodata">' + vars['photodata'] + '</div>');
											}
											
											if (!buttons.is(':empty') || vars['caption']) {
												// Adding caption
												caption = $('<div>', {
														'class':	settings.captionClass + ' ' + settings.captionStyle + ' ' + settings.captionPlacement
													})
													.appendTo(card);
													
												if (vars['caption']) {
													$(vars['caption']).addClass('text').appendTo(caption);
												}
	
												// Appending buttons if any
												if (!buttons.is(':empty')) {
													caption.append(buttons);
													buttons.children().addTooltip({
															touchToggle:	true
														});
												}
											}
											
											// Right click protect
											if (settings.rightClickProtect) {
												img.on('contextmenu', function(e) {
														e.preventDefault()
														return false;
													});
											}
											
											wait = wait.add(img);
										
										} else {
											
											// Already loaded
											
											// Stop video
											if (img[0].nodeName === 'VIDEO') {
												img[0].pause();
											}
											
										}
										
										
									} else {
										
										// Empty card
										card.addClass('blank').show();
										card.find('img').addClass(settings.loadedClass);
									}
									
									
									if (i === nextCard) {
										// Current card
										scard = getRelatedCard(i + 1);
										
										if (scard.length) {
											scard.before(card);
										} else {
											scard = getRelatedCard(i - 1);
											if (scard.length) {
												scard.after(card);
											} else {
												cards.append(card);
											}
										}
										
									} else if (i < nextCard) {
										// Insert before
										scard = getRelatedCard(i + 1);
										
										if (scard.length) {
											scard.before(card);
										} else {
											cards.prepend(card);
										}
										
									} else {
										// Insert after
										scard = getRelatedCard(i - 1);
										
										if (scard.length) {
											scard.after(card);
										} else {
											cards.append(card);
										}
									}
									
								},
								
							// Add card before and after target
							
							addCard = function() {
									// Adding current card
									createCard(nextCard);
									
									// Has to wait?
									if (!wait.length) {
										//console.log('Firing cardReady immediately: ' + n);
										cardReady();
									} else if (wait.length === 1) {
										// Wait for one image
										wait.one(wait[0].nodeName === 'VIDEO'? 'loadedmetadata' : 'load', cardReady); // loadedmetadata
									} else {
										// Wait for multiple images to be loaded
										wait.waitAllImg(cardReady);
									}
								};
														
						// Adding new card
						
						//window.requestAnimationFrame(addCard);
						addCard();
						
					},
					
							
				// Cards ready
				
				cardReady = function() {
						var currCard = getRelatedCard(nextCard),
							vars = thumbs.eq(nextCard).data('vars'),
							cat = vars['type'] || 'image',
							oldCard = (curr !== nextCard)? getRelatedCard(curr) : $(),
							//prevCards = currCard.prevAll(),
							//nextCards = currCard.nextAll(),
							
							getDim = function() {
									return [ (currCard.outerWidth() || cards.width()), (currCard.outerHeight() || cards.height()) ];
								},
					
							cleanup = function() {
									cards.children().not(currCard).remove();
								};
						
						//console.log('Card ' + nextCard + '. vars = { ' + JSON.stringify(vars) + ' }');
						
						cards.removeClass('bwd fwd');
						
						if (dir) {
							cards.addClass((dir > 0)? 'fwd' : 'bwd');
						}
						
						$('.map-root').trigger('resetMarkers');
						
						// Image is ready: adding class
						currCard.addClass(id.imageReady);
						
						// Saving transition start date
						cards.data('ts', new Date());
						
						oldCard.destroyAllTooltips();
						
						// Removing old cards hanging around
						cards.children('.' + settings.cardClass).not(currCard).not(oldCard).remove();
						
						// Add swipe gesture handling
						if (thumbs.length > 1 || settings.afterLast !== 'donothing') {
							
							//oldCard.trigger('removeSwipe');
							
							currCard.swipe({
									onSwipedLeft: 	function(e) {
															next();
														},
												
									onSwipedRight: 	function() {
															previous();
														},
													
									onSwipeStart:	function() {
															stopAuto();
														},

									onClick:		function(e) {
															if (e.target.nodeName === 'BUTTON' || e.target.nodeName === 'A' || e.target.nodeName === 'AUDIO') {
																return true;
															} else if (e.target.nodeName === 'VIDEO') {
																var coords = getCoords(e);
																if (!e.target.controls || coords.y > (e.target.offsetHeight - 60)) {
																	// Toggling video controls on
																	e.target.controls = 1;
																	return false;
																}
															}
															
															if (settings.auto) {
																// Stopping slideshow
																stopAuto();
															} else if (settings.clickAction === 'nextimage') {
																// Next image
																next();
															} else if (settings.clickAction === 'togglecontrols') {
																// Toggling controls
																if (main.hasClass(settings.hideControlsClass)) {
																	showControls();
																} else {
																	hideControls();
																}
															}
															
															return false;
														},
														
									onDoubleClick:	settings.doubleClickZoom? toggleZoom : null,
														
									onFinished:		function() {
															if ($(this).data('rel') !== curr) {
																$(this).remove();
															}
														}
								});
						}
						
						if (oldCard.data('scrolling')) {
							// Image was swiped
							
							oldCard = $();
							
						}
						
						switch (settings.transitionType) {
							
							case 'crossfade':
								
								if (oldCard.length) {
									
									currCard.css({
												opacity:		0,
												transform:		'translateX(0)'
											})
										.css({
												transition:		'opacity ' + transition.adaptiveSpeed + 'ms ease-in-out'
											})
										.css({
												opacity:		1,
											});
									
									currCard.one('transitionend', transitionCompleted);
									
									oldCard.one('transitionend', function() {
												$(this).remove();
											})
										.css({
												transition:		'opacity ' + transition.adaptiveSpeed + 'ms ease-in-out'
											})
										.css({
												opacity:		0
											});
									
								} else {
									
									currCard.show().css({
											opacity:		1,
											transition:		'opacity ' + transition.adaptiveSpeed + 'ms ease-in-out'
										});
								}
								
								break;
								
							case 'zoom':
								
								var init = function() {
									
											currCard.css({
													transform:		'scale(' + ((dir > 0)? 0.9 : 1.11) + ')',
													opacity:		0
												});
											
											oldCard.one('transitionend', function() {
													$(this).remove();
												})
												.css({
													opacity:		0,
													transform:		'scale(' + ((dir > 0)? 1.11 : 0.9) + ')',
													transition:		'transform ' + transition.adaptiveSpeed + 'ms ease-out, opacity ' + transition.adaptiveSpeed + 'ms ease-in-out'
												});
												
											window.requestAnimationFrame(start);
										},
										
									start = function() {
										
											currCard.css({
													transition:		'transform ' + transition.adaptiveSpeed + 'ms ease-out, opacity ' + transition.adaptiveSpeed + 'ms ease-in-out'
												})
												.css({
													transform:		'scale(1)',
													opacity:		1
												});
												
											currCard.one('transitionend', transitionCompleted);
										};
								
								window.requestAnimationFrame(init);
									
								break;
								
							case 'stack':
								
								var dim = getDim(),
									a = random(60),
									r = (a + 90) / 180 * Math.PI,
									d = Math.min(dim[0], dim[1]) / 4;
								
									init = function() {
										
											currCard.css({
													transition:		'all ' + transition.adaptiveSpeed + 'ms ease-out, opacity ' + (transition.adaptiveSpeed / 2) + 'ms linear',
													opacity:		0,
													transform:		'scale(0.92)'
												});
											
											oldCard.css({
													transition:		'all ' + (transition.adaptiveSpeed / 2) + 'ms ease-in, opacity ' + (transition.adaptiveSpeed / 2) + 'ms ease-in',
													opacity:		1
												});
											
											window.requestAnimationFrame(start);
										},
										
									start = function() {
										
											currCard.css({
													transform:		'scale(1)',
													opacity:		1
												});
										
											currCard.one('transitionend', transitionCompleted);
											
											oldCard.one('transitionend', function() {
														$(this).remove();
													})
												.css({
														opacity:		0,
														transform:		'rotate(' + (0 - a / 10) + 'deg) translate(' + Math.round(d * Math.cos(r)) + 'px,' + dir * Math.round(0 - d * Math.sin(r)) + 'px)'
													});
										};
	
								window.requestAnimationFrame(init);	
								
								break;
								
							case 'kenburns':
								
								var dim = getDim(),
									img = currCard.children('img,video,audio,object').eq(0),
									caption = currCard.children('.caption'),
									sc,
									tsc,
									a = random(1) * Math.PI,
									d = (Math.random() + 0.2) * Math.min(dim[0], dim[1]) * 0.08,
									dx = Math.round(d * Math.cos(a)),
									dy = Math.round(d * Math.sin(a)),
									lw = lightbox.width(),
									lh = lightbox.height(),
									td,
									
									init = function() {
											currCard.css({
														transition:		'none',
														transform: 		'scale(' + sc + ')  translate(' + dx + 'px,' + dy + 'px)',
														opacity:		0
													});
											
											// Adjusting caption position to avoid honging over
											if (captionPlacement[1] === 'top' || captionPlacement[1] === 'bottom') {
												if ((td = (dim[1] - lh / tsc) / 2) > 0) {
													caption.css(captionPlacement[1], td);
												}
											}
											
											if (captionPlacement[0] === 'left' || captionPlacement[0] === 'right') {
												if ((td = (dim[0] - lw / tsc) / 2) > 0) {
													caption.css(captionPlacement[0], td);
												}
											}
												
											window.requestAnimationFrame(start);
										},
								
									start = function() {
										
											currCard.show()
												.css({
														transition:		'all ' + transition.adaptiveSpeed + 'ms cubic-bezier(0.39, 0.575, 0.565, 1)',
														transform:		'scale(' + tsc + ') translate(0,0)',
														opacity: 		1
													});
											
												currCard.one('transitionend', transitionCompleted);
												
											if (oldCard.length) {
																									
												oldCard.css({
															transition:		'all ' + transition.adaptiveSpeed + 'ms cubic-bezier(0.47, 0, 0.745, 0.715)',
															transform: 		'scale(' + oldCard.data('sc') + ') translate(' + oldCard.data('dx') + 'px,' + oldCard.data('dy') + 'px)',
															opacity: 		0
														})
													.one('transitionend', function() {
															$(this).remove();
														});
											}
										};
								
								// Setting before and after transition scaling
								if (img[0].nodeName === 'IMG') {
									sc = 1.14 + random(0.2);
									tsc = (sc > 1.14)? (sc - 0.1) : (sc + 0.1);
								} else {
									sc = 1.1;
									tsc = 1.0;
								}
									
								// Saving parameters for continuous transition in the next change
								currCard.data({
										'sc': 			(sc > 1.14)? (sc - 0.2) : (sc + 0.2),
										'dx':			-dx,
										'dy':			-dy
									});
								
								//console.log(curr + '. ' + [oldCard.data('sc'),oldCard.data('dx'),oldCard.data('dy')].join(':'));
								//console.log(nextCard + '. ' + [currCard.data('sc'),currCard.data('dx'),currCard.data('dy')].join(':'));
								window.requestAnimationFrame(init);
											
								break;
						}		
						
						// Next image
						if (settings.auto && !slideshowSuspended) {
							clearTimeout(slideshow);
							animateProgress(transition.delay + transition.speed);
							slideshow = setTimeout(next, transition.delay + transition.speed);
						} else {
							stopAuto(true);
							if (settings.hideControls && main.hasClass(settings.hideControlsClass)) {
								hideControlsLater();
							}
						}
						
						preloadNext();
						
						curr = nextCard;
						
						cards.children('.' + settings.currClass).removeClass(settings.currClass);
						currCard.addClass(settings.currClass);
						updateShares();
					},
				
				// Reading the string renditions array
				
				getRenditionsArray = function(s) {
						var a = [];
						
						if (s[0] === '[') {
							s = s.slice(1, -1);
						}
						
						s.split(/,\s?/).forEach(function(r) {
								r = r.match(/(.+) \((\d+)x(\d+)\)/);
								if (r && r.length > 3) {
									a.push([ r[1], r[2], r[3] ]);
								}
							});
						
						return a;
					},
				
				// Gets the closest rendition for a given window
				
				getClosestRendition = function(ra, dim) {
						var dim = (typeof dim === UNDEF)? [ $body.width(), $body.height() ] : dim,
							w = dim[0] * PIXELRATIO,
							h = dim[1] * PIXELRATIO;
						
						if (ra && ra.length > 1) {
							 
							if (!WEBP_LOSSY) {
								// No webp support: filtering out
								ra = ra.filter(function(r) {
										return !r[0].endsWith('.webp');
									});
							}
							
							ra.forEach(function(r) {
									var s = Math.min(w / r[1], h / r[2]);
							
									r[3] = (s > 1)? 
											// Upscaling penalized by factor 3: 1x -> 120% == 2x -> 50%
											(3 * (1 - 1 / s))
											:
											// Downscaling
											(1 - s)
								});
						
							// Sort by scale
							ra.sort(function(r1, r2) { 
									return r1[3] - r2[3];
								});
						}
						
						return ra[0][0];
					},
					
				// Get optimal source
				
				getImageSource = function(thumb, dim) {
						var vars = thumb.data('vars');
						
						if (vars) {
							
							if (vars.hasOwnProperty('imageRenditions')) {
								return settings.slidesDir + '/' + getClosestRendition(vars['imageRenditions'], (typeof dim === UNDEF)? [ $body.width(), $body.height() ] : dim);
							}
						}	
						
						return thumb.attr('href');
					},
				
				// Get optimal source
				
				getThumbSource = function(thumb, dim) {
						var vars = thumb.data('vars');
						
						if (vars) {
							
							if (vars.hasOwnProperty('thumbRenditions')) {
								return settings.thumbsDir + '/' + getClosestRendition(vars['thumbRenditions'], (typeof dim === UNDEF)? settings.thumbDims : dim);
							}
							
							return settings.thumbsDir + '/' + vars['thumb']
						}	
						
						return thumb.attr('src');
					},
				
				// Function executed on transition complete
				
				transitionCompleted = function(e) {
						var n = (typeof e !== UNDEF && e.hasOwnProperty('target'))? $(e.target).data('rel') : curr;
						
						autoStartMedia(n);
						
						$('.map-root').trigger('setActive', n);
						
					},
					
				// Preloading the next image (direction sensitive)
				
				preloadNext = function() {
						var nextThumb,
							src,
							vars;
						
						if (dir < 0 && curr > 0) {
							nextThumb = thumbs.eq(curr - 1);
						} else if (curr < thumbs.length - 2) {
							nextThumb = thumbs.eq(curr + 1);
						}
						
						if (nextThumb) {
							if (src = getImageSource(nextThumb)) {
								var next = new Image();
								next.src = src;
							}
						}
					},
			
				// Focus image
				
				focus = function(n) {
						//console.log('focus(' + n + ')');
						var dt = new Date();
						
						main.hideAllTooltips();
						
						if (n < 0 || n >= thumbs.length || ((dt - lastChangeDate) < 200)) {
							return;
						}
						
						nextCard = n;
						dir = setDir();
						
						transition.adaptiveSpeed = Math.minMax(transition.speed / 5, dt - lastChangeDate, transition.speed);
						lastChangeDate = dt;
						
						pauseMedia(curr);
						
						if (curr !== nextCard || cards.is(':empty')) {
							prepareCard();
						} else {
							// Curr card is already in focus
							var cc = getRelatedCard(curr);
							// Reset position after swipe
							cc.css({
									transition:		'transform 300ms ease-out, opacity 300ms linear',
									transform:		'translateX(0)',
									opacity:		1
								});
							//cardReady();
						}
						
						focusThumb(nextCard);
						
						updateUrl(nextCard);
						
						if (settings.showNumbers) {
							updateNumber(nextCard + 1);
						}
											
					},
												
				// Previous card
				
				previous = function() {
						
						hideAllPanels();
						if (settings.autoHideControls) {
							hideControlsLater();
						}
						
						if (curr) {
							// Not first
							focus(curr - 1);
							return;
							
						} else {
							
							stopAuto();
							
							// First
							switch (settings.afterLast) {
								
								case 'donothing':
									
									stopAuto();
									recenter();
									break;
									
								case 'nextfolder':
									
									if (settings.previousFoldersLast) {
										window.location.href = settings.previousFoldersLast;
									} else {
										focus(0);
									}
									break;
									
								case 'startover':
									
									focus(thumbs.length - 1);
									return;
									
							}
							
						}
						
					},
					
				// Next card
				
				next = function() {
						
						hideAllPanels();
						
						if (settings.autoHideControls) {
							hideControlsLater();
						}
						
						if (curr < (thumbs.length - 1)) {
							// Not at end
							focus(curr + 1);
							return;
							
						} else {
							// At the end
							switch (settings.afterLast) {
								
								case 'donothing':
									
									stopAuto();
									recenter();
									break;
									
								case 'startover':
									
									focus(0);
									return;
									
								case 'onelevelup':
									
									stopAuto();
									
									if (settings.level) {
										window.location.href = '../' + settings.indexName;
									}
									break;
									
								case 'nextfolder':
									
									var ss = settings.auto;
									
									stopAuto();
									
									if (settings.nextFoldersFirst) {
										window.location.href = settings.nextFoldersFirst + (ss? '&slideshow' : '');
									} else {
										focus(thumbs.length - 1);
									}
									break;
									
								case 'ask':
									
									var ss = settings.auto;
									
									stopAuto();
									
									var buttons = new Array();
									
									if (thumbs.length > 1) {
										// Start over
										buttons.push({
												t: 	text.startOver,
												c:	'icon-loop',
												h: 	function() {
														focus(0);
														settings.auto = ss;
														
														if (ss) {
															slideshowSuspended = false;
															requestWakeLock();
															main.removeClass(settings.pausedClass).addClass(settings.playingClass);
														
															if (settings.backgroundAudioSlideshowControl) {
																backgroundAudioRoot.trigger('fadeInPlayer');
															}
														}
													}
											});
									}
									
									if (settings.level) {
										// Up one level
										buttons.push({
												t: 	text.upOneLevel, 
												c:	'icon-one-level-up',	
												h: 	function() { 
														window.location.href = '../' + settings.indexName;
													}
											});
									}
																		
									if (settings.nextFoldersFirst) {
										// Go to next folder
										buttons.push({
												t: 	text.nextFolder,
												c:	'icon-arrow-right',
												h: 	function() {
														window.location.href = settings.nextFoldersFirst + (ss? '&slideshow' : '');
													}
											});
									}
							
									main.modal($('<p>', {
											'class':	'text-center',
											text: 		text.atLastPageQuestion
										}), buttons, {
											onClose:	function() {
															focus(thumbs.length - 1);
														},
											'class': 	'secondary',
											title: 		text.atLastPage
										});
									
									return;
									
							}
							
						}
						
					},
										
				// Initializing lightbox
				
				initLightbox = function() {
					
						// Extending settings with data-settings="" attribute
						
						if (lightbox.data('settings')) {
							$.extend(setting, lightbox.data('settings'));
						}
						
						// Finding / Creating cards container
						
						cards = lightbox.find('.' + settings.cardsClass);
						
						if (!cards.length) {
							cards = $('<div>', {
									'class':	settings.cardsClass
								}).appendTo(lightbox);
						}
						
						captionPlacement = settings.captionPlacement.split(' ');
						
						// Collecting transition properties
						
						transition = settings.defaults[settings.transitionType || 'kenburns'];
						
						if (settings.hasOwnProperty('transitionSpeed')) {
							transition.speed = settings.transitionSpeed;
						}
						
						if (settings.hasOwnProperty('slideshowDelay')) {
							transition.delay = settings.slideshowDelay;
						}
						
						if (settings.hasOwnProperty('fillHorz')) {
							transition.fillHorz = settings.fillHorz / 100;
						}
						
						if (settings.hasOwnProperty('fillVert')) {
							transition.fillVert = settings.fillVert / 100;
						}
						
						// Looping conveyor
						
						loop = settings.afterLast === 'startover';
						
						// Reducing before/after count if too few items
						
						if (thumbs.length === 1) {
							transition.after = transition.before = 0;
						} else {
							transition.after = Math.min(transition.after, Math.ceil(thumbs.length / 2));
							transition.before = Math.min(transition.before, Math.floor(thumbs.length / 2));
						}
						
						// Mouse wheel
						
						if (settings.enableMouseWheel) {
							var wheelTimeout = null;
							lightbox.on('mousewheel.' + ns + ' DOMMouseScroll.' + ns, function(e) {
									wheelTimeout = clearTimeout(wheelTimeout);
									wheelTimeout = setTimeout((e.originalEvent.wheelDelta > 0 || e.originalEvent.detail < 0)? previous : next, 100); 
									return false;
								});
						}
						
						// Buttons
						
						if (thumbs.length > 1) {
							// Previous button
							prevBtn	= lbcontrols.find('.previous');
							prevBtn.on('click.' + ns, previous);
							
							// Play button
							playBtn = lbcontrols.find('.play');
							playBtn.on('click.' + ns, startAuto);
							
							// Pause button
							pauseBtn = lbcontrols.find('.pause');
							pauseBtn.on('click.' + ns, stopAuto);
							
							// Next button
							nextBtn = lbcontrols.find('.next');
							nextBtn.on('click.' + ns, next);
						}
						
						// On mouse-driven devices
						
						if (whatInput.ask('intent') === 'mouse') {
							
							// Allow tooltips
							$('#lbcontrols [data-tooltip]').addTooltip({
									delay:		1000,
									stay:		2000,
									pos:		[1, 2, 1, 0]
								});
							
						}
						
						// Keyboard
						
						if (settings.enableKeyboard) {
							
							$document.on('keyup.' + ns, function(e) {
								if (!main.length || 
										lightbox.children('.modal:visible').length || 
										document.activeElement && (document.activeElement.nodeName === 'INPUT' || document.activeElement.nodeName === 'TEXTAREA')
									) {
									return true;
								}
								
								e.preventDefault();
								
								switch (e.keyCode) {
									
									case 10:
									case 13:
										hidePanel(thumbnails);
										break;
										
									case 33:
									case 37:
										previous();
										break;
										
									case 34:
									case 39:
										next();
										break;
										
									case 97:
									case 35:
										focus(thumbs.length - 1);
										break;
										
									case 103:
									case 36:
										focus(0);
										break;
									
									case 106:
									case 179:
										if (slideshow) {
											stopAuto();
										} else {
											startAuto();
										}
										break;
										
									case 32:
										if (typeCheck('video,audio')) {
											toggleMedia();
										} else {
											if (slideshow) {
												stopAuto();
											} else {
												startAuto();
											}
										}
										break;
										
									default:
										return true;
														
								}
								
								return false;
							});
						}
						
						// Handling clicks on lightbox
						
						var lastClick = 0,
						
							lightboxClick = function(e) {
									var d = new Date();
									
									if ((d - lastClick) > 100) {
										
										// Handle only if separate click 
										lastClick = d;
										
										if (thumbnails.is(':visible')) {
											// Hide thumbnails panel
											hidePanel(thumbnails);
											return false;
											
										} else if (e.target.nodeName !== 'BUTTON' && e.target.nodeName !== 'A') {
											var el = $(e.target).closest('.' + settings.cardClass);
											
											if (el.length && el.data('rel') !== curr) {
												// Focus previous / next card
												focus(el.data('rel'));
												return false;
											} else {
												// Handle video start / stop
												if (typeCheck('video,audio')) {
													toggleMedia();
													return false;
												}
											}
										}
									}
									
									return true;
								};
						
						
						// Lightbox resize event listener 
						
						$window.on('resize.' + ns, function() {
								clearTimeout(layoutTo);
								layoutTo = setTimeout(layoutRefresh, 100);
							});
						
					},
				
				/********************************************************
				 *
				 *						Initialization
				 *
				 ********************************************************/
				 
				init = function() {
					
						// Removing search string
						
						if (window.location.search) {
							removeSearch();
						}
						
						// Background music
						
						initBackgroundMusic(backgroundAudioRoot);
						
						// Preparing share buttons
						
						if (settings.hasOwnProperty('share')) {
							initShares(settings.share);
						}
						
						// Hash change listener
						
						$window.on('hashchange.' + ns, stateChange);
						
						if (window.location.hash) {
							var m = window.location.hash.match(/img=([^&]+)/);
							if (m) {
								curr = getImageByName(m[1]);
								if (curr !== -1) {
									initByHash = true;
									settings.auto = !!window.location.hash.match(/\&slideshow$/);
								} else {
									curr = 0;
								}
							}
						}
						
						// Controls
						
						initControls();
						
						// Thumbnails
						
						if (thumbnails.length) {
							initThumbs();
						}
						
						// Initializing map
						
						if (settings.hasOwnProperty('map')) {
							initMap();
						}
					
						// Lightbox
						
						if (lightbox.length) {
							
							initLightbox();
							focus(curr);
							
						}
			
						// Tooltips
						if (whatInput.ask('intent') === 'mouse') {
							$('#controls [data-tooltip]').addTooltip({
									delay:		1000,
									stay:		2000,
									pos:		[0, 1, 2, 1]
								});
						}
						
						// Autoplay
						
						if (settings.auto && lightbox.length && thumbs.length > 1) {
							startAuto();
						}
					};
					
			/********************************************************
			 *
			 *					Execution starts here
			 *
			 ********************************************************/
				
			init();
			
			return this;
		};
		
	/********************************************************
	 *
	 *						Defaults
	 *
	 ********************************************************/
				
	$.fn.skin.defaults = {
			activeClass: 					'active',
			lazyloadClass:					'lazyload',
			thumbsDir:						'thumbs',
			slidesDir:						'slides',
			indexName:						'index.html',
			cardsClass:						'cards',
			cardClass:						'card',
			currClass:						'curr',
			prevClass:						'prev',
			nextClass:						'next',
			loadedClass:					'loaded',
			captionClass:					'caption',
			buttonsClass:					'buttons',
			hideImageClass:					'hide-image',
			showImageClass:					'show-image',
			fullscreenClass:				'fullscreen',
			hideControlsClass:				'hide-controls',
			playingClass:					'playing',
			pausedClass:					'paused',
			
			transTime:						500, 				// Menu transition length
			clickForNext:					true,
			backgroundAudioSlideshowControl:false,
			muteBackgroundAudio:			true,
			videoAuto:						false,
			videoLoop:						false,
			volume:							0.5,
			auto:							false,
			afterLast:						'donothing',
			clickAction:					'donothing',
			doubleClickZoom:				true,
			
			rightClickProtect: 				false,
			allowDownloadOthers:			false,
			
			autoHideControls:				false,
			hideControlsDelay:				0,
			
			showNumbers:					true,
			
			transitionType:					'kenburns',
			captionStyle:					'dark',
			captionPlacement:				'center bottom',
			
			showFullscreen:					true,
			enableKeyboard:					true,
			enableMouseWheel:				true,
			
			scaling:						'fit',
			maxScale:						1.4,
			
			fitPadding:						0,
			
			defaults: 						{
												'kenburns':				{
																			before:		1,
																			after:		0,
																			delay:		0,
																			speed:		4000
																		},
												'crossfade':			{
																			before:		1,
																			after:		0,
																			delay:		1000,
																			speed:		2000
																		},
												'zoom':					{
																			before:		1,
																			after:		0,
																			delay:		1000,
																			speed:		2000
																		},
												'stack':				{
																			before:		1,
																			after:		0,
																			delay:		1000,
																			speed:		2000
																		}
											},
											
			fit:							'both'								// 'cover', 'fit'
			
		};
			
})(jQuery, jQuery(window), jQuery(document), jQuery('body'));
