/*	
 *	thumbScroll() :: adds horizontal scrolling to layer
 *
 *	Copyright by Lazaworx
 *	http://www.lazaworx.com
 *	Author: Laszlo Molnar
 *
 *	Dual licensed under the MIT and GPL licenses.
 *	- http://www.opensource.org/licenses/mit-license.php
 *	- http://www.gnu.org/copyleft/gpl.html
 *
 *	Usage: $(element).thumbScroll( options );
 *	options:
		speed: 				1000,
		extraSpeed:			600,
		headRoom: 			0.67,
		enableMouseWheel: 	true
 */

(function($, $document, $window) {
	'use strict';
	
	$.fn.thumbScroll = function(settings) {
		
		settings = $.extend({}, $.fn.thumbScroll.defaults, settings);
				
		return this.each(function() {
			var self 		= $(this),
				ns 			= '_lts_' + Math.floor(Math.random() * 10000),
				options 	= $.extend({}, settings, JSON.parse(self.data('thumbscroll') || '{}')),
				backBtn,
				forwardBtn,
				clip 		= self.find('.' + options.clipClass),
				thumbs		= clip.find('.' + options.thumbContClass),
				
				vertical	= self.closest('.cont').hasClass('thumbs-left') || self.closest('.cont').hasClass('thumbs-right'),
				
				p0, 
				tP, 
				tT, 
				tP1, 
				dist, 
				min, 
				seek,
				
				resizeTimeout,
				
				eventsWired	= false,
										
				noAction = function(e) {
						e.preventDefault();
						return false;
					},
				
				// Disabling the left and right buttons if needed 
				setButtons = function(p) {
						p = (typeof p !== UNDEF)? p : (vertical? thumbs.position().top : thumbs.position().left);
						backBtn.toggleClass('disabled', p > -1);
						forwardBtn.toggleClass('disabled',  
							p < (vertical? (clip.height() - thumbs.height() + 1) : (clip.width() - thumbs.width() + 1)));
					},
				
				// Stops CSS transition at the current stage
				stopAtCurrentPos = function() {
						thumbs.css(vertical?
							{
								top:			thumbs.position().top,
								transition:		'top 0ms linear'
							}
							:
							{
								left:			thumbs.position().left,
								transition:		'left 0ms linear'
							}
						);
					},
						
				// Moving to "p" position
				scrollTo = function(p) {
						var cs, 
							ts;
							
						if (vertical) {
							cs = clip.height(); 
							ts = thumbs.height();
						} else {
							cs = clip.width(); 
							ts = thumbs.width();
						}
						
						if (!cs || !ts || cs >= ts || typeof p !== 'number') {
							return;
						} 
						
						p = Math.minMax(cs - ts, p, 0);
						
						self.addClass(options.scrollingClass);
						
						thumbs.css(vertical?
							{
								top:			p,
								transition:		'top ' + options.speed + 'ms ease-out'
							}
							:
							{
								left:			p,
								transition:		'left ' + options.speed + 'ms ease-out'
							}).one('transitionend', function() {
								self.removeClass(options.scrollingClass);
								self.trigger('lazyloadThumbs', p);
							});
						
						setButtons(p);
					},	
			
				// Scrolling back
				scrollBack = function(e) {
						seekOn();
						scrollTo(vertical? 
							(thumbs.position().left + clip.width()) 
							:
							(thumbs.position().top + clip.height())
						); 
						return false; 
					},
				
				// Scrolling forward
				scrollForward = function(e) {
						seekOn();
						scrollTo(vertical?
							(thumbs.position().top - clip.height())
							:
							(thumbs.position().left - clip.width())
						); 
						return false; 
					},
				
				// User performs a seek - don't move active into view
				seekOn = function(state) {
						if (typeof seek === 'number') {
							clearTimeout(seek);
						}
						if (typeof state !== 'undefined') {
							seek = state;
						} else {
							seek = setTimeout(function() {
								seek = false;
							}, options.seekStay);
						}
					},
			
				// Scrolling to the active thumb
				scrollToActive = function(e, t) {
						if (self.hasClass(options.scrollingClass) || seek) {
							return;
						}
						
						var e = t? $(t) : thumbs.find(options.activeClass);
						
						if (e.length) {
							var ep,
								ed,
								hr,
								tp,
								td,
								cd;
								
							if (vertical) {
								ep = e.position().top; 
								ed = e.outerHeight(true);
								hr = Math.round(ed * options.headRoom);
								tp = thumbs.position().top;
								td = thumbs.height();
								cd = clip.height();
							} else {
								ep = e.position().left; 
								ed = e.outerWidth(true);
								hr = Math.round(ed * options.headRoom);
								tp = thumbs.position().left;
								td = thumbs.width();
								cd = clip.width();
							}
							
							if (cd >= td) {
								// Smaller
								return;
							} else if (ep > (cd - ed - hr - tp)) {
								tp = Math.max(cd - ed - hr - ep, cd - td);
							} else if (ep < -tp + hr) {
								tp = -ep + hr;
							} else { 
								return;
							}
							
							scrollTo(tp);
						}
					},
			
				// Mousewheel action
				mousewheel = function(e, d) {
					
						e.preventDefault();
						
						if (d) {
							stopAtCurrentPos();
							seekOn();
							if (vertical) {
								scrollTo(thumbs.position().top + clip.height() * ((d < 0)? -1 : 1));
							} else {
								scrollTo(thumbs.position().left + clip.width() * ((d < 0)? -1 : 1));
							}
						}
						
						return false;
					},
					
				// Extra movement after drag
				dragExtra = function(d) {
						
						d = Math.minMax(min, tP1 + d, 0);
						
						setButtons(d);
						
						thumbs.css(vertical?
							{
								top: 			d,
								transition:		'top ' + options.extraSpeed + 'ms ease-out'
							}
							:
							{
								left: 			d,
								transition:		'left ' + options.extraSpeed + 'ms ease-out'
							}).one('transitionend', function() {
								self.removeClass(options.scrollingClass);
							});
						
						self.trigger('lazyloadThumbs', d);
					},
			
				// Moving thumbs
				dragMove = function(e) {
					
						if (tP) {
							var dP = getCoords(e.originalEvent)[vertical? 'y' : 'x'] - tP;
							if (dP) {
								thumbs.css(vertical? 'top' : 'left', Math.minMax(min, p0 + dP, 0));
							}
						} else {
							// No dragtart?
							tP = getCoords(e.originalEvent)[vertical? 'y' : 'x'];
							self.addClass(options.scrollingClass);
						}
						
						return false;
					},
				
				// Stop moving
				dragStop = function(e) {
						e.preventDefault();
						
						tP1 = thumbs.position()[vertical? 'top' : 'left'];
						
						var dP = getCoords(e.originalEvent)[vertical? 'y' : 'x'] - tP,
							dT = new Date().getTime() - tT;
						
						if (e.type === 'mouseup') {
							$document.off('mousemove.' + ns + ' mouseup.' + ns);
						} else {
							thumbs.off(TOUCH.MOVE + '.' + ns + ' ' + TOUCH.END + '.' + ns);
						}
						
						seekOn();
						
						if (Math.abs(dP) < 10) {
							
							// No dragging: trigger click
							self.removeClass(options.scrollingClass);
							$(e.target).off('click.' + ns);
							$(e.target).closest('a').trigger('click', e);
							
						} else {
							
							dragExtra(1000 * dP / dT);
							
							// Delay 30ms after a dragging
							setTimeout(function(){
									self.removeClass(options.scrollingClass);
									$(e.target).off('click.' + ns);
								}, 30);
						}
						
						return false;
					},
			
				// Start moving the thumbs 
				dragStart = function(e) {
						
						if (e.type === 'mousedown' && e.which !== 1) {
							return true;
						}
						
						e.preventDefault();
						
						if (e.type === 'touchstart' && 
							(!e.originalEvent.touches || e.originalEvent.touches.length > 1 || 
								self.hasClass(options.scrollingClass))) {
							return true;
						}
						
						self.addClass(options.scrollingClass);
						
						stopAtCurrentPos();
						
						p0 = thumbs.position()[vertical? 'top' : 'left'];
						tP = getCoords(e.originalEvent)[vertical? 'y' : 'x'];
						tT = new Date().getTime();
						dist = 0;
						min = vertical? (clip.height() - thumbs.height()) : (clip.width() - thumbs.width());
						
						if (min >= 0) {
							return true;
						}
						
						if (e.type === 'mousedown') {
							$document
								.on('mousemove.' + ns, dragMove)
								.on('mouseup.' + ns, dragStop);
							$(e.target).on('click.' + ns, noAction);
						} else {
							$(e.target).closest('a').focus();
							thumbs
								.on(TOUCH.MOVE + '.' + ns, dragMove)
								.on(TOUCH.END + '.' + ns, dragStop);
						}
						
						return false;
					},
											
				// Initialization
				init = function() {
					
						if (clip.length && thumbs.length) {
							
							// Left button
							backBtn = self.find('.' + options.backClass);
							if (!backBtn.length) {
								backBtn = $('<button>', { 
										'class': 	options.backClass + ' icon-arrow-' + (vertical? 'up' : 'left') + ' ' + options.disabledClass 
									}).appendTo(self);
							}
							
							// Right button
							forwardBtn = self.find('.' + options.forwardClass);
							if (!forwardBtn.length) {
								forwardBtn = $('<button>', { 
										'class': 	options.forwardClass + ' icon-arrow-' + (vertical? 'down' : 'right') + ' ' + options.disabledClass
									}).appendTo(self);
							}
							
							$window.on('resize.' + ns, function() {
									clearTimeout(resizeTimeout);
									resizeTimeout = setTimeout(refresh, 100);
								});
							
							self.on('scrollToActive', scrollToActive);
							self.trigger('lazyloadThumbs', 0);
							
							refresh();
						}
					},
				
				// Refreshing
				refresh = function() {
						
						self.addClass(options.fillClass);
						
						var tp,
							td,
							cd;
							
						if (vertical) {
							tp = thumbs.position().top;
							td = thumbs.height(),
							cd = clip.height();
						} else {
							tp = thumbs.position().left;
							td = thumbs.width(),
							cd = clip.width();
						}
						
						if (td > cd) {
							
							// Wider than window
							
							if (!eventsWired) {
								
								// Reattach events
								
								// Left/Right buttons
								backBtn.on('click.' + ns, scrollBack);
								forwardBtn.on('click.' + ns, scrollForward);
								
								// Drag start event
								thumbs.on(TOUCH.START + '.' + ns + ' mousedown.' + ns, dragStart);
								thumbs.add(backBtn).add(forwardBtn).on('selectstart.' + ns, noAction);
								
								// Mouse wheel
								if (options.enableMouseWheel) {
									clip.on('mousewheel.' + ns + ' DOMMouseScroll.' + ns, mousewheel);
								}
								
								eventsWired = true;
							}
							
							if ((cd - td) > tp) {
								
								// Adjust position so no empty space left on the right / bottom
								
								stopAtCurrentPos();
								
								thumbs.css(vertical?
									{
										top:			cd - td,
										transition:		'top 0ms linear'
									}
									:
									{
										left:			cd - td,
										transition:		'left 0ms linear'
									});
							}
							
							setButtons();
							
						} else {
							
							// Narrower: remove events
							
							self.removeClass(options.fillClass);
							
							backBtn.add(forwardBtn).add(thumbs).add(clip).off('.' + ns);
							eventsWired = false;
							
							scrollTo(0);
						}
						
					};
			
			init();									
		});
	};
	
	$.fn.thumbScroll.defaults = {
			activeClass: 			'active',
			disabledClass:			'disabled',
			backClass: 				'scroll-back',
			forwardClass:			'scroll-forward',
			clipClass:				'clip',
			thumbContClass:			'thumb-cont',
			scrollingClass:			'scrolling',
			fillClass:				'fill',
			thumbsPosition:			'top',
			speed:					1000,
			extraSpeed:				600,
			seekStay: 				3000,
			headRoom: 				1.75,
			enableMouseWheel: 		true
		};

})(jQuery, $(document), $(window));
