/*	
 *	swipe() :: Swipe action using CSS transforms with jQuery position fallback
 *
 *	Copyright by Lazaworx
 *	http://www.lazaworx.com
 *	Author: Laszlo Molnar
 *
 *	Dual licensed under the MIT and GPL licenses.
 *	- http://www.opensource.org/licenses/mit-license.php
 *	- http://www.gnu.org/copyleft/gpl.html
 *
 *	Usage: $(element).swipe( options );
 */
 
;(function($, $window, $document, undefined) {
	'use strict';
	
	$.fn.swipe = function(options) {
		
		options = $.extend({
						onSwipedLeft:			false,			// Swiped left event
						onSwipedRight:			false,			// Swiped right event
						onSmallSwipe:			false,			// Small swipe (< threshold px)
						onSwipeStart:			false,			// Start swiping event
						onClick:				false,			// Clicked
						onDoubleClick:			false,			// Doble-clicked
						onFinished:				false,			// Swiped out - can remove
						threshold:				20,				// Small swipe threshold
						//clickThreshold:			150,			
						doubleClickThreshold:	240,			// Timespan to tell apart single and double click
						noSwipeInside:			'.caption',		// Avoid swipe inside this container
						swipeoutSpeed:			300,			// Swipe out speed in ms
						overThreshold:			0.25			// Oversized pictures must moved beyond (this * screen width) to trigger swipe 
					}, options);
		
		// Getting coordinates of a touch or mouse event
		var	getDim = function(el) {
					return [ el.outerWidth(), el.outerHeight() ];
				},
				
			getDistance = function(diff) {
					return Array.isArray(diff)?
						Math.sqrt(Math.pow(diff[0], 2) + Math.pow(diff[1], 2))
						:
						Math.abs(diff);
				},
				
			// Getting coordinates of a touch or mouse event	
			getXY = function(e) {
					if (e.touches && e.touches.length === 1) {
						return [ 
							Math.round(e.touches[0].pageX),
							Math.round(e.touches[0].pageY)
						];
					} else if (e.clientX !== null) {
						return [
							Math.round(e.pageX),
							Math.round(e.pageY)
						];
					}
					return null;
				},
			
			getX = function(e) {
					if (e.touches && e.touches.length === 1) {
						return Math.round(e.touches[0].pageX);
					} else if (e.pageX !== null) {
						return Math.round(e.pageX);
					}
					return null;
				};
				
		return this.each(function() {
				
				var self = $(this),							// target element
					clip = self.parent(),					// clipping window
					ns = self.data('lsw_ns') ||				// namespace 
						('lsw_' + Math.floor(Math.random() * 10000)),
					selfDim,								// item dimensions
					clipDim,								// clip dimensions
					rto,									// window resize update timeout
					startTime,								// start time
					endTime,								// end time
					startPos,								// start position
					diff,									// move difference
					pos,									// current position
					started = false,						// ongoing swipe
					ended = true,							// ended swiping
					constrainX = true,						// only horizontal move
					nonSwipeable = true,					// target element swipeable?
					clickedOnce = false,					// Check for double click
					chkDoubleClick = null,					// Checking double-click (timeout)
					timer = null,
					media = self.hasClass('audio') || self.hasClass('video'), 
					
					// Handling swipe start event
					handleSwipeStart = function(e) {
							if (typeof options.onSwipeStart === FUNCTION) {
								options.onSwipeStart.call(self[0], e);
							}
						},
						
					// Handling swipeout movement finished event
					handleFinished = function(e) {
							//log('handleFinished');
							if (typeof options.onFinished === FUNCTION) {
								options.onFinished.call(self[0], e);
							}
						},
						
					// Handling small swipe
					handleSmallSwipe = function(e) {
							//log('handleSmallSwipe');
							unregisterEvents(e);
							self.data('scrolling', false);	
							if (typeof options.onSmallSwipe === FUNCTION) {
								options.onSmallSwipe.call(self[0], e);
							}
						},
						
					// Handling click event
					handleClick = function(e) {
							//log('handleClick');
							unregisterEvents(e);
							self.data('scrolling', false);	
							if (typeof options.onClick === FUNCTION) {
								options.onClick.call(self[0], e);
							}
						},
						
					// Handling doubleclick event
					handleDoubleClick = function(e) {
							//log('handleDoubleClick');
							unregisterEvents(e);
							self.data('scrolling', false);	
							if (typeof options.onDoubleClick === FUNCTION) {
								options.onDoubleClick.call(self[0], e);
							}
						},
						
					// Handling swipe		
					handleSwipe = function(right, e) {
							//log('handleSwipe:'+(right?'right':'left'));
							unregisterEvents(e);
							if (right) {
								if (typeof options.onSwipedRight === FUNCTION) {
									options.onSwipedRight.call(self[0], e);
								}
							} else {
								if (typeof options.onSwipedLeft === FUNCTION) {
									options.onSwipedLeft.call(self[0], e);
								}
							}
						},
							
					// Handling right click event
					handleRightClick = function(e) {
							//log('handleRightClick');
							unregisterEvents(e);
							self.data('scrolling', false);	
							if (typeof options.onRightClick === FUNCTION) {
								options.onRightClick.call(self[0], e);
							}
						},
					
					// Registering events
					registerEvents = function(e) { 
							//log('registerEvents');
							if (e.type === 'mousedown') {
								self.on('mouseup.' + ns + ' mouseout.' + ns, swipeEnd);
								$document.on('mousemove.' + ns, swipeMove);
							} else if (e.type === TOUCH.START) {
								self.on(TOUCH.END + '.' + ns, swipeEnd);
							}
							clickedOnce = false;
						},
						
					unregisterEvents = function(e) {
							//log('unregisterEvents');
							if (e.type === 'mouseup' || e.type === 'mouseout') {
								self.off('mouseup.' + ns + ' mouseout.' + ns);
								$document.off('mousemove.' + ns);
							} else if (e.type === TOUCH.END) {
								self.off(TOUCH.END + '.' + ns);
							}
						},
					
					// Starting swipe
					swipeStart = function(e) {
							//log('swipeStart:' + e.type);
							
							if (clickedOnce) {
								// No mouseup / touchend registered yet
								return false;
							}
							
							if (e.type !== 'touchstart') {
								// Cancel bubbling, but not break long tap
								e.preventDefault();
							}
							
							if (e.type === 'mousedown' && e.which !== 1) {									
								// Right click
								handleRightClick(e);
								return true;
							}
							
							//log('Touch start, fingers: ' + (e.originalEvent.touches? e.originalEvent.touches.length : 0));
							if (pageZoomed() ||
								e.originalEvent.touches && e.originalEvent.touches.length > 1 ||			// multi finger touch
								(options.noSwipeInside && $(e.target).closest(options.noSwipeInside).length) ||
								e.target.nodeName === 'A' ||
								e.target.nodeName === 'INPUT' ||
								e.target.nodeName === 'BUTTON' ||
								e.target.nodeName === 'AUDIO' || 
								(e.target.nodeName === 'VIDEO' && 
									e.target.controls && 
									(ISIOSDEVICE || e.offsetY > ($(e.target).height() - 60)))
								) {
								// Don't swipe buttons and links and other non-swipeable elements
								// No swiping on iOS video player
								nonSwipeable = true;
								return true;
							}
							
							nonSwipeable = false;
							ended = false;
							clickedOnce = false;
							
							//self.stop(true, false);
							self.data({
									scrolling: 		true,
									swipeEnded:		false,
									taplength:		0
								});
							
							startTime = new Date().getTime();
							startPos = getXY(e.originalEvent);
							
							clipDim = getDim(clip);
							selfDim = getDim(self);
							
							constrainX = clipDim[1] >= selfDim[1];
							
							diff = [0, 0];
							pos = self.translate();
							
							registerEvents(e);
							
							// Do not cancel long tap
							return e.type === 'touchstart';
						},
					
					// Moving in progress
					swipeMove = function(e) {
							//log('swipeMove:' + e.type);
							
							if (ended || nonSwipeable || 
								e.originalEvent.touches && e.originalEvent.touches.length > 1) {
								// not active, prohibited area or multi finger touch
								return true;
							}
							
							e.preventDefault();
							
							if (constrainX) {
								diff = [ getX(e.originalEvent) - startPos[0], 0 ];
							} else {
								diff = getXY(e.originalEvent);
								diff[0] -= startPos[0];
								diff[1] -= startPos[1];
							}
							
							self.translate([pos[0] + diff[0], pos[1] + diff[1]]);
							
							return false;
						},
					
					// Cancel swipe (e.g. moved out of the screen)
					swipeCancel = function(e) {
							//log('swipeCancel:' + e.type);
							
							ended = true;
							
							setTimeout(function() { 
									self.data({
											scrolling:	 	false,
											swipeEnded:		true
										});
								}, 100);
							
							self.translate([0, 0], 100);
							
							return false;
						},
						
					// Stopped moving or clicked : mouse/touch up
					swipeEnd = function(e) {
							//log('swipeEnd:' + e.type);
							if (nonSwipeable) {
								// Prohibited area
								return true;
							}
							
							e.preventDefault();
							
							if (chkDoubleClick) {
								clearTimeout(chkDoubleClick);
								chkDoubleClick = null;
							}
							
							ended = true;
							endTime = new Date().getTime();
							
							if ((endTime - startTime) > 50 && (getDistance(diff) > options.threshold)) {
								//log('swiped: ' + (endTime - startTime) + 'ms ' + getDistance(diff) + 'px');
								// Not a click, because it took longer or the distance is larger than the threshold
								var speed = 1 + options.swipeoutSpeed / (endTime - startTime);
								
								self.data({
										scrolling:	 	true,
										swipeEnded:		true,
										taplength:		endTime - startTime
									});
								
								if (constrainX) {
									var targetPos = pos[0] + diff[0] * speed;
									
									if ((clipDim[0] >= selfDim[0]) ||
											(diff[0] > 0) &&
											((targetPos - selfDim[0] / 2) > clipDim[0] * (options.overThreshold - 0.5)) ||
											((targetPos + selfDim[0] / 2) < clipDim[0] * (0.5 - options.overThreshold))
										) {
										self.translateXAndFade(targetPos, 0, options.swipeoutSpeed);
										timer = setTimeout(handleFinished, options.swipeoutSpeed + 20, e);
										handleSwipe(diff[0] > 0, e);
										
									} else {
										// Panoramic :: Hasn't reached the edge yet
										self.translateX(targetPos, options.swipeoutSpeed);
									}
									
								} else {
									var top = parseInt(self.css('top')),
										targetPos = [
												pos[0] + diff[0] * speed,
												((diff[1] > 0)? 
													Math.min(-top, pos[1] + diff[1] * speed)
													:
													Math.max(-top + clipDim[1] - selfDim[1], pos[1] + diff[1] * speed))
											];
										
									if ((Math.abs(diff[0]) > Math.abs(diff[1])) &&
											(
												((targetPos[0] - selfDim[0] / 2) > clipDim[0] * (options.overThreshold - 0.5)) ||
												((targetPos[0] + selfDim[0] / 2) < clipDim[0] * (0.5 - options.overThreshold))
											)
										) { 
											
										// Swipe action: only for horizontal move
										//self.data('scrolling', true);
										self.translateAndFade(targetPos, 0, options.swipeoutSpeed);
										timer = setTimeout(handleFinished, options.swipeoutSpeed + 20, e);
										handleSwipe(diff[0] >= 0, e);
										
									} else {
										// Hasn't reached the edge yet
										self.translate(targetPos, options.swipeoutSpeed);
									}
								}	
								
								clickedOnce = false;
								
							} else if (diff[0] || diff[1]) {
								// Small swipe: no click triggered
								handleSmallSwipe(e);
								self.translateX(0, 100);
														
								clickedOnce = false;
								
							} else if (options.onDoubleClick) {
								
								// Has doubleclick handler event
								if (clickedOnce) {
									// One click already registered
									if ((new Date().getTime() - endTime) < options.doubleClickThreshold) {
										// Within threshold: Double click
										handleDoubleClick(e);
									} else {
										// Over threshold
										handleClick(e);
									}
									clickedOnce = false;
										
								} else { 
									// Wait if second click happens
									clickedOnce = true;
									// Triggering single click if no second click comes
									chkDoubleClick = setTimeout(function(e) {
											handleClick(e);
											clickedOnce = false;
										}, options.doubleClickThreshold + 10, e);
								}
								
							} else {
								// Handling only single clicks
								handleClick(e);
							}
							
							return false;
						},
					
					noAction = function(e) {
							e.preventDefault();
							return false;
						},
					
					// Tearing off the swipe handler
					removeSwipe = function() {
							setTimeout(function() { 
									self.data('scrolling', false);
								}, 20);
							self.removeAttr('draggable');
							self.add($document).off('.' + ns);
						};
					
					
				// clean up events that may hang around (in theory this should never happen)
				self.add($document).off('.' + ns);
				
				// Storing namespace
				self.data('lsw_ns', ns);
				
				// Need to be updated about the window size
				$window.on('resize.' + ns, function() {
						clearTimeout(rto);
						rto = setTimeout(function() {
							clipDim = getDim(clip);		
						}, 50);
					});
				
				self.attr('draggable', 'false')
					.on(TOUCH.START + '.' + ns + ' dragstart.' + ns + ' mousedown.' + ns, swipeStart)
					//.on(TOUCH.CANCEL + '.' + ns + ' dragcancel.' + ns, swipeCancel)
					.on(TOUCH.MOVE + '.' + ns + ' drag.' + ns, swipeMove)
					// External remove request
					.on('removeSwipe.' + ns, removeSwipe)
					// Do not select
					.on('selectstart.' + ns, noAction);
				
				//self.logEvents();
		
			});
		};
			
})(jQuery, jQuery(window), jQuery(document));
