import se.datadosen.jalbum.*;

import java.awt.*;
import java.awt.image.BufferedImage;

import java.util.*;

/**
 * JAlbum filter that makes sure that images have their aspect ratio constrained
 * within certain bounds
 * @version 1.0
 */
public class ConstrainRatioFilter implements JAFilter, ModifiesSize {
    
    private float xWeight = 0.5f;
    private float yWeight = 0.2f;
    private float minRatio = 0.5f;
    private float maxRatio = 2f;

    public ConstrainRatioFilter() {} // Default constructor for Beans

    public ConstrainRatioFilter(float minRatio, float maxRatio) {
        setMinRatio(minRatio);
        setMaxRatio(maxRatio);
    }

    public float getXWeight() {
        return xWeight;
    }

    public float getMinRatio() {
        return minRatio;
    }

    public float getMaxRatio() {
        return maxRatio;
    }

    public void setMinRatio(float minRatio) {
        this.minRatio = minRatio;
    }

    public void setMaxRatio(float maxRatio) {
        this.maxRatio = maxRatio;
    }

    /**
     * Control what area of the image to use when cropping occurs
     * Defaults to 50%, i.e focus on the middle part of the source image
     * Can be overridden by the "xWeight" user variable
     */
    public void setXWeight(float xWeight) {
        this.xWeight = xWeight;
    }

    public float getYWeight() {
        return yWeight;
    }

    /**
     * Control what area of the image to use when cropping occurs
     * Defaults to 20%, i.e focus on the topmost part of the source image
     * Can be overridden by the "yWeight" user variable
     */
    public void setYWeight(float yWeight) {
        this.yWeight = yWeight;
    }    
    
    @Override
    public String getName() {
        return "Constrain ratio filter";
    }

    // Implements JAFilter
    @Override
    public String getDescription() {
        return "Constrains the aspect ratio of images to be within the specificd min- and max ratios";
    }

    // Implements JAFilter
    @Override
    public BufferedImage filter(BufferedImage bi, java.util.Map vars) {
    	bi = doFilter(bi, vars);
    	return bi;
    }

    // Implements JAFilter
    private BufferedImage doFilter(BufferedImage bi, java.util.Map vars) {
        Dimension sDim = new Dimension(bi.getWidth(), bi.getHeight());
        Rectangle bounds = getBounds(sDim, vars);

        if (bounds.x != 0 || bounds.y != 0 || bounds.width != bi.getWidth() || bounds.height != bi.getHeight()) {
            return bi.getSubimage(bounds.x, bounds.y, bounds.width, bounds.height);
        } else {
            return bi; // Already right shape
        }
    }

    private Rectangle getBounds(Dimension sDim, Map vars) {
        AlbumBean engine = (AlbumBean)vars.get("engine");

        float hw = xWeight;
        if (vars.containsKey("xWeight")) {
            Object xWeight = vars.get("xWeight");
            hw = (xWeight instanceof Double) ? ((Double)xWeight).floatValue() : Float.parseFloat((String)xWeight);
        }
        float vw = yWeight;
        if (vars.containsKey("yWeight")) {
            Object yWeight = vars.get("yWeight");
            vw = (yWeight instanceof Double) ? ((Double)yWeight).floatValue() : Float.parseFloat((String)yWeight);
        }
        
        double sAspekt = ((double)sDim.width) / sDim.height;
        double dAspekt = Math.min(maxRatio, Math.max(minRatio, sAspekt));

        if (sAspekt > dAspekt) { // Crop image to the left and right to fit

            int newWidth = (int)(dAspekt * (sDim.height));

            return new Rectangle((int)((sDim.width - newWidth) * hw), 0, newWidth, sDim.height);
        } else if (sAspekt < dAspekt) { // Crop image to the top and bottom to fit

            int newHeight = (int)((sDim.width) / dAspekt);

            return new Rectangle(0, (int)((sDim.height - newHeight) * vw), sDim.width, newHeight);
        } else {
            return new Rectangle(0,0, sDim.width, sDim.height); // Already right shape
        }        
    }


    // Implements ModifiesSize
    @Override
    public Dimension getModifiedSize(Dimension originalSize, Map vars) {
        return getBounds(originalSize, vars).getSize();
    }
}
