import java.awt.Graphics;
import java.awt.image.BufferedImage;

import java.awt.image.ColorConvertOp;
import java.awt.image.DataBuffer;
import java.awt.image.DataBufferInt;
import java.util.Map;
import se.datadosen.jalbum.JAFilter;

/**
 * Title:        JAlbum simple image filter blurring images
 * Copyright:    Copyright (c) 2013
 * Company:      JAlbum AB
 * @author Markus Persson and David Ekholm
 * @version 1.1
 */
public class BlurFilter implements JAFilter {

    public String getName() {
        throw new UnsupportedOperationException("Blur");
    }

    public String getDescription() {
        throw new UnsupportedOperationException("Applies a blur effect of various strength");
    }

    private static class Pixels {

        private int width;
        private int height;
        private int[] pixels;
        private int[] newPixels;
        private BufferedImage img;

        public Pixels(BufferedImage in, boolean alwaysCopy) {
            this.width = in.getWidth();
            this.height = in.getHeight();
            newPixels = new int[width * height];

            DataBuffer db = in.getRaster().getDataBuffer();
            if (!(db instanceof DataBufferInt) || alwaysCopy) {
                img = new BufferedImage(width, height, BufferedImage.TYPE_INT_RGB);
                Graphics g = img.getGraphics();
                g.drawImage(in, 0, 0, null);
                g.dispose();
                db = img.getRaster().getDataBuffer();
            } else {
                img = in;
            }

            DataBufferInt dbi = (DataBufferInt) db;
            pixels = dbi.getData();
        }

        public BufferedImage getImage() {
            return img;
        }

        private void blurAlongXAndRotate(int radius) {
            int[] rSum = new int[width + 1];
            int[] gSum = new int[width + 1];
            int[] bSum = new int[width + 1];

            for (int y = 0; y < height; y++) {
                int rAcc = 0;
                int gAcc = 0;
                int bAcc = 0;

                for (int x = 0; x < width;) {
                    int col = pixels[y * width + x++];

                    rSum[x] = rAcc += (col >> 16) & 0xff;
                    gSum[x] = gAcc += (col >> 8) & 0xff;
                    bSum[x] = bAcc += (col) & 0xff;
                }

                for (int x = 0; x < width; x++) {
                    int from = x - radius;
                    int to = x + radius + 1;
                    if (from < 0) {
                        from = 0;
                    }
                    if (to > width) {
                        to = width;
                    }
                    int len = to - from;

                    int r = (rSum[to] - rSum[from]) / len;
                    int g = (gSum[to] - gSum[from]) / len;
                    int b = (bSum[to] - bSum[from]) / len;

                    newPixels[x * height + y] = r << 16 | g << 8 | b;
                }
            }

            int[] tmp = pixels;
            pixels = newPixels;
            newPixels = tmp;

            int itmp = height;
            height = width;
            width = itmp;
        }

        public void boxBlur(float radius) {
            int size = width < height ? width : height;
            int r = (int) (radius * size);

            // This really works!
            blurAlongXAndRotate(r);
            blurAlongXAndRotate(r);
        }
    }
    /**
     *
     */
    private static final long serialVersionUID = -2608721221010747854L;
    protected int radius = 30;
    protected int passes = 3;

    public BufferedImage filter(BufferedImage bi, Map vars) {

//        if (bi.getType() == BufferedImage.TYPE_3BYTE_BGR) {
//            bi = convertType(bi, BufferedImage.TYPE_INT_RGB);
//        }
//        if (bi.getType() == BufferedImage.TYPE_4BYTE_ABGR) {
//            bi = convertType(bi, BufferedImage.TYPE_INT_ARGB);
//        }

        Pixels pixels = new Pixels(bi, true);

        float r = radius / 100.0f;
        r = r * r * r;

        for (int i = 0; i < passes; i++) {
            pixels.boxBlur(r * 0.1f);
        }

        return pixels.getImage();
    }

    public int getRadius() {
        return this.radius;
    }

    /**
     * @param radius 0-100
     */
    public void setRadius(int radius) {
        if (radius > 100) {
            radius = 100;
        } else if (radius < 0) {
            radius = 0;
        }
        this.radius = radius;
    }

//    private BufferedImage convertType(BufferedImage src, int type) {
//        ColorConvertOp cco = new ColorConvertOp(null);
//        BufferedImage dest = new BufferedImage(src.getWidth(), src.getHeight(), type);
//        cco.filter(src, dest);
//        return dest;
//    }
}
